'use client';

import React, { createContext, useContext, useState, useEffect } from 'react';
import { externalApiClient } from '@/lib/apiClient';

// 언어 파일 타입 정의
interface Translations {
  [key: string]: any;
}

interface LanguageContextType {
  language: string;
  translations: Translations;
  setLanguage: (lang: string) => void;
  t: (key: string, fallback?: string) => string;
  isLoading: boolean;
}

const LanguageContext = createContext<LanguageContextType | undefined>(undefined);

export const useLanguage = () => {
  const context = useContext(LanguageContext);
  if (!context) {
    throw new Error('useLanguage must be used within a LanguageProvider');
  }
  return context;
};

export const LanguageProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  // ~/.airun/airun.conf의 LANGUAGE 설정을 기본으로 사용
  const getDefaultLanguage = () => {
    if (typeof window !== 'undefined') {
      // Next.js 환경변수 체크 (클라이언트 사이드에서 접근 가능)
      const envLanguage = process.env.NEXT_PUBLIC_LANGUAGE;
      if (envLanguage) return envLanguage;

      // 브라우저 언어 체크
      const browserLang = navigator.language.split('-')[0];
      const supportedLanguages = ['ko', 'en', 'ja', 'zh', 'fr', 'es'];
      return supportedLanguages.includes(browserLang) ? browserLang : 'en';
    }
    return 'en';
  };

  const [language, setLanguageState] = useState<string>(getDefaultLanguage());
  const [translations, setTranslations] = useState<Translations>({});
  const [isLoading, setIsLoading] = useState(true);

  // 번역 파일 로드
  const loadTranslations = async (lang: string) => {
    try {
      const response = await fetch(`/lang/${lang}.json`);
      if (!response.ok) {
        throw new Error(`Failed to load ${lang} translations`);
      }
      const translations = await response.json();
      setTranslations(translations);
    } catch (error) {
      console.error('Error loading translations:', error);
      // 기본값으로 영어 로드 시도
      if (lang !== 'en') {
        try {
          const response = await fetch('/lang/en.json');
          const fallbackTranslations = await response.json();
          setTranslations(fallbackTranslations);
        } catch (fallbackError) {
          console.error('Error loading fallback translations:', fallbackError);
        }
      }
    }
  };

  // 사용자 언어 설정 불러오기
  const loadUserLanguage = async () => {
    try {
      // 먼저 서버 설정에서 LANGUAGE 값 확인
      try {
        const configRes = await externalApiClient.get('/config');
        if (configRes.success && configRes.data && configRes.data.configs && configRes.data.configs.LANGUAGE) {
          setLanguageState(configRes.data.configs.LANGUAGE);
          await loadTranslations(configRes.data.configs.LANGUAGE);
          return;
        }
      } catch (error) {
        console.error('Error loading server language config:', error);
      }

      // 로그인 상태 확인 (localStorage에서 토큰 체크)
      const token = typeof window !== 'undefined' ? localStorage.getItem('userToken') : null;

      if (token) {
        // 로그인된 경우에만 API 호출
        try {
          const res = await externalApiClient.get('/auth/me');

          if (res.success && res.data && res.data.language) {
            setLanguageState(res.data.language);
            await loadTranslations(res.data.language);
            return;
          }
        } catch (error) {
          console.error('Error loading user language from API:', error);
          // API 호출 실패 시 기본값 사용
        }
      }

      // 비로그인 상태이거나 API 호출 실패 시 기본값 사용
      const defaultLang = getDefaultLanguage();
      setLanguageState(defaultLang);
      await loadTranslations(defaultLang);
    } catch (error) {
      console.error('Error in loadUserLanguage:', error);
      const defaultLang = getDefaultLanguage();
      setLanguageState(defaultLang);
      await loadTranslations(defaultLang);
    } finally {
      setIsLoading(false);
    }
  };

  // 언어 변경
  const setLanguage = async (lang: string) => {
    setLanguageState(lang);
    await loadTranslations(lang);
  };

  // 번역 함수
  const t = (key: string, fallback?: string): string => {
    const keys = key.split('.');
    let value = translations;
    
    for (const k of keys) {
      if (value && typeof value === 'object' && k in value) {
        value = value[k];
      } else {
        return fallback || key;
      }
    }
    
    return typeof value === 'string' ? value : fallback || key;
  };

  useEffect(() => {
    loadUserLanguage();
    
    // 언어 변경 이벤트 리스너 추가
    const handleLanguageChange = () => {
      loadUserLanguage();
    };
    
    window.addEventListener('languageChanged', handleLanguageChange);
    
    return () => {
      window.removeEventListener('languageChanged', handleLanguageChange);
    };
  }, []);

  const value: LanguageContextType = {
    language,
    translations,
    setLanguage,
    t,
    isLoading,
  };

  return (
    <LanguageContext.Provider value={value}>
      {children}
    </LanguageContext.Provider>
  );
}; 