import { apiUsageQueue, licenseCheckQueue, apiRequestQueue, chatRequestQueue, reportGenerationQueue, cleanup } from './index.js';
import Bull from 'bull';
import Redis from 'ioredis';
import fs from 'fs';
import path from 'path';

// 설정 파일 로드
const configPath = path.join(path.dirname(new URL(import.meta.url).pathname), 'config.json');
let config = { redis: { host: '127.0.0.1', port: 6379 } };

try {
  if (fs.existsSync(configPath)) {
    const configData = fs.readFileSync(configPath, 'utf8');
    config = JSON.parse(configData);
  } else {
    console.warn('큐 유틸리티: 설정 파일을 찾을 수 없습니다. 기본 설정을 사용합니다.');
  }
} catch (error) {
  console.error('큐 유틸리티: 설정 파일 로드 중 오류:', error);
}

// 모든 큐 목록
const queues = [
  apiUsageQueue,
  licenseCheckQueue,
  apiRequestQueue,
  chatRequestQueue,
  reportGenerationQueue
];

// Redis 클라이언트 생성
const redisClient = new Redis(config.redis);

// 큐 상태 모니터링
export async function monitorQueueHealth() {
  try {
    const stats = await getQueueStatistics();
    console.log('=== 큐 상태 모니터링 ===');
    
    // 모든 큐의 작업 목록 수집
    const allJobs = [];
    
    for (const queue of queues) {
      try {
        // 다양한 상태의 작업들을 모두 가져오기
        const [activeJobs, waitingJobs, completedJobs, failedJobs] = await Promise.all([
          queue.getJobs(['active'], 0, 50),
          queue.getJobs(['waiting'], 0, 50),
          queue.getJobs(['completed'], 0, 50),
          queue.getJobs(['failed'], 0, 50)
        ]);
        
        // 모든 작업을 하나의 배열로 합치기
        const queueJobs = [...activeJobs, ...waitingJobs, ...completedJobs, ...failedJobs];
        
        // 작업 정보 변환
        for (const job of queueJobs) {
          const state = await job.getState();
          allJobs.push({
            id: job.id,
            name: job.name,
            queue: queue.name,
            state: state,
            data: job.data,
            timestamp: job.timestamp,
            processedOn: job.processedOn,
            finishedOn: job.finishedOn,
            createdAt: new Date(job.timestamp).toISOString(),
            jobId: job.id,
            attemptsMade: job.attemptsMade,
            returnvalue: job.returnvalue,
            failedReason: job.failedReason
          });
        }
        
        const queueStats = stats[queue.name];
        console.log(`${queue.name}:`);
        console.log(`  - 대기 중: ${queueStats.waiting}`);
        console.log(`  - 처리 중: ${queueStats.active}`);
        console.log(`  - 완료됨: ${queueStats.completed}`);
        console.log(`  - 실패함: ${queueStats.failed}`);
        console.log(`  - 지연됨: ${queueStats.delayed}`);
        console.log(`  - 총 작업 수: ${queueJobs.length}`);
      } catch (error) {
        console.error(`큐 ${queue.name} 작업 목록 조회 오류:`, error);
      }
    }
    
    // 최신 작업 순으로 정렬
    allJobs.sort((a, b) => {
      const timeA = b.processedOn || b.timestamp || 0;
      const timeB = a.processedOn || a.timestamp || 0;
      return timeA - timeB;
    });
    
    return {
      ...stats,
      jobs: allJobs,
      totalJobs: allJobs.length
    };
  } catch (error) {
    console.error('큐 상태 모니터링 오류:', error);
    throw error;
  }
}

// 큐 통계 수집
export async function getQueueStatistics() {
  const stats = {};
  
  for (const queue of queues) {
    const [waiting, active, completed, failed, delayed] = await Promise.all([
      queue.getWaitingCount(),
      queue.getActiveCount(),
      queue.getCompletedCount(),
      queue.getFailedCount(),
      queue.getDelayedCount()
    ]);
    
    stats[queue.name] = {
      waiting,
      active,
      completed,
      failed,
      delayed
    };
  }
  
  return stats;
}

// 특정 큐의 모든 작업 목록 조회
export async function listJobs(queueName, status = 'active') {
  const queue = queues.find(q => q.name === queueName);
  
  if (!queue) {
    throw new Error(`큐를 찾을 수 없습니다: ${queueName}`);
  }
  
  try {
    // 상태별 작업 조회 (active, waiting, completed, failed, delayed)
    const jobs = await queue.getJobs([status], 0, 100); // 최대 100개 조회
    
    return jobs.map(job => ({
      id: job.id,
      data: job.data,
      timestamp: job.timestamp,
      processedOn: job.processedOn,
      finishedOn: job.finishedOn,
      attemptsMade: job.attemptsMade,
      returnvalue: job.returnvalue,
      failedReason: job.failedReason
    }));
  } catch (error) {
    console.error(`작업 목록 조회 오류 (${queueName}/${status}):`, error);
    throw error;
  }
}

// 특정 작업의 상태 조회
export async function getJobStatus(queueName, jobId) {
  const queue = queues.find(q => q.name === queueName);
  
  if (!queue) {
    throw new Error(`큐를 찾을 수 없습니다: ${queueName}`);
  }
  
  try {
    const job = await queue.getJob(jobId);
    
    if (!job) {
      return { exists: false, message: '작업을 찾을 수 없습니다.' };
    }
    
    const state = await job.getState();
    
    return {
      exists: true,
      id: job.id,
      state,
      data: job.data,
      timestamp: job.timestamp,
      processedOn: job.processedOn,
      finishedOn: job.finishedOn,
      attemptsMade: job.attemptsMade,
      returnvalue: job.returnvalue,
      failedReason: job.failedReason
    };
  } catch (error) {
    console.error(`작업 상태 조회 오류 (${queueName}/${jobId}):`, error);
    throw error;
  }
}

// 모든 큐에서 특정 작업 찾기
export async function getJobFromAllQueues(jobId) {
  for (const queue of queues) {
    try {
      const job = await queue.getJob(jobId);
      if (job) {
        const state = await job.getState();
        console.log(`작업 ${jobId}를 큐 ${queue.name}에서 발견 (상태: ${state})`);
        return {
          job,
          queueName: queue.name,
          state,
          exists: true
        };
      }
    } catch (error) {
      // 이 큐에서 작업을 찾지 못했으므로 다음 큐 확인
      console.debug(`큐 ${queue.name}에서 작업 ${jobId} 조회 중 오류:`, error.message);
      continue;
    }
  }
  
  console.log(`모든 큐에서 작업 ${jobId}를 찾을 수 없습니다.`);
  return null;
}

// 실패한 작업 재시도
export async function retryFailedJob(queueName, jobId) {
  const queue = queues.find(q => q.name === queueName);
  
  if (!queue) {
    throw new Error(`큐를 찾을 수 없습니다: ${queueName}`);
  }
  
  try {
    const job = await queue.getJob(jobId);
    
    if (!job) {
      throw new Error('작업을 찾을 수 없습니다.');
    }
    
    const state = await job.getState();
    
    if (state !== 'failed') {
      throw new Error(`실패한 작업만 재시도할 수 있습니다. 현재 상태: ${state}`);
    }
    
    // 작업 재시도
    await job.retry();
    console.log(`${queueName} 큐의 작업 ${jobId}를 재시도했습니다.`);
    
    return { 
      success: true, 
      message: `작업 ${jobId}이 재시도되었습니다.`,
      jobId: jobId,
      queueName: queueName
    };
  } catch (error) {
    console.error(`작업 재시도 오류 (${queueName}/${jobId}):`, error);
    throw error;
  }
}

// 모든 실패한 작업 재시도
export async function retryAllFailedJobs(queueName) {
  const queue = queues.find(q => q.name === queueName);
  
  if (!queue) {
    throw new Error(`큐를 찾을 수 없습니다: ${queueName}`);
  }
  
  try {
    const failedJobs = await queue.getJobs(['failed'], 0, 100);
    const retryResults = [];
    
    for (const job of failedJobs) {
      try {
        await job.retry();
        retryResults.push({ jobId: job.id, success: true });
        console.log(`${queueName} 큐의 작업 ${job.id}를 재시도했습니다.`);
      } catch (error) {
        retryResults.push({ jobId: job.id, success: false, error: error.message });
        console.error(`작업 ${job.id} 재시도 실패:`, error);
      }
    }
    
    const successCount = retryResults.filter(r => r.success).length;
    const failureCount = retryResults.filter(r => !r.success).length;
    
    return {
      success: true,
      message: `${successCount}개 작업 재시도 성공, ${failureCount}개 실패`,
      queueName: queueName,
      totalJobs: failedJobs.length,
      successCount,
      failureCount,
      results: retryResults
    };
  } catch (error) {
    console.error(`모든 실패한 작업 재시도 오류 (${queueName}):`, error);
    throw error;
  }
}

// 큐 초기화 (모든 작업 제거)
export async function cleanupQueue(queueName) {
  const queue = queues.find(q => q.name === queueName);
  
  if (!queue) {
    throw new Error(`큐를 찾을 수 없습니다: ${queueName}`);
  }
  
  try {
    // 모든 상태의 작업 제거
    await queue.empty();
    console.log(`${queueName} 큐의 모든 작업이 제거되었습니다.`);
    
    return { success: true, message: `${queueName} 큐가 초기화되었습니다.` };
  } catch (error) {
    console.error(`큐 초기화 오류 (${queueName}):`, error);
    throw error;
  }
}

// Redis 상태 확인
export async function checkRedisStatus() {
  try {
    const pong = await redisClient.ping();
    const info = await redisClient.info();
    const memory = await redisClient.info('memory');
    
    // 메모리 사용량 파싱
    const usedMemoryMatch = memory.match(/used_memory:(\d+)/);
    const usedMemoryHuman = memory.match(/used_memory_human:([^\r\n]+)/);
    
    return {
      status: pong === 'PONG' ? 'connected' : 'error',
      memoryUsage: usedMemoryHuman ? usedMemoryHuman[1] : 'unknown',
      memoryBytes: usedMemoryMatch ? parseInt(usedMemoryMatch[1]) : 0,
      info: {
        redis_version: info.match(/redis_version:([^\r\n]+)/)?.[1],
        uptime_in_seconds: parseInt(info.match(/uptime_in_seconds:(\d+)/)?.[1] || '0'),
        connected_clients: parseInt(info.match(/connected_clients:(\d+)/)?.[1] || '0')
      }
    };
  } catch (error) {
    console.error('Redis 상태 확인 오류:', error);
    return {
      status: 'error',
      error: error.message
    };
  }
}

// 모든 리소스 정리
export async function shutdownQueueSystem() {
  try {
    console.log('큐 시스템 종료 중...');
    
    // 모든 큐 닫기
    await cleanup();
    
    // Redis 연결 종료
    await redisClient.quit();
    
    console.log('큐 시스템이 안전하게 종료되었습니다.');
    return { success: true };
  } catch (error) {
    console.error('큐 시스템 종료 오류:', error);
    throw error;
  }
}

// 큐 시스템 상태 취합
export async function getSystemStatus() {
  try {
    const queueStats = await getQueueStatistics();
    const redisStatus = await checkRedisStatus();
    
    return {
      timestamp: new Date().toISOString(),
      redis: redisStatus,
      queues: queueStats
    };
  } catch (error) {
    console.error('시스템 상태 조회 오류:', error);
    throw error;
  }
} 