import { spawn } from 'child_process';
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import { getMainWindow } from './index.js';
import ProcessManager from '../../processManager.js';
import path from 'path';
import os from 'os';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

export const executeTask = async (taskData) => {
    const window = getMainWindow();
    
    try {
        if (!taskData || !taskData.prompt) {
            throw new Error('Invalid task data: prompt is required');
        }

        // Windows 환경에서 프로세스 상태 확인 및 정리
        if (process.platform === 'win32') {
            await ProcessManager.checkAndCleanupProcesses();
        }

        // 프롬프트 문자열 이스케이프 처리
        const sanitizedPrompt = taskData.prompt
            .replace(/\\/g, '\\\\')
            .replace(/"/g, '\\"')
            .replace(/'/g, '')
            .replace(/\$/g, '\\$')
            .replace(/`/g, '\\`')
            .replace(/~/g, '\\~')
            .replace(/\n/g, ' ')
            .replace(/\r/g, ' ')
            .replace(/\t/g, ' ')
            .replace(/\&/g, '\\&')
            .replace(/\|/g, '\\|')
            .replace(/>/g, '\\>')
            .replace(/</g, '\\<')
            .replace(/\(/g, '\\(')
            .replace(/\)/g, '\\)')
            .replace(/\{/g, '\\{')
            .replace(/\}/g, '\\}')
            .replace(/([A-Za-z]):\\/, '$1:\\\\');

        const executionLog = {
            timestamp: new Date().toISOString(),
            prompt: sanitizedPrompt,
            command: taskData.prompt,
            taskName: taskData.name || 'CLI Execution',
            status: 'Running',
            success: true,
            result: 'Executing in new terminal...'
        };

        // Electron 환경에서 실행
        if (window) {
            window.webContents.send('execute-prompt', { 
                prompt: sanitizedPrompt,
                taskName: taskData.name || 'Scheduled Task'
            });
            
            return {
                success: true,
                message: 'Task is being processed in chat window'
            };
        }

        // CLI 환경에서 실행
        const indexPath = path.join(dirname(dirname(__dirname)), 'index.js');
        const command = `node "${indexPath}" "${sanitizedPrompt}"`;

        return new Promise((resolve, reject) => {
            let terminalCommand;
            if (process.platform === 'win32') {
                terminalCommand = `start cmd.exe /K "${command}"`;
            } else if (process.platform === 'darwin') {
                terminalCommand = `osascript -e 'tell app "Terminal" to do script "${command}"'`;
            } else {
                const shell = process.env.SHELL && process.env.SHELL.includes('zsh') ? 'zsh' : 'bash';
                terminalCommand = `x-terminal-emulator -e ${shell} -c '${command}; exec ${shell}'`;
            }

            const spawnOptions = {
                ...ProcessManager.getWindowsSpawnOptions(),
                shell: true,
                windowsHide: false
            };

            const child = spawn(terminalCommand, [], spawnOptions);

            if (process.platform !== 'win32') {
                child.unref();
            }

            child.on('error', (error) => {
                console.error('Task execution error:', error);
                reject(error);
            });

            resolve({
                success: true,
                result: 'Task started in new terminal window'
            });
        });
    } catch (error) {
        console.error('Task execution failed:', error);
        throw error;
    }
}; 