import swaggerJsdoc from 'swagger-jsdoc';
import swaggerUi from 'swagger-ui-express';
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import { glob } from 'glob';
import { schemas, tags, securitySchemes, tagGroups } from './swagger/index.js';

// 프로젝트 루트 디렉토리 찾기
const APP_ROOT = dirname(fileURLToPath(import.meta.url));

const options = {
    definition: {
        openapi: '3.1.0',
        info: {
            title: 'API',
            version: '1.0.0',
            description: `
## 인증 방식
이 API는 두 가지 인증 방식을 사용합니다:

1. API 키 인증 (X-API-Key)
   - AI 기능 관련 엔드포인트(chat, code, agent, report, rag, web, etc)에 사용
   - API 키는 헤더의 X-API-Key에 전달

2. JWT 토큰 인증 (Bearer Token)
   - 사용자 관리, 설정 관리 등 관리자 기능에 사용
   - 로그인 후 받은 토큰을 Authorization 헤더에 Bearer 형식으로 전달
`,
        },
        servers: [
            {
                url: 'http://localhost:5500',
                description: '로컬 API 서버'
            },
            {
                url: 'http://0.0.0.0:5500',
                description: '프로덕션 서버'
            }
        ],
        tags,
        'x-tagGroups': tagGroups,
        components: {
            schemas,
            securitySchemes
        },
        security: [] // 전역 보안 설정은 제거하고 각 API별로 설정
    },
    apis: [
        // 모든 스키마 파일들을 동적으로 로드
        ...glob.sync(join(APP_ROOT, 'swagger/**/*.js')),
        join(APP_ROOT, 'api_server.js')
    ]
};

const specs = swaggerJsdoc(options);


export const setupSwagger = (app) => {
    // OpenAPI 스펙 JSON 엔드포인트
    app.get('/api/v1/docs.json', (req, res) => {
        // 프록시를 통한 프로토콜 감지 (X-Forwarded-Proto 헤더 우선 확인)
        const protocol = req.headers['x-forwarded-proto'] || req.protocol;
        const host = req.get('host');

        specs.servers = [
            {
                url: `${protocol}://${host}`,
                description: '현재 접속한 서버'
            }
        ];

        // console.log('host-', host); 
        res.setHeader('Content-Type', 'application/json');
        res.send(specs);
    });

    // Swagger UI 설정 (💡 중요: specs 대신 null)
    app.use('/api/v1/docs', swaggerUi.serve, swaggerUi.setup(null, {
        swaggerOptions: {
            url: '/api/v1/docs.json',
            persistAuthorization: true,
            filter: true,
            displayOperationId: true,
            docExpansion: 'list'
        },
        explorer: true,
        customCss: `
            .swagger-ui .topbar { display: none }
            .swagger-ui section.models { display: none }
            .swagger-ui .auth-wrapper .authorize {
                margin-right: 10px;
                background: rgba(97, 175, 254, .1);
            }
            .swagger-ui .auth-container input[type=text] {
                min-width: 350px;
            }
            .swagger-ui .auth-container .wrapper {
                margin: 0;
                padding: 10px 0;
            }
            .swagger-ui .auth-btn-wrapper {
                display: flex;
                padding: 10px 0;
            }
            .swagger-ui .auth-btn-wrapper .btn {
                margin: 0 5px;
            }
            .swagger-ui .authorization__btn.locked {
                background-color: #49cc90;
            }
            .swagger-ui .authorization__btn.unlocked {
                background-color: #f93e3e;
            }
            .swagger-ui .auth-container .locked {
                background-color: #e8f6f0;
                border: 1px solid #49cc90;
                padding: 10px;
                border-radius: 4px;
            }
            .swagger-ui .auth-container .locked::before { 
                content: "✓ 인증되었습니다";
                color: #49cc90;
                font-weight: bold;
                display: block;
                margin-bottom: 5px;
            }
            .swagger-ui .auth-container .unlocked {
                background-color: #fff1f0;
                border: 1px solid #f93e3e;
                padding: 10px;
                border-radius: 4px;
            }
        `,
        customSiteTitle: 'API 문서'
    }));
};

// export const setupSwagger = (app) => {
//     // Swagger UI 설정
//     app.use('/api/v1/docs', swaggerUi.serve, swaggerUi.setup(specs, {
//         explorer: true,
//         customCss: `
//             .swagger-ui .topbar { display: none }
//             .swagger-ui section.models { display: none }
//             .swagger-ui .auth-wrapper .authorize {
//                 margin-right: 10px;
//                 background: rgba(97, 175, 254, .1);
//             }
//             .swagger-ui .auth-container input[type=text] {
//                 min-width: 350px;
//             }
//             .swagger-ui .auth-container .wrapper {
//                 margin: 0;
//                 padding: 10px 0;
//             }
//             .swagger-ui .auth-btn-wrapper {
//                 display: flex;
//                 padding: 10px 0;
//             }
//             .swagger-ui .auth-btn-wrapper .btn {
//                 margin: 0 5px;
//             }
//             .swagger-ui .authorization__btn.locked {
//                 background-color: #49cc90;
//             }
//             .swagger-ui .authorization__btn.unlocked {
//                 background-color: #f93e3e;
//             }
//             .swagger-ui .auth-container .locked {
//                 background-color: #e8f6f0;
//                 border: 1px solid #49cc90;
//                 padding: 10px;
//                 border-radius: 4px;
//             }
//             .swagger-ui .auth-container .locked::before {
//                 content: "✓ 인증되었습니다";
//                 color: #49cc90;
//                 font-weight: bold;
//                 display: block;
//                 margin-bottom: 5px;
//             }
//             .swagger-ui .auth-container .unlocked {
//                 background-color: #fff1f0;
//                 border: 1px solid #f93e3e;
//                 padding: 10px;
//                 border-radius: 4px;
//             }
//         `,
//         customSiteTitle: 'AI.RUN API 문서',
//         swaggerOptions: {
//             persistAuthorization: true,
//             filter: true,
//             displayOperationId: true,
//             docExpansion: 'list'
//         }
//     }));

//     // OpenAPI 스펙 JSON 엔드포인트
//     app.get('/api/v1/docs.json', (req, res) => {
//         res.setHeader('Content-Type', 'application/json');
//         res.send(specs);
//     });
// }; 

