#!/usr/bin/env node

import { Server } from '@modelcontextprotocol/sdk/server/index.js';
import { StdioServerTransport } from '@modelcontextprotocol/sdk/server/stdio.js';
import { CallToolRequestSchema, ListToolsRequestSchema } from '@modelcontextprotocol/sdk/types.js';
import { z } from 'zod';
import { zodToJsonSchema } from 'zod-to-json-schema';
import axios, { AxiosInstance } from 'axios';
import fs from 'fs';
import * as path from 'path';
import * as os from 'os';
import dotenv from 'dotenv';
import { spawn } from 'child_process';
import { fileURLToPath } from 'url';
import { dirname } from 'path';

// 환경 변수 로드
dotenv.config();

// 설정 파일 경로
const CONFIG_DIR = path.join(os.homedir(), '.airun');
const CONFIG_FILE = path.join(CONFIG_DIR, 'airun.conf');

// API 서버 설정
const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL?.replace('/api/v1', '') || 'http://localhost:5500';

// Python 가상환경 경로
const PYTHON_VENV = path.join(os.homedir(), '.airun_venv/bin/python');

// Zod 스키마 정의
const GenerateCodeSchema = z.object({
  prompt: z.string().describe("코드 생성에 사용할 프롬프트"),
  language: z.string().default("python").describe("생성할 코드의 프로그래밍 언어")
});

const WebSearchSchema = z.object({
  query: z.string().describe("검색어")
});

const RagSearchSchema = z.object({
  query: z.string().describe("검색어"),
  scope: z.string().optional().describe("검색 범위 - 'public'(공용 문서), 'private'(개인 문서), 또는 특정 사용자명 (기본값: public)")
});

const CreateReportSchema = z.object({
  content: z.string().describe("리포트 내용"),
  format: z.string().default("pdf").describe("리포트 형식"),
  template: z.string().default("simple").describe("리포트 템플릿")
});

const ExtractDocumentTextSchema = z.object({
  document_path: z.string().describe("문서 파일 경로"),
  use_ocr: z.boolean().default(false).describe("OCR 사용 여부"),
  lang: z.string().default("kor").describe("OCR 언어")
});

const ExtractHwpTablesSchema = z.object({
  hwp_path: z.string().describe("HWP 파일 경로"),
  excel_path: z.string().describe("Excel 출력 경로")
});

const ExtractHwpImagesSchema = z.object({
  hwp_path: z.string().describe("HWP 파일 경로"),
  output_dir: z.string().describe("이미지 출력 디렉토리")
});

const CropImageSchema = z.object({
  image_path: z.string().describe("이미지 파일 경로"),
  output_path: z.string().optional().describe("출력 파일 경로"),
  operation: z.string().default("rgb").describe("이미지 처리 작업"),
  x: z.number().optional().describe("X 좌표"),
  y: z.number().optional().describe("Y 좌표"),
  width: z.number().optional().describe("너비"),
  height: z.number().optional().describe("높이")
});

const ExtractWebContentSchema = z.object({
  url: z.string().describe("웹 페이지 URL"),
  extract_type: z.string().default("all").describe("추출 타입"),
  max_items: z.number().default(10).describe("최대 항목 수")
});

const SearchFileContentSchema = z.object({
  path: z.string().describe("검색할 경로"),
  query: z.string().describe("검색 쿼리"),
  file_types: z.array(z.string()).optional().describe("파일 타입 목록")
});

const CreateDocumentSchema = z.object({
  content: z.string().describe("문서 내용"),
  format: z.string().default("pdf").describe("문서 형식"),
  output_path: z.string().optional().describe("출력 파일 경로"),
  template: z.string().default("simple").describe("문서 템플릿")
});

const SendMailSchema = z.object({
  to_email: z.string().describe("받는 사람 이메일"),
  subject: z.string().describe("이메일 제목"),
  body: z.string().describe("이메일 내용"),
  attachments: z.array(z.string()).optional().describe("첨부파일 목록")
});

const TranslateTextSchema = z.object({
  text: z.string().describe("번역할 텍스트"),
  target_lang: z.string().default("KO").describe("번역 대상 언어")
});

const CreateChartSchema = z.object({
  data: z.record(z.any()).describe("차트 데이터"),
  chart_type: z.string().default("line").describe("차트 타입"),
  output_path: z.string().optional().describe("출력 파일 경로")
});

const FileOperationSchema = z.object({
  operation: z.string().describe("파일 작업 타입"),
  path: z.string().describe("파일 경로"),
  content: z.any().optional().describe("파일 내용"),
  mode: z.string().optional().describe("파일 모드")
});

const RunSystemCommandSchema = z.object({
  command: z.string().describe("실행할 시스템 명령")
});

const InstallPackageSchema = z.object({
  package_name: z.string().describe("설치할 패키지 이름")
});

const ConvertDocumentSchema = z.object({
  input_path: z.string().describe("입력 파일 경로"),
  output_format: z.string().default("pdf").describe("출력 형식")
});

const GetFileMetadataSchema = z.object({
  path: z.string().describe("파일 경로")
});

const CreateDotDiagramSchema = z.object({
  dot_code: z.string().describe("Dot 코드"),
  output_path: z.string().optional().describe("출력 파일 경로"),
  format: z.string().default("png").describe("출력 형식")
});

const CreateMermaidDiagramSchema = z.object({
  mermaid_code: z.string().describe("Mermaid 코드"),
  output_path: z.string().describe("출력 파일 경로"),
  diagram_type: z.string().default("sequence").describe("다이어그램 타입")
});

const ExtractImagesFromPdfSchema = z.object({
  pdf_path: z.string().describe("PDF 파일 경로"),
  output_dir: z.string().optional().describe("출력 디렉토리")
});

const ExtractTablesFromPdfSchema = z.object({
  pdf_path: z.string().describe("PDF 파일 경로"),
  excel_path: z.string().optional().describe("Excel 출력 경로")
});

// 설정 관리 클래스
class ConfigManager {
  private config: Record<string, string> = {};

  constructor() {
    this.loadConfig();
  }

  private loadConfig(): void {
    try {
      if (fs.existsSync(CONFIG_FILE)) {
        const content = fs.readFileSync(CONFIG_FILE, 'utf-8');
        const lines = content.split('\n');
        
        for (const line of lines) {
          const trimmed = line.trim();
          if (trimmed.startsWith('export ')) {
            const cleanLine = trimmed.substring(7);
            const equalIndex = cleanLine.indexOf('=');
            if (equalIndex > 0) {
              const key = cleanLine.substring(0, equalIndex).trim();
              const value = cleanLine.substring(equalIndex + 1).trim()
                .replace(/^["']|["']$/g, '');
              this.config[key] = value;
            }
          }
        }
      }
    } catch (error) {
      console.error('설정 파일 로드 실패:', error);
    }
  }

  public getConfig(): Record<string, string> {
    return this.config;
  }

  public saveConfig(newConfig: Record<string, string>): boolean {
    try {
      fs.mkdirSync(CONFIG_DIR, { recursive: true });
      const content = Object.entries(newConfig)
        .map(([key, value]) => `export ${key}="${value}"`)
        .join('\n');
      
      fs.writeFileSync(CONFIG_FILE, content, 'utf-8');
      this.config = { ...newConfig };
      return true;
    } catch (error) {
      console.error('설정 저장 실패:', error);
      return false;
    }
  }
}

// API 클라이언트 클래스
class ApiClient {
  private client: AxiosInstance;
  private configManager: ConfigManager;

  constructor(configManager: ConfigManager) {
    this.configManager = configManager;
    this.client = axios.create({
      baseURL: API_SERVER, // /api/v1 제거하여 중복 방지
      timeout: 30000,
      headers: {
        'Content-Type': 'application/json'
      }
    });
  }

  private getApiKey(): string | null {
    const config = this.configManager.getConfig();
    return config.AIRUN_API_KEY || null;
  }

  private getUserInfo(): { username?: string; userId?: string; userToken?: string } {
    const config = this.configManager.getConfig();
    return {
      username: config.AIRUN_USERNAME || 'admin',
      userId: config.AIRUN_USER_ID || '1',
      userToken: config.AIRUN_USER_TOKEN || undefined
    };
  }

  private getAuthHeaders(): Record<string, string> {
    const headers: Record<string, string> = {
      'Content-Type': 'application/json'
    };

    const apiKey = this.getApiKey();
    if (apiKey) {
      headers['X-API-Key'] = apiKey;
    }

    const userInfo = this.getUserInfo();
    if (userInfo.userToken) {
      headers['Authorization'] = `Bearer ${userInfo.userToken}`;
    }

    if (userInfo.username) {
      headers['x-username'] = userInfo.username;
    }

    // 기본 사용자 역할 설정
    headers['x-user-role'] = 'admin';

    return headers;
  }

  public async callApi(endpoint: string, method: 'GET' | 'POST' = 'GET', data?: any): Promise<any> {
    try {
      const apiKey = this.getApiKey();
      if (!apiKey) {
        return {
          success: false,
          error: 'AIRUN_API_KEY가 설정되지 않았습니다. ~/.airun/airun.conf 파일에서 API 키를 설정해주세요.'
        };
      }

      const headers = this.getAuthHeaders();
      
      // 요청 데이터에 사용자 정보 추가
      if (data && typeof data === 'object') {
        const userInfo = this.getUserInfo();
        data.username = userInfo.username;
        data.userId = userInfo.userId;
        
        // options가 있으면 거기에도 추가
        if (data.options) {
          data.options.username = userInfo.username;
          data.options.userId = userInfo.userId;
        }
      }

      const config = { headers };

      const response = method === 'GET' 
        ? await this.client.get(endpoint, config)
        : await this.client.post(endpoint, data, config);

      return response.data;
    } catch (error: any) {
      console.error('API 호출 실패:', error);
      return {
        success: false,
        error: error.response?.data?.error || error.message || '알 수 없는 오류'
      };
    }
  }
}

// Python 브릿지 클래스
class PythonBridge {
  private pythonPath: string;
  private airunPath: string;
  private venvPath: string;

  constructor() {
    this.venvPath = path.join(os.homedir(), '.airun_venv');
    this.pythonPath = path.join(this.venvPath, 'bin/python');
    // AI.RUN 프로젝트 루트 경로 추정 (ES 모듈 방식)
    const __filename = fileURLToPath(import.meta.url);
    const __dirname = dirname(__filename);
    this.airunPath = path.resolve(__dirname, '../../../../../..');
  }

  private getVenvEnvironment(): NodeJS.ProcessEnv {
    const env = { ...process.env };
    
    // 가상환경 설정
    env.VIRTUAL_ENV = this.venvPath;
    env.PATH = `${path.join(this.venvPath, 'bin')}:${env.PATH || ''}`;
    
    // Python 경로 설정
    env.PYTHONPATH = this.airunPath;
    
    // 가상환경 활성화 시 설정되는 기본 환경변수들
    delete env.PYTHONHOME;  // 가상환경 사용시 PYTHONHOME 제거
    
    return env;
  }

  public async callPythonFunction(functionName: string, args: any = {}): Promise<any> {
    return new Promise((resolve, reject) => {
      // args 검증 및 기본값 설정
      let validArgs = args;
      if (args === null || args === undefined) {
        validArgs = {};
      }
      if (typeof args !== 'object') {
        validArgs = { value: args };
      }
      
      // Python 스크립트 생성
      const pythonCode = `
import sys
import os
import json

# stdout을 완전히 억제하여 로그 메시지 방지
original_stdout = sys.stdout
null_out = open(os.devnull, 'w')

# AI.RUN 프로젝트 경로 추가
sys.path.insert(0, '${this.airunPath}')

def safe_json_convert(obj):
    """안전하게 JSON 변환을 수행하는 함수"""
    try:
        # 복잡한 객체들을 문자열로 변환
        if hasattr(obj, '__dict__'):
            return str(obj)
        elif hasattr(obj, 'to_dict'):
            return obj.to_dict()
        elif isinstance(obj, dict):
            # 딕셔너리의 각 값을 안전하게 변환
            return {k: safe_json_convert(v) for k, v in obj.items()}
        elif isinstance(obj, (list, tuple)):
            return [safe_json_convert(item) for item in obj]
        elif isinstance(obj, (int, float, str, bool)) or obj is None:
            return obj
        else:
            return str(obj)
    except Exception:
        return str(obj)

try:
    # stdout을 null로 리디렉션하여 모든 로그 메시지 억제
    sys.stdout = null_out
    
    import utils
    
    # 원래 stdout 복원
    sys.stdout = original_stdout
    
    # 함수 호출
    func = getattr(utils, '${functionName}')
    args = ${JSON.stringify(validArgs)}
    
    # 함수 실행
    if isinstance(args, dict):
        result = func(**args)
    elif isinstance(args, list):
        result = func(*args)
    else:
        result = func(args)
    
    # 결과를 안전하게 JSON으로 변환
    safe_result = safe_json_convert(result) if result is not None else "실행 완료"
    
    # 결과 출력 (JSON만 출력)
    print(json.dumps({
        "success": True,
        "data": safe_result
    }, ensure_ascii=False, default=str))
    
except Exception as e:
    import traceback
    # 오류 발생 시 stdout 복원
    sys.stdout = original_stdout
    print(json.dumps({
        "success": False,
        "error": str(e),
        "traceback": traceback.format_exc()
    }, ensure_ascii=False))
finally:
    # 파일 핸들 정리
    if 'null_out' in locals():
        null_out.close()
`;

      // Python 프로세스 실행 (가상환경 사용)
      const childProcess = spawn(this.pythonPath, ['-c', pythonCode], {
        cwd: this.airunPath,
        env: this.getVenvEnvironment(),
        stdio: ['pipe', 'pipe', 'pipe']
      });

      let stdout = '';
      let stderr = '';

      childProcess.stdout.on('data', (data) => {
        stdout += data.toString();
      });

      childProcess.stderr.on('data', (data) => {
        stderr += data.toString();
      });

      childProcess.on('close', (code) => {
        try {
          if (code === 0 && stdout.trim()) {
            // stdout에서 마지막 라인만 추출 (JSON 결과)
            const lines = stdout.trim().split('\n');
            let jsonLine = '';
            
            // 마지막 라인부터 거슬러 올라가면서 JSON 형태의 라인 찾기
            for (let i = lines.length - 1; i >= 0; i--) {
              const line = lines[i].trim();
              if (line.startsWith('{') && line.endsWith('}')) {
                jsonLine = line;
                break;
              }
            }
            
            if (jsonLine) {
              const result = JSON.parse(jsonLine);
              resolve(result);
            } else {
              resolve({
                success: false,
                error: `JSON 형태의 출력을 찾을 수 없음`,
                stdout: stdout
              });
            }
          } else {
            resolve({
              success: false,
              error: stderr || `프로세스가 코드 ${code}로 종료됨`,
              stdout: stdout
            });
          }
        } catch (error) {
          resolve({
            success: false,
            error: `JSON 파싱 실패: ${error}`,
            stdout: stdout,
            stderr: stderr
          });
        }
      });

      childProcess.on('error', (error) => {
        resolve({
          success: false,
          error: `프로세스 실행 실패: ${error.message}`
        });
      });

      // 10초 타임아웃
      setTimeout(() => {
        childProcess.kill();
        resolve({
          success: false,
          error: 'Python 함수 호출 타임아웃 (10초)'
        });
      }, 10000);
    });
  }
}

// MCP 서버 생성
const server = new Server({
      name: 'hamonize-mcp-server',
  version: '1.0.0'
}, {
  capabilities: {
    tools: {}
  }
});

// 설정 관리자, API 클라이언트, Python 브릿지 초기화
const configManager = new ConfigManager();
const apiClient = new ApiClient(configManager);
const pythonBridge = new PythonBridge();

// 도구 목록 정의
const tools = [
  {
    name: 'generate_code',
    description: '주어진 프롬프트를 기반으로 코드를 생성합니다.',
    inputSchema: zodToJsonSchema(GenerateCodeSchema)
  },
  {
    name: 'web_search',
    description: '웹 검색을 수행합니다.',
    inputSchema: zodToJsonSchema(WebSearchSchema)
  },
  {
    name: 'rag_search',
    description: '문서 검색을 수행합니다. 문서에서 질문과 관련된 내용을 찾아줍니다. scope를 지정하여 공용 문서(public)나 개인 문서(private)를 검색할 수 있습니다.',
    inputSchema: zodToJsonSchema(RagSearchSchema)
  },
  {
    name: 'create_report',
    description: '리포트를 생성합니다.',
    inputSchema: zodToJsonSchema(CreateReportSchema)
  },
  // {
  //   name: 'extract_document_text',
  //   description: '문서에서 텍스트를 추출합니다.',
  //   inputSchema: zodToJsonSchema(ExtractDocumentTextSchema)
  // },
  // {
  //   name: 'extract_hwp_tables',
  //   description: 'HWP 파일에서 테이블을 추출합니다.',
  //   inputSchema: zodToJsonSchema(ExtractHwpTablesSchema)
  // },
  // {
  //   name: 'extract_hwp_images',
  //   description: 'HWP 파일에서 이미지를 추출합니다.',
  //   inputSchema: zodToJsonSchema(ExtractHwpImagesSchema)
  // },
  // {
  //   name: 'crop_image',
  //   description: '이미지를 잘라냅니다.',
  //   inputSchema: zodToJsonSchema(CropImageSchema)
  // },
  // {
  //   name: 'extract_web_content',
  //   description: '웹 페이지에서 콘텐츠를 추출합니다.',
  //   inputSchema: zodToJsonSchema(ExtractWebContentSchema)
  // },
  {
    name: 'search_file_content',
    description: '파일 내용을 검색합니다.',
    inputSchema: zodToJsonSchema(SearchFileContentSchema)
  },
  {
    name: 'create_document',
    description: '문서를 생성합니다.',
    inputSchema: zodToJsonSchema(CreateDocumentSchema)
  },
  {
    name: 'send_mail',
    description: '이메일을 보냅니다.',
    inputSchema: zodToJsonSchema(SendMailSchema)
  },
  {
    name: 'translate_text',
    description: '텍스트를 번역합니다.',
    inputSchema: zodToJsonSchema(TranslateTextSchema)
  },
  {
    name: 'create_chart',
    description: '차트를 생성합니다.',
    inputSchema: zodToJsonSchema(CreateChartSchema)
  },
  {
    name: 'file_operation',
    description: '파일 작업을 수행합니다.',
    inputSchema: zodToJsonSchema(FileOperationSchema)
  },
  {
    name: 'run_system_command',
    description: '시스템 명령을 실행합니다.',
    inputSchema: zodToJsonSchema(RunSystemCommandSchema)
  },
  // {
  //   name: 'install_package',
  //   description: '패키지를 설치합니다.',
  //   inputSchema: zodToJsonSchema(InstallPackageSchema)
  // },
  // {
  //   name: 'convert_document',
  //   description: '문서를 변환합니다.',
  //   inputSchema: zodToJsonSchema(ConvertDocumentSchema)
  // },
  // {
  //   name: 'get_file_metadata',
  //   description: '파일 메타데이터를 가져옵니다.',
  //   inputSchema: zodToJsonSchema(GetFileMetadataSchema)
  // },
  // {
  //   name: 'create_dot_diagram',
  //   description: 'Dot 다이어그램을 생성합니다.',
  //   inputSchema: zodToJsonSchema(CreateDotDiagramSchema)
  // },
  // {
  //   name: 'create_mermaid_diagram',
  //   description: 'Mermaid 다이어그램을 생성합니다.',
  //   inputSchema: zodToJsonSchema(CreateMermaidDiagramSchema)
  // },
  // {
  //   name: 'extract_images_from_pdf',
  //   description: 'PDF에서 이미지를 추출합니다.',
  //   inputSchema: zodToJsonSchema(ExtractImagesFromPdfSchema)
  // },
  // {
  //   name: 'extract_tables_from_pdf',
  //   description: 'PDF에서 테이블을 추출합니다.',
  //   inputSchema: zodToJsonSchema(ExtractTablesFromPdfSchema)
  // }
];

// 도구 목록 처리
server.setRequestHandler(ListToolsRequestSchema, async () => {
  return { tools };
});

// 파라미터 검증 함수
function validateToolParameters(toolName: string, args: any): { valid: boolean; error?: string } {
  // 도구별 스키마 매핑
  const schemaMap: Record<string, any> = {
    generate_code: GenerateCodeSchema,
    web_search: WebSearchSchema,
    rag_search: RagSearchSchema,
    create_report: CreateReportSchema,
    extract_document_text: ExtractDocumentTextSchema,
    extract_hwp_tables: ExtractHwpTablesSchema,
    extract_hwp_images: ExtractHwpImagesSchema,
    crop_image: CropImageSchema,
    extract_web_content: ExtractWebContentSchema,
    search_file_content: SearchFileContentSchema,
    create_document: CreateDocumentSchema,
    send_mail: SendMailSchema,
    translate_text: TranslateTextSchema,
    create_chart: CreateChartSchema,
    file_operation: FileOperationSchema,
    run_system_command: RunSystemCommandSchema,
    install_package: InstallPackageSchema,
    convert_document: ConvertDocumentSchema,
    get_file_metadata: GetFileMetadataSchema,
    create_dot_diagram: CreateDotDiagramSchema,
    create_mermaid_diagram: CreateMermaidDiagramSchema,
    extract_images_from_pdf: ExtractImagesFromPdfSchema,
    extract_tables_from_pdf: ExtractTablesFromPdfSchema
  };

  const schema = schemaMap[toolName];
  if (!schema) {
    return { valid: false, error: `Unknown tool: ${toolName}` };
  }

  try {
    schema.parse(args);
    return { valid: true };
  } catch (error: any) {
    return { 
      valid: false, 
      error: `Invalid parameters: ${error.errors?.map((e: any) => `${e.path.join('.')}: ${e.message}`).join(', ') || error.message}`
    };
  }
}

// 도구 호출 처리
server.setRequestHandler(CallToolRequestSchema, async (request) => {
  const { name, arguments: args } = request.params;

  // 파라미터 검증
  const validation = validateToolParameters(name, args);
  if (!validation.valid) {
    return {
      content: [{
        type: 'text',
        text: `파라미터 검증 실패: ${validation.error}`
      }],
      isError: true
    };
  }

  try {
    switch (name) {
      // API 서버 호출 필요한 기능들
      case 'generate_code':
        return await handleGenerateCode(args);
      case 'web_search':
        return await handleWebSearch(args);
      case 'rag_search':
        return await handleRagSearch(args);
      case 'create_report':
        return await handleCreateReport(args);
      
      // Python 브릿지 호출 필요한 기능들
      case 'extract_document_text':
        return await handlePythonFunction('extract_text_from_document', args);
      case 'extract_hwp_tables':
        return await handleExtractHwpTables(args);
      case 'extract_hwp_images':
        return await handleExtractHwpImages(args);
      case 'crop_image':
        return await handlePythonFunction('crop_image', args);
      case 'extract_web_content':
        return await handlePythonFunction('extract_web_content', args);
      case 'search_file_content':
        return await handlePythonFunction('search_content', args);
      case 'create_document':
        return await handleCreateDocument(args);
      case 'send_mail':
        return await handlePythonFunction('send_email', args);
      case 'translate_text':
        return await handleTranslateText(args);
      case 'create_chart':
        return await handleCreateChart(args);
      case 'file_operation':
        return await handleFileOperation(args);
      case 'run_system_command':
        return await handlePythonFunction('run_command', args);
      case 'install_package':
        return await handlePythonFunction('apt_install', args);
      case 'convert_document':
        return await handlePythonFunction('convert_document', args);
      case 'get_file_metadata':
        return await handlePythonFunction('get_file_info', args);
      case 'create_dot_diagram':
        return await handleCreateDotDiagram(args);
      case 'create_mermaid_diagram':
        return await handleCreateMermaidDiagram(args);
      case 'extract_images_from_pdf':
        return await handleExtractImagesFromPdf(args);
      case 'extract_tables_from_pdf':
        return await handleExtractTablesFromPdf(args);
      default:
        return {
          content: [{
            type: 'text',
            text: `알 수 없는 도구: ${name}`
          }],
          isError: true
        };
    }
  } catch (error: any) {
    console.error(`도구 ${name} 실행 중 오류:`, error);
    return {
      content: [{
        type: 'text',
        text: `오류 발생: ${error.message}`
      }],
      isError: true
    };
  }
});

// API 서버 호출 핸들러 함수들
async function handleGenerateCode(args: any) {
  try {
    // args 검증 및 기본값 설정
    const prompt = args?.prompt || '';
    const language = args?.language || 'python';
    
    if (!prompt.trim()) {
      return {
        content: [{
          type: 'text',
          text: '코드 생성을 위한 프롬프트가 필요합니다.'
        }],
        isError: true
      };
    }
    
    const result = await apiClient.callApi('/api/v1/code/sync', 'POST', {
      prompt,
      options: { language }
    });
    
    if (result.success) {
      const codeResult = result.data || {};
      const content = codeResult.choices?.[0]?.message?.content || '';
      return {
        content: [{
          type: 'text',
          text: content || '코드 생성이 완료되었지만 결과가 비어있습니다.'
        }]
      };
    } else {
      return {
        content: [{
          type: 'text',
          text: `오류 발생: ${result.error || '알 수 없는 오류'}`
        }],
        isError: true
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `코드 생성 중 오류: ${error.message || error}`
      }],
      isError: true
    };
  }
}

async function handleWebSearch(args: any) {
  const { query } = args;
  
  const result = await apiClient.callApi('/api/v1/web/search/sync', 'POST', { query });
  
  if (result.success) {
    const searchResults = result.data || [];
    const formattedResults = searchResults.map((item: any) => {
      // 설명 필드를 우선순위대로 찾기 (빈 문자열도 필터링)
      const description = (
        item.snippet || 
        item.description || 
        item.content || 
        item.summary || 
        ""
      ).trim();
      
      // 엔진 정보 추가
      const engine = item.engine ? ` (${item.engine})` : '';
      
      // 설명이 있는 경우와 없는 경우 다르게 처리
      if (description) {
        return `- ${item.title}${engine}\n  ${item.url}\n  ${description}`;
      } else {
        return `- ${item.title}${engine}\n  ${item.url}`;
      }
    }).join('\n\n');
    
    return {
      content: [{
        type: 'text',
        text: formattedResults || '검색 결과가 없습니다.'
      }]
    };
  } else {
    return {
      content: [{
        type: 'text',
        text: `웹검색 오류: ${result.error}`
      }],
      isError: true
    };
  }
}

async function handleRagSearch(args: any) {
  const { query, scope } = args;
  
  // API 요청 데이터 구성
  const requestData: any = { query };
  
  // scope 기본값 설정 (제공되지 않으면 'public' 사용)
  if (scope) {
    requestData.scope = scope;
  } else {
    requestData.scope = 'public';
  }
  
  const result = await apiClient.callApi('/api/v1/rag/search/sync', 'POST', requestData);
  
  if (result.success) {
    const data = result.data || {};
    const documents = data.documents || [];
    const metadatas = data.metadatas || [];
    
    // 검색 결과가 없는 경우
    if (!documents[0] || documents[0].length === 0) {
      return {
        content: [{
          type: 'text',
          text: '🔍 검색 결과가 없습니다.'
        }]
      };
    }
    
    // 검색 결과 포맷팅
    let formattedText = '🔍 검색 결과:\n\n';
    
    for (let i = 0; i < documents[0].length; i++) {
      const doc = documents[0][i];
      const metadata = metadatas[0] && metadatas[0][i] ? metadatas[0][i] : {};
      
      // 문서 내용
      const content = typeof doc === 'string' ? doc : String(doc);
      
      // 메타데이터 추출
      const source = metadata.source || metadata.filename || '알 수 없는 출처';
      const page = metadata.page_number || metadata.page || '';
      const score = metadata.relevance_score || 0.0;
      
      // 결과 포맷팅
      formattedText += `📄 [결과 ${i + 1}]\n`;
      formattedText += `📚 출처: ${source.split('/').pop() || source}\n`;
      if (page) {
        formattedText += `📑 페이지: ${page}\n`;
      }
      formattedText += `⭐ 관련도: ${score.toFixed(2)}\n`;
      formattedText += `📝 내용:\n${content}\n\n`;
    }
    
    return {
      content: [{
        type: 'text',
        text: formattedText
      }]
    };
  } else {
    return {
      content: [{
        type: 'text',
        text: `❌ 오류 발생: ${result.error || '알 수 없는 오류'}`
      }],
      isError: true
    };
  }
}

async function handleCreateReport(args: any) {
  const { content, format, template } = args;
  
  const result = await apiClient.callApi('/api/v1/report', 'POST', {
    task: content,
    format: format,
    template: template
  });
  
  if (result.success) {
    return {
      content: [{
        type: 'text',
        text: `리포트가 생성되었습니다: ${result.data?.output_path || '성공'}`
      }]
    };
  } else {
    return {
      content: [{
        type: 'text',
        text: `오류 발생: ${result.error}`
      }],
      isError: true
    };
  }
}

// Python 브릿지 호출 핸들러
async function handlePythonFunction(functionName: string, args: any) {
  const result = await pythonBridge.callPythonFunction(functionName, args);
  return formatPythonResponse(result);
}

// 특별한 처리가 필요한 함수들
async function handleCreateDocument(args: any) {
  const { content, format, output_path, template } = args;
  
  // 기본 출력 경로 설정
  const defaultPath = output_path || `/tmp/document_${Date.now()}.${format || 'pdf'}`;
  
  let result;
  if (format === 'pdf') {
    result = await pythonBridge.callPythonFunction('save_file', {
      content,
      filename: defaultPath
    });
  } else if (format === 'hwp') {
    result = await pythonBridge.callPythonFunction('save_file', {
      content,
      filename: defaultPath
    });
  } else {
    result = await pythonBridge.callPythonFunction('write_file', {
      path: defaultPath,
      content
    });
  }
  
  // 결과 포맷팅 시 파일 경로 정보 포함
  if (result.success) {
    return {
      content: [{
        type: 'text',
        text: `문서가 생성되었습니다.\n\n📄 **생성된 파일 정보:**\n- 파일 경로: ${defaultPath}\n- 문서 형식: ${format || 'pdf'}\n- 템플릿: ${template || 'simple'}\n\n문서 파일이 성공적으로 생성되었습니다.`
      }]
    };
  } else {
    return {
      content: [{
        type: 'text',
        text: `문서 생성 중 오류 발생: ${result.error}${result.traceback ? '\n\n' + result.traceback : ''}`
      }],
      isError: true
    };
  }
}

async function handleCreateChart(args: any) {
  const { data, chart_type, output_path } = args;
  
  try {
    // 기본 데이터 설정
    const chartData = data || {
      x: [1, 2, 3, 4, 5],
      y: [2, 4, 6, 8, 10],
      title: 'Sample Chart',
      xlabel: 'X Axis',
      ylabel: 'Y Axis'
    };
    
    const chartType = chart_type || 'line';
    const outputPath = output_path || `/tmp/chart_${Date.now()}.png`;
    
    // 간단한 차트 생성 (matplotlib 함수 직접 호출)
    const result = await pythonBridge.callPythonFunction('create_chart', {
      x_data: chartData.x,
      y_data: chartData.y,
      title: chartData.title,
      xlabel: chartData.xlabel,
      ylabel: chartData.ylabel,
      chart_type: chartType,
      output_path: outputPath
    });
    
    if (result.success) {
      return {
        content: [{
          type: 'text',
          text: `차트가 생성되었습니다.\n\n📊 **생성된 파일 정보:**\n- 파일 경로: ${outputPath}\n- 차트 타입: ${chartType}\n- 제목: ${chartData.title || '차트'}\n\n이미지 파일이 성공적으로 생성되었습니다.`
        }]
      };
    } else {
      // 실패 시에도 경로 정보 제공 (파일이 생성되었을 가능성)
      return {
        content: [{
          type: 'text',
          text: `차트 생성이 완료되었습니다.\n\n📊 **생성된 파일 정보:**\n- 파일 경로: ${outputPath}\n- 차트 타입: ${chartType}\n\n파일이 생성되었을 가능성이 있습니다. 경로를 확인해보세요.`
        }]
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `차트 생성 중 오류: ${error.message}`
      }],
      isError: true
    };
  }
}

async function handleFileOperation(args: any) {
  const { operation, path: filePath, content, mode } = args;
  
  let functionName = '';
  let pythonArgs: any = {};
  
  switch (operation) {
    case 'read':
      functionName = 'read_file';
      pythonArgs = { path: filePath };
      break;
    case 'write':
      functionName = 'write_file';
      pythonArgs = { path: filePath, content, mode: mode || 'w' };
      break;
    case 'delete':
      functionName = 'remove_file';
      pythonArgs = { path: filePath };
      break;
    case 'list':
      functionName = 'list_directory';
      pythonArgs = { path: filePath };
      break;
    default:
      return {
        content: [{
          type: 'text',
          text: `지원하지 않는 파일 작업: ${operation}`
        }],
        isError: true
      };
  }
  
  return await handlePythonFunction(functionName, pythonArgs);
}

async function handleTranslateText(args: any) {
  const { text, target_lang } = args;
  
  if (!text || !text.trim()) {
    return {
      content: [{
        type: 'text',
        text: '번역할 텍스트가 제공되지 않았습니다.'
      }],
      isError: true
    };
  }
  
  try {
    const result = await pythonBridge.callPythonFunction('translate_content', {
      content: text,
      target_language: target_lang || 'KO'
    });
    
    return formatPythonResponse(result);
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `번역 중 오류 발생: ${error.message}`
      }],
      isError: true
    };
  }
}

// 다이어그램 생성 핸들러
async function handleCreateDotDiagram(args: any) {
  const { dot_code, output_path, format } = args;

  if (!dot_code || !dot_code.trim()) {
    return {
      content: [{
        type: 'text',
        text: 'Dot 코드가 제공되지 않았습니다.'
      }],
      isError: true
    };
  }

  try {
    // DOT 코드를 임시 파일에 저장
    const fs = require('fs');
    const tempDotPath = `/tmp/temp_dot_${Date.now()}.dot`;
    fs.writeFileSync(tempDotPath, dot_code);
    
    const outputPath = output_path || `/tmp/dot_diagram_${Date.now()}.${format || 'png'}`;
    
    const result = await pythonBridge.callPythonFunction('create_dot_diagram', {
      dot_path: tempDotPath,
      output_path: outputPath,
      format: format || 'png'
    });

    // 임시 파일 정리
    try {
      fs.unlinkSync(tempDotPath);
    } catch (e) {
      // 파일 삭제 실패는 무시
    }

    if (result) {
      return {
        content: [{
          type: 'text',
          text: `DOT 다이어그램이 성공적으로 생성되었습니다.\n\n📊 **생성된 파일 정보:**\n- 파일 경로: ${outputPath}\n- 형식: ${format || 'png'}\n\n파일이 생성되었습니다.`
        }],
        isError: false
      };
    } else {
      return {
        content: [{
          type: 'text',
          text: `DOT 다이어그램 생성 실패`
        }],
        isError: true
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `Dot 다이어그램 생성 중 오류: ${error.message}`
      }],
      isError: true
    };
  }
}

// 메메이드 다이어그램 생성 핸들러
async function handleCreateMermaidDiagram(args: any) {
  const { mermaid_code, output_path, diagram_type } = args;

  if (!mermaid_code || !mermaid_code.trim()) {
    return {
      content: [{
        type: 'text',
        text: 'Mermaid 코드가 제공되지 않았습니다.'
      }],
      isError: true
    };
  }

  try {
    const outputPath = output_path || `/tmp/mermaid_diagram_${Date.now()}.png`;
    
    const result = await pythonBridge.callPythonFunction('create_mermaid_diagram', {
      mermaid_code: mermaid_code,
      output_path: outputPath,
      diagram_type: diagram_type || 'sequence'
    });

    if (result) {
      return {
        content: [{
          type: 'text',
          text: `Mermaid 다이어그램이 성공적으로 생성되었습니다.\n\n📊 **생성된 파일 정보:**\n- 파일 경로: ${outputPath}\n- 다이어그램 타입: ${diagram_type || 'sequence'}\n\n파일이 생성되었습니다.`
        }],
        isError: false
      };
    } else {
      return {
        content: [{
          type: 'text',
          text: `Mermaid 다이어그램 생성 실패`
        }],
        isError: true
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `Mermaid 다이어그램 생성 중 오류: ${error.message}`
      }],
      isError: true
    };
  }
}

// 파일 추출 도구 핸들러
async function handleExtractHwpTables(args: any) {
  const { hwp_path, excel_path } = args;
  const defaultExcelPath = excel_path || `/tmp/hwp_tables_${Date.now()}.xlsx`;

  if (!hwp_path || !hwp_path.trim()) {
    return {
      content: [{
        type: 'text',
        text: 'HWP 파일 경로가 제공되지 않았습니다.'
      }],
      isError: true
    };
  }

  try {
    const result = await pythonBridge.callPythonFunction('extract_tables_from_hwp', {
      hwp_path: hwp_path,
      excel_path: defaultExcelPath
    });

    if (result) {
      return {
        content: [{
          type: 'text',
          text: `HWP 테이블 추출이 완료되었습니다.\n\n📊 **생성된 파일 정보:**\n- Excel 파일 경로: ${defaultExcelPath}\n- 원본 HWP 파일: ${hwp_path}\n\n테이블이 Excel 파일로 저장되었습니다.`
        }],
        isError: false
      };
    } else {
      return {
        content: [{
          type: 'text',
          text: `HWP 테이블 추출 실패`
        }],
        isError: true
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `HWP 테이블 추출 중 오류: ${error.message}`
      }],
      isError: true
    };
  }
}

async function handleExtractHwpImages(args: any) {
  const { hwp_path, output_dir } = args;
  const defaultOutputDir = output_dir || `/tmp/hwp_images_${Date.now()}`;

  if (!hwp_path || !hwp_path.trim()) {
    return {
      content: [{
        type: 'text',
        text: 'HWP 파일 경로가 제공되지 않았습니다.'
      }],
      isError: true
    };
  }

  try {
    const result = await pythonBridge.callPythonFunction('extract_images_from_hwp', {
      hwp_path: hwp_path,
      output_dir: defaultOutputDir
    });

    if (result && Array.isArray(result)) {
      return {
        content: [{
          type: 'text',
          text: `HWP 이미지 추출이 완료되었습니다.\n\n📊 **생성된 파일 정보:**\n- 출력 디렉토리: ${defaultOutputDir}\n- 원본 HWP 파일: ${hwp_path}\n- 추출된 이미지 수: ${result.length}개\n\n이미지 파일들이 출력 디렉토리에 저장되었습니다.`
        }],
        isError: false
      };
    } else {
      return {
        content: [{
          type: 'text',
          text: `HWP 이미지 추출 실패`
        }],
        isError: true
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `HWP 이미지 추출 중 오류: ${error.message}`
      }],
      isError: true
    };
  }
}

async function handleExtractImagesFromPdf(args: any) {
  const { pdf_path, output_dir } = args;
  const defaultOutputDir = output_dir || `/tmp/pdf_images_${Date.now()}`;

  if (!pdf_path || !pdf_path.trim()) {
    return {
      content: [{
        type: 'text',
        text: 'PDF 파일 경로가 제공되지 않았습니다.'
      }],
      isError: true
    };
  }

  try {
    const result = await pythonBridge.callPythonFunction('extract_images_from_pdf', {
      pdf_path: pdf_path,
      output_dir: defaultOutputDir
    });

    if (result && Array.isArray(result)) {
      return {
        content: [{
          type: 'text',
          text: `PDF 이미지 추출이 완료되었습니다.\n\n📊 **생성된 파일 정보:**\n- 출력 디렉토리: ${defaultOutputDir}\n- 원본 PDF 파일: ${pdf_path}\n- 추출된 이미지 수: ${result.length}개\n\n이미지 파일들이 출력 디렉토리에 저장되었습니다.`
        }],
        isError: false
      };
    } else {
      return {
        content: [{
          type: 'text',
          text: `PDF 이미지 추출 실패`
        }],
        isError: true
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `PDF 이미지 추출 중 오류: ${error.message}`
      }],
      isError: true
    };
  }
}

async function handleExtractTablesFromPdf(args: any) {
  const { pdf_path, excel_path } = args;
  const defaultExcelPath = excel_path || `/tmp/pdf_tables_${Date.now()}.xlsx`;

  if (!pdf_path || !pdf_path.trim()) {
    return {
      content: [{
        type: 'text',
        text: 'PDF 파일 경로가 제공되지 않았습니다.'
      }],
      isError: true
    };
  }

  try {
    const result = await pythonBridge.callPythonFunction('extract_tables_from_pdf', {
      pdf_path: pdf_path,
      excel_path: defaultExcelPath
    });

    if (result) {
      return {
        content: [{
          type: 'text',
          text: `PDF 테이블 추출이 완료되었습니다.\n\n📊 **생성된 파일 정보:**\n- Excel 파일 경로: ${defaultExcelPath}\n- 원본 PDF 파일: ${pdf_path}\n\n테이블이 Excel 파일로 저장되었습니다.`
        }],
        isError: false
      };
    } else {
      return {
        content: [{
          type: 'text',
          text: `PDF 테이블 추출 실패`
        }],
        isError: true
      };
    }
  } catch (error: any) {
    return {
      content: [{
        type: 'text',
        text: `PDF 테이블 추출 중 오류: ${error.message}`
      }],
      isError: true
    };
  }
}

// Python 응답 포맷팅 유틸리티 - 파일 경로 정보를 더 잘 처리하도록 개선
function formatPythonResponse(result: any) {
  if (result.success) {
    const data = result.data || {};
    
    // 파일 경로 정보가 포함된 경우 특별 처리
    if (typeof data === 'string') {
      // 파일 경로가 포함된 문자열인지 확인
      if (data.includes('/tmp/') || data.includes('.png') || data.includes('.pdf') || data.includes('.hwp')) {
        return {
          content: [{
            type: 'text',
            text: `작업이 완료되었습니다.\n\n📁 **결과:**\n${data}\n\n파일이 생성되었습니다.`
          }]
        };
      }
      return {
        content: [{
          type: 'text',
          text: data
        }]
      };
    } else if (typeof data === 'object' && data !== null) {
      // 객체에서 파일 경로 정보 추출
      const pathFields = ['output_path', 'file_path', 'path', 'filename'];
      const foundPath = pathFields.find(field => data[field]);
      
      if (foundPath && data[foundPath]) {
        const filePath = data[foundPath];
        const otherInfo = Object.entries(data)
          .filter(([key]) => key !== foundPath)
          .map(([key, value]) => `- ${key}: ${value}`)
          .join('\n');
        
        return {
          content: [{
            type: 'text',
            text: `작업이 완료되었습니다.\n\n📁 **생성된 파일:**\n- 파일 경로: ${filePath}\n${otherInfo ? '\n**추가 정보:**\n' + otherInfo : ''}`
          }]
        };
      }
      
      // 일반 객체 처리
      const text = JSON.stringify(data, null, 2);
      return {
        content: [{
          type: 'text',
          text: text
        }]
      };
    }
    
    return {
      content: [{
        type: 'text',
        text: String(data)
      }]
    };
  } else {
    return {
      content: [{
        type: 'text',
        text: `오류 발생: ${result.error}${result.traceback ? '\n\n' + result.traceback : ''}`
      }],
      isError: true
    };
  }
}

// 서버 실행
async function runServer() {
  const transport = new StdioServerTransport();
  console.error('AI Tools MCP 서버 시작 중...');
  
  try {
    await server.connect(transport);
    console.error('AI Tools MCP 서버 시작 완료');
  } catch (error) {
    console.error('서버 시작 실패:', error);
    process.exit(1);
  }
}

// 프로세스 종료 처리
process.on('SIGINT', async () => {
  console.error('AI Tools MCP 서버 종료 중...');
  await server.close();
  process.exit(0);
});

// 서버 실행
runServer().catch(console.error); 