#!/usr/bin/env node

/**
 * AI.RUN MCP Tools 전체 테스트 스위트
 * 모든 도구들의 정상 동작을 테스트합니다.
 */

import { spawn } from 'child_process';
import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

class MCPToolsTester {
  constructor() {
    this.mcpProcess = null;
    this.messageId = 1;
    this.testResults = [];
    this.testOutputDir = path.join(__dirname, 'test-output');
    
    // 테스트 출력 디렉토리 생성
    if (!fs.existsSync(this.testOutputDir)) {
      fs.mkdirSync(this.testOutputDir, { recursive: true });
    }
  }

  async startMCPServer() {
    console.log('🚀 MCP 서버 시작 중...');
    
    const serverPath = path.join(__dirname, '../dist/index.js');
    if (!fs.existsSync(serverPath)) {
      throw new Error('MCP 서버 빌드 파일이 없습니다. npm run build를 실행하세요.');
    }

    this.mcpProcess = spawn('node', [serverPath], {
      stdio: ['pipe', 'pipe', 'pipe'],
      cwd: path.dirname(serverPath)
    });

    this.mcpProcess.stderr.on('data', (data) => {
      console.log('MCP 서버 로그:', data.toString());
    });

    // 서버 초기화 대기
    await new Promise(resolve => setTimeout(resolve, 2000));

    // 서버 초기화
    await this.sendRequest('initialize', {
      protocolVersion: '2024-11-05',
      capabilities: {
        roots: { listChanged: true },
        sampling: {}
      },
      clientInfo: {
        name: 'airun-mcp-tester',
        version: '1.0.0'
      }
    });

    console.log('✅ MCP 서버가 시작되었습니다.');
  }

  async sendRequest(method, params = {}) {
    if (!this.mcpProcess) {
      throw new Error('MCP 서버가 실행되지 않았습니다.');
    }

    const request = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method,
      params
    };

    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error(`요청 시간 초과: ${method}`));
      }, 30000);

      let buffer = '';
      const onData = (data) => {
        buffer += data.toString();
        
        while (true) {
          const newlineIndex = buffer.indexOf('\n');
          if (newlineIndex === -1) break;
          
          const line = buffer.slice(0, newlineIndex);
          buffer = buffer.slice(newlineIndex + 1);
          
          if (line.trim()) {
            try {
              const response = JSON.parse(line);
              if (response.id === request.id) {
                clearTimeout(timeout);
                this.mcpProcess.stdout.off('data', onData);
                
                if (response.error) {
                  reject(new Error(`MCP 오류: ${response.error.message}`));
                } else {
                  resolve(response.result);
                }
                return;
              }
            } catch (e) {
              // JSON 파싱 오류 무시
            }
          }
        }
      };

      this.mcpProcess.stdout.on('data', onData);
      this.mcpProcess.stdin.write(JSON.stringify(request) + '\n');
    });
  }

  async listTools() {
    const response = await this.sendRequest('tools/list');
    return response.tools || [];
  }

  async callTool(name, args = {}) {
    const response = await this.sendRequest('tools/call', {
      name,
      arguments: args
    });
    return response;
  }

  async runTest(testName, testFunction) {
    console.log(`\n🧪 테스트 시작: ${testName}`);
    const startTime = Date.now();
    
    try {
      const result = await testFunction();
      const duration = Date.now() - startTime;
      
      console.log(`✅ 테스트 성공: ${testName} (${duration}ms)`);
      this.testResults.push({
        name: testName,
        status: 'success',
        duration,
        result
      });
      
      return result;
    } catch (error) {
      const duration = Date.now() - startTime;
      
      console.log(`❌ 테스트 실패: ${testName} (${duration}ms)`);
      console.log(`   오류: ${error.message}`);
      
      this.testResults.push({
        name: testName,
        status: 'failed',
        duration,
        error: error.message
      });
      
      throw error;
    }
  }

  // 개별 도구 테스트 케이스들
  async testGenerateCode() {
    return await this.runTest('generate_code', async () => {
      const result = await this.callTool('generate_code', {
        prompt: 'Hello World를 출력하는 Python 코드를 작성해주세요',
        language: 'python'
      });
      
      if (result.isError) {
        throw new Error(`코드 생성 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testWebSearch() {
    return await this.runTest('web_search', async () => {
      const result = await this.callTool('web_search', {
        query: 'AI.RUN 인공지능 플랫폼'
      });
      
      if (result.isError) {
        throw new Error(`웹 검색 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testRagSearch() {
    return await this.runTest('rag_search', async () => {
      const result = await this.callTool('rag_search', {
        query: '인공지능 AI 머신러닝'
      });
      
      if (result.isError) {
        throw new Error(`RAG 검색 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testCreateReport() {
    return await this.runTest('create_report', async () => {
      const result = await this.callTool('create_report', {
        content: '테스트 리포트입니다. 이것은 MCP 도구 테스트의 일부입니다.',
        format: 'pdf',
        template: 'simple'
      });
      
      if (result.isError) {
        throw new Error(`리포트 생성 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testCreateChart() {
    return await this.runTest('create_chart', async () => {
      const result = await this.callTool('create_chart', {
        data: {
          x: [1, 2, 3, 4, 5],
          y: [2, 4, 6, 8, 10],
          title: '테스트 차트',
          xlabel: 'X 축',
          ylabel: 'Y 축'
        },
        chart_type: 'line',
        output_path: path.join(this.testOutputDir, 'test-chart.png')
      });
      
      if (result.isError) {
        throw new Error(`차트 생성 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testFileOperation() {
    return await this.runTest('file_operation', async () => {
      const testFile = path.join(this.testOutputDir, 'test-file.txt');
      const testContent = 'MCP 도구 테스트 파일입니다.\n현재 시간: ' + new Date().toISOString();
      
      // 파일 쓰기 테스트
      const writeResult = await this.callTool('file_operation', {
        operation: 'write',
        path: testFile,
        content: testContent
      });
      
      if (writeResult.isError) {
        throw new Error(`파일 쓰기 실패: ${writeResult.content[0].text}`);
      }
      
      // 파일 읽기 테스트
      const readResult = await this.callTool('file_operation', {
        operation: 'read',
        path: testFile
      });
      
      if (readResult.isError) {
        throw new Error(`파일 읽기 실패: ${readResult.content[0].text}`);
      }
      
      return { write: writeResult, read: readResult };
    });
  }

  async testTranslateText() {
    return await this.runTest('translate_text', async () => {
      const result = await this.callTool('translate_text', {
        text: 'Hello, this is a test message for translation.',
        target_lang: 'KO'
      });
      
      if (result.isError) {
        throw new Error(`번역 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testGetFileMetadata() {
    return await this.runTest('get_file_metadata', async () => {
      const testFile = path.join(__dirname, '../src/index.ts');
      const result = await this.callTool('get_file_metadata', {
        path: testFile
      });
      
      if (result.isError) {
        throw new Error(`파일 메타데이터 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testRunSystemCommand() {
    return await this.runTest('run_system_command', async () => {
      const result = await this.callTool('run_system_command', {
        command: 'echo "MCP 도구 테스트 명령어"'
      });
      
      if (result.isError) {
        throw new Error(`시스템 명령 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testCreateDocument() {
    return await this.runTest('create_document', async () => {
      const result = await this.callTool('create_document', {
        content: 'MCP 도구 테스트 문서\n\n이것은 테스트를 위한 문서입니다.',
        format: 'txt',
        output_path: path.join(this.testOutputDir, 'test-document.txt')
      });
      
      if (result.isError) {
        throw new Error(`문서 생성 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testConvertToChartData() {
    return await this.runTest('convert_to_chart_data', async () => {
      const result = await this.callTool('convert_to_chart_data', {
        data_text: 'Month,Sales\nJan,100\nFeb,150\nMar,200\nApr,180\nMay,220',
        x_column: 0,
        y_column: 1,
        title: '월별 판매량',
        xlabel: '월',
        ylabel: '판매량',
        skip_header: true,
        delimiter: ','
      });
      
      if (result.isError) {
        throw new Error(`차트 데이터 변환 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  // 무거운 테스트들 (선택적 실행)
  async testExtractWebContent() {
    return await this.runTest('extract_web_content', async () => {
      const result = await this.callTool('extract_web_content', {
        url: 'https://httpbin.org/html',
        extract_type: 'text',
        max_items: 5
      });
      
      if (result.isError) {
        throw new Error(`웹 콘텐츠 추출 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async testSearchFileContent() {
    return await this.runTest('search_file_content', async () => {
      const result = await this.callTool('search_file_content', {
        path: path.join(__dirname, '../src'),
        query: 'generate_code',
        file_types: ['ts', 'js']
      });
      
      if (result.isError) {
        throw new Error(`파일 내용 검색 실패: ${result.content[0].text}`);
      }
      
      return result;
    });
  }

  async runAllTests() {
    console.log('🔥 AI.RUN MCP 도구 전체 테스트 시작');
    console.log('=' * 50);

    try {
      await this.startMCPServer();
      
      // 도구 목록 확인
      const tools = await this.listTools();
      console.log(`📋 총 ${tools.length}개의 도구가 등록되었습니다:`);
      tools.forEach(tool => console.log(`  - ${tool.name}: ${tool.description}`));
      
      // 기본 테스트들 (빠른 실행)
      const basicTests = [
        () => this.testGenerateCode(),
        () => this.testCreateChart(),
        () => this.testFileOperation(),
        () => this.testGetFileMetadata(),
        () => this.testRunSystemCommand(),
        () => this.testCreateDocument(),
        () => this.testSearchFileContent(),
      ];

      // 네트워크 테스트들 (느린 실행)
      const networkTests = [
        () => this.testWebSearch(),
        () => this.testRagSearch(),
        () => this.testCreateReport(),
        () => this.testTranslateText(),
        () => this.testExtractWebContent(),
      ];

      console.log('\n🏃‍♂️ 기본 테스트 실행 중...');
      for (const test of basicTests) {
        try {
          await test();
        } catch (error) {
          console.log(`   (계속 진행...)`);
        }
      }

      console.log('\n🌐 네트워크 테스트 실행 중...');
      for (const test of networkTests) {
        try {
          await test();
        } catch (error) {
          console.log(`   (계속 진행...)`);
        }
      }

      // 결과 요약
      this.printTestSummary();

    } catch (error) {
      console.error('❌ 테스트 실행 중 오류:', error);
    } finally {
      if (this.mcpProcess) {
        this.mcpProcess.kill();
        console.log('\n🔚 MCP 서버가 종료되었습니다.');
      }
    }
  }

  printTestSummary() {
    console.log('\n' + '=' * 50);
    console.log('📊 테스트 결과 요약');
    console.log('=' * 50);

    const totalTests = this.testResults.length;
    const successfulTests = this.testResults.filter(t => t.status === 'success').length;
    const failedTests = this.testResults.filter(t => t.status === 'failed').length;

    console.log(`총 테스트: ${totalTests}`);
    console.log(`성공: ${successfulTests} ✅`);
    console.log(`실패: ${failedTests} ❌`);
    console.log(`성공률: ${Math.round((successfulTests / totalTests) * 100)}%`);

    if (failedTests > 0) {
      console.log('\n❌ 실패한 테스트들:');
      this.testResults
        .filter(t => t.status === 'failed')
        .forEach(t => console.log(`  - ${t.name}: ${t.error}`));
    }

    console.log('\n✅ 성공한 테스트들:');
    this.testResults
      .filter(t => t.status === 'success')
      .forEach(t => console.log(`  - ${t.name} (${t.duration}ms)`));

    // 테스트 결과를 파일로 저장
    const reportPath = path.join(this.testOutputDir, 'test-results.json');
    fs.writeFileSync(reportPath, JSON.stringify(this.testResults, null, 2));
    console.log(`\n📄 상세 결과가 저장되었습니다: ${reportPath}`);
  }
}

// 메인 실행
async function main() {
  const tester = new MCPToolsTester();
  await tester.runAllTests();
}

// 직접 실행시
if (import.meta.url === `file://${process.argv[1]}`) {
  main().catch(console.error);
}

export default MCPToolsTester; 