/** @type {import('next').NextConfig} */
const nextConfig = {
  // ESLint 오류 무시 (프로덕션 빌드를 위해)
  eslint: {
    ignoreDuringBuilds: true,
  },
  
  // TypeScript 오류 무시 (프로덕션 빌드를 위해)
  typescript: {
    ignoreBuildErrors: true,
  },

  // 개발 환경 설정 (원격 서버 환경 최적화)
  webpack: (config, { dev, isServer }) => {
    if (dev && !isServer) {
      // 파일 변경 감지 설정 (폴링 방식)
      config.watchOptions = {
        poll: 1000,
        aggregateTimeout: 300,
      };
      
      // 개발 환경에서 안정성을 위한 설정
      config.optimization = {
        ...config.optimization,
        splitChunks: {
          ...config.optimization.splitChunks,
          cacheGroups: {
            ...config.optimization.splitChunks.cacheGroups,
            default: false,
            vendors: false,
          },
        },
      };
    }
    return config;
  },
  
  // React Strict Mode 활성화
  reactStrictMode: true,
  
  // Body size limit 설정 (PDF 미리보기를 위해 증가)
  experimental: {
    serverActions: {
      bodySizeLimit: '50mb',
    },
  },
  
  // 운영 환경에서 HMR 비활성화
  experimental: {
    // 실험적 기능 설정 (필요시 추가)
  },
  
  // 개발 환경에서만 허용할 외부 도메인/IP 설정
  ...(process.env.NODE_ENV === 'development' && {
    allowedDevOrigins: [
      '192.168.100.*',    // 현재 사용 중인 대역 (192.168.100.0/24)
      '192.168.1.*',      // 일반적인 홈 네트워크 (192.168.1.0/24)
      '192.168.0.*',      // 또 다른 홈 네트워크 (192.168.0.0/24)
      '10.0.0.*',         // 기업 네트워크 (10.0.0.0/24)
      '172.16.*.*',       // 사설 네트워크 B 클래스 (172.16.0.0/12)
      'localhost',        // 로컬호스트
      '127.0.0.1',        // 로컬 IP
      '*.hamonize.com',
      '*.ngrok-free.app', // ngrok 무료 도메인 허용
      '*.ngrok.io',       // ngrok 유료 도메인 허용
      'profound-balanced-mammoth.ngrok-free.app', // 현재 사용 중인 ngrok 도메인
    ],
  }),

  async rewrites() {
    // 환경변수에서 API 서버 URL 가져오기 (이미 /api/v1 포함된 경우 고려)
    const fullApiServerUrl = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';
    const baseApiServerUrl = fullApiServerUrl.replace(/\/api\/v1$/, '');
    
    // RAG 서버, 웹검색 서버, 보고서 서버 URL 설정 (통일된 환경변수 사용)
    const ragServerUrl = process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';
    const websearchServerUrl = process.env.NEXT_PUBLIC_WEB_API_SERVER || 'http://localhost:5610';
    const reportServerUrl = process.env.NEXT_PUBLIC_REPORT_API_SERVER || 'http://localhost:5620';
    
    return [
      // v1 API 프록시 (환경변수에 이미 /api/v1이 포함되어 있으므로 직접 사용)
      {
        source: '/api/v1/:path*',
        destination: `${fullApiServerUrl}/:path*`,
      },
      // 일반 API 프록시 (기본 서버 URL + 경로)
      {
        source: '/api/sessions/:path*',
        destination: `${baseApiServerUrl}/api/v1/sessions/:path*`,
      },
      {
        source: '/api/queues/:path*',
        destination: `${baseApiServerUrl}/api/v1/queues/:path*`,
      },
      {
        source: '/api/agent/:path*',
        destination: `${baseApiServerUrl}/api/v1/agent/:path*`,
      },
      {
        source: '/api/tasks/:path*',
        destination: `${baseApiServerUrl}/api/v1/tasks/:path*`,
      },
      {
        source: '/api/schedules/:path*',
        destination: `${baseApiServerUrl}/api/v1/schedules/:path*`,
      },
      // 보고서 서버 프록시 (SSE 스트리밍 포함)
      {
        source: '/report/:path*',
        destination: `${reportServerUrl}/:path*`,
      },
      // RAG 서버 캐시 삭제 프록시
      {
        source: '/rag/clear-cache',
        destination: `${ragServerUrl}/clear-cache`,
      },
      // 웹검색 서버 캐시 삭제 프록시
      {
        source: '/websearch/cache/clear',
        destination: `${websearchServerUrl}/cache/clear`,
      },
    ];
  },
};

module.exports = nextConfig; 