import { NextRequest, NextResponse } from 'next/server';
import pg from 'pg';
import { getDatabaseConfig } from '@/config/serverConfig';

const { Pool } = pg;

// 데이터베이스 연결
function getDbConnection() {
  const config = getDatabaseConfig();
  return new Pool({
    host: config.host,
    port: config.port,
    user: config.user,
    password: config.password,
    database: config.name,
    max: 5
  });
}

export async function POST(request: NextRequest) {
  try {
    const dbManager = getDbConnection();
    const body = await request.json();
    const { jobId, patientIds } = body;

    // 임시로 인증 건너뛰기 (기존 방식 유지)
    const userId = 'admin';

    // 작업 존재 확인
    const jobQuery = await dbManager.query(
      'SELECT * FROM health_batch_jobs WHERE id = $1 AND user_id = $2',
      [jobId, userId]
    );

    if (jobQuery.rows.length === 0) {
      return NextResponse.json({
        success: false,
        error: '작업을 찾을 수 없습니다.'
      }, { status: 404 });
    }

    const job = jobQuery.rows[0];

    // 재분석할 환자들 조회 (실패한 환자들만)
    let patientsQuery;
    if (patientIds && patientIds.length > 0) {
      // 특정 환자들만 재분석
      const placeholders = patientIds.map((_: any, index: number) => `$${index + 2}`).join(',');
      patientsQuery = await dbManager.query(
        `SELECT * FROM health_batch_patients
         WHERE job_id = $1 AND id IN (${placeholders}) AND status = 'failed'`,
        [jobId, ...patientIds]
      );
    } else {
      // 모든 실패한 환자들 재분석
      patientsQuery = await dbManager.query(
        'SELECT * FROM health_batch_patients WHERE job_id = $1 AND status = $2',
        [jobId, 'failed']
      );
    }

    const failedPatients = patientsQuery.rows;

    if (failedPatients.length === 0) {
      return NextResponse.json({
        success: false,
        error: '재분석할 실패한 환자가 없습니다.'
      }, { status: 400 });
    }

    // 건강검진 평가 기준
    const HEALTH_EVALUATION_CRITERIA = {
      bloodPressure: {
        systolic: 140,    // mmHg 이상 시 해당(1)
        diastolic: 90     // mmHg 이상 시 해당(1)
      },
      lipid: {
        triglycerides: 150,  // mg/dl 이상 시 해당(1)
        hdl: 40,            // mg/dl 미만 시 해당(1)
        ldl: 140            // mg/dl 이상 시 해당(1)
      },
      bloodSugar: {
        glucose: 100,       // mg/dl 이상 시 해당(1)
        hba1c: 5.7         // % 이상 시 해당(1)
      },
      obesity: {
        bmi: 25            // kg/m² 이상 시 해당(1)
      },
      urineProtein: {
        threshold: 1       // 1+ 이상 시 해당(1)
      }
    };

    // 환자 데이터 정규화 함수
    function normalizePatientData(rawData: any) {
      const normalized: any = {};

      // 숫자값 정규화 (문자열로 된 숫자를 숫자로 변환)
      const parseValue = (value: any): number | null => {
        if (value === null || value === undefined || value === '') return null;
        const parsed = parseFloat(String(value).replace(/[^\d.-]/g, ''));
        return isNaN(parsed) ? null : parsed;
      };

      // 일반적인 필드명 매핑
      const fieldMappings = {
        // 혈압
        systolic: ['수축기혈압', '최고혈압', 'SBP', 'systolic', '수축기', '최고', 'Systolic'],
        diastolic: ['이완기혈압', '최저혈압', 'DBP', 'diastolic', '이완기', '최저', 'Diastolic'],

        // 지질
        triglycerides: ['중성지방', 'TG', 'triglycerides', 'Triglycerides'],
        hdl: ['HDL콜레스테롤', 'HDL-C', 'HDL', 'hdl'],
        ldl: ['LDL콜레스테롤', 'LDL-C', 'LDL', 'ldl'],
        totalCholesterol: ['총콜레스테롤', 'Total-C', 'TC', 'totalCholesterol'],

        // 혈당
        glucose: ['공복혈당', 'FBS', 'glucose', 'Glucose', 'GLU'],
        hba1c: ['당화혈색소', 'HbA1c', 'hba1c', 'A1C'],

        // 비만
        height: ['신장', '키', 'height', 'Height'],
        weight: ['체중', 'weight', 'Weight'],
        bmi: ['BMI', 'bmi', '체질량지수'],
        waistCircumference: ['허리둘레', '복부둘레', 'waist', 'Waist'],

        // 요단백
        urineProtein: ['요단백', '소변단백', 'protein', 'Protein', 'PRO']
      };

      // 필드 매핑을 통한 값 추출
      for (const [standardField, possibleNames] of Object.entries(fieldMappings)) {
        for (const possibleName of possibleNames) {
          if (rawData[possibleName] !== undefined && rawData[possibleName] !== null && rawData[possibleName] !== '') {
            const value = parseValue(rawData[possibleName]);
            if (value !== null) {
              normalized[standardField] = value;
              break;
            }
          }
        }
      }

      // BMI 계산 (신장과 체중이 있지만 BMI가 없는 경우)
      if (!normalized.bmi && normalized.height && normalized.weight) {
        const heightInM = normalized.height / 100;
        normalized.bmi = normalized.weight / (heightInM * heightInM);
      }

      return normalized;
    }

    // AI 분석 함수
    function analyzePatientHealth(normalizedData: any, patientName: string) {
      const result = {
        patientName,
        bloodPressureScore: 0,
        cholesterolScore: 0,
        hba1cReceived: normalizedData.hba1c ? 1 : 0,
        bloodSugarScore: 0,
        obesityScore: 0,
        urineProteinScore: 0,
        pastHistoryTreatment: '무',
        currentHistoryTreatment: '무',
        finalJudgment: 0
      };

      // 1. 혈압 판정
      if (normalizedData.systolic >= HEALTH_EVALUATION_CRITERIA.bloodPressure.systolic ||
          normalizedData.diastolic >= HEALTH_EVALUATION_CRITERIA.bloodPressure.diastolic) {
        result.bloodPressureScore = 1;
      }

      // 2. 지질 판정 (하나라도 해당되면 1)
      let lipidIssues = 0;
      if (normalizedData.triglycerides >= HEALTH_EVALUATION_CRITERIA.lipid.triglycerides) lipidIssues++;
      if (normalizedData.hdl < HEALTH_EVALUATION_CRITERIA.lipid.hdl) lipidIssues++;
      if (normalizedData.ldl >= HEALTH_EVALUATION_CRITERIA.lipid.ldl) lipidIssues++;

      if (lipidIssues > 0) {
        result.cholesterolScore = 1;
      }

      // 3. 혈당 판정
      if (normalizedData.glucose >= HEALTH_EVALUATION_CRITERIA.bloodSugar.glucose ||
          normalizedData.hba1c >= HEALTH_EVALUATION_CRITERIA.bloodSugar.hba1c) {
        result.bloodSugarScore = 1;
      }

      // 4. 비만 판정
      if (normalizedData.bmi >= HEALTH_EVALUATION_CRITERIA.obesity.bmi) {
        result.obesityScore = 1;
      }

      // 5. 요단백 판정
      if (normalizedData.urineProtein >= HEALTH_EVALUATION_CRITERIA.urineProtein.threshold) {
        result.urineProteinScore = 1;
      }

      // 6. 최종 판정 (총합)
      result.finalJudgment = result.bloodPressureScore + result.cholesterolScore +
                           result.bloodSugarScore + result.obesityScore + result.urineProteinScore;

      return result;
    }

    // 작업 상태를 processing으로 변경
    await dbManager.query(
      'UPDATE health_batch_jobs SET status = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
      ['processing', jobId]
    );

    let successCount = 0;
    let totalReanalyzed = 0;

    // 각 실패한 환자에 대해 재분석 수행
    for (const patient of failedPatients) {
      totalReanalyzed++;

      try {
        // 원시 데이터 정규화
        const normalizedData = normalizePatientData(patient.raw_data);

        // AI 분석 수행
        const analysisResult = analyzePatientHealth(normalizedData, patient.patient_name);

        // 분석 결과를 데이터베이스에 업데이트
        await dbManager.query(
          `UPDATE health_batch_patients
           SET status = $1, analysis_result = $2, error_message = NULL, updated_at = CURRENT_TIMESTAMP
           WHERE id = $3`,
          ['completed', JSON.stringify(analysisResult), patient.id]
        );

        successCount++;

      } catch (error) {
        console.error(`환자 ${patient.patient_name} 재분석 실패:`, error);

        // 실패한 경우 에러 메시지 업데이트
        await dbManager.query(
          `UPDATE health_batch_patients
           SET error_message = $1, updated_at = CURRENT_TIMESTAMP
           WHERE id = $2`,
          [error instanceof Error ? error.message : '알 수 없는 오류', patient.id]
        );
      }
    }

    // 작업의 전체 통계 업데이트
    const statsQuery = await dbManager.query(
      `SELECT
         COUNT(*) as total,
         COUNT(*) FILTER (WHERE status = 'completed') as completed,
         COUNT(*) FILTER (WHERE status = 'failed') as failed
       FROM health_batch_patients WHERE job_id = $1`,
      [jobId]
    );

    const stats = statsQuery.rows[0];

    await dbManager.query(
      `UPDATE health_batch_jobs
       SET status = $1, processed_records = $2, successful_records = $3, failed_records = $4, updated_at = CURRENT_TIMESTAMP
       WHERE id = $5`,
      ['completed', stats.total, stats.completed, stats.failed, jobId]
    );

    return NextResponse.json({
      success: true,
      data: {
        jobId,
        totalReanalyzed,
        successCount,
        failureCount: totalReanalyzed - successCount,
        message: `${totalReanalyzed}명의 환자 중 ${successCount}명이 성공적으로 재분석되었습니다.`
      }
    });

  } catch (error) {
    console.error('재분석 API 오류:', error);
    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : '재분석 중 오류가 발생했습니다.'
    }, { status: 500 });
  }
}