import { NextRequest, NextResponse } from 'next/server';
import { withTokenAuth } from '@/app/api/middlewares';
import { externalApiClient } from '@/lib/apiClient';
import type { ProcessType } from '@/types/flowai';
import { getApiServerUrl, getWebsearchServerUrl, getRagServerUrl, getReportServerUrl } from '@/config/serverConfig';

interface ExecuteRequest {
  nodeType: string;
  processType: ProcessType;
  config: any;
  input?: any;
}

// 노드 실행
export async function POST(request: NextRequest) {
  return withTokenAuth(request, async (req, user) => {
    try {

    const body: ExecuteRequest = await request.json();
    const { nodeType, processType, config, input } = body;

    let result;

    // 노드 타입별 실행 로직
    switch (nodeType) {
      case 'input':
        result = await executeInputNode(config, input);
        break;
      
      case 'process':
        result = await executeProcessNode(processType, config, input, user);
        break;
      
      case 'output':
        result = await executeOutputNode(config, input, user);
        break;
      
      case 'condition':
        result = await executeConditionNode(config, input);
        break;
      
      case 'loop':
        result = await executeLoopNode(config, input);
        break;
      
      default:
        throw new Error(`지원하지 않는 노드 타입: ${nodeType}`);
    }

    return NextResponse.json({
      success: true,
      data: result,
      timestamp: new Date().toISOString(),
    });

    } catch (error) {
      console.error('노드 실행 오류:', error);
      return NextResponse.json(
        { 
          success: false, 
          error: { 
            code: 'EXECUTION_ERROR',
            message: error instanceof Error ? error.message : '노드 실행에 실패했습니다' 
          } 
        },
        { status: 500 }
      );
    }
  });
}

// 입력 노드 실행
async function executeInputNode(config: any, input?: any): Promise<any> {
  return config.value || input || '';
}

// 처리 노드 실행
async function executeProcessNode(processType: ProcessType, config: any, input: any, user: any): Promise<any> {
  switch (processType) {
    case 'text-generation':
      return await executeTextGeneration(config, input, user);
    
    case 'image-generation':
      return await executeImageGeneration(config, input, user);
    
    case 'code-generation':
      return await executeCodeGeneration(config, input, user);
    
    case 'document-analysis':
      return await executeDocumentAnalysis(config, input, user);
    
    case 'web-search':
      return await executeWebSearch(config, input, user);
    
    case 'rag-search':
      return await executeRagSearch(config, input, user);
    
    case 'data-transform':
      return await executeDataTransform(config, input);
    
    case 'api-call':
      return await executeApiCall(config, input);
    
    default:
      throw new Error(`지원하지 않는 처리 타입: ${processType}`);
  }
}

// 텍스트 생성
async function executeTextGeneration(config: any, input: any, user: any): Promise<string> {
  const prompt = config.prompt || '';
  const finalPrompt = prompt + (input ? `\n\n입력: ${input}` : '');

  try {
    // 내부 채팅 API 호출
    const response = await fetch(`${getApiServerUrl().replace('/api/v1', '')}/api/chat`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        message: finalPrompt,
        model: config.model || 'gpt-4',
        temperature: config.temperature || 0.7,
        max_tokens: config.maxTokens || 2000,
        userId: user.id,
        username: user.username,
      }),
    });

    if (!response.ok) {
      throw new Error('텍스트 생성 API 호출 실패');
    }

    const result = await response.json();
    return result.response || '';
  } catch (error) {
    console.error('텍스트 생성 오류:', error);
    throw new Error('텍스트 생성에 실패했습니다');
  }
}

// 이미지 생성
async function executeImageGeneration(config: any, input: any, user: any): Promise<string> {
  const prompt = config.prompt || input || '';
  
  try {
    // 이미지 생성 API 호출
    const response = await fetch(`${getApiServerUrl().replace('/api/v1', '')}/api/image/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        prompt: prompt,
        size: config.size || '1024x1024',
        model: config.model || 'dall-e-3',
        userId: user.id,
      }),
    });

    if (!response.ok) {
      throw new Error('이미지 생성 API 호출 실패');
    }

    const result = await response.json();
    return result.imageUrl || '';
  } catch (error) {
    console.error('이미지 생성 오류:', error);
    throw new Error('이미지 생성에 실패했습니다');
  }
}

// 코드 생성
async function executeCodeGeneration(config: any, input: any, user: any): Promise<string> {
  const prompt = `${config.prompt || '다음 기능을 구현하는 코드를 작성해주세요:'}\n\n${input || ''}`;
  const language = config.language || 'javascript';
  
  const fullPrompt = `언어: ${language}\n요청사항: ${prompt}\n\n위 요청사항에 맞는 ${language} 코드를 작성해주세요. 코드만 반환하고 설명은 주석으로 포함해주세요.`;

  try {
    const response = await fetch(`${getApiServerUrl().replace('/api/v1', '')}/api/chat`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        message: fullPrompt,
        model: config.model || 'gpt-4',
        temperature: config.temperature || 0.3,
        max_tokens: config.maxTokens || 4000,
        userId: user.id,
        username: user.username,
      }),
    });

    if (!response.ok) {
      throw new Error('코드 생성 API 호출 실패');
    }

    const result = await response.json();
    return result.response || '';
  } catch (error) {
    console.error('코드 생성 오류:', error);
    throw new Error('코드 생성에 실패했습니다');
  }
}

// 문서 분석
async function executeDocumentAnalysis(config: any, input: any, user: any): Promise<string> {
  const analysisType = config.analysisType || 'summary';
  let prompt = '';

  switch (analysisType) {
    case 'summary':
      prompt = '다음 문서를 요약해주세요:';
      break;
    case 'keywords':
      prompt = '다음 문서에서 주요 키워드들을 추출해주세요:';
      break;
    case 'sentiment':
      prompt = '다음 문서의 감정을 분석해주세요:';
      break;
    default:
      prompt = '다음 문서를 분석해주세요:';
  }

  const fullPrompt = `${prompt}\n\n${input || ''}`;

  try {
    const response = await fetch(`${getApiServerUrl().replace('/api/v1', '')}/api/chat`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        message: fullPrompt,
        model: config.model || 'gpt-4',
        temperature: 0.3,
        max_tokens: 2000,
        userId: user.id,
        username: user.username,
      }),
    });

    if (!response.ok) {
      throw new Error('문서 분석 API 호출 실패');
    }

    const result = await response.json();
    return result.response || '';
  } catch (error) {
    console.error('문서 분석 오류:', error);
    throw new Error('문서 분석에 실패했습니다');
  }
}

// 웹 검색
async function executeWebSearch(config: any, input: any, user: any): Promise<string> {
  const query = input || config.query || '';
  const maxResults = config.maxResults || 10;

  try {
    const response = await fetch(`${getWebsearchServerUrl()}/search`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        query: query,
        maxResults: maxResults,
        searchEngine: config.searchEngine || 'google',
      }),
    });

    if (!response.ok) {
      throw new Error('웹 검색 API 호출 실패');
    }

    const result = await response.json();
    return JSON.stringify(result.results || [], null, 2);
  } catch (error) {
    console.error('웹 검색 오류:', error);
    throw new Error('웹 검색에 실패했습니다');
  }
}

// RAG 검색
async function executeRagSearch(config: any, input: any, user: any): Promise<string> {
  const query = input || config.query || '';

  try {
    const response = await fetch(`${getRagServerUrl()}/search`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        query: query,
        username: user.username,
        maxResults: config.maxResults || 10,
        scope: config.scope || 'personal',
      }),
    });

    if (!response.ok) {
      throw new Error('RAG 검색 API 호출 실패');
    }

    const result = await response.json();
    return JSON.stringify(result.documents || [], null, 2);
  } catch (error) {
    console.error('RAG 검색 오류:', error);
    throw new Error('RAG 검색에 실패했습니다');
  }
}

// 데이터 변환
async function executeDataTransform(config: any, input: any): Promise<any> {
  const transformType = config.transformType || 'json';
  
  try {
    switch (transformType) {
      case 'json':
        return typeof input === 'string' ? JSON.parse(input) : input;
      case 'string':
        return typeof input === 'object' ? JSON.stringify(input, null, 2) : String(input);
      case 'csv':
        if (Array.isArray(input)) {
          const headers = Object.keys(input[0] || {});
          const rows = input.map(item => headers.map(header => item[header] || '').join(','));
          return [headers.join(','), ...rows].join('\n');
        }
        return String(input);
      case 'xml':
        // 간단한 JSON to XML 변환
        const jsonToXml = (obj: any, rootName = 'root'): string => {
          let xml = `<${rootName}>`;
          for (const [key, value] of Object.entries(obj)) {
            if (typeof value === 'object') {
              xml += jsonToXml(value, key);
            } else {
              xml += `<${key}>${value}</${key}>`;
            }
          }
          xml += `</${rootName}>`;
          return xml;
        };
        return jsonToXml(typeof input === 'string' ? JSON.parse(input) : input);
      default:
        return input;
    }
  } catch (error) {
    console.error('데이터 변환 오류:', error);
    throw new Error('데이터 변환에 실패했습니다');
  }
}

// API 호출
async function executeApiCall(config: any, input: any): Promise<any> {
  const method = config.method || 'GET';
  const url = config.apiEndpoint;
  
  if (!url) {
    throw new Error('API URL이 설정되지 않았습니다');
  }

  try {
    const options: RequestInit = {
      method: method,
      headers: {
        'Content-Type': 'application/json',
        ...config.headers,
      },
    };

    if (method !== 'GET' && input) {
      options.body = typeof input === 'string' ? input : JSON.stringify(input);
    }

    const response = await fetch(url, options);
    
    if (!response.ok) {
      throw new Error(`API 호출 실패: ${response.status} ${response.statusText}`);
    }

    const contentType = response.headers.get('content-type');
    if (contentType?.includes('application/json')) {
      return await response.json();
    } else {
      return await response.text();
    }
  } catch (error) {
    console.error('API 호출 오류:', error);
    throw new Error('API 호출에 실패했습니다');
  }
}

// 출력 노드 실행
async function executeOutputNode(config: any, input: any, user: any): Promise<any> {
  const outputFormat = config.outputFormat || 'text';
  const filename = config.filename || 'output';
  
  try {
    switch (outputFormat) {
      case 'text':
        return { format: 'text', content: String(input), filename: `${filename}.txt` };
      
      case 'pdf':
        return await generatePDF(input, config, user);
      
      case 'docx':
        return await generateDOCX(input, config, user);
      
      case 'pptx':
        return await generatePPTX(input, config, user);
      
      case 'hwpx':
        return await generateHWPX(input, config, user);
      
      case 'html':
        return { format: 'html', content: `<html><body>${input}</body></html>`, filename: `${filename}.html` };
      
      case 'markdown':
        return { format: 'markdown', content: String(input), filename: `${filename}.md` };
      
      case 'json':
        return { format: 'json', content: JSON.stringify(input, null, 2), filename: `${filename}.json` };
      
      case 'csv':
        if (Array.isArray(input)) {
          const headers = Object.keys(input[0] || {});
          const rows = input.map(item => headers.map(header => item[header] || '').join(','));
          const csvContent = [headers.join(','), ...rows].join('\n');
          return { format: 'csv', content: csvContent, filename: `${filename}.csv` };
        }
        return { format: 'csv', content: String(input), filename: `${filename}.csv` };
      
      default:
        return { format: 'text', content: String(input), filename: `${filename}.txt` };
    }
  } catch (error) {
    console.error('출력 노드 실행 오류:', error);
    throw new Error('출력 생성에 실패했습니다');
  }
}

// PDF 생성
async function generatePDF(content: any, config: any, user: any): Promise<any> {
  try {
    const response = await fetch(`${getReportServerUrl()}/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        content: String(content),
        format: 'pdf',
        template: config.template || 'default',
        filename: config.filename || 'output',
        userId: user.id,
      }),
    });

    if (!response.ok) {
      throw new Error('PDF 생성 API 호출 실패');
    }

    const result = await response.json();
    return { format: 'pdf', url: result.url, filename: result.filename };
  } catch (error) {
    console.error('PDF 생성 오류:', error);
    throw new Error('PDF 생성에 실패했습니다');
  }
}

// DOCX 생성
async function generateDOCX(content: any, config: any, user: any): Promise<any> {
  try {
    const response = await fetch(`${getReportServerUrl()}/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        content: String(content),
        format: 'docx',
        template: config.template || 'default',
        filename: config.filename || 'output',
        userId: user.id,
      }),
    });

    if (!response.ok) {
      throw new Error('DOCX 생성 API 호출 실패');
    }

    const result = await response.json();
    return { format: 'docx', url: result.url, filename: result.filename };
  } catch (error) {
    console.error('DOCX 생성 오류:', error);
    throw new Error('DOCX 생성에 실패했습니다');
  }
}

// PPTX 생성
async function generatePPTX(content: any, config: any, user: any): Promise<any> {
  try {
    const response = await fetch(`${getReportServerUrl()}/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        content: String(content),
        format: 'pptx',
        template: config.template || 'presentation',
        filename: config.filename || 'output',
        userId: user.id,
      }),
    });

    if (!response.ok) {
      throw new Error('PPTX 생성 API 호출 실패');
    }

    const result = await response.json();
    return { format: 'pptx', url: result.url, filename: result.filename };
  } catch (error) {
    console.error('PPTX 생성 오류:', error);
    throw new Error('PPTX 생성에 실패했습니다');
  }
}

// HWPX 생성
async function generateHWPX(content: any, config: any, user: any): Promise<any> {
  try {
    const response = await fetch(`${getReportServerUrl()}/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        content: String(content),
        format: 'hwpx',
        template: config.template || 'default',
        filename: config.filename || 'output',
        userId: user.id,
      }),
    });

    if (!response.ok) {
      throw new Error('HWPX 생성 API 호출 실패');
    }

    const result = await response.json();
    return { format: 'hwpx', url: result.url, filename: result.filename };
  } catch (error) {
    console.error('HWPX 생성 오류:', error);
    throw new Error('HWPX 생성에 실패했습니다');
  }
}

// 조건 노드 실행
async function executeConditionNode(config: any, input: any): Promise<boolean> {
  const condition = config.condition || '';
  const operator = config.operator || 'equals';
  const value = config.value || '';

  try {
    switch (operator) {
      case 'equals':
        return String(input) === String(value);
      case 'not_equals':
        return String(input) !== String(value);
      case 'greater':
        return Number(input) > Number(value);
      case 'less':
        return Number(input) < Number(value);
      case 'contains':
        return String(input).includes(String(value));
      case 'regex':
        const regex = new RegExp(value);
        return regex.test(String(input));
      default:
        // 조건식 평가 (간단한 경우만)
        const safeEval = (expr: string, inputValue: any): boolean => {
          // 보안상 eval 사용을 피하고 간단한 비교만 처리
          if (expr.includes('>')) {
            const [left, right] = expr.split('>');
            return Number(left.trim() === 'input' ? inputValue : left.trim()) > Number(right.trim());
          }
          if (expr.includes('<')) {
            const [left, right] = expr.split('<');
            return Number(left.trim() === 'input' ? inputValue : left.trim()) < Number(right.trim());
          }
          if (expr.includes('==')) {
            const [left, right] = expr.split('==');
            return String(left.trim() === 'input' ? inputValue : left.trim()) === String(right.trim());
          }
          return Boolean(inputValue);
        };
        
        return safeEval(condition, input);
    }
  } catch (error) {
    console.error('조건 평가 오류:', error);
    return false;
  }
}

// 반복 노드 실행
async function executeLoopNode(config: any, input: any): Promise<any[]> {
  const loopCount = config.loopCount || 1;
  const results = [];

  for (let i = 0; i < loopCount; i++) {
    results.push({
      iteration: i + 1,
      input: input,
      result: `반복 ${i + 1} 완료`
    });
  }

  return results;
}