import { NextRequest, NextResponse } from 'next/server';

import { getLLMFinetuneServerUrl } from '@/config/serverConfig';

const LLM_FINETUNE_API_BASE_URL = getLLMFinetuneServerUrl();

export async function GET() {
  try {
    console.log('[LLM Models API] 모델 목록 요청 시작');
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/models`);
    console.log('[LLM Models API] 응답 상태:', response.status, response.statusText);
    
    if (!response.ok) {
      console.error('[LLM Models API] HTTP 오류:', response.status, response.statusText);
      const errorText = await response.text();
      console.error('[LLM Models API] 오류 내용:', errorText);
      
      return NextResponse.json(
        { 
          success: false, 
          error: `LLM 모델 서버 오류: ${response.status} ${response.statusText}`,
          details: errorText
        },
        { status: response.status }
      );
    }
    
    const data = await response.json();
    console.log('[LLM Models API] 성공적으로 데이터 받음');
    
    return NextResponse.json(data);
  } catch (error) {
    console.error('[LLM Models API] 모델 목록 호출 실패:', error);
    
    const isNetworkError = error instanceof Error && (
      error.message.includes('ECONNREFUSED') ||
      error.message.includes('ENOTFOUND') ||
      error.message.includes('ETIMEDOUT') ||
      error.message.includes('fetch failed')
    );
    
    return NextResponse.json(
      { 
        success: false, 
        error: isNetworkError 
          ? `LLM 파인튜닝 서버(${LLM_FINETUNE_API_BASE_URL})에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.`
          : '모델 목록을 불러올 수 없습니다.',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: isNetworkError ? 503 : 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const body = await request.json();
    const { modelId } = body;
    
    if (!modelId) {
      return NextResponse.json(
        { success: false, error: '모델 ID가 필요합니다.' },
        { status: 400 }
      );
    }
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/models/${modelId}`, {
      method: 'DELETE'
    });
    
    const result = await response.json();
    
    if (response.ok && result.success) {
      return NextResponse.json({
        success: true,
        message: '파인튜닝된 모델이 성공적으로 삭제되었습니다.'
      });
    } else {
      return NextResponse.json(
        { success: false, error: result.error || '모델 삭제에 실패했습니다.' },
        { status: response.status || 500 }
      );
    }
    
  } catch (error) {
    console.error('[LLM Models API] 모델 삭제 실패:', error);
    return NextResponse.json(
      { success: false, error: '모델 삭제 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}