import { NextRequest, NextResponse } from 'next/server';

import { getLLMFinetuneServerUrl } from '@/config/serverConfig';

const LLM_FINETUNE_API_BASE_URL = getLLMFinetuneServerUrl();

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    console.log('[LLM Playground API] 모델 테스트 요청:', body);
    
    // 필수 필드 검증
    const { modelId, prompt } = body;
    if (!modelId || !prompt) {
      return NextResponse.json(
        { success: false, error: '모델 ID와 프롬프트는 필수입니다.' },
        { status: 400 }
      );
    }
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/playground/generate`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        modelId,
        prompt,
        
        // 생성 설정
        maxLength: body.maxLength || 512,
        temperature: body.temperature || 0.7,
        topP: body.topP || 0.9,
        topK: body.topK || 50,
        repetitionPenalty: body.repetitionPenalty || 1.0,
        
        // 스트리밍 설정
        stream: body.stream || false,
        
        // 시스템 메시지
        systemMessage: body.systemMessage || '',
        
        // 대화 히스토리
        conversationHistory: body.conversationHistory || []
      })
    });
    
    // 스트리밍 응답인 경우
    if (body.stream && response.body) {
      // ReadableStream을 클라이언트에 전달
      return new Response(response.body, {
        headers: {
          'Content-Type': 'text/event-stream',
          'Cache-Control': 'no-cache',
          'Connection': 'keep-alive',
        },
      });
    }
    
    const data = await response.json();
    console.log('[LLM Playground API] 생성 응답:', data);
    
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('[LLM Playground API] 텍스트 생성 실패:', error);
    
    const isNetworkError = error instanceof Error && (
      error.message.includes('ECONNREFUSED') ||
      error.message.includes('ENOTFOUND') ||
      error.message.includes('ETIMEDOUT') ||
      error.message.includes('fetch failed')
    );
    
    return NextResponse.json(
      { 
        success: false, 
        error: isNetworkError 
          ? `LLM 파인튜닝 서버(${LLM_FINETUNE_API_BASE_URL})에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.`
          : '텍스트 생성에 실패했습니다.',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: isNetworkError ? 503 : 500 }
    );
  }
}

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const modelId = searchParams.get('modelId');
    
    if (!modelId) {
      return NextResponse.json(
        { success: false, error: '모델 ID가 필요합니다.' },
        { status: 400 }
      );
    }
    
    console.log('[LLM Playground API] 모델 정보 요청:', modelId);
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/playground/model-info?modelId=${modelId}`);
    
    if (!response.ok) {
      console.error('[LLM Playground API] HTTP 오류:', response.status, response.statusText);
      const errorText = await response.text();
      
      return NextResponse.json(
        { 
          success: false, 
          error: `모델 정보 조회 오류: ${response.status} ${response.statusText}`,
          details: errorText
        },
        { status: response.status }
      );
    }
    
    const data = await response.json();
    console.log('[LLM Playground API] 모델 정보 응답:', data);
    
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('[LLM Playground API] 모델 정보 조회 실패:', error);
    
    return NextResponse.json(
      { success: false, error: '모델 정보를 조회할 수 없습니다.' },
      { status: 500 }
    );
  }
}