import { NextRequest, NextResponse } from 'next/server';
import { exec } from 'child_process';
import { promisify } from 'util';
import path from 'path';
import os from 'os';
import fs from 'fs';

const execAsync = promisify(exec);

// 기본 MCP 설정 (JavaScript 버전)
const DEFAULT_MCP_CONFIG = {
  "hamonize": {
    "command": "node",
    "args": ["./mcp_tools/hamonize/dist/index.js"],
    "transport": "stdio"
  },
  "postgres": {
    "transport": "stdio",
    "command": "node",
    "args": ["./mcp_tools/mcp-postgres-server/build/index.js"]
  }
};



// 전역 MCP 클라이언트 임포트
import { getGlobalMCPClient } from '@/lib/mcpClient';

// 인증 정보 저장 함수
const saveAuthInfo = async (authInfo: any) => {
  try {
    const configDir = path.join(os.homedir(), '.airun');
    const configFile = path.join(configDir, 'airun.conf');
    
    // 디렉토리가 없으면 생성
    if (!fs.existsSync(configDir)) {
      fs.mkdirSync(configDir, { recursive: true });
    }
    
    // 기존 설정 파일 읽기
    let existingConfig = '';
    if (fs.existsSync(configFile)) {
      existingConfig = fs.readFileSync(configFile, 'utf8');
    }
    
    // 새로운 설정 항목들
    const newConfigItems = [];
    
    if (authInfo.apiKey) {
      newConfigItems.push(`export AIRUN_API_KEY="${authInfo.apiKey}"`);
    }
    
    if (authInfo.username) {
      newConfigItems.push(`export AIRUN_USERNAME="${authInfo.username}"`);
    }
    
    if (authInfo.userId) {
      newConfigItems.push(`export AIRUN_USER_ID="${authInfo.userId}"`);
    }
    
    if (authInfo.userToken) {
      newConfigItems.push(`export AIRUN_USER_TOKEN="${authInfo.userToken}"`);
    }
    
    // 기존 설정에서 같은 키 제거
    const lines = existingConfig.split('\n');
    const filteredLines = lines.filter(line => {
      const trimmed = line.trim();
      return !trimmed.startsWith('export AIRUN_API_KEY=') &&
             !trimmed.startsWith('export AIRUN_USERNAME=') &&
             !trimmed.startsWith('export AIRUN_USER_ID=') &&
             !trimmed.startsWith('export AIRUN_USER_TOKEN=');
    });
    
    // 새로운 설정 추가
    const finalConfig = [...filteredLines, ...newConfigItems].join('\n');
    
    // 설정 파일 저장
    fs.writeFileSync(configFile, finalConfig, 'utf8');
    
    console.log(`인증 정보 저장 완료: ${configFile}`);
  } catch (error) {
    console.error('인증 정보 저장 실패:', error);
    throw error;
  }
};

// MCP 설정 로드
const loadMCPConfig = () => {
  try {
    // 프로젝트 루트의 mcp_config.json 파일 사용
    const configPath = path.join(process.cwd(), 'mcp_config.json');
    
    if (fs.existsSync(configPath)) {
      const configData = fs.readFileSync(configPath, 'utf8');
      const config = JSON.parse(configData);
      
      console.log(`MCP 설정 로드 완료: ${configPath}`);
      console.log(`서버 수: ${Object.keys(config).length}개`);
      
      return config;
    } else {
      console.log(`MCP 설정 파일을 찾을 수 없습니다: ${configPath}`);
      // 설정 파일이 없으면 빈 설정 사용
      return {};
    }
  } catch (error) {
    console.error('MCP 설정 로드 중 오류:', error);
    return {};
  }
};

export async function POST(request: NextRequest) {
  try {
    const { config, authInfo } = await request.json();
    
    // 인증 정보가 있으면 설정 파일에 저장
    if (authInfo) {
      try {
        await saveAuthInfo(authInfo);
        console.log('인증 정보 저장 완료:', {
          hasToken: !!authInfo.userToken,
          hasApiKey: !!authInfo.apiKey,
          username: authInfo.username,
          userId: authInfo.userId
        });
      } catch (error) {
        console.error('인증 정보 저장 실패:', error);
      }
    }
    
    const mcpConfig = config || loadMCPConfig();
    
    // 전역 MCP 클라이언트 초기화
    try {
      console.log('🔄 전역 MCP 클라이언트 초기화 시작...');
      const client = await getGlobalMCPClient();
      
      if (client) {
        const serverCount = client.getServerCount();
        console.log(`✅ MCP 초기화 완료: ${serverCount}개 서버`);
        
        // 도구 정보도 함께 가져오기
        try {
          const allTools = await client.getAllTools();
          
          // 서버별로 도구 정보 그룹화 (/api/mcp/tools와 동일한 로직)
          const serverToolsMap = new Map<string, any[]>();
          
          for (const tool of allTools) {
            const serverName = tool.server || tool.serverName || 'unknown';
            if (!serverToolsMap.has(serverName)) {
              serverToolsMap.set(serverName, []);
            }
            serverToolsMap.get(serverName)!.push({
              name: tool.name,
              description: tool.description || '설명 없음'
            });
          }
          
          // 서버별 도구 정보 생성
          const serverToolsInfo = Array.from(serverToolsMap.entries()).map(([serverName, tools]) => ({
            serverName,
            toolCount: tools.length,
            tools: tools,
            transport: 'stdio'
          }));
          
          const totalTools = allTools.length;
          
          return NextResponse.json({
            success: true,
            message: `MCP 서버 초기화 완료 (${serverCount}개 서버, ${totalTools}개 도구 사용 가능)`,
            config: mcpConfig,
            initialized: true,
            toolsData: { servers: serverToolsInfo, totalToolCount: totalTools } // 도구 정보 추가
          });
        } catch (toolsError) {
          console.error('도구 정보 가져오기 실패:', toolsError);
          
          // 도구 정보 가져오기 실패해도 초기화는 성공으로 처리
          return NextResponse.json({
            success: true,
            message: `MCP 서버 초기화 완료 (${serverCount}개 서버, 도구 정보 로드 실패)`,
            config: mcpConfig,
            initialized: true,
            toolsData: null
          });
        }
      } else {
        throw new Error('전역 MCP 클라이언트 초기화 실패');
      }
      
    } catch (mcpError) {
      console.error('❌ 전역 MCP 클라이언트 초기화 실패:', mcpError);
      
      // 실패해도 설정 정보는 반환
      return NextResponse.json({
        success: false,
        error: 'MCP 클라이언트 초기화 실패',
        details: mcpError instanceof Error ? mcpError.message : String(mcpError),
        config: mcpConfig
      }, { status: 500 });
    }
    
  } catch (error) {
    console.error('MCP 초기화 중 오류:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: 'MCP 초기화 중 오류가 발생했습니다',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: 500 }
    );
  }
}

// GET: MCP 상태 확인
export async function GET() {
  try {
    const mcpConfig = loadMCPConfig();
    const toolCount = Object.keys(mcpConfig).length;
    
    return NextResponse.json({
      success: true,
      initialized: true,
      toolCount: toolCount,
      availableTools: Object.keys(mcpConfig),
      config: mcpConfig
    });
  } catch (error) {
    console.error('MCP 상태 확인 실패:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: 'MCP 상태 확인에 실패했습니다.',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: 500 }
    );
  }
} 