'use client';

import { useState, useEffect, useRef, useMemo, Suspense, useCallback } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { externalApiClient, internalApiClient } from '@/lib/apiClient';
import { streamChatWithWebSocket, StreamCallbacks, getAuthHeaders } from '@/utils/api';
import { ChatMarkdownRenderer } from '@/components/ChatMarkdownRenderer';
import { useAuth } from '@/hooks/useAuth';
import { useLanguage } from '@/contexts/LanguageContext';
import { useInfiniteScroll } from '@/hooks/useInfiniteScroll';
import ClientOnly from '@/components/ClientOnly';
import { BackgroundJobStatus } from '@/components/BackgroundJobStatus';
import type { BackgroundJob } from '@/types/support';
import { DocumentPreviewModal } from '@/components/DocumentPreviewModal';
import { getApiServerUrl } from '@/config/serverConfig';
import ExcelPreprocessModal from '@/components/rag/ExcelPreprocessModal';
import GraphRAGModal from '@/components/GraphRAGModal';
import { FileManagementModal } from '@/components/FileManagementModal';


interface DownloadInfo {
  filename: string;
  pages?: number[];  // Optional for web sources
  url?: string;      // URL for web sources
  title?: string;    // Title for web sources
  source?: 'web' | 'rag' | 'doc';  // Source type
}

interface Message {
  id: string;
  role: 'user' | 'assistant';
  content: string;
  timestamp: string;
  originalContent?: string; // 멘션된 문서가 추가되기 전 원본 질문 (히스토리 표시용)
  attachedImages?: {
    file: File;
    url: string;
    name: string;
    base64?: string;
  }[];
  downloadInfo?: DownloadInfo[];
  thinking?: string; // AI의 사고 과정
}

interface ChatSession {
  id: string;
  title: string;
  lastMessage: string;
  updatedAt: string;
  createdAt: string; // 추가
  messageCount: number;
  provider?: string;
  model?: string;
  history: Message[];
}

interface ChatOptions {
  provider: string;
  model: string;
  temperature: number;
  maxTokens: number;
  enableRag: boolean;
  enableWebSearch: boolean;
  enableHealthAnalysis: boolean;

  ragSearchScope: 'personal' | 'shared' | 'all';
  userId: number;
  history: Message[];
  reasoningLevel: 'low' | 'medium' | 'high';
}

interface RagFile {
  name: string;
  size: number;
  modified?: string;
  type?: string;
  status: 'uploading' | 'processing' | 'completed' | 'error';
  progress?: number;
  error?: string;
}

interface MentionedDocument {
  name: string;
  displayName: string;
}

interface DocumentContext {
  documentName: string;
  totalChunks: number;
  selectedChunks: any[];
  metadata: {
    fileName: string;
    chunkCount: number;
    selectedCount: number;
  };
}

interface ApiSession {
  id: string;
  type: string;
  firstAIResponse?: string;
  lastMessage?: string;
  updated_at?: string;
  messageCount?: number;
  provider?: string;
  model?: string;
  history?: Message[];
}

// 세션 목록을 머지하는 함수 (중복 id는 새로 받은 sessionList 기준으로 덮어씀)
function mergeSessions(oldSessions: ChatSession[], newSessions: ChatSession[]): ChatSession[] {
  const sessionMap = new Map<string, ChatSession>();
  oldSessions.forEach(s => sessionMap.set(s.id, s));
  newSessions.forEach(s => sessionMap.set(s.id, s)); // 새 세션이 덮어씀
  // updatedAt 기준 내림차순 정렬
  return Array.from(sessionMap.values()).sort((a, b) => b.updatedAt.localeCompare(a.updatedAt));
}

// useSearchParams를 사용하는 컴포넌트를 분리
function ChatPageContent() {
  const { user, isAuthenticated, isInitialized } = useAuth();
  const { t } = useLanguage();
  const router = useRouter();
  const searchParams = useSearchParams();

  // URL에서 세션 ID를 읽어와서 상태로 관리
  const [currentSessionId, setCurrentSessionId] = useState<string | null>(searchParams.get('session'));

  const [sessions, setSessions] = useState<ChatSession[]>([]);
  const [sessionPage, setSessionPage] = useState(1);
  const [hasMoreSessions, setHasMoreSessions] = useState(true);
  const [currentMessages, setCurrentMessages] = useState<Message[]>([]);
  const [inputMessage, setInputMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [isLoadingSessions, setIsLoadingSessions] = useState(false);
  const [error, setError] = useState<string | null>(null);
  // 모바일에서는 사이드바를 기본적으로 숨김
  const [showSidebar, setShowSidebar] = useState(() => {
    if (typeof window !== 'undefined') {
      return window.innerWidth >= 768; // 768px 이상에서만 기본 표시
    }
    return false; // SSR 시에는 숨김 상태로 시작
  });
  const [showOptions, setShowOptions] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');
  const [options, setOptions] = useState<ChatOptions>({
    provider: 'ollama',
    model: 'hamonize:latest',
    temperature: 1.0,
    maxTokens: 2000,
    enableRag: false, // 문서검색은 사용자가 선택적으로 활성화
    enableWebSearch: false,
    enableHealthAnalysis: false, // 건강검진 분석은 사용자가 선택적으로 활성화

    ragSearchScope: 'personal', // 기본값: 개인 문서만 검색
    userId: user?.id ? Number(user.id) : 0,
    history: [],
    reasoningLevel: 'low'
  });

  // 프로바이더 및 모델 관련 상태
  const [providers, setProviders] = useState<any>({});
  const [availableProviders, setAvailableProviders] = useState<any[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(false);

  const [uploadedFiles, setUploadedFiles] = useState<File[]>([]);
  const [uploadedFileUrls, setUploadedFileUrls] = useState<Map<string, string>>(new Map());
  const [isUploading, setIsUploading] = useState(false);
  const [uploadingFiles, setUploadingFiles] = useState<Set<string>>(new Set());

  const [isDragOver, setIsDragOver] = useState(false);
  const [progressMessages, setProgressMessages] = useState<string[]>([]);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [sessionToDelete, setSessionToDelete] = useState<string | null>(null);
  const [showDeleteAllModal, setShowDeleteAllModal] = useState(false);
  const [typingMessageId, setTypingMessageId] = useState<string | null>(null);
  const [displayedContent, setDisplayedContent] = useState<string>('');

  const [isNewChatMode, setIsNewChatMode] = useState(true);
  const isNewSessionJustCreatedRef = useRef(false);

  // 음성 인식 관련 상태
  const [isListening, setIsListening] = useState<boolean>(false);
  const [isProcessingAudio, setIsProcessingAudio] = useState<boolean>(false);
  const mediaRecorderRef = useRef<MediaRecorder | null>(null);
  const audioChunksRef = useRef<Blob[]>([]);

  // RAG 파일 브라우저 상태
  const [ragFiles, setRagFiles] = useState<RagFile[]>([]);
  const [isLoadingRagFiles, setIsLoadingRagFiles] = useState<boolean>(false);
  const [showRagBrowser, setShowRagBrowser] = useState<boolean>(false);

  // 프로젝트 관련 상태
  const [projects, setProjects] = useState<any[]>([]);
  const [selectedProject, setSelectedProject] = useState<any | null>(null);
  const [isLoadingProjects, setIsLoadingProjects] = useState<boolean>(false);
  const [showProjectBrowser, setShowProjectBrowser] = useState<boolean>(false);
  const [showProjectModal, setShowProjectModal] = useState<boolean>(false);
  const [projectModalType, setProjectModalType] = useState<'create' | 'edit' | 'delete'>('create');
  const [projectToEdit, setProjectToEdit] = useState<any | null>(null);
  const [newProjectName, setNewProjectName] = useState<string>('');
  const [newProjectDescription, setNewProjectDescription] = useState<string>('');

  // 메시지 액션 상태
  const [messageReactions, setMessageReactions] = useState<Record<string, { liked?: boolean; disliked?: boolean }>>({});
  const [showToast, setShowToast] = useState<{ message: string; type: 'success' | 'error' } | null>(null);
  const [copiedMessageId, setCopiedMessageId] = useState<string | null>(null);
  const [pdfProcessingId, setPdfProcessingId] = useState<string | null>(null);

  // 프로필 이미지 상태
  const [userProfileImage, setUserProfileImage] = useState<string>('/images/user.png');

  // @ 멘션 관련 상태
  const [mentionedDocs, setMentionedDocs] = useState<MentionedDocument[]>([]);
  const [showDocSuggestions, setShowDocSuggestions] = useState(false);
  const [mentionQuery, setMentionQuery] = useState('');
  const [mentionStartPos, setMentionStartPos] = useState(0);
  const [selectedSuggestionIndex, setSelectedSuggestionIndex] = useState(0);
  
  // 문서 미리보기 모달 상태
  const [showDocPreview, setShowDocPreview] = useState(false);
  const [previewFileName, setPreviewFileName] = useState<string>('');

  // GraphRAG 모달 상태
  const [showGraphRAGModal, setShowGraphRAGModal] = useState(false);
  const [currentGraphQuery, setCurrentGraphQuery] = useState<string>('');
  
  // 파일관리 모달 상태
  const [showFileManagementModal, setShowFileManagementModal] = useState(false);
  
  // GraphRAG 활성화 상태
  const [graphRAGEnabled, setGraphRAGEnabled] = useState(false);

  // 백그라운드 작업 관련 상태
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  // 임베딩 상태 추적
  const [embeddingStatuses, setEmbeddingStatuses] = useState<{[filename: string]: {step: string; step_label: string; isProcessing: boolean}}>({});

  // 현재 파일 목록 참조
  const currentFilesRef = useRef<File[]>([]);
  const [backgroundJobs, setBackgroundJobs] = useState<BackgroundJob[]>([]);

  // GraphRAG 설정 로드
  useEffect(() => {
    const loadGraphRAGSettings = async () => {
      try {
        const res = await externalApiClient.get('/config');
        if (res.success && res.data && res.data.configs) {
          const enableGraphRAG = res.data.configs['RAG_ENABLE_GRAPHRAG'];
          setGraphRAGEnabled(enableGraphRAG === 'yes' || enableGraphRAG === true);
        }
      } catch (error) {
        console.error('GraphRAG settings load failed:', error);
        setGraphRAGEnabled(false);
      }
    };
    
    loadGraphRAGSettings();
  }, []);

  // 백그라운드 작업 완료 시 콜백
  const handleJobComplete = useCallback(async (job: BackgroundJob) => {
    console.log('Background job completed:', job);

    if (job.type === 'file_upload') {
      // RAG 파일 목록 새로고침을 위한 트리거 업데이트
      setRefreshTrigger(prev => prev + 1);

      // 토스트 메시지 제거 - 사용자 요청
      // const baseMessage = `파일 "${job.filename}" 업로드 ${job.error ? '실패' : '완료'}`;
      // const fullMessage = job.error ? `${baseMessage}\n오류: ${job.error}` : baseMessage;

      // 오류 발생 시에만 콘솔 로그
      if (job.error) {
        console.error(`File upload failed: ${job.filename}`, job.error);
      } else {
        console.log(`File upload completed: ${job.filename}`);
      }
    }
  }, []);

  // RAG 서버에서 임베딩 상태 가져오기
  const fetchEmbeddingStatuses = useCallback(async () => {
    try {
      const response = await fetch('/api/rag/progress');
      if (response.ok) {
        const data = await response.json();
        const statusMap: {[filename: string]: {step: string; step_label: string; isProcessing: boolean}} = {};

        if (data.documents && Array.isArray(data.documents)) {
          data.documents.forEach((doc: any) => {
            statusMap[doc.filename] = {
              step: doc.step,
              step_label: doc.step_label,
              isProcessing: doc.step !== 'done'
            };
          });
        }

        // 현재 대화창에 있는 파일들만 필터링하여 상태 업데이트
        const currentFileNames = new Set(currentFilesRef.current.map(file => file.name));

        setEmbeddingStatuses(prevStatuses => {
          const newStatuses = { ...prevStatuses };

          // 현재 대화창에 있는 파일들의 상태만 업데이트
          Object.keys(statusMap).forEach(filename => {
            if (currentFileNames.has(filename)) {
              newStatuses[filename] = statusMap[filename];
            }
          });

          // 삭제된 파일들의 상태는 제거
          Object.keys(newStatuses).forEach(filename => {
            if (!currentFileNames.has(filename)) {
              delete newStatuses[filename];
            }
          });

          return newStatuses;
        });
      }
    } catch (error) {
      console.error('Embedding status fetch failed:', error);
    }
  }, []);

  // RAG 검색 범위 드롭다운 상태

  
  // 시스템 프롬프트 관련 상태
  const [systemPrompt, setSystemPrompt] = useState<string>('');
  const [showSystemPromptModal, setShowSystemPromptModal] = useState(false);
  const [isLoadingSystemPrompt, setIsLoadingSystemPrompt] = useState(false);
  const [systemPromptProvider, setSystemPromptProvider] = useState<string>('');
  const [isSystemPromptCustom, setIsSystemPromptCustom] = useState(false);

  // 건강검진 분석 관련 상태
  const [showHealthAnalysisModal, setShowHealthAnalysisModal] = useState(false);
  const [isHealthAnalysisLoading, setIsHealthAnalysisLoading] = useState(false);
  const [healthAnalysisResult, setHealthAnalysisResult] = useState<any>(null);
  const [patientNameInput, setPatientNameInput] = useState<string>('');

  // 파일 업로드 팝업 메뉴 관련 상태
  const [showUploadOptions, setShowUploadOptions] = useState(false);
  const [showExcelModal, setShowExcelModal] = useState(false);
  
  // GraphRAG 시각화 모달 상태

  // 검색 범위 라벨과 아이콘 반환 함수
  const getRagScopeInfo = (scope: 'personal' | 'shared' | 'all') => {
    switch (scope) {
      case 'personal':
        return {
          label: t('chat.personalDocuments', '개인문서'),
          icon: '🔒',
          description: '개인 문서만 검색'
        };
      case 'shared':
        return {
          label: t('chat.sharedDocuments', '공용문서'),
          icon: '🌐',
          description: '공용 문서만 검색'
        };
      case 'all':
        return {
          label: t('chat.globalSearch', '전체검색'),
          icon: '🔍',
          description: '개인 + 공용 문서 검색'
        };
      default:
        return {
          label: t('chat.personalDocuments', '개인문서'),
          icon: '🔒',
          description: '개인 문서만 검색'
        };
    }
  };

  const messagesEndRef = useRef<HTMLDivElement>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const textareaRef = useRef<HTMLTextAreaElement>(null);
  const uploadOptionsRef = useRef<HTMLDivElement>(null);
  const uploadOptionsBottomRef = useRef<HTMLDivElement>(null);

  // WebSocket 참조 관리 (채팅 중지 기능용)
  const wsConnectionRef = useRef<WebSocket | null>(null);
  const isStreamingRef = useRef<boolean>(false);
  const stopButtonDebounceRef = useRef<boolean>(false);

  // 스트리밍 중지 함수 (디바운싱 포함)
  const stopStreaming = useCallback(() => {
    // 빠른 중지 버튼 클릭 방지를 위한 디바운싱
    if (stopButtonDebounceRef.current) {
      return;
    }

    stopButtonDebounceRef.current = true;

    try {
      // 중지 버튼을 클릭하기 전에 WebSocket이 닫히는 엣지 케이스 처리
      if (wsConnectionRef.current) {
        if (wsConnectionRef.current.readyState === WebSocket.OPEN) {
          wsConnectionRef.current.close();
        }
        wsConnectionRef.current = null;
      }

      // 중지 시 상태 재설정 (WebSocket이 이미 닫혀있어도 상태는 정리)
      setIsLoading(false);
      setProgressMessages([]);
      isStreamingRef.current = false;
    } catch (error) {
      console.error('Error occurred while stopping streaming:', error);
      // 오류 발생 시에도 상태 정리 보장
      setIsLoading(false);
      setProgressMessages([]);
      isStreamingRef.current = false;
      wsConnectionRef.current = null;
    } finally {
      // 500ms 후 디바운스 해제
      setTimeout(() => {
        stopButtonDebounceRef.current = false;
      }, 500);
    }
  }, []);

  const [isTyping, setIsTyping] = useState(false);
  const lastTypingEndRef = useRef<number>(0);

  // 입력창 높이 초기화 함수
  const resetTextareaHeight = () => {
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
      textarea.style.height = 'auto';
      textarea.style.height = '48px';
    });
  };

  // 음성 처리 함수
  const processAudio = async (audioBlob: Blob, contentType: string) => {
    setIsProcessingAudio(true);

    const whisperApiUrl = "/api/stt";

    try {
      const res = await fetch(whisperApiUrl, {
        method: "POST",
        headers: {
          'Content-Type': contentType,
        },
        body: audioBlob,
      });

      if (res.ok) {
        const result = await res.json();
        const recognizedText = result.text || "";
        if (recognizedText.trim()) {
          setInputMessage(prev => prev + (prev ? ' ' : '') + recognizedText);
          setShowToast({ message: t('chat.voiceRecognitionCompleted', '음성 인식이 완료되었습니다.'), type: 'success' });
        } else {
          setShowToast({ message: t('chat.voiceNotRecognized', '음성이 인식되지 않았습니다.'), type: 'error' });
        }
      } else {
        const errorText = await res.text();
        console.error(`Server response error: ${res.status} ${res.statusText}. Content: ${errorText}`);
        setShowToast({ message: t('chat.voiceRecognitionError', '음성 인식 중 오류가 발생했습니다.'), type: 'error' });
      }
    } catch (error) {
      console.error("Whisper API fetch error:", error);
      setShowToast({ message: t('chat.voiceRecognitionServerError', '음성 인식 서버에 연결할 수 없습니다.'), type: 'error' });
    } finally {
      setIsProcessingAudio(false);
      setTimeout(() => setShowToast(null), 3000);
    }
  };

  // 음성 녹음 시작/중지 함수
  const toggleSpeechRecognition = async () => {
    if (isListening) {
      // 녹음 중지
      if (mediaRecorderRef.current) {
        mediaRecorderRef.current.stop();
        // 스트림의 모든 트랙을 중지하여 마이크 사용 표시를 끔
        mediaRecorderRef.current.stream.getTracks().forEach(track => track.stop());
      }
      setIsListening(false);
    } else {
      // 녹음 시작
      try {
        const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
        const mediaRecorder = new MediaRecorder(stream);

        audioChunksRef.current = [];
        mediaRecorder.ondataavailable = (event) => {
          if (event.data.size > 0) {
            audioChunksRef.current.push(event.data);
          }
        };

        mediaRecorder.onstop = async () => {
          const audioBlob = new Blob(audioChunksRef.current, { type: "audio/webm" });
          await processAudio(audioBlob, "audio/webm");
        };

        mediaRecorder.start();
        mediaRecorderRef.current = mediaRecorder;
        setIsListening(true);
        setShowToast({ message: t('chat.voiceRecordingStarted', '음성 녹음을 시작합니다. 말씀해 주세요.'), type: 'success' });
        setTimeout(() => setShowToast(null), 3000);
      } catch (error) {
        console.error("getUserMedia error:", error);
        setShowToast({ message: t('chat.microphoneAccessError', '마이크에 접근할 수 없습니다. 브라우저의 마이크 권한을 확인해주세요.'), type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
      }
    }
  };

  // 인증 확인
  useEffect(() => {
    if (isInitialized && !isAuthenticated) {
      router.push('/login');
    }

    // user 객체 디버그 출력
    if (user) {
      // console.log('=== 사용자 정보 디버그 ===');
      // console.log('user.id:', user.id, typeof user.id);
      // console.log('user.username:', user.username, typeof user.username);
      // console.log('user.email:', user.email, typeof user.email);
      // console.log('user.role:', user.role, typeof user.role);
      // console.log('전체 user 객체:', user);
      // console.log('========================');

      // options의 userId도 함께 출력
      // console.log('=== ChatOptions 디버그 ===');
      // console.log('options.userId:', options.userId, typeof options.userId);
      // console.log('========================');
    }
  }, [isInitialized, isAuthenticated, router, user, options]);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  // 출처는 이미 백엔드에서 정리되어 전달됨

  // 타이핑 효과 함수 (onDone 콜백 추가)
  const typeMessage = (messageId: string, fullContent: string, speed: number = 5, onDone?: () => void) => {
    setTypingMessageId(messageId);
    setDisplayedContent('');
    setIsTyping(true);
    let currentIndex = 0;
    const typeInterval = setInterval(() => {
      if (currentIndex < fullContent.length) {
        // 한 번에 여러 글자씩 표시하여 더 빠르게
        const charsToAdd = Math.min(3, fullContent.length - currentIndex);
        setDisplayedContent(fullContent.slice(0, currentIndex + charsToAdd));
        currentIndex += charsToAdd;
        setTimeout(scrollToBottom, 10);
      } else {
        clearInterval(typeInterval);
        setTypingMessageId(null);
        setDisplayedContent('');
        setIsTyping(false);
        lastTypingEndRef.current = Date.now();
        if (onDone) onDone();
      }
    }, speed);
    return () => clearInterval(typeInterval);
  };

  useEffect(() => {
    scrollToBottom();
  }, [currentMessages]);

  // 로딩 상태가 변경될 때도 스크롤
  useEffect(() => {
    if (isLoading) {
      // 로딩 메시지가 표시된 후 스크롤
      setTimeout(scrollToBottom, 100);
    }
  }, [isLoading]);

  // 메시지가 업데이트될 때 강제 스크롤 (긴 응답 처리)
  useEffect(() => {
    const timer = setTimeout(scrollToBottom, 200);
    return () => clearTimeout(timer);
  }, [currentMessages.length]);


  // uploadedFiles 변경 시 currentFilesRef 업데이트
  useEffect(() => {
    currentFilesRef.current = uploadedFiles;
  }, [uploadedFiles]);

  // 임베딩 상태 추적을 위한 폴링
  useEffect(() => {
    if (!isAuthenticated) return;

    // 초기 로드
    fetchEmbeddingStatuses();

    // 3초마다 상태 업데이트
    const interval = setInterval(fetchEmbeddingStatuses, 3000);

    return () => clearInterval(interval);
  }, [isAuthenticated, fetchEmbeddingStatuses]);

  // URL의 세션 ID가 변경될 때마다 해당 세션의 메시지를 불러옵니다.
  useEffect(() => {
    // console.log('🔍 [DEBUG] useEffect currentSessionId 변경됨:', {
    //   currentSessionId,
    //   isInitialized,
    //   isAuthenticated,
    //   hasCurrentMessages: currentMessages.length > 0
    // });
    
    if (isInitialized && isAuthenticated) {
      if (currentSessionId) {
        // 새로 생성된 세션인 경우 selectSession 건너뛰기 (thinking 유지를 위해)
        if (isNewSessionJustCreatedRef.current) {
          // console.log('🔍 [DEBUG] 새로 생성된 세션이므로 selectSession 건너뛰기 - thinking 유지를 위해');
          return;
        }
        
        // console.log('🔍 [DEBUG] selectSession 호출 예정:', currentSessionId);
        // 기존 세션을 선택하는 로직
        selectSession(currentSessionId);
      } else {
        console.log('🔍 [DEBUG] Initialize messages for new conversation state');
        // 새 대화 상태
        setCurrentMessages([]);
      }
      // 세션 목록은 항상 불러옵니다.
      // loadSessions().then(setSessions);
    }
  }, [currentSessionId, isInitialized, isAuthenticated]);

  // 컴포넌트 마운트 시 초기 데이터 로드
  useEffect(() => {
    loadProviders();
    initializeDefaultConfig();
  }, []);

  // 인증 상태가 변경될 때 RAG 파일 로드 (세션은 위에서 처리)
  useEffect(() => {
    if (isAuthenticated) {
      loadRagFiles();
      loadProjects(); // 프로젝트 목록도 함께 로드
    }
  }, [isAuthenticated]);

  // 백그라운드 작업 완료 시 RAG 파일 목록 새로고침
  useEffect(() => {
    if (isAuthenticated && refreshTrigger > 0) {
      loadRagFiles();
    }
  }, [isAuthenticated, refreshTrigger]);

  // 선택된 프로젝트가 변경될 때 RAG 파일 목록 새로고침
  useEffect(() => {
    if (isAuthenticated) {
      loadRagFiles();
    }
  }, [selectedProject, isAuthenticated]);

  // RAG 파일 상태 변화를 실시간으로 반영하기 위한 주기적 새로 고침
  useEffect(() => {
    if (!isAuthenticated) return;

    const interval = setInterval(() => {
      // pending 또는 processing 상태의 파일이 있는지 확인
      const hasProcessingFiles = ragFiles.some(file =>
        file.status === 'pending' || file.status === 'processing' || file.status === 'uploading'
      );

      if (hasProcessingFiles) {
        console.log('[RAG Polling] 처리 중인 파일이 있어 목록을 새로 고침합니다.');
        loadRagFiles();
      }
    }, 3000); // 3초마다 체크

    return () => clearInterval(interval);
  }, [isAuthenticated, ragFiles]);

  // user가 로드되면 options 업데이트 및 프로필 이미지 로드
  useEffect(() => {
    if (user?.id) {
      setOptions(prev => ({
        ...prev,
        userId: Number(user.id)
      }));
      // 프로필 이미지 불러오기
      fetchUserProfileImage();
    }
  }, [user]);



  // ESC 키로 스트리밍 중지
  useEffect(() => {
    const handleKeyDown = (event: KeyboardEvent) => {
      // ESC 키가 눌리고 현재 스트리밍 중일 때만 중지
      if (event.key === 'Escape' && isStreamingRef.current && isLoading) {
        event.preventDefault();
        stopStreaming();
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [isLoading, stopStreaming]);

  // 외부 클릭 시 팝업 닫기
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (showUploadOptions) {
        const target = event.target as Node;
        const isInsideTopPopup = uploadOptionsRef.current && uploadOptionsRef.current.contains(target);
        const isInsideBottomPopup = uploadOptionsBottomRef.current && uploadOptionsBottomRef.current.contains(target);

        if (!isInsideTopPopup && !isInsideBottomPopup) {
          setShowUploadOptions(false);
        }
      }
    };

    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, [showUploadOptions]);

  // 컴포넌트 언마운트 시 정리
  useEffect(() => {
    return () => {
      // 컴포넌트 언마운트 시에만 현재 업로드 중인 파일들의 URL 정리
      // 메시지에 저장된 URL은 정리하지 않음
      uploadedFileUrls.forEach(url => {
        URL.revokeObjectURL(url);
      });

      // WebSocket 연결 정리
      if (wsConnectionRef.current) {
        try {
          if (wsConnectionRef.current.readyState === WebSocket.OPEN) {
            wsConnectionRef.current.close();
          }
        } catch (error) {
          console.error('WebSocket cleanup error during component unmount:', error);
        } finally {
          wsConnectionRef.current = null;
        }
      }

      // 음성 녹음 정리
      if (mediaRecorderRef.current) {
        try {
          if (mediaRecorderRef.current.state === 'recording') {
            mediaRecorderRef.current.stop();
          }
          mediaRecorderRef.current.stream?.getTracks().forEach(track => track.stop());
        } catch (error) {
          console.error('Voice recording cleanup error during component unmount:', error);
        }
      }
    };
  }, []); // 의존성 배열을 비워서 언마운트 시에만 실행

  // 시스템 프롬프트 불러오기 함수
  const fetchSystemPrompt = async () => {
    try {
      setIsLoadingSystemPrompt(true);
      // 명시적으로 chat 모드임을 지정하여 code 모드용 프롬프트를 사용하지 않도록 함
      const authHeaders = getAuthHeaders();
      const apiServerUrl = getApiServerUrl();
      console.log('[DEBUG] fetchSystemPrompt - authHeaders:', authHeaders);
      console.log('[DEBUG] fetchSystemPrompt - localStorage userToken:', localStorage.getItem('userToken'));
      console.log('[DEBUG] fetchSystemPrompt - localStorage apiKey:', localStorage.getItem('apiKey'));
      const response = await fetch(`${apiServerUrl}/user/system-prompt?mode=chat`, {
        method: 'GET',
        headers: {
          ...authHeaders
        },
        credentials: 'include'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data) {
          setSystemPrompt(data.data.content);
          setSystemPromptProvider(data.data.provider || '');
          setIsSystemPromptCustom(data.data.isCustom || false);
        }
      } else {
        console.error('시스템 프롬프트 불러오기 실패:', response.status);
      }
    } catch (error) {
      console.error('시스템 프롬프트 불러오기 중 오류:', error);
    } finally {
      setIsLoadingSystemPrompt(false);
    }
  };

  // 시스템 프롬프트 저장 함수
  const saveSystemPrompt = async () => {
    try {
      setIsLoadingSystemPrompt(true);
      const authHeaders = getAuthHeaders();
      const apiServerUrl = getApiServerUrl();
      const response = await fetch(`${apiServerUrl}/user/system-prompt`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          ...authHeaders
        },
        credentials: 'include',
        body: JSON.stringify({ content: systemPrompt })
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          setShowToast({
            message: '시스템 프롬프트가 저장되었습니다.',
            type: 'success'
          });
          setTimeout(() => setShowToast(null), 3000);
        } else {
          throw new Error(data.error?.message || '저장 실패');
        }
      } else {
        throw new Error('서버 응답 오류');
      }
    } catch (error) {
      console.error('시스템 프롬프트 저장 실패:', error);
      setShowToast({
        message: '시스템 프롬프트 저장에 실패했습니다.',
        type: 'error'
      });
      setTimeout(() => setShowToast(null), 3000);
    } finally {
      setIsLoadingSystemPrompt(false);
    }
  };

  // 프로필 이미지 불러오기 함수
  const fetchUserProfileImage = async () => {
    try {
      const userId = localStorage.getItem('userId');
      if (!userId) return;

      const response = await fetch(`/api/profile/image?userId=${userId}`, {
        method: 'GET',
        credentials: 'include'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data.imageUrl) {
          setUserProfileImage(data.data.imageUrl);
        }
      }
    } catch (error) {
      console.error('프로필 이미지 불러오기 실패:', error);
    }
  };

  // 프로바이더 정보 로드
  const loadProviders = async () => {
    try {
      setIsLoadingProviders(true);

      // 먼저 사용 가능한 프로바이더 목록을 가져옴 (인증 불필요)
      const availableResponse = await externalApiClient.getAvailableProviders();

      if (availableResponse.success && availableResponse.data) {
        setAvailableProviders(availableResponse.data);

        // 각 프로바이더의 모델 목록 로드
        const providersData: any = {};
        for (const provider of availableResponse.data) {
          try {
            const modelsResponse = await externalApiClient.getProviderModels(provider.key);
            if (modelsResponse.success && modelsResponse.data) {
              providersData[provider.key] = {
                name: provider.name,
                requiresApiKey: provider.requiresApiKey,
                apiKeyConfigured: provider.apiKeyConfigured !== false, // 백엔드에서 제공된 값 사용
                isDynamic: provider.isDynamic,
                isAvailable: true,
                models: modelsResponse.data
              };
            }
          } catch (modelError) {
            console.warn(`${provider.key} 모델 목록 로드 실패:`, modelError);
            // 모델 목록 로드 실패해도 프로바이더는 표시
            providersData[provider.key] = {
              name: provider.name,
              requiresApiKey: provider.requiresApiKey,
              apiKeyConfigured: provider.apiKeyConfigured !== false, // 백엔드에서 제공된 값 사용
              isDynamic: provider.isDynamic,
              isAvailable: true,
              models: []
            };
          }
        }

        setProviders(providersData);
      }
    } catch (error: any) {
      console.error('프로바이더 정보 로드 실패:', error);

      // 오류 발생 시 기본 프로바이더 정보 설정 (Ollama + 기본 모델)
      const defaultModel = {
        id: 'hamonize:latest',
        name: 'hamonize:latest',
        size: 0
      };

      const defaultProviders = {
        ollama: {
          name: 'Ollama',
          isAvailable: true,
          requiresApiKey: false,
          apiKeyConfigured: true, // API 키가 필요없으므로 true
          isDynamic: true,
          models: [defaultModel] // 기본 모델 추가
        }
      };
      setProviders(defaultProviders);
      setAvailableProviders([
        { key: 'ollama', name: 'Ollama', requiresApiKey: false, apiKeyConfigured: true, isDynamic: true, models: [defaultModel] }
      ]);

      // 기본 모델로 설정 (옵션이 ollama인 경우)
      if (options.provider === 'ollama' || !options.model) {
        setOptions(prev => ({
          ...prev,
          provider: 'ollama',
          model: 'hamonize:latest'
        }));
        localStorage.setItem('airun-selected-provider', 'ollama');
        localStorage.setItem('airun-selected-model', 'hamonize:latest');
      }
    } finally {
      setIsLoadingProviders(false);
    }
  };

  // 기본 프로바이더/모델 설정 (사용자별 로컬 설정)
  const initializeDefaultConfig = () => {
    // 로컬 스토리지에서 사용자 설정 불러오기 (기본값: ollama, hamonize:latest)
    const savedProvider = localStorage.getItem('airun-selected-provider') || 'ollama';
    const savedModel = localStorage.getItem('airun-selected-model') || 'hamonize:latest';

    setOptions(prev => ({
      ...prev,
      provider: savedProvider,
      model: savedModel
    }));

    // 기본값이 사용된 경우 로컬 스토리지에 저장
    if (!localStorage.getItem('airun-selected-provider')) {
      localStorage.setItem('airun-selected-provider', 'ollama');
    }
    if (!localStorage.getItem('airun-selected-model')) {
      localStorage.setItem('airun-selected-model', 'hamonize:latest');
    }
  };

  // 프로바이더 변경 시 모델 목록 업데이트
  const handleProviderChange = async (newProvider: string) => {
    // 로컬 스토리지에 선택한 프로바이더 저장
    localStorage.setItem('airun-selected-provider', newProvider);

    setOptions(prev => ({ ...prev, provider: newProvider }));

    // 해당 프로바이더의 모델 목록이 비어있거나 없으면 실시간으로 다시 로드
    const providerData = providers[newProvider];
    if (!providerData || !providerData.models || providerData.models.length === 0) {
      console.log(`프로바이더 ${newProvider}의 모델 목록이 없어서 실시간 로드를 시도합니다.`);

      try {
        // 해당 프로바이더의 모델 목록을 실시간으로 로드
        const modelsResponse = await externalApiClient.getProviderModels(newProvider);
        if (modelsResponse.success && modelsResponse.data && modelsResponse.data.length > 0) {
          // 프로바이더 정보 업데이트
          setProviders((prev: any) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: modelsResponse.data
            }
          }));

          // 첫 번째 모델로 자동 설정
          const newModel = modelsResponse.data[0].id;
          localStorage.setItem('airun-selected-model', newModel);

          setOptions(prev => ({
            ...prev,
            provider: newProvider,
            model: newModel
          }));

          console.log(`프로바이더 ${newProvider} 모델 실시간 로드 성공:`, modelsResponse.data.length, '개');
          return;
        } else {
          console.warn(`프로바이더 ${newProvider}의 모델 목록 실시간 로드 실패:`, modelsResponse);

          // Ollama인 경우 기본 모델로 fallback
          if (newProvider === 'ollama') {
            const defaultModel = {
              id: 'hamonize:latest',
              name: 'hamonize:latest',
              size: 0
            };

            setProviders((prev: any) => ({
              ...prev,
              [newProvider]: {
                ...prev[newProvider],
                models: [defaultModel]
              }
            }));

            setOptions(prev => ({
              ...prev,
              provider: newProvider,
              model: 'hamonize:latest'
            }));

            localStorage.setItem('airun-selected-model', 'hamonize:latest');
            console.log(`프로바이더 ${newProvider} 실시간 로드 실패, 기본 모델로 fallback 완료`);
            return;
          }
        }
      } catch (error) {
        console.error(`프로바이더 ${newProvider}의 모델 목록 실시간 로드 중 오류:`, error);

        // Ollama인 경우 기본 모델로 fallback
        if (newProvider === 'ollama') {
          const defaultModel = {
            id: 'hamonize:latest',
            name: 'hamonize:latest',
            size: 0
          };

          setProviders((prev: any) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: [defaultModel]
            }
          }));

          setOptions(prev => ({
            ...prev,
            provider: newProvider,
            model: 'hamonize:latest'
          }));

          localStorage.setItem('airun-selected-model', 'hamonize:latest');
          console.log(`프로바이더 ${newProvider} 실시간 로드 오류, 기본 모델로 fallback 완료`);
          return;
        }
      }
    }

    // 기존 로직: 프로바이더에 모델이 있는 경우
    if (providerData && providerData.models && providerData.models.length > 0) {
      const newModel = providerData.models[0].id;

      // 로컬 스토리지에 선택한 모델 저장
      localStorage.setItem('airun-selected-model', newModel);

      setOptions(prev => ({
        ...prev,
        provider: newProvider,
        model: newModel
      }));

      console.log(`프로바이더 변경: ${newProvider}, 모델: ${newModel}`);
    } else {
      console.warn(`프로바이더 ${newProvider}의 모델 목록이 없습니다:`, providerData);

      // Ollama 프로바이더인 경우 기본 모델 설정
      if (newProvider === 'ollama') {
        const defaultModel = 'hamonize:latest';

        // 프로바이더에 기본 모델 추가
        setProviders((prev: any) => ({
          ...prev,
          [newProvider]: {
            ...prev[newProvider],
            models: [{
              id: defaultModel,
              name: defaultModel,
              size: 0
            }]
          }
        }));

        setOptions(prev => ({
          ...prev,
          provider: newProvider,
          model: defaultModel
        }));

        localStorage.setItem('airun-selected-model', defaultModel);
        console.log(`프로바이더 ${newProvider}에 기본 모델 ${defaultModel} 설정 완료`);
      } else {
        // 다른 프로바이더인 경우 빈 모델로 설정
        setOptions(prev => ({
          ...prev,
          provider: newProvider,
          model: '' // 모델을 빈 문자열로 설정
        }));
      }
    }
  };

  // 새 대화 버튼 클릭 시 바로 새 대화창을 보여줍니다.
  const handleNewChatClick = async () => {
    if (isTyping) return;

    // URL에서 세션 파라미터 제거
    router.push('/chat');

    // 모든 상태 초기화
    setCurrentSessionId(null); // 세션 ID 초기화 추가
    setCurrentMessages([]);
    setUploadedFiles([]);
    setInputMessage('');
    setError(null);
    setProgressMessages([]); // 새 대화 시작 시 진행 상황 메시지 초기화
    setIsNewChatMode(true);

    // 새 대화 시작 시 세션 목록 새로고침
    try {
      // 무한 스크롤 상태 초기화
      setSessionPage(1);
      setHasMoreSessions(true);
      resetInfiniteScroll();

      const latestSessions = await loadSessions(1, 20);
      setSessions(latestSessions);
      setHasMoreSessions(latestSessions.length >= 20);
      // console.log('✅ 새 대화 시작 - 세션 목록 새로고침 완료');
    } catch (error) {
      console.error('❌ 새 대화 시작 시 세션 목록 새로고침 실패:', error);
    }

    // console.log('새 대화 시작 - 모든 상태 초기화 완료');
  };

  // 세션 목록을 불러오고 최신 세션 배열만 반환 (setSessions는 호출하지 않음)
  const loadSessions = async (page: number = 1, limit: number = 20): Promise<ChatSession[]> => {
    if (isTyping) return [];
    try {
      setIsLoadingSessions(true);
      setError(null);
      const response = await internalApiClient.getSessions('chat', page, limit);
      if (!response || !response.success || !response.data) {
        console.error('API 응답이 유효하지 않음:', response);
        throw new Error('API 응답이 유효하지 않습니다.');
      }
      const sessionsArray: ApiSession[] = Object.values(response.data as Record<string, ApiSession>);
      const sessionList: ChatSession[] = sessionsArray
        .filter((session: ApiSession) => {
          if (!session || typeof session !== 'object') return false;
          if (!Array.isArray(session.history)) return false;
          // 의도된 동작: assistant 메시지가 있는 세션만 표시
          return session.history.some(msg => msg.role === 'assistant' && msg.content && typeof msg.content === 'string' && msg.content.trim().length > 0);
        })
        .map((session: ApiSession): ChatSession => {
          // 안전하게 content를 문자열로 변환하는 함수
          const safeStringify = (content: any): string => {
            if (typeof content === 'string') {
              // 이미지 데이터 제거
              const imagePattern = /\[IMAGE:[^\]]+\]data:image\/[^;]+;base64,[A-Za-z0-9+/=]+/g;
              let cleanContent = content.replace(imagePattern, '').trim();
              
              // 마크다운 특수문자 제거 (세션 제목용)
              cleanContent = cleanContent.replace(/\*\*([^*]+)\*\*/g, '$1'); // **bold**
              cleanContent = cleanContent.replace(/\*([^*]+)\*/g, '$1'); // *italic*
              cleanContent = cleanContent.replace(/__([^_]+)__/g, '$1'); // __underline__
              cleanContent = cleanContent.replace(/~~([^~]+)~~/g, '$1'); // ~~strikethrough~~
              cleanContent = cleanContent.replace(/```[^`]*```/g, ''); // code blocks
              cleanContent = cleanContent.replace(/`([^`]+)`/g, '$1'); // inline code
              cleanContent = cleanContent.replace(/^#+ /gm, ''); // headers
              cleanContent = cleanContent.replace(/^[*\-+] /gm, ''); // list items
              cleanContent = cleanContent.replace(/\[([^\]]+)\]\([^)]+\)/g, '$1'); // links
              cleanContent = cleanContent.replace(/!\[([^\]]*)\]\([^)]+\)/g, ''); // images
              
              return cleanContent.trim();
            }
            if (content === null || content === undefined) return '';
            try {
              return typeof content === 'object' ? JSON.stringify(content) : String(content);
            } catch {
              return String(content);
            }
          };

          // 사용자 메시지와 어시스턴트 메시지 필터링
          const userMessages = (session.history ?? []).filter(msg => {
            return msg && msg.role === 'user' && msg.content;
          });

          const assistantMessages = (session.history ?? []).filter(msg => {
            return msg && msg.role === 'assistant' && msg.content;
          });

          // 기본 오류 메시지 필터링 함수
          const isDefaultErrorMessage = (content: string): boolean => {
            const defaultErrorMessages = [
              t('chat.sorryCannotGenerate', '죄송합니다. 응답을 생성할 수 없었습니다. 다시 시도해주세요.'),
              t('chat.responseGenerationError', '응답 생성 중 오류가 발생했습니다. 다시 시도해주세요.'),
              t('chat.sorryResponseProblem', '죄송합니다. 응답을 생성하는 중 문제가 발생했습니다.'),
              t('chat.sorryResponseErrorRetry', '죄송합니다. 응답 생성 중 오류가 발생했습니다. 잠시 후 다시 시도해주세요.')
            ];
            return defaultErrorMessages.some(errorMsg => content.includes(errorMsg));
          };

          // 제목은 첫 번째 assistant 응답을 기준으로 생성 (일관성 있는 제목 표시)
          const title = assistantMessages.length > 0
            ? (() => {
                const content = safeStringify(assistantMessages[0].content);
                // 기본 오류 메시지가 아닌 경우에만 사용
                if (!isDefaultErrorMessage(content)) {
                  return content.slice(0, 30) + (content.length > 30 ? '...' : '');
                }
                // 오류 메시지인 경우 사용자 질문을 대신 사용
                if (userMessages.length > 0) {
                  const userContent = safeStringify(userMessages[0].content);
                  return userContent.slice(0, 30) + (userContent.length > 30 ? '...' : '');
                }
                return t('chat.newConversation', '새 대화');
              })()
            : userMessages.length > 0
              ? (() => {
                  const content = safeStringify(userMessages[0].content);
                  return content.slice(0, 30) + (content.length > 30 ? '...' : '');
                })()
              : t('chat.newConversation', '새 대화');

          const lastMessage = assistantMessages.length > 0
            ? (() => {
              // 기본 오류 메시지가 아닌 마지막 assistant 메시지 찾기
              const validAssistantMessage = [...assistantMessages].reverse().find(msg => {
                const content = safeStringify(msg.content);
                return !isDefaultErrorMessage(content);
              });
              
              if (validAssistantMessage) {
                const content = safeStringify(validAssistantMessage.content);
                return content.slice(0, 50) + (content.length > 50 ? '...' : '');
              } else {
                // 모든 assistant 메시지가 기본 오류 메시지인 경우
                const lastContent = safeStringify(assistantMessages[assistantMessages.length - 1].content);
                return lastContent.slice(0, 50) + (lastContent.length > 50 ? '...' : '');
              }
            })()
            : t('chat.newConversation', '새 대화');

          return {
            id: session.id,
            title,
            lastMessage,
            updatedAt: session.updated_at || new Date().toISOString(),
            createdAt: new Date().toISOString(),
            messageCount: session.history?.length ?? 0,
            provider: session.provider,
            model: session.model,
            history: session.history ?? []
          };
        });
      return sessionList;
    } catch (error) {
      console.error('세션 목록 로드 실패:', error);
      setError(t('chat.sessionLoadFailed', '세션 목록을 불러오는데 실패했습니다.'));
      setTimeout(() => setError(null), 3000);
      return [];
    } finally {
      setIsLoadingSessions(false);
    }
  };

  // 무한 스크롤을 위한 추가 세션 로드 함수
  const loadMoreSessions = useCallback(async (): Promise<boolean> => {
    if (!hasMoreSessions || isLoadingSessions) return false;

    try {
      const newSessions = await loadSessions(sessionPage + 1, 20);

      if (newSessions.length === 0) {
        setHasMoreSessions(false);
        return false;
      }

      // 중복 세션 제거
      let hasUniqueNewSessions = false;
      setSessions(prev => {
        const existingIds = new Set(prev.map(session => session.id));
        const uniqueNewSessions = newSessions.filter(session => !existingIds.has(session.id));

        // 중복 제거 후에도 새로운 세션이 있는지 확인
        if (uniqueNewSessions.length === 0) {
          console.log('모든 새 세션이 중복이어서 추가하지 않음');
          hasUniqueNewSessions = false;
          return prev;
        }

        hasUniqueNewSessions = true;
        return [...prev, ...uniqueNewSessions];
      });

      // 고유한 새 세션이 없다면 무한 스크롤 중단
      if (!hasUniqueNewSessions) {
        setHasMoreSessions(false);
        return false;
      }

      setSessionPage(prev => prev + 1);
      return true;
    } catch (error) {
      console.error('추가 세션 로드 실패:', error);
      return false;
    }
  }, [sessionPage, hasMoreSessions, isLoadingSessions, loadSessions]);

  // 무한 스크롤 훅 사용
  const {
    ref: infiniteScrollRef,
    isLoading: isInfiniteLoading,
    hasMore: hasMoreInfinite,
    reset: resetInfiniteScroll,
  } = useInfiniteScroll(loadMoreSessions, {
    threshold: 0.8,
    rootMargin: '50px',
  });

  // 인증 상태 변경 시에만 세션 목록 로드 (초기화)
  useEffect(() => {
    if (isAuthenticated) {
      // 무한 스크롤 상태 초기화
      setSessionPage(1);
      setHasMoreSessions(true);
      setSessions([]);
      resetInfiniteScroll();

      // 첫 페이지 로드
      loadSessions(1, 20).then((initialSessions) => {
        setSessions(initialSessions);
        setHasMoreSessions(initialSessions.length >= 20);
      });
    }
  }, [isAuthenticated, resetInfiniteScroll]);

  const selectSession = async (sessionId: string) => {
    // console.log('🔍 [DEBUG] selectSession 호출됨:', {
    //   sessionId,
    //   currentSessionId,
    //   isTyping,
    //   currentMessagesLength: currentMessages.length
    // });
    
    if (isTyping || !sessionId) return;
    // if (sessionId === currentSessionId) return;

    // URL에 세션 파라미터 반영
    const url = new URL(window.location.href);
    url.searchParams.set('session', sessionId);
    window.history.replaceState({}, '', url.toString());

    try {
      setIsNewChatMode(false);
      setIsLoading(true);
      setCurrentSessionId(sessionId);
      setProgressMessages([]); // 다른 세션 선택 시 진행 상황 메시지 초기화

      const response = await externalApiClient.getSession(sessionId);

      console.log('🔍 [selectSession] API 응답:', {
        sessionId,
        success: response.success,
        historyLength: response.data?.history?.length,
        firstAssistantMessage: response.data?.history?.find((m: any) => m.role === 'assistant'),
        rawResponse: response.data
      });

      if (response.success && response.data) {
        const sessionData = response.data;
        if (Array.isArray(sessionData.history)) {
          // user와 assistant role만 필터링 (tool 메시지는 백엔드에서만 사용, UI에서는 숨김)
          const filteredHistory = sessionData.history.filter((message: any) => {
            // 기본 역할 필터링 - tool 메시지는 제외
            // 만약 Message 타입에 'tool'이 포함되지 않는다면 아래 조건만 사용
            if (message.role !== 'user' && message.role !== 'assistant') {
              return false;
            }

            // internal 플래그가 있는 메시지 제외 (도구 호출 응답 등)
            if ((message as any).internal === true) {
              // console.log('내부 메시지 제외 (도구 호출 응답):', message.content?.slice(0, 100) + '...');
              return false;
            }

            // tool_calls가 있는 assistant 메시지 제외 (도구 호출 메시지)
            if (message.role === 'assistant' && (message as any).tool_calls) {
              return false;
            }

            // 시스템 프롬프트 필터링 제거 - 히스토리 누락 방지를 위해 모든 사용자/어시스턴트 메시지 유지
            // originalContent가 있으면 UI에서는 그것을 표시하고, 백엔드에는 전체 content 전송
            
            return true;
          });

          // 히스토리에서 downloadInfo 직접 사용 (HTML 주석 방식 제거)
          const messagesWithReferences = filteredHistory.map((message: any) => {
            if (message.role === 'assistant') {
              // 디버깅: 메시지 구조 확인
              console.log('📋 [selectSession] Assistant 메시지 구조:', {
                messageId: message.id,
                hasDownloadInfo: 'downloadInfo' in message,
                downloadInfoValue: message.downloadInfo,
                messageKeys: Object.keys(message),
                fullMessage: message
              });
              
              // downloadInfo가 메시지에 직접 저장되어 있으면 사용
              if (message.downloadInfo && Array.isArray(message.downloadInfo)) {
                // console.log('✅ [selectSession] downloadInfo 발견 및 추가:', {
                //   messageId: message.id,
                //   count: message.downloadInfo.length
                // });
                return {
                  ...message,
                  downloadInfo: message.downloadInfo
                };
              }
              
              // 백워드 호환성: HTML 주석 방식도 지원
              if (message.content) {
                const content = typeof message.content === 'string' ? message.content : String(message.content);
                const referenceMatch = content.match(/<!-- REFERENCE_INFO:(.+?) -->/);
                if (referenceMatch) {
                  try {
                    const referenceInfo = JSON.parse(referenceMatch[1]);
                    return {
                      ...message,
                      content: content.replace(/\n\n<!-- REFERENCE_INFO:.+? -->/, ''),
                      downloadInfo: referenceInfo.downloadInfo || []
                    };
                  } catch (error) {
                    console.error('참조 정보 파싱 실패:', error);
                  }
                }
              }
            }
            return message;
          });

          // 참조 정보가 복원된 메시지들을 설정
          // console.log('📊 [selectSession] 최종 메시지 설정:', {
          //   totalMessages: messagesWithReferences.length,
          //   messagesWithDownloadInfo: messagesWithReferences.filter((m: any) => m.downloadInfo && m.downloadInfo.length > 0).length,
          //   downloadInfoDetails: messagesWithReferences
          //     .filter((m: any) => m.downloadInfo && m.downloadInfo.length > 0)
          //     .map((m: any) => ({
          //       messageId: m.id,
          //       role: m.role,
          //       downloadInfoCount: m.downloadInfo.length
          //     }))
          // });
          setCurrentMessages(messagesWithReferences);
        } else {
          setCurrentMessages([]);
        }
      } else {
        setError(t('chat.sessionSelectFailed', '세션을 불러오는데 실패했습니다.'));
        setCurrentMessages([]);
      }
    } catch (error) {
      setError(t('chat.sessionSelectFailed', '세션을 불러오는데 실패했습니다.'));
      setCurrentMessages([]);
    } finally {
      setIsLoading(false);
    }
  };



  const createNewSession = async (): Promise<string | null> => {
    // 이미 생성 중이면 중복 실행 방지
    if (isLoading) return null;

    try {
      setIsLoading(true);

      const sessionData = {
        provider: options.provider,
        model: options.model,
        parameters: {
          temperature: options.temperature,
          max_tokens: options.maxTokens,
        },
        history: [],
        type: 'chat',
        title: t('chat.newConversation', '새 대화'),
        userId: user?.id ? Number(user.id) : 0, // 숫자 ID
        username: user?.username, // 문자열 username
      };

      // 클라이언트에서 ID를 만들지 않고 서버에 생성을 요청합니다.
      const response = await externalApiClient.createSession(sessionData);

      // 서버로부터 받은 ID가 유효한지 확인합니다.
      if (response && response.id) {
        const newSessionId = response.id;

        const newSession: ChatSession = {
          id: newSessionId,
          title: t('chat.newConversation', '새 대화'),
          lastMessage: t('chat.newConversation', '새 대화'),
          updatedAt: new Date().toISOString(),
          createdAt: new Date().toISOString(),
          messageCount: 0,
          provider: options.provider,
          model: options.model,
          history: [],
        };

        // 새로운 세션으로 상태를 업데이트합니다.
        setCurrentMessages([]);
        setSessions(prev => [newSession, ...prev]);

        // URL도 새로운 세션 ID로 업데이트합니다.
        const url = new URL(window.location.href);
        url.searchParams.set('session', newSessionId);
        window.history.replaceState({}, '', url.toString());

        return newSessionId;
      } else {
        // 서버에서 ID를 받지 못한 경우 에러를 발생시킵니다.
        throw new Error('API 서버에서 세션 ID를 받지 못했습니다.');
      }
    } catch (error) {
      console.error('세션 생성 실패:', error);
      setError(t('chat.newChatFailed', '새 대화를 시작하지 못했습니다. 잠시 후 다시 시도해주세요.'));
      // 실패 시 null을 반환하여 후속 작업을 막습니다.
      return null;
    } finally {
      setIsLoading(false);
    }
  };

  // 파일 업로드 처리
  const handleFileSelect = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files || files.length === 0) return;

    // 이미 업로드 중이면 중단
    if (isUploading) {
      // console.log('이미 파일 업로드 중입니다.');
      return;
    }

    // dataset.mode 확인 (fast, rich, 또는 기본값)
    const uploadMode = event.target.dataset.mode || 'default';
    console.log('파일 업로드 모드:', uploadMode);

    const selectedFiles = Array.from(files);
    const maxDocuments = 10; // 문서 최대 개수 제한

    // 파일 개수 및 타입 검증
    const imageFiles = selectedFiles.filter(isImageFile);
    const documentFiles = selectedFiles.filter(file => !isImageFile(file));

    // 이미지 선택 제한 (1개만)
    if (imageFiles.length > 1) {
      setError('이미지는 한 번에 1개만 선택할 수 있습니다.');
      return;
    }

    // 기존 업로드된 파일과 합쳐서 총 개수 확인
    const totalFilesAfterUpload = uploadedFiles.length + selectedFiles.length;
    if (totalFilesAfterUpload > maxDocuments) {
      setError(t('chat.fileUploadError', '파일을 선택하셨습니다. 최대 {max}개까지만 업로드 가능합니다.').replace('{count}', selectedFiles.length.toString()).replace('{max}', maxDocuments.toString()));
      return;
    }

    // 이미지가 이미 있는 경우 추가 이미지 업로드 방지
    const existingImages = uploadedFiles.filter(isImageFile);
    if (existingImages.length > 0 && imageFiles.length > 0) {
      setError(t('chat.imageUploadLimit', '이미지는 한 번에 1개만 업로드할 수 있습니다.'));
      return;
    }

    // 에러 메시지 초기화
    setError('');

    const processedFiles: File[] = [];

    // 파일들을 순차적으로 처리 (이미지는 압축, 일반 파일은 그대로)
    for (const file of selectedFiles) {
      if (isImageFile(file)) {
        try {
          console.log(`이미지 압축 시작: ${file.name}`);
          const compressedFile = await compressImage(file);
          processedFiles.push(compressedFile);
        } catch (error) {
          console.error(`이미지 압축 실패 (${file.name}):`, error);
          // 압축 실패 시 원본 파일 사용
          processedFiles.push(file);
        }
      } else {
        processedFiles.push(file);
      }
    }

    setUploadedFiles(prev => [...prev, ...processedFiles]);

    // 이미지 파일의 경우 썸네일 URL 생성 (압축된 파일 기준)
    const newUrls = new Map<string, string>();
    processedFiles.forEach(file => {
      if (isImageFile(file)) {
        const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
        const url = URL.createObjectURL(file);
        newUrls.set(fileKey, url);
      }
    });

    if (newUrls.size > 0) {
      setUploadedFileUrls(prev => new Map([...prev, ...newUrls]));
    }

    // 이미지가 아닌 파일들은 모드에 따라 업로드
    for (const file of processedFiles) {
      if (!file.type.startsWith('image/')) {
        if (uploadMode === 'fast') {
          // RAG_MODE fast로 문서 업로드
          await uploadDocumentWithFastMode(file);
        } else if (uploadMode === 'rich') {
          // RAG_MODE rich로 문서 업로드
          await uploadDocumentWithRichMode(file);
        } else {
          // 기본 RAG 업로드
          await uploadFileToRag(file);
        }
      }
    }

    // 파일 입력 초기화
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
      // dataset.mode도 초기화
      delete fileInputRef.current.dataset.mode;
    }
  };

  const uploadFileToRag = async (file: File) => {
    const fileKey = `${file.name}-${file.size}-${file.lastModified}`;

    // 이미 업로드 중인 파일이면 중단
    if (uploadingFiles.has(fileKey)) {
      console.log(`파일 ${file.name}은 이미 업로드 중입니다.`);
      return;
    }

    if (!user?.username) {
      setError('로그인이 필요합니다.');
      return;
    }

    try {
      setUploadingFiles(prev => new Set(prev).add(fileKey));

      // 공통 uploadFile 함수 사용
      await externalApiClient.uploadFile(
        file,
        user.username,
        'chat', // 타겟 폴더
        (progress, status) => {
          // 파일 업로드 진행상태 로깅
          console.log(`파일 ${file.name}: ${progress}% (${status})`);
        },
        undefined, // ragModeOrMultiple
        selectedProject?.name // 선택된 프로젝트명 전달
      );

      // 즉시 백그라운드 작업 상태 새로고침 트리거
      setRefreshTrigger(prev => prev + 1);
      
      console.log(`파일 ${file.name} 업로드 및 임베딩 완료`);
      return 'completed';
      
    } catch (error) {
      console.error(`파일 ${file.name} 업로드 실패:`, error);
      setError(`${t('chat.fileUploadFailed', '파일 업로드에 실패했습니다.')}: ${error instanceof Error ? error.message : 'Unknown error'}`);
    } finally {
      setUploadingFiles(prev => {
        const newSet = new Set(prev);
        newSet.delete(fileKey);
        return newSet;
      });
    }
  };

  // 각 업로드 옵션별 처리 함수들
  const handleDocumentUpload = () => {
    setShowUploadOptions(false);
    // RAG_MODE fast 설정으로 문서 업로드 (다수 선택 가능)
    if (fileInputRef.current) {
      fileInputRef.current.value = ''; // 이전 선택 초기화
      fileInputRef.current.accept = '.pdf,.doc,.docx,.txt,.md,.ppt,.pptx,.hwp,.hwpx,.xls,.xlsx';
      fileInputRef.current.dataset.mode = 'fast';
      fileInputRef.current.multiple = true;
      fileInputRef.current.click();
    }
  };

  const handleImageUpload = () => {
    setShowUploadOptions(false);
    // 이미지 업로드 (1개만 선택 가능)
    if (fileInputRef.current) {
      fileInputRef.current.value = ''; // 이전 선택 초기화
      fileInputRef.current.accept = 'image/*';
      fileInputRef.current.multiple = false;
      fileInputRef.current.dataset.mode = 'image';
      fileInputRef.current.click();
    }
  };

  const handleRichRagUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files || files.length === 0) return;

    if (!user?.username) {
      setError('로그인이 필요합니다.');
      return;
    }

    for (const file of Array.from(files)) {
      try {
        // Rich RAG 모드로 업로드 (더 상세한 분석)
        await externalApiClient.uploadFile(
          file,
          user.username,
          'chat', // 타겟 폴더
          (progress, status) => {
            console.log(`Rich RAG 파일 ${file.name}: ${progress}% (${status})`);
          },
          'rich', // RAG_MODE rich 지정
          selectedProject?.name // 선택된 프로젝트명 전달
        );

        setRefreshTrigger(prev => prev + 1);
        console.log(`Rich RAG 파일 ${file.name} 업로드 완료`);
      } catch (error) {
        console.error(`Rich RAG 파일 ${file.name} 업로드 실패:`, error);
        setError(`Rich RAG 업로드 실패: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
      }
    }

    // 파일 입력 초기화
    if (event.target) {
      event.target.value = '';
    }
  };

  const handleRichRagClick = () => {
    setShowUploadOptions(false);
    // Rich RAG 문서 업로드 (fileInputRef 사용)
    if (fileInputRef.current) {
      fileInputRef.current.value = ''; // 이전 선택 초기화
      fileInputRef.current.accept = '.pdf,.doc,.docx,.txt,.md,.ppt,.pptx';
      fileInputRef.current.dataset.mode = 'rich';
      fileInputRef.current.multiple = true;
      fileInputRef.current.click();
    }
  };

  const handleExcelUpload = () => {
    setShowUploadOptions(false);
    setShowExcelModal(true);
  };

  // RAG_MODE fast로 문서 업로드
  const uploadDocumentWithFastMode = async (file: File) => {
    const fileKey = `${file.name}-${file.size}-${file.lastModified}`;

    // 이미 업로드 중인 파일이면 중단
    if (uploadingFiles.has(fileKey)) {
      console.log(`파일 ${file.name}은 이미 업로드 중입니다.`);
      return;
    }

    if (!user?.username) {
      setError('로그인이 필요합니다.');
      return;
    }

    try {
      setUploadingFiles(prev => new Set(prev).add(fileKey));

      // RAG_MODE fast로 업로드 (externalApiClient.uploadFile에 'fast' 전달)
      await externalApiClient.uploadFile(
        file,
        user.username,
        'chat', // 타겟 폴더
        (progress, status) => {
          console.log(`Fast 모드 파일 ${file.name}: ${progress}% (${status})`);
        },
        'fast', // RAG_MODE fast 지정
        selectedProject?.name // 선택된 프로젝트명 전달
      );

      // 즉시 백그라운드 작업 상태 새로고침 트리거
      setRefreshTrigger(prev => prev + 1);

      console.log(`파일 ${file.name} Fast 모드 업로드 완료`);
    } catch (error) {
      console.error(`파일 ${file.name} Fast 모드 업로드 실패:`, error);
      setError(error instanceof Error ? error.message : t('chat.fileUploadFailed', '파일 업로드에 실패했습니다.'));
    } finally {
      setUploadingFiles(prev => {
        const newSet = new Set(prev);
        newSet.delete(fileKey);
        return newSet;
      });
    }
  };

  // RAG_MODE rich로 문서 업로드
  const uploadDocumentWithRichMode = async (file: File) => {
    const fileKey = `${file.name}-${file.size}-${file.lastModified}`;

    // 이미 업로드 중인 파일이면 중단
    if (uploadingFiles.has(fileKey)) {
      console.log(`파일 ${file.name}은 이미 업로드 중입니다.`);
      return;
    }

    if (!user?.username) {
      setError('로그인이 필요합니다.');
      return;
    }

    try {
      setUploadingFiles(prev => new Set(prev).add(fileKey));

      // RAG_MODE rich로 업로드 (externalApiClient.uploadFile에 'rich' 전달)
      await externalApiClient.uploadFile(
        file,
        user.username,
        'chat', // 타겟 폴더
        (progress, status) => {
          console.log(`Rich 모드 파일 ${file.name}: ${progress}% (${status})`);
        },
        'rich', // RAG_MODE rich 지정
        selectedProject?.name // 선택된 프로젝트명 전달
      );

      // 즉시 백그라운드 작업 상태 새로고침 트리거
      setRefreshTrigger(prev => prev + 1);

      console.log(`파일 ${file.name} Rich 모드 업로드 완료`);
    } catch (error) {
      console.error(`파일 ${file.name} Rich 모드 업로드 실패:`, error);
      setError(error instanceof Error ? error.message : t('chat.fileUploadFailed', '파일 업로드에 실패했습니다.'));
    } finally {
      setUploadingFiles(prev => {
        const newSet = new Set(prev);
        newSet.delete(fileKey);
        return newSet;
      });
    }
  };

  const handleExcelProcess = async (options: {
    file: File;
    mode: 'ai' | 'manual';
    selectedSheet: string;
    selectedColumns: string[];
    dataDescription?: string;
    rowFilter?: {
      column: string;
      operator: 'equals' | 'contains' | 'greaterThan' | 'lessThan';
      value: string;
    };
    headerNormalization: boolean;
    preserveFormatting: boolean;
    aiRecommendations?: {
      structure: string;
      columns: string[];
      filters: any[];
    };
    dataSplitting?: {
      enabled: boolean;
      maxRowsPerFile: number;
      preserveHeaders: boolean;
    };
  }) => {
    try {
      setShowExcelModal(false);

      console.log('엑셀 전처리 옵션:', options);

      // 파일 검증
      if (!options.file) {
        throw new Error('처리할 엑셀 파일이 없습니다.');
      }

      // 로딩 상태 설정
      setIsUploading(true);

      // ExcelPreprocessModal의 모든 옵션을 활용한 전처리
      const processedData = await processExcelFile(options);

      // 처리된 파일들을 RAG 시스템에 업로드
      if (processedData && processedData.processedFiles && processedData.processedFiles.length > 0) {
        console.log(`${processedData.processedFiles.length}개의 마크다운 파일을 RAG 시스템에 업로드 중...`);
        
        for (let i = 0; i < processedData.processedFiles.length; i++) {
          const file = processedData.processedFiles[i];
          try {
            await uploadFileToRag(file);
            console.log(`파일 ${i + 1}/${processedData.processedFiles.length} 업로드 완료: ${file.name}`);
          } catch (uploadError) {
            console.error(`파일 ${file.name} 업로드 실패:`, uploadError);
            throw new Error(`파일 "${file.name}" 업로드 실패: ${uploadError instanceof Error ? uploadError.message : '알 수 없는 오류'}`);
          }
        }
      }

      const summary = processedData?.summary || {};
      setShowToast({
        message: `${t('chat.excelProcessingCompleted', '엑셀 파일 전처리가 완료되었습니다.')}
모드: ${options.mode}
시트: ${options.selectedSheet}
${t('chat.generatedFiles', '생성된 파일')}: ${summary.fileCount || 1}개
총 데이터: ${summary.rows || 0}행, ${summary.columns || 0}열`,
        type: 'success'
      });
      setTimeout(() => setShowToast(null), 8000);

    } catch (error) {
      console.error('엑셀 전처리 실패:', error);
      setShowToast({
        message: error instanceof Error ? error.message : '엑셀 전처리에 실패했습니다.',
        type: 'error'
      });
      setTimeout(() => setShowToast(null), 3000);
    } finally {
      setIsUploading(false);
    }
  };

  // 엑셀 파일 전처리 함수 (ExcelPreprocessModal의 모든 기능 활용)
  const processExcelFile = async (options: {
    file: File;
    mode: 'ai' | 'manual';
    selectedSheet: string;
    selectedColumns: string[];
    dataDescription?: string;
    rowFilter?: {
      column: string;
      operator: 'equals' | 'contains' | 'greaterThan' | 'lessThan';
      value: string;
    };
    headerNormalization: boolean;
    preserveFormatting: boolean;
    aiRecommendations?: {
      structure: string;
      columns: string[];
      filters: any[];
    };
    dataSplitting?: {
      enabled: boolean;
      maxRowsPerFile: number;
      preserveHeaders: boolean;
    };
  }) => {
    console.log('엑셀 파일 전처리 시작:', {
      mode: options.mode,
      selectedSheet: options.selectedSheet,
      selectedColumns: options.selectedColumns,
      dataDescription: options.dataDescription,
      rowFilter: options.rowFilter,
      headerNormalization: options.headerNormalization,
      preserveFormatting: options.preserveFormatting,
      aiRecommendations: options.aiRecommendations,
      dataSplitting: options.dataSplitting
    });

    // API가 기대하는 형식에 맞게 FormData 구성
    const formData = new FormData();
    formData.append('file', options.file);
    
    // options 객체를 JSON 문자열로 변환하여 전달 (API에서 JSON.parse로 파싱함)
    const optionsForApi = {
      mode: options.mode,
      selectedSheet: options.selectedSheet,
      selectedColumns: options.selectedColumns,
      dataDescription: options.dataDescription,
      rowFilter: options.rowFilter,
      headerNormalization: options.headerNormalization,
      preserveFormatting: options.preserveFormatting,
      aiRecommendations: options.aiRecommendations,
      dataSplitting: options.dataSplitting
    };
    
    formData.append('options', JSON.stringify(optionsForApi));
    
    if (user?.username) {
      formData.append('user_id', user.username);
    }

    // 실제 엑셀 전처리 API 호출
    const response = await fetch('/api/preprocess/excel', {
      method: 'POST',
      body: formData
    });

    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.error || `API 요청 실패: ${response.status}`);
    }

    const result = await response.json();
    
    if (!result.success) {
      throw new Error(result.error || '엑셀 전처리 실패');
    }

    const processedData = result.data;
    
    // 마크다운 파일들을 RAG 시스템에 업로드할 수 있도록 File 객체로 변환
    const processedFiles: File[] = [];
    
    if (processedData.processedFiles && processedData.processedFiles.length > 0) {
      for (const fileData of processedData.processedFiles) {
        // Base64 디코딩하여 Blob 생성
        const binaryData = atob(fileData.content);
        const bytes = new Uint8Array(binaryData.length);
        for (let i = 0; i < binaryData.length; i++) {
          bytes[i] = binaryData.charCodeAt(i);
        }
        
        const blob = new Blob([bytes], { type: 'text/markdown' });
        const file = new File([blob], fileData.name, { type: 'text/markdown' });
        processedFiles.push(file);
      }
    }

    return {
      processedFiles: processedFiles,
      summary: {
        mode: options.mode,
        sheet: options.selectedSheet,
        columns: processedData.columnCount,
        rows: processedData.totalRowCount,
        fileCount: processedData.fileCount,
        splitEnabled: processedData.splitEnabled,
        filters: options.rowFilter ? 1 : 0,
        normalizedHeaders: options.headerNormalization,
        preservedFormatting: options.preserveFormatting
      }
    };
  };

  const removeFile = (index: number) => {
    setUploadedFiles(prev => {
      const fileToRemove = prev[index];

      // 이미지 파일인 경우 생성된 URL 해제
      if (fileToRemove && isImageFile(fileToRemove)) {
        const fileKey = `${fileToRemove.name}-${fileToRemove.size}-${fileToRemove.lastModified}`;
        setUploadedFileUrls(urlMap => {
          const url = urlMap.get(fileKey);
          if (url) {
            URL.revokeObjectURL(url);
          }
          const newMap = new Map(urlMap);
          newMap.delete(fileKey);
          return newMap;
        });
      }

      // 해당 파일의 임베딩 상태도 제거
      if (fileToRemove) {
        setEmbeddingStatuses(prevStatuses => {
          const newStatuses = { ...prevStatuses };
          delete newStatuses[fileToRemove.name];
          return newStatuses;
        });
      }

      return prev.filter((_, i) => i !== index);
    });
  };

  const isImageFile = (file: File) => {
    return file.type.startsWith('image/');
  };

  // 이미지 압축 함수 추가
  const compressImage = async (file: File, maxWidth: number = 1024, maxHeight: number = 1024, quality: number = 0.8): Promise<File> => {
    return new Promise((resolve, reject) => {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const img = new Image();

      img.onload = () => {
        // 원본 크기
        const { width, height } = img;

        // 비율 유지하면서 크기 조정
        let newWidth = width;
        let newHeight = height;

        if (width > maxWidth || height > maxHeight) {
          const ratio = Math.min(maxWidth / width, maxHeight / height);
          newWidth = width * ratio;
          newHeight = height * ratio;
        }

        // Canvas 크기 설정
        canvas.width = newWidth;
        canvas.height = newHeight;

        // 이미지 그리기
        ctx?.drawImage(img, 0, 0, newWidth, newHeight);

        // Blob으로 변환
        canvas.toBlob(
          (blob) => {
            if (blob) {
              // 압축된 File 객체 생성
              const compressedFile = new File([blob], file.name, {
                type: file.type,
                lastModified: Date.now(),
              });

              console.log(`이미지 압축 완료: ${file.name}`);
              console.log(`  원본 크기: ${(file.size / 1024 / 1024).toFixed(2)}MB (${width}x${height})`);
              console.log(`  압축 후: ${(compressedFile.size / 1024 / 1024).toFixed(2)}MB (${newWidth}x${newHeight})`);
              console.log(`  압축률: ${((1 - compressedFile.size / file.size) * 100).toFixed(1)}%`);

              resolve(compressedFile);
            } else {
              reject(new Error('이미지 압축 실패'));
            }
          },
          file.type,
          quality
        );
      };

      img.onerror = () => reject(new Error('이미지 로드 실패'));
      img.src = URL.createObjectURL(file);
    });
  };

  // 드래그앤드롭 이벤트 핸들러들
  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    // 드래그 영역을 완전히 벗어났을 때만 상태 변경
    if (!e.currentTarget.contains(e.relatedTarget as Node)) {
      setIsDragOver(false);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(false);

    const files = Array.from(e.dataTransfer.files);
    if (files.length === 0) return;

    // 이미 업로드 중이면 중단
    if (isUploading) {
      // console.log('이미 파일 업로드 중입니다.');
      return;
    }

    // console.log('드래그앤드롭으로 파일 업로드:', files.map(f => f.name));

    const processedFiles: File[] = [];

    // 파일들을 순차적으로 처리 (이미지는 압축, 일반 파일은 그대로)
    for (const file of files) {
      if (isImageFile(file)) {
        try {
          console.log(`이미지 압축 시작: ${file.name}`);
          const compressedFile = await compressImage(file);
          processedFiles.push(compressedFile);
        } catch (error) {
          console.error(`이미지 압축 실패 (${file.name}):`, error);
          // 압축 실패 시 원본 파일 사용
          processedFiles.push(file);
        }
      } else {
        processedFiles.push(file);
      }
    }

    setUploadedFiles(prev => [...prev, ...processedFiles]);

    // 이미지 파일의 경우 썸네일 URL 생성 (압축된 파일 기준)
    const newUrls = new Map<string, string>();
    processedFiles.forEach(file => {
      if (isImageFile(file)) {
        const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
        const url = URL.createObjectURL(file);
        newUrls.set(fileKey, url);
      }
    });

    if (newUrls.size > 0) {
      setUploadedFileUrls(prev => new Map([...prev, ...newUrls]));
    }

    // 이미지가 아닌 파일들은 RAG 업로드
    for (const file of processedFiles) {
      if (!file.type.startsWith('image/')) {
        await uploadFileToRag(file);
      }
    }
  };

  // sendMessage에서 타이핑 효과가 끝난 후에만 loadSessions 호출
  const sendMessage = async () => {
    if ((!inputMessage.trim() && uploadedFiles.length === 0) || isLoading) return;


    // @ 멘션된 문서들 추출 - mentionedDocs 상태를 우선 사용하고, 없으면 정규표현식으로 추출
    let mentions: string[] = [];

    // 1. 먼저 상태에서 관리되는 멘션된 문서들 사용 (더 정확함)
    if (mentionedDocs.length > 0) {
      mentions = mentionedDocs.map(doc => doc.name);
      console.log('🔍 상태 기반 멘션 문서:', mentions);
    } else {
      // 2. 상태에 없으면 정규표현식으로 추출 (폴백)
      const mentionPattern = /@([^\s@]+(?:\.[^\s@]*)*)/g;
      let match;

      while ((match = mentionPattern.exec(inputMessage)) !== null) {
        const mentionText = match[1];
        // 이메일 주소 패턴인지 확인 (@ 이후에 도메인 형식이 아닌 경우만 포함)
        // 파일명은 보통 .pdf, .txt 등의 확장자를 가지므로 이를 허용
        if (!mentionText.match(/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/)) {
          mentions.push(mentionText);
        }
      }
      console.log('🔍 정규표현식 기반 멘션 문서:', mentions);
      console.log('🔍 원본 메시지:', inputMessage);
    }

    console.log('🔍 최종 멘션 목록:', mentions);

    // 세션 ID 처리: currentSessionId 우선, 없으면 URL에서 가져오기
    let sessionId = currentSessionId || searchParams.get('session');

    // console.log('세션 ID 상태:', {
    //   currentSessionId,
    //   urlSessionId: searchParams.get('session'),
    //   finalSessionId: sessionId,
    //   messageCount: currentMessages.length
    // });

    const imageFiles = uploadedFiles.filter(isImageFile);

    // 이미지 파일들을 base64로 인코딩하여 메시지에 첨부
    const attachedImages = await Promise.all(
      imageFiles.map(async (file) => {
        const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
        const url = uploadedFileUrls.get(fileKey);

        // 파일을 base64로 인코딩
        const base64Data = await new Promise<string>((resolve, reject) => {
          const reader = new FileReader();
          reader.onload = () => {
            const result = reader.result as string;
            resolve(result); // data:image/jpeg;base64,... 형태
          };
          reader.onerror = reject;
          reader.readAsDataURL(file);
        });

        return {
          file,
          url: url || '',
          name: file.name,
          base64: base64Data // base64 데이터 추가
        };
      })
    );

    // 사용자 메시지는 원본 전체 내용 저장 (멘션 포함)
    let messageContent = inputMessage.trim();
    if (attachedImages.length > 0) {
      // 텍스트 메시지와 이미지를 함께 저장
      const imageDataString = attachedImages.map(img =>
        `[IMAGE:${img.name}]${img.base64}`
      ).join('\n');

      messageContent = messageContent
        ? `${messageContent}\n\n${imageDataString}`
        : imageDataString;
    }

    const userMessage: Message = {
      id: `msg-${Date.now()}-user`,
      role: 'user',
      content: messageContent, // 원본 메시지 (멘션 포함)
      timestamp: new Date().toISOString(),
      attachedImages: attachedImages.filter(img => img.url).length > 0
        ? attachedImages.filter(img => img.url).map(img => ({
          file: img.file,
          url: img.url,
          name: img.name,
          base64: img.base64
        }))
        : undefined,
    };

    // 사용자 메시지 추가
    setCurrentMessages(prev => [...prev, userMessage]);
    setInputMessage('');

    // 업로드된 파일 목록과 URL 맵 즉시 정리하여 중복 표시 방지
    setUploadedFiles([]);
    setUploadedFileUrls(new Map());

    setMentionedDocs([]); // 멘션 상태 초기화
    setShowDocSuggestions(false);
    setMentionQuery('');
    setSelectedSuggestionIndex(0);
    resetTextareaHeight();
    setIsLoading(true);
    setError(null);

    try {
      // 스트리밍 모드에서는 서버에서 세션 저장을 처리하므로 프론트엔드에서 미리 저장하지 않음
      // 이는 중복 저장을 방지하기 위함
      if (sessionId) {
        // console.log('기존 세션 ID 확인:', sessionId, '- 서버에서 스트리밍과 함께 저장 처리됨');
      } else {
        // console.log('새 대화 - 백엔드에서 세션 생성 예정');
      }

      // 첫 번째 이미지 파일 (있는 경우)
      const imageFile = imageFiles.length > 0 ? imageFiles[0] : undefined;

      // 멘션된 문서들의 전체 내용 가져오기
      let cleanMessage = inputMessage.trim();

      if (mentions.length > 0) {
        try {
          // 멘션된 문서가 실제로 존재하는지 검증
          const availableDocs = ragFiles.filter(f => f.status === 'completed').map(f => f.name);
          const validMentions = mentions.filter(mention => availableDocs.includes(mention));
          const invalidMentions = mentions.filter(mention => !availableDocs.includes(mention));

          if (invalidMentions.length > 0) {
            console.warn('⚠️ 존재하지 않는 멘션된 문서들:', invalidMentions);
            console.log('📂 사용 가능한 문서들:', availableDocs);
          }

          if (validMentions.length > 0) {
            console.log('✅ 유효한 멘션된 문서들:', validMentions);
            // 백엔드에서 문서 내용을 처리하므로 프론트엔드에서는 문서명만 전달
            mentions.splice(0, mentions.length, ...validMentions);
          } else {
            console.warn('⚠️ 유효한 멘션된 문서가 없습니다.');
            mentions.length = 0; // 배열 비우기
          }

          // @ 멘션 제거한 깔끔한 질문 (실제 파일명 기반으로 정확히 매칭)
          // 사용 가능한 문서명들을 길이 순으로 정렬 (긴 파일명부터 매칭하여 부분 매칭 방지)
          const sortedAvailableDocs = availableDocs.sort((a, b) => b.length - a.length);

          for (const docName of sortedAvailableDocs) {
            // @ + 파일명 패턴을 찾아서 제거 (대소문자 구분 없이)
            const mentionPattern = new RegExp(`@${docName.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}`, 'gi');
            cleanMessage = cleanMessage.replace(mentionPattern, '');
          }

          // 이메일 주소가 아닌 남은 @ 멘션들 제거 (공백 포함 파일명 대응)
          cleanMessage = cleanMessage.replace(/@([^\s@]+(?:\s+[^\s@]+)*)/g, (match, captured) => {
            // 이메일 주소 패턴인지 확인
            if (captured.includes('.') && captured.match(/^[a-zA-Z0-9._%+-]+\.[a-zA-Z]{2,}$/)) {
              return match; // 이메일 주소는 그대로 유지
            }
            return ''; // 나머지 멘션 제거
          });

          // 연속된 공백을 하나로 정규화하고 앞뒤 공백 제거
          cleanMessage = cleanMessage.replace(/\s+/g, ' ').trim();
          
          // 사용자 메시지 내용은 변경하지 않음 (원본 유지)
          // cleanMessage는 백엔드 전송용으로만 사용
        } catch (error) {
          console.error('❌ 멘션된 문서 검증 실패:', error);
        }
      }

      // 이미지가 첨부된 경우 검색 옵션 자동 비활성화
      const hasImageFiles = imageFiles.length > 0;

      if (hasImageFiles) {
        // console.log('🖼️ 이미지 파일 감지 - 검색 옵션 자동 비활성화:', imageFiles.map(f => f.name));
      }

      // 사용자 정보 확인
      const username = user?.username;

      const chatOptions = {
        provider: options.provider,
        model: options.model,
        parameters: {
          temperature: options.temperature,
          max_tokens: options.maxTokens,
        },
        rag: (mentions.length > 0 || hasImageFiles) ? false : options.enableRag, // 멘션 사용 시 또는 이미지 첨부 시 RAG 비활성화
        ragSearchScope: options.ragSearchScope, // RAG 검색 범위 추가
        web: hasImageFiles ? false : options.enableWebSearch, // 이미지 첨부 시 웹 검색 비활성화
        healthAnalysis: options.enableHealthAnalysis, // 건강검진 분석 옵션

        userId: Number(options.userId), // 정수형 ID (세션 관리용)
        username: username, // 문자열 사용자명 (RAG 검색용)
        type: 'chat', // 세션 타입 명시
        mentionedDocuments: mentions, // 멘션된 문서 목록 (백엔드에서 내용 처리)
      };

      // 디버그: 전송할 데이터 로그
      console.log('🔍 백엔드로 전송할 데이터:', {
        originalMessage: messageContent,
        cleanMessage: cleanMessage,
        mentions: mentions,
        chatOptions: {
          ...chatOptions,
          mentionedDocuments: mentions
        }
      });

      // 디버깅을 위한 로그 추가
      // console.log('=== 프론트엔드 전송 데이터 디버깅 ===');
      // console.log('options.ragSearchScope:', options.ragSearchScope);
      // console.log('options.userId:', options.userId, 'type:', typeof options.userId);
      // console.log('user?.id:', user?.id, 'type:', typeof user?.id);
      // console.log('username:', username);
      // console.log('chatOptions.ragSearchScope:', chatOptions.ragSearchScope);
      // console.log('chatOptions.userId:', chatOptions.userId, 'type:', typeof chatOptions.userId);
      // console.log('chatOptions.username:', chatOptions.username);
      // console.log('전체 chatOptions:', JSON.stringify(chatOptions, null, 2));
      // console.log('========================');

      // 웹소켓 스트리밍 API 호출
      const assistantMessage: Message = {
        id: `msg-${Date.now()}-assistant`,
        role: 'assistant',
        content: '', // 스트리밍으로 실시간 업데이트
        timestamp: new Date().toISOString(),
      };

      // UI에 빈 어시스턴트 메시지 추가
      setCurrentMessages(prev => [...prev, assistantMessage]);

      let streamingContent = '';
      let thinkingContent = '';
      let responseSessionId: string | null = null;

      // 스트림 시작 시 isStreamingRef.current를 true로 설정
      isStreamingRef.current = true;

      await streamChatWithWebSocket(
        messageContent || (imageFile ? t('chat.describeImage', '이 이미지에 대해 설명해주세요.') : ''),
        {
          ...chatOptions,
          sessionId: sessionId || undefined,
          settings: {
            USE_LLM: options.provider,
            model: options.model,
            temperature: options.temperature,
            max_tokens: options.maxTokens,
          },
          // Reasoning Level 추가
          reasoningLevel: options.reasoningLevel,
          // 선택된 프로젝트 정보 추가
          selectedProject: selectedProject ? {
            id: selectedProject.id,
            name: selectedProject.name
          } : null,
          // 이미지 파일이 있으면 첫 번째 이미지의 base64 데이터 전달
          image: imageFiles.length > 0 ? attachedImages[0]?.base64 : null,
          // 현재 메시지에는 이미 userMessage가 포함되어 있으므로 그대로 전달
          // 메타데이터도 함께 전송하여 백엔드에서 컨텍스트를 이해할 수 있도록 함
          history: currentMessages.map(msg => ({
            role: msg.role,
            content: msg.content,
            ...(msg.originalContent && { originalContent: msg.originalContent }),
            ...(msg.downloadInfo && { downloadInfo: msg.downloadInfo }),
            ...(msg.thinking && { thinking: msg.thinking }),
            ...(msg.timestamp && { timestamp: msg.timestamp })
          }))
        },
        {
          onConnect: () => {
            // console.log('웹소켓 연결 성공');
            // setProgressMessages(['서버에 연결 중...']);
          },
          onContent: (content: string) => {
            streamingContent += content;
            // 실시간으로 메시지 업데이트
            setCurrentMessages(prev =>
              prev.map(msg =>
                msg.id === assistantMessage.id
                  ? { ...msg, content: streamingContent, thinking: thinkingContent }
                  : msg
              )
            );
            setTimeout(scrollToBottom, 10);
          },
          onThinking: (thinking: string) => {
            // console.log('Frontend onThinking 콜백 호출됨:', thinking);
            
            // thinking 내용 누적
            thinkingContent += thinking;
            
            // 누적된 thinking 내용으로 메시지 업데이트
            setCurrentMessages(prev => {
              const updatedMessages = prev.map(msg =>
                msg.id === assistantMessage.id
                  ? { ...msg, thinking: thinkingContent }  // 누적된 thinking 내용 사용
                  : msg
              );
              
              // console.log('🔍 [DEBUG] onThinking에서 메시지 업데이트:', {
              //   assistantMessageId: assistantMessage.id,
              //   thinkingContentLength: thinkingContent?.length || 0,
              //   newThinkingChunk: thinking
              // });
              
              return updatedMessages;
            });
            
            // thinking 업데이트 후 즉시 스크롤
            setTimeout(scrollToBottom, 10);
          },
          onProgress: (message: string, percent?: number, stage?: string) => {
            // console.log('진행 상황 업데이트:', { message, percent, stage });
            // 진행 상황 메시지 업데이트
            setProgressMessages(prev => {
              const newMessages = [...prev];
              // 같은 stage의 메시지가 있으면 교체, 없으면 추가
              if (stage) {
                const existingIndex = newMessages.findIndex(msg => msg.includes(stage));
                if (existingIndex !== -1) {
                  newMessages[existingIndex] = message;
                } else {
                  newMessages.push(message);
                }
              } else {
                newMessages.push(message);
              }
              return newMessages;
            });
          },
          onComplete: (response: any) => {
            console.log('🔍 [DEBUG] onComplete 호출됨:', response);

            // 세션 ID 추출 - response가 문자열인지 객체인지 확인
            if (typeof response === 'string') {
              // response가 문자열이면 그대로 사용하고 sessionId는 기존 값 유지
              responseSessionId = sessionId;
            } else if (response && typeof response === 'object') {
              // response가 객체이면 data.sessionId 또는 sessionId에서 추출
              responseSessionId = response.data?.sessionId || response.sessionId || sessionId;
            } else {
              responseSessionId = sessionId;
            }

            // 최종 메시지 업데이트 (downloadInfo 포함)
            let downloadInfo = response?.data?.downloadInfo || response?.downloadInfo || undefined;

            // 멘션된 문서가 있지만 downloadInfo가 없는 경우, 멘션된 문서에 대한 다운로드 정보 생성
            if (mentions.length > 0 && (!downloadInfo || downloadInfo.length === 0)) {
              downloadInfo = mentions.map(doc => ({
                filename: doc,
                pages: [] // 페이지 정보는 백엔드에서 제공하지 않으므로 빈 배열
              }));
              console.log('🔍 멘션된 문서에 대한 다운로드 정보 생성:', downloadInfo);
            }

            let finalContent = streamingContent;

            console.log('🔍 [DEBUG] 참조 정보 확인:', {
              hasResponse: !!response,
              responseType: typeof response,
              hasData: !!(response?.data),
              downloadInfoLength: downloadInfo?.length || 0,
              downloadInfo: downloadInfo
            });

            // 실시간 스트리밍에서는 HTML 주석을 추가하지 않고 downloadInfo만 설정
            // (HTML 주석은 DB 저장용이고, UI 표시에서는 downloadInfo prop 사용)
            console.log('🔍 [DEBUG] 실시간 스트리밍 참조 정보:', {
              downloadInfoLength: downloadInfo?.length || 0,
              downloadInfo: downloadInfo
            });

            // console.log('🔍 [DEBUG] onComplete에서 thinking 내용 확인:', {
            //   thinkingContentLength: thinkingContent?.length || 0,
            //   thinkingContent: thinkingContent?.substring(0, 100) + '...',
            //   finalContentLength: finalContent?.length || 0,
            //   assistantMessageId: assistantMessage.id,
            //   responseType: typeof response
            // });

            setCurrentMessages(prev => {
              const updatedMessages = prev.map(msg =>
                msg.id === assistantMessage.id
                  ? {
                    ...msg,
                    content: finalContent,
                    thinking: thinkingContent,  // Thinking 내용 유지
                    downloadInfo: downloadInfo
                  }
                  : msg
              );
              
              // console.log('🔍 [DEBUG] onComplete 후 메시지 업데이트:', {
              //   assistantMessageId: assistantMessage.id,
              //   thinkingContentLength: thinkingContent?.length || 0,
              //   updatedMessageThinking: updatedMessages.find(m => m.id === assistantMessage.id)?.thinking?.length || 0
              // });
              
              return updatedMessages;
            });

            setIsLoading(false);
            setProgressMessages([]);
            // onComplete 콜백에서 스트리밍 상태 재설정 보장
            isStreamingRef.current = false;

            // 세션 관리 (새 세션인 경우만 URL 업데이트)
            if (responseSessionId && responseSessionId !== sessionId) {
              console.log('🔍 [DEBUG] 새 세션 생성됨 - URL 업데이트:', {
                oldSessionId: sessionId,
                newSessionId: responseSessionId
              });
              
              // 새 세션 생성 플래그 설정 (ref 사용으로 재렌더링 방지)
              isNewSessionJustCreatedRef.current = true;
              setCurrentSessionId(responseSessionId);
              
              // URL 업데이트를 history.replaceState로 변경하여 페이지 재렌더링 방지
              const url = new URL(window.location.href);
              url.searchParams.set('session', responseSessionId);
              window.history.replaceState({}, '', url.toString());
              
              // 플래그를 3초 후 리셋
              setTimeout(() => {
                isNewSessionJustCreatedRef.current = false;
              }, 3000);
            }

            // 세션 목록 새로고침
            const refreshSessions = async () => {
              try {
                const latestSessions = await loadSessions();
                setSessions(latestSessions);
                
                // 현재 세션의 제목도 업데이트
                if (responseSessionId) {
                  const currentSession = latestSessions.find(s => s.id === responseSessionId);
                  if (currentSession) {
                    updateSessionInList(responseSessionId, currentSession.history);
                  }
                }
              } catch (error) {
                console.error('세션 목록 새로고침 실패:', error);
              }
            };
            refreshSessions();
          },
          onError: (error: Error) => {
            console.error('스트리밍 오류:', error);
            // error.message가 객체일 수 있으므로 안전하게 처리
            const errorMessage = typeof error.message === 'string'
              ? error.message
              : (error.message ? JSON.stringify(error.message) : '알 수 없는 오류가 발생했습니다.');
            setError(errorMessage);

            // 모든 오류 시나리오에서 적절한 상태 정리 보장
            setIsLoading(false);
            setProgressMessages([]);
            isStreamingRef.current = false;

            // WebSocket 연결 정리
            if (wsConnectionRef.current) {
              try {
                if (wsConnectionRef.current.readyState === WebSocket.OPEN) {
                  wsConnectionRef.current.close();
                }
              } catch (closeError) {
                console.error('오류 처리 중 WebSocket 정리 실패:', closeError);
              } finally {
                wsConnectionRef.current = null;
              }
            }
          },
          onDisconnect: () => {
            // console.log('🔍 [DEBUG] onDisconnect 호출됨 - thinking 내용 유지 처리');
            
            // 연결 종료 시에도 thinking 내용 유지
            // console.log('🔍 [DEBUG] onDisconnect에서 thinking 내용 확인:', {
            //   thinkingContentLength: thinkingContent?.length || 0,
            //   thinkingContent: thinkingContent?.substring(0, 100) + '...',
            //   assistantMessageId: assistantMessage.id
            // });

            setCurrentMessages(prev =>
              prev.map(msg =>
                msg.id === assistantMessage.id
                  ? {
                    ...msg,
                    thinking: thinkingContent  // onDisconnect에서도 thinking 내용 유지
                  }
                  : msg
              )
            );

            setIsLoading(false);
            setProgressMessages([]);
            isStreamingRef.current = false;

            // WebSocket이 예기치 않게 닫힌 경우 ref 정리
            if (wsConnectionRef.current) {
              wsConnectionRef.current = null;
            }
          }
        },
        String(options.userId),
        wsConnectionRef // streamChatWithWebSocket 함수 호출에 wsConnectionRef 전달
      );

      return; // 여기서 함수 종료

    } catch (error) {
      console.error('채팅 요청 중 오류:', error);
      setError(error instanceof Error ? error.message : '알 수 없는 오류가 발생했습니다.');

      // 모든 오류 시나리오에서 적절한 상태 정리 보장
      setIsLoading(false);
      setProgressMessages([]);
      isStreamingRef.current = false;

      // WebSocket 연결 정리
      if (wsConnectionRef.current) {
        try {
          if (wsConnectionRef.current.readyState === WebSocket.OPEN) {
            wsConnectionRef.current.close();
          }
        } catch (closeError) {
          console.error('오류 처리 중 WebSocket 정리 실패:', closeError);
        } finally {
          wsConnectionRef.current = null;
        }
      }
    }
  };

  const updateSessionInList = (sessionId: string, messages: Message[]) => {
    // 안전하게 content를 문자열로 변환하는 함수
    const safeStringify = (content: any): string => {
      if (typeof content === 'string') {
        // 이미지 데이터 제거
        const imagePattern = /\[IMAGE:[^\]]+\]data:image\/[^;]+;base64,[A-Za-z0-9+/=]+/g;
        let cleanContent = content.replace(imagePattern, '').trim();
        
        // 마크다운 특수문자 제거 (세션 제목용)
        cleanContent = cleanContent.replace(/\*\*([^*]+)\*\*/g, '$1'); // **bold**
        cleanContent = cleanContent.replace(/\*([^*]+)\*/g, '$1'); // *italic*
        cleanContent = cleanContent.replace(/__([^_]+)__/g, '$1'); // __underline__
        cleanContent = cleanContent.replace(/~~([^~]+)~~/g, '$1'); // ~~strikethrough~~
        cleanContent = cleanContent.replace(/```[^`]*```/g, ''); // code blocks
        cleanContent = cleanContent.replace(/`([^`]+)`/g, '$1'); // inline code
        cleanContent = cleanContent.replace(/^#+ /gm, ''); // headers
        cleanContent = cleanContent.replace(/^[*\-+] /gm, ''); // list items
        cleanContent = cleanContent.replace(/\[([^\]]+)\]\([^)]+\)/g, '$1'); // links
        cleanContent = cleanContent.replace(/!\[([^\]]*)\]\([^)]+\)/g, ''); // images
        
        return cleanContent.trim();
      }
      if (content === null || content === undefined) return '';
      try {
        return typeof content === 'object' ? JSON.stringify(content) : String(content);
      } catch {
        return String(content);
      }
    };

    setSessions(prev => prev.map(session => {
      if (session.id === sessionId) {
        const lastMessage = messages.length > 0
          ? safeStringify(messages[messages.length - 1]?.content) || t('chat.newConversation', '새 대화')
          : t('chat.newConversation', '새 대화');

        return {
          ...session,
          lastMessage: lastMessage.length > 50 ? lastMessage.slice(0, 50) + '...' : lastMessage,
          messageCount: messages.length,
          updatedAt: new Date().toISOString(),
          history: messages,
          title: session.title === t('chat.newConversation', '새 대화') && messages.length > 0
            ? (() => {
              // 첫 번째 사용자 메시지를 찾아서 제목으로 사용
              const firstUserMessage = messages.find(msg =>
                msg && msg.role === 'user' && msg.content
              );
              if (firstUserMessage) {
                // originalContent가 있으면 그것을 우선 사용 (멘션된 문서가 포함되기 전 원본 질문)
                const content = firstUserMessage.originalContent
                  ? safeStringify(firstUserMessage.originalContent)
                  : safeStringify(firstUserMessage.content);

                // 이미지만 있고 텍스트가 없는 경우
                if (content.length === 0 && firstUserMessage.content.includes('[IMAGE:')) {
                  return '이미지 질문';
                }
                // 시스템 프롬프트나 긴 내용은 간단히 처리 (원본 질문 사용 시에는 이 조건이 덜 걸림)
                if (content.length > 1000 || content.includes('다음 문서들의 내용을 바탕으로') || content.includes('검색 결과:')) {
                  return t('chat.question', '질문'); // 시스템 프롬프트인 경우 기본 제목
                }
                return content.slice(0, 30) + (content.length > 30 ? '...' : '');
              } else {
                // 사용자 메시지가 없으면 AI 응답을 사용
                const firstAIResponse = messages.find(msg =>
                  msg && msg.role === 'assistant' && msg.content
                );
                const content = firstAIResponse?.content ? safeStringify(firstAIResponse.content) : t('chat.newConversation', '새 대화');
                return content.slice(0, 30) + (content.length > 30 ? '...' : '');
              }
            })()
            : session.title,
        };
      }
      return session;
    }));
  };

  const handleDeleteClick = (sessionId: string) => {
    // console.log('[DEBUG] 삭제 아이콘 클릭, sessionId:', sessionId);
    setSessionToDelete(sessionId);
    setShowDeleteModal(true);
  };

  // confirmDelete에서 loadSessions의 반환값을 받아 setSessions로 강제 동기화
  const confirmDelete = async () => {
    // console.log('[DEBUG] confirmDelete 진입, sessionToDelete:', sessionToDelete);
    if (!sessionToDelete) return;
    try {
      // console.log('[DEBUG] 세션 삭제 요청:', sessionToDelete);
      const response = await externalApiClient.deleteSession(sessionToDelete);
      // console.log('[DEBUG] 세션 삭제 응답:', response);
      // 삭제 후 서버에서 최신 목록을 다시 불러오고, 최신 세션 배열을 받음
      const latestSessions = await loadSessions();
      setSessions(latestSessions); // 최신 세션 배열로 강제 동기화
      if (sessionToDelete === currentSessionId) {
        setCurrentSessionId(null);
        setCurrentMessages([]);
        const url = new URL(window.location.href);
        url.searchParams.delete('session');
        window.history.replaceState({}, '', url.toString());
        setIsNewChatMode(true);
      } else {
        const exists = latestSessions.some(s => s.id === currentSessionId);
        if (!exists) {
          setCurrentSessionId(null);
          setCurrentMessages([]);
          const url = new URL(window.location.href);
          url.searchParams.delete('session');
          window.history.replaceState({}, '', url.toString());
          setIsNewChatMode(true);
        }
      }
    } catch (error) {
      console.error('세션 삭제 실패:', error);
      setError(t('chat.deleteSessionFailed', '대화 삭제에 실패했습니다.'));
    } finally {
      setShowDeleteModal(false);
      setSessionToDelete(null);
    }
  };

  const cancelDelete = () => {
    setShowDeleteModal(false);
    setSessionToDelete(null);
  };

  // 전체 대화목록 삭제 핸들러
  const handleDeleteAllClick = () => {
    setShowDeleteAllModal(true);
  };

  const confirmDeleteAll = async () => {
    try {
      await externalApiClient.deleteAllSessions();

      // 상태 초기화
      setSessions([]);
      setCurrentMessages([]);
      setIsNewChatMode(true);

      // URL에서 세션 파라미터 제거
      const url = new URL(window.location.href);
      url.searchParams.delete('session');
      window.history.replaceState({}, '', url.toString());

      setShowDeleteAllModal(false);

      // 성공 토스트 표시
      setShowToast({ message: t('chat.allChatsDeleted', '모든 대화가 삭제되었습니다.'), type: 'success' });
      setTimeout(() => setShowToast(null), 3000);

    } catch (error) {
      console.error('전체 세션 삭제 실패:', error);
      setError(t('chat.deleteAllFailed', '전체 대화 삭제에 실패했습니다.'));

      // 에러 토스트 표시
      setShowToast({ message: t('chat.deleteAllFailed', '전체 대화 삭제에 실패했습니다.'), type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    }
  };

  const cancelDeleteAll = () => {
    setShowDeleteAllModal(false);
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    // 문서 자동완성이 표시된 상태에서 키보드 네비게이션 처리
    if (showDocSuggestions && filteredDocsForMention.length > 0) {
      if (e.key === 'ArrowDown') {
        e.preventDefault();
        setSelectedSuggestionIndex(prev =>
          prev < filteredDocsForMention.length - 1 ? prev + 1 : 0
        );
        return;
      }

      if (e.key === 'ArrowUp') {
        e.preventDefault();
        setSelectedSuggestionIndex(prev =>
          prev > 0 ? prev - 1 : filteredDocsForMention.length - 1
        );
        return;
      }

      if (e.key === 'Enter' && !e.shiftKey) {
        e.preventDefault();
        const selectedDoc = filteredDocsForMention[selectedSuggestionIndex];
        if (selectedDoc) {
          selectDocument(selectedDoc.name);
        }
        return;
      }

      if (e.key === 'Escape') {
        e.preventDefault();
        setShowDocSuggestions(false);
        setMentionQuery('');
        setSelectedSuggestionIndex(0);
        return;
      }
    }

    // 일반 메시지 전송
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  // filteredSessions를 useMemo로 감싸서 항상 updatedAt 내림차순 정렬을 보장
  const filteredSessions = useMemo(() => {
    return sessions
      .filter(session => {
        const safeTitle = typeof session.title === 'string' ? session.title : '';
        const safeLastMessage = typeof session.lastMessage === 'string' ? session.lastMessage : '';
        const safeSearchTerm = typeof searchTerm === 'string' ? searchTerm : '';
        return safeTitle.toLowerCase().includes(safeSearchTerm.toLowerCase()) ||
          safeLastMessage.toLowerCase().includes(safeSearchTerm.toLowerCase());
      });
    // .sort((a, b) => {
    //   // createdAt 기준 내림차순 정렬
    //   const aTime = a.createdAt ? new Date(a.createdAt).getTime() : 0;
    //   const bTime = b.createdAt ? new Date(b.createdAt).getTime() : 0;
    //   return bTime - aTime;
    // });
  }, [sessions, searchTerm]);

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    const now = new Date();
    const diffInHours = (now.getTime() - date.getTime()) / (1000 * 60 * 60);

    if (diffInHours < 1) {
      return t('chat.justNow', '방금 전');
    } else if (diffInHours < 24) {
      return `${Math.floor(diffInHours)}시간 전`;
    } else if (diffInHours < 24 * 7) {
      return `${Math.floor(diffInHours / 24)}일 전`;
    } else {
      return date.toLocaleDateString('ko-KR');
    }
  };

  // RAG 파일 목록 로드
  const loadChatDocument = async (userId: string) => {
    const res = await fetch('/api/chat/documents?userId=' + userId, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      }
    });
    const result = await res.json();
    if (!res.ok || !result.success) {
      throw new Error(result.error || '문서 로드 실패');
    }
    return result;
  };

  // RAG 파일 목록 정렬 헬퍼 함수
  const sortRagFiles = (files: RagFile[]): RagFile[] => {
    // 업로딩/처리중 파일과 완료된 파일 분리
    const processingFiles = files.filter(f => f.status === 'uploading' || f.status === 'processing' || f.status === 'pending' || f.status === 'error');
    const completedFiles = files.filter(f => f.status === 'completed');
    
    // 완료된 파일만 이름순 정렬 (대소문자 구분 없이)
    const sortedCompletedFiles = completedFiles.sort((a, b) => 
      a.name.toLowerCase().localeCompare(b.name.toLowerCase())
    );
    
    // 처리중 파일들도 이름순 정렬
    const sortedProcessingFiles = processingFiles.sort((a, b) => 
      a.name.toLowerCase().localeCompare(b.name.toLowerCase())
    );
    
    return [...sortedProcessingFiles, ...sortedCompletedFiles];
  };

  const loadRagFiles = async () => {
    if (!user?.username) return;

    try {
      setIsLoadingRagFiles(true);
      const result = await loadChatDocument(user.username);
      // const res = await fetch('/api/chat/documents');
      // if (!res.ok) {
      //   throw new Error(`서버 응답 오류: ${res.status}`);
      // }

      // const result = await res.json();
      // console.log("RAG 파일 목록 응답:", result);

      if (result.success && Array.isArray(result.data)) {
        let files = result.data;

        // 선택된 프로젝트가 있으면 해당 프로젝트의 문서만 필터링
        if (selectedProject) {
          files = files.filter((file: any) => file.project_name === selectedProject.name);
        }

        const completedFiles = files.map((file: any) => {
          const mappedStatus = file.embedding_status === 'completed' ? 'completed' as const :
                              file.embedding_status === 'pending' ? 'pending' as const :
                              file.embedding_status === 'processing' ? 'processing' as const :
                              file.embedding_status === 'failed' ? 'error' as const : 'completed' as const;

          // 디버그 로그: 상태 매핑 확인
          console.log(`[RAG Files] ${file.filename}: ${file.embedding_status} -> ${mappedStatus}`);

          return {
            id: file.id,
            name: file.filename,
            size: file.filesize,
            type: file.mimetype,
            path: file.filepath,
            status: mappedStatus,
            createdAt: file.created_at,
            projectName: file.project_name // 프로젝트명 추가
          };
        });

        setRagFiles(prev => {
          const processingFiles = prev.filter(
            (f: RagFile) => f.status === 'uploading' || f.status === 'processing' || f.status === 'pending'
          );
          const existingNames = new Set(completedFiles.map((f: RagFile) => f.name));
          const uniqueProcessingFiles = processingFiles.filter(
            (f: RagFile) => !existingNames.has(f.name)
          );
          // sortRagFiles 헬퍼 함수 사용
          return sortRagFiles([...uniqueProcessingFiles, ...completedFiles]);
        });
      } else {
        console.error('파일 목록을 불러오지 못했습니다:', result);
      }
    } catch (error) {
      console.error('RAG 파일 목록 로드 중 오류:', error);
    } finally {
      setIsLoadingRagFiles(false);
    }
  };

  // 프로젝트 목록 로드
  const loadProjects = async () => {
    if (!user?.username) return;

    try {
      setIsLoadingProjects(true);
      const headers = await getAuthHeaders();
      const response = await fetch('/api/projects', {
        method: 'GET',
        headers
      });

      if (!response.ok) {
        throw new Error(`프로젝트 목록 조회 실패: ${response.status}`);
      }

      const result = await response.json();
      if (result.success && Array.isArray(result.data)) {
        setProjects(result.data);
      } else {
        console.error('프로젝트 목록을 불러오지 못했습니다:', result);
      }
    } catch (error) {
      console.error('프로젝트 목록 로드 중 오류:', error);
    } finally {
      setIsLoadingProjects(false);
    }
  };

  // 프로젝트 생성
  const createProject = async (name: string, description?: string) => {
    if (!user?.username) return;

    try {
      const headers = await getAuthHeaders();
      const response = await fetch('/api/projects', {
        method: 'POST',
        headers: {
          ...headers,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ name, description })
      });

      const result = await response.json();
      if (result.success) {
        await loadProjects(); // 프로젝트 목록 새로고침
        setShowToast({ message: '프로젝트가 성공적으로 생성되었습니다.', type: 'success' });
        return result.data;
      } else {
        throw new Error(result.error?.message || '프로젝트 생성 실패');
      }
    } catch (error) {
      console.error('프로젝트 생성 중 오류:', error);
      setShowToast({ message: `프로젝트 생성 실패: ${error}`, type: 'error' });
      throw error;
    }
  };

  // 프로젝트 삭제
  const deleteProject = async (projectName: string) => {
    if (!user?.username) return;

    try {
      const headers = await getAuthHeaders();
      const response = await fetch(`/api/projects/${encodeURIComponent(projectName)}`, {
        method: 'DELETE',
        headers
      });

      const result = await response.json();
      if (result.success) {
        // 삭제된 프로젝트가 현재 선택된 프로젝트인 경우 선택 해제
        if (selectedProject?.name === projectName) {
          setSelectedProject(null);
        }
        await loadProjects(); // 프로젝트 목록 새로고침
        await loadRagFiles(); // 문서 목록도 새로고침
        setShowToast({ message: `프로젝트 "${projectName}"가 성공적으로 삭제되었습니다.`, type: 'success' });
        setTimeout(() => setShowToast(null), 3000);
        return result.data;
      } else {
        throw new Error(result.error?.message || '프로젝트 삭제 실패');
      }
    } catch (error) {
      console.error('프로젝트 삭제 중 오류:', error);
      setShowToast({ message: `프로젝트 삭제 실패: ${error}`, type: 'error' });
      setTimeout(() => setShowToast(null), 5000);
      throw error;
    }
  };

  // RAG chat 문서 삭제 (chat_documents DB 및 파일 제거)
  const deleteChatDocument = async (filename: string, username: string) => {
    const res = await fetch('/api/chat/documents', {
      method: 'DELETE',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ filename, id: username }),
    });

    const result = await res.json();
    if (!res.ok || !result.success) {
      throw new Error(result.error || '문서 삭제 실패');
    }
    return result;
  };

  // RAG 파일 삭제
  const deleteRagFile = async (filename: string) => {
    if (!user?.username) return;

    try {
      // 1. RAG 파일 삭제 (외부 API)
      const response = await externalApiClient.deleteRagFile(filename, user.username);
      if (response.success) {
        // 2. chat_documents에서 DB 및 파일 삭제
        const resPg = await deleteChatDocument(filename, user.username);

        // 3. 대화창의 uploadedFiles에서도 해당 파일 삭제
        setUploadedFiles(prev => {
          const filteredFiles = prev.filter(file => file.name !== filename);
          console.log(`대화창에서 파일 ${filename} 제거 완료`);
          return filteredFiles;
        });

        // 3-1. 해당 파일의 임베딩 상태도 제거
        setEmbeddingStatuses(prev => {
          const newStatuses = { ...prev };
          delete newStatuses[filename];
          console.log(`임베딩 상태에서 파일 ${filename} 제거 완료`);
          return newStatuses;
        });

        // 4. 해당 파일의 이미지 URL도 정리
        setUploadedFileUrls(prev => {
          const newUrls = new Map(prev);
          for (const [key, url] of prev.entries()) {
            if (key.startsWith(filename + '-')) {
              URL.revokeObjectURL(url);
              newUrls.delete(key);
            }
          }
          return newUrls;
        });

        // 5. 파일 목록 새로고침
        await loadRagFiles();
      } else {
        console.error('RAG 파일 삭제 실패:', response.error?.message);
      }
    } catch (error) {
      console.error('RAG 파일 삭제 오류:', error);
    }
  };


  // 사이드바 드래그 앤 드롭 기능 제거됨 - 대화창 하단의 파일 버튼 사용

  // 파일 크기 포맷팅
  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  // @ 멘션 처리 함수들
  const handleInputChange = (e: React.ChangeEvent<HTMLTextAreaElement>) => {
    const value = e.target.value;
    const cursorPos = e.target.selectionStart;

    setInputMessage(value);

    // 자동 높이 조절
    const target = e.target as HTMLTextAreaElement;
    target.style.height = 'auto';
    target.style.height = `${Math.min(Math.max(target.scrollHeight, 48), 200)}px`;

    // @ 멘션 감지 (이메일 주소 제외)
    const textBeforeCursor = value.substring(0, cursorPos);
    const mentionMatch = textBeforeCursor.match(/(\s|^)@([^\s@]*)$/);

    if (mentionMatch) {
      const mentionText = mentionMatch[2]; // 두 번째 캡처 그룹 (@ 뒤의 텍스트)
      // 이메일 주소 패턴 감지 - 이미 도메인 형태가 입력되고 있으면 자동완성 비활성화
      const isEmailPattern = mentionText.includes('.') && mentionText.match(/^[a-zA-Z0-9._%+-]*\.[a-zA-Z]*$/);

      if (!isEmailPattern) {
        setMentionQuery(mentionText);
        // @ 위치 계산 시 공백 또는 시작 위치 고려
        const atPosition = textBeforeCursor.lastIndexOf('@');
        setMentionStartPos(atPosition);
        setShowDocSuggestions(true);
        setSelectedSuggestionIndex(0); // 자동완성 표시 시 첫 번째 항목 선택
      } else {
        setShowDocSuggestions(false);
        setMentionQuery('');
        setSelectedSuggestionIndex(0);
      }
    } else {
      setShowDocSuggestions(false);
      setMentionQuery('');
      setSelectedSuggestionIndex(0);
    }
  };

  const selectDocument = (docName: string) => {
    if (!textareaRef.current) return;

    const textarea = textareaRef.current;
    const cursorPos = textarea.selectionStart;
    const textBeforeCursor = inputMessage.substring(0, cursorPos);
    const textAfterCursor = inputMessage.substring(cursorPos);

    // @ 멘션 부분을 문서명으로 교체
    const beforeMention = textBeforeCursor.substring(0, mentionStartPos);
    const newMessage = `${beforeMention}@${docName} ${textAfterCursor}`;

    setInputMessage(newMessage);

    // 기존 멘션된 문서를 모두 제거하고 새로 선택된 문서만 설정
    const newDoc: MentionedDocument = {
      name: docName,
      displayName: docName
    };

    setMentionedDocs([newDoc]); // 배열을 새 문서 하나로 교체

    setShowDocSuggestions(false);
    setMentionQuery('');
    setSelectedSuggestionIndex(0);

    // 커서 위치 조정
    setTimeout(() => {
      const newCursorPos = beforeMention.length + docName.length + 2; // @ + docName + space
      textarea.setSelectionRange(newCursorPos, newCursorPos);
      textarea.focus();
    }, 0);
  };

  const removeMentionedDoc = (docName: string) => {
    console.log(`🗑️ 멘션된 문서 제거: ${docName}`);
    setMentionedDocs(prev => prev.filter(doc => doc.name !== docName));

    // 입력창에서도 해당 멘션 제거 (더 정확한 패턴 사용)
    const mentionPattern = new RegExp(`@${docName.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\s*`, 'g');
    setInputMessage(prev => {
      const newMessage = prev.replace(mentionPattern, '');
      console.log(`📝 입력 메시지 업데이트: "${prev}" → "${newMessage}"`);
      return newMessage;
    });
  };

  // 문서 목록 필터링
  const filteredDocsForMention = ragFiles
    .filter(file => file.status === 'completed')
    .filter(file =>
      file.name.toLowerCase().includes(mentionQuery.toLowerCase())
    );

  // 더 이상 사용하지 않음 - 백엔드에서 처리
  // const getDocumentChunksByName = async (documentName: string) => {
  //   // 백엔드에서 처리하므로 제거됨
  // };

  // 토큰 수 추정
  const estimateTokens = (text: string) => {
    const englishChars = (text.match(/[a-zA-Z\s]/g) || []).length;
    const koreanChars = (text.match(/[가-힣]/g) || []).length;
    const otherChars = text.length - englishChars - koreanChars;

    return Math.ceil(englishChars / 4 + koreanChars / 2 + otherChars / 3);
  };

  // 토큰 제한으로 청크 선별
  const limitChunksByTokens = (chunks: any[], maxTokens: number) => {
    let totalTokens = 0;
    const limitedChunks: any[] = [];

    for (const chunk of chunks) {
      const chunkTokens = estimateTokens(chunk.content || chunk.text || '');
      if (totalTokens + chunkTokens <= maxTokens) {
        limitedChunks.push(chunk);
        totalTokens += chunkTokens;
      } else {
        break;
      }
    }

    return limitedChunks;
  };

  // AI 응답에서 존재하지 않는 이미지 참조를 제거하는 함수
  const cleanupImageReferences = async (content: string): Promise<string> => {
    // console.log('[이미지 정리] 시작, 원본 길이:', content.length);

    // 마크다운 이미지 패턴 매칭 - 더 포괄적인 패턴 사용
    const imagePattern = /!\[([^\]]*)\]\(([^)]+)\)/g;
    let cleanedContent = content;
    const matches = Array.from(content.matchAll(imagePattern));

    // console.log('[이미지 정리] 발견된 이미지 참조:', matches.length);

    for (const match of matches) {
      const [fullMatch, altText, imageSrc] = match;
      // console.log('[이미지 정리] 처리 중:', { fullMatch, altText, imageSrc });

      // 절대 URL이나 data URL은 그대로 유지
      if (imageSrc.startsWith('http://') || imageSrc.startsWith('https://') || imageSrc.startsWith('data:')) {
        // console.log('[이미지 정리] 절대 URL 또는 data URL, 유지:', imageSrc);
        continue;
      }

      // 모든 상대 경로 이미지는 제거 (존재 확인 없이)
      // console.log('[이미지 정리] 상대 경로 이미지 제거:', imageSrc);

      // 이미지 참조 제거
      if (altText && altText.trim()) {
        // alt 텍스트가 있으면 강조 텍스트로 대체
        const replacement = `**[이미지: ${altText}]**`;
        cleanedContent = cleanedContent.replace(fullMatch, replacement);
        // console.log('[이미지 정리] 텍스트로 대체:', replacement);
      } else {
        // alt 텍스트가 없으면 완전히 제거
        cleanedContent = cleanedContent.replace(fullMatch, '');
        // console.log('[이미지 정리] 완전히 제거');
      }
    }

    // 연속된 빈 줄 정리
    cleanedContent = cleanedContent.replace(/\n\s*\n\s*\n/g, '\n\n');

    // console.log('[이미지 정리] 완료, 정리된 길이:', cleanedContent.length);
    // console.log('[이미지 정리] 정리된 내용 미리보기:', cleanedContent.substring(0, 200) + '...');

    return cleanedContent;
  };

  // 메시지 액션 함수들
  const copyToClipboard = async (content: string, messageId: string) => {
    try {
      // 최신 Clipboard API 사용
      if (navigator.clipboard && window.isSecureContext) {
        await navigator.clipboard.writeText(content);
        setShowToast({ message: t('chat.copiedToClipboard', '클립보드에 복사되었습니다'), type: 'success' });
      } else {
        // 폴백: 구형 브라우저나 비보안 컨텍스트에서 사용
        const textArea = document.createElement('textarea');
        textArea.value = content;
        textArea.style.position = 'fixed';
        textArea.style.left = '-999999px';
        textArea.style.top = '-999999px';
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();

        const successful = document.execCommand('copy');
        document.body.removeChild(textArea);

        if (successful) {
          setShowToast({ message: t('chat.copiedToClipboard', '클립보드에 복사되었습니다'), type: 'success' });
        } else {
          throw new Error(t('chat.copyFailed', '복사 명령 실행 실패'));
        }
      }

      // 복사 완료 상태 표시
      setCopiedMessageId(messageId);
      setTimeout(() => setCopiedMessageId(null), 2000); // 2초 후 원래 아이콘으로 복원
      setTimeout(() => setShowToast(null), 3000);
    } catch (error) {
      console.error('클립보드 복사 실패:', error);
      setShowToast({ message: t('chat.copyFailed', '클립보드 복사에 실패했습니다. 수동으로 복사해주세요.'), type: 'error' });
      setTimeout(() => setShowToast(null), 4000);
    }
  };

  const saveToPDF = async (content: string, messageId: string) => {
    try {
      setPdfProcessingId(messageId);
      setShowToast({ message: t('chat.generatingPdf', 'PDF 생성 중...'), type: 'success' });

      // 브라우저의 인쇄 기능을 사용하여 PDF 생성
      const printWindow = window.open('', '_blank');
      if (!printWindow) {
        setPdfProcessingId(null);
        throw new Error(t('chat.popupBlocked', '팝업이 차단되었습니다. 팝업을 허용해주세요.'));
      }

      // 코드 블록 처리를 위한 함수
      const formatContent = (text: string) => {
        return text
          .replace(/```(\w+)?\n([\s\S]*?)```/g, '<div class="code-block"><pre><code>$2</code></pre></div>')
          .replace(/`([^`]+)`/g, '<code class="inline-code">$1</code>')
          .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
          .replace(/\*(.*?)\*/g, '<em>$1</em>')
          .replace(/\n/g, '<br>');
      };

      const htmlContent = `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <title>AI 응답 - ${new Date().toLocaleDateString()}</title>
          <style>
            @media print {
              body { margin: 0; }
              .no-print { display: none; }
            }
            body {
              font-family: 'Noto Sans KR', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
              line-height: 1.6;
              color: #333;
              max-width: 800px;
              margin: 0 auto;
              padding: 20px;
              background: white;
            }
            .header {
              border-bottom: 2px solid #e5e7eb;
              padding-bottom: 20px;
              margin-bottom: 30px;
            }
            .title {
              font-size: 24px;
              font-weight: bold;
              color: #1f2937;
              margin-bottom: 10px;
            }
            .meta {
              color: #6b7280;
              font-size: 14px;
            }
            .content {
              font-size: 16px;
              line-height: 1.8;
              word-wrap: break-word;
            }
            .code-block {
              background-color: #f8f9fa;
              border: 1px solid #e9ecef;
              border-radius: 8px;
              margin: 16px 0;
              overflow: hidden;
            }
            .code-block pre {
              margin: 0;
              padding: 16px;
              overflow-x: auto;
              font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
              font-size: 14px;
              line-height: 1.4;
            }
            .inline-code {
              background-color: #f3f4f6;
              padding: 2px 6px;
              border-radius: 4px;
              font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
              font-size: 14px;
            }
            .footer {
              margin-top: 40px;
              padding-top: 20px;
              border-top: 1px solid #e5e7eb;
              text-align: center;
              color: #6b7280;
              font-size: 12px;
            }
            .print-button {
              position: fixed;
              top: 20px;
              right: 20px;
              background: #3b82f6;
              color: white;
              border: none;
              padding: 10px 20px;
              border-radius: 6px;
              cursor: pointer;
              font-size: 14px;
              z-index: 1000;
            }
            .print-button:hover {
              background: #2563eb;
            }
            @media print {
              .print-button { display: none; }
            }
          </style>
        </head>
        <body>
          <button class="print-button no-print" onclick="window.print()">PDF로 저장</button>
          <div class="header">
            <div class="title">AI 응답</div>
            <div class="meta">
              생성일: ${new Date().toLocaleString('ko-KR')}
              <br>메시지 ID: ${messageId}
            </div>
          </div>
          <div class="content">${formatContent(content)}</div>
          <div class="footer">
                              Generated by AI 워크스페이스 - ${window.location.origin}
          </div>
          <script>
            // 자동으로 인쇄 대화상자 열기
            window.onload = function() {
              setTimeout(() => {
                window.print();
              }, 500);
            };
          </script>
        </body>
        </html>
      `;

      printWindow.document.write(htmlContent);
      printWindow.document.close();

      setShowToast({ message: t('chat.printWindowOpened', '인쇄 창이 열렸습니다. PDF로 저장을 선택하세요.'), type: 'success' });
      setTimeout(() => {
        setShowToast(null);
        setPdfProcessingId(null);
      }, 4000);

    } catch (error) {
      console.error('PDF 저장 실패:', error);
      setPdfProcessingId(null);
      setShowToast({ message: 'PDF 저장에 실패했습니다', type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    }
  };

  const handleLike = async (messageId: string) => {
    const isCurrentlyLiked = messageReactions[messageId]?.liked;

    // UI 상태 즉시 업데이트
    setMessageReactions(prev => ({
      ...prev,
      [messageId]: {
        ...prev[messageId],
        liked: !isCurrentlyLiked,
        disliked: false // 좋아요 누르면 싫어요 해제
      }
    }));

    // 좋아요를 새로 누른 경우에만 Langfuse에 전송
    if (!isCurrentlyLiked) {
      try {
        // 해당 메시지와 이전 사용자 메시지 찾기
        const messageIndex = currentMessages.findIndex(msg => msg.id === messageId);
        const aiMessage = currentMessages[messageIndex];

        // AI 메시지 이전에서 가장 가까운 사용자 메시지 찾기
        let userMessage = null;
        for (let i = messageIndex - 1; i >= 0; i--) {
          if (currentMessages[i] && currentMessages[i].role === 'user') {
            userMessage = currentMessages[i];
            break;
          }
        }

        // console.log('좋아요 클릭 - 디버그 정보:', {
        //   messageId,
        //   currentSessionId,
        //   messageIndex,
        //   aiMessage: aiMessage ? { id: aiMessage.id, role: aiMessage.role, contentLength: aiMessage.content.length } : null,
        //   userMessage: userMessage ? { id: userMessage.id, role: userMessage.role, contentLength: userMessage.content.length } : null,
        //   totalMessages: currentMessages.length
        // });

        if (aiMessage && aiMessage.role === 'assistant' && userMessage && userMessage.role === 'user') {
          const response = await externalApiClient.sendFeedbackToLangfuse({
            messageId: messageId,
            sessionId: currentSessionId || 'unknown',
            rating: 5, // 좋아요는 5점
            userMessage: userMessage.content,
            aiResponse: aiMessage.content,
            timestamp: new Date().toISOString()
          });

          // 학습 결과가 있으면 표시
          if (response.learning) {
            const learningMessage = response.learning.message;
            const suggestionsCount = response.learning.suggestionsGenerated;
            const mode = response.learning.mode;
            const cached = response.learning.cached;

            let statusIcon = '';
            let additionalInfo = '';

            switch (mode) {
              case 'cloud':
                statusIcon = '☁️';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 패턴 학습됨)` : '';
                break;
              case 'local':
                statusIcon = '📦';
                additionalInfo = `(로컬 모드${cached ? ', 캐시됨' : ''})`;
                break;
              case 'fallback':
                statusIcon = '⚠️';
                additionalInfo = '(제한된 기능)';
                break;
              case 'basic':
                statusIcon = '💾';
                additionalInfo = '(기본 저장)';
                break;
              default:
                statusIcon = '✅';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 패턴 학습됨)` : '';
            }

            setShowToast({
              message: `${statusIcon} ${learningMessage} ${additionalInfo}`,
              type: 'success'
            });
            // console.log('피드백 학습 결과:', response.learning);
          } else {
            setShowToast({ message: t('chat.feedbackSaved', '피드백이 저장되었습니다'), type: 'success' });
          }
          setTimeout(() => setShowToast(null), 4000);
        } else {
          // 메시지를 찾을 수 없는 경우 UI 상태만 업데이트하고 조용히 처리
          console.warn('메시지 쌍을 찾을 수 없어 Langfuse 전송을 건너뜁니다:', {
            messageId,
            aiMessage: !!aiMessage,
            userMessage: !!userMessage,
            userMessageRole: userMessage?.role
          });
          setShowToast({ message: t('chat.likeMarked', '좋아요가 표시되었습니다'), type: 'success' });
          setTimeout(() => setShowToast(null), 2000);
        }
      } catch (error) {
        console.error('피드백 전송 실패:', error);
        setShowToast({ message: t('chat.feedbackSaveFailed', '피드백 저장에 실패했습니다'), type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
      }
    }
  };

  const handleDislike = async (messageId: string) => {
    const isCurrentlyDisliked = messageReactions[messageId]?.disliked;

    // UI 상태 즉시 업데이트
    setMessageReactions(prev => ({
      ...prev,
      [messageId]: {
        ...prev[messageId],
        disliked: !isCurrentlyDisliked,
        liked: false // 싫어요 누르면 좋아요 해제
      }
    }));

    // 싫어요를 새로 누른 경우에만 Langfuse에 전송
    if (!isCurrentlyDisliked) {
      try {
        // 해당 메시지와 이전 사용자 메시지 찾기
        const messageIndex = currentMessages.findIndex(msg => msg.id === messageId);
        const aiMessage = currentMessages[messageIndex];

        // AI 메시지 이전에서 가장 가까운 사용자 메시지 찾기
        let userMessage = null;
        for (let i = messageIndex - 1; i >= 0; i--) {
          if (currentMessages[i] && currentMessages[i].role === 'user') {
            userMessage = currentMessages[i];
            break;
          }
        }

        // console.log('싫어요 클릭 - 디버그 정보:', {
        //   messageId,
        //   currentSessionId,
        //   messageIndex,
        //   aiMessage: aiMessage ? { id: aiMessage.id, role: aiMessage.role, contentLength: aiMessage.content.length } : null,
        //   userMessage: userMessage ? { id: userMessage.id, role: userMessage.role, contentLength: userMessage.content.length } : null,
        //   totalMessages: currentMessages.length
        // });

        if (aiMessage && aiMessage.role === 'assistant' && userMessage && userMessage.role === 'user') {
          const response = await externalApiClient.sendFeedbackToLangfuse({
            messageId: messageId,
            sessionId: currentSessionId || 'unknown',
            rating: 1, // 싫어요는 1점
            userMessage: userMessage.content,
            aiResponse: aiMessage.content,
            timestamp: new Date().toISOString()
          });

          // 학습 결과가 있으면 표시
          if (response.learning) {
            const learningMessage = response.learning.message;
            const suggestionsCount = response.learning.suggestionsGenerated;
            const mode = response.learning.mode;
            const cached = response.learning.cached;

            let statusIcon = '';
            let additionalInfo = '';

            switch (mode) {
              case 'cloud':
                statusIcon = '☁️';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 개선점 분석됨)` : '';
                break;
              case 'local':
                statusIcon = '📦';
                additionalInfo = `(로컬 모드${cached ? ', 캐시됨' : ''})`;
                break;
              case 'fallback':
                statusIcon = '⚠️';
                additionalInfo = '(제한된 기능)';
                break;
              case 'basic':
                statusIcon = '💾';
                additionalInfo = '(기본 저장)';
                break;
              default:
                statusIcon = '✅';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 개선점 분석됨)` : '';
            }

            setShowToast({
              message: `${statusIcon} ${learningMessage} ${additionalInfo}`,
              type: 'success'
            });
            // console.log('피드백 학습 결과:', response.learning);
          } else {
            setShowToast({ message: t('chat.feedbackSaved', '피드백이 저장되었습니다'), type: 'success' });
          }
          setTimeout(() => setShowToast(null), 4000);
        } else {
          // 메시지를 찾을 수 없는 경우 UI 상태만 업데이트하고 조용히 처리
          console.warn('메시지 쌍을 찾을 수 없어 Langfuse 전송을 건너뜁니다:', {
            messageId,
            aiMessage: !!aiMessage,
            userMessage: !!userMessage,
            userMessageRole: userMessage?.role
          });
          setShowToast({ message: t('chat.dislikeMarked', '싫어요가 표시되었습니다'), type: 'success' });
          setTimeout(() => setShowToast(null), 2000);
        }
      } catch (error) {
        console.error('피드백 전송 실패:', error);
        setShowToast({ message: t('chat.feedbackSaveFailed', '피드백 저장에 실패했습니다'), type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
      }
    }
  };

  // 파일 유형별 아이콘 반환
  const getFileIcon = (fileName: string, status: string) => {
    const extension = fileName.split('.').pop()?.toLowerCase() || '';
    const iconClass = "w-4 h-4 flex-shrink-0";

    // 디버그 로그: 아이콘 렌더링 확인
    console.log(`[File Icon] ${fileName}: status=${status}`);

    // 상태별 아이콘 (업로딩/처리 중)
    if (status === 'uploading') {
      return (
        <div className="w-4 h-4 flex-shrink-0">
          <div className="animate-spin rounded-full h-4 w-4 border-2 border-blue-500 border-t-transparent"></div>
        </div>
      );
    }

    if (status === 'processing') {
      return (
        <div className="w-4 h-4 flex-shrink-0">
          <div className="animate-pulse rounded-full h-4 w-4 bg-yellow-500"></div>
        </div>
      );
    }

    if (status === 'pending') {
      return (
        <div className="w-4 h-4 flex-shrink-0">
          <div className="animate-pulse rounded-full h-4 w-4 bg-orange-400"></div>
        </div>
      );
    }

    if (status === 'error') {
      return (
        <svg className={`${iconClass} text-red-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
        </svg>
      );
    }

    // 파일 유형별 아이콘
    switch (extension) {
      case 'pdf':
        return (
          <svg className={`${iconClass} text-red-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M8.267 14.68c-.184 0-.308.018-.372.036v1.178c.076.018.171.023.302.023.479 0 .774-.242.774-.651 0-.366-.254-.586-.704-.586zm3.487.012c-.2 0-.33.018-.407.036v2.61c.077.018.201.018.313.018.817.006 1.349-.444 1.349-1.396.006-.83-.479-1.268-1.255-1.268z" />
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM9.498 16.19c-.309.29-.765.42-1.296.42a2.23 2.23 0 0 1-.308-.018v.426H7v-3.936A7.558 7.558 0 0 1 8.219 14c.557 0 .953.106 1.22.319.254.202.426.533.426.923-.001.392-.131.723-.367.948zm3.807 1.355c-.42.349-1.059.515-1.84.515-.468 0-.799-.03-1.024-.06v-3.917A7.947 7.947 0 0 1 11.66 14c.757 0 1.249.136 1.633.426.415.308.675.799.675 1.504 0 .763-.279 1.29-.663 1.615zM17 14.77h-1.532v.911H16.9v.734h-1.432v1.604h-.906V14.03H17v.74z" />
          </svg>
        );
      case 'doc':
      case 'docx':
        return (
          <svg className={`${iconClass} text-blue-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM16 18H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z" />
          </svg>
        );
      case 'xls':
      case 'xlsx':
        return (
          <svg className={`${iconClass} text-green-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM12 18H8v-2h4v2zm4-4H8v-2h8v2zm0-4H8V8h8v2zm-3-5V3.5L18.5 9H13z" />
          </svg>
        );
      case 'ppt':
      case 'pptx':
        return (
          <svg className={`${iconClass} text-orange-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM11 15H9v3H7v-7h4c1.1 0 2 .9 2 2s-.9 2-2 2zm2-6V3.5L18.5 9H13z" />
            <path d="M9 11v2h2c.6 0 1-.4 1-1s-.4-1-1-1H9z" />
          </svg>
        );
      case 'txt':
      case 'md':
        return (
          <svg className={`${iconClass} text-gray-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
        );
      case 'jpg':
      case 'jpeg':
      case 'png':
      case 'gif':
      case 'bmp':
      case 'svg':
        return (
          <svg className={`${iconClass} text-purple-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
          </svg>
        );
      case 'zip':
      case 'rar':
      case '7z':
      case 'tar':
      case 'gz':
        return (
          <svg className={`${iconClass} text-yellow-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10l-8 4" />
          </svg>
        );
      case 'mp4':
      case 'avi':
      case 'mov':
      case 'wmv':
      case 'flv':
        return (
          <svg className={`${iconClass} text-red-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z" />
          </svg>
        );
      case 'mp3':
      case 'wav':
      case 'flac':
      case 'aac':
        return (
          <svg className={`${iconClass} text-green-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3" />
          </svg>
        );
      default:
        return (
          <svg className={`${iconClass} text-gray-400`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
        );
    }
  };

  // 인증 초기화 중인 경우
  if (!isInitialized) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div
            className="animate-spin rounded-full h-8 w-8 mx-auto mb-4"
            style={{
              border: '2px solid #e5e7eb',
              borderBottomColor: '#ea580c'
            }}
          ></div>
          <p className="text-gray-600 dark:text-gray-400">{t('chat.loading', '로딩 중...')}</p>
        </div>
      </div>
    );
  }

  // 인증되지 않은 경우 (리다이렉트 처리됨)
  if (!isAuthenticated) {
    return null;
  }

  return (
    <ClientOnly fallback={
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div
            className="animate-spin rounded-full h-8 w-8 mx-auto mb-4"
            style={{
              border: '2px solid #e5e7eb',
              borderBottomColor: '#ea580c'
            }}
          ></div>
          <p className="text-gray-600 dark:text-gray-400">{t('chat.loading', '로딩 중...')}</p>
        </div>
      </div>
    }>
      <div className="h-full flex" style={{ backgroundColor: 'var(--body-bg)' }}>
        {/* 사이드바 */}
        <div className={`${showSidebar ? 'w-80' : 'w-0'} h-full transition-all duration-300 border-r border-gray-200 dark:border-gray-700 flex flex-col overflow-hidden`} style={{ backgroundColor: 'var(--sidebar-bg)' }}>
          {showSidebar && (
            <>
              {/* 폴딩 메뉴 - 최상단 */}
              <div className="p-3 border-b border-gray-200 dark:border-gray-700">
                <div className="flex items-center justify-start">
                  <button
                    onClick={() => setShowSidebar(false)}
                    className="p-1.5 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md"
                    title={t('chat.closeSidebar', '사이드바 닫기')}
                  >
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                    </svg>
                  </button>
                </div>
              </div>

              {/* 컨테이너 0: 프로젝트 - 프로젝트별 RAG 검색 기능이 구현되지 않아 임시 비활성화 */}
              {/*
              <div className="border-b border-gray-200 dark:border-gray-700 p-2">
                <div className="p-3" style={{ backgroundColor: 'var(--sidebar-bg)', borderRadius: '8px' }}>
                  <div className="flex items-center justify-between mb-2">
                    <button
                      onClick={() => setShowProjectBrowser(!showProjectBrowser)}
                      className="flex items-center space-x-2 text-left hover:bg-gray-100 dark:hover:bg-gray-700 rounded p-1 transition-colors"
                    >
                      <svg className="w-4 h-4 text-gray-600 dark:text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                      </svg>
                      <span className="text-base font-medium text-gray-900 dark:text-white">{t('chat.projects', '프로젝트')}</span>
                    </button>
                    <div className="flex items-center space-x-1">
                      <button
                        onClick={() => {
                          setProjectModalType('create');
                          setNewProjectName('');
                          setNewProjectDescription('');
                          setShowProjectModal(true);
                        }}
                        className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors"
                        title="새 프로젝트 만들기"
                      >
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
                        </svg>
                      </button>
                      <button
                        onClick={() => setShowProjectBrowser(!showProjectBrowser)}
                        className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors"
                      >
                        <svg className={`w-4 h-4 transition-transform ${showProjectBrowser ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                        </svg>
                      </button>
                    </div>
                  </div>

                  {showProjectBrowser && (
                    <div className="space-y-3">
                      {selectedProject && (
                        <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-2 mb-3">
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-2">
                              <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                              </svg>
                              <span className="text-sm font-medium text-blue-900 dark:text-blue-100">{selectedProject.name}</span>
                            </div>
                            <button
                              onClick={() => setSelectedProject(null)}
                              className="p-1 text-blue-500 hover:text-blue-700 dark:text-blue-400 dark:hover:text-blue-200 hover:bg-blue-100 dark:hover:bg-blue-800 rounded transition-colors"
                              title="프로젝트 선택 해제"
                            >
                              <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                              </svg>
                            </button>
                          </div>
                        </div>
                      )}

                      <div className="max-h-40 overflow-y-auto">
                        {isLoadingProjects ? (
                          <div className="flex justify-center items-center h-20">
                            <div className="relative">
                              <div
                                className="animate-spin rounded-full h-4 w-4 dark:hidden"
                                style={{
                                  border: '2px solid #e5e7eb',
                                  borderBottomColor: '#ea580c'
                                }}
                              ></div>
                              <div
                                className="animate-spin rounded-full h-4 w-4 hidden dark:block"
                                style={{
                                  border: '2px solid #4b5563',
                                  borderBottomColor: '#fb923c'
                                }}
                              ></div>
                            </div>
                          </div>
                        ) : projects.length === 0 ? (
                          <div className="text-center text-xs text-gray-500 dark:text-gray-400 py-4">
                            {t('chat.noProjects', '생성된 프로젝트가 없습니다')}
                          </div>
                        ) : (
                          <div className="space-y-1 text-sm">
                            {projects.map((project) => (
                              <div
                                key={project.id}
                                className={`group flex items-center justify-between p-2 rounded-md transition-colors cursor-pointer ${
                                  selectedProject?.id === project.id
                                    ? 'bg-blue-100 dark:bg-blue-900/40'
                                    : 'hover:bg-gray-100 dark:hover:bg-gray-700'
                                }`}
                                onClick={() => setSelectedProject(project)}
                              >
                                <div className="flex items-center space-x-2 flex-1 min-w-0">
                                  <svg className="w-4 h-4 text-gray-600 dark:text-gray-400 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                                  </svg>
                                  <div className="flex-1 min-w-0">
                                    <p className="text-gray-700 dark:text-gray-300 truncate font-medium" title={project.name}>
                                      {project.name}
                                    </p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                      {project.document_count}개 문서
                                    </p>
                                  </div>
                                </div>
                                <button
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    setProjectToEdit(project);
                                    setProjectModalType('delete');
                                    setShowProjectModal(true);
                                  }}
                                  className="opacity-0 group-hover:opacity-100 p-1 text-red-500 hover:text-red-700 dark:text-red-400 dark:hover:text-red-200 hover:bg-red-100 dark:hover:bg-red-900/40 rounded transition-all"
                                  title="프로젝트 삭제"
                                >
                                  <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                  </svg>
                                </button>
                              </div>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>
                  )}
                </div>
              </div>
              */}

              {/* 컨테이너 1: 내 문서 */}
              <div className="border-b border-gray-200 dark:border-gray-700 p-2">
                <div className="p-3" style={{ backgroundColor: 'var(--sidebar-bg)', borderRadius: '8px' }}>
                  <div className="flex items-center justify-between mb-2">
                    <button
                      onClick={() => setShowRagBrowser(!showRagBrowser)}
                      className="flex items-center space-x-2 text-left hover:bg-gray-100 dark:hover:bg-gray-700 rounded p-1 transition-colors"
                    >
                      <svg className="w-4 h-4 text-gray-600 dark:text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                      </svg>
                      <span className="text-base font-medium text-gray-900 dark:text-white">
                        {t('chat.myDocuments', '내 문서')}
                        {/* 프로젝트 기능 임시 비활성화
                        {selectedProject && (
                          <span className="text-sm text-blue-600 dark:text-blue-400 ml-1">
                            ({selectedProject.name})
                          </span>
                        )}
                        */}
                      </span>
                    </button>
                    <div className="flex items-center space-x-1">
                      {/* <button
                        onClick={() => setShowFileManagementModal(true)}
                        className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors"
                        title={t('chat.fileManagement', '파일관리')}
                      >
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4" />
                        </svg>
                      </button> */}
                      <button
                        onClick={() => setShowRagBrowser(!showRagBrowser)}
                        className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors"
                      >
                        <svg className={`w-4 h-4 transition-transform ${showRagBrowser ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                        </svg>
                      </button>
                    </div>
                  </div>

                  {showRagBrowser && (
                    <div className="space-y-3">

                      {/* 백그라운드 작업 상태 */}
                      <div className="mb-3">
                        <BackgroundJobStatus
                          userId={user?.username}
                          onJobComplete={handleJobComplete}
                          refreshTrigger={refreshTrigger}
                        />
                      </div>

                      {/* 파일 목록 */}
                      <div className="max-h-60 min-h-40 overflow-y-auto">
                        {isLoadingRagFiles ? (
                          <div className="flex justify-center items-center h-20">
                            <div className="relative">
                              <div
                                className="animate-spin rounded-full h-4 w-4 dark:hidden"
                                style={{
                                  border: '2px solid #e5e7eb',
                                  borderBottomColor: '#ea580c'
                                }}
                              ></div>
                              <div
                                className="animate-spin rounded-full h-4 w-4 hidden dark:block"
                                style={{
                                  border: '2px solid #4b5563',
                                  borderBottomColor: '#fb923c'
                                }}
                              ></div>
                            </div>
                          </div>
                        ) : ragFiles.length === 0 ? (
                          <div className="text-center text-xs text-gray-500 dark:text-gray-400 py-4">
                            {t('chat.noDocuments', '업로드된 문서가 없습니다')}
                          </div>
                        ) : (
                          <div className="space-y-1 text-sm">
                            {ragFiles.map((file, index) => (
                              <div
                                key={`${file.name}-${index}`}
                                className="group flex flex-col p-0.5 rounded-md hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors cursor-pointer"
                                onClick={() => {
                                  if (file.status === 'completed') {
                                    setPreviewFileName(file.name);
                                    setShowDocPreview(true);
                                  }
                                }}
                              >
                                <div className="flex items-center space-x-2">
                                  {/* 파일 유형별 아이콘 */}
                                  {getFileIcon(file.name, file.status)}

                                  <div className="flex-1 min-w-0">
                                    <div className="flex items-center space-x-1">
                                      <p
                                        className="file-browser-filename text-gray-700 dark:text-gray-300 truncate flex-1"
                                        title={file.name}
                                      >
                                        {file.name}
                                      </p>
                                    </div>
                                  </div>

                                  {/* 삭제 버튼 (완료된 파일만) */}
                                  {file.status === 'completed' && (
                                    <button
                                      onClick={(e) => {
                                        e.preventDefault();
                                        e.stopPropagation();
                                        deleteRagFile(file.name);
                                      }}
                                      className="opacity-0 group-hover:opacity-100 p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-opacity flex-shrink-0 ml-auto"
                                      title={t('chat.deleteFile', '파일 삭제')}
                                    >
                                      <svg className="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                      </svg>
                                    </button>
                                  )}
                                </div>

                              </div>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>
                  )}
                </div>
              </div>

              {/* 새 대화 버튼 */}
              <div className="p-3 border-b border-gray-200 dark:border-gray-700">
                <button
                  onClick={handleNewChatClick}
                  className="w-full px-3 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white text-sm font-medium rounded-lg transition-colors flex items-center justify-center"
                >
                  <span>{t('chat.newChat', 'New Chat')}</span>
                </button>
              </div>

              {/* 컨테이너 2: 대화목록 */}
              <div className="flex flex-col border-b border-gray-200 dark:border-gray-700 flex-1 min-h-0 p-2">
                <div className="p-3 border-b border-gray-200 dark:border-gray-700 flex-shrink-0" style={{ backgroundColor: 'var(--sidebar-bg)', borderRadius: '8px' }}>
                  <div className="flex items-center justify-between mb-2">
                    <div className="flex items-center space-x-2">
                      <svg className="w-4 h-4 text-gray-600 dark:text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z" />
                      </svg>
                      <span className="text-base font-medium text-gray-900 dark:text-white">{t('chat.chatSessions', '대화목록')}</span>
                    </div>
                    {sessions.length > 0 && (
                      <button
                        onClick={handleDeleteAllClick}
                        className="p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-colors"
                        title={t('chat.deleteAllSessions', '모든 대화 삭제')}
                      >
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                        </svg>
                      </button>
                    )}
                  </div>

                  {/* 검색 */}
                  <input
                    type="text"
                    placeholder=""
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full px-2 py-1 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    style={{ fontSize: '11px' }}
                  />
                </div>

                {/* 세션 목록 */}
                <div className="flex-1 overflow-y-auto min-h-0 p-3" style={{ backgroundColor: 'var(--sidebar-bg)', borderRadius: '8px' }}>
                  {isLoadingSessions ? (
                    <div className="flex justify-center items-center h-32">
                      <div className="relative">
                        <div
                          className="animate-spin rounded-full h-6 w-6 dark:hidden"
                          style={{
                            border: '2px solid #e5e7eb',
                            borderBottomColor: '#ea580c'
                          }}
                        ></div>
                        <div
                          className="animate-spin rounded-full h-6 w-6 hidden dark:block"
                          style={{
                            border: '2px solid #4b5563',
                            borderBottomColor: '#fb923c'
                          }}
                        ></div>
                      </div>
                    </div>
                  ) : filteredSessions.length === 0 ? (
                    <div className="p-3 text-center text-xs text-gray-500 dark:text-gray-400">
                      {searchTerm ? t('chat.noResults', '검색 결과가 없습니다') : t('chat.noConversations', '대화가 없습니다')}
                    </div>
                  ) : (
                    <div className="space-y-0.5">
                      {filteredSessions.map((session) => (
                        <div
                          key={session.id}
                          onClick={async () => await selectSession(session.id)}
                          className={`group flex items-center justify-between px-2 py-2 rounded-md cursor-pointer transition-colors ${session.id === currentSessionId
                            ? 'bg-gray-200 dark:bg-gray-700'
                            : 'hover:bg-gray-100 dark:hover:bg-gray-800'
                            }`}
                        >
                          <div className="flex-1 min-w-0 mr-2">
                            <p className="text-sm text-gray-700 dark:text-gray-300 truncate chat-sidebar-list-text">
                              {session.title || session.lastMessage}
                            </p>
                          </div>
                          <button
                            onClick={(e) => {
                              e.stopPropagation();
                              handleDeleteClick(session.id);
                            }}
                            className="opacity-0 group-hover:opacity-100 p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-opacity flex-shrink-0"
                          >
                            <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                            </svg>
                          </button>
                        </div>
                      ))}

                      {/* 무한 스크롤 트리거 */}
                      {!searchTerm && hasMoreSessions && (
                        <div ref={infiniteScrollRef} className="h-4 flex items-center justify-center">
                          {isInfiniteLoading && (
                            <div className="text-xs text-gray-500 dark:text-gray-400">
                              로딩중...
                            </div>
                          )}
                        </div>
                      )}
                    </div>
                  )}
                </div>
              </div>

              {/* 컨테이너 3: Settings */}
              <div className="relative p-2 space-y-3 flex-shrink-0">
                <div className="flex items-center justify-between p-3" style={{ backgroundColor: 'var(--body-bg)', borderRadius: '8px' }}>
                  <button
                    onClick={() => setShowOptions(!showOptions)}
                    className="flex items-center space-x-2 text-left hover:bg-gray-100 dark:hover:bg-gray-700 rounded p-1 transition-colors"
                  >
                    <svg className="w-4 h-4 text-gray-600 dark:text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                    </svg>
                    <span className="text-sm font-medium text-gray-900 dark:text-white">{t('header.settings', 'Settings')}</span>
                  </button>
                  <button
                    onClick={() => setShowOptions(!showOptions)}
                    className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors"
                  >
                    <svg className={`w-4 h-4 transition-transform ${showOptions ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                    </svg>
                  </button>
                </div>

                {showOptions && (
                  <div className="absolute bottom-full left-2 right-2 mb-2 border border-gray-200 dark:border-gray-700 rounded-lg shadow-lg p-4 space-y-4 z-10" style={{ backgroundColor: 'var(--card-bg)' }}>
                    {/* 프로바이더 선택 */}
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Provider
                      </label>
                      {isLoadingProviders ? (
                        <div className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-gray-100 dark:bg-gray-600 text-gray-500 dark:text-gray-400">
                          Loading...
                        </div>
                      ) : (
                        <select
                          value={options.provider}
                          onChange={(e) => handleProviderChange(e.target.value)}
                          className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                        >
                          {availableProviders.map((provider) => (
                            <option key={provider.key} value={provider.key}>
                              {provider.name}
                              {provider.requiresApiKey && !providers[provider.key]?.apiKeyConfigured && ' (API 키 필요)'}
                            </option>
                          ))}
                        </select>
                      )}
                    </div>

                    {/* 모델 선택 */}
                    <div>
                      <div className="flex items-center justify-between mb-2">
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                          Model
                        </label>
                        {(options.provider === 'ollama' || !providers[options.provider]?.models?.length) && (
                          <button
                            onClick={() => handleProviderChange(options.provider)}
                            className="text-xs text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 flex items-center gap-1"
                            title={t('chat.reloadModelList', '모델 목록을 다시 불러옵니다')}
                          >
                            <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                            </svg>
                            새로고침
                          </button>
                        )}
                      </div>
                      <select
                        value={options.model}
                        onChange={(e) => {
                          const newModel = e.target.value;
                          // 로컬 스토리지에 선택한 모델 저장
                          localStorage.setItem('airun-selected-model', newModel);
                          setOptions(prev => ({ ...prev, model: newModel }));
                        }}
                        className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                        disabled={!providers[options.provider]?.models?.length}
                      >
                        {providers[options.provider]?.models?.length > 0 ? (
                          providers[options.provider].models.map((model: any) => (
                            <option key={model.id} value={model.id}>
                              {model.name}
                              {model.size && ` (${(model.size / 1024 / 1024 / 1024).toFixed(1)}GB)`}
                            </option>
                          ))
                        ) : options.provider === 'ollama' ? (
                          <option value="hamonize:latest">hamonize:latest (기본 모델)</option>
                        ) : (
                          <option value="">
                            {isLoadingProviders ? 'Loading...' : 'No models available'}
                          </option>
                        )}
                      </select>

                      {/* API 키 상태 표시 */}
                      {providers[options.provider]?.requiresApiKey && (
                        <div className="mt-2 p-2 rounded-md bg-gray-50 dark:bg-gray-800">
                          {providers[options.provider]?.apiKeyConfigured ? (
                            <div className="flex items-center text-sm text-green-600 dark:text-green-400">
                              <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                              </svg>
                              API Key Set
                            </div>
                          ) : (
                            <div className="flex items-center text-sm text-yellow-600 dark:text-yellow-400">
                              <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                              </svg>
                              API Key Required
                            </div>
                          )}
                        </div>
                      )}
                    </div>

                    {/* Temperature 설정 */}
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Temperature: {options.temperature}
                      </label>
                      <input
                        type="range"
                        min="0"
                        max="2"
                        step="0.1"
                        value={options.temperature}
                        onChange={(e) => setOptions(prev => ({ ...prev, temperature: parseFloat(e.target.value) }))}
                        className="w-full h-2 bg-gray-200 dark:bg-gray-600 rounded-lg appearance-none cursor-pointer slider"
                      />
                      <div className="flex justify-between mt-1">
                        <span className="temperature-label text-gray-500 dark:text-gray-400">{t('chat.accurate', '정확')}</span>
                        <span className="temperature-label text-gray-500 dark:text-gray-400">{t('chat.creative', '창의적')}</span>
                      </div>
                    </div>

                    {/* Reasoning Level 설정 - hamonize 모델일 때만 표시 */}
                    {options.model === 'hamonize:latest' && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                          Reasoning Level: {options.reasoningLevel}
                        </label>
                        <select
                          value={options.reasoningLevel}
                          onChange={(e) => setOptions(prev => ({ ...prev, reasoningLevel: e.target.value as 'low' | 'medium' | 'high' }))}
                          className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:outline-none focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400"
                        >
                          
                          <option value="low">Low (빠른 추론)</option>
                          <option value="medium">Medium (균형 잡힌 추론)</option>
                          <option value="high">High (신중한 추론)</option>
                        </select>
                        <div className="mt-1">
                          <span className="text-xs text-gray-500 dark:text-gray-400">
                            {options.reasoningLevel === 'low' && '빠르고 간결한 추론 과정'}
                            {options.reasoningLevel === 'medium' && '적절한 수준의 추론 과정'}
                            {options.reasoningLevel === 'high' && '상세하고 신중한 추론 과정'}
                          </span>
                        </div>
                      </div>
                    )}

                    {/* 시스템 프롬프트 섹션 */}
                    <div className="border-t border-gray-200 dark:border-gray-600 pt-4">
                      <div className="flex items-center justify-between mb-2">
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                          시스템 프롬프트
                        </label>
                        <button
                          onClick={() => {
                            fetchSystemPrompt();
                            setShowSystemPromptModal(true);
                          }}
                          className="text-xs text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                        >
                          편집하기
                        </button>
                      </div>
                    </div>

                  </div>
                )}
              </div>
            </>
          )}
        </div>

        {/* 메인 채팅 영역 */}
        <div className="flex-1 flex flex-col" style={{ backgroundColor: 'var(--body-bg)' }}>
          {/* 채팅 헤더 */}
                        <div className="border-b border-gray-200 dark:border-gray-700 px-4 py-3" style={{ backgroundColor: 'var(--body-bg)' }}>
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-3">
                {!showSidebar && (
                  <button
                    onClick={() => setShowSidebar(true)}
                    className="p-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md"
                    title="사이드바 열기"
                  >
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                    </svg>
                  </button>
                )}

                {/* 현재 모델 정보 표시 */}
                <div className="flex items-center space-x-2 text-sm text-gray-600 dark:text-gray-400">
                  <div className="flex items-center space-x-1">
                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19.428 15.428a2 2 0 00-1.022-.547l-2.387-.477a6 6 0 00-3.86.517l-.318.158a6 6 0 01-3.86.517L6.05 15.21a2 2 0 00-1.806.547M8 4h8l-1 1v5.172a2 2 0 00.586 1.414l5 5c1.26 1.26.367 3.414-1.415 3.414H4.828c-1.782 0-2.674-2.154-1.414-3.414l5-5A2 2 0 009 10.172V5L8 4z" />
                    </svg>
                    <span>{providers[options.provider]?.models?.find((m: any) => m.id === options.model)?.name || options.model}</span>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* 채팅 컨텐츠 영역 */}
          <div className="flex-1 flex flex-col min-h-0">
            {currentMessages.length === 0 ? (
              /* 메시지가 없는 화면 - 중앙 입력창 */
              <div className="flex-1 flex items-center justify-center p-4">
                <div className="w-full max-w-4xl">
                  <div className="text-center mb-8">
                    <h2 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
                      How can I help?
                    </h2>
                    <p className="text-gray-600 dark:text-gray-400">
                      {t('chat.welcomeMessage', '아이디어부터 문제 해결까지, AI와 함께 시작해보세요! ✨')}
                    </p>
                  </div>

                  {/* 기능 안내 카드들 - 3x2 배치 (6개 카드) */}
                  <div className="hidden md:grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mb-8">
                    {/* 지식검색 카드 */}
                    <button
                      onClick={() => {
                        setInputMessage(t('chat.knowledgeSearchPrompt', '문서에서 소버린 AI 구축 로드맵을 찾아서 알려주세요'));
                        setOptions(prev => ({
                          ...prev,
                          enableRag: true,
                          ragSearchScope: 'all',
                          enableWebSearch: false,
                        }));
                      }}
                      className="p-6 bg-gradient-to-br from-blue-50 to-blue-100 dark:from-blue-900/20 dark:to-blue-800/20 border border-blue-200 dark:border-blue-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                    >
                      <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">📋</div>
                      <h4 className="font-semibold text-gray-900 dark:text-white mb-2">{t('chat.knowledgeSearch', '지식검색')}</h4>
                      <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                        {t('chat.documentSearchDescription', '업로드된 문서에서 정보를 검색하여 정확한 답변을 제공합니다')}
                      </p>
                    </button>

                    {/* 웹검색 카드 */}
                    <button
                      onClick={() => {
                        setInputMessage(t('chat.webSearchPrompt', '최근 인공지능 에이전트 기술동향을 웹검색으로 찾아서 알려주세요'));
                        setOptions(prev => ({
                          ...prev,
                          enableWebSearch: true,
                          enableRag: false,
                        }));
                      }}
                      className="p-6 bg-gradient-to-br from-orange-50 to-orange-100 dark:from-orange-900/20 dark:to-orange-800/20 border border-orange-200 dark:border-orange-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                    >
                      <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">🌐</div>
                      <h4 className="font-semibold text-gray-900 dark:text-white mb-2">{t('chat.webSearch', '웹검색')}</h4>
                      <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                        {t('chat.webSearchDescription', '실시간 웹 검색으로 최신 뉴스, 시장 동향, 기술 정보를 제공합니다')}
                      </p>
                    </button>

                    {/* 이미지 분석 카드 */}
                    <button
                      onClick={() => {
                        setInputMessage(t('chat.imageAnalysisPrompt', '이미지 내용을 분석해주세요'));
                        setOptions(prev => ({
                          ...prev,
                          enableRag: false,
                          enableWebSearch: false,
                        }));
                      }}
                      className="p-6 bg-gradient-to-br from-purple-50 to-purple-100 dark:from-purple-900/20 dark:to-purple-800/20 border border-purple-200 dark:border-purple-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                    >
                      <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">🖼️</div>
                      <h4 className="font-semibold text-gray-900 dark:text-white mb-2">{t('chat.imageAnalysis', '이미지 분석')}</h4>
                      <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                        {t('chat.fileAnalysisDescription', '이미지, 문서, PDF 등 다양한 파일을 분석하고 내용을 이해합니다')}
                      </p>
                    </button>

                    {/* 건강검진 분석 카드 */}
                    <button
                      onClick={() => {
                        setInputMessage(t('chat.healthAnalysisPrompt', '"신동욱" 환자의 건강검진 분석을 수행해주세요.'));
                        setOptions(prev => ({
                          ...prev,
                          enableHealthAnalysis: true,
                          enableRag: false,
                          enableWebSearch: false,
                        }));
                      }}
                      className="p-6 bg-gradient-to-br from-green-50 to-green-100 dark:from-green-900/20 dark:to-green-800/20 border border-green-200 dark:border-green-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                    >
                      <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">🏥</div>
                      <h4 className="font-semibold text-gray-900 dark:text-white mb-2">{t('chat.healthAnalysis', '건강검진 분석')}</h4>
                      <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                        {t('chat.healthAnalysisDescription', '의료 데이터를 분석하여 건강 상태를 평가하고 맞춤형 건강 조언을 제공합니다')}
                      </p>
                    </button>

                    {/* 민원신청서 카드 */}
                    <button
                      onClick={() => {
                        setInputMessage('아래 내용으로 민원신청서를 작성해주세요.\n민원인: 김철수\n주소: 부산시 해운대구 센텀중앙로 97\n전화: 010-9876-5432\n민원내용: 공원 시설 개선 요청');
                        setOptions(prev => ({
                          ...prev,
                          enableRag: false,
                          enableWebSearch: false,
                          enableHealthAnalysis: false,
                        }));
                      }}
                      className="p-6 bg-gradient-to-br from-indigo-50 to-indigo-100 dark:from-indigo-900/20 dark:to-indigo-800/20 border border-indigo-200 dark:border-indigo-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                    >
                      <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">📋</div>
                      <h4 className="font-semibold text-gray-900 dark:text-white mb-2">{t('chat.civilPetition', '민원신청서')}</h4>
                      <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                        {t('chat.civilPetitionDescription', '공공기관에 제출할 민원신청서를 자동으로 작성합니다')}
                      </p>
                    </button>

                    {/* 공문작성 카드 */}
                    <button
                      onClick={() => {
                        setInputMessage('아래 내용으로 공문을 작성해주세요.\n신청자: 김철수\n주소: 서울시 강남구\n연락처: 010-1234-5678\n제목: 시설 개선 요청\n내용: 공공시설의 안전성 개선을 위한 예산 지원 요청');
                        setOptions(prev => ({
                          ...prev,
                          enableRag: false,
                          enableWebSearch: false,
                          enableHealthAnalysis: false,
                        }));
                      }}
                      className="p-6 bg-gradient-to-br from-teal-50 to-teal-100 dark:from-teal-900/20 dark:to-teal-800/20 border border-teal-200 dark:border-teal-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                    >
                      <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">📝</div>
                      <h4 className="font-semibold text-gray-900 dark:text-white mb-2">{t('chat.officialDocument', '공문작성')}</h4>
                      <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                        {t('chat.officialDocumentDescription', '공식적인 업무 처리를 위한 공문서를 작성합니다')}
                      </p>
                    </button>

                  </div>

                  {/* 업로드된 파일 목록 */}
                  {uploadedFiles.length > 0 && (
                    <div className="mb-4 flex flex-wrap gap-2 justify-center">
                      {uploadedFiles.map((file, index) => {
                        const isImage = isImageFile(file);
                        const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
                        const imageUrl = isImage ? uploadedFileUrls.get(fileKey) : null;

                        // 임베딩 상태 확인
                        const embeddingStatus = embeddingStatuses[file.name];
                        const isEmbedding = embeddingStatus?.isProcessing || false;
                        const embeddingStep = embeddingStatus?.step_label || '';

                        return (
                          <div key={`${file.name}-${file.size}-${index}`} className="flex items-center space-x-2 bg-gray-100 dark:bg-gray-700 px-3 py-1 rounded-full text-sm">
                            {isImage && imageUrl ? (
                              <img
                                src={imageUrl}
                                alt={file.name}
                                className="w-6 h-6 rounded-full object-cover"
                              />
                            ) : (
                              // 임베딩 완료/미완료 상태 구분
                              <span className={`w-2 h-2 rounded-full ${
                                isImage
                                  ? 'bg-green-500'
                                  : (embeddingStatus && !isEmbedding)
                                    ? 'bg-green-500'
                                    : 'bg-orange-500 animate-pulse'
                              }`}></span>
                            )}
                            <span className="text-gray-700 dark:text-gray-300 truncate max-w-32">
                              {file.name}
                            </span>
                            <button
                              onClick={() => removeFile(index)}
                              className="text-gray-500 hover:text-red-500 ml-1"
                            >
                              <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                              </svg>
                            </button>
                          </div>
                        );
                      })}
                    </div>
                  )}

                  {/* 통합된 입력 영역 */}
                  <div
                    className={`bg-gray-100 dark:bg-gray-700 rounded-2xl border-0 dark:border dark:border-gray-600 p-4 transition-all ${isDragOver ? 'ring-2 ring-blue-500 ring-opacity-50 border-blue-400' : ''}`}
                    onDragEnter={handleDragEnter}
                    onDragLeave={handleDragLeave}
                    onDragOver={handleDragOver}
                    onDrop={handleDrop}
                  >
                    {/* 도구 버튼들 */}
                    <div className="flex items-center justify-start gap-2 mb-3">
                      {/* 파일 업로드 버튼 */}
                      <div ref={uploadOptionsRef} className="relative">
                        <button
                          onClick={() => setShowUploadOptions(!showUploadOptions)}
                          disabled={isUploading || isLoading || isLoadingSessions}
                          className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${isUploading ? 'bg-gray-300 dark:bg-gray-600 text-gray-600 dark:text-gray-500' :
                            'bg-gray-300 dark:bg-gray-600 text-gray-800 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                            }`}
                          title={t('chat.fileUpload', '파일')}
                        >
                          <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                          </svg>
                          <span>{t('chat.fileUpload', '파일')}</span>
                        </button>

                        {/* 파일 업로드 옵션 팝업 메뉴 */}
                        {showUploadOptions && (
                          <div
                            className="absolute top-full left-0 mt-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-600 rounded-lg shadow-lg py-2 z-50 min-w-64"
                            style={{ minWidth: '280px' }}
                          >
                            <button
                              onClick={handleDocumentUpload}
                              className="w-full flex items-center gap-3 px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                            >
                              <svg className="w-5 h-5 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                              </svg>
                              <div className="text-left">
                                <span className="font-medium">{t('chat.document', '문서')}</span>
                                <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- {t('chat.fastDocumentProcessing', '빠른 문서 처리')}</span>
                              </div>
                            </button>

                            <button
                              onClick={handleImageUpload}
                              className="w-full flex items-center gap-3 px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                            >
                              <svg className="w-5 h-5 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                              </svg>
                              <div className="text-left">
                                <span className="font-medium">{t('chat.image', '이미지')}</span>
                                <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- {t('chat.imageAnalysis', '이미지 분석')}</span>
                              </div>
                            </button>

                            <button
                              onClick={handleRichRagClick}
                              className="w-full flex items-center gap-3 px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                            >
                              <svg className="w-5 h-5 text-purple-600 dark:text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                              </svg>
                              <div className="text-left">
                                <span className="font-medium">시맨틱 RAG</span>
                                <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- 멀티모달, Graph, 고급 분석</span>
                              </div>
                            </button>

                            <button
                              onClick={handleExcelUpload}
                              className="w-full flex items-center gap-3 px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                            >
                              <svg className="w-5 h-5 text-orange-600 dark:text-orange-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 10h18M3 14h18m-9-4v8m-7 0V4a2 2 0 012-2h14a2 2 0 012 2v16a2 2 0 01-2 2H5a2 2 0 01-2-2z" />
                              </svg>
                              <div className="text-left">
                                <span className="font-medium">{t('chat.smartData', '스마트 데이터')}</span>
                                <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- XLS,CSV 데이터 AI 전처리</span>
                              </div>
                            </button>
                          </div>
                        )}
                      </div>

                      {/* 추론 수준 선택 버튼 - hamonize 모델일 때만 표시 */}
                      {options.model === 'hamonize:latest' && (
                        <div className="relative">
                          <button
                            onClick={() => {
                              const levels = ['low', 'medium', 'high'] as const;
                              const currentIndex = levels.indexOf(options.reasoningLevel);
                              const nextIndex = (currentIndex + 1) % levels.length;
                              setOptions(prev => ({ ...prev, reasoningLevel: levels[nextIndex] }));
                            }}
                            disabled={isLoading || isLoadingSessions}
                            className="flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all bg-purple-300 dark:bg-purple-800 text-purple-800 dark:text-purple-100 ring-1 ring-purple-300 dark:ring-purple-600 hover:bg-purple-400 dark:hover:bg-purple-700"
                            title={`추론 수준: ${
                              options.reasoningLevel === 'low' ? '빠른 추론' :
                              options.reasoningLevel === 'medium' ? '균형 추론' : '상세 추론'
                            }`}
                          >
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                            </svg>
                            <span>
                              {options.reasoningLevel === 'low' && t('chat.fastInference', '빠른추론')}
                              {options.reasoningLevel === 'medium' && t('chat.balancedReasoning', '균형추론')}
                              {options.reasoningLevel === 'high' && t('chat.deepReasoning', '심층추론')}
                            </span>
                          </button>
                        </div>
                      )}

                      {/* 문서검색 드롭다운 버튼 */}
                      <div className="relative rag-dropdown">
                        <button
                          onClick={() => {
                            if (!options.enableRag) {
                              // 비활성화 상태에서 클릭하면 개인문서로 활성화
                              setOptions(prev => ({
                                ...prev,
                                enableRag: true,
                                ragSearchScope: 'personal'
                              }));
                            } else {
                              // 활성화 상태에서 클릭하면 다음 검색 범위로 순환
                              const scopes = ['personal', 'shared', 'all'] as const;
                              const currentIndex = scopes.indexOf(options.ragSearchScope);
                              const nextIndex = (currentIndex + 1) % scopes.length;
                              
                              if (nextIndex === 0) {
                                // 마지막 범위(전체검색)에서 다시 클릭하면 비활성화
                                setOptions(prev => ({
                                  ...prev,
                                  enableRag: false
                                }));
                              } else {
                                // 다음 범위로 이동
                                setOptions(prev => ({
                                  ...prev,
                                  ragSearchScope: scopes[nextIndex]
                                }));
                              }
                            }
                          }}
                          disabled={isLoading || isLoadingSessions}
                          className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${options.enableRag ?
                            'bg-green-300 dark:bg-green-800 text-green-900 dark:text-green-100 ring-1 ring-green-300 dark:ring-green-600' :
                            'bg-gray-300 dark:bg-gray-600 text-gray-900 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                            }`}
                          title={options.enableRag ? getRagScopeInfo(options.ragSearchScope).description : t('chat.documentSearch', '문서검색')}
                        >
                          <span className="text-sm">{options.enableRag ? getRagScopeInfo(options.ragSearchScope).icon : '📄'}</span>
                          <span>{options.enableRag ? getRagScopeInfo(options.ragSearchScope).label : t('chat.documentSearch', '문서검색')}</span>
                        </button>


                      </div>

                      {/* 웹검색 버튼 */}
                      <button
                        onClick={() => setOptions(prev => ({
                          ...prev,
                          enableWebSearch: !prev.enableWebSearch,

                        }))}
                        disabled={isLoading || isLoadingSessions}
                        className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${options.enableWebSearch ?
                          'bg-blue-300 dark:bg-blue-800 text-blue-900 dark:text-blue-100 ring-1 ring-blue-300 dark:ring-blue-600' :
                          'bg-gray-300 dark:bg-gray-600 text-gray-900 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                          }`}
                        title={t('chat.webSearch', '웹검색')}
                      >
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9v-9m0-9v9" />
                        </svg>
                        <span>{t('chat.webSearch', '웹검색')}</span>
                      </button>

                      {/* 건강검진 분석 버튼 */}
                      <button
                        onClick={() => setOptions(prev => ({
                          ...prev,
                          enableHealthAnalysis: !prev.enableHealthAnalysis,
                        }))}
                        disabled={isLoading || isLoadingSessions}
                        className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${options.enableHealthAnalysis ?
                          'bg-green-300 dark:bg-green-800 text-green-900 dark:text-green-100 ring-1 ring-green-300 dark:ring-green-600' :
                          'bg-gray-300 dark:bg-gray-600 text-gray-900 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                          }`}
                        title="건강검진 분석"
                      >
                        <span className="text-sm">🏥</span>
                        <span>{t('chat.healthAnalysis', '검진분석')}</span>
                      </button>

                    </div>

                    {/* 멘션된 문서 표시 */}
                    {mentionedDocs.length > 0 && (
                      <div className="mb-3 flex flex-wrap gap-1">
                        {mentionedDocs.map((doc, index) => (
                          <span
                            key={index}
                            className="inline-flex items-center px-2 py-1 bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200 rounded-full text-xs"
                          >
                            <svg className="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                              <path d="M4 4a2 2 0 00-2 2v8a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2H4z" />
                            </svg>
                            {doc.displayName}
                            <button
                              onClick={() => removeMentionedDoc(doc.name)}
                              className="ml-1 text-blue-600 hover:text-blue-800"
                            >
                              ×
                            </button>
                          </span>
                        ))}
                      </div>
                    )}

                    {/* 입력창 영역 */}
                    <div className="relative">
                      <textarea
                        ref={(el) => {
                          if (el) {
                            textareaRef.current = el;
                            el.style.height = 'auto';
                            el.style.height = `${Math.min(Math.max(el.scrollHeight, 48), 200)}px`;
                          }
                        }}
                        value={inputMessage}
                        onChange={handleInputChange}
                        onKeyDown={handleKeyDown}
                        placeholder={isLoadingSessions ? t('chat.loadingSessions', '세션을 로드하는 중...') : isDragOver ? t('chat.dropFiles', '파일을 여기에 놓으세요...') : t('chat.askAnything', '무엇이든 물어보세요...')}
                        className="w-full px-0 py-2 pr-12 bg-transparent text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none resize-none border-0 text-base"
                        rows={1}
                        disabled={isLoading || isLoadingSessions}
                        style={{ minHeight: '48px', maxHeight: '200px' }}
                      />

                      {/* 문서 자동완성 드롭다운 */}
                      {showDocSuggestions && filteredDocsForMention.length > 0 && (
                        <div className="absolute bottom-full left-0 right-0 mb-1 bg-gray-50 dark:bg-gray-800 border border-gray-300 dark:border-gray-500 rounded-lg shadow-xl max-h-60 overflow-y-auto z-10 scrollbar-thin scrollbar-thumb-gray-300 dark:scrollbar-thumb-gray-600 scrollbar-track-transparent">
                          {filteredDocsForMention.map((doc, index) => (
                            <button
                              key={index}
                              onClick={() => selectDocument(doc.name)}
                              className={`w-full text-left px-3 py-2 flex items-center space-x-2 transition-colors ${index === selectedSuggestionIndex
                                ? 'bg-blue-100 dark:bg-blue-900/40 text-blue-900 dark:text-blue-100'
                                : 'hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-800 dark:text-gray-200'
                                }`}
                            >
                              {getFileIcon(doc.name, doc.status)}
                              <span className="text-sm truncate flex-1" title={doc.name}>{doc.name}</span>
                            </button>
                          ))}
                        </div>
                      )}

                      <div className="absolute right-0 bottom-2 flex items-center space-x-2">
                        {/* 마이크 버튼 */}
                        <button
                          onClick={toggleSpeechRecognition}
                          disabled={isLoading || isLoadingSessions || isProcessingAudio}
                          className={`p-2 rounded-lg transition-colors ${isListening
                            ? 'bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 animate-pulse'
                            : isProcessingAudio
                              ? 'bg-yellow-100 dark:bg-yellow-900/30 text-yellow-600 dark:text-yellow-400'
                              : 'bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-400 hover:bg-gray-200 dark:hover:bg-gray-600'
                            } disabled:opacity-50 disabled:cursor-not-allowed`}
                          title={isListening ? t('chat.voiceRecordingStop', '음성 녹음 중지') : isProcessingAudio ? t('chat.voiceProcessing', '음성 처리 중') : t('chat.voiceRecordingStart', '음성 녹음 시작')}
                        >
                          {isListening ? (
                            <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                              <path d="M12 14c1.66 0 3-1.34 3-3V5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm-1-9c0-.55.45-1 1-1s1 .45 1 1v6c0 .55-.45 1-1 1s-1-.45-1-1V5zm6 6c0 2.76-2.24 5-5 5s-5-2.24-5-5H5c0 3.53 2.61 6.43 6 6.92V21h2v-3.08c3.39-.49 6-3.39 6-6.92h-2z" />
                            </svg>
                          ) : isProcessingAudio ? (
                            <svg className="w-4 h-4 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                            </svg>
                          ) : (
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 11a7 7 0 01-7 7m0 0a7 7 0 00-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z" />
                            </svg>
                          )}
                        </button>

                        {/* 전송 버튼 */}
                        <button
                          onClick={isLoading && isStreamingRef.current ? stopStreaming : sendMessage}
                          disabled={
                            isLoading && isStreamingRef.current
                              ? false // 스트리밍 중에는 중지 버튼을 클릭할 수 있도록 활성화
                              : ((!inputMessage.trim() && uploadedFiles.length === 0) || isLoadingSessions)
                          }
                          className={`p-2 text-white rounded-lg focus:outline-none focus:ring-2 transition-colors ${isLoading && isStreamingRef.current
                            ? 'bg-red-600 hover:bg-red-700 focus:ring-red-500'
                            : 'bg-blue-600 hover:bg-blue-700 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed'
                            }`}
                        >
                          {isLoading && isStreamingRef.current ? (
                            <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                              <rect x="6" y="6" width="12" height="12" />
                            </svg>
                          ) : (
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                            </svg>
                          )}
                        </button>
                      </div>
                    </div>
                  </div>

                  {/* 숨겨진 파일 입력 */}
                  <input
                    ref={fileInputRef}
                    type="file"
                    multiple
                    accept="image/*,.pdf,.doc,.docx,.txt,.md,.csv,.json"
                    onChange={handleFileSelect}
                    className="hidden"
                  />

                  {/* 오류 메시지 */}
                  {error && (
                    <div className="mt-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                      <p className="text-sm text-red-800 dark:text-red-200">{error}</p>
                    </div>
                  )}
                </div>
              </div>
            ) : (
              /* 메시지가 있는 화면 - 분리된 레이아웃 */
              <>

                {/* 메시지 영역 */}
                <div className="flex-1 overflow-y-auto">
                  <div className="max-w-5xl mx-auto px-4 py-6 pb-24">
                    {/* 메시지 목록 */}
                    <div className="space-y-6">
                      {currentMessages.map((message, index) => {
                        // Debug log for each message being rendered
                        // if (message.role === 'assistant' && currentSessionId) {
                        //   console.log('🎨 [Rendering Message]:', {
                        //     messageId: message.id,
                        //     sessionId: currentSessionId,
                        //     hasDownloadInfo: !!message.downloadInfo,
                        //     downloadInfoCount: message.downloadInfo?.length || 0,
                        //     downloadInfoSample: message.downloadInfo?.[0]
                        //   });
                        // }
                        return (
                        <div
                          key={`${message.id}-${index}`}
                          className="flex justify-start"
                        >
                          <div className="flex max-w-5xl w-full flex-row">
                            {/* 아바타 */}
                            <div className="flex-shrink-0 mr-3">
                              <div className={`w-8 h-8 rounded-full flex items-center justify-center ${message.role === 'user'
                                ? 'bg-gray-200 dark:bg-gray-700 text-gray-600 dark:text-gray-300'
                                : 'bg-gray-200 dark:bg-gray-700 text-gray-600 dark:text-gray-300'
                                }`}>
                                {message.role === 'user' ? (
                                  <img
                                    src={userProfileImage}
                                    alt="User"
                                    className="w-7 h-7 object-cover rounded-full"
                                  />
                                ) : (
                                  <img
                                    src="/images/ai.png"
                                    alt="AI 워크스페이스"
                                    className="w-7 h-7 object-contain"
                                  />
                                )}
                              </div>
                            </div>
                            {/* 메시지 내용 */}
                            <div className="flex-1 min-w-0 text-left">
                              {message.role === 'assistant' ? (
                                <div className="text-gray-900 dark:text-white">
                                  {/* Thinking 과정 및 진행상태 표시 - AI 응답 완료 후 숨김 */}
                                  {(() => {
                                    // 로딩 중이고 현재 메시지가 마지막 메시지인 경우에만 표시
                                    const isCurrentMessage = message.id === currentMessages[currentMessages.length - 1]?.id;
                                    const shouldShow = isLoading && isCurrentMessage && (message.thinking || progressMessages.length > 0);
                                    
                                    return shouldShow;
                                  })() && (
                                    <div className="mb-4 p-4 bg-blue-50 dark:bg-blue-900/30 border border-blue-200 dark:border-blue-700 rounded-lg">
                                      <div className="flex items-center mb-2">
                                        {/* <svg className="w-4 h-4 text-blue-600 dark:text-blue-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                                        </svg> */}
                                        <span className="text-xs font-medium text-blue-800 dark:text-blue-300">💭 Chain of Thought</span>
                                      </div>
                                      
                                      {/* 진행상태 메시지들 - 로딩 중일 때만 표시 */}
                                      {isLoading && message.id === currentMessages[currentMessages.length - 1]?.id && progressMessages.length > 0 && (
                                        <div className="mb-3 space-y-1">
                                          {progressMessages.map((progressMessage, index) => (
                                            <div key={index} className="flex items-center space-x-2">
                                              <div className="w-1.5 h-1.5 rounded-full bg-blue-400 dark:bg-blue-300 animate-pulse flex-shrink-0"></div>
                                              <span className="text-xs text-blue-600 dark:text-blue-400 font-medium">
                                                {progressMessage}
                                              </span>
                                            </div>
                                          ))}
                                        </div>
                                      )}
                                      
                                      {/* Thinking 내용 */}
                                      {message.thinking && (
                                        <div className="text-xs leading-relaxed text-blue-700/90 dark:text-blue-200/90 whitespace-pre-wrap italic opacity-90">
                                          {message.thinking}
                                        </div>
                                      )}
                                    </div>
                                  )}
                                  
                                  {/* 메인 응답 내용 */}
                                  <ChatMarkdownRenderer
                                    content={typingMessageId === message.id ? displayedContent : message.content}
                                    className=""
                                    downloadInfo={(() => {
                                      // 디버깅용 로그
                                      // if (message.downloadInfo && message.downloadInfo.length > 0) {
                                      //   console.log('✅ [ChatMarkdownRenderer] downloadInfo 발견:', {
                                      //     messageId: message.id,
                                      //     sessionId: currentSessionId,
                                      //     downloadInfoCount: message.downloadInfo.length,
                                      //     files: message.downloadInfo.map((info: any) => info.filename || info.title).join(', '),
                                      //     sources: message.downloadInfo.map((info: any) => info.source).join(', ')
                                      //   });
                                      // } 
                                      return message.downloadInfo;
                                    })()}
                                  />
                                  {typingMessageId === message.id && (
                                    <span className="inline-block w-2 h-5 bg-gray-400 dark:bg-gray-500 ml-1 animate-pulse"></span>
                                  )}
                                </div>
                              ) : (
                                <div className="text-blue-900 dark:text-blue-200">
                                  {/* 첨부된 이미지들 표시 */}
                                  {(() => {
                                    // attachedImages가 있으면 우선 사용 (실시간 표시용)
                                    if (message.attachedImages && message.attachedImages.length > 0) {
                                      return (
                                        <div className="mb-3 flex flex-wrap gap-2">
                                          {message.attachedImages.map((image, imgIndex) => (
                                            <div key={`${message.id}-img-${imgIndex}`} className="relative group">
                                              <img
                                                src={image.url || image.base64}
                                                alt={image.name}
                                                className="max-w-xs max-h-48 rounded-lg object-cover border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md transition-shadow cursor-pointer"
                                                onClick={() => {
                                                  // 이미지 클릭 시 새 탭에서 크게 보기
                                                  window.open(image.url || image.base64, '_blank');
                                                }}
                                                title={`${image.name} - ${t('chat.clickToEnlarge', '클릭하여 크게 보기')}`}
                                              />
                                              <div className="absolute bottom-1 left-1 bg-black bg-opacity-50 text-white text-xs px-2 py-1 rounded opacity-0 group-hover:opacity-100 transition-opacity">
                                                {image.name}
                                              </div>
                                            </div>
                                          ))}
                                        </div>
                                      );
                                    }

                                    // attachedImages가 없으면 content에서 이미지 데이터 추출 (저장된 대화 표시용)
                                    const imagePattern = /\[IMAGE:([^\]]+)\](data:image\/[^;]+;base64,[A-Za-z0-9+/=]+)/g;
                                    const images: { name: string; base64: string }[] = [];
                                    let match;

                                    while ((match = imagePattern.exec(message.content || '')) !== null) {
                                      images.push({
                                        name: match[1],
                                        base64: match[2]
                                      });
                                    }

                                    return images.length > 0 ? (
                                      <div className="mb-3 flex flex-wrap gap-2">
                                        {images.map((image, imgIndex) => (
                                          <div key={`${message.id}-img-${imgIndex}`} className="relative group">
                                            <img
                                              src={image.base64}
                                              alt={image.name}
                                              className="max-w-xs max-h-48 rounded-lg object-cover border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md transition-shadow cursor-pointer"
                                              onClick={() => {
                                                // 이미지 클릭 시 새 탭에서 크게 보기
                                                // window.open(image.base64, '_blank');
                                                openImageInNewTab(image.base64);
                                              }}
                                              title={`${image.name} - ${t('chat.clickToEnlarge', '클릭하여 크게 보기')}`}
                                            />
                                            <div className="absolute bottom-1 left-1 bg-black bg-opacity-50 text-white text-xs px-2 py-1 rounded opacity-0 group-hover:opacity-100 transition-opacity">
                                              {image.name}
                                            </div>
                                          </div>
                                        ))}
                                      </div>
                                    ) : null;
                                  })()}
                                  <ChatMarkdownRenderer
                                    content={(() => {
                                      // originalContent가 있으면 그것을 우선 표시 (멘션된 문서가 포함되기 전 원본 질문)
                                      const displayContent = message.originalContent || message.content || ' ';
                                      // content에서 이미지 데이터 제거한 텍스트만 표시
                                      const imagePattern = /\[IMAGE:[^\]]+\]data:image\/[^;]+;base64,[A-Za-z0-9+/=]+/g;
                                      return displayContent.replace(imagePattern, '').trim() || ' ';
                                    })()}
                                    className=""
                                    downloadInfo={message.downloadInfo}
                                  />
                                </div>
                              )}
                              <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                {(() => {
                                  const date = new Date(message.timestamp);
                                  return isNaN(date.getTime()) ? '' : date.toLocaleTimeString('ko-KR');
                                })()}
                              </p>
                              {/* AI 응답 액션 버튼들 */}
                              {message.role === 'assistant' && typingMessageId !== message.id && (
                                <div className="flex items-center justify-end space-x-2 mt-2">
                                  {/* 클립보드 복사 */}
                                  <button
                                    onClick={() => copyToClipboard(message.content, message.id)}
                                    className={`p-1.5 rounded transition-colors group ${copiedMessageId === message.id
                                      ? 'text-green-600 bg-green-50 dark:bg-green-900/20'
                                      : 'text-gray-500 hover:text-green-600 hover:bg-gray-100 dark:text-gray-300 dark:hover:text-green-400 dark:hover:bg-gray-700'
                                      }`}
                                    title={copiedMessageId === message.id ? t('chat.copyCompleted', '복사 완료!') : t('chat.copyToClipboard', '응답 내용을 클립보드에 복사')}
                                  >
                                    {copiedMessageId === message.id ? (
                                      // 복사 완료 체크 아이콘
                                      <svg className="w-4 h-4 group-hover:scale-110 transition-transform" fill="currentColor" viewBox="0 0 24 24">
                                        <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                      </svg>
                                    ) : (
                                      // 기본 복사 아이콘
                                      <svg className="w-4 h-4 group-hover:scale-110 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                                      </svg>
                                    )}
                                  </button>

                                  {/* PDF 저장 */}
                                  <button
                                    onClick={() => saveToPDF(message.content, message.id)}
                                    disabled={pdfProcessingId === message.id}
                                    className={`p-1.5 rounded transition-colors group ${pdfProcessingId === message.id
                                      ? 'text-orange-600 bg-orange-50 dark:bg-orange-900/20 cursor-wait'
                                      : 'text-gray-500 hover:text-green-600 hover:bg-gray-100 dark:text-gray-300 dark:hover:text-green-400 dark:hover:bg-gray-700'
                                      }`}
                                    title={pdfProcessingId === message.id ? t('chat.generatingPdf', 'PDF 생성 중...') : t('chat.saveToPdf', '응답을 PDF로 저장 (인쇄 창 열림)')}
                                  >
                                    {pdfProcessingId === message.id ? (
                                      // 로딩 스피너
                                      <svg className="w-4 h-4 animate-spin" fill="none" viewBox="0 0 24 24">
                                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                      </svg>
                                    ) : (
                                      // 기본 PDF 아이콘
                                      <svg className="w-4 h-4 group-hover:scale-110 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                                      </svg>
                                    )}
                                  </button>

                                  {/* 좋아요 */}
                                  <button
                                    onClick={() => handleLike(message.id)}
                                    className={`p-1.5 rounded transition-colors ${messageReactions[message.id]?.liked
                                      ? 'text-green-600 bg-green-50 dark:bg-green-900/20 hover:bg-green-100 dark:hover:bg-green-900/30'
                                      : 'text-gray-500 hover:text-green-600 hover:bg-gray-100 dark:text-gray-300 dark:hover:text-green-400 dark:hover:bg-gray-700'
                                      }`}
                                    title={t('chat.likeMessage', '좋아요')}
                                  >
                                    <svg className="w-4 h-4" fill={messageReactions[message.id]?.liked ? "currentColor" : "none"} stroke="currentColor" viewBox="0 0 24 24">
                                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14 10h4.764a2 2 0 011.789 2.894l-3.5 7A2 2 0 0115.263 21h-4.017c-.163 0-.326-.02-.485-.06L7 20m7-10V5a2 2 0 00-2-2h-.095c-.5 0-.905.405-.905.905 0 .714-.211 1.412-.608 2.006L7 11v9m7-10h-2M7 20H5a2 2 0 01-2-2v-6a2 2 0 012-2h2.5" />
                                    </svg>
                                  </button>

                                  {/* 싫어요 */}
                                  <button
                                    onClick={() => handleDislike(message.id)}
                                    className={`p-1.5 rounded transition-colors ${messageReactions[message.id]?.disliked
                                      ? 'text-red-600 bg-red-50 dark:bg-red-900/20 hover:bg-red-100 dark:hover:bg-red-900/30'
                                      : 'text-gray-500 hover:text-red-600 hover:bg-gray-100 dark:text-gray-300 dark:hover:text-red-400 dark:hover:bg-gray-700'
                                      }`}
                                    title={t('chat.dislikeMessage', '싫어요')}
                                  >
                                    <svg className="w-4 h-4" fill={messageReactions[message.id]?.disliked ? "currentColor" : "none"} stroke="currentColor" viewBox="0 0 24 24">
                                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 14H5.236a2 2 0 01-1.789-2.894l3.5-7A2 2 0 018.736 3h4.018c.163 0 .326.02.485.06L17 4m-7 10v2a2 2 0 002 2h.095c.5 0 .905-.405.905-.905 0-.714.211-1.412.608-2.006L17 13V4m-7 10h2m5-10h2a2 2 0 012 2v6a2 2 0 01-2 2h-2.5" />
                                    </svg>
                                  </button>

                                  {/* GraphRAG 그래프 보기 (GraphRAG 활성화된 경우에만) */}
                                  {graphRAGEnabled && (
                                    <button
                                      onClick={() => {
                                        // 현재 AI 응답 메시지 바로 전의 사용자 질문을 찾기
                                        const currentIndex = currentMessages.findIndex(msg => msg.id === message.id);
                                        let userQuery = '';
                                        
                                        // 현재 메시지부터 거슬러 올라가면서 사용자 메시지 찾기
                                        for (let i = currentIndex - 1; i >= 0; i--) {
                                          if (currentMessages[i].role === 'user') {
                                            userQuery = currentMessages[i].content || '';
                                            break;
                                          }
                                        }
                                        
                                        setCurrentGraphQuery(userQuery);
                                        setShowGraphRAGModal(true);
                                      }}
                                      className="p-1.5 rounded transition-colors text-gray-400 hover:text-purple-600 dark:hover:text-purple-400 hover:bg-purple-50 dark:hover:bg-purple-900/20"
                                      title="지식 그래프 보기"
                                    >
                                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 12h14M5 12a2 2 0 01-2-2V6a2 2 0 012-2h6a2 2 0 012 2v4a2 2 0 01-2 2M5 12a2 2 0 00-2 2v4a2 2 0 002 2h6a2 2 0 002-2v-4a2 2 0 00-2-2m8-8a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V4z" />
                                      </svg>
                                    </button>
                                  )}
                                </div>
                              )}
                            </div>
                          </div>
                        </div>
                      )})}



                    </div>
                    <div ref={messagesEndRef} className="h-4" />
                  </div>
                </div>



                {/* 하단 입력 영역 - 고정 */}
                <div className="flex-shrink-0 border-t border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--body-bg)' }}>
                  <div className="max-w-5xl mx-auto px-4 py-4 pb-8">

                    {/* 업로드된 파일 목록 */}
                    {uploadedFiles.length > 0 && (
                      <div className="mb-3">
                        <div className="flex flex-wrap gap-2">
                          {uploadedFiles.map((file, index) => {
                            const isImage = isImageFile(file);
                            const imageFiles = uploadedFiles.filter(isImageFile);
                            const isFirstImage = isImage && imageFiles.indexOf(file) === 0;
                            const willBeProcessed = !isImage || isFirstImage;
                            const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
                            const imageUrl = isImage ? uploadedFileUrls.get(fileKey) : null;

                            // 임베딩 상태 확인
                            const embeddingStatus = embeddingStatuses[file.name];
                            const isEmbedding = embeddingStatus?.isProcessing || false;
                            const embeddingStep = embeddingStatus?.step_label || '';

                            return (
                              <div
                                key={`${file.name}-${file.size}-${index}`}
                                className={`flex items-center space-x-2 px-3 py-1 rounded-full text-sm ${willBeProcessed
                                  ? 'bg-gray-100 dark:bg-gray-700'
                                  : 'bg-yellow-50 dark:bg-yellow-900/30 border border-yellow-200 dark:border-yellow-700'
                                  }`}
                              >
                                {isImage && imageUrl ? (
                                  <img
                                    src={imageUrl}
                                    alt={file.name}
                                    className="w-6 h-6 rounded-full object-cover"
                                  />
                                ) : (
                                  // 임베딩 완료/미완료 상태 구분
                                  isImage ? (
                                    <span className={`w-2 h-2 rounded-full ${
                                      willBeProcessed ? 'bg-green-500' : 'bg-yellow-500'
                                    }`}></span>
                                  ) : (
                                    // 비이미지 파일: 임베딩 완료되지 않은 상태는 깜빡이는 주황색, 완료된 상태는 초록색
                                    <span className={`w-2 h-2 rounded-full ${
                                      embeddingStatus && !isEmbedding
                                        ? 'bg-green-500'
                                        : 'bg-orange-500 animate-pulse'
                                    }`}></span>
                                  )
                                )}
                                <span className={`truncate max-w-32 ${willBeProcessed
                                  ? 'text-gray-700 dark:text-gray-300'
                                  : 'text-yellow-700 dark:text-yellow-300'
                                  }`}>
                                  {file.name}
                                  {isImage && !willBeProcessed && ' (대기)'}
                                </span>
                                <button
                                  onClick={() => removeFile(index)}
                                  className="text-gray-500 hover:text-red-500 ml-1"
                                >
                                  <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                                  </svg>
                                </button>
                              </div>
                            );
                          })}
                        </div>

                        {/* 여러 이미지 업로드 시 안내 메시지 */}
                        {uploadedFiles.filter(isImageFile).length > 1 && (
                          <div className="mt-2 p-2 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-700 rounded-md">
                            <p className="text-xs text-yellow-700 dark:text-yellow-300 flex items-center">
                              <svg className="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                              </svg>
                              {t('chat.multipleImagesWarning', '여러 이미지가 업로드되었습니다. 현재는 첫 번째 이미지만 분석됩니다.')}
                            </p>
                          </div>
                        )}
                      </div>
                    )}

                    {/* 통합된 입력 영역 */}
                    <div
                      className={`bg-gray-100 dark:bg-gray-700 rounded-2xl border-0 dark:border dark:border-gray-600 p-4 transition-all ${isDragOver ? 'ring-2 ring-blue-500 ring-opacity-50 border-blue-400' : ''}`}
                      onDragEnter={handleDragEnter}
                      onDragLeave={handleDragLeave}
                      onDragOver={handleDragOver}
                      onDrop={handleDrop}
                    >
                      {/* 도구 버튼들 */}
                      <div className="flex items-center justify-start gap-2 mb-3">
                        {/* 파일 업로드 버튼 */}
                        <div ref={uploadOptionsBottomRef} className="relative">
                          <button
                            onClick={() => setShowUploadOptions(!showUploadOptions)}
                            disabled={isUploading || isLoading || isLoadingSessions}
                            className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${isUploading ? 'bg-gray-300 dark:bg-gray-600 text-gray-600 dark:text-gray-500' :
                            'bg-gray-300 dark:bg-gray-600 text-gray-800 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                            }`}
                            title={t('chat.fileUpload', '파일')}
                          >
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                            </svg>
                            <span>{t('chat.fileUpload', '파일')}</span>
                            <svg className="w-3 h-3 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                            </svg>
                          </button>

                          {/* 파일 업로드 옵션 팝업 */}
                          {showUploadOptions && (
                            <div className="absolute bottom-full left-0 mb-2 w-72 bg-white dark:bg-gray-800 rounded-lg shadow-lg border border-gray-200 dark:border-gray-700 z-50">
                              <div className="py-2">
                                <button
                                  onClick={handleDocumentUpload}
                                  className="flex items-center gap-3 w-full px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
                                >
                                  <svg className="w-5 h-5 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                                  </svg>
                                  <div className="text-left">
                                    <span className="font-medium">{t('chat.document', '문서')}</span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- {t('chat.fastDocumentProcessing', '빠른 문서 처리')}</span>
                                  </div>
                                </button>
                                <button
                                  onClick={handleImageUpload}
                                  className="flex items-center gap-3 w-full px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
                                >
                                  <svg className="w-5 h-5 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                  </svg>
                                  <div className="text-left">
                                    <span className="font-medium">{t('chat.image', '이미지')}</span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- {t('chat.imageAnalysis', '이미지 분석')}</span>
                                  </div>
                                </button>
                                <button
                                  onClick={() => {
                                    setShowUploadOptions(false);
                                    if (fileInputRef.current) {
                                      fileInputRef.current.accept = '.pdf,.doc,.docx,.txt,.md,.ppt,.pptx';
                                      fileInputRef.current.dataset.mode = 'rich';
                                      fileInputRef.current.multiple = true;
                                      fileInputRef.current.click();
                                    }
                                  }}
                                  className="flex items-center gap-3 w-full px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
                                >
                                  <svg className="w-5 h-5 text-purple-600 dark:text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                                  </svg>
                                  <div className="text-left">
                                    <span className="font-medium">시맨틱 RAG</span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- 멀티모달, Graph, 고급 분석</span>
                                  </div>
                                </button>
                                <button
                                  onClick={handleExcelUpload}
                                  className="flex items-center gap-3 w-full px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
                                >
                                  <svg className="w-5 h-5 text-orange-600 dark:text-orange-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 10h18M3 14h18m-9-4v8m-7 0V4a2 2 0 012-2h14a2 2 0 012 2v16a2 2 0 01-2 2H5a2 2 0 01-2-2z" />
                                  </svg>
                                  <div className="text-left">
                                    <span className="font-medium">{t('chat.smartData', '스마트 데이터')}</span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-2">- XLS,CSV 데이터 AI 전처리</span>
                                  </div>
                                </button>
                              </div>
                            </div>
                          )}
                        </div>

                        {/* 추론 수준 선택 버튼 - hamonize 모델일 때만 표시 */}
                        {options.model === 'hamonize:latest' && (
                          <div className="relative">
                            <button
                              onClick={() => {
                                const levels = ['low', 'medium', 'high'] as const;
                                const currentIndex = levels.indexOf(options.reasoningLevel);
                                const nextIndex = (currentIndex + 1) % levels.length;
                                setOptions(prev => ({ ...prev, reasoningLevel: levels[nextIndex] }));
                              }}
                                                              disabled={isLoading || isLoadingSessions}
                                className="flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all bg-purple-300 dark:bg-purple-800 text-purple-800 dark:text-purple-100 ring-1 ring-purple-300 dark:ring-purple-600 hover:bg-purple-400 dark:hover:bg-purple-700"
                                title={`추론 수준: ${
                                options.reasoningLevel === 'low' ? '빠른 추론' :
                                options.reasoningLevel === 'medium' ? '균형 추론' : '상세 추론'
                              }`}
                            >
                              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                              </svg>
                              <span>
                                {options.reasoningLevel === 'low' && t('chat.fastInference', '빠른추론')}
                                {options.reasoningLevel === 'medium' && t('chat.balancedReasoning', '균형추론')}
                                {options.reasoningLevel === 'high' && t('chat.deepReasoning', '심층추론')}
                              </span>
                            </button>
                          </div>
                        )}

                        {/* 문서검색 드롭다운 버튼 */}
                        <div className="relative rag-dropdown">
                          <button
                            onClick={() => {
                              if (!options.enableRag) {
                                // 비활성화 상태에서 클릭하면 개인문서로 활성화
                                setOptions(prev => ({
                                  ...prev,
                                  enableRag: true,
                                  ragSearchScope: 'personal'
                                }));
                              } else {
                                // 활성화 상태에서 클릭하면 다음 검색 범위로 순환
                                const scopes = ['personal', 'shared', 'all'] as const;
                                const currentIndex = scopes.indexOf(options.ragSearchScope);
                                const nextIndex = (currentIndex + 1) % scopes.length;
                                
                                if (nextIndex === 0) {
                                  // 마지막 범위(전체검색)에서 다시 클릭하면 비활성화
                                  setOptions(prev => ({
                                    ...prev,
                                    enableRag: false
                                  }));
                                } else {
                                  // 다음 범위로 이동
                                  setOptions(prev => ({
                                    ...prev,
                                    ragSearchScope: scopes[nextIndex]
                                  }));
                                }
                              }
                            }}
                            disabled={isLoading || isLoadingSessions}
                            className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${options.enableRag ?
                              'bg-green-300 dark:bg-green-800 text-green-900 dark:text-green-100 ring-1 ring-green-300 dark:ring-green-600' :
                              'bg-gray-300 dark:bg-gray-600 text-gray-900 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                              }`}
                            title={options.enableRag ? getRagScopeInfo(options.ragSearchScope).description : t('chat.documentSearch', '문서검색')}
                          >
                            <span className="text-sm">{options.enableRag ? getRagScopeInfo(options.ragSearchScope).icon : '📄'}</span>
                            <span>{options.enableRag ? getRagScopeInfo(options.ragSearchScope).label : t('chat.documentSearch', '문서검색')}</span>
                          </button>


                        </div>

                        {/* 웹검색 버튼 */}
                        <button
                          onClick={() => setOptions(prev => ({
                            ...prev,
                            enableWebSearch: !prev.enableWebSearch,
}))}
                                                    disabled={isLoading || isLoadingSessions}
                          className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${options.enableWebSearch ?
                            'bg-blue-300 dark:bg-blue-800 text-blue-900 dark:text-blue-100 ring-1 ring-blue-300 dark:ring-blue-600' :
                            'bg-gray-300 dark:bg-gray-600 text-gray-900 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                            }`}
                          title={t('chat.webSearch', '웹검색')}
                        >
                          <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9v-9m0-9v9" />
                          </svg>
                          <span>{t('chat.webSearch', '웹검색')}</span>
                        </button>

                        {/* 건강검진 분석 버튼 */}
                        <button
                          onClick={() => setOptions(prev => ({
                            ...prev,
                            enableHealthAnalysis: !prev.enableHealthAnalysis,
                          }))}
                          disabled={isLoading || isLoadingSessions}
                          className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${options.enableHealthAnalysis ?
                            'bg-green-300 dark:bg-green-800 text-green-900 dark:text-green-100 ring-1 ring-green-300 dark:ring-green-600' :
                            'bg-gray-300 dark:bg-gray-600 text-gray-900 dark:text-gray-300 hover:bg-gray-400 dark:hover:bg-gray-500'
                            }`}
                          title="건강검진 분석"
                        >
                          <span className="text-sm">🏥</span>
                          <span>{t('chat.healthAnalysis', '검진분석')}</span>
                        </button>

  
                      </div>

                      {/* 멘션된 문서 표시 */}
                      {mentionedDocs.length > 0 && (
                        <div className="mb-3 flex flex-wrap gap-1">
                          {mentionedDocs.map((doc, index) => (
                            <span
                              key={index}
                              className="inline-flex items-center px-2 py-1 bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200 rounded-full text-xs"
                            >
                              <svg className="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                <path d="M4 4a2 2 0 00-2 2v8a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2H4z" />
                              </svg>
                              {doc.displayName}
                              <button
                                onClick={() => removeMentionedDoc(doc.name)}
                                className="ml-1 text-blue-600 hover:text-blue-800"
                              >
                                ×
                              </button>
                            </span>
                          ))}
                        </div>
                      )}


                      {/* 입력창 영역 */}
                      <div className="relative">
                        <textarea
                          ref={(el) => {
                            if (el) {
                              textareaRef.current = el;
                              el.style.height = 'auto';
                              el.style.height = `${Math.min(Math.max(el.scrollHeight, 48), 200)}px`;
                            }
                          }}
                          value={inputMessage}
                          onChange={handleInputChange}
                          onKeyDown={handleKeyDown}
                          placeholder={isLoadingSessions ? t('chat.loadingSessions', '세션을 로드하는 중...') : isDragOver ? t('chat.dropFiles', '파일을 여기에 놓으세요...') : t('chat.askAnything', '무엇이든 물어보세요...')}
                          className="w-full px-0 py-2 pr-12 bg-transparent text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none resize-none border-0 text-base"
                          rows={1}
                          disabled={isLoading || isLoadingSessions}
                          style={{ minHeight: '48px', maxHeight: '200px' }}
                        />

                        {/* 문서 자동완성 드롭다운 */}
                        {showDocSuggestions && filteredDocsForMention.length > 0 && (
                          <div className="absolute bottom-full left-0 right-0 mb-1 bg-gray-50 dark:bg-gray-800 border border-gray-300 dark:border-gray-500 rounded-lg shadow-xl max-h-60 overflow-y-auto z-10 scrollbar-thin scrollbar-thumb-gray-300 dark:scrollbar-thumb-gray-600 scrollbar-track-transparent">
                            {filteredDocsForMention.map((doc, index) => (
                              <button
                                key={index}
                                onClick={() => selectDocument(doc.name)}
                                className={`w-full text-left px-3 py-2 flex items-center space-x-2 transition-colors ${index === selectedSuggestionIndex
                                  ? 'bg-blue-100 dark:bg-blue-900/40 text-blue-900 dark:text-blue-100'
                                  : 'hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-800 dark:text-gray-200'
                                  }`}
                              >
                                {getFileIcon(doc.name, doc.status)}
                                <span className="text-sm truncate flex-1" title={doc.name}>{doc.name}</span>
                              </button>
                            ))}
                          </div>
                        )}

                        <div className="absolute right-0 bottom-2 flex items-center space-x-2">
                          {/* 마이크 버튼 */}
                          <button
                            onClick={toggleSpeechRecognition}
                            disabled={isLoading || isLoadingSessions || isProcessingAudio}
                            className={`p-2 rounded-lg transition-colors ${isListening
                              ? 'bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 animate-pulse'
                              : isProcessingAudio
                                ? 'bg-yellow-100 dark:bg-yellow-900/30 text-yellow-600 dark:text-yellow-400'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-400 hover:bg-gray-200 dark:hover:bg-gray-600'
                              } disabled:opacity-50 disabled:cursor-not-allowed`}
                            title={isListening ? t('chat.voiceRecordingStop', '음성 녹음 중지') : isProcessingAudio ? t('chat.voiceProcessing', '음성 처리 중') : t('chat.voiceRecordingStart', '음성 녹음 시작')}
                          >
                            {isListening ? (
                              <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M12 14c1.66 0 3-1.34 3-3V5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm-1-9c0-.55.45-1 1-1s1 .45 1 1v6c0 .55-.45 1-1 1s-1-.45-1-1V5zm6 6c0 2.76-2.24 5-5 5s-5-2.24-5-5H5c0 3.53 2.61 6.43 6 6.92V21h2v-3.08c3.39-.49 6-3.39 6-6.92h-2z" />
                              </svg>
                            ) : isProcessingAudio ? (
                              <svg className="w-4 h-4 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                              </svg>
                            ) : (
                              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 11a7 7 0 01-7 7m0 0a7 7 0 00-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z" />
                              </svg>
                            )}
                          </button>

                          {/* 전송 버튼 */}
                          <button
                            onClick={isLoading && isStreamingRef.current ? stopStreaming : sendMessage}
                            disabled={
                              isLoading && isStreamingRef.current
                                ? false // 스트리밍 중에는 중지 버튼을 클릭할 수 있도록 활성화
                                : ((!inputMessage.trim() && uploadedFiles.length === 0) || isLoadingSessions)
                            }
                            className={`p-2 text-white rounded-lg focus:outline-none focus:ring-2 transition-colors ${isLoading && isStreamingRef.current
                              ? 'bg-red-600 hover:bg-red-700 focus:ring-red-500'
                              : 'bg-blue-600 hover:bg-blue-700 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed'
                              }`}
                          >
                            {isLoading && isStreamingRef.current ? (
                              <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                                <rect x="6" y="6" width="12" height="12" />
                              </svg>
                            ) : (
                              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                              </svg>
                            )}
                          </button>
                        </div>
                      </div>
                    </div>

                    {/* 숨겨진 파일 입력 */}
                    <input
                      ref={fileInputRef}
                      type="file"
                      accept="image/*,.pdf,.doc,.docx,.txt,.md,.csv,.json"
                      onChange={handleFileSelect}
                      className="hidden"
                    />

                    {/* 오류 메시지 */}
                    {error && (
                      <div className="mt-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                        <p className="text-sm text-red-800 dark:text-red-200">{error}</p>
                      </div>
                    )}
                  </div>
                </div>
              </>
            )}
          </div>
        </div>

        {/* 삭제 확인 모달 */}
        {showDeleteModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="delete-modal" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="delete-modal-header">
                <div className="delete-modal-icon">
                  <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                  </svg>
                </div>
                <div className="delete-modal-text">
                  <h3 className="delete-modal-title">
                    {t('chat.deleteConversation', '대화 삭제')}
                  </h3>
                  <p className="delete-modal-subtitle">
                    {t('chat.actionCannotBeUndone', '이 작업은 되돌릴 수 없습니다.')}
                  </p>
                </div>
              </div>

              <p className="delete-modal-content">
                {t('chat.deleteConversationConfirm', '선택한 대화를 영구적으로 삭제하시겠습니까? 대화 내용과 모든 메시지가 삭제됩니다.')}
              </p>

              <div className="delete-modal-buttons">
                <button
                  onClick={cancelDelete}
                  className="delete-modal-button delete-modal-button-cancel"
                >
                  Cancel
                </button>
                <button
                  onClick={confirmDelete}
                  className="delete-modal-button delete-modal-button-delete"
                >
                  Delete
                </button>
              </div>
            </div>
          </div>
        )}

        {/* 전체 삭제 확인 모달 */}
        {showDeleteAllModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="delete-modal" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="delete-modal-header">
                <div className="delete-modal-icon">
                  <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                  </svg>
                </div>
                <div className="delete-modal-text">
                  <h3 className="delete-modal-title">
                    {t('chat.deleteAllConversations', '모든 대화 삭제')}
                  </h3>
                  <p className="delete-modal-subtitle">
                    {t('chat.actionCannotBeUndone', '이 작업은 되돌릴 수 없습니다.')}
                  </p>
                </div>
              </div>

              <p className="delete-modal-content">
                {t('chat.deleteAllConversationsConfirm', '모든 대화를 영구적으로 삭제하시겠습니까? 전체 대화 목록과 모든 메시지가 삭제됩니다.')}
              </p>

              <div className="delete-modal-buttons">
                <button
                  onClick={cancelDeleteAll}
                  className="delete-modal-button delete-modal-button-cancel"
                >
                  Cancel
                </button>
                <button
                  onClick={confirmDeleteAll}
                  className="delete-modal-button delete-modal-button-delete"
                >
                  Delete
                </button>
              </div>
            </div>
          </div>
        )}

        {/* 토스트 알림 */}
        {showToast && (
          <div className="fixed top-1/4 left-1/2 transform -translate-x-1/2 -translate-y-1/2 z-50">
            <div
              className={`px-6 py-5 rounded-lg shadow-lg flex items-start space-x-6 ${showToast.type === 'success'
                  ? 'bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-200 border border-green-200 dark:border-green-700'
                  : 'bg-red-100 dark:bg-red-900/30 text-red-800 dark:text-red-200 border border-red-200 dark:border-red-700'
                }`}
            >
              {showToast.type === 'success' ? (
                <svg className="w-6 h-6 mt-1 shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path
                    fillRule="evenodd"
                    d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                    clipRule="evenodd"
                  />
                </svg>
              ) : (
                <svg className="w-6 h-6 mt-1 shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path
                    fillRule="evenodd"
                    d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z"
                    clipRule="evenodd"
                  />
                </svg>
              )}

              {/* ✅ 변경된 메시지 렌더링 */}
              <div
                className={`text-sm font-semibold whitespace-pre-wrap leading-relaxed
              ${showToast.type === 'success' ? 'text-green-800 dark:text-green-100' : 'text-red-800 dark:text-red-100'}
            `}
              >
                {showToast.message}
              </div>

              {/* 닫기 버튼 */}
              <button
                onClick={() => setShowToast(null)}
                className="ml-auto text-sm font-bold rounded px-2 py-1 hover:bg-black/10 dark:hover:bg-white/10 transition"
              >
                [close]
              </button>
            </div>
          </div>
        )}

        {/* 시스템 프롬프트 편집 모달 */}
        {showSystemPromptModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl w-full max-w-6xl max-h-[90vh] overflow-hidden">
              {/* 모달 헤더 */}
              <div className="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-700">
                <div>
                  <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                    시스템 프롬프트 편집
                  </h2>
                  {systemPromptProvider && (
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                      프로바이더: {systemPromptProvider.toUpperCase()} 
                      {isSystemPromptCustom ? ' (사용자 정의)' : ' (기본 템플릿)'}
                    </p>
                  )}
                </div>
                <button
                  onClick={() => setShowSystemPromptModal(false)}
                  className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors"
                >
                  <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                  </svg>
                </button>
              </div>

              {/* 모달 내용 */}
              <div className="p-6 overflow-y-auto max-h-[70vh]">
                {isLoadingSystemPrompt ? (
                  <div className="flex items-center justify-center py-12">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                    <span className="ml-3 text-gray-600 dark:text-gray-400">로딩 중...</span>
                  </div>
                ) : (
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        프롬프트 내용
                      </label>
                      <textarea
                        value={systemPrompt}
                        onChange={(e) => setSystemPrompt(e.target.value)}
                        className="w-full h-96 px-4 py-3 text-sm border border-gray-300 dark:border-gray-600 rounded-lg bg-[var(--card-bg)] text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none font-mono"
                        placeholder="시스템 프롬프트를 입력하세요..."
                      />
                    </div>
                    
                    {!isSystemPromptCustom && (
                      <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                        <div className="flex items-start">
                          <div className="flex-shrink-0">
                            <svg className="w-5 h-5 text-blue-400" fill="currentColor" viewBox="0 0 20 20">
                              <path fillRule="evenodd" d="M18 10a8 8 0 100-16 8 8 0 000 16zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                            </svg>
                          </div>
                          <div className="ml-3">
                            <h3 className="text-sm font-medium text-blue-800 dark:text-blue-200">
                              기본 템플릿 사용 중
                            </h3>
                            <div className="mt-2 text-sm text-blue-700 dark:text-blue-300">
                              <p>현재 {systemPromptProvider.toUpperCase()} 프로바이더의 기본 템플릿을 사용하고 있습니다. 수정하면 사용자 정의 프롬프트로 저장됩니다.</p>
                            </div>
                          </div>
                        </div>
                      </div>
                    )}
                  </div>
                )}
              </div>

              {/* 모달 푸터 */}
              <div className="flex items-center justify-end space-x-3 p-6 border-t border-gray-200 dark:border-gray-700">
                <button
                  onClick={() => setShowSystemPromptModal(false)}
                  className="px-4 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 transition-colors"
                >
                  취소
                </button>
                <button
                  onClick={async () => {
                    await saveSystemPrompt();
                    setShowSystemPromptModal(false);
                  }}
                  disabled={isLoadingSystemPrompt}
                  className="px-4 py-2 text-sm bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  {isLoadingSystemPrompt ? '저장 중...' : '저장'}
                </button>
              </div>
            </div>
          </div>
        )}

        {/* 문서 미리보기 모달 */}
        <DocumentPreviewModal
          isOpen={showDocPreview}
          onClose={() => setShowDocPreview(false)}
          fileName={previewFileName}
          userId={user?.username || ''}
        />

        {/* Excel 전처리 모달 */}
        <ExcelPreprocessModal
          isOpen={showExcelModal}
          onClose={() => setShowExcelModal(false)}
          onProcess={handleExcelProcess}
        />

        {/* GraphRAG 모달 (GraphRAG 활성화된 경우에만) */}
        {graphRAGEnabled && (
          <GraphRAGModal
            isOpen={showGraphRAGModal}
            onClose={() => setShowGraphRAGModal(false)}
            query={currentGraphQuery}
            userId={user?.username || ''}
          />
        )}

        {/* 프로젝트 관리 모달 */}
        {showProjectModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl w-full max-w-md">
              <div className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                    {projectModalType === 'create' && '새 프로젝트 만들기'}
                    {projectModalType === 'edit' && '프로젝트 수정'}
                    {projectModalType === 'delete' && '프로젝트 삭제'}
                  </h3>
                  <button
                    onClick={() => setShowProjectModal(false)}
                    className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200"
                  >
                    <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                    </svg>
                  </button>
                </div>

                {projectModalType === 'delete' ? (
                  <div>
                    <p className="text-gray-600 dark:text-gray-300 mb-6">
                      프로젝트 "<span className="font-medium">{projectToEdit?.name}</span>"을(를) 삭제하시겠습니까?
                      <br />
                      <span className="text-sm text-red-600 dark:text-red-400">
                        이 작업은 되돌릴 수 없으며, 프로젝트에 포함된 모든 문서도 함께 삭제됩니다.
                      </span>
                    </p>
                    <div className="flex justify-end space-x-3">
                      <button
                        onClick={() => setShowProjectModal(false)}
                        className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 rounded-md transition-colors"
                      >
                        취소
                      </button>
                      <button
                        onClick={async () => {
                          try {
                            await deleteProject(projectToEdit.name);
                            setShowProjectModal(false);
                          } catch (error) {
                            // 에러는 deleteProject 함수에서 토스트로 표시됨
                          }
                        }}
                        className="px-4 py-2 text-sm font-medium text-white bg-red-600 hover:bg-red-700 rounded-md transition-colors"
                      >
                        삭제
                      </button>
                    </div>
                  </div>
                ) : (
                  <div>
                    <div className="mb-4">
                      <label htmlFor="projectName" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        프로젝트명 *
                      </label>
                      <input
                        type="text"
                        id="projectName"
                        value={newProjectName}
                        onChange={(e) => setNewProjectName(e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                        placeholder="프로젝트명을 입력하세요"
                        autoFocus
                      />
                    </div>
                    <div className="mb-6">
                      <label htmlFor="projectDescription" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        설명 (선택사항)
                      </label>
                      <textarea
                        id="projectDescription"
                        value={newProjectDescription}
                        onChange={(e) => setNewProjectDescription(e.target.value)}
                        rows={3}
                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                        placeholder="프로젝트에 대한 간단한 설명을 입력하세요"
                      />
                    </div>
                    <div className="flex justify-end space-x-3">
                      <button
                        onClick={() => setShowProjectModal(false)}
                        className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 rounded-md transition-colors"
                      >
                        취소
                      </button>
                      <button
                        onClick={async () => {
                          if (!newProjectName.trim()) {
                            setShowToast({ message: '프로젝트명을 입력해주세요.', type: 'error' });
                            return;
                          }
                          try {
                            await createProject(newProjectName.trim(), newProjectDescription.trim() || undefined);
                            setShowProjectModal(false);
                            setNewProjectName('');
                            setNewProjectDescription('');
                          } catch (error) {
                            // 에러는 createProject 함수에서 토스트로 표시됨
                          }
                        }}
                        disabled={!newProjectName.trim()}
                        className="px-4 py-2 text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed rounded-md transition-colors"
                      >
                        {projectModalType === 'create' ? '생성' : '수정'}
                      </button>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </div>
        )}
      </div>

      {/* 파일관리 모달 */}
      {showFileManagementModal && (
        <FileManagementModal
          isOpen={showFileManagementModal}
          onClose={() => setShowFileManagementModal(false)}
          userId={user?.username || ''}
          userRole={user?.role || 'user'}
        />
      )}
    </ClientOnly>
  );
}

// 메인 컴포넌트를 Suspense로 감싸기 - 개선된 버전
export default function ChatPage() {
  return (
    <Suspense fallback={
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-2 text-gray-600 dark:text-gray-400">채팅을 로딩하고 있습니다...</p>
        </div>
      </div>
    }>
      <ChatPageContent />
    </Suspense>
  );
}
//이미지 탑 프레임 내비게이션 금지 회피
export async function openImageInNewTab(dataOrBase64: string, mime = 'image/png') {
  const dataUrl = dataOrBase64.startsWith('data:')
    ? dataOrBase64
    : `data:${mime};base64,${dataOrBase64}`;

  // data: URL -> Blob
  const blob = await (await fetch(dataUrl)).blob();
  const objectUrl = URL.createObjectURL(blob);

  // 새 탭으로 보기 (다운로드 아님)
  // noopener/noreferrer로 opener 참조 차단
  const w = window.open(objectUrl, '_blank', 'noopener,noreferrer');

  // 메모리 누수 방지 (탭 열린 뒤 적당히 해제)
  setTimeout(() => URL.revokeObjectURL(objectUrl), 60_000);

  // 팝업 차단 시 대체 (사용자 클릭 핸들러 안에서 호출하면 차단 가능성↓)
  if (!w) {
    const a = document.createElement('a');
    a.href = objectUrl;
    a.target = '_blank';
    a.rel = 'noopener noreferrer';
    document.body.appendChild(a);
    a.click();
    a.remove();
    setTimeout(() => URL.revokeObjectURL(objectUrl), 60_000);
  }
}