"use client";
import React, { useState, useEffect } from "react";
import { getAuthHeaders } from "@/utils/api";

interface PdfIframeViewerProps {
  filePath: string;
}

function PdfIframeViewer({ filePath }: PdfIframeViewerProps) {
  const [blobUrl, setBlobUrl] = useState<string>('');
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchPdfFile = async () => {
      if (!filePath) {
        setError('파일 경로가 없습니다.');
        setLoading(false);
        return;
      }

      try {
        setLoading(true);
        setError(null);
        
        console.log('[PdfViewer] PDF 파일 가져오기 시작:', filePath);
        
        // 인증 헤더와 함께 파일 fetch
        const headers = getAuthHeaders();
        const response = await fetch(filePath, {
          method: 'GET',
          headers
        });
        
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        // Response를 Blob으로 변환
        const blob = await response.blob();
        console.log('[PdfViewer] PDF Blob 생성 완료, 크기:', blob.size);
        
        // Blob URL 생성
        const url = URL.createObjectURL(blob);
        // PDF 뷰어에 제목 설정을 위한 파라미터 추가
        const pdfUrl = `${url}#toolbar=1&navpanes=0&scrollbar=1&view=FitH`;
        setBlobUrl(pdfUrl);
        
      } catch (err) {
        console.error('[PdfViewer] PDF 로딩 오류:', err);
        setError(err instanceof Error ? err.message : '파일을 불러올 수 없습니다.');
      } finally {
        setLoading(false);
      }
    };

    fetchPdfFile();

    // 컴포넌트 언마운트 시 Blob URL 정리
    return () => {
      if (blobUrl) {
        URL.revokeObjectURL(blobUrl);
      }
    };
  }, [filePath]);

  // 로딩 중
  if (loading) {
    return (
      <div className="w-full flex-1 flex flex-col items-center justify-center bg-gray-50 dark:bg-gray-900 rounded-lg" style={{ minHeight: 'calc(100vh - 200px)' }}>
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600 dark:text-gray-400">PDF 문서를 불러오는 중...</p>
        </div>
      </div>
    );
  }

  // 오류 발생
  if (error) {
    return (
      <div className="w-full flex-1 flex flex-col items-center justify-center bg-gray-50 dark:bg-gray-900 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600" style={{ minHeight: 'calc(100vh - 200px)' }}>
        <div className="text-center p-8">
          <div className="text-6xl mb-4">❌</div>
          <h3 className="text-lg font-semibold mb-2 text-gray-700 dark:text-gray-300">PDF 로딩 오류</h3>
          <p className="text-gray-500 dark:text-gray-400 mb-4">{error}</p>
          <button 
            onClick={() => window.location.reload()} 
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            새로고침
          </button>
        </div>
      </div>
    );
  }

  // PDF 표시
  return (
    <iframe
      src={blobUrl}
      style={{ width: '100%', height: 'calc(100vh - 200px)', minHeight: 'calc(100vh - 200px)', border: 'none' }}
      title="PDF 미리보기"
    />
  );
}

// React DevTools에서 컴포넌트 이름 명확하게 표시
PdfIframeViewer.displayName = 'PdfIframeViewer';

export default PdfIframeViewer; 