'use client';

import React from 'react';
import { Handle, Position } from 'reactflow';
import type { NodeType } from '@/types/flowai';

interface CustomNodeData {
  label: string;
  description?: string;
  nodeType: NodeType;
  selected: boolean;
  isRunning?: boolean;
  hasError?: boolean;
  onSelect: () => void;
  onUpdate: (updates: any) => void;
  onDelete: () => void;
  onExecute?: () => void;
  onStop?: () => void;
  onDuplicate?: () => void;
}

interface CustomNodeProps {
  data: CustomNodeData;
  selected?: boolean;
}

// 노드 아이콘 가져오기
const getNodeIcon = (type: NodeType): string => {
  switch (type) {
    case 'input':
      return '📝';
    case 'process':
      return '🤖';
    case 'output':
      return '📋';
    case 'condition':
      return '🔀';
    case 'loop':
      return '🔄';
    default:
      return '⭕';
  }
};

// 노드 색상 클래스 가져오기
const getNodeColorClass = (type: NodeType): string => {
  switch (type) {
    case 'input':
      return 'flowai-node-input-card';
    case 'process':
      return 'flowai-node-process-card';
    case 'output':
      return 'flowai-node-output-card';
    case 'condition':
      return 'flowai-node-condition-card';
    case 'loop':
      return 'flowai-node-loop-card';
    default:
      return 'bg-gray-50 border-gray-200 dark:bg-gray-800 dark:border-gray-700';
  }
};

export default function CustomNode({ data, selected }: CustomNodeProps) {
  const [isHovered, setIsHovered] = React.useState(false);

  return (
    <div
      className={`relative w-56 min-h-[96px] p-4 border rounded-xl cursor-pointer transition-all duration-150 ${
        getNodeColorClass(data.nodeType)
      } ${
        selected || data.selected
          ? 'ring-2 ring-blue-500 shadow-lg scale-[1.02]'
          : 'shadow-sm hover:shadow-md hover:scale-[1.01]'
      }`}
      onClick={data.onSelect}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
    >
      {/* Hover Action Buttons (n8n style) */}
      {isHovered && (
        <div className="absolute -top-10 left-1/2 transform -translate-x-1/2 flex items-center space-x-1 bg-gray-800 dark:bg-gray-800 rounded-lg shadow-xl border border-gray-700 dark:border-gray-600 p-1 z-50 animate-fadeIn">
          {/* Execute Node Button */}
          <button
            onClick={(e) => {
              e.stopPropagation();
              if (data.onExecute) {
                data.onExecute();
              }
            }}
            className="group flex items-center justify-center w-8 h-8 rounded-md bg-transparent hover:bg-blue-600/30 transition-all duration-150"
            title="노드 실행"
          >
            <svg 
              className="w-5 h-5 text-gray-300 dark:text-gray-400 group-hover:text-blue-400 dark:group-hover:text-blue-400 transition-colors" 
              fill="none" 
              stroke="currentColor" 
              viewBox="0 0 24 24"
            >
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z" />
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
          </button>

          {/* Pause/Stop Button (for running nodes) */}
          {data.isRunning && (
            <button
              onClick={(e) => {
                e.stopPropagation();
                if (data.onStop) {
                  data.onStop();
                }
              }}
              className="group flex items-center justify-center w-8 h-8 rounded-md bg-transparent hover:bg-orange-600/30 transition-all duration-150"
              title="실행 중지"
            >
              <svg 
                className="w-5 h-5 text-gray-300 dark:text-gray-400 group-hover:text-orange-400 dark:group-hover:text-orange-400 transition-colors" 
                fill="none" 
                stroke="currentColor" 
                viewBox="0 0 24 24"
              >
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 9v6m4-6v6m7-3a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            </button>
          )}

          {/* Duplicate Node Button */}
          <button
            onClick={(e) => {
              e.stopPropagation();
              if (data.onDuplicate) {
                data.onDuplicate();
              }
            }}
            className="group flex items-center justify-center w-8 h-8 rounded-md bg-transparent hover:bg-green-600/30 transition-all duration-150"
            title="노드 복제"
          >
            <svg 
              className="w-5 h-5 text-gray-300 dark:text-gray-400 group-hover:text-green-400 dark:group-hover:text-green-400 transition-colors" 
              fill="none" 
              stroke="currentColor" 
              viewBox="0 0 24 24"
            >
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
            </svg>
          </button>

          {/* Delete Node Button */}
          <button
            onClick={(e) => {
              e.stopPropagation();
              data.onDelete();
            }}
            className="group flex items-center justify-center w-8 h-8 rounded-md bg-transparent hover:bg-red-600/30 transition-all duration-150"
            title="노드 삭제"
          >
            <svg 
              className="w-5 h-5 text-gray-300 dark:text-gray-400 group-hover:text-red-400 dark:group-hover:text-red-400 transition-colors" 
              fill="none" 
              stroke="currentColor" 
              viewBox="0 0 24 24"
            >
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
            </svg>
          </button>
        </div>
      )}

      {/* Execution Status Indicator */}
      {data.isRunning && (
        <div className="absolute -top-2 -right-2 w-4 h-4 bg-green-500 rounded-full animate-pulse border-2 border-white dark:border-gray-800" />
      )}

      {/* Error Status Indicator */}
      {data.hasError && (
        <div className="absolute -top-2 -right-2 w-4 h-4 bg-red-500 rounded-full border-2 border-white dark:border-gray-800" />
      )}

      {/* 입력 핸들들 */}
      {/* 좌측 입력 핸들 (input 타입이 아닌 경우) */}
      {data.nodeType !== 'input' && (
        <Handle
          type="target"
          position={Position.Left}
          id="left"
          className="w-4 h-4 !bg-gray-400 !border-2 !border-white hover:!bg-green-500 transition-colors"
          style={{ left: -8 }}
        />
      )}

      {/* 상단 입력 핸들 (input 타입이 아닌 경우) */}
      {data.nodeType !== 'input' && (
        <Handle
          type="target"
          position={Position.Top}
          id="top"
          className="w-4 h-4 !bg-gray-400 !border-2 !border-white hover:!bg-green-500 transition-colors"
          style={{ top: -8 }}
        />
      )}

      {/* 출력 핸들들 */}
      {/* 우측 출력 핸들 (output 타입이 아닌 경우) */}
      {data.nodeType !== 'output' && (
        <Handle
          type="source"
          position={Position.Right}
          id="right"
          className="w-4 h-4 !bg-blue-500 !border-2 !border-white hover:!bg-blue-600 transition-colors"
          style={{ right: -8 }}
        />
      )}

      {/* 하단 출력 핸들 (output 타입이 아닌 경우) */}
      {data.nodeType !== 'output' && (
        <Handle
          type="source"
          position={Position.Bottom}
          id="bottom"
          className="w-4 h-4 !bg-blue-500 !border-2 !border-white hover:!bg-blue-600 transition-colors"
          style={{ bottom: -8 }}
        />
      )}

      {/* 노드 헤더 */}
      <div className="flex items-center justify-between mb-3">
        <div className="flex items-center space-x-3">
          <span className="text-xl">{getNodeIcon(data.nodeType)}</span>
          <span className="font-semibold text-gray-800 dark:text-gray-200">
            {data.label}
          </span>
        </div>
      </div>

      {/* 노드 내용 */}
      <div className="text-sm text-gray-600 dark:text-gray-400 line-clamp-2 leading-relaxed">
        {data.description || '설정 필요'}
      </div>
    </div>
  );
}