'use client';

import React, { useState, useEffect, useCallback } from 'react';
import { X, Upload, FileSpreadsheet, Brain, Settings2, Eye, Columns, Filter, CheckSquare, Square, Sparkles, User } from 'lucide-react';

interface ExcelPreprocessModalProps {
  isOpen: boolean;
  onClose: () => void;
  onProcess: (options: ExcelProcessOptions) => void;
}

interface ExcelProcessOptions {
  file: File;
  mode: 'ai' | 'manual';
  selectedSheet: string;
  selectedColumns: string[];
  dataDescription?: string;
  rowFilter?: {
    column: string;
    operator: 'equals' | 'contains' | 'greaterThan' | 'lessThan';
    value: string;
  };
  headerNormalization: boolean;
  preserveFormatting: boolean;
  aiRecommendations?: {
    structure: string;
    columns: string[];
    filters: any[];
  };
  dataSplitting?: {
    enabled: boolean;
    maxRowsPerFile: number;
    preserveHeaders: boolean;
  };
}

interface SheetData {
  name: string;
  headers: string[];
  preview: any[][];
  rowCount: number;
  columnCount: number;
}

export default function ExcelPreprocessModal({ isOpen, onClose, onProcess }: ExcelPreprocessModalProps) {
  const [processingStatus, setProcessingStatus] = useState<{
    isProcessing: boolean;
    step: string;
    progress: number;
  }>({
    isProcessing: false,
    step: '',
    progress: 0
  });
  const [file, setFile] = useState<File | null>(null);
  const [mode, setMode] = useState<'ai' | 'manual'>('manual');
  const [sheets, setSheets] = useState<SheetData[]>([]);
  const [selectedSheet, setSelectedSheet] = useState<string>('');
  const [selectedColumns, setSelectedColumns] = useState<string[]>([]);
  const [headerNormalization, setHeaderNormalization] = useState(true);
  const [preserveFormatting, setPreserveFormatting] = useState(true);
  const [isProcessing, setIsProcessing] = useState(false);
  const [aiRecommendations, setAiRecommendations] = useState<any>(null);
  const [previewData, setPreviewData] = useState<any[][]>([]);
  const [dataDescription, setDataDescription] = useState('');
  const [rowFilter, setRowFilter] = useState<any>({
    enabled: false,
    column: '',
    operator: 'contains',
    value: ''
  });
  const [dataSplitting, setDataSplitting] = useState({
    enabled: false,
    maxRowsPerFile: 100,
    preserveHeaders: true
  });

  // Case 2 수정: 모달 상태 초기화 함수
  const resetModalState = useCallback(() => {
    setFile(null);
    setMode('manual');
    setSheets([]);
    setSelectedSheet('');
    setSelectedColumns([]);
    setHeaderNormalization(true);
    setPreserveFormatting(true);
    setIsProcessing(false);
    setAiRecommendations(null);
    setPreviewData([]);
    setDataDescription('');
    setRowFilter({
      enabled: false,
      column: '',
      operator: 'contains',
      value: ''
    });
    setDataSplitting({
      enabled: false,
      maxRowsPerFile: 100,
      preserveHeaders: true
    });
  }, []);

  // Case 2 수정: 모달이 닫힐 때 상태 초기화
  useEffect(() => {
    if (!isOpen) {
      resetModalState();
    }
  }, [isOpen, resetModalState]);

  // 파일 업로드 처리
  const handleFileUpload = useCallback(async (e: React.ChangeEvent<HTMLInputElement>) => {
    const uploadedFile = e.target.files?.[0];
    if (!uploadedFile) return;

    setFile(uploadedFile);
    setIsProcessing(true);

    try {
      // 서버에서 파일 분석 요청
      const formData = new FormData();
      formData.append('file', uploadedFile);

      const response = await fetch('/api/preprocess/excel/analyze', {
        method: 'POST',
        body: formData
      });

      if (response.ok) {
        const result = await response.json();
        const sheetsData = result.data.sheets;

        setSheets(sheetsData);
        if (sheetsData.length > 0) {
          setSelectedSheet(sheetsData[0].name);
          setSelectedColumns(sheetsData[0].headers);
          setPreviewData(sheetsData[0].preview);
        }

        // 파일 업로드 단계에서는 AI 분석을 실행하지 않음
        // 사용자가 처리 모드를 선택한 후에 실행됨
      } else {
        throw new Error('파일 분석 실패');
      }
    } catch (error) {
      console.error('파일 읽기 오류:', error);
      alert('Excel 파일을 읽는 중 오류가 발생했습니다.');
    } finally {
      setIsProcessing(false);
    }
  }, []);

  // AI 추천 요청
  const requestAiRecommendations = async (file: File, sheets: SheetData[]) => {
    try {
      // 현재 선택된 시트에 대해서만 AI 분석 수행
      const currentSheet = sheets.find((s) => s.name === selectedSheet);
      if (!currentSheet) {
        console.error('❌ 선택된 시트를 찾을 수 없음:', selectedSheet);
        return;
      }

      console.log('🎯 선택된 시트에 대한 AI 분석 시작:', selectedSheet);
      console.log('📊 시트 정보:', {
        이름: currentSheet.name,
        행수: currentSheet.rowCount,
        열수: currentSheet.columnCount,
        헤더: currentSheet.headers.slice(0, 5)
      });

      const formData = new FormData();
      formData.append('file', file);
      // 선택된 시트만 전달
      formData.append(
        'sheets',
        JSON.stringify([
          {
            name: currentSheet.name,
            headers: currentSheet.headers,
            sampleData: currentSheet.preview.slice(0, 5),
            rowCount: currentSheet.rowCount, // 실제 전체 행 수 추가
            columnCount: currentSheet.columnCount // 실제 전체 열 수 추가
          }
        ])
      );

      const response = await fetch('/api/preprocess/excel/recommend', {
        method: 'POST',
        body: formData
      });

      if (response.ok) {
        const recommendations = await response.json();

        // 🔍 개발자 콘솔에서 AI 응답 확인
        console.log('🤖 AI 분석 응답:', recommendations);
        console.log('📊 AI 생성 데이터:', {
          제목: recommendations.data.aiGeneratedTitle,
          설명: recommendations.data.aiGeneratedDescription,
          카테고리: recommendations.data.dataCategory,
          추천시트: recommendations.data.selectedSheet,
          추천열: recommendations.data.columns,
          신뢰도: recommendations.data.confidence,
          핵심인사이트: recommendations.data.keyInsights
        });

        setAiRecommendations(recommendations.data);

        // AI 추천사항 자동 적용 (시트는 이미 선택되어 있으므로 열만 업데이트)
        if (recommendations.data.columns) {
          console.log('✅ AI 추천 열 자동 선택:', recommendations.data.columns);
          setSelectedColumns(recommendations.data.columns);
        }

        // 선택된 시트 확인 (디버깅용)
        console.log('✅ 현재 선택된 시트 유지:', selectedSheet);

        // AI 생성 데이터 설명 자동 적용
        if (recommendations.data.aiGeneratedDescription) {
          console.log('✅ 데이터 설명 자동 입력:', recommendations.data.aiGeneratedDescription);
          setDataDescription(recommendations.data.aiGeneratedDescription);
        }
      }
    } catch (error) {
      console.error('AI 추천 요청 실패:', error);
    }
  };

  // 처리 모드 변경 처리
  const handleModeChange = async (newMode: 'ai' | 'manual') => {
    setMode(newMode);

    // AI 모드로 변경하고 파일이 있으면 AI 분석 실행
    if (newMode === 'ai' && file && sheets.length > 0) {
      // 기존 AI 추천 초기화 (시트가 변경되었을 수 있으므로)
      setAiRecommendations(null);
      setDataDescription('');

      console.log('🤖 AI 모드 전환 - 새로운 분석 시작, 선택된 시트:', selectedSheet);
      setIsProcessing(true);
      await requestAiRecommendations(file, sheets);
      setIsProcessing(false);
    }

    // 수동 모드로 변경하면 AI 추천 초기화
    if (newMode === 'manual') {
      setAiRecommendations(null);
      setDataDescription('');
      // 현재 선택된 시트 기준으로 설정 유지 (첫 번째 시트로 강제 변경하지 않음)
      const currentSheet = sheets.find((s) => s.name === selectedSheet);
      if (currentSheet) {
        setSelectedColumns(currentSheet.headers);
      }
    }
  };

  // 시트 변경 처리
  const handleSheetChange = (sheetName: string) => {
    setSelectedSheet(sheetName);
    const sheet = sheets.find((s) => s.name === sheetName);
    if (sheet) {
      setSelectedColumns(sheet.headers);
      setPreviewData(sheet.preview);
    }

    // 시트 변경 시 AI 추천 상태 초기화
    setAiRecommendations(null);
    setDataDescription('');

    console.log('🔄 시트 변경:', sheetName, '- AI 추천 상태 초기화됨');
  };

  // 열 선택 토글
  const toggleColumn = (column: string) => {
    setSelectedColumns((prev) => (prev.includes(column) ? prev.filter((c) => c !== column) : [...prev, column]));
  };

  // 전처리 실행
  const handleProcess = async () => {
    if (!file) return;

    const options: ExcelProcessOptions = {
      file,
      mode,
      selectedSheet,
      selectedColumns,
      dataDescription,
      headerNormalization,
      preserveFormatting,
      rowFilter: rowFilter.enabled ? rowFilter : undefined,
      aiRecommendations,
      dataSplitting: dataSplitting.enabled ? dataSplitting : undefined
    };

    // 처리 중 상태로 설정
    setProcessingStatus({
      isProcessing: true,
      step: '파일 분석 중...',
      progress: 10
    });

    try {
      // 진행 상태를 시뮬레이션 (실제로는 서버에서 진행률을 받아와야 함)
      setTimeout(() => setProcessingStatus((prev) => ({ ...prev, step: '데이터 추출 중...', progress: 30 })), 1000);
      setTimeout(() => setProcessingStatus((prev) => ({ ...prev, step: '형식 변환 중...', progress: 60 })), 2000);
      setTimeout(() => setProcessingStatus((prev) => ({ ...prev, step: '최종 처리 중...', progress: 90 })), 3000);

      // 전처리 실행
      await onProcess(options);

      // 완료 후 상태 초기화 및 모달 닫기
      setTimeout(() => {
        setProcessingStatus({ isProcessing: false, step: '', progress: 0 });
        resetModalState();
        onClose();
      }, 4000);
    } catch (error) {
      console.error('전처리 실패:', error);
      setProcessingStatus({ isProcessing: false, step: '', progress: 0 });
      alert('전처리 중 오류가 발생했습니다.');
    }
  };

  if (!isOpen) return null;

  // 전처리 중일 때 표시할 오버레이
  if (processingStatus.isProcessing) {
    return (
      <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div
          className="rounded-lg shadow-xl w-full max-w-md p-8"
          style={{
            backgroundColor: 'var(--sidebar-bg)',
            borderColor: 'var(--card-border)'
          }}
        >
          <div className="text-center">
            {/* 로딩 스피너 */}
            <div className="mb-6 flex justify-center">
              <div className="relative">
                <div className="w-16 h-16 border-4 rounded-full animate-pulse" style={{ borderColor: 'var(--primary-200)' }}></div>
                <div className="absolute top-0 left-0 w-16 h-16 border-4 rounded-full border-t-transparent animate-spin" style={{ borderColor: 'var(--primary-600)' }}></div>
              </div>
            </div>

            {/* 상태 메시지 */}
            <h3 className="text-xl font-semibold mb-2" style={{ color: 'var(--text-primary)' }}>
              Excel 파일 전처리 중
            </h3>
            <p className="text-sm mb-6" style={{ color: 'var(--text-secondary)' }}>
              {processingStatus.step || '처리 중입니다...'}
            </p>

            {/* 프로그레스 바 */}
            <div className="w-full rounded-full h-2 mb-4" style={{ backgroundColor: 'var(--neutral-200)' }}>
              <div
                className="h-2 rounded-full transition-all duration-300"
                style={{
                  width: `${processingStatus.progress}%`,
                  backgroundColor: 'var(--primary-600)'
                }}
              ></div>
            </div>
            <p className="text-sm" style={{ color: 'var(--text-muted)' }}>
              {processingStatus.progress}% 완료
            </p>

            {/* 안내 메시지 */}
            <div className="mt-6 p-3 rounded-lg" style={{ backgroundColor: 'var(--info-bg)', borderColor: 'var(--info-border)' }}>
              <p className="text-xs" style={{ color: 'var(--text-secondary)' }}>
                💡 파일 크기에 따라 처리 시간이 다를 수 있습니다.
              </p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="rounded-lg shadow-xl w-full max-w-7xl max-h-[95vh] overflow-hidden" style={{ backgroundColor: 'var(--sidebar-bg)', borderColor: 'var(--card-border)' }}>
        {/* 헤더 */}
        <div
          className="flex items-center justify-between p-6 border-b"
          style={{
            backgroundColor: 'var(--sidebar-bg)',
            borderColor: 'var(--card-border)'
          }}
        >
          <div className="flex items-center gap-3">
            <FileSpreadsheet className="w-6 h-6 text-blue-600 dark:text-blue-400" />
            <h2 className="text-2xl font-bold" style={{ color: 'var(--text-primary)' }}>
              스마트 데이터
            </h2>
          </div>
          <button
            onClick={onClose}
            className="p-2 rounded-lg transition-colors"
            style={{
              backgroundColor: 'transparent',
              color: 'var(--text-secondary)'
            }}
            onMouseEnter={(e) => {
              e.currentTarget.style.backgroundColor = 'var(--card-hover)';
              e.currentTarget.style.color = 'var(--text-primary)';
            }}
            onMouseLeave={(e) => {
              e.currentTarget.style.backgroundColor = 'transparent';
              e.currentTarget.style.color = 'var(--text-secondary)';
            }}
          >
            <X className="w-6 h-6" />
          </button>
        </div>

        {/* 본문 */}
        <div className="p-6 overflow-y-auto max-h-[calc(95vh-180px)]" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
          {/* 1단계: 파일 선택 및 시트 선택 (1x2 레이아웃) */}
          <div className="mb-8">
            <div className="flex items-center gap-2 mb-4">
              <div className="w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold">1</div>
              <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                파일 및 시트 선택
              </h3>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* 파일 업로드 */}
              <div className="card p-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                <div className="flex items-center gap-2 mb-3">
                  <Upload className="w-5 h-5 text-blue-600" />
                  <label className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                    XLS,CSV 파일 선택
                  </label>
                </div>
                <div className="relative">
                  <input type="file" accept=".xlsx,.xls,.csv" onChange={handleFileUpload} className="block w-full text-base file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-base file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100 dark:file:bg-blue-900/20 dark:file:text-blue-300" style={{ color: 'var(--text-secondary)' }} />
                  {file && (
                    <div className="mt-2 flex items-center gap-2 text-sm" style={{ color: 'var(--text-secondary)' }}>
                      <FileSpreadsheet className="w-4 h-4 text-green-500" />
                      <span>{file.name}</span>
                    </div>
                  )}
                </div>
              </div>

              {/* 시트 선택 */}
              <div className="card p-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                <div className="flex items-center gap-2 mb-3">
                  <Columns className="w-5 h-5 text-blue-600" />
                  <label className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                    시트 선택
                  </label>
                </div>
                {sheets.length > 0 ? (
                  <select value={selectedSheet} onChange={(e) => handleSheetChange(e.target.value)} className="w-full px-3 py-2 border rounded-lg input" style={{ borderColor: 'var(--card-border)', backgroundColor: 'var(--card-bg)', color: 'var(--text-primary)' }}>
                    {sheets.map((sheet) => (
                      <option key={sheet.name} value={sheet.name}>
                        {sheet.name} ({sheet.rowCount}행 × {sheet.columnCount}열)
                      </option>
                    ))}
                  </select>
                ) : (
                  <div className="text-center text-sm py-4" style={{ color: 'var(--text-muted)' }}>
                    Excel 파일을 먼저 선택해주세요
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* 2단계: 처리 모드 선택 */}
          {sheets.length > 0 && (
            <div className="mb-8">
              <div className="flex items-center gap-2 mb-4">
                <div className="w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold">2</div>
                <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                  처리 모드 선택
                </h3>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <label
                  className="card p-4 cursor-pointer transition-all duration-200"
                  style={{
                    backgroundColor: mode === 'manual' ? 'var(--primary-100)' : 'var(--card-bg)',
                    borderColor: mode === 'manual' ? 'var(--primary-500)' : 'var(--card-border)',
                    borderWidth: mode === 'manual' ? '2px' : '1px'
                  }}
                >
                  <div className="flex items-center gap-3">
                    <input type="radio" value="manual" checked={mode === 'manual'} onChange={(e) => handleModeChange(e.target.value as 'manual' | 'ai')} className="sr-only" />
                    <div
                      className="w-6 h-6 rounded-full border-2 flex items-center justify-center"
                      style={{
                        borderColor: mode === 'manual' ? 'var(--primary-500)' : 'var(--neutral-300)',
                        backgroundColor: mode === 'manual' ? 'var(--primary-500)' : 'transparent'
                      }}
                    >
                      {mode === 'manual' && <div className="w-2 h-2 bg-white rounded-full" />}
                    </div>
                    <User
                      className="w-5 h-5"
                      style={{
                        color: mode === 'manual' ? 'var(--primary-600)' : 'var(--primary-600)'
                      }}
                    />
                    <div>
                      <div
                        className="text-base font-medium"
                        style={{
                          color: mode === 'manual' ? 'var(--primary-900)' : 'var(--text-primary)'
                        }}
                      >
                        수동 설정
                      </div>
                      <div
                        className="text-sm"
                        style={{
                          color: mode === 'manual' ? 'var(--primary-700)' : 'var(--text-secondary)'
                        }}
                      >
                        직접 열과 옵션을 선택합니다
                      </div>
                    </div>
                  </div>
                </label>
                <label
                  className="card p-4 cursor-pointer transition-all duration-200"
                  style={{
                    backgroundColor: mode === 'ai' ? 'var(--secondary-100)' : 'var(--card-bg)',
                    borderColor: mode === 'ai' ? 'var(--secondary-500)' : 'var(--card-border)',
                    borderWidth: mode === 'ai' ? '2px' : '1px'
                  }}
                >
                  <div className="flex items-center gap-3">
                    <input type="radio" value="ai" checked={mode === 'ai'} onChange={(e) => handleModeChange(e.target.value as 'manual' | 'ai')} className="sr-only" />
                    <div
                      className="w-6 h-6 rounded-full border-2 flex items-center justify-center"
                      style={{
                        borderColor: mode === 'ai' ? 'var(--secondary-500)' : 'var(--neutral-300)',
                        backgroundColor: mode === 'ai' ? 'var(--secondary-500)' : 'transparent'
                      }}
                    >
                      {mode === 'ai' && <div className="w-2 h-2 bg-white rounded-full" />}
                    </div>
                    <Sparkles
                      className="w-5 h-5"
                      style={{
                        color: mode === 'ai' ? 'var(--secondary-600)' : 'var(--secondary-600)'
                      }}
                    />
                    <div>
                      <div
                        className="text-base font-medium"
                        style={{
                          color: mode === 'ai' ? 'var(--secondary-900)' : 'var(--text-primary)'
                        }}
                      >
                        AI 자동 추천
                      </div>
                      <div
                        className="text-sm"
                        style={{
                          color: mode === 'ai' ? 'var(--secondary-700)' : 'var(--text-secondary)'
                        }}
                      >
                        AI가 데이터를 분석해서 추천합니다
                      </div>
                    </div>
                  </div>
                </label>
              </div>
            </div>
          )}

          {/* 데이터 설명 */}
          {sheets.length > 0 && (
            <div className="mb-8">
              <div className="flex items-center gap-2 mb-4">
                <div className="w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold">3</div>
                <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                  데이터 설명 (선택사항)
                </h3>
              </div>
              <div className="card p-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                <textarea value={dataDescription} onChange={(e) => setDataDescription(e.target.value)} placeholder="이 데이터가 무엇인지 설명해주세요. 예: 고객 구매 데이터, 직원 정보 목록 등..." className="w-full px-3 py-2 border rounded-lg resize-none input" style={{ borderColor: 'var(--card-border)', backgroundColor: 'var(--background)', color: 'var(--text-primary)' }} rows={3} />
                <div className="mt-2 text-sm" style={{ color: 'var(--text-muted)' }}>
                  💡 데이터에 대한 설명을 추가하면 RAG 검색 시 더 정확한 결과를 얻을 수 있습니다.
                </div>
              </div>
            </div>
          )}

          {/* AI 추천 결과 */}
          {mode === 'ai' && aiRecommendations && (
            <div className="mb-8">
              <div className="card p-4" style={{ backgroundColor: 'var(--primary-500)', color: 'white' }}>
                <div className="flex items-center gap-2 mb-3">
                  <Brain className="w-5 h-5" />
                  <h3 className="font-semibold text-white">AI 추천 사항</h3>
                </div>
                <p className="text-white/90">{aiRecommendations.description}</p>
              </div>
            </div>
          )}

          {/* 4단계: 열 선택 및 전처리 옵션 (2x1 레이아웃) */}
          {selectedSheet && (
            <div className="mb-8">
              <div className="flex items-center gap-2 mb-4">
                <div className="w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold">4</div>
                <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                  세부 설정
                </h3>
              </div>
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {/* 열 선택 */}
                <div className="card p-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="flex items-center gap-2 mb-4">
                    <Columns className="w-5 h-5 text-green-600" />
                    <h4 className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                      열 선택 ({sheets.find((s) => s.name === selectedSheet)?.headers.length || 0}개 열)
                    </h4>
                  </div>
                  <div className="max-h-48 overflow-y-auto border rounded-lg p-3" style={{ borderColor: 'var(--card-border)' }}>
                    <div className="flex items-center mb-3 pb-2 border-b" style={{ borderColor: 'var(--card-border)' }}>
                      <button
                        onClick={() => {
                          const currentSheet = sheets.find((s) => s.name === selectedSheet);
                          if (currentSheet) {
                            if (selectedColumns.length === currentSheet.headers.length) {
                              setSelectedColumns([]);
                            } else {
                              setSelectedColumns(currentSheet.headers);
                            }
                          }
                        }}
                        className="flex items-center gap-2 p-1 rounded transition-colors"
                        style={{
                          backgroundColor: 'transparent',
                          color: 'var(--text-primary)'
                        }}
                        onMouseEnter={(e) => {
                          e.currentTarget.style.backgroundColor = 'var(--card-hover)';
                        }}
                        onMouseLeave={(e) => {
                          e.currentTarget.style.backgroundColor = 'transparent';
                        }}
                      >
                        {selectedColumns.length === sheets.find((s) => s.name === selectedSheet)?.headers.length ? <CheckSquare className="w-4 h-4 text-blue-600" /> : <Square className="w-4 h-4 text-gray-400" />}
                        <span className="text-sm font-medium text-blue-600 dark:text-blue-400">전체 선택/해제</span>
                      </button>
                    </div>
                    {sheets
                      .find((s) => s.name === selectedSheet)
                      ?.headers.map((header, index) => (
                        <div
                          key={header}
                          className="flex items-center p-2 rounded transition-colors"
                          style={{
                            backgroundColor: 'transparent'
                          }}
                          onMouseEnter={(e) => {
                            e.currentTarget.style.backgroundColor = 'var(--card-hover)';
                          }}
                          onMouseLeave={(e) => {
                            e.currentTarget.style.backgroundColor = 'transparent';
                          }}
                        >
                          <button
                            onClick={() => toggleColumn(header)}
                            className="flex items-center gap-2 w-full"
                            style={{
                              backgroundColor: 'transparent',
                              color: 'var(--text-primary)'
                            }}
                          >
                            {selectedColumns.includes(header) ? <CheckSquare className="w-4 h-4 text-blue-600" /> : <Square className="w-4 h-4 text-gray-400" />}
                            <span className="text-sm font-medium min-w-0 flex-1 text-left" style={{ color: 'var(--text-primary)' }}>
                              {header || `열 ${index + 1}`}
                            </span>
                            <span className="text-xs" style={{ color: 'var(--text-muted)' }}>
                              ({String.fromCharCode(65 + index)} 열)
                            </span>
                          </button>
                        </div>
                      ))}
                  </div>
                  <div className="mt-2 text-sm" style={{ color: 'var(--text-secondary)' }}>
                    선택된 열: {selectedColumns.length}개
                  </div>
                </div>

                {/* 전처리 옵션 */}
                <div className="space-y-4">
                  {/* 행 필터 */}
                  <div className="card p-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                    <div className="flex items-center gap-2 mb-3">
                      <Filter className="w-5 h-5 text-orange-600" />
                      <h4 className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                        행 필터
                      </h4>
                    </div>
                    <label className="flex items-center mb-3">
                      <input type="checkbox" checked={rowFilter.enabled} onChange={(e) => setRowFilter({ ...rowFilter, enabled: e.target.checked })} className="mr-2" />
                      <span className="text-sm" style={{ color: 'var(--text-primary)' }}>
                        행 필터 사용
                      </span>
                    </label>

                    {rowFilter.enabled && (
                      <div className="space-y-2">
                        <select value={rowFilter.column} onChange={(e) => setRowFilter({ ...rowFilter, column: e.target.value })} className="w-full px-3 py-2 border rounded-lg input text-sm" style={{ borderColor: 'var(--card-border)', backgroundColor: 'var(--card-bg)', color: 'var(--text-primary)' }}>
                          <option value="">열 선택</option>
                          {selectedColumns.map((col) => (
                            <option key={col} value={col}>
                              {col}
                            </option>
                          ))}
                        </select>
                        <div className="flex gap-2">
                          <select value={rowFilter.operator} onChange={(e) => setRowFilter({ ...rowFilter, operator: e.target.value })} className="px-3 py-2 border rounded-lg input text-sm flex-1" style={{ borderColor: 'var(--card-border)', backgroundColor: 'var(--card-bg)', color: 'var(--text-primary)' }}>
                            <option value="contains">포함</option>
                            <option value="equals">같음</option>
                            <option value="greaterThan">크다</option>
                            <option value="lessThan">작다</option>
                          </select>
                          <input type="text" value={rowFilter.value} onChange={(e) => setRowFilter({ ...rowFilter, value: e.target.value })} placeholder="값 입력" className="px-3 py-2 border rounded-lg input text-sm flex-1" style={{ borderColor: 'var(--card-border)', backgroundColor: 'var(--card-bg)', color: 'var(--text-primary)' }} />
                        </div>
                      </div>
                    )}
                  </div>

                  {/* 데이터 분할 옵션 */}
                  <div className="card p-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                    <div className="flex items-center gap-2 mb-3">
                      <div className="w-5 h-5 flex items-center justify-center">
                        <svg className="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                      </div>
                      <h4 className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                        데이터 분할
                      </h4>
                    </div>
                    <label className="flex items-center mb-3">
                      <input type="checkbox" checked={dataSplitting.enabled} onChange={(e) => setDataSplitting({ ...dataSplitting, enabled: e.target.checked })} className="mr-2" />
                      <span className="text-sm" style={{ color: 'var(--text-primary)' }}>
                        큰 데이터를 여러 파일로 분할
                      </span>
                    </label>

                    {dataSplitting.enabled && (
                      <div className="space-y-3 pl-6 border-l-2" style={{ borderColor: 'var(--primary-200)' }}>
                        <div>
                          <label className="block text-sm font-medium mb-1" style={{ color: 'var(--text-primary)' }}>
                            파일당 최대 행 수
                          </label>
                          <div className="flex items-center gap-2">
                            <input type="number" value={dataSplitting.maxRowsPerFile} onChange={(e) => setDataSplitting({ ...dataSplitting, maxRowsPerFile: parseInt(e.target.value) || 100 })} min="10" max="1000" className="px-3 py-2 border rounded-lg input text-sm w-24" style={{ borderColor: 'var(--card-border)', backgroundColor: 'var(--card-bg)', color: 'var(--text-primary)' }} />
                            <span className="text-sm" style={{ color: 'var(--text-secondary)' }}>
                              행
                            </span>
                          </div>
                          <div className="text-xs mt-1" style={{ color: 'var(--text-muted)' }}>
                            💡 추천: 100행 이하 (AI가 더 정확하게 이해할 수 있습니다)
                          </div>
                        </div>
                        <label className="flex items-center">
                          <input type="checkbox" checked={dataSplitting.preserveHeaders} onChange={(e) => setDataSplitting({ ...dataSplitting, preserveHeaders: e.target.checked })} className="mr-2" />
                          <span className="text-sm" style={{ color: 'var(--text-primary)' }}>
                            각 파일에 헤더 포함
                          </span>
                        </label>
                      </div>
                    )}

                    {dataSplitting.enabled && (
                      <div className="mt-3 p-3 rounded-lg border" style={{ backgroundColor: 'var(--info-bg)', borderColor: 'var(--info-border)' }}>
                        <div className="text-sm" style={{ color: 'var(--info)' }}>
                          <strong>분할 예상 결과:</strong>
                          <div className="mt-1 text-xs" style={{ color: 'var(--text-secondary)' }}>
                            • 총 {sheets.find((s) => s.name === selectedSheet)?.rowCount || 0}행 → 약 {Math.ceil((sheets.find((s) => s.name === selectedSheet)?.rowCount || 0) / dataSplitting.maxRowsPerFile)}개 파일로 분할
                            <br />• 각 파일명: {file?.name?.replace(/\.[^/.]+$/, '')}_part1.md, {file?.name?.replace(/\.[^/.]+$/, '')}_part2.md ...
                          </div>
                        </div>
                      </div>
                    )}
                  </div>

                  {/* 기타 옵션 */}
                  <div className="card p-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                    <div className="flex items-center gap-2 mb-3">
                      <Settings2 className="w-5 h-5 text-purple-600" />
                      <h4 className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                        기타 옵션
                      </h4>
                    </div>
                    <div className="space-y-3">
                      <label className="flex items-center">
                        <input type="checkbox" checked={headerNormalization} onChange={(e) => setHeaderNormalization(e.target.checked)} className="mr-2" />
                        <span className="text-sm" style={{ color: 'var(--text-primary)' }}>
                          헤더 정규화 (공백/개행 제거)
                        </span>
                      </label>
                      <label className="flex items-center">
                        <input type="checkbox" checked={preserveFormatting} onChange={(e) => setPreserveFormatting(e.target.checked)} className="mr-2" />
                        <span className="text-sm" style={{ color: 'var(--text-primary)' }}>
                          숫자/날짜 포맷 유지
                        </span>
                      </label>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}

          {/* 5단계: 미리보기 */}
          {previewData.length > 0 && (
            <div className="mb-6">
              <div className="flex items-center gap-2 mb-4">
                <div className="w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold">5</div>
                <div className="flex items-center gap-2">
                  <Eye className="w-5 h-5 text-blue-600" />
                  <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                    데이터 미리보기
                  </h3>
                </div>
                <div className="text-sm ml-auto" style={{ color: 'var(--text-muted)' }}>
                  전체 {previewData.length - 1}행 중 상위 5행
                </div>
              </div>
              <div className="card overflow-hidden" style={{ backgroundColor: 'var(--card-bg)' }}>
                <div className="overflow-x-auto">
                  <table className="min-w-full divide-y" style={{ borderColor: 'var(--card-border)' }}>
                    <thead style={{ backgroundColor: 'var(--card-bg)' }}>
                      <tr>
                        {sheets
                          .find((s) => s.name === selectedSheet)
                          ?.headers.map((col, index) => (
                            <th
                              key={col}
                              className="px-3 py-3 text-left text-sm font-semibold tracking-wider border-r last:border-r-0"
                              style={{
                                borderColor: 'var(--card-border)',
                                backgroundColor: selectedColumns.includes(col) ? (document.documentElement.classList.contains('dark') ? '#1e40af' : '#dbeafe') : 'var(--card-bg)',
                                color: selectedColumns.includes(col) ? (document.documentElement.classList.contains('dark') ? '#ffffff' : '#1e3a8a') : 'var(--text-secondary)'
                              }}
                            >
                              <div className="flex items-center space-x-2">
                                <span>{col || `열 ${index + 1}`}</span>
                                <span className="text-xs opacity-60">({String.fromCharCode(65 + index)})</span>
                                {selectedColumns.includes(col) && (
                                  <CheckSquare
                                    className="w-4 h-4"
                                    style={{
                                      color: document.documentElement.classList.contains('dark') ? '#ffffff' : '#2563eb'
                                    }}
                                  />
                                )}
                              </div>
                            </th>
                          ))}
                      </tr>
                    </thead>
                    <tbody className="divide-y">
                      {previewData.slice(1, 6).map((row, idx) => (
                        <tr key={idx}>
                          {sheets
                            .find((s) => s.name === selectedSheet)
                            ?.headers.map((col, colIndex) => (
                              <td
                                key={colIndex}
                                className="px-3 py-2 text-sm border-r last:border-r-0"
                                style={{
                                  borderColor: 'var(--card-border)',
                                  backgroundColor: selectedColumns.includes(col) ? (document.documentElement.classList.contains('dark') ? '#1e40af' : '#dbeafe') : 'var(--card-bg)',
                                  color: selectedColumns.includes(col) ? (document.documentElement.classList.contains('dark') ? '#ffffff' : '#1e3a8a') : 'var(--text-primary)',
                                  fontWeight: selectedColumns.includes(col) ? '500' : 'normal'
                                }}
                              >
                                <div className="max-w-xs truncate" title={String(row[colIndex] || '')}>
                                  {row[colIndex] || '-'}
                                </div>
                              </td>
                            ))}
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
                <div
                  className="p-3 border-t text-sm flex items-center justify-between"
                  style={{
                    borderColor: 'var(--card-border)',
                    color: document.documentElement.classList.contains('dark') ? '#ffffff' : '#1e3a8a',
                    backgroundColor: document.documentElement.classList.contains('dark') ? '#1e40af' : '#dbeafe'
                  }}
                >
                  <span>🔵 파란색으로 표시된 열이 선택된 열입니다.</span>
                  <span>선택된 열: {selectedColumns.length}개</span>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* 푸터 */}
        <div
          className="flex justify-end gap-3 p-6 border-t"
          style={{
            backgroundColor: 'var(--sidebar-bg)',
            borderColor: 'var(--card-border)'
          }}
        >
          <button
            onClick={onClose}
            className="btn btn-secondary px-6 py-2"
            style={{
              backgroundColor: 'var(--btn-outline-bg)',
              color: 'var(--btn-outline-fg)',
              border: '1px solid var(--btn-outline-border)'
            }}
          >
            취소
          </button>
          <button
            onClick={handleProcess}
            disabled={!file || selectedColumns.length === 0 || isProcessing}
            className="btn btn-primary px-6 py-2 disabled:opacity-50 disabled:cursor-not-allowed"
            style={{
              backgroundColor: 'var(--btn-primary-bg)',
              color: 'var(--btn-primary-fg)'
            }}
          >
            {isProcessing ? '처리 중...' : '전처리 시작'}
          </button>
        </div>
      </div>
    </div>
  );
}
