/**
 * AI.RUN 시스템 URL 관리
 * 개발환경(localhost:port)과 운영환경(nginx proxy + HTTPS) 모두 지원
 */

// 환경변수 로드
const isProduction = process.env.PRODUCTION === 'true';
const useNginxProxy = process.env.NGINX_PROXY === 'true';
const baseUrl = process.env.BASE_URL || 'http://localhost';

// 포트 설정
const PORTS = {
    API_SERVER: parseInt(process.env.API_SERVER_PORT) || 5500,
    RAG_SERVER: parseInt(process.env.RAG_SERVER_PORT) || 5600,
    WEB_SEARCH_SERVER: parseInt(process.env.WEB_SEARCH_PORT) || 5610,
    REPORT_SERVER: parseInt(process.env.REPORT_SERVER_PORT) || 5620,
    WEB_PORTAL: parseInt(process.env.WEB_PORTAL_PORT) || 5700,
    MONITORING_SERVER: parseInt(process.env.MONITORING_SERVER_PORT) || 5750,
    WEBHOOK_SERVER: parseInt(process.env.WEBHOOK_SERVER_PORT) || 5800
};

/**
 * URL 생성 함수
 * @param {string} service - 서비스 이름 (API_SERVER, RAG_SERVER 등)
 * @param {string} path - API 경로 (선택사항)
 * @returns {string} 완전한 URL
 */
function getServiceUrl(service, path = '') {
    if (isProduction && useNginxProxy) {
        // 운영환경: nginx proxy를 통한 HTTPS 접근
        const servicePaths = {
            API_SERVER: '/api',
            RAG_SERVER: '/rag',
            WEB_SEARCH_SERVER: '/websearch',
            REPORT_SERVER: '/report',
            WEB_PORTAL: '/portal',
            MONITORING_SERVER: '/monitor',
            WEBHOOK_SERVER: '/webhook'
        };
        
        const servicePath = servicePaths[service] || '';
        return `${baseUrl}${servicePath}${path}`;
    } else {
        // 개발환경: 직접 포트 접근
        const port = PORTS[service];
        if (!port) {
            throw new Error(`Unknown service: ${service}`);
        }
        
        return `http://localhost:${port}${path}`;
    }
}

/**
 * 내부 서비스 간 통신용 URL (항상 localhost 사용)
 * @param {string} service - 서비스 이름
 * @param {string} path - API 경로
 * @returns {string} 내부 통신용 URL
 */
function getInternalUrl(service, path = '') {
    const port = PORTS[service];
    if (!port) {
        throw new Error(`Unknown service: ${service}`);
    }
    
    return `http://127.0.0.1:${port}${path}`;
}

// 편의 함수들
export const URLs = {
    // 외부 접근용 (환경에 따라 localhost 또는 nginx proxy)
    API_SERVER: (path = '') => getServiceUrl('API_SERVER', path),
    RAG_SERVER: (path = '') => getServiceUrl('RAG_SERVER', path),
    WEB_SEARCH_SERVER: (path = '') => getServiceUrl('WEB_SEARCH_SERVER', path),
    REPORT_SERVER: (path = '') => getServiceUrl('REPORT_SERVER', path),
    WEB_PORTAL: (path = '') => getServiceUrl('WEB_PORTAL', path),
    
    // 내부 서비스 간 통신용 (항상 localhost)
    INTERNAL: {
        API_SERVER: (path = '') => getInternalUrl('API_SERVER', path),
        RAG_SERVER: (path = '') => getInternalUrl('RAG_SERVER', path),
        WEB_SEARCH_SERVER: (path = '') => getInternalUrl('WEB_SEARCH_SERVER', path),
        REPORT_SERVER: (path = '') => getInternalUrl('REPORT_SERVER', path)
    }
};

// CommonJS 지원
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { URLs, PORTS, getServiceUrl, getInternalUrl };
}

// 환경 정보 출력
export function printEnvironmentInfo() {
    console.log('🌐 AI.RUN 환경 설정:');
    console.log(`   환경: ${isProduction ? '운영' : '개발'}`);
    console.log(`   Proxy: ${useNginxProxy ? 'nginx' : '직접접근'}`);
    console.log(`   Base URL: ${baseUrl}`);
    console.log('');
    console.log('🔧 서비스 URL:');
    console.log(`   📡 API 서버: ${URLs.API_SERVER()}`);
    console.log(`   🤖 RAG 서버: ${URLs.RAG_SERVER()}`);
    console.log(`   🌐 웹검색 서버: ${URLs.WEB_SEARCH_SERVER()}`);
    console.log(`   📄 문서 서버: ${URLs.REPORT_SERVER()}`);
    console.log(`   🌐 웹포털: ${URLs.WEB_PORTAL()}`);
}

export { PORTS }; 