import { Langfuse } from 'langfuse';
import { sharedContext } from '../aiFeatures.js';
import dotenv from 'dotenv';
import path from 'path';
import { fileURLToPath } from 'url';
import fs from 'fs';
import { saveUsageData } from '../pages/usageTracker.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// 환경 변수 로드
dotenv.config();

export class UserExperienceManager {
    constructor() {
        this.langfuse = null;
        this.isInitialized = false;
        this.qualityManager = null;
        this.currentSession = null;
        this.recentTraces = []; // 로컬 캐시
        this.connectionStatus = 'unknown'; // 'connected', 'disconnected', 'unknown'
        this.lastConnectionCheck = null;
        this.connectionCheckInterval = 30000; // 30초마다 연결 확인
        this.fallbackMode = false; // Langfuse 없이 동작하는 모드
        this.localFeedbackCache = []; // 오프라인 피드백 캐시
        this.dbManager = null; // 데이터베이스 매니저
        
        // 연결 상태 주기적 확인
        this.startConnectionMonitoring();
        
        // 초기화 시작
        this.initializeLangfuse();
    }

    // 데이터베이스 매니저 설정
    setDatabaseManager(dbManager) {
        this.dbManager = dbManager;
        
        // qualityManager에도 dbManager 설정
        if (this.qualityManager && this.qualityManager.setDatabaseManager) {
            this.qualityManager.setDatabaseManager(dbManager);
        }
        
        console.log('🗄️ UserExperienceManager에 데이터베이스 매니저가 설정되었습니다.');
    }

    // 연결 상태 모니터링 시작
    startConnectionMonitoring() {
        setInterval(async () => {
            await this.checkLangfuseConnection();
        }, this.connectionCheckInterval);
    }

    // Langfuse 연결 상태 확인
    async checkLangfuseConnection() {
        try {
            if (!this.langfuse) {
                this.connectionStatus = 'disconnected';
                return false;
            }

            // 간단한 연결 테스트 (trace 생성 시도)
            const testTrace = this.langfuse.trace({
                name: 'connection_test',
                input: { test: true },
                metadata: { timestamp: new Date().toISOString() }
            });

            // flush 시도로 실제 연결 확인
            await this.langfuse.flush();
            
            this.connectionStatus = 'connected';
            this.fallbackMode = false;
            this.lastConnectionCheck = new Date();
            
            // 캐시된 피드백이 있으면 전송 시도
            if (this.localFeedbackCache.length > 0) {
                await this.flushCachedFeedback();
            }
            
            return true;
        } catch (error) {
            console.warn('Langfuse 연결 확인 실패:', error.message);
            this.connectionStatus = 'disconnected';
            this.fallbackMode = true;
            return false;
        }
    }

    // 캐시된 피드백 전송
    async flushCachedFeedback() {
        if (this.localFeedbackCache.length === 0) return;

        console.log(`캐시된 피드백 ${this.localFeedbackCache.length}개 전송 시도...`);
        
        const successfullyFlushed = [];
        
        for (const feedbackData of this.localFeedbackCache) {
            try {
                await this.submitFeedbackToLangfuse(feedbackData);
                successfullyFlushed.push(feedbackData);
            } catch (error) {
                console.warn('캐시된 피드백 전송 실패:', error.message);
                break; // 하나라도 실패하면 중단
            }
        }

        // 성공적으로 전송된 피드백 제거
        this.localFeedbackCache = this.localFeedbackCache.filter(
            item => !successfullyFlushed.includes(item)
        );

        if (successfullyFlushed.length > 0) {
            console.log(`캐시된 피드백 ${successfullyFlushed.length}개 전송 완료`);
        }
    }

    // Langfuse에 실제 피드백 전송 (내부 메서드)
    async submitFeedbackToLangfuse(feedbackData) {
        if (!this.isInitialized || !this.langfuse) {
            throw new Error('Langfuse not initialized');
        }

        const sessionId = feedbackData.sessionId;
        const messageId = feedbackData.messageId;
        const timestamp = feedbackData.timestamp;

        // 1. 먼저 대화 Trace를 생성하여 Score가 연결될 수 있도록 함
        const conversationTraceId = `conversation-${sessionId}-${Date.now()}`;
        
        try {
            // 2. 대화 Trace 생성 (Score 연결용)
            const conversationTrace = this.langfuse.trace({
                id: conversationTraceId,
                name: 'user_conversation',
                sessionId: sessionId,
                userId: feedbackData.userId || 'user',
                input: {
                    userMessage: feedbackData.userMessage,
                    timestamp: timestamp
                },
                output: {
                    aiResponse: feedbackData.aiResponse,
                    timestamp: timestamp
                },
                metadata: {
                    source: 'airun',
                    environment: process.env.NODE_ENV || 'development',
                    messageId: messageId,
                    conversationType: 'chat_with_feedback',
                    sessionId: sessionId
                }
            });

            // 3. Generation 추가 (Observation 생성)
            const generation = conversationTrace.generation({
                name: 'ai_response',
                input: feedbackData.userMessage,
                output: feedbackData.aiResponse,
                model: 'gpt-4', // 실제 모델명으로 교체 가능
                metadata: {
                    provider: 'openai',
                    messageId: messageId,
                    timestamp: timestamp
                }
            });

            // 4. Score 생성 (Conversation Trace와 연결)
            const score = this.langfuse.score({
                traceId: conversationTraceId,
                name: 'user_feedback',
                value: feedbackData.rating,
                comment: feedbackData.rating === 5 ? 'positive_feedback' : 'negative_feedback',
                metadata: {
                    source: 'user_interaction',
                    feedbackType: feedbackData.rating === 5 ? 'like' : 'dislike',
                    messageId: messageId,
                    sessionId: sessionId,
                    timestamp: timestamp,
                    userMessage: feedbackData.userMessage?.substring(0, 100),
                    aiResponse: feedbackData.aiResponse?.substring(0, 100)
                }
            });

            await this.langfuse.flush();
            
            // console.log(`✅ Langfuse 완전한 데이터 생성 완료:
            // - Conversation Trace: ${conversationTraceId}
            // - Generation (Observation): ${generation.id}
            // - Score: ${score.id} (Trace와 연결됨)
            // - 피드백: ${feedbackData.rating === 5 ? '👍 좋아요' : '👎 싫어요'}`);
            
            return { 
                trace: conversationTrace, 
                generation: generation,
                score: score, 
                traceId: conversationTraceId 
            };
            
        } catch (error) {
            console.error('Langfuse 데이터 생성 실패:', error.message);
            throw error;
        }
    }

    async initializeLangfuse() {
        try {
            // 환경변수에서 Langfuse 설정 읽기
            const secretKey = process.env.LANGFUSE_SECRET_KEY;
            const publicKey = process.env.LANGFUSE_PUBLIC_KEY;
            const baseUrl = process.env.LANGFUSE_BASE_URL || 'http://localhost:3002';

            if (!secretKey || !publicKey) {
                // console.warn('Langfuse 환경변수가 설정되지 않았습니다. 로컬 모드로 동작합니다.');
                this.fallbackMode = true;
                this.connectionStatus = 'disconnected';
                this.initializeFallbackMode();
                return;
            }

            const { Langfuse } = await import('langfuse');
            
            this.langfuse = new Langfuse({
                secretKey,
                publicKey,
                baseUrl,
                flushAt: 1,
                flushInterval: 1000
            });

            // 연결 테스트
            const isConnected = await this.checkLangfuseConnection();
            
            if (isConnected) {
                this.isInitialized = true;
                this.qualityManager = new AIResponseQualityManager(this.langfuse);
                // console.log('Langfuse 초기화 완료');
            } else {
                throw new Error('Langfuse 연결 실패');
            }

        } catch (error) {
            // console.warn('Langfuse 초기화 실패, 로컬 모드로 전환:', error.message);
            this.fallbackMode = true;
            this.connectionStatus = 'disconnected';
            this.initializeFallbackMode();
        }
    }

    // Fallback 모드 초기화
    initializeFallbackMode() {
        // console.log('🔄 Langfuse Fallback 모드 활성화');
        this.isInitialized = true; // 로컬 모드에서도 기능 제공
        this.qualityManager = new AIResponseQualityManager(null); // null langfuse로 초기화
    }

    async submitFeedback(feedbackData) {
        try {
            // 로컬 캐시에 항상 저장 (백업용)
            this.addToRecentTraces({
                input: {
                    userMessage: feedbackData.userMessage
                },
                output: {
                    aiResponse: feedbackData.aiResponse
                },
                feedback: feedbackData.rating,
                timestamp: feedbackData.timestamp,
                sessionId: feedbackData.sessionId
            });

            // Langfuse 연결 상태 확인
            if (this.connectionStatus === 'connected' && this.langfuse) {
                try {
                    await this.submitFeedbackToLangfuse(feedbackData);
                    // console.log('✅ Langfuse 피드백 전송 성공');
                } catch (error) {
                    console.warn('Langfuse 피드백 전송 실패, 캐시에 저장:', error.message);
                    this.localFeedbackCache.push(feedbackData);
                    this.connectionStatus = 'disconnected';
                }
            } else {
                // Langfuse 연결 안됨 - 캐시에 저장
                console.log('📦 Langfuse 연결 안됨, 로컬 캐시에 피드백 저장');
                this.localFeedbackCache.push(feedbackData);
            }

            // 품질 관리자에게 피드백 데이터 전달 (항상 실행)
            if (this.qualityManager) {
                await this.qualityManager.processFeedback(feedbackData);
            }

            return true;
        } catch (error) {
            console.warn('피드백 제출 중 오류 발생:', error.message);
            // 오류가 발생해도 로컬 캐시에는 저장
            this.localFeedbackCache.push(feedbackData);
            return true; // 사용자에게는 성공으로 표시
        }
    }

    async logMessage(message, type) {
        try {
            if (!this.isInitialized || !this.langfuse) {
                console.warn('Langfuse is not initialized. Skipping message logging.');
                return true;
            }

            const trace = this.langfuse.trace({
                name: 'conversation_message',
                input: {
                    messageType: type,
                    content: message
                },
                output: {
                    processed: true,
                    timestamp: new Date().toISOString()
                },
                metadata: {
                    source: 'airun',
                    environment: process.env.NODE_ENV || 'development',
                    messageType: type,
                    content: message,
                    timestamp: new Date().toISOString()
                }
            });

            const span = trace.span({
                name: 'message_processing',
                metadata: {
                    processingTime: Date.now(),
                    status: 'success'
                }
            });

            console.log('메시지 로깅됨:', { type, message });
            return true;
        } catch (error) {
            console.warn('메시지 로깅 중 오류 발생:', error.message);
            return true;
        }
    }

    // 세션 시작 메서드 추가 (Sessions 메뉴에 표시되도록 개선)
    async startSession(userId) {
        try {
            if (!this.isInitialized || !this.langfuse) {
                console.warn('Langfuse is not initialized. Skipping session start.');
                return null;
            }

            const sessionId = `session-${userId}-${Date.now()}`;
            const timestamp = new Date().toISOString();
            
            // 1. Session 정보 Trace 생성 (Sessions 메뉴 표시용)
            const sessionInfoTrace = this.langfuse.trace({
                id: `session-info-${sessionId}`,
                name: 'session_start',
                sessionId: sessionId, // 중요: Session과 연결
                userId: userId,
                input: {
                    sessionStart: true,
                    userId: userId,
                    sessionType: 'user_feedback',
                    startTime: timestamp
                },
                output: {
                    sessionId: sessionId,
                    status: 'started',
                    sessionCreated: true
                },
                metadata: {
                    userId: userId,
                    startTime: timestamp,
                    source: 'airun',
                    environment: process.env.NODE_ENV || 'development',
                    sessionType: 'user_feedback',
                    prompt: {
                        name: 'code-assistant-prompt',
                        version: 1,
                        template: 'You are a helpful coding assistant...'
                    }
                }
            });

            // 2. 메인 세션 Trace 생성
            this.currentSession = this.langfuse.trace({
                id: sessionId,
                name: 'user_session',
                sessionId: sessionId, // 중요: Session과 연결
                userId: userId,
                input: {
                    userId: userId,
                    sessionType: 'user_feedback',
                    startTime: timestamp
                },
                output: {
                    sessionId: sessionId,
                    status: 'started'
                },
                metadata: {
                    userId: userId,
                    startTime: timestamp,
                    source: 'airun',
                    environment: process.env.NODE_ENV || 'development',
                    sessionType: 'user_feedback'
                }
            });

            await this.langfuse.flush();
            console.log(`🎯 Langfuse Session 시작: ${sessionId} (Sessions 메뉴에 표시됨)`);
            return sessionId;
        } catch (error) {
            console.warn('Failed to start session:', error.message);
            return null;
        }
    }

    // 세션 종료 메서드 추가
    async endSession() {
        try {
            if (this.currentSession) {
                await this.currentSession.update({
                    metadata: {
                        endTime: new Date().toISOString(),
                        status: 'completed'
                    }
                });
                this.currentSession = null;
                console.log('Session ended');
            }
        } catch (error) {
            console.warn('Failed to end session:', error.message);
        }
    }

    // Langfuse에서 관련 대화 검색
    async searchConversations(searchQuery) {
        try {
            if (!this.isInitialized || !this.langfuse) {
                console.warn('Langfuse가 초기화되지 않았습니다.');
                return [];
            }

            // 현재는 Langfuse SDK에서 직접 검색 API를 제공하지 않으므로
            // 메모리에 저장된 최근 대화들을 검색합니다
            const recentTraces = this.recentTraces || [];
            
            if (recentTraces.length === 0) {
                console.log('검색할 최근 대화가 없습니다.');
                return [];
            }

            // 검색 쿼리와 관련성이 높은 대화 필터링
            const relevantTraces = recentTraces.filter(trace => {
                if (!trace.input?.userMessage) return false;
                
                // 간단한 키워드 매칭
                const userMessage = trace.input.userMessage.toLowerCase();
                const query = searchQuery.query.toLowerCase();
                
                // 공통 키워드 찾기
                const queryWords = query.split(' ').filter(word => word.length > 2);
                const messageWords = userMessage.split(' ');
                
                const commonWords = queryWords.filter(word => 
                    messageWords.some(msgWord => msgWord.includes(word) || word.includes(msgWord))
                );
                
                // 30% 이상 키워드가 일치하면 관련성 있다고 판단
                return commonWords.length / queryWords.length >= 0.3;
            });

            // 최신순으로 정렬하고 제한
            const sortedTraces = relevantTraces
                .sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp))
                .slice(0, searchQuery.limit || 3);

            console.log(`${sortedTraces.length}개의 관련 대화를 찾았습니다.`);
            return sortedTraces;

        } catch (error) {
            console.error('대화 검색 실패:', error);
            return [];
        }
    }

    // 최근 대화 저장을 위한 메서드 추가
    addToRecentTraces(trace) {
        if (!this.recentTraces) {
            this.recentTraces = [];
        }
        
        this.recentTraces.unshift(trace);
        
        // 최대 50개까지만 유지
        if (this.recentTraces.length > 50) {
            this.recentTraces = this.recentTraces.slice(0, 50);
        }
    }
}

export class ConversationMonitor {
    constructor(langfuse, traceId) {
        // console.log('[ConversationMonitor] 초기화 시작:', { traceId, hasLangfuse: !!langfuse });
        this.langfuse = langfuse;
        this.traceId = traceId;
        this.isInitialized = !!langfuse;
        this.lastUserMessage = null;
        this.traceQueue = [];
        this.flushInterval = 5000; // 5초마다 flush
        this.startBatchProcessing();
        // console.log('[ConversationMonitor] 초기화 완료:', { isInitialized: this.isInitialized });
    }

    startBatchProcessing() {
        if (this.batchProcessingInterval) {
            clearInterval(this.batchProcessingInterval);
        }

        this.batchProcessingInterval = setInterval(async () => {
            if (this.traceQueue.length === 0) return;

            // console.log('[ConversationMonitor] 배치 처리 시작:', {
            //     queueLength: this.traceQueue.length,
            //     timestamp: new Date().toISOString()
            // });

            try {
                // 현재 큐의 모든 trace를 처리
                const traces = [...this.traceQueue];
                this.traceQueue = []; // 큐 비우기

                // langfuse 인스턴스의 flush 메서드 호출
                if (this.langfuse && typeof this.langfuse.flush === 'function') {
                    await this.langfuse.flush();
                    // console.log('[ConversationMonitor] 배치 처리 완료:', {
                    //     processedCount: traces.length,
                    //     timestamp: new Date().toISOString()
                    // });
                } else {
                    console.warn('[ConversationMonitor] Langfuse flush 메서드를 사용할 수 없음');
                }
            } catch (error) {
                console.error('[ConversationMonitor] 배치 처리 중 오류:', {
                    error: error.message,
                    stack: error.stack,
                    timestamp: new Date().toISOString()
                });
                // 실패한 trace들을 다시 큐에 추가
                this.traceQueue = [...this.traceQueue, ...traces];
            }
        }, 5000); // 5초마다 실행
    }

    async trackMessage(message, type, usageData = {}) {
        const startTime = Date.now();
        // console.log('[ConversationMonitor] 메시지 추적 시작:', { 
        //     type, 
        //     messageLength: message?.length,
        //     hasLangfuse: !!this.langfuse,
        //     isInitialized: this.isInitialized 
        // });

        try {
            if (!this.isInitialized || !this.langfuse) {
                // console.warn('[ConversationMonitor] Langfuse가 초기화되지 않음');
                return message;
            }

            const session = await sharedContext.getSession(this.traceId);
            // console.log('[ConversationMonitor] 세션 정보:', {
            //     hasSession: !!session,
            //     sessionId: session?.getId(),
            //     provider: session?.getProvider()
            // });

            if (!session) {
                console.warn('세션을 찾을 수 없습니다:', this.traceId);
                return message;
            }

            // trace 객체를 먼저 생성
            const trace = this.langfuse.trace({
                id: this.traceId,
                name: 'chat_message',
                input: type === 'user' ? {
                    messageType: type,
                    content: message,
                    timestamp: new Date().toISOString()
                } : {
                    messageType: 'user',
                    content: this.lastUserMessage,
                    timestamp: new Date().toISOString()
                },
                output: type === 'assistant' ? {
                    messageType: type,
                    content: message,
                    timestamp: new Date().toISOString()
                } : null,
                metadata: {
                    source: 'airun',
                    environment: process.env.NODE_ENV || 'development',
                    messageType: type,
                    timestamp: new Date().toISOString(),
                    sessionId: session.getId(),
                    provider: session.getProvider() || 'unknown',
                    model: session.getModel() || 'unknown',
                    parameters: session.getParameters() || {},
                    requestType: session.getRequestType() || 'unknown',
                    usage: {
                        promptTokens: usageData.promptTokens || 0,
                        completionTokens: usageData.completionTokens || 0,
                        totalTokens: usageData.totalTokens || 0,
                        duration: usageData.duration || 0
                    }
                }
            });

            // 큐에 추가
            this.traceQueue.push(trace);

            // 비동기적으로 flush 실행
            (async () => {
                try {
                    await this.langfuse.flush();
                    const duration = Date.now() - startTime;
                    // console.log('[ConversationMonitor] 추적 완료:', {
                    //     duration: `${duration}ms`,
                    //     queueLength: this.traceQueue?.length
                    // });
                } catch (flushError) {
                    console.error('[ConversationMonitor] Flush 중 오류:', {
                        error: flushError.message,
                        stack: flushError.stack
                    });
                }
            })();

            return message;
        } catch (error) {
            console.error('[ConversationMonitor] 추적 중 오류:', {
                error: error.message,
                stack: error.stack,
                duration: `${Date.now() - startTime}ms`
            });
            return message; // 오류가 발생해도 메인 로직은 계속 진행
        }
    }

    async trackConversation(userMessage, aiResponse, usageData = {}) {
        try {
            // 세션 정보 가져오기
            const session = await sharedContext.getSession(this.traceId);
            if (!session) {
                console.warn('세션을 찾을 수 없습니다:', this.traceId);
                return;
            }

            if (this.isInitialized && this.langfuse) {
                const sessionId = session.getId();
                const timestamp = new Date().toISOString();
                
                // 1. Session 생성 (Langfuse Sessions 메뉴에 표시되도록)
                await this.createLangfuseSession(sessionId, session, timestamp);
                
                // 2. 메인 Conversation Trace 생성 (Session과 연결, 고유 ID 사용)
                const conversationTraceId = `conversation-${sessionId}-${Date.now()}`;
                const mainTrace = this.langfuse.trace({
                    id: conversationTraceId,
                    name: 'conversation',
                    sessionId: sessionId, // 중요: Session과 연결
                    userId: 'user',
                    input: {
                        userMessage: userMessage,
                        timestamp: timestamp,
                        sessionType: 'chat'
                    },
                    output: {
                        aiResponse: aiResponse,
                        timestamp: timestamp
                    },
                    metadata: {
                        source: 'airun',
                        environment: process.env.NODE_ENV || 'development',
                        provider: session.getProvider() || 'unknown',
                        model: session.getModel() || 'unknown',
                        requestType: session.getRequestType() || 'unknown',
                        parameters: session.getParameters() || {},
                        usage: {
                            promptTokens: usageData.promptTokens || 0,
                            completionTokens: usageData.completionTokens || 0,
                            totalTokens: usageData.totalTokens || 0,
                            duration: usageData.duration || 0
                        }
                    }
                });

                // 3. Generation 생성 (메인 Trace 하위)
                const generation = mainTrace.generation({
                    name: 'ai_response',
                    input: userMessage,
                    output: aiResponse,
                    model: session.getModel() || 'unknown',
                    metadata: {
                        provider: session.getProvider() || 'unknown',
                        usage: {
                            promptTokens: usageData.promptTokens || 0,
                            completionTokens: usageData.completionTokens || 0,
                            totalTokens: usageData.totalTokens || 0
                        }
                    }
                });

                // 4. 큐에 추가하여 배치 처리
                this.traceQueue.push({ mainTrace, generation });
                
                // console.log(`📊 Langfuse 대화 추적 완료:
                // - Session: ${sessionId} (Sessions 메뉴에 표시됨)
                // - Conversation Trace: ${conversationTraceId}
                // - Generation: ${generation.id}
                // - 피드백 연결 준비됨 ✅`);
            }

            // 사용자 메시지 저장
            this.lastUserMessage = userMessage;
        } catch (error) {
            console.warn('대화 추적 중 오류 발생:', error.message);
        }
    }

    // Langfuse Session 생성 메서드
    async createLangfuseSession(sessionId, session, timestamp) {
        try {
            // Session 생성을 위한 별도 API 호출이 필요하지만
            // Langfuse JS SDK에서는 sessionId를 포함한 trace 생성으로 자동 생성됨
            // 하지만 명시적으로 Session 정보를 기록하기 위해 별도 처리
            
            if (!this.createdSessions) {
                this.createdSessions = new Set();
            }
            
            // 이미 생성된 세션인지 확인
            if (this.createdSessions.has(sessionId)) {
                return;
            }
            
            // Session 정보를 별도 Trace로 생성 (Sessions 메뉴 표시용)
            const sessionTrace = this.langfuse.trace({
                id: `session-info-${sessionId}`,
                name: 'session_info',
                sessionId: sessionId,
                userId: 'user',
                input: {
                    sessionStart: true,
                    sessionId: sessionId,
                    timestamp: timestamp
                },
                output: {
                    sessionCreated: true,
                    provider: session.getProvider() || 'unknown',
                    model: session.getModel() || 'unknown'
                },
                metadata: {
                    source: 'airun',
                    environment: process.env.NODE_ENV || 'development',
                    sessionType: 'chat_session',
                    provider: session.getProvider() || 'unknown',
                    model: session.getModel() || 'unknown',
                    parameters: session.getParameters() || {},
                    sessionCreated: true
                }
            });
            
            this.createdSessions.add(sessionId);
            // console.log(`🎯 Langfuse Session 생성: ${sessionId}`);
            
        } catch (error) {
            console.warn('Session 생성 중 오류:', error.message);
        }
    }
}

// 새로운 품질 관리 전용 클래스
export class AIResponseQualityManager {
    constructor(langfuse) {
        this.langfuse = langfuse;
        this.isLangfuseAvailable = !!langfuse;
        this.dbManager = null; // 데이터베이스 매니저
        this.feedbackPatterns = new Map(); // 피드백 패턴 저장 (메모리 캐시)
        this.responseTemplates = new Map(); // 성공적인 응답 템플릿 (메모리 캐시)
        this.avoidPatterns = new Set(); // 피해야 할 패턴들 (메모리 캐시)
        this.qualityThresholds = {
            high: 0.8,
            medium: 0.6,
            low: 0.4
        };
        this.localAnalytics = {
            totalFeedback: 0,
            positiveFeedback: 0,
            negativeFeedback: 0,
            patterns: []
        };
        
        // console.log(`🎯 AIResponseQualityManager 초기화 (Langfuse: ${this.isLangfuseAvailable ? '연결됨' : '로컬 모드'})`);
    }

    // 데이터베이스 매니저 설정
    setDatabaseManager(dbManager) {
        this.dbManager = dbManager;
        console.log('🗄️ 데이터베이스 매니저가 설정되었습니다.');
    }

    async processFeedback(feedbackData, existingTrace = null) {
        try {
            console.log('🔍 processFeedback 호출됨:', {
                rating: feedbackData.rating,
                sessionId: feedbackData.sessionId
            });

            // 피드백 패턴 학습
            await this.learnFromFeedback(feedbackData);
            
            // 간단한 메트릭 반환
            const metrics = {
                score: feedbackData.rating,
                quality: feedbackData.rating >= 4 ? 'high' : feedbackData.rating >= 3 ? 'medium' : 'low',
                timestamp: new Date().toISOString()
            };

            console.log('✅ 피드백 처리 완료:', metrics);
            return metrics;
        } catch (error) {
            console.error('피드백 처리 중 오류:', error);
            return null;
        }
    }

    // 피드백으로부터 학습하는 메서드
    async learnFromFeedback(feedbackData) {
        try {
            const { userMessage, aiResponse, rating, sessionId } = feedbackData;
            
            console.log('🔍 learnFromFeedback 호출됨:', {
                rating,
                sessionId,
                userMessageLength: userMessage?.length,
                aiResponseLength: aiResponse?.length
            });

            // 간단한 패턴 생성
            const pattern = {
                rating,
                timestamp: new Date().toISOString(),
                sessionId,
                responseLength: aiResponse?.length || 0
            };

            // 메모리 캐시에 저장 (기존 기능 유지)
            const patternKey = `session-${sessionId}-${rating}`;
            if (!this.feedbackPatterns.has(patternKey)) {
                this.feedbackPatterns.set(patternKey, []);
            }
            this.feedbackPatterns.get(patternKey).push(pattern);

            // 데이터베이스에 피드백 저장 (새로운 기능)
            console.log('🔍 dbManager 상태 확인:', {
                hasDbManager: !!this.dbManager,
                dbManagerType: this.dbManager ? typeof this.dbManager : 'undefined',
                hasQueryMethod: this.dbManager ? typeof this.dbManager.query : 'undefined'
            });
            
            if (this.dbManager) {
                try {
                    await this.saveFeedbackToDatabase(feedbackData);
                    console.log('💾 피드백이 데이터베이스에 저장되었습니다.');
                } catch (dbError) {
                    console.warn('데이터베이스 저장 실패 (메모리 캐시는 유지):', dbError.message);
                }
            } else {
                console.warn('⚠️ dbManager가 설정되지 않아 데이터베이스 저장을 건너뜁니다.');
            }

            console.log(`피드백 패턴 학습 완료: ${patternKey} (평점: ${rating})`);
            
        } catch (error) {
            console.error('피드백 학습 중 오류:', error);
        }
    }

    // 데이터베이스에 피드백 저장
    async saveFeedbackToDatabase(feedbackData) {
        console.log('🔍 saveFeedbackToDatabase 호출됨:', {
            hasDbManager: !!this.dbManager,
            feedbackData: {
                userMessage: feedbackData.userMessage?.substring(0, 50) + '...',
                aiResponse: feedbackData.aiResponse?.substring(0, 50) + '...',
                rating: feedbackData.rating,
                userId: feedbackData.userId
            }
        });

        if (!this.dbManager) {
            console.error('❌ 데이터베이스 매니저가 설정되지 않음');
            return;
        }

        try {
            const {
                sessionId, rating, userMessage, aiResponse, userId
            } = feedbackData;

            console.log('📝 데이터베이스 저장 시작:', {
                sessionId,
                rating,
                userId,
                userMessageLength: userMessage?.length,
                aiResponseLength: aiResponse?.length
            });

            // QA 데이터셋에 직접 저장 (피드백 포함)
            const qaQuery = `
                INSERT INTO qa_dataset (
                    user_question, ai_answer, user_id, rating, feedback_type
                ) VALUES ($1, $2, $3, $4, $5)
                RETURNING id
            `;

            const queryParams = [
                userMessage,
                aiResponse,
                userId || 'unknown',
                rating,
                'thumbs'
            ];

            console.log('🔍 SQL 쿼리 실행:', {
                query: qaQuery,
                params: queryParams
            });

            const qaResult = await this.dbManager.query(qaQuery, queryParams);

            console.log('✅ 데이터베이스 저장 성공:', {
                qaId: qaResult.rows[0].id,
                rating,
                rowsAffected: qaResult.rowCount
            });

        } catch (error) {
            console.error('❌ 데이터베이스 피드백 저장 실패:', error);
            console.error('❌ 오류 상세:', {
                message: error.message,
                code: error.code,
                detail: error.detail,
                hint: error.hint,
                stack: error.stack
            });
            throw error;
        }
    }
} 