#!/usr/bin/env python3
"""
FLOP 라이선스 관리 CLI 스크립트

Node.js API 서버에서 Python FLOP 라이선스 시스템을 호출하기 위한
명령행 인터페이스입니다.
"""

import sys
import json
import logging
import argparse
from typing import Dict, Any, Optional

# 로깅 설정 (에러만 출력)
logging.basicConfig(level=logging.ERROR, format='%(levelname)s: %(message)s')

try:
    from flop_license_manager import flop_license_manager
except ImportError:
    # 상대 경로로 다시 시도
    import os
    sys.path.append(os.path.dirname(os.path.abspath(__file__)))
    from flop_license_manager import flop_license_manager

def track_usage(data: Dict[str, Any]) -> Dict[str, Any]:
    """AI 사용량 추적"""
    try:
        result = flop_license_manager.track_ai_usage(
            provider=data.get('provider', ''),
            model=data.get('model', ''),
            prompt_tokens=data.get('prompt_tokens', 0),
            completion_tokens=data.get('completion_tokens', 0),
            duration_ms=data.get('duration_ms', 0.0),
            session_id=data.get('session_id'),
            user_id=data.get('user_id'),
            username=data.get('username'),
            feature=data.get('feature')
        )
        
        if result:
            return {
                'success': True,
                'data': result
            }
        else:
            return {
                'success': False,
                'error': 'FLOP 사용량 추적 실패'
            }
            
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def get_usage_report(data: Dict[str, Any]) -> Dict[str, Any]:
    """사용량 리포트 조회"""
    try:
        period = data.get('period', 'monthly')
        start_date = data.get('start_date')
        end_date = data.get('end_date')
        
        report = flop_license_manager.get_usage_report(period, start_date, end_date)
        
        return {
            'success': True,
            'data': report
        }
        
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def check_license(data: Dict[str, Any]) -> Dict[str, Any]:
    """라이선스 체크"""
    try:
        user_id = data.get('user_id')
        username = data.get('username')
        
        is_valid = flop_license_manager.check_license_limits(
            user_id=user_id,
            username=username
        )
        
        if is_valid:
            return {
                'success': True,
                'message': 'FLOP 라이선스 유효'
            }
        else:
            return {
                'success': False,
                'error': 'FLOP_LIMIT_EXCEEDED',
                'message': 'FLOP 사용 한도 초과'
            }
            
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def get_usage(data: Dict[str, Any]) -> Dict[str, Any]:
    """사용량 조회"""
    try:
        # user_id = data.get('user_id')
        user_id = str(data.get("user_id"))
        username = data.get('username')
        period = data.get('period', 'all')
        
        usage_data = flop_license_manager.get_usage_summary(
            user_id=user_id,
            username=username,
            period=period
        )
        
        return {
            'success': True,
            'data': usage_data
        }
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def get_license_info(data: Dict[str, Any]) -> Dict[str, Any]:
    """라이선스 정보 조회"""
    try:
        # user_id = data.get('user_id')
        user_id = str(data.get("user_id"))
        username = data.get('username')
        
        license_info = flop_license_manager.get_license_status(
            user_id=user_id,
            username=username
        )
        
        return {
            'success': True,
            'data': license_info
        }
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def estimate_monthly_cost(data: Dict[str, Any]) -> Dict[str, Any]:
    """월간 비용 추정"""
    try:
        estimated_gflops = data.get('estimated_gflops', 0)
        plan = data.get('plan', 'basic')
        
        estimate = flop_license_manager.estimate_monthly_cost(estimated_gflops, plan)
        
        return {
            'success': True,
            'data': estimate
        }
        
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def get_pricing_info(data: Dict[str, Any] = None) -> Dict[str, Any]:
    """가격 정책 정보 조회"""
    try:
        pricing = flop_license_manager.get_pricing_info()
        
        return {
            'success': True,
            'data': pricing
        }
        
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def update_pricing_config(data: Dict[str, Any]) -> Dict[str, Any]:
    """가격 정책 업데이트"""
    try:
        success = flop_license_manager.update_pricing_config(data.get('config', {}))
        
        return {
            'success': success,
            'message': '가격 정책 업데이트 완료' if success else '가격 정책 업데이트 실패'
        }
        
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def get_usage_stats(data: Dict[str, Any]) -> Dict[str, Any]:
    """사용량 통계 조회"""
    try:
        start_date = data.get('start_date')
        end_date = data.get('end_date')
        
        stats = flop_license_manager.flop_tracker.get_usage_stats(start_date, end_date)
        
        return {
            'success': True,
            'data': stats
        }
        
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def cleanup_old_data(data: Dict[str, Any]) -> Dict[str, Any]:
    """오래된 데이터 정리"""
    try:
        days_to_keep = data.get('days_to_keep', 90)
        flop_license_manager.flop_tracker.cleanup_old_data(days_to_keep)
        
        return {
            'success': True,
            'message': f'{days_to_keep}일 이전 데이터 정리 완료'
        }
        
    except Exception as e:
        return {
            'success': False,
            'error': str(e)
        }

def main():
    """메인 함수"""
    parser = argparse.ArgumentParser(description='FLOP 라이선스 관리 CLI')
    parser.add_argument('command', help='실행할 명령', choices=[
        'track_usage', 'check_license', 'get_usage', 'get_license_info',
        'get_usage_report', 'estimate_monthly_cost', 'get_pricing_info', 
        'update_pricing_config', 'get_usage_stats', 'cleanup_old_data'
    ])
    parser.add_argument('--input', help='JSON 입력 파일 경로')
    parser.add_argument('--output', help='JSON 출력 파일 경로')
    
    args = parser.parse_args()
    
    # 입력 데이터 읽기
    input_data = {}
    if args.input:
        try:
            with open(args.input, 'r', encoding='utf-8') as f:
                input_data = json.load(f)
        except Exception as e:
            print(json.dumps({
                'success': False,
                'error': f'입력 파일 읽기 실패: {str(e)}'
            }))
            sys.exit(1)
    else:
        # stdin에서 입력 데이터 읽기
        try:
            input_text = sys.stdin.read().strip()
            if input_text:
                input_data = json.loads(input_text)
        except json.JSONDecodeError:
            # JSON이 아닌 경우 빈 딕셔너리 사용
            input_data = {}
        except Exception as e:
            print(json.dumps({
                'success': False,
                'error': f'입력 데이터 읽기 실패: {str(e)}'
            }))
            sys.exit(1)
    
    # 명령 실행
    command_functions = {
        'track_usage': track_usage,
        'check_license': check_license,
        'get_usage': get_usage,
        'get_license_info': get_license_info,
        'get_usage_report': get_usage_report,
        'estimate_monthly_cost': estimate_monthly_cost,
        'get_pricing_info': get_pricing_info,
        'update_pricing_config': update_pricing_config,
        'get_usage_stats': get_usage_stats,
        'cleanup_old_data': cleanup_old_data
    }
    
    try:
        command_func = command_functions[args.command]
        result = command_func(input_data)
        
        # 결과 출력
        if args.output:
            with open(args.output, 'w', encoding='utf-8') as f:
                json.dump(result, f, ensure_ascii=False, indent=2)
        else:
            print(json.dumps(result, ensure_ascii=False))
            
    except KeyError:
        print(json.dumps({
            'success': False,
            'error': f'알 수 없는 명령: {args.command}'
        }))
        sys.exit(1)
    except Exception as e:
        print(json.dumps({
            'success': False,
            'error': f'명령 실행 실패: {str(e)}'
        }))
        sys.exit(1)

if __name__ == '__main__':
    main() 