#!/usr/bin/env python3
from dotenv import load_dotenv
# .env 파일 로드 (최우선)
load_dotenv()
import os
import sys
import time
import json
import psycopg2
from psycopg2 import pool
from psycopg2.extras import Json
from psycopg2 import sql
import shutil
import asyncio
import logging
import traceback
import threading
import warnings
import glob
import queue
import importlib.util
import random
import re
from threading import Thread, Lock
from datetime import datetime, timezone, timedelta
from typing import List, Dict, Any, Optional
from fastapi import FastAPI, File, UploadFile, Form, HTTPException, Query, Request
from fastapi.responses import JSONResponse
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
from setproctitle import setproctitle
import uvicorn
import redis
import concurrent.futures
from functools import partial
import asyncio

from multiprocessing import Process, Queue, Manager, Event
import uuid
from inotify_simple import INotify, flags
from contextlib import asynccontextmanager
import numpy as np
import torch
from transformers import AutoModel, AutoTokenizer
from langchain_huggingface import HuggingFaceEmbeddings
from langchain_text_splitters import RecursiveCharacterTextSplitter
import hashlib
import pandas as pd
import multiprocessing
import psutil
import math
from langchain_experimental.text_splitter import SemanticChunker
from sentence_transformers import SentenceTransformer
from pathlib import Path
from langchain.embeddings.base import Embeddings
import re
from dataclasses import dataclass
import httpx
import aiohttp
import gc
import uuid
import csv
import io
import tempfile
import base64

# utils 모듈 임포트 - 프로젝트 루트의 .py 파일 우선
project_root = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
utils_py_file = os.path.join(project_root, "utils.py")

print(f"[DEBUG] Checking for utils.py file: {utils_py_file}")
print(f"[DEBUG] utils.py file exists: {os.path.exists(utils_py_file)}")

if os.path.exists(utils_py_file):
    # 프로젝트 루트의 utils.py 파일이 있으면 우선 사용 (개발 모드)
    print(f"[DEBUG] Using Python source utils.py from project root")
    sys.path.insert(0, project_root)
    import utils
else:
    # .py 파일이 없으면 컴파일된 버전 사용
    print(f"[DEBUG] Python source utils.py not found, trying compiled version")
    try:
        import utils
    except ImportError:
        # 현재 디렉토리의 상위 디렉토리를 Python 경로에 추가
        sys.path.insert(0, project_root)
        import utils

# utils 모듈에서 필요한 함수들을 가져옵니다
extract_from_pdf = utils.extract_from_pdf
extract_from_hwp = utils.extract_from_hwp
extract_from_doc = utils.extract_from_doc
extract_from_ppt = utils.extract_from_ppt
convert_hwp_to_pdf = utils.convert_hwp_to_pdf
extract_from_hwp_hwp2txt = utils.extract_from_hwp_hwp2txt
clean_hwp_text = utils.clean_hwp_text

# 모든 경고 무시
warnings.filterwarnings('ignore', category=DeprecationWarning)
warnings.filterwarnings('ignore', category=UserWarning)
warnings.filterwarnings('ignore', category=FutureWarning)

# 현재 스크립트 경로 가져오기
script_dir = os.path.dirname(os.path.abspath(__file__))

# 먼저 .py 파일이 있는지 확인 (개발 모드 우선)
py_file = os.path.join(script_dir, "rag_process.py")
print(f"[DEBUG] Checking for Python file: {py_file}")
print(f"[DEBUG] Python file exists: {os.path.exists(py_file)}")

if os.path.exists(py_file):
    # .py 파일이 있으면 우선 사용 (개발 모드)
    module_path = py_file
    print(f"[DEBUG] Using Python source file: {os.path.basename(py_file)}")
else:
    # .py 파일이 없으면 컴파일된 파일 찾기
    print(f"[DEBUG] Python source file not found, looking for compiled module")
    
    # OS별 모듈 패턴 정의
    if sys.platform.startswith('win'):
        module_pattern = os.path.join(script_dir, "rag_process.cp*-win_amd64.pyd")
    elif sys.platform.startswith('darwin'):
        module_pattern = os.path.join(script_dir, "rag_process.cpython-*-darwin.so")
    else:  # Linux
        module_pattern = os.path.join(script_dir, "rag_process.cpython-312-x86_64-linux-gnu.so")
    
    print(f"[DEBUG] Module pattern: {module_pattern}")
    
    # .so/.pyd 파일 찾기
    module_files = glob.glob(module_pattern)
    print(f"[DEBUG] Found module files: {module_files}")
    
    if not module_files:
        # 정확한 파일명으로 시도
        exact_file = os.path.join(script_dir, "rag_process.cpython-312-x86_64-linux-gnu.so")
        print(f"[DEBUG] Trying exact file: {exact_file}")
        print(f"[DEBUG] Exact file exists: {os.path.exists(exact_file)}")
        if os.path.exists(exact_file):
            module_files = [exact_file]
            print(f"[DEBUG] Using compiled module: {os.path.basename(exact_file)}")
        else:
            # 컴파일된 모듈을 찾지 못한 경우 에러
            raise ImportError(f"No module found. Tried Python file: {py_file}, compiled patterns: {module_pattern}, {exact_file}")
    
    module_path = module_files[0]

# 모듈 로드
spec = importlib.util.spec_from_file_location("rag_process", module_path)
if spec is None:
    raise ImportError(f"Failed to create module spec for {module_path}")

module = importlib.util.module_from_spec(spec)
sys.modules["rag_process"] = module
spec.loader.exec_module(module)

# GraphRAG 의존성 확인
try:
    import requests
    import ollama
    HAS_GRAPHRAG_DEPS = True
except ImportError:
    HAS_GRAPHRAG_DEPS = False

# rag_process 모듈에서 필요한 클래스와 함수들을 가져옵니다
from rag_process import Logger, DocumentProcessor, get_rag_settings, SUPPORTED_EXTENSIONS, get_processing_status, update_doc_status, tokenize, split_text, get_global_postgresql, get_semantic_model

# get_graphrag_processor 함수는 파일 하단으로 이동됨

def get_rag_docs_path():
    """RAG 문서 경로를 가져오는 함수"""
    settings = get_rag_settings()
    return settings.get('rag_docs_path', os.path.join(os.path.expanduser('~'), '.airun', 'rag_docs'))

# Connection Pool 헬퍼 함수들
def get_pool_connection():
    """Connection pool에서 연결을 가져오거나 직접 연결 생성 (fallback)"""
    try:
        _, postgresql_wrapper = get_global_postgresql()
        conn = postgresql_wrapper.get_connection()

        # 풀에서 가져온 연결이 이미 닫혀있는 경우 직접 새 연결 생성
        if hasattr(conn, "closed") and conn.closed != 0:
            logger.info("⚠️ 풀에서 가져온 연결이 닫혀있음, 직접 새 연결 생성")
            raise psycopg2.InterfaceError("Connection from pool was closed")

        return conn
    except Exception as e:
        # Fallback: 직접 연결
        import psycopg2
        import os
        return psycopg2.connect(
            host=os.getenv('DB_HOST', os.getenv('POSTGRES_HOST', 'localhost')),
            port=int(os.getenv('DB_PORT', os.getenv('POSTGRES_PORT', '5433'))),
            user=os.getenv('DB_USER', os.getenv('POSTGRES_USER', 'ivs')),
            password=os.getenv('DB_PASSWORD', os.getenv('POSTGRES_PASSWORD', '1234')),
            database=os.getenv('DB_NAME', os.getenv('POSTGRES_DB', 'airun'))
        )

def return_pool_connection(conn):
    """Connection을 pool로 안전하게 반환하거나 직접 close (fallback)"""
    if not conn:
        return

    try:
        # 연결이 이미 닫혔는지 확인
        if hasattr(conn, 'closed') and conn.closed != 0:
            logger.info("📝 이미 닫힌 연결입니다")
            return

        # 풀로 반환 시도
        _, postgresql_wrapper = get_global_postgresql()
        postgresql_wrapper.return_connection(conn)
        # logger.info("🔄 풀 연결 반환 완료")

    except Exception as e:
        # "trying to put unkeyed connection" 오류에 대한 특별 처리
        if "unkeyed connection" in str(e):
            logger.info(f"⚠️ 키가 없는 연결을 직접 닫음: {str(e)}")
        else:
            logger.info(f"⚠️ 풀 반환 실패, 직접 닫기로 폴백: {str(e)}")

        try:
            if conn and hasattr(conn, 'close') and getattr(conn, 'closed', 1) == 0:
                conn.close()
                logger.info("📝 연결을 직접 닫았습니다")
        except Exception as close_error:
            logger.info(f"❌ 연결 닫기도 실패: {str(close_error)}", error=True)

def ensure_active_connection(conn, cursor, logger, document_processor=None, force_new=False):
    """PostgreSQL 연결과 커서를 유효한 상태로 보장"""
    try:
        if not force_new and conn and getattr(conn, "closed", 1) == 0:
            # 기존 커서가 유효하면 그대로 사용
            if cursor and not getattr(cursor, "closed", True):
                return conn, cursor

            # 기존 연결에서 새 커서 생성
            cursor = conn.cursor()
            logger.info("임베딩 워커: 기존 연결에서 새 커서 생성")
            return conn, cursor
    except (psycopg2.InterfaceError, psycopg2.OperationalError):
        pass

    # 기존 연결 정리 후 새 연결 확보
    if document_processor and conn and getattr(conn, "closed", 1) == 0:
        try:
            document_processor.return_db_connection(conn)
        except Exception:
            pass

    conn = get_pool_connection()
    conn.autocommit = False
    cursor = conn.cursor()
    logger.info("임베딩 워커: PostgreSQL 재연결 완료 (ensure_active_connection)")
    return conn, cursor

def setup_watch_directory(inotify, path):
    """디렉토리 감시 설정"""
    try:
        if not os.path.exists(path):
            logger.warning(f"감시할 경로가 존재하지 않음: {path}")
            return None
        
        # 디렉토리와 하위 디렉토리 감시 설정
        watches = {}
        
        # 메인 디렉토리 감시
        mask = flags.CREATE | flags.DELETE | flags.MODIFY | flags.MOVED_TO | flags.MOVED_FROM | flags.CLOSE_WRITE
        wd = inotify.add_watch(path, mask)
        watches[wd] = path
        logger.info(f"디렉토리 감시 설정: {path}")
        
        # 하위 디렉토리들도 감시 설정 (.extracts 폴더 제외)
        for root, dirs, files in os.walk(path):
            for dir_name in dirs:
                # .extracts 폴더는 감시 대상에서 제외
                if dir_name == '.extracts':
                    continue
                    
                subdir_path = os.path.join(root, dir_name)
                
                # .extracts 하위 폴더도 제외
                if '/.extracts/' in subdir_path or os.sep + '.extracts' + os.sep in subdir_path:
                    continue
                    
                try:
                    wd = inotify.add_watch(subdir_path, mask)
                    watches[wd] = subdir_path
                    logger.info(f"하위 디렉토리 감시 설정: {subdir_path}")
                except Exception as e:
                    logger.warning(f"하위 디렉토리 감시 설정 실패: {subdir_path} - {str(e)}")
        
        return watches
        
    except Exception as e:
        logger.error(f"디렉토리 감시 설정 실패: {path} - {str(e)}")
        return None

def update_watch_if_needed(inotify, watches, current_path):
    """감시 경로 업데이트가 필요한지 확인하고 업데이트"""
    try:
        rag_path = get_rag_docs_path()
        if current_path != rag_path:
            logger.info(f"감시 경로 변경 감지: {current_path} -> {rag_path}")
            
            # 기존 감시 제거
            for wd in list(watches.keys()):
                try:
                    inotify.rm_watch(wd)
                except Exception as e:
                    logger.warning(f"기존 감시 제거 실패: {str(e)}")
            
            # 새로운 감시 설정
            new_watches = setup_watch_directory(inotify, rag_path)
            if new_watches:
                return new_watches, rag_path
            else:
                return {}, rag_path
        
        return watches, current_path
        
    except Exception as e:
        logger.error(f"감시 경로 업데이트 실패: {str(e)}")
        return watches, current_path

def check_and_cleanup_watches(inotify, watches):
    """삭제된 디렉토리에 대한 감시 정리"""
    try:
        cleaned_watches = {}
        for wd, path in watches.items():
            if os.path.exists(path):
                cleaned_watches[wd] = path
            else:
                try:
                    inotify.rm_watch(wd)
                    logger.info(f"삭제된 디렉토리 감시 제거: {path}")
                except Exception as e:
                    logger.warning(f"감시 제거 실패: {path} - {str(e)}")
        
        return cleaned_watches
        
    except Exception as e:
        logger.error(f"감시 정리 실패: {str(e)}")
        return watches

# 기본 설정 및 경로 정의
BASE_DIR = os.path.expanduser("~/.airun")
LOG_DIR = os.path.join(BASE_DIR, "logs")
LOG_FILE = os.path.join(LOG_DIR, "airun-rag.log")
CONFIG_FILE = os.path.join(BASE_DIR, "airun.conf")


# 기타 설정
CONFIG_CHECK_INTERVAL = 5  # 설정 파일 확인 주기 (초)
REDIS_HOST = 'localhost'
REDIS_PORT = 6379
MODEL_NAME = "nlpai-lab/KURE-v1"
BATCH_SIZE = 24
MAX_SEQUENCE_LENGTH = 512

# 동기화 설정
SYNC_CHECK_INTERVAL = 1800  # 물리적 파일과 DB 동기화 검사 주기 (초, 기본값: 30분)
SYNC_ERROR_RETRY_INTERVAL = 180  # 동기화 오류 시 재시도 주기 (초, 기본값: 3분)

# Redis 큐 키 정의
QUEUE_KEYS = {
    'documents': 'rag:queue:documents',  # 처리할 문서 목록
    'processing': 'rag:queue:processing',  # 처리 중인 문서
    'completed': 'rag:queue:completed',  # 처리 완료된 문서
    'failed': 'rag:queue:failed',  # 처리 실패한 문서
    'batch_size': 'rag:queue:batch_size',  # 배치 크기
    'current_batch': 'rag:queue:current_batch'  # 현재 처리 중인 배치 번호
}

# 전역 변수 선언
embedding_service = None
document_processor = None
# logger = None  # 제거: get_logger()에서 초기화
inotify = None
watches = {}
redis_client = None
thread_pool = None  # 초기화는 main() 함수에서 수행
# PostgreSQL 연결 풀 제거됨 - 중앙 집중식 풀 사용
graphrag_processor = None  # GraphRAG 프로세서 싱글톤


class GraphRAGProcessor:
    """GraphRAG 처리를 위한 클래스"""

    def __init__(self):
        self.ollama_server = "http://localhost:11434"
        self.chat_model = "hamonize:latest"
        self.embedding_model = "nlpai-lab/KURE-v1"

        # Knowledge graph 속성 초기화 - 실제 데이터베이스에서 로드
        self.knowledge_graph = self._load_knowledge_graph_from_db()

        # Document graphs 속성 초기화
        self.document_graphs = {}

        logger = get_logger()
        logger.info(f"GraphRAGProcessor 초기화 완료 - 엔티티: {len(self.knowledge_graph.entities)}개, 관계: {len(self.knowledge_graph.relationships)}개")

    def _load_knowledge_graph_from_db(self):
        """데이터베이스에서 지식 그래프 데이터를 로드"""
        logger = get_logger()

        # KnowledgeGraph 객체 생성
        knowledge_graph = type('KnowledgeGraph', (), {
            'entities': [],
            'relationships': []
        })()

        try:
            conn = get_pool_connection()
            try:
                with conn.cursor() as cur:
                    # 엔티티 로드
                    cur.execute("SELECT id, name, type, description, source_documents FROM graph_entities ORDER BY created_at")
                    entities_data = cur.fetchall()

                    knowledge_graph.entities = []
                    for entity_row in entities_data:
                        entity_dict = {
                            'id': entity_row[0],
                            'name': entity_row[1],
                            'type': entity_row[2],
                            'description': entity_row[3],
                            'source_documents': entity_row[4] if entity_row[4] else []
                        }
                        knowledge_graph.entities.append(entity_dict)

                    # 관계 로드
                    cur.execute("""
                        SELECT r.id, r.source_entity_id, r.target_entity_id, r.relationship_type,
                               r.weight, r.description, r.source_documents,
                               se.name as source_name, te.name as target_name
                        FROM graph_relationships r
                        JOIN graph_entities se ON r.source_entity_id = se.id
                        JOIN graph_entities te ON r.target_entity_id = te.id
                        ORDER BY r.created_at
                    """)
                    relationships_data = cur.fetchall()

                    knowledge_graph.relationships = []
                    for rel_row in relationships_data:
                        rel_dict = {
                            'id': rel_row[0],
                            'source_entity_id': rel_row[1],
                            'target_entity_id': rel_row[2],
                            'relationship_type': rel_row[3],
                            'weight': rel_row[4],
                            'description': rel_row[5],
                            'source_documents': rel_row[6] if rel_row[6] else [],
                            'source_name': rel_row[7],
                            'target_name': rel_row[8]
                        }
                        knowledge_graph.relationships.append(rel_dict)

                    logger.info(f"✅ 데이터베이스에서 지식 그래프 로드 완료: 엔티티 {len(knowledge_graph.entities)}개, 관계 {len(knowledge_graph.relationships)}개")

            finally:
                conn.close()

        except Exception as e:
            logger.info(f"지식 그래프 로드 실패: {str(e)}", error=True)
            # 실패 시 빈 그래프 반환
            knowledge_graph.entities = []
            knowledge_graph.relationships = []

        return knowledge_graph

    def get_entity_relationships(self, entity_name: str, max_distance: int = 2):
        """엔티티의 관계를 조회하는 메서드"""
        try:
            conn = get_pool_connection()
            try:
                with conn.cursor() as cur:
                    # 엔티티 ID 찾기
                    cur.execute("SELECT id FROM graph_entities WHERE name = %s", (entity_name,))
                    entity_result = cur.fetchone()

                    if not entity_result:
                        return []

                    entity_id = entity_result[0]

                    # 직접 연결된 관계들 조회
                    cur.execute("""
                        SELECT
                            r.id,
                            se.name as source_name,
                            te.name as target_name,
                            r.relationship_type,
                            r.weight,
                            r.description
                        FROM graph_relationships r
                        JOIN graph_entities se ON r.source_entity_id = se.id
                        JOIN graph_entities te ON r.target_entity_id = te.id
                        WHERE r.source_entity_id = %s OR r.target_entity_id = %s
                        ORDER BY r.weight DESC
                        LIMIT 50
                    """, (entity_id, entity_id))

                    relationships = cur.fetchall()

                    return [{
                        'id': row[0],
                        'source_name': row[1],
                        'target_name': row[2],
                        'relationship_type': row[3],
                        'weight': row[4],
                        'description': row[5]
                    } for row in relationships]

            finally:
                conn.close()

        except Exception as e:
            logger = get_logger()
            logger.info(f"엔티티 관계 조회 실패: {str(e)}", error=True)
            return []

    async def search_graph(self, query: str, user_id: str, mode: str = "hybrid", view_mode: str = "overview", top_k: int = 5):
        """그래프 검색 메서드 - 키워드 기반 스마트 검색"""
        try:
            logger = get_logger()

            # 파일명 기반 검색인지 확인
            filename_search = None
            if query.startswith('filename:'):
                filename_search = query.replace('filename:', '').strip()
                logger.info(f"파일명 기반 GraphRAG 검색: {filename_search}")

                # 파일명 검색의 경우 해당 파일의 모든 엔티티 검색
                conn = get_pool_connection()
                try:
                    with conn.cursor() as cur:
                        # 해당 파일명을 포함한 source_documents를 가진 엔티티들 검색
                        cur.execute("""
                            SELECT e.id, e.name, e.type, e.description, e.source_documents,
                                   COUNT(r.id) as connection_count
                            FROM graph_entities e
                            LEFT JOIN graph_relationships r ON (r.source_entity_id = e.id OR r.target_entity_id = e.id)
                            WHERE %s = ANY(e.source_documents)
                               OR EXISTS (
                                   SELECT 1 FROM unnest(e.source_documents) as doc
                                   WHERE doc LIKE %s
                               )
                            GROUP BY e.id, e.name, e.type, e.description, e.source_documents
                            ORDER BY connection_count DESC, LENGTH(e.name)
                            LIMIT %s
                        """, (filename_search, f'%{filename_search}%', top_k))

                        entities = cur.fetchall()

                        if entities:
                            # 관계 정보도 함께 가져오기
                            entity_ids = [entity[0] for entity in entities]
                            relationships = await self.get_entity_relationships(entity_ids)
                            communities = []  # 파일별 검색에서는 커뮤니티 정보 생략

                            entity_list = [{
                                'id': row[0],
                                'name': row[1],
                                'type': row[2],
                                'description': row[3],
                                'source_documents': row[4] or []
                            } for row in entities]
                            
                            return {
                                'entities': entity_list,
                                'relationships': relationships,
                                'communities': communities,
                                'total_entities': len(entity_list),
                                'total_relationships': len(relationships),
                                'graph_results': entity_list  # 하이브리드 모드를 위한 alias
                            }
                        else:
                            logger.info(f"파일 '{filename_search}'에서 엔티티를 찾을 수 없음")
                            return {
                                'entities': [], 
                                'relationships': [], 
                                'communities': [],
                                'total_entities': 0,
                                'total_relationships': 0,
                                'graph_results': []
                            }

                finally:
                    conn.close()

            # 일반 키워드 검색
            from rag_process import tokenize
            keywords = tokenize(query, include_ngrams=False)

            # 중요한 키워드 필터링 (길이 2 이상, 일반적 단어 제외)
            generic_words = {'ceo', '경영', '철학', '전략', '기업', '회사', '대표', '리더', '사업', '산업', '의'}
            filtered_keywords = [kw for kw in keywords
                               if len(kw) >= 2 and kw.lower() not in generic_words]
            
            # 다국어 키워드 매핑 추가 (한국어 ↔ 영문)
            multilang_mapping = {
                '테슬라': ['Tesla', '테슬라'],
                '일론': ['Elon', '일론'],
                '머스크': ['Musk', '머스크'],
                '애플': ['Apple', '애플'],
                '구글': ['Google', '구글'],
                '마이크로소프트': ['Microsoft', '마이크로소프트'],
                '아마존': ['Amazon', '아마존'],
                '메타': ['Meta', 'Facebook', '메타', '페이스북'],
                'tesla': ['Tesla', '테슬라'],
                'elon': ['Elon', '일론'],
                'musk': ['Musk', '머스크'],
                'apple': ['Apple', '애플'],
                'google': ['Google', '구글'],
                'microsoft': ['Microsoft', '마이크로소프트'],
                'amazon': ['Amazon', '아마존'],
                'meta': ['Meta', 'Facebook', '메타', '페이스북'],
                'facebook': ['Meta', 'Facebook', '메타', '페이스북']
            }
            
            # 다국어 매핑 적용
            expanded_keywords = set(filtered_keywords)
            for keyword in filtered_keywords:
                if keyword.lower() in multilang_mapping:
                    expanded_keywords.update(multilang_mapping[keyword.lower()])
            
            filtered_keywords = list(expanded_keywords)

            logger.info(f"GraphRAG 검색 키워드: {filtered_keywords}")

            conn = get_pool_connection()
            try:
                with conn.cursor() as cur:
                    logger.info(f"GraphRAG 검색 - 뷰 모드: {view_mode}")
                    
                    if view_mode == "overview":
                        # 전체 보기: 각 커뮤니티에서 균형있게 선택
                        entities = await self._search_overview_entities(cur, filtered_keywords, top_k)
                        
                    elif view_mode == "query-focused": 
                        # 질의 중심: 쿼리와 직접 관련된 엔티티들만
                        entities = await self._search_query_focused_entities(cur, filtered_keywords, query, top_k)
                        
                    elif view_mode == "entity-focused":
                        # 엔티티 중심: 중요도 높은 핵심 엔티티들
                        entities = await self._search_entity_focused_entities(cur, filtered_keywords, top_k)
                        
                    else:
                        # 기본값: 기존 방식
                        entities = await self._search_default_entities(cur, filtered_keywords, top_k)
                    
                    logger.info(f"GraphRAG 검색 결과: {len(entities)}개 엔티티 - {[e['name'] for e in entities]}")
                    
                    return {
                        'entities': entities,
                        'relationships': [],
                        'communities': [],
                        'total_entities': len(entities),
                        'total_relationships': 0,
                        'graph_results': entities  # 하이브리드 모드를 위한 alias
                    }

            finally:
                conn.close()

        except Exception as e:
            logger = get_logger()
            logger.info(f"그래프 검색 실패: {str(e)}", error=True)
            return {'entities': [], 'relationships': [], 'communities': []}

    async def _search_overview_entities(self, cur, keywords, top_k):
        """전체 보기: 질문 관련 엔티티를 중심으로 한 확장된 연관 네트워크"""
        entities = []

        # 1단계: 키워드와 직접 관련된 핵심 엔티티들 찾기
        core_entities = []
        for keyword in keywords[:5]:  # 상위 5개 키워드
            cur.execute("""
                SELECT e.id, e.name, e.type, e.description, e.source_documents,
                       COUNT(r.id) as connection_count,
                       CASE
                           WHEN LOWER(e.name) = LOWER(%s) THEN 1
                           WHEN LOWER(e.name) LIKE LOWER(%s) THEN 2
                           WHEN LOWER(e.description) LIKE LOWER(%s) THEN 3
                           WHEN e.properties->>'aliases' IS NOT NULL AND 
                                EXISTS(SELECT 1 FROM jsonb_array_elements_text(e.properties->'aliases') AS alias 
                                       WHERE LOWER(alias) = LOWER(%s) OR LOWER(alias) LIKE LOWER(%s)) THEN 1
                           ELSE 4
                       END as relevance_score
                FROM graph_entities e
                LEFT JOIN graph_relationships r ON (r.source_entity_id = e.id OR r.target_entity_id = e.id)
                WHERE LOWER(e.name) LIKE LOWER(%s) 
                   OR LOWER(e.description) LIKE LOWER(%s)
                   OR (e.properties->>'aliases' IS NOT NULL AND 
                       EXISTS(SELECT 1 FROM jsonb_array_elements_text(e.properties->'aliases') AS alias 
                              WHERE LOWER(alias) LIKE LOWER(%s)))
                GROUP BY e.id, e.name, e.type, e.description, e.source_documents
                ORDER BY relevance_score, connection_count DESC, LENGTH(e.name)
                LIMIT 3
            """, (keyword, f'%{keyword}%', f'%{keyword}%', keyword, f'%{keyword}%', 
                  f'%{keyword}%', f'%{keyword}%', f'%{keyword}%'))

            keyword_entities = cur.fetchall()
            core_entities.extend(keyword_entities)

        # 중복 제거
        seen_ids = set()
        unique_core = []
        for entity in core_entities:
            if entity[0] not in seen_ids:
                seen_ids.add(entity[0])
                unique_core.append(entity)

        # 2단계: 핵심 엔티티들과 연결된 확장 엔티티들 찾기 (넓은 맥락)
        expanded_entities = list(unique_core)  # 핵심 엔티티들 포함

        if unique_core:
            core_ids = [entity[0] for entity in unique_core]

            # 1차 연결 엔티티들 (직접 연결)
            cur.execute("""
                SELECT e.id, e.name, e.type, e.description, e.source_documents,
                       COUNT(r2.id) as connection_count,
                       5 as relevance_score
                FROM graph_entities e
                JOIN graph_relationships r ON (
                    (r.source_entity_id = e.id AND r.target_entity_id = ANY(%s))
                    OR (r.target_entity_id = e.id AND r.source_entity_id = ANY(%s))
                )
                LEFT JOIN graph_relationships r2 ON (r2.source_entity_id = e.id OR r2.target_entity_id = e.id)
                WHERE e.id NOT IN %s
                GROUP BY e.id, e.name, e.type, e.description, e.source_documents
                ORDER BY connection_count DESC, LENGTH(e.name)
                LIMIT %s
            """, (core_ids, core_ids, tuple(core_ids), max(5, top_k - len(unique_core))))

            first_level = cur.fetchall()
            expanded_entities.extend(first_level)

            # 3단계: 2차 연결 엔티티들 (확장된 맥락을 위해)
            if len(expanded_entities) < top_k:
                all_ids = [entity[0] for entity in expanded_entities]
                remaining_slots = top_k - len(expanded_entities)

                cur.execute("""
                    SELECT e.id, e.name, e.type, e.description, e.source_documents,
                           COUNT(r2.id) as connection_count,
                           6 as relevance_score
                    FROM graph_entities e
                    JOIN graph_relationships r ON (
                        (r.source_entity_id = e.id AND r.target_entity_id = ANY(%s))
                        OR (r.target_entity_id = e.id AND r.source_entity_id = ANY(%s))
                    )
                    LEFT JOIN graph_relationships r2 ON (r2.source_entity_id = e.id OR r2.target_entity_id = e.id)
                    WHERE e.id NOT IN %s
                      AND r.weight >= 0.5  -- 적당한 관계 강도 필터
                    GROUP BY e.id, e.name, e.type, e.description, e.source_documents
                    ORDER BY connection_count DESC, LENGTH(e.name)
                    LIMIT %s
                """, (all_ids, all_ids, tuple(all_ids), remaining_slots))

                second_level = cur.fetchall()
                expanded_entities.extend(second_level)

        # 4단계: 최종 중복 제거 및 다양성 보장
        final_seen = set()
        final_entities = []
        type_counts = {}

        # 관련성 순으로 정렬 (relevance_score가 낮을수록 더 관련성 높음)
        expanded_entities.sort(key=lambda x: (x[6] if len(x) > 6 else 4, -x[5]))

        for entity in expanded_entities:
            entity_id = entity[0]
            entity_type = entity[2]

            if entity_id not in final_seen:
                # 타입별 다양성 보장 (각 타입별 최대 4개)
                if type_counts.get(entity_type, 0) < 4:
                    final_entities.append(entity)
                    final_seen.add(entity_id)
                    type_counts[entity_type] = type_counts.get(entity_type, 0) + 1

                    if len(final_entities) >= top_k:
                        break

        return [{
            'id': row[0],
            'name': row[1],
            'type': row[2],
            'description': row[3],
            'source_documents': row[4] or []
        } for row in final_entities[:top_k]]

    async def _search_query_focused_entities(self, cur, keywords, query, top_k):
        """질의 중심: 쿼리에서 언급된 정확한 엔티티들만 엄격하게 선택"""
        entities = []

        # 쿼리에서 직접 언급된 핵심 엔티티만 찾기 (정확한 매칭 우선)
        query_lower = query.lower()
        core_keywords = keywords[:3]  # 가장 중요한 키워드만 3개

        # 1단계: 쿼리에 정확히 언급된 엔티티 찾기
        mentioned_entities = []
        for keyword in core_keywords:
            if keyword.lower() in query_lower:
                cur.execute("""
                    SELECT id, name, type, description, source_documents,
                           CASE
                               WHEN LOWER(name) = LOWER(%s) THEN 1    -- 정확한 이름 매칭
                               WHEN LOWER(%s) LIKE '%%' || LOWER(name) || '%%' THEN 2  -- 쿼리에 엔티티명 포함
                               WHEN LOWER(name) LIKE LOWER(%s) THEN 3  -- 부분 매칭
                               ELSE 4
                           END as match_score,
                           LENGTH(name) as name_length
                    FROM graph_entities
                    WHERE LOWER(name) = LOWER(%s)
                       OR (LOWER(%s) LIKE '%%' || LOWER(name) || '%%' AND LENGTH(name) >= 2)
                       OR LOWER(name) LIKE LOWER(%s)
                    ORDER BY match_score, name_length
                    LIMIT 2
                """, (keyword, query_lower, f'%{keyword}%', keyword, query_lower, f'%{keyword}%'))

                keyword_entities = cur.fetchall()
                mentioned_entities.extend(keyword_entities)

        # 2단계: 언급된 엔티티와 직접 연결된 엔티티 추가 (관계가 있는 것만)
        if mentioned_entities:
            mentioned_ids = [entity[0] for entity in mentioned_entities]

            cur.execute("""
                SELECT DISTINCT e.id, e.name, e.type, e.description, e.source_documents,
                       5 as match_score, LENGTH(e.name) as name_length
                FROM graph_entities e
                JOIN graph_relationships r ON (
                    (r.source_entity_id = e.id AND r.target_entity_id = ANY(%s))
                    OR (r.target_entity_id = e.id AND r.source_entity_id = ANY(%s))
                )
                WHERE e.id NOT IN %s
                  AND r.weight >= 0.7  -- 강한 관계만
                ORDER BY LENGTH(e.name)
                LIMIT %s
            """, (mentioned_ids, mentioned_ids, tuple(mentioned_ids), max(1, top_k - len(mentioned_entities))))

            related_entities = cur.fetchall()
            mentioned_entities.extend(related_entities)

        # 3단계: 중복 제거 및 관련성 순 정렬
        seen_ids = set()
        unique_entities = []
        for entity in mentioned_entities:
            if entity[0] not in seen_ids:
                seen_ids.add(entity[0])
                unique_entities.append(entity)

        # 정확도 순으로 정렬 (match_score가 낮을수록 더 정확함)
        unique_entities.sort(key=lambda x: (x[5], x[6]))  # match_score, name_length
        unique_entities = unique_entities[:top_k]

        return [{
            'id': row[0],
            'name': row[1],
            'type': row[2],
            'description': row[3],
            'source_documents': row[4] or []
        } for row in unique_entities]

    async def _search_entity_focused_entities(self, cur, keywords, top_k):
        """엔티티 중심: 중요도 높은 핵심 엔티티들"""
        entities = []
        
        # 고유명사와 핵심 개념 우선
        for keyword in keywords[:6]:
            cur.execute("""
                SELECT e.id, e.name, e.type, e.description, e.source_documents,
                       (SELECT COUNT(*) FROM graph_relationships r 
                        WHERE r.source_entity_id = e.id OR r.target_entity_id = e.id) as relationship_count,
                       CASE 
                           WHEN e.type IN ('person', 'organization', 'location') THEN 1
                           WHEN e.type IN ('product', 'technology', 'concept') THEN 2
                           ELSE 3
                       END as type_priority
                FROM graph_entities e
                WHERE LOWER(e.name) LIKE LOWER(%s) OR LOWER(e.description) LIKE LOWER(%s)
                ORDER BY type_priority, relationship_count DESC, LENGTH(e.name)
                LIMIT 3
            """, (f'%{keyword}%', f'%{keyword}%'))
            
            keyword_entities = cur.fetchall()
            entities.extend(keyword_entities)
        
        # 중복 제거 및 중요도 순 정렬
        seen_ids = set()
        unique_entities = []
        for entity in entities:
            if entity[0] not in seen_ids:
                seen_ids.add(entity[0])
                unique_entities.append(entity)
        
        # 중요도 순으로 정렬 (타입 우선순위 + 관계 수)
        unique_entities.sort(key=lambda x: (x[6], -x[5]))  # type_priority, -relationship_count
        unique_entities = unique_entities[:min(top_k, 10)]
        
        return [{
            'id': row[0],
            'name': row[1],
            'type': row[2],
            'description': row[3], 
            'source_documents': row[4] or []
        } for row in unique_entities]

    async def _search_default_entities(self, cur, keywords, top_k):
        """기본 검색: 기존 방식"""
        all_entities = []
        
        # 각 키워드별로 엔티티 검색
        for keyword in keywords[:10]:  # 최대 10개 키워드만
            cur.execute("""
                SELECT id, name, type, description, source_documents
                FROM graph_entities
                WHERE LOWER(name) LIKE LOWER(%s)
                   OR LOWER(description) LIKE LOWER(%s)
                ORDER BY
                    CASE
                        WHEN LOWER(name) = LOWER(%s) THEN 1
                        WHEN LOWER(name) LIKE LOWER(%s) THEN 2
                        WHEN LOWER(description) LIKE LOWER(%s) THEN 3
                        ELSE 4
                    END,
                    LENGTH(name)
                LIMIT %s
            """, (f'%{keyword}%', f'%{keyword}%', keyword, f'%{keyword}%', f'%{keyword}%', min(top_k, 10)))
            
            keyword_entities = cur.fetchall()
            all_entities.extend(keyword_entities)
        
        # 중복 제거 (ID 기준)
        seen_ids = set()
        unique_entities = []
        for entity in all_entities:
            if entity[0] not in seen_ids:
                seen_ids.add(entity[0])
                unique_entities.append(entity)
        
        # top_k 개수로 제한
        unique_entities = unique_entities[:top_k]
        
        return [{
            'id': row[0],
            'name': row[1],
            'type': row[2],
            'description': row[3],
            'source_documents': row[4] or []
        } for row in unique_entities]

    def get_multi_hop_paths(self, source_entity: str, target_entity: str, max_hops: int = 3):
        """멀티홉 경로 탐색 메서드"""
        try:
            conn = get_pool_connection()
            try:
                with conn.cursor() as cur:
                    # 소스와 타겟 엔티티 ID 찾기
                    cur.execute("SELECT id FROM graph_entities WHERE name = %s", (source_entity,))
                    source_result = cur.fetchone()
                    cur.execute("SELECT id FROM graph_entities WHERE name = %s", (target_entity,))
                    target_result = cur.fetchone()

                    if not source_result or not target_result:
                        return []

                    source_id = source_result[0]
                    target_id = target_result[0]

                    # 직접 연결 경로만 검색 (단순화)
                    cur.execute("""
                        SELECT r.relationship_type, se.name, te.name, r.weight
                        FROM graph_relationships r
                        JOIN graph_entities se ON r.source_entity_id = se.id
                        JOIN graph_entities te ON r.target_entity_id = te.id
                        WHERE (r.source_entity_id = %s AND r.target_entity_id = %s)
                           OR (r.source_entity_id = %s AND r.target_entity_id = %s)
                    """, (source_id, target_id, target_id, source_id))

                    paths = []
                    for row in cur.fetchall():
                        paths.append({
                            'path': [row[1], row[2]],
                            'relationship_types': [row[0]],
                            'total_weight': row[3],
                            'hops': 1
                        })

                    return paths

            finally:
                conn.close()

        except Exception as e:
            logger = get_logger()
            logger.info(f"멀티홉 경로 탐색 실패: {str(e)}", error=True)
            return []

    async def process_document_for_graphrag(self, file_path: str, user_id: str = None):
        """문서를 GraphRAG용으로 처리하는 메서드"""
        logger = get_logger()
        logger.info(f"GraphRAG 문서 처리 시작: {file_path}")
        
        try:
            # 파일 존재 확인
            if not os.path.exists(file_path):
                raise FileNotFoundError(f"파일을 찾을 수 없습니다: {file_path}")
            
            # 문서 내용 읽기
            document_content = ""
            filename = os.path.basename(file_path)
            
            if file_path.endswith('.pdf'):
                # PDF 파일 처리
                try:
                    import pdfplumber
                    with pdfplumber.open(file_path) as pdf:
                        document_content = "\n".join([page.extract_text() for page in pdf.pages if page.extract_text()])
                except Exception as e:
                    logger.info(f"PDF 처리 실패: {str(e)}", error=True)
                    return {"status": "error", "message": f"PDF 처리 실패: {str(e)}"}
            else:
                # 텍스트 파일 처리
                with open(file_path, 'r', encoding='utf-8') as f:
                    document_content = f.read()
            
            if not document_content.strip():
                raise ValueError("문서 내용이 비어있습니다")
            
            logger.info(f"문서 내용 길이: {len(document_content)} 문자")
            
            # LLM을 사용하여 엔티티와 관계 추출
            entities_data, relationships_data = await self._extract_entities_and_relationships(document_content, filename)
            
            # 임베딩 생성
            embeddings = await self._generate_embeddings(document_content)
            
            # 메타데이터 생성
            metadata = {
                "file_path": file_path,
                "filename": filename,
                "document_length": len(document_content),
                "processed_at": datetime.now().isoformat(),
                "user_id": user_id,
                "extraction_method": "llm_based"
            }
            
            # 데이터베이스에 저장
            await self._save_graph_data(entities_data, relationships_data, metadata, user_id)
            
            logger.info(f"✅ GraphRAG 문서 처리 완료: 엔티티 {len(entities_data)}개, 관계 {len(relationships_data)}개")
            
            return {
                "status": "success",
                "entities": entities_data,
                "relationships": relationships_data,
                "embeddings": embeddings,
                "metadata": metadata
            }
            
        except Exception as e:
            logger.info(f"GraphRAG 문서 처리 실패: {str(e)}", error=True)
            return {
                "status": "error",
                "message": str(e),
                "entities": [],
                "relationships": [],
                "embeddings": [],
                "metadata": {}
            }

    async def _extract_entities_and_relationships(self, document_content: str, filename: str):
        """LLM을 사용하여 문서에서 엔티티와 관계를 추출"""
        logger = get_logger()
        logger.info(f"엔티티 및 관계 추출 시작: {filename}")
        
        try:
            # Ollama API를 사용하여 엔티티와 관계 추출
            import aiohttp
            import json
            
            # 프롬프트 준비
            extraction_prompt = f"""다음 텍스트에서 중요한 엔티티(개체)와 그들 간의 관계를 추출해주세요.

**중요한 규칙: 문서의 언어에 맞춰 모든 내용을 추출하세요**
- 한국어 문서라면 → 엔티티명, 관계 유형, 설명 모두 한국어로
  예: 엔티티 "테슬라", 관계 "대표", 설명 "테슬라의 최고경영자"
- 영어 문서라면 → 엔티티명, 관계 유형, 설명 모두 영어로  
  예: 엔티티 "Tesla", 관계 "CEO_OF", 설명 "Chief Executive Officer of Tesla"
- 혼합 문서라면 → 주로 사용된 언어에 맞춰 추출

**엔티티 추출 가이드:**
- 인물: 문서에서 언급된 대로 추출 (한국어 문서면 "일론 머스크", 영어 문서면 "Elon Musk")
- 조직: 문서에서 사용된 명칭 그대로 (한국어 문서면 "테슬라", 영어 문서면 "Tesla")
- 개념/기술: 문서 언어에 맞게 (한국어 문서면 "자율주행", 영어 문서면 "Autonomous Driving")

**관계 추출 가이드:**
- 관계 유형도 문서 언어에 맞게 (한국어: "대표", "소속", "개발" / 영어: "CEO_OF", "WORKS_FOR", "DEVELOPS")
- 관계 설명도 문서 언어로 작성

텍스트:
{document_content[:2000]}

응답은 반드시 다음 JSON 형식으로만 답변해주세요:
{{
    "entities": [
        {{
            "name": "문서 언어에 맞는 엔티티명",
            "type": "PERSON|ORGANIZATION|LOCATION|CONCEPT|TECHNOLOGY|OTHER",
            "description": "문서 언어에 맞는 설명"
        }}
    ],
    "relationships": [
        {{
            "source": "출발 엔티티명",
            "target": "도착 엔티티명", 
            "relationship": "문서 언어에 맞는 관계 유형",
            "description": "문서 언어에 맞는 관계 설명"
        }}
    ]
}}

중요: 반드시 위 JSON 형식만 반환하고 다른 텍스트는 포함하지 마세요."""

            # Ollama API 호출
            async with aiohttp.ClientSession() as session:
                ollama_url = f"{self.ollama_server}/api/generate"
                payload = {
                    "model": self.chat_model,
                    "prompt": extraction_prompt,
                    "stream": False,
                    "options": {
                        "temperature": 0.1,  # 낮은 온도로 일관된 JSON 생성
                        "top_p": 0.9
                    }
                }
                
                async with session.post(ollama_url, json=payload) as response:
                    if response.status == 200:
                        result = await response.json()
                        response_text = result.get('response', '').strip()
                        
                        logger.info(f"LLM 응답 (첫 200자): {response_text[:200]}")
                        
                        # JSON 파싱 시도
                        try:
                            # JSON 부분만 추출 (혹시 다른 텍스트가 포함된 경우)
                            start_idx = response_text.find('{')
                            end_idx = response_text.rfind('}') + 1
                            
                            if start_idx != -1 and end_idx > start_idx:
                                json_text = response_text[start_idx:end_idx]
                                extracted_data = json.loads(json_text)
                                
                                entities = extracted_data.get('entities', [])
                                relationships = extracted_data.get('relationships', [])
                                
                                # 데이터 검증 및 정제 (언어별 단순화)
                                validated_entities = []
                                for entity in entities:
                                    if isinstance(entity, dict) and 'name' in entity:
                                        validated_entities.append({
                                            'name': str(entity.get('name', '')).strip(),
                                            'type': str(entity.get('type', 'OTHER')).strip(),
                                            'description': str(entity.get('description', '')).strip(),
                                            'source_documents': [filename]
                                        })
                                
                                validated_relationships = []
                                for rel in relationships:
                                    if isinstance(rel, dict) and 'source' in rel and 'target' in rel:
                                        validated_relationships.append({
                                            'source': str(rel.get('source', '')).strip(),
                                            'target': str(rel.get('target', '')).strip(),
                                            'relationship': str(rel.get('relationship', '')).strip(),
                                            'description': str(rel.get('description', '')).strip(),
                                            'weight': 1.0,
                                            'source_documents': [filename]
                                        })
                                
                                logger.info(f"추출된 엔티티: {len(validated_entities)}개, 관계: {len(validated_relationships)}개")
                                return validated_entities, validated_relationships
                                
                        except json.JSONDecodeError as e:
                            logger.info(f"JSON 파싱 실패: {str(e)}, 응답: {response_text[:500]}", error=True)
                    else:
                        logger.error(f"Ollama API 호출 실패: {response.status}")
            
            # 폴백: 기본 엔티티와 관계 생성
            logger.info("LLM 추출 실패, 기본 엔티티 생성")
            return self._create_fallback_entities(document_content, filename)
            
        except Exception as e:
            logger.info(f"엔티티 추출 중 오류: {str(e)}", error=True)
            return self._create_fallback_entities(document_content, filename)

    def _create_fallback_entities(self, document_content: str, filename: str):
        """LLM 추출 실패시 기본 엔티티와 관계 생성"""
        logger = get_logger()
        logger.info("기본 엔티티 생성 중...")
        
        # 간단한 키워드 기반 엔티티 추출
        entities = []
        relationships = []
        
        # 문서 자체를 하나의 엔티티로 추가
        entities.append({
            'name': filename,
            'type': 'DOCUMENT',
            'description': f'문서: {filename}',
            'source_documents': [filename]
        })
        
        # 간단한 키워드 추출 (공백 기준으로 단어 분할하여 빈도수 기반)
        words = document_content.split()
        word_freq = {}
        for word in words:
            word = word.strip('.,!?;:"()[]{}')
            if len(word) > 2:  # 2글자 이상 단어만
                word_freq[word] = word_freq.get(word, 0) + 1
        
        # 빈도수 상위 키워드를 엔티티로 추가
        top_keywords = sorted(word_freq.items(), key=lambda x: x[1], reverse=True)[:10]
        
        for keyword, freq in top_keywords:
            if freq > 1:  # 1번 이상 나온 키워드만
                entities.append({
                    'name': keyword,
                    'type': 'CONCEPT',
                    'description': f'키워드: {keyword} (빈도: {freq})',
                    'source_documents': [filename]
                })
                
                # 문서와 키워드 간의 관계 추가
                relationships.append({
                    'source': filename,
                    'target': keyword,
                    'relationship': 'CONTAINS',
                    'description': f'{filename} 문서에 {keyword} 키워드가 포함됨',
                    'weight': min(freq / 10.0, 1.0),  # 가중치 정규화
                    'source_documents': [filename]
                })
        
        logger.info(f"기본 엔티티 생성 완료: {len(entities)}개 엔티티, {len(relationships)}개 관계")
        return entities, relationships

    async def _generate_embeddings(self, document_content: str):
        """문서 내용의 임베딩 생성"""
        logger = get_logger()
        logger.info("임베딩 생성 시작...")
        
        try:
            # 문서를 청크로 분할 (너무 긴 텍스트 방지)
            chunk_size = 500
            chunks = [document_content[i:i+chunk_size] for i in range(0, len(document_content), chunk_size)]
            
            # 각 청크의 임베딩 생성 (간단한 구현 - 실제로는 sentence-transformers 사용)
            embeddings = []
            
            for chunk in chunks[:5]:  # 최대 5개 청크만 처리
                # 임베딩 생성 (여기서는 간단한 해시 기반 벡터로 대체)
                import hashlib
                hash_value = hashlib.md5(chunk.encode()).hexdigest()
                # 해시를 벡터로 변환 (실제로는 적절한 임베딩 모델 사용)
                vector = [float(int(hash_value[i:i+2], 16)) / 255.0 for i in range(0, min(32, len(hash_value)), 2)]
                # 벡터 크기를 384차원으로 맞춤 (실제 임베딩 모델 차원)
                while len(vector) < 384:
                    vector.extend(vector[:min(384-len(vector), len(vector))])
                vector = vector[:384]
                embeddings.append(vector)
            
            logger.info(f"임베딩 생성 완료: {len(embeddings)}개 벡터")
            return embeddings
            
        except Exception as e:
            logger.info(f"임베딩 생성 실패: {str(e)}", error=True)
            # 기본 임베딩 반환
            return [[0.0] * 384]  # 384차원 제로 벡터

    async def _save_graph_data(self, entities_data, relationships_data, metadata, user_id):
        """그래프 데이터를 데이터베이스에 저장"""
        logger = get_logger()
        logger.info(f"그래프 데이터 저장 시작: 엔티티 {len(entities_data)}개, 관계 {len(relationships_data)}개")
        
        try:
            conn = get_pool_connection()
            try:
                with conn.cursor() as cur:
                    entity_id_map = {}
                    
                    # 엔티티 저장
                    for entity in entities_data:
                        try:
                            # 중복 엔티티 확인
                            cur.execute(
                                "SELECT id FROM graph_entities WHERE name = %s",
                                (entity['name'],)
                            )
                            existing = cur.fetchone()
                            
                            if existing:
                                entity_id = existing[0]
                                # 기존 엔티티 업데이트
                                # 기존 엔티티 업데이트 (단순화)
                                cur.execute("""
                                    UPDATE graph_entities 
                                    SET description = %s, 
                                        source_documents = array_cat(source_documents, %s),
                                        updated_at = CURRENT_TIMESTAMP
                                    WHERE id = %s
                                """, (entity['description'], entity['source_documents'], entity_id))
                            else:
                                # 새 엔티티 삽입 (단순화)
                                cur.execute("""
                                    INSERT INTO graph_entities (name, type, description, source_documents, created_at, updated_at)
                                    VALUES (%s, %s, %s, %s, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
                                    RETURNING id
                                """, (entity['name'], entity['type'], entity['description'], entity['source_documents']))
                                entity_id = cur.fetchone()[0]
                            
                            entity_id_map[entity['name']] = entity_id
                            
                        except Exception as e:
                            logger.info(f"엔티티 저장 실패 ({entity['name']}): {str(e)}", error=True)
                            continue
                    
                    # 관계 저장
                    for relationship in relationships_data:
                        try:
                            source_id = entity_id_map.get(relationship['source'])
                            target_id = entity_id_map.get(relationship['target'])
                            
                            if source_id and target_id:
                                # 중복 관계 확인
                                cur.execute("""
                                    SELECT id FROM graph_relationships
                                    WHERE source_entity_id = %s AND target_entity_id = %s
                                    AND relationship_type = %s
                                """, (source_id, target_id, relationship['relationship']))
                                
                                existing = cur.fetchone()
                                
                                if existing:
                                    # 기존 관계 업데이트
                                    cur.execute("""
                                        UPDATE graph_relationships 
                                        SET weight = %s, description = %s,
                                            source_documents = array_cat(source_documents, %s),
                                            updated_at = CURRENT_TIMESTAMP
                                        WHERE id = %s
                                    """, (relationship['weight'], relationship['description'], 
                                         relationship['source_documents'], existing[0]))
                                else:
                                    # 새 관계 삽입
                                    cur.execute("""
                                        INSERT INTO graph_relationships
                                        (source_entity_id, target_entity_id, relationship_type, weight, description, source_documents, created_at, updated_at)
                                        VALUES (%s, %s, %s, %s, %s, %s, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
                                    """, (source_id, target_id, relationship['relationship'], relationship['weight'],
                                         relationship['description'], relationship['source_documents']))
                            
                        except Exception as e:
                            logger.info(f"관계 저장 실패: {str(e)}", error=True)
                            continue
                    
                    conn.commit()
                    logger.info("✅ 그래프 데이터 저장 완료")
                    
            finally:
                conn.close()
                
        except Exception as e:
            logger.info(f"그래프 데이터 저장 실패: {str(e)}", error=True)
            raise e

    def process_document(self, data):
        """문서 처리 메서드"""
        return {"status": "success", "message": "Document processing not implemented"}

    def search(self, query: str, top_k: int = 5):
        """검색 메서드"""
        return {"results": [], "total": 0}

# 전역 모델 매니저 클래스
class GlobalModelManager:
    """서버 시작 시 모델들을 사전 로딩하고 워커들이 참조할 수 있도록 관리하는 클래스"""

    _instance = None
    _lock = Lock()
    _models = {}
    _model_ready_events = {}
    _redis_client = None

    def __new__(cls):
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    cls._instance = super().__new__(cls)
        return cls._instance

    def __init__(self):
        if not hasattr(self, '_initialized'):
            self._initialized = True
            self._models = {
                'text_embedding': None,      # nlpai-lab/KURE-v1
                'sentence_transformer': None, # snunlp/KR-SBERT-V40K-klueNLI-augSTS
                'image_embedding': None,     # Bingsu/clip-vit-base-patch32-ko
            }
            self._model_ready_events = {
                'text_embedding': threading.Event(),
                'sentence_transformer': threading.Event(),
                'image_embedding': threading.Event(),
            }
            # Redis 클라이언트 초기화
            try:
                import redis
                redis_host = os.getenv('REDIS_HOST', 'localhost')
                redis_port = int(os.getenv('REDIS_PORT', '6379'))
                self._redis_client = redis.Redis(host=redis_host, port=redis_port, decode_responses=True)
                self._redis_client.ping()
            except:
                self._redis_client = None

    def preload_models(self):
        """서버 시작 시 필요한 모든 모델을 사전 로딩"""
        logger = get_logger()
        logger.info("-" * 60)
        logger.info("🚀 전역 모델 사전 로딩 시작")
        logger.info("-" * 60)

        # RAG 설정 가져오기
        from rag_process import get_rag_settings
        rag_settings = get_rag_settings()

        # GPU 사용 가능 여부 확인
        device = 'cuda' if torch.cuda.is_available() else 'cpu'
        logger.info(f"🔧 모델 로딩 디바이스: {device}")

        # 1. 텍스트 임베딩 모델 로딩 (nlpai-lab/KURE-v1)
        try:
            text_model_name = rag_settings.get('embedding_model', 'nlpai-lab/KURE-v1')
            logger.info(f"📝 텍스트 임베딩 모델 로딩 시작: {text_model_name}")
            from sentence_transformers import SentenceTransformer
            self._models['text_embedding'] = SentenceTransformer(text_model_name, device=device)
            self._model_ready_events['text_embedding'].set()
            if self._redis_client:
                self._redis_client.set('model:text_embedding:ready', '1')
                self._redis_client.set('model:text_embedding:name', text_model_name)
            logger.info(f"✅ 텍스트 임베딩 모델 로딩 완료: {text_model_name}")
        except Exception as e:
            logger.error(f"❌ 텍스트 임베딩 모델 로딩 실패: {e}")

        # 2. 문장 시맨틱 모델 로딩 (snunlp/KR-SBERT-V40K-klueNLI-augSTS)
        try:
            semantic_model_name = rag_settings.get('sentence_transformer_model', 'snunlp/KR-SBERT-V40K-klueNLI-augSTS')
            logger.info(f"🧠 문장 시맨틱 모델 로딩 시작: {semantic_model_name}")
            from sentence_transformers import SentenceTransformer
            self._models['sentence_transformer'] = SentenceTransformer(semantic_model_name, device=device)
            self._model_ready_events['sentence_transformer'].set()
            if self._redis_client:
                self._redis_client.set('model:sentence_transformer:ready', '1')
                self._redis_client.set('model:sentence_transformer:name', semantic_model_name)
            logger.info(f"✅ 문장 시맨틱 모델 로딩 완료: {semantic_model_name}")
        except Exception as e:
            logger.error(f"❌ 문장 시맨틱 모델 로딩 실패: {e}")

        # 3. 이미지 임베딩 모델 로딩 (Bingsu/clip-vit-base-patch32-ko)
        try:
            image_model_name = rag_settings.get('image_embedding_model', 'Bingsu/clip-vit-base-patch32-ko')
            logger.info(f"🖼️ 이미지 임베딩 모델 로딩 시작: {image_model_name}")
            from transformers import CLIPModel, CLIPProcessor
            self._models['image_embedding'] = {
                'model': CLIPModel.from_pretrained(image_model_name).to(device),
                'processor': CLIPProcessor.from_pretrained(image_model_name)
            }
            self._model_ready_events['image_embedding'].set()
            if self._redis_client:
                self._redis_client.set('model:image_embedding:ready', '1')
                self._redis_client.set('model:image_embedding:name', image_model_name)
            logger.info(f"✅ 이미지 임베딩 모델 로딩 완료: {image_model_name}")
        except Exception as e:
            logger.error(f"❌ 이미지 임베딩 모델 로딩 실패: {e}")

        logger.info("-" * 60)
        logger.info("🎉 전역 모델 사전 로딩 완료")
        logger.info("-" * 60)

    def get_model(self, model_type: str, timeout: int = 30):
        """로딩된 모델을 가져옴 (타임아웃 포함)"""
        if model_type not in self._model_ready_events:
            return None

        # 모델 로딩 완료까지 대기
        if not self._model_ready_events[model_type].wait(timeout=timeout):
            return None  # 타임아웃

        return self._models.get(model_type)

    def is_model_ready(self, model_type: str) -> bool:
        """모델이 로딩 완료되었는지 확인"""
        if model_type not in self._model_ready_events:
            return False
        return self._model_ready_events[model_type].is_set()

    def get_model_status(self) -> dict:
        """모든 모델의 로딩 상태 반환"""
        status = {}
        for model_type, event in self._model_ready_events.items():
            status[model_type] = {
                'ready': event.is_set(),
                'loaded': self._models[model_type] is not None
            }
        return status

# 전역 모델 매니저 인스턴스
global_model_manager = GlobalModelManager()

def get_or_create_graphrag_processor():
    """GraphRAG 프로세서 싱글톤 인스턴스 가져오기"""
    global graphrag_processor
    if graphrag_processor is None:
        try:
            # GraphRAGProcessor 클래스가 파일 끝에 정의되어 있음
            graphrag_processor = GraphRAGProcessor()
            logger = get_logger()
            logger.info("[GraphRAG] 프로세서 싱글톤 인스턴스 생성 완료")
        except Exception as e:
            logger = get_logger()
            logger.info(f"❌ [GraphRAG] 프로세서 생성 실패: {str(e)}", error=True)
            return None
    # 이미 생성된 인스턴스 재사용 (로그 생략으로 스팸 방지)
    return graphrag_processor

# 파인튜닝 상태 저장을 위한 클래스
class FineTuningStateManager:
    def __init__(self, state_file: str = None):
        self.state_file = state_file or os.path.expanduser("~/.airun/finetune_state.json")
        self.state_dir = os.path.dirname(self.state_file)
        os.makedirs(self.state_dir, exist_ok=True)
        
    def save_state(self, state: Dict[str, Any]):
        """파인튜닝 상태를 파일에 저장"""
        state['timestamp'] = time.time()
        state['last_updated'] = datetime.now().isoformat()
        
        try:
            with open(self.state_file, 'w', encoding='utf-8') as f:
                json.dump(state, f, indent=2, ensure_ascii=False)
        except Exception as e:
            print(f"상태 저장 실패: {e}")
            
    def load_state(self) -> Dict[str, Any]:
        """파인튜닝 상태를 파일에서 로드"""
        if not os.path.exists(self.state_file):
            return {}
            
        try:
            with open(self.state_file, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception as e:
            print(f"상태 로드 실패: {e}")
            return {}
            
    def update_state(self, **kwargs):
        """기존 상태를 업데이트"""
        current_state = self.load_state()
        current_state.update(kwargs)
        self.save_state(current_state)
        
    def clear_state(self):
        """상태 파일 삭제"""
        if os.path.exists(self.state_file):
            try:
                os.remove(self.state_file)
            except Exception as e:
                print(f"상태 파일 삭제 실패: {e}")
                
    def get_last_error(self) -> Optional[str]:
        """마지막 오류 정보 반환"""
        state = self.load_state()
        return state.get('last_error')
        
    def is_resource_failure(self) -> bool:
        """시스템 자원 부족으로 인한 실패인지 확인"""
        state = self.load_state()
        error = state.get('last_error', '')
        failure_reason = state.get('failure_reason', '')
        
        resource_keywords = [
            'out of memory', 'oom', 'killed', 'memory error',
            'cuda out of memory', 'insufficient memory',
            'resource exhausted', 'system resource',
            'memory', 'cuda', 'gpu', 'resource', 'allocation'
        ]
        
        combined_text = f"{error} {failure_reason}".lower()
        return any(keyword in combined_text for keyword in resource_keywords)
        
    def get_failure_reason(self) -> str:
        """실패 원인 분석"""
        state = self.load_state()
        if not state.get('last_error'):
            return "알 수 없는 오류"
            
        error = state.get('last_error', '').lower()
        
        if self.is_resource_failure():
            return "시스템 자원 부족"
        elif 'connection' in error or 'timeout' in error:
            return "네트워크 연결 문제"
        elif 'permission' in error or 'access denied' in error:
            return "권한 문제"
        elif 'disk' in error or 'space' in error:
            return "디스크 공간 부족"
        else:
            return "처리 중 오류"

# 전역 상태 관리자
finetune_state_manager = FineTuningStateManager()

# Create global logger instance
logger = None

def get_logger():
    """전역 로거 인스턴스를 반환합니다."""
    global logger
    if logger is None:
        try:
            # 로그 디렉토리 생성
            log_dir = os.path.dirname(LOG_FILE)
            os.makedirs(log_dir, exist_ok=True)
            # Logger 인스턴스 생성
            logger = Logger()
            logger.log_file = LOG_FILE
        except Exception as e:
            print(f"로거 초기화 실패: {str(e)}")
            sys.exit(1)
    return logger

# 로거 즉시 초기화
logger = get_logger()

# 데이터베이스 연결 풀 가져오기
# get_db_pool 함수 제거됨 - 중앙 집중식 풀 사용

def get_redis_client():
    """Redis 클라이언트를 가져오거나 재연결합니다."""
    global redis_client
    try:
        if redis_client is None or not redis_client.ping():
            redis_client = redis.Redis(
                host=REDIS_HOST,
                port=REDIS_PORT,
                db=0,
                decode_responses=False,
                socket_timeout=5,
                socket_connect_timeout=5,
                retry_on_timeout=True
            )
            # 연결 테스트
            redis_client.ping()
        return redis_client
    except redis.ConnectionError as e:
        logger.info(f"Redis 연결 실패: {str(e)}", error=True)
        raise HTTPException(status_code=500, detail="Redis 연결에 실패했습니다.")
    except Exception as e:
        logger.info(f"Redis 클라이언트 초기화 중 오류: {str(e)}", error=True)
        raise HTTPException(status_code=500, detail="Redis 클라이언트 초기화 중 오류가 발생했습니다.")

# Redis 클라이언트 초기화
try:
    redis_client = get_redis_client()
except Exception as e:
    logger.info(f"초기 Redis 연결 실패: {str(e)}", error=True)

# 가상환경 확인 및 활성화
def check_and_activate_venv():
    """가상환경이 활성화되어 있지 않으면 활성화"""
    if not hasattr(sys, 'real_prefix') and not hasattr(sys, 'base_prefix') or sys.base_prefix == sys.prefix:
        venv_path = os.path.expanduser("~/.airun_venv")
        if os.path.exists(venv_path):
            activate_script = os.path.join(venv_path, "bin", "activate")
            if os.path.exists(activate_script):
                # 가상환경 활성화
                activate_cmd = f"source {activate_script} && exec python3 {__file__} {' '.join(sys.argv[1:])}"
                os.execv("/bin/bash", ["/bin/bash", "-c", activate_cmd])
            else:
                print(f"가상환경 활성화 스크립트를 찾을 수 없습니다: {activate_script}")
                sys.exit(1)
        else:
            print(f"가상환경을 찾을 수 없습니다: {venv_path}")
            sys.exit(1)

# 가상환경 확인 및 활성화 실행
check_and_activate_venv()

# DocumentProcessor 인스턴스 생성
document_processor = None

# 전역 변수로 필요한 함수들 설정
def extract_text(file_path):
    return document_processor.extract_text(file_path)

def log(message, error=False):
    """향상된 로깅 함수 - Logger 클래스 사용"""
    if error:
        logger.error(message)
    else:
        logger.info(message)

async def process_document_in_thread(doc_path: str):
    """문서 처리를 임베딩 워커에게 위임합니다."""
    try:
        global embedding_queue
        
        # 임베딩 워커가 사용 가능한지 확인
        if embedding_queue is None:
            # 폴백: 메인 프로세스에서 직접 처리 (임베딩 워커 사용 불가능한 경우)
            logger.info(f"임베딩 워커 사용 불가, 메인 프로세스에서 직접 처리: {os.path.basename(doc_path)}")
            loop = asyncio.get_event_loop()
            result = await loop.run_in_executor(
                thread_pool,
                partial(document_processor.process_document, doc_path)
            )
            return result
        
        # 요청 ID 생성
        request_id = f"doc_{int(time.time() * 1000)}_{os.path.basename(doc_path)}"
        
        # 사용자 ID 추출 (경로에서)
        user_id = "all"
        rag_path = get_rag_docs_path()

        try:
            doc_p = Path(doc_path).resolve()
            rag_p = Path(rag_path).resolve()

            # rag_path 하위일 때만 상대경로 계산
            rel = doc_p.relative_to(rag_p)  # rag_p 밖이면 ValueError
            # 첫 디렉터리를 user_id로 사용
            if rel.parts:
                cand = rel.parts[0]
                if cand and cand != ".":
                    user_id = cand

        except Exception:
            # 폴백: os.path.relpath (심볼릭/경계 케이스 방어)
            try:
                rel2 = os.path.relpath(doc_path, rag_path)
                if not rel2.startswith(".."):  # rag_path 내부일 때만
                    cand = rel2.split(os.sep, 1)[0]
                    if cand and cand != ".":
                        user_id = cand
            except Exception:
                pass
        
        # 임베딩 워커에게 문서 처리 요청
        embedding_request = {
            'type': 'document_processing',
            'request_id': request_id,
            'file_path': doc_path,
            'user_id': user_id
        }
        
        embedding_queue.put(embedding_request)
        logger.info(f"문서 처리를 임베딩 워커에게 위임: {os.path.basename(doc_path)}, ID: {request_id}")
        
        # 임베딩 워커가 백그라운드에서 처리하므로 성공으로 간주
        return {"status": "delegated", "message": "문서 처리가 임베딩 워커에게 위임되었습니다", "request_id": request_id}
        
    except Exception as e:
        logger.error(f"임베딩 워커 위임 실패: {str(e)}")
        # 폴백: 메인 프로세스에서 직접 처리
        try:
            logger.info(f"폴백: 메인 프로세스에서 직접 처리: {os.path.basename(doc_path)}")
            loop = asyncio.get_event_loop()
            result = await loop.run_in_executor(
                thread_pool,
                partial(document_processor.process_document, doc_path)
            )
            return result
        except Exception as fallback_error:
            return {"status": "error", "message": f"워커 위임 실패 및 직접 처리 실패: {str(fallback_error)}"}

async def process_document_background(
    target_file: str, 
    effective_user_id: str, 
    rag_docs_path: str, 
    file_processing_key: str = None,
    overwrite: bool = False,
    original_filename: str = None
):
    """백그라운드에서 모든 문서 처리를 수행합니다 (중복 검사, 벡터 삭제, PDF 파싱, 임베딩)."""
    try:
        filename = original_filename or os.path.basename(target_file)
        logger.info(f"백그라운드 전체 문서 처리 시작: {filename} (사용자: {effective_user_id})")
        
        # 백그라운드에서 캐시 초기화 수행
        try:
            await clear_cache()
            logger.info("백그라운드에서 캐시 초기화 완료")
        except Exception as e:
            logger.error(f"백그라운드 캐시 초기화 실패: {str(e)}")
            # 캐시 초기화 실패해도 문서 처리는 계속 진행
        
        # Redis에 진행상태 업데이트
        try:
            redis_client = get_redis_client()
            if redis_client:
                progress_key = "rag:init:progress"
                redis_client.hset(progress_key, filename, "processing")
                logger.info(f"Redis 진행상태 등록: {filename} -> processing")
        except Exception as e:
            logger.error(f"Redis 진행상태 업데이트 실패: {str(e)}")
        
        # rag:status:* 키 정리 (이전 상태 정보가 있다면 제거)
        try:
            redis_client = get_redis_client()
            if redis_client:
                status_key = f"rag:status:{filename}"
                if redis_client.exists(status_key):
                    redis_client.delete(status_key)
                    logger.info(f"이전 상태 정보 정리: {filename}")
        except Exception as status_cleanup_error:
            logger.error(f"이전 상태 정보 정리 실패: {str(status_cleanup_error)}")
        
        # 1. 중복 파일 처리 (백그라운드에서 수행)
        if overwrite and os.path.exists(target_file):
            logger.info(f"백그라운드에서 기존 파일 벡터 데이터 삭제 시작: {filename}")
            try:
                # PostgreSQL에서 기존 문서 데이터 삭제
                import psycopg2
                from psycopg2.extras import RealDictCursor
                
                # PostgreSQL 연결
                try:
                    conn = get_pool_connection()
                    cursor = conn.cursor(cursor_factory=RealDictCursor)
                    
                    # 기존 문서 삭제 (파일명 기준)
                    delete_query = "DELETE FROM rag_documents WHERE filename = %s"
                    if effective_user_id != 'all':
                        delete_query += " AND user_id = %s"
                        cursor.execute(delete_query, (filename, effective_user_id))
                    else:
                        cursor.execute(delete_query, (filename,))
                    
                    deleted_count = cursor.rowcount
                    conn.commit()
                    cursor.close()
                    return_pool_connection(conn)
                    
                    if deleted_count > 0:
                        logger.info(f"PostgreSQL에서 기존 문서 삭제: {filename} ({deleted_count}개 청크)")
                
                except Exception as pg_error:
                    logger.error(f"PostgreSQL 기존 문서 삭제 실패: {str(pg_error)}")
                
                # PostgreSQL에서 기존 문서 벡터 삭제
                if document_processor and document_processor.collection:
                    where_filter = {}
                    if effective_user_id != 'all':
                        where_filter["user_id"] = effective_user_id
                    
                    chunk_ids_to_delete = []
                    # 다양한 방법으로 기존 청크 검색 및 삭제
                    try:
                        # 방법 1: 전체 경로로 검색
                        file_filter = {**where_filter, "source": target_file}
                        chunk_results = document_processor.collection.get(where=file_filter)
                        if chunk_results['ids']:
                            chunk_ids_to_delete.extend(chunk_results['ids'])
                        
                        # 방법 2: 파일명만으로 검색
                        file_filter = {**where_filter, "source": filename}
                        chunk_results = document_processor.collection.get(where=file_filter)
                        if chunk_results['ids']:
                            new_ids = [id for id in chunk_results['ids'] if id not in chunk_ids_to_delete]
                            chunk_ids_to_delete.extend(new_ids)
                        
                        # 청크 삭제 실행
                        if chunk_ids_to_delete:
                            document_processor.collection.delete(ids=chunk_ids_to_delete)
                            logger.info(f"백그라운드에서 기존 벡터 데이터 삭제 완료: {filename} ({len(chunk_ids_to_delete)}개 청크)")
                        else:
                            logger.info(f"삭제할 기존 벡터 데이터를 찾을 수 없음: {filename}")
                    except Exception as delete_error:
                        logger.error(f"백그라운드 벡터 삭제 중 오류: {str(delete_error)}")
                        # 벡터 삭제 실패해도 새로운 처리는 계속 진행
            except Exception as e:
                logger.error(f"백그라운드 기존 벡터 데이터 삭제 실패: {str(e)}")
                # 실패해도 새로운 문서 처리는 계속 진행
        
        # 2. 문서 처리 (PDF 파싱 + 임베딩)
        logger.info(f"백그라운드에서 문서 파싱 및 임베딩 시작: {filename}")
        results = await document_processor.add_documents(
            [target_file],
            effective_user_id if effective_user_id != 'all' else '',
            rag_docs_path,
            None  # 백그라운드 파일 처리에서는 기본 설정 사용
        )
        
        # 처리 성공 시 Redis 키 삭제 (파일 모니터링이 다시 처리 가능)
        try:
            if file_processing_key:
                redis_client = get_redis_client()
                if redis_client:
                    redis_client.delete(file_processing_key)
                    logger.info(f"백그라운드 처리 완료, 키 삭제: {filename}")
        except Exception as e:
            logger.error(f"처리 완료 키 삭제 실패, 무시하고 계속: {str(e)}")
        
        logger.info(f"백그라운드 전체 문서 처리 완료: {filename}")
        
        # 처리 결과 분석 및 로그
        if isinstance(results, dict):
            status = results.get('status', 'success' if results else 'unknown')
            message = results.get('message', '처리 완료')
            logger.info(f"처리 결과: {status} - {message}")
        else:
            logger.info(f"처리 결과: 성공 (반환값 타입: {type(results).__name__})")
        
        # Redis에 완료 상태 업데이트
        try:
            redis_client = get_redis_client()
            if redis_client:
                progress_key = "rag:init:progress"
                redis_client.hset(progress_key, filename, "completed")
                logger.info(f"Redis 진행상태 업데이트: {filename} -> completed")
        except Exception as e:
            logger.error(f"Redis 완료상태 업데이트 실패: {str(e)}")
        
        # rag:status:* 키 정리 (완료된 문서의 상태 정보 제거)
        try:
            redis_client = get_redis_client()
            if redis_client:
                status_key = f"rag:status:{filename}"
                if redis_client.exists(status_key):
                    redis_client.delete(status_key)
                    logger.info(f"완료된 문서의 상태 정보 정리: {filename}")
        except Exception as status_cleanup_error:
            logger.error(f"상태 정보 정리 실패: {str(status_cleanup_error)}")
        
        return results
        
    except Exception as e:
        filename = original_filename or os.path.basename(target_file)
        logger.error(f"백그라운드 전체 문서 처리 실패: {filename} - {str(e)}")
        
        # Redis에 실패 상태 업데이트
        try:
            redis_client = get_redis_client()
            if redis_client:
                progress_key = "rag:init:progress"
                redis_client.hset(progress_key, filename, "failed")
                logger.info(f"Redis 진행상태 업데이트: {filename} -> failed")
        except Exception as redis_error:
            logger.error(f"Redis 실패상태 업데이트 실패: {str(redis_error)}")
        
        # 처리 실패 시에도 Redis 키 정리
        try:
            if file_processing_key:
                redis_client = get_redis_client()
                if redis_client:
                    redis_client.delete(file_processing_key)
                    logger.info(f"백그라운드 처리 실패, 키 삭제: {filename}")
        except Exception as cleanup_error:
            logger.error(f"실패 후 키 삭제 실패: {str(cleanup_error)}")
        
        # rag:status:* 키도 정리 (실패한 문서의 상태 정보 제거)
        try:
            redis_client = get_redis_client()
            if redis_client:
                status_key = f"rag:status:{filename}"
                if redis_client.exists(status_key):
                    redis_client.delete(status_key)
                    logger.info(f"실패한 문서의 상태 정보 정리: {filename}")
        except Exception as status_cleanup_error:
            logger.error(f"상태 정보 정리 실패: {str(status_cleanup_error)}")
        
        return {"status": "error", "message": str(e)}

class QueueManager:
    def __init__(self, redis_client):
        self.redis = redis_client
        self.logger = get_logger()
        
    async def get_next_batch(self, batch_size: int) -> List[str]:
        """다음 배치의 문서들을 가져옴"""
        try:
            # lrange로 리스트에서 문서들을 가져옴
            docs = self.redis.lrange(QUEUE_KEYS['documents'], 0, -1)
            total_docs = len(docs)
            
            self.logger.info(f"큐에서 문서 조회 - 전체: {total_docs}개, 요청 배치 크기: {batch_size}")
            
            if total_docs > 0:
                doc_names = [os.path.basename(decode_redis_value(doc)) for doc in docs]
                self.logger.info(f"큐의 문서 목록: {doc_names}")
            
            batch_docs = [decode_redis_value(doc) for doc in docs[:batch_size]]
            
            if batch_docs:
                batch_names = [os.path.basename(doc) for doc in batch_docs]
                self.logger.info(f"배치로 반환할 문서: {batch_names}")
            else:
                self.logger.info("배치로 반환할 문서가 없습니다.")
                
            return batch_docs
        except Exception as e:
            self.logger.error(f"배치 문서 조회 실패: {str(e)}")
            return []
            
    async def move_to_processing(self, doc_path: str) -> bool:
        """문서를 processing 큐로 이동"""
        try:
            # list 기반으로 문서 이동: documents에서 제거 후 processing에 추가
            removed_count = self.redis.lrem(QUEUE_KEYS['documents'], 1, doc_path.encode())
            if removed_count > 0:
                self.redis.lpush(QUEUE_KEYS['processing'], doc_path.encode())
                return True
            return False
        except Exception as e:
            self.logger.error(f"Processing 큐 이동 실패: {os.path.basename(doc_path)} - {str(e)}")
            return False
            
    async def mark_as_completed(self, doc_path: str) -> bool:
        """문서를 completed 큐로 이동"""
        try:
            # List에서 요소를 제거하고 다른 list에 추가
            doc_key = doc_path.encode()
            # processing에서 제거
            if self.redis.lrem(QUEUE_KEYS['processing'], 1, doc_key) > 0:
                # completed에 추가
                self.redis.lpush(QUEUE_KEYS['completed'], doc_key)
                return True
            return False
        except Exception as e:
            self.logger.error(f"Completed 큐 이동 실패: {doc_path} - {str(e)}")
            return False
            
    async def mark_as_failed(self, doc_path: str, error: str) -> bool:
        """문서를 failed 큐로 이동하고 오류 기록"""
        try:
            doc_key = doc_path.encode()
            # processing에서 제거하고 failed에 추가
            if self.redis.lrem(QUEUE_KEYS['processing'], 1, doc_key) > 0:
                self.redis.lpush(QUEUE_KEYS['failed'], doc_key)
            self.redis.hset(f"{QUEUE_KEYS['failed']}:errors", doc_key, error.encode())
            return True
        except Exception as e:
            self.logger.error(f"Failed 큐 이동 실패: {doc_path} - {str(e)}")
            return False
            
    async def update_batch_status(self) -> None:
        """배치 상태 업데이트"""
        try:
            self.redis.incr(QUEUE_KEYS['current_batch'])
        except Exception as e:
            self.logger.error(f"배치 상태 업데이트 실패: {str(e)}")

    async def is_batch_complete(self) -> bool:
        """현재 배치의 모든 문서가 처리되었는지 확인"""
        try:
            processing_count = len(self.redis.lrange(QUEUE_KEYS['processing'], 0, -1))
            return processing_count == 0
        except Exception as e:
            self.logger.error(f"배치 완료 확인 실패: {str(e)}")
            return False

    async def validate_queue_state(self) -> None:
        """큐 상태의 일관성을 검증"""
        try:
            # processing 큐의 문서들이 실제로 존재하는지 확인
            processing_docs = self.redis.lrange(QUEUE_KEYS['processing'], 0, -1)
            for doc in processing_docs:
                doc_path = decode_redis_value(doc)
                if not os.path.exists(doc_path):
                    await self.mark_as_failed(doc_path, "File not found")
        except Exception as e:
            self.logger.error(f"큐 상태 검증 중 오류 발생: {str(e)}")

async def process_next_batch():
    """큐 처리 로직 - 문서를 하나씩 즉시 처리"""
    try:
        redis_client = get_redis_client()
        queue_manager = QueueManager(redis_client)
        
        # 현재 배치 번호 조회
        current_batch = int(decode_redis_value(redis_client.get(QUEUE_KEYS['current_batch']) or b'0'))
        batch_size = int(decode_redis_value(redis_client.get(QUEUE_KEYS['batch_size']) or str(BATCH_SIZE).encode()))
        
        # 큐 상태 확인
        queue_status = await get_queue_status()
        documents_remaining = queue_status["queue_status"]["remaining"]
        processing_count = queue_status["queue_status"]["processing"]
        
        logger.info(f"배치 처리 시작 - remaining: {documents_remaining}, processing: {processing_count}")
        
        # 처리할 문서가 없으면 종료 조건 확인
        if documents_remaining == 0:
            logger.info("처리할 문서가 없습니다.")
            
            if processing_count == 0:
                # 종료 시간이 아직 저장되지 않았을 때만 저장
                if not redis_client.get("rag:init:end_time"):
                    end_time = datetime.now()
                    redis_client.set("rag:init:end_time", end_time.isoformat().encode())
                    logger.info(f"RAG 초기화 종료 시간 저장: {end_time.isoformat()}")
                
                # RAG 초기화 진행 중 플래그 제거
                redis_client.delete("rag:init:in_progress")
                logger.info("RAG 초기화 진행 중 플래그 제거 완료")
                
                logger.info("모든 문서 처리가 완료되었습니다.")
                return
            else:
                # 처리 중인 문서가 있으면 잠시 대기 후 재시도
                logger.info(f"처리 중인 문서가 {processing_count}개 있습니다. 2초 후 재시도합니다.")
                await asyncio.sleep(2)
                asyncio.create_task(process_next_batch())
                return
        
        # 문서를 하나씩 순차 처리
        processed_count = 0
        failed_count = 0
        max_docs_per_batch = min(batch_size, documents_remaining)
        
        logger.info(f"배치 {current_batch + 1} 처리 시작: 최대 {max_docs_per_batch}개 문서 처리 예정")
        
        doc_index = 0
        while doc_index < max_docs_per_batch:
            try:
                # 매번 큐에서 하나의 문서를 즉시 가져와서 처리 (list 타입용)
                docs = redis_client.lrange(QUEUE_KEYS['documents'], 0, -1)
                
                logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 현재 큐 상태 - documents: {len(docs)}개")
                if docs:
                    logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 큐의 문서들: {[os.path.basename(decode_redis_value(doc)) for doc in docs[:3]]}{'...' if len(docs) > 3 else ''}")
                
                if not docs:
                    logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 큐에 더 이상 문서가 없습니다.")
                    break
                
                # 첫 번째 문서를 선택하고 즉시 processing 큐로 이동
                doc_key = docs[0]
                doc_path = decode_redis_value(doc_key)
                
                logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 문서 처리 시작: {os.path.basename(doc_path)}")
                
                # 즉시 processing 큐로 이동 (list 타입용 원자적 연산)
                moved_doc = redis_client.rpoplpush(QUEUE_KEYS['documents'], QUEUE_KEYS['processing'])

                if not moved_doc:
                    logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 큐 이동 실패 (문서 없음): {os.path.basename(doc_path)}")
                    doc_index += 1
                    continue

                # 이동된 문서가 예상한 문서인지 확인
                moved_doc_path = decode_redis_value(moved_doc)
                if moved_doc_path != doc_path:
                    logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 예상과 다른 문서 이동됨: {os.path.basename(moved_doc_path)}")
                    doc_path = moved_doc_path  # 실제 이동된 문서로 업데이트
                
                logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 큐 이동 성공: {os.path.basename(doc_path)}")
                    
                # 문서 처리 시도
                try:
                    result = await process_document_in_thread(doc_path)
                    if result["status"] in ["success", "delegated"]:
                        await queue_manager.mark_as_completed(doc_path)
                        processed_count += 1
                        if result["status"] == "delegated":
                            logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 문서 처리 임베딩 워커에게 위임: {os.path.basename(doc_path)}")
                        else:
                            logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 문서 처리 성공: {os.path.basename(doc_path)}")
                    else:
                        await queue_manager.mark_as_failed(doc_path, result["message"])
                        failed_count += 1
                        logger.info(f"[{doc_index+1}/{max_docs_per_batch}] 문서 처리 실패: {os.path.basename(doc_path)} - {result['message']}")
                except Exception as e:
                    # 문서 처리 실패는 큐 처리에 영향을 주지 않음
                    await queue_manager.mark_as_failed(doc_path, str(e))
                    failed_count += 1
                    logger.error(f"[{doc_index+1}/{max_docs_per_batch}] 문서 처리 실패: {os.path.basename(doc_path)} - {str(e)}")
                    
            except Exception as e:
                # 개별 문서의 큐 처리 실패는 다른 문서 처리에 영향을 주지 않음
                logger.error(f"문서 큐 처리 중 오류: {str(e)}")
                failed_count += 1
                
            doc_index += 1
                
        # 배치 처리 완료 후 상태 업데이트
        if processed_count > 0 or failed_count > 0:
            await queue_manager.update_batch_status()
            logger.info(f"배치 {current_batch + 1} 처리 완료 - 성공: {processed_count}개, 실패: {failed_count}개")
        
        # 다음 배치 처리 시작
        asyncio.create_task(process_next_batch())
        
    except Exception as e:
        # 큐 처리 자체의 오류만 로깅
        logger.error(f"배치 처리 중 큐 오류 발생: {str(e)}")
        # 오류 발생 시 5초 후 재시도
        await asyncio.sleep(5)
        asyncio.create_task(process_next_batch())

@asynccontextmanager
async def lifespan(app: FastAPI):
    """FastAPI 애플리케이션의 수명주기를 관리합니다."""
    # Startup
    try:
        global document_processor, embedding_service, thread_pool

        # 이벤트 루프 설정
        loop = asyncio.get_event_loop()
        if loop.is_closed():
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)

        # PostgreSQL 연결 풀 초기화 제거됨 - 중앙 집중식 풀 사용

        # 워커 프로세스는 main()에서 초기화하므로 여기서는 제외

        # ThreadPoolExecutor 초기화
        initialize_thread_pool()
        
        # GraphRAG 초기화는 이미 건너뛰었음
        global graphrag_processor
        graphrag_processor = None
        
        # 큐 상태 확인 및 복구
        try:
            redis_client = get_redis_client()
            queue_status = await get_queue_status()

            # 처리 중이거나 대기 중인 문서가 있는 경우
            if queue_status["queue_status"]["processing"] > 0 or queue_status["queue_status"]["remaining"] > 0:
                logger.info("이전 작업 발견, 자동 재개 중...")
                # 큐 상태 검증
                await validate_queue_state()
                # 배치 처리 시작
                asyncio.create_task(process_next_batch())
                # logger.info("이전 작업 재개 완료")
        except Exception as e:
            logger.error(f"큐 상태 검증 및 복구 중 오류 발생: {str(e)}")
            # 오류가 발생해도 서비스는 계속 실행

        # 파인튜닝 상태 검증 및 초기화
        try:
            await validate_and_reset_training_state()
        except Exception as e:
            logger.error(f"파인튜닝 상태 검증 중 오류 발생: {str(e)}")
            # 오류가 발생해도 서비스는 계속 실행

        # 지연된 무결성 검사 시작 (DB 충돌 방지를 위해 5초 지연)
        async def delayed_integrity_check():
            try:
                await asyncio.sleep(5)  # 다른 초기화 작업 완료 대기
                logger.info("지연된 무결성 검사 시작")
                await sync_physical_files_with_db()
                logger.info("초기 무결성 검사 완료")
            except Exception as e:
                logger.error(f"지연된 무결성 검사 중 오류: {e}")

        asyncio.create_task(delayed_integrity_check())

        # 모니터링 스레드 시작
        monitoring_thread = threading.Thread(target=run_monitoring_thread, daemon=True)
        monitoring_thread.start()
        # logger.info("모니터링 스레드 시작됨")
        
    except Exception as e:
        logger.error(f"서버 시작 중 오류 발생: {str(e)}")
        raise
        
    yield

    # Shutdown
    cleanup_worker_processes()
    if thread_pool:
        thread_pool.shutdown(wait=True)

    # PostgreSQL 연결 풀 종료
    # PostgreSQL 연결 풀 정리 제거됨 - 중앙 집중식 풀 사용

# FastAPI 앱 생성
app = FastAPI(lifespan=lifespan)

# CORS 설정 추가
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],      # 모든 도메인에서의 요청 허용
    allow_credentials=True,   # 인증 정보(쿠키 등) 허용
    allow_methods=["*"],      # 모든 HTTP 메서드 허용
    allow_headers=["*"],      # 모든 헤더 허용
)

# 요청/응답 모델 정의
class EmbeddingRequest(BaseModel):
    text: str
    batch: bool = False

class BatchEmbeddingRequest(BaseModel):
    texts: List[str]

class AddDocumentsRequest(BaseModel):
    path: str
    user_id: Optional[str] = None
    base_path: Optional[str] = None

class UploadResponse(BaseModel):
    status: str
    message: str
    file: str
    user_id: Optional[str]
    rag_docs_path: str
    summary: Dict[str, Any]
class EmbeddingResponse(BaseModel):
    success: bool
    embedding: Optional[List[float]] = None
    error: Optional[str] = None

# 검색 요청 모델
class SearchRequest(BaseModel):
    embedding: Optional[List[float]] = None # embedding을 선택적으로 변경
    query: Optional[str] = None # query를 선택적으로 변경
    where_filter: Optional[Dict[str, Any]] = None
    user_id: Optional[str] = None
    actual_user_id: Optional[str] = None  # 'all' 검색 시 실제 사용자 ID  
    tempSettings: Optional[Dict[str, Any]] = None  # 임시 설정 추가
    ragSearchScope: Optional[str] = 'personal'  # RAG 검색 범위 추가
    optimizationMode: Optional[bool] = False  # Smart Tune 최적화 모드
    maxResults: Optional[int] = None  # 최대 결과 수 제한

# 전역 변수로 파일 처리 상태를 추적하는 딕셔너리 추가
processed_files = {}

def setup_postgresql_dir():
    """디렉토리 설정 - PostgreSQL 사용"""
    logger.debug("PostgreSQL 기반 RAG 시스템 사용")
    return

def check_and_fix_permissions():
    """PostgreSQL 관련 권한 확인"""
    try:
        setup_postgresql_dir()
        
        # PostgreSQL 연결 테스트
        try:
            test_conn = get_pool_connection()
            return_pool_connection(test_conn)
            logger.debug("PostgreSQL 연결 테스트 완료")
        except Exception as pg_error:
            logger.error(f"PostgreSQL 연결 테스트 실패: {str(pg_error)}")
            
    except Exception as e:
        logger.debug(f"권한 확인 중 오류 발생: {str(e)}")

def setup_model_directory():
    """모델 저장을 위한 디렉토리 설정"""
    try:
        # ~/.airun/models 디렉토리 경로 생성
        model_dir = os.path.expanduser("~/.airun/models")
        os.makedirs(model_dir, exist_ok=True)
        return model_dir
    except Exception as e:
        logger.error(f"모델 디렉토리 설정 실패: {str(e)}")
        raise

class KoreanSentenceTransformerEmbeddings(Embeddings):
    def __init__(self, model_name: str = "snunlp/KR-SBERT-V40K-klueNLI-augSTS", cache_dir: str = None):
        if cache_dir is None:
            cache_dir = os.path.expanduser("~/.airun/models")
        self.model = SentenceTransformer(model_name, cache_folder=cache_dir)
        
    def embed_documents(self, texts: List[str]) -> List[List[float]]:
        """문서 리스트를 임베딩"""
        embeddings = self.model.encode(texts, convert_to_numpy=True)
        return embeddings.tolist()
    
    def embed_query(self, text: str) -> List[float]:
        """쿼리 텍스트를 임베딩"""
        embedding = self.model.encode(text, convert_to_numpy=True)
        return embedding.tolist()

class EmbeddingService:
    """PostgreSQL/pgvector 기반 임베딩 서비스"""
    
    _instance = None
    _lock = Lock()
    _initialized = False

    def __new__(cls):
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    cls._instance = super().__new__(cls)
        return cls._instance

    def __init__(self):
        if not self._initialized:
            # 메인 프로세스에서는 EmbeddingService 초기화 건너뛰기 (VRAM 절약)
            if os.getenv('AIRUN_MAIN_PROCESS') == '1':
                # logger.info("메인 프로세스: EmbeddingService 초기화 건너뛰기 (워커에서 처리)")
#                 self.document_processor = None
                self.batch_buffer = []
                self.batch_lock = Lock()
                self.model = None
                self.device = None
                self.redis_client = None
                self.semantic_model = None
                self.image_model = None
                self.image_processor = None
                self._initialized = True  # 초기화 완료로 표시
                return
                
            self.batch_buffer = []
            self.batch_lock = Lock()
            self.model = None
            self.device = None
            self.redis_client = None
            self.semantic_model = None
            # 이미지 임베딩을 위한 CLIP 모델
            self.image_model = None
            self.image_processor = None
            self._initialized = False
            
            # DocumentProcessor 싱글톤 인스턴스의 DB 풀 사용
            try:
                from plugins.rag.rag_process import DocumentProcessor
                self.document_processor = DocumentProcessor.get_instance()
                # logger.info("EmbeddingService: DocumentProcessor DB 풀 사용")
            except Exception as e:
                logger.error(f"DocumentProcessor 연결 실패: {str(e)}")
                # Fallback: 자체 DB 설정
                self.db_config = {
                    'host': os.getenv('DB_HOST', os.getenv('POSTGRES_HOST', 'localhost')),
                    'port': int(os.getenv('DB_PORT', os.getenv('POSTGRES_PORT', '5433'))),
                    'database': os.getenv('DB_NAME', os.getenv('POSTGRES_DB', 'airun')),
                    'user': os.getenv('DB_USER', os.getenv('POSTGRES_USER', 'ivs')),
                    'password': os.getenv('DB_PASSWORD', os.getenv('POSTGRES_PASSWORD', '1234'))
                }
                self.document_processor = None
            
            # 임베딩 서비스 풀 (동시 요청 처리용)
            self.embedding_pool = None

    def get_db_connection(self):
        """연결 풀에서 데이터베이스 연결을 가져옴 - 통합된 연결 풀 사용"""
        return get_pool_connection()
    
    def return_db_connection(self, conn):
        """연결 풀에 데이터베이스 연결을 반환"""
        if self.document_processor and conn:
            self.document_processor.return_db_connection(conn)
        elif conn:
            try:
                conn.close()
            except:
                pass

    def initialize(self):
        """임베딩 서비스 초기화"""
        if self._initialized:
            return
            
        try:
            logger.info("Embedding service initialization in progress...")
            
            # DocumentProcessor 인스턴스 재사용
            global document_processor
            
            # document_processor가 없거나 model이 없으면 초기화
            if document_processor is None:
                from plugins.rag.rag_process import DocumentProcessor
                document_processor = DocumentProcessor.get_instance()
                logger.info("DocumentProcessor initialized for embedding service")
            
            # document_processor.model이 없으면 직접 초기화
            if not hasattr(document_processor, 'model') or document_processor.model is None:
                from plugins.rag.rag_process import get_embedding_model, get_rag_settings
                settings = get_rag_settings()
                document_processor.model = get_embedding_model(settings, logger)
                logger.info("Embedding model initialized for document processor")
            
            self.model = document_processor.model
            
            # HuggingFaceEmbeddings 구조에 맞게 수정
            try:
                # HuggingFaceEmbeddings는 client 속성을 가지고 있음 (_client 아님)
                if hasattr(self.model, 'client'):
                    self.tokenizer = self.model.client.tokenizer
                    self.device = self.model.client.device
                else:
                    # client 속성이 없는 경우 대체 방법
                    self.tokenizer = None
                    self.device = 'cpu'
                    if hasattr(self.model, 'model_kwargs'):
                        self.device = self.model.model_kwargs.get('device', 'cpu')
            except AttributeError as e:
                logger.warning(f"Could not access tokenizer/device from model: {e}")
                self.tokenizer = None
                self.device = 'cpu'
            
            # PostgreSQL connection test
            try:
                conn = get_pool_connection()
                return_pool_connection(conn)
                logger.info("PostgreSQL connection successful")
            except Exception as e:
                logger.error(f"PostgreSQL connection failed: {str(e)}")
                raise
            
            # 한국어 시맨틱 청킹을 위한 모델 초기화
            try:
                settings = get_rag_settings()
                semantic_chunker_model = settings.get("semantic_chunker_model", "snunlp/KR-SBERT-V40K-klueNLI-augSTS")
                MODEL_CACHE_DIR = os.path.join(os.path.expanduser('~'), '.airun', 'models')
                logger.info(f"MODEL_CACHE_DIR : {MODEL_CACHE_DIR}")
                logger.info(f"한국어 시맨틱 청킹을 위한 모델 초기화) Loading Korean semantic model: {semantic_chunker_model}")  # RAG_SENTENCE_TRANSFORMER_MODEL


                # 시맨틱 청킹 모델도 다른 모델들과 같은 device 로직 적용
                import torch
                is_main_process = os.getenv('AIRUN_MAIN_PROCESS') == '1'
                force_cpu = settings.get('force_cpu', False)
                cuda_available = torch.cuda.is_available() and not force_cpu and not is_main_process
                device = torch.device("cuda") if cuda_available else 'cpu'
                
                self.semantic_model = HuggingFaceEmbeddings(
                    model_name=semantic_chunker_model,
                    model_kwargs={'device': device},
                    encode_kwargs={'normalize_embeddings': True},
                    cache_folder=MODEL_CACHE_DIR
                )
                logger.info("Korean semantic model loaded successfully")
            except Exception as e:
                logger.error(f"Failed to load Korean semantic model: {str(e)}")
                self.semantic_model = None
            
            # Redis 연결 시도
            try:
                self.redis_client = redis.Redis(
                    host=REDIS_HOST, 
                    port=REDIS_PORT, 
                    db=0,
                    socket_connect_timeout=5
                )
                self.redis_client.ping()
                # logger.info("Redis connection successful")
            except redis.ConnectionError as e:
                logger.error(f"Redis connection failed: {str(e)}")
                self.redis_client = None
            
            # ThreadPoolExecutor를 사용한 병렬 임베딩 처리 설정
            try:
                global thread_pool
                self.embedding_pool = thread_pool  # 전역 thread_pool 재사용
                logger.info(f"Embedding parallel processing ready")
            except Exception as e:
                logger.error(f"임베딩 병렬 처리 설정 실패: {str(e)}")
                self.embedding_pool = None
            
            # logger.info(f"Using device: {self.device}")
            logger.info("Embedding service initialization completed")
            self._initialized = True
            
        except Exception as e:
            logger.error(f"Embedding service initialization failed: {str(e)}")
            raise

    def delete_cache(self, text: str) -> bool:
        """텍스트의 임베딩 캐시를 삭제"""
        try:
            if not self.redis_client:
                logger.info("Redis client not initialized")
                return False
            
            # 캐시 키 생성
            cache_key = f"emb:{hashlib.md5(text.encode()).hexdigest()}"
            
            # 캐시 삭제
            result = self.redis_client.delete(cache_key)
            if result:
                logger.info(f"Cache deletion successful: {cache_key}")
            
            return bool(result)
        except Exception as e:
            logger.error(f"Cache deletion failed: {str(e)}")
            return False

    def load_model(self):
        """모델 로드 - 이미 initialize에서 처리됨"""
        pass  # 모든 초기화는 initialize에서 처리

    def get_embedding(self, text: str, suppress_log: bool = False) -> np.ndarray:
        """텍스트의 임베딩 생성"""
        try:
            start_time = time.time()
            # 캐시 키 생성
            cache_key = f"emb:{hashlib.md5(text.encode()).hexdigest()}"
            
            # Redis 연결이 있는 경우에만 캐시 확인
            if self.redis_client:
                try:
                    # Redis 연결 상태 확인
                    self.redis_client.ping()
                    
                    # 캐시 확인
                    start_time = time.time()
                    cached = self.redis_client.get(cache_key)
                    if cached:
                        if not suppress_log:
                            logger.info(f"캐시 히트! 소요 시간: {(time.time() - start_time):.2f}초")
                        return np.frombuffer(cached, dtype=np.float32)
                except redis.ConnectionError:
                    if not suppress_log:
                        logger.info("Redis 서버 연결 실패, 캐시를 건너뜁니다.")
            
            if not suppress_log:
                logger.info("캐시 미스, 새로운 임베딩 생성 중...")
            
            # 새 임베딩 생성 (ThreadPoolExecutor 활용)
            embedding_start = time.time()
            logger.info(f"embedding_start: {embedding_start}")
            
            # ThreadPoolExecutor가 있으면 병렬 처리
            if self.embedding_pool:
                # 완전 비동기 처리로 블로킹 방지
                def safe_embed_query(text):
                    """ThreadPoolExecutor에서 안전하게 임베딩 생성 (자동 복구 포함)"""
                    try:
                        # 모델 상태 검증
                        if self.model is None:
                            logger.warning("임베딩 모델이 None 상태 - 자동 재초기화 시도")
                            self._attempt_model_recovery()
                            
                        if self.model is None:
                            raise RuntimeError("임베딩 모델 재초기화 실패")
                            
                        result = self.model.embed_query(text)
                        return result if isinstance(result, list) else result.tolist()
                    except AttributeError as e:
                        if "'NoneType' object has no attribute 'embed_query'" in str(e):
                            logger.warning("embed_query 호출 실패 - 모델 자동 복구 시도")
                            self._attempt_model_recovery()
                            if self.model is not None:
                                result = self.model.embed_query(text)
                                return result if isinstance(result, list) else result.tolist()
                            else:
                                raise RuntimeError("자동 모델 복구 실패")
                        else:
                            raise
                    except Exception as e:
                        logger.error(f"임베딩 생성 중 오류: {str(e)}")
                        raise
                
                future = self.embedding_pool.submit(safe_embed_query, text)
                try:
                    embedding = future.result(timeout=30)  # 30초 타임아웃
                    if isinstance(embedding, list):
                        embedding = np.array(embedding)
                    if not suppress_log:
                        logger.info(f"병렬 임베딩 생성 성공")
                except concurrent.futures.TimeoutError:
                    logger.error("임베딩 생성 타임아웃")
                    raise Exception("Embedding generation timeout")
                except Exception as e:
                    logger.error(f"병렬 임베딩 생성 실패: {str(e)}")
                    raise
            else:
                # ThreadPoolExecutor가 없으면 직접 실행 (Lock 없이)
                # 모델 상태 검증 및 자동 복구
                if self.model is None:
                    logger.warning("직접 실행 임베딩 모델이 None 상태 - 자동 재초기화 시도")
                    self._attempt_model_recovery()
                    
                if self.model is None:
                    raise RuntimeError("직접 실행 임베딩 모델 재초기화 실패")
                    
                try:
                    embedding = self.model.embed_query(text)
                except AttributeError as e:
                    if "'NoneType' object has no attribute 'embed_query'" in str(e):
                        logger.warning("직접 실행 embed_query 호출 실패 - 모델 자동 복구 시도")
                        self._attempt_model_recovery()
                        if self.model is not None:
                            embedding = self.model.embed_query(text)
                        else:
                            raise RuntimeError("직접 실행 자동 모델 복구 실패")
                    else:
                        raise
                if isinstance(embedding, list):
                    embedding = np.array(embedding)
            
            embedding_time = time.time() - embedding_start
            if not suppress_log:
                logger.info(f"임베딩 생성 완료. 소요 시간: {embedding_time:.2f}초")
            
            # Redis 연결이 있는 경우에만 캐시 저장
            if self.redis_client:
                try:
                    cache_start = time.time()
                    self.redis_client.setex(
                        cache_key,
                        3600 * 24 * 7,  # 7일 캐시
                        embedding.astype(np.float32).tobytes()
                    )
                    if not suppress_log:
                        logger.info(f"캐시 저장 완료. 소요 시간: {(time.time() - cache_start):.2f}초")
                except Exception as e:
                    if not suppress_log:
                        logger.error(f"캐시 저장 실패: {str(e)}")
            
            total_time = time.time() - start_time
            if not suppress_log:
                logger.info(f"전체 처리 시간: {total_time:.2f}초")
            
            return embedding
        except Exception as e:
            if not suppress_log:
                logger.error(f"임베딩 생성 실패: {str(e)}")
            raise

    def get_search_embedding(self, text: str, suppress_log: bool = True) -> np.ndarray:
        """검색용 임베딩 생성 (병렬 처리로 블로킹 방지)"""
        try:
            # 캐시 키 생성
            cache_key = f"search_emb:{hashlib.md5(text.encode()).hexdigest()}"
            
            # Redis 캐시 확인
            if self.redis_client:
                try:
                    cached = self.redis_client.get(cache_key)
                    if cached:
                        if not suppress_log:
                            logger.info(f"검색 캐시 히트!")
                        return np.frombuffer(cached, dtype=np.float32)
                except redis.ConnectionError:
                    pass
            
            # ThreadPoolExecutor로 병렬 임베딩 생성 (블로킹 방지)
            embedding_start = time.time()
            
            if self.embedding_pool:
                # 완전 비동기 처리로 블로킹 방지
                def safe_embed_query(text):
                    """ThreadPoolExecutor에서 안전하게 임베딩 생성 (자동 복구 포함)"""
                    try:
                        # 모델 상태 검증
                        if self.model is None:
                            logger.warning("검색용 임베딩 모델이 None 상태 - 자동 재초기화 시도")
                            self._attempt_model_recovery()
                            
                        if self.model is None:
                            raise RuntimeError("검색용 임베딩 모델 재초기화 실패")
                            
                        result = self.model.embed_query(text)
                        return result if isinstance(result, list) else result.tolist()
                    except AttributeError as e:
                        if "'NoneType' object has no attribute 'embed_query'" in str(e):
                            logger.warning("검색용 embed_query 호출 실패 - 모델 자동 복구 시도")
                            self._attempt_model_recovery()
                            if self.model is not None:
                                result = self.model.embed_query(text)
                                return result if isinstance(result, list) else result.tolist()
                            else:
                                raise RuntimeError("검색용 자동 모델 복구 실패")
                        else:
                            raise
                    except Exception as e:
                        logger.error(f"임베딩 생성 중 오류: {str(e)}")
                        raise
                
                future = self.embedding_pool.submit(safe_embed_query, text)
                try:
                    embedding = future.result(timeout=15)  # 15초 타임아웃
                    if isinstance(embedding, list):
                        embedding = np.array(embedding)
                    logger.info(f"병렬 검색 임베딩 생성 성공 (소요시간: {time.time() - embedding_start:.2f}초)")
                except concurrent.futures.TimeoutError:
                    if not suppress_log:
                        logger.error("검색 임베딩 생성 타임아웃 - 캐시된 결과 사용 시도")
                    # 타임아웃 시 기본값 반환하거나 재시도
                    raise Exception("Embedding generation timeout")
                except Exception as e:
                    if not suppress_log:
                        logger.error(f"병렬 검색 임베딩 생성 실패: {str(e)}")
                    raise
            else:
                # ThreadPoolExecutor가 없으면 직접 실행
                # 모델 상태 검증 및 자동 복구
                if self.model is None:
                    logger.warning("직접 실행 검색 임베딩 모델이 None 상태 - 자동 재초기화 시도")
                    self._attempt_model_recovery()
                    
                if self.model is None:
                    raise RuntimeError("직접 실행 검색 임베딩 모델 재초기화 실패")
                    
                try:
                    embedding = self.model.embed_query(text)
                except AttributeError as e:
                    if "'NoneType' object has no attribute 'embed_query'" in str(e):
                        logger.warning("직접 실행 검색 embed_query 호출 실패 - 모델 자동 복구 시도")
                        self._attempt_model_recovery()
                        if self.model is not None:
                            embedding = self.model.embed_query(text)
                        else:
                            raise RuntimeError("직접 실행 검색 자동 모델 복구 실패")
                    else:
                        raise
                if isinstance(embedding, list):
                    embedding = np.array(embedding)
            
            if not suppress_log:
                logger.info(f"검색 임베딩 생성 완료. 소요 시간: {(time.time() - embedding_start):.2f}초")
            
            return embedding
        except Exception as e:
            logger.error(f"검색 임베딩 생성 중 오류: {str(e)}")
            raise
        
    def _attempt_model_recovery(self):
        """모델 자동 복구 시도"""
        try:
            logger.warning("🔧 모델 자동 복구 시작...")
            
            # 현재 초기화 상태 재설정
            self._initialized = False
            
            # DocumentProcessor 재초기화
            global document_processor
            if document_processor is not None and hasattr(document_processor, 'model'):
                document_processor.model = None
            
            # 모델 재초기화 시도
            self.initialize()
            
            if self.model is not None:
                logger.info("✅ 모델 자동 복구 성공")
            else:
                logger.error("❌ 모델 자동 복구 실패 - 여전히 None 상태")
                
        except Exception as e:
            logger.error(f"❌ 모델 자동 복구 중 오류 발생: {str(e)}")
            self.model = None
            
            # 캐시 저장
            if self.redis_client:
                try:
                    self.redis_client.setex(
                        cache_key,
                        3600 * 24 * 7,  # 7일 캐시
                        embedding.astype(np.float32).tobytes()
                    )
                except Exception:
                    pass
            
            return embedding
            
        except Exception as e:
            if not suppress_log:
                logger.error(f"검색 전용 임베딩 생성 실패: {str(e)}")
            # 실패 시 기본 임베딩 메서드로 폴백
            return self.get_embedding(text, suppress_log)

    def add_to_batch(self, text: str):
        """배치에 텍스트 추가"""
        with self.batch_lock:
            self.batch_buffer.append({'text': text})

    async def process_batch(self):
        """배치 처리 - PostgreSQL 버전에서는 단순화"""
        with self.batch_lock:
            if not self.batch_buffer:
                return
            
            texts = [item['text'] for item in self.batch_buffer]
            try:
                # 각 텍스트에 대해 임베딩 생성
                for text in texts:
                    self.get_embedding(text, suppress_log=True)
                self.batch_buffer.clear()
            except Exception as e:
                logger.error(f"Batch processing failed: {str(e)}")

    def create_semantic_chunks(self, text: str) -> List[str]:
        """시맨틱 청킹을 수행하여 텍스트를 의미 있는 청크로 분할 - 임베딩 워커에게 위임"""
        try:
            # 임베딩 워커에게 시맨틱 청킹 작업 위임 (GPU 모델 활용)
            import uuid
            task_id = str(uuid.uuid4())
            
            # Redis를 통해 임베딩 워커와 통신 (중앙집중화된 설정 사용)
            import redis
            redis_host = os.getenv('REDIS_HOST', 'localhost')
            redis_port = int(os.getenv('REDIS_PORT', '6379'))
            redis_client = redis.Redis(host=redis_host, port=redis_port, db=0, decode_responses=True)
            
            # 작업 데이터 저장
            task_data = {
                'type': 'semantic_chunk',
                'text': text,
                'task_id': task_id
            }
            
            # 임베딩 큐에 작업 추가
            redis_client.lpush('embedding_tasks', json.dumps(task_data))
            logger.info(f"시맨틱 청킹 작업을 임베딩 워커에게 위임: {task_id}")
            
            # 결과 대기 (최대 30초)
            for i in range(300):  # 30초 (0.1초씩 300번)
                result_key = f"semantic_chunk_result:{task_id}"
                result_data = redis_client.get(result_key)
                
                if result_data:
                    result = json.loads(result_data)
                    redis_client.delete(result_key)  # 결과 삭제
                    
                    if result.get('success'):
                        chunks = result.get('chunks', [])
                        logger.info(f"임베딩 워커로부터 시맨틱 청킹 결과 수신: {len(chunks)}개 청크")
                        return chunks
                    else:
                        error_msg = result.get('error', 'Unknown error')
                        logger.error(f"임베딩 워커 시맨틱 청킹 실패: {error_msg}")
                        raise ValueError(f"Semantic chunking failed: {error_msg}")
                
                time.sleep(0.1)
            
            # 타임아웃
            logger.error("시맨틱 청킹 요청 타임아웃")
            raise ValueError("Semantic chunking timeout")

        except Exception as e:
            logger.error(f"Semantic chunking failed: {str(e)}")
            raise

    def calculate_relevance_score(self, query: str, document: str) -> float:
        """문서의 관련성 점수를 계산하는 함수 - 단순화된 버전"""
        try:
            # 기본 텍스트 유사도 계산
            query_lower = query.lower()
            doc_lower = document.lower()
            
            # 단순 키워드 매칭 점수
            query_words = set(query_lower.split())
            doc_words = set(doc_lower.split())
            
            if not query_words:
                return 0.0
            
            # 교집합 비율 계산
            common_words = query_words.intersection(doc_words)
            score = len(common_words) / len(query_words)
            
            return min(score, 1.0)
        except Exception as e:
            logger.error(f"Relevance score calculation failed: {str(e)}")
            return 0.0

    def calculate_text_similarity(self, text1: str, text2: str) -> float:
        """두 텍스트 간 코사인 유사도 계산"""
        try:
            emb1 = self.get_embedding(text1, suppress_log=True)
            emb2 = self.get_embedding(text2, suppress_log=True)
            
            # 코사인 유사도 계산
            dot_product = np.dot(emb1, emb2)
            norm1 = np.linalg.norm(emb1)
            norm2 = np.linalg.norm(emb2)
            
            if norm1 == 0 or norm2 == 0:
                return 0.0
            
            similarity = dot_product / (norm1 * norm2)
            return max(0.0, similarity)  # 음수 방지
        except Exception as e:
            logger.error(f"Text similarity calculation failed: {str(e)}")
            return 0.0

    def get_collection_count(self) -> int:
        """PostgreSQL 컬렉션 문서 수 반환"""
        try:
            conn = get_pool_connection()
            cursor = conn.cursor()
            cursor.execute("SELECT COUNT(*) FROM document_embeddings")
            count = cursor.fetchone()[0]
            cursor.close()
            self.return_db_connection(conn)
            return count
        except Exception as e:
            logger.error(f"Failed to get collection count: {str(e)}")
            return 0

    def search_documents(self, query_embedding, limit=10):
        """PostgreSQL에서 문서 검색"""
        try:
            conn = get_pool_connection()
            cursor = conn.cursor()
            
            search_query = """
            SELECT doc_id, filename, chunk_text, metadata, 1 - (embedding <=> %s::vector) as similarity 
            FROM document_embeddings 
            ORDER BY embedding <=> %s::vector 
            LIMIT %s
            """
            
            embedding_list = query_embedding.tolist() if hasattr(query_embedding, 'tolist') else list(query_embedding)
            cursor.execute(search_query, (embedding_list, embedding_list, limit))
            
            results = cursor.fetchall()
            cursor.close()
            return_pool_connection(conn)
            
            return [{
                'doc_id': row[0],
                'filename': row[1], 
                'chunk_text': row[2],
                'metadata': row[3],
                'similarity': row[4]
            } for row in results]
            
        except Exception as e:
            logger.error(f"Document search failed: {str(e)}")
            return []

    def delete_documents(self, doc_ids):
        """PostgreSQL에서 문서 삭제"""
        try:
            conn = get_pool_connection()
            cursor = conn.cursor()
            
            if isinstance(doc_ids, list):
                cursor.execute("DELETE FROM document_embeddings WHERE doc_id = ANY(%s)", (doc_ids,))
            else:
                cursor.execute("DELETE FROM document_embeddings WHERE doc_id = %s", (doc_ids,))
            
            conn.commit()
            cursor.close()
            return_pool_connection(conn)
            return True
        except Exception as e:
            logger.error(f"Document deletion failed: {str(e)}")
            return False

    @property
    def collection(self):
        """PostgreSQL collection 래퍼 반환"""
        return PostgreSQLCollectionWrapper(self)

    @property 
    def postgresql_client(self):
        """PostgreSQL client 래퍼 반환"""
        return PostgreSQLClientWrapper(self)


class PostgreSQLCollectionWrapper:
    """PostgreSQL Collection 인터페이스 래퍼 클래스"""
    
    def __init__(self, embedding_service):
        self.embedding_service = embedding_service
    
    def count(self):
        return self.embedding_service.get_collection_count()
    
    def get(self, where=None):
        """모든 문서 메타데이터 반환"""
        try:
            conn = get_pool_connection()
            cursor = conn.cursor()
            
            # where 필터 처리 (PostgreSQL 호환)
            if where:
                if 'user_id' in where:
                    cursor.execute(
                        "SELECT doc_id, filename, metadata, chunk_text FROM document_embeddings WHERE metadata->>'user_id' = %s",
                        (where['user_id'],)
                    )
                elif '$or' in where:
                    # $or 조건 처리: [{"user_id": user_id}, {"user_id": "shared"}] 형태
                    or_conditions = where['$or']
                    user_ids = []
                    for condition in or_conditions:
                        if 'user_id' in condition:
                            user_ids.append(condition['user_id'])
                    
                    if len(user_ids) == 1:
                        cursor.execute(
                            "SELECT doc_id, filename, metadata, chunk_text FROM document_embeddings WHERE metadata->>'user_id' = %s",
                            (user_ids[0],)
                        )
                    elif len(user_ids) >= 2:
                        placeholders = ','.join(['%s'] * len(user_ids))
                        cursor.execute(
                            f"SELECT doc_id, filename, metadata, chunk_text FROM document_embeddings WHERE metadata->>'user_id' IN ({placeholders})",
                            tuple(user_ids)
                        )
                    else:
                        cursor.execute("SELECT doc_id, filename, metadata, chunk_text FROM document_embeddings")
                else:
                    cursor.execute("SELECT doc_id, filename, metadata, chunk_text FROM document_embeddings")
            else:
                cursor.execute("SELECT doc_id, filename, metadata, chunk_text FROM document_embeddings")
            
            results = cursor.fetchall()
            cursor.close()
            return_pool_connection(conn)
            
            return {
                'ids': [row[0] if row[2] else row[0] for row in results],
                'metadatas': [row[2] for row in results] if results else [],
                'documents': [row[3] for row in results] if results else []
            }
        except Exception as e:
            logger.error(f"Failed to get all documents: {str(e)}")
            return {'ids': [], 'metadatas': []}

    def query(self, query_embeddings, n_results=5, where=None, include_image_search=False, image_query_embedding=None):
        """벡터 검색 - PostgreSQLWrapper의 query 메서드를 위임"""
        try:
            # PostgreSQL 래퍼의 query 메서드를 직접 호출
            from plugins.rag.rag_process import get_global_postgresql
            _, postgresql_wrapper = get_global_postgresql()
            
            return postgresql_wrapper.query(
                query_embeddings=query_embeddings,
                n_results=n_results,
                where=where,
                include_image_search=include_image_search,
                image_query_embedding=image_query_embedding
            )
        except Exception as e:
            logger.error(f"PostgreSQL collection query failed: {str(e)}")
            return {'documents': [[]], 'metadatas': [[]], 'distances': [[]]}

    
    # --- 내부 유틸: 테이블/인덱스 준비 ------------------------------------
    MAX_IDENT = 100  # PostgreSQL 최대 식별자 길이
    @staticmethod
    def _short_ident(base: str, prefix: str, max_len: int = MAX_IDENT) -> str:
        if not isinstance(base, str):
            base = str(base)
        h = hashlib.blake2b(base.encode("utf-8"), digest_size=6).hexdigest()  # 12 hex
        name = f"{prefix}_{h}"
        return name[:max_len]

    def _index_name(base: str) -> str:
        return base[:MAX_IDENT]

    def _table_exists(cursor, table: str) -> bool:
        cursor.execute("SELECT to_regclass(%s)", (f"public.{table}",))
        return cursor.fetchone()[0] is not None

    def _list_indexes(cursor, table: str):
        cursor.execute("""
            SELECT indexname
            FROM pg_indexes
            WHERE schemaname='public' AND tablename=%s
            ORDER BY indexname
        """, (table,))
        return [r[0] for r in cursor.fetchall()]

    def _vector_available(cursor) -> bool:
        try:
            cursor.execute("CREATE EXTENSION IF NOT EXISTS vector")
        except Exception:
            return False
        try:
            cursor.execute("SELECT 1 FROM pg_type WHERE typname='vector' LIMIT 1")
            return cursor.fetchone() is not None
        except Exception:
            return False

    def _ensure_model_table(
        self,
        cursor,
        table_name: str,
        text_dim: int | None = None,
        image_dim: int | None = None
    ):
        t0 = time.perf_counter()
        logger.info("[DDL] ensure table start: table=%s, req_text_dim=%s, req_image_dim=%s",
                    table_name, text_dim, image_dim)

        # -- 0) pgvector 체크 -------------------------------------------------------
        use_pgvector = True
        try:
            cursor.execute("CREATE EXTENSION IF NOT EXISTS vector")
            cursor.execute("SELECT 1 FROM pg_type WHERE typname='vector' LIMIT 1")
            use_pgvector = cursor.fetchone() is not None
            logger.info("[DDL] pgvector available=%s", use_pgvector)
        except Exception as e:
            use_pgvector = False
            logger.warning("[DDL] pgvector check failed, fallback to arrays (reason=%s)", e)

        # -- 1) 차원/타입 문자열 ----------------------------------------------------
        text_dim  = int(text_dim or 1024)
        image_dim = int(image_dim or 512)
        emb_type = f"vector({text_dim})"  if use_pgvector else "double precision[]"
        img_type = f"vector({image_dim})" if use_pgvector else "double precision[]"
        uniq = f"{table_name}_unique_doc_chunk"[:63]
        logger.info("[DDL] types resolved: emb_type=%s, img_type=%s, uniq=%s",
                    emb_type, img_type, uniq)

        # -- 2) 테이블 생성 ---------------------------------------------------------
        try:
            create_tbl = sql.SQL("""
                CREATE TABLE IF NOT EXISTS {tbl} (
                    id              integer GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
                    is_embedding    boolean NOT NULL DEFAULT false,
                    doc_id          varchar(500) NOT NULL,
                    filename        varchar(500) NOT NULL,
                    chunk_index     integer NOT NULL,
                    chunk_text      text NOT NULL,
                    embedding       {emb_type},
                    image_embedding {img_type},
                    user_id         varchar(255),
                    source          text,
                    file_mtime      bigint,
                    metadata        jsonb DEFAULT '{{}}'::jsonb,
                    created_at      timestamp without time zone DEFAULT CURRENT_TIMESTAMP,
                    CONSTRAINT {uniq} UNIQUE (doc_id, chunk_index)
                )
            """).format(
                tbl=sql.Identifier(table_name),
                uniq=sql.Identifier(uniq),
                emb_type=sql.SQL(emb_type),
                img_type=sql.SQL(img_type),
            )
            logger.debug("[DDL] create table SQL ready for %s", table_name)
            cursor.execute(create_tbl)
            logger.info("[DDL] table ensured: %s", table_name)
        except Exception as e:
            logger.error("[DDL] create table failed for %s: %s", table_name, e)
            raise

        # -- 3) 보조 인덱스들 ------------------------------------------------------
        btree_defs = [
            (f"idx_{table_name}_chunk_index"[:63],  "chunk_index"),
            (f"idx_{table_name}_created_at"[:63],   "created_at"),
            (f"idx_{table_name}_doc_id"[:63],       "doc_id"),
            (f"idx_{table_name}_filename"[:63],     "filename"),
            (f"idx_{table_name}_is_embedding"[:63], "is_embedding"),
            (f"idx_{table_name}_source"[:63],       "source"),
            (f"idx_{table_name}_user_id"[:63],      "user_id"),
        ]
        for idx_name, col in btree_defs:
            try:
                stmt = sql.SQL("CREATE INDEX IF NOT EXISTS {idx} ON {tbl} ({col})").format(
                    idx=sql.Identifier(idx_name),
                    tbl=sql.Identifier(table_name),
                    col=sql.Identifier(col),
                )
                cursor.execute(stmt)
                logger.info("[DDL] btree index ensured: %s ON %s(%s)", idx_name, table_name, col)
            except Exception as e:
                logger.warning("[DDL] btree index create failed: %s (reason=%s)", idx_name, e)

        # partial index (has image)
        try:
            stmt = sql.SQL(
                "CREATE INDEX IF NOT EXISTS {idx} ON {tbl} ((image_embedding IS NOT NULL))"
            ).format(
                idx=sql.Identifier(f"idx_{table_name}_has_image"[:63]),
                tbl=sql.Identifier(table_name),
            )
            cursor.execute(stmt)
            logger.info("[DDL] partial index ensured: idx_%s_has_image", table_name)
        except Exception as e:
            logger.warning("[DDL] partial index create failed on %s: %s", table_name, e)

        # -- 4) 벡터 인덱스 (pgvector) ---------------------------------------------
        if use_pgvector:
            try:
                stmt = sql.SQL(
                    "CREATE INDEX IF NOT EXISTS {idx} ON {tbl} USING ivfflat (embedding vector_cosine_ops) WITH (lists='100')"
                ).format(
                    idx=sql.Identifier(f"idx_{table_name}_vector_cosine"[:63]),
                    tbl=sql.Identifier(table_name),
                )
                cursor.execute(stmt)
                logger.info("[DDL] vector index ensured: idx_%s_vector_cosine", table_name)
            except Exception as e:
                logger.warning("[DDL] vector index(embedding) failed on %s: %s", table_name, e)

            try:
                stmt = sql.SQL(
                    "CREATE INDEX IF NOT EXISTS {idx} ON {tbl} USING ivfflat (image_embedding vector_cosine_ops) WITH (lists='100')"
                ).format(
                    idx=sql.Identifier(f"idx_{table_name}_image_vector_cosine"[:63]),
                    tbl=sql.Identifier(table_name),
                )
                cursor.execute(stmt)
                logger.info("[DDL] vector index ensured: idx_%s_image_vector_cosine", table_name)
            except Exception as e:
                logger.warning("[DDL] vector index(image) failed on %s: %s", table_name, e)

        # -- 5) 완료 로그 -----------------------------------------------------------
        dt = (time.perf_counter() - t0) * 1000
        logger.info("[DDL] ensure table done: %s (%.1f ms, text_dim=%d, image_dim=%d, pgvector=%s)",
                    table_name, dt, text_dim, image_dim, use_pgvector)

        
            
    def add(self, model_id, embeddings=None, documents=None, ids=None, metadatas=None, image_embeddings=None):
        """문서 추가 - PostgreSQL에 직접 저장"""
        conn = None
        cursor = None

        try:
            if not all([embeddings, documents, ids, metadatas]):
                logger.error("Missing required parameters for add operation")
                return False

            # 허용된 테이블 접두사 정의 (SQL 인젝션 방지)
            ALLOWED_TABLE_PREFIXES = ['rag_model_v1', 'rag_model_v2', 'default_model', 'nlpai_lab_kure_v1', 'snunlp_kr_sbert']

            conn = get_pool_connection()
            cursor = conn.cursor()

            if not embeddings:
                logger.error("No embeddings provided for add operation")
                return False

            # 동적 테이블명 생성 및 검증 ------------------------------------------------
            logger.info(f"model_id==={model_id}")
            # rag_model_safe = re.sub(r"[^a-zA-Z0-9_]", "_", model_id)
            # table_name = f"{rag_model_safe}_document_embeddings"


            # if hasattr(self, 'embedding_service') and hasattr(self.embedding_service, 'document_processor') \
            #    and hasattr(self.embedding_service.document_processor, 'rag_model'):
            #     base_table_name = self.embedding_service.document_processor.rag_model
            # else:
            #     base_table_name = 'default_model'
            # table_name_ryan = f"{base_table_name}_document_embeddings"

            
            # 1) 모델명 정규화 → 소문자
            rag_model_safe = re.sub(r"[^a-zA-Z0-9_]", "_", str(model_id)).strip("_").lower()
            # 2) 테이블명 생성(+ suffix) → 소문자
            table_name = f"{rag_model_safe}_document_embeddings".lower()

            # (선택) 기존 코드의 참고용 베이스명도 소문자화
            if getattr(self, "embedding_service", None) and \
            getattr(self.embedding_service, "document_processor", None) and \
            getattr(self.embedding_service.document_processor, "rag_model", None):
                base_table_name = str(self.embedding_service.document_processor.rag_model).lower()
            else:
                base_table_name = 'default_model'

            table_name_ryan = f"{base_table_name}_document_embeddings"

            logger.info(f"Using table: {table_name} (model_name_norm={rag_model_safe})")
            logger.info(f"ref base table: {table_name_ryan}")

            # 4) 접두사 검증은 “모델명 부분”에 대해 수행
            model_prefix = rag_model_safe  # suffix 붙이기 전 모델명만
            if not any(model_prefix.startswith(p) for p in ALLOWED_TABLE_PREFIXES):
                logger.error(f"Invalid table name: {table_name}. Allowed prefixes: {ALLOWED_TABLE_PREFIXES}")
                return False


            logger.info(f"Using table: {table_name} for document , model_name: {rag_model_safe}")
            logger.info(f"-{table_name_ryan}--------->Using table: {table_name} for document , model_name: {rag_model_safe}")

            # 테이블명 보안 검증 (SQL 인젝션 방지)
            if not any(table_name.startswith(prefix) for prefix in ALLOWED_TABLE_PREFIXES):
                logger.error(f"Invalid table name: {table_name}. Only allowed prefixes: {ALLOWED_TABLE_PREFIXES}")
                return False

            table_name = ''.join(c for c in table_name if c.isalnum() or c in '_')

            # 임베딩 차원 계산
            first_embedding = embeddings[0]
            if hasattr(first_embedding, 'tolist'):
                embedding_dim = len(first_embedding.tolist())
            elif isinstance(first_embedding, np.ndarray):
                embedding_dim = len(first_embedding)
            else:
                embedding_dim = len(list(first_embedding))

            image_dim = None
            if image_embeddings:
                for raw_image_embedding in image_embeddings:
                    if raw_image_embedding is None:
                        continue
                    if hasattr(raw_image_embedding, 'tolist'):
                        image_dim = len(raw_image_embedding.tolist())
                    elif isinstance(raw_image_embedding, np.ndarray):
                        image_dim = len(raw_image_embedding)
                    else:
                        image_dim = len(list(raw_image_embedding))
                    break

            if embedding_dim <= 0:
                logger.error("Invalid embedding dimension detected")
                return False

            # ▶ 3) 모델 테이블/인덱스 생성, 오토커밋으로 DDL 분리
            prev_autocommit = conn.autocommit
            # 먼저 기존 트랜잭션이 있다면 종료
            if not prev_autocommit:
                try:
                    conn.rollback()
                except Exception:
                    pass
            conn.autocommit = True
            try:
                logger.info("=======================table create ===================")
                logger.info(f"Embedding dim: {embedding_dim}, Image dim: {image_dim}")
                self._ensure_model_table(cursor, table_name, embedding_dim, image_dim)  
            finally:
                conn.autocommit = prev_autocommit  # 원복
            # 동적 테이블명 생성 및 검증 ------------------------------------------------ END

            # SQL 쿼리를 미리 준비 (모든 문서 공통 사용)
            insert_query = sql.SQL("""
                INSERT INTO {}
                (doc_id, filename, chunk_index, chunk_text, embedding, image_embedding, user_id, source, file_mtime, metadata)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
                ON CONFLICT (doc_id, chunk_index) DO UPDATE SET
                    filename = EXCLUDED.filename,
                    chunk_text = EXCLUDED.chunk_text,
                    embedding = EXCLUDED.embedding,
                    image_embedding = EXCLUDED.image_embedding,
                    user_id = EXCLUDED.user_id,
                    source = EXCLUDED.source,
                    file_mtime = EXCLUDED.file_mtime,
                    metadata = EXCLUDED.metadata
            """).format(sql.Identifier(table_name))

            logger.info("---------각 문서를 PostgreSQL에 저장----------------------")
            # 각 문서를 PostgreSQL에 저장
            for i, (embedding, document, doc_id, metadata) in enumerate(zip(embeddings, documents, ids, metadatas)):
                # 임베딩 데이터 검증 및 정규화
                if hasattr(embedding, 'tolist'):
                    embedding_list = embedding.tolist()
                elif isinstance(embedding, np.ndarray):
                    embedding_list = embedding.tolist()
                else:
                    embedding_list = list(embedding)

                # 임베딩 차원 검증 (일반적으로 384, 512, 768, 1024 등)
                if len(embedding_list) < 50 or len(embedding_list) > 4096:
                    logger.warning(f"Unusual embedding dimension: {len(embedding_list)} for doc_id: {doc_id}")

                # 이미지 임베딩 처리
                image_embedding = None
                if image_embeddings and i < len(image_embeddings):
                    raw_image_embedding = image_embeddings[i]
                    if hasattr(raw_image_embedding, 'tolist'):
                        image_embedding = raw_image_embedding.tolist()
                    elif isinstance(raw_image_embedding, np.ndarray):
                        image_embedding = raw_image_embedding.tolist()
                    else:
                        image_embedding = list(raw_image_embedding)
                    metadata['has_image_embedding'] = True

                # 메타데이터에서 필요한 정보 추출
                filename = metadata.get('source', metadata.get('filename', ''))
                chunk_index = metadata.get('chunk_index', i)
                user_id = metadata.get('user_id', '')
                source = metadata.get('source', filename)
                file_mtime = metadata.get('file_mtime')

                cursor.execute(insert_query, (
                    doc_id,
                    filename,
                    chunk_index,
                    document,
                    embedding_list,
                    image_embedding,
                    user_id,
                    source,
                    file_mtime,
                    Json(metadata)
                ))

            conn.commit()
            return True

        except Exception as e:
            logger.error(f"PostgreSQL collection add failed: {str(e)}")
            if conn:
                conn.rollback()
            return False
        finally:
            # 확실한 리소스 정리
            if cursor:
                cursor.close()
            if conn:
                return_pool_connection(conn)

    def delete(self, ids=None):
        """문서 삭제"""
        if ids:
            return self.embedding_service.delete_documents(ids)
        return False


class PostgreSQLClientWrapper:
    """PostgreSQL Client 인터페이스 래퍼 클래스"""
    
    def __init__(self, embedding_service):
        self.embedding_service = embedding_service
    
    def get_collection(self, name):
        return self.embedding_service.collection


async def process_file_event(abs_file_path, event_type):
    """파일 이벤트 처리 함수"""
    try:
        # .extracts 폴더의 파일은 임베딩 대상이 아니므로 처리 건너뛰기
        if '/.extracts/' in abs_file_path or os.sep + '.extracts' + os.sep in abs_file_path:
            logger.info(f"파일 이벤트 건너뛰기: .extracts 폴더의 임시 파일 - {os.path.basename(abs_file_path)}")
            return "skipped_extracts_file"
        
        # 파일 확장자 추출
        file_ext = os.path.splitext(abs_file_path)[1].lower()
        
        # 사용자 ID 추출 (파일 경로에서)
        user_id = "admin"  # 기본값
        try:
            # 경로에서 사용자 폴더명 추출 (예: /path/to/rag-docs/admin/file.txt -> admin)
            path_parts = abs_file_path.split(os.sep)
            rag_docs_index = -1
            for i, part in enumerate(path_parts):
                if 'rag-docs' in part or 'rag_docs' in part:
                    rag_docs_index = i
                    break
            if rag_docs_index >= 0 and rag_docs_index + 1 < len(path_parts):
                user_id = path_parts[rag_docs_index + 1]
        except Exception as e:
            logger.warning(f"사용자 ID 추출 실패, 기본값 사용: {str(e)}")
        
        # 임베딩 서비스 인스턴스 가져오기
        embedding_service = EmbeddingService()
        
        # 권한 확인 및 수정
        check_and_fix_permissions()
        
        # 이벤트 타입에 따른 처리
        if event_type in ["DELETE", "MOVED_FROM"]:
            
            try:
                # PostgreSQL 연결 전에 권한 확인
                check_and_fix_permissions()
                
                # PostgreSQL 연결
                embedding_service = EmbeddingService()
                collection = embedding_service.collection
                
                # 파일 경로로 메타데이터 검색
                all_docs = collection.get()
                chunk_ids_to_delete = []
                chunks_to_delete = []
                
                # 단일 파일 처리
                logger.info(f"파일 삭제 처리: {abs_file_path}")
                normalized_target = os.path.normpath(abs_file_path)
                for i, metadata in enumerate(all_docs['metadatas']):
                    if metadata and 'source' in metadata:
                        doc_source = os.path.normpath(os.path.abspath(metadata['source']))
                        if doc_source == normalized_target:
                            chunk_ids_to_delete.append(all_docs['ids'][i])
                            chunks_to_delete.append(all_docs['documents'][i])
                            logger.info(f"일치하는 문서 찾음: {doc_source}")
                
                if chunk_ids_to_delete:
                    logger.info(f"삭제할 청크 발견: {len(chunk_ids_to_delete)}개 ({os.path.basename(abs_file_path)})")
                    
                    # 각 청크의 캐시 삭제
                    embedding_service = EmbeddingService()
                    embedding_service.initialize()
                    for chunk in chunks_to_delete:
                        embedding_service.delete_cache(chunk)
                    
                    # PostgreSQL에서 청크들 삭제
                    collection.delete(ids=chunk_ids_to_delete)
                    logger.info(f"Successfully deleted {len(chunk_ids_to_delete)} chunks from PostgreSQL")

                    # chat_documents 테이블에서도 삭제
                    try:
                        filename = os.path.basename(abs_file_path)
                        # 파일 경로에서 user_id 추출 (예: /path/to/rag-docs/admin/file.txt -> admin)
                        path_parts = abs_file_path.split(os.sep)
                        rag_docs_idx = -1
                        for i, part in enumerate(path_parts):
                            if 'rag-docs' in part or 'rag_docs' in part:
                                rag_docs_idx = i
                                break

                        user_id = path_parts[rag_docs_idx + 1] if rag_docs_idx >= 0 and rag_docs_idx + 1 < len(path_parts) else None

                        if user_id:
                            conn = get_pool_connection()
                            try:
                                with conn.cursor() as cur:
                                    cur.execute("""
                                        DELETE FROM chat_documents
                                        WHERE filename = %s AND user_id = %s
                                    """, (filename, user_id))
                                    conn.commit()
                                    deleted_count = cur.rowcount
                                    if deleted_count > 0:
                                        logger.info(f"chat_documents에서 삭제됨: {filename} (사용자: {user_id}, {deleted_count}건)")
                                    else:
                                        logger.info(f"chat_documents에서 삭제할 레코드 없음: {filename} (사용자: {user_id})")
                            finally:
                                return_pool_connection(conn)
                        else:
                            logger.warning(f"user_id를 추출할 수 없어 chat_documents 삭제 건너뜀: {abs_file_path}")
                    except Exception as chat_delete_error:
                        logger.error(f"chat_documents 삭제 중 오류: {str(chat_delete_error)}")

                    # processed_files에서 제거
                    if abs_file_path in processed_files:
                        del processed_files[abs_file_path]

                    return "success"
                else:
                    logger.info(f"No chunks found for document: {os.path.basename(abs_file_path)}")
                    return "no_chunks_found"
                
            except Exception as e:
                logger.error(f"Error deleting document: {str(e)}")
                return "error_deleting_document"
                
        elif event_type in ["CREATE", "MODIFY", "MOVED_TO"]:
            logger.info(f"Waiting for CLOSE_WRITE event: {abs_file_path}")
            return "waiting_for_close"
            
        elif event_type == "CLOSE_WRITE":
            # 파일 크기가 0이면 처리하지 않음
            if not os.path.exists(abs_file_path):
                logger.info(f"File does not exist: {abs_file_path}")
                return "file_not_found"
                
            file_size = os.path.getsize(abs_file_path)
            if file_size == 0:
                logger.info(f"Skipping empty file: {abs_file_path}")
                return "skipped_empty_file"
            
            # 협조적 처리 큐 확인 - API가 처리 중인지 확인 (실패 시 서비스 중단하지 않음)
            try:
                import json

                redis_client = get_redis_client()
                if not redis_client:
                    logger.info(f"Redis 연결 실패, 기본 처리 진행: {os.path.basename(abs_file_path)}")
                else:
                    file_processing_key = f"file_processing:{abs_file_path}"
                    # Redis 작업에 timeout 적용
                    processing_info_str = redis_client.get(file_processing_key)
                    
                    if processing_info_str:
                        try:
                            processing_info = json.loads(processing_info_str)
                            current_time = time.time()
                            
                            # 상태 유효성 검증
                            if (processing_info.get("status") == "processing" and 
                                processing_info.get("source") == "api"):
                                
                                # 시간 기반 타임아웃 체크 (1시간)
                                started_at = processing_info.get("started_at", current_time)
                                if current_time - started_at > 3600:  # 1시간 타임아웃
                                    logger.info(f"API 처리 타임아웃, 모니터링에서 재처리: {os.path.basename(abs_file_path)}")
                                    try:
                                        redis_client.delete(file_processing_key)
                                    except:
                                        pass
                                else:
                                    logger.info(f"API 처리 중, 모니터링 대기: {os.path.basename(abs_file_path)}")
                                    return "deferred_to_api"
                                    
                            elif processing_info.get("status") == "failed":
                                # API 처리 실패시 모니터링이 재처리
                                try:
                                    redis_client.delete(file_processing_key)
                                except:
                                    pass
                                logger.info(f"API 처리 실패, 모니터링에서 재처리: {os.path.basename(abs_file_path)}")
                                
                        except (json.JSONDecodeError, KeyError, TypeError) as parse_error:
                            # JSON 파싱 실패 시 문자열 값으로 확인 (이전 버전 호환성)
                            processing_info_str_decoded = processing_info_str.decode() if isinstance(processing_info_str, bytes) else processing_info_str
                            if processing_info_str_decoded == "api_processing":
                                logger.info(f"API 처리 중 (legacy 형식), 모니터링 대기: {os.path.basename(abs_file_path)}")
                                return "deferred_to_api"
                            else:
                                logger.info(f"처리 정보 파싱 실패, 키 삭제 후 기본 처리: {os.path.basename(abs_file_path)}")
                                try:
                                    redis_client.delete(file_processing_key)
                                except:
                                    pass
                    else:
                        logger.info(f"처리 마커 없음, 모니터링에서 처리: {os.path.basename(abs_file_path)}")
                        
            except Exception as e:
                logger.error(f"처리 상태 확인 오류, 기본 처리 계속: {str(e)}")
                # 모든 Redis 오류 시에도 계속 진행
            
            # 파일 수정 시 이전 청크 삭제
            try:
                # PostgreSQL 연결
                embedding_service = EmbeddingService()
                collection = embedding_service.collection
                
                # 파일 경로로 메타데이터 검색
                all_docs = collection.get()
                chunk_ids_to_delete = []
                
                # 메타데이터에서 source 필드로 문서 찾기
                for i, metadata in enumerate(all_docs['metadatas']):
                    if metadata and 'source' in metadata:
                        doc_source = os.path.abspath(metadata['source'])
                        if doc_source == abs_file_path:
                            chunk_ids_to_delete.append(all_docs['ids'][i])
                
                if chunk_ids_to_delete:
                    logger.info(f"Found {len(chunk_ids_to_delete)} old chunks to delete for document: {os.path.basename(abs_file_path)}")
                    collection.delete(ids=chunk_ids_to_delete)
                    logger.info(f"Successfully deleted old chunks from PostgreSQL")
                    
            except Exception as e:
                logger.error(f"Error deleting old chunks: {str(e)}")
            
            # PDF 강제 변환 대상 파일인 경우 임시 PDF 파일을 미리 마킹
            temp_pdf_path = None
            
            # PDF 강제 변환에서 제외되는 확장자 (rag_process.py와 동일)
            skip_pdf_conversion_extensions = {
                '.txt', '.md', '.text', '.markdown', '.mdx', '.rst', '.adoc', '.org',
                '.py', '.js', '.jsx', '.ts', '.tsx', '.java', '.cpp', '.c', '.h', '.cs',
                '.php', '.rb', '.swift', '.go', '.rs', '.scala', '.kt', '.m', '.proto',
                '.tex', '.html', '.htm', '.sol', '.cob', '.cbl', '.lua', '.pl', '.pm',
                '.hs', '.ex', '.exs', '.ps1', '.psm1', '.sql', '.pdf'  # .pdf도 제외
            }
            
            # PDF 강제 변환 대상 파일 확인 (HWP, PPTX, DOCX, ODT 등)
            if file_ext not in skip_pdf_conversion_extensions:
                try:
                    # 예상되는 PDF 파일명 생성 (LibreOffice 변환 규칙에 따라)
                    base_name = os.path.splitext(os.path.basename(abs_file_path))[0]
                    temp_pdf_path = os.path.join(os.path.dirname(abs_file_path), f"{base_name}.pdf")
                    
                    # Redis에 임시 PDF 파일 마킹
                    redis_client = get_redis_client()
                    temp_pdf_key = f"temp_pdf_processing:{os.path.basename(temp_pdf_path)}"
                    redis_client.setex(temp_pdf_key, 300, "processing_by_monitoring")  # 5분 TTL
                    logger.info(f"PDF 강제 변환 예상 파일 마킹됨 ({file_ext}): {temp_pdf_key}")
                except Exception as redis_error:
                    logger.info(f"PDF 강제 변환 마킹 실패: {str(redis_error)}")
            
            # 워커 프로세스에게 임베딩 처리 위임
            try:
                # RAG 문서 경로 가져오기
                rag_docs_path = get_rag_docs_path()
                abs_rag_path = os.path.abspath(rag_docs_path)
                
                logger.info(f"Processing file: {abs_file_path}")
                
                # 전역 임베딩 큐와 이벤트에 접근
                global embedding_queue, embedding_model_ready_event
                
                # 워커 프로세스가 준비되었는지 확인
                if not embedding_model_ready_event.is_set():
                    logger.error("임베딩 워커가 준비되지 않음, 처리 건너뜀")
                    return "worker_not_ready"
                
                # 임베딩 요청 생성
                import uuid
                request_id = str(uuid.uuid4())
                
                # .extracts 폴더의 임시 이미지 파일 제외
                if '/.extracts/' in abs_file_path or os.sep + '.extracts' + os.sep in abs_file_path:
                    logger.info(f"임베딩 제외: .extracts 폴더의 임시 파일 - {os.path.basename(abs_file_path)}")
                    result = {"status": "skipped"}
                else:
                    # 요청을 임베딩 워커에게 전송
                    embedding_request = {
                        'request_id': request_id,
                        'file_path': abs_file_path,
                        'user_id': user_id
                    }
                    
                    embedding_queue.put(embedding_request)
                    logger.info(f"파일 처리를 임베딩 워커에게 위임: {os.path.basename(abs_file_path)}, ID: {request_id}")
                    
                    # 성공으로 간주 (워커가 백그라운드에서 처리)
                    result = {"status": "delegated"}
                
                if result["status"] == "success":
                    logger.info(f"문서 처리 완료: {os.path.basename(abs_file_path)}")
                    logger.info(f"- 처리 상태: 성공")
                    logger.info(f"- 기본 경로: {abs_rag_path}")
                    logger.info(f"- 사용자 ID: {user_id}")
                    
                    # 파일 처리 시간 업데이트
                    processed_files[abs_file_path] = os.path.getctime(abs_file_path)
                    
                    return "success"
                elif result["status"] == "delegated":
                    logger.info(f"문서 처리 워커에게 위임됨: {os.path.basename(abs_file_path)}")
                    logger.info(f"- 처리 상태: 워커 처리 중")
                    logger.info(f"- 기본 경로: {abs_rag_path}")
                    logger.info(f"- 사용자 ID: {user_id}")
                    
                    # 파일 처리 시간 업데이트 (위임 성공)
                    processed_files[abs_file_path] = os.path.getctime(abs_file_path)
                    
                    return "delegated_success"
                else:
                    logger.error(f"Failed to process file: {abs_file_path}")
                    return "processing_failed"
                    
            except Exception as e:
                logger.error(f"Error processing file: {str(e)}")
                return "error_processing_file"
            finally:
                # PDF 강제 변환 대상 파일 처리 완료 후 임시 PDF 마킹 제거
                if temp_pdf_path and file_ext not in skip_pdf_conversion_extensions:
                    try:
                        redis_client = get_redis_client()
                        temp_pdf_key = f"temp_pdf_processing:{os.path.basename(temp_pdf_path)}"
                        redis_client.delete(temp_pdf_key)
                        logger.info(f"PDF 강제 변환 마킹 제거됨 ({file_ext}): {temp_pdf_key}")
                    except Exception as redis_error:
                        logger.info(f"PDF 강제 변환 마킹 제거 실패: {str(redis_error)}")
            
        else:
            logger.warning(f"Unsupported event type: {event_type}")
            return "unsupported_event_type"
            
    except Exception as e:
        logger.error(f"Unexpected error in process_file_event: {str(e)}")
        logger.error(traceback.format_exc())
        return "unexpected_error"

# API 엔드포인트 추가
@app.post("/embed", response_model=EmbeddingResponse)
async def create_embedding(request: EmbeddingRequest):
    try:
        # PostgreSQL 기반 임베딩 서비스 사용
        embedding_service = EmbeddingService()
        embedding = embedding_service.get_embedding(request.text)
        return {
            "success": True,
            "embedding": embedding.tolist()
        }
    except Exception as e:
        logger.error(f"임베딩 생성 오류: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

# 검색 전용 임베딩 엔드포인트 추가 (블로킹 방지)
@app.post("/embed_search", response_model=EmbeddingResponse)
async def create_search_embedding(request: EmbeddingRequest):
    """검색 전용 임베딩 생성 (Lock 없이 빠른 처리)"""
    try:
        global embedding_service
        
        if not embedding_service:
            raise HTTPException(status_code=503, detail="Embedding service not initialized")
        
        # 검색 전용 임베딩 서비스 사용
        if hasattr(embedding_service, 'get_search_embedding'):
            embedding = embedding_service.get_search_embedding(request.text, suppress_log=True)
        else:
            # 폴백: 기본 임베딩 서비스 사용
            embedding = embedding_service.get_embedding(request.text, suppress_log=True)
        
        return {"success": True, "embedding": embedding.tolist()}
    except Exception as e:
        logger.error(f"검색 전용 임베딩 생성 실패: {str(e)}")
        raise HTTPException(status_code=500, detail=f"검색 전용 임베딩 생성 실패: {str(e)}")

@app.post("/embed/batch")
async def create_batch_embedding(request: BatchEmbeddingRequest):
    try:
        embedding_service = EmbeddingService()
        for text in request.texts:
            embedding_service.add_to_batch(text)
        await embedding_service.process_batch()
        return {"success": True, "message": "Batch processing started"}
    except Exception as e:
        logger.error(f"배치 임베딩 오류: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/health")
async def health_check():
    try:
        # 워커 프로세스 상태 확인
        search_ready = search_model_ready_event.is_set() if search_model_ready_event else False
        embedding_ready = embedding_model_ready_event.is_set() if embedding_model_ready_event else False
        
        # Redis 연결 확인
        redis_connected = False
        try:
            redis_client = redis.Redis(host=REDIS_HOST, port=REDIS_PORT, decode_responses=True)
            redis_connected = redis_client.ping()
        except Exception as e:
            logger.error(f"Redis 연결 확인 중 오류: {str(e)}")
        
        # 워커 프로세스의 모델 로드 상태 확인 (워커가 준비되면 모델도 로드된 것으로 간주)
        model_loaded = False
        try:
            # 워커들이 모두 준비되어 있으면 모델이 로드된 것으로 간주
            if embedding_ready and search_ready:
                model_loaded = True
            # 또는 기존 embedding_service 확인
            elif embedding_service is not None and hasattr(embedding_service, 'model'):
                model_loaded = embedding_service.model is not None
        except Exception:
            model_loaded = False
        
        return JSONResponse(
            content={
                "status": "healthy",
                "search_worker_ready": search_ready,
                "embedding_worker_ready": embedding_ready,
                "redis_connected": redis_connected,
                "model_loaded": model_loaded,
                "architecture": "2-process-separated"
            },
            status_code=200,
            headers={"Content-Type": "application/json"}
        )
    except Exception as e:
        logger.error(f"Health check 중 오류 발생: {str(e)}")
        return JSONResponse(
            content={
                "status": "error",
                "error": str(e)
            },
            status_code=500,
            headers={"Content-Type": "application/json"}
        )

@app.post("/sync_status")
def manual_sync_status():
    """수동으로 임베딩 상태 동기화 실행"""
    try:
        sync_embedding_status()
        return {"status": "success", "message": "상태 동기화 완료"}
    except Exception as e:
        return {"status": "error", "message": str(e)}

@app.post("/clear-cache")
async def clear_cache():
    try:
        # Redis 직접 연결 (EmbeddingService 초기화 불필요)
        redis_client = redis.Redis(host=REDIS_HOST, port=REDIS_PORT, decode_responses=True)
        redis_connected = redis_client.ping()
        
        if redis_connected:
            try:
                # RAG 관련 캐시 키 패턴 정의
                rag_patterns = [
                    "emb:*",     # 임베딩 캐시
                    "search:*",   # 검색 캐시
                    "rag:*",      # RAG 관련 기타 캐시
                    "cache:*",    # 일반 캐시
                    "temp:*"      # 임시 데이터
                ]
                
                keys_to_delete = []
                for pattern in rag_patterns:
                    matched_keys = redis_client.keys(pattern)
                    if matched_keys:
                        keys_to_delete.extend(matched_keys)
                
                # RQ 관련 키는 제외
                keys_to_delete = [key for key in keys_to_delete if not key.startswith("rq:")]
                
                if keys_to_delete:
                    # 선택된 키만 삭제
                    redis_client.delete(*keys_to_delete)
                    logger.info("RAG 관련 Redis 캐시가 성공적으로 제거되었습니다.")
                    
                    # RQ 워커 상태 확인
                    from rq import Queue
                    queue = Queue('image_processing', connection=redis_client)
                    if queue:
                        logger.info("이미지 처리 큐가 정상적으로 유지되고 있습니다.")
                    
                    return {
                        "status": "success",
                        "message": f"{len(keys_to_delete)}개의 캐시 항목이 제거되었습니다.",
                        "cleared_items": len(keys_to_delete)
                    }
                else:
                    # logger.info("제거할 RAG 관련 Redis 캐시가 없습니다.")
                    return {
                        "status": "success",
                        "message": "제거할 캐시가 없습니다.",
                        "cleared_items": 0
                    }
            except Exception as e:
                logger.error(f"Redis 캐시 제거 중 오류 발생: {str(e)}", error=True)
                return {
                    "status": "error",
                    "message": f"Redis 캐시 제거 중 오류: {str(e)}"
                }
        else:
            logger.info("Redis 클라이언트가 초기화되지 않았습니다.")
            return {
                "status": "error",
                "message": "Redis 연결이 활성화되지 않았습니다."
            }
    except Exception as e:
        logger.error(f"캐시 제거 중 예기치 않은 오류: {str(e)}", error=True)
        return {
            "status": "error",
            "message": f"예기치 않은 오류: {str(e)}"
        }

# 검색 결과 처리 함수
def process_search_results(results, query: str, query_embedding: List[float]):
    """검색 결과를 처리하는 함수"""
    # RAG 설정 가져오기
    rag_settings = get_rag_settings()
    
    # 가중치 설정
    weights = {
        'exact_match': float(rag_settings['exact_match_weight']),
        'partial_match': float(rag_settings['partial_match_weight']),
        'similarity': float(rag_settings['similarity_score_weight']),
        'structure': float(rag_settings['structure_score_weight']),
        'content': float(rag_settings['content_score_weight'])
    }
    
    # 임계값 설정
    thresholds = {
        'exact_match': float(rag_settings['exact_match_threshold']),
        'similarity': float(rag_settings['similarity_threshold']),
        'relevance': float(rag_settings['relevance_threshold']),
        'keyword_match': float(rag_settings['keyword_match_threshold']),
        'strong_match': float(rag_settings['strong_keyword_match_threshold'])
    }
    
    processed_results = []
    
    # 임계값 로깅
    logger.info("\n=== 문서 평가 결과 ===")
    logger.info("- 적용된 임계값:")
    logger.info(f"  * 유사도 임계값: {thresholds['similarity']:.2f}")
    logger.info(f"  * 키워드 매칭 임계값: {thresholds['keyword_match']:.2f}")
    logger.info(f"  * 강한 키워드 매칭 임계값: {thresholds['strong_match']:.2f}")
    logger.info(f"  * 관련성 점수 임계값: {thresholds['relevance']:.2f}")
    
    try:
        # 기존 처리 로직 유지
        for i, (metadata, doc, dist) in enumerate(zip(results['metadatas'][0], results['documents'][0], results['distances'][0]), 1):
            # 문서 내용이 None인 경우 건너뛰기
            if doc is None:
                continue
            
            # 현재 시간 가져오기
            current_time = time.time()
            current_date = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(current_time))
            
            # 메타데이터가 None인 경우 기본값 설정
            if metadata is None:
                metadata = {
                    'source': 'unknown',
                    'chunk_index': i,
                    'total_chunks': 1,
                    'timestamp': current_time,
                    'date': current_date,
                    'file_size': 0,
                    'quality_score': 0.0,
                    'user_id': 'all'
                }
            else:
                # 기존 메타데이터에 date 필드 추가
                if 'date' not in metadata:
                    timestamp = metadata.get('timestamp', current_time)
                    if isinstance(timestamp, str):
                        try:
                            timestamp = float(timestamp)
                        except (ValueError, TypeError):
                            timestamp = current_time
                    metadata['date'] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(timestamp))
            
            # 유사도 점수 계산 (0~1 범위로 정규화)
            similarity_score = min(1.0, max(0.0, 1.0 - dist))
            
            # 키워드 매칭 분석
            query_keywords = tokenize(query)
            content_keywords = tokenize(doc)
            matches = find_keyword_matches(query_keywords, content_keywords)
            
            # 정확한 매칭과 부분 매칭 분리
            exact_matches = [m for m in matches if m[2] == 1.0]
            partial_matches = [m for m in matches if m[2] < 1.0]
            
            # 키워드 품질 점수 (0~1 범위로 정규화)
            exact_match_score = len(exact_matches) / max(1, len(query_keywords))
            partial_match_score = sum(m[2] for m in partial_matches) / max(1, len(query_keywords) * 2)
            keyword_quality = min(1.0, exact_match_score + partial_match_score)
            
            # 구조 점수
            structure_score = calculate_structure_score(doc)
            
            # 내용 점수
            content_score = calculate_content_score(doc)
            
            # 길이 점수
            length_score = calculate_length_score(doc)
            
            # 최종 관련성 점수 계산 (0~1 범위로 정규화)
            relevance_score = min(1.0, (
                weights['similarity'] * similarity_score +
                weights['exact_match'] * exact_match_score +
                weights['partial_match'] * partial_match_score +
                weights['structure'] * structure_score +
                weights['content'] * content_score
            ))
            
            # 이미지 관련 필드 추출
            image_path = metadata.get('image_path')
            # source에서 파일명만 추출
            source_path = metadata.get('source', 'unknown')
            source_name = os.path.basename(source_path) if source_path != 'unknown' else 'unknown'
            result = {
                'content': doc,
                'source': source_name,
                'page_number': metadata.get('page_number'),
                'similarity_score': similarity_score,
                'relevance_score': relevance_score,
                'keyword_quality': keyword_quality,
                'structure_score': structure_score,
                'content_score': content_score,
                'length_score': length_score,
                'matches': {
                    'exact_matches': [{'query': m[0], 'content': m[1], 'score': m[2]} for m in exact_matches],
                    'partial_matches': [{'query': m[0], 'content': m[1], 'score': m[2]} for m in partial_matches]
                },
                'distance': dist,
                'user_id': metadata.get('user_id', 'all'),
                'timestamp': metadata.get('timestamp', time.time()),  # timestamp 필드 추가
                'thresholds': {
                    'exact_match': 0.3,
                    'keyword_match': 0.6,
                    'strong_match': 0.9,
                    'similarity': 0.5,
                    'relevance': 0.5
                },
                'main_weights': {
                    'exact_match': weights['exact_match'],
                    'similarity': weights['similarity'],
                    'relevance': weights['exact_match']
                },
                'sub_weights': {
                    'structure': weights['structure'],
                    'content': weights['content'],
                    'partial_match': weights['partial_match']
                },
                # 기타 메타데이터
                'user_id': metadata.get('user_id', 'all'),
                'final_score': metadata.get('final_score', 0)
            }
            if image_path:
                result['image_path'] = image_path
                result['type'] = metadata.get('type', 'text')
                result['parent_document'] = metadata.get('parent_document')
                result['timestamp'] = metadata.get('timestamp')
            processed_results.append(result)
        
        # 유사도와 관련성 점수 기반 필터링
        filtered_results = []
        rag_settings = get_rag_settings()
        logger.info("\n=== 문서 평가 결과 ===")
        
        # 임계값 정보 표시
        logger.info("- 적용된 임계값:")
        logger.info(f"  * 정확 매칭 임계값: {rag_settings['exact_match_threshold']}")
        logger.info(f"  * 유사도 임계값: {rag_settings['similarity_threshold']}")
        logger.info(f"  * 관련성 점수 임계값: {rag_settings['relevance_threshold']}")
        logger.info(f"  * 키워드 매칭 임계값: {rag_settings['keyword_match_threshold']}")
        logger.info(f"  * 강한 키워드 매칭 임계값: {rag_settings['strong_keyword_match_threshold']}")
        
        # 통과/실패한 문서 수 카운트
        passed_count = 0
        failed_count = 0
        
        for result in processed_results:
            file_name = os.path.basename(result['source'])
            source_path = result['source']
            sim_score = result['similarity_score']
            rel_score = result['relevance_score']
            
            # 파일 존재 여부 확인
            file_exists = os.path.exists(source_path) if source_path and source_path != 'unknown' else False
            
            # 판단 근거 결정
            reason = []
            logger.info(f"\n[문서] {file_name}")
            logger.info(f"- 파일 경로: {source_path}")
            logger.info(f"- 파일 존재: {'예' if file_exists else '아니오'}")
            
            # 정확 매칭 점수 계산
            exact_match_score = len(result['matches']['exact_matches']) / len(tokenize(query)) if tokenize(query) else 0
            
            logger.info(f"- 정확 매칭: {exact_match_score:.2f} (임계값: {thresholds['exact_match']:.2f})")
            logger.info(f"- 유사도: {sim_score:.2f} (임계값: {thresholds['similarity']:.2f})")
            logger.info(f"- 관련성: {rel_score:.2f} (임계값: {thresholds['relevance']:.2f})")
            
            # 파일이 존재하지 않는 경우 제외
            if not file_exists:
                logger.info("→ 결과 제외 (파일이 존재하지 않음)")
                failed_count += 1
                continue
            
            if exact_match_score >= thresholds['exact_match']:
                reason.append("정확 매칭 통과")
            if sim_score >= thresholds['similarity']:
                reason.append("유사도 통과")
            if rel_score >= thresholds['relevance']:
                reason.append("관련성 통과")
                
            if reason:
                logger.info(f"- 판단 근거: {', '.join(reason)}")
                
            # 정확 매칭 점수 계산 (중복 제거)
            # exact_match_score는 이미 위에서 계산됨
            
            # 최종 판단: (정확 매칭 OR 유사도) AND 관련성 조건으로 통과해야 함
            # 정확 매칭이나 유사도 중 하나는 통과해야 하고, 관련성은 반드시 통과해야 함
            if ((exact_match_score >= thresholds['exact_match'] or 
                 sim_score >= thresholds['similarity']) and 
                rel_score >= thresholds['relevance']):
                filtered_results.append(result)
                logger.info("→ 결과 포함")
                passed_count += 1
            else:
                logger.info("→ 결과 제외 (임계값 미달)")
                failed_count += 1
                
            # 이미 위에서 필터링 완료되었으므로 이 중복 코드는 제거
        
        # 최종 결과 요약
        logger.info(f"\n최종 결과: 총 {len(processed_results)}개 중 {passed_count}개 통과, {failed_count}개 제외")
        
        def get_sort_key(result):
            # 수정일자 처리
            modification_date = None
            source_path = result.get('source')
            
            if source_path and os.path.exists(source_path):
                # 실제 파일의 수정일자 사용
                modification_date = os.path.getmtime(source_path)
            else:
                # 파일이 없는 경우 메타데이터에서 수정일자 확인
                if 'metadata' in result:
                    modification_date = (
                        result['metadata'].get('modification_date') or
                        result['metadata'].get('file_mtime') or
                        result['metadata'].get('timestamp') or
                        time.time()
                    )
                else:
                    modification_date = (
                        result.get('modification_date') or
                        result.get('file_mtime') or
                        result.get('timestamp') or
                        time.time()
                    )
            
            # 수정일자를 datetime 객체로 변환
            try:
                from datetime import datetime
                if isinstance(modification_date, (int, float)):
                    mod_date_dt = datetime.fromtimestamp(modification_date)
                elif isinstance(modification_date, str):
                    # ISO 형식 문자열인 경우
                    if 'T' in modification_date:
                        mod_date_dt = datetime.fromisoformat(modification_date.replace('Z', '+00:00'))
                    else:
                        mod_date_dt = datetime.strptime(modification_date, '%Y-%m-%d %H:%M:%S')
                else:
                    mod_date_dt = datetime.fromtimestamp(time.time())
                
                # 최신 문서가 우선되도록 음수로 변환
                date_score = -mod_date_dt.timestamp()
            except:
                date_score = 0
            
            # 유사도 점수 (높을수록 우선)
            sim_score = result.get('similarity_score', 0)
            
            # 관련성 점수 (높을수록 우선)
            rel_score = result.get('relevance_score', 0)
            
            # 종합 점수 계산 (설정된 가중치 사용)
            settings = get_rag_settings()
            total_score = (
                sim_score * settings['similarity_score_weight'] +
                rel_score * settings['relevance_score_weight'] +
                date_score * settings['date_weight']
            )
            return total_score
            
        # 종합 점수 기준으로 내림차순 정렬
        filtered_results.sort(key=get_sort_key, reverse=True)
        
        # 정렬 결과 로깅
        logger.info("\n최종 검색 결과:")
        for idx, result in enumerate(filtered_results[:5], 1):  # 상위 5개 결과만 로깅
            # 수정일자 처리
            modification_date = None
            source_path = result.get('source')
            
            if source_path and os.path.exists(source_path):
                # 실제 파일의 수정일자 사용
                modification_date = os.path.getmtime(source_path)
                try:
                    from datetime import datetime
                    modification_date = datetime.fromtimestamp(modification_date).strftime('%Y-%m-%d %H:%M:%S')
                except:
                    modification_date = '날짜 변환 실패'
            else:
                # 파일이 없는 경우 메타데이터에서 수정일자 확인
                if 'metadata' in result:
                    modification_date = (
                        result['metadata'].get('modification_date') or
                        result['metadata'].get('file_mtime') or
                        result['metadata'].get('timestamp') or
                        '날짜 없음'
                    )
                else:
                    modification_date = (
                        result.get('modification_date') or
                        result.get('file_mtime') or
                        result.get('timestamp') or
                        '날짜 없음'
                    )
                
                # ISO 형식 문자열을 일반 형식으로 변환
                try:
                    from datetime import datetime
                    if isinstance(modification_date, str) and 'T' in modification_date:
                        dt = datetime.fromisoformat(modification_date.replace('Z', '+00:00'))
                        modification_date = dt.strftime('%Y-%m-%d %H:%M:%S')
                except:
                    pass
            
            sim_score = result.get('similarity_score', 0)
            rel_score = result.get('relevance_score', 0)
            logger.info(f"{idx}위: {result.get('source', '파일명 없음')}")
            logger.info(f"  - 유사도: {sim_score:.2f}")
            logger.info(f"  - 관련성: {rel_score:.2f}")
            logger.info(f"  - 수정일: {modification_date}")            
        
        return filtered_results
        
    except Exception as e:
        logger.error(f"검색 오류: {str(e)}")
        return {
            "status": "success",
            "results": [],
            "cache_hit": False
        }

# Helper 함수들 추가
def find_keyword_matches(query_keywords: List[str], content_keywords: List[str]) -> List[tuple]:
    """키워드 매칭 쌍과 점수를 찾는 함수"""
    matches = []
    matched_content_keywords = set()
    
    # RAG 설정 가져오기
    rag_settings = get_rag_settings()
    
    for qk in query_keywords:
        best_match = None
        best_score = 0.0
        
        for ck in content_keywords:
            if ck in matched_content_keywords:
                continue
                
            # 토큰화
            q_tokens = set(qk.split())
            c_tokens = set(ck.split())
            
            # 토큰 기반 매칭
            if q_tokens and c_tokens:
                common_tokens = q_tokens & c_tokens
                if common_tokens:
                    # 토큰 매칭 점수 계산
                    token_score = len(common_tokens) / len(q_tokens) * float(rag_settings['exact_match_weight'])
                    
                    # 연속 토큰 보너스
                    q_list = qk.split()
                    c_list = ck.split()
                    max_consecutive = 0
                    for i in range(len(q_list)):
                        for j in range(len(c_list)):
                            consecutive = 0
                            while (i + consecutive < len(q_list) and 
                                   j + consecutive < len(c_list) and 
                                   q_list[i + consecutive] == c_list[j + consecutive]):
                                consecutive += 1
                            max_consecutive = max(max_consecutive, consecutive)
                    
                    consecutive_bonus = float(rag_settings['partial_match_weight']) * max_consecutive
                    
                    # 포함 관계 보너스
                    contains_bonus = 0
                    if qk in ck:
                        contains_bonus = float(rag_settings['keyword_quality_weight']) * (len(qk) / len(ck))
                    elif ck in qk:
                        contains_bonus = float(rag_settings['keyword_quality_weight']) * (len(ck) / len(qk))
                    
                    # 유사도 보너스
                    similarity_bonus = 0
                    if len(qk) > 2 and len(ck) > 2:
                        from difflib import SequenceMatcher
                        similarity = SequenceMatcher(None, qk, ck).ratio()
                        similarity_bonus = float(rag_settings['similarity_score_weight']) * similarity
                    
                    score = token_score + consecutive_bonus + contains_bonus + similarity_bonus
                    if score > best_score:
                        best_score = score
                        best_match = (qk, ck, min(0.98, score))  # 최대 점수는 0.98로 제한
            
            if best_match and best_match[2] >= float(rag_settings['keyword_match_threshold']):
                matches.append(best_match)
                matched_content_keywords.add(best_match[1])
        
    return matches

def calculate_structure_score(doc: str) -> float:
    """구조적 품질 점수 계산"""
    if doc is None or not isinstance(doc, str):
        return 0.0
            
    # RAG 설정 가져오기
    rag_settings = get_rag_settings()
    
    # 기본 점수 (구조 점수 가중치의 절반)
    score = float(rag_settings['structure_score_weight']) / 2
    
    # 제목 구조 점수 (구조 점수 가중치의 비율로 계산)
    title_weight = float(rag_settings['structure_score_weight'])
    if re.search(r'^(?:제\s*)?[0-9０-９]+[.\s]*장\s+', doc):  # 대제목
        score += title_weight * 0.75
    elif re.search(r'^(?:제\s*)?[0-9０-９]+[.\s]*절\s+', doc):  # 중제목
        score += title_weight * 0.5
    elif re.search(r'^[0-9０-９]+[.][0-9０-９]+[.]*\s+[^\n]+', doc):  # 소제목
        score += title_weight * 0.25
    
    # 문단 구조 점수 (구조 점수 가중치의 비율로 계산)
    has_paragraphs = '\n\n' in doc
    has_sentences = len(re.findall(r'[.!?]+', doc)) > 1
    if has_paragraphs: score += title_weight * 0.5
    if has_sentences: score += title_weight * 0.25
    
    return min(1.0, score)

def calculate_content_score(doc: str) -> float:
    """문서 내용 점수 계산"""
    score = 0.5
    if len(doc.strip()) > 100:  # 최소 길이
        score += 0.2
    if not re.search(r'[^\w\s가-힣]', doc):  # 특수문자 비율
        score += 0.3
    return min(1.0, score)

def calculate_length_score(doc: str) -> float:
    """문서 길이 점수 계산"""
    return min(1.0, len(doc) / 1000)

def to_builtin(obj: Any):
    """np.* → python 기본 타입으로 재귀 변환"""
    if isinstance(obj, np.generic):          # np.float32, np.int64 ...
        return obj.item()
    if isinstance(obj, np.ndarray):          # 배열
        return obj.tolist()
    if isinstance(obj, (list, tuple, set)):  # 시퀀스
        return [to_builtin(x) for x in obj]
    if isinstance(obj, dict):                # 매핑
        return {k: to_builtin(v) for k, v in obj.items()}
    return obj


#!/usr/bin/env python3

import os
import sys
import time
import json
import psycopg2
from psycopg2 import pool
import shutil
import asyncio
import logging
import traceback
import threading
import warnings
import glob
import queue
import importlib.util
import random
import re
from threading import Thread, Lock
from datetime import datetime, timezone, timedelta
from typing import List, Dict, Any, Optional
from fastapi import FastAPI, File, UploadFile, Form, HTTPException, Query, Request
from fastapi.responses import JSONResponse
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
from setproctitle import setproctitle
import uvicorn
import redis
import concurrent.futures
from functools import partial
import asyncio

from multiprocessing import Process, Queue, Manager, Event
import uuid
from inotify_simple import INotify, flags
from contextlib import asynccontextmanager
import numpy as np
import torch
from transformers import AutoModel, AutoTokenizer
from langchain_huggingface import HuggingFaceEmbeddings
from langchain_text_splitters import RecursiveCharacterTextSplitter
import hashlib
import pandas as pd
import multiprocessing
import psutil
import math
from langchain_experimental.text_splitter import SemanticChunker
from sentence_transformers import SentenceTransformer
from pathlib import Path
from langchain.embeddings.base import Embeddings
import re
from dataclasses import dataclass
import httpx
import aiohttp
import gc
import uuid
import csv
import io
import tempfile
import base64

# utils 모듈 임포트 - 프로젝트 루트의 .py 파일 우선
project_root = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
utils_py_file = os.path.join(project_root, "utils.py")

print(f"[DEBUG] Checking for utils.py file: {utils_py_file}")
print(f"[DEBUG] utils.py file exists: {os.path.exists(utils_py_file)}")

if os.path.exists(utils_py_file):
    # 프로젝트 루트의 utils.py 파일이 있으면 우선 사용 (개발 모드)
    print(f"[DEBUG] Using Python source utils.py from project root")
    sys.path.insert(0, project_root)
    import utils
else:
    # .py 파일이 없으면 컴파일된 버전 사용
    print(f"[DEBUG] Python source utils.py not found, trying compiled version")
    try:
        import utils
    except ImportError:
        # 현재 디렉토리의 상위 디렉토리를 Python 경로에 추가
        sys.path.insert(0, project_root)
        import utils

# utils 모듈에서 필요한 함수들을 가져옵니다
extract_from_pdf = utils.extract_from_pdf
extract_from_hwp = utils.extract_from_hwp
extract_from_doc = utils.extract_from_doc
extract_from_ppt = utils.extract_from_ppt
convert_hwp_to_pdf = utils.convert_hwp_to_pdf
extract_from_hwp_hwp2txt = utils.extract_from_hwp_hwp2txt
clean_hwp_text = utils.clean_hwp_text

# 모든 경고 무시
warnings.filterwarnings('ignore', category=DeprecationWarning)
warnings.filterwarnings('ignore', category=UserWarning)
warnings.filterwarnings('ignore', category=FutureWarning)

# 현재 스크립트 경로 가져오기
script_dir = os.path.dirname(os.path.abspath(__file__))

# 먼저 .py 파일이 있는지 확인 (개발 모드 우선)
py_file = os.path.join(script_dir, "rag_process.py")
print(f"[DEBUG] Checking for Python file: {py_file}")
print(f"[DEBUG] Python file exists: {os.path.exists(py_file)}")

if os.path.exists(py_file):
    # .py 파일이 있으면 우선 사용 (개발 모드)
    module_path = py_file
    print(f"[DEBUG] Using Python source file: {os.path.basename(py_file)}")
else:
    # .py 파일이 없으면 컴파일된 파일 찾기
    print(f"[DEBUG] Python source file not found, looking for compiled module")
    
    # OS별 모듈 패턴 정의
    if sys.platform.startswith('win'):
        module_pattern = os.path.join(script_dir, "rag_process.cp*-win_amd64.pyd")
    elif sys.platform.startswith('darwin'):
        module_pattern = os.path.join(script_dir, "rag_process.cpython-*-darwin.so")
    else:  # Linux
        module_pattern = os.path.join(script_dir, "rag_process.cpython-312-x86_64-linux-gnu.so")
    
    print(f"[DEBUG] Module pattern: {module_pattern}")
    
    # .so/.pyd 파일 찾기
    module_files = glob.glob(module_pattern)
    print(f"[DEBUG] Found module files: {module_files}")
    
    if not module_files:
        # 정확한 파일명으로 시도
        exact_file = os.path.join(script_dir, "rag_process.cpython-312-x86_64-linux-gnu.so")
        print(f"[DEBUG] Trying exact file: {exact_file}")
        print(f"[DEBUG] Exact file exists: {os.path.exists(exact_file)}")
        if os.path.exists(exact_file):
            module_files = [exact_file]
            print(f"[DEBUG] Using compiled module: {os.path.basename(exact_file)}")
        else:
            # 컴파일된 모듈을 찾지 못한 경우 에러
            raise ImportError(f"No module found. Tried Python file: {py_file}, compiled patterns: {module_pattern}, {exact_file}")
    
    module_path = module_files[0]

# 모듈 로드
spec = importlib.util.spec_from_file_location("rag_process", module_path)
if spec is None:
    raise ImportError(f"Failed to create module spec for {module_path}")

module = importlib.util.module_from_spec(spec)
sys.modules["rag_process"] = module
spec.loader.exec_module(module)



# FastAPI 엔드포인트 추가
@app.post("/batch-search")
async def batch_search(request: Request):
    """여러 키워드로 빠른 LIKE 검색만 수행 (임베딩 없이)"""
    try:
        data = await request.json()
        keywords = data.get('keywords', [])
        settings = data.get('settings', {})
        user_id = data.get('user_id', 'admin')
        
        if not keywords:
            return {"success": False, "error": "No keywords provided"}
        
        logger.info(f"배치 검색 시작: {len(keywords)}개 키워드 (LIKE 검색만)")
        
        # 데이터베이스 직접 연결
        import psycopg2
        from psycopg2.extras import RealDictCursor
        
        conn = get_pool_connection()
        cursor = conn.cursor(cursor_factory=RealDictCursor)
        
        results = []
        top_k = settings.get('top_k', 5)  # 설정된 값 그대로 사용
        
        # 각 키워드에 대해 빠른 LIKE 검색 수행 (최대 7개)
        for idx, keyword in enumerate(keywords[:7]):
            try:
                logger.info(f"LIKE 검색 중 ({idx+1}/{min(len(keywords), 7)}): '{keyword}'")
                
                # 키워드를 포함하는 문서 검색 (LIKE 검색)
                query = """
                    SELECT 
                        chunk_text as content,
                        metadata->>'file_name' as file_name,
                        COALESCE((metadata->>'chunk_index')::int, 0) as chunk_index,
                        LENGTH(chunk_text) as content_length,
                        created_at
                    FROM document_embeddings
                    WHERE user_id = %s 
                    AND LOWER(chunk_text) LIKE LOWER(%s)
                    ORDER BY created_at DESC
                    LIMIT %s
                """
                
                search_pattern = f'%{keyword}%'
                cursor.execute(query, (user_id, search_pattern, top_k))
                rows = cursor.fetchall()
                
                if rows:
                    # 간단한 점수 계산 (키워드 출현 빈도 기반)
                    scores = []
                    for row in rows:
                        content = row['content'].lower() if row['content'] else ''
                        keyword_lower = keyword.lower()
                        # 키워드 출현 횟수
                        occurrences = content.count(keyword_lower)
                        content_length = row['content_length']
                        # TF 스코어 (Term Frequency)
                        tf_score = occurrences / max(content_length / 100, 1) if content_length > 0 else 0
                        # 0-1 범위로 정규화
                        score = min(1.0, tf_score)
                        scores.append(score)
                    
                    avg_score = sum(scores) / len(scores) if scores else 0
                    
                    results.append({
                        "keyword": keyword,
                        "avgScore": avg_score * 100,  # 100점 만점
                        "numResults": len(rows),
                        "topScore": max(scores) * 100 if scores else 0
                    })
                    logger.info(f"✓ '{keyword}': {len(rows)}개 결과, 평균 점수: {avg_score*100:.1f}")
                else:
                    results.append({
                        "keyword": keyword,
                        "avgScore": 0,
                        "numResults": 0,
                        "topScore": 0
                    })
                    logger.info(f"✗ '{keyword}': 결과 없음")
                    
            except Exception as e:
                logger.error(f"키워드 '{keyword}' 검색 실패: {e}")
                results.append({
                    "keyword": keyword,
                    "avgScore": 0,
                    "numResults": 0,
                    "topScore": 0,
                    "error": str(e)
                })
        
        cursor.close()
        return_pool_connection(conn)
        
        logger.info(f"배치 검색 완료: {len(results)}개 결과")
        
        # 전체 평균 점수 계산
        total_avg = sum(r["avgScore"] for r in results) / len(results) if results else 0
        
        return {
            "success": True,
            "results": results,
            "summary": {
                "total_keywords": len(keywords[:7]),
                "total_searches": len(results),
                "average_score": total_avg
            }
        }
        
    except Exception as e:
        logger.error(f"배치 검색 실패: {e}")
        import traceback
        traceback.print_exc()
        if 'conn' in locals():
            conn.close()
        return {
            "success": False,
            "error": str(e)
        }

# GraphRAG 전용 요청/응답 모델
class GraphRAGProcessRequest(BaseModel):
    file_path: str
    user_id: str
    options: Optional[Dict[str, Any]] = {}

class GraphRAGSearchRequest(BaseModel):
    query: str
    user_id: str
    mode: Optional[str] = "hybrid"  # hybrid, vector, graph
    top_k: Optional[int] = 5

@app.post("/graphrag/process")
async def process_document_graphrag(request: GraphRAGProcessRequest):
    """
    Microsoft GraphRAG 지식 그래프 구축 엔드포인트
    문서에서 엔티티와 관계를 추출하여 실제 지식 그래프를 구축합니다.
    """
    try:
        # 런타임에 직접 GraphRAGProcessor 클래스 찾기
        import sys
        import importlib.util
        
        GraphRAGProcessorClass = None
        
        # GraphRAG 클래스 강제 로드 (파일 끝에 정의된 클래스)
        if 'GraphRAGProcessor' not in globals():
            try:
                # 파일 끝 부분의 GraphRAGProcessor 클래스 정의를 강제로 실행
                with open(__file__, 'r', encoding='utf-8') as f:
                    lines = f.readlines()
                
                # KnowledgeGraph + GraphRAGProcessor 클래스 모두 로드
                kg_start = None
                grag_start = None
                
                for i, line in enumerate(lines):
                    if line.strip().startswith('class KnowledgeGraph:'):
                        kg_start = i
                    elif line.strip().startswith('class GraphRAGProcessor:'):
                        grag_start = i
                        break
                
                if kg_start and grag_start:
                    # KnowledgeGraph부터 파일 끝까지 모든 클래스 로드
                    class_code = ''.join(lines[kg_start:])
                    exec(class_code, globals())
                    logger.info("✅ KnowledgeGraph + GraphRAGProcessor 클래스 강제 로드 완료")
            except Exception as e:
                logger.info(f"GraphRAG 클래스 강제 로드 실패: {str(e)}", error=True)
        
        # 클래스 찾기
        if 'GraphRAGProcessor' in globals():
            GraphRAGProcessorClass = globals()['GraphRAGProcessor']
            logger.info("✅ GraphRAGProcessor 클래스 발견")
        else:
            logger.error("❌ GraphRAGProcessor 클래스를 찾을 수 없음")
        
        if not GraphRAGProcessorClass:
            return JSONResponse(
                content={
                    "status": "error", 
                    "message": "GraphRAG processor class not found. Service may still be loading."
                },
                status_code=503
            )
        
        # GraphRAGProcessor 인스턴스 생성
        graphrag_processor = GraphRAGProcessorClass()
        logger.info("✅ GraphRAGProcessor 인스턴스 생성 완료")
        
        logger.info(f"GraphRAG 문서 처리 요청: {request.file_path}")
        
        # 파일 존재 확인
        if not os.path.exists(request.file_path):
            return JSONResponse(
                content={
                    "status": "error",
                    "message": f"File not found: {request.file_path}"
                },
                status_code=404
            )
        
        # GraphRAG 처리 실행
        result = await graphrag_processor.process_document_for_graphrag(
            request.file_path, 
            request.user_id
        )
        
        return JSONResponse(
            content={
                "status": "success",
                "data": {
                    "entities": result['entities'],
                    "relationships": result['relationships'],
                    "entity_count": len(result['entities']),
                    "relationship_count": len(result['relationships']),
                    "embedding_dimensions": len(result['embeddings'][0]) if result['embeddings'] else 0,
                    "metadata": result['metadata']
                }
            }
        )
        
    except Exception as e:
        logger.info(f"GraphRAG processing error: {str(e)}", error=True)
        return JSONResponse(
            content={
                "status": "error",
                "message": str(e)
            },
            status_code=500
        )

@app.post("/graphrag/search")
async def search_graphrag(request: GraphRAGSearchRequest):
    """
    Microsoft GraphRAG 검색 엔드포인트
    - graph: 지식 그래프 기반 관계형 추론 검색
    - hybrid: 벡터 검색 + 지식 그래프 검색 통합
    - vector: 기존 벡터 검색 (폴백)
    """
    try:
        # GraphRAG 클래스 강제 로드 (파일 끝에 정의된 클래스)
        import sys
        GraphRAGProcessorClass = None
        
        if 'GraphRAGProcessor' not in globals():
            try:
                # 파일 끝 부분의 GraphRAGProcessor 클래스 정의를 강제로 실행
                with open(__file__, 'r', encoding='utf-8') as f:
                    lines = f.readlines()
                
                # KnowledgeGraph + GraphRAGProcessor 클래스 모두 로드
                kg_start = None
                grag_start = None
                
                for i, line in enumerate(lines):
                    if line.strip().startswith('class KnowledgeGraph:'):
                        kg_start = i
                    elif line.strip().startswith('class GraphRAGProcessor:'):
                        grag_start = i
                        break
                
                if kg_start and grag_start:
                    # KnowledgeGraph부터 파일 끝까지 모든 클래스 로드
                    class_code = ''.join(lines[kg_start:])
                    exec(class_code, globals())
                    logger.info("✅ GraphRAG 검색용 KnowledgeGraph + GraphRAGProcessor 강제 로드 완료")
            except Exception as e:
                logger.info(f"GraphRAG 클래스 강제 로드 실패: {str(e)}", error=True)
        
        # 클래스 찾기
        if 'GraphRAGProcessor' in globals():
            GraphRAGProcessorClass = globals()['GraphRAGProcessor']
        
        # GraphRAG 클래스가 없더라도 GraphRAG 스타일 검색 구현
        if not GraphRAGProcessorClass:
            logger.info("💡 GraphRAG 클래스 없음 - GraphRAG 스타일 벡터 검색으로 대체")
            # 기본 벡터 검색 + GraphRAG 스타일 응답
            fallback_request = SearchRequest(
                query=request.query,
                user_id=request.user_id,
                top_k=request.top_k
            )
            vector_results = await search_documents(fallback_request)
            
            # GraphRAG 스타일 응답으로 변환
            return JSONResponse(
                content={
                    "status": "success",
                    "search_mode": request.mode,
                    "results": vector_results.get('results', []),
                    "total_entities": len(vector_results.get('results', [])),
                    "total_relationships": 0,
                    "note": f"GraphRAG 대체: 벡터 검색 + 엔티티 메타데이터 ({request.mode} 모드)"
                }
            )
        
        # GraphRAGProcessor 인스턴스 생성
        graphrag_processor = GraphRAGProcessorClass()
        logger.info("✅ GraphRAG 검색용 프로세서 생성 완료")
        
        logger.info(f"GraphRAG 검색 요청: {request.query} (모드: {request.mode})")
        
        start_time = time.time()
        
        # 모드에 따른 검색 실행
        if request.mode == "vector":
            # 기존 벡터 검색 사용
            fallback_request = SearchRequest(
                query=request.query,
                user_id=request.user_id,
                top_k=request.top_k
            )
            vector_result = await search_documents(fallback_request)
            search_time = time.time() - start_time
            
            # GraphRAG 형식으로 응답 변환
            return JSONResponse(
                content={
                    "status": "success",
                    "search_mode": "vector",
                    "search_time": round(search_time, 2),
                    "results": vector_result,
                    "entities": [],
                    "relationships": []
                }
            )
            
        elif request.mode == "graph":
            # 실제 그래프 검색 실행
            graph_result = await graphrag_processor.search_graph(
                request.query, 
                request.user_id, 
                mode="graph", 
                top_k=request.top_k
            )
            
            search_time = time.time() - start_time
            
            return JSONResponse(
                content={
                    "status": "success",
                    "search_mode": "graph",
                    "search_time": round(search_time, 3),
                    "results": graph_result.get('results', []),
                    "total_entities": graph_result.get('total_entities', 0),
                    "total_relationships": graph_result.get('total_relationships', 0),
                    "note": "실제 지식 그래프 기반 관계형 추론 검색"
                }
            )
            
        else:  # hybrid mode (기본값)
            # 실제 하이브리드 검색: 그래프 + 벡터
            # 1. 벡터 검색 실행
            fallback_request = SearchRequest(
                query=request.query,
                user_id=request.user_id,
                top_k=request.top_k
            )
            vector_result = await search_documents(fallback_request)
            
            # 2. 그래프 검색 실행
            graph_result = await graphrag_processor.search_graph(
                request.query, 
                request.user_id, 
                mode="hybrid", 
                top_k=request.top_k
            )
            
            search_time = time.time() - start_time
            
            return JSONResponse(
                content={
                    "status": "success",
                    "search_mode": "hybrid",
                    "search_time": round(search_time, 3),
                    "vector_results": vector_result,
                    "graph_results": graph_result.get('graph_results', []),
                    "total_entities": graph_result.get('total_entities', 0),
                    "total_relationships": graph_result.get('total_relationships', 0),
                    "note": "진짜 하이브리드: 벡터 검색 + 지식 그래프 관계형 추론"
                }
            )
        
    except Exception as e:
        logger.info(f"GraphRAG search error: {str(e)}", error=True)
        return JSONResponse(
            content={
                "status": "error",
                "message": str(e)
            },
            status_code=500
        )

@app.get("/graphrag/status")
async def get_graphrag_status():
    """GraphRAG 시스템 상태 확인"""
    try:
        # GraphRAG 검색과 동일하게 새로운 인스턴스 생성하여 실제 데이터 로딩
        GraphRAGProcessorClass = None
        
        # GraphRAG 클래스 강제 로드 (파일 끝에 정의된 클래스)
        if 'GraphRAGProcessor' not in globals():
            try:
                # 파일 끝 부분의 GraphRAGProcessor 클래스 정의를 강제로 실행
                with open(__file__, 'r', encoding='utf-8') as f:
                    lines = f.readlines()
                
                # KnowledgeGraph + GraphRAGProcessor 클래스 모두 로드
                kg_start = None
                grag_start = None
                
                for i, line in enumerate(lines):
                    if line.strip().startswith('class KnowledgeGraph:'):
                        kg_start = i
                    elif line.strip().startswith('class GraphRAGProcessor:'):
                        grag_start = i
                        break
                
                if kg_start and grag_start:
                    # KnowledgeGraph부터 파일 끝까지 모든 클래스 로드
                    class_code = ''.join(lines[kg_start:])
                    exec(class_code, globals())
                    logger.info("✅ GraphRAG 상태용 KnowledgeGraph + GraphRAGProcessor 강제 로드 완료")
            except Exception as e:
                logger.info(f"GraphRAG 클래스 강제 로드 실패: {str(e)}", error=True)
        
        # 클래스 찾기
        if 'GraphRAGProcessor' in globals():
            GraphRAGProcessorClass = globals()['GraphRAGProcessor']
        
        if not GraphRAGProcessorClass:
            graphrag_processor = None
        else:
            # 검색과 동일하게 새로운 인스턴스 생성
            graphrag_processor = GraphRAGProcessorClass()
        
        status = {
            "graphrag_enabled": graphrag_processor is not None,
            "dependencies_available": HAS_GRAPHRAG_DEPS,
            "ollama_server": None,
            "models": {
                "chat_model": None,
                "embedding_model": None
            }
        }
        
        if graphrag_processor:
            status["ollama_server"] = graphrag_processor.ollama_server
            status["models"]["chat_model"] = graphrag_processor.chat_model
            status["models"]["embedding_model"] = graphrag_processor.embedding_model
            
            # 지식 그래프 통계 추가
            status["knowledge_graph"] = {
                "total_entities": len(graphrag_processor.knowledge_graph.entities),
                "total_relationships": len(graphrag_processor.knowledge_graph.relationships),
                "users_with_documents": len(graphrag_processor.document_graphs),
                "core_features": {
                    "entity_extraction": True,
                    "knowledge_graph_construction": True,
                    "relational_inference_search": True,
                    "multi_hop_traversal": True
                }
            }
        
        return JSONResponse(content={"status": "success", "data": status})
        
    except Exception as e:
        logger.info(f"GraphRAG status error: {str(e)}", error=True)
        return JSONResponse(
            content={"status": "error", "message": str(e)},
            status_code=500
        )

class MultiHopRequest(BaseModel):
    start_entity: str
    end_entity: str
    max_hops: Optional[int] = 3

class EntityRelationshipRequest(BaseModel):
    entity_name: str
    max_distance: Optional[int] = 2

@app.post("/graphrag/multihop")
async def find_multihop_paths(request: MultiHopRequest):
    """멀티홉 경로 탐색 엔드포인트"""
    try:
        # GraphRAG 프로세서 싱글톤 인스턴스 가져오기
        graphrag_processor = get_or_create_graphrag_processor()
        
        if not graphrag_processor:
            return JSONResponse(
                content={
                    "status": "error",
                    "message": "GraphRAG processor not initialized"
                },
                status_code=400
            )
        
        logger.info(f"멀티홉 경로 탐색: {request.start_entity} -> {request.end_entity} (최대 {request.max_hops}홉)")
        
        paths = graphrag_processor.get_multi_hop_paths(
            request.start_entity,
            request.end_entity,
            request.max_hops
        )
        
        return JSONResponse(
            content={
                "status": "success",
                "start_entity": request.start_entity,
                "end_entity": request.end_entity,
                "max_hops": request.max_hops,
                "paths_found": len(paths),
                "paths": paths,
                "note": "실제 멀티홉 그래프 탐색 결과"
            }
        )
        
    except Exception as e:
        logger.info(f"MultiHop path error: {str(e)}", error=True)
        return JSONResponse(
            content={"status": "error", "message": str(e)},
            status_code=500
        )

@app.post("/graphrag/entity-relations")
async def get_entity_relations(request: EntityRelationshipRequest):
    """특정 엔티티의 관계형 정보 조회 엔드포인트"""
    try:
        # GraphRAG 프로세서 싱글톤 인스턴스 가져오기
        graphrag_processor = get_or_create_graphrag_processor()
        
        if not graphrag_processor:
            return JSONResponse(
                content={
                    "status": "error",
                    "message": "GraphRAG processor not initialized"
                },
                status_code=400
            )
        
        logger.info(f"엔티티 관계 조회: {request.entity_name} (최대 거리: {request.max_distance})")
        
        relationships = graphrag_processor.get_entity_relationships(
            request.entity_name,
            request.max_distance
        )
        
        total_related = len(relationships) if isinstance(relationships, list) else sum(len(entities) for entities in relationships.values())
        
        return JSONResponse(
            content={
                "status": "success",
                "entity_name": request.entity_name,
                "max_distance": request.max_distance,
                "total_related_entities": total_related,
                "relationships": relationships,
                "note": "관계형 추론을 통한 연관 엔티티 탐색"
            }
        )
        
    except Exception as e:
        logger.info(f"Entity relations error: {str(e)}", error=True)
        return JSONResponse(
            content={"status": "error", "message": str(e)},
            status_code=500
        )

@app.post("/graphrag/chunk-entities")
async def get_chunk_entities(request: Request):
    """특정 문서의 청크에서 엔티티를 추출하여 반환"""
    try:
        body = await request.json()
        filename = body.get('filename')
        chunk_index = body.get('chunk_index')
        user_id = body.get('user_id', 'admin')

        if not filename or chunk_index is None:
            return JSONResponse(
                status_code=400,
                content={'error': 'filename과 chunk_index가 필요합니다'}
            )

        logger.info(f"청크 엔티티 추출 요청: {filename}, chunk_index={chunk_index}, user_id={user_id}")

        # 1. document_embeddings에서 해당 청크의 텍스트 가져오기
        conn = get_pool_connection()
        try:
            with conn.cursor() as cur:
                cur.execute("""
                    SELECT chunk_text, metadata
                    FROM document_embeddings
                    WHERE (filename = %s OR filename LIKE %s OR filename LIKE %s)
                      AND chunk_index = %s AND user_id = %s
                    LIMIT 1
                """, (filename, f'%/{filename}', f'%{filename}', chunk_index, user_id))
                chunk_result = cur.fetchone()

                if not chunk_result:
                    return JSONResponse(
                        status_code=404,
                        content={'error': f'청크를 찾을 수 없습니다: {filename} chunk {chunk_index}'}
                    )

                chunk_text = chunk_result[0]
                chunk_metadata = chunk_result[1] or {}

                logger.info(f"청크 텍스트 길이: {len(chunk_text)}")

                # 2. 청크 텍스트에서 키워드 추출
                keywords = extract_keywords_from_text(chunk_text)
                logger.info(f"추출된 키워드: {keywords[:10]}")  # 상위 10개만 로깅

                # 3. 키워드와 관련된 엔티티들을 graph_entities에서 검색
                if not keywords:
                    return JSONResponse(content={
                        'success': True,
                        'data': {
                            'entities': [],
                            'relationships': [],
                            'communities': [],
                            'chunk_info': {
                                'filename': filename,
                                'chunk_index': chunk_index,
                                'chunk_text_length': len(chunk_text),
                                'keywords': []
                            }
                        }
                    })

                # 키워드들로 엔티티 검색 (OR 조건)
                keyword_conditions = []
                params = []
                for i, keyword in enumerate(keywords[:20]):  # 상위 20개 키워드만 사용
                    keyword_conditions.append(f"""
                        (LOWER(e.name) LIKE LOWER(%s) OR
                         LOWER(e.description) LIKE LOWER(%s))
                    """)
                    params.extend([f'%{keyword}%', f'%{keyword}%'])

                # source_documents에 해당 파일명이 포함된 엔티티도 검색
                keyword_conditions.append("(%s = ANY(e.source_documents))")
                params.append(filename.replace('.txt', '').replace('.pdf', ''))

                entity_query = f"""
                    SELECT DISTINCT
                        e.id,
                        e.name as entity_name,
                        e.type as entity_type,
                        e.description as entity_description,
                        e.source_documents,
                        e.properties
                    FROM graph_entities e
                    WHERE ({' OR '.join(keyword_conditions)})
                    ORDER BY e.name
                    LIMIT 50
                """

                cur.execute(entity_query, params)
                entity_results = cur.fetchall()

                # 4. 찾은 엔티티들의 관계 조회
                entity_ids = [row[0] for row in entity_results]
                relationships = []
                communities = []

                if entity_ids:
                    # 관계 조회
                    relationship_query = """
                        SELECT DISTINCT
                            r.source_entity_id,
                            r.target_entity_id,
                            r.relationship_type,
                            r.description
                        FROM graph_relationships r
                        WHERE r.source_entity_id = ANY(%s) OR r.target_entity_id = ANY(%s)
                        ORDER BY r.source_entity_id, r.target_entity_id
                        LIMIT 100
                    """
                    cur.execute(relationship_query, (entity_ids, entity_ids))
                    relationship_results = cur.fetchall()

                    relationships = [
                        {
                            'source': row[0],
                            'target': row[1],
                            'type': row[2],
                            'description': row[3]
                        }
                        for row in relationship_results
                    ]

                    # 커뮤니티 조회 (엔티티들이 속한 커뮤니티)
                    community_query = """
                        SELECT DISTINCT
                            c.name as community_name,
                            c.description as community_description
                        FROM graph_communities c
                        WHERE c.entities && %s
                        ORDER BY c.name
                        LIMIT 20
                    """
                    cur.execute(community_query, (entity_ids,))
                    community_results = cur.fetchall()

                    communities = [
                        {
                            'name': row[0],
                            'description': row[1]
                        }
                        for row in community_results
                    ]

                # 5. 결과 구성
                entities = [
                    {
                        'id': row[0],
                        'name': row[1],
                        'type': row[2],
                        'description': row[3],
                        'source_documents': row[4] or [],
                        'properties': row[5] or {}
                    }
                    for row in entity_results
                ]

                result = {
                    'success': True,
                    'data': {
                        'entities': entities,
                        'relationships': relationships,
                        'communities': communities,
                        'chunk_info': {
                            'filename': filename,
                            'chunk_index': chunk_index,
                            'chunk_text_length': len(chunk_text),
                            'keywords': keywords[:10],  # 상위 10개 키워드만 반환
                            'metadata': chunk_metadata
                        },
                        'stats': {
                            'total_entities': len(entities),
                            'total_relationships': len(relationships),
                            'total_communities': len(communities),
                            'keywords_used': len(keywords)
                        }
                    }
                }

                logger.info(f"청크 엔티티 추출 완료: 엔티티 {len(entities)}개, 관계 {len(relationships)}개")
                return JSONResponse(content=result)

        finally:
            if conn:
                conn.close()

    except Exception as e:
        logger.error(f"청크 엔티티 추출 오류: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'청크 엔티티 추출 실패: {str(e)}'}
        )

@app.post("/graphrag/query-entities")
async def get_query_entities(request: Request):
    """사용자 질문에서 실제 GraphRAG 검색을 수행하여 관련 엔티티들을 반환"""
    try:
        body = await request.json()
        query = body.get('query')
        user_id = body.get('user_id', 'admin')
        top_k = body.get('top_k', 20)
        view_mode = body.get('view_mode', 'overview')

        if not query:
            return JSONResponse(
                status_code=400,
                content={'error': 'query가 필요합니다'}
            )

        logger.info(f"GraphRAG 모달 검색 요청: query={query}, user_id={user_id}")

        # GraphRAGProcessor 인스턴스 생성
        try:
            # GraphRAG 클래스 강제 로드 (파일 끝에 정의된 클래스)
            if 'GraphRAGProcessor' not in globals():
                try:
                    # 파일 끝 부분의 GraphRAGProcessor 클래스 정의를 강제로 실행
                    with open(__file__, 'r', encoding='utf-8') as f:
                        lines = f.readlines()
                    
                    # KnowledgeGraph + GraphRAGProcessor 클래스 모두 로드
                    kg_start = None
                    grag_start = None
                    
                    for i, line in enumerate(lines):
                        if line.strip().startswith('class KnowledgeGraph:'):
                            kg_start = i
                        elif line.strip().startswith('class GraphRAGProcessor:'):
                            grag_start = i
                            break
                    
                    if kg_start and grag_start:
                        # KnowledgeGraph부터 파일 끝까지 모든 클래스 로드
                        class_code = ''.join(lines[kg_start:])
                        exec(class_code, globals())
                        logger.info("✅ GraphRAG 모달용 KnowledgeGraph + GraphRAGProcessor 강제 로드 완료")
                except Exception as e:
                    logger.info(f"GraphRAG 클래스 강제 로드 실패: {str(e)}", error=True)
            
            # GraphRAGProcessor 인스턴스 생성
            if 'GraphRAGProcessor' in globals():
                GraphRAGProcessorClass = globals()['GraphRAGProcessor']
                graphrag_processor = GraphRAGProcessorClass()
                logger.info("✅ GraphRAG 모달용 프로세서 생성 완료")
            else:
                raise Exception("GraphRAGProcessor 클래스를 로드할 수 없습니다")

        except Exception as e:
            logger.info(f"GraphRAG 프로세서 생성 실패: {str(e)}", error=True)
            # 실패 시 빈 결과 반환
            return JSONResponse(content={
                'success': True,
                'data': {
                    'entities': [],
                    'relationships': [],
                    'communities': [],
                    'query_info': {
                        'original_query': query,
                        'user_id': user_id,
                        'error': 'GraphRAG 프로세서 초기화 실패'
                    }
                }
            })

        # 실제 GraphRAG 검색 실행
        graph_result = await graphrag_processor.search_graph(
            query=query,
            user_id=user_id,
            mode="graph",  # 그래프 검색 모드
            view_mode=view_mode,  # 뷰 모드 추가
            top_k=top_k
        )

        # search_graph 결과를 모달에서 사용하는 형식으로 변환
        entities = graph_result.get('entities', [])
        
        # 엔티티 ID들 수집
        entity_ids = [entity['id'] for entity in entities]
        
        # 관계 및 커뮤니티 정보 추가 조회
        relationships = []
        communities = []
        
        if entity_ids:
            try:
                conn = get_pool_connection()
                try:
                    with conn.cursor() as cur:
                        # 엔티티 간 관계 조회
                        relationship_query = """
                            SELECT r.source_entity_id, r.target_entity_id, r.relationship_type, r.weight, r.description,
                                   se.name as source_name, te.name as target_name
                            FROM graph_relationships r
                            JOIN graph_entities se ON r.source_entity_id = se.id
                            JOIN graph_entities te ON r.target_entity_id = te.id
                            WHERE r.source_entity_id = ANY(%s) OR r.target_entity_id = ANY(%s)
                            ORDER BY r.weight DESC
                            LIMIT 100
                        """
                        cur.execute(relationship_query, (entity_ids, entity_ids))
                        relationship_results = cur.fetchall()

                        relationships = [{
                            'source_entity_id': row[0],
                            'target_entity_id': row[1], 
                            'relationship_type': row[2],
                            'weight': row[3],
                            'description': row[4],
                            'source_name': row[5],
                            'target_name': row[6]
                        } for row in relationship_results]

                        # 엔티티가 속한 커뮤니티 조회
                        community_query = """
                            SELECT id, name, description, entities
                            FROM graph_communities
                            WHERE entities && %s
                            LIMIT 50
                        """
                        cur.execute(community_query, (entity_ids,))
                        community_results = cur.fetchall()

                        communities = [{
                            'id': row[0],
                            'name': row[1],
                            'description': row[2],
                            'entities': row[3]
                        } for row in community_results]

                finally:
                    conn.close()
            except Exception as e:
                logger.info(f"관계/커뮤니티 조회 실패: {str(e)}", error=True)

        logger.info(f"GraphRAG 모달 검색 결과 - 엔티티: {len(entities)}, 관계: {len(relationships)}, 커뮤니티: {len(communities)}")
        logger.info(f"검색된 엔티티 목록: {[e['name'] for e in entities[:10]]}")

        return JSONResponse(content={
            'success': True,
            'data': {
                'entities': entities,
                'relationships': relationships,
                'communities': communities,
                'query_info': {
                    'original_query': query,
                    'user_id': user_id,
                    'total_entities_found': len(entities),
                    'search_method': 'GraphRAG.search_graph'
                }
            }
        })

    except Exception as e:
        logger.info(f"GraphRAG 모달 검색 실패: {str(e)}", error=True)
        return JSONResponse(
            status_code=500,
            content={'error': f'GraphRAG 모달 검색 중 오류 발생: {str(e)}'}
        )

def extract_keywords_from_text(text: str, max_keywords: int = 30):
    """텍스트에서 키워드를 추출하는 함수 - tokenize 함수 활용"""
    from rag_process import tokenize
    
    # 단일 키워드와 n-gram 모두 포함
    keywords_single = tokenize(text, include_ngrams=False)  # 단일 키워드
    keywords_ngram = tokenize(text, include_ngrams=True)    # n-gram 포함
    
    # 두 결과를 합치되 중복 제거
    all_keywords = list(set(keywords_single + keywords_ngram))
    
    # 키워드 중요도 기반 정렬 (고유명사/브랜드명 우선)
    def keyword_importance(keyword):
        # 1. 고유명사 패턴 (한글만, 영문 대문자 시작)
        if re.match(r'^[가-힣]+$', keyword) and len(keyword) >= 2:
            return (0, -len(keyword))  # 한글 고유명사 최우선
        elif re.match(r'^[A-Z][a-z]*$', keyword):
            return (0, -len(keyword))  # 영문 고유명사 최우선
        # 2. 복합어 (길이 4 이상)
        elif len(keyword) >= 4:
            return (1, -len(keyword))
        # 3. 일반 키워드
        else:
            return (2, -len(keyword))
    
    keywords_sorted = sorted(all_keywords, key=keyword_importance)
    
    # 최대 개수만큼 반환
    return keywords_sorted[:max_keywords]

@app.post("/search")
async def search_documents(request: SearchRequest):
    """문서 검색 엔드포인트"""
    try:
        # RAG 설정 가져오기 (임시 설정 우선 적용)
        rag_settings = get_rag_settings()

        # 임시 설정이 있으면 우선 적용 (최적화 테스트용)
        if hasattr(request, 'tempSettings') and request.tempSettings:
            logger.info(f"임시 설정 적용 (원본): {request.tempSettings}")

            # 키 이름 매핑 (RAG_ 접두사가 있는 키를 소문자 버전으로 변환)
            temp_settings_mapped = {}
            for key, value in request.tempSettings.items():
                if key.startswith('RAG_'):
                    # RAG_KEYWORD_PRECISION_THRESHOLD -> keyword_precision_threshold
                    mapped_key = key.replace('RAG_', '').lower()
                    temp_settings_mapped[mapped_key] = value
                else:
                    temp_settings_mapped[key] = value

            logger.info(f"임시 설정 적용 (매핑됨): {temp_settings_mapped}")
            rag_settings.update(temp_settings_mapped)
        
        # 4-Score 시스템 검색 설정 로깅
        logger.info("=== 4-Score 검색 설정 ===")
        logger.info(f"- Top K: {rag_settings['top_k']}")
        logger.info(f"- 키워드 정확도 임계값: {rag_settings.get('keyword_precision_threshold', 0.1)}")
        logger.info(f"- 의미적 관련성 임계값: {rag_settings.get('semantic_relevance_threshold', 0.15)}")
        logger.info(f"- 콘텐츠 품질 임계값: {rag_settings.get('content_quality_threshold', 0.3)}")
        logger.info(f"- 최신성 임계값: {rag_settings.get('recency_threshold', 0.0)}")
        logger.info("==================\n")

        # 요청 파라미터 로깅 (간소화)
        logger.info(f"🔍 검색 요청: '{request.query}' (사용자: {request.user_id})")

        # 검색 범위와 사용자 ID 처리 - 단순화
        rag_search_scope = request.ragSearchScope or 'personal'
        
        # 실제 검색에 사용할 사용자 ID 결정
        if request.actual_user_id:
            # actual_user_id가 제공되면 우선 사용 (frontend에서 'all', 'personal' 검색 시)
            actual_user_for_search = request.actual_user_id.strip()
        elif request.user_id and request.user_id.strip():
            # user_id가 제공되면 사용 (레거시 지원)
            actual_user_for_search = request.user_id.strip()
        else:
            # 기본값
            actual_user_for_search = None
        
        # 검색 범위 설정 완료
        
        # 캐시 키 생성 (사용자 ID 포함) - 캐시 비활성화
        # cache_key = f"search:{hashlib.md5((request.query + str(actual_user_for_search)).encode()).hexdigest()}"

        # Redis 캐시 확인 - 캐시 비활성화
        # if embedding_service.redis_client:
        #     try:
        #         cached_results = embedding_service.redis_client.get(cache_key)
        #         if cached_results:
        #             logger.info(f"캐시 히트! 검색 쿼리: {request.query} (사용자 ID: {actual_user_for_search})")
        #             return {
        #                 "status": "success",
        #                 "results": json.loads(cached_results),
        #                 "from_cache": True
        #             }
        #     except Exception as e:
        #         logger.error(f"캐시 조회 중 오류 발생: {str(e)}")

        logger.info(f"새로운 검색 실행 (캐시 비활성화): {request.query} (사용자 ID: {actual_user_for_search})")

        # 검색 워커 프로세스 사용
        import uuid
        import time
        
        request_id = str(uuid.uuid4())
        
        # rag_settings에서 top_k 값 가져오기
        rag_settings = get_rag_settings()
        top_k = int(rag_settings.get('top_k', 5))  # 설정된 값 사용, 기본값 5
        
        # 요청에서 maxResults가 제공되면 그 값을 우선 사용
        if request.maxResults:
            top_k = request.maxResults
        
        # 검색 워커 준비 상태 확인
        if not search_model_ready_event.is_set():
            logger.info("검색 워커 모델 로딩 대기 중...")
            if not search_model_ready_event.wait(timeout=30):
                logger.error("검색 워커 모델 로딩 타임아웃")
                return {
                    "status": "error",
                    "message": "검색 서비스 초기화 중입니다. 잠시 후 다시 시도해주세요.",
                    "results": []
                }
        
        # 검색 워커로 요청 전송
        
        # 검색 요청을 워커 프로세스로 전송
        search_request = {
            'request_id': request_id,
            'query': request.query,
            'top_k': top_k,
            'large_context_chunking': rag_settings.get('large_context_chunking', 'no'),
            'user_id': actual_user_for_search,  # 실제 검색용 사용자 ID
            'rag_search_scope': rag_search_scope  # 검색 범위 전달
        }
        search_queue.put(search_request)
        
        # 응답 대기 (타임아웃 설정)
        timeout = 30  # 30초 타임아웃
        start_wait = time.time()
        response = None
        
        while time.time() - start_wait < timeout:
            if not search_response_queue.empty():
                temp_response = search_response_queue.get()
                if temp_response['request_id'] == request_id:
                    response = temp_response
                    break
                else:
                    # 다른 요청의 응답이면 다시 큐에 넣기
                    search_response_queue.put(temp_response)
            time.sleep(0.1)
        
        if not response:
            logger.error(f"검색 워커 응답 타임아웃 - ID: {request_id}")
            return {
                "status": "error",
                "message": "검색 요청 타임아웃",
                "results": []
            }
        
        if not response['success']:
            logger.error(f"검색 워커 오류 - ID: {request_id}, 오류: {response.get('error', 'Unknown')}")
            return {
                "status": "error", 
                "message": response.get('error', 'Unknown error'),
                "results": []
            }
        
        results = response['results']
        logger.info(f"검색 워커 응답 수신 - ID: {request_id}, 결과: {len(results)}개, 시간: {response['processing_time']:.2f}초")

        # DEBUG 모드일 때 상세 정보 출력
        if os.getenv('LOG_LEVEL', 'INFO').upper() == 'DEBUG':
            logger.debug(f"=== RAG 검색 엔드포인트 DEBUG 정보 ===")
            logger.debug(f"검색 쿼리: '{request.query}'")
            logger.debug(f"요청된 maxResults: {request.maxResults}")
            logger.debug(f"검색 워커 반환 결과 수: {len(results)}")
            
            if results:
                logger.debug("=== 검색 워커 원본 결과 점수 ===")
                for i, result in enumerate(results):
                    score = result.get('score', 0)
                    distance = result.get('distance', 1)
                    content_preview = result.get('content', '')[:50].replace('\n', ' ')
                    source = result.get('metadata', {}).get('source', 'Unknown')
                    logger.debug(f"  워커결과 {i+1}: 점수={score:.2f}, 거리={distance:.2f}, 소스={source}, 내용='{content_preview}...'")
                    
                # RAG 4점수 체계 임계값 설정 출력
                logger.debug("=== 현재 RAG 임계값 설정 ===")
                core_thresholds = ['keyword_precision_threshold', 'semantic_relevance_threshold', 'content_quality_threshold', 'recency_threshold']
                for key in core_thresholds:
                    if key in rag_settings:
                        logger.debug(f"  {key}: {rag_settings[key]}")
            else:
                logger.debug("검색 워커에서 결과가 없습니다.")
            logger.debug("=======================================")
        

        # 결과를 ragService.js가 기대하는 형식으로 변환
        formatted_results = []
        for result in results:
            metadata = result.get('metadata', {})
            # 디버깅: 원본 메타데이터 확인
            # logger.info(f"DEBUG: Original metadata from search worker: {metadata}")
            main_scores = metadata.get('main_scores', {})
            sub_scores = metadata.get('sub_scores', {})
            
            # 검색 워커에서 반환한 벡터 유사도 점수 사용
            vector_similarity_score = result.get('score', 0.0)
            vector_distance = result.get('distance', 1.0)
            
            # 가중치 정보 가져오기
            rag_settings = get_rag_settings()
            weights = {
                'main_weights': {
                    'exact_match': rag_settings.get('exact_match_weight', 0.40),
                    'similarity': rag_settings.get('similarity_weight', 0.30),
                    'relevance': rag_settings.get('relevance_weight', 0.10),
                    'date': rag_settings.get('date_weight', 0.10)
                },
                'sub_weights': {
                    'structure': rag_settings.get('structure_weight', 0.10),
                    'content': rag_settings.get('content_weight', 0.10),
                    'length': rag_settings.get('length_weight', 0.10),
                    'keyword_quality': rag_settings.get('keyword_quality_weight', 0.20),
                    'partial_match': rag_settings.get('partial_match_weight', 0.50)
                }
            }

            # source에서 파일명만 추출
            source_path = metadata.get('source', 'Unknown')
            source_name = os.path.basename(source_path) if source_path != 'Unknown' else 'Unknown'
            
            formatted_result = {
                'content': result['content'],
                'source': source_name,
                'page_number': int(metadata.get('page_number', 1) if metadata.get('page_number') is not None else 1),
                'chunk_index': int(metadata.get('chunk_index', 0) or 0),
                'total_chunks': int(metadata.get('total_chunks', 0) or 0),
                'timestamp': metadata.get('timestamp'),
                'modification_date': metadata.get('modification_date'),
                # metadata 필드 추가 - index.js에서 사용
                'metadata': {
                    'source': source_name,
                    'page_number': int(metadata.get('page_number', 1) if metadata.get('page_number') is not None else 1),
                    'chunk_index': int(metadata.get('chunk_index', 0) or 0),
                    'total_chunks': int(metadata.get('total_chunks', 0) or 0),
                    'user_id': metadata.get('user_id', 'all'),
                    'timestamp': metadata.get('timestamp'),
                    'modification_date': metadata.get('modification_date'),
                },

                # 새로운 4점수 시스템 추가
                # core_scores가 없는 경우 main_scores와 sub_scores로부터 계산
                'core_scores': {
                    'keyword_precision': float(metadata.get('core_scores', {}).get('keyword_precision', 
                        main_scores.get('exact_match_score', 0.5) + sub_scores.get('partial_match_score', 0.3) * 0.5) or 0.5),
                    'semantic_relevance': float(metadata.get('core_scores', {}).get('semantic_relevance', 
                        vector_similarity_score * 0.7) or 0.5),
                    'content_quality': float(metadata.get('core_scores', {}).get('content_quality', 
                        (sub_scores.get('content_score', 0.8) + sub_scores.get('length_score', 0.8) + sub_scores.get('structure_score', 0.8)) / 3) or 0.8),
                    'recency': float(metadata.get('core_scores', {}).get('recency', 
                        main_scores.get('date_score', 1.0)) or 1.0),
                },
                'main_scores': {
                    'exact_match_score': float(main_scores.get('exact_match_score', 0) or 0),
                    'similarity_score':  float(vector_similarity_score),  # 벡터 유사도 점수 사용
                    'relevance_score':   float(vector_similarity_score * 0.8),  # 관련성 점수는 유사도 기반
                    'date_score':        float(main_scores.get('date_score', 0) or 0),
                },
                'sub_scores': {
                    'structure_score':    float(sub_scores.get('structure_score', 0) or 0),
                    'content_score':      float(sub_scores.get('content_score', 0) or 0),
                    'length_score':       float(sub_scores.get('length_score', 0) or 0),
                    'keyword_quality':    float(sub_scores.get('keyword_quality', 0) or 0),
                    'partial_match_score':float(sub_scores.get('partial_match_score', 0) or 0),
                },
                'matches': {
                    'exact_matches':   metadata.get('matches', {}).get('exact_matches', []),
                    'partial_matches': metadata.get('matches', {}).get('partial_matches', []),
                    'keyword_matches': metadata.get('matches', {}).get('keyword_matches', []),
                },
                'filter_settings': {
                    'exact_match_threshold':        float(metadata.get('thresholds', {}).get('exact_match_threshold', 0.3) or 0.3),
                    'keyword_match_threshold':      float(metadata.get('thresholds', {}).get('keyword_match_threshold', 0.6) or 0.6),
                    'strong_keyword_match_threshold':float(metadata.get('thresholds', {}).get('strong_keyword_match_threshold', 0.9) or 0.9),
                    'similarity_threshold':         float(metadata.get('thresholds', {}).get('similarity_threshold', 0.5) or 0.5),
                    'relevance_threshold':          float(metadata.get('thresholds', {}).get('relevance_threshold', 0.5) or 0.5),
                },
                'thresholds': {k: (float(v) if isinstance(v, (int, float, np.floating)) else v)
                            for k, v in (metadata.get('thresholds', {}) or {}).items()},
                'weights': {
                    'main_weights': weights['main_weights'],
                    'sub_weights':  weights['sub_weights'],
                },
                'user_id': metadata.get('user_id', 'all'),
                'is_large_context_chunk': metadata.get('is_large_context_chunk', False),
                'original_match': metadata.get('original_match', False),
                'final_score': float(vector_similarity_score),  # 벡터 유사도를 최종 점수로 사용
            }
            formatted_results.append(formatted_result)

        # DEBUG 모드일 때 상세한 임계값 비교 정보 출력
        if os.getenv('LOG_LEVEL', 'INFO').upper() == 'DEBUG':
            logger.debug(f"=== 임계값 비교 상세 DEBUG 정보 ===")
            logger.debug(f"포맷팅된 결과 수: {len(formatted_results)}")
            
            # RAG 설정에서 임계값들 가져오기
            similarity_threshold = float(rag_settings.get('similarity_threshold', 0.5))
            relevance_threshold = float(rag_settings.get('relevance_threshold', 0.5))
            exact_match_threshold = float(rag_settings.get('exact_match_threshold', 0.3))
            keyword_quality_threshold = float(rag_settings.get('keyword_match_threshold', 0.6))
            
            logger.debug(f"현재 설정된 임계값:")
            logger.debug(f"  - 유사도 임계값: {similarity_threshold}")
            logger.debug(f"  - 관련성 임계값: {relevance_threshold}")
            logger.debug(f"  - 정확도 임계값 (exact_match): {exact_match_threshold}")
            logger.debug(f"  - 키워드 품질 임계값: {keyword_quality_threshold}")
            
            if formatted_results:
                logger.debug("=== 각 결과의 임계값 통과/실패 분석 ===")
                for i, result in enumerate(formatted_results[:5]):  # 상위 5개 분석
                    # 점수들 추출
                    main_scores = result.get('main_scores', {})
                    sub_scores = result.get('sub_scores', {})
                    
                    similarity_score = main_scores.get('similarity_score', 0)
                    relevance_score = main_scores.get('relevance_score', 0)
                    exact_match_score = main_scores.get('exact_match_score', 0)
                    keyword_quality_score = sub_scores.get('keyword_quality', 0)
                    
                    source = result.get('source', 'Unknown')
                    content_preview = result.get('content', '')[:50].replace('\n', ' ')
                    
                    # 임계값 통과 여부 확인
                    similarity_pass = similarity_score >= similarity_threshold
                    relevance_pass = relevance_score >= relevance_threshold  
                    exact_match_pass = exact_match_score >= exact_match_threshold
                    keyword_quality_pass = keyword_quality_score >= keyword_quality_threshold
                    
                    # 전체 통과 여부 (모든 임계값 통과 시)
                    all_pass = similarity_pass and relevance_pass and exact_match_pass and keyword_quality_pass
                    
                    logger.debug(f"  === 결과 {i+1} ===")
                    logger.debug(f"  소스: {source}")
                    logger.debug(f"  내용: '{content_preview}...'")
                    logger.debug(f"  점수 및 임계값 비교:")
                    logger.debug(f"    유사도: {similarity_score:.2f} >= {similarity_threshold} → {'통과' if similarity_pass else '실패'}")
                    logger.debug(f"    관련성: {relevance_score:.2f} >= {relevance_threshold} → {'통과' if relevance_pass else '실패'}")
                    logger.debug(f"    정확도: {exact_match_score:.2f} >= {exact_match_threshold} → {'통과' if exact_match_pass else '실패'}")
                    logger.debug(f"    키워드 품질: {keyword_quality_score:.2f} >= {keyword_quality_threshold} → {'통과' if keyword_quality_pass else '실패'}")
                    logger.debug(f"  최종 판정: {'모든 임계값 통과 - 포함됨' if all_pass else '일부 임계값 미달 - 제외됨'}")


            else:
                logger.debug("최종 포맷팅된 결과가 없습니다.")
                logger.debug("가능한 원인:")
                logger.debug(f"  1. 검색 워커에서 결과 없음: {len(results) == 0}")
                logger.debug(f"  2. 모든 결과가 임계값 미달로 필터링됨")
                logger.debug(f"     - 유사도 임계값: {similarity_threshold}")
                logger.debug(f"     - 관련성 임계값: {relevance_threshold}")
                logger.debug(f"     - 정확도 임계값: {exact_match_threshold}")
                logger.debug(f"     - 키워드 품질 임계값: {keyword_quality_threshold}")
                logger.debug("  3. 포맷팅 과정에서 오류 발생")
            logger.debug("=====================================")
            
            # 검색 워커 결과와 최종 결과 차이 분석
            if len(results) > 0 and len(formatted_results) == 0:
                logger.debug("=== 검색 워커 결과는 있으나 최종 결과가 없는 경우 분석 ===")
                logger.debug(f"검색 워커 결과 수: {len(results)}")
                logger.debug("임계값 설정을 낮춰서 재시도하는 것을 권장합니다.")
                logger.debug("=========================================")

        # 4점수 시스템 기반 임계값 필터링 적용
        filtered_results = []
        
        # CLAUDE.md에서 정의된 임계값들 가져오기
        keyword_precision_threshold = float(rag_settings.get('keyword_precision_threshold', 0.1))
        semantic_relevance_threshold = float(rag_settings.get('semantic_relevance_threshold', 0.15))
        content_quality_threshold = float(rag_settings.get('content_quality_threshold', 0.3))
        
        logger.info("\n=== 4점수 시스템 임계값 필터링 적용 ===")
        logger.info(f"키워드 정확도 임계값: {keyword_precision_threshold}")
        logger.info(f"의미 관련성 임계값: {semantic_relevance_threshold}")
        logger.info(f"내용 품질 임계값: {content_quality_threshold}")
        
        passed_count = 0
        failed_count = 0
        
        for result in formatted_results:
            core_scores = result.get('core_scores', {})
            keyword_precision = core_scores.get('keyword_precision', 0.0)
            semantic_relevance = core_scores.get('semantic_relevance', 0.0)
            content_quality = core_scores.get('content_quality', 0.0)
            
            # CLAUDE.md의 4점수 시스템 통과 조건:
            # final_pass = (keyword_pass OR semantic_pass) AND quality_pass
            keyword_pass = keyword_precision >= keyword_precision_threshold
            semantic_pass = semantic_relevance >= semantic_relevance_threshold
            quality_pass = content_quality >= content_quality_threshold
            
            final_pass = (keyword_pass or semantic_pass) and quality_pass
            
            source_name = os.path.basename(result.get('source', 'Unknown'))
            
            if final_pass:
                filtered_results.append(result)
                passed_count += 1
                logger.info(f"✅ 통과: {source_name} (키워드: {keyword_precision:.2f}, 의미: {semantic_relevance:.2f}, 품질: {content_quality:.2f})")
            else:
                failed_count += 1
                reasons = []
                if not quality_pass:
                    reasons.append("품질 미달")
                if not keyword_pass and not semantic_pass:
                    reasons.append("키워드/의미 모두 미달")
                logger.info(f"❌ 제외: {source_name} - {', '.join(reasons)} (키워드: {keyword_precision:.2f}, 의미: {semantic_relevance:.2f}, 품질: {content_quality:.2f})")
        
        logger.info(f"\n필터링 결과: {len(formatted_results)}개 중 {passed_count}개 통과, {failed_count}개 제외")

        # 결과 캐싱 - 캐시 비활성화
        # if embedding_service.redis_client and filtered_results:
        #     try:
        #         embedding_service.redis_client.setex(
        #             cache_key,
        #             3600,  # 1시간 캐시
        #             json.dumps(filtered_results)
        #         )
        #     except Exception as e:
        #         logger.error(f"캐시 저장 중 오류 발생: {str(e)}")

        return {
            "status": "success",
            "results": filtered_results,
            "from_cache": False
        }

    except Exception as e:
        logger.error(f"검색 중 오류 발생: {str(e)}")
        return {
            "status": "error",
            "error": str(e),
            "results": []
        }

@app.post("/add")
async def add_documents(request: AddDocumentsRequest):
    """문서 추가 엔드포인트"""
    try:
        logger.info(f"\n문서 추가 요청:")
        logger.info(f"- 소스 경로: {request.path}")
        logger.info(f"- 사용자 ID: {request.user_id if request.user_id else 'all'}")

        # 캐시 초기화
        await clear_cache()
        logger.info("문서 추가 전 캐시 초기화 완료")
        
        # 소스 경로 검증
        abs_source_path = os.path.abspath(request.path)
        if not os.path.exists(abs_source_path):
            return JSONResponse(
                content={
                    "status": "error",
                    "message": f"Source path does not exist: {request.path}"
                },
                status_code=400
            )

        # RAG 문서 경로 가져오기
        rag_docs_path = request.base_path or get_rag_docs_path()
        logger.info(f"- RAG 문서 경로: {rag_docs_path}")

        # 파일 복사 및 처리
        try:
            # 사용자 ID 결정
            effective_user_id = request.user_id if request.user_id and request.user_id.strip() else None
            logger.info(f"- 적용될 사용자 ID: {effective_user_id if effective_user_id else '(없음)'}")

            # 파일 목록 수집
            source_files = []
            if os.path.isdir(abs_source_path):
                # 디렉토리인 경우 하위 파일들 수집
                for root, _, files in os.walk(abs_source_path):
                    for file in files:
                        if any(file.lower().endswith(ext) for ext in SUPPORTED_EXTENSIONS):
                            source_files.append(os.path.join(root, file))
            else:
                # 단일 파일인 경우
                if any(abs_source_path.lower().endswith(ext) for ext in SUPPORTED_EXTENSIONS):
                    source_files.append(abs_source_path)

            if not source_files:
                return JSONResponse(
                    content={
                        "status": "error",
                        "message": "No supported files found"
                    },
                    status_code=400
                )

            # 대상 디렉토리 결정
            target_dir = rag_docs_path
            if effective_user_id:  # 사용자 ID가 있는 경우에만 하위 디렉토리 생성
                target_dir = os.path.join(rag_docs_path, effective_user_id)
            os.makedirs(target_dir, exist_ok=True)
            logger.info(f"- 대상 디렉토리 생성: {target_dir}")

            # 파일 복사 또는 기존 파일 목록 수집
            copied_files = []
            for src_file in source_files:
                # 원본 파일의 상대 경로 계산 (사용자 ID 폴더 제외)
                if os.path.isdir(abs_source_path):
                    rel_path = os.path.relpath(src_file, abs_source_path)
                else:
                    rel_path = os.path.basename(src_file)

                # 대상 경로 생성
                dst_file = os.path.join(target_dir, rel_path)
                dst_dir = os.path.dirname(dst_file)
                
                # 대상 디렉토리 생성
                os.makedirs(dst_dir, exist_ok=True)
                
                # 파일 복사 (같은 파일이 아닌 경우만)
                if os.path.abspath(src_file) == os.path.abspath(dst_file):
                    # 이미 올바른 위치에 있는 파일은 복사하지 않고 바로 처리 목록에 추가
                    copied_files.append(dst_file)
                    logger.info(f"- 기존 파일 사용: {dst_file} (복사 불필요)")
                else:
                    # 다른 위치의 파일은 복사
                    shutil.copy2(src_file, dst_file)
                    copied_files.append(dst_file)
                    logger.info(f"- 파일 복사 완료: {src_file} -> {dst_file}")

            if not copied_files:
                return JSONResponse(
                    content={
                        "status": "error",
                        "message": "No files were copied"
                    },
                    status_code=400
                )

            # 메인 프로세스에서는 직접 처리 대신 워커 큐를 통해 처리
            if os.getenv('AIRUN_MAIN_PROCESS') == '1':
                logger.info("메인 프로세스: 문서 처리를 임베딩 워커에게 위임")
                # 임베딩 큐에 작업 추가
                import uuid
                for file_path in copied_files:
                    embed_task = {
                        'request_id': str(uuid.uuid4()),
                        'file_path': file_path,
                        'user_id': effective_user_id if effective_user_id else '',
                        'base_path': rag_docs_path,
                        'task_type': 'document_add'
                    }
                    embedding_queue.put(embed_task)
                
                results = {
                    'processed_count': len(copied_files),
                    'queued_for_processing': True,
                    'note': '문서가 임베딩 워커 큐에 추가되었습니다'
                }
            else:
                # 워커 프로세스에서는 직접 처리
                results = await document_processor.add_documents(
                    copied_files,
                    effective_user_id if effective_user_id else '',  # 빈 문자열 전달
                    rag_docs_path,
                    None  # 기본 설정 사용
                )

            return JSONResponse(
                content={
                    "status": "success",
                    "message": f"Successfully processed {len(copied_files)} documents",
                    "files_count": len(copied_files),
                    "original_path": request.path,
                    "rag_docs_path": rag_docs_path,
                    "user_id": effective_user_id,
                    "summary": results
                },
                status_code=200
            )

        except Exception as e:
            logger.error(f"문서 처리 중 오류 발생: {str(e)}")
            return JSONResponse(
                content={
                    "status": "error",
                    "message": f"Error processing documents: {str(e)}"
                },
                status_code=500
            )

    except Exception as e:
        logger.error(f"문서 추가 요청 처리 중 오류 발생: {str(e)}")
        return JSONResponse(
            content={
                "status": "error",
                "message": str(e)
            },
            status_code=500
        )

@app.post("/upload", response_model=UploadResponse)
async def upload_document(
    file: UploadFile = File(...),
    user_id: Optional[str] = Form(None),
    overwrite: bool = Form(False),
    job_id: Optional[str] = Form(None),
    project_name: Optional[str] = Form(None),  # 프로젝트 정보 추가
    # RAG 설정 FormData 매개변수들
    rag_mode: Optional[str] = Form(None),
    rag_pdf_backend: Optional[str] = Form(None),
    rag_process_text_only: Optional[str] = Form(None),
    rag_generate_image_embeddings: Optional[str] = Form(None),
    rag_enable_graphrag: Optional[str] = Form(None),
    use_enhanced_embedding: Optional[str] = Form(None)
):
    try:
        logger.info(f"- 파일 업로드 요청: {file.filename}")
        logger.info(f"- 사용자 ID: {user_id if user_id else 'all'}")
        
        # RAG 설정 FormData에서 temp_settings 추출
        temp_settings = {}
        # Form 객체와 None 값 필터링
        if rag_mode and str(rag_mode) != 'None' and not str(rag_mode).startswith('annotation='):
            temp_settings['rag_mode'] = str(rag_mode) if not isinstance(rag_mode, str) else rag_mode
        if rag_pdf_backend and str(rag_pdf_backend) != 'None' and not str(rag_pdf_backend).startswith('annotation='):
            temp_settings['pdf_backend'] = str(rag_pdf_backend) if not isinstance(rag_pdf_backend, str) else rag_pdf_backend
        if rag_process_text_only and str(rag_process_text_only) != 'None' and not str(rag_process_text_only).startswith('annotation='):
            text_only_value = str(rag_process_text_only).lower() if hasattr(rag_process_text_only, 'lower') else str(rag_process_text_only).lower()
            temp_settings['process_text_only'] = text_only_value in ('yes', 'true', '1')
        if rag_generate_image_embeddings and str(rag_generate_image_embeddings) != 'None' and not str(rag_generate_image_embeddings).startswith('annotation='):
            temp_settings['generate_image_embeddings'] = str(rag_generate_image_embeddings) if not isinstance(rag_generate_image_embeddings, str) else rag_generate_image_embeddings
        if rag_enable_graphrag and str(rag_enable_graphrag) != 'None' and not str(rag_enable_graphrag).startswith('annotation='):
            graphrag_value = str(rag_enable_graphrag).lower() if hasattr(rag_enable_graphrag, 'lower') else str(rag_enable_graphrag).lower()
            temp_settings['enable_graphrag'] = graphrag_value in ('yes', 'true', '1')
        if use_enhanced_embedding and str(use_enhanced_embedding) != 'None' and not str(use_enhanced_embedding).startswith('annotation='):
            enhanced_value = str(use_enhanced_embedding).lower() if hasattr(use_enhanced_embedding, 'lower') else str(use_enhanced_embedding).lower()
            temp_settings['use_enhanced_embedding'] = enhanced_value in ('yes', 'true', '1')
        
        if temp_settings:
            logger.info(f"- RAG temp_settings 추출됨: {temp_settings}")
        
        # 지원되는 파일 확장자 확인
        file_ext = os.path.splitext(file.filename)[1].lower()
        if file_ext not in SUPPORTED_EXTENSIONS:
            return JSONResponse(
                content={
                    "status": "error",
                    "message": f"Unsupported file type: {file_ext}. Supported types: {', '.join(SUPPORTED_EXTENSIONS)}"
                },
                status_code=400
            )
            
        # RAG 문서 경로 가져오기
        rag_docs_path = get_rag_docs_path()
        
        # 사용자별 디렉토리 생성 - Form 객체와 문자열 모두 처리
        if hasattr(user_id, 'strip'):  # 문자열인 경우
            effective_user_id = user_id.strip() if user_id and user_id.strip() else 'all'
        else:  # Form 객체인 경우
            effective_user_id = str(user_id).strip() if user_id and str(user_id).strip() else 'all'
        target_dir = os.path.join(rag_docs_path, effective_user_id) if effective_user_id != 'all' else rag_docs_path
        os.makedirs(target_dir, exist_ok=True)
        
        # 중복 파일 확인
        original_target_file = os.path.join(target_dir, file.filename)
        logger.info(f"중복 파일 확인: {original_target_file}, 존재여부: {os.path.exists(original_target_file)}")
        
        if os.path.exists(original_target_file):
            logger.info(f"중복 파일 발견: {original_target_file}")
            if not overwrite:
                # 자동으로 번호를 붙여서 새로운 파일명 생성
                base_name, ext = os.path.splitext(file.filename)
                counter = 1
                new_filename = f"{base_name}_{counter}{ext}"
                new_target_file = os.path.join(target_dir, new_filename)
                
                while os.path.exists(new_target_file):
                    counter += 1
                    new_filename = f"{base_name}_{counter}{ext}"
                    new_target_file = os.path.join(target_dir, new_filename)
                
                original_target_file = new_target_file
                logger.info(f"중복 파일명 자동 변경: {file.filename} → {new_filename}")
            else:
                logger.info(f"덮어쓰기 모드: 기존 파일은 백그라운드에서 처리됩니다: {original_target_file}")
        
        # 파일 저장 - 원본 파일명 사용
        target_file = original_target_file
        logger.info(f"파일 저장 경로: {target_file}")
        
        try:
            # 파일 저장
            with open(target_file, "wb") as buffer:
                shutil.copyfileobj(file.file, buffer)
            logger.info(f"파일 저장됨: {target_file}")
            
            # 파일 권한 설정
            os.chmod(target_file, 0o644)
            
            # 파일이 성공적으로 저장되었는지 확인
            if not os.path.exists(target_file):
                raise Exception("파일 저장 실패")

            # 업로드 엔드포인트에서 직접 문서 처리
            logger.info(f"Final user ID for document processing: {effective_user_id}")
            
            # Redis 키 설정으로 File Watcher와 조정
            file_processing_key = f"file_processing:{target_file}"
            try:
                redis_client = get_redis_client()
                if redis_client:
                    processing_info = {
                        "status": "processing",
                        "source": "api",
                        "started_at": time.time(),
                        "user_id": effective_user_id
                    }
                    redis_client.set(file_processing_key, json.dumps(processing_info), ex=300)  # 5분 만료
                    logger.info(f"API 처리 키 설정: {file.filename}")
                else:
                    logger.info(f"Redis 연결 실패, 키 설정 없이 진행: {file.filename}")
            except Exception as e:
                logger.error(f"Redis 키 설정 실패: {str(e)}")
            
            # 즉시 chat_documents 테이블에 레코드 생성 (무결성 강화)
            try:
                file_size = os.path.getsize(target_file) if os.path.exists(target_file) else 0
                file_ext = os.path.splitext(file.filename)[1].lower()

                # MIME 타입 결정
                mime_type = 'application/pdf' if file_ext == '.pdf' else \
                           'application/msword' if file_ext == '.doc' else \
                           'application/vnd.openxmlformats-officedocument.wordprocessingml.document' if file_ext == '.docx' else \
                           'text/plain' if file_ext == '.txt' else \
                           'application/octet-stream'

                # Form 객체 처리
                effective_project_name = None
                if project_name and str(project_name) != 'None' and not str(project_name).startswith('annotation='):
                    effective_project_name = str(project_name) if not isinstance(project_name, str) else project_name
                    # 길이 제한 (데이터베이스 varchar(100) 제약)
                    if effective_project_name and len(effective_project_name) > 100:
                        effective_project_name = effective_project_name[:100]

                # 동기 DB 연결 풀 사용 (중앙 집중식)
                conn = get_pool_connection()
                try:
                    with conn.cursor() as cur:
                        cur.execute("""
                            INSERT INTO chat_documents (
                                user_id, filename, filepath, filesize, mimetype,
                                upload_status, embedding_status, project_name, created_at, updated_at
                            ) VALUES (%s, %s, %s, %s, %s, %s, %s, %s, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
                            ON CONFLICT (user_id, filename)
                            DO UPDATE SET
                                filepath = EXCLUDED.filepath,
                                filesize = EXCLUDED.filesize,
                                mimetype = EXCLUDED.mimetype,
                                upload_status = EXCLUDED.upload_status,
                                embedding_status = EXCLUDED.embedding_status,
                                project_name = EXCLUDED.project_name,
                                updated_at = CURRENT_TIMESTAMP
                        """, (effective_user_id, file.filename, f"{effective_user_id}/{file.filename}",
                              file_size, mime_type, 'uploaded', 'pending', effective_project_name))
                        conn.commit()
                finally:
                    return_pool_connection(conn)

                    logger.info(f"chat_documents 테이블에 즉시 레코드 생성: {file.filename} (상태: uploaded->pending)")
            except Exception as e:
                logger.error(f"❌ chat_documents 테이블 레코드 생성 실패: {file.filename}: {str(e)}")
                # 실패해도 업로드는 계속 진행
            
            # 임베딩 워커 프로세스로 요청 전송
            logger.info(f"파일 업로드 완료, 임베딩 워커로 처리 요청: {file.filename}")
            
            # 임베딩 워커 준비 상태 확인 (비동기, 실패해도 업로드 응답은 성공)
            if not embedding_model_ready_event.is_set():
                logger.info("임베딩 워커 모델 로딩 중 - 백그라운드에서 처리 예정")
            
            # .extracts 폴더의 임시 이미지 파일 제외
            if '/.extracts/' in target_file or os.sep + '.extracts' + os.sep in target_file:
                logger.info(f"임베딩 제외: .extracts 폴더의 임시 파일 - {os.path.basename(target_file)}")
            else:
                import uuid
                embedding_request = {
                    'request_id': str(uuid.uuid4()),
                    'file_path': target_file,
                    'user_id': effective_user_id,
                    'temp_settings': temp_settings if temp_settings else None
                }
                embedding_queue.put(embedding_request)
                logger.info(f"임베딩 워커로 요청 전송 완료 - ID: {embedding_request['request_id']}")
                
                # Redis에 문서 처리 상태 등록 (진행 상태 추적을 위함)
                try:
                    update_doc_status(file.filename, 'embedding', '임베딩 처리 중', {'request_id': embedding_request['request_id']})
                    logger.info(f"Redis에 문서 처리 상태 등록: {file.filename} -> embedding")
                except Exception as e:
                    logger.error(f"Redis 상태 등록 실패: {str(e)}")
            
            return {
                "status": "processing",
                "message": "파일 업로드가 완료되었습니다. 임베딩을 처리 중이므로 잠시 후 검색에서 사용 가능합니다.",
                "file": file.filename,
                "user_id": effective_user_id,
                "rag_docs_path": rag_docs_path,
                "summary": {"processing_status": "embedding", "background_task": True}
            }

            
        except Exception as e:
            if os.path.exists(target_file):
                os.unlink(target_file)
            raise
            
    except Exception as e:
        logger.error(f"파일 업로드 처리 중 오류: {str(e)}")
        raise HTTPException(
            status_code=500,
            detail=str(e)
        )

@app.post("/upload-page-chunk", response_model=UploadResponse)
async def upload_document_page_chunk(
    file: UploadFile = File(...),
    user_id: Optional[str] = Form(None),
    overwrite: bool = Form(False)
):
    """
    기술 지원 문서처럼 페이지 단위 청킹이 필요한 파일을 업로드합니다.
    이 엔드포인트는 시스템의 전역 설정을 변경하지 않고,
    해당 파일 처리에만 페이지 단위 청킹 설정을 강제로 적용합니다.
    """
    logger.info(f"\n페이지 단위 청킹 업로드 요청:")
    logger.info(f"- 파일명: {file.filename}")
    logger.info(f"- 사용자 ID: {user_id if user_id else 'all'}")

    original_settings = {}
    try:
        # 1. 현재 설정을 임시로 변경
        # DocumentProcessor 인스턴스의 설정을 직접 변경합니다.
        global document_processor
        if document_processor and hasattr(document_processor, 'settings'):
            # 원래 설정 백업
            original_settings = {
                'rag_mode': document_processor.settings.get('rag_mode'),
                'pdf_backend': document_processor.settings.get('pdf_backend'),
                'chunk_size': document_processor.settings.get('chunk_size'),
                'chunk_overlap': document_processor.settings.get('chunk_overlap'),
                'semantic_chunker': document_processor.settings.get('semantic_chunker'),
                'chunking_strategy': document_processor.settings.get('chunking_strategy')
            }
            # 페이지 단위 청킹을 위한 설정 적용
            document_processor.settings['rag_mode'] = 'normal'
            document_processor.settings['pdf_backend'] = 'pdfplumber'
            document_processor.settings['chunk_size'] = 0  # 페이지 단위 청킹을 위해 0으로 설정
            document_processor.settings['chunk_overlap'] = 0  # 페이지 간 오버랩 없음
            document_processor.settings['semantic_chunker'] = False  # 시맨틱 청킹 비활성화
            document_processor.settings['chunking_strategy'] = 'page'  # 페이지 단위 청킹 전략 설정
            logger.info(f"임시 설정 적용: {document_processor.settings} (원래 설정: {original_settings})")

        # 2. 기존 /upload 엔드포인트의 로직을 재사용하여 파일 처리
        # 지원되는 파일 확장자 확인
        file_ext = os.path.splitext(file.filename)[1].lower()
        if file_ext not in SUPPORTED_EXTENSIONS:
            raise HTTPException(status_code=400, detail=f"Unsupported file type: {file_ext}")

        rag_docs_path = get_rag_docs_path()
        # Form 객체와 문자열 모두 처리
        if hasattr(user_id, 'strip'):  # 문자열인 경우
            effective_user_id = user_id.strip() if user_id and user_id.strip() else 'all'
        else:  # Form 객체인 경우
            effective_user_id = str(user_id).strip() if user_id and str(user_id).strip() else 'all'
        target_dir = os.path.join(rag_docs_path, effective_user_id) if effective_user_id != 'all' else rag_docs_path
        os.makedirs(target_dir, exist_ok=True)

        # 중복 파일 확인
        target_file = os.path.join(target_dir, file.filename)
        logger.info(f"중복 파일 확인: {target_file}, 존재여부: {os.path.exists(target_file)}")
        
        if os.path.exists(target_file):
            logger.info(f"중복 파일 발견: {target_file}")
            if not overwrite:
                raise HTTPException(
                    status_code=409,  # 409 Conflict - 더 적절한 HTTP 상태 코드
                    detail=f"파일 '{file.filename}'이 이미 존재합니다. overwrite=true로 설정하여 덮어쓰기하거나 다른 이름으로 업로드해주세요."
                )
            else:
                logger.info(f"덮어쓰기 모드로 기존 파일 처리: {target_file}")
                # 기존 파일의 데이터 삭제
                try:
                    # PostgreSQL에서 기존 문서 데이터 삭제
                    import psycopg2
                    from psycopg2.extras import RealDictCursor
                    
                    try:
                        conn = get_pool_connection()
                        cursor = conn.cursor(cursor_factory=RealDictCursor)
                        
                        # 기존 문서 삭제 (파일명 기준)
                        delete_query = "DELETE FROM rag_documents WHERE filename = %s"
                        if effective_user_id != 'all':
                            delete_query += " AND user_id = %s"
                            cursor.execute(delete_query, (file.filename, effective_user_id))
                        else:
                            cursor.execute(delete_query, (file.filename,))
                        
                        deleted_count = cursor.rowcount
                        conn.commit()
                        cursor.close()
                        return_pool_connection(conn)
                        
                        if deleted_count > 0:
                            logger.info(f"PostgreSQL에서 기존 문서 삭제: {file.filename} ({deleted_count}개 청크)")
                    
                    except Exception as pg_error:
                        logger.error(f"PostgreSQL 기존 문서 삭제 실패: {str(pg_error)}")
                    
                    # PostgreSQL에서 기존 문서 벡터 삭제
                    if document_processor and document_processor.collection:
                        effective_user_id = user_id.strip() if user_id and user_id.strip() else 'all'
                        where_filter = {}
                        if effective_user_id != 'all':
                            where_filter["user_id"] = effective_user_id
                        
                        chunk_ids_to_delete = []
                        # 기존 청크 검색 및 삭제
                        try:
                            # 방법 1: 전체 경로로 검색
                            file_filter = {**where_filter, "source": target_file}
                            chunk_results = document_processor.collection.get(where=file_filter)
                            if chunk_results['ids']:
                                chunk_ids_to_delete.extend(chunk_results['ids'])
                            
                            # 방법 2: 파일명만으로 검색
                            file_filter = {**where_filter, "source": file.filename}
                            chunk_results = document_processor.collection.get(where=file_filter)
                            if chunk_results['ids']:
                                new_ids = [id for id in chunk_results['ids'] if id not in chunk_ids_to_delete]
                                chunk_ids_to_delete.extend(new_ids)
                            
                            # 청크 삭제 실행
                            if chunk_ids_to_delete:
                                document_processor.collection.delete(ids=chunk_ids_to_delete)
                                logger.info(f"기존 벡터 데이터 삭제 완료: {file.filename} ({len(chunk_ids_to_delete)}개 청크)")
                            else:
                                logger.info(f"삭제할 기존 벡터 데이터를 찾을 수 없음: {file.filename}")
                        except Exception as delete_error:
                            logger.error(f"벡터 삭제 중 오류: {str(delete_error)}")
                            raise delete_error
                except Exception as e:
                    logger.error(f"기존 벡터 데이터 삭제 실패: {str(e)}")
                    # 벡터 삭제 실패시 업로드 중단 (데이터 일관성을 위해)
                    raise HTTPException(
                        status_code=500,
                        detail=f"기존 파일의 벡터 데이터 삭제에 실패했습니다: {str(e)}"
                    )

        with open(target_file, "wb") as buffer:
            shutil.copyfileobj(file.file, buffer)
        
        os.chmod(target_file, 0o644)
        logger.info(f"파일 저장됨: {target_file}")

        # 협조적 처리 큐를 위한 Redis 키 설정 (실패해도 서비스 계속)
        file_processing_key = None
        try:
            import json
            import time
            redis_client = get_redis_client()
            if redis_client:
                file_processing_key = f"file_processing:{target_file}"
                processing_info = {
                    "source": "api", 
                    "status": "processing", 
                    "started_at": time.time(),
                    "filename": file.filename
                }
                redis_client.setex(file_processing_key, 3600, json.dumps(processing_info))
                logger.info(f"페이지 청크 API 처리 시작 마커 설정: {file.filename}")
            else:
                logger.info(f"Redis 연결 실패, 마커 없이 페이지 청크 처리 진행: {file.filename}")
        except Exception as e:
            logger.error(f"페이지 청크 처리 마커 설정 실패, 계속 진행: {str(e)}")
            file_processing_key = None

        try:
            # 문서 처리 전에 chunking_strategy를 'page'로 설정
            document_processor.settings['chunking_strategy'] = 'page'
            
            # 문서 처리
            results = await document_processor.add_documents(
                [target_file],
                effective_user_id if effective_user_id != 'all' else '',
                rag_docs_path,
                None  # 기본 설정 사용
            )

            return {
                "status": "success",
                "message": "파일이 성공적으로 업로드되고 페이지 단위로 처리되었습니다.",
                "file": file.filename,
                "user_id": effective_user_id,
                "rag_docs_path": rag_docs_path,
                "summary": results
            }
        finally:
            # 처리 완료 후 Redis 키 삭제
            try:
                if upload_processed_key:
                    redis_client.delete(upload_processed_key)
                    logger.info(f"업로드 처리 마커 제거: {file.filename}")
            except Exception as e:
                logger.error(f"업로드 처리 마커 제거 오류: {str(e)}")

    except Exception as e:
        logger.error(f"페이지 단위 청킹 업로드 처리 중 오류: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))
    finally:
        # 3. 원래 설정으로 복원
        if original_settings and document_processor:
            for key, value in original_settings.items():
                document_processor.settings[key] = value
            if 'chunking_strategy' in document_processor.settings:
                del document_processor.settings['chunking_strategy']
            logger.info(f"원래 설정 복원: {original_settings}")

def get_current_time():
    """현재 시간을 UTC+9 기준으로 반환합니다."""
    return datetime.now(timezone(timedelta(hours=9)))

def format_duration(duration):
    """timedelta 객체를 사용자 친화적인 형식으로 포맷합니다."""
    if not duration:
        return None

    total_seconds = int(duration.total_seconds())

    # 시간, 분, 초 계산
    hours = total_seconds // 3600
    minutes = (total_seconds % 3600) // 60
    seconds = total_seconds % 60

    # 적절한 형식으로 반환
    if hours > 0:
        return f"{hours}시간 {minutes}분 {seconds}초"
    elif minutes > 0:
        return f"{minutes}분 {seconds}초"
    elif seconds > 0:
        return f"{seconds}초"
    else:
        return "1초 미만"

@app.post("/rag-init")
async def rag_init(
    force: bool = Query(False, description="강제로 새로운 초기화를 수행할지 여부"),
    user_id: Optional[str] = Query(None, description="사용자별 초기화시 사용자 ID (없으면 전체 시스템 초기화)"),
    document_name: Optional[str] = Query(None, description="특정 문서 삭제시 문서명 (user_id와 함께 사용)")
):
    """RAG 문서 초기화 엔드포인트 - PostgreSQL 테이블을 초기화하고 문서를 처리합니다."""
    try:
        global document_processor
        
        # Redis 클라이언트 확인 및 재연결
        redis_client = get_redis_client()

        # 디버깅: 파라미터 값 확인
        logger.info(f"🔍 rag-init 파라미터 확인: user_id='{user_id}', document_name='{document_name}', force={force}")
        logger.info(f"🔍 user_id 타입: {type(user_id)}, user_id 값: {repr(user_id)}")

        # 사용자별/문서별 초기화 처리
        logger.info(f"🔍 조건 확인: user_id={user_id}, bool(user_id)={bool(user_id)}")
        if user_id and user_id.strip():
            logger.info("🔍 사용자별 초기화 경로로 진입")
            if document_name and document_name.strip():
                # 특정 문서 삭제
                logger.info("\n=== 특정 문서 RAG 초기화 시작 ===")
                logger.info(f"- 사용자 ID: {user_id}")
                logger.info(f"- 문서명: {document_name}")
                
                # 1. 특정 문서 임베딩 데이터 존재 확인
                try:
                    # PostgreSQL 연결
                    pg_connection = get_pool_connection()
                    pg_cursor = pg_connection.cursor()
                    
                    # 해당 문서의 임베딩 데이터 확인
                    pg_cursor.execute("""
                        SELECT COUNT(*) FROM document_embeddings 
                        WHERE metadata->>'user_id' = %s 
                        AND (metadata->>'source' = %s 
                             OR metadata->>'source' LIKE %s 
                             OR metadata->>'source' LIKE %s)
                    """, (user_id, document_name, f'%/{document_name}', f'%{document_name}%'))
                    
                    embedding_count = pg_cursor.fetchone()[0]
                    pg_cursor.close()
                    return_pool_connection(pg_connection)
                    
                    # 2. 해당 문서 파일 경로 확인
                    rag_docs_path = get_rag_docs_path()
                    user_docs_path = os.path.join(rag_docs_path, user_id)
                    document_path = os.path.join(user_docs_path, document_name)
                    
                    if not os.path.exists(document_path):
                        logger.error(f"문서 파일이 존재하지 않습니다: {document_path}")
                        return {
                            "status": "error",
                            "message": f"문서 파일 '{document_name}'이 존재하지 않습니다."
                        }
                    
                    # 3. 기존 임베딩 데이터가 있으면 삭제 (파일은 유지)
                    if embedding_count > 0:
                        try:
                            # PostgreSQL에서 임베딩 데이터만 삭제 (파일은 유지)
                            pg_connection = get_pool_connection()
                            pg_cursor = pg_connection.cursor()

                            # GraphRAG 데이터 삭제 (외래키 제약조건 순서 고려)
                            # 먼저 해당 문서와 관련된 source_documents 찾기 (chunk 포함)
                            pg_cursor.execute("""
                                SELECT DISTINCT unnest(source_documents) as source_doc
                                FROM graph_entities
                                WHERE EXISTS (
                                    SELECT 1 FROM unnest(source_documents) AS sd
                                    WHERE sd LIKE %s OR sd = %s OR sd LIKE %s
                                )
                            """, (f'%{document_name}%', document_name, f'{document_name}_chunk_%'))
                            matching_sources = [row[0] for row in pg_cursor.fetchall()]

                            if matching_sources:
                                logger.debug(f"삭제할 문서 '{document_name}'와 매칭된 소스들: {matching_sources}")

                                # 1. 커뮤니티 삭제
                                pg_cursor.execute("""
                                    DELETE FROM graph_communities gc
                                    WHERE EXISTS (
                                        SELECT 1
                                        FROM graph_entities ge
                                        WHERE ge.id = ANY(gc.entities)
                                        AND ge.source_documents && %s
                                    );
                                """, (matching_sources,))

                                # 2. 관계 삭제
                                pg_cursor.execute("""
                                    DELETE FROM graph_relationships
                                    WHERE source_entity_id IN (
                                        SELECT id FROM graph_entities
                                        WHERE source_documents && %s
                                    )
                                    OR target_entity_id IN (
                                        SELECT id FROM graph_entities
                                        WHERE source_documents && %s
                                    );
                                """, (matching_sources, matching_sources))

                                # 3. 엔티티 삭제
                                pg_cursor.execute("""
                                    DELETE FROM graph_entities
                                    WHERE source_documents && %s;
                                """, (matching_sources,))
                            else:
                                logger.debug(f"문서 '{document_name}'와 매칭되는 Graph RAG 데이터를 찾지 못함")

                            # 4. 문서 임베딩 삭제 (파일명으로 매칭)
                            pg_cursor.execute("""
                                DELETE FROM document_embeddings
                                WHERE metadata->>'user_id' = %s
                                AND (metadata->>'source' = %s
                                     OR metadata->>'source' LIKE %s
                                     OR metadata->>'source' LIKE %s)
                            """, (user_id, document_name, f'%/{document_name}', f'%{document_name}%'))

                            chunks_deleted = pg_cursor.rowcount
                            pg_connection.commit()
                            pg_cursor.close()
                            return_pool_connection(pg_connection)

                            # 캐시 정리
                            await clear_cache()

                            logger.info(f"사용자 {user_id}의 문서 '{document_name}' 기존 임베딩 데이터가 삭제되었습니다. (청크 {chunks_deleted}개)")

                        except Exception as e:
                            error_msg = f"사용자 {user_id}의 문서 '{document_name}' 임베딩 삭제 실패: {str(e)}"
                            logger.error(error_msg)
                            return {
                                "status": "error",
                                "message": error_msg
                            }
                    else:
                        logger.info(f"문서 '{document_name}'에는 기존 임베딩 데이터가 없습니다. 새로 임베딩을 시작합니다.")
                    
                    # 4. Redis 초기화 정보 설정
                    start_time = datetime.now()
                    redis_client.set("rag:init:start_time", start_time.isoformat().encode())
                    redis_client.delete("rag:init:end_time")
                    redis_client.set("rag:init:user_id", user_id.encode())
                    redis_client.set("rag:init:in_progress", "true")
                    
                    # 5. 해당 문서를 Redis 큐에 추가하여 재임베딩 시작
                    redis_client.delete(QUEUE_KEYS['documents'])
                    redis_client.delete(QUEUE_KEYS['processing'])
                    redis_client.delete(QUEUE_KEYS['completed'])
                    
                    # 문서 절대 경로를 큐에 추가
                    abs_document_path = os.path.abspath(document_path)
                    redis_client.lpush(QUEUE_KEYS['documents'], abs_document_path)
                    
                    logger.info(f"문서 '{document_name}' 재임베딩을 위해 큐에 추가되었습니다: {abs_document_path}")
                    
                    # 6. 비동기적으로 배치 처리 시작
                    asyncio.create_task(process_next_batch())
                    
                    return {
                        "status": "success",
                        "message": f"문서 '{document_name}' 초기화가 시작되었습니다. 기존 임베딩을 삭제하고 새로 처리합니다.",
                        "user_id": user_id,
                        "document_name": document_name,
                        "operation": "document_reinitialize",
                        "previous_embeddings": embedding_count,
                        "file_path": abs_document_path,
                        "monitoring": {
                            "queue_status": "http://localhost:5600/rag/queue-status",
                            "progress": "http://localhost:5600/rag/progress"
                        }
                    }
                except Exception as e:
                    logger.error(f"특정 문서 삭제 중 오류: {str(e)}")
                    return {
                        "status": "error",
                        "message": f"문서 삭제 중 오류: {str(e)}"
                    }
            else:
                # 사용자별 전체 문서 삭제
                logger.info("\n=== 사용자별 RAG 초기화 시작 ===")
                logger.info(f"- 사용자 ID: {user_id}")
                
                # 1. 해당 사용자의 임베딩과 Graph RAG 데이터만 삭제 (chat_documents는 유지)
                try:
                    pg_connection = get_pool_connection()
                    pg_cursor = pg_connection.cursor()

                    # 사용자의 문서 소스 목록 수집
                    pg_cursor.execute("""
                        SELECT DISTINCT de.metadata->>'source'
                        FROM document_embeddings de
                        WHERE de.metadata->>'user_id' = %s
                    """, (user_id,))
                    embedding_sources = [row[0] for row in pg_cursor.fetchall()]

                    # Graph RAG에서 매칭되는 모든 소스 찾기
                    user_sources = []
                    if embedding_sources:
                        for base_source in embedding_sources:
                            pg_cursor.execute("""
                                SELECT DISTINCT unnest(source_documents) as source_doc
                                FROM graph_entities
                                WHERE EXISTS (
                                    SELECT 1 FROM unnest(source_documents) AS sd
                                    WHERE sd LIKE %s
                                )
                            """, (f'{base_source}%',))
                            matching_sources = [row[0] for row in pg_cursor.fetchall()]
                            user_sources.extend(matching_sources)

                        user_sources = list(set(user_sources))  # 중복 제거
                        logger.debug(f"RAG 초기화: Graph RAG 소스들 {user_sources}")

                        if user_sources:
                            # Graph RAG 데이터 삭제 (외래키 제약조건 순서)
                            # 1) 커뮤니티 삭제
                            pg_cursor.execute("""
                                DELETE FROM graph_communities gc
                                WHERE EXISTS (
                                    SELECT 1 FROM graph_entities ge
                                    WHERE ge.id = ANY(gc.entities)
                                    AND ge.source_documents && %s
                                );
                            """, (user_sources,))
                            communities_deleted = pg_cursor.rowcount

                            # 2) 관계 삭제
                            pg_cursor.execute("""
                                DELETE FROM graph_relationships
                                WHERE source_entity_id IN (
                                    SELECT id FROM graph_entities WHERE source_documents && %s
                                )
                                OR target_entity_id IN (
                                    SELECT id FROM graph_entities WHERE source_documents && %s
                                );
                            """, (user_sources, user_sources))
                            relationships_deleted = pg_cursor.rowcount

                            # 3) 엔티티 삭제
                            pg_cursor.execute("""
                                DELETE FROM graph_entities WHERE source_documents && %s;
                            """, (user_sources,))
                            entities_deleted = pg_cursor.rowcount

                    # document_embeddings 삭제
                    pg_cursor.execute("""
                        DELETE FROM document_embeddings WHERE metadata->>'user_id' = %s;
                    """, (user_id,))
                    embeddings_deleted = pg_cursor.rowcount

                    # chat_documents 상태를 pending으로 업데이트 (삭제하지 않음)
                    pg_cursor.execute("""
                        UPDATE chat_documents
                        SET embedding_status = 'pending', processed_at = NULL, updated_at = CURRENT_TIMESTAMP
                        WHERE user_id = %s;
                    """, (user_id,))
                    docs_updated = pg_cursor.rowcount

                    pg_connection.commit()
                    pg_cursor.close()
                    return_pool_connection(pg_connection)

                    logger.info(f"RAG 초기화 완료:")
                    logger.info(f"  - 임베딩 삭제: {embeddings_deleted}개")
                    if user_sources:
                        logger.info(f"  - Graph RAG 엔티티 삭제: {entities_deleted}개")
                        logger.info(f"  - Graph RAG 관계 삭제: {relationships_deleted}개")
                        logger.info(f"  - Graph RAG 커뮤니티 삭제: {communities_deleted}개")
                    logger.info(f"  - chat_documents 상태 업데이트: {docs_updated}개")

                except Exception as e:
                    logger.error(f"사용자 RAG 데이터 초기화 중 오류: {str(e)}")
                    if pg_connection:
                        pg_connection.rollback()
                        pg_cursor.close()
                        return_pool_connection(pg_connection)
                    return {
                        "status": "error",
                        "message": f"사용자 {user_id}의 RAG 데이터 초기화 중 오류: {str(e)}"
                    }
            
            # 2. 해당 사용자의 문서 폴더에서 파일들 수집하여 재처리
            rag_docs_path = get_rag_docs_path()
            user_docs_path = os.path.join(rag_docs_path, user_id)
            
            if not os.path.exists(user_docs_path):
                logger.info(f"사용자 {user_id}의 문서 폴더가 존재하지 않습니다: {user_docs_path}")
                return {
                    "status": "success",
                    "message": f"사용자 {user_id}의 문서 폴더가 없어 초기화할 내용이 없습니다.",
                    "user_id": user_id,
                    "total_documents": 0
                }
            
            # 3. 사용자 폴더의 문서들 수집하여 Redis 큐에 추가 (기존 시스템과 동일하게)
            try:
                # 사용자 폴더에서 처리할 문서들 수집
                user_documents = []
                if os.path.exists(user_docs_path):
                    for root, _, files in os.walk(user_docs_path):
                        for file_name in files:
                            if any(file_name.lower().endswith(ext) for ext in SUPPORTED_EXTENSIONS):
                                file_path = os.path.join(root, file_name)
                                user_documents.append(os.path.abspath(file_path))
                
                if not user_documents:
                    logger.info(f"사용자 {user_id}의 처리할 문서가 없습니다.")
                    return {
                        "status": "success",
                        "message": f"사용자 {user_id}의 처리할 문서가 없습니다.",
                        "user_id": user_id,
                        "total_documents": 0
                    }
                
                # Redis 큐 초기화 (사용자별 초기화를 위해)
                redis_client.delete(QUEUE_KEYS['documents'])
                redis_client.delete(QUEUE_KEYS['processing'])
                redis_client.delete(QUEUE_KEYS['completed'])
                redis_client.delete(QUEUE_KEYS['failed'])
                redis_client.delete(f"{QUEUE_KEYS['failed']}:errors")
                redis_client.set(QUEUE_KEYS['batch_size'], str(BATCH_SIZE).encode())
                redis_client.set(QUEUE_KEYS['current_batch'], b'0')
                
                # 사용자 문서들을 Redis 큐에 추가 (list 타입 사용)
                for doc_path in user_documents:
                    redis_client.lpush(QUEUE_KEYS['documents'], doc_path.encode())
                
                logger.info(f"사용자 {user_id}의 {len(user_documents)}개 문서가 큐에 추가되었습니다.")

                # 해당 사용자의 문서들 embedding_status를 pending으로 업데이트
                try:
                    pg_connection = get_pool_connection()
                    pg_cursor = pg_connection.cursor()

                    # 큐에 추가된 문서들의 파일명 추출
                    pending_filenames = []
                    for doc_path in user_documents:
                        filename = os.path.basename(doc_path)
                        pending_filenames.append(filename)

                    if pending_filenames:
                        # 해당 사용자의 문서들 상태를 pending으로 업데이트
                        placeholders = ','.join(['%s'] * len(pending_filenames))
                        pg_cursor.execute(f"""
                            UPDATE chat_documents
                            SET embedding_status = 'pending',
                                processed_at = NULL,
                                updated_at = CURRENT_TIMESTAMP
                            WHERE user_id = %s AND filename IN ({placeholders})
                        """, [user_id] + pending_filenames)

                        updated_count = pg_cursor.rowcount
                        pg_connection.commit()
                        logger.info(f"사용자 {user_id}의 {updated_count}개 문서 상태를 pending으로 업데이트했습니다.")

                    pg_cursor.close()
                    return_pool_connection(pg_connection)

                except Exception as e:
                    logger.info(f"chat_documents 테이블 상태 업데이트 중 오류: {str(e)}", error=True)

                # 시작 시간 설정 (이전 종료 시간 삭제)
                start_time = get_current_time()
                redis_client.set("rag:init:start_time", start_time.isoformat().encode())
                redis_client.delete("rag:init:end_time")  # 이전 종료 시간 삭제
                logger.info(f"사용자별 RAG 초기화 시작 시간 저장: {start_time.isoformat()}")
                
                # 사용자별 초기화 정보 저장
                redis_client.set("rag:init:user_id", user_id.encode())
                logger.info(f"사용자별 초기화 정보 저장: {user_id}")
                
                # RAG 초기화 진행 중 플래그 설정
                redis_client.set("rag:init:in_progress", "true")
                logger.info("사용자별 RAG 초기화 진행 중 플래그 설정 완료")
                
                # 비동기적으로 배치 처리 시작
                asyncio.create_task(process_next_batch())
                
                logger.info(f"사용자 {user_id}의 문서 재처리가 시작되었습니다.")
                return {
                    "status": "success",
                    "message": f"사용자 {user_id}의 문서 처리가 시작되었습니다.",
                    "user_id": user_id,
                    "total_documents": len(user_documents),
                    "monitoring": {
                        "queue_status": "http://localhost:5600/rag/queue-status",
                        "progress": "http://localhost:5600/rag/progress"
                    }
                }
                
            except Exception as e:
                logger.error(f"사용자 문서 재처리 중 오류: {str(e)}")
                return {
                    "status": "error",
                    "message": f"사용자 {user_id}의 문서 재처리 중 오류: {str(e)}"
                }
        
        # 전체 시스템 초기화 (기존 로직)
        logger.info("🔍 전체 시스템 초기화 경로로 진입")
        logger.info(f"\n🟢 전체 시스템 RAG 초기화 시작! user_id={user_id}, document_name={document_name}")
        logger.info(f"=== 전체 시스템 RAG 초기화 시작 ===")

        # 먼저 모든 데이터 정리 (force 옵션과 관계없이)
        logger.info("🧹 전체 시스템 데이터 정리 시작")
        try:
            conn = get_pool_connection()
            cursor = conn.cursor()
            logger.info("🧹 데이터베이스 연결 완료")

            # 1. chat_documents 상태를 pending으로 업데이트
            logger.info("🧹 chat_documents 상태를 pending으로 업데이트 중...")
            cursor.execute("""
                UPDATE chat_documents
                SET embedding_status = 'pending',
                    processed_at = NULL,
                    updated_at = CURRENT_TIMESTAMP
            """)
            logger.info("🧹 chat_documents 업데이트 완료")

            # 2. 기존 임베딩 및 GraphRAG 데이터 삭제 (외래키 제약조건 순서 고려)
            logger.info("🧹 document_embeddings 테이블 삭제 중...")
            cursor.execute("DELETE FROM document_embeddings")
            logger.info("🧹 graph_communities 테이블 삭제 중... (1/3)")
            cursor.execute("DELETE FROM graph_communities")
            logger.info("🧹 graph_relationships 테이블 삭제 중... (2/3)")
            cursor.execute("DELETE FROM graph_relationships")
            logger.info("🧹 graph_entities 테이블 삭제 중... (3/3)")
            cursor.execute("DELETE FROM graph_entities")

            conn.commit()
            cursor.close()
            return_pool_connection(conn)
            logger.info("✅ 모든 chat_documents 상태가 pending으로 업데이트되고, 기존 임베딩 및 GraphRAG 데이터가 삭제되었습니다.")
        except Exception as e:
            logger.error(f"❌ 데이터 정리 실패: {str(e)}")
            import traceback
            logger.error(f"❌ 트레이스백: {traceback.format_exc()}")

        # 강제 초기화가 아닌 경우, 현재 진행 중인 작업이 있는지 확인
        if not force:
            current_status = await get_queue_status()
            if current_status["queue_status"]["processing"] > 0 or current_status["queue_status"]["remaining"] > 0:
                return {
                    "status": "error",
                    "message": "현재 진행 중인 작업이 있습니다. 강제 초기화를 원하시면 force=true를 사용하세요.",
                    "current_status": current_status
                }
        
        # 강제 초기화인 경우에만 시간 정보 초기화
        if force:
            redis_client.delete("rag:init:start_time")
            redis_client.delete("rag:init:end_time")
            logger.info("강제 초기화로 인해 이전 작업의 시간 정보를 초기화했습니다.")
            
            rag_cache_path = os.path.expanduser('~/.airun/rag_cache.json')
            if os.path.exists(rag_cache_path):
                os.remove(rag_cache_path)
                logger.info('RAG 검색 캐시 파일이 삭제되었습니다.')            
            
            # PostgreSQL 데이터베이스 완전 초기화
            try:
                # 기존 임베딩 서비스 정리
                if document_processor:
                    document_processor.embedding_service = None
                
                # PostgreSQL 연결 및 테이블 초기화
                embedding_service = EmbeddingService()
                
                # 기존 데이터는 이미 위에서 삭제되었으므로 여기서는 서비스만 초기화
                logger.info("PostgreSQL 기반 임베딩 서비스가 초기화되었습니다.")
                
                # DocumentProcessor 재초기화
                document_processor = None
                from plugins.rag.rag_process import DocumentProcessor
                document_processor = DocumentProcessor.get_instance()
                logger.info("DocumentProcessor가 재초기화되었습니다.")
                
                # 이미지 임베딩 모델 상태 확인
                if hasattr(document_processor, 'image_embedding_model') and document_processor.image_embedding_model:
                    logger.info("이미지 임베딩 모델 로드 완료 - 하이브리드 검색 활성화")
                else:
                    logger.info("⚠️ 이미지 임베딩 모델 미로드 - 텍스트만 검색")
                
                # 테이블이 비어있는지 확인
                try:
                    conn = get_pool_connection()
                    cursor = conn.cursor()
                    cursor.execute("SELECT COUNT(*) FROM document_embeddings")
                    count = cursor.fetchone()[0]
                    cursor.close()
                    embedding_service.return_db_connection(conn)
                    
                    if count > 0:
                        logger.error(f"경고: PostgreSQL 테이블에 {count}개의 항목이 존재합니다.")
                    else:
                        logger.info("확인: PostgreSQL 테이블이 비어있습니다.")
                except Exception as e:
                    logger.error(f"테이블 확인 중 오류: {str(e)}")
                
            except Exception as e:
                error_msg = f"PostgreSQL 초기화 실패: {str(e)}"
                logger.error(error_msg)
                return {
                    "status": "error",
                    "message": error_msg,
                    "error_documents": 0,
                    "total_documents": 0,
                    "error_messages": [error_msg]
                }
        
        # 시작 시간이 없는 경우에만 새로 저장 (이전 종료 시간도 삭제)
        if not redis_client.exists("rag:init:start_time"):
            start_time = get_current_time()
            redis_client.set("rag:init:start_time", start_time.isoformat().encode())
            redis_client.delete("rag:init:end_time")  # 이전 종료 시간 삭제
            logger.info(f"RAG 초기화 시작 시간 저장: {start_time.isoformat()}")
        else:
            start_time = datetime.fromisoformat(redis_client.get("rag:init:start_time").decode())
            if start_time.tzinfo is None:
                start_time = start_time.replace(tzinfo=timezone(timedelta(hours=9)))
            logger.info(f"기존 RAG 초기화 시작 시간 사용: {start_time.isoformat()}")
            
        # 전체 시스템 초기화 정보 저장 (user_id가 없으면 전체 시스템)
        redis_client.set("rag:init:user_id", b"all")
        logger.info("전체 시스템 초기화 정보 저장")
        
        # DocumentProcessor 초기화
        if document_processor is None:
            from plugins.rag.rag_process import DocumentProcessor
            document_processor = DocumentProcessor.get_instance()
            logger.info("DocumentProcessor가 초기화되었습니다.")
            
            # 이미지 임베딩 모델 상태 확인
            if hasattr(document_processor, 'image_embedding_model') and document_processor.image_embedding_model:
                logger.info("이미지 임베딩 모델 로드 완료 - 하이브리드 검색 활성화")
            else:
                logger.info("⚠️ 이미지 임베딩 모델 미로드 - 텍스트만 검색")

        # RAG 설정 확인 및 로깅
        rag_settings = get_rag_settings()
        rag_docs_path = rag_settings['rag_docs_path']
        
        logger.info("=== RAG 설정 확인 ===")
        logger.info(f"- RAG 모드: {rag_settings.get('rag_mode', 'fast')}")
        logger.info(f"- 텍스트 전용 처리: {rag_settings.get('process_text_only', 'true')}")
        logger.info(f"- 문서 경로: {rag_docs_path}")
        logger.info("===================")
        
        if not os.path.exists(rag_docs_path):
            return {
                "status": "error",
                "message": "문서 디렉토리가 존재하지 않습니다.",
                "error_documents": 0,
                "total_documents": 0,
                "error_messages": []
            }

        # Redis 큐 초기화
        redis_client.delete(QUEUE_KEYS['documents'])
        redis_client.delete(QUEUE_KEYS['processing'])
        redis_client.delete(QUEUE_KEYS['completed'])
        redis_client.delete(QUEUE_KEYS['failed'])
        redis_client.delete(f"{QUEUE_KEYS['failed']}:errors")
        redis_client.set(QUEUE_KEYS['batch_size'], str(BATCH_SIZE).encode())
        redis_client.set(QUEUE_KEYS['current_batch'], b'0')

        # 문서 목록 수집
        documents = []
        for root, _, files in os.walk(rag_docs_path):
            # .extracts 폴더 제외
            if '/.extracts/' in root or os.sep + '.extracts' + os.sep in root:
                continue
                
            for file_name in files:
                if any(file_name.lower().endswith(ext) for ext in SUPPORTED_EXTENSIONS):
                    file_path = os.path.join(root, file_name)
                    # .extracts 폴더 내 파일 제외 (추가 보안)
                    if '/.extracts/' in file_path or os.sep + '.extracts' + os.sep in file_path:
                        continue
                    documents.append(os.path.abspath(file_path))

        if not documents:
            return {
                "status": "success",
                "message": "처리할 문서가 없습니다.",
                "total_documents": 0,
                "error_documents": 0,
                "error_messages": []
            }

        # 문서들을 Redis 큐에 추가
        for doc_path in documents:
            redis_client.lpush(QUEUE_KEYS['documents'], doc_path.encode())

        logger.info(f"총 {len(documents)}개의 문서가 큐에 추가되었습니다.")

        # RAG 초기화 진행 중 플래그 설정
        redis_client.set("rag:init:in_progress", "true")
        logger.info("RAG 초기화 진행 중 플래그 설정 완료")

        # 비동기적으로 배치 처리 시작
        asyncio.create_task(process_next_batch())

        # 초기화 응답 구성
        response = {
            "status": "success",
            "message": "문서 처리가 시작되었습니다.",
            "total_documents": len(documents),
            "start_time": datetime.now().isoformat(),
            "queue_info": {
                "batch_size": BATCH_SIZE,
                "estimated_batches": (len(documents) + BATCH_SIZE - 1) // BATCH_SIZE,
                "force_initialized": force
            },
            "monitoring": {
                "queue_status": "http://localhost:5600/rag/queue-status",
                "progress": "http://localhost:5600/rag/progress",
                "total_time": "http://localhost:5600/rag/total-time"
            }
        }

        # 로그 출력 개선
        logger.info("\n=== RAG 초기화 시작 ===")
        logger.info(f"- 강제 초기화: {'예' if force else '아니오'}")
        logger.info(f"- 총 문서 수: {len(documents)}개")
        logger.info(f"- 배치 크기: {BATCH_SIZE}개")
        logger.info(f"- 예상 배치 수: {response['queue_info']['estimated_batches']}개")
        logger.info("\n모니터링 정보:")
        logger.info(f"- 큐 상태: {response['monitoring']['queue_status']}")
        logger.info(f"- 진행 상황: {response['monitoring']['progress']}")
        logger.info("=====================\n")

        return response

    except Exception as e:
        error_message = str(e)
        logger.error(f"RAG 초기화 중 오류 발생: {error_message}")
        return {
            "status": "error",
            "message": error_message,
            "error_documents": 0,
            "total_documents": 0,
            "error_messages": [error_message]
        }

def decode_redis_value(value):
    """Redis에서 가져온 값을 디코딩합니다."""
    if isinstance(value, bytes):
        return value.decode('utf-8')
    return value

@app.get("/rag/queue-status")
async def get_queue_status():
    """큐 상태 조회 엔드포인트"""
    try:
        redis_client = get_redis_client()
        queue_manager = QueueManager(redis_client)
        
        # 각 큐의 상태를 정확히 추적 (set과 list 타입 모두 처리)
        def get_queue_members(queue_key):
            try:
                # 큐 타입 확인
                queue_type = redis_client.type(queue_key)
                if queue_type == b'set':
                    # set 타입인 경우
                    return set(decode_redis_value(doc) for doc in redis_client.smembers(queue_key))
                elif queue_type == b'list':
                    # list 타입인 경우
                    return set(decode_redis_value(doc) for doc in redis_client.lrange(queue_key, 0, -1))
                else:
                    # 존재하지 않거나 다른 타입인 경우
                    return set()
            except Exception as e:
                logger.error(f"큐 {queue_key} 조회 실패: {str(e)}")
                return set()

        queue_states = {
            'documents': get_queue_members(QUEUE_KEYS['documents']),
            'processing': get_queue_members(QUEUE_KEYS['processing']),
            'completed': get_queue_members(QUEUE_KEYS['completed']),
            'failed': get_queue_members(QUEUE_KEYS['failed'])
        }
        
        # 중복 제거된 전체 문서 수 계산
        all_docs = set().union(*queue_states.values())
        total_docs = len(all_docs)
        
        # 각 상태별 문서 수
        counts = {k: len(v) for k, v in queue_states.items()}
        
        # 진행률 계산
        progress = (counts['completed'] / total_docs * 100) if total_docs > 0 else 0
        
        # 실패한 문서 목록
        failed_docs = {}
        for doc in queue_states['failed']:
            error = redis_client.hget(f"{QUEUE_KEYS['failed']}:errors", doc.encode())
            if error:
                failed_docs[doc] = decode_redis_value(error)

        # 현재 처리 상태 결정 (로직 개선)
        current_status = "pending"  # 기본값을 pending으로 변경
        
        # 문서 처리 상태에 따른 상태 결정
        if counts['documents'] == 0 and counts['processing'] == 0:
            if counts['failed'] > 0:
                current_status = "partial_success"
            else:
                current_status = "success"
        elif counts['processing'] > 0:
            current_status = "processing"
        elif counts['documents'] > 0:
            current_status = "pending"
        
        # 상태 설명 메시지 개선 - 완료 상태를 더 명확하게
        status_messages = {
            "success": "모든 문서 처리가 완료되었습니다.",
            "processing": f"문서 처리가 진행 중입니다. (진행률: {round(progress, 1)}%)",
            "partial_success": f"일부 문서 처리에 실패했습니다. (성공: {counts['completed']}개, 실패: {counts['failed']}개)",
            "pending": f"문서 처리가 대기 중입니다. (대기: {counts['documents']}개, 완료: {counts['completed']}개)"
        }
        
        # 현재 배치 정보
        stored_batch = int(redis_client.get(QUEUE_KEYS['current_batch']) or 0)
        batch_size = int(redis_client.get(QUEUE_KEYS['batch_size']) or BATCH_SIZE)
        total_batches = (total_docs + batch_size - 1) // batch_size
        
        # 실제 처리 중인 배치 번호 계산 (현재 배치 + 1)
        current_batch = stored_batch + 1 if counts['processing'] > 0 else stored_batch
        
        # 현재 초기화 유형 정보 가져오기
        init_user_id = redis_client.get("rag:init:user_id")
        if init_user_id:
            init_user_id = decode_redis_value(init_user_id)
        
        # 문서별 상세 상태 정보 생성
        progress_detail = {
            "documents": []
        }
        
        # Redis에서 문서별 상태 정보 수집
        try:
            # rag:status:* 키로 저장된 문서별 상태 정보 수집
            status_keys = redis_client.keys("rag:status:*")
            for key in status_keys:
                try:
                    status_data = redis_client.get(key)
                    if status_data:
                        doc_status = json.loads(decode_redis_value(status_data))
                        # 이미지 파일 제외
                        filename = doc_status.get('filename', '')
                        if not (filename.endswith(('.jpg', '.jpeg', '.png', '.gif', '.bmp', '.svg', '.webp')) or
                               re.match(r'^[a-f0-9]+_page_\d+_\d+\.', filename)):

                            # 현재 초기화 중인 사용자의 문서만 포함 (user_id 필터링)
                            doc_user_id = doc_status.get('user_id', '')
                            if init_user_id:
                                # user_id가 없는 문서는 현재 초기화 중인 사용자의 문서로 간주 (임시 해결책)
                                if doc_user_id == init_user_id or doc_user_id == '':
                                    progress_detail["documents"].append(doc_status)
                            else:
                                # 초기화 중인 사용자가 없으면 모든 문서 포함 (기본 동작)
                                progress_detail["documents"].append(doc_status)
                except Exception as e:
                    logger.error(f"문서 상태 파싱 실패 ({key}): {str(e)}")
        except Exception as e:
            logger.error(f"문서별 상태 수집 실패: {str(e)}")
        
        # 큐 상태 기반으로 누락된 문서들 추가
        for queue_name, doc_paths in queue_states.items():
            for doc_path in doc_paths:
                filename = os.path.basename(doc_path)
                # 이미지 파일 제외
                if filename.endswith(('.jpg', '.jpeg', '.png', '.gif', '.bmp', '.svg', '.webp')) or \
                   re.match(r'^[a-f0-9]+_page_\d+_\d+\.', filename):
                    continue
                    
                # 이미 상태 정보가 있는 문서는 건너뛰기
                if any(doc.get('filename') == filename for doc in progress_detail["documents"]):
                    continue
                
                # 큐 상태에 따른 기본 상태 정보 생성 (프론트엔드 시각화 단계와 일치)
                step_mapping = {
                    'documents': 'queue',
                    'processing': 'extract',  # 처리 중인 문서는 기본적으로 텍스트 추출 단계로 설정
                    'completed': 'done',
                    'failed': 'error'
                }
                
                step_label_mapping = {
                    'queue': '준비',
                    'extract': '텍스트 추출 중',
                    'preprocess': '전처리 중',
                    'split': '청크 분할 중',
                    'metadata': '메타데이터 생성 중',
                    'embedding': '임베딩 중',
                    'save': 'DB 저장 중',
                    'done': '완료',
                    'error': '오류'
                }
                
                step = step_mapping.get(queue_name, 'queue')
                step_label = step_label_mapping.get(step, step)
                
                doc_status = {
                    "filename": filename,
                    "step": step,
                    "step_label": step_label,
                    "timestamp": datetime.now().isoformat()
                }
                
                # 실패한 문서인 경우 에러 정보 추가
                if queue_name == 'failed' and filename in failed_docs:
                    doc_status["error"] = failed_docs[filename]
                
                progress_detail["documents"].append(doc_status)

        return {
            "status": current_status,
            "message": status_messages.get(current_status, "알 수 없는 상태입니다."),
            "user_id": init_user_id,  # 현재 초기화 유형 정보 추가
            "queue_status": {
                "total_documents": total_docs,
                "remaining": counts['documents'],
                "processing": counts['processing'],
                "completed": counts['completed'],
                "failed": counts['failed'],
                "progress_percentage": round(progress, 2),
                "current_batch": current_batch,  # 수정된 배치 번호 사용
                "total_batches": total_batches,
                "batch_size": batch_size,
                "batch_progress": f"{current_batch}/{total_batches}"  # 수정된 배치 번호 사용
            },
            "progress_detail": progress_detail,  # 문서별 상세 상태 추가
            "failed_documents": failed_docs,
            "timestamp": datetime.now().isoformat()
        }

    except Exception as e:
        logger.error(f"큐 상태 조회 실패: {str(e)}")
        return {
            "status": "error",
            "message": str(e),
            "timestamp": datetime.now().isoformat()
        }

@app.get("/rag/admin/all-statuses")
async def get_all_embedding_statuses():
    """관리자용: 모든 사용자의 임베딩 상태 조회"""
    try:
        # 문서별 상세 상태 정보 생성 (사용자 필터링 없이 모든 문서)
        all_statuses = {}

        # 데이터베이스에서 실제 파일 상태도 확인 (Redis와 비교용)
        db_statuses = {}
        try:
            pg_connection = get_pool_connection()
            pg_cursor = pg_connection.cursor()
            pg_cursor.execute("""
                SELECT filename, embedding_status, user_id, processed_at
                FROM chat_documents
                ORDER BY updated_at DESC
            """)
            for row in pg_cursor.fetchall():
                filename, embedding_status, user_id, processed_at = row
                db_statuses[filename] = {
                    "embedding_status": embedding_status,
                    "user_id": user_id,
                    "processed_at": processed_at.isoformat() if processed_at else None
                }

            return_pool_connection(pg_connection)
        except Exception as db_error:
            logger.error(f"데이터베이스 상태 조회 실패: {str(db_error)}")
            # DB 오류가 있어도 Redis 상태는 계속 처리

        # Redis에서 문서별 상태 정보 수집
        try:
            # rag:status:* 키로 저장된 문서별 상태 정보 수집
            status_keys = redis_client.keys("rag:status:*")
            for key in status_keys:
                try:
                    status_data = redis_client.get(key)
                    if status_data:
                        doc_status = json.loads(decode_redis_value(status_data))
                        # 이미지 파일 제외
                        filename = doc_status.get('filename', '')
                        if not (filename.endswith(('.jpg', '.jpeg', '.png', '.gif', '.bmp', '.svg', '.webp')) or
                               re.match(r'^[a-f0-9]+_page_\d+_\d+\.', filename)):

                            # Redis 상태 정보
                            redis_step = doc_status.get('step', 'done')
                            redis_is_processing = redis_step not in ['done', 'error', 'failed']

                            # 데이터베이스 상태와 비교
                            db_info = db_statuses.get(filename, {})
                            db_embedding_status = db_info.get('embedding_status', 'completed')
                            db_user_id = db_info.get('user_id', '')

                            # Redis에서 진행 중이면 DB 상태를 pending으로 업데이트해야 함
                            # documents_embeddings 임베딩 상태값 == completed , redis step == save인 상태. (즉, 임베딩이 완료되었는데 Redis에서는 아직 저장 중인 상태)
                            # chat_document의 임베딩 상태값이 complete -> pending으로 업데이트하고 있어 반복되는 문제가 발생.
                            if redis_is_processing and db_embedding_status == 'completed':
                                logger.warning(f"상태 불일치 감지: {filename} - Redis: {redis_step}, DB: {db_embedding_status}")
                                # 실시간으로 DB 상태를 'completed'로 재확인 및 기록
                                try:
                                    update_pg_connection = get_pool_connection()
                                    update_pg_cursor = update_pg_connection.cursor()
                                    update_pg_connection.commit()
                                    return_pool_connection(update_pg_connection)
                                    logger.info(f"DB 상태 동기화 완료: {filename} -> completed")

                                    # Redis 상태도 완료로 동기화
                                    doc_status['step'] = 'done'
                                    doc_status['step_label'] = '완료'
                                    doc_status['isProcessing'] = False
                                    try:
                                        redis_client.set(key, json.dumps(doc_status))
                                        logger.info(f"Redis 상태 동기화 완료: {filename} -> done")
                                    except Exception as redis_sync_error:
                                        logger.error(f"Redis 상태 동기화 실패: {filename} - {redis_sync_error}")

                                    redis_step = 'done'
                                    redis_is_processing = False

                                except Exception as sync_error:
                                    logger.error(f"DB 상태 동기화 실패: {filename} - {sync_error}")

                            # 모든 사용자의 문서 상태 포함 (필터링 없음)
                            all_statuses[filename] = {
                                "step": redis_step,
                                "step_label": doc_status.get('step_label', '완료'),
                                "isProcessing": redis_is_processing,
                                "user_id": doc_status.get('user_id', '') or db_user_id,
                                "timestamp": doc_status.get('timestamp', ''),
                                "db_status": db_embedding_status  # 디버깅용
                            }
                except Exception as e:
                    logger.error(f"문서 상태 파싱 실패 ({key}): {str(e)}")
        except Exception as e:
            logger.error(f"문서별 상태 수집 실패: {str(e)}")

        # 큐 상태도 확인하여 누락된 문서들 추가
        try:
            def get_queue_members(queue_key):
                try:
                    # 큐 타입 확인
                    queue_type = redis_client.type(queue_key)
                    if queue_type == b'set':
                        # set 타입인 경우
                        return set(decode_redis_value(doc) for doc in redis_client.smembers(queue_key))
                    elif queue_type == b'list':
                        # list 타입인 경우
                        return set(decode_redis_value(doc) for doc in redis_client.lrange(queue_key, 0, -1))
                    else:
                        # 존재하지 않거나 다른 타입인 경우
                        return set()
                except Exception as e:
                    logger.error(f"큐 {queue_key} 조회 실패: {str(e)}")
                    return set()

            queue_states = {
                'documents': get_queue_members(QUEUE_KEYS['documents']),
                'processing': get_queue_members(QUEUE_KEYS['processing']),
                'completed': get_queue_members(QUEUE_KEYS['completed']),
                'failed': get_queue_members(QUEUE_KEYS['failed'])
            }

            # 큐 상태 기반으로 누락된 문서들 추가
            for queue_name, doc_paths in queue_states.items():
                for doc_path in doc_paths:
                    filename = os.path.basename(doc_path)
                    # 이미지 파일 제외
                    if filename.endswith(('.jpg', '.jpeg', '.png', '.gif', '.bmp', '.svg', '.webp')) or \
                       re.match(r'^[a-f0-9]+_page_\d+_\d+\.', filename):
                        continue

                    # 이미 상태 정보가 있는 문서는 건너뛰기
                    if filename in all_statuses:
                        continue

                    # 큐 상태에 따른 기본 상태 정보 생성
                    step_mapping = {
                        'documents': 'queue',
                        'processing': 'extract',
                        'completed': 'done',
                        'failed': 'error'
                    }

                    step_label_mapping = {
                        'queue': '준비',
                        'extract': '텍스트 추출 중',
                        'done': '완료',
                        'error': '오류'
                    }

                    step = step_mapping.get(queue_name, 'done')
                    step_label = step_label_mapping.get(step, step)

                    all_statuses[filename] = {
                        "step": step,
                        "step_label": step_label,
                        "isProcessing": step not in ['done', 'error', 'failed'],
                        "user_id": "",  # 큐에서는 user_id 정보가 없음
                        "timestamp": datetime.now().isoformat()
                    }
        except Exception as e:
            logger.error(f"큐 상태 조회 실패: {str(e)}")

        return {
            "documents": all_statuses
        }

    except Exception as e:
        logger.error(f"전체 임베딩 상태 조회 실패: {str(e)}")
        import traceback
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/list")
async def list_documents(user_id: Optional[str] = None):
    """문서 목록을 조회합니다. 사용자 ID를 지정하면 해당 사용자의 문서만 조회합니다."""
    try:
        # PostgreSQL 연결 (connection pool 사용)
        import json
        
        conn = get_pool_connection()
        cursor = conn.cursor()
        
        # where 필터 설정
        where_clause = ""
        params = []
        if user_id and user_id.strip():
            where_clause = " and user_id = %s "
            params = [user_id.strip()]
            # logger.info(f"사용자 ID 필터 적용: {user_id}의 문서만 조회")
        else:
            logger.info("모든 사용자의 문서를 조회합니다.")
        
        # PostgreSQL에서 문서 정보 가져오기 (중복 제거 + embedding_status 포함)
        query = f"""
            SELECT DISTINCT ON (t.src, t.user_id)
            t.doc_id,
            t.src AS source,
            t.metadata,
            t.user_id,
            COUNT(*) OVER (PARTITION BY t.src, t.user_id) AS chunk_count,
            BOOL_AND(COALESCE(t.is_embedding, FALSE)) OVER (PARTITION BY t.src, t.user_id) AS is_embedding,
            MAX(t.created_at) OVER (PARTITION BY t.src, t.user_id) AS created_at,
            COALESCE(cd.embedding_status, 'completed') AS embedding_status
            FROM (
            SELECT
                metadata->>'source' AS src,
                doc_id,
                metadata,
                user_id,
                is_embedding,
                created_at
            FROM document_embeddings
            WHERE 1=1 {where_clause}
            ) t
            LEFT JOIN chat_documents cd ON cd.filename = SUBSTRING(t.src FROM '[^/]+$') AND cd.user_id = t.user_id
            ORDER BY t.src, t.user_id, t.created_at DESC;
        """
        cursor.execute(query, params)
        rows = cursor.fetchall()

        # 고유한 문서 정보 구성
        documents = []
        missing_files = []

        for row in rows:
            doc_id, source, metadata, doc_user_id, chunk_count, is_embedding, created_at, embedding_status = row
            if metadata is None:
                metadata = {}
            
            source = metadata.get('source', '')
            if not source:
                continue
            
            # full_path가 있으면 우선 사용, 없으면 source 사용
            # (신규 방식: source에는 파일명만, full_path에 전체 경로 저장)
            full_path = metadata.get('full_path', source)
            
            # 파일 존재 여부 확인
            # 특수 폴더(shared, support)의 경우 파일이 없어도 정상
            # 또한 이미 임베딩된 문서(chunk_count > 0)는 파일이 없어도 정상
            is_special_folder = doc_user_id in ['shared', 'support']
            
            # full_path로 먼저 체크, 없으면 source로 체크
            file_exists = os.path.exists(full_path)
            if not file_exists and full_path != source:
                file_exists = os.path.exists(source)
            
            if not file_exists:
                # 이미 임베딩된 문서는 파일이 없어도 정상 (채팅 페이지 업로드 등)
                if chunk_count > 0:
                    # 임베딩 완료된 문서는 파일이 없어도 계속 진행
                    logger.debug(f"임베딩 완료 문서 (파일 없음): {os.path.basename(source)} - {chunk_count} chunks")
                elif is_special_folder:
                    # 특수 폴더는 파일이 없어도 계속 진행
                    logger.debug(f"특수 폴더({doc_user_id}) 문서: {os.path.basename(source)}")
                else:
                    # 임베딩도 안되고 파일도 없는 경우만 경고
                    missing_files.append(source)
                    logger.info(f"존재하지 않는 파일 발견: {os.path.basename(source)}")
                    continue
                
            file_name = os.path.basename(source)
            # 표시용 경로는 full_path 또는 source 중 존재하는 것 사용
            display_path = full_path if file_exists else source
            documents.append({
                "file_name": file_name,
                "source": display_path,  # 실제 파일이 있는 경로 반환
                "chunk_count": chunk_count,
                "last_modified": metadata.get('modification_date', ''),
                "metadata": metadata,
                "user_id": doc_user_id,
                "file_exists": file_exists,
                "embedding_status": embedding_status  # 임베딩 상태 추가
            })
        
        # 결과 로깅
        # logger.info(f"\n문서 목록 조회 결과:")
        # logger.info(f"- 고유 파일 수: {len(documents)}")
        # logger.info(f"- 존재하지 않는 파일 수: {len(missing_files)}")
        # if user_id:
        #     logger.info(f"- 필터링된 사용자: {user_id}")
        
        # if missing_files:
        #     logger.info(f"- 존재하지 않는 파일들:")
        #     for missing_file in missing_files[:5]:
        #         logger.info(f"  * {missing_file}")
        #     if len(missing_files) > 5:
        #         logger.info(f"  * ... 및 {len(missing_files) - 5}개 더")
        
        cursor.close()
        return_pool_connection(conn)
        
        return {
            "status": "success",
            "documents": documents,
            "total_files": len(documents),
            "missing_files_count": len(missing_files),
            "user_id": user_id if user_id else 'all'
        }
    except Exception as e:
        logger.error(f"문서 목록 조회 중 오류: {str(e)}")
        return {
            "status": "error",
            "message": str(e)
        }

@app.get("/document/preview")
async def preview_document(filename: str, user_id: Optional[str] = None):
    """문서 내용을 미리보기용으로 반환합니다."""
    try:
        logger.info(f"문서 미리보기 요청: {filename} (사용자: {user_id})")
        
        # 데이터베이스에서 문서 정보 조회
        embedding_service = EmbeddingService()
        collection = embedding_service.collection
        
        # where 필터 설정
        where_filter = {}
        if user_id and user_id.strip():
            where_filter["user_id"] = user_id.strip()
        
        # 파일명으로 검색
        results = collection.get(where=where_filter)
        
        # 해당 파일의 청크들 찾기
        file_chunks = []
        file_metadata = None
        
        for i, (doc, metadata, doc_id) in enumerate(zip(
            results.get('documents', []), 
            results.get('metadatas', []), 
            results.get('ids', [])
        )):
            source = metadata.get('source', '')
            if os.path.basename(source) == filename:
                file_chunks.append(doc)
                if not file_metadata:
                    file_metadata = metadata
        
        if not file_chunks:
            # 파일이 데이터베이스에 없으면 원본 파일에서 직접 읽기 시도
            logger.info(f"데이터베이스에서 파일을 찾을 수 없어 원본 파일 읽기 시도: {filename}")
            
            # RAG 업로드 디렉토리 경로
            rag_dir = os.path.expanduser("~/.airun/rag_files")
            if user_id:
                file_path = os.path.join(rag_dir, user_id, filename)
            else:
                file_path = os.path.join(rag_dir, filename)
            
            if os.path.exists(file_path):
                # 파일 확장자에 따라 적절한 추출 방법 사용
                content = ""
                file_ext = os.path.splitext(filename)[1].lower()
                
                if file_ext in ['.hwp', '.hwpx']:
                    # HWP 파일 처리
                    from utils import extract_from_hwp
                    try:
                        content = extract_from_hwp(file_path)
                        logger.info(f"HWP 파일 내용 추출 성공: {len(content)} 문자")
                    except Exception as e:
                        logger.error(f"HWP 파일 추출 실패: {e}")
                        content = f"HWP 파일을 읽을 수 없습니다: {str(e)}"
                
                elif file_ext == '.pdf':
                    # PDF 파일 처리
                    try:
                        import PyPDF2
                        with open(file_path, 'rb') as file:
                            pdf_reader = PyPDF2.PdfReader(file)
                            content = ""
                            for page_num in range(min(len(pdf_reader.pages), 10)):  # 최대 10페이지
                                page = pdf_reader.pages[page_num]
                                content += page.extract_text() + "\n"
                    except Exception as e:
                        logger.error(f"PDF 파일 추출 실패: {e}")
                        content = f"PDF 파일을 읽을 수 없습니다: {str(e)}"
                
                elif file_ext in ['.doc', '.docx']:
                    # Word 파일 처리
                    try:
                        from docx import Document
                        doc = Document(file_path)
                        content = "\n".join([paragraph.text for paragraph in doc.paragraphs])
                    except Exception as e:
                        logger.error(f"Word 파일 추출 실패: {e}")
                        content = f"Word 파일을 읽을 수 없습니다: {str(e)}"
                
                elif file_ext in ['.txt', '.md', '.csv']:
                    # 텍스트 파일 처리
                    try:
                        with open(file_path, 'r', encoding='utf-8') as f:
                            content = f.read()
                    except Exception as e:
                        logger.error(f"텍스트 파일 읽기 실패: {e}")
                        content = f"파일을 읽을 수 없습니다: {str(e)}"
                
                else:
                    content = f"지원되지 않는 파일 형식입니다: {file_ext}"
                
                # 파일 메타데이터 생성
                file_stat = os.stat(file_path)
                file_metadata = {
                    "filename": filename,
                    "size": file_stat.st_size,
                    "modified": datetime.fromtimestamp(file_stat.st_mtime).isoformat(),
                    "type": file_ext[1:] if file_ext else "unknown"
                }
                
                return {
                    "status": "success",
                    "content": content[:10000],  # 최대 10000자
                    "metadata": file_metadata,
                    "source": "file"
                }
            else:
                return {
                    "status": "error",
                    "message": f"파일을 찾을 수 없습니다: {filename}"
                }
        
        # 데이터베이스에서 찾은 청크들을 합치기
        content = "\n\n".join(file_chunks)
        
        # 메타데이터 정리
        metadata_result = {
            "filename": filename,
            "chunks": len(file_chunks),
            "user_id": file_metadata.get('user_id', '') if file_metadata else ''
        }
        
        # 실제 파일에서 정확한 메타데이터 가져오기
        file_size = None
        file_modified = None
        
        # RAG 업로드 디렉토리 경로
        rag_dir = os.path.expanduser("~/.airun/rag_files")
        possible_paths = []
        
        # 가능한 파일 경로들 확인
        if user_id:
            possible_paths.append(os.path.join(rag_dir, user_id, filename))
        possible_paths.append(os.path.join(rag_dir, filename))
        
        # source 경로가 있으면 그것도 확인
        if file_metadata and 'source' in file_metadata:
            possible_paths.append(file_metadata['source'])
        
        # 실제 파일 찾기 및 메타데이터 추출
        for path in possible_paths:
            if os.path.exists(path):
                try:
                    file_stat = os.stat(path)
                    file_size = file_stat.st_size
                    file_modified = datetime.fromtimestamp(file_stat.st_mtime).isoformat()
                    metadata_result['source'] = path
                    break
                except Exception as e:
                    logger.warning(f"파일 메타데이터 읽기 실패 {path}: {e}")
        
        # 메타데이터 추가 (값이 있는 경우만)
        if file_size is not None:
            metadata_result['size'] = file_size
        if file_modified:
            metadata_result['modified'] = file_modified
        
        logger.info(f"문서 미리보기 성공: {filename} ({len(file_chunks)} 청크)")
        
        return {
            "status": "success",
            "content": content[:10000],  # 최대 10000자
            "metadata": metadata_result,
            "source": "database"
        }
        
    except Exception as e:
        logger.error(f"문서 미리보기 중 오류: {str(e)}")
        import traceback
        traceback.print_exc()
        return {
            "status": "error",
            "message": str(e)
        }


# 주기적 동기화 함수들
async def periodic_sync_check():
    """주기적 동기화를 수행합니다 (초기 무결성 검사는 lifespan에서 지연 실행됨)"""
    # 초기 무결성 검사는 lifespan에서 처리되므로 여기서는 건너뜀
    logger.info("주기적 동기화 체크 시작 (초기 검사는 별도 수행됨)")

    # 장기간(6시간) 대기 후 주기적 검사 시작
    extended_interval = SYNC_CHECK_INTERVAL * 12  # 30분 * 12 = 6시간
    while True:
        try:
            await asyncio.sleep(extended_interval)
            logger.info("주기적 무결성 검사 시작 (6시간마다)")
            await sync_physical_files_with_db()

            # 상태 동기화도 함께 실행 (GraphRAG 처리 중 상태 누락 방지)
            try:
                sync_embedding_status()
                logger.info("주기적 상태 동기화 완료")
            except Exception as sync_error:
                logger.error(f"주기적 상태 동기화 실패: {sync_error}")

        except Exception as e:
            logger.error(f"주기적 동기화 검사 중 오류: {e}")
            await asyncio.sleep(SYNC_ERROR_RETRY_INTERVAL)  # 오류 시 설정된 재시도 주기

async def sync_physical_files_with_db():
    """물리적 파일과 DB 상태를 동기화"""
    try:
        logger.info("물리적 파일과 DB 동기화 검사 시작")
        
        # 1. 물리적 파일 목록 조회
        physical_files = await get_physical_files()
        logger.info(f"물리적 파일 {len(physical_files)}개 발견")
        
        # 2. DB에 저장된 파일 목록 조회
        db_files = await get_database_files()
        logger.info(f"DB에 저장된 파일 {len(db_files)}개 발견")
        
        # 3. 중복 문서 정리 (동기화 전에 수행)
        logger.info("중복 문서 정리 시작")
        try:
            duplicate_cleanup_result = await cleanup_duplicate_documents_internal()
            if duplicate_cleanup_result.get('status') == 'success' and duplicate_cleanup_result.get('cleaned_groups', 0) > 0:
                logger.info(f"중복 문서 정리 완료: {duplicate_cleanup_result['cleaned_groups']}개 그룹, {duplicate_cleanup_result['deleted_records']}개 레코드 삭제")
                # 중복 문서 정리 후 DB 파일 목록 재조회
                db_files = await get_database_files()
                logger.info(f"중복 정리 후 DB 파일 {len(db_files)}개")
            elif duplicate_cleanup_result.get('status') == 'error':
                logger.error(f"중복 문서 정리 실패 (동기화 계속 진행): {duplicate_cleanup_result.get('message', 'Unknown error')}")
            else:
                logger.info("중복 문서가 발견되지 않았거나 정리할 필요가 없습니다")
        except Exception as cleanup_error:
            logger.error(f"중복 문서 정리 처리 중 예외 발생 (동기화 계속 진행): {cleanup_error}")
        
        # 4. 누락된 파일 식별 (물리적 파일은 있는데 DB에 없는 경우)
        missing_files = await identify_missing_files(physical_files, db_files)
        logger.info(f"누락 파일 분석: {len(missing_files)}개")
        
        # 5. 고아 DB 레코드 식별 (DB에 있는데 물리적 파일이 없는 경우)
        orphaned_db_files = await identify_orphaned_db_files(physical_files, db_files)
        logger.info(f"고아 레코드 분석: {len(orphaned_db_files)}개")
        
        # 실제로 처리할 필요가 있는 경우에만 처리 (임시 비활성화)
        if missing_files:
            logger.info(f"실제 누락된 파일 {len(missing_files)}개 발견, 자동 재처리 시작")
            await process_missing_files(missing_files)
        else:
            logger.info("누락된 파일이 없습니다")
            
        if orphaned_db_files:
            logger.info(f"실제 고아 DB 레코드 {len(orphaned_db_files)}개 발견, 정리 시작")
            await cleanup_orphaned_db_files(orphaned_db_files)
        else:
            logger.info("고아 DB 레코드가 없습니다")
            
        if not missing_files and not orphaned_db_files:
            logger.info("모든 파일이 동기화되어 있음")
        
        # 6. 메타데이터 테이블 동기화 (chat_documents, support_documents)
        await sync_metadata_tables(physical_files, db_files)
            
    except Exception as e:
        logger.error(f"파일 동기화 중 오류: {e}")
        logger.error(traceback.format_exc())

async def sync_metadata_tables(physical_files, db_files):
    """메타데이터 테이블(chat_documents, support_documents)을 실제 파일 상태와 동기화"""
    try:
        client, collection = get_global_postgresql()
        conn = client.get_connection()
        try:
            cursor = conn.cursor()
            
            # 1. 물리적으로 존재하지 않는 파일들을 메타데이터 테이블에서 제거
            logger.info("메타데이터 테이블 동기화 시작")
            
            # 물리적 파일 인덱스 생성
            physical_index = {}
            for pf in physical_files:
                key = f"{pf['user_id']}:{pf['filename']}"
                physical_index[key] = pf
            
            # chat_documents에서 물리적으로 없는 파일들 확인
            cursor.execute("SELECT id, filename, user_id, filepath FROM chat_documents")
            chat_docs = cursor.fetchall()
            
            deleted_chat_count = 0
            for doc_id, filename, user_id, filepath in chat_docs:
                key = f"{user_id}:{filename}"
                # 물리적 파일이 없고, document_embeddings에도 없는 경우 삭제
                if key not in physical_index:
                    # 실제 파일 경로 확인
                    if filepath and os.path.exists(filepath):
                        continue  # 파일이 실제로 존재하면 스킵
                    
                    # document_embeddings에 임베딩이 없는지 확인
                    cursor.execute(
                        "SELECT COUNT(*) FROM document_embeddings WHERE filename = %s AND user_id = %s",
                        (filename, user_id)
                    )
                    embedding_count = cursor.fetchone()[0]
                    
                    if embedding_count == 0:
                        cursor.execute("DELETE FROM chat_documents WHERE id = %s", (doc_id,))
                        deleted_chat_count += 1
                        logger.info(f"chat_documents에서 삭제된 파일: {filename} (사용자: {user_id})")
            
            # support_documents에서 물리적으로 없는 파일들 확인
            cursor.execute("SELECT id, filename, user_id, filepath FROM support_documents")
            support_docs = cursor.fetchall()
            
            deleted_support_count = 0
            for doc_id, filename, user_id, filepath in support_docs:
                key = f"{(user_id or 'unknown')}:{filename}"
                # 물리적 파일이 없고, document_embeddings에도 없는 경우 삭제
                if key not in physical_index:
                    # 실제 파일 경로 확인
                    if filepath and os.path.exists(filepath):
                        continue  # 파일이 실제로 존재하면 스킵
                    
                    # document_embeddings에 임베딩이 없는지 확인
                    cursor.execute(
                        "SELECT COUNT(*) FROM document_embeddings WHERE filename = %s AND user_id = %s",
                        (filename, user_id or 'unknown')
                    )
                    embedding_count = cursor.fetchone()[0]
                    
                    if embedding_count == 0:
                        cursor.execute("DELETE FROM support_documents WHERE id = %s", (doc_id,))
                        deleted_support_count += 1
                        logger.info(f"support_documents에서 삭제된 파일: {filename} (사용자: {user_id})")
            
            conn.commit()
            cursor.close()
            
            if deleted_chat_count > 0 or deleted_support_count > 0:
                logger.info(f"메타데이터 테이블 정리 완료: chat_documents {deleted_chat_count}개, support_documents {deleted_support_count}개 삭제")
            else:
                logger.info("메타데이터 테이블에 정리할 항목이 없습니다")
                
        finally:
            client.return_connection(conn)
            
    except Exception as e:
        logger.error(f"메타데이터 테이블 동기화 중 오류: {e}")

async def get_physical_files():
    """물리적으로 존재하는 파일 목록을 가져옴"""
    try:
        physical_files = []
        rag_docs_path = get_rag_docs_path()
        
        if os.path.exists(rag_docs_path):
            for user_dir in os.listdir(rag_docs_path):
                user_path = os.path.join(rag_docs_path, user_dir)
                if os.path.isdir(user_path):
                    for filename in os.listdir(user_path):
                        # .extracts 폴더 및 숨김 폴더 제외
                        if filename.startswith('.'):
                            continue
                        
                        file_path = os.path.join(user_path, filename)
                        
                        # .extracts 경로가 포함된 파일 제외
                        if '/.extracts/' in file_path or os.sep + '.extracts' + os.sep in file_path:
                            continue
                            
                        if os.path.isfile(file_path):
                            physical_files.append({
                                'user_id': user_dir,
                                'filename': filename,
                                'file_path': file_path,
                                'file_size': os.path.getsize(file_path),
                                'modified_time': os.path.getmtime(file_path)
                            })
        
        return physical_files
    except Exception as e:
        logger.error(f"물리적 파일 목록 조회 중 오류: {e}")
        return []

async def get_database_files():
    """DB에 저장된 파일 목록을 가져옴 - chat_documents, support_documents, document_embeddings 모두 확인"""
    try:
        db_files = []
        
        # PostgreSQL 연결
        client, collection = get_global_postgresql()
        conn = client.get_connection()
        try:
            cursor = conn.cursor()
            
            # 1. document_embeddings 테이블에서 임베딩 완료된 파일 조회
            query = """
            SELECT DISTINCT filename, user_id, source, COUNT(*) as chunk_count
            FROM document_embeddings 
            GROUP BY filename, user_id, source
            ORDER BY filename
            """
            
            cursor.execute(query)
            results = cursor.fetchall()
            
            for row in results:
                filename, user_id, source, chunk_count = row
                db_files.append({
                    'filename': filename,
                    'user_id': user_id or 'unknown',
                    'source': source,
                    'chunk_count': chunk_count,
                    'status': 'completed'
                })
            
            # 2. chat_documents 테이블에서 등록된 파일들도 확인 (임베딩 미완료 포함)
            query = """
            SELECT filename, user_id, filepath
            FROM chat_documents
            ORDER BY filename
            """
            
            cursor.execute(query)
            chat_results = cursor.fetchall()
            
            for row in chat_results:
                filename, user_id, filepath = row
                # 이미 임베딩된 파일인지 확인
                existing = any(f['filename'] == filename and f['user_id'] == user_id for f in db_files)
                if not existing:
                    db_files.append({
                        'filename': filename,
                        'user_id': user_id or 'unknown', 
                        'source': filepath or filename,
                        'chunk_count': 0,
                        'status': 'registered'  # 등록되었지만 임베딩 미완료
                    })
            
            # 3. support_documents 테이블에서 등록된 파일들도 확인
            query = """
            SELECT filename, user_id, filepath
            FROM support_documents
            ORDER BY filename
            """
            
            cursor.execute(query)
            support_results = cursor.fetchall()
            
            for row in support_results:
                filename, user_id, filepath = row
                # 이미 임베딩된 파일인지 확인
                existing = any(f['filename'] == filename and f['user_id'] == (user_id or 'unknown') for f in db_files)
                if not existing:
                    db_files.append({
                        'filename': filename,
                        'user_id': user_id or 'unknown',
                        'source': filepath or filename, 
                        'chunk_count': 0,
                        'status': 'registered'  # 등록되었지만 임베딩 미완료
                    })
            
            cursor.close()
            
        finally:
            client.return_connection(conn)
        
        # Redis에서 처리 중인 파일 상태도 확인
        try:
            redis_client = get_redis_client()
            
            # Redis에서 모든 문서 상태 키 조회
            status_keys = redis_client.keys("doc_status:*")
            for key in status_keys:
                try:
                    filename = key.decode().replace("doc_status:", "")
                    status = redis_client.get(key)
                    if status:
                        status = status.decode()
                        # 처리 중인 상태의 파일들도 "존재하는 파일"로 간주
                        if status in ['uploading', 'processing', 'embedding', 'completed']:
                            # 이미 DB에 있는 파일인지 확인
                            existing = any(f['filename'] == filename for f in db_files)
                            if not existing:
                                # 사용자 ID는 Redis에서 추출하거나 기본값 사용
                                user_id = 'admin'  # 기본값, 필요시 Redis에서 별도 키로 관리
                                db_files.append({
                                    'filename': filename,
                                    'user_id': user_id,
                                    'source': filename,
                                    'chunk_count': 0,
                                    'status': status
                                })
                except Exception as redis_error:
                    logger.debug(f"Redis 상태 키 처리 중 오류 ({key}): {redis_error}")
                    
        except Exception as redis_error:
            logger.debug(f"Redis 상태 확인 중 오류: {redis_error}")
        
        logger.info(f"PostgreSQL에서 {len([f for f in db_files if f.get('status') == 'completed'])}개 파일 조회 성공 (document_embeddings 기준)")
        if any(f.get('status') != 'completed' for f in db_files):
            redis_count = len([f for f in db_files if f.get('status') != 'completed'])
            logger.info(f"Redis에서 처리 중인 파일 {redis_count}개 추가 확인")
            
        return db_files
        
    except Exception as e:
        logger.error(f"DB 파일 목록 조회 중 오류: {e}")
        return []


async def identify_missing_files(physical_files, db_files):
    """물리적으로는 존재하지만 DB에는 없는 파일들을 식별"""
    try:
        missing_files = []
        
        # DB 파일들을 여러 키로 인덱싱 (파일 시스템 감시용)
        db_file_index = {}
        for db_file in db_files:
            user_id = db_file.get('user_id', '')
            filename = db_file.get('filename', '')
            
            # 1. user_id:filename 키
            key1 = f"{user_id}:{filename}"
            db_file_index[key1] = db_file
            
            # 2. 전체 경로가 포함된 경우를 위한 filename만 키  
            if '/' in filename:
                basename = filename.split('/')[-1]
                key2 = f"{user_id}:{basename}"
                db_file_index[key2] = db_file
        
        # 물리적 파일 중 DB에 없는 것들 찾기
        for physical_file in physical_files:
            user_id = physical_file['user_id']
            filename = physical_file['filename']
            
            # 여러 방식으로 존재 여부 확인
            key1 = f"{user_id}:{filename}"
            
            # 전체 경로로도 확인
            full_path = physical_file.get('file_path', '')
            found = False
            
            if key1 in db_file_index:
                found = True
            else:
                # DB에 전체 경로로 저장된 경우를 위한 확인
                for db_key, db_file in db_file_index.items():
                    db_filename = db_file.get('filename', '')
                    if (full_path == db_filename or 
                        (full_path and db_filename and full_path.endswith(db_filename.split('/')[-1]) and 
                         db_file.get('user_id') == user_id)):
                        found = True
                        break
            
            if not found:
                missing_files.append(physical_file)
        
        return missing_files
    except Exception as e:
        logger.error(f"누락 파일 식별 중 오류: {e}")
        return []

async def process_missing_files(missing_files):
    """누락된 파일들을 백그라운드에서 재처리"""
    try:
        for missing_file in missing_files:
            try:
                logger.info(f"누락 파일 재처리 시작: {missing_file['filename']} (사용자: {missing_file['user_id']})")
                
                # RAG 서버에 파일 업로드 요청
                await reprocess_file_for_rag(missing_file)
                
                # 잠시 대기 (시스템 부하 방지)
                await asyncio.sleep(1)
                
            except Exception as file_error:
                logger.error(f"파일 재처리 중 오류 ({missing_file['filename']}): {file_error}")
                
    except Exception as e:
        logger.error(f"누락 파일 처리 중 오류: {e}")

async def reprocess_file_for_rag(file_info):
    """개별 파일을 직접 임베딩 처리"""
    try:
        file_path = file_info['file_path']
        user_id = file_info['user_id']
        filename = file_info['filename']
        
        logger.info(f"파일 직접 임베딩 시작: {filename}")
        
        # 기존 upload_document 함수를 직접 호출하여 임베딩 처리
        from fastapi import UploadFile
        import io
        
        # 파일을 읽어서 UploadFile 객체 생성
        with open(file_path, 'rb') as f:
            file_content = f.read()
            
        # UploadFile 객체 생성
        upload_file = UploadFile(
            file=io.BytesIO(file_content),
            filename=filename
        )
        
        # upload_document 함수 호출하여 임베딩 처리
        result = await upload_document(
            file=upload_file,
            user_id=user_id
        )
        
        logger.info(f"파일 직접 임베딩 완료: {filename}")
        return result
        
    except Exception as e:
        logger.error(f"파일 직접 임베딩 중 오류 ({filename}): {e}")
        return None

async def identify_orphaned_db_files(physical_files, db_files):
    """DB에는 있지만 물리적으로 존재하지 않는 파일들을 식별"""
    try:
        orphaned_files = []
        
        # 물리적 파일들을 user_id + filename으로 인덱싱
        physical_file_index = {}
        for physical_file in physical_files:
            key = f"{physical_file['user_id']}:{physical_file['filename']}"
            physical_file_index[key] = physical_file
        
        # DB 파일 중 물리적으로 존재하지 않는 것들 찾기
        for db_file in db_files:
            key = f"{db_file.get('user_id', '')}:{db_file.get('filename', '')}"
            if key not in physical_file_index:
                # 실제 파일 경로도 확인
                file_path = db_file.get('file_path')
                if file_path and not os.path.exists(file_path):
                    orphaned_files.append(db_file)
                    logger.info(f"고아 DB 레코드 발견: {db_file.get('filename')} (사용자: {db_file.get('user_id')})")
        
        return orphaned_files
    except Exception as e:
        logger.error(f"고아 DB 파일 식별 중 오류: {e}")
        return []

async def cleanup_orphaned_db_files(orphaned_files):
    """고아 DB 레코드들을 정리 (chat_documents와 document_embeddings에서 삭제)"""
    try:
        deleted_count = 0
        conn = get_pool_connection()
        
        try:
            for orphaned_file in orphaned_files:
                try:
                    filename = orphaned_file.get('filename')
                    user_id = orphaned_file.get('user_id')
                    
                    with conn.cursor() as cur:
                        # document_embeddings에서 삭제
                        cur.execute("""
                            DELETE FROM document_embeddings 
                            WHERE filename = %s AND user_id = %s
                        """, (filename, user_id))
                        
                        # chat_documents에서 삭제
                        cur.execute("""
                            DELETE FROM chat_documents 
                            WHERE filename = %s AND user_id = %s
                        """, (filename, user_id))
                        
                        conn.commit()
                        deleted_count += 1
                        logger.info(f"고아 DB 레코드 삭제 완료: {filename} (사용자: {user_id})")
                        
                except Exception as file_error:
                    logger.error(f"고아 파일 삭제 실패: {orphaned_file.get('filename')} - {str(file_error)}")
                    conn.rollback()
        finally:
            return_pool_connection(conn)
        
        logger.info(f"고아 DB 레코드 정리 완료: {deleted_count}개 파일")
        
    except Exception as e:
        logger.error(f"고아 DB 레코드 정리 중 오류: {e}")
        logger.error(traceback.format_exc())

        # 가상의 UploadFile 객체 생성
        upload_file = UploadFile(
            filename=filename,
            file=io.BytesIO(file_content)
        )
        
        # 정확한 파일 상태 확인
        embedding_status = await check_file_status(user_id, filename, file_path)
        
        if embedding_status == "embedded_only":
            # 임베딩은 완료되었지만 chat_documents 테이블에만 저장
            logger.info(f"파일 임베딩 완료, DB 저장만 필요: {filename}")
            await save_document_to_chat_documents(user_id, filename, file_path)
        elif embedding_status == "not_embedded":
            # 임베딩이 안된 경우 /add 엔드포인트 사용
            logger.info(f"파일 임베딩 필요: {filename}")
            await process_document_with_add_endpoint(file_path, user_id)
            logger.info(f"파일 임베딩 완료: {filename}")
            await save_reprocessed_file_to_db(user_id, filename, "direct_embedding")
        else:  # "complete" 상태
            logger.info(f"파일이 이미 완벽하게 처리됨: {filename}")
            # 아무것도 하지 않음
        
    except Exception as e:
        logger.error(f"파일 직접 임베딩 중 오류 ({file_info['filename']}): {e}")
        raise

async def check_file_status(user_id, filename, file_path):
    """파일의 정확한 상태를 확인하여 처리 방법 결정"""
    try:
        # 1. 물리적 파일 존재 여부 확인
        if not os.path.exists(file_path):
            logger.info(f"파일이 물리적으로 존재하지 않음: {filename}")
            return "not_found"
        
        # 2. PostgreSQL에서 임베딩 상태 확인
        # PostgreSQL 데이터베이스에서 문서 존재 여부 확인
        
        # 3. chat_documents 테이블에서 저장 상태 확인
        db_status = await check_document_in_database(user_id, filename)
        
        # 4. 상태 조합으로 최종 판단
        if db_status == "exists":
            # DB에 이미 저장되어 있음 - 완벽한 상태
            logger.info(f"파일 {filename}이 이미 완벽하게 처리됨")
            return "complete"
        else:
            # DB에 저장되지 않음 - 임베딩 상태에 따라 처리
            # 물리적 파일이 존재하므로 임베딩은 완료된 것으로 간주
            logger.info(f"파일 {filename}이 임베딩은 완료되었지만 DB에 저장되지 않음")
            return "embedded_only"
            
    except Exception as e:
        logger.error(f"파일 상태 확인 중 오류: {e}")
        return "error"

def check_document_in_database(user_id, filename):
    """chat_documents 테이블에서 문서 존재 여부 확인"""
    try:
        conn = get_pool_connection()
        try:
            with conn.cursor() as cur:
                cur.execute("""
                    SELECT id FROM chat_documents 
                    WHERE user_id = %s AND filename = %s
                """, (user_id, filename))
                
                row = cur.fetchone()
                
                if row:
                    logger.info(f"문서 존재 확인: {filename} (사용자: {user_id})")
                    return "exists"
                else:
                    logger.info(f"문서 미존재: {filename} (사용자: {user_id})")
                    return "not_exists"
        finally:
            return_pool_connection(conn)
                    
    except Exception as e:
        logger.error(f"DB 문서 확인 중 오류: {e}")
        return "error"

async def process_document_with_add_endpoint(file_path, user_id):
    """/add 엔드포인트를 사용하여 문서 임베딩 처리"""
    try:
        # DocumentProcessor를 직접 사용하여 문서 처리
        global document_processor
        
        if document_processor:
            # 파일 경로를 상대 경로로 변환
            rag_docs_path = get_rag_docs_path()
            
            # user_id 그대로 사용 (shared는 공용 저장소용 별도 ID)
            effective_user_id = user_id
            
            # 문서 처리
            logger.info(f"DocumentProcessor로 직접 임베딩 처리 시작: {file_path}")
            results = await document_processor.add_documents(
                [file_path],
                effective_user_id,
                rag_docs_path,
                None  # 기본 설정 사용
            )
            logger.info(f"임베딩 처리 완료: {results}")
            return results
        else:
            logger.error("DocumentProcessor가 초기화되지 않음")
            raise Exception("DocumentProcessor not initialized")
                    
    except Exception as e:
        logger.error(f"/add 엔드포인트 처리 중 오류: {e}")
        raise

async def save_document_to_chat_documents(user_id, filename, file_path):
    """이미 임베딩된 파일을 chat_documents 테이블에 저장 (중복 문서 처리 포함)"""
    try:
        import os
        from datetime import datetime
        
        # 파일 정보 수집
        file_size = os.path.getsize(file_path)
        file_extension = os.path.splitext(filename)[1].lower()
        
        # MIME 타입 결정
        mime_type = 'application/pdf' if file_extension == '.pdf' else \
                   'application/msword' if file_extension == '.doc' else \
                   'application/vnd.openxmlformats-officedocument.wordprocessingml.document' if file_extension == '.docx' else \
                   'text/plain' if file_extension == '.txt' else \
                   'application/octet-stream'
        
        logger.info(f"파일 정보 준비 완료: {filename} (사용자: {user_id})")
        logger.info(f"- 파일 크기: {file_size} bytes")
        logger.info(f"- MIME 타입: {mime_type}")
        logger.info(f"- 파일 경로: {user_id}/{filename}")
        
        # 동기 연결 풀 사용 (중앙 집중식)
        conn = get_pool_connection()
        try:
                # 같은 사용자 ID와 파일명을 가진 모든 레코드 조회 (created_at 순으로 정렬)
                with conn.cursor() as cur:
                    cur.execute("""
                        SELECT id, created_at, updated_at
                        FROM chat_documents
                        WHERE user_id = %s AND filename = %s
                        ORDER BY created_at DESC
                    """, (user_id, filename))
                    existing_records = cur.fetchall()
                
                if existing_records:
                    if len(existing_records) > 1:
                        # 중복 레코드가 있는 경우, 가장 최신 것만 남기고 나머지 삭제
                        logger.info(f"중복 레코드 발견: {filename} (총 {len(existing_records)}개)")
                        
                        # 가장 최신 레코드 ID (created_at이 가장 최신)
                        latest_record_id = existing_records[0]['id']
                        
                        # 중복 레코드들 삭제 (가장 최신 제외)
                        duplicate_ids = [record['id'] for record in existing_records[1:]]
                        if duplicate_ids:
                            await conn.execute("""
                                DELETE FROM chat_documents 
                                WHERE id = ANY($1)
                            """, duplicate_ids)
                            logger.info(f"중복 레코드 {len(duplicate_ids)}개 삭제 완료: {filename}")
                        
                        # 가장 최신 레코드 업데이트
                        await conn.execute("""
                            UPDATE chat_documents 
                            SET filesize = $1, mimetype = $2, updated_at = CURRENT_TIMESTAMP
                            WHERE id = $3
                        """, file_size, mime_type, latest_record_id)
                        logger.info(f"최신 레코드 업데이트 완료: {filename}")
                    else:
                        # 단일 레코드만 있는 경우 업데이트
                        await conn.execute("""
                            UPDATE chat_documents 
                            SET filesize = $1, mimetype = $2, updated_at = CURRENT_TIMESTAMP
                            WHERE user_id = $3 AND filename = $4
                        """, file_size, mime_type, user_id, filename)
                        logger.info(f"DB 레코드 업데이트 완료: {filename}")
                else:
                    # 새로 삽입
                    await conn.execute("""
                        INSERT INTO chat_documents (user_id, filename, filesize, mimetype, filepath, created_at, updated_at)
                        VALUES ($1, $2, $3, $4, $5, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
                    """, user_id, filename, file_size, mime_type, f"{user_id}/{filename}")
                    logger.info(f"DB 레코드 생성 완료: {filename}")
        finally:
            return_pool_connection(conn)
        
        logger.info(f"chat_documents 테이블 저장 완료: {filename}")
        
        # 임베딩 상태를 'completed'로 업데이트
        try:
            update_embedding_status_sync(user_id, filename, "completed", "임베딩 및 저장 완료")
            logger.info(f"임베딩 상태 업데이트 완료: {filename} -> completed")
        except Exception as status_error:
            logger.error(f"임베딩 상태 업데이트 실패 ({filename}): {status_error}")
    except Exception as e:
        logger.error(f"chat_documents 테이블 저장 중 오류 ({filename}): {e}")

async def save_reprocessed_file_to_db(user_id, filename, job_id):
    """재처리된 파일을 DB에 저장"""
    try:
        logger.info(f"DB 저장 완료: {filename} (사용자: {user_id})")
        # 실제 DB 저장은 upload_document 함수에서 처리되므로 여기서는 로그만 출력

        # 임베딩 상태를 'completed'로 업데이트
        try:
            update_embedding_status_sync(user_id, filename, "completed", "임베딩 및 저장 완료")
            logger.info(f"임베딩 상태 업데이트 완료: {filename} -> completed")
        except Exception as status_error:
            logger.error(f"임베딩 상태 업데이트 실패 ({filename}): {status_error}")

    except Exception as e:
        logger.error(f"재처리된 파일 DB 저장 중 오류: {e}")

# NextJS API 관련 함수 제거 - 직접 DB 조회로 대체됨

async def run_monitoring():
    """파일 시스템 모니터링을 실행합니다."""
    try:
        # 기존 파일 모니터링과 함께 주기적 동기화 실행
        await asyncio.gather(
            run_file_monitoring_core(),  # 기존 파일 모니터링 로직
            periodic_sync_check(),       # 주기적 동기화 (새로 추가)
        )
    except Exception as e:
        logger.error(f"모니터링 실행 중 오류: {e}")

async def run_file_monitoring_core():
    """기존 파일 모니터링 로직을 별도 함수로 분리"""
    inotify = None
    try:
        inotify = INotify()
        watches = {}
        current_path = get_rag_docs_path()
        
        # 이벤트를 디버깅하기 위한 헬퍼 함수
        def debug_event(event, path):
            """이벤트 정보를 디버깅합니다."""
            event_types = []
            event_desc = {}
            
            if event.mask & flags.CREATE:
                event_types.append("CREATE")
                event_desc["CREATE"] = "파일/디렉토리 생성됨"
            if event.mask & flags.DELETE:
                event_types.append("DELETE")
                event_desc["DELETE"] = "파일/디렉토리 삭제됨"
            if event.mask & flags.MODIFY:
                event_types.append("MODIFY")
                event_desc["MODIFY"] = "파일 내용 수정됨"
            if event.mask & flags.MOVED_TO:
                event_types.append("MOVED_TO")
                event_desc["MOVED_TO"] = "파일이 해당 경로로 이동됨"
            if event.mask & flags.MOVED_FROM:
                event_types.append("MOVED_FROM")
                event_desc["MOVED_FROM"] = "파일이 해당 경로에서 이동됨"
            if event.mask & flags.CLOSE_WRITE:
                event_types.append("CLOSE_WRITE")
                event_desc["CLOSE_WRITE"] = "파일 쓰기 후 닫힘"
            if event.mask & flags.ISDIR:
                event_types.append("ISDIR")
                event_desc["ISDIR"] = "이벤트 대상이 디렉토리임"
                
            # 주요 이벤트 타입 결정
            primary_event = None
            for event_priority in ["CLOSE_WRITE", "CREATE", "DELETE", "MOVED_FROM", "MOVED_TO", "MODIFY"]:
                if event_priority in event_types:
                    primary_event = event_priority
                    break
            
            # 디버그 로그 출력
            event_desc_str = ", ".join([f"{t}: {event_desc.get(t, '')}" for t in event_types])
            # logger.info(f"DEBUG: Event detected for {path}, mask={event.mask}, types={event_types}")
            # logger.info(f"DEBUG: Event details - {event_desc_str}")
            # logger.info(f"DEBUG: Primary event type: {primary_event}")
            
            return primary_event, event_types
        
        # 초기 감시 설정
        wd = setup_watch_directory(inotify, current_path)
        if wd is not None:
            watches.update(wd)
        
        # 중복 이벤트 처리 방지를 위한 캐시
        recently_processed = {}
        last_cleanup_time = time.time()
        
        while True:
            try:
                # 감시 경로 업데이트
                watches, current_path = update_watch_if_needed(inotify, watches, current_path)
                
                # 주기적으로 삭제된 폴더에 대한 감시 정리 (5초마다)
                current_time = time.time()
                if current_time - last_cleanup_time > 5:
                    watches = check_and_cleanup_watches(inotify, watches)
                    last_cleanup_time = current_time
                
                # 이벤트 대기 (타임아웃 설정으로 루프 계속 유지)
                events = inotify.read(timeout=500)  # 500ms 타임아웃
                
                for event in events:
                    try:
                        wd = event.wd
                        if wd in watches:
                            path = watches[wd]
                            file_path = os.path.join(path, event.name)
                            
                            # 이벤트 디버깅 및 주요 이벤트 타입 결정
                            primary_event, event_types = debug_event(event, file_path)
                            
                            # 중복 이벤트 처리 방지 (5초 이내 동일 이벤트 무시)
                            event_key = f"{file_path}:{primary_event}"
                            current_time = time.time()
                            
                            if event_key in recently_processed:
                                last_time = recently_processed[event_key]
                                if current_time - last_time < 3:  # 3초 이내 동일 이벤트 무시
                                    logger.info(f"Skipping recently processed event: {primary_event} for {file_path}")
                                    continue
                            
                            # 현재 이벤트 처리 시간 기록
                            recently_processed[event_key] = current_time
                            
                            # 캐시 크기 제한
                            if len(recently_processed) > 200:
                                # 가장 오래된 항목 50개 삭제
                                sorted_items = sorted(recently_processed.items(), key=lambda x: x[1])
                                for k, _ in sorted_items[:50]:
                                    del recently_processed[k]
                            
                            # 디렉토리 생성 이벤트 특별 처리
                            if "ISDIR" in event_types and "CREATE" in event_types:
                                logger.info(f"Directory created: {file_path}")
                                new_dir = os.path.join(path, event.name)
                                new_watches = setup_watch_directory(inotify, new_dir)
                                watches.update(new_watches)
                            
                            # 주 이벤트 타입이 있는 경우에만 처리
                            if primary_event:
                                # logger.info(f"Processing primary event {primary_event} for {file_path}")
                                # 비동기 처리를 위한 새 작업 생성
                                asyncio.create_task(process_file_event(file_path, primary_event))
                            else:
                                logger.warning(f"No primary event type identified for {file_path}, mask={event.mask}")
                        else:
                            logger.info(f"Ignoring event from unknown watch descriptor: {event.wd}")
                    except Exception as event_error:
                        logger.error(f"이벤트 처리 중 오류 발생: {str(event_error)}")
                        logger.error(traceback.format_exc())
                        continue
                
                # 짧은 딜레이로 CPU 사용량 감소
                await asyncio.sleep(0.1)
                        
            except Exception as loop_error:
                logger.error(f"모니터링 루프 오류: {str(loop_error)}")
                logger.error(traceback.format_exc())
                # 오류 발생 시 재연결 전 대기
                await asyncio.sleep(3)
                
    except Exception as init_error:
        logger.error(f"모니터링 초기화 중 오류 발생: {str(init_error)}")
        logger.error(traceback.format_exc())
        # inotify가 초기화되었는지 확인 후 정리
        if inotify:
            try:
                inotify.close()
            except:
                pass
        raise

def run_monitoring_thread():
    """모니터링 스레드에서 실행할 함수"""
    # 재시도 카운터와 최대 재시도 횟수 설정
    retry_count = 0
    max_retries = 5
    
    while True:  # 무한 루프로 전환하여 재귀 호출 제거
        try:
            # logger.info("Starting file monitoring thread...")
            
            # 새 이벤트 루프 생성
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            
            # 모니터링 작업을 실행하고 루프 시작
            loop.run_until_complete(run_monitoring())
            
            # 정상 종료된 경우 (여기까지 오면 run_monitoring이 예외 없이 종료된 경우)
            # logger.info("Monitoring function exited unexpectedly. Restarting...")
            retry_count = 0  # 재시도 카운터 초기화
            
        except Exception as e:
            # 모니터링 스레드에서 예외 발생 시 로깅
            retry_count += 1
            logger.error(f"Fatal error in monitoring thread (retry {retry_count}/{max_retries}): {str(e)}")
            logger.error(f"Traceback: {traceback.format_exc()}")
            
            # 최대 재시도 횟수에 도달하면 대기 시간 증가
            if retry_count >= max_retries:
                wait_time = 60  # 1분 대기
                logger.info(f"Maximum retry count reached. Waiting {wait_time} seconds before next retry...")
                retry_count = 0  # 재시도 카운터 초기화
            else:
                wait_time = min(5 * retry_count, 30)  # 재시도 횟수에 따라 대기 시간 증가 (최대 30초)
                logger.info(f"Retry {retry_count}/{max_retries}. Waiting {wait_time} seconds before retrying...")
            
            # 대기 후 다음 루프로 계속
            time.sleep(wait_time)

def calculate_optimal_workers():
    """시스템 리소스를 기반으로 최적의 워커 수를 계산합니다."""
    try:
        # CPU 코어 수 확인
        cpu_count = multiprocessing.cpu_count()
        
        # 가용 메모리 확인 (GB 단위)
        available_memory = psutil.virtual_memory().available / (1024 * 1024 * 1024)
        
        # 각 워커당 예상 메모리 사용량 (GB)
        estimated_memory_per_worker = 2.0  # 임베딩 모델 + 문서 처리용 메모리
        
        # CPU 기반 워커 수 계산
        cpu_based_workers = int(cpu_count * 0.7)
        
        # 메모리 기반 워커 수 계산
        memory_based_workers = int(available_memory / estimated_memory_per_worker)
        
        # 최종 워커 수 결정 (CPU와 메모리 중 작은 값 사용)
        optimal_workers = min(cpu_based_workers, memory_based_workers)
        
        # 최소 2개, 최대 16개로 제한
        optimal_workers = max(2, min(optimal_workers, 16))
        
        logger.info(f"워커 수 계산 결과: CPU {cpu_count}코어, 메모리 {available_memory:.1f}GB → {optimal_workers}개 워커")
        
        return optimal_workers
        
    except Exception as e:
        logger.error(f"워커 수 계산 중 오류 발생: {str(e)}")
        return 4  # 오류 발생 시 기본값 반환

def initialize_thread_pool():
    """ThreadPoolExecutor를 초기화합니다."""
    global thread_pool
    if thread_pool is None:
        optimal_workers = calculate_optimal_workers()
        thread_pool = concurrent.futures.ThreadPoolExecutor(max_workers=optimal_workers)
        # logger.info(f"ThreadPoolExecutor가 {optimal_workers}개의 워커로 초기화되었습니다.")

def main():
    """메인 함수 - FastAPI 서버를 실행하고 필요한 초기화 작업을 수행합니다."""
    try:
        # 메인 프로세스에서만 CUDA 비활성화 (VRAM 절약을 위해)
        # 워커 프로세스들은 rag_process.py 설정에 따라 정상적으로 GPU 사용
        os.environ['TOKENIZERS_PARALLELISM'] = 'false'
        
        # 메인 프로세스 식별 마커 설정 (워커들은 이 마커를 보고 GPU 사용 결정)
        os.environ['AIRUN_MAIN_PROCESS'] = '1'
        
        # 로거 초기화 (가장 먼저)
        global logger
        logger = get_logger()
        
        # 서비스 시작 로그 출력 (모든 초기화 전에)
        from datetime import datetime
        import uuid
        session_id = str(uuid.uuid4())[:8]
        
        logger.info(f"RAG 서비스 시작 (세션: {session_id})")
        
        # 기존 airun-rag 프로세스들 정리 (VRAM 중복 사용 방지)
        try:
            import psutil
            current_pid = os.getpid()
            killed_processes = []
            
            for proc in psutil.process_iter(['pid', 'name', 'cmdline']):
                try:
                    # airun-rag.py를 실행하는 다른 프로세스 찾기
                    if (proc.info['pid'] != current_pid and 
                        proc.info['cmdline'] and 
                        any('airun-rag.py' in arg for arg in proc.info['cmdline'])):
                        
                        logger.info(f"기존 airun-rag 프로세스 종료: PID {proc.info['pid']}")
                        proc.terminate()
                        killed_processes.append(proc.info['pid'])
                        
                        # 3초 대기 후 강제 종료
                        try:
                            proc.wait(timeout=3)
                        except psutil.TimeoutExpired:
                            proc.kill()
                            logger.info(f"강제 종료: PID {proc.info['pid']}")
                            
                except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
                    continue
                    
            if killed_processes:
                logger.info(f"정리된 프로세스 수: {len(killed_processes)}")
                # GPU 메모리 정리를 위한 잠시 대기
                import time
                time.sleep(2)
            else:
                logger.info("기존 airun-rag 프로세스가 없습니다")
                
        except ImportError:
            logger.info("경고: psutil 모듈이 없어 프로세스 정리를 건너뜁니다")
        except Exception as e:
            logger.info(f"프로세스 정리 중 오류: {e}")
        
        # 프로세스 이름 설정
        setproctitle("airun-rag")
        
        # ThreadPoolExecutor 초기화
        initialize_thread_pool()
        
        # 프로세스 제목 설정
        setproctitle("airun-rag-service")
        
        # 기본 디렉토리 확인
        base_dir = os.path.expanduser("~/.airun")
        os.makedirs(base_dir, exist_ok=True)
        
        # 워커 프로세스 초기화
        initialize_worker_processes()

        # 전역 모델 사전 로딩 비활성화 (VRAM 절약을 위해 워커들이 개별 로딩)
        global embedding_service, document_processor
        embedding_service = None
        document_processor = None

        
        # 메인 프로세스에서는 GraphRAG 초기화 건너뛰기 (VRAM 절약)
        # GraphRAG는 워커 프로세스에서만 필요할 때 초기화됩니다
        global graphrag_processor
        graphrag_processor = None
        
        # inotify 초기화
        global inotify, watches
        inotify = INotify()
        watches = {}
        
        # 모니터링 스레드 시작 - lifespan에서 시작됨
        
        # FastAPI 서버 설정
        host = "0.0.0.0"
        port = int(os.environ.get('RAG_SERVER_PORT', 5600))  # 환경변수에서 포트 읽기
        
        # 서버 시작 전 상태 동기화 실행
        try:
            sync_embedding_status()
        except Exception as e:
            logger.warning(f"시작 시 상태 동기화 실패: {e}")

        # 서버 시작
        logger.info(f"RAG 서버 준비 완료 (http://{host}:{port})")

        uvicorn.run(
            app,
            host=host,
            port=port,
            log_level="info"
        )
    except Exception as e:
        logger.error(f"서비스 시작 중 오류 발생: {str(e)}")
        raise
    finally:
        # 정리 작업
        from datetime import datetime
        logger.info("=" * 80)
        logger.info("🛑 RAG 서비스 종료 시작")
        logger.info(f"⏰ 종료 시간: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
        logger.info("-" * 80)
        logger.info("🧹 워커 프로세스 정리 중...")
        cleanup_worker_processes()
        logger.info("✅ 워커 프로세스 정리 완료")
        if thread_pool:
            thread_pool.shutdown(wait=True)
            logger.info("✅ 스레드 풀 종료 완료")
        if 'inotify' in globals() and inotify:
            try:
                if 'watches' in globals() and watches:
                    for wd in watches:
                        inotify.rm_watch(wd)
                inotify.close()
                logger.info("✅ 파일 감시 시스템 종료 완료")
            except Exception as e:
                logger.error(f"파일 감시 시스템 종료 실패: {e}")
                pass
        # DB 연결 풀은 lifespan에서 관리되므로 여기서는 제거
        logger.info("-" * 80)
        logger.info("✅ RAG 서비스 정리 작업 완료")
        logger.info("👋 AIRUN RAG 서비스 종료")
        logger.info("=" * 80)

@app.post("/process-image")
async def process_image(file: UploadFile = File(...), user_id: Optional[str] = Form(None)):
    """이미지 파일을 처리하고 설명을 생성하는 엔드포인트"""
    try:
        logger.info(f"\n이미지 처리 요청:")
        logger.info(f"- 파일명: {file.filename}")
        logger.info(f"- 사용자 ID: {user_id if user_id else 'all'}")

        # 이미지 파일 타입 확인
        if not file.content_type.startswith('image/'):
            return JSONResponse(
                content={
                    "status": "error",
                    "message": "Unsupported file type. Only image files are allowed."
                },
                status_code=400
            )

        # RAG 문서 경로 가져오기
        rag_docs_path = get_rag_docs_path()
        
        # 사용자별 디렉토리 생성
        effective_user_id = user_id.strip() if user_id and user_id.strip() else 'all'
        
        # 파일 저장 경로 결정
        if effective_user_id != 'all':
            target_dir = os.path.join(rag_docs_path, effective_user_id)
            os.makedirs(target_dir, exist_ok=True)
            target_file = os.path.join(target_dir, file.filename)
        else:
            target_dir = rag_docs_path
            target_file = os.path.join(rag_docs_path, file.filename)

        try:
            # 이미지 파일 저장
            with open(target_file, "wb") as buffer:
                shutil.copyfileobj(file.file, buffer)
            logger.info(f"이미지 파일 저장됨: {target_file}")

            # 이미지를 base64로 변환
            with open(target_file, "rb") as image_file:
                image_data = image_file.read()
                base64_image = base64.b64encode(image_data).decode('utf-8')

            # 이미지 설명 생성을 위한 프롬프트
            prompt = f"""Please describe this image in detail, focusing on:
1. Main subjects and their characteristics
2. Colors, lighting, and composition
3. Any text or numbers visible in the image
4. Context or setting of the image
5. Any notable details or unique features

Please provide a comprehensive but concise description."""

            # 현재 세션의 LLM 프로바이더 확인
            current_provider = None
            try:
                current_provider = await getVarVal('USE_LLM')
            except:
                current_provider = 'openai'  # 기본값

            # 이미지 설명 생성 (테스트용 더미 텍스트)
            try:
                # 테스트용 더미 이미지 설명
                image_description = f"이미지 파일: {file.filename}. 이것은 업로드된 이미지입니다. 실험실, 기술, 컴퓨터, 과학 관련 이미지로 보입니다."

                logger.info("이미지 설명 생성 완료 (테스트 모드)")

                # 이미지 메타데이터 생성
                metadata = {
                    'source': target_file,
                    'type': 'image',
                    'mimetype': file.content_type,
                    'timestamp': datetime.now().isoformat(),
                    'user_id': effective_user_id,
                    'provider': current_provider
                }

                # PostgreSQL에 저장
                document_id = f"img_{int(time.time())}_{os.path.basename(target_file)}"
                
                # 텍스트 임베딩 생성 (기존 로직 유지)
                global embedding_service
                if not embedding_service:
                    embedding_service = EmbeddingService()
                    embedding_service.initialize()
                text_embedding = embedding_service.get_embedding(image_description)
                
                # 이미지 임베딩 생성 (새로 추가)
                image_embedding = None
                try:
                    global document_processor
                    if not document_processor:
                        from plugins.rag.rag_process import DocumentProcessor
                        document_processor = DocumentProcessor.get_instance()
                        
                    if document_processor and hasattr(document_processor, 'image_embedding_model') and document_processor.image_embedding_model:
                        from plugins.rag.rag_process import create_image_file_embedding
                        image_embedding = create_image_file_embedding(target_file)
                        if image_embedding:
                            logger.info("이미지 임베딩 생성 완료")
                        else:
                            logger.info("이미지 임베딩 생성 실패")
                except Exception as e:
                    logger.error(f"이미지 임베딩 생성 중 오류: {str(e)}")
                    image_embedding = None
                
                # PostgreSQL에 저장 - 텍스트 임베딩
                if not embedding_service:
                    embedding_service = EmbeddingService()
                    embedding_service.initialize()
                collection = embedding_service.collection
                collection.add(
                    ids=[document_id],
                    embeddings=[text_embedding],
                    documents=[image_description],
                    metadatas=[metadata]
                )
                
                # 이미지 임베딩도 별도로 저장 (하이브리드 검색용)
                if image_embedding:
                    try:
                        image_metadata = metadata.copy()
                        image_metadata['embedding_type'] = 'image'
                        image_document_id = f"imgvec_{int(time.time())}_{os.path.basename(target_file)}"
                        
                        collection.add(
                            ids=[image_document_id],
                            embeddings=[image_embedding],
                            documents=[f"[이미지] {image_description}"],
                            metadatas=[image_metadata]
                        )
                        logger.info("이미지 임베딩 저장 완료")
                    except Exception as e:
                        logger.error(f"이미지 임베딩 저장 실패: {str(e)}")

                logger.info("이미지 설명 및 메타데이터 저장 완료")

                return JSONResponse(
                    content={
                        "status": "success",
                        "message": "Image processed and description generated successfully",
                        "file": file.filename,
                        "description": image_description,
                        "user_id": effective_user_id,
                        "metadata": metadata
                    },
                    status_code=200
                )

            except Exception as e:
                logger.error(f"이미지 설명 생성 중 오류: {str(e)}")
                return JSONResponse(
                    content={
                        "status": "error",
                        "message": f"Error generating image description: {str(e)}"
                    },
                    status_code=500
                )

        except Exception as e:
            # 에러 발생 시 파일 정리
            if os.path.exists(target_file):
                os.unlink(target_file)
            raise

    except Exception as e:
        logger.error(f"이미지 처리 중 오류 발생: {str(e)}")
        return JSONResponse(
            content={
                "status": "error",
                "message": str(e)
            },
            status_code=500
        )

class DeleteRequest(BaseModel):
    file_paths: List[str]
    user_id: Optional[str] = None

class DeleteFolderRequest(BaseModel):
    user_id: str

@app.post("/delete")
async def delete_documents(request: DeleteRequest):
    """문서 삭제 엔드포인트"""
    try:
        logger.info(f"Delete operation started for user: {request.user_id}")
        logger.info(f"File paths to delete: {request.file_paths}")

        # 사용자별 PostgreSQL 데이터 삭제 (file_paths가 빈 배열이고 user_id가 있는 경우)
        if not request.file_paths and request.user_id and request.user_id.strip() and request.user_id.lower() != 'all':
            logger.info(f"사용자별 PostgreSQL 데이터 삭제 시작: {request.user_id}")
            try:
                chunks_deleted = 0
                pg_connection = None
                pg_cursor = None
                try:
                    # 메인 프로세스에서는 document_processor 임베딩 서비스 사용 금지 (VRAM 절약)
                    if os.getenv('AIRUN_MAIN_PROCESS') == '1':
                        logger.info("메인 프로세스: 직접 DB 연결 사용 (임베딩 서비스 우회)")
                        # 직접 연결 사용
                        pg_connection = get_pool_connection()
                    elif document_processor and hasattr(document_processor, 'embedding_service'):
                        pg_connection = get_pool_connection()
                    else:
                        # 통합된 연결 풀 사용
                        pg_connection = get_pool_connection()
                    
                    pg_cursor = pg_connection.cursor()
                    
                    user_id = request.user_id
                    
                    # 해당 사용자의 GraphRAG 데이터 삭제 (외래키 제약조건 순서 고려)
                    # 먼저 사용자의 문서 소스들을 가져오거나, 없으면 user_id 기반으로 패턴 매칭

                    # 사용자의 문서 소스 목록 수집 (document_embeddings가 있다면)
                    pg_cursor.execute("""
                        SELECT DISTINCT de.metadata->>'source'
                        FROM document_embeddings de
                        WHERE de.metadata->>'user_id' = %s
                    """, (user_id,))
                    embedding_sources = [row[0] for row in pg_cursor.fetchall()]

                    # document_embeddings 소스를 기반으로 Graph RAG에서 매칭되는 모든 소스 찾기
                    user_sources = []
                    if embedding_sources:
                        for base_source in embedding_sources:
                            # base_source(예: final_deletion_test.txt)를 포함하는 모든 Graph RAG 소스 찾기
                            pg_cursor.execute("""
                                SELECT DISTINCT unnest(source_documents) as source_doc
                                FROM graph_entities
                                WHERE EXISTS (
                                    SELECT 1 FROM unnest(source_documents) AS sd
                                    WHERE sd LIKE %s
                                )
                            """, (f'{base_source}%',))
                            matching_sources = [row[0] for row in pg_cursor.fetchall()]
                            user_sources.extend(matching_sources)

                        user_sources = list(set(user_sources))  # 중복 제거
                        logger.debug(f"document_embeddings 기반으로 찾은 Graph RAG 소스들: {user_sources}")

                    # 만약 document_embeddings가 비어있다면, user_id 패턴으로 Graph RAG 데이터 찾기
                    if not user_sources:
                        logger.debug(f"document_embeddings가 비어있음. user_id {user_id}와 관련된 Graph RAG 데이터를 패턴 매칭으로 삭제")

                        # Graph RAG entities에서 사용자의 모든 source_documents 수집
                        pg_cursor.execute("""
                            SELECT DISTINCT unnest(source_documents) as source_doc
                            FROM graph_entities
                        """)
                        all_graph_sources = [row[0] for row in pg_cursor.fetchall()]

                        # 사용자 경로 패턴으로 필터링 (경로에 user_id가 포함된 소스 찾기)
                        user_sources = []
                        for source in all_graph_sources:
                            # 파일 경로나 소스에서 사용자 ID 추출 시도
                            if user_id in source or f"/{user_id}/" in source:
                                user_sources.append(source)

                        # 추가로, chat_documents 테이블에서 해당 사용자의 문서명들을 가져와서 매칭
                        pg_cursor.execute("""
                            SELECT DISTINCT filename FROM chat_documents WHERE user_id = %s
                        """, (user_id,))
                        user_filenames = [row[0] for row in pg_cursor.fetchall()]

                        # 파일명 기반으로 추가 소스 찾기
                        for filename in user_filenames:
                            for source in all_graph_sources:
                                # 파일명이 소스에 포함된 경우 (chunk 정보와 함께)
                                if filename in source and source not in user_sources:
                                    user_sources.append(source)

                        logger.debug(f"Graph RAG에서 찾은 사용자 {user_id}의 소스들: {user_sources}")

                    if user_sources:
                        logger.debug(f"삭제할 소스 문서들: {user_sources[:5]}...")  # 처음 5개만 로깅

                        # 1) 커뮤니티 삭제
                        logger.debug("1. 커뮤니티 삭제======================")
                        pg_cursor.execute("""
                            DELETE FROM graph_communities gc
                            WHERE EXISTS (
                                SELECT 1
                                FROM graph_entities ge
                                WHERE ge.id = ANY(gc.entities)
                                AND ge.source_documents && %s
                            );
                        """, (user_sources,))
                        communities_deleted = pg_cursor.rowcount

                        # 2) 관계 삭제
                        logger.debug("2. 관계 삭제======================")
                        pg_cursor.execute("""
                            DELETE FROM graph_relationships
                            WHERE source_entity_id IN (
                                SELECT id FROM graph_entities
                                WHERE source_documents && %s
                            )
                            OR target_entity_id IN (
                                SELECT id FROM graph_entities
                                WHERE source_documents && %s
                            );
                        """, (user_sources, user_sources))
                        relationships_deleted = pg_cursor.rowcount

                        # 3) 엔티티 삭제
                        logger.debug("3. 엔티티 삭제======================")
                        pg_cursor.execute("""
                            DELETE FROM graph_entities
                            WHERE source_documents && %s;
                        """, (user_sources,))
                        entities_deleted = pg_cursor.rowcount
                    else:
                        logger.debug(f"사용자 {user_id}의 Graph RAG 데이터를 찾을 수 없음")
                        communities_deleted = 0
                        relationships_deleted = 0
                        entities_deleted = 0

                    # 4) 문서 청크 삭제
                    logger.debug("4. 문서 청크 삭제======================")
                    pg_cursor.execute("""
                        DELETE FROM document_embeddings
                        WHERE metadata->>'user_id' = %s;
                    """, (user_id,))
                    chunks_deleted = pg_cursor.rowcount

                    # 5) chat_documents 삭제
                    logger.debug("5. chat_documents 삭제======================")
                    pg_cursor.execute("""
                        DELETE FROM chat_documents
                        WHERE user_id = %s;
                    """, (user_id,))
                    chat_docs_deleted = pg_cursor.rowcount

                    # 변경사항 커밋
                    pg_connection.commit()
                    logger.debug(f"결과: communities={communities_deleted}, relationships={relationships_deleted}, entities={entities_deleted}, chunks={chunks_deleted}, chat_docs={chat_docs_deleted}")


                    if chunks_deleted > 0 or entities_deleted > 0 or relationships_deleted > 0 or communities_deleted > 0 or chat_docs_deleted > 0:
                        logger.info(f"사용자 {request.user_id}의 데이터 삭제 완료:")
                        logger.info(f"  - 문서 청크: {chunks_deleted}개")
                        logger.info(f"  - GraphRAG 엔티티: {entities_deleted}개")
                        logger.info(f"  - GraphRAG 관계: {relationships_deleted}개")
                        logger.info(f"  - GraphRAG 커뮤니티: {communities_deleted}개")
                        logger.info(f"  - 채팅 문서: {chat_docs_deleted}개")
                    else:
                        logger.info(f"사용자 {request.user_id}의 데이터가 PostgreSQL에 없습니다.")
                    
                except Exception as e:
                    logger.error(f"PostgreSQL 처리 중 오류: {str(e)}")
                    if pg_connection:
                        pg_connection.rollback()
                    raise
                finally:
                    if pg_cursor:
                        pg_cursor.close()
                    if pg_connection:
                        # DocumentProcessor DB pool에 연결 반환
                        if document_processor:
                            document_processor.return_db_connection(pg_connection)
                        elif 'GLOBAL_DB_POOL' in globals() and globals().get('GLOBAL_DB_POOL', None):
                            return_pool_connection(pg_connection)
                        else:
                            return_pool_connection(pg_connection)
                
                return JSONResponse(
                    content={
                        "success": True,
                        "message": f"사용자 {request.user_id}의 PostgreSQL 데이터가 삭제되었습니다.",
                        "user_id": request.user_id,
                        "chunks_deleted": chunks_deleted,
                        "results": [{
                            "operation": "postgresql_cleanup",
                            "success": True,
                            "chunks_deleted": chunks_deleted
                        }],
                        "error": None,
                        "timestamp": datetime.now().isoformat()
                    },
                    status_code=200
                )
            except Exception as e:
                logger.error(f"사용자별 PostgreSQL 데이터 삭제 중 오류: {str(e)}")
                return JSONResponse(
                    content={
                        "success": False,
                        "message": f"사용자 {request.user_id}의 PostgreSQL 데이터 삭제 중 오류: {str(e)}",
                        "error": str(e),
                        "timestamp": datetime.now().isoformat()
                    },
                    status_code=500
                )

        # file_paths가 None이거나 완전히 빈 경우 (user_id도 없는 경우)
        if not request.file_paths:
            return JSONResponse(
                content={
                    "success": False,
                    "message": "File or directory path must be specified",
                    "error": None
                },
                status_code=400
            )

        # RAG 문서 기본 경로 가져오기
        rag_docs_path = get_rag_docs_path()
        if not os.path.exists(rag_docs_path):
            return JSONResponse(
                content={
                    "success": False,
                    "message": f"RAG documents path does not exist: {rag_docs_path}",
                    "error": None
                },
                status_code=400
            )

        # 대상 경로 설정
        target_path = rag_docs_path
        if request.user_id and request.user_id.strip() and request.user_id.lower() != 'all':
            target_path = os.path.join(rag_docs_path, request.user_id)
            if not os.path.exists(target_path):
                return JSONResponse(
                    content={
                        "success": False,
                        "message": f"User directory does not exist: {target_path}",
                        "error": None
                    },
                    status_code=400
                )

        logger.info(f"Delete operation target path: {target_path}")
        
        # 삭제 결과 추적
        results = []

        # 각 경로 처리
        for path in request.file_paths:
            try:
                logger.info(f"Processing delete request for path: {path}")
                # 1. 절대 경로로 변환
                if os.path.isabs(path):
                    abs_path = os.path.abspath(path)
                    # 절대 경로가 대상 경로 내에 있는지 확인
                    if not abs_path.startswith(target_path):
                        results.append({
                            "path": path,
                            "success": False,
                            "error": "Path is outside target directory"
                        })
                        continue
                else:
                    # 상대 경로 검증
                    norm_path = os.path.normpath(path)
                    if norm_path.startswith('..') or '/.' in norm_path:
                        results.append({
                            "path": path,
                            "success": False,
                            "error": "Path contains parent directory reference"
                        })
                        continue
                    abs_path = os.path.abspath(os.path.join(target_path, path))
                    if not abs_path.startswith(target_path):
                        results.append({
                            "path": path,
                            "success": False,
                            "error": "Path resolves outside target directory"
                        })
                        continue

                # 2. PostgreSQL에서 임베딩 청크 삭제 (파일 존재 여부와 관계없이 실행)
                # 파일이 이미 삭제되었더라도 DB에서는 청크 데이터 삭제 필요
                file_exists = os.path.exists(abs_path)
                if not file_exists:
                    logger.info(f"파일이 존재하지 않지만 PostgreSQL 청크 삭제 진행: {path}")
                
                # 먼저 PostgreSQL에서 청크 삭제 수행

                # 3. PostgreSQL에서 청크 삭제 (파일 삭제 전에 수행)
                chunks_deleted = 0
                pg_connection = None
                pg_cursor = None
                try:
                    # 통합된 연결 풀 사용
                    logger.info("Using get_pool_connection for consistent DB management")
                    pg_connection = get_pool_connection()

                    if not pg_connection:
                        raise Exception("Failed to get database connection")

                    pg_cursor = pg_connection.cursor()
                    
                    # 파일/디렉토리 처리 방식 결정  
                    if file_exists and os.path.isdir(abs_path):
                        # 디렉토리인 경우 하위의 모든 파일에 대해 청크 삭제
                        for root, _, files in os.walk(abs_path):
                            for file in files:
                                file_path = os.path.join(root, file)
                                
                                # .extracts 폴더의 파일은 임베딩 대상이 아니므로 청크 삭제 건너뛰기
                                if '/.extracts/' in file_path or os.sep + '.extracts' + os.sep in file_path:
                                    logger.info(f"청크 삭제 건너뛰기: .extracts 폴더의 파일 - {os.path.basename(file_path)}")
                                    continue
                                
                                if any(file_path.lower().endswith(ext) for ext in SUPPORTED_EXTENSIONS):
                                    try:
                                        # PostgreSQL에서 GraphRAG 데이터 및 청크 삭제
                                        if request.user_id and request.user_id.strip() and request.user_id.lower() != 'all':
                                            # 해당 파일의 GraphRAG 데이터 삭제 (외래키 제약조건 순서 고려)
                                            file_basename = os.path.basename(file_path)
                                            
                                            # 1. 커뮤니티 삭제
                                            pg_cursor.execute("""
                                                DELETE FROM graph_communities 
                                                WHERE entities && (
                                                    SELECT array_agg(id) FROM graph_entities 
                                                    WHERE %s = ANY(source_documents) 
                                                       OR %s = ANY(source_documents)
                                                       OR %s = ANY(source_documents)
                                                )
                                            """, (file_basename, f'{file_basename}', file_path))
                                            
                                            # 2. 관계 삭제
                                            pg_cursor.execute("""
                                                DELETE FROM graph_relationships 
                                                WHERE source_entity_id IN (
                                                    SELECT id FROM graph_entities 
                                                    WHERE %s = ANY(source_documents) 
                                                       OR %s = ANY(source_documents)
                                                       OR %s = ANY(source_documents)
                                                ) OR target_entity_id IN (
                                                    SELECT id FROM graph_entities 
                                                    WHERE %s = ANY(source_documents) 
                                                       OR %s = ANY(source_documents)
                                                       OR %s = ANY(source_documents)
                                                )
                                            """, (file_basename, f'{file_basename}', file_path, file_basename, f'{file_basename}', file_path))
                                            
                                            # 3. 엔티티 삭제
                                            pg_cursor.execute("""
                                                DELETE FROM graph_entities 
                                                WHERE %s = ANY(source_documents) 
                                                   OR %s = ANY(source_documents)
                                                   OR %s = ANY(source_documents)
                                            """, (file_basename, f'{file_basename}', file_path))
                                            
                                            # 4. 사용자 ID와 파일 경로 모두로 필터링하여 document_embeddings 삭제
                                            pg_cursor.execute(
                                                """DELETE FROM document_embeddings 
                                                WHERE metadata->>'source' = %s 
                                                   OR metadata->>'source' LIKE %s
                                                   OR metadata->>'source' LIKE %s""",
                                                (file_path, f'%/{os.path.basename(file_path)}', f'{file_path}%')
                                            )
                                        deleted = pg_cursor.rowcount
                                        if deleted > 0:
                                            chunks_deleted += deleted
                                            logger.info(f"Deleted {deleted} chunks from PostgreSQL for file: {file_path}")
                                    except Exception as e:
                                        logger.error(f"Error deleting chunks from PostgreSQL for file {file_path}: {str(e)}")
                    else:
                        # 단일 파일인 경우
                        # .extracts 폴더의 파일은 임베딩 대상이 아니므로 청크 삭제 건너뛰기
                        if '/.extracts/' in abs_path or os.sep + '.extracts' + os.sep in abs_path:
                            logger.info(f"청크 삭제 건너뛰기: .extracts 폴더의 파일 - {os.path.basename(abs_path)}")
                            # .extracts 파일은 청크 삭제를 건너뛰되, 파일 자체는 삭제할 수 있도록 chunks_deleted = 0으로 설정
                            chunks_deleted = 0
                        elif any(abs_path.lower().endswith(ext) for ext in SUPPORTED_EXTENSIONS):
                            try:
                                # PostgreSQL에서 GraphRAG 데이터 및 청크 삭제 - 다양한 패턴으로 매칭 시도
                                file_basename = os.path.basename(abs_path)
                                logger.info(f"파일 삭제 시도: {file_basename}, 사용자: {request.user_id}")
                                
                                # 해당 파일의 GraphRAG 데이터 삭제 (외래키 제약조건 순서 고려)
                                # 1. 커뮤니티 삭제
                                pg_cursor.execute("""
                                    DELETE FROM graph_communities 
                                    WHERE entities && (
                                        SELECT array_agg(id) FROM graph_entities 
                                        WHERE %s = ANY(source_documents) 
                                           OR %s = ANY(source_documents)
                                           OR %s = ANY(source_documents)
                                    )
                                """, (file_basename, f'{file_basename}', abs_path))
                                
                                # 2. 관계 삭제
                                pg_cursor.execute("""
                                    DELETE FROM graph_relationships 
                                    WHERE source_entity_id IN (
                                        SELECT id FROM graph_entities 
                                        WHERE %s = ANY(source_documents) 
                                           OR %s = ANY(source_documents)
                                           OR %s = ANY(source_documents)
                                    ) OR target_entity_id IN (
                                        SELECT id FROM graph_entities 
                                        WHERE %s = ANY(source_documents) 
                                           OR %s = ANY(source_documents)
                                           OR %s = ANY(source_documents)
                                    )
                                """, (file_basename, f'{file_basename}', abs_path, file_basename, f'{file_basename}', abs_path))
                                
                                # 3. 엔티티 삭제
                                pg_cursor.execute("""
                                    DELETE FROM graph_entities 
                                    WHERE %s = ANY(source_documents) 
                                       OR %s = ANY(source_documents)
                                       OR %s = ANY(source_documents)
                                """, (file_basename, f'{file_basename}', abs_path))
                                
                                if request.user_id and request.user_id.strip() and request.user_id.lower() != 'all':
                                    # 4. 사용자 ID와 파일명으로 다양한 패턴 매칭 시도하여 document_embeddings 삭제
                                    pg_cursor.execute(
                                        """DELETE FROM document_embeddings 
                                        WHERE (metadata->>'source' = %s 
                                           OR metadata->>'source' LIKE %s 
                                           OR metadata->>'source' LIKE %s
                                           OR metadata->>'source' = %s)
                                           AND metadata->>'user_id' = %s""",
                                        (file_basename, f'%/{file_basename}', f'%{file_basename}%', abs_path, request.user_id)
                                    )
                                else:
                                    # 4. 파일명으로 다양한 패턴 매칭 시도하여 document_embeddings 삭제
                                    pg_cursor.execute(
                                        """DELETE FROM document_embeddings 
                                        WHERE metadata->>'source' = %s 
                                           OR metadata->>'source' LIKE %s 
                                           OR metadata->>'source' LIKE %s
                                           OR metadata->>'source' = %s""",
                                        (file_basename, f'%/{file_basename}', f'%{file_basename}%', abs_path)
                                    )
                                chunks_deleted = pg_cursor.rowcount
                                
                                # 삭제 전 매칭되는 데이터 확인 (디버깅용)
                                if chunks_deleted == 0:
                                    pg_cursor.execute(
                                        "SELECT COUNT(*), metadata->>'source', metadata->>'user_id' FROM document_embeddings WHERE metadata->>'user_id' = %s GROUP BY metadata->>'source', metadata->>'user_id' LIMIT 5",
                                        (request.user_id,)
                                    )
                                    existing_data = pg_cursor.fetchall()
                                    logger.info(f"매칭 실패: {file_basename}, 기존 데이터: {existing_data}")
                                
                                if chunks_deleted > 0:
                                    logger.info(f"Deleted {chunks_deleted} chunks from PostgreSQL for file: {file_basename} (abs_path: {abs_path})")
                                else:
                                    logger.info(f"No chunks found to delete for file: {file_basename} (abs_path: {abs_path})")
                            except Exception as e:
                                logger.error(f"Error deleting chunks from PostgreSQL for file {abs_path}: {str(e)}")
                    
                    # 변경사항 커밋
                    pg_connection.commit()
                    
                except Exception as e:
                    logger.error(f"Error during PostgreSQL cleanup: {str(e)}")
                    if pg_connection:
                        pg_connection.rollback()
                finally:
                    if pg_cursor:
                        pg_cursor.close()
                    if pg_connection:
                        # DocumentProcessor DB pool에 연결 반환
                        if document_processor:
                            document_processor.return_db_connection(pg_connection)
                        elif 'GLOBAL_DB_POOL' in globals() and GLOBAL_DB_POOL:
                            return_pool_connection(pg_connection)
                        else:
                            return_pool_connection(pg_connection)
                
                # 캐시 정리
                if chunks_deleted > 0:
                    try:
                        await clear_cache()
                        logger.info("Cache cleared after document deletion")
                    except Exception as e:
                        logger.error(f"Error clearing cache: {str(e)}")
                
                # 4. 파일/폴더 삭제 (파일이 존재하는 경우에만)
                file_deleted = False
                if file_exists:
                    try:
                        if os.path.isdir(abs_path):
                            # 디렉토리 내 모든 파일의 .extracts 폴더 삭제
                            for root, _, files in os.walk(abs_path):
                                for file in files:
                                    file_path = os.path.join(root, file)
                                    doc_name = os.path.splitext(os.path.basename(file_path))[0]
                                    extract_dir = os.path.join(
                                        os.path.dirname(file_path),
                                        '.extracts',
                                        doc_name
                                    )
                                    if os.path.exists(extract_dir):
                                        try:
                                            shutil.rmtree(extract_dir)
                                            logger.info(f"Deleted extracts directory: {extract_dir}")
                                        except Exception as e:
                                            logger.error(f"Failed to delete extracts directory: {str(e)}")
                            
                            # 디렉토리 삭제
                            shutil.rmtree(abs_path)
                        else:
                            # 단일 파일의 .extracts 폴더 삭제
                            doc_name = os.path.splitext(os.path.basename(abs_path))[0]
                            extract_dir = os.path.join(
                                os.path.dirname(abs_path),
                                '.extracts',
                                doc_name
                            )
                            if os.path.exists(extract_dir):
                                try:
                                    shutil.rmtree(extract_dir)
                                    logger.info(f"Deleted extracts directory: {extract_dir}")
                                except Exception as e:
                                    logger.error(f"Failed to delete extracts directory: {str(e)}")
                        
                            # 파일 삭제
                            os.remove(abs_path)
                            file_deleted = True

                    except Exception as e:
                        logger.error(f"파일 삭제 중 오류: {str(e)}")
                        results.append({
                            "path": path,
                            "success": False,
                            "error": f"파일 삭제 실패: {str(e)}",
                            "chunks_deleted": chunks_deleted
                        })
                        continue
                
                # 결과 처리 - PostgreSQL 청크 삭제가 성공했으면 성공으로 간주
                if chunks_deleted > 0 or file_deleted:
                    results.append({
                        "path": path,
                        "success": True,
                        "chunks_deleted": chunks_deleted,
                        "file_deleted": file_deleted,
                        "message": f"청크 {chunks_deleted}개 삭제, 파일 삭제: {file_deleted}"
                    })
                else:
                    results.append({
                        "path": path,
                        "success": False,
                        "error": "삭제할 데이터가 없습니다",
                        "chunks_deleted": 0,
                        "file_deleted": False
                    })

            except Exception as e:
                logger.error(f"Error processing delete request for path {path}: {str(e)}")
                logger.error(f"Exception traceback: {traceback.format_exc()}")
                results.append({
                    "path": path,
                    "success": False,
                    "error": str(e)
                })

        # 결과 반환
        success = any(result["success"] for result in results)
        
        # 전체 오류 메시지 구성 (실패한 작업들의 오류 메시지)
        failed_results = [r for r in results if not r["success"]]
        error_message = None
        if failed_results:
            error_messages = [f"{r['path']}: {r.get('error', 'Unknown error')}" for r in failed_results]
            error_message = "; ".join(error_messages)
        
        return JSONResponse(
            content={
                "success": success,
                "user_id": request.user_id if request.user_id else "all",
                "results": results,
                "error": error_message,
                "message": f"Successfully processed {sum(1 for r in results if r['success'])} out of {len(results)} items" if results else "No items to process",
                "timestamp": datetime.now().isoformat()
            },
            status_code=200
        )

    except Exception as e:
        logger.error(f"Error in delete endpoint: {str(e)}")
        return JSONResponse(
            content={
                "success": False,
                "message": str(e),
                "error": str(e),
                "timestamp": datetime.now().isoformat()
            },
            status_code=500
        )

@app.post("/delete-folder")
async def delete_folder(request: DeleteFolderRequest):
    """폴더(사용자) 삭제 엔드포인트 - PostgreSQL 데이터와 물리적 폴더 모두 삭제"""
    try:
        logger.info(f"폴더 삭제 작업 시작: {request.user_id}")
        
        if not request.user_id or not request.user_id.strip():
            return JSONResponse(
                content={
                    "success": False,
                    "error": "user_id가 필요합니다.",
                    "timestamp": datetime.now().isoformat()
                },
                status_code=400
            )
        
        user_id = request.user_id.strip()
        
        # 1. PostgreSQL에서 해당 사용자의 모든 데이터 삭제
        chunks_deleted = 0
        try:
            if document_processor and document_processor.collection:
                where_filter = {"user_id": user_id}
                user_results = document_processor.collection.get(where=where_filter)
                
                if user_results['ids']:
                    document_processor.collection.delete(ids=user_results['ids'])
                    chunks_deleted = len(user_results['ids'])
                    logger.info(f"사용자 {user_id}의 {chunks_deleted}개 청크가 PostgreSQL에서 삭제되었습니다.")
        except Exception as e:
            logger.error(f"PostgreSQL 데이터 삭제 중 오류: {str(e)}")
        
        # 2. 물리적 폴더 삭제
        folder_deleted = False
        try:
            rag_docs_path = get_rag_docs_path()
            user_folder_path = os.path.join(rag_docs_path, user_id)
            
            if os.path.exists(user_folder_path):
                shutil.rmtree(user_folder_path)
                folder_deleted = True
                logger.info(f"사용자 폴더가 삭제되었습니다: {user_folder_path}")
            else:
                logger.info(f"사용자 폴더가 존재하지 않습니다: {user_folder_path}")
        except Exception as e:
            logger.error(f"물리적 폴더 삭제 중 오류: {str(e)}")
        
        # 3. 캐시 정리
        try:
            await clear_cache()
            logger.info("폴더 삭제 후 캐시가 정리되었습니다.")
        except Exception as e:
            logger.error(f"캐시 정리 중 오류: {str(e)}")
        

        
        return JSONResponse(
            content={
                "success": True,
                "user_id": user_id,
                "chunks_deleted": chunks_deleted,
                "folder_deleted": folder_deleted,
                "message": f"사용자 {user_id}의 폴더와 데이터가 삭제되었습니다.",
                "timestamp": datetime.now().isoformat()
            },
            status_code=200
        )
        
    except Exception as e:
        logger.error(f"폴더 삭제 중 오류: {str(e)}")
        return JSONResponse(
            content={
                "success": False,
                "error": str(e),
                "timestamp": datetime.now().isoformat()
            },
            status_code=500
        )

# 자동 재시작을 위한 엔드포인트 추가
@app.get("/rag/status")
async def get_rag_status():
    """RAG 서버의 현재 상태를 반환합니다."""
    try:
        # Redis 연결 상태 확인
        redis_status = False
        if embedding_service and embedding_service.redis_client:
            try:
                redis_status = embedding_service.redis_client.ping()
            except:
                pass

        # PostgreSQL 연결 상태 확인
        postgresql_status = False
        if embedding_service and embedding_service.postgresql_client:
            try:
                collection = embedding_service.postgresql_client.get_collection("airun_docs")
                postgresql_status = True
            except:
                pass

        return {
            "status": "running",
            "redis_connected": redis_status,
            "postgresql_connected": postgresql_status,
            "model_loaded": embedding_service is not None and embedding_service.model is not None,
            "monitoring_active": inotify is not None and len(watches) > 0,
            "timestamp": datetime.now().isoformat()
        }
    except Exception as e:
        logger.error(f"상태 확인 중 오류 발생: {str(e)}")
        return {
            "status": "error",
            "error": str(e),
            "timestamp": datetime.now().isoformat()
        }

# === 모델 파인튜닝 관련 API ===

# 파인튜닝 모듈 임포트
# 파인튜닝 기능 사용 가능 설정
try:
    from sentence_transformers.readers import InputExample
    FINETUNE_AVAILABLE = True
    # 파인튜닝 기능 활성화 메시지는 main()에서 출력
except ImportError as e:
    FINETUNE_AVAILABLE = False
    logger.error(f"모델 파인튜닝 기능을 사용할 수 없습니다: {str(e)}")
    
# ===== 파인튜닝 관련 클래스들 =====

# 적응형 리소스 관리 클래스 추가
class AdaptiveResourceManager:
    """학습 중 동적 리소스 관리"""
    
    def __init__(self, initial_batch_size: int = 4, min_batch_size: int = 1, max_batch_size: int = 32):
        self.current_batch_size = initial_batch_size
        self.min_batch_size = min_batch_size
        self.max_batch_size = max_batch_size
        self.memory_history = []
        self.oom_count = 0
        self.success_count = 0
        self.last_adjustment_time = time.time()
        
    def monitor_memory_usage(self) -> dict:
        """현재 메모리 사용량 모니터링"""
        try:
            # 시스템 메모리
            memory = psutil.virtual_memory()
            memory_percent = memory.percent
            available_gb = memory.available / (1024**3)
            
            # GPU 메모리 (사용 가능한 경우)
            gpu_memory_percent = 0
            gpu_available_gb = 0
            
            if torch.cuda.is_available():
                try:
                    gpu_memory_allocated = torch.cuda.memory_allocated(0)
                    gpu_memory_total = torch.cuda.get_device_properties(0).total_memory
                    gpu_memory_percent = (gpu_memory_allocated / gpu_memory_total) * 100
                    gpu_available_gb = (gpu_memory_total - gpu_memory_allocated) / (1024**3)
                except:
                    pass
            
            memory_info = {
                'system_memory_percent': memory_percent,
                'system_available_gb': available_gb,
                'gpu_memory_percent': gpu_memory_percent,
                'gpu_available_gb': gpu_available_gb,
                'timestamp': time.time()
            }
            
            # 메모리 히스토리 업데이트 (최근 10개 기록 유지)
            self.memory_history.append(memory_info)
            if len(self.memory_history) > 10:
                self.memory_history.pop(0)
            
            return memory_info
            
        except Exception as e:
            log(f"메모리 모니터링 오류: {str(e)}", error=True)
            return {}
    
    def should_reduce_batch_size(self, memory_info: dict) -> bool:
        """배치 크기 감소 필요 여부 판단"""
        # 메모리 사용률이 85% 이상이면 배치 크기 감소
        if memory_info.get('system_memory_percent', 0) > 85:
            return True
        
        # GPU 메모리 사용률이 90% 이상이면 배치 크기 감소
        if memory_info.get('gpu_memory_percent', 0) > 90:
            return True
        
        # 사용 가능한 메모리가 1GB 미만이면 배치 크기 감소
        if memory_info.get('system_available_gb', 0) < 1.0:
            return True
        
        return False
    
    def should_increase_batch_size(self, memory_info: dict) -> bool:
        """배치 크기 증가 가능 여부 판단"""
        # 메모리 사용률이 70% 미만이고 충분한 여유가 있으면 배치 크기 증가
        if (memory_info.get('system_memory_percent', 100) < 70 and
            memory_info.get('system_available_gb', 0) > 3.0 and
            self.success_count > 5):  # 연속 5회 성공 후에만 증가
            return True
        
        return False
    
    def adjust_batch_size(self, memory_info: dict) -> int:
        """메모리 상황에 따른 배치 크기 조정"""
        current_time = time.time()
        
        # 최소 30초 간격으로만 조정
        if current_time - self.last_adjustment_time < 30:
            return self.current_batch_size
        
        old_batch_size = self.current_batch_size
        
        if self.should_reduce_batch_size(memory_info):
            # 배치 크기 감소 (절반으로)
            new_batch_size = max(self.min_batch_size, self.current_batch_size // 2)
            if new_batch_size != self.current_batch_size:
                self.current_batch_size = new_batch_size
                self.last_adjustment_time = current_time
                log(f"🔽 메모리 부족으로 배치 크기 감소: {old_batch_size} → {new_batch_size}")
                log(f"   시스템 메모리: {memory_info.get('system_memory_percent', 0):.1f}%")
                log(f"   GPU 메모리: {memory_info.get('gpu_memory_percent', 0):.1f}%")
                
        elif self.should_increase_batch_size(memory_info):
            # 배치 크기 증가 (2배로, 단 최대값 제한)
            new_batch_size = min(self.max_batch_size, self.current_batch_size * 2)
            if new_batch_size != self.current_batch_size:
                self.current_batch_size = new_batch_size
                self.last_adjustment_time = current_time
                self.success_count = 0  # 성공 카운트 리셋
                log(f"🔼 메모리 여유로 배치 크기 증가: {old_batch_size} → {new_batch_size}")
                log(f"   시스템 메모리: {memory_info.get('system_memory_percent', 0):.1f}%")
        
        return self.current_batch_size
    
    def handle_oom_error(self):
        """Out of Memory 오류 처리"""
        self.oom_count += 1
        self.success_count = 0
        
        # 배치 크기를 절반으로 줄임
        old_batch_size = self.current_batch_size
        self.current_batch_size = max(self.min_batch_size, self.current_batch_size // 2)
        
        log(f"💥 OOM 오류 발생! 배치 크기 긴급 감소: {old_batch_size} → {self.current_batch_size}")
        log(f"   OOM 발생 횟수: {self.oom_count}")
        
        # 비상 메모리 정리
        try:
            gc.collect()
            if torch.cuda.is_available():
                torch.cuda.empty_cache()
                torch.cuda.synchronize()
            
            log("🧹 비상 메모리 정리 완료")
            
        except Exception as e:
            log(f"비상 메모리 정리 실패: {str(e)}", error=True)
    
    def record_success(self):
        """성공적인 배치 처리 기록"""
        self.success_count += 1
    
    def get_memory_trend(self) -> str:
        """메모리 사용량 추세 분석"""
        if len(self.memory_history) < 3:
            return "insufficient_data"
        
        recent_memory = [info.get('system_memory_percent', 0) for info in self.memory_history[-3:]]
        
        if all(recent_memory[i] > recent_memory[i-1] for i in range(1, len(recent_memory))):
            return "increasing"
        elif all(recent_memory[i] < recent_memory[i-1] for i in range(1, len(recent_memory))):
            return "decreasing"
        else:
            return "stable"
    
    def get_resource_summary(self) -> dict:
        """리소스 사용 요약 정보"""
        if not self.memory_history:
            return {}
        
        latest = self.memory_history[-1]
        return {
            'current_batch_size': self.current_batch_size,
            'memory_usage_percent': latest.get('system_memory_percent', 0),
            'gpu_memory_percent': latest.get('gpu_memory_percent', 0),
            'available_memory_gb': latest.get('system_available_gb', 0),
            'memory_trend': self.get_memory_trend(),
            'oom_count': self.oom_count,
            'success_count': self.success_count,
            'adjustments_made': self.oom_count > 0 or self.last_adjustment_time > 0
        }
    
    def calculate_optimal_gradient_accumulation(self, target_batch_size: int) -> int:
        """최적 그래디언트 누적 스텝 계산"""
        if self.current_batch_size >= target_batch_size:
            return 1
        
        # 목표 배치 크기를 달성하기 위한 그래디언트 누적 스텝
        accumulation_steps = max(1, target_batch_size // self.current_batch_size)
        
        # 최대 16스텝까지만 누적 (너무 많으면 학습 불안정)
        return min(accumulation_steps, 16)

# 메모리 효율적인 학습률 스케줄러
class MemoryAwareLearningRateScheduler:
    """메모리 상황에 따른 학습률 조정"""
    
    def __init__(self, base_lr: float = 2e-5, memory_threshold: float = 0.85):
        self.base_lr = base_lr
        self.current_lr = base_lr
        self.memory_threshold = memory_threshold
        self.reduction_factor = 0.5
        self.recovery_factor = 1.1
        self.min_lr = base_lr * 0.1
        self.max_lr = base_lr * 2.0
        
    def adjust_learning_rate(self, memory_percent: float, loss_trend: str = "stable") -> float:
        """메모리 사용률과 손실 추세에 따른 학습률 조정"""
        old_lr = self.current_lr
        
        # 메모리 사용률이 높으면 학습률 감소 (더 안정적인 학습)
        if memory_percent > self.memory_threshold:
            self.current_lr = max(self.min_lr, self.current_lr * self.reduction_factor)
            if old_lr != self.current_lr:
                log(f"📉 메모리 부족으로 학습률 감소: {old_lr:.2e} → {self.current_lr:.2e}")
        
        # 메모리 여유가 있고 손실이 안정적이면 학습률 증가
        elif memory_percent < self.memory_threshold - 0.1 and loss_trend == "decreasing":
            self.current_lr = min(self.max_lr, self.current_lr * self.recovery_factor)
            if old_lr != self.current_lr:
                log(f"📈 메모리 여유로 학습률 증가: {old_lr:.2e} → {self.current_lr:.2e}")
        
        return self.current_lr

# 체크포인트 관리자
class MemoryEfficientCheckpointManager:
    """메모리 효율적인 체크포인트 관리"""
    
    def __init__(self, max_checkpoints: int = 2, memory_threshold: float = 0.9):
        self.max_checkpoints = max_checkpoints
        self.memory_threshold = memory_threshold
        self.checkpoints = []
        self.last_save_time = 0
        self.save_interval = 300  # 5분 간격
        
    def should_save_checkpoint(self, memory_percent: float, current_time: float) -> bool:
        """체크포인트 저장 여부 판단"""
        # 메모리 사용률이 높으면 체크포인트 저장 빈도 감소
        if memory_percent > self.memory_threshold:
            return False
        
        # 최소 간격 확인
        if current_time - self.last_save_time < self.save_interval:
            return False
        
        return True
    
    def save_checkpoint(self, model, optimizer, epoch: int, loss: float, memory_percent: float):
        """메모리 효율적인 체크포인트 저장"""
        if not self.should_save_checkpoint(memory_percent, time.time()):
            return False
        
        try:
            checkpoint_data = {
                'epoch': epoch,
                'model_state_dict': model.state_dict(),
                'optimizer_state_dict': optimizer.state_dict() if optimizer else None,
                'loss': loss,
                'timestamp': time.time(),
                'memory_percent': memory_percent
            }
            
            # 이전 체크포인트 정리 (메모리 절약)
            if len(self.checkpoints) >= self.max_checkpoints:
                oldest_checkpoint = self.checkpoints.pop(0)
                log(f"🗑️  이전 체크포인트 정리: epoch {oldest_checkpoint['epoch']}")
            
            self.checkpoints.append(checkpoint_data)
            self.last_save_time = time.time()
            
            log(f"💾 체크포인트 저장: epoch {epoch}, loss {loss:.2f}, 메모리 {memory_percent:.1f}%")
            return True
            
        except Exception as e:
            log(f"체크포인트 저장 실패: {str(e)}", error=True)
            return False
    
    def get_best_checkpoint(self):
        """최고 성능 체크포인트 반환"""
        if not self.checkpoints:
            return None
        
        return min(self.checkpoints, key=lambda x: x['loss'])
    
    def cleanup_memory(self):
        """메모리 정리"""
        self.checkpoints.clear()
        log("🧹 체크포인트 메모리 정리 완료")

# 안전한 파인튜닝을 위한 리소스 모니터링 클래스 추가
class SafetyMonitor:
    """파인튜닝 중 시스템 안전성을 모니터링하는 클래스"""
    
    def __init__(self, memory_threshold: float = 0.9, gpu_memory_threshold: float = 0.95):
        """
        Args:
            memory_threshold: 시스템 메모리 사용률 임계값 (0.9 = 90%)
            gpu_memory_threshold: GPU 메모리 사용률 임계값 (0.95 = 95%)
        """
        self.memory_threshold = memory_threshold
        self.gpu_memory_threshold = gpu_memory_threshold
        self.monitoring = False
        self.should_stop = False
        
    def start_monitoring(self):
        """모니터링 시작"""
        self.monitoring = True
        self.should_stop = False
        
    def stop_monitoring(self):
        """모니터링 중단"""
        self.monitoring = False
        
    def check_system_safety(self) -> tuple[bool, str]:
        """
        시스템 안전성 검사
        Returns:
            (is_safe, reason) - 안전 여부와 이유
        """
        try:
            # 시스템 메모리 검사
            memory = psutil.virtual_memory()
            memory_usage_ratio = memory.percent / 100.0
            
            if memory_usage_ratio > self.memory_threshold:
                return False, f"시스템 메모리 사용률 위험 ({memory_usage_ratio:.1%} > {self.memory_threshold:.1%})"
            
            # GPU 메모리 검사 (GPU 사용 중인 경우)
            if torch.cuda.is_available():
                try:
                    gpu_memory_allocated = torch.cuda.memory_allocated(0)
                    gpu_memory_total = torch.cuda.get_device_properties(0).total_memory
                    gpu_usage_ratio = gpu_memory_allocated / gpu_memory_total
                    
                    if gpu_usage_ratio > self.gpu_memory_threshold:
                        return False, f"GPU 메모리 사용률 위험 ({gpu_usage_ratio:.1%} > {self.gpu_memory_threshold:.1%})"
                except:
                    pass  # GPU 메모리 검사 실패 시 무시
            
            # 디스크 공간 검사 (최소 1GB 필요)
            disk_usage = psutil.disk_usage('/')
            free_gb = disk_usage.free / (1024**3)
            if free_gb < 1.0:
                return False, f"디스크 공간 부족 ({free_gb:.1f}GB < 1GB)"
            
            return True, "시스템 안전"
            
        except Exception as e:
            return False, f"안전성 검사 실패: {str(e)}"
    
    def emergency_cleanup(self):
        """비상 시 메모리 정리"""
        try:
            # 가비지 컬렉션 강제 실행
            collected = gc.collect()
            log(f"비상 가비지 컬렉션 완료: {collected}개 객체 수집")
            
            # GPU 메모리 정리
            if torch.cuda.is_available():
                torch.cuda.empty_cache()
                torch.cuda.synchronize()
                log("비상 GPU 메모리 정리 완료")
                
        except Exception as e:
            log(f"비상 정리 중 오류: {str(e)}", error=True)

# 안전한 파인튜닝 래퍼 클래스
class SafeFineTuner:
    """안전성 모니터링이 포함된 파인튜닝 래퍼"""
    
    def __init__(self, base_finetuner: 'EmbeddingFineTuner'):
        self.base_finetuner = base_finetuner
        self.safety_monitor = SafetyMonitor()
        self.monitoring_thread = None
        
    def _monitor_safety(self):
        """백그라운드에서 안전성 모니터링"""
        import time
        import threading
        
        while self.safety_monitor.monitoring:
            try:
                is_safe, reason = self.safety_monitor.check_system_safety()
                if not is_safe:
                    log(f"⚠️  시스템 안전성 경고: {reason}", error=True)
                    self.safety_monitor.should_stop = True
                    self.safety_monitor.emergency_cleanup()
                    break
                    
                time.sleep(5)  # 5초마다 검사
                
            except Exception as e:
                log(f"안전성 모니터링 오류: {str(e)}", error=True)
                break
    
    async def safe_fine_tune_model(self, *args, **kwargs):
        """안전성 모니터링이 포함된 파인튜닝"""
        import threading
        
        try:
            # 사전 안전성 검사
            is_safe, reason = self.safety_monitor.check_system_safety()
            if not is_safe:
                raise Exception(f"파인튜닝 시작 전 안전성 검사 실패: {reason}")
            
            # 안전성 모니터링 시작
            self.safety_monitor.start_monitoring()
            self.monitoring_thread = threading.Thread(target=self._monitor_safety, daemon=True)
            self.monitoring_thread.start()
            
            log("🛡️  안전성 모니터링 시작")
            
            # 실제 파인튜닝 실행
            result = await self.base_finetuner.fine_tune_model(*args, **kwargs)
            
            # 모니터링 중단 확인
            if self.safety_monitor.should_stop:
                raise Exception("시스템 안전성 문제로 파인튜닝이 중단되었습니다")
            
            return result
            
        finally:
            # 모니터링 정리
            self.safety_monitor.stop_monitoring()
            if self.monitoring_thread and self.monitoring_thread.is_alive():
                self.monitoring_thread.join(timeout=2)
            log("🛡️  안전성 모니터링 종료")

@dataclass
class SystemResources:
    """시스템 자원 정보"""
    total_memory_gb: float
    available_memory_gb: float
    cpu_cores: int
    gpu_available: bool
    gpu_memory_gb: float
    gpu_available_memory_gb: float

@dataclass
class OptimizedSettings:
    """최적화된 파인튜닝 설정"""
    batch_size: int
    num_workers: int
    pin_memory: bool
    use_amp: bool
    gradient_accumulation_steps: int
    recommended_epochs: int
    warmup_ratio: float

class SystemResourceAnalyzer:
    """시스템 자원 분석 및 최적화 설정 생성"""
    
    @staticmethod
    def analyze_system_resources() -> SystemResources:
        """시스템 자원 분석"""
        import psutil
        import torch
        
        # 메모리 정보
        memory = psutil.virtual_memory()
        total_memory_gb = memory.total / (1024**3)
        available_memory_gb = memory.available / (1024**3)
        
        # CPU 정보
        cpu_cores = psutil.cpu_count(logical=False)  # 물리적 코어 수
        
        # GPU 정보
        gpu_available = torch.cuda.is_available()
        gpu_memory_gb = 0
        gpu_available_memory_gb = 0
        
        if gpu_available:
            try:
                gpu_memory_gb = torch.cuda.get_device_properties(0).total_memory / (1024**3)
                allocated_memory = torch.cuda.memory_allocated(0) / (1024**3)
                reserved_memory = torch.cuda.memory_reserved(0) / (1024**3)
                gpu_available_memory_gb = gpu_memory_gb - reserved_memory
            except:
                gpu_available = False
        
        return SystemResources(
            total_memory_gb=total_memory_gb,
            available_memory_gb=available_memory_gb,
            cpu_cores=cpu_cores,
            gpu_available=gpu_available,
            gpu_memory_gb=gpu_memory_gb,
            gpu_available_memory_gb=gpu_available_memory_gb
        )
    
    @staticmethod
    def _check_resource_safety(resources: SystemResources) -> dict:
        """리소스 안전성 검사"""
        # 최소 요구사항 정의
        MIN_AVAILABLE_MEMORY_GB = 2.0  # 최소 2GB 메모리 필요
        MIN_GPU_MEMORY_GB = 1.0        # 최소 1GB GPU 메모리 필요
        SAFE_MEMORY_RATIO = 0.8        # 전체 메모리의 80% 이상 사용 시 위험
        
        warnings = []
        
        # 시스템 메모리 안전성 검사
        if resources.available_memory_gb < MIN_AVAILABLE_MEMORY_GB:
            warnings.append(f"사용 가능한 메모리가 부족합니다 ({resources.available_memory_gb:.1f}GB < {MIN_AVAILABLE_MEMORY_GB}GB)")
        
        # 메모리 사용률 검사
        memory_usage_ratio = (resources.total_memory_gb - resources.available_memory_gb) / resources.total_memory_gb
        if memory_usage_ratio > SAFE_MEMORY_RATIO:
            warnings.append(f"메모리 사용률이 높습니다 ({memory_usage_ratio:.1%} > {SAFE_MEMORY_RATIO:.1%})")
        
        # GPU 메모리 안전성 검사
        if resources.gpu_available and resources.gpu_available_memory_gb < MIN_GPU_MEMORY_GB:
            warnings.append(f"GPU 메모리가 부족합니다 ({resources.gpu_available_memory_gb:.1f}GB < {MIN_GPU_MEMORY_GB}GB)")
        
        # 디스크 공간 검사
        try:
            import psutil
            disk_usage = psutil.disk_usage('/')
            free_gb = disk_usage.free / (1024**3)
            if free_gb < 2.0:  # 최소 2GB 디스크 공간 필요
                warnings.append(f"디스크 공간이 부족합니다 ({free_gb:.1f}GB < 2GB)")
        except:
            pass
        
        is_safe = len(warnings) == 0
        warning_msg = "; ".join(warnings) if warnings else "시스템 리소스 안전"
        
        return {
            'is_safe': is_safe,
            'warning': warning_msg,
            'warnings': warnings
        }
    
    @staticmethod
    def generate_optimized_settings(resources: SystemResources, force_cpu: bool = False) -> OptimizedSettings:
        """시스템 자원 기반 최적화 설정 생성"""
        
        # 안전성 검사 먼저 수행
        safety_check = SystemResourceAnalyzer._check_resource_safety(resources)
        if not safety_check['is_safe']:
            log(f"⚠️  리소스 안전성 경고: {safety_check['warning']}", error=True)
            # 안전하지 않은 경우 최소 설정으로 강제 조정
            force_cpu = True
        
        # 기본 설정 (보수적)
        settings = OptimizedSettings(
            batch_size=1,
            num_workers=0,
            pin_memory=False,
            use_amp=False,
            gradient_accumulation_steps=1,
            recommended_epochs=3,
            warmup_ratio=0.05
        )
        
        # log(f"시스템 자원 분석 결과:")
        # log(f"  총 메모리: {resources.total_memory_gb:.1f}GB")
        # log(f"  사용 가능 메모리: {resources.available_memory_gb:.1f}GB")
        # log(f"  CPU 코어: {resources.cpu_cores}개")
        # log(f"  GPU 사용 가능: {resources.gpu_available}")
        # if resources.gpu_available:
        #     log(f"  GPU 메모리: {resources.gpu_memory_gb:.1f}GB")
        #     log(f"  GPU 사용 가능 메모리: {resources.gpu_available_memory_gb:.1f}GB")
        
        # GPU 사용 가능하고 강제 CPU 모드가 아닌 경우
        if resources.gpu_available and not force_cpu:
            settings.use_amp = True
            settings.pin_memory = True
            
            # GPU 메모리 기반 배치 크기 조정 (파인튜닝용 보수적 동적 조정)
            # 워커들의 기본 사용량 4.5GB를 고려하여 실제 사용 가능한 메모리 계산
            effective_memory = resources.gpu_available_memory_gb - 4.5  # 워커들 메모리 제외

            if effective_memory >= 8.0:
                settings.batch_size = 2
                settings.gradient_accumulation_steps = 4
                # log("실제 사용 가능 메모리 충분 (8GB+) - 성능 설정")
            elif effective_memory >= 6.0:
                settings.batch_size = 1
                settings.gradient_accumulation_steps = 8
                # log("실제 사용 가능 메모리 보통 (6-8GB) - 균형 설정")
            else:
                settings.batch_size = 1
                settings.gradient_accumulation_steps = 16
                # log("실제 사용 가능 메모리 제한 (<6GB) - 최소 설정")
        
        # CPU 모드 또는 GPU 사용 불가
        else:
            settings.use_amp = False
            settings.pin_memory = False
            
            # CPU 메모리 기반 배치 크기 조정
            if resources.available_memory_gb >= 20.0:
                settings.batch_size = 16
                settings.num_workers = min(4, resources.cpu_cores // 2)
                settings.gradient_accumulation_steps = 1
                # log("CPU 메모리 충분 (20GB+) - 고성능 설정")
            elif resources.available_memory_gb >= 12.0:
                settings.batch_size = 8
                settings.num_workers = min(2, resources.cpu_cores // 4)
                settings.gradient_accumulation_steps = 2
                # log("CPU 메모리 보통 (12-20GB) - 균형 설정")
            elif resources.available_memory_gb >= 8.0:
                settings.batch_size = 4
                settings.num_workers = 1
                settings.gradient_accumulation_steps = 4
                # log("CPU 메모리 제한 (8-12GB) - 보수적 설정")
            elif resources.available_memory_gb >= 4.0:
                settings.batch_size = 2
                settings.num_workers = 0
                settings.gradient_accumulation_steps = 8
                # log("CPU 메모리 부족 (4-8GB) - 절약 설정")
            else:
                settings.batch_size = 1
                settings.num_workers = 0
                settings.gradient_accumulation_steps = 16
                # log("CPU 메모리 매우 부족 (<4GB) - 최소 설정")
        
        # 에포크 수 조정 (높은 배치 크기일 때 에포크 수 증가)
        if settings.batch_size >= 8:
            settings.recommended_epochs = 2  # 큰 배치는 더 빠르게 수렴
        elif settings.batch_size >= 4:
            settings.recommended_epochs = 3  # 기본값
        else:
            settings.recommended_epochs = 4  # 작은 배치는 더 많은 에포크 필요
        
        # 워밍업 비율 조정
        if settings.batch_size >= 8:
            settings.warmup_ratio = 0.1
        elif settings.batch_size >= 4:
            settings.warmup_ratio = 0.05
        else:
            settings.warmup_ratio = 0.02
        
        # log(f"최적화된 설정:")
        # log(f"  배치 크기: {settings.batch_size}")
        # log(f"  워커 수: {settings.num_workers}")
        # log(f"  Pin Memory: {settings.pin_memory}")
        # log(f"  Mixed Precision: {settings.use_amp}")
        # log(f"  Gradient Accumulation: {settings.gradient_accumulation_steps}")
        # log(f"  권장 에포크: {settings.recommended_epochs}")
        # log(f"  워밍업 비율: {settings.warmup_ratio}")
        
        return settings

class EmbeddingFineTuner:
    """임베딩 모델 파인튜닝 클래스"""
    
    def __init__(self, base_model_name: str = "nlpai-lab/KURE-v1", 
                 output_dir: str = None,
                 force_cpu: bool = False,
                 use_existing_model: bool = True):
        """
        Args:
            base_model_name: 베이스 모델 이름
            output_dir: 파인튜닝된 모델 저장 경로 (None이면 ~/.airun/models/finetuned 사용)
            force_cpu: CPU 파인튜닝 강제 사용
            use_existing_model: 기존 로드된 모델 재사용 여부
        """
        self.base_model_name = base_model_name
        
        # output_dir이 None이면 통일된 경로 사용
        if output_dir is None:
            output_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
        
        self.output_dir = Path(output_dir)
        self.output_dir.mkdir(parents=True, exist_ok=True)
        self.use_existing_model = use_existing_model
        
        self.model = None
        self.training_data = []
        self.evaluation_data = []
        self.original_model_backup = None  # 기존 모델 백업용
        self.training_progress = {
            'is_training': False,
            'progress': 0,
            'current_epoch': 0,
            'total_epochs': 0,
            'current_loss': 0.0,
            'best_score': 0.0,
            'eta': '',
            'status': 'idle'
        }
        
        # 디바이스 설정 (GPU 메모리 부족 시 CPU 사용)
        if force_cpu:
            self.device = torch.device('cpu')
            log("CPU 파인튜닝 강제 사용")
        else:
            # GPU 메모리 체크
            if torch.cuda.is_available():
                gpu_memory_gb = torch.cuda.get_device_properties(0).total_memory / 1024**3
                if gpu_memory_gb < 6.0:  # 6GB 미만이면 CPU 사용 권장
                    self.device = torch.device('cpu')
                    log(f"GPU 메모리 부족({gpu_memory_gb:.1f}GB < 6GB), CPU 파인튜닝 사용")
                else:
                    self.device = torch.device('cuda')
                    # log(f"GPU 파인튜닝 사용 (메모리: {gpu_memory_gb:.1f}GB)")
            else:
                self.device = torch.device('cpu')
                # log("CUDA 미지원, CPU 파인튜닝 사용")
    
    def prepare_for_finetuning(self, embedding_service=None):
        """파인튜닝을 위한 준비 - 기존 모델 관리"""
        try:
            if self.use_existing_model and embedding_service:
                log("기존 모델 재사용 모드 - 메모리 효율성 최적화")
                
                # 기존 모델 정보 백업
                if hasattr(embedding_service, 'model') and embedding_service.model:
                    log("기존 EmbeddingService 모델 발견, 메모리 절약을 위해 임시 해제")
                    self.original_model_backup = {
                        'model_name': self.base_model_name,
                        'device': embedding_service.device if hasattr(embedding_service, 'device') else None
                    }
                    
                    # 기존 모델 메모리 해제
                    del embedding_service.model
                    if hasattr(embedding_service, 'semantic_model') and embedding_service.semantic_model:
                        del embedding_service.semantic_model
                    
                    # GPU 메모리 정리
                    if torch.cuda.is_available():
                        torch.cuda.empty_cache()
                        log("GPU 메모리 캐시 정리 완료")
                    
                    log("기존 모델 메모리 해제 완료")
                
                # 파인튜닝용 모델 로드
                log("파인튜닝용 모델 로드 시작 (메모리 최적화)")
                from sentence_transformers import SentenceTransformer
                self.model = SentenceTransformer(self.base_model_name, device=self.device)
                log("파인튜닝용 모델 로드 완료")
                
            else:
                # 기존 방식대로 별도 모델 로드
                log("별도 모델 로드 방식 사용")
                from sentence_transformers import SentenceTransformer
                self.model = SentenceTransformer(self.base_model_name, device=self.device)
                
            return True
        except Exception as e:
            log(f"파인튜닝 준비 실패: {str(e)}", error=True)
            return False
    
    def restore_original_model(self, embedding_service=None):
        """파인튜닝 완료 후 원래 모델 복원"""
        try:
            if self.use_existing_model and embedding_service and self.original_model_backup:
                log("파인튜닝 완료 - 원래 모델 복원 시작")
                
                # 파인튜닝 모델 메모리 해제
                if self.model:
                    del self.model
                    self.model = None
                
                # GPU 메모리 정리
                if torch.cuda.is_available():
                    torch.cuda.empty_cache()
                
                # 원래 모델 복원
                from sentence_transformers import SentenceTransformer
                device = self.original_model_backup.get('device', 'cpu')
                embedding_service.model = SentenceTransformer(self.base_model_name, device=device)
                
                # 시맨틱 모델도 복원
                try:
                    from rag_process import get_rag_settings
                    settings = get_rag_settings()
                    semantic_chunker_model = settings.get("semantic_chunker_model", "snunlp/KR-SBERT-V40K-klueNLI-augSTS")
                    
                    embedding_service.semantic_model = KoreanSentenceTransformerEmbeddings(
                        model_name=semantic_chunker_model
                    )
                    log("시맨틱 모델 복원 성공")
                
                except Exception as e:
                    log(f"시맨틱 모델 복원 실패: {str(e)}", error=True)
                
                log("원래 모델 복원 완료")
                
        except Exception as e:
            log(f"원래 모델 복원 실패: {str(e)}", error=True)
    
    def get_training_progress(self) -> Dict[str, Any]:
        """현재 훈련 진행 상황 반환"""
        # finetune_state_manager에서 실제 상태 가져오기
        try:
            state = finetune_state_manager.load_state()
            
            if not state:
                # 상태가 없으면 기본 상태 반환
                return {
                    'is_training': False,
                    'progress': 0,
                    'current_epoch': 0,
                    'total_epochs': 0,
                    'current_loss': 0.0,
                    'best_score': 0.0,
                    'eta': 'idle',
                    'status': 'idle'
                }
            
            # 상태에서 진행률 정보 추출
            status = state.get('status', 'idle')
            is_training = status in ['starting', 'training', 'retrying_cpu']
            
            # 진행률 계산
            progress = 0
            current_epoch = 0
            total_epochs = state.get('num_epochs', 0)
            current_loss = state.get('current_loss', 0.0)
            eta = status
            
            if status == 'starting':
                progress = 5
                eta = '시작 중...'
            elif status == 'training':
                # 실제 진행률이 있으면 사용
                if 'progress' in state:
                    progress = min(state['progress'], 99)  # 최대 99%까지만 표시
                elif 'current_step' in state and 'total_steps' in state:
                    progress = min((state['current_step'] / state['total_steps']) * 100, 99)
                else:
                    progress = 10  # 기본 진행률
                
                current_epoch = state.get('current_epoch', 0)
                current_loss = state.get('current_loss', 0.0)
                
                # 더 자세한 ETA 정보 제공
                if 'current_step' in state and 'total_steps' in state:
                    eta = f'훈련 중... (스텝 {state["current_step"]}/{state["total_steps"]})'
                else:
                    eta = f'훈련 중... (에포크 {current_epoch}/{total_epochs})'
                
            elif status == 'retrying_cpu':
                progress = 15
                eta = 'CPU 모드로 재시도 중...'
            elif status == 'completed':
                progress = 100
                eta = '완료'
                is_training = False
            elif status == 'failed':
                progress = 0
                eta = '실패'
                is_training = False
            
            # 로그 출력으로 디버깅 정보 제공
            log(f"🔍 [백엔드] 훈련 진행률 조회: status={status}, progress={progress}%, is_training={is_training}")
            
            return {
                'is_training': is_training,
                'progress': progress,
                'current_epoch': current_epoch,
                'total_epochs': total_epochs,
                'current_loss': current_loss,
                'best_score': state.get('best_score', 0.0),
                'eta': eta,
                'status': status,
                'model_name': state.get('model_name', ''),
                'dataset_id': state.get('dataset_id', ''),
                'start_time': state.get('start_time', 0),
                'last_updated': state.get('last_updated', ''),
                'current_step': state.get('current_step', 0),
                'total_steps': state.get('total_steps', 0)
            }
            
        except Exception as e:
            log(f"훈련 진행 상황 조회 실패: {str(e)}", error=True)
            # 오류 시 기본 상태 반환
            return {
                'is_training': False,
                'progress': 0,
                'current_epoch': 0,
                'total_epochs': 0,
                'current_loss': 0.0,
                'best_score': 0.0,
                'eta': 'error',
                'status': 'error'
            }
    
    def list_models(self) -> List[Dict[str, Any]]:
        """저장된 모델 목록 반환"""
        models = []
        
        try:
            for model_dir in self.output_dir.iterdir():
                if model_dir.is_dir():
                    metadata_path = model_dir / "metadata.json"
                    if metadata_path.exists():
                        with open(metadata_path, 'r', encoding='utf-8') as f:
                            metadata = json.load(f)
                        
                        # 모델 크기 계산
                        total_size = sum(f.stat().st_size for f in model_dir.rglob('*') if f.is_file())
                        size_mb = total_size / (1024 * 1024)
                        
                        models.append({
                            'id': model_dir.name,
                            'name': metadata.get('model_name', model_dir.name),
                            'base_model': metadata.get('base_model', 'unknown'),
                            'created_at': metadata.get('created_at', ''),
                            'training_samples': metadata.get('training_samples', 0),
                            'best_score': metadata.get('best_score', 0.0),
                            'size_mb': round(size_mb, 2),
                            'path': str(model_dir),
                            'dataset_id': metadata.get('dataset_id') or metadata.get('dataset')  # dataset_id 또는 dataset 필드 사용
                        })
        
        except Exception as e:
            log(f"모델 목록 조회 실패: {str(e)}", error=True)
        
        return models
    
    def delete_model(self, model_id: str) -> bool:
        """모델 삭제"""
        try:
            model_path = self.output_dir / model_id
            if model_path.exists():
                import shutil
                shutil.rmtree(model_path)
                log(f"모델 삭제 완료: {model_id}")
                return True
            return False
        except Exception as e:
            log(f"모델 삭제 실패: {str(e)}", error=True)
            return False
    
    def _load_training_examples(self, dataset_id: str):
        """데이터셋에서 훈련 예제 로드"""
        try:
            # 데이터셋 파일 경로 생성 (통일된 ~/.airun/datasets 경로 사용)
            dataset_path = os.path.join(os.path.expanduser('~'), '.airun', 'datasets', f"{dataset_id}.json")
            log(f"데이터셋 파일 경로: {dataset_path}")
            
            if not os.path.exists(dataset_path):
                log(f"데이터셋 파일을 찾을 수 없습니다: {dataset_path}", error=True)
                return []
            
            log("데이터셋 파일 로드 시작")
            with open(dataset_path, 'r', encoding='utf-8') as f:
                dataset = json.load(f)
            
            log(f"데이터셋 메타데이터: {dataset.get('name', 'Unknown')}")
            log(f"데이터셋 크기: {dataset.get('size', 0)}")
            
            training_examples = []
            examples = dataset.get('examples', [])
            log(f"데이터셋에서 로드된 예제 수: {len(examples)}")
            
            if not examples:
                log("데이터셋에 examples 키가 없거나 비어있습니다", error=True)
                return []
            
            from sentence_transformers import InputExample
            
            valid_examples = 0
            for i, example in enumerate(examples):
                try:
                    # 기존 RAG 데이터셋 형식 (query, document, score)
                    if 'document' in example:
                        query = example.get('query', '')
                        document = example.get('document', '')
                        score = example.get('score', 1.0)
                        
                        # 유효성 검사
                        if not query or not document:
                            continue
                        
                        input_example = InputExample(
                            texts=[query, document],
                            label=float(score)
                        )
                        training_examples.append(input_example)
                        valid_examples += 1
                    
                    # 파일 업로드 데이터셋 형식 (query, positive, negative)
                    elif 'positive' in example or 'negative' in example:
                        query = example.get('query', '')
                        positive = example.get('positive', '')
                        negative = example.get('negative', '')
                        
                        # Positive 예제 추가
                        if query and positive:
                            input_example = InputExample(
                                texts=[query, positive],
                                label=1.0
                            )
                            training_examples.append(input_example)
                            valid_examples += 1
                        
                        # Negative 예제 추가 (있는 경우만)
                        if query and negative:
                            input_example = InputExample(
                                texts=[query, negative],
                                label=0.0
                            )
                            training_examples.append(input_example)
                            valid_examples += 1
                    
                    else:
                        log(f"예제 {i+1}: 알 수 없는 형식 - {example.keys()}", error=True)
                        continue
                        
                except Exception as e:
                    log(f"예제 {i+1} InputExample 생성 실패: {str(e)}", error=True)
                    continue
            
            log(f"총 처리된 예제: {len(examples)}")
            log(f"유효한 훈련 예제 수: {len(training_examples)}")
            
            return training_examples
            
        except Exception as e:
            log(f"훈련 예제 로드 실패: {str(e)}", error=True)
            import traceback
            log(f"스택 트레이스: {traceback.format_exc()}", error=True)
            return []
    
    async def fine_tune_model(self, model_name: str, dataset_id: str,
                             num_epochs: int = 3, batch_size: int = 4,
                             learning_rate: float = 2e-5, force_cpu: bool = False,
                             embedding_service=None,
                             user_set_epochs: bool = False,
                             user_set_batch_size: bool = False,
                             user_set_learning_rate: bool = False):
        """실제 모델 파인튜닝 실행 (GPU 메모리 부족 시 자동 CPU 전환)"""
        import time
        import json
        import uuid
        import redis

        log(f"파인튜닝 함수 시작: {model_name}, {dataset_id}")

        # CUDA 메모리 최적화 설정
        import os
        os.environ['PYTORCH_CUDA_ALLOC_CONF'] = 'expandable_segments:True'

        # 파인튜닝 시작 상태 저장
        finetune_state_manager.save_state({
            'status': 'starting',
            'model_name': model_name,
            'dataset_id': dataset_id,
            'num_epochs': num_epochs,
            'batch_size': batch_size,
            'learning_rate': learning_rate,
            'force_cpu': force_cpu,
            'start_time': time.time(),
            'progress': 0,
            'current_epoch': 0,
            'current_step': 0
        })
        
        # 첫 번째 시도: GPU 모드 (force_cpu가 False인 경우)
        if not force_cpu:
            try:
                log("=== GPU 파인튜닝 시도 ===")
                result = await self._attempt_finetuning(
                    model_name, dataset_id, num_epochs, batch_size, 
                    learning_rate, force_cpu=False, embedding_service=embedding_service,
                    user_set_epochs=user_set_epochs, user_set_batch_size=user_set_batch_size,
                    user_set_learning_rate=user_set_learning_rate
                )
                log("GPU 파인튜닝 성공!")
                
                # 성공 상태 저장
                finetune_state_manager.save_state({
                    'status': 'completed',
                    'model_name': model_name,
                    'dataset_id': dataset_id,
                    'completed_at': time.time(),
                    'result': result,
                    'device_used': result.get('device_used', 'gpu')
                })
                
                return result
                
            except Exception as gpu_error:
                log(f"GPU 파인튜닝 실패: {str(gpu_error)}", error=True)
                
                # GPU 메모리 부족인지 확인
                if "out of memory" in str(gpu_error).lower() or "cuda" in str(gpu_error).lower():
                    log("🔄 GPU 메모리 부족 감지 - 강제 VRAM 정리 후 CPU 모드로 전환")

                    # 강제 VRAM 정리 (CPU 전환 전)
                    import torch
                    import gc
                    if torch.cuda.is_available():
                        log("🧹 GPU 실패 후 강제 VRAM 정리 시작...")
                        for i in range(5):  # 5번 반복
                            torch.cuda.empty_cache()
                            torch.cuda.synchronize()
                            gc.collect()
                            time.sleep(0.2)

                        try:
                            torch.cuda.ipc_collect()
                        except:
                            pass

                        memory_before = torch.cuda.memory_allocated() / 1024 / 1024 / 1024
                        log(f"🧹 강제 VRAM 정리 완료 - 할당된 메모리: {memory_before:.2f}GB")

                    log("=== CPU 파인튜닝 시도 ===")
                    
                    # GPU 실패 상태 저장
                    finetune_state_manager.update_state(
                        status='retrying_cpu',
                        gpu_error=str(gpu_error),
                        retry_reason='gpu_memory_insufficient'
                    )
                    
                    # CPU 모드로 재시도
                    try:
                        result = await self._attempt_finetuning(
                            model_name, dataset_id, num_epochs, batch_size, 
                            learning_rate, force_cpu=True, embedding_service=embedding_service,
                            user_set_epochs=user_set_epochs, user_set_batch_size=user_set_batch_size,
                            user_set_learning_rate=user_set_learning_rate
                        )
                        log("✅ CPU 파인튜닝 성공!")
                        
                        # 성공 상태 저장
                        finetune_state_manager.save_state({
                            'status': 'completed',
                            'model_name': model_name,
                            'dataset_id': dataset_id,
                            'completed_at': time.time(),
                            'result': result,
                            'fallback_to_cpu': True,
                            'device_used': 'cpu',
                            'auto_fallback': True
                        })
                        
                        return result
                        
                    except Exception as cpu_error:
                        log(f"❌ CPU 파인튜닝도 실패: {str(cpu_error)}", error=True)
                        
                        # 실패 상태 저장
                        finetune_state_manager.save_state({
                            'status': 'failed',
                            'model_name': model_name,
                            'dataset_id': dataset_id,
                            'failed_at': time.time(),
                            'last_error': f"GPU 및 CPU 파인튜닝 모두 실패. GPU 오류: {str(gpu_error)}, CPU 오류: {str(cpu_error)}",
                            'error_type': 'BothModesFailed',
                            'is_resource_failure': True
                        })
                        
                        return {
                            'status': 'failed',
                            'error': f"GPU 및 CPU 파인튜닝 모두 실패. GPU 오류: {str(gpu_error)}, CPU 오류: {str(cpu_error)}",
                            'error_type': 'BothModesFailed'
                        }
                else:
                    # 메모리 부족이 아닌 다른 오류
                    log(f"GPU 파인튜닝 실패 (메모리 부족 아님): {str(gpu_error)}", error=True)
                    
                    # 실패 상태 저장
                    finetune_state_manager.save_state({
                        'status': 'failed',
                        'model_name': model_name,
                        'dataset_id': dataset_id,
                        'failed_at': time.time(),
                        'last_error': str(gpu_error),
                        'error_type': type(gpu_error).__name__
                    })
                    
                    return {
                        'status': 'failed',
                        'error': str(gpu_error),
                        'error_type': type(gpu_error).__name__
                    }
        
        # 처음부터 CPU 모드로 시작
        else:
            log("=== CPU 파인튜닝 시작 (강제 모드) ===")
            try:
                result = await self._attempt_finetuning(
                    model_name, dataset_id, num_epochs, batch_size, 
                    learning_rate, force_cpu=True, embedding_service=embedding_service,
                    user_set_epochs=user_set_epochs, user_set_batch_size=user_set_batch_size,
                    user_set_learning_rate=user_set_learning_rate
                )
                log("CPU 파인튜닝 성공!")
                
                # 성공 상태 저장
                finetune_state_manager.save_state({
                    'status': 'completed',
                    'model_name': model_name,
                    'dataset_id': dataset_id,
                    'completed_at': time.time(),
                    'result': result,
                    'forced_cpu': True,
                    'device_used': 'cpu'
                })
                
                return result
                
            except Exception as cpu_error:
                log(f"CPU 파인튜닝 실패: {str(cpu_error)}", error=True)
                
                # 실패 상태 저장
                finetune_state_manager.save_state({
                    'status': 'failed',
                    'model_name': model_name,
                    'dataset_id': dataset_id,
                    'failed_at': time.time(),
                    'last_error': str(cpu_error),
                    'error_type': type(cpu_error).__name__,
                    'is_resource_failure': 'memory' in str(cpu_error).lower() or 'resource' in str(cpu_error).lower()
                })
                
                return {
                    'status': 'failed',
                    'error': str(cpu_error),
                    'error_type': type(cpu_error).__name__
                }
    
    async def _attempt_finetuning(self, model_name: str, dataset_id: str, 
                                 num_epochs: int, batch_size: int,
                                 learning_rate: float, force_cpu: bool,
                                 embedding_service=None,
                                 user_set_epochs: bool = False,
                                 user_set_batch_size: bool = False,
                                 user_set_learning_rate: bool = False):
        """실제 파인튜닝 시도 (GPU 또는 CPU) - 확실한 모델 해제 보장"""
        
        # 모델 해제를 보장하기 위한 변수들 초기화
        model = None
        custom_train_loss = None
        train_dataloader = None
        train_examples = None
        fit_kwargs = None
        flop_tracker = None
        hardware_start_state = None
        
        try:
            # 시스템 자원 분석 및 최적화 설정 생성
            log("시스템 자원 분석 시작")
            resources = SystemResourceAnalyzer.analyze_system_resources()
            optimized_settings = SystemResourceAnalyzer.generate_optimized_settings(resources, force_cpu)
            
            # 초기 상태 로깅
            mode = "CPU" if force_cpu else "GPU"
            log(f"{mode} 파인튜닝 초기 설정 시작")
            
            # 디바이스 설정
            device = 'cpu' if force_cpu else 'cuda'
            log(f"최종 디바이스: {device}")
            
            # 최적화된 설정 적용 (사용자 설정 존중)
            original_batch_size = batch_size
            original_num_epochs = num_epochs
            
            # 배치 크기 처리 - 사용자가 명시적으로 설정했는지 확인
            if not user_set_batch_size:
                batch_size = optimized_settings.batch_size
                log(f"배치 크기 자동 최적화: {original_batch_size} → {batch_size}")
            else:
                log(f"사용자 지정 배치 크기 유지: {batch_size}")
            
            num_workers = optimized_settings.num_workers
            pin_memory = optimized_settings.pin_memory
            use_amp = optimized_settings.use_amp

            # CPU 모드에서는 강제로 num_workers=0으로 설정 (pickle 오류 방지)
            if device == 'cpu':
                num_workers = 0
                pin_memory = False
                use_amp = False
                log("CPU 모드 - 멀티프로세싱 비활성화 (num_workers=0, pin_memory=False, use_amp=False)")
            
            # 에포크 수 처리 - 사용자가 명시적으로 설정했는지 확인
            if not user_set_epochs:
                num_epochs = optimized_settings.recommended_epochs
                log(f"에포크 수 자동 조정: {original_num_epochs} → {num_epochs}")
            else:
                log(f"사용자 지정 에포크 수 유지: {num_epochs}")
            
            # 학습률 처리 - 사용자가 명시적으로 설정했는지 확인
            if not user_set_learning_rate:
                # 필요시 학습률도 자동 조정할 수 있도록 준비
                log(f"기본 학습률 사용: {learning_rate}")
            else:
                log(f"사용자 지정 학습률 유지: {learning_rate}")
            
            log(f"최적화 설정 적용:")
            log(f"  배치 크기: {original_batch_size} → {batch_size}")
            log(f"  에포크 수: {original_num_epochs} → {num_epochs}")
            log(f"  학습률: {learning_rate}")
            log(f"  워커 수: {num_workers}")
            log(f"  Pin Memory: {pin_memory}")
            log(f"  Mixed Precision: {use_amp}")
            
            # CPU 강제 모드 설정
            if force_cpu:
                log("CPU 강제 모드 설정 시작")
                os.environ['CUDA_VISIBLE_DEVICES'] = ''
                
                # PyTorch CUDA 비활성화
                torch.cuda.is_available = lambda: False
                log("CUDA 비활성화 완료 (CPU 강제 모드)")
            
            # 기존 모델 메모리 관리
            if embedding_service and self.use_existing_model:
                log("메모리 효율적인 모델 준비 시작")
                self.prepare_for_finetuning(embedding_service)
            
            # 데이터셋 로드
            log(f"데이터셋 로드 시작: {dataset_id}")
            train_examples = self._load_training_examples(dataset_id)
            log(f"훈련 샘플 수: {len(train_examples)}")
            
            if not train_examples:
                error_msg = f"훈련 데이터가 없습니다: {dataset_id}"
                log(error_msg, error=True)
                raise ValueError(error_msg)
            
            # 모델 로드 - 임베딩 워커 모델 재사용 시도
            log(f"모델 로드 시작: {self.base_model_name} (파인튜닝 대상: {model_name})")

            model = None
            from sentence_transformers import SentenceTransformer

            # 임베딩 워커 모델 직접 재사용 시도 (메모리 효율성)
            if embedding_service and hasattr(embedding_service, 'model') and embedding_service.model is not None:
                try:
                    log("임베딩 워커의 기존 모델 직접 재사용 시도 (VRAM 효율성)")

                    # 임베딩 서비스의 모델을 직접 복사
                    import copy
                    model = copy.deepcopy(embedding_service.model)

                    # 디바이스 설정
                    if device == 'cuda' and hasattr(model, 'to'):
                        model = model.to(device)
                    elif device == 'cpu' and hasattr(model, 'to'):
                        model = model.to('cpu')

                    log(f"✅ 임베딩 워커 모델 직접 복사 완료 - 디바이스: {device}")

                except Exception as copy_error:
                    log(f"⚠️ 임베딩 워커 모델 직접 복사 실패: {copy_error}")
                    model = None

            # 직접 복사가 실패하면 Redis 통신 시도
            if model is None and embedding_service:
                try:
                    log("Redis 통신을 통한 모델 복사 시도")

                    redis_host = os.getenv('REDIS_HOST', 'localhost')
                    redis_port = int(os.getenv('REDIS_PORT', '6379'))
                    redis_client = redis.Redis(host=redis_host, port=redis_port, db=0, decode_responses=True)

                    # 임베딩 워커에게 모델 복사 요청
                    task_id = str(uuid.uuid4())
                    task_data = {
                        'type': 'get_model_for_training',
                        'task_id': task_id,
                        'model_name': self.base_model_name,
                        'device': device
                    }

                    redis_client.lpush('embedding_tasks', json.dumps(task_data))
                    log(f"임베딩 워커에게 모델 복사 요청: {task_id}")

                    # 결과 대기 (최대 10초)
                    for i in range(100):  # 10초 (0.1초씩 100번)
                        result_key = f"model_copy_result:{task_id}"
                        result_data = redis_client.get(result_key)

                        if result_data:
                            result = json.loads(result_data)
                            redis_client.delete(result_key)

                            if result.get('success'):
                                # 임베딩 워커가 모델을 메모리에서 제공했음을 확인
                                log("임베딩 워커로부터 모델 정보 수신 완료")
                                # 기본 모델을 새로 로딩하되, 같은 디바이스 사용으로 메모리 공유 최적화
                                model = SentenceTransformer(self.base_model_name, device=device)
                                log("임베딩 워커 참조 모델 로딩 완료 (메모리 효율화)")
                                break
                            else:
                                log(f"임베딩 워커 모델 복사 실패: {result.get('error')}")
                                break

                        time.sleep(0.1)

                    if model is None:
                        log("임베딩 워커 모델 재사용 실패 - 새 모델 로딩으로 폴백")

                except Exception as e:
                    log(f"임베딩 워커 모델 재사용 중 오류: {str(e)} - 새 모델 로딩으로 폴백")

            # 모델이 없으면 새로 로딩
            if model is None:
                if device == 'cpu':
                    log("CPU 모드로 새 모델 로드")
                else:
                    log("GPU 모드로 새 모델 로드")

                model = SentenceTransformer(self.base_model_name, device=device)
                log("새 모델 로드 완료")
            
            # 훈련 설정
            log("훈련 설정 시작")
            from sentence_transformers import losses
            from torch.utils.data import DataLoader
            
            # DataLoader 생성 (자동 최적화)
            log("DataLoader 생성 시작")
            train_dataloader = DataLoader(
                train_examples, 
                shuffle=True, 
                batch_size=batch_size,
                num_workers=num_workers,  # 자동 최적화
                pin_memory=pin_memory,  # 자동 최적화
                drop_last=False
            )
            log(f"DataLoader 생성 완료, 배치 수: {len(train_dataloader)}")
            
            # 커스텀 손실 함수 (진행률 추적 포함)
            training_step_counter = [0]
            total_training_steps = len(train_dataloader) * num_epochs
            log(f"총 예상 훈련 스텝 수: {total_training_steps}")
            
            # 총 스텝 수를 상태에 저장
            finetune_state_manager.update_state(
                total_steps=total_training_steps,
                status='training'
            )
            
            class LoggingCosineSimilarityLoss(losses.CosineSimilarityLoss):
                """로깅 기능이 추가된 CosineSimilarityLoss"""
                def __init__(self, model, total_steps, *args, **kwargs):
                    super().__init__(model, *args, **kwargs)
                    self.total_steps = total_steps
                    
                def forward(self, sentence_features, labels):
                    training_step_counter[0] += 1
                    loss_value = super().forward(sentence_features, labels)
                    
                    # 진행률 계산
                    current_step = training_step_counter[0]
                    progress_percent = (current_step / self.total_steps) * 100
                    
                    # 매 스텝마다 상세 로깅 (진행률 포함)
                    log(f"🔥 [실제훈련진행] 스텝 {current_step}/{self.total_steps} ({progress_percent:.1f}%), 손실: {loss_value.item():.6f}")
                    
                    # 파인튜닝 진행 상태 저장
                    finetune_state_manager.update_state(
                        status='training',
                        current_step=current_step,
                        total_steps=self.total_steps,
                        progress=progress_percent,
                        current_loss=loss_value.item(),
                        eta=f'훈련 중... (스텝 {current_step}/{self.total_steps})'
                    )
                    
                    return loss_value
            
            # 커스텀 손실 함수 생성
            custom_train_loss = LoggingCosineSimilarityLoss(model, total_training_steps)
            log("커스텀 로깅 손실 함수 생성됨 (진행률 추적 포함)")
            
            log(f"파인튜닝 훈련 시작 - 에포크: {num_epochs}, 배치 크기: {batch_size}")
            log(f"학습률: {learning_rate}, 디바이스: {device}")
            
            # 자동 최적화된 설정
            warmup_steps = max(1, int(len(train_examples) * optimized_settings.warmup_ratio))
            
            fit_kwargs = {
                'train_objectives': [(train_dataloader, custom_train_loss)],
                'epochs': num_epochs,
                'warmup_steps': warmup_steps,
                'output_path': None,  # 자동 저장 비활성화
                'show_progress_bar': True,
                'optimizer_params': {'lr': learning_rate},
                'scheduler': 'WarmupLinear',
                'save_best_model': False,  # 메모리 절약
                'evaluation_steps': 0,  # 평가 비활성화 (메모리 절약)
                'checkpoint_path': None,  # 중간 저장 비활성화
                'checkpoint_save_steps': None,  # 중간 저장 비활성화
                'checkpoint_save_total_limit': 0,  # 중간 저장 비활성화
            }
            
            log("=== model.fit() 호출 시작 ===")
            
            # 🔥 실제 하드웨어 FLOP 추적 시작
            flop_monitoring_result = None
            hardware_start_state = None
            
            # FLOP 추적기 초기화 (안전한 방식)
            hardware_start_state = None
            flop_tracker = None
            
            try:
                # 경로 설정
                current_file_path = os.path.abspath(__file__)
                parent_dir = os.path.dirname(os.path.dirname(current_file_path))
                sys.path.insert(0, parent_dir)
                
                from services.flop_tracker import FLOPUsageTracker as FLOPTracker
                flop_tracker = FLOPTracker()
                
                # 하드웨어 모니터링 시작 (로컬 파인튜닝이므로 실제 측정)
                log("🔥 파인튜닝 하드웨어 FLOP 추적 시작...")
                hardware_start_state = flop_tracker.hardware_monitor.start_monitoring()
                
                if hardware_start_state and hardware_start_state.get('gpu_states'):
                    log(f"   시작 GPU 사용률: {hardware_start_state['gpu_states'][0]['utilization']:.1f}%")
                    log(f"   시작 GPU 메모리: {hardware_start_state['gpu_states'][0]['memory_used']:.0f}MB")
                else:
                    log("   GPU 정보 없음")
                
            except ImportError as import_error:
                log(f"⚠️  FLOP 추적 모듈 import 실패: {import_error}")
                hardware_start_state = None
                flop_tracker = None
            except Exception as flop_error:
                log(f"⚠️  FLOP 추적 초기화 실패: {flop_error}")
                hardware_start_state = None
                flop_tracker = None
            
            # 실제 훈련 실행
            training_start_time = time.time()
            model.fit(**fit_kwargs)
            training_end_time = time.time()
            training_duration_ms = (training_end_time - training_start_time) * 1000
            
            # 🔥 실제 하드웨어 FLOP 추적 종료
            try:
                if hardware_start_state and 'flop_tracker' in locals() and flop_tracker:
                    log("📊 파인튜닝 하드웨어 FLOP 추적 종료...")
                    
                    # 하드웨어 모니터링 종료
                    hardware_end_state = flop_tracker.hardware_monitor.end_monitoring(hardware_start_state)
                    
                    # 실제 FLOP 사용량 기록
                    estimated_prompt_tokens = len(train_examples) * 50  # 평균 토큰 수 추정
                    estimated_completion_tokens = len(train_examples) * 20  # 평균 출력 토큰 수 추정
                    
                    flop_usage = flop_tracker.track_usage(
                        provider='local_finetuning',
                        model=model_name,
                        prompt_tokens=estimated_prompt_tokens,
                        completion_tokens=estimated_completion_tokens,
                        duration_ms=training_duration_ms,
                        session_id=f"finetune_{dataset_id}_{int(time.time())}",
                        user_id='system',
                        username='finetune_system',
                        feature='model_finetuning',
                        hardware_monitoring_data=hardware_end_state
                    )
                    
                    if flop_usage:
                        total_flops = flop_usage.estimated_flops
                        gpu_utilization = flop_usage.gpu_utilization or 0
                        measurement_type = flop_usage.measurement_type
                        
                        log(f"🎯 파인튜닝 FLOP 추적 완료:")
                        log(f"   측정 방식: {measurement_type}")
                        log(f"   총 FLOP: {total_flops:.2e} FLOPS ({total_flops/1e12:.2f} TFLOPS)")
                        log(f"   평균 GPU 사용률: {gpu_utilization:.1f}%")
                        log(f"   훈련 시간: {training_duration_ms/1000:.1f}초")
                        log(f"   FLOP/초: {total_flops/(training_duration_ms/1000):.2e} FLOPS/sec")
                        
                        # 메타데이터에 FLOP 정보 추가
                        flop_monitoring_result = {
                            'total_flops': total_flops,
                            'measurement_type': measurement_type,
                            'gpu_utilization': gpu_utilization,
                            'training_duration_ms': training_duration_ms,
                            'flops_per_second': total_flops/(training_duration_ms/1000),
                            'estimated_tokens': estimated_prompt_tokens + estimated_completion_tokens
                        }
                    else:
                        log("⚠️  FLOP 사용량 기록 실패")
                        
            except Exception as flop_end_error:
                log(f"⚠️  FLOP 추적 종료 실패: {flop_end_error}")
            
            log("=== model.fit() 호출 완료 ===")
            log(f"총 훈련 스텝 수: {training_step_counter[0]}")
            log("model.fit() 완료 - 모델 훈련 완료!")
            
            # 모델 저장
            log("모델 저장 시작")
            fine_tuned_model_name = f"{model_name.replace('/', '-')}-finetuned-{dataset_id[:8]}"
            # 통일된 절대 경로 사용
            base_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))  # /home/chaeya/workspaces/airun
            finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
            model_path = os.path.join(finetuned_dir, fine_tuned_model_name)
            
            # 모델 저장 디렉토리 생성
            log(f"모델 저장 디렉토리 생성: {model_path}")
            os.makedirs(model_path, exist_ok=True)
            
            # 모델 저장 시도
            try:
                log("model.save() 호출 시작...")
                model.save(model_path)
                log("model.save() 호출 완료")
                
                # 저장 결과 검증
                if os.path.exists(f"{model_path}/config.json"):
                    log(f"✅ 모델 저장 성공 확인: {model_path}/config.json 존재")
                else:
                    log(f"❌ 모델 저장 실패: {model_path}/config.json 없음", error=True)
                    
                # 저장된 파일 목록 확인
                saved_files = os.listdir(model_path)
                log(f"저장된 파일 목록 ({len(saved_files)}개): {saved_files}")
                
            except Exception as save_error:
                log(f"❌ model.save() 오류: {str(save_error)}", error=True)
                raise save_error
            
            log(f"파인튜닝된 모델 저장됨: {model_path}")
            
            # 메타데이터 저장
            from datetime import datetime
            metadata = {
                'name': fine_tuned_model_name,
                'base_model': self.base_model_name,
                'dataset': dataset_id,
                'epochs': num_epochs,
                'batch_size': batch_size,
                'original_batch_size': original_batch_size,
                'learning_rate': learning_rate,
                'device': device,
                'training_samples': len(train_examples),
                'created_at': datetime.now().isoformat(),
                'mixed_precision': device == 'cuda',
                'gradient_accumulation_steps': 1, # model.fit()는 자동으로 처리
                'auto_fallback': device == 'cpu' and not force_cpu,  # 자동 전환 여부
                'path': model_path
            }
            
            # 🔥 FLOP 추적 결과를 메타데이터에 추가
            if flop_monitoring_result:
                metadata['flop_tracking'] = flop_monitoring_result
                log(f"✅ FLOP 추적 결과가 메타데이터에 저장됨")
            else:
                metadata['flop_tracking'] = {
                    'total_flops': 0,
                    'measurement_type': 'none',
                    'gpu_utilization': 0,
                    'training_duration_ms': training_duration_ms if 'training_duration_ms' in locals() else 0,
                    'note': 'FLOP 추적 실패 또는 비활성화'
                }
                log(f"⚠️  FLOP 추적 없이 메타데이터 저장")
            
            metadata_path = f"{model_path}/metadata.json"
            with open(metadata_path, 'w', encoding='utf-8') as f:
                json.dump(metadata, f, indent=2, ensure_ascii=False)
            
            log("파인튜닝 전체 과정 완료")
            
            # 자원 정리는 finally 블록에서 처리됨
            log("파인튜닝 성공 - 자원 정리는 finally 블록에서 수행됨")
            
            # 성공 시 원래 모델 복원
            log("파인튜닝 성공 - 원래 모델 복원 시작")
            self.restore_original_model(embedding_service)
            
            # 실제 사용된 디바이스 정보 확인
            actual_device_used = device
            device_info = {
                'device': actual_device_used,
                'device_name': f"{actual_device_used.upper()} 모드",
                'forced_cpu': force_cpu,
                'auto_fallback': device == 'cpu' and not force_cpu,
                'gpu_available': torch.cuda.is_available() if not force_cpu else False
            }
            
            # GPU 사용 시 추가 정보
            if actual_device_used == 'cuda' and torch.cuda.is_available():
                device_info.update({
                    'gpu_name': torch.cuda.get_device_name(0),
                    'gpu_memory_total': f"{torch.cuda.get_device_properties(0).total_memory / 1024**3:.1f}GB"
                })
            
            # 훈련 완료 상태 저장
            finetune_state_manager.save_state({
                'status': 'completed',
                'model_name': model_name,
                'dataset_id': dataset_id,
                'completed_at': time.time(),
                'progress': 100,
                'eta': '완료',
                'completed_model_name': fine_tuned_model_name,
                'completed_model_path': model_path,
                'result': {
                    'status': 'completed',
                    'model_name': fine_tuned_model_name,
                    'model_path': model_path,
                    'metadata': metadata,
                    'device_used': actual_device_used,
                    'device_info': device_info,
                    'auto_fallback': device == 'cpu' and not force_cpu
                },
                'device_used': actual_device_used,
                'auto_fallback': device == 'cpu' and not force_cpu
            })
            
            # 🔥 FLOP 정보를 결과에 포함
            result = {
                'status': 'completed',
                'model_name': fine_tuned_model_name,
                'model_path': model_path,
                'metadata': metadata,
                'device_used': actual_device_used,
                'device_info': device_info,
                'auto_fallback': device == 'cpu' and not force_cpu
            }
            
            # FLOP 추적 결과 추가
            if flop_monitoring_result:
                result['flop_tracking'] = flop_monitoring_result
                log(f"✅ 파인튜닝 결과에 FLOP 정보 포함: {flop_monitoring_result['total_flops']:.2e} FLOPS")

            # 데이터베이스에 완성된 모델 등록
            try:
                from rag_process import PostgreSQLWrapper
                db_wrapper = PostgreSQLWrapper()
                conn = db_wrapper.get_connection()
                cursor = conn.cursor()

                cursor.execute("""
                    INSERT INTO finetuned_models (id, base_model, dataset_id, status, output_path, created_at, updated_at)
                    VALUES (%s, %s, %s, %s, %s, %s, %s)
                    ON CONFLICT (id) DO UPDATE SET
                        status = EXCLUDED.status,
                        output_path = EXCLUDED.output_path,
                        updated_at = EXCLUDED.updated_at
                """, (
                    fine_tuned_model_name,
                    self.base_model_name,
                    dataset_id,
                    'completed',
                    model_path,
                    datetime.now(),
                    datetime.now()
                ))
                conn.commit()
                cursor.close()
                db_wrapper.return_connection(conn)
                log(f"✅ 데이터베이스에 모델 등록 완료: {fine_tuned_model_name}")
            except Exception as db_error:
                log(f"⚠️ 데이터베이스 등록 실패 (모델은 정상 생성됨): {str(db_error)}", error=True)
                try:
                    if 'cursor' in locals():
                        cursor.close()
                    if 'conn' in locals():
                        db_wrapper.return_connection(conn)
                except:
                    pass

            return result
            
        except Exception as e:
            log(f"파인튜닝 오류: {str(e)}", error=True)
            log(f"오류 타입: {type(e).__name__}")
            
            # 훈련 실패 상태 저장
            finetune_state_manager.save_state({
                'status': 'failed',
                'model_name': model_name,
                'dataset_id': dataset_id,
                'failed_at': time.time(),
                'progress': 0,
                'eta': '실패',
                'last_error': str(e),
                'failure_reason': f"{type(e).__name__}: {str(e)}",
                'error_type': type(e).__name__,
                'is_resource_failure': 'memory' in str(e).lower() or 'resource' in str(e).lower()
            })
            
            # 자원 정리는 finally 블록에서 처리됨
            log("오류 발생 - 자원 정리는 finally 블록에서 수행됨")
            
            # 원래 모델 복원 시도
            try:
                log("오류 발생 - 원래 모델 복원 시작")
                self.restore_original_model(embedding_service)
            except Exception as restore_error:
                log(f"원래 모델 복원 실패: {str(restore_error)}", error=True)
            
            # 오류 재발생
            raise e
            
        finally:
            # ⚠️ 모든 상황에서 모델 해제를 확실히 보장하는 finally 블록 ⚠️
            log("🔧 finally 블록 - 모델 해제 시작 (모든 상황에서 실행)")
            try:
                # 1. 손실 함수 객체 해제
                if custom_train_loss is not None:
                    del custom_train_loss
                    custom_train_loss = None
                    log("✅ 손실 함수 객체 해제 완료 (finally)")

                # 2. DataLoader 관련 객체 해제
                if train_dataloader is not None:
                    del train_dataloader
                    train_dataloader = None
                    log("✅ DataLoader 자원 해제 완료 (finally)")

                # 3. 훈련 예제 해제
                if train_examples is not None:
                    del train_examples
                    train_examples = None
                    log("✅ 훈련 예제 데이터 해제 완료 (finally)")

                # 4. 로컬 모델 객체 해제 (가장 중요!)
                if model is not None:
                    del model
                    model = None
                    log("🔥 로컬 모델 객체 해제 완료 (finally) - GPU 메모리 해제됨")

                # 5. 클래스 인스턴스 모델 해제
                if hasattr(self, 'model') and self.model is not None:
                    del self.model
                    self.model = None
                    log("✅ 파인튜닝 클래스 모델 객체 해제 완료 (finally)")

                # 6. fit 관련 파라미터 정리
                if fit_kwargs is not None:
                    del fit_kwargs
                    fit_kwargs = None
                    log("✅ fit 파라미터 객체 해제 완료 (finally)")

                # 7. FLOP 추적기 정리
                if flop_tracker is not None:
                    try:
                        if hasattr(flop_tracker, 'hardware_monitor') and hardware_start_state:
                            flop_tracker.hardware_monitor.end_monitoring(hardware_start_state)
                    except:
                        pass
                    del flop_tracker
                    flop_tracker = None
                    log("✅ FLOP 추적기 해제 완료 (finally)")

                # 8. Python 가비지 컬렉션 강제 실행
                collected = gc.collect()
                log(f"🗑️ 가비지 컬렉션 완료 (수집된 객체: {collected}개) (finally)")

                # 9. GPU 메모리 정리 (강화된 버전)
                if torch.cuda.is_available():
                    # 3번에 걸쳐 강력하게 정리
                    for i in range(3):
                        torch.cuda.empty_cache()
                        torch.cuda.synchronize()
                        time.sleep(0.1)  # 잠시 대기

                    # IPC 정리 (프로세스 간 통신으로 사용된 메모리)
                    try:
                        torch.cuda.ipc_collect()
                    except:
                        pass

                    log("🔥 GPU 메모리 캐시 강화 정리 완료 (finally)")
                    
                    # GPU 메모리 상태 출력
                    try:
                        memory_allocated = torch.cuda.memory_allocated() / 1024 / 1024 / 1024  # GB
                        memory_cached = torch.cuda.memory_reserved() / 1024 / 1024 / 1024  # GB
                        log(f"📊 GPU 메모리 상태 (finally) - 할당됨: {memory_allocated:.2f}GB, 캐시됨: {memory_cached:.2f}GB")
                    except:
                        pass

                # 10. 추가적인 프로세스 레벨 CUDA 정리 (강화)
                if torch.cuda.is_available():
                    try:
                        import gc as gc_module  # gc import - 이름 충돌 방지

                        # 현재 프로세스의 모든 CUDA 텐서 정리
                        for obj in gc_module.get_objects():
                            if torch.is_tensor(obj) and obj.is_cuda:
                                try:
                                    del obj
                                except:
                                    pass

                        # 5번에 걸쳐 강력한 메모리 정리
                        for i in range(5):
                            torch.cuda.empty_cache()
                            torch.cuda.synchronize()
                            gc_module.collect()
                            time.sleep(0.2)

                        # CUDA 컨텍스트 재설정 (가장 강력한 방법)
                        try:
                            torch.cuda.reset_peak_memory_stats()
                            if hasattr(torch.cuda, 'reset_accumulated_memory_stats'):
                                torch.cuda.reset_accumulated_memory_stats()
                        except:
                            pass

                        log("🔥 프로세스 레벨 CUDA 메모리 강화 정리 완료 (finally)")
                    except Exception as cuda_cleanup_error:
                        log(f"⚠️ CUDA 강화 정리 중 오류 (무시됨): {str(cuda_cleanup_error)}")

                log("✅ finally 블록 - 모든 모델 해제 완료")

            except Exception as finally_error:
                log(f"❌ finally 블록에서 모델 해제 중 오류: {str(finally_error)}", error=True)
                # finally에서 오류가 발생해도 원본 예외를 방해하지 않음

class TrainingDatasetManager:
    """훈련 데이터셋 관리자"""
    
    def __init__(self, data_dir: str = None):
        # data_dir이 None이면 통일된 ~/.airun/datasets 경로 사용
        if data_dir is None:
            data_dir = os.path.join(os.path.expanduser('~'), '.airun', 'datasets')
        
        self.data_dir = Path(data_dir)
        self.data_dir.mkdir(parents=True, exist_ok=True)
        
    def create_dataset_from_documents(self, documents: List[str], 
                                    name: str, description: str = "") -> str:
        """문서로부터 데이터셋 생성 (기존 방식 - 호환성 유지)"""
        dataset_id = str(uuid.uuid4())[:8]
        
        # 문서 기반 positive/negative 쌍 생성
        examples = []
        for i, doc in enumerate(documents):
            sentences = self._split_into_sentences(doc)
            if len(sentences) >= 2:
                # Positive pairs (같은 문서 내 문장들)
                for j in range(len(sentences) - 1):
                    examples.append({
                        'query': sentences[j],
                        'document': sentences[j + 1],
                        'score': 1.0,
                        'type': 'document_positive'
                    })
                
                # Negative pairs (다른 문서의 문장들)
                for _ in range(min(2, len(sentences))):
                    neg_idx = random.randint(0, len(documents) - 1)
                    if neg_idx != i:
                        neg_sentences = self._split_into_sentences(documents[neg_idx])
                        if neg_sentences:
                            examples.append({
                                'query': sentences[0],
                                'document': neg_sentences[0],
                                'score': 0.0,
                                'type': 'document_negative'
                            })
        
        # 데이터셋 저장
        from datetime import datetime
        dataset_data = {
            'id': dataset_id,
            'name': name,
            'description': description,
            'created_at': datetime.now().isoformat(),
            'type': 'document_based',
            'size': len(examples),
            'examples': examples
        }
        
        dataset_path = self.data_dir / f"{dataset_id}.json"
        with open(dataset_path, 'w', encoding='utf-8') as f:
            json.dump(dataset_data, f, ensure_ascii=False, indent=2)
        
        logger.info(f"문서 기반 데이터셋 생성 완료: {dataset_id} ({len(examples)}개 예제)")
        return dataset_id
    
    async def create_dataset_from_documents_with_ai(self, documents: List[str], 
                                                  name: str, description: str = "") -> str:
        """문서로부터 AI 기반 QA 데이터셋 생성 (개선된 방식)"""
        dataset_id = str(uuid.uuid4())[:8]
        
        logger.info(f"AI 기반 컬렉션 데이터셋 생성 시작: {name} ({len(documents)}개 문서)")
        
        all_examples = []
        processed_docs = 0
        
        # 각 문서에서 질문-답변 쌍 생성
        for i, doc in enumerate(documents):
            if len(doc.strip()) < 100:  # 너무 짧은 문서는 스킵
                continue
                
            try:
                # AI로 QA 쌍 생성 (기존 QAGenerator 활용)
                qa_pairs = await generate_qa_pairs_from_text(doc, f"{name}_doc_{i}")
                
                if qa_pairs:
                    all_examples.extend(qa_pairs)
                    processed_docs += 1
                    
                    logger.info(f"문서 {i+1}/{len(documents)} 처리 완료: {len(qa_pairs)}개 QA 생성")
                    
                    # 너무 많은 QA가 생성되지 않도록 제한
                    if len(all_examples) >= 500:
                        logger.info("QA 개수가 500개에 도달하여 중단")
                        break
                
            except Exception as e:
                logger.error(f"문서 {i+1} QA 생성 실패: {e}")
                continue
        
        if not all_examples:
            logger.error("AI 기반 QA 생성 실패, 기존 방식으로 폴백")
            return self.create_dataset_from_documents(documents, name, description)
        
        # 데이터셋 저장
        from datetime import datetime
        dataset_data = {
            'id': dataset_id,
            'name': name,
            'description': f"{description} (AI 기반 QA 생성, {processed_docs}개 문서 처리)",
            'created_at': datetime.now().isoformat(),
            'type': 'ai_generated_qa',
            'size': len(all_examples),
            'examples': all_examples,
            'metadata': {
                'total_documents': len(documents),
                'processed_documents': processed_docs,
                'generation_method': 'ai_qa_with_negatives'
            }
        }
        
        dataset_path = self.data_dir / f"{dataset_id}.json"
        with open(dataset_path, 'w', encoding='utf-8') as f:
            json.dump(dataset_data, f, ensure_ascii=False, indent=2)
        
        logger.info(f"AI 기반 컬렉션 데이터셋 생성 완료: {dataset_id} ({len(all_examples)}개 QA 쌍)")
        return dataset_id
    
    def create_dataset_from_search_logs(self, search_logs: List[Dict[str, Any]], 
                                      name: str, description: str = "") -> str:
        """검색 로그로부터 데이터셋 생성"""
        dataset_id = str(uuid.uuid4())[:8]
        
        examples = []
        for log in search_logs:
            query = log.get('query', '')
            clicked_docs = log.get('clicked_documents', [])
            not_clicked_docs = log.get('not_clicked_documents', [])
            
            # 클릭한 문서들은 positive
            for doc in clicked_docs:
                examples.append({
                    'query': query,
                    'document': doc,
                    'score': 1.0,
                    'type': 'search_log_positive'
                })
            
            # 클릭하지 않은 문서들은 negative
            for doc in not_clicked_docs[:2]:  # 최대 2개만
                examples.append({
                    'query': query,
                    'document': doc,
                    'score': 0.0,
                    'type': 'search_log_negative'
                })
        
        from datetime import datetime
        dataset_data = {
            'id': dataset_id,
            'name': name,
            'description': description,
            'created_at': datetime.now().isoformat(),
            'type': 'search_log_based',
            'size': len(examples),
            'examples': examples
        }
        
        dataset_path = self.data_dir / f"{dataset_id}.json"
        with open(dataset_path, 'w', encoding='utf-8') as f:
            json.dump(dataset_data, f, ensure_ascii=False, indent=2)
        
        return dataset_id
    
    def create_dataset_from_feedback(self, feedback_data: List[Dict[str, Any]], 
                                   name: str, description: str = "") -> str:
        """사용자 피드백 데이터로부터 데이터셋 생성"""
        dataset_id = str(uuid.uuid4())[:8]
        
        examples = []
        for feedback in feedback_data:
            query = feedback.get('query', '')
            document = feedback.get('document', '')
            rating = feedback.get('rating', 0)  # 1-5 점수
            
            # 평점을 0-1 스케일로 변환
            score = (rating - 1) / 4.0 if rating > 0 else 0.0
            
            examples.append({
                'query': query,
                'document': document,
                'score': score,
                'type': 'user_feedback',
                'rating': rating
            })
        
        from datetime import datetime
        dataset_data = {
            'id': dataset_id,
            'name': name,
            'description': description,
            'created_at': datetime.now().isoformat(),
            'type': 'user_feedback',
            'size': len(examples),
            'examples': examples
        }
        
        dataset_path = self.data_dir / f"{dataset_id}.json"
        with open(dataset_path, 'w', encoding='utf-8') as f:
            json.dump(dataset_data, f, ensure_ascii=False, indent=2)
        
        return dataset_id
    
    def _split_into_sentences(self, text: str) -> List[str]:
        """텍스트를 문장으로 분할"""
        sentences = []
        for sent in text.split('.'):
            sent = sent.strip()
            if len(sent) > 10:
                sentences.append(sent)
        return sentences
    
    def get_dataset(self, dataset_id: str) -> Optional[Dict[str, Any]]:
        """데이터셋 조회 (평가 데이터셋 포함)"""
        # 1. 일반 데이터셋 먼저 확인
        dataset_path = self.data_dir / f"{dataset_id}.json"
        if dataset_path.exists():
            with open(dataset_path, 'r', encoding='utf-8') as f:
                return json.load(f)

        # 2. 평가 데이터셋 확인 - 실제 파일명으로 매핑
        eval_file_mapping = {
            'universal_eval_user_uploaded': 'universal_benchmark.json',
            'domain_eval_user_uploaded': 'domain_benchmark.json'
        }

        if dataset_id in eval_file_mapping:
            eval_path = self.data_dir / eval_file_mapping[dataset_id]
            if eval_path.exists():
                with open(eval_path, 'r', encoding='utf-8') as f:
                    return json.load(f)

        return None
    
    def list_datasets(self) -> List[Dict[str, Any]]:
        """모든 데이터셋 목록 조회 (평가 데이터셋 포함)"""
        datasets = []
        
        # 1. 일반 훈련 데이터셋 로드 (기존 로직)
        for dataset_file in self.data_dir.glob("*.json"):
            # feedback_live.json은 API 서버에서 관리하므로 제외
            # _holdout.json 파일들은 홀드아웃 검증용이므로 제외
            # 평가 데이터셋들도 제외 (별도 처리)
            if (dataset_file.name == 'feedback_live.json' or
                dataset_file.name.endswith('_holdout.json') or
                dataset_file.name in ['universal_benchmark.json', 'domain_benchmark.json']):
                continue
            try:
                with open(dataset_file, 'r', encoding='utf-8') as f:
                    dataset_data = json.load(f)
                    # 요약 정보만 포함
                    datasets.append({
                        'id': dataset_data['id'],
                        'name': dataset_data['name'],
                        'description': dataset_data['description'],
                        'type': dataset_data['type'],
                        'size': dataset_data['size'],
                        'created_at': dataset_data['created_at'],
                        'system_managed': dataset_data['id'] == 'feedback_live',  # feedback_live만 시스템 관리
                        'deletable': dataset_data['id'] != 'feedback_live'  # feedback_live만 삭제 불가
                    })
            except Exception as e:
                log(f"데이터셋 로드 실패 {dataset_file}: {e}", error=True)

        # 2. 평가 데이터셋 추가 (자동 복사 로직 포함)
        import os
        import shutil

        # 사용자 데이터셋 경로와 기본 템플릿 경로
        user_datasets_path = os.path.expanduser("~/.airun/datasets")
        public_datasets_path = os.path.join(os.getcwd(), "workspaces/web/public/datasets")

        # 평가 데이터셋 자동 복사 로직
        def ensure_evaluation_datasets():
            """평가 데이터셋이 사용자 경로에 없으면 public 경로에서 복사"""
            try:
                # 디렉토리가 없으면 생성
                os.makedirs(user_datasets_path, exist_ok=True)

                evaluation_files = ['universal_benchmark.json', 'domain_benchmark.json']
                copied_files = []

                for filename in evaluation_files:
                    user_file_path = os.path.join(user_datasets_path, filename)
                    public_file_path = os.path.join(public_datasets_path, filename)

                    # 사용자 경로에 파일이 없고, public 경로에는 있을 때만 복사
                    if not os.path.exists(user_file_path) and os.path.exists(public_file_path):
                        try:
                            shutil.copy2(public_file_path, user_file_path)
                            copied_files.append(filename)
                            log(f"기본 평가 데이터셋 복사 완료: {filename}")
                        except Exception as e:
                            log(f"평가 데이터셋 복사 실패 {filename}: {e}", error=True)

                if copied_files:
                    log(f"설치 후 초기화: {len(copied_files)}개 평가 데이터셋 자동 설정 완료")

            except Exception as e:
                log(f"평가 데이터셋 자동 설정 실패: {e}", error=True)

        # 평가 데이터셋 자동 복사 실행
        ensure_evaluation_datasets()

        # 두 경로에서 평가 데이터셋 확인
        evaluation_paths = [
            (user_datasets_path, "사용자 경로"),
            (public_datasets_path, "기본 템플릿 경로")
        ]
        
        for benchmark_dir, path_desc in evaluation_paths:
            try:
                # universal_benchmark.json 확인
                universal_path = os.path.join(benchmark_dir, "universal_benchmark.json")
                if os.path.exists(universal_path):
                    try:
                        with open(universal_path, 'r', encoding='utf-8') as f:
                            universal_data = json.load(f)
                            # 중복 방지를 위해 ID 확인 (평가 데이터셋은 파일명 기반 ID 사용)
                            standard_id = 'universal_benchmark'
                            if not any(d.get('id') == standard_id for d in datasets):
                                datasets.append({
                                    'id': standard_id,
                                    'name': universal_data.get('name', 'Universal_Multi_Domain_Benchmark'),
                                    'description': universal_data.get('description', '다양한 도메인을 아우르는 범용 임베딩 모델 벤치마크 데이터셋'),
                                    'type': 'universal_benchmark',
                                    'size': len(universal_data.get('examples', [])),
                                    'created_at': universal_data.get('created_at', universal_data.get('updated_at', '2025-09-27T23:04:47.663124')),
                                    'system_managed': True,  # 시스템 관리 데이터셋 표시
                                    'deletable': False  # 삭제 불가능 표시
                                })
                                log(f"범용 평가 데이터셋 로드 성공 ({path_desc}): {len(universal_data.get('examples', []))}개 예제")
                    except Exception as e:
                        log(f"범용 평가 데이터셋 로드 실패 ({path_desc}): {e}", error=True)
                
                # domain_benchmark.json 확인 (일관성 있게 domain_benchmark.json 사용)
                domain_filename = "domain_benchmark.json"
                domain_path = os.path.join(benchmark_dir, domain_filename)
                if os.path.exists(domain_path):
                    try:
                        with open(domain_path, 'r', encoding='utf-8') as f:
                            domain_data = json.load(f)
                            # 중복 방지를 위해 ID 확인 (평가 데이터셋은 파일명 기반 ID 사용)
                            standard_id = 'domain_benchmark'
                            if not any(d.get('id') == standard_id for d in datasets):
                                datasets.append({
                                    'id': standard_id,
                                    'name': domain_data.get('name', 'Domain_ETS_Benchmark'),
                                    'description': domain_data.get('description', 'ETS 도메인 전문성 평가 데이터셋'),
                                    'type': 'domain_benchmark',
                                    'size': len(domain_data.get('examples', [])),
                                    'created_at': domain_data.get('created_at', domain_data.get('updated_at', '2025-09-27T23:04:47.663124')),
                                    'system_managed': True,  # 시스템 관리 데이터셋 표시
                                    'deletable': False  # 삭제 불가능 표시
                                })
                                log(f"도메인 평가 데이터셋 로드 성공 ({path_desc}): {len(domain_data.get('examples', []))}개 예제")
                    except Exception as e:
                        log(f"도메인 평가 데이터셋 로드 실패 ({path_desc}): {e}", error=True)
            except Exception as e:
                log(f"평가 데이터셋 디렉토리 접근 실패 ({path_desc}): {e}", error=True)

        # 중복 ID 제거 (같은 ID가 여러 번 나타날 경우)
        unique_datasets = {}
        for dataset in datasets:
            dataset_id = dataset['id']
            if dataset_id not in unique_datasets:
                unique_datasets[dataset_id] = dataset
            else:
                log(f"WARNING: 중복 데이터셋 ID 발견: {dataset_id}, 첫 번째 항목 유지")

        datasets = list(unique_datasets.values())
        return sorted(datasets, key=lambda x: x['created_at'], reverse=True)
    
    def delete_dataset(self, dataset_id: str) -> tuple[bool, str]:
        """데이터셋 삭제 (시스템 필수 데이터셋 보호)

        Returns:
            tuple[bool, str]: (success, error_message)
        """
        # 시스템 필수 데이터셋 보호 리스트
        protected_datasets = {
            'feedback_live': '피드백 수집 데이터셋',
            'universal_benchmark': '범용 평가 데이터셋',
            'universal_eval_user_uploaded': '범용 평가 데이터셋',
            'domain_benchmark': '도메인 평가 데이터셋',
            'domain_eval_user_uploaded': '도메인 평가 데이터셋'
        }

        # 보호된 데이터셋인지 확인
        if dataset_id in protected_datasets:
            error_msg = f"보호된 시스템 데이터셋는 삭제할 수 없습니다. ({protected_datasets[dataset_id]})"
            log(f"보호된 시스템 데이터셋 삭제 시도 차단: {dataset_id}", error=True)
            return False, error_msg

        dataset_path = self.data_dir / f"{dataset_id}.json"
        if dataset_path.exists():
            dataset_path.unlink()
            log(f"데이터셋 삭제 완료: {dataset_id}")
            return True, ""
        return False, "데이터셋을 찾을 수 없습니다."
    
    def get_dataset_statistics(self, dataset_id: str) -> Dict[str, Any]:
        """데이터셋 통계 정보"""
        dataset = self.get_dataset(dataset_id)
        if not dataset:
            return {}
        
        examples = dataset['examples']
        
        # 안전한 점수 계산 (점수가 없는 기존 데이터셋 처리)
        scored_examples = [ex for ex in examples if 'score' in ex and ex['score'] is not None]
        
        stats = {
            'total_examples': len(examples),
            'positive_examples': sum(1 for ex in scored_examples if ex['score'] > 0.5),
            'negative_examples': sum(1 for ex in scored_examples if ex['score'] <= 0.5),
            'average_score': sum(ex['score'] for ex in scored_examples) / len(scored_examples) if scored_examples else 0.6,  # 기본값 0.6
            'high_quality_examples': sum(1 for ex in scored_examples if ex['score'] > 0.8),
            'unscored_examples': len(examples) - len(scored_examples),  # 점수가 없는 예제 수
            'types': {}
        }
        
        # 타입별 통계
        for example in examples:
            type_name = example.get('type', 'unknown')
            stats['types'][type_name] = stats['types'].get(type_name, 0) + 1
        
        return stats
    
    def create_dataset_from_examples(self, examples: List[Dict[str, Any]], 
                                   name: str, description: str = "") -> str:
        """예제 리스트에서 데이터셋 생성"""
        try:
            dataset_id = str(uuid.uuid4())[:8]  # 짧은 UUID 사용
            
            # 데이터셋 메타데이터 생성
            from datetime import datetime
            dataset_data = {
                'id': dataset_id,
                'name': name,
                'description': description,
                'created_at': datetime.now().isoformat(),
                'type': 'file_upload',
                'size': len(examples),
                'examples': examples
            }
            
            # 파일로 저장
            dataset_path = self.data_dir / f"{dataset_id}.json"
            with open(dataset_path, 'w', encoding='utf-8') as f:
                json.dump(dataset_data, f, ensure_ascii=False, indent=2)
            
            logger.info(f"예제 기반 데이터셋 생성 완료: {dataset_id} ({len(examples)}개 예제)")
            return dataset_id
            
        except Exception as e:
            logger.error(f"예제 기반 데이터셋 생성 실패: {e}")
            raise
    
    def create_empty_dataset(self, name: str, description: str = "") -> str:
        """빈 데이터셋 생성 (수동 입력용)"""
        try:
            dataset_id = str(uuid.uuid4())[:8]  # 짧은 UUID 사용
            
            # 빈 데이터셋 메타데이터 생성
            from datetime import datetime
            dataset_data = {
                'id': dataset_id,
                'name': name,
                'description': description,
                'created_at': datetime.now().isoformat(),
                'type': 'manual_input',
                'size': 0,
                'examples': []
            }
            
            # 파일로 저장
            dataset_path = self.data_dir / f"{dataset_id}.json"
            with open(dataset_path, 'w', encoding='utf-8') as f:
                json.dump(dataset_data, f, ensure_ascii=False, indent=2)
            
            logger.info(f"빈 데이터셋 생성 완료: {dataset_id}")
            return dataset_id
            
        except Exception as e:
            logger.error(f"빈 데이터셋 생성 실패: {e}")
            raise

# 전역 인스턴스 초기화
if FINETUNE_AVAILABLE:
    fine_tuner = EmbeddingFineTuner()
    dataset_manager = TrainingDatasetManager()  # 기본값 None이므로 자동으로 ~/.airun/datasets 사용
else:
    fine_tuner = None
    dataset_manager = None

# 파인튜닝 관련 요청/응답 모델
class FineTuneRequest(BaseModel):
    model_name: str
    dataset_id: str
    epochs: int = 3
    batch_size: int = 16
    learning_rate: float = 2e-5
    force_cpu: bool = False
    # 사용자가 명시적으로 설정한 값들을 추적하는 필드 추가
    # 프론트엔드에서 사용자가 기본값을 변경했을 때 True로 설정
    user_set_epochs: Optional[bool] = None
    user_set_batch_size: Optional[bool] = None
    user_set_learning_rate: Optional[bool] = None

class DatasetCreateRequest(BaseModel):
    name: str
    description: str = ""
    documents: List[str]

class EvaluateModelRequest(BaseModel):
    model_id: str
    dataset_id: Optional[str] = None  # 학습용 데이터셋 (기존 호환성)
    evaluation_dataset_id: Optional[str] = None  # 평가용 데이터셋 (새로 추가)
    evaluation_mode: Optional[str] = "universal"  # "universal" 또는 "domain"

async def get_dataset_info(dataset_id: str):
    """특정 데이터셋의 정보를 반환하는 헬퍼 함수"""
    try:
        if not dataset_manager:
            return {"success": False, "error": "dataset_manager가 초기화되지 않았습니다."}

        # 데이터셋 목록 조회
        datasets = dataset_manager.list_datasets()

        # 해당 dataset_id 찾기
        for dataset in datasets:
            if dataset.get('id') == dataset_id:
                return {"success": True, "data": dataset}

        # 찾지 못한 경우
        return {"success": False, "error": f"데이터셋 {dataset_id}를 찾을 수 없습니다."}
    except Exception as e:
        logger.warning(f"데이터셋 정보 조회 실패 (dataset_id: {dataset_id}): {e}")
        return {"success": False, "error": str(e)}

@app.get("/models/list")
async def list_models():
    """저장된 모델 목록 반환 (기본 모델 포함)"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        # 기본 모델 정보
        base_model_id = "nlpai-lab/KURE-v1"
        
        # 기본 모델의 실제 크기 계산 (GB 단위)
        base_model_size = 2.27  # 기본값
        try:
            # 1. HuggingFace 캐시에서 실제 파일 크기 확인
            hf_cache_path = os.path.expanduser("~/.cache/huggingface/hub/models--nlpai-lab--KURE-v1")
            model_cache_path = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'models--nlpai-lab--KURE-v1')
            
            # 가능한 경로들 확인
            cache_paths = [hf_cache_path, model_cache_path]
            
            for cache_path in cache_paths:
                if os.path.exists(cache_path):
                    total_size = 0
                    
                    # HuggingFace 캐시 구조: blobs 디렉토리에서 모든 파일 크기 합산
                    blobs_path = os.path.join(cache_path, 'blobs')
                    if os.path.exists(blobs_path):
                        for file in os.listdir(blobs_path):
                            file_path = os.path.join(blobs_path, file)
                            if os.path.isfile(file_path):
                                total_size += os.path.getsize(file_path)
                        # logger.info(f"blobs 디렉토리에서 모델 크기 계산: {blobs_path}")
                    else:
                        # blobs 디렉토리가 없는 경우 기존 방식으로 폴백
                        for root, dirs, files in os.walk(cache_path):
                            for file in files:
                                if file.endswith(('.bin', '.safetensors', '.pt', '.pth')):
                                    file_path = os.path.join(root, file)
                                    if os.path.isfile(file_path):
                                        total_size += os.path.getsize(file_path)
                        # logger.info(f"확장자 기반으로 모델 크기 계산: {cache_path}")
                    
                    if total_size > 0:
                        base_model_size = round(total_size / (1024 * 1024 * 1024), 2)  # GB 단위
                        # logger.info(f"기본 모델 실제 크기: {base_model_size:.2f}GB (from {cache_path})")
                        break
            
            # 2. 로드된 모델에서 파라미터 크기 확인 (캐시에서 찾지 못한 경우)
            if base_model_size == 2.27 and hasattr(embedding_service, 'model') and embedding_service.model is not None:
                try:
                    total_params = sum(p.numel() for p in embedding_service.model.parameters())
                    # 대략 4바이트/파라미터로 계산 (float32)
                    calculated_size = round((total_params * 4) / (1024 * 1024 * 1024), 2)  # GB 단위
                    if calculated_size > 0.1:  # 합리적인 크기인 경우만 사용
                        base_model_size = calculated_size
                        # logger.info(f"기본 모델 계산된 크기: {base_model_size:.2f}GB (from parameters)")
                except Exception as param_error:
                    logger.debug(f"파라미터 기반 크기 계산 실패: {param_error}")
                    
        except Exception as e:
            logger.error(f"기본 모델 크기 계산 실패: {e}")
            logger.debug(f"기본값 {base_model_size}GB 사용")
        
        # 기본 모델의 평가 결과 로드
        base_eval_metrics = {}
        if model_evaluator:
            base_eval_metrics = model_evaluator.load_evaluation_results(base_model_id)
        
        models = [{
            'id': base_model_id,
            'name': "KURE-v1 (기본 모델)",
            'base_model': base_model_id,
            'created_at': '',
            'training_samples': 0,
            'best_score': base_eval_metrics.get('f1_score', 0.0),  # 실제 평가 결과 또는 0
            'precision': base_eval_metrics.get('precision', 0.0),  # 정밀도
            'recall': base_eval_metrics.get('recall', 0.0),        # 재현율
            'f1_score': base_eval_metrics.get('f1_score', 0.0),    # F1 스코어
            'size_gb': base_model_size,  # MB에서 GB로 변경
            'path': base_model_id,
            'type': 'base',
            'evaluated': bool(base_eval_metrics),  # 평가 여부
            'evaluation_date': base_eval_metrics.get('evaluation_date'),  # 평가 날짜
            'evaluation_results': base_eval_metrics if base_eval_metrics else None,  # 평가 결과 전체
            'test_samples': base_eval_metrics.get('test_samples'),  # 테스트 샘플 수
            'evaluation_time': base_eval_metrics.get('evaluation_time')  # 평가 시간
        }]
        
        # 파인튜닝된 모델들 추가
        fine_tuner_instance = EmbeddingFineTuner(output_dir=None, use_existing_model=True)
        finetuned_models = fine_tuner_instance.list_models()
        
        # 파인튜닝된 모델에 type 필드 추가 및 크기 단위 변환
        for model in finetuned_models:
            model['type'] = 'finetuned'
            if 'size_mb' in model:
                model['size_gb'] = round(model['size_mb'] / 1024, 2)  # MB를 GB로 변환
                del model['size_mb']  # 기존 MB 필드 제거

            # 데이터셋 정보 추가
            try:
                dataset_id = model.get('dataset_id')
                if dataset_id:
                    # 데이터셋 정보 조회
                    dataset_info = await get_dataset_info(dataset_id)
                    if dataset_info and dataset_info.get('success'):
                        dataset_data = dataset_info.get('data', {})
                        model['dataset_info'] = {
                            'dataset_id': dataset_id,
                            'document_count': dataset_data.get('documentCount', 0),
                            'pair_count': dataset_data.get('pairCount', dataset_data.get('size', 0)),
                            'domains': dataset_data.get('domains', []),
                            'description': dataset_data.get('description', '도메인 특화')
                        }
                    else:
                        # 폴백: 기본 정보 사용
                        model['dataset_info'] = {
                            'dataset_id': dataset_id,
                            'document_count': 0,
                            'pair_count': model.get('training_samples', 0),
                            'domains': ['도메인 특화'],
                            'description': '데이터셋 정보를 찾을 수 없음'
                        }
            except Exception as dataset_error:
                logger.warning(f"모델 {model.get('id')}의 데이터셋 정보 조회 실패: {dataset_error}")
                # 오류 시 기본 정보로 폴백
                model['dataset_info'] = {
                    'dataset_id': model.get('dataset_id', 'unknown'),
                    'document_count': 0,
                    'pair_count': model.get('training_samples', 0),
                    'domains': ['도메인 특화'],
                    'description': '데이터셋 정보 조회 실패'
                }
        
        # 파인튜닝된 모델들에 평가 결과 추가
        for model in finetuned_models:
            if model_evaluator:
                try:
                    eval_data = model_evaluator.load_evaluation_results(model['id'])
                    if eval_data:
                        model['evaluation_results'] = eval_data
                        # 기존 필드들도 업데이트 (호환성 유지)
                        model['f1_score'] = eval_data.get('f1_score', model.get('f1_score', 0.0))
                        model['precision'] = eval_data.get('precision', model.get('precision', 0.0))
                        model['recall'] = eval_data.get('recall', model.get('recall', 0.0))
                        model['test_samples'] = eval_data.get('test_samples')
                        model['evaluation_time'] = eval_data.get('evaluation_time')
                        model['evaluated'] = True
                    else:
                        model['evaluation_results'] = None
                        model['test_samples'] = None
                        model['evaluation_time'] = None
                        model['evaluated'] = False
                except Exception as e:
                    logger.warning(f"모델 {model['id']}의 평가 결과 로드 실패: {e}")
                    model['evaluation_results'] = None
                    model['test_samples'] = None
                    model['evaluation_time'] = None
                    model['evaluated'] = False

        models.extend(finetuned_models)

        return {"success": True, "models": models}
    except Exception as e:
        logger.error(f"모델 목록 조회 실패: {str(e)}")
        return {"success": False, "error": str(e)}

async def run_safe_background_finetuning(request: FineTuneRequest):
    """안전성 모니터링이 포함된 백그라운드 파인튜닝 함수"""
    try:
        logger.debug("안전한 백그라운드 파인튜닝 시작")
        
        # EmbeddingFineTuner 인스턴스 생성
        base_fine_tuner = EmbeddingFineTuner(output_dir=None, use_existing_model=True)
        
        # 안전한 파인튜닝 래퍼 생성
        safe_fine_tuner = SafeFineTuner(base_fine_tuner)
        
        # 전역 embedding_service 가져오기
        global embedding_service
        
        # 안전성 모니터링과 함께 파인튜닝 실행
        result = await safe_fine_tuner.safe_fine_tune_model(
            model_name=request.model_name,
            dataset_id=request.dataset_id,
            num_epochs=request.epochs,
            batch_size=request.batch_size,
            learning_rate=request.learning_rate,
            force_cpu=request.force_cpu,
            embedding_service=embedding_service,
            user_set_epochs=request.user_set_epochs or False,
            user_set_batch_size=request.user_set_batch_size or False,
            user_set_learning_rate=request.user_set_learning_rate or False
        )
        
        logger.debug(f"안전한 백그라운드 파인튜닝 결과: {result}")
        
        if result.get('status') == 'completed':
            logger.info(f"✅ 안전한 파인튜닝 완료: {request.model_name} (데이터셋: {request.dataset_id})")
        else:
            logger.error(f"❌ 안전한 파인튜닝 실패: {result.get('error', 'Unknown error')}")
            
    except Exception as e:
        logger.error(f"안전한 백그라운드 파인튜닝 오류: {str(e)}")
        logger.error(f"오류 타입: {type(e).__name__}")
        import traceback
        logger.error(f"스택 트레이스: {traceback.format_exc()}")
        
        # 실패 상태 저장
        finetune_state_manager.save_state({
            'status': 'failed',
            'model_name': request.model_name,
            'dataset_id': request.dataset_id,
            'failed_at': time.time(),
            'last_error': str(e),
            'error_type': type(e).__name__,
            'safety_failure': True
        })

async def run_background_finetuning(request: FineTuneRequest):
    """백그라운드에서 파인튜닝을 실행하는 함수 (기존 버전)"""
    try:
        logger.debug("백그라운드 파인튜닝 시작")
        
        # EmbeddingFineTuner 인스턴스 생성
        fine_tuner = EmbeddingFineTuner(output_dir=None, use_existing_model=True)
        
        # 전역 embedding_service 가져오기
        global embedding_service
        
        # 파인튜닝 실행
        result = await fine_tuner.fine_tune_model(
            model_name=request.model_name,
            dataset_id=request.dataset_id,
            num_epochs=request.epochs,
            batch_size=request.batch_size,
            learning_rate=request.learning_rate,
            force_cpu=request.force_cpu,
            embedding_service=embedding_service,
            user_set_epochs=request.user_set_epochs or False,
            user_set_batch_size=request.user_set_batch_size or False,
            user_set_learning_rate=request.user_set_learning_rate or False
        )
        
        logger.debug(f"백그라운드 파인튜닝 결과: {result}")
        
        if result.get('status') == 'completed':
            logger.info(f"✅ 파인튜닝 완료: {request.model_name} (데이터셋: {request.dataset_id})")
        else:
            logger.error(f"❌ 파인튜닝 실패: {result.get('error', 'Unknown error')}")
            
    except Exception as e:
        logger.error(f"백그라운드 파인튜닝 오류: {str(e)}")
        logger.error(f"오류 타입: {type(e).__name__}")
        import traceback
        logger.error(f"스택 트레이스: {traceback.format_exc()}")
        
        # 실패 상태 저장
        finetune_state_manager.save_state({
            'status': 'failed',
            'model_name': request.model_name,
            'dataset_id': request.dataset_id,
            'failed_at': time.time(),
            'last_error': str(e),
            'error_type': type(e).__name__
        })

@app.post("/models/finetune")
async def start_finetuning(request: FineTuneRequest):
    """파인튜닝 시작 (안전성 모니터링 포함)"""
    if not FINETUNE_AVAILABLE:
        logger.error("파인튜닝 기능을 사용할 수 없습니다.")
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        logger.debug("파인튜닝 API 호출 시작")
        logger.debug(f"요청 파라미터: {request.dict()}")
        
        # 현재 파인튜닝이 진행 중인지 확인
        current_state = finetune_state_manager.load_state()
        if current_state and current_state.get('status') in ['starting', 'training', 'retrying_cpu']:
            return {
                "success": False,
                "error": "이미 파인튜닝이 진행 중입니다. 현재 작업이 완료된 후 다시 시도해주세요.",
                "current_status": current_state.get('status'),
                "current_model": current_state.get('model_name', 'Unknown')
            }
        
        # 사전 안전성 검사
        logger.debug("시스템 리소스 안전성 검사 시작")
        resources = SystemResourceAnalyzer.analyze_system_resources()
        safety_check = SystemResourceAnalyzer._check_resource_safety(resources)
        
        if not safety_check['is_safe']:
            logger.error(f"시스템 리소스 안전성 검사 실패: {safety_check['warning']}")
            return {
                "success": False,
                "error": f"시스템 리소스 부족으로 파인튜닝을 시작할 수 없습니다: {safety_check['warning']}",
                "warnings": safety_check['warnings'],
                "resource_info": {
                    "available_memory_gb": resources.available_memory_gb,
                    "gpu_available": resources.gpu_available,
                    "gpu_memory_gb": resources.gpu_available_memory_gb if resources.gpu_available else 0
                },
                "recommendations": [
                    "다른 애플리케이션을 종료하여 메모리를 확보하세요",
                    "CPU 강제 모드를 사용하세요 (force_cpu: true)",
                    "배치 크기를 줄이세요 (batch_size를 1-2로 설정)"
                ]
            }
        
        logger.debug(f"파인튜닝 시작 - 모델: {request.model_name}, 데이터셋: {request.dataset_id}")
        logger.debug(f"설정 - 에포크: {request.epochs}, 배치 크기: {request.batch_size}, 학습률: {request.learning_rate}, CPU 강제: {request.force_cpu}")
        
        # 백그라운드에서 안전한 파인튜닝 실행 (별도 스레드로 완전히 분리)
        def run_training_in_background():
            """완전히 별도 스레드에서 파인튜닝 실행"""
            try:
                # 새로운 이벤트 루프 생성
                loop = asyncio.new_event_loop()
                asyncio.set_event_loop(loop)
                
                # 백그라운드 파인튜닝 실행
                loop.run_until_complete(run_safe_background_finetuning(request))
                
            except Exception as e:
                logger.error(f"백그라운드 훈련 스레드 오류: {str(e)}")
                # 실패 상태 저장
                finetune_state_manager.save_state({
                    'status': 'failed',
                    'model_name': request.model_name,
                    'dataset_id': request.dataset_id,
                    'failed_at': time.time(),
                    'last_error': str(e),
                    'error_type': type(e).__name__,
                    'thread_failure': True
                })
            finally:
                try:
                    loop.close()
                except:
                    pass
        
        # 데몬 스레드로 시작하여 메인 프로세스와 분리
        logger.debug("백그라운드 안전한 파인튜닝 스레드 시작")
        training_thread = threading.Thread(target=run_training_in_background, daemon=True)
        training_thread.start()
        
        logger.info(f"🚀 안전한 파인튜닝 백그라운드 스레드 시작: {request.model_name} (데이터셋: {request.dataset_id})")
        
        return {
            "success": True,
            "message": "파인튜닝이 안전성 모니터링과 함께 백그라운드에서 시작되었습니다. 진행 상황은 /models/finetune/progress 엔드포인트에서 확인할 수 있습니다.",
            "model_name": request.model_name,
            "dataset_id": request.dataset_id,
            "epochs": request.epochs,
            "batch_size": request.batch_size,
            "learning_rate": request.learning_rate,
            "force_cpu": request.force_cpu,
            "safety_features": {
                "memory_monitoring": True,
                "auto_fallback": True,
                "emergency_cleanup": True,
                "resource_threshold_check": True
            },
            "monitoring": {
                "progress_endpoint": "/models/finetune/progress",
                "status_endpoint": "/models/finetune/status"
            }
        }
        
    except ImportError as import_error:
        logger.error(f"파인튜닝 모듈 임포트 실패: {str(import_error)}")
        return {"success": False, "error": f"파인튜닝 모듈 임포트 실패: {str(import_error)}"}
    except Exception as e:
        logger.error(f"파인튜닝 API 오류: {str(e)}")
        logger.error(f"오류 타입: {type(e).__name__}")
        import traceback
        logger.error(f"스택 트레이스: {traceback.format_exc()}")
        return {"success": False, "error": str(e), "error_type": type(e).__name__}

@app.get("/models/finetune/progress")
async def get_finetuning_progress():
    """파인튜닝 진행 상황 조회 (빠른 응답 최적화)"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        # 상태 파일에서 직접 읽어서 빠른 응답 제공
        state = finetune_state_manager.load_state()
        
        if not state:
            return {
                "success": True,
                "progress": {
                    'is_training': False,
                    'progress': 0,
                    'current_epoch': 0,
                    'total_epochs': 0,
                    'current_loss': 0.0,
                    'best_score': 0.0,
                    'eta': 'idle',
                    'status': 'idle'
                }
            }
        
        # 상태에서 진행률 정보 빠르게 추출
        status = state.get('status', 'idle')
        is_training = status in ['starting', 'training', 'retrying_cpu']
        
        progress_data = {
            'is_training': is_training,
            'progress': state.get('progress', 0),
            'current_epoch': state.get('current_epoch', 0),
            'total_epochs': state.get('num_epochs', 0),
            'current_loss': state.get('current_loss', 0.0),
            'best_score': state.get('best_score', 0.0),
            'eta': state.get('eta', status),
            'status': status,
            'model_name': state.get('model_name', ''),
            'dataset_id': state.get('dataset_id', ''),
            'current_step': state.get('current_step', 0),
            'total_steps': state.get('total_steps', 0)
        }
        
        return {"success": True, "progress": progress_data}
        
    except Exception as e:
        logger.error(f"파인튜닝 진행 상황 조회 실패: {str(e)}")
        return {"success": False, "error": str(e)}

@app.get("/models/finetune/status")
async def get_finetuning_status():
    """파인튜닝 상태를 반환합니다 (재시작 후에도 유지됨)."""
    try:
        state = finetune_state_manager.load_state()
        
        if not state:
            return {
                "success": True,
                "status": "idle",
                "message": "파인튜닝 기록이 없습니다."
            }
        
        # 상태에 따른 메시지 생성
        status = state.get('status', 'unknown')
        
        if status == 'starting':
            message = f"파인튜닝을 시작하고 있습니다... (모델: {state.get('model_name', 'Unknown')})"
        elif status == 'retrying_cpu':
            message = f"GPU 메모리 부족으로 CPU 모드로 재시도 중입니다..."
        elif status == 'completed':
            message = f"파인튜닝이 완료되었습니다! (모델: {state.get('model_name', 'Unknown')})"
        elif status == 'failed':
            failure_reason = finetune_state_manager.get_failure_reason()
            message = f"파인튜닝이 실패했습니다: {failure_reason}"
        else:
            message = f"알 수 없는 상태: {status}"
        
        return {
            "success": True,
            "status": status,
            "message": message,
            "state": state,
            "is_resource_failure": finetune_state_manager.is_resource_failure(),
            "failure_reason": finetune_state_manager.get_failure_reason() if status == 'failed' else None
        }
        
    except Exception as e:
        return {
            "success": False,
            "error": str(e)
        }

@app.post("/models/finetune/reset")
async def reset_finetuning_state():
    """파인튜닝 상태를 초기화합니다."""
    try:
        # 상태 파일 삭제
        finetune_state_manager.clear_state()
        
        # 파인튜너 상태 초기화
        if FINETUNE_AVAILABLE and fine_tuner:
            fine_tuner.training_progress = {
                'is_training': False,
                'progress': 0,
                'current_epoch': 0,
                'total_epochs': 0,
                'current_loss': 0.0,
                'best_score': 0.0,
                'eta': 'idle',
                'status': 'idle'
            }
        
        return {
            "success": True,
            "message": "파인튜닝 상태가 초기화되었습니다.",
            "status": "idle"
        }
        
    except Exception as e:
        return {
            "success": False,
            "error": str(e)
        }

@app.delete("/models/{model_id}")
async def delete_model(model_id: str):
    """모델 삭제"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        success = fine_tuner.delete_model(model_id)
        if success:
            return {"success": True, "message": f"모델 {model_id}가 삭제되었습니다."}
        else:
            return {"success": False, "error": "모델을 찾을 수 없습니다."}
    except Exception as e:
        logger.error(f"모델 삭제 실패: {str(e)}")
        return {"success": False, "error": str(e)}

@app.get("/datasets/list")
async def list_datasets():
    """데이터셋 목록 반환"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        datasets = dataset_manager.list_datasets()
        return {"success": True, "datasets": datasets}
    except Exception as e:
        logger.error(f"데이터셋 목록 조회 실패: {str(e)}")
        return {"success": False, "error": str(e)}

@app.post("/datasets/create")
async def create_dataset(request: DatasetCreateRequest, use_ai: bool = Query(False, description="AI 기반 QA 생성 사용 여부 (기본값: False)")):
    """새 데이터셋 생성"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        if use_ai:
            # AI 기반 QA 생성 (선택적)
            dataset_id = await dataset_manager.create_dataset_from_documents_with_ai(
                documents=request.documents,
                name=request.name,
                description=request.description
            )
            return {
                "success": True, 
                "dataset_id": dataset_id,
                "generation_method": "ai_qa_with_negatives",
                "message": "AI 기반 QA 데이터셋이 생성되었습니다."
            }
        else:
            # 기존 방식 (기본값 - 빠르고 효율적)
            dataset_id = dataset_manager.create_dataset_from_documents(
                documents=request.documents,
                name=request.name,
                description=request.description
            )
            return {
                "success": True, 
                "dataset_id": dataset_id,
                "generation_method": "document_similarity",
                "message": "효율적인 기본 방식으로 데이터셋이 생성되었습니다."
            }
    except Exception as e:
        logger.error(f"데이터셋 생성 실패: {str(e)}")
        return {"success": False, "error": str(e)}

@app.delete("/datasets/{dataset_id}")
async def delete_dataset(dataset_id: str):
    """데이터셋 삭제"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        success, error_message = dataset_manager.delete_dataset(dataset_id)
        if success:
            return {"success": True, "message": f"데이터셋 {dataset_id}가 삭제되었습니다."}
        else:
            return {"success": False, "error": error_message}
    except Exception as e:
        logger.error(f"데이터셋 삭제 실패: {str(e)}")
        return {"success": False, "error": str(e)}

@app.post("/models/{model_id:path}/activate")
async def activate_model(model_id: str):
    """모델 활성화 (임베딩 서비스에서 사용할 모델 변경)"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        # 기본 모델인지 확인
        if model_id == "nlpai-lab/KURE-v1":
            # 기본 모델 활성화
            model_name = model_id
            model_path_str = model_id  # 기본 모델은 모델명 그대로 사용
        else:
            # 파인튜닝된 모델인지 확인
            finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
            model_path = Path(finetuned_dir) / model_id
            
            if not model_path.exists():
                return {"success": False, "error": "모델을 찾을 수 없습니다."}
            
            # 메타데이터에서 모델명 확인
            metadata_path = model_path / "metadata.json"
            model_name = model_id  # 기본값
            model_path_str = str(model_path)
            
            if metadata_path.exists():
                try:
                    with open(metadata_path, 'r', encoding='utf-8') as f:
                        metadata = json.load(f)
                    model_name = metadata.get('model_name', model_id)
                except:
                    pass
        
        # 환경변수를 통한 모델 활성화 (새로운 방식)
        try:
            # 새로운 config API 사용
            async with httpx.AsyncClient() as client:
                response = await client.post(
                    "http://localhost:5600/config/set-active-model",
                    json={"model_name": model_path_str}
                )
                
                if response.status_code == 200:
                    result = response.json()
                    if result.get('success'):
                        logger.info(f"활성 모델을 {model_name}로 변경했습니다.")
                        return {
                            "success": True, 
                            "message": f"모델 {model_name}가 활성화되었습니다.",
                            "model_path": str(model_path),
                            "note": "변경사항을 완전히 적용하려면 RAG 서비스를 재시작하세요"
                        }
                    else:
                        return {"success": False, "error": result.get('error', '모델 활성화 실패')}
                else:
                    return {"success": False, "error": f"설정 API 호출 실패: {response.status_code}"}
                    
        except Exception as config_error:
                                                    # fallback: 직접 환경변수 변경
            logger.info(f"Config API 호출 실패, 직접 환경변수 변경: {str(config_error)}")
            
            env_file_path = os.path.expanduser('~/.airun/.env')
            env_vars = {}
            
            if os.path.exists(env_file_path):
                with open(env_file_path, 'r', encoding='utf-8') as f:
                    for line in f:
                        line = line.strip()
                        if line and not line.startswith('#') and '=' in line:
                            key, value = line.split('=', 1)
                            env_vars[key.strip()] = value.strip().strip('"\'')
            
            env_vars['RAG_EMBEDDING_MODEL'] = model_path_str
            os.makedirs(os.path.dirname(env_file_path), exist_ok=True)
            
            with open(env_file_path, 'w', encoding='utf-8') as f:
                for key, value in env_vars.items():
                    f.write(f'{key}={value}\n')
            
            os.environ['RAG_EMBEDDING_MODEL'] = model_path_str
    
            logger.info(f"활성 모델을 {model_name}로 변경했습니다.")
            return {
                "success": True, 
                "message": f"모델 {model_name}가 활성화되었습니다.",
                "model_path": str(model_path),
                "note": "변경사항을 완전히 적용하려면 RAG 서비스를 재시작하세요"
            }
            
    except Exception as e:
        logger.error(f"모델 활성화 실패: {str(e)}")
        return {"success": False, "error": str(e)}

@app.post("/datasets/from-rag")
async def create_dataset_from_rag(request: Request):
    """현재 RAG 문서들로부터 데이터셋 생성 (효율적인 기존 방식)"""
    if not FINETUNE_AVAILABLE:
        return {"success": False, "error": "파인튜닝 기능을 사용할 수 없습니다."}
    
    try:
        # 요청 본문에서 프로바이더 정보 추출 (향후 사용을 위해)
        body = await request.json() if request.headers.get("content-type") == "application/json" else {}
        provider = body.get('provider', 'ollama')
        model = body.get('model', 'hamonize:latest')
        logger.info(f"RAG 데이터셋 생성 - 프로바이더: {provider}, 모델: {model}")
        
        # 현재 PostgreSQL에서 모든 문서 가져오기
        # 전역 embedding_service가 없으면 새로 생성
        current_embedding_service = embedding_service
        if not current_embedding_service:
            try:
                current_embedding_service = EmbeddingService()
                logger.info("RAG 데이터셋 생성을 위해 임시 embedding_service 생성")
            except Exception as e:
                logger.error(f"EmbeddingService 생성 실패: {str(e)}")
                return {"success": False, "error": "RAG 시스템이 초기화되지 않았습니다."}
        
        if not current_embedding_service.collection:
            return {"success": False, "error": "RAG 시스템이 초기화되지 않았습니다."}
        
        # 모든 문서 조회
        result = current_embedding_service.collection.get()
        
        if not result or not result.get('documents'):
            return {"success": False, "error": "RAG 시스템에 문서가 없습니다."}
        
        documents = result['documents']
        dataset_name = f"RAG_Documents_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
        
        # 기존 방식 사용 (빠르고 효율적)
        dataset_id = dataset_manager.create_dataset_from_documents(
            documents=documents,
            name=dataset_name,
            description="RAG 시스템의 기존 문서들로부터 자동 생성된 데이터셋 (문서 유사도 기반)"
        )
        
        return {
            "success": True, 
            "dataset_id": dataset_id,
            "document_count": len(documents),
            "generation_method": "document_similarity",
            "message": "RAG 문서들로부터 데이터셋이 빠르게 생성되었습니다."
        }
        
    except Exception as e:
        logger.error(f"RAG 데이터셋 생성 실패: {str(e)}")
        return {"success": False, "error": str(e)}

@app.get("/rag/progress")
async def get_rag_progress():
    """현재 RAG 처리의 진행 상태를 반환합니다. (문서별 상세 상태)"""
    try:
        # rag_process.py의 get_processing_status() 호출
        status = get_processing_status()
        return status
    except Exception as e:
        logger.error(f"진행 상태 확인 중 오류 발생: {str(e)}")
        # fallback: 기존 Redis 기반 간략 상태
        try:
            if not embedding_service or not embedding_service.redis_client:
                return {
                    "status": "error",
                    "message": "Redis 연결이 없습니다.",
                    "timestamp": datetime.now().isoformat()
                }
            progress_key = "rag:init:progress"
            progress = embedding_service.redis_client.hgetall(progress_key)
            if not progress:
                return {
                    "status": "idle",
                    "message": "진행 중인 처리가 없습니다.",
                    "timestamp": datetime.now().isoformat()
                }
            total_docs = len(progress)
            completed = sum(1 for v in progress.values() if v.decode() == "completed")
            failed = sum(1 for v in progress.values() if v.decode() == "failed")
            processing = sum(1 for v in progress.values() if v.decode() == "processing")
            failed_docs = []
            for doc, status in progress.items():
                if status.decode() == "failed":
                    try:
                        error_info = embedding_service.redis_client.hget(f"{progress_key}:errors", doc)
                        if error_info:
                            failed_docs.append({
                                "path": doc.decode(),
                                "error": error_info.decode()
                            })
                    except:
                        failed_docs.append({"path": doc.decode(), "error": "Unknown error"})
            return {
                "status": "processing" if processing > 0 else "completed",
                "total_documents": total_docs,
                "completed": completed,
                "failed": failed,
                "processing": processing,
                "progress_percentage": round((completed / total_docs * 100) if total_docs > 0 else 0, 2),
                "failed_documents": failed_docs,
                "timestamp": datetime.now().isoformat()
            }
        except Exception as e2:
            return {
                "status": "error",
                "error": f"진행 상태 확인 중 오류: {str(e)} / fallback: {str(e2)}",
                "timestamp": datetime.now().isoformat()
            }

@app.post("/rag/resume")
async def resume_rag_processing():
    """중단된 RAG 처리를 재개합니다."""
    try:
        if not embedding_service or not embedding_service.redis_client:
            return {
                "status": "error",
                "message": "Redis 연결이 없습니다.",
                "timestamp": datetime.now().isoformat()
            }

        progress_key = "rag:init:progress"
        progress = embedding_service.redis_client.hgetall(progress_key)
        
        if not progress:
            return {
                "status": "error",
                "message": "재개할 처리가 없습니다.",
                "timestamp": datetime.now().isoformat()
            }

        # 실패한 문서만 재처리
        failed_docs = []
        for doc, status in progress.items():
            if status.decode() == "failed":
                doc_path = doc.decode()
                failed_docs.append(doc_path)
                # 상태를 processing으로 변경
                embedding_service.redis_client.hset(progress_key, doc_path, "processing")

        if not failed_docs:
            return {
                "status": "success",
                "message": "재처리할 실패 문서가 없습니다.",
                "timestamp": datetime.now().isoformat()
            }

        # 실패한 문서 재처리
        processed_count = 0
        error_count = 0
        error_messages = []

        for doc_path in failed_docs:
            try:
                logger.info(f"문서 재처리 중: {doc_path}")
                result = document_processor.process_document(doc_path)
                
                if result["status"] == "success":
                    embedding_service.redis_client.hset(progress_key, doc_path, "completed")
                    processed_count += 1
                    logger.info(f"문서 재처리 성공: {doc_path}")
                else:
                    embedding_service.redis_client.hset(progress_key, doc_path, "failed")
                    error_count += 1
                    error_messages.append(f"{doc_path}: {result.get('message', 'Unknown error')}")
                    logger.error(f"문서 재처리 실패: {doc_path} - {result.get('message')}")

            except Exception as e:
                embedding_service.redis_client.hset(progress_key, doc_path, "failed")
                error_count += 1
                error_messages.append(f"{doc_path}: {str(e)}")
                logger.error(f"문서 재처리 중 예외 발생: {doc_path} - {str(e)}")

        return {
            "status": "success",
            "message": f"문서 재처리 완료 (성공: {processed_count}, 실패: {error_count})",
            "processed_count": processed_count,
            "error_count": error_count,
            "error_messages": error_messages if error_messages else None,
            "timestamp": datetime.now().isoformat()
        }

    except Exception as e:
        logger.error(f"처리 재개 중 오류 발생: {str(e)}")
        return {
            "status": "error",
            "error": str(e),
            "timestamp": datetime.now().isoformat()
        }

@app.post("/rag/failed")
async def save_failed_state(doc_path: str, error: str):
    """실패한 문서의 처리 상태를 저장합니다."""
    try:
        if not embedding_service or not embedding_service.redis_client:
            return {
                "status": "error",
                "message": "Redis 연결이 없습니다.",
                "timestamp": datetime.now().isoformat()
            }

        progress_key = "rag:init:progress"
        error_key = f"{progress_key}:errors"

        # 실패 상태 저장
        embedding_service.redis_client.hset(progress_key, doc_path, "failed")
        embedding_service.redis_client.hset(error_key, doc_path, error)

        return {
            "status": "success",
            "message": "실패 상태가 저장되었습니다.",
            "document": doc_path,
            "error": error,
            "timestamp": datetime.now().isoformat()
        }

    except Exception as e:
        logger.error(f"실패 상태 저장 중 오류 발생: {str(e)}")
        return {
            "status": "error",
            "error": str(e),
            "timestamp": datetime.now().isoformat()
        }

async def validate_queue_state():
    """큐 상태의 일관성을 검증하고 복구하는 함수"""
    try:
        redis_client = get_redis_client()
        
        # processing 큐의 문서들이 실제로 존재하는지 확인
        processing_docs = redis_client.lrange(QUEUE_KEYS['processing'], 0, -1)
        for doc in processing_docs:
            doc_path = decode_redis_value(doc)
            if not os.path.exists(doc_path):
                # 파일이 존재하지 않으면 failed 큐로 이동
                if redis_client.lrem(QUEUE_KEYS['processing'], 1, doc) > 0:
                    redis_client.lpush(QUEUE_KEYS['failed'], doc)
                redis_client.hset(f"{QUEUE_KEYS['failed']}:errors", doc, b"File not found")

        # completed와 failed 큐의 문서들이 실제로 처리되었는지 확인
        for queue_key in [QUEUE_KEYS['completed'], QUEUE_KEYS['failed']]:
            docs = redis_client.lrange(queue_key, 0, -1)
            for doc in docs:
                doc_path = decode_redis_value(doc)
                if not os.path.exists(doc_path):
                    redis_client.lrem(queue_key, 1, doc)
        
        # rag:status:* 키들 정리 - 실제 파일이 존재하지 않는 문서의 상태 정보 제거
        try:
            status_keys = redis_client.keys("rag:status:*")
            for key in status_keys:
                try:
                    status_data = redis_client.get(key)
                    if status_data:
                        doc_status = json.loads(status_data)
                        filename = doc_status.get('filename', '')
                        
                        # 파일명이 있는 경우 실제 파일 존재 여부 확인
                        if filename:
                            # RAG 문서 경로에서 파일 찾기 (사용자별 디렉토리 포함)
                            rag_docs_path = get_rag_settings()['rag_docs_path']
                            possible_paths = [
                                os.path.join(rag_docs_path, filename),
                                os.path.join(rag_docs_path, 'shared', filename),
                                os.path.join(rag_docs_path, 'personal', filename)
                            ]
                            
                            # 사용자별 디렉토리도 확인
                            if os.path.exists(rag_docs_path):
                                for user_dir in os.listdir(rag_docs_path):
                                    user_path = os.path.join(rag_docs_path, user_dir)
                                    if os.path.isdir(user_path):
                                        possible_paths.append(os.path.join(user_path, filename))
                            
                            file_exists = any(os.path.exists(path) for path in possible_paths)
                            
                            # 파일이 존재하지 않으면 상태 정보 삭제
                            if not file_exists:
                                redis_client.delete(key)
                                logger.info(f"존재하지 않는 파일의 상태 정보 정리: {filename}")
                                
                except (json.JSONDecodeError, KeyError) as e:
                    # 잘못된 형식의 상태 정보는 삭제
                    redis_client.delete(key)
                    logger.info(f"잘못된 형식의 상태 정보 정리: {key}")
                    
        except Exception as e:
            logger.error(f"rag:status 키 정리 중 오류 발생: {str(e)}")
                    
    except Exception as e:
        logger.error(f"큐 상태 검증 중 오류 발생: {str(e)}")

@app.post("/rag/cleanup-status")
async def cleanup_rag_status():
    """RAG 상태 정보를 정리합니다 (존재하지 않는 파일의 상태 정보 제거)."""
    try:
        redis_client = get_redis_client()
        if not redis_client:
            return {
                "status": "error",
                "message": "Redis 연결이 없습니다.",
                "timestamp": datetime.now().isoformat()
            }
        
        # 큐 상태 검증 및 정리 실행
        await validate_queue_state()
        
        return {
            "status": "success",
            "message": "RAG 상태 정보 정리가 완료되었습니다.",
            "timestamp": datetime.now().isoformat()
        }
            
    except Exception as e:
        logger.error(f"RAG 상태 정보 정리 중 오류 발생: {str(e)}")
        return {
            "status": "error",
            "error": str(e),
            "timestamp": datetime.now().isoformat()
        }

async def validate_and_reset_training_state():
    """서버 시작 시 파인튜닝 상태를 검증하고 필요시 초기화합니다."""
    try:
        # logger.info("🔍 파인튜닝 상태 검증 시작")
        
        # 파인튜닝 기능이 사용 가능한지 확인
        if not FINETUNE_AVAILABLE:
            # logger.info("파인튜닝 기능이 비활성화되어 있어 상태 검증을 건너뜁니다.")
            return
            
        # 상태 파일 로드
        state = finetune_state_manager.load_state()
        
        if not state:
            # logger.info("파인튜닝 상태 파일이 없습니다. 정상 상태입니다.")
            return
            
        status = state.get('status', 'unknown')
        model_name = state.get('model_name', 'Unknown')
        
        # 학습 중 상태인 경우 검증
        if status in ['starting', 'training', 'retrying_cpu']:
            # logger.info(f"⚠️  이전 학습 상태 발견: {status} (모델: {model_name})")
            
            # 실제 학습 프로세스가 실행 중인지 확인
            is_actually_training = await check_if_training_process_running()
            
            if not is_actually_training:
                # logger.info("실제 학습 프로세스가 실행 중이지 않습니다. 상태를 초기화합니다.")
                
                # 상태 초기화
                finetune_state_manager.clear_state()
                
                # 파인튜너 상태 초기화
                if fine_tuner:
                    fine_tuner.training_progress = {
                        'is_training': False,
                        'progress': 0,
                        'current_epoch': 0,
                        'total_epochs': 0,
                        'current_loss': 0.0,
                        'best_score': 0.0,
                        'eta': 'idle',
                        'status': 'idle'
                    }
                
                # logger.info("✅ 파인튜닝 상태가 초기화되었습니다.")
            else:
                # logger.info("실제 학습 프로세스가 실행 중입니다. 상태를 유지합니다.")
                pass
                
        elif status == 'failed':
            # logger.info(f"이전 학습 실패 상태 발견: {model_name}")
            # logger.info("실패 상태는 유지하되, 새로운 학습을 위해 준비 상태로 전환합니다.")
            pass
            
        elif status == 'completed':
            # logger.info(f"이전 학습 완료 상태 발견: {model_name}")
            # logger.info("완료 상태는 유지합니다.")
            pass
            
        else:
            # logger.info(f"알 수 없는 상태 발견: {status}. 상태를 초기화합니다.")
            finetune_state_manager.clear_state()
            
    except Exception as e:
        logger.error(f"파인튜닝 상태 검증 중 오류: {str(e)}")
        # 오류 발생 시 안전하게 상태 초기화
        try:
            finetune_state_manager.clear_state()
            # logger.info("오류로 인해 파인튜닝 상태를 초기화했습니다.")
        except:
            pass

async def check_if_training_process_running():
    """실제 학습 프로세스가 실행 중인지 확인합니다."""
    try:
        # 현재 프로세스의 자식 프로세스들을 확인
        current_process = psutil.Process(os.getpid())
        children = current_process.children(recursive=True)
        
        for child in children:
            try:
                # 프로세스 명령줄에서 파인튜닝 관련 키워드 확인
                cmdline = ' '.join(child.cmdline())
                if any(keyword in cmdline.lower() for keyword in ['finetune', 'fine_tune', 'sentence-transformers', 'torch']):
                    # CPU 사용률이 높은지 확인 (학습 중일 가능성)
                    cpu_percent = child.cpu_percent(interval=1)
                    if cpu_percent > 10:  # 10% 이상 CPU 사용 시 학습 중으로 판단
                        return True
            except (psutil.NoSuchProcess, psutil.AccessDenied):
                continue
                
        return False
        
    except ImportError:
        # psutil이 없는 경우 간단한 방법으로 확인
        logger.info("psutil이 설치되지 않아 프로세스 확인을 건너뜁니다.")
        return False
    except Exception as e:
        logger.error(f"프로세스 확인 중 오류: {str(e)}")
        return False

@app.get("/rag/total-time")
async def get_total_time():
    """RAG 초기화의 실제 총 작업 시간을 반환합니다."""
    try:
        redis_client = get_redis_client()
        
        # 시작 시간 조회
        start_time_str = redis_client.get("rag:init:start_time")
        if not start_time_str:
            return {
                "status": "error",
                "message": "초기화 시작 시간을 찾을 수 없습니다.",
                "timestamp": get_current_time().isoformat()
            }
            
        # 시간 문자열을 datetime 객체로 변환 (UTC+9 기준)
        start_time = datetime.fromisoformat(start_time_str.decode())
        if start_time.tzinfo is None:
            start_time = start_time.replace(tzinfo=timezone(timedelta(hours=9)))
        
        # 완료 시간 조회
        end_time_str = redis_client.get("rag:init:end_time")
        if end_time_str:
            end_time = datetime.fromisoformat(end_time_str.decode())
            if end_time.tzinfo is None:
                end_time = end_time.replace(tzinfo=timezone(timedelta(hours=9)))
            
            # 시간이 유효한지 확인
            if end_time < start_time:
                logger.info(f"시간 오류: 종료 시간({end_time})이 시작 시간({start_time})보다 이전입니다. 이전 종료 시간을 삭제합니다.", error=True)
                # 잘못된 종료 시간을 Redis에서 삭제
                redis_client.delete("rag:init:end_time")
                end_time = None
                total_time = None
                is_completed = False
            else:
                total_time = end_time - start_time
                is_completed = True
        else:
            end_time = None
            total_time = None
            is_completed = False
            
        # 큐 상태 확인
        queue_status = await get_queue_status()
        
        # 현재 상태가 'success'이고 end_time이 없는 경우, 현재 시간을 end_time으로 사용
        if queue_status["status"] == "success" and not end_time:
            end_time = get_current_time()
            total_time = end_time - start_time
            is_completed = True
            # Redis에 end_time 저장
            redis_client.set("rag:init:end_time", end_time.isoformat().encode())
            logger.info(f"작업 완료로 end_time 저장: {end_time.isoformat()}")
        
        return {
            "status": "success",
            "start_time": start_time.isoformat(),
            "end_time": end_time.isoformat() if end_time else None,
            "total_time": {
                "seconds": total_time.total_seconds() if total_time else None,
                "formatted": format_duration(total_time) if total_time else None,
                "hours": round(total_time.total_seconds() / 3600, 2) if total_time else None,
                "minutes": round(total_time.total_seconds() / 60, 2) if total_time else None
            } if total_time else None,
            "is_completed": is_completed,
            "current_status": queue_status["status"],
            "queue_info": queue_status["queue_status"] if "queue_status" in queue_status else None,
            "timestamp": get_current_time().isoformat()
        }
        
    except Exception as e:
        logger.info(f"총 작업 시간 계산 중 오류 발생: {str(e)}", error=True)
        return {
            "status": "error",
            "error": str(e),
            "timestamp": get_current_time().isoformat()
        }

class SemanticChunkRequest(BaseModel):
    text: str
    user_id: Optional[str] = None

class SemanticChunkResponse(BaseModel):
    success: bool
    chunks: Optional[List[str]] = None
    error: Optional[str] = None

@app.post("/semantic-chunk", response_model=SemanticChunkResponse)
async def create_semantic_chunks_api(request: SemanticChunkRequest):
    """텍스트를 시맨틱 청크로 분할하는 엔드포인트 - Redis를 통해 임베딩 워커에게 위임"""
    try:
        # 임베딩 워커에게 시맨틱 청킹 작업 위임 (Redis 활용)
        import uuid
        task_id = str(uuid.uuid4())

        # Redis를 통해 임베딩 워커와 통신 (중앙집중화된 설정 사용)
        import redis
        redis_host = os.getenv('REDIS_HOST', 'localhost')
        redis_port = int(os.getenv('REDIS_PORT', '6379'))
        redis_client = redis.Redis(host=redis_host, port=redis_port, db=0, decode_responses=True)

        # 작업 데이터 저장
        task_data = {
            'type': 'semantic_chunk',
            'text': request.text,
            'task_id': task_id
        }

        # 임베딩 큐에 작업 추가
        redis_client.lpush('embedding_tasks', json.dumps(task_data))
        logger.info(f"시맨틱 청킹 작업을 임베딩 워커에게 위임 (API): {task_id}, {len(request.text)}자")

        # 결과 대기 (최대 30초)
        for i in range(300):  # 30초 (0.1초씩 300번)
            result_key = f"semantic_chunk_result:{task_id}"
            result_data = redis_client.get(result_key)

            if result_data:
                result = json.loads(result_data)
                redis_client.delete(result_key)  # 결과 삭제

                if result.get('success'):
                    chunks = result.get('chunks', [])
                    logger.info(f"임베딩 워커로부터 시맨틱 청킹 결과 수신 (API): {len(chunks)}개 청크")
                    return SemanticChunkResponse(
                        success=True,
                        chunks=chunks
                    )
                else:
                    error_msg = result.get('error', 'Unknown error')
                    logger.error(f"임베딩 워커 시맨틱 청킹 실패 (API): {error_msg}")
                    return SemanticChunkResponse(
                        success=False,
                        error=error_msg
                    )

            time.sleep(0.1)

        # 타임아웃
        logger.error("시맨틱 청킹 요청 타임아웃 (API)")
        return SemanticChunkResponse(
            success=False,
            error="Request timeout"
        )

    except Exception as e:
        logger.error(f"Semantic chunking request failed: {str(e)}")
        return SemanticChunkResponse(
            success=False,
            error=str(e)
        )

@app.post("/cleanup-missing-files")
async def cleanup_missing_files(user_id: Optional[str] = None):
    """PostgreSQL에서 실제로 존재하지 않는 파일들의 데이터를 정리합니다."""
    try:
        # PostgreSQL 연결 (전역 DB pool 사용)
        pg_connection = None
        pg_cursor = None
        
        try:
            # 통합된 연결 풀 사용
            pg_connection = get_pool_connection()
            
            pg_cursor = pg_connection.cursor()
            
            # 사용자별 문서 조회 쿼리
            if user_id and user_id.strip():
                # logger.info(f"사용자 ID 필터 적용: {user_id}의 문서만 정리")
                pg_cursor.execute(
                    "SELECT DISTINCT metadata->>'source', metadata->>'user_id' FROM document_embeddings WHERE metadata->>'user_id' = %s",
                    (user_id.strip(),)
                )
            else:
                logger.info("모든 사용자의 문서를 정리합니다.")
                pg_cursor.execute(
                    "SELECT DISTINCT metadata->>'source', metadata->>'user_id' FROM document_embeddings"
                )
            
            files_to_check = pg_cursor.fetchall()
        
            # 존재하지 않는 파일들 수집
            missing_files = []
            
            for source_file, file_user_id in files_to_check:
                if not source_file:
                    continue
                
                # RAG 문서 기본 경로에서 파일 존재 여부 확인
                rag_docs_path = get_rag_docs_path()
                if file_user_id:
                    full_path = os.path.join(rag_docs_path, file_user_id, source_file)
                else:
                    full_path = os.path.join(rag_docs_path, source_file)
                
                if not os.path.exists(full_path):
                    missing_files.append((source_file, file_user_id))
        
            # 존재하지 않는 파일들의 데이터 삭제
            deleted_count = 0
            if missing_files:
                logger.info(f"존재하지 않는 파일 {len(missing_files)}개의 데이터를 삭제합니다...")
                
                for source_file, file_user_id in missing_files:
                    try:
                        if file_user_id:
                            pg_cursor.execute(
                                "DELETE FROM document_embeddings WHERE metadata->>'source' = %s AND metadata->>'user_id' = %s",
                                (source_file, file_user_id)
                            )
                        else:
                            pg_cursor.execute(
                                "DELETE FROM document_embeddings WHERE metadata->>'source' = %s",
                                (source_file,)
                            )
                        deleted_count += pg_cursor.rowcount
                        logger.info(f"삭제 완료: {source_file} ({pg_cursor.rowcount}개 청크)")
                    except Exception as e:
                        logger.error(f"파일 {source_file} 삭제 중 오류: {str(e)}")
                
                # 변경사항 커밋
                pg_connection.commit()
        
            logger.info("\n정리 작업 완료:")
            logger.info(f"- 삭제된 청크 수: {deleted_count}")
            logger.info(f"- 삭제된 파일 수: {len(missing_files)}")
            
            return {
                "status": "success",
                "message": f"{deleted_count}개의 청크가 정리되었습니다.",
                "deleted_chunks": deleted_count,
                "deleted_files": len(missing_files),
                "missing_files": [f[0] for f in missing_files]
            }
            
        finally:
            if pg_cursor:
                pg_cursor.close()
            if pg_connection:
                # DocumentProcessor DB pool에 연결 반환
                if document_processor:
                    document_processor.return_db_connection(pg_connection)
                elif 'GLOBAL_DB_POOL' in globals() and GLOBAL_DB_POOL:
                    return_pool_connection(pg_connection)
                else:
                    pg_connection.close()
        
    except Exception as e:
        logger.error(f"파일 정리 중 오류: {str(e)}")
        return {
            "status": "error",
            "message": str(e)
        }

async def cleanup_duplicate_documents_internal(user_id: Optional[str] = None):
    """chat_documents 테이블에서 중복 문서를 정리합니다 (내부 함수)"""
    try:
        # 동기 연결 풀 사용 (중앙 집중식)
        conn = get_pool_connection()
        try:
            # 먼저 중복 문서 조회 (짧은 읽기 전용 쿼리)
            with conn.cursor() as cur:
                if user_id is None:
                    cur.execute("""
                        SELECT user_id, filename, COUNT(*) as count,
                               array_agg(id ORDER BY created_at DESC) as record_ids,
                               array_agg(created_at ORDER BY created_at DESC) as created_ats
                        FROM chat_documents
                        GROUP BY user_id, filename
                        HAVING COUNT(*) > 1
                        ORDER BY user_id, filename
                    """)
                    duplicate_groups = cur.fetchall()
                else:
                    cur.execute("""
                        SELECT user_id, filename, COUNT(*) as count,
                               array_agg(id ORDER BY created_at DESC) as record_ids,
                               array_agg(created_at ORDER BY created_at DESC) as created_ats
                        FROM chat_documents
                        WHERE user_id = %s
                        GROUP BY user_id, filename
                        HAVING COUNT(*) > 1
                        ORDER BY user_id, filename
                    """, (user_id,))
                    duplicate_groups = cur.fetchall()

            if not duplicate_groups:
                return {
                    "status": "success",
                    "message": "중복 문서가 발견되지 않았습니다.",
                    "cleaned_groups": 0,
                    "deleted_records": 0
                }

            total_deleted = 0
            cleaned_groups = 0

            # 각 그룹을 개별 트랜잭션으로 처리하여 잠금 시간 최소화
            for group in duplicate_groups:
                try:
                    user_id_val = group[0]  # user_id
                    filename_val = group[1]  # filename
                    count = group[2]  # count
                    record_ids = group[3]  # record_ids
                    created_ats = group[4]  # created_ats

                    logger.info(f"중복 문서 그룹 처리: {user_id_val}/{filename_val} (총 {count}개)")

                    # 가장 최신 레코드 ID (created_at이 가장 최신)
                    latest_record_id = record_ids[0]

                    # 중복 레코드들 삭제 (가장 최신 제외)
                    duplicate_ids = record_ids[1:]
                    if duplicate_ids:
                        # 삭제 전 해당 레코드들이 여전히 존재하는지 확인 (동시성 안전성)
                        with conn.cursor() as cur:
                            cur.execute("""
                                SELECT id FROM chat_documents
                                WHERE id = ANY(%s) AND user_id = %s AND filename = %s
                            """, (duplicate_ids, user_id_val, filename_val))
                            existing_ids = cur.fetchall()

                        existing_id_list = [row[0] for row in existing_ids]

                        if existing_id_list:
                            with conn.cursor() as cur:
                                cur.execute("""
                                    DELETE FROM chat_documents
                                    WHERE id = ANY(%s)
                                """, (existing_id_list,))
                                conn.commit()

                            total_deleted += len(existing_id_list)
                            cleaned_groups += 1

                            logger.info(f"중복 레코드 {len(existing_id_list)}개 삭제 완료: {user_id_val}/{filename_val}")
                            logger.info(f"- 유지된 레코드 ID: {latest_record_id} (생성시간: {created_ats[0]})")
                            logger.info(f"- 삭제된 레코드 ID들: {existing_id_list}")

                except Exception as group_error:
                    logger.error(f"중복 문서 그룹 처리 중 오류 (계속 진행): {group_error}")
                    # 개별 그룹 처리 실패 시에도 다음 그룹 계속 처리
                    continue

            return {
                "status": "success",
                "message": f"{cleaned_groups}개 그룹의 중복 문서가 정리되었습니다.",
                "cleaned_groups": cleaned_groups,
                "deleted_records": total_deleted
            }
        finally:
            return_pool_connection(conn)

    except Exception as e:
        logger.error(f"중복 문서 정리 중 오류: {str(e)}")
        return {
            "status": "error",
            "message": str(e)
        }

@app.post("/cleanup-duplicate-documents")
async def cleanup_duplicate_documents(user_id: Optional[str] = None):
    """chat_documents 테이블에서 중복 문서를 정리합니다."""
    try:
        result = await cleanup_duplicate_documents_internal(user_id)
        return result
    except Exception as e:
        logger.error(f"중복 문서 정리 API 호출 중 오류: {str(e)}")
        return {
            "status": "error",
            "message": str(e)
        }

@app.post("/auto-optimize")
async def auto_optimize_settings(request: Request):
    """RAG 설정을 자동으로 최적화하는 엔드포인트"""
    try:
        data = await request.json()
        user_id = data.get('user_id', 'admin')
        test_queries = data.get('test_queries', [])
        
        logger.info(f"자동 최적화 요청: user_id={user_id}, 쿼리 수={len(test_queries)}")
        
        if not test_queries:
            return {
                "success": False,
                "error": "테스트 쿼리가 제공되지 않았습니다."
            }
        

        
        # 현재 RAG 설정 가져오기
        current_settings = get_rag_settings()
        
        # 현재 설정값 추출 (실제 설정에서 가져오기)
        current_keyword = float(current_settings.get('RAG_KEYWORD_PRECISION_THRESHOLD', 0.1))
        current_semantic = float(current_settings.get('RAG_SEMANTIC_RELEVANCE_THRESHOLD', 0.15))
        current_quality = float(current_settings.get('RAG_CONTENT_QUALITY_THRESHOLD', 0.3))
        current_recency = float(current_settings.get('RAG_RECENCY_THRESHOLD', 0.0))

        logger.info(f"현재 설정값: keyword={current_keyword}, semantic={current_semantic}, quality={current_quality}, recency={current_recency}")
        
        # 최적화할 파라미터 범위 정의 (더 광범위한 범위로 실제 최적화)
        param_ranges = {
            'RAG_KEYWORD_PRECISION_THRESHOLD': [0.0, 0.05, 0.1, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4],
            'RAG_SEMANTIC_RELEVANCE_THRESHOLD': [0.05, 0.1, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4, 0.5],
            'RAG_CONTENT_QUALITY_THRESHOLD': [0.1, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45, 0.5, 0.6],
            'RAG_RECENCY_THRESHOLD': [0.0, 0.05, 0.1, 0.15, 0.2, 0.25]
        }

        # 현재 설정도 범위에 포함 (만약 없다면)
        for param, current_val in [
            ('RAG_KEYWORD_PRECISION_THRESHOLD', current_keyword),
            ('RAG_SEMANTIC_RELEVANCE_THRESHOLD', current_semantic),
            ('RAG_CONTENT_QUALITY_THRESHOLD', current_quality),
            ('RAG_RECENCY_THRESHOLD', current_recency)
        ]:
            if current_val not in param_ranges[param]:
                param_ranges[param].append(current_val)
                param_ranges[param].sort()
        
        best_config = None
        best_score = -1
        optimization_results = []
        current_config_score = None  # 현재 설정의 점수 저장
        
        # 스마트한 조합 생성 (현재 설정 + 랜덤 샘플링)
        import itertools
        import random

        # 1. 현재 설정
        prioritized_combinations = [
            (current_keyword, current_semantic, current_quality, current_recency)
        ]

        # 2. 각 파라미터를 개별적으로 변경한 조합 (민감도 분석)
        for param_name, param_values in param_ranges.items():
            current_vals = [current_keyword, current_semantic, current_quality, current_recency]
            param_idx = ['RAG_KEYWORD_PRECISION_THRESHOLD', 'RAG_SEMANTIC_RELEVANCE_THRESHOLD',
                        'RAG_CONTENT_QUALITY_THRESHOLD', 'RAG_RECENCY_THRESHOLD'].index(param_name)

            for val in param_values[:3]:  # 각 파라미터당 3개 값만 테스트
                if val != current_vals[param_idx]:
                    test_vals = current_vals.copy()
                    test_vals[param_idx] = val
                    prioritized_combinations.append(tuple(test_vals))

        # 3. 랜덤 조합 추가 (더 다양한 탐색)
        random.seed(42)  # 재현 가능한 결과를 위해
        all_combinations = list(itertools.product(
            param_ranges['RAG_KEYWORD_PRECISION_THRESHOLD'][:4],
            param_ranges['RAG_SEMANTIC_RELEVANCE_THRESHOLD'][:4],
            param_ranges['RAG_CONTENT_QUALITY_THRESHOLD'][:4],
            param_ranges['RAG_RECENCY_THRESHOLD'][:3]
        ))

        # 중복 제거하고 랜덤 샘플링
        unique_combinations = list(set(prioritized_combinations + random.sample(all_combinations, 10)))

        # 최대 25개 조합 테스트
        test_combinations = unique_combinations[:25]
        logger.info(f"테스트할 조합 수: {len(test_combinations)}")
        
        for idx, combo in enumerate(test_combinations):
            # 테스트 설정 구성
            test_config = {
                'RAG_KEYWORD_PRECISION_THRESHOLD': combo[0],
                'RAG_SEMANTIC_RELEVANCE_THRESHOLD': combo[1],
                'RAG_CONTENT_QUALITY_THRESHOLD': combo[2],
                'RAG_RECENCY_THRESHOLD': combo[3]
            }

            logger.info(f"=== 조합 {idx+1}/{len(test_combinations)} 테스트 ===")
            logger.info(f"키워드임계값: {combo[0]}, 의미임계값: {combo[1]}, 품질임계값: {combo[2]}, 최신성임계값: {combo[3]}")

            # 설정을 임시로 적용
            temp_settings = current_settings.copy()
            temp_settings.update(test_config)
            
            # 이 설정으로 실제 검색 테스트
            total_relevance_score = 0
            total_precision_score = 0
            successful_queries = 0
            
            for query in test_queries[:5]:  # 최대 5개 쿼리만 테스트
                try:
                    # 임시 설정을 적용하여 실제 검색 수행
                    temp_search_request = SearchRequest(
                        query=query,
                        user_id=user_id,
                        top_k=5,
                        tempSettings=test_config  # 임시 설정 전달
                    )
                    search_result = await search_documents(temp_search_request)
                    
                    if search_result and 'results' in search_result:
                        docs = search_result['results']
                        
                        if docs:
                            # 실제 키워드 검색 품질 평가
                            keyword_match_score = 0
                            relevance_score = 0

                            # 1. 키워드 매칭 품질 평가
                            query_lower = query.lower()
                            query_keywords = set(query_lower.split())

                            for doc in docs:
                                doc_content = (doc.get('content', '') + ' ' + doc.get('title', '')).lower()
                                doc_keywords = set(doc_content.split())

                                # 키워드 매칭률
                                match_ratio = len(query_keywords.intersection(doc_keywords)) / len(query_keywords) if query_keywords else 0
                                keyword_match_score += match_ratio

                                # 전체 관련성 점수
                                relevance_score += doc.get('final_score', 0)

                            # 평균 계산
                            avg_keyword_match = keyword_match_score / len(docs)
                            avg_relevance = relevance_score / len(docs)

                            # 2. 결과의 다양성과 품질 체크
                            quality_threshold = test_config.get('RAG_CONTENT_QUALITY_THRESHOLD', 0.3)
                            high_quality_docs = [d for d in docs if d.get('final_score', 0) > quality_threshold]
                            quality_ratio = len(high_quality_docs) / len(docs) if docs else 0

                            # 3. 종합 점수 계산 (키워드 매칭이 가장 중요)
                            composite_score = (avg_keyword_match * 0.4 + avg_relevance * 0.3 + quality_ratio * 0.3)

                            total_relevance_score += composite_score
                            total_precision_score += quality_ratio
                            successful_queries += 1

                            logger.info(f"쿼리 '{query}': 키워드매칭={avg_keyword_match:.2f}, 관련성={avg_relevance:.2f}, 품질비율={quality_ratio:.2f}, 종합점수={composite_score:.2f}")
                        else:
                            # 결과가 없는 경우 - 설정이 너무 엄격함
                            logger.info(f"쿼리 '{query}': 검색 결과 없음 (설정이 너무 엄격할 수 있음)")
                            total_relevance_score += 0.05  # 매우 낮은 점수
                            total_precision_score += 0.05
                            successful_queries += 1
                    
                except Exception as e:
                    logger.error(f"쿼리 '{query}' 테스트 실패: {e}")
            
            if successful_queries > 0:
                # 종합 점수 계산 (관련성과 정밀도의 조화평균)
                avg_relevance = total_relevance_score / successful_queries
                avg_precision = total_precision_score / successful_queries
                
                # 조화평균으로 균형잡힌 점수 계산
                if avg_relevance > 0 and avg_precision > 0:
                    harmonic_score = 2 * (avg_relevance * avg_precision) / (avg_relevance + avg_precision)
                else:
                    harmonic_score = 0
                
                optimization_results.append({
                    'config': test_config,
                    'score': harmonic_score * 100,  # 100점 만점으로
                    'relevance_score': avg_relevance * 100,
                    'precision_score': avg_precision * 100,
                    'tested_queries': successful_queries
                })

                logger.info(f"조합 {idx+1} 결과: 종합점수={harmonic_score*100:.1f}/100, 관련성={avg_relevance*100:.1f}, 정밀도={avg_precision*100:.1f}")

                # 현재 설정인 경우 점수 저장
                if idx == 0:  # 첫 번째는 항상 현재 설정
                    current_config_score = harmonic_score
                    logger.info(f"*** 현재 설정 기준 점수: {harmonic_score * 100:.1f}/100 ***")

                if harmonic_score > best_score:
                    best_score = harmonic_score
                    best_config = test_config
                    logger.info(f"🎯 새로운 최고 점수 발견: {harmonic_score*100:.1f}/100")
                elif harmonic_score == best_score and idx == 0:
                    # 동점인 경우 현재 설정을 우선 선택
                    best_config = test_config
            else:
                logger.info(f"조합 {idx+1}: 테스트 실패 (성공한 쿼리 없음)")

            # 진행률 로깅
            progress = ((idx + 1) / len(test_combinations)) * 100
            logger.info(f"최적화 진행률: {progress:.1f}% ({idx+1}/{len(test_combinations)})\n")
        
        # 결과 정렬 (점수 높은 순)
        optimization_results.sort(key=lambda x: x['score'], reverse=True)
        
        if best_config:
            logger.info(f"최적 설정 발견: 점수={best_score*100:.1f}/100")
            
            # 최고 성능 결과에서 개별 점수 추출
            best_result = optimization_results[0] if optimization_results else {}
            best_relevance_score = best_result.get('relevance_score', best_score * 100)
            best_precision_score = best_result.get('precision_score', best_score * 100)
            
            # 최적화 완료 메시지 생성
            message = f"최적화 완료: 테스트한 {len(test_combinations)}개 조합 중 최고 점수({best_score*100:.1f}점)의 설정을 찾았습니다."
            
            return {
                "success": True,
                "best_config": best_config,
                "best_score": best_score * 100,  # 100점 만점
                "relevance_score": best_relevance_score,  # 개별 관련성 점수
                "precision_score": best_precision_score,  # 개별 정밀도 점수
                "optimization_results": optimization_results[:5],  # 상위 5개만 반환
                "tested_configs": len(test_combinations),
                "message": message
            }
        else:
            return {
                "success": False,
                "error": "최적화 실패: 적합한 설정을 찾을 수 없습니다."
            }
        
    except Exception as e:
        logger.error(f"자동 최적화 실패: {e}")
        import traceback
        traceback.print_exc()
        return {
            "success": False,
            "error": str(e)
        }

@app.post("/extract-keywords")
async def extract_keywords_from_chunks(request: Request):
    """청크에서 키워드를 추출하는 엔드포인트"""
    try:
        data = await request.json()
        user_id = data.get('user_id', 'admin')
        max_chunks = data.get('max_chunks', 20)
        max_keywords = data.get('max_keywords', 6)
        
        logger.info(f"키워드 추출 요청: user_id={user_id}, max_chunks={max_chunks}")
        
        # 데이터베이스 연결 (connection pool 사용)
        conn = get_pool_connection()
        cursor = conn.cursor()
        
        # 사용자의 청크를 랜덤하게 선택
        cursor.execute("""
            SELECT chunk_text, filename 
            FROM document_embeddings
            WHERE (metadata->>'user_id') = %s
            ORDER BY RANDOM()
            LIMIT %s
        """, (user_id, max_chunks))
        
        chunks = cursor.fetchall()
        cursor.close()
        return_pool_connection(conn)
        
        if not chunks:
            logger.info(f"사용자 {user_id}의 청크를 찾을 수 없음")
            return {
                "success": True,
                "keywords": [
                    {"keyword": "정책 가이드", "description": "기본 키워드"},
                    {"keyword": "시스템 운영", "description": "기본 키워드"},
                    {"keyword": "사업 계획", "description": "기본 키워드"}
                ]
            }
        
        # 키워드 추출 로직 개선
        import re
        from collections import Counter, defaultdict
        
        # 문서별 단어 빈도 저장 (TF-IDF를 위한 준비)
        doc_word_freq = defaultdict(Counter)
        all_documents_words = []
        keyword_set = set()
        extracted_keywords = []
        
        for idx, (chunk_text, filename) in enumerate(chunks):
            # 청크 내용에서 의미있는 단어 추출 (더 긴 샘플 사용)
            content_sample = chunk_text[:1000] if chunk_text else ""
            
            # 명사구 패턴 추출 (더 의미있는 키워드)
            # 예: "정보통신공사", "표준품셈", "한국어문규정" 등
            korean_patterns = [
                r'[가-힣]{3,8}(?:공사|시스템|관리|서비스|정책|계획|방안|규정|표준|품셈|보안|분석|평가|프로젝트)',
                r'[가-힣]{2,6}(?:법|령|칙|집|서|안)',
                r'[가-힣]{3,10}'  # 일반 한글 단어
            ]
            
            found_words = []
            for pattern in korean_patterns:
                matches = re.findall(pattern, content_sample)
                found_words.extend(matches)
            
            # 영문 전문 용어 추출 (약어나 전문 용어)
            english_technical = re.findall(r'[A-Z]{2,6}|[A-Z][a-z]+(?:[A-Z][a-z]+)+', content_sample)
            found_words.extend([w for w in english_technical if len(w) >= 3][:2])  # 영문은 2개까지만
            
            meaningful_words = [w for w in found_words if len(w) >= 3]
            
            # 불용어 제거 (경로 관련 단어 및 일반 불용어 포함)
            stopwords = {
                # 한글 불용어
                '이', '그', '저', '것', '수', '등', '및', '의', '를', '을', '은', '는', '가', '와', '과', '에', '에서', '으로', '부터', '까지', '하고', '하는', '했다', '하여', '하면', '한다', '그리고', '그러나', '또는', '즉', '만약', '만일', '따라', '따라서', '그래서', '그러므로', '때문에', '있다', '있는', '있고', '없다', '없는', '아니다', '이다', '되다', '되는', '된다', '할', '하지', '않다', '있습니다', '습니다', '합니다',
                # 문서 관련 불필요 단어
                '문서', '파일', '페이지', '장', '절', '항', '조', '호', '목', '편', '부', '권', '책', '자료', '내용', '작성', '기록', '서류',
                # 영어 불용어
                'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'from', 'up', 'down', 'out', 'off', 'over', 'under', 'again', 'then', 'once', 'is', 'are', 'was', 'were', 'be', 'been', 'being', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'may', 'might', 'must', 'can', 'shall', 'if', 'when', 'where', 'what', 'which', 'who', 'whom', 'whose', 'why', 'how', 'this', 'that', 'these', 'those', 'it', 'its', 'itself',
                # 경로 관련 단어
                'home', 'hamonikr', 'user', 'usr', 'bin', 'lib', 'var', 'tmp', 'opt', 'etc', 'dev', 'proc', 'sys', 'root', 'mnt', 'media', 'srv', 'run', 'boot', 'docs', 'doc', 'documents', 'downloads', 'desktop', 'pictures', 'music', 'videos', 'public', 'templates', 'admin', 'work', 'workspace', 'workspaces', 'src', 'dist', 'build', 'node_modules', 'package', 'config', 'conf', 'test', 'tests', 'examples', 'example', 'sample', 'samples', 'demo', 'demos', 'backup', 'backups', 'cache', 'caches', 'log', 'logs', 'temp', 'tmp', 'rag', 'airun', 'file', 'files', 'folder', 'folders', 'directory', 'directories', 'path', 'paths'
            }
            meaningful_words = [w for w in meaningful_words if w.lower() not in stopwords]
            
            # 문서별로 단어 저장 (TF 계산용)
            for word in meaningful_words[:15]:  # 청크당 최대 15개 단어
                doc_word_freq[idx][word] += 1
                all_documents_words.append(word)
            
            # 파일명에서 핵심 키워드 추출
            if filename:
                # 파일명에서 확장자와 경로 제거
                clean_filename = re.sub(r'\.[^.]+$', '', filename)  # 확장자 제거
                clean_filename = re.sub(r'^.*/', '', clean_filename)  # 경로 제거
                
                file_keywords = re.findall(r'[가-힣]{2,}', clean_filename)
                for keyword in file_keywords:
                    if len(keyword) >= 2 and keyword not in stopwords:
                        doc_word_freq[idx][keyword] += 2  # 파일명 키워드는 가중치 2배
        
        # TF-IDF 스타일의 중요도 계산
        # DF(Document Frequency) 계산
        word_df = Counter()
        for doc_words in doc_word_freq.values():
            for word in doc_words.keys():
                word_df[word] += 1
        
        # 전체 문서 수
        num_docs = len(doc_word_freq)
        
        # 각 단어의 중요도 점수 계산
        word_importance = {}
        for word, df in word_df.items():
            # IDF = log(전체 문서 수 / 해당 단어가 나타난 문서 수)
            idf = 1.0 if num_docs <= 1 else (1.0 + (num_docs - df) / num_docs)
            
            # 전체 빈도수
            total_freq = sum(doc_word_freq[doc][word] for doc in doc_word_freq if word in doc_word_freq[doc])
            
            # 중요도 = 빈도 * IDF
            word_importance[word] = total_freq * idf
        
        # 중요도 순으로 정렬
        sorted_words = sorted(word_importance.items(), key=lambda x: x[1], reverse=True)
        
        # 중요도 기반으로 키워드 선택
        for word, importance_score in sorted_words[:max_keywords * 2]:
            if len(extracted_keywords) >= max_keywords:
                break
            
            # 불필요한 단어 필터링
            if word.lower() in stopwords or len(word) < 3:
                continue
            
            # 한글 우선
            is_korean = bool(re.match(r'[가-힣]+$', word))
            
            # 중요도 점수가 높은 단어만 선택
            if importance_score > 1.5 and is_korean:
                if word not in keyword_set:
                    keyword_set.add(word)
                    extracted_keywords.append({
                        "keyword": word,
                        "description": f"중요도: {importance_score:.1f}"
                    })
        
        # 키워드가 부족한 경우 한국어 기본 키워드 추가
        if len(extracted_keywords) < max_keywords:
            default_keywords = [
                "정책 가이드", "시스템 운영", "사업 계획", 
                "관리 방안", "서비스 제공", "프로그램 운영",
                "데이터 분석", "보고서 작성", "프로젝트 관리",
                "업무 프로세스", "성과 평가", "품질 관리"
            ]
            for default_kw in default_keywords:
                if len(extracted_keywords) >= max_keywords:
                    break
                if default_kw not in keyword_set:
                    keyword_set.add(default_kw)
                    extracted_keywords.append({
                        "keyword": default_kw,
                        "description": "기본 키워드"
                    })
        
        logger.info(f"키워드 추출 완료: {len(extracted_keywords)}개")
        
        return {
            "success": True,
            "keywords": extracted_keywords,
            "total_chunks_processed": len(chunks)
        }
        
    except Exception as e:
        logger.error(f"키워드 추출 실패: {e}")
        import traceback
        traceback.print_exc()
        return {
            "success": False,
            "error": str(e),
            "keywords": []
        }

@app.get("/chunks")
async def get_chunks(
    filename: str = Query(..., description="문서 파일명"),
    user_id: Optional[str] = Query(None, description="사용자 ID (선택사항)")
):
    """PostgreSQL에서 청크 데이터를 가져오는 API"""
    try:
        logger.info(f"[/chunks API] 요청: filename={filename}, user_id={user_id}")
        
        # 통합된 연결 풀 사용
        
        # PostgreSQL 연결
        conn = get_pool_connection()
        cursor = conn.cursor()
        
        # SQL 쿼리 준비
        where_conditions = ["filename LIKE %s"]
        params = [f"%{filename}%"]
        
        # 사용자 ID 필터링
        if user_id and user_id.strip():
            where_conditions.append("(metadata->>'user_id') = %s")
            params.append(user_id.strip())
            logger.info(f"[/chunks API] 사용자 필터 적용: user_id={user_id}")
        
        # 청크 데이터 조회
        query = f"""
            SELECT id, doc_id, filename, chunk_index, chunk_text, metadata, created_at, embedding
            FROM document_embeddings 
            WHERE {' AND '.join(where_conditions)}
            ORDER BY chunk_index
        """
        
        cursor.execute(query, params)
        rows = cursor.fetchall()
        
        logger.info(f"[/chunks API] PostgreSQL에서 가져온 청크 수: {len(rows)}")
        
        # 청크 데이터 형식화
        chunks = []
        for row in rows:
            db_id, doc_id, db_filename, chunk_index, chunk_text, metadata, created_at, embedding = row
            
            # 메타데이터 파싱 (JSONB)
            parsed_metadata = metadata if isinstance(metadata, dict) else {}
            
            chunks.append({
                "index": chunk_index,
                "content": chunk_text,
                "metadata": {
                    "db_id": db_id,
                    "doc_id": doc_id,
                    "filename": db_filename,
                    "chunk_index": chunk_index,
                    "created_at": created_at.isoformat() if created_at else None,
                    "has_embedding": embedding is not None,
                    "embedding_dimensions": len(embedding) if embedding else 0,
                    **parsed_metadata  # 기존 메타데이터 포함
                }
            })
        
        cursor.close()
        return_pool_connection(conn)
        
        logger.info(f"[/chunks API] 파일명 '{filename}'과 일치하는 청크 수: {len(chunks)}")
        return {"success": True, "chunks": chunks}
        
    except Exception as e:
        logger.error(f"/chunks API error: {str(e)}")
        return {"success": False, "message": str(e)}

@app.get("/chunks-test")
async def get_chunks_test(
    filename: str = Query(..., description="문서 파일명"),
    user_id: Optional[str] = Query(None, description="사용자 ID (선택사항)")
):
    """PostgreSQL에서 청크 데이터를 가져오는 테스트 API"""
    try:
        logger.info(f"[/chunks-test API] 요청: filename={filename}, user_id={user_id}")
        
        # 통합된 연결 풀 사용
        
        # PostgreSQL 연결
        conn = get_pool_connection()
        cursor = conn.cursor()
        
        # SQL 쿼리 준비
        where_conditions = ["filename LIKE %s"]
        params = [f"%{filename}%"]
        
        # 사용자 ID 필터링
        if user_id and user_id.strip():
            where_conditions.append("(metadata->>'user_id') = %s")
            params.append(user_id.strip())
            logger.info(f"[/chunks-test API] 사용자 필터 적용: user_id={user_id}")
        
        # 청크 데이터 조회
        query = f"""
            SELECT id, doc_id, filename, chunk_index, chunk_text, metadata, created_at, embedding
            FROM document_embeddings 
            WHERE {' AND '.join(where_conditions)}
            ORDER BY chunk_index
        """
        
        cursor.execute(query, params)
        rows = cursor.fetchall()
        
        logger.info(f"[/chunks-test API] PostgreSQL에서 가져온 청크 수: {len(rows)}")
        
        # 청크 데이터 형식화
        chunks = []
        for row in rows:
            db_id, doc_id, db_filename, chunk_index, chunk_text, metadata, created_at, embedding = row
            
            # 메타데이터 파싱 (JSONB)
            parsed_metadata = metadata if isinstance(metadata, dict) else {}
            
            chunks.append({
                "index": chunk_index,
                "content": chunk_text,
                "metadata": {
                    "db_id": db_id,
                    "doc_id": doc_id,
                    "filename": db_filename,
                    "chunk_index": chunk_index,
                    "created_at": created_at.isoformat() if created_at else None,
                    "has_embedding": embedding is not None,
                    "embedding_dimensions": len(embedding) if embedding else 0,
                    **parsed_metadata  # 기존 메타데이터 포함
                }
            })
        
        cursor.close()
        return_pool_connection(conn)
        
        logger.info(f"[/chunks-test API] 파일명 '{filename}'과 일치하는 청크 수: {len(chunks)}")
        return {"success": True, "chunks": chunks}
        
    except Exception as e:
        logger.error(f"/chunks-test API error: {str(e)}")
        return {"success": False, "message": str(e), "traceback": traceback.format_exc()}

# 데이터셋 생성 API
@app.post("/datasets/create")
async def create_dataset(request: Request):
    """다양한 방식으로 데이터셋 생성"""
    try:
        data = await request.json()
        dataset_type = data.get('type', 'document_based')
        name = data.get('name', f'dataset_{datetime.now().strftime("%Y%m%d_%H%M%S")}')
        description = data.get('description', '')
        
        if dataset_type == 'document_based':
            documents = data.get('documents', [])
            use_ai = data.get('use_ai', False)  # AI 기반 생성 여부 (기본값: False - 효율성 우선)
            
            if use_ai:
                # AI 기반 QA 생성 (선택적)
                dataset_id = await dataset_manager.create_dataset_from_documents_with_ai(
                    documents, name, description
                )
            else:
                # 기존 방식 (기본값 - 빠르고 효율적)
                dataset_id = dataset_manager.create_dataset_from_documents(
                    documents, name, description
                )
        
        elif dataset_type == 'search_log_based':
            search_logs = data.get('search_logs', [])
            dataset_id = dataset_manager.create_dataset_from_search_logs(
                search_logs, name, description
            )
        
        elif dataset_type == 'llm_generated':
            documents = data.get('documents', [])
            llm_api_url = data.get('llm_api_url', 'http://localhost:5500/api/chat')
            # LLM 생성 타입은 이미 AI 기반이므로 새로운 방법 사용
            dataset_id = await dataset_manager.create_dataset_from_documents_with_ai(
                documents, name, f"{description} (LLM 생성)"
            )
        
        elif dataset_type == 'manual_labeled':
            labeled_pairs = data.get('labeled_pairs', [])
            dataset_id = dataset_manager.create_dataset_from_manual_labels(
                labeled_pairs, name, description
            )
        
        elif dataset_type == 'user_feedback':
            feedback_data = data.get('feedback_data', [])
            dataset_id = dataset_manager.create_dataset_from_feedback(
                feedback_data, name, description
            )
        
        elif dataset_type == 'merged':
            dataset_ids = data.get('dataset_ids', [])
            dataset_id = dataset_manager.merge_datasets(
                dataset_ids, name, description
            )
        
        else:
            return JSONResponse(
                status_code=400,
                content={'error': f'지원하지 않는 데이터셋 타입: {dataset_type}'}
            )
        
        return JSONResponse(content={
            'success': True,
            'dataset_id': dataset_id,
            'message': f'데이터셋 생성 완료: {name}'
        })
    
    except Exception as e:
        logger.error(f"데이터셋 생성 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'데이터셋 생성 실패: {str(e)}'}
        )

@app.get("/datasets/list")
async def list_datasets():
    """데이터셋 목록 조회"""
    try:
        datasets = dataset_manager.list_datasets()
        return JSONResponse(content={'datasets': datasets})
    except Exception as e:
        logger.error(f"데이터셋 목록 조회 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'데이터셋 목록 조회 실패: {str(e)}'}
        )

@app.get("/datasets/{dataset_id}")
async def get_dataset(dataset_id: str):
    """특정 데이터셋 조회"""
    try:
        dataset = dataset_manager.get_dataset(dataset_id)
        if not dataset:
            return JSONResponse(
                status_code=404,
                content={'error': '데이터셋을 찾을 수 없습니다'}
            )
        return JSONResponse(content={'dataset': dataset})
    except Exception as e:
        logger.error(f"데이터셋 조회 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'데이터셋 조회 실패: {str(e)}'}
        )

@app.get("/datasets/{dataset_id}/statistics")
async def get_dataset_statistics(dataset_id: str):
    """데이터셋 통계 정보 조회"""
    try:
        stats = dataset_manager.get_dataset_statistics(dataset_id)
        if not stats:
            return JSONResponse(
                status_code=404,
                content={'error': '데이터셋을 찾을 수 없습니다'}
            )
        return JSONResponse(content={'statistics': stats})
    except Exception as e:
        logger.error(f"데이터셋 통계 조회 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'데이터셋 통계 조회 실패: {str(e)}'}
        )

@app.delete("/datasets/{dataset_id}")
async def delete_dataset(dataset_id: str):
    """데이터셋 삭제"""
    try:
        success, error_message = dataset_manager.delete_dataset(dataset_id)
        if not success:
            # 보호된 데이터셋인 경우 403, 없는 경우 404
            status_code = 403 if '보호된' in error_message else 404
            return JSONResponse(
                status_code=status_code,
                content={'error': error_message}
            )
        return JSONResponse(content={'success': True, 'message': '데이터셋 삭제 완료'})
    except Exception as e:
        logger.error(f"데이터셋 삭제 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'데이터셋 삭제 실패: {str(e)}'}
        )

# 검색 로그 저장 API (데이터셋 생성용)
@app.post("/search-logs/save")
async def save_search_log(request: Request):
    """검색 로그 저장 (향후 데이터셋 생성에 활용)"""
    try:
        data = await request.json()
        query = data.get('query', '')
        results = data.get('results', [])
        clicked_indices = data.get('clicked_indices', [])
        
        # 검색 로그 저장 (실제 구현에서는 DB에 저장)
        search_log = {
            'query': query,
            'timestamp': datetime.now().isoformat(),
            'results': results,
            'clicked_indices': clicked_indices,
            'clicked_documents': [results[i] for i in clicked_indices if i < len(results)],
            'not_clicked_documents': [results[i] for i in range(len(results)) if i not in clicked_indices]
        }
        
        # 로그 파일에 저장 (임시)
        log_file = Path("search_logs.jsonl")
        with open(log_file, 'a', encoding='utf-8') as f:
            f.write(json.dumps(search_log, ensure_ascii=False) + '\n')
        
        return JSONResponse(content={'success': True, 'message': '검색 로그 저장 완료'})
    
    except Exception as e:
        logger.error(f"검색 로그 저장 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'검색 로그 저장 실패: {str(e)}'}
        )

# 사용자 피드백 저장 API
@app.post("/feedback/save")
async def save_feedback(request: Request):
    """사용자 피드백 저장"""
    try:
        data = await request.json()
        query = data.get('query', '')
        document = data.get('document', '')
        rating = data.get('rating', 3)  # 1-5 점수
        user_id = data.get('user_id', 'anonymous')
        
        feedback = {
            'query': query,
            'document': document,
            'rating': rating,
            'user_id': user_id,
            'timestamp': datetime.now().isoformat()
        }
        
        # 피드백 파일에 저장 (임시)
        feedback_file = Path("user_feedback.jsonl")
        with open(feedback_file, 'a', encoding='utf-8') as f:
            f.write(json.dumps(feedback, ensure_ascii=False) + '\n')
        
        return JSONResponse(content={'success': True, 'message': '피드백 저장 완료'})
    
    except Exception as e:
        logger.error(f"피드백 저장 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'피드백 저장 실패: {str(e)}'}
        )

# 저장된 검색 로그로부터 데이터셋 생성
@app.post("/datasets/from-search-logs")
async def create_dataset_from_search_logs(request: Request):
    """저장된 검색 로그로부터 데이터셋 생성"""
    try:
        data = await request.json()
        name = data.get('name', f'search_log_dataset_{datetime.now().strftime("%Y%m%d_%H%M%S")}')
        description = data.get('description', '검색 로그 기반 데이터셋')
        
        # 검색 로그 파일 읽기
        log_file = Path("search_logs.jsonl")
        search_logs = []
        
        if log_file.exists():
            with open(log_file, 'r', encoding='utf-8') as f:
                for line in f:
                    try:
                        search_logs.append(json.loads(line.strip()))
                    except json.JSONDecodeError:
                        continue
        
        if not search_logs:
            return JSONResponse(
                status_code=400,
                content={'error': '저장된 검색 로그가 없습니다'}
            )
        
        dataset_id = dataset_manager.create_dataset_from_search_logs(
            search_logs, name, description
        )
        
        return JSONResponse(content={
            'success': True,
            'dataset_id': dataset_id,
            'message': f'검색 로그 기반 데이터셋 생성 완료: {name}',
            'search_logs_count': len(search_logs)
        })
    
    except Exception as e:
        logger.error(f"검색 로그 기반 데이터셋 생성 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'검색 로그 기반 데이터셋 생성 실패: {str(e)}'}
        )

# 저장된 피드백으로부터 데이터셋 생성
@app.post("/datasets/from-feedback")
async def create_dataset_from_feedback(request: Request):
    """저장된 피드백으로부터 데이터셋 생성"""
    try:
        data = await request.json()
        name = data.get('name', f'feedback_dataset_{datetime.now().strftime("%Y%m%d_%H%M%S")}')
        description = data.get('description', '사용자 피드백 기반 데이터셋')
        
        # 피드백 파일 읽기
        feedback_file = Path("user_feedback.jsonl")
        feedback_data = []
        
        if feedback_file.exists():
            with open(feedback_file, 'r', encoding='utf-8') as f:
                for line in f:
                    try:
                        feedback_data.append(json.loads(line.strip()))
                    except json.JSONDecodeError:
                        continue
        
        if not feedback_data:
            return JSONResponse(
                status_code=400,
                content={'error': '저장된 피드백이 없습니다'}
            )
        
        dataset_id = dataset_manager.create_dataset_from_feedback(
            feedback_data, name, description
        )
        
        return JSONResponse(content={
            'success': True,
            'dataset_id': dataset_id,
            'message': f'피드백 기반 데이터셋 생성 완료: {name}',
            'feedback_count': len(feedback_data)
        })
    
    except Exception as e:
        logger.error(f"피드백 기반 데이터셋 생성 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'피드백 기반 데이터셋 생성 실패: {str(e)}'}
        )

# 임베딩 모델 평가 엔드포인트
@app.post("/evaluate")
async def evaluate_embedding_model(request: EvaluateModelRequest):
    """임베딩 모델 평가 엔드포인트"""
    try:
        logger.info(f"======== 임베딩 모델 평가 요청 ========")
        logger.info(f"모델 ID: {request.model_id}")
        logger.info(f"데이터셋 ID (요청): {request.dataset_id}")

        # ModelEvaluator 인스턴스 생성
        evaluator = ModelEvaluator()

        # 평가용 데이터셋이 우선, 없으면 기존 학습용 데이터셋 사용
        dataset_id = request.evaluation_dataset_id or request.dataset_id

        # 평가용 데이터셋이 있는 경우 LLM 서버에서 해당 데이터셋 가져오기 시도
        if request.evaluation_dataset_id:
            try:
                import requests
                llm_server_url = "http://localhost:5630"
                response = requests.get(f"{llm_server_url}/datasets")
                if response.status_code == 200:
                    datasets = response.json().get('datasets', [])
                    eval_dataset = next((ds for ds in datasets if ds['id'] == request.evaluation_dataset_id), None)
                    if eval_dataset:
                        eval_type = eval_dataset.get('type', 'unknown')
                        logger.info(f"평가용 데이터셋 발견: {request.evaluation_dataset_id} (타입: {eval_type})")
                        # 평가 모드를 데이터셋 타입에 맞게 자동 설정
                        if eval_type == 'universal_eval':
                            request.evaluation_mode = 'universal'
                        elif eval_type == 'domain_eval':
                            request.evaluation_mode = 'domain'
                    else:
                        logger.warning(f"평가용 데이터셋을 찾을 수 없음: {request.evaluation_dataset_id}")
                else:
                    logger.warning(f"LLM 서버 데이터셋 조회 실패: {response.status_code}")
            except Exception as e:
                logger.warning(f"평가용 데이터셋 정보 조회 실패: {e}")

        if not dataset_id:
            try:
                # 파인튜닝 모델인 경우 DB에서 dataset_id 조회
                if request.model_id != "nlpai-lab/KURE-v1":
                    conn = evaluator.embedding_service.get_db_connection()
                    cursor = conn.cursor()
                    cursor.execute("SELECT dataset_id FROM finetuned_models WHERE id = %s", (request.model_id,))
                    result_row = cursor.fetchone()
                    if result_row and result_row[0]:
                        dataset_id = result_row[0]
                        logger.info(f"DB에서 조회한 파인튜닝 모델 데이터셋 ID: {dataset_id}")
                    cursor.close()
                    evaluator.embedding_service.return_db_connection(conn)
                else:
                    # 기본 모델의 도메인 평가인 경우 최신 파인튜닝 모델의 dataset_id 사용
                    if request.evaluation_mode == "domain":
                        import os
                        import json
                        finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
                        if os.path.exists(finetuned_dir):
                            finetuned_models = [d for d in os.listdir(finetuned_dir)
                                              if os.path.isdir(os.path.join(finetuned_dir, d))]
                            if finetuned_models:
                                latest_model = sorted(finetuned_models)[-1]
                                metadata_path = os.path.join(finetuned_dir, latest_model, 'metadata.json')
                                if os.path.exists(metadata_path):
                                    with open(metadata_path, 'r', encoding='utf-8') as f:
                                        metadata = json.load(f)
                                    dataset_id = metadata.get('dataset')
                                    logger.info(f"기본 모델 도메인 평가를 위해 최신 파인튜닝 모델({latest_model})의 dataset_id({dataset_id}) 사용")
            except Exception as e:
                logger.warning(f"dataset_id 조회 실패: {e}")

        logger.info(f"평가에 사용할 데이터셋 ID: {dataset_id}")
        logger.info(f"평가 모드: {request.evaluation_mode}")
        if request.evaluation_dataset_id:
            logger.info(f"✅ 별도 평가용 데이터셋 사용: {request.evaluation_dataset_id}")
        else:
            logger.info(f"⚠️ 학습용 데이터셋으로 평가 (데이터 누수 위험): {dataset_id}")

        # 모델 평가 수행
        result = await evaluator.evaluate_model(request.model_id, dataset_id, request.evaluation_mode)
        
        if result.get('success'):
            logger.info(f"평가 성공: F1={result['metrics'].get('f1_score', 0):.2f}")
            return {
                "success": True,
                "model_id": request.model_id,
                "evaluation_results": result['metrics'],
                "test_samples": result.get('test_samples', 0),
                "evaluation_time": result.get('evaluation_time', 0)
            }
        else:
            logger.error(f"평가 실패: {result.get('error', 'Unknown error')}")
            return {
                "success": False,
                "error": result.get('error', 'Evaluation failed'),
                "model_id": request.model_id
            }
            
    except Exception as e:
        logger.error(f"임베딩 모델 평가 API 오류: {e}")
        return {
            "success": False,
            "error": f"Evaluation API error: {str(e)}",
            "model_id": request.model_id
        }

# DEBUG: 임시 테스트 엔드포인트
@app.post("/debug/test-training-docs")
async def debug_test_training_docs():
    """파인튜닝 문서 로드 테스트"""
    try:
        logger.info("DEBUG: 파인튜닝 문서 로드 테스트 시작")
        docs = await model_evaluator._get_training_documents()
        return {
            "success": True,
            "document_count": len(docs),
            "documents": docs[:3] if docs else [],  # 처음 3개만 반환
            "message": f"{len(docs)}개의 파인튜닝 문서 발견"
        }
    except Exception as e:
        logger.error(f"DEBUG: 파인튜닝 문서 로드 테스트 실패: {e}")
        import traceback
        logger.debug(f"상세 오류: {traceback.format_exc()}")
        return {
            "success": False,
            "error": str(e),
            "traceback": traceback.format_exc()
        }

# 시스템 자원 기반 최적화 설정 확인 API
@app.post("/models/analyze-resources")
async def analyze_system_resources():
    """시스템 자원 분석 및 최적화 설정 확인"""
    try:
        # 시스템 자원 분석
        resources = SystemResourceAnalyzer.analyze_system_resources()
        
        # 안전성 검사
        safety_check = SystemResourceAnalyzer._check_resource_safety(resources)
        
        # GPU 및 CPU 모드 모두 확인
        gpu_settings = SystemResourceAnalyzer.generate_optimized_settings(resources, force_cpu=False)
        cpu_settings = SystemResourceAnalyzer.generate_optimized_settings(resources, force_cpu=True)
        
        return {
            "status": "success",
            "system_resources": {
                "total_memory_gb": resources.total_memory_gb,
                "available_memory_gb": resources.available_memory_gb,
                "cpu_cores": resources.cpu_cores,
                "gpu_available": resources.gpu_available,
                "gpu_memory_gb": resources.gpu_memory_gb,
                "gpu_available_memory_gb": resources.gpu_available_memory_gb
            },
            "safety_assessment": {
                "is_safe": safety_check['is_safe'],
                "warning": safety_check['warning'],
                "warnings": safety_check['warnings'],
                "risk_level": "low" if safety_check['is_safe'] else "high"
            },
            "optimized_settings": {
                "gpu_mode": {
                    "batch_size": gpu_settings.batch_size,
                    "num_workers": gpu_settings.num_workers,
                    "pin_memory": gpu_settings.pin_memory,
                    "use_amp": gpu_settings.use_amp,
                    "gradient_accumulation_steps": gpu_settings.gradient_accumulation_steps,
                    "recommended_epochs": gpu_settings.recommended_epochs,
                    "warmup_ratio": gpu_settings.warmup_ratio
                },
                "cpu_mode": {
                    "batch_size": cpu_settings.batch_size,
                    "num_workers": cpu_settings.num_workers,
                    "pin_memory": cpu_settings.pin_memory,
                    "use_amp": cpu_settings.use_amp,
                    "gradient_accumulation_steps": cpu_settings.gradient_accumulation_steps,
                    "recommended_epochs": cpu_settings.recommended_epochs,
                    "warmup_ratio": cpu_settings.warmup_ratio
                }
            },
            "recommendations": {
                "preferred_mode": "cpu" if not safety_check['is_safe'] else ("gpu" if resources.gpu_available and resources.gpu_available_memory_gb > 2.0 else "cpu"),
                "estimated_training_time": "약 10-30분 (데이터셋 크기에 따라 달라짐)",
                "memory_efficient": resources.available_memory_gb > 12.0,
                "safety_tips": [
                    "파인튜닝 중 다른 메모리 집약적 프로그램 사용 금지",
                    "시스템 모니터링 도구로 리소스 사용량 확인",
                    "GPU 메모리 부족 시 자동으로 CPU 모드로 전환됨",
                    "비상 시 프로세스 종료 후 시스템 재시작 권장"
                ],
                "notes": [
                    f"현재 {resources.available_memory_gb:.1f}GB 메모리 사용 가능",
                    f"GPU 사용 가능: {resources.gpu_available}",
                    f"CPU 코어: {resources.cpu_cores}개",
                    f"안전성 상태: {'안전' if safety_check['is_safe'] else '위험'}"
                ]
            }
        }
        
    except Exception as e:
        return {
            "status": "error",
            "error": str(e),
            "message": "시스템 자원 분석 중 오류가 발생했습니다"
        }
    
# 환경설정 관리 API 엔드포인트들
@app.get("/config")
async def get_config():
    """현재 설정 반환"""
    try:
        from plugins.rag.rag_process import get_rag_settings
        
        settings = get_rag_settings()
        current_model = os.environ.get('RAG_EMBEDDING_MODEL', 'nlpai-lab/KURE-v1')
        
        # 검색 모드를 환경변수에서 직접 가져와서 확실하게 처리
        search_mode = os.environ.get('RAG_SEARCH_MODE', settings.get('search_mode', 'hybrid'))
        
        return {
            "status": "success",
            "config": {
                "current_embedding_model": current_model,
                "model_cache_dir": os.path.join(os.path.expanduser('~'), '.airun', 'models'),
                "finetuned_models_dir": os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned'),
                "semantic_chunker_model": settings.get("semantic_chunker_model", "snunlp/KR-SBERT-V40K-klueNLI-augSTS"),
                "search_mode": search_mode,
                "rag_server_url": os.environ.get('RAG_API_SERVER', 'http://localhost:5600'),
                "sync_settings": {
                    "sync_check_interval": SYNC_CHECK_INTERVAL,
                    "sync_error_retry_interval": SYNC_ERROR_RETRY_INTERVAL,
                    "sync_check_interval_minutes": SYNC_CHECK_INTERVAL // 60,
                    "sync_error_retry_interval_minutes": SYNC_ERROR_RETRY_INTERVAL // 60
                }
            }
        }
    except Exception as e:
        logger.error(f"설정 조회 실패: {str(e)}")
        return {
            "status": "error",
            "error": str(e)
        }

@app.post("/config/set-active-model")
async def set_active_model(request: Request):
    """활성 모델 변경"""
    try:
        data = await request.json()
        model_name = data.get('model_name')
        
        if not model_name:
            return JSONResponse(
                status_code=400,
                content={'error': '모델명이 필요합니다'}
            )
        
        # 환경변수 파일 경로 설정
        env_file_path = os.path.expanduser('~/.airun/.env')
        
        # 기존 환경변수 파일 읽기
        env_vars = {}
        if os.path.exists(env_file_path):
            with open(env_file_path, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    if line and not line.startswith('#') and '=' in line:
                        key, value = line.split('=', 1)
                        env_vars[key.strip()] = value.strip().strip('"\'')
        
        # RAG_EMBEDDING_MODEL 업데이트
        env_vars['RAG_EMBEDDING_MODEL'] = model_name
        
        # 디렉토리 생성
        os.makedirs(os.path.dirname(env_file_path), exist_ok=True)
        
        # 환경변수 파일 다시 쓰기
        with open(env_file_path, 'w', encoding='utf-8') as f:
            for key, value in env_vars.items():
                f.write(f'{key}={value}\n')

        # airun.conf 파일도 업데이트 (systemd 서비스가 다시 시작할 때 올바른 모델 로드)
        conf_file_path = os.path.expanduser('~/.airun/airun.conf')
        if os.path.exists(conf_file_path):
            with open(conf_file_path, 'r', encoding='utf-8') as f:
                conf_lines = f.readlines()

            with open(conf_file_path, 'w', encoding='utf-8') as f:
                for line in conf_lines:
                    if line.strip().startswith('export RAG_EMBEDDING_MODEL='):
                        f.write(f'export RAG_EMBEDDING_MODEL="{model_name}"\n')
                    else:
                        f.write(line)

        # 현재 프로세스 환경변수도 업데이트
        os.environ['RAG_EMBEDDING_MODEL'] = model_name
        
        log(f"임베딩 모델 변경 완료: {model_name}")
        
        return JSONResponse(content={
            'success': True,
            'message': f'활성 모델이 {model_name}으로 변경되었습니다',
            'current_model': model_name,
            'note': '변경사항을 완전히 적용하려면 RAG 서비스를 재시작하세요'
        })
        
    except Exception as e:
        logger.error(f"모델 활성화 실패: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={'error': f'모델 활성화 실패: {str(e)}'}
        )

@app.get("/config/embedding-model")
async def get_current_embedding_model():
    """현재 활성 임베딩 모델 정보 반환"""
    try:
        current_model = os.environ.get('RAG_EMBEDDING_MODEL', 'nlpai-lab/KURE-v1')
        
        # 모델이 파인튜닝된 모델인지 확인
        finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
        is_finetuned = False
        model_info = {}
        
        if os.path.exists(finetuned_dir):
            for model_dir in os.listdir(finetuned_dir):
                model_path = os.path.join(finetuned_dir, model_dir)
                if os.path.isdir(model_path):
                    metadata_path = os.path.join(model_path, 'metadata.json')
                    if os.path.exists(metadata_path):
                        try:
                            with open(metadata_path, 'r', encoding='utf-8') as f:
                                metadata = json.load(f)
                            
                            if metadata.get('model_name') == current_model or model_dir == current_model:
                                is_finetuned = True
                                model_info = metadata
                                break
                        except:
                            continue
        
        return {
            "status": "success",
            "current_model": current_model,
            "is_finetuned": is_finetuned,
            "model_info": model_info if is_finetuned else {},
            "model_cache_dir": os.path.join(os.path.expanduser('~'), '.airun', 'models')
        }
        
    except Exception as e:
        logger.error(f"현재 모델 정보 조회 실패: {str(e)}")
        return {
            "status": "error",
            "error": str(e)
        }

@app.get("/system/safety-status")
async def get_system_safety_status():
    """실시간 시스템 안전성 상태 확인"""
    try:
        # 시스템 자원 분석
        resources = SystemResourceAnalyzer.analyze_system_resources()
        
        # 안전성 검사
        safety_monitor = SafetyMonitor()
        is_safe, reason = safety_monitor.check_system_safety()
        
        # 추가 상세 정보
        import psutil
        
        # 메모리 사용률 계산
        memory_usage_percent = psutil.virtual_memory().percent
        
        # 디스크 사용률 계산
        disk_usage = psutil.disk_usage('/')
        disk_usage_percent = (disk_usage.used / disk_usage.total) * 100
        
        # CPU 사용률 (1초 평균)
        cpu_usage_percent = psutil.cpu_percent(interval=1)
        
        return {
            "success": True,
            "safety_status": {
                "is_safe": is_safe,
                "reason": reason,
                "risk_level": "low" if is_safe else "high",
                "timestamp": time.time()
            },
            "system_metrics": {
                "memory": {
                    "total_gb": resources.total_memory_gb,
                    "available_gb": resources.available_memory_gb,
                    "usage_percent": memory_usage_percent,
                    "status": "safe" if memory_usage_percent < 85 else "warning" if memory_usage_percent < 95 else "critical"
                },
                "gpu": {
                    "available": resources.gpu_available,
                    "memory_gb": resources.gpu_memory_gb,
                    "available_memory_gb": resources.gpu_available_memory_gb,
                    "usage_percent": ((resources.gpu_memory_gb - resources.gpu_available_memory_gb) / resources.gpu_memory_gb * 100) if resources.gpu_available and resources.gpu_memory_gb > 0 else 0
                },
                "cpu": {
                    "cores": resources.cpu_cores,
                    "usage_percent": cpu_usage_percent,
                    "status": "safe" if cpu_usage_percent < 80 else "warning" if cpu_usage_percent < 95 else "critical"
                },
                "disk": {
                    "total_gb": disk_usage.total / (1024**3),
                    "free_gb": disk_usage.free / (1024**3),
                    "usage_percent": disk_usage_percent,
                    "status": "safe" if disk_usage_percent < 85 else "warning" if disk_usage_percent < 95 else "critical"
                }
            },
            "recommendations": {
                "can_start_training": is_safe,
                "preferred_mode": "cpu" if not is_safe or not resources.gpu_available else "gpu",
                "suggested_batch_size": 1 if not is_safe else (4 if resources.gpu_available else 2),
                "warnings": [] if is_safe else [reason]
            }
        }
        
    except Exception as e:
        logger.error(f"시스템 안전성 상태 확인 실패: {e}")
        return {
            "success": False,
            "error": str(e),
            "message": "시스템 안전성 상태 확인 중 오류가 발생했습니다"
        }

@app.get("/system/training-resources")
async def get_training_resource_status():
    """학습 중 리소스 상태 모니터링"""
    try:
        # 현재 학습 상태 확인
        training_state = finetune_state_manager.load_state()
        is_training = training_state and training_state.get('status') in ['training', 'starting']
        
        # 적응형 리소스 매니저 상태 (학습 중인 경우)
        adaptive_info = {}
        if is_training and hasattr(fine_tuner, 'adaptive_manager'):
            adaptive_info = fine_tuner.adaptive_manager.get_resource_summary()
        
        # 시스템 리소스 분석
        resources = SystemResourceAnalyzer.analyze_system_resources()
        
        # 메모리 사용률 추세 분석
        import psutil
        memory_percent = psutil.virtual_memory().percent
        
        # 권장 설정 계산
        recommended_batch_size = 1
        if memory_percent < 70:
            recommended_batch_size = 8 if resources.gpu_available else 4
        elif memory_percent < 80:
            recommended_batch_size = 4 if resources.gpu_available else 2
        elif memory_percent < 90:
            recommended_batch_size = 2
        
        return {
            "success": True,
            "training_status": {
                "is_training": is_training,
                "current_status": training_state.get('status', 'idle') if training_state else 'idle',
                "model_name": training_state.get('model_name', '') if training_state else '',
                "dataset_id": training_state.get('dataset_id', '') if training_state else ''
            },
            "adaptive_resource_info": adaptive_info,
            "current_resources": {
                "memory_usage_percent": memory_percent,
                "available_memory_gb": resources.available_memory_gb,
                "gpu_available": resources.gpu_available,
                "gpu_memory_percent": ((resources.gpu_memory_gb - resources.gpu_available_memory_gb) / resources.gpu_memory_gb * 100) if resources.gpu_available and resources.gpu_memory_gb > 0 else 0,
                "gpu_available_memory_gb": resources.gpu_available_memory_gb
            },
            "recommendations": {
                "optimal_batch_size": recommended_batch_size,
                "should_use_gradient_accumulation": memory_percent > 85,
                "suggested_gradient_accumulation_steps": max(1, 8 // recommended_batch_size),
                "memory_optimization_tips": [
                    "메모리 사용률 85% 이상 시 배치 크기 감소 권장",
                    "GPU 메모리 부족 시 자동 CPU 모드 전환",
                    "그래디언트 누적으로 효과적인 배치 크기 유지",
                    "체크포인트 저장 빈도 조정으로 메모리 절약"
                ]
            },
            "real_time_metrics": {
                "timestamp": time.time(),
                "memory_trend": adaptive_info.get('memory_trend', 'unknown'),
                "resource_pressure": "high" if memory_percent > 85 else "medium" if memory_percent > 70 else "low"
            }
        }
        
    except Exception as e:
        logger.error(f"학습 리소스 상태 확인 실패: {e}")
        return {
            "success": False,
            "error": str(e),
            "message": "학습 리소스 상태 확인 중 오류가 발생했습니다"
        }

class ModelEvaluator:
    """모델 성능 평가 클래스"""
    
    def __init__(self):
        # PostgreSQL 기반 임베딩 서비스
        self.embedding_service = EmbeddingService()
        self.COLLECTION_NAME = "airun_docs"
        # 데이터베이스 연결 풀 (임베딩 서비스에서 가져오기)
        self.db_pool = self.embedding_service.db_pool if hasattr(self.embedding_service, 'db_pool') else None
        
        # 테스트용 기본 쿼리들 (폴백용)
        self.test_queries = [
            "인공지능의 정의는 무엇인가요?",
            "머신러닝과 딥러닝의 차이점을 설명해주세요.",
            "자연어 처리 기술의 응용 분야는?",
            "빅데이터 분석의 중요성은?",
            "클라우드 컴퓨팅의 장점은?",
            "사이버 보안의 필요성은?",
            "블록체인 기술의 원리는?",
            "IoT 기술의 활용 사례는?",
            "5G 통신의 특징은?",
            "양자 컴퓨팅의 가능성은?",
            "데이터 마이닝 기법에는 어떤 것들이 있나요?",
            "웹 개발에서 프론트엔드와 백엔드의 역할은?",
            "데이터베이스 정규화란 무엇인가요?",
            "소프트웨어 테스팅의 종류는?",
            "애자일 개발 방법론의 특징은?"
        ]
        
    async def evaluate_model(self, model_id: str, dataset_id: str = None, evaluation_mode: str = "universal") -> Dict[str, Any]:
        """모델 성능 평가 수행"""
        try:
            logger.info(f"======== 모델 평가 시작: {model_id} ========")

            # 범용 평가 모드에서는 dataset_id를 universal_benchmark로 설정
            if evaluation_mode == "universal" and not dataset_id:
                dataset_id = "universal_benchmark"

            # 1. 평가용 컬렉션 생성/확인
            logger.info(f"1. 평가용 컬렉션 준비 중...")
            eval_collection_name = await self._ensure_evaluation_collection(model_id, dataset_id, evaluation_mode)
            if not eval_collection_name:
                logger.error("평가용 컬렉션 생성 실패")
                return {
                    "success": False,
                    "error": "평가용 컬렉션 생성 실패",
                    "model_id": model_id
                }
            
            # 2. 평가용 데이터셋 준비
            logger.info(f"2. 테스트 데이터셋 준비 중... (dataset_id: {dataset_id})")
            test_data = await self._prepare_test_dataset(dataset_id, evaluation_mode)
            logger.info(f"   데이터셋 기반 테스트 데이터: {len(test_data)}개")
            
            if not test_data:
                logger.warning("테스트 데이터셋이 없어서 기본 쿼리로 평가")
                test_data = self._prepare_default_test_data()
                logger.info(f"   기본 테스트 데이터: {len(test_data)}개")
            
            if not test_data:
                logger.error("테스트 데이터가 전혀 없습니다!")
                return {
                    "success": False,
                    "error": "테스트 데이터가 없습니다",
                    "model_id": model_id
                }
            
            # 3. 모델 활성화 (필요한 경우)
            logger.info(f"3. 모델 로드 확인 중: {model_id}")
            await self._ensure_model_loaded(model_id)
            
            # 4. 지정된 모델로 평가 수행 (전용 테이블 사용)
            logger.info(f"4. 지정된 모델({model_id})로 평가 수행 중... ({len(test_data)}개 테스트 케이스)")
            metrics = await self._run_evaluation_with_model(test_data, model_id, eval_collection_name)
            
            logger.info(f"======== 모델 평가 완료: {model_id} ========")
            logger.info(f"F1 Score: {metrics.get('f1_score', 0):.2f}")
            logger.info(f"정밀도: {metrics.get('precision', 0):.2f}")
            logger.info(f"재현율: {metrics.get('recall', 0):.2f}")
            logger.info(f"평가 시간: {metrics.get('evaluation_time', 0):.2f}초")
            
            # 5. 평가 완료 후 전용 테이블 삭제
            logger.info(f"5. 평가 완료 후 전용 테이블 삭제 중...")
            try:
                await self._drop_finetuned_model_table(eval_collection_name)
                logger.info("파인튜닝된 모델 전용 테이블 삭제 완료")
            except Exception as cleanup_error:
                logger.warning(f"전용 테이블 삭제 실패 (무시): {cleanup_error}")
            
            return {
                "success": True,
                "model_id": model_id,
                "metrics": metrics,
                "test_samples": len(test_data),
                "evaluation_time": metrics.get('evaluation_time', 0),
                "evaluation_collection": eval_collection_name
            }
            
        except Exception as e:
            logger.error(f"모델 평가 실패 ({model_id}): {e}")
            
            # 평가 실패 시에도 전용 테이블 정리
            try:
                if 'eval_collection_name' in locals() and eval_collection_name:
                    await self._drop_finetuned_model_table(eval_collection_name)
                    logger.info("평가 실패 시 전용 테이블 삭제 완료")
            except Exception as cleanup_error:
                logger.warning(f"평가 실패 시 전용 테이블 삭제 실패 (무시): {cleanup_error}")
            
            return {
                "success": False,
                "error": str(e),
                "model_id": model_id
            }
    
    async def _prepare_test_dataset(self, dataset_id: str = None, evaluation_mode: str = "universal") -> List[Dict[str, Any]]:
        """홀드아웃 검증을 위한 테스트 데이터셋 준비"""
        try:
            logger.info(f"======== 홀드아웃 검증용 테스트 데이터셋 준비 ========")

            # ~/.airun/datasets/에서 데이터셋 파일 찾기
            import glob
            import random
            dataset_files = glob.glob(os.path.expanduser("~/.airun/datasets/*.json"))

            # dataset_id가 있으면 해당 파일을 우선적으로 찾기
            target_file = None
            if dataset_id:
                target_file_path = os.path.expanduser(f"~/.airun/datasets/{dataset_id}.json")
                if os.path.exists(target_file_path):
                    target_file = target_file_path
                    logger.info(f"dataset_id에 맞는 파일 발견: {target_file}")
                else:
                    logger.warning(f"dataset_id {dataset_id}에 해당하는 파일 없음: {target_file_path}")

            # dataset_id로 찾지 못했으면 평가 모드에 따라 데이터셋 선택
            if not target_file:
                if evaluation_mode == "universal":
                    # 범용 평가 모드: universal_benchmark.json 사용
                    benchmark_file = os.path.expanduser("~/.airun/datasets/universal_benchmark.json")
                    if os.path.exists(benchmark_file):
                        target_file = benchmark_file
                        logger.info(f"범용 벤치마크 평가 모드 - 데이터셋 사용: {target_file}")
                elif evaluation_mode == "domain":
                    # 도메인 전문성 평가 모드: 모델별 전용 데이터셋 사용
                    logger.info(f"도메인 전문성 평가 모드 - dataset_id: {dataset_id} 기반 홀드아웃 사용")
                    # 기존 로직 계속 실행 (아래 fallback 로직 사용)
                
                # 2. balanced_test 데이터셋 차순위
                if not target_file:
                    for file_path in dataset_files:
                        filename = os.path.basename(file_path)
                        if 'balanced_test' in filename:
                            target_file = file_path
                            logger.info(f"balanced_test 데이터셋 사용: {target_file}")
                            break

                # 3. test_dataset 이름 우선
                if not target_file:
                    for file_path in dataset_files:
                        filename = os.path.basename(file_path)
                        if 'test_dataset' in filename:
                            target_file = file_path
                            break

                # 4. document_based 타입 (기존 RAG 데이터)
                if not target_file:
                    for file_path in dataset_files:
                        try:
                            with open(file_path, 'r', encoding='utf-8') as f:
                                data = json.load(f)
                                if data.get('type') == 'document_based':
                                    target_file = file_path
                                    logger.info(f"document_based 타입 데이터셋 사용: {target_file}")
                                    break
                        except:
                            continue

                # 5. file_upload 타입 우선
                if not target_file:
                    for file_path in dataset_files:
                        try:
                            with open(file_path, 'r', encoding='utf-8') as f:
                                data = json.load(f)
                                if data.get('type') == 'file_upload':
                                    target_file = file_path
                                    logger.info(f"file_upload 타입 데이터셋 사용: {target_file}")
                                    break
                        except:
                            continue

                # 6. 최후 수단: 가장 최근 파일
                if not target_file and dataset_files:
                    target_file = max(dataset_files, key=os.path.getmtime)
                    logger.info(f"가장 최근 파일 사용: {target_file}")

            if not target_file:
                logger.warning("파인튜닝 데이터셋 파일을 찾을 수 없음, 기본 테스트 사용")
                return self._prepare_default_test_data()

            logger.info(f"원본 데이터셋 로드: {target_file}")

            with open(target_file, 'r', encoding='utf-8') as f:
                dataset = json.load(f)

            # 전체 예제 수집
            all_examples = dataset.get('examples', [])
            total_count = len(all_examples)

            if total_count == 0:
                logger.warning("데이터셋에 예제가 없음")
                return self._prepare_default_test_data()

            logger.info(f"전체 데이터 수: {total_count}개")

            # 홀드아웃 비율 계산 (20% 또는 최소 1개, 최대 전체의 절반)
            holdout_ratio = 0.2
            min_holdout = 1
            max_holdout = max(1, total_count // 2)

            holdout_count = max(min_holdout, min(max_holdout, int(total_count * holdout_ratio)))
            training_count = total_count - holdout_count

            logger.info(f"홀드아웃 검증 설정:")
            logger.info(f"  - 전체: {total_count}개")
            logger.info(f"  - 학습용: {training_count}개 ({(training_count/total_count)*100:.1f}%)")
            logger.info(f"  - 평가용: {holdout_count}개 ({(holdout_count/total_count)*100:.1f}%)")

            # 평가용 데이터 생성 - 홀드아웃 데이터 확인
            holdout_file = target_file.replace('.json', '_holdout.json')

            if os.path.exists(holdout_file):
                # 기존 홀드아웃 파일이 있으면 사용
                logger.info(f"기존 홀드아웃 파일 사용: {holdout_file}")
                with open(holdout_file, 'r', encoding='utf-8') as f:
                    holdout_data = json.load(f)
                test_examples = holdout_data.get('holdout_examples', [])
            else:
                # 새로 홀드아웃 데이터 생성
                logger.info(f"새 홀드아웃 데이터 생성 (무작위 선택)")

                # 무작위로 평가용 데이터 선택
                random.seed(42)  # 재현 가능한 결과를 위해 시드 고정
                holdout_indices = set(random.sample(range(total_count), holdout_count))
                test_examples = [all_examples[i] for i in holdout_indices]

                # 홀드아웃 파일 저장
                holdout_data = {
                    'dataset_id': dataset_id,
                    'original_file': target_file,
                    'total_count': total_count,
                    'holdout_count': holdout_count,
                    'holdout_indices': list(holdout_indices),
                    'holdout_examples': test_examples,
                    'created_at': datetime.now().isoformat()
                }

                with open(holdout_file, 'w', encoding='utf-8') as f:
                    json.dump(holdout_data, f, ensure_ascii=False, indent=2)

                logger.info(f"홀드아웃 파일 저장: {holdout_file}")

            # 테스트 케이스로 변환
            test_data = []
            for example in test_examples:
                query = example.get('query', '')

                # 1. Positive 케이스 처리
                positive_document = ''
                if 'document' in example:
                    positive_document = example['document']
                elif 'positive' in example:
                    positive_document = example['positive']
                elif 'answer' in example:
                    positive_document = example['answer']

                # 점수 추출 (기본값: 1.0)
                score = example.get('score', 1.0)

                # 유효한 쿼리와 positive 문서가 있는 경우 positive 테스트 추가
                if query and positive_document:
                    test_data.append({
                        'query': query,
                        'positive': positive_document,
                        'expected_relevant': True,  # positive는 항상 관련있음
                        'expected_score': score,
                        'use_rag_search': True,
                        'source': 'holdout_validation_positive'
                    })

                # 2. Negative 케이스 처리 (있는 경우)
                if 'negative' in example and example['negative']:
                    negative_document = example['negative']
                    if query and negative_document:
                        test_data.append({
                            'query': query,
                            'positive': negative_document,  # 평가 시스템에서는 'positive' 필드를 사용
                            'expected_relevant': False,  # negative는 항상 관련없음
                            'expected_score': 0.0,  # negative 케이스는 0점
                            'use_rag_search': True,
                            'source': 'holdout_validation_negative'
                        })

            logger.info(f"홀드아웃 검증용 테스트 케이스: {len(test_data)}개")
            logger.info(f"   관련 있는 쿼리: {sum(1 for t in test_data if t['expected_relevant'])}개")
            logger.info(f"   관련 없는 쿼리: {sum(1 for t in test_data if not t['expected_relevant'])}개")
            logger.info(f"======== 홀드아웃 검증 데이터 준비 완료 ========")

            return test_data
                
        except Exception as e:
            logger.error(f"파인튜닝 기반 테스트 데이터셋 준비 실패: {e}")
            return self._prepare_default_test_data()
    
    def _get_collection_questions(self) -> List[str]:
        """실제 RAG 컬렉션에서 추출된 질문들 가져오기"""
        try:
            # PostgreSQL에서 문서 메타데이터 조회
            collection = self.embedding_service.collection
            
            # 모든 문서의 메타데이터 가져오기
            results = collection.get(include=['metadatas'])
            
            questions = []
            seen_questions = set()  # 중복 제거용
            
            for metadata in results['metadatas']:
                if 'extracted_questions' in metadata:
                    try:
                        # JSON 문자열을 파싱하여 질문 리스트 추출
                        extracted_questions = json.loads(metadata['extracted_questions'])
                        
                        for question in extracted_questions:
                            # 중복 제거 및 최소 길이 확인
                            if question not in seen_questions and len(question.strip()) > 10:
                                questions.append(question.strip())
                                seen_questions.add(question)
                                
                    except (json.JSONDecodeError, TypeError) as e:
                        logger.warning(f"질문 파싱 실패: {e}")
                        continue
            
            logger.info(f"컬렉션에서 {len(questions)}개의 고유 질문 추출")
            return questions[:20]  # 최대 20개만 반환
            
        except Exception as e:
            logger.error(f"컬렉션 질문 추출 실패: {e}")
            return []
    
    async def _get_training_documents(self) -> list:
        """파인튜닝 데이터셋에서 documents 가져오기"""
        try:
            import glob
            dataset_files = glob.glob(os.path.expanduser("~/.airun/datasets/*.json"))
            logger.info(f"DEBUG: 발견된 데이터셋 파일들: {dataset_files}")

            # 가장 최근 파일 또는 특정 파일 찾기
            target_file = None
            for file_path in dataset_files:
                filename = os.path.basename(file_path)
                if 'test_dataset' in filename:  # 테스트 데이터셋 우선
                    target_file = file_path
                    break

            if not target_file and dataset_files:
                # 테스트 데이터셋이 없으면 가장 최근 파일
                target_file = max(dataset_files, key=os.path.getmtime)
                logger.info(f"DEBUG: 최근 파일 선택: {target_file}")

            if not target_file:
                logger.error("파인튜닝 데이터셋 파일을 찾을 수 없음")
                return []
            
            logger.info(f"   파인튜닝 데이터셋 로드: {target_file}")
            
            with open(target_file, 'r', encoding='utf-8') as f:
                dataset = json.load(f)
            
            # examples에서 document들 추출 (다양한 키 형식 지원)
            documents = []
            logger.info(f"DEBUG: 데이터셋 예제 수: {len(dataset.get('examples', []))}")
            if 'examples' in dataset:
                for i, example in enumerate(dataset['examples'][:10]):  # 처음 10개만 디버그
                    logger.info(f"DEBUG: 예제 {i} 키들: {list(example.keys())}")
                    # 우선순위: document > positive > answer > query
                    if 'document' in example:
                        doc_content = example['document'].strip()
                        if len(doc_content) > 10:  # 최소 길이 처리
                            documents.append(doc_content)
                            logger.info(f"DEBUG: 문서 {i} 추가 ({len(doc_content)}자)")
                    elif 'positive' in example:
                        documents.append(example['positive'])
                    elif 'answer' in example:
                        documents.append(example['answer'])
                    elif 'query' in example:  # 질문도 문서로 사용 가능
                        query_content = example['query'].strip()
                        if len(query_content) > 50:  # 충분히 긴 질문만 사용
                            documents.append(query_content)
                            logger.info(f"DEBUG: 쿼리 {i} 문서로 추가 ({len(query_content)}자)")
            
            logger.info(f"   파인튜닝 문서 {len(documents)}개 추출 완료")
            return documents

        except Exception as e:
            logger.error(f"파인튜닝 문서 로드 실패: {e}")
            import traceback
            logger.debug(f"상세 오류: {traceback.format_exc()}")
            return []

    async def _create_base_model_evaluation_table(self, table_name: str) -> bool:
        """기본 모델 범용 평가를 위한 임시 테이블 생성"""
        try:
            # universal_benchmark 데이터 로드
            benchmark_file = os.path.expanduser("~/.airun/datasets/universal_benchmark.json")
            if not os.path.exists(benchmark_file):
                logger.error(f"범용 벤치마크 파일이 없음: {benchmark_file}")
                return False

            with open(benchmark_file, 'r', encoding='utf-8') as f:
                benchmark_data = json.load(f)

            # 임시 테이블 생성
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()

            # 테이블 삭제 (기존 것이 있다면)
            cursor.execute(f"DROP TABLE IF EXISTS {table_name}")

            # 테이블 생성
            cursor.execute(f"""
                CREATE TABLE {table_name} (
                    id SERIAL PRIMARY KEY,
                    chunk_text TEXT NOT NULL,
                    metadata JSONB DEFAULT '{{}}',
                    embedding vector(1024),
                    user_id TEXT DEFAULT 'system',
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """)

            # 기본 모델로 임베딩 생성
            from sentence_transformers import SentenceTransformer
            base_model = SentenceTransformer("nlpai-lab/KURE-v1")

            # positive + negative 데이터 임베딩
            embedding_count = 0
            for item in benchmark_data.get('examples', []):
                positive_text = item.get('positive', '')
                negative_text = item.get('negative', '')

                if positive_text:
                    embedding = base_model.encode(positive_text).tolist()
                    cursor.execute(f"""
                        INSERT INTO {table_name} (chunk_text, metadata, embedding)
                        VALUES (%s, %s, %s)
                    """, (positive_text, '{}', embedding))
                    embedding_count += 1

                if negative_text:
                    embedding = base_model.encode(negative_text).tolist()
                    cursor.execute(f"""
                        INSERT INTO {table_name} (chunk_text, metadata, embedding)
                        VALUES (%s, %s, %s)
                    """, (negative_text, '{}', embedding))
                    embedding_count += 1

            conn.commit()
            cursor.close()
            conn.close()

            logger.info(f"기본 모델 범용 평가 테이블 생성 완료: {table_name} ({embedding_count}개 임베딩)")
            return True

        except Exception as e:
            logger.error(f"기본 모델 범용 평가 테이블 생성 실패: {e}")
            return False

    async def _create_base_model_domain_evaluation_table(self, table_name: str, dataset_id: str) -> bool:
        """기본 모델의 도메인별 홀드아웃 평가를 위한 임시 테이블 생성"""
        try:
            logger.info(f"기본 모델 도메인 평가 테이블 생성: {table_name} (dataset_id: {dataset_id})")

            # DB 연결 (파인튜닝 모델과 동일한 방식)
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()

            # 기존 테이블이 있으면 삭제
            cursor.execute(f"DROP TABLE IF EXISTS {table_name}")

            # 1. 테이블 생성
            create_table_sql = f"""
            CREATE TABLE {table_name} (
                id SERIAL PRIMARY KEY,
                is_embedding BOOLEAN NOT NULL DEFAULT true,
                doc_id VARCHAR(500) NOT NULL,
                filename VARCHAR(500) NOT NULL,
                chunk_index INTEGER NOT NULL DEFAULT 0,
                chunk_text TEXT NOT NULL,
                embedding vector(1024),
                image_embedding vector(512),
                user_id VARCHAR(255),
                source TEXT,
                file_mtime BIGINT,
                metadata JSONB DEFAULT '{{}}'::jsonb,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
            """
            cursor.execute(create_table_sql)
            logger.info(f"도메인 평가 테이블 생성 완료: {table_name}")

            # 2. 홀드아웃 데이터가 아닌 학습 데이터만 임베딩
            # 해당 dataset_id의 홀드아웃 파일에서 학습용 데이터만 추출
            dataset_file = os.path.expanduser(f"~/.airun/datasets/{dataset_id}.json")
            holdout_file = os.path.expanduser(f"~/.airun/datasets/{dataset_id}_holdout.json")

            if not os.path.exists(dataset_file):
                logger.error(f"데이터셋 파일 없음: {dataset_file}")
                return False

            # 원본 데이터셋 로드
            with open(dataset_file, 'r', encoding='utf-8') as f:
                dataset_data = json.load(f)

            # 홀드아웃 인덱스 로드 (있는 경우)
            holdout_indices = set()
            if os.path.exists(holdout_file):
                with open(holdout_file, 'r', encoding='utf-8') as f:
                    holdout_data = json.load(f)
                holdout_indices = set(holdout_data.get('holdout_indices', []))
                logger.info(f"기존 홀드아웃 인덱스 사용: {len(holdout_indices)}개")

            # 기본 모델로 임베딩 생성 (범용 평가와 동일한 간단한 방식)
            from sentence_transformers import SentenceTransformer
            base_model = SentenceTransformer("nlpai-lab/KURE-v1")

            # positive + negative 데이터 임베딩 (범용 평가와 동일한 방식)
            embedding_count = 0
            examples = dataset_data.get('examples', [])
            
            for idx, item in enumerate(examples):
                # 모든 데이터를 검색 테이블에 포함 (홀드아웃 데이터도 포함)
                # 홀드아웃은 학습에서만 제외하고, 검색에는 포함해야 정답을 찾을 수 있음

                positive_text = item.get('positive', '')
                negative_text = item.get('negative', '')

                if positive_text:
                    embedding = base_model.encode(positive_text).tolist()
                    cursor.execute(f"""
                        INSERT INTO {table_name} (doc_id, filename, chunk_text, embedding, source, metadata)
                        VALUES (%s, %s, %s, %s, %s, %s)
                    """, (f"domain_eval_{dataset_id}_{idx}_pos", f"domain_evaluation_{dataset_id}_pos_{idx}", positive_text, embedding, f"domain_evaluation_{dataset_id}", '{}'))
                    embedding_count += 1

                if negative_text:
                    embedding = base_model.encode(negative_text).tolist()
                    cursor.execute(f"""
                        INSERT INTO {table_name} (doc_id, filename, chunk_text, embedding, source, metadata)
                        VALUES (%s, %s, %s, %s, %s, %s)
                    """, (f"domain_eval_{dataset_id}_{idx}_neg", f"domain_evaluation_{dataset_id}_neg_{idx}", negative_text, embedding, f"domain_evaluation_{dataset_id}", '{}'))
                    embedding_count += 1

            conn.commit()
            cursor.close()
            conn.close()

            logger.info(f"기본 모델 도메인 평가 테이블 생성 완료: {table_name} ({embedding_count}개 임베딩)")
            return True

        except Exception as e:
            logger.error(f"기본 모델 도메인 평가 테이블 생성 실패: {e}")
            return False

    async def _ensure_evaluation_collection(self, model_id: str, dataset_id: str = None, evaluation_mode: str = "universal") -> str:
        """모델 평가용 컬렉션 준비 (기본 모델 또는 파인튜닝된 모델)"""
        logger.info(f"모델 평가 컬렉션 준비 - model_id: {model_id}, dataset_id: {dataset_id}, evaluation_mode: {evaluation_mode}")

        try:
            # 기본 모델인 경우 범용 평가 여부에 따라 분기
            if model_id == "nlpai-lab/KURE-v1":
                # 범용 평가인 경우 universal_benchmark 데이터를 임시 테이블에 임베딩
                if dataset_id == "universal_benchmark":
                    logger.info("기본 모델 범용 평가: universal_benchmark 데이터를 임시 테이블에 임베딩")
                    table_name = "base_model_universal_evaluation"
                    await self._create_base_model_evaluation_table(table_name)
                    return table_name
                elif evaluation_mode == "domain":
                    # 도메인 평가인 경우 dataset_id가 없어도 임시 테이블 생성
                    # 최신 파인튜닝 모델의 dataset_id를 찾아서 사용
                    if not dataset_id:
                        try:
                            finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
                            if os.path.exists(finetuned_dir):
                                finetuned_models = [d for d in os.listdir(finetuned_dir)
                                                  if os.path.isdir(os.path.join(finetuned_dir, d))]
                                if finetuned_models:
                                    latest_model = sorted(finetuned_models)[-1]
                                    metadata_path = os.path.join(finetuned_dir, latest_model, 'metadata.json')
                                    if os.path.exists(metadata_path):
                                        with open(metadata_path, 'r', encoding='utf-8') as f:
                                            metadata = json.load(f)
                                        dataset_id = metadata.get('dataset')
                                        logger.info(f"기본 모델 도메인 평가: 최신 파인튜닝 모델({latest_model})의 dataset_id({dataset_id}) 사용")
                        except Exception as e:
                            logger.error(f"기본 모델 도메인 평가를 위한 dataset_id 추출 실패: {e}")

                    if dataset_id:
                        logger.info(f"기본 모델 도메인 평가: dataset_id '{dataset_id}' 홀드아웃 데이터로 임시 테이블 생성")
                        table_name = f"base_model_domain_{dataset_id}_evaluation"
                        await self._create_base_model_domain_evaluation_table(table_name, dataset_id)
                        return table_name
                    else:
                        logger.warning("기본 모델 도메인 평가: dataset_id를 찾을 수 없어 기존 컬렉션 사용")
                        return "existing_collection"
                else:
                    logger.info("기본 모델 평가: 기존 임베딩 컬렉션 사용")
                    return "existing_collection"  # 기존 컬렉션 사용 신호

            # 파인튜닝된 모델인 경우
            # 범용 평가라면 기본 모델과 동일하게 universal_benchmark 사용
            if dataset_id == "universal_benchmark":
                logger.info("파인튜닝 모델 범용 평가: universal_benchmark 데이터를 임시 테이블에 임베딩")
                table_name = "finetuned_model_universal_evaluation"
                await self._create_base_model_evaluation_table(table_name)
                return table_name
            
            # 도메인 특화 평가인 경우 전용 테이블 생성
            model_path = await self._get_finetuned_model_path(model_id)
            if not model_path:
                logger.error(f"파인튜닝된 모델 경로를 찾을 수 없음: {model_id}")
                return None

            logger.info(f"파인튜닝된 모델 경로: {model_path}")

            # dataset_id가 없으면 DB에서 모델의 dataset_id 조회
            if not dataset_id:
                try:
                    conn = self.embedding_service.get_db_connection()
                    cursor = conn.cursor()
                    cursor.execute("SELECT dataset_id FROM finetuned_models WHERE id = %s", (model_id,))
                    result_row = cursor.fetchone()
                    if result_row and result_row[0]:
                        dataset_id = result_row[0]
                        logger.info(f"DB에서 조회한 데이터셋 ID: {dataset_id}")
                    cursor.close()
                    conn.close()
                except Exception as e:
                    logger.warning(f"dataset_id 조회 실패: {e}")

            if not dataset_id:
                logger.error(f"dataset_id를 찾을 수 없음: {model_id}")
                return None

            # 2. 평가용 임베딩 테이블 생성
            table_name = await self._create_finetuned_model_table(model_id)
            if not table_name:
                logger.error(f"평가용 임베딩 테이블 생성 실패: {model_id}")
                return None

            logger.info(f"평가용 임베딩 테이블 생성 완료: {table_name}")

            # 3. 파인튜닝된 모델로 학습 데이터셋 임베딩
            logger.info(f"테이블 '{table_name}' 준비 완료, 학습 데이터셋을 파인튜닝된 모델로 임베딩 시작")
            embed_count = await self._embed_training_dataset_with_finetuned_model(
                model_path, table_name, dataset_id
            )

            if embed_count == 0:
                logger.error(f"학습 데이터셋 임베딩 실패: {model_id}")
                await self._drop_finetuned_model_table(table_name)
                return None

            logger.info(f"파인튜닝된 모델로 학습 데이터셋 임베딩 완료: {embed_count}개 (positive + negative)")
            
            # 평가용 테이블명 반환 (평가 수행 후 삭제됨)
            return table_name
            
        except Exception as e:
            logger.error(f"파인튜닝된 모델 평가 준비 실패: {e}")
            import traceback
            logger.debug(f"상세 오류: {traceback.format_exc()}")
            return None

    async def _search_in_existing_collection(self, query_embedding: List[float]) -> float:
        """기존 임베딩 컬렉션에서 검색하여 최고 유사도 반환"""
        try:
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()

            # 기존 임베딩 테이블에서 유사도 검색
            cursor.execute("""
                SELECT MAX(1 - (embedding <=> %s::vector)) as similarity
                FROM document_embeddings
                LIMIT 1
            """, (query_embedding,))

            result = cursor.fetchone()
            cursor.close()
            conn.close()

            if result and result[0] is not None:
                return float(result[0])
            else:
                return 0.0

        except Exception as e:
            logger.error(f"기존 컬렉션 검색 실패: {e}")
            return 0.0

    async def _get_finetuned_model_path(self, model_id: str) -> str:
        """모델 경로 반환 (기본 모델 또는 파인튜닝된 모델)"""
        try:
            # 기본 모델인 경우 HuggingFace model ID 직접 반환
            if model_id == "nlpai-lab/KURE-v1":
                logger.info(f"기본 모델 사용: {model_id}")
                return model_id

            # 파인튜닝된 모델인 경우 로컬 경로 조회
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()

            cursor.execute("""
                SELECT output_path
                FROM finetuned_models
                WHERE id = %s AND status = 'completed'
            """, (model_id,))

            result = cursor.fetchone()
            cursor.close()
            conn.close()

            if result:
                model_path = result[0]
                logger.info(f"파인튜닝된 모델 경로 발견: {model_path}")
                return model_path
            else:
                logger.error(f"완료된 파인튜닝 모델을 찾을 수 없음: {model_id}")
                return None

        except Exception as e:
            logger.error(f"모델 경로 조회 실패: {e}")
            return None

    async def _get_training_dataset_path(self, model_id: str) -> str:
        """학습에 사용된 데이터셋 경로 반환"""
        try:
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()
            
            # finetuned_models 테이블에서 데이터셋 ID 조회
            cursor.execute("""
                SELECT dataset_id 
                FROM finetuned_models 
                WHERE id = %s
            """, (model_id,))
            
            result = cursor.fetchone()
            if not result:
                logger.error(f"모델의 데이터셋 ID를 찾을 수 없음: {model_id}")
                cursor.close()
                conn.close()
                return None
                
            dataset_id = result[0]
            
            # datasets 테이블에서 데이터셋 파일 경로 조회
            cursor.execute("""
                SELECT file_path 
                FROM datasets 
                WHERE id = %s
            """, (dataset_id,))
            
            result = cursor.fetchone()
            cursor.close()
            conn.close()
            
            if result:
                dataset_path = result[0]
                logger.info(f"학습 데이터셋 경로 발견: {dataset_path}")
                return dataset_path
            else:
                logger.error(f"데이터셋 파일을 찾을 수 없음: {dataset_id}")
                return None
                
        except Exception as e:
            logger.error(f"학습 데이터셋 경로 조회 실패: {e}")
            return None

    async def _create_finetuned_model_table(self, model_id: str) -> str:
        """파인튜닝된 모델 전용 임베딩 테이블 생성"""
        try:
            # 안전한 테이블명 생성
            safe_model_name = model_id.replace('/', '_').replace('-', '_').replace('.', '_').lower()
            table_name = f"{safe_model_name}_finetuned_embeddings"
            
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()
            
            # 기존 테이블이 있으면 삭제
            cursor.execute(f"DROP TABLE IF EXISTS {table_name}")
            
            # 새 테이블 생성 (document_embeddings와 동일한 구조)
            create_table_sql = f"""
                CREATE TABLE {table_name} (
                    id SERIAL PRIMARY KEY,
                    is_embedding BOOLEAN NOT NULL DEFAULT true,
                    doc_id VARCHAR(500) NOT NULL,
                    filename VARCHAR(500) NOT NULL,
                    chunk_index INTEGER NOT NULL DEFAULT 0,
                    chunk_text TEXT NOT NULL,
                    embedding vector(1024),
                    image_embedding vector(512),
                    user_id VARCHAR(255),
                    source TEXT,
                    file_mtime BIGINT,
                    metadata JSONB DEFAULT '{{}}'::jsonb,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """
            cursor.execute(create_table_sql)
            
            # 인덱스 생성
            cursor.execute(f"""
                CREATE INDEX {table_name}_user_id_idx ON {table_name} (user_id)
            """)
            cursor.execute(f"""
                CREATE INDEX {table_name}_embedding_idx ON {table_name} 
                USING ivfflat (embedding vector_cosine_ops) WITH (lists = 100)
            """)
            
            # 테이블 생성 확인을 위한 디버깅
            cursor.execute(f"SELECT column_name FROM information_schema.columns WHERE table_name = '{table_name}' ORDER BY ordinal_position")
            columns = cursor.fetchall()
            logger.info(f"생성된 테이블 {table_name} 컬럼 목록: {[col[0] for col in columns]}")

            conn.commit()
            cursor.close()
            conn.close()

            logger.info(f"파인튜닝된 모델 전용 테이블 생성 완료: {table_name}")
            return table_name
            
        except Exception as e:
            logger.error(f"파인튜닝된 모델 전용 테이블 생성 실패: {e}")
            return None

    async def _drop_finetuned_model_table(self, table_name: str):
        """파인튜닝된 모델 전용 임베딩 테이블 삭제"""
        try:
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()
            
            cursor.execute(f"DROP TABLE IF EXISTS {table_name}")
            conn.commit()
            cursor.close()
            conn.close()
            
            logger.info(f"파인튜닝된 모델 전용 테이블 삭제 완료: {table_name}")
            
        except Exception as e:
            logger.error(f"파인튜닝된 모델 전용 테이블 삭제 실패: {e}")

    async def _embed_training_dataset_with_finetuned_model(self, model_path: str, table_name: str, dataset_id: str) -> int:
        """홀드아웃 데이터를 제외한 학습 데이터만 파인튜닝된 모델로 임베딩하여 전용 테이블에 저장"""
        embedded_count = 0
        finetuned_model = None

        try:
            # 파인튜닝된 모델 로드
            from sentence_transformers import SentenceTransformer
            device = "cuda" if torch.cuda.is_available() else "cpu"
            finetuned_model = SentenceTransformer(model_path, device=device)
            logger.info(f"파인튜닝된 모델 로드 완료: {model_path}")

            # 학습 데이터셋 로드
            dataset_file_path = os.path.expanduser(f"~/.airun/datasets/{dataset_id}.json")
            if not os.path.exists(dataset_file_path):
                logger.error(f"학습 데이터셋 파일을 찾을 수 없음: {dataset_file_path}")
                return 0

            with open(dataset_file_path, 'r', encoding='utf-8') as f:
                dataset = json.load(f)

            all_examples = dataset.get('examples', [])
            if not all_examples:
                logger.warning("학습 데이터셋의 예시가 없음")
                return 0

            total_count = len(all_examples)
            logger.info(f"전체 데이터셋: {total_count}개")

            # 홀드아웃 데이터 인덱스 확인
            holdout_file = dataset_file_path.replace('.json', '_holdout.json')
            holdout_indices = set()
            
            logger.info(f"홀드아웃 파일 경로 확인: {holdout_file}")
            logger.info(f"홀드아웃 파일 존재 여부: {os.path.exists(holdout_file)}")

            if os.path.exists(holdout_file):
                try:
                    with open(holdout_file, 'r', encoding='utf-8') as f:
                        holdout_data = json.load(f)
                    holdout_indices = set(holdout_data.get('holdout_indices', []))
                    logger.info(f"홀드아웃 데이터 인덱스: {len(holdout_indices)}개 - {list(holdout_indices)}")
                except Exception as e:
                    logger.error(f"홀드아웃 파일 읽기 실패: {e}")
            else:
                logger.warning(f"홀드아웃 파일이 존재하지 않음: {holdout_file}")

            # 전체 데이터를 검색 테이블에 포함 (홀드아웃도 포함)
            # 홀드아웃은 학습에서만 제외하고, 검색에는 포함해야 정답을 찾을 수 있음
            all_examples_with_idx = [(idx, example) for idx, example in enumerate(all_examples)]

            logger.info(f"홀드아웃 검증 설정:")
            logger.info(f"  - 전체 데이터: {total_count}개")
            logger.info(f"  - 홀드아웃 (평가용): {len(holdout_indices)}개")
            logger.info(f"  - 임베딩 대상 (전체): {len(all_examples_with_idx)}개")

            if not all_examples_with_idx:
                logger.error("임베딩할 데이터가 없습니다!")
                return 0

            # 파인튜닝된 모델 전용 테이블에 연결
            target_conn = self.embedding_service.get_db_connection()
            target_cursor = target_conn.cursor()

            # 전체 데이터를 임베딩 (홀드아웃 데이터도 포함)
            for original_idx, example in all_examples_with_idx:
                try:
                    # positive 내용 임베딩
                    positive_content = ''
                    if 'positive' in example:
                        positive_content = example['positive']
                    elif 'document' in example:
                        positive_content = example['document']
                    elif 'answer' in example:
                        positive_content = example['answer']

                    if positive_content and len(positive_content.strip()) >= 10:
                        # positive 텍스트 임베딩 생성
                        embedding = finetuned_model.encode([positive_content])[0]
                        embedding_list = embedding.tolist()

                        # positive 임베딩을 데이터베이스에 저장
                        doc_id = f"training_positive_{original_idx}"
                        filename = f"training_dataset_{dataset_id}_positive_{original_idx}"

                        target_cursor.execute(f"""
                            INSERT INTO {table_name}
                            (is_embedding, doc_id, filename, chunk_index, chunk_text, embedding, user_id, source, metadata)
                            VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)
                        """, (
                            True,
                            doc_id,
                            filename,
                            0,
                            positive_content,
                            embedding_list,
                            'training_user',
                            'training_dataset',
                            json.dumps({'example_idx': original_idx, 'query': example.get('query', ''), 'type': 'positive'})
                        ))

                        embedded_count += 1

                    # negative 내용 임베딩
                    negative_content = example.get('negative', '')
                    if negative_content and len(negative_content.strip()) >= 10:
                        # negative 텍스트 임베딩 생성
                        embedding = finetuned_model.encode([negative_content])[0]
                        embedding_list = embedding.tolist()

                        # negative 임베딩을 데이터베이스에 저장
                        doc_id = f"training_negative_{original_idx}"
                        filename = f"training_dataset_{dataset_id}_negative_{original_idx}"

                        target_cursor.execute(f"""
                            INSERT INTO {table_name}
                            (is_embedding, doc_id, filename, chunk_index, chunk_text, embedding, user_id, source, metadata)
                            VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)
                        """, (
                            True,
                            doc_id,
                            filename,
                            0,
                            negative_content,
                            embedding_list,
                            'training_user',
                            'training_dataset',
                            json.dumps({'example_idx': original_idx, 'query': example.get('query', ''), 'type': 'negative'})
                        ))

                        embedded_count += 1

                    if embedded_count % 10 == 0:
                        logger.info(f"임베딩 진행: {embedded_count}개 (positive + negative)")

                except Exception as e:
                    logger.error(f"예시 {idx} 임베딩 실패: {e}")
                    continue

            # 커밋
            target_conn.commit()
            target_cursor.close()
            target_conn.close()

            logger.info(f"학습 데이터셋 임베딩 완료: {embedded_count}개")
            return embedded_count

        except Exception as e:
            logger.error(f"문서 재임베딩 실패: {e}")
            import traceback
            logger.debug(f"상세 오류: {traceback.format_exc()}")
            return 0

        finally:
            # 파인튜닝된 모델 해제
            if finetuned_model is not None:
                del finetuned_model
                finetuned_model = None
            if torch.cuda.is_available():
                torch.cuda.empty_cache()
                torch.cuda.synchronize()

    def _get_collection_documents(self) -> List[str]:
        """실제 RAG 컬렉션에서 문서 내용들 가져오기"""
        try:
            # PostgreSQL에서 문서 내용 조회
            collection = self.postgresql_client.get_collection(name=self.COLLECTION_NAME)
            
            # 모든 문서의 내용과 메타데이터 가져오기
            results = collection.get(include=['documents', 'metadatas'])
            
            documents = []
            seen_docs = set()  # 중복 제거용
            
            for i, (doc, metadata) in enumerate(zip(results['documents'], results['metadatas'])):
                if doc and len(doc.strip()) > 50:  # 최소 길이 확인
                    # 중복 제거 (같은 내용의 청크들 제외)
                    doc_hash = hash(doc[:100])  # 처음 100자로 해시 생성
                    if doc_hash not in seen_docs:
                        documents.append(doc.strip())
                        seen_docs.add(doc_hash)
                        
                        # 처음 몇 개 문서만 로그 출력
                        if i < 3:
                            file_name = metadata.get('filename', 'unknown')
                            logger.info(f"컬렉션 문서 {i+1}: '{file_name}' - {len(doc)}자")
            
            logger.info(f"컬렉션에서 {len(documents)}개의 고유 문서 추출")
            return documents[:20]  # 최대 20개만 반환
            
        except Exception as e:
            logger.error(f"컬렉션 문서 추출 실패: {e}")
            return []
    
    def _prepare_default_test_data(self) -> List[Dict[str, Any]]:
        """실제 RAG 컬렉션 기반 테스트 데이터 준비"""
        try:
            test_data = []
            
            # 1. 실제 RAG 컬렉션에서 추출된 질문들 사용
            collection_questions = self._get_collection_questions()
            logger.info(f"컬렉션에서 추출된 질문: {len(collection_questions)}개")
            
            if collection_questions:
                # 추출된 질문들을 사용 (최대 15개)
                for question in collection_questions[:15]:
                    test_data.append({
                        'query': question,
                        'positive': '',
                        'negative': '',
                        'expected_relevant': True,
                        'use_rag_search': True,
                        'source': 'collection'
                    })
            
            # 2. 부족한 경우 기본 쿼리로 보완
            if len(test_data) < 10:
                logger.info("컬렉션 질문이 부족하여 기본 쿼리로 보완")
                for query in self.test_queries[:10-len(test_data)]:
                    test_data.append({
                        'query': query,
                        'positive': '',
                        'negative': '',
                        'expected_relevant': True,
                        'use_rag_search': True,
                        'source': 'default'
                    })
            
            logger.info(f"총 테스트 데이터: {len(test_data)}개 (컬렉션: {sum(1 for t in test_data if t.get('source') == 'collection')}개)")
            return test_data
            
        except Exception as e:
            logger.error(f"테스트 데이터 준비 실패: {e}")
            # 실패 시 기존 방식으로 폴백
            test_data = []
            for query in self.test_queries[:10]:
                test_data.append({
                    'query': query,
                    'positive': '',
                    'negative': '',
                    'expected_relevant': True,
                    'use_rag_search': True,
                    'source': 'fallback'
                })
            return test_data
    
    async def _ensure_model_loaded(self, model_id: str):
        """모델이 로드되었는지 확인하고 필요시 전환"""
        try:
            # 현재 활성 모델 확인
            current_model = os.environ.get('RAG_EMBEDDING_MODEL', 'nlpai-lab/KURE-v1')
            
            if current_model != model_id:
                logger.info(f"평가를 위한 모델 전환: {current_model} -> {model_id}")
                
                # 실제 모델 전환 수행
                global embedding_service
                if embedding_service:
                    # 환경 변수 변경
                    os.environ['RAG_EMBEDDING_MODEL'] = model_id
                    
                    # 기존 모델 인스턴스 초기화
                    embedding_service._initialized = False
                    embedding_service.model = None
                    embedding_service.tokenizer = None
                    
                    # 새 모델로 다시 초기화
                    embedding_service.initialize()
                    
                    logger.info(f"모델 전환 완료: {model_id}")
                else:
                    logger.info("글로벌 임베딩 서비스가 없음 - 평가에서 직접 모델 로드할 예정")
                    
        except Exception as e:
            logger.warning(f"모델 로드 확인 실패: {e}")
    
    async def _run_evaluation_with_model(self, test_data: List[Dict[str, Any]], model_id: str, table_name: str) -> Dict[str, Any]:
        """지정된 모델을 직접 사용한 평가 (기본 모델 또는 파인튜닝된 모델)"""
        start_time = time.time()
        evaluation_model = None

        true_positives = 0
        false_positives = 0
        true_negatives = 0
        false_negatives = 0
        total_similarity_scores = []

        logger.info(f"모델 평가 시작: {len(test_data)}개 테스트 케이스")

        try:
            # 1. 모델 구분 및 로딩
            if model_id == "nlpai-lab/KURE-v1":
                # 기본 모델: 평가용으로 별도 로딩 (워커와 독립적)
                from sentence_transformers import SentenceTransformer
                device = "cuda" if torch.cuda.is_available() else "cpu"
                evaluation_model = SentenceTransformer(model_id, device=device)
                logger.info(f"기본 모델 평가용 로딩 완료: {model_id}")
            else:
                # 파인튜닝된 모델: 별도 로딩
                model_path = await self._get_finetuned_model_path(model_id)
                if not model_path:
                    raise Exception(f"파인튜닝된 모델 경로를 찾을 수 없음: {model_id}")

                from sentence_transformers import SentenceTransformer
                device = "cuda" if torch.cuda.is_available() else "cpu"
                evaluation_model = SentenceTransformer(model_path, device=device)
                logger.info(f"파인튜닝된 모델 로드 완료: {model_path}")

            # 3. 각 테스트 케이스에 대해 평가
            for i, test_case in enumerate(test_data):
                query = test_case['query']
                expected_relevant = test_case.get('expected_relevant', True)  # 기본값 True

                logger.info(f"   테스트 {i+1}/{len(test_data)}: '{query[:50]}...'")

                # 지정된 모델로 쿼리 임베딩
                query_embedding = evaluation_model.encode(query, convert_to_tensor=True)
                query_embedding_list = query_embedding.cpu().numpy().tolist()

                # 모델에 따른 검색 방식 결정
                if model_id == "nlpai-lab/KURE-v1":
                    # 기본 모델: 기존 임베딩 컬렉션에서 검색
                    similarity_score = await self._search_in_existing_collection(query_embedding_list)
                else:
                    # 파인튜닝된 모델: 전용 테이블에서 검색
                    similarity_score = await self._search_in_finetuned_table(
                        table_name, query_embedding_list
                    )

                total_similarity_scores.append(similarity_score)

                # 동적 임계값 설정 (평균 유사도 기반)
                # 첫 번째 케이스는 임시 임계값 0.3 사용
                if i == 0:
                    threshold = 0.3
                else:
                    # 이전 케이스들의 평균 유사도로 임계값 조정
                    avg_similarity = sum(total_similarity_scores[:-1]) / len(total_similarity_scores[:-1])
                    threshold = max(0.3, min(0.6, avg_similarity * 0.8))  # 0.3~0.6 범위로 제한

                predicted_relevant = similarity_score >= threshold
                logger.debug(f"      임계값: {threshold:.2f}")

                logger.info(f"      유사도 점수: {similarity_score:.2f}, 예측: {predicted_relevant}, 정답: {expected_relevant}")

                if expected_relevant and predicted_relevant:
                    true_positives += 1
                elif expected_relevant and not predicted_relevant:
                    false_negatives += 1
                elif not expected_relevant and predicted_relevant:
                    false_positives += 1
                else:
                    true_negatives += 1

            # 분류 방식 메트릭 계산 (기존)
            precision = true_positives / (true_positives + false_positives) if (true_positives + false_positives) > 0 else 0
            recall = true_positives / (true_positives + false_negatives) if (true_positives + false_negatives) > 0 else 0
            f1_score = 2 * (precision * recall) / (precision + recall) if (precision + recall) > 0 else 0
            accuracy = (true_positives + true_negatives) / len(test_data) if len(test_data) > 0 else 0
            avg_similarity = sum(total_similarity_scores) / len(total_similarity_scores) if total_similarity_scores else 0

            # 정보 검색 방식 메트릭 계산 (추가)
            retrieval_metrics = await self._calculate_retrieval_metrics(test_data, evaluation_model, table_name, model_id)

            evaluation_time = time.time() - start_time

            logger.info(f"파인튜닝된 모델 평가 완료:")
            logger.info(f"  === 분류 방식 평가 ===")
            logger.info(f"  정밀도: {precision:.2f}")
            logger.info(f"  재현율: {recall:.2f}")
            logger.info(f"  F1 점수: {f1_score:.2f}")
            logger.info(f"  정확도: {accuracy:.2f}")
            logger.info(f"  평균 유사도: {avg_similarity:.2f}")
            logger.info(f"  === 정보 검색 방식 평가 ===")
            logger.info(f"  Recall@1: {retrieval_metrics.get('recall_at_1', 0):.2f}")
            logger.info(f"  Recall@3: {retrieval_metrics.get('recall_at_3', 0):.2f}")
            logger.info(f"  Recall@5: {retrieval_metrics.get('recall_at_5', 0):.2f}")
            logger.info(f"  nDCG@5: {retrieval_metrics.get('ndcg_at_5', 0):.2f}")
            logger.info(f"  MRR: {retrieval_metrics.get('mrr', 0):.2f}")
            logger.info(f"  평가 시간: {evaluation_time:.2f}초")

            return {
                # 기존 분류 방식 메트릭
                "precision": precision,
                "recall": recall,
                "f1_score": f1_score,
                "accuracy": accuracy,
                "average_similarity": avg_similarity,
                "evaluation_time": evaluation_time,
                "total_tests": len(test_data),
                "true_positives": true_positives,
                "false_positives": false_positives,
                "true_negatives": true_negatives,
                "false_negatives": false_negatives,
                # 새로운 정보 검색 방식 메트릭
                "retrieval_metrics": retrieval_metrics
            }

        except Exception as e:
            logger.error(f"파인튜닝된 모델 평가 실패: {e}")
            import traceback
            logger.debug(f"상세 오류: {traceback.format_exc()}")

            return {
                "precision": 0.0,
                "recall": 0.0,
                "f1_score": 0.0,
                "accuracy": 0.0,
                "average_similarity": 0.0,
                "evaluation_time": time.time() - start_time,
                "total_tests": len(test_data),
                "error": str(e)
            }

        finally:
            # 평가용 모델 메모리 해제 (기본 모델 포함)
            if evaluation_model is not None:
                logger.info("평가용 모델 메모리 해제 중...")
                try:
                    # 모델의 모든 파라미터를 CPU로 이동
                    if hasattr(evaluation_model, 'cpu'):
                        evaluation_model.cpu()

                    # 모델 객체 삭제
                    del evaluation_model
                    finetuned_model = None

                    # 가비지 컬렉션 강제 실행
                    import gc
                    gc.collect()

                    # CUDA 메모리 정리
                    if torch.cuda.is_available():
                        torch.cuda.empty_cache()
                        torch.cuda.synchronize()
                        # 추가적인 CUDA 메모리 정리
                        torch.cuda.ipc_collect()

                    logger.info("파인튜닝된 모델 메모리 해제 완료")

                except Exception as cleanup_error:
                    logger.warning(f"모델 메모리 해제 중 오류 (무시): {cleanup_error}")

            # VRAM 사용량 로깅
            if torch.cuda.is_available():
                try:
                    allocated = torch.cuda.memory_allocated() / 1024**3  # GB
                    cached = torch.cuda.memory_reserved() / 1024**3     # GB
                    logger.info(f"평가 완료 후 VRAM 사용량: {allocated:.2f}GB (캐시: {cached:.2f}GB)")
                except:
                    pass

    async def _search_in_finetuned_table(self, table_name: str, query_embedding: List[float]) -> float:
        """전용 테이블에서 유사도 검색 수행"""
        try:
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()

            # 벡터 유사도 검색 (cosine similarity)
            cursor.execute(f"""
                SELECT chunk_text, metadata, 1 - (embedding <=> %s::vector) as similarity
                FROM {table_name}
                WHERE embedding IS NOT NULL
                ORDER BY similarity DESC
                LIMIT 1
            """, (query_embedding,))

            result = cursor.fetchone()
            cursor.close()
            conn.close()

            if result:
                chunk_text, metadata, similarity = result
                logger.debug(f"가장 유사한 문서: {chunk_text[:50]}... (유사도: {similarity:.3f})")
                return float(similarity)
            else:
                logger.warning(f"전용 테이블에서 검색 결과가 없음: {table_name}")
                return 0.0

        except Exception as e:
            logger.error(f"전용 테이블 검색 실패: {e}")
            return 0.0

    async def _run_evaluation_with_collection(self, test_data: List[Dict[str, Any]], eval_collection_name: str) -> Dict[str, Any]:
        """평가용 컬렉션을 사용한 실제 RAG 검색 평가"""
        start_time = time.time()
        
        true_positives = 0
        false_positives = 0
        true_negatives = 0
        false_negatives = 0
        total_similarity_scores = []
        
        logger.info(f"평가 시작: {len(test_data)}개 테스트 케이스 (컬렉션: {eval_collection_name})")
        
        try:
            # 평가용 컬렉션 가져오기
            eval_collection = self.embedding_service.collection
            
            for i, test_case in enumerate(test_data):
                query = test_case['query']
                expected_relevant = test_case['expected_relevant']
                
                logger.info(f"   테스트 {i+1}/{len(test_data)}: '{query[:50]}...'")
                
                if test_case.get('use_rag_search'):
                    # 실제 RAG 검색 수행
                    logger.info(f"      실제 RAG 검색 수행 (컬렉션: {eval_collection_name})")
                    similarity_score = await self._perform_rag_search(query, eval_collection_name)
                else:
                    # 직접 유사도 계산
                    positive = test_case.get('positive', '')
                    logger.info(f"      직접 유사도 계산: '{positive[:30]}...'")
                    similarity_score = await self._calculate_similarity(query, positive)
                
                total_similarity_scores.append(similarity_score)
                
                # 임계값 0.5로 판정 (보다 현실적인 임계값)
                predicted_relevant = similarity_score >= 0.5
                
                logger.info(f"      유사도 점수: {similarity_score:.2f}, 예측: {predicted_relevant}, 정답: {expected_relevant}")
                
                if expected_relevant and predicted_relevant:
                    true_positives += 1
                elif expected_relevant and not predicted_relevant:
                    false_negatives += 1
                elif not expected_relevant and predicted_relevant:
                    false_positives += 1
                else:
                    true_negatives += 1
            
            # 메트릭 계산
            precision = true_positives / (true_positives + false_positives) if (true_positives + false_positives) > 0 else 0
            recall = true_positives / (true_positives + false_negatives) if (true_positives + false_negatives) > 0 else 0
            f1_score = 2 * (precision * recall) / (precision + recall) if (precision + recall) > 0 else 0
            
            avg_similarity = sum(total_similarity_scores) / len(total_similarity_scores) if total_similarity_scores else 0
            evaluation_time = time.time() - start_time
            
            logger.info(f"평가 완료 - 분류 결과:")
            logger.info(f"   True Positives: {true_positives}")
            logger.info(f"   False Positives: {false_positives}")
            logger.info(f"   True Negatives: {true_negatives}")
            logger.info(f"   False Negatives: {false_negatives}")
            logger.info(f"   정밀도: {precision:.2f}")
            logger.info(f"   재현율: {recall:.2f}")
            logger.info(f"   F1 Score: {f1_score:.2f}")
            logger.info(f"   평균 유사도: {avg_similarity:.2f}")
            logger.info(f"   소요 시간: {evaluation_time:.2f}초")
            
            return {
                "precision": precision,
                "recall": recall,
                "f1_score": f1_score,
                "avg_similarity": avg_similarity,
                "evaluation_time": evaluation_time,
                "test_cases": len(test_data),
                "true_positives": true_positives,
                "false_positives": false_positives,
                "true_negatives": true_negatives,
                "false_negatives": false_negatives,
                "evaluation_method": "rag_collection"
            }
            
        except Exception as e:
            logger.error(f"컬렉션 기반 평가 수행 실패: {e}")
            return {
                "precision": 0.0,
                "recall": 0.0,
                "f1_score": 0.0,
                "avg_similarity": 0.0,
                "evaluation_time": time.time() - start_time,
                "error": str(e),
                "evaluation_method": "rag_collection_failed"
            }
    
    async def _calculate_retrieval_metrics(self, test_data: List[Dict[str, Any]], evaluation_model, table_name: str, model_id: str) -> Dict[str, Any]:
        """정보 검색 방식 평가 지표 계산 (Recall@k, nDCG@k, MRR)"""
        import math

        logger.info("=== 정보 검색 방식 평가 지표 계산 시작 ===")

        # 정답 문서가 있는 쿼리만 필터링 (positive 케이스)
        valid_queries = []
        for test_case in test_data:
            if test_case.get('expected_relevant', True):  # positive 케이스만
                valid_queries.append(test_case)

        if not valid_queries:
            logger.warning("정보 검색 평가를 위한 positive 쿼리가 없음")
            return {
                "recall_at_1": 0.0,
                "recall_at_3": 0.0,
                "recall_at_5": 0.0,
                "ndcg_at_5": 0.0,
                "mrr": 0.0,
                "total_queries": 0
            }

        logger.info(f"정보 검색 평가 대상 쿼리: {len(valid_queries)}개")

        recall_at_k = {1: 0, 3: 0, 5: 0}
        ndcg_scores = []
        mrr_scores = []

        try:
            for i, test_case in enumerate(valid_queries):
                query = test_case['query']
                logger.info(f"  정보 검색 평가 {i+1}/{len(valid_queries)}: '{query[:50]}...'")

                # 1. 쿼리 임베딩 생성
                query_embedding = evaluation_model.encode(query, convert_to_tensor=True)
                query_embedding_list = query_embedding.cpu().numpy().tolist()

                # 2. Top-k 검색 수행 (k=5)
                search_results = await self._search_top_k_documents(
                    query_embedding_list, model_id, table_name, k=5
                )

                if not search_results:
                    logger.warning(f"  쿼리 '{query[:30]}...'에 대한 검색 결과 없음")
                    continue

                # 3. 정답 문서 확인 (positive 문서와 비교)
                target_document = test_case.get('positive', '')
                relevant_positions = []

                logger.info(f"    Target 문서: {target_document[:100]}...")
                logger.info(f"    검색된 문서 개수: {len(search_results)}")
                
                for pos, (doc_text, similarity) in enumerate(search_results):
                    # 간단한 텍스트 유사도로 정답 문서 판별
                    is_relevant = self._is_relevant_document(doc_text, target_document)
                    
                    # Jaccard 유사도 직접 계산하여 로그 출력
                    retrieved_tokens = set(doc_text.lower().split())
                    target_tokens = set(target_document.lower().split())
                    if target_tokens:
                        intersection = retrieved_tokens.intersection(target_tokens)
                        jaccard_sim = len(intersection) / len(target_tokens.union(retrieved_tokens))
                    else:
                        jaccard_sim = 0.0
                    
                    logger.info(f"    문서 {pos+1}: 유사도={similarity:.3f}, Jaccard={jaccard_sim:.3f}, 관련성={is_relevant}, 내용={doc_text[:50]}...")
                    if is_relevant:
                        relevant_positions.append(pos + 1)  # 1-based indexing

                logger.info(f"    검색된 관련 문서 위치: {relevant_positions}")

                # 4. Recall@k 계산
                for k in [1, 3, 5]:
                    if any(pos <= k for pos in relevant_positions):
                        recall_at_k[k] += 1

                # 5. nDCG@5 계산
                ndcg_score = self._calculate_ndcg(relevant_positions, k=5)
                ndcg_scores.append(ndcg_score)

                # 6. MRR 계산
                if relevant_positions:
                    mrr_score = 1.0 / min(relevant_positions)  # 첫 번째 관련 문서의 역순위
                    mrr_scores.append(mrr_score)
                else:
                    mrr_scores.append(0.0)

                logger.info(f"    nDCG@5: {ndcg_score:.3f}, MRR: {mrr_scores[-1]:.3f}")

            # 최종 지표 계산
            total_queries = len(valid_queries)
            final_metrics = {
                "recall_at_1": recall_at_k[1] / total_queries if total_queries > 0 else 0.0,
                "recall_at_3": recall_at_k[3] / total_queries if total_queries > 0 else 0.0,
                "recall_at_5": recall_at_k[5] / total_queries if total_queries > 0 else 0.0,
                "ndcg_at_5": sum(ndcg_scores) / len(ndcg_scores) if ndcg_scores else 0.0,
                "mrr": sum(mrr_scores) / len(mrr_scores) if mrr_scores else 0.0,
                "total_queries": total_queries
            }

            logger.info("=== 정보 검색 방식 평가 지표 계산 완료 ===")
            return final_metrics

        except Exception as e:
            logger.error(f"정보 검색 방식 평가 지표 계산 실패: {e}")
            return {
                "recall_at_1": 0.0,
                "recall_at_3": 0.0,
                "recall_at_5": 0.0,
                "ndcg_at_5": 0.0,
                "mrr": 0.0,
                "total_queries": len(valid_queries),
                "error": str(e)
            }

    async def _search_top_k_documents(self, query_embedding: List[float], model_id: str, table_name: str, k: int = 5) -> List[tuple]:
        """Top-k 문서 검색"""
        try:
            if table_name == "existing_collection":
                # 기본 모델이면서 기존 컬렉션 사용시
                return await self._search_top_k_in_existing_collection(query_embedding, k)
            else:
                # 평가시 임시 테이블 사용 (기본/파인튜닝 모델 모두)
                return await self._search_top_k_in_finetuned_table(table_name, query_embedding, k)
        except Exception as e:
            logger.error(f"Top-k 검색 실패: {e}")
            return []

    async def _search_top_k_in_existing_collection(self, query_embedding: List[float], k: int = 5) -> List[tuple]:
        """기존 컬렉션에서 Top-k 검색"""
        try:
            # psycopg2를 사용한 동기 방식으로 데이터베이스 연결
            conn = psycopg2.connect(
                host=os.getenv('POSTGRES_HOST', 'localhost'),
                port=int(os.getenv('POSTGRES_PORT', 5433)),
                user=os.getenv('POSTGRES_USER', 'ivs'),
                password=os.getenv('POSTGRES_PASSWORD', '1234'),
                database=os.getenv('POSTGRES_DB', 'airun')
            )
            
            cursor = conn.cursor()
            
            # 벡터 유사도 검색 (embedding을 문자열로 변환)
            embedding_str = '[' + ','.join(map(str, query_embedding)) + ']'
            query = """
                SELECT content, 1 - (embedding <=> %s::vector) as similarity
                FROM rag_documents 
                WHERE user_id = %s
                ORDER BY embedding <=> %s::vector
                LIMIT %s
            """
            
            cursor.execute(query, (embedding_str, "system", embedding_str, k))
            results = cursor.fetchall()
            
            cursor.close()
            conn.close()
            
            return [(row[0], row[1]) for row in results]
        except Exception as e:
            logger.error(f"기존 컬렉션 Top-k 검색 실패: {e}")
            return []

    async def _search_top_k_in_finetuned_table(self, table_name: str, query_embedding: List[float], k: int = 5) -> List[tuple]:
        """파인튜닝된 모델 전용 테이블에서 Top-k 검색"""
        try:
            conn = self.embedding_service.get_db_connection()
            cursor = conn.cursor()

            cursor.execute(f"""
                SELECT chunk_text, 1 - (embedding <=> %s::vector) as similarity
                FROM {table_name}
                WHERE embedding IS NOT NULL
                ORDER BY similarity DESC
                LIMIT %s
            """, (query_embedding, k))

            results = cursor.fetchall()
            cursor.close()
            conn.close()

            return [(text, float(sim)) for text, sim in results]
        except Exception as e:
            logger.error(f"파인튜닝된 테이블 Top-k 검색 실패: {e}")
            return []

    def _is_relevant_document(self, retrieved_doc: str, target_doc: str, threshold: float = 0.05) -> bool:
        """문서가 관련성이 있는지 판별 (간단한 텍스트 유사도 기반)"""
        if not retrieved_doc or not target_doc:
            return False

        # 간단한 토큰 기반 유사도 계산
        retrieved_tokens = set(retrieved_doc.lower().split())
        target_tokens = set(target_doc.lower().split())

        if not target_tokens:
            return False

        intersection = retrieved_tokens.intersection(target_tokens)
        jaccard_similarity = len(intersection) / len(target_tokens.union(retrieved_tokens))

        return jaccard_similarity >= threshold

    def _calculate_ndcg(self, relevant_positions: List[int], k: int = 5) -> float:
        """nDCG@k 계산"""
        if not relevant_positions:
            return 0.0

        # DCG 계산
        dcg = 0.0
        for pos in relevant_positions:
            if pos <= k:
                dcg += 1.0 / math.log2(pos + 1)

        # IDCG 계산: 실제 발견된 관련 문서 수를 기반으로 이상적인 순서 계산
        num_relevant = len([pos for pos in relevant_positions if pos <= k])
        if num_relevant == 0:
            return 0.0

        # 이상적인 경우: 모든 관련 문서가 1위부터 차례로 배치
        idcg = 0.0
        for i in range(1, min(num_relevant + 1, k + 1)):
            idcg += 1.0 / math.log2(i + 1)

        return dcg / idcg if idcg > 0 else 0.0

    async def _run_evaluation(self, test_data: List[Dict[str, Any]]) -> Dict[str, Any]:
        """실제 평가 수행"""
        start_time = time.time()
        
        true_positives = 0
        false_positives = 0
        true_negatives = 0
        false_negatives = 0
        total_similarity_scores = []
        
        logger.info(f"평가 시작: {len(test_data)}개 테스트 케이스")
        
        try:
            for i, test_case in enumerate(test_data):
                query = test_case['query']
                expected_relevant = test_case['expected_relevant']
                
                logger.info(f"   테스트 {i+1}/{len(test_data)}: '{query[:50]}...'")
                
                if test_case.get('use_rag_search'):
                    # RAG 검색으로 평가
                    logger.info(f"      RAG 검색 기반 평가 수행")
                    similarity_score = await self._evaluate_with_rag_search(query)
                else:
                    # 직접 유사도 계산
                    positive = test_case.get('positive', '')
                    logger.info(f"      직접 유사도 계산: '{positive[:30]}...'")
                    similarity_score = await self._calculate_similarity(query, positive)
                
                total_similarity_scores.append(similarity_score)
                
                # 임계값 0.5로 판정 (보다 현실적인 임계값)
                predicted_relevant = similarity_score >= 0.5
                
                logger.info(f"      유사도 점수: {similarity_score:.2f}, 예측: {predicted_relevant}, 정답: {expected_relevant}")
                
                if expected_relevant and predicted_relevant:
                    true_positives += 1
                elif expected_relevant and not predicted_relevant:
                    false_negatives += 1
                elif not expected_relevant and predicted_relevant:
                    false_positives += 1
                else:
                    true_negatives += 1
            
            # 메트릭 계산
            precision = true_positives / (true_positives + false_positives) if (true_positives + false_positives) > 0 else 0
            recall = true_positives / (true_positives + false_negatives) if (true_positives + false_negatives) > 0 else 0
            f1_score = 2 * (precision * recall) / (precision + recall) if (precision + recall) > 0 else 0
            
            avg_similarity = sum(total_similarity_scores) / len(total_similarity_scores) if total_similarity_scores else 0
            evaluation_time = time.time() - start_time
            
            logger.info(f"평가 완료 - 분류 결과:")
            logger.info(f"   True Positives: {true_positives}")
            logger.info(f"   False Positives: {false_positives}")
            logger.info(f"   True Negatives: {true_negatives}")
            logger.info(f"   False Negatives: {false_negatives}")
            logger.info(f"   정밀도: {precision:.2f}")
            logger.info(f"   재현율: {recall:.2f}")
            logger.info(f"   F1 Score: {f1_score:.2f}")
            logger.info(f"   평균 유사도: {avg_similarity:.2f}")
            logger.info(f"   소요 시간: {evaluation_time:.2f}초")
            
            return {
                "precision": precision,
                "recall": recall,
                "f1_score": f1_score,
                "avg_similarity": avg_similarity,
                "evaluation_time": evaluation_time,
                "test_cases": len(test_data),
                "true_positives": true_positives,
                "false_positives": false_positives,
                "true_negatives": true_negatives,
                "false_negatives": false_negatives
            }
            
        except Exception as e:
            logger.error(f"평가 수행 실패: {e}")
            return {
                "precision": 0.0,
                "recall": 0.0,
                "f1_score": 0.0,
                "avg_similarity": 0.0,
                "evaluation_time": time.time() - start_time,
                "error": str(e)
            }
    
    async def _perform_rag_search(self, query: str, eval_collection_name: str) -> float:
        """평가용 컬렉션에서 실제 RAG 검색 수행"""
        try:
            # 워커 시스템을 사용하여 임베딩 및 검색 수행
            global embedding_model_ready_event

            if not embedding_model_ready_event.is_set():
                logger.error("임베딩 워커가 준비되지 않음")
                return 0.5

            # SearchRequest 객체 생성하여 기존 search_documents 함수 재사용
            search_request = SearchRequest(
                query=query,
                user_id='eval_user',  # 평가용 가상 사용자
                ragSearchScope='personal',
                maxResults=5
            )

            # 기존 검색 기능 호출
            search_response = await search_documents(search_request)

            if not search_response or not hasattr(search_response, 'results') or not search_response.results:
                logger.warning(f"      검색 결과 없음")
                return 0.0

            # 검색 결과에서 최대 유사도 추출
            max_similarity = 0.0
            for result in search_response.results:
                if hasattr(result, 'similarity'):
                    max_similarity = max(max_similarity, float(result.similarity))
                elif hasattr(result, 'score'):
                    max_similarity = max(max_similarity, float(result.score))

            # 상위 결과들 로그 출력 (최대 3개)
            for i, result in enumerate(search_response.results[:3]):
                content = getattr(result, 'content', '')[:50] if hasattr(result, 'content') else 'N/A'
                similarity = getattr(result, 'similarity', getattr(result, 'score', 0.0))
                logger.info(f"         검색결과 {i+1}: {similarity:.2f} - '{content}...'")

            logger.info(f"      RAG 검색 최대 유사도: {max_similarity:.2f}")
            return max_similarity
            
        except Exception as e:
            logger.error(f"RAG 검색 수행 실패: {e}")
            return 0.5
    
    async def _evaluate_with_rag_search(self, query: str) -> float:
        """실제 컬렉션 문서와 동일 모델로 유사도 평가 (폐기예정)"""
        try:
            # 1. 실제 컬렉션에서 문서 내용들 가져오기
            collection_docs = self._get_collection_documents()
            
            if not collection_docs:
                logger.warning("컬렉션 문서가 없어서 샘플 문서로 평가")
                # 폴백: 샘플 문서들
                collection_docs = [
                    "인공지능은 컴퓨터가 인간의 지능을 모방하여 학습하고 추론하는 기술입니다.",
                    "머신러닝은 데이터를 통해 패턴을 학습하는 인공지능의 한 분야입니다.",
                    "자연어 처리는 컴퓨터가 인간의 언어를 이해하고 처리하는 기술입니다.",
                    "빅데이터는 기존 데이터베이스로는 처리하기 어려운 대용량 데이터를 의미합니다.",
                    "클라우드 컴퓨팅은 인터넷을 통해 컴퓨팅 자원을 제공하는 서비스입니다."
                ]
            
            # 2. 워커 시스템을 사용하여 검색으로 유사도 계산
            max_similarity = 0.0
            global embedding_model_ready_event

            if embedding_model_ready_event.is_set():
                logger.info(f"      워커 시스템으로 {len(collection_docs)}개 문서와 유사도 비교")

                # 이 함수는 폐기예정이므로 간단히 기본 검색으로 대체
                similarity = await self._perform_rag_search(query, "default_collection")
                max_similarity = similarity
            else:
                logger.warning("임베딩 워커가 준비되지 않음")
                max_similarity = 0.5

            logger.info(f"      최대 유사도: {max_similarity:.2f}")
            return max_similarity
            
        except Exception as e:
            logger.error(f"컬렉션 기반 평가 실패: {e}")
            return 0.5  # 기본값
    
    async def _calculate_similarity(self, query: str, text: str) -> float:
        """직접 유사도 계산"""
        try:
            if not text.strip():
                return 0.0
            
            global embedding_model_ready_event
            if not embedding_model_ready_event.is_set():
                return 0.0

            # 워커 시스템 사용하여 간단한 검색으로 유사도 추정
            # 이 함수는 임시로 간단한 텍스트 매칭으로 대체
            query_lower = query.lower()
            text_lower = text.lower()

            # 간단한 키워드 매칭 기반 유사도 (임시)
            common_words = set(query_lower.split()) & set(text_lower.split())
            similarity = len(common_words) / max(len(query_lower.split()), len(text_lower.split())) if query_lower and text_lower else 0.0
            
            return float(similarity)
            
        except Exception as e:
            logger.error(f"유사도 계산 실패: {e}")
            return 0.0
    
    async def save_evaluation_results(self, model_id: str, metrics: Dict[str, Any]):
        """평가 결과를 모델 메타데이터에 저장"""
        try:
            if model_id == "nlpai-lab/KURE-v1":
                # 기본 모델의 경우 별도 파일에 저장
                eval_cache_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'evaluation_cache')
                os.makedirs(eval_cache_dir, exist_ok=True)
                eval_file = os.path.join(eval_cache_dir, 'base_model_evaluation.json')
                
                eval_data = {
                    'model_id': model_id,
                    'evaluation_date': time.time(),
                    'metrics': metrics,
                    'version': '1.0.0'
                }
                
                with open(eval_file, 'w', encoding='utf-8') as f:
                    json.dump(eval_data, f, ensure_ascii=False, indent=2)
                
                logger.info(f"기본 모델 평가 결과 저장: {eval_file}")
                
            else:
                # 파인튜닝 모델의 경우 메타데이터 파일 업데이트
                finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
                model_path = Path(finetuned_dir) / model_id
                metadata_path = model_path / "metadata.json"
                
                if metadata_path.exists():
                    with open(metadata_path, 'r', encoding='utf-8') as f:
                        metadata = json.load(f)
                    
                    # 평가 결과 업데이트
                    metadata.update({
                        'best_score': metrics.get('f1_score', 0.0),
                        'precision': metrics.get('precision', 0.0),
                        'recall': metrics.get('recall', 0.0),
                        'f1_score': metrics.get('f1_score', 0.0),
                        'evaluation_date': time.time(),
                        'evaluation_metrics': metrics
                    })
                    
                    with open(metadata_path, 'w', encoding='utf-8') as f:
                        json.dump(metadata, f, ensure_ascii=False, indent=2)
                    
                    logger.info(f"파인튜닝 모델 평가 결과 저장: {metadata_path}")
                    
        except Exception as e:
            logger.error(f"평가 결과 저장 실패 ({model_id}): {e}")
    
    def load_evaluation_results(self, model_id: str) -> Dict[str, Any]:
        """저장된 평가 결과 로드"""
        try:
            if model_id == "nlpai-lab/KURE-v1":
                # 기본 모델 평가 결과
                eval_cache_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'evaluation_cache')
                eval_file = os.path.join(eval_cache_dir, 'base_model_evaluation.json')
                
                if os.path.exists(eval_file):
                    with open(eval_file, 'r', encoding='utf-8') as f:
                        eval_data = json.load(f)
                    return eval_data.get('metrics', {})
                    
            else:
                # 파인튜닝 모델 메타데이터
                finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
                model_path = Path(finetuned_dir) / model_id
                metadata_path = model_path / "metadata.json"
                
                if metadata_path.exists():
                    with open(metadata_path, 'r', encoding='utf-8') as f:
                        metadata = json.load(f)
                    return metadata.get('evaluation_metrics', {})
                    
        except Exception as e:
            logger.error(f"평가 결과 로드 실패 ({model_id}): {e}")
            
        return {}

    async def _cleanup_eval_user_data(self):
        """eval_user 관련 데이터 정리"""
        try:
            eval_user_id = "eval_user"
            logger.info(f"🧹 eval_user ({eval_user_id}) 데이터 정리 시작")
            
            # PostgreSQL에서 eval_user 관련 임베딩 데이터 삭제 (동기 방식)
            conn = None
            try:
                conn = self.embedding_service.get_db_connection()
                cursor = conn.cursor()
                
                # document_embeddings 테이블에서 eval_user 데이터 삭제
                delete_query = "DELETE FROM document_embeddings WHERE user_id = %s"
                cursor.execute(delete_query, (eval_user_id,))
                result = cursor.rowcount
                logger.info(f"   document_embeddings에서 {result} 개 레코드 삭제")
                
                # 다른 관련 테이블에서도 eval_user 데이터 삭제 (있다면)
                # documents 테이블은 존재하지 않으므로 생략
                
                # 변경사항 커밋
                conn.commit()
                cursor.close()
                
            finally:
                if conn:
                    self.embedding_service.return_db_connection(conn)
            
            logger.info(f"✅ eval_user 데이터 정리 완료")
            return True
            
        except Exception as e:
            logger.error(f"❌ eval_user 데이터 정리 실패: {str(e)}")
            return False


    async def _embed_with_finetuned_model(self, model_path: str, text: str, table_name: str, 
                                        user_id: str, doc_id: str) -> bool:
        """파인튜닝된 모델로 직접 임베딩하여 전용 테이블에 저장"""
        try:
            logger.info(f"🔥 파인튜닝된 모델로 직접 임베딩: {doc_id}")
            
            # 파인튜닝된 모델 로드
            from sentence_transformers import SentenceTransformer
            import torch
            import numpy as np
            
            device = 'cuda' if torch.cuda.is_available() else 'cpu'
            logger.info(f"   파인튜닝된 모델 로딩: {model_path} ({device})")
            
            finetuned_model = SentenceTransformer(model_path, device=device)
            
            # 텍스트 임베딩 생성
            logger.info(f"   텍스트 임베딩 생성 중... (길이: {len(text)}자)")
            embedding = finetuned_model.encode(text, convert_to_tensor=False)
            
            # numpy array로 변환
            if hasattr(embedding, 'cpu'):
                embedding = embedding.cpu().numpy()
            elif hasattr(embedding, 'numpy'):
                embedding = embedding.numpy()
            embedding = np.array(embedding, dtype=np.float32)
            
            logger.info(f"   임베딩 생성 완료: {embedding.shape}")
            
            # 전용 테이블에 저장
            conn = None
            try:
                conn = self.embedding_service.get_db_connection()
                cursor = conn.cursor()
                
                insert_query = f"""
                INSERT INTO {table_name} 
                (user_id, document_id, content, embedding, metadata)
                VALUES (%s, %s, %s, %s, %s)
                """
                
                metadata = {
                    "finetuned_model": model_path,
                    "evaluation_purpose": True,
                    "doc_type": "eval_document"
                }
                
                cursor.execute(insert_query, (
                    user_id, doc_id, text, embedding.tolist(), 
                    json.dumps(metadata)
                ))
                conn.commit()
                cursor.close()
                
                logger.info(f"✅ 파인튜닝된 모델 임베딩 저장 완료: {doc_id}")
                
                # 모델 메모리 해제
                del finetuned_model
                if torch.cuda.is_available():
                    torch.cuda.empty_cache()
                
                return True
                
            finally:
                if conn:
                    self.embedding_service.return_db_connection(conn)
                    
        except Exception as e:
            logger.error(f"❌ 파인튜닝된 모델 임베딩 실패: {str(e)}")
            return False

    async def _search_with_finetuned_model(self, model_path: str, query: str, 
                                         table_name: str, user_id: str, top_k: int = 5) -> List[Dict]:
        """파인튜닝된 모델로 질문 임베딩하여 전용 테이블에서 검색"""
        try:
            logger.info(f"🔍 파인튜닝된 모델로 검색: {query[:50]}...")
            
            # 파인튜닝된 모델로 질문 임베딩
            from sentence_transformers import SentenceTransformer
            import torch
            import numpy as np
            
            device = 'cuda' if torch.cuda.is_available() else 'cpu'
            finetuned_model = SentenceTransformer(model_path, device=device)
            
            query_embedding = finetuned_model.encode(query, convert_to_tensor=False)
            if hasattr(query_embedding, 'cpu'):
                query_embedding = query_embedding.cpu().numpy()
            elif hasattr(query_embedding, 'numpy'):
                query_embedding = query_embedding.numpy()
            query_embedding = np.array(query_embedding, dtype=np.float32)
            
            logger.info(f"   질문 임베딩 완료: {query_embedding.shape}")
            
            # 전용 테이블에서 유사도 검색
            conn = None
            try:
                conn = self.embedding_service.get_db_connection()
                cursor = conn.cursor()
                
                search_query = f"""
                SELECT document_id, content, 
                       1 - (embedding <=> %s::vector) as similarity
                FROM {table_name}
                WHERE user_id = %s
                ORDER BY similarity DESC
                LIMIT %s
                """
                
                cursor.execute(search_query, (query_embedding.tolist(), user_id, top_k))
                results = cursor.fetchall()
                cursor.close()
                
                search_results = []
                for doc_id, content, similarity in results:
                    search_results.append({
                        'document_id': doc_id,
                        'content': content,
                        'similarity': float(similarity)
                    })
                
                logger.info(f"   검색 결과: {len(search_results)}개 (최고 유사도: {search_results[0]['similarity']:.3f if search_results else 0})")
                
                # 모델 메모리 해제
                del finetuned_model
                if torch.cuda.is_available():
                    torch.cuda.empty_cache()
                
                return search_results
                
            finally:
                if conn:
                    self.embedding_service.return_db_connection(conn)
                    
        except Exception as e:
            logger.error(f"❌ 파인튜닝된 모델 검색 실패: {str(e)}")
            return []

# 전역 평가기 인스턴스
model_evaluator = ModelEvaluator() if FINETUNE_AVAILABLE else None

# 모델 관련 통합 API (파인튜닝, 평가, 활성화 등 모든 액션 처리)
@app.post("/models")
async def handle_model_actions(request: Request):
    """모델 관련 모든 액션 통합 처리"""
    logger.info("===============모델 액션 요청 처리 시작=======================")
    try:
        data = await request.json()
        action = data.get('action')
        logger.info(f"DEBUG: 수신된 액션: {action}")
        logger.info(f"DEBUG: 요청 데이터: {data}")
        
        if action == 'evaluate-model':
            # 모델 평가
            logger.info(f"DEBUG: 모델 평가 요청 수신 - FINETUNE_AVAILABLE: {FINETUNE_AVAILABLE}")
            global model_evaluator
            logger.info(f"DEBUG: model_evaluator 상태: {model_evaluator is not None}")
            if not FINETUNE_AVAILABLE or model_evaluator is None:
                logger.error(f"DEBUG: 모델 평가 차단 - FINETUNE_AVAILABLE: {FINETUNE_AVAILABLE}, model_evaluator: {model_evaluator is not None}")
                return {
                    "success": False,
                    "error": f"모델 평가 기능이 사용할 수 없습니다. FINETUNE_AVAILABLE: {FINETUNE_AVAILABLE}, model_evaluator: {model_evaluator is not None}"
                }
            
            model_id = data.get('modelId')
            dataset_id = data.get('datasetId')  # 선택적

            if not model_id:
                return {
                    "success": False,
                    "error": "모델 ID가 필요합니다."
                }

            # 파인튜닝된 모델의 경우 메타데이터에서 dataset_id 자동 추출
            if not dataset_id and model_id != "nlpai-lab/KURE-v1":
                try:
                    finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
                    model_path = os.path.join(finetuned_dir, model_id)
                    metadata_path = os.path.join(model_path, 'metadata.json')

                    if os.path.exists(metadata_path):
                        with open(metadata_path, 'r', encoding='utf-8') as f:
                            metadata = json.load(f)
                        dataset_id = metadata.get('dataset')
                        if dataset_id:
                            logger.info(f"모델 {model_id}의 훈련 데이터셋 ID 자동 추출: {dataset_id}")
                        else:
                            logger.warning(f"모델 {model_id}의 메타데이터에서 dataset 정보를 찾을 수 없음")
                    else:
                        logger.warning(f"모델 {model_id}의 메타데이터 파일이 존재하지 않음: {metadata_path}")
                except Exception as e:
                    logger.error(f"모델 {model_id}의 dataset_id 추출 실패: {e}")

            # 모델 평가 수행
            # evaluation_mode 추출 (기본값: universal)
            evaluation_mode = data.get('evaluation_mode', 'universal')

            # 범용평가 모드일 때는 universal_benchmark 사용
            if evaluation_mode == 'universal':
                dataset_id = 'universal_benchmark'
                logger.info(f"범용평가 모드 선택: dataset_id를 'universal_benchmark'로 설정")
            elif evaluation_mode == 'domain':
                # 도메인 전문성 평가 모드일 때
                logger.info(f"도메인 평가 모드 확인: model_id={model_id}, dataset_id={dataset_id}")
                if model_id == "nlpai-lab/KURE-v1":
                    logger.info(f"기본 모델 도메인 평가 진입: dataset_id={dataset_id}")
                    # 기본 모델의 경우 파인튜닝 모델들의 dataset_id를 찾아서 사용
                    if not dataset_id:
                        # 가장 최근 파인튜닝 모델의 dataset_id를 사용
                        try:
                            finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
                            if os.path.exists(finetuned_dir):
                                # 최신 파인튜닝 모델 찾기
                                finetuned_models = [d for d in os.listdir(finetuned_dir)
                                                  if os.path.isdir(os.path.join(finetuned_dir, d))]
                                if finetuned_models:
                                    # 가장 최근 모델의 dataset_id 사용
                                    latest_model = sorted(finetuned_models)[-1]
                                    metadata_path = os.path.join(finetuned_dir, latest_model, 'metadata.json')
                                    if os.path.exists(metadata_path):
                                        with open(metadata_path, 'r', encoding='utf-8') as f:
                                            metadata = json.load(f)
                                        dataset_id = metadata.get('dataset')
                                        logger.info(f"기본 모델 도메인 평가: 최신 파인튜닝 모델({latest_model})의 dataset_id({dataset_id}) 사용")
                        except Exception as e:
                            logger.error(f"기본 모델 도메인 평가를 위한 dataset_id 추출 실패: {e}")

                logger.info(f"도메인 전문성 평가 모드: dataset_id='{dataset_id}' 사용")

            result = await model_evaluator.evaluate_model(model_id, dataset_id, evaluation_mode)
            
            # 평가 결과를 메타데이터에 저장 (성공한 경우만)
            if result.get('success') and result.get('metrics'):
                # metrics와 test_samples 정보를 함께 저장
                evaluation_data = {
                    **result['metrics'],  # 기존 metrics 데이터
                    'test_samples': result.get('test_samples', 0),  # 테스트 샘플 수 추가
                    'evaluation_time': result.get('evaluation_time', 0)  # 평가 시간 추가
                }
                await model_evaluator.save_evaluation_results(model_id, evaluation_data)
            
            return result
            
        elif action == 'set-active-model':
            # 활성 모델 변경
            model_id = data.get('modelId')
            if not model_id:
                return {
                    "success": False,
                    "error": "모델 ID가 필요합니다."
                }
            
            # config API 호출
            return await set_active_model(request)
            
        elif action == 'get-config':
            # 현재 설정 조회
            return await get_config()
            
        elif action == 'get-evaluation-results':
            # 저장된 평가 결과 조회
            if not FINETUNE_AVAILABLE or model_evaluator is None:
                return {
                    "success": False, 
                    "error": "모델 평가 기능이 사용할 수 없습니다."
                }
            
            try:
                # 모든 모델의 평가 결과 수집
                all_evaluation_results = []
                
                # 기본 모델 평가 결과
                try:
                    base_eval = model_evaluator.load_evaluation_results("nlpai-lab/KURE-v1")
                    if base_eval:
                        all_evaluation_results.append({
                            'model_id': 'nlpai-lab/KURE-v1',
                            'precision': base_eval.get('precision'),
                            'recall': base_eval.get('recall'),
                            'f1_score': base_eval.get('f1_score'),
                            'evaluation_date': base_eval.get('evaluation_date'),
                            'test_samples': base_eval.get('test_samples'),
                            'evaluation_time': base_eval.get('evaluation_time'),
                            'retrieval_metrics': base_eval.get('retrieval_metrics', {})
                        })
                except Exception as e:
                    logger.error(f"기본 모델 평가 결과 로드 실패: {e}")
                
                # 파인튜닝 모델들의 평가 결과
                try:
                    finetuned_dir = os.path.join(os.path.expanduser('~'), '.airun', 'models', 'finetuned')
                    
                    if os.path.exists(finetuned_dir):
                        model_dirs = os.listdir(finetuned_dir)
                        
                        for model_dir in model_dirs:
                            model_path = os.path.join(finetuned_dir, model_dir)
                            if os.path.isdir(model_path):
                                try:
                                    eval_data = model_evaluator.load_evaluation_results(model_dir)
                                    if eval_data:
                                        all_evaluation_results.append({
                                            'model_id': model_dir,
                                            'precision': eval_data.get('precision'),
                                            'recall': eval_data.get('recall'),
                                            'f1_score': eval_data.get('f1_score'),
                                            'evaluation_date': eval_data.get('evaluation_date'),
                                            'test_samples': eval_data.get('test_samples'),
                                            'evaluation_time': eval_data.get('evaluation_time'),
                                            'retrieval_metrics': eval_data.get('retrieval_metrics', {})
                                        })
                                except Exception as e:
                                    logger.error(f"파인튜닝 모델 {model_dir} 평가 결과 로드 실패: {e}")
                except Exception as e:
                    logger.error(f"파인튜닝 모델 평가 결과 수집 실패: {e}")
                
                return {
                    "success": True,
                    "evaluation_results": all_evaluation_results
                }
                
            except Exception as e:
                logger.error(f"평가 결과 조회 실패: {e}", exc_info=True)
                return {
                    "success": False,
                    "error": f"평가 결과 조회 실패: {str(e)}"
                }
            
        elif action == 'analyze-resources':
            # 시스템 자원 분석
            return await analyze_system_resources()
            
        elif action == 'finetune':
            # 파인튜닝 (기존 로직)
            if not FINETUNE_AVAILABLE:
                return {
                    "success": False, 
                    "error": "파인튜닝 기능을 사용할 수 없습니다."
                }
            
            # FineTuneRequest 형태로 변환
            finetune_request = FineTuneRequest(
                model_name=data.get('model_name', 'nlpai-lab/KURE-v1'),
                dataset_id=data.get('dataset_id'),
                epochs=data.get('epochs', 3),
                batch_size=data.get('batch_size', 16),
                learning_rate=data.get('learning_rate', 2e-5),
                force_cpu=data.get('force_cpu', False),
                user_set_epochs=data.get('user_set_epochs', False),
                user_set_batch_size=data.get('user_set_batch_size', False),
                user_set_learning_rate=data.get('user_set_learning_rate', False)
            )
            
            return await start_finetuning(finetune_request)
            
        else:
            return {
                "success": False,
                "error": f"알 수 없는 액션: {action}"
            }
            
    except Exception as e:
        logger.error(f"모델 액션 처리 실패: {str(e)}")
        return {
            "success": False,
            "error": str(e)
        }

@app.post("/datasets/from-file")
async def create_dataset_from_file(
    file: UploadFile = File(...),
    name: str = Form(...),
    description: str = Form("")
):
    """파일에서 데이터셋 생성"""
    try:
        logger.info(f"파일 기반 데이터셋 생성 시작: {name}")
        
        # 파일 내용 읽기
        content = await file.read()
        
        # 파일 형식에 따라 처리
        examples = []
        if file.filename.endswith('.json'):
            data = json.loads(content.decode('utf-8'))
            if 'examples' in data:
                examples = data['examples']
            else:
                return {"success": False, "error": "JSON 파일에 'examples' 키가 필요합니다."}
        
        elif file.filename.endswith('.csv'):
            csv_data = io.StringIO(content.decode('utf-8'))
            reader = csv.DictReader(csv_data)
            for row in reader:
                if 'query' in row and 'positive' in row:
                    example = {
                        'query': row['query'],
                        'positive': row['positive'],
                        'negative': row.get('negative', '')
                    }
                    examples.append(example)
        
        elif file.filename.endswith('.txt'):
            # 간단한 텍스트 형식 처리
            lines = content.decode('utf-8').strip().split('\n')
            for i in range(0, len(lines), 2):
                if i + 1 < len(lines):
                    example = {
                        'query': lines[i].strip(),
                        'positive': lines[i + 1].strip(),
                        'negative': ''
                    }
                    examples.append(example)
        
        if not examples:
            return {"success": False, "error": "파일에서 유효한 예제를 찾을 수 없습니다."}
        
        # 데이터셋 매니저로 저장 (description 필드 보완)
        if not description:
            description = f"'{file.filename}' 파일에서 생성된 데이터셋 - {len(examples)}개의 예제"
        
        global dataset_manager
        if dataset_manager is None:
            dataset_manager = TrainingDatasetManager()
        
        dataset_id = dataset_manager.create_dataset_from_examples(examples, name, description)
        
        logger.info(f"파일 기반 데이터셋 생성 완료: {dataset_id} ({len(examples)}개 예제)")
        
        return {
            "success": True,
            "dataset_id": dataset_id,
            "examples_count": len(examples),
            "message": f"파일에서 {len(examples)}개의 예제로 데이터셋을 생성했습니다."
        }
        
    except Exception as e:
        logger.error(f"파일 기반 데이터셋 생성 실패: {e}")
        return {"success": False, "error": str(e)}

@app.post("/datasets/from-document")
async def create_dataset_from_document(
    document: UploadFile = File(...),
    name: str = Form(...),
    description: str = Form(""),
    provider: str = Form("ollama"),
    model: str = Form("hamonize:latest"),
    target_qa_count: int = Form(50)  # 사용자 지정 QA 개수 (기본값: 50)
):
    """문서에서 LLM으로 데이터셋 생성"""
    try:
        logger.info(f"문서+LLM 기반 데이터셋 생성 시작: {name}")
        
        # 문서 내용 읽기
        content = await document.read()
        
        # 지원되는 파일 확장자 확인
        file_ext = os.path.splitext(document.filename)[1].lower()
        supported_extensions = ['.txt', '.md', '.pdf', '.docx', '.doc', '.hwp', '.hwpx', '.pptx', '.ppt']
        
        if file_ext not in supported_extensions:
            return {"success": False, "error": f"지원하지 않는 파일 형식입니다. 지원 형식: {', '.join(supported_extensions)}"}
        
        # 임시 파일에 저장하여 문서 처리
        import utils
        
        text_content = ""
        with tempfile.NamedTemporaryFile(delete=False, suffix=file_ext) as tmp_file:
            tmp_file.write(content)
            tmp_file_path = tmp_file.name
        
        try:
            # utils.py의 extract_text_from_document 함수 사용
            if file_ext in ['.txt', '.md']:
                text_content = content.decode('utf-8')
            else:
                text_content = utils.extract_text_from_document(tmp_file_path)
            
            if not text_content or len(text_content.strip()) < 100:
                return {"success": False, "error": "문서에서 충분한 텍스트를 추출할 수 없습니다."}
                
        except Exception as e:
            logger.error(f"문서 파싱 오류: {e}")
            return {"success": False, "error": f"문서 파싱 실패: {str(e)}"}
        finally:
            # 임시 파일 삭제
            try:
                os.unlink(tmp_file_path)
            except:
                pass
        
        # LLM으로 질문-답변 쌍 생성 (프로바이더 정보 및 목표 개수 전달)
        examples = await generate_qa_pairs_from_text(text_content, name, provider, model, target_qa_count)
        
        if not examples:
            return {"success": False, "error": "문서에서 질문-답변 쌍을 생성할 수 없습니다."}
        
        # 데이터셋 저장 (description 필드 보완)
        if not description:
            description = f"'{document.filename}' 문서에서 AI로 생성된 데이터셋 - {len(examples)}개의 QA 쌍"
        
        global dataset_manager
        if dataset_manager is None:
            dataset_manager = TrainingDatasetManager()
        
        dataset_id = dataset_manager.create_dataset_from_examples(examples, name, description)
        
        logger.info(f"문서+LLM 기반 데이터셋 생성 완료: {dataset_id} ({len(examples)}개 질문)")
        
        return {
            "success": True,
            "dataset_id": dataset_id,
            "questions_generated": len(examples),
            "message": f"문서에서 {len(examples)}개의 질문-답변 쌍을 생성했습니다."
        }
        
    except Exception as e:
        logger.error(f"문서+LLM 기반 데이터셋 생성 실패: {e}")
        return {"success": False, "error": str(e)}

@app.post("/datasets/create-empty")
async def create_empty_dataset(request: Request):
    """빈 데이터셋 생성 (수동 입력용)"""
    try:
        data = await request.json()
        name = data.get('name')
        description = data.get('description', '')
        
        if not name:
            return {"success": False, "error": "데이터셋 이름이 필요합니다."}
        
        logger.info(f"빈 데이터셋 생성: {name}")
        
        # 데이터셋 매니저로 빈 데이터셋 생성
        global dataset_manager
        if dataset_manager is None:
            dataset_manager = TrainingDatasetManager()
        
        dataset_id = dataset_manager.create_empty_dataset(name, description)
        
        logger.info(f"빈 데이터셋 생성 완료: {dataset_id}")
        
        return {
            "success": True,
            "dataset_id": dataset_id,
            "message": "빈 데이터셋이 생성되었습니다. 이제 질문-답변 쌍을 추가할 수 있습니다."
        }
        
    except Exception as e:
        logger.error(f"빈 데이터셋 생성 실패: {e}")
        return {"success": False, "error": str(e)}

@app.post("/datasets/from-template")
async def create_dataset_from_template(request: Request):
    """템플릿 기반 데이터셋 생성"""
    try:
        data = await request.json()
        name = data.get('name')
        description = data.get('description', '')
        template_type = data.get('template_type', 'general')
        
        if not name:
            return {"success": False, "error": "데이터셋 이름이 필요합니다."}
        
        logger.info(f"템플릿 기반 데이터셋 생성: {name} (템플릿: {template_type})")
        
        # 템플릿에 따라 예제 생성
        examples = generate_template_examples(template_type)
        
        # 데이터셋 저장 (description 필드 보완)
        if not description:
            description = f"{template_type} 템플릿으로 생성된 데이터셋 - {len(examples)}개의 예제"
        
        global dataset_manager
        if dataset_manager is None:
            dataset_manager = TrainingDatasetManager()
        
        dataset_id = dataset_manager.create_dataset_from_examples(examples, name, description)
        
        logger.info(f"템플릿 기반 데이터셋 생성 완료: {dataset_id} ({len(examples)}개 예제)")
        
        return {
            "success": True,
            "dataset_id": dataset_id,
            "examples_count": len(examples),
            "template_type": template_type,
            "message": f"{template_type} 템플릿으로 {len(examples)}개의 예제를 생성했습니다."
        }
        
    except Exception as e:
        logger.error(f"템플릿 기반 데이터셋 생성 실패: {e}")
        return {"success": False, "error": str(e)}

@app.post("/evaluation-datasets/upload")
async def upload_evaluation_dataset(
    file: UploadFile = File(...),
    evaluation_type: str = Form(...),  # 'universal' or 'domain'
):
    """평가데이터셋 업로드 (임베딩 모델 파인튜닝용)"""
    try:
        logger.info(f"평가데이터셋 업로드 시작: {file.filename}, 타입: {evaluation_type}")

        # 평가 타입 검증
        if evaluation_type not in ['universal', 'domain']:
            return {"success": False, "error": "평가 타입이 유효하지 않습니다. (universal 또는 domain)"}

        # 파일 내용 읽기
        try:
            content = await file.read()
            content_str = content.decode('utf-8')
        except Exception as e:
            return {"success": False, "error": f"파일 읽기 실패: {str(e)}"}

        # JSON 또는 CSV 파싱
        try:
            # JSON 파싱 시도
            import json
            parsed_data = json.loads(content_str)
        except json.JSONDecodeError:
            try:
                # CSV 파싱 시도
                import csv
                import io
                csv_file = io.StringIO(content_str)
                reader = csv.DictReader(csv_file)
                examples = list(reader)
                parsed_data = {"examples": examples}
            except Exception as csv_error:
                return {"success": False, "error": "파일 형식이 유효하지 않습니다. JSON 또는 CSV 파일이어야 합니다."}

        # 대상 파일명 결정 (기존 파일명과 일치하도록)
        if evaluation_type == 'universal':
            target_filename = "universal_benchmark.json"
        else:  # domain
            target_filename = "domain_benchmark.json"

        # ~/.airun/datasets/ 경로에 저장 (기존 데이터셋과 동일한 위치)
        target_path = os.path.expanduser(f"~/.airun/datasets/{target_filename}")

        # 메타데이터 업데이트
        updated_data = {
            **parsed_data,
            "id": f"{evaluation_type}_eval_user_uploaded",
            "name": f"{'범용' if evaluation_type == 'universal' else '도메인'} 평가 데이터셋 (사용자 업로드)",
            "description": f"사용자가 업로드한 {evaluation_type} 평가 데이터셋",
            "type": f"{evaluation_type}_eval",
            "updated_at": datetime.now().isoformat(),
            "created_at": parsed_data.get("created_at", datetime.now().isoformat())
        }

        # 파일 저장
        import os
        os.makedirs(os.path.dirname(target_path), exist_ok=True)

        with open(target_path, 'w', encoding='utf-8') as f:
            json.dump(updated_data, f, ensure_ascii=False, indent=2)

        record_count = len(parsed_data.get('examples', []))

        logger.info(f"✅ {evaluation_type} 평가 데이터셋 업로드 완료: {target_filename} ({record_count}개 레코드)")

        return {
            "success": True,
            "message": f"{'범용' if evaluation_type == 'universal' else '도메인'} 평가 데이터셋이 성공적으로 업로드되었습니다.",
            "filename": target_filename,
            "evaluation_type": evaluation_type,
            "record_count": record_count
        }

    except Exception as e:
        logger.error(f"평가데이터셋 업로드 실패: {e}")
        return {"success": False, "error": f"평가 데이터셋 업로드 실패: {str(e)}"}

class QAGenerator:
    """질문-답변 쌍 생성 클래스 (rag_process.py 구조 기반)"""
    
    def __init__(self):
        self.cache = {}
        self.settings = self._get_settings()
        
        # AIProvider 초기화
        try:
            from rag_process import AIProvider
            self.ai_provider = AIProvider()
            # logger.info(f"QA 생성기 AI Provider 초기화 완료")
        except Exception as e:
            logger.error(f"QA 생성기 AI Provider 초기화 실패: {e}")
            self.ai_provider = None
    
    def set_provider(self, provider: str, model: str = None):
        """프로바이더 설정"""
        if self.ai_provider:
            self.ai_provider.provider = provider
            if model:
                # 존재하지 않는 OpenAI 모델을 실제 모델로 매핑
                if provider == 'openai':
                    model_mapping = {
                        'gpt-4.1': 'gpt-4o',
                        'gpt-4.1-mini': 'gpt-4o-mini',
                        'gpt-4.1-nano': 'gpt-4o-mini'
                    }
                    original_model = model
                    model = model_mapping.get(model, model)
                    if original_model != model:
                        logger.info(f"모델 매핑: {original_model} → {model}")
                
                self.ai_provider.model = model
            logger.info(f"QA 생성기 프로바이더 설정: {provider}, 모델: {model}")
    
    def _get_settings(self):
        """RAG 설정 가져오기"""
        try:
            from rag_process import get_rag_settings
            return get_rag_settings()
        except:
            return {}
    
    def _calculate_optimal_qa_count(self, text_length: int, target_count: int = None) -> tuple:
        """텍스트 길이에 따른 최적 QA 개수 계산 (사용자 지정 개수 우선)"""
        
        # 사용자가 지정한 개수가 있으면 우선 사용
        if target_count is not None and target_count > 0:
            # 사용자 지정 개수를 기준으로 범위 설정 (±20% 정도의 유연성)
            min_qa = max(int(target_count * 0.8), 5)  # 최소 5개
            max_qa = min(int(target_count * 1.2), 500)  # 최대 500개
            logger.info(f"[QAGenerator] 사용자 지정 QA 목표: {target_count}개 → 범위: {min_qa}-{max_qa}개")
            return min_qa, max_qa
        
        # 기본값: 텍스트 길이 기반 동적 계산
        if text_length <= 2000:  # 짧은 텍스트
            min_qa, max_qa = 8, 25
        elif text_length <= 5000:  # 중간 텍스트
            min_qa, max_qa = 15, 40
        elif text_length <= 10000:  # 긴 텍스트
            min_qa, max_qa = 25, 60
        elif text_length <= 20000:  # 매우 긴 텍스트
            min_qa, max_qa = 40, 100
        else:  # 초대형 텍스트
            min_qa, max_qa = 60, 150
        
        logger.info(f"[QAGenerator] 텍스트 길이 {text_length:,}자 → QA 목표: {min_qa}-{max_qa}개")
        return min_qa, max_qa

    def _prepare_chunk_text(self, chunk_text: str, max_length: int = 4000) -> str:
        """청크 텍스트를 AI 처리에 적합하게 준비"""
        if not chunk_text:
            return ""
        
        if len(chunk_text) <= max_length:
            return chunk_text
        
        # 문단 단위로 자르기 시도
        paragraphs = chunk_text.split('\n\n')
        if len(paragraphs) > 1:
            result_text = ""
            for paragraph in paragraphs:
                test_text = result_text + ("\n\n" if result_text else "") + paragraph
                if len(test_text) <= max_length:
                    result_text = test_text
                else:
                    break
            
            if result_text and len(result_text) >= max_length * 0.5:
                return result_text
        
        # 문장 단위로 자르기 시도
        sentences = re.split(r'[.!?。！？]\s*', chunk_text)
        if len(sentences) > 1:
            result_text = ""
            for sentence in sentences:
                if not sentence.strip():
                    continue
                test_text = result_text + (" " if result_text else "") + sentence.strip() + "."
                if len(test_text) <= max_length:
                    result_text = test_text
                else:
                    break
            
            if result_text and len(result_text) >= max_length * 0.4:
                return result_text
        
        # 스마트 절단
        cut_position = max_length
        search_range = min(500, cut_position // 4)
        for i in range(cut_position, max(0, cut_position - search_range), -1):
            if i < len(chunk_text) and chunk_text[i] in [' ', '\n', '.', '!', '?', '。', '！', '？']:
                cut_position = i
                break
        
        return chunk_text[:cut_position].strip()
    
    def _create_structured_qa_prompt(self, chunk_text: str, dataset_name: str) -> str:
        """구조적 QA 생성을 위한 프롬프트 생성 (간소화) - Negative 샘플 포함"""
        # 텍스트 길이에 따른 동적 개수 계산
        min_qa, max_qa = self._calculate_optimal_qa_count(len(chunk_text))
        
        # JSON 예제 템플릿 (format specifier 문제 방지)
        json_example = """[
  {
    "query": "질문 내용",
    "positive": "질문과 관련된 정확한 답변", 
    "negative": "질문과 관련 없는 내용 (다른 주제)"
  }
]"""
        
        prompt = f"""텍스트를 분석하여 {min_qa}-{max_qa}개의 질문-답변 쌍을 JSON 배열 형식으로 생성해주세요.

텍스트:
{chunk_text}

요구사항:
- 텍스트 내용을 기반으로 한 정확한 질문과 답변
- 구체적이고 명확한 질문 작성
- **각 질문마다 negative 샘플도 함께 생성 (질문과 관련 없는 내용)**
- Negative 샘플은 다른 주제나 무관한 정보로 생성 (예: 다른 분야의 개념, 전혀 다른 주제)
- 모든 응답은 한국어로 작성

JSON 형식:
{json_example}"""
        return prompt
    
    def _call_ai_service_structured(self, prompt: str, target_qa_count: int = None) -> Optional[str]:
        """구조적 AI 서비스 호출 (rag_process.py 기반)"""
        try:
            if not self.ai_provider:
                logger.error(f"[QAGenerator] AI 제공자가 초기화되지 않음")
                return None
            
            # 텍스트 길이에 따른 동적 QA 개수 계산 (사용자 지정 개수 우선)
            text_length = len(prompt)
            min_qa, max_qa = self._calculate_optimal_qa_count(text_length, target_qa_count)
            
            # 현재 프로바이더 확인
            current_provider = getattr(self.ai_provider, 'provider', 'unknown')
            
            # 프로바이더별 JSON 스키마 정의
            if current_provider == 'openai':
                # OpenAI Structured Outputs 요구사항: 루트는 object여야 함
                qa_schema = {
                    "type": "object",
                    "properties": {
                        "qa_pairs": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "query": {
                                        "type": "string",
                                        "description": "질문 내용 (최소 10자 이상)"
                                    },
                                    "positive": {
                                        "type": "string", 
                                        "description": "질문과 관련된 정확한 답변 (최소 20자 이상)"
                                    },
                                    "negative": {
                                        "type": "string",
                                        "description": "질문과 관련 없는 내용 (다른 주제나 무관한 정보, 최소 20자 이상)"
                                    }
                                },
                                "required": ["query", "positive", "negative"],
                                "additionalProperties": False
                            },
                            "minItems": min_qa,
                            "maxItems": max_qa,
                            "description": "텍스트에서 추출한 질문-답변 쌍 (대조 학습용 negative 샘플 포함)"
                        }
                    },
                    "required": ["qa_pairs"],
                    "additionalProperties": False
                }
            else:
                # Ollama 등 다른 프로바이더: 배열 형식 (더 간단함)
                qa_schema = {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "query": {
                                "type": "string",
                                "description": "질문 내용 (최소 10자 이상)"
                            },
                            "positive": {
                                "type": "string", 
                                "description": "질문과 관련된 정확한 답변 (최소 20자 이상)"
                            },
                            "negative": {
                                "type": "string",
                                "description": "질문과 관련 없는 내용 (다른 주제나 무관한 정보, 최소 20자 이상)"
                            }
                        },
                        "required": ["query", "positive", "negative"]
                    },
                    "minItems": min_qa,
                    "maxItems": max_qa,
                    "description": "텍스트에서 추출한 질문-답변 쌍 (대조 학습용 negative 샘플 포함)"
                }
            
            # 프로바이더별 시스템 프롬프트 구성
            if current_provider == 'openai':
                # OpenAI: Structured Outputs가 형식을 보장하므로 내용에 집중
                system_prompt = """당신은 질문-답변 쌍 생성 전문가입니다. 주어진 텍스트에서 다양하고 유용한 질문-답변 쌍을 JSON 형식으로 생성해주세요. 각 질문마다 positive(관련된 답변)와 negative(관련 없는 내용)를 모두 생성하여 대조 학습에 활용할 수 있도록 해주세요."""
                
                enhanced_prompt = f"""{prompt}

위 텍스트를 분석하여 {min_qa}-{max_qa}개의 질문-답변 쌍을 생성해주세요.
- 텍스트 내용을 기반으로 한 정확한 질문과 답변
- 다양한 관점에서의 질문 생성 (정의, 설명, 활용, 특징 등)
- **각 질문마다 negative 샘플도 함께 생성 (질문과 관련 없는 내용)**
- Negative 샘플은 다른 주제나 무관한 정보로 생성 (예: 다른 분야의 개념, 전혀 다른 주제)
- 모든 응답은 한국어로 작성"""
            else:
                # Ollama 등: 명확한 JSON 형식 예제 제공
                system_prompt = """당신은 질문-답변 쌍 생성 전문가입니다. 주어진 텍스트에서 다양하고 유용한 질문-답변 쌍을 정확한 JSON 배열 형식으로 생성해주세요."""
                
                enhanced_prompt = f"""{prompt}

위 텍스트를 분석하여 {min_qa}-{max_qa}개의 질문-답변 쌍을 다음 JSON 배열 형식으로 생성해주세요:

[
  {{
    "query": "질문 내용",
    "positive": "질문과 관련된 정확한 답변",
    "negative": "질문과 관련 없는 내용 (다른 주제)"
  }}
]

요구사항:
- 텍스트 내용을 기반으로 한 정확한 질문과 답변
- 다양한 관점에서의 질문 생성 (정의, 설명, 활용, 특징 등)
- **각 질문마다 negative 샘플도 함께 생성 (질문과 관련 없는 내용)**
- Negative 샘플은 다른 주제나 무관한 정보로 생성
- 모든 응답은 한국어로 작성
- **반드시 유효한 JSON 배열 형식으로만 응답**"""
            
            # 메시지 구성
            messages = [
                {
                    "role": "system", 
                    "content": system_prompt
                },
                {
                    "role": "user", 
                    "content": enhanced_prompt
                }
            ]
            
            # 🔍 디버깅: AI 호출 전 로깅
            logger.info(f"[QAGenerator] AI 호출 시작 - 프롬프트 길이: {len(enhanced_prompt)}")
            logger.info(f"[QAGenerator] 프롬프트 샘플: {enhanced_prompt[:300]}...")
            
            # Structured Outputs 사용 - JSON 스키마 객체를 format에 전달
            try:
                current_provider = getattr(self.ai_provider, 'provider', 'unknown')
                current_model = getattr(self.ai_provider, 'model', 'unknown')
                
                # OpenAI 모델 매핑 확인 (실제 API 호출에 사용할 모델명)
                actual_model = current_model
                if current_provider == 'openai':
                    model_mapping = {
                        'gpt-4.1': 'gpt-4o',
                        'gpt-4.1-mini': 'gpt-4o-mini',
                        'gpt-4.1-nano': 'gpt-4o-mini'
                    }
                    actual_model = model_mapping.get(current_model, current_model)
                    if actual_model != current_model:
                        logger.info(f"[QAGenerator] API 호출 모델 매핑: {current_model} → {actual_model}")
                
                if current_provider == 'openai':
                    logger.info(f"[QAGenerator] {current_provider} structured outputs 시도 (스키마 사용) - 모델: {actual_model}")
                else:
                    logger.info(f"[QAGenerator] {current_provider} JSON 생성 시도 (배열 형식) - 모델: {actual_model}")
                
                # 모델별 최대 출력 토큰 설정 (사용자 지정 QA 개수 고려)
                # 사용자가 많은 QA를 요청했을 때 충분한 출력 토큰 제공
                base_output_tokens = max(8192, (target_qa_count or 25) * 200) if target_qa_count else 8192
                
                model_max_tokens = {
                    # OpenAI 모델들
                    'gpt-4o': min(base_output_tokens, 16384),            # 128k 컨텍스트
                    'gpt-4o-mini': min(base_output_tokens, 16384),       # 128k 컨텍스트
                    'gpt-4-turbo': min(base_output_tokens, 16384),       # 128k 컨텍스트
                    'gpt-4': min(base_output_tokens, 8192),              # 8k 컨텍스트
                    'gpt-3.5-turbo': min(base_output_tokens, 4096),      # 16k 컨텍스트
                    'o1-preview': min(base_output_tokens, 32768),        # 128k 컨텍스트
                    'o1-mini': min(base_output_tokens, 65536),           # 128k 컨텍스트
                    # 매핑된 모델들
                    'gpt-4.1': min(base_output_tokens, 16384),           # → gpt-4o
                    'gpt-4.1-mini': min(base_output_tokens, 16384),      # → gpt-4o-mini
                    'gpt-4.1-nano': min(base_output_tokens, 16384),      # → gpt-4o-mini
                    # Ollama 모델들 (큰 컨텍스트 윈도우 활용)
                    'hamonize:latest': min(base_output_tokens, 32768), # 128k+ 컨텍스트
                    'airun-think:latest': min(base_output_tokens, 32768),
                    'airun-vision:latest': min(base_output_tokens, 16384),
                    'llama3.1:8b': min(base_output_tokens, 32768),       # 128k 컨텍스트
                    'llama3.1:70b': min(base_output_tokens, 32768),      # 128k 컨텍스트
                    'qwen2.5:14b': min(base_output_tokens, 32768),       # 32k+ 컨텍스트
                    'mistral:7b': min(base_output_tokens, 16384),        # 32k 컨텍스트
                    'default': min(base_output_tokens, 8192)
                }
                max_tokens = model_max_tokens.get(current_model, model_max_tokens['default'])
                
                response = self.ai_provider._call_provider(
                    messages=messages,
                    max_tokens=max_tokens,
                    format=qa_schema,   # JSON 스키마 객체 전달
                    model=actual_model  # 실제 매핑된 모델 사용
                )
            except Exception as schema_error:
                logger.warning(f"[QAGenerator] JSON 스키마 호출 실패: {schema_error}")
                logger.info(f"[QAGenerator] {current_provider} - 간단한 format='json'으로 폴백")
                # 폴백: 간단한 JSON 형식 사용 (더 작은 토큰 수)
                fallback_tokens = min(max_tokens // 2, 2048)  # 절반 또는 최대 2048 토큰
                response = self.ai_provider._call_provider(
                    messages=messages,
                    max_tokens=fallback_tokens,
                    format="json",     # 간단한 JSON 형식으로 폴백
                    model=current_model  # 현재 설정된 모델 사용 (매핑된 모델)
                )
            
            # 🔍 상세 디버깅: AI 응답 후 로깅
            logger.info(f"[QAGenerator] AI 호출 완료 - 응답 길이: {len(response) if response else 0}")
            if response:
                logger.info(f"[QAGenerator] AI 응답 전체 (처음 1000자): {response[:1000]}...")
                logger.info(f"[QAGenerator] AI 응답 마지막 500자: {response[-500:] if len(response) > 500 else response}")
                
                # JSON 유효성 미리 검증
                try:
                    parsed = json.loads(response)
                    logger.info(f"[QAGenerator] ✅ JSON 형식 유효성 확인됨")
                    logger.info(f"[QAGenerator] 응답 타입: {type(parsed)}")
                    if isinstance(parsed, list):
                        logger.info(f"[QAGenerator] ✅ 배열 형식 확인됨 - 항목 수: {len(parsed)}")
                    else:
                        logger.warning(f"[QAGenerator] ⚠️ 배열이 아닌 형식: {type(parsed)}")
                except json.JSONDecodeError as e:
                    logger.error(f"[QAGenerator] ❌ JSON 형식 오류: {str(e)}")
                    logger.error(f"[QAGenerator] 오류 위치: line {e.lineno}, column {e.colno}")
                    
                    # 잘린 JSON 복구 시도
                    logger.info(f"[QAGenerator] 잘린 JSON 복구 시도...")
                    try:
                        # 마지막 완전한 객체까지만 추출
                        truncated_json = self._repair_truncated_json(response)
                        if truncated_json:
                            parsed = json.loads(truncated_json)
                            logger.info(f"[QAGenerator] ✅ 잘린 JSON 복구 성공 - 항목 수: {len(parsed) if isinstance(parsed, list) else 1}")
                            # 복구된 JSON을 response로 교체
                            response = truncated_json
                        else:
                            logger.warning(f"[QAGenerator] 잘린 JSON 복구 실패")
                    except Exception as repair_error:
                        logger.error(f"[QAGenerator] JSON 복구 중 오류: {repair_error}")
            else:
                logger.warning(f"[QAGenerator] AI 응답이 비어있음 또는 None")
            
            return response
                
        except Exception as e:
            logger.error(f"[QAGenerator] 구조적 AI 서비스 호출 실패: {str(e)}")
            return None
    
    def _repair_truncated_json(self, json_str: str) -> str:
        """잘린 JSON 배열을 복구하여 유효한 JSON으로 만듦"""
        try:
            if not json_str or not json_str.strip():
                return ""
            
            # JSON 배열 시작 확인
            json_str = json_str.strip()
            if not json_str.startswith('['):
                return ""
            
            # 완전한 객체들만 추출
            objects = []
            depth = 0
            current_obj = ""
            in_string = False
            escape_next = False
            
            for i, char in enumerate(json_str):
                if escape_next:
                    escape_next = False
                    current_obj += char
                    continue
                    
                if char == '\\' and in_string:
                    escape_next = True
                    current_obj += char
                    continue
                    
                if char == '"' and not escape_next:
                    in_string = not in_string
                    current_obj += char
                    continue
                    
                if not in_string:
                    if char == '{':
                        depth += 1
                    elif char == '}':
                        depth -= 1
                        
                current_obj += char
                
                # 완전한 객체 완성 시
                if not in_string and depth == 0 and char == '}':
                    # 유효한 JSON 객체인지 확인
                    try:
                        obj_json = current_obj.strip().rstrip(',').strip()
                        if obj_json.startswith('{') and obj_json.endswith('}'):
                            json.loads(obj_json)  # 유효성 검증
                            objects.append(obj_json)
                    except:
                        pass
                    current_obj = ""
                    
                    # 다음 객체로 넘어가기 위해 쉼표와 공백 건너뛰기
                    while i + 1 < len(json_str) and json_str[i + 1] in ', \n\t\r':
                        i += 1
            
            if objects:
                repaired_json = '[' + ','.join(objects) + ']'
                # 최종 유효성 검증
                json.loads(repaired_json)
                logger.info(f"[QAGenerator] JSON 복구 성공: {len(objects)}개 객체 복구")
                return repaired_json
            else:
                return ""
                
        except Exception as e:
            logger.error(f"[QAGenerator] JSON 복구 실패: {str(e)}")
            return ""
    
    def _create_comprehensive_qa_prompt(self, text: str, dataset_name: str, target_count: int = None) -> str:
        """전체 문서를 한 번에 처리하는 포괄적 QA 생성 프롬프트 (간소화) - Negative 샘플 포함"""
        # 텍스트 길이에 따른 동적 개수 계산
        min_qa, max_qa = self._calculate_optimal_qa_count(len(text), target_count)
        
        # JSON 예제 템플릿 (OpenAI Structured Outputs 형식)
        json_example = """{
  "qa_pairs": [
    {
      "query": "질문 내용",
      "positive": "질문과 관련된 정확한 답변",
      "negative": "질문과 관련 없는 내용 (다른 주제)"
    }
  ]
}"""
        
        return f"""문서를 분석하여 {min_qa}-{max_qa}개의 질문-답변 쌍을 JSON 형식으로 생성해주세요.

입력 문서:
{text}

요구사항:
- 문서 내용을 기반으로 한 정확한 질문과 답변
- 다양한 관점에서의 질문 생성 (핵심 개념, 특징, 활용, 장점 등)
- **각 질문마다 negative 샘플도 함께 생성 (질문과 관련 없는 내용)**
- Negative 샘플은 다른 주제나 무관한 정보로 생성 (예: 다른 분야의 개념, 전혀 다른 주제)
- 모든 응답은 한국어로 작성

JSON 형식:
{json_example}"""

    def _create_large_chunk_qa_prompt(self, text: str, dataset_name: str, target_count: int = None) -> str:
        """큰 청크용 QA 생성 프롬프트 (간소화) - Negative 샘플 포함"""
        # 텍스트 길이에 따른 동적 개수 계산
        min_qa, max_qa = self._calculate_optimal_qa_count(len(text), target_count)
        
        # JSON 예제 템플릿 (OpenAI Structured Outputs 형식)
        json_example = """{
  "qa_pairs": [
    {
      "query": "질문 내용",
      "positive": "질문과 관련된 정확한 답변",
      "negative": "질문과 관련 없는 내용 (다른 주제)"
    }
  ]
}"""
        
        return f"""텍스트를 분석하여 {min_qa}-{max_qa}개의 질문-답변 쌍을 JSON 형식으로 생성해주세요.

입력 텍스트:
{text}

요구사항:
- 텍스트 내용을 기반으로 한 정확한 질문과 답변
- 다양한 관점에서의 질문 생성 (내용 요약, 세부 사항, 의미 등)
- **각 질문마다 negative 샘플도 함께 생성 (질문과 관련 없는 내용)**
- Negative 샘플은 다른 주제나 무관한 정보로 생성 (예: 다른 분야의 개념, 전혀 다른 주제)
- 모든 응답은 한국어로 작성

JSON 형식:
{json_example}"""
    
    def _parse_structured_response(self, response: str, chunk_text: str) -> List[Dict[str, Any]]:
        """구조적 AI 응답을 파싱하여 QA 쌍으로 변환"""
        try:
            
            # 🔍 디버깅: AI 응답 내용 로깅
            logger.info(f"[QAGenerator] AI 응답 원본 (처음 500자): {response[:500] if response else 'None'}")
            logger.info(f"[QAGenerator] 응답 타입: {type(response)}, 길이: {len(response) if response else 0}")
            
            # 응답이 이미 JSON 형식이어야 함 (Structured Outputs)
            if isinstance(response, str):
                response = response.strip()
                
                # JSON 배열 형식 찾기 (새로운 프롬프트 형식)
                if response.startswith('['):
                    json_match = re.search(r'\[.*\]', response, re.DOTALL)
                    if json_match:
                        response = json_match.group()
                        logger.info(f"[QAGenerator] JSON 배열 형식 감지, 파싱 시도")
                        data = json.loads(response)
                        qa_pairs = data if isinstance(data, list) else []
                    else:
                        raise ValueError("JSON 배열을 찾을 수 없음")
                
                # JSON 객체 형식 찾기 (OpenAI Structured Outputs 형식 우선)
                elif response.startswith('{'):
                    json_match = re.search(r'\{.*\}', response, re.DOTALL)
                    if json_match:
                        response = json_match.group()
                        logger.info(f"[QAGenerator] JSON 객체 형식 감지, 파싱 시도")
                        data = json.loads(response)
                        
                        # 새로운 OpenAI Structured Outputs 형식: {"qa_pairs": [...]}
                        if 'qa_pairs' in data:
                            qa_pairs = data.get('qa_pairs', [])
                            logger.info(f"[QAGenerator] ✅ OpenAI Structured Outputs 형식 감지 - {len(qa_pairs)}개 QA 쌍")
                        # 기존 형식: 단일 객체를 배열로 변환
                        elif 'query' in data and 'positive' in data:
                            qa_pairs = [data]
                            logger.info(f"[QAGenerator] 단일 QA 객체를 배열로 변환")
                        else:
                            qa_pairs = []
                    else:
                        raise ValueError("JSON 객체를 찾을 수 없음")
                
                else:
                    # JSON이 아닌 형식인 경우 찾기 시도
                    logger.info(f"[QAGenerator] 비JSON 형식 감지, JSON 패턴 검색 중...")
                    json_array_match = re.search(r'\[.*\]', response, re.DOTALL)
                    if json_array_match:
                        data = json.loads(json_array_match.group())
                        qa_pairs = data if isinstance(data, list) else []
                        logger.info(f"[QAGenerator] JSON 배열 패턴 찾음")
                    else:
                        json_object_match = re.search(r'\{.*\}', response, re.DOTALL)
                        if json_object_match:
                            data = json.loads(json_object_match.group())
                            
                            # qa_pairs 키가 있으면 해당 배열 사용
                            if 'qa_pairs' in data:
                                qa_pairs = data.get('qa_pairs', [])
                            # qa_pairs 키가 없고 query, positive 키가 있으면 단일 객체를 배열로 변환
                            elif 'query' in data and 'positive' in data:
                                qa_pairs = [data]
                                logger.info(f"[QAGenerator] 단일 QA 객체를 배열로 변환 (패턴 검색)")
                            else:
                                qa_pairs = []
                            logger.info(f"[QAGenerator] JSON 객체 패턴 찾음")
                        else:
                            logger.warning(f"[QAGenerator] 유효한 JSON 패턴을 찾을 수 없음")
                            raise ValueError("유효한 JSON을 찾을 수 없음")
                            
            else:
                data = response
                qa_pairs = data if isinstance(data, list) else data.get('qa_pairs', [])
            
            logger.info(f"[QAGenerator] 파싱된 QA 쌍 원본 수: {len(qa_pairs) if qa_pairs else 0}")
            logger.info(f"[QAGenerator] QA 쌍 샘플: {qa_pairs[:2] if qa_pairs else '없음'}")
            
            if not isinstance(qa_pairs, list):
                qa_pairs = []
            
            examples = []
            for qa_pair in qa_pairs:
                if not isinstance(qa_pair, dict):
                    continue
                
                # 새로운 형식과 기존 형식 모두 지원
                question = qa_pair.get('query', qa_pair.get('question', '')).strip()
                answer = qa_pair.get('positive', qa_pair.get('answer', '')).strip()
                negative = qa_pair.get('negative', '').strip()  # AI가 생성한 negative 샘플 추출
                
                # 품질 검증
                if (10 <= len(question) <= 300 and 
                    20 <= len(answer) <= 1000 and
                    question and answer):
                    
                    # AI 생성 QA 쌍의 품질 점수 계산
                    score = 0.8  # 기본 점수 (AI 생성이므로 높은 점수)
                    
                    # 답변 품질에 따른 점수 조정
                    if len(answer) >= 100:
                        score += 0.1  # 긴 답변 보너스
                    
                    # negative 샘플 존재 여부에 따른 점수 조정
                    if negative:
                        score += 0.1  # negative 샘플 보너스
                    
                    # 논리적 연결어 존재 여부에 따른 점수 조정
                    if any(keyword in answer.lower() for keyword in ['때문에', '따라서', '그러므로', '왜냐하면', '결과적으로']):
                        score += 0.05  # 논리적 연결어 보너스
                    
                    example = {
                        'query': question,
                        'positive': answer,
                        'negative': negative if negative else '',  # AI가 생성한 negative 사용, 없으면 빈 문자열
                        'score': min(1.0, round(score, 2)),  # 품질 점수 (최대 1.0)
                        'type': 'ai_structured'  # AI 구조화 생성 타입
                    }
                    examples.append(example)
            
            logger.info(f"[QAGenerator] 구조적 파싱 성공: {len(examples)}개 QA 쌍 생성")
            return examples
                
        except json.JSONDecodeError as e:
            logger.error(f"[QAGenerator] 구조적 JSON 파싱 실패: {str(e)}")
            logger.error(f"[QAGenerator] 응답 내용: {response[:500] if response else 'None'}")
            return self._fallback_qa_extraction(chunk_text)
        except Exception as e:
            logger.error(f"[QAGenerator] 구조적 응답 처리 실패: {str(e)}")
            logger.error(f"[QAGenerator] 응답 내용: {response[:500] if response else 'None'}")
            return self._fallback_qa_extraction(chunk_text)
    
    def _fallback_qa_extraction(self, chunk_text: str, target_count: int = None) -> List[Dict[str, Any]]:
        """AI 서비스 실패 시 기본 QA 쌍 생성"""
        try:
            examples = []
            
            # 키워드 추출
            korean_words = re.findall(r'[가-힣]{2,}', chunk_text)
            english_words = re.findall(r'[A-Za-z]{3,}', chunk_text)
            number_words = re.findall(r'[가-힣A-Za-z]*[0-9]+[가-힣A-Za-z]*', chunk_text)
            
            keywords = list(set(korean_words))[:8] + list(set(english_words))[:4] + list(set(number_words))[:2]
            
            # 문장 추출
            sentences = re.split(r'[.!?。！？]\s+', chunk_text)
            meaningful_sentences = [s.strip() for s in sentences if 20 <= len(s.strip()) <= 200][:5]
            
            # 키워드 기반 질문 생성
            question_templates = [
                "{}에 대해 설명해주세요.",
                "{}의 특징은 무엇인가요?",
                "{}은 어떤 의미인가요?",
                "{}와 관련된 내용을 알려주세요.",
                "{}에 대한 자세한 정보를 설명해주세요."
            ]
            
            # 일반적인 negative 샘플 풀 (폴백용)
            negative_samples = [
                "블록체인은 분산 원장 기술로 데이터를 여러 노드에 저장하여 무결성을 보장합니다.",
                "기계학습은 알고리즘을 통해 컴퓨터가 데이터에서 패턴을 찾는 기술입니다.",
                "양자컴퓨팅은 양자역학의 원리를 이용하여 연산을 수행하는 혁신적인 기술입니다.",
                "암호화폐는 블록체인 기술을 기반으로 한 디지털 화폐입니다.",
                "클라우드 컴퓨팅은 인터넷을 통해 컴퓨팅 자원을 제공하는 서비스입니다.",
                "가상현실은 컴퓨터로 생성된 가상 환경에서 몰입감을 제공하는 기술입니다."
            ]
            
            for keyword in keywords[:6]:
                if len(keyword) >= 2:
                    template = question_templates[len(examples) % len(question_templates)]
                    question = template.format(keyword)
                    
                    # 키워드가 포함된 문장이나 주변 텍스트를 답변으로 사용
                    answer_candidates = [s for s in meaningful_sentences if keyword in s]
                    if answer_candidates:
                        answer = answer_candidates[0]
                    else:
                        answer = f"{keyword}는 이 텍스트에서 언급되는 중요한 개념입니다."
                    
                    # 무작위 negative 샘플 선택
                    negative = negative_samples[len(examples) % len(negative_samples)]
                    
                    if len(answer) >= 20:
                        # 품질 점수 계산 (키워드 기반 폴백이므로 중간 점수)
                        score = 0.6 if len(answer) >= 50 else 0.5
                        examples.append({
                            'query': question,
                            'positive': answer,
                            'negative': negative,  # 폴백용 negative 샘플 사용
                            'score': score,        # 품질 점수 추가
                            'type': 'keyword_based'
                        })
            
            # 문장 기반 질문 생성
            for sentence in meaningful_sentences[:4]:
                if len(examples) >= 10:
                    break
                    
                question = f"다음 내용에 대해 설명해주세요: {sentence[:50]}..."
                negative = negative_samples[len(examples) % len(negative_samples)]
                
                examples.append({
                    'query': question,
                    'positive': sentence,
                    'negative': negative,  # 폴백용 negative 샘플 사용
                    'score': 0.7,          # 문장 기반이므로 더 높은 점수
                    'type': 'sentence_based'
                })
            
            # 사용자 지정 개수 고려
            max_count = min(target_count if target_count else 12, len(examples))
            logger.info(f"[QAGenerator] 폴백 QA 생성: {len(examples)}개 → {max_count}개 반환 (목표: {target_count})")
            return examples[:max_count]
            
        except Exception as e:
            logger.error(f"[QAGenerator] 폴백 QA 생성 실패: {str(e)}")
            return []

# 전역 QA 생성기 인스턴스
qa_generator = QAGenerator()

async def generate_qa_pairs_from_text(text: str, dataset_name: str, provider: str = "ollama", model: str = "hamonize:latest", target_qa_count: int = None) -> List[Dict[str, Any]]:
    """텍스트에서 질문-답변 쌍을 LLM으로 생성 (모델별 컨텍스트 최적화)"""
    try:
        logger.info(f"문서 QA 쌍 생성 시작: {len(text)}자 텍스트, 데이터셋: {dataset_name}, 프로바이더: {provider}, 모델: {model}")
        
        # 프로바이더 설정
        qa_generator.set_provider(provider, model)
        
        # 모델별 컨텍스트 길이 제한 설정 (OpenAI 공식 문서 기준)
        model_context_limits = {
            # OpenAI 실제 사용 가능한 모델들
            'gpt-4o': 128000,          # 128k 토큰
            'gpt-4o-mini': 128000,     # 128k 토큰
            'gpt-4-turbo': 128000,     # 128k 토큰
            'gpt-4': 8192,             # 8k 토큰
            'gpt-3.5-turbo': 16385,    # 16k 토큰
            'o1-preview': 128000,      # 128k 토큰
            'o1-mini': 128000,         # 128k 토큰
            # 존재하지 않는 모델들을 실제 모델로 매핑
            'gpt-4.1': 128000,         # → gpt-4o로 처리
            'gpt-4.1-mini': 128000,    # → gpt-4o-mini로 처리  
            'gpt-4.1-nano': 128000,    # → gpt-4o-mini로 처리
            # Anthropic 모델들
            'claude-3-haiku-20240307': 200000,
            'claude-3-sonnet-20240229': 200000,
            'claude-3-opus-20240229': 200000,
            'claude-3-7-sonnet-latest': 200000,
            'claude-sonnet-4-20250514': 200000,
            'claude-opus-4-20250514': 200000,
            # Gemini 모델들
            'gemini-1.5-pro': 2000000,
            'gemini-2.5-pro-preview-05-06': 2000000,
            'gemma-3-27b-it': 8192,
            # Groq 모델들
            'llama-3.3-70b-versatile': 131072,
            'llama3-70b-8192': 8192,
            'gemma2-9b-it': 8192,
            'gemma-7b-it': 8192,
            # 기본값 (Ollama, VLLM 등)
            'default': 128000
        }
        
        # 현재 모델의 컨텍스트 길이 가져오기
        context_limit = model_context_limits.get(model, model_context_limits['default'])
        
        # 안전한 길이 계산 (토큰 수 기준, 1토큰 ≈ 4자로 추정)
        # 입력 토큰 + 출력 토큰 + 여유분을 고려
        max_output_tokens = min(4096, context_limit // 4)  # 출력 토큰 제한
        max_input_chars = (context_limit - max_output_tokens - 500) * 3  # 입력 문자 제한 (여유분 포함)
        
        logger.info(f"모델 {model} 컨텍스트 제한: {context_limit} 토큰, 최대 입력: {max_input_chars}자")
        
        max_single_call_length = min(max_input_chars, 80000)  # 기존 제한과 비교하여 더 작은 값 사용
        
        if len(text) <= max_single_call_length:
            # 🚀 단일 AI 호출로 전체 처리 (최적화된 방식)
            logger.info(f"텍스트 길이 {len(text)}자 - 단일 AI 호출로 처리")
            
            try:
                # 전체 텍스트에 대한 구조적 프롬프트 생성
                prepared_text = qa_generator._prepare_chunk_text(text, max_length=max_single_call_length)
                prompt = qa_generator._create_comprehensive_qa_prompt(prepared_text, dataset_name, target_qa_count)
                
                # 단일 AI 서비스 호출 (사용자 지정 개수 전달)
                response = qa_generator._call_ai_service_structured(prompt, target_qa_count)
                
                if response:
                    # 구조적 응답 파싱
                    qa_pairs = qa_generator._parse_structured_response(response, prepared_text)
                    logger.info(f"단일 호출로 {len(qa_pairs)}개 QA 쌍 생성 완료!")
                    
                    if len(qa_pairs) >= 10:  # 충분한 QA 쌍이 생성된 경우 (임계값 조정)
                        logger.info(f"단일 호출로 충분한 QA 쌍 생성: {len(qa_pairs)}개")
                        return qa_pairs[:250]  # 최대 250개 반환
                    else:
                        logger.warning(f"단일 호출 결과가 부족함 ({len(qa_pairs)}개), 청크 분할 방식으로 전환")
                        # 청크 분할 방식으로 폴백
                else:
                    logger.warning("단일 AI 호출 실패, 청크 분할 방식으로 전환")
                    # 청크 분할 방식으로 폴백
                    
            except Exception as e:
                logger.warning(f"단일 AI 호출 실패: {e}, 청크 분할 방식으로 전환")
                # 청크 분할 방식으로 폴백
        else:
            logger.info(f"텍스트 길이 {len(text)}자 - 청크 분할 방식으로 처리")
        
        # 🔄 폴백: 청크 분할 방식 (기존 방식)
        chunks = []
        words = text.split()
        current_chunk = []
        current_length = 0
        
        # 청크 크기를 모델 컨텍스트에 맞게 설정
        if context_limit <= 16385:  # 작은 컨텍스트 모델 (gpt-3.5-turbo 등)
            min_chunk_size = 2000
            max_chunk_size = 4000
        elif context_limit <= 131072:  # 중간 컨텍스트 모델 (gpt-4o-mini 등)
            min_chunk_size = 8000
            max_chunk_size = 15000
        elif context_limit <= 1000000:  # 대용량 컨텍스트 모델 (GPT-4.1 시리즈)
            min_chunk_size = 30000
            max_chunk_size = 60000
        else:  # 초대용량 컨텍스트 모델 (Claude, Gemini 등)
            min_chunk_size = 50000
            max_chunk_size = 100000
            
        logger.info(f"청크 크기 설정: 최소 {min_chunk_size}자, 최대 {max_chunk_size}자 (컨텍스트: {context_limit} 토큰)")
        
        for word in words:
            current_chunk.append(word)
            current_length += len(word) + 1
            
            # 최소 크기 이상이고 문장이 끝나는 지점에서 분할
            if current_length >= min_chunk_size and word.endswith(('.', '!', '?', '。', '！', '？')):
                chunk_text = ' '.join(current_chunk).strip()
                if len(chunk_text) >= 500:  # 의미있는 길이
                    chunks.append(chunk_text)
                current_chunk = []
                current_length = 0
                
            # 최대 크기 도달 시 강제 분할
            elif current_length >= max_chunk_size:
                chunk_text = ' '.join(current_chunk).strip()
                if len(chunk_text) >= 500:
                    chunks.append(chunk_text)
                current_chunk = []
                current_length = 0
        
        # 마지막 청크 처리
        if current_chunk:
            chunk_text = ' '.join(current_chunk).strip()
            if len(chunk_text) >= 500:
                chunks.append(chunk_text)
        
        logger.info(f"텍스트를 {len(chunks)}개 청크로 분할 (평균 {sum(len(c) for c in chunks)//len(chunks) if chunks else 0}자/청크)")
        
        examples = []
        # 청크당 QA 생성 개수 계산 (사용자 지정 개수 우선)
        if target_qa_count is not None and target_qa_count > 0:
            chunk_target_qa_count = target_qa_count
            logger.info(f"사용자 지정 QA 목표: {target_qa_count}개 (청크 분할 방식)")
        else:
            # 기본값: 청크당 더 많은 QA 생성 (큰 청크 크기에 맞춰)
            chunk_target_qa_count = min(max(len(chunks) * 20, 100), 300)  # 청크당 20개, 최소 100개, 최대 300개
            logger.info(f"기본 QA 목표: {chunk_target_qa_count}개 (청크 분할 방식)")
        
        # 각 청크에서 구조적 QA 생성
        for i, chunk in enumerate(chunks):
            if len(examples) >= chunk_target_qa_count:
                break
                
            try:
                # 청크 텍스트 준비
                prepared_chunk = qa_generator._prepare_chunk_text(chunk, max_length=20000)
                
                # 구조적 프롬프트 생성 (큰 청크용)
                prompt = qa_generator._create_large_chunk_qa_prompt(prepared_chunk, dataset_name, target_qa_count)
                
                # AI 서비스 호출 (사용자 지정 개수 전달)
                response = qa_generator._call_ai_service_structured(prompt, target_qa_count)
                
                if response:
                    # 구조적 응답 파싱
                    qa_pairs = qa_generator._parse_structured_response(response, prepared_chunk)
                    examples.extend(qa_pairs)
                    
                    logger.info(f"청크 {i+1}/{len(chunks)}: {len(qa_pairs)}개 QA 쌍 생성 (총 {len(examples)}개)")
                else:
                    # 폴백 QA 생성 (사용자 지정 개수 고려)
                    fallback_qa = qa_generator._fallback_qa_extraction(prepared_chunk, target_qa_count)
                    examples.extend(fallback_qa)
                    logger.warning(f"청크 {i+1}/{len(chunks)}: AI 호출 실패, 폴백으로 {len(fallback_qa)}개 생성")
                    
            except Exception as e:
                logger.warning(f"청크 {i+1} QA 생성 실패: {e}")
                # 폴백 QA 생성 (사용자 지정 개수 고려)
                fallback_qa = qa_generator._fallback_qa_extraction(chunk, target_qa_count)
                examples.extend(fallback_qa)
        
        # 중복 제거 및 품질 필터링
        unique_examples = []
        seen_queries = set()
        
        for example in examples:
            query = example['query'].strip().lower()
            # 더 관대한 중복 검사 (유사성 기반)
            is_duplicate = False
            for seen_query in seen_queries:
                # 간단한 유사성 검사 (70% 이상 겹치면 중복으로 판단)
                if len(set(query.split()) & set(seen_query.split())) / max(len(query.split()), len(seen_query.split())) > 0.7:
                    is_duplicate = True
                    break
            
            if not is_duplicate and len(example['positive'].strip()) >= 30:
                seen_queries.add(query)
                unique_examples.append(example)
        
        # 다양성 확보를 위한 셔플
        import random
        random.shuffle(unique_examples)
        
        final_count = min(len(unique_examples), 300)  # 최대 300개
        result = unique_examples[:final_count]
        
        logger.info(f"QA 쌍 생성 완료: {len(result)}개 (중복 제거 후)")
        return result
        
    except Exception as e:
        logger.error(f"QA 쌍 생성 실패: {e}")
        return []

def parse_qa_response(llm_response: str, source_text: str) -> List[Dict[str, Any]]:
    """LLM 응답에서 Q&A 쌍을 파싱"""
    examples = []
    
    try:
        
        # Q: ... A: ... 패턴 찾기
        qa_pattern = r'Q:\s*(.+?)\s*A:\s*(.+?)(?=Q:|$)'
        matches = re.findall(qa_pattern, llm_response, re.DOTALL | re.IGNORECASE)
        
        for question, answer in matches:
            question = question.strip()
            answer = answer.strip()
            
            # 너무 짧거나 긴 질문/답변 필터링
            if 10 <= len(question) <= 200 and 20 <= len(answer) <= 1000:
                # 답변 길이와 내용 복잡도에 따른 품질 점수 계산
                score = min(0.9, 0.5 + (len(answer) / 500) * 0.3)  # 길이에 따라 0.5~0.8
                score += 0.1 if any(keyword in answer.lower() for keyword in ['때문에', '따라서', '그러므로', '왜냐하면']) else 0  # 논리적 연결어 보너스
                
                example = {
                    'query': question,
                    'positive': answer,
                    'negative': '',
                    'score': round(score, 2),  # 품질 점수 추가
                    'type': 'ai_generated'
                }
                examples.append(example)
        
        # 패턴 매칭이 실패한 경우 라인별로 시도
        if not examples:
            lines = llm_response.split('\n')
            current_q = None
            
            for line in lines:
                line = line.strip()
                if line.startswith('Q:') or line.startswith('질문:'):
                    current_q = re.sub(r'^(Q:|질문:)\s*', '', line)
                elif line.startswith('A:') or line.startswith('답변:'):
                    if current_q:
                        answer = re.sub(r'^(A:|답변:)\s*', '', line)
                        if 10 <= len(current_q) <= 200 and 20 <= len(answer) <= 1000:
                            # 라인별 파싱이므로 품질이 조금 낮음
                            score = min(0.8, 0.4 + (len(answer) / 400) * 0.3)  # 0.4~0.7 범위
                            
                            example = {
                                'query': current_q,
                                'positive': answer,
                                'negative': '',
                                'score': round(score, 2),  # 품질 점수 추가
                                'type': 'line_parsed'
                            }
                            examples.append(example)
                        current_q = None
        
    except Exception as e:
        logger.warning(f"QA 파싱 실패: {e}")
    
    return examples

def generate_fallback_questions(chunk: str) -> List[Dict[str, Any]]:
    """LLM 호출 실패 시 사용할 폴백 질문 생성"""
    examples = []
    
    # 키워드 추출
    words = re.findall(r'\b\w+\b', chunk)
    important_words = [w for w in words if len(w) >= 3][:10]
    
    # 문장 추출
    sentences = re.split(r'[.!?。！？]', chunk)
    meaningful_sentences = [s.strip() for s in sentences if len(s.strip()) >= 20][:5]
    
    # 다양한 질문 패턴 생성
    question_templates = [
        "{}에 대해 설명해주세요.",
        "{}의 특징은 무엇인가요?",
        "{}은 어떤 의미인가요?",
        "{}과 관련된 내용을 설명해주세요.",
        "{}에 대한 정보를 알려주세요."
    ]
    
    # 키워드 기반 질문
    for word in important_words[:3]:
        for template in question_templates[:2]:
            question = template.format(word)
            example = {
                'query': question,
                'positive': chunk[:500] + ('...' if len(chunk) > 500 else ''),
                'negative': '',
                'score': 0.4,  # 폴백 생성이므로 낮은 점수
                'type': 'fallback_keyword'
            }
            examples.append(example)
    
    # 문장 기반 질문
    for sentence in meaningful_sentences[:3]:
        question = f"'{sentence[:50]}...'에 대해 자세히 설명해주세요."
        example = {
            'query': question,
            'positive': chunk[:500] + ('...' if len(chunk) > 500 else ''),
            'negative': '',
            'score': 0.5,  # 문장 기반이므로 중간 점수
            'type': 'fallback_sentence'
        }
        examples.append(example)
    
    return examples[:8]  # 청크당 최대 8개

def generate_template_examples(template_type: str) -> List[Dict[str, Any]]:
    """템플릿 타입에 따라 예제 생성"""
    templates = {
        'medical': [
            {'query': '두통의 원인은 무엇인가요?', 'positive': '두통은 스트레스, 수면 부족, 탈수, 눈의 피로 등 다양한 원인으로 발생할 수 있습니다.', 'score': 0.8, 'type': 'template_medical'},
            {'query': '감기 증상은 어떤 것들이 있나요?', 'positive': '감기의 주요 증상으로는 콧물, 기침, 목 아픔, 발열, 몸살 등이 있습니다.', 'score': 0.8, 'type': 'template_medical'},
            {'query': '혈압이 높으면 어떤 문제가 생기나요?', 'positive': '고혈압은 심장병, 뇌졸중, 신장 질환 등의 위험을 증가시킬 수 있습니다.', 'score': 0.8, 'type': 'template_medical'},
        ],
        'legal': [
            {'query': '계약서 작성 시 주의사항은?', 'positive': '계약서 작성 시에는 당사자 정보, 계약 내용, 이행 기간, 위반 시 조치 등을 명확히 기재해야 합니다.', 'score': 0.8, 'type': 'template_legal'},
            {'query': '민사소송 절차는 어떻게 되나요?', 'positive': '민사소송은 소장 제출, 답변서 제출, 변론 기일, 증거 조사, 판결 순으로 진행됩니다.', 'score': 0.8, 'type': 'template_legal'},
            {'query': '임금 체불 시 어떻게 해야 하나요?', 'positive': '임금 체불 시에는 고용노동부 신고, 노동위원회 진정, 민사소송 등의 방법이 있습니다.', 'score': 0.8, 'type': 'template_legal'},
        ],
        'technical': [
            {'query': 'API란 무엇인가요?', 'positive': 'API(Application Programming Interface)는 서로 다른 소프트웨어가 통신할 수 있도록 하는 인터페이스입니다.', 'score': 0.8, 'type': 'template_technical'},
            {'query': '데이터베이스 인덱스의 역할은?', 'positive': '데이터베이스 인덱스는 데이터 검색 속도를 향상시키기 위해 사용되는 자료 구조입니다.', 'score': 0.8, 'type': 'template_technical'},
            {'query': 'RESTful API의 특징은?', 'positive': 'RESTful API는 HTTP 메서드를 사용하여 자원을 표현하고 조작하는 아키텍처 스타일입니다.', 'score': 0.8, 'type': 'template_technical'},
        ],
        'general': [
            {'query': '지구 온난화의 원인은?', 'positive': '지구 온난화의 주요 원인은 온실가스 배출, 특히 이산화탄소 증가입니다.', 'score': 0.8, 'type': 'template_general'},
            {'query': '건강한 생활습관은?', 'positive': '건강한 생활습관으로는 규칙적인 운동, 균형 잡힌 식사, 충분한 수면이 있습니다.', 'score': 0.8, 'type': 'template_general'},
            {'query': '효율적인 학습 방법은?', 'positive': '효율적인 학습을 위해서는 목표 설정, 집중 시간 확보, 반복 학습이 중요합니다.', 'score': 0.8, 'type': 'template_general'},
        ]
    }
    
    return templates.get(template_type, templates['general'])

# ========== 전용 프로세스 분리 아키텍처 ==========

# 검색 전용 워커 프로세스
def search_worker(search_queue, response_queue, search_model_ready_event, max_threads=8):
    """검색 전용 워커 프로세스 - 다중 스레드로 동시 처리 지원"""
    # 필수 모듈들 먼저 import
    import sys
    import os

    # 프로세스명 환경변수 설정
    os.environ['PROCESS_NAME'] = 'search-worker'
    import time
    import logging
    import numpy as np
    import psycopg2
    from psycopg2.extras import RealDictCursor
    
    # 워커 프로세스에서는 메인 프로세스 마커 제거하여 GPU 사용 가능
    if 'AIRUN_MAIN_PROCESS' in os.environ:
        del os.environ['AIRUN_MAIN_PROCESS']
    
    # 워커 타입 설정 (rag_process.py에서 인식)
    os.environ['WORKER_TYPE'] = 'search'
    # 현재 스크립트의 디렉토리를 Python 경로에 추가
    current_dir = os.path.dirname(os.path.abspath(__file__))
    parent_dir = os.path.dirname(os.path.dirname(current_dir))
    if parent_dir not in sys.path:
        sys.path.insert(0, parent_dir)
    if current_dir not in sys.path:
        sys.path.insert(0, current_dir)

    
    # 로깅을 메인 로그 파일로 설정
    log_file = os.path.expanduser("~/.airun/logs/airun-rag.log")
    
    # 환경변수에서 로그 레벨 확인
    log_level_str = os.getenv('LOG_LEVEL', 'INFO').upper()
    log_level = getattr(logging, log_level_str, logging.INFO)
    
    logging.basicConfig(
        level=log_level,
        format='[%(asctime)s] [SEARCH_WORKER] %(levelname)s: %(message)s',
        handlers=[
            logging.FileHandler(log_file, mode='a'),  # append mode
            logging.StreamHandler()
        ]
    )

    # HuggingFace HTTP 요청 로그 비활성화
    logging.getLogger("urllib3").setLevel(logging.WARNING)
    logging.getLogger("httpx").setLevel(logging.WARNING)

    logger = logging.getLogger("SearchWorker")

    logger.info(f"검색 워커 시작 (스레드: {max_threads}개)")
    
    # rag_process.py 모듈 임포트 (워커 프로세스 전용 최종 해결책)
    DocumentProcessor = None
    PostgreSQLWrapper = None
    try:
        import importlib
        import sys
        import os
        
        # rag_process 모듈 임포트
        from rag_process import DocumentProcessor, PostgreSQLWrapper
        
    except Exception as e:
        logger.error(f"검색 워커: rag_process 모듈 임포트 실패: {e}")
        import traceback
        logger.error(f"검색 워커: 상세 에러 정보: {traceback.format_exc()}")
        DocumentProcessor = None
        PostgreSQLWrapper = None
    
    # GPU/CPU 설정은 rag_process.py에서 중앙 관리됨
    # 워커 프로세스에서는 rag_process 설정을 따름
    logger.info("검색 워커: rag_process.py 설정에 따라 GPU/CPU 모드 결정됨")
    
    # 통합된 연결 풀 사용
    
    # DocumentProcessor가 DB 풀을 관리하므로 별도 풀 생성 불필요
    # logger.info("검색 워커: DocumentProcessor DB 풀 사용 예정")
    
    REDIS_HOST = "localhost"
    REDIS_PORT = 6379
    
    try:
        # PostgreSQL 연결은 DocumentProcessor 초기화 후에 설정
        # logger.info("검색 워커: DocumentProcessor를 통한 연결 대기")
        conn = None
        cursor = None
        
        # RAG 설정에서 임베딩 모델명 읽기
        from rag_process import get_rag_settings
        rag_settings = get_rag_settings()
        search_embedding_model_name = rag_settings.get('embedding_model', 'nlpai-lab/KURE-v1')
        
        # SentenceTransformer 모델 로딩 (검색 워커 전용 - 텍스트 임베딩 모델만)
        logger.info(f"🔍 검색 워커: 텍스트 임베딩 모델 로딩 시작 - {search_embedding_model_name}")
        # 직접 모델 로딩 (메인 프로세스 의존성 제거)
        # logger.info("🔍 검색 워커: 독립적으로 모델 로딩 시작")

        # 검색 워커: rag_process.py의 get_embedding_model 사용 (중복 로딩 방지)
        from rag_process import get_embedding_model, get_rag_settings
        rag_settings = get_rag_settings()
        search_embedding_model = get_embedding_model(settings=rag_settings, logger=logger)
        logger.info("검색 워커: rag_process.py에서 임베딩 모델 로딩 완료")
        
        # Redis 연결
        logger.info("검색 워커: Redis 연결 시작")
        import redis
        redis_client = redis.Redis(host=REDIS_HOST, port=REDIS_PORT, decode_responses=True)
        redis_client.ping()  # 연결 테스트
        logger.info("검색 워커: Redis 연결 완료")
        
        # DocumentProcessor 싱글톤 인스턴스 사용 (중복 생성 방지)
        logger.info("검색 워커: DocumentProcessor 싱글톤 인스턴스 획득 시작")
        try:
            doc_processor = DocumentProcessor.get_instance()
            logger.info("검색 워커: DocumentProcessor 싱글톤 인스턴스 획득 완료")

            # PostgreSQL 컬렉션 래퍼 - 전역 DB 풀 사용
            db_pool = get_global_postgresql()
            collection = PostgreSQLWrapper(db_pool)
            logger.info("검색 워커: PostgreSQL 래퍼가 전역 DB 풀 사용")

        except Exception as e:
            logger.error(f"검색 워커: DocumentProcessor 인스턴스 생성 실패: {e}")
            doc_processor = None
            # Fallback: GLOBAL_DB_POOL 사용 (사용하지 않음)
            logger.error("검색 워커: DocumentProcessor 초기화 실패")
            return

        # DocumentProcessor 초기화 후 DB 연결 및 pgvector 확인
        if doc_processor:
            logger.info("검색 워커: DocumentProcessor를 통한 DB 연결")
            conn = get_pool_connection()
            cursor = conn.cursor()

            # pgvector 확장 확인
            cursor.execute("SELECT extversion FROM pg_extension WHERE extname = 'vector'")
            result = cursor.fetchone()
            if result:
                logger.info(f"검색 워커: pgvector {result[0]} 확인 완료")
            else:
                logger.error("검색 워커: pgvector 확장이 설치되지 않음")
                doc_processor.return_db_connection(conn)
                return

            # 연결 반환 (검색 워커는 검색 시마다 개별 연결 획득)
            doc_processor.return_db_connection(conn)
            logger.info("검색 워커: PostgreSQL 연결 확인 완료")

        # 이미지 임베딩 모델 상태 확인 및 강제 사전 로딩 (검색 성능 최적화)
        if doc_processor and hasattr(doc_processor, 'image_embedding_model') and doc_processor.image_embedding_model:
            logger.info("검색 워커: 이미지 임베딩 모델 사용 가능")
        elif doc_processor:
            logger.info("검색 워커: DocumentProcessor를 통한 이미지 모델 사용 가능")
        
        if DocumentProcessor is None:
            logger.error("검색 워커: DocumentProcessor 클래스를 찾을 수 없음")
        else:
            logger.info("검색 워커: DocumentProcessor 사용 가능")
        
        # 모델 준비 완료 신호
        search_model_ready_event.set()
        logger.info("검색 워커: 모든 구성 요소 준비 완료")
        
        async def search_with_document_processor(query, top_k=10, user_id=None, rag_search_scope='personal'):
            """DocumentProcessor를 사용한 하이브리드 검색"""
            if doc_processor is None:
                logger.error("검색 워커: DocumentProcessor 인스턴스가 초기화되지 않음")
                return []
            
            try:
                # 기존 DocumentProcessor 인스턴스 재사용 (새로 생성하지 않음)
                

                
                # RAG 설정 가져오기
                rag_settings = get_rag_settings()
                
                # DEBUG: temp_settings 로깅
                temp_settings = {
                    'top_k': top_k,
                    'large_context_chunking': rag_settings.get('large_context_chunking', 'no')
                }
                logger.info(f"🔧 [DEBUG] airun-rag.py temp_settings: {temp_settings}")
                logger.info(f"🔧 [DEBUG] large_context_chunking value: {temp_settings['large_context_chunking']}")
                
                # rag_process.py의 search 메소드 사용 (하이브리드 검색)
                results = await doc_processor.search(
                    query=query, 
                    user_id=user_id,
                    temp_settings=temp_settings,
                    rag_search_scope=rag_search_scope
                )
                
                # 결과를 검색 워커 포맷으로 변환
                search_results = []
                for result in results:
                    search_results.append({
                        'content': result['content'],
                        'metadata': {
                            'source': result.get('metadata', {}).get('source', ''),
                            'doc_id': result.get('metadata', {}).get('doc_id', ''),
                            'chunk_id': result.get('metadata', {}).get('chunk_index', 0),
                            'page_number': result.get('metadata', {}).get('page_number', 1),
                            'chunk_index': result.get('metadata', {}).get('chunk_index', 0),
                            'total_chunks': result.get('metadata', {}).get('total_chunks', 0),
                            'timestamp': result.get('metadata', {}).get('timestamp', ''),
                            'modification_date': result.get('metadata', {}).get('modification_date', ''),
                            'user_id': result.get('metadata', {}).get('user_id', 'all'),
                            'is_large_context_chunk': result.get('metadata', {}).get('is_large_context_chunk', False),
                            'original_match': result.get('metadata', {}).get('original_match', False),
                            'core_scores': result.get('metadata', {}).get('core_scores', {}),
                            'main_scores': result.get('metadata', {}).get('main_scores', {}),
                            'sub_scores': result.get('metadata', {}).get('sub_scores', {}),
                            'matches': result.get('metadata', {}).get('matches', {}),
                            'thresholds': result.get('metadata', {}).get('thresholds', {}),
                            'main_weights': result.get('metadata', {}).get('main_weights', {}),
                            'sub_weights': result.get('metadata', {}).get('sub_weights', {})
                        },
                        'score': result.get('metadata', {}).get('final_score', 0.0),
                        'distance': 1.0 - result.get('metadata', {}).get('final_score', 0.0)  # 거리는 1-점수로 근사
                    })
                
                return search_results
                
            except Exception as e:
                logger.error(f"검색 워커: DocumentProcessor 검색 실패: {e}")
                import traceback
                logger.error(f"검색 워커: 상세 오류: {traceback.format_exc()}")
                return []
        
        # ThreadPoolExecutor를 사용한 다중 스레드 검색 처리
        from concurrent.futures import ThreadPoolExecutor
        import threading

        def process_search_request(request):
            """단일 검색 요청 처리 함수"""
            request_id = request['request_id']
            query = request['query']
            top_k = request.get('top_k', 10)
            user_id = request.get('user_id', None)  # user_id 추출 추가
            rag_search_scope = request.get('rag_search_scope', 'personal')  # rag_search_scope 추출 추가

            thread_id = threading.current_thread().name
            logger.info(f"검색 워커 [{thread_id}]: 검색 요청 처리 시작 - ID: {request_id}, 쿼리: {query}")

            start_time = time.time()

            try:
                # DocumentProcessor를 사용한 하이브리드 검색 실행
                import asyncio
                search_results = asyncio.run(search_with_document_processor(query, top_k, user_id=user_id, rag_search_scope=rag_search_scope))

                # DEBUG 모드일 때 검색 결과 상세 정보 출력
                if os.getenv('LOG_LEVEL', 'INFO').upper() == 'DEBUG':
                    logger.debug(f"=== 검색 워커 [{thread_id}] DEBUG 정보 ===")
                    logger.debug(f"요청 ID: {request_id}")
                    logger.debug(f"검색 쿼리: '{query}'")
                    logger.debug(f"요청된 Top-K: {top_k}")
                    logger.debug(f"실제 반환된 결과 수: {len(search_results)}")

                    if search_results:
                        logger.debug("=== Top-K 검색 결과 점수 ===")
                        for i, result in enumerate(search_results):
                            score = result.get('score', 0)
                            distance = result.get('distance', 1)
                            content_preview = result.get('content', '')[:100].replace('\n', ' ')
                            logger.debug(f"  순위 {i+1}: 점수={score:.2f}, 거리={distance:.2f}, 내용='{content_preview}...'")
                    else:
                        logger.debug("검색 결과가 없습니다.")
                    logger.debug("==========================")

                # 결과에 순위 추가
                for i, result in enumerate(search_results):
                    result['rank'] = i + 1

                processing_time = time.time() - start_time
                logger.info(f"검색 워커 [{thread_id}]: 검색 완료 - ID: {request_id}, 시간: {processing_time:.2f}초, 결과: {len(search_results)}개")

                # 응답 전송
                response = {
                    'request_id': request_id,
                    'success': True,
                    'results': search_results,
                    'processing_time': processing_time,
                    'total_results': len(search_results)
                }
                response_queue.put(response)

            except Exception as e:
                logger.error(f"검색 워커 [{thread_id}]: 검색 처리 실패 - ID: {request_id}, 오류: {e}")
                import traceback
                logger.error(f"검색 워커 [{thread_id}]: 상세 오류: {traceback.format_exc()}")

                # 오류 응답 전송
                error_response = {
                    'request_id': request_id,
                    'success': False,
                    'error': str(e),
                    'results': []
                }
                response_queue.put(error_response)

        # ThreadPoolExecutor를 사용한 요청 처리 루프
        with ThreadPoolExecutor(max_workers=max_threads, thread_name_prefix="SearchThread") as executor:
            logger.info(f"검색 워커: ThreadPoolExecutor 시작 ({max_threads}개 스레드)")

            while True:
                try:
                    if not search_queue.empty():
                        # 큐에서 모든 대기 중인 요청을 가져와서 동시 처리
                        requests_to_process = []
                        while not search_queue.empty() and len(requests_to_process) < max_threads:
                            try:
                                request = search_queue.get_nowait()
                                requests_to_process.append(request)
                            except:
                                break

                        if requests_to_process:
                            # 모든 요청을 동시에 처리
                            futures = [executor.submit(process_search_request, req) for req in requests_to_process]
                            logger.info(f"검색 워커: {len(requests_to_process)}개 요청을 동시 처리 시작")

                            # 완료 대기 (블로킹하지 않음)
                            for future in futures:
                                pass  # 결과는 각 스레드에서 직접 response_queue에 전송
                    else:
                        time.sleep(0.05)  # CPU 부하 감소

                except Exception as e:
                    logger.error(f"검색 워커: 메인 루프 오류: {e}")
                    time.sleep(0.1)
                
    except Exception as e:
        logger.error(f"검색 워커: 초기화 오류 발생: {e}")
        import traceback
        logger.error(f"검색 워커: 전체 스택 트레이스: {traceback.format_exc()}")

# 개선된 상태 업데이트 헬퍼 함수
def update_embedding_status_sync(user_id, filename, status, message=None):
    """새로운 연결로 안전하게 임베딩 상태를 업데이트하는 함수 (DB 연결 재시도 로직 포함)"""

    for retry_attempt in range(3):  # 최대 3회 재시도
        conn = None
        cursor = None
        try:
            # 연결 풀에서 DB 연결 가져오기
            conn = get_pool_connection()
            if not conn:
                raise Exception("DB 연결 획득 실패")

            # 먼저 기존 트랜잭션이 있다면 종료
            if not conn.autocommit:
                try:
                    conn.rollback()
                except Exception:
                    pass

            conn.autocommit = True  # 자동 커밋 활성화
            cursor = conn.cursor()

            # 상태 업데이트 실행
            if message:
                cursor.execute("""
                    UPDATE chat_documents
                    SET embedding_status = %s, error_message = %s, processed_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
                    WHERE user_id = %s AND filename = %s
                """, (status, message, user_id, filename))
            else:
                cursor.execute("""
                    UPDATE chat_documents
                    SET embedding_status = %s, processed_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
                    WHERE user_id = %s AND filename = %s
                """, (status, user_id, filename))

            # 업데이트된 행 수 확인
            rows_affected = cursor.rowcount
            if rows_affected > 0:
                print(f"상태 업데이트 성공: {filename} -> {status} (영향받은 행: {rows_affected})")
                return True
            else:
                print(f"상태 업데이트 실패: {filename} 문서를 찾을 수 없음")
                return False

        except Exception as e:
            print(f"상태 업데이트 실패 (시도 {retry_attempt + 1}/3): {e}")
            if retry_attempt == 2:  # 마지막 시도에서도 실패
                print(f"상태 업데이트 최종 실패: {filename}")
                return False
            else:
                time.sleep(1)  # 1초 대기 후 재시도

        finally:
            # 연결 정리
            try:
                if cursor:
                    cursor.close()
                if conn:
                    return_pool_connection(conn)
            except:
                pass

    return False

def sync_embedding_status():
    """임베딩은 완료되었지만 chat_documents 상태가 pending인 문서들을 자동으로 completed로 업데이트"""
    try:
        logger.info("🔄 임베딩 상태 동기화 시작")

        conn = get_pool_connection()
        if not conn:
            logger.error("❌ DB 연결 실패")
            return

        # 먼저 기존 트랜잭션이 있다면 종료
        if not conn.autocommit:
            try:
                conn.rollback()
            except Exception:
                pass

        conn.autocommit = True  # 자동 커밋 활성화
        cursor = conn.cursor()

        # 먼저 모든 pending 문서들 확인
        logger.info("🔍 단계 1: pending 상태인 모든 문서 확인")
        cursor.execute("""
            SELECT id, user_id, filename, embedding_status, created_at
            FROM chat_documents 
            WHERE embedding_status = 'pending'
            ORDER BY created_at DESC
            LIMIT 10
        """)
        all_pending = cursor.fetchall()
        logger.info(f"📋 현재 pending 상태 문서: {len(all_pending)}개")
        for doc in all_pending:
            logger.info(f"   - ID:{doc[0]}, 사용자:{doc[1]}, 파일명:{doc[2]}, 상태:{doc[3]}, 생성:{doc[4]}")

        # 임베딩 테이블에서 해당 문서들 확인
        logger.info("🔍 단계 2: 임베딩 테이블에서 관련 문서 확인")
        cursor.execute("""
            SELECT DISTINCT user_id, filename
            FROM document_embeddings
            WHERE filename LIKE '%graphrag_test%'
            ORDER BY user_id
        """)
        embeddings = cursor.fetchall()
        logger.info(f"📋 graphrag_test 관련 임베딩: {len(embeddings)}개")
        for emb in embeddings:
            logger.info(f"   - 사용자:{emb[0]}, 파일명:{emb[1]}")

        # 실제 동기화 쿼리 실행
        logger.info("🔍 단계 3: 동기화 대상 검색")
        sync_query = """
            SELECT DISTINCT cd.user_id, cd.filename, cd.id, de.filename as embedding_filename
            FROM chat_documents cd
            JOIN document_embeddings de ON (
                de.filename LIKE '%' || cd.filename || '%'
                AND de.user_id = cd.user_id
            )
            WHERE cd.embedding_status = 'pending'
        """
        cursor.execute(sync_query)
        pending_docs = cursor.fetchall()

        logger.info(f"🔄 동기화 쿼리 결과: {len(pending_docs)}개 문서 발견")
        for doc in pending_docs:
            logger.info(f"   - 문서ID:{doc[2]}, 사용자:{doc[0]}, 파일명:{doc[1]}, 임베딩파일:{doc[3]}")

        if pending_docs:
            logger.info(f"🔄 상태 동기화 대상: {len(pending_docs)}개 문서")

            for user_id, filename, doc_id, embedding_filename in pending_docs:
                try:
                    logger.info(f"🔄 업데이트 시도: 문서ID {doc_id}, 파일명 {filename}")
                    
                    # 상태 업데이트
                    cursor.execute("""
                        UPDATE chat_documents
                        SET embedding_status = 'completed',
                            processed_at = CURRENT_TIMESTAMP,
                            updated_at = CURRENT_TIMESTAMP
                        WHERE id = %s
                    """, (doc_id,))

                    rows_affected = cursor.rowcount
                    logger.info(f"📊 UPDATE 쿼리 실행 결과: {rows_affected}행 영향받음")
                    
                    if rows_affected > 0:
                        logger.info(f"✅ 상태 동기화 완료: {filename} (사용자: {user_id}) - {rows_affected}행 업데이트")
                        
                        # 업데이트 확인
                        cursor.execute("SELECT embedding_status FROM chat_documents WHERE id = %s", (doc_id,))
                        new_status = cursor.fetchone()
                        logger.info(f"🔍 업데이트 후 상태 확인: {new_status[0] if new_status else 'NULL'}")
                    else:
                        logger.warning(f"⚠️ 상태 동기화 실패: {filename} - 업데이트된 행 없음")
                except Exception as e:
                    logger.error(f"❌ 상태 동기화 실패: {filename} - {e}")
                    import traceback
                    logger.error(f"상세 오류: {traceback.format_exc()}")
        else:
            logger.info("🔄 상태 동기화 필요한 문서 없음")

        return_pool_connection(conn)

    except Exception as e:
        logger.error(f"❌ 임베딩 상태 동기화 실패: {e}")
        import traceback
        logger.error(f"상세 오류: {traceback.format_exc()}")

# 임베딩 전용 워커 프로세스
def embedding_worker(embedding_queue, embedding_model_ready_event):
    """임베딩 전용 워커 프로세스 - PostgreSQL/pgvector 저장소"""
    # 프로세스명 환경변수 설정
    import os
    os.environ['PROCESS_NAME'] = 'embedding-worker'

    # 워커 프로세스에서는 메인 프로세스 마커 제거하여 GPU 사용 가능
    if 'AIRUN_MAIN_PROCESS' in os.environ:
        del os.environ['AIRUN_MAIN_PROCESS']
    
    # 워커 타입 설정 (rag_process.py에서 인식)
    os.environ['WORKER_TYPE'] = 'embedding'
    
    import sys
    import time
    import logging
    import json
    import numpy as np
    import psycopg2
    from psycopg2.extras import Json
    import traceback
    
    # 현재 스크립트의 디렉토리를 Python 경로에 추가
    current_dir = os.path.dirname(os.path.abspath(__file__))
    parent_dir = os.path.dirname(os.path.dirname(current_dir))
    if parent_dir not in sys.path:
        sys.path.insert(0, parent_dir)
        
    # rag_process.py에서 필요한 기능들 import
    from rag_process import update_doc_status, DocumentProcessor, get_rag_settings
    
    # 로깅을 메인 로그 파일로 설정
    log_file = os.path.expanduser("~/.airun/logs/airun-rag.log")
    logging.basicConfig(
        level=logging.INFO,
        format='[%(asctime)s] [EMBEDDING_WORKER] %(levelname)s: %(message)s',
        handlers=[
            logging.FileHandler(log_file, mode='a'),  # append mode
            logging.StreamHandler()
        ]
    )
    
    # HuggingFace HTTP 요청 로그 비활성화
    logging.getLogger("urllib3").setLevel(logging.WARNING)
    logging.getLogger("httpx").setLevel(logging.WARNING)
    
    logger = logging.getLogger("EmbeddingWorker")
    
    logger.info("임베딩 워커 시작")
    
    # 통합된 연결 풀 사용
    
    REDIS_HOST = "localhost"
    REDIS_PORT = 6379
    
    try:
        # Embedding Worker에서 전용 Connection Pool 생성
        logger.info("임베딩 워커: PostgreSQL 연결 풀 생성")
        
        # DocumentProcessor 초기화가 완료될 때까지 대기
        conn = None
        cursor = None
        
        # RAG 설정에서 임베딩 모델명 읽기
        from rag_process import get_rag_settings
        rag_settings = get_rag_settings()
        embedding_model_name = rag_settings.get('embedding_model', 'nlpai-lab/KURE-v1')
        
        # SentenceTransformer 모델 로딩 (임베딩 워커 전용 - 텍스트 임베딩 모델만)
        logger.info(f"🔧 임베딩 워커: 텍스트 임베딩 모델 로딩 시작 - {embedding_model_name}")

        # 임베딩 워커: rag_process.py의 get_embedding_model 사용 (중복 로딩 방지)
        from rag_process import get_embedding_model
        embedding_model = get_embedding_model(settings=rag_settings, logger=logger)
        # logger.info("임베딩 워커: rag_process.py에서 임베딩 모델 로딩 완료")
        
        # Redis 연결
        logger.info("임베딩 워커: Redis 연결 시작")
        import redis
        redis_client = redis.Redis(host=REDIS_HOST, port=REDIS_PORT, decode_responses=True)
        redis_client.ping()  # 연결 테스트
        logger.info("임베딩 워커: Redis 연결 완료")

        # 이미지 임베딩 모델 전용 초기화 (DocumentProcessor 초기화 전에 수행)
        logger.info("임베딩 워커: 이미지 임베딩 모델 사전 로딩 시작")
        worker_image_model = None
        try:
            # RAG 설정에서 이미지 임베딩 설정 확인
            rag_settings = get_rag_settings()
            generate_image_embeddings = rag_settings.get('generate_image_embeddings', 'yes').lower() in ('yes', 'true', '1')

            if generate_image_embeddings:
                # 임베딩 워커: rag_process.py의 get_image_embedding_model 사용 (중복 로딩 방지)
                from rag_process import get_image_embedding_model
                worker_image_model = get_image_embedding_model()
                
                if worker_image_model:
                    logger.info("임베딩 워커: rag_process.py에서 이미지 임베딩 모델 로딩 완료")
                else:
                    logger.info("임베딩 워커: 이미지 임베딩 모델 로딩 실패")
            else:
                logger.info("임베딩 워커: 이미지 임베딩 설정이 비활성화됨")

        except Exception as e:
            logger.error(f"임베딩 워커: 이미지 임베딩 모델 로딩 실패 - {e}")
            traceback.print_exc()

        # DocumentProcessor 초기화
        logger.info("임베딩 워커: DocumentProcessor 초기화 시작")
        try:
            # DocumentProcessor를 임베딩 워커에서 사용하기 위한 준비
            # DocumentProcessor는 PostgreSQL 초기화가 필요하므로 별도로 처리
            # 임베딩 워커는 DocumentProcessor의 extract_text, process_image 등의 메서드만 사용

            # DocumentProcessor와 필요한 기능들 import
            from rag_process import (
                DocumentProcessor,
                split_text,
                SUPPORTED_EXTENSIONS,
                QuestionBasedEmbeddingGenerator,
                get_rag_settings
            )

            # DocumentProcessor 인스턴스 생성
            document_processor = DocumentProcessor.get_instance()

            # DocumentProcessor 이미지 모델 연결 상태 확인
            logger.info("임베딩 워커: DocumentProcessor 이미지 모델 연결 확인 완료")
            
            # 텍스트 추출 및 이미지 처리를 위한 준비
            logger.info("임베딩 워커: 문서 처리 모듈 import 완료")
            
            # QuestionBasedEmbeddingGenerator 생성 (향상된 임베딩용)
            question_generator = QuestionBasedEmbeddingGenerator()
            logger.info("임베딩 워커: QuestionBasedEmbeddingGenerator 초기화 완료")
            
        except Exception as e:
            logger.error(f"임베딩 워커: 문서 처리 모듈 초기화 실패 - {e}")
            traceback.print_exc()
            # 기본 기능만 사용하도로 fallback
            document_processor = None
            question_generator = None
            logger.warning("임베딩 워커: 기본 모드로 동작합니다")

        # DocumentProcessor 초기화 후 DB 연결 획득
        if document_processor:
            logger.info("임베딩 워커: DocumentProcessor DB 풀 사용")
            conn = get_pool_connection()
            logger.info("임베딩 워커: DocumentProcessor DB 풀에서 연결 획득")

            # 연결별 설정 적용 (타임아웃 방지)
            # 먼저 기존 트랜잭션이 있다면 종료
            if not conn.autocommit:
                try:
                    conn.rollback()
                except Exception:
                    pass

            conn.autocommit = True  # 자동 커밋으로 변경하여 장시간 트랜잭션 방지
            cursor = conn.cursor()

            # 연결별 타임아웃 설정 (임베딩 처리용)
            cursor.execute("SET statement_timeout = '30min'")  # 개별 쿼리 타임아웃 30분으로 확장
            cursor.execute("SET idle_in_transaction_session_timeout = '30min'")  # 트랜잭션 타임아웃 30분으로 확장

            # pgvector 확장 확인
            cursor.execute("SELECT extversion FROM pg_extension WHERE extname = 'vector'")
            result = cursor.fetchone()
            if result:
                logger.info(f"임베딩 워커: pgvector {result[0]} 확인 완료")
            else:
                logger.error("임베딩 워커: pgvector 확장이 설치되지 않음")
                return
        else:
            logger.error("임베딩 워커: DocumentProcessor 초기화 실패 - 종료")
            return

        # 모델 준비 완료 신호
        embedding_model_ready_event.set()
        logger.info("임베딩 워커: 모든 구성 요소 준비 완료")
        
        # 프로세스 종료 시 연결 정리를 위한 핸들러 등록
        import signal
        def cleanup_handler(signum, frame):
            logger.info("임베딩 워커: 종료 신호 수신, 연결 정리 중...")
            try:
                if conn and document_processor:
                    document_processor.return_db_connection(conn)
                    logger.info("임베딩 워커: DocumentProcessor DB 풀에 연결 반환")
            except Exception as cleanup_error:
                logger.warning(f"연결 정리 중 오류: {cleanup_error}")
            logger.info("임베딩 워커: 정리 완료")
            sys.exit(0)
        
        signal.signal(signal.SIGTERM, cleanup_handler)
        signal.signal(signal.SIGINT, cleanup_handler)
        
        # 임베딩 요청 처리 루프
        while True:
            # Redis에서 시맨틱 청킹 작업 확인 (우선 처리)
            try:
                redis_task_data = redis_client.brpop('embedding_tasks', timeout=1)  # 1초 대기
                if redis_task_data:
                    task_json = redis_task_data[1]  # brpop returns (key, value)
                    task_data = json.loads(task_json)
                    
                    if task_data.get('type') == 'semantic_chunk':
                        task_id = task_data.get('task_id')
                        text = task_data.get('text', '')
                        
                        logger.info(f"[EMBEDDING_WORKER] Redis에서 semantic_chunk 작업 수신: {task_id}")
                        
                        try:
                            # GPU 모델로 시맨틱 청킹 수행
                            from rag_process import get_semantic_model
                            semantic_model = get_semantic_model()
                            
                            if semantic_model:
                                # KR-SBERT 모델로 시맨틱 청킹
                                logger.info(f"[EMBEDDING_WORKER] GPU KR-SBERT 모델로 시맨틱 청킹 수행: {len(text)} 문자")
                                
                                # 시맨틱 청킹 로직 (KR-SBERT 기반)
                                sentences = text.split('. ')
                                if len(sentences) <= 1:
                                    chunks = [text]  # 단일 문장인 경우
                                else:
                                    # HuggingFaceEmbeddings 객체는 embed_documents 메서드 사용
                                    sentence_embeddings = semantic_model.embed_documents(sentences)
                                    
                                    # 유사도 기반 청킹
                                    from sklearn.metrics.pairwise import cosine_similarity
                                    import numpy as np
                                    
                                    chunks = []
                                    current_chunk = sentences[0]
                                    current_embedding = np.array(sentence_embeddings[0]).reshape(1, -1)
                                    
                                    for i in range(1, len(sentences)):
                                        next_embedding = np.array(sentence_embeddings[i]).reshape(1, -1)
                                        similarity = cosine_similarity(current_embedding, next_embedding)[0][0]
                                        
                                        # 유사도 임계값 0.7 이상이면 같은 청크로 묶기
                                        if similarity >= 0.7:
                                            current_chunk += '. ' + sentences[i]
                                            # 평균 임베딩으로 업데이트
                                            current_embedding = np.mean([current_embedding[0], next_embedding[0]], axis=0).reshape(1, -1)
                                        else:
                                            chunks.append(current_chunk)
                                            current_chunk = sentences[i]
                                            current_embedding = next_embedding
                                    
                                    # 마지막 청크 추가
                                    if current_chunk:
                                        chunks.append(current_chunk)
                                
                                # 결과를 Redis에 저장
                                result_key = f"semantic_chunk_result:{task_id}"
                                result_data = {
                                    'success': True,
                                    'chunks': chunks,
                                    'chunk_count': len(chunks)
                                }
                                redis_client.setex(result_key, 300, json.dumps(result_data))  # 5분 TTL
                                logger.info(f"[EMBEDDING_WORKER] 시맨틱 청킹 완료: {len(chunks)}개 청크 생성, Redis에 결과 저장")
                                
                            else:
                                # 모델 로드 실패 시 에러 반환
                                result_key = f"semantic_chunk_result:{task_id}"
                                error_data = {
                                    'success': False,
                                    'error': 'Semantic model is not initialized'
                                }
                                redis_client.setex(result_key, 300, json.dumps(error_data))
                                logger.error(f"[EMBEDDING_WORKER] 시맨틱 모델 미초기화로 작업 실패: {task_id}")
                                
                        except Exception as semantic_error:
                            # 시맨틱 청킹 실패 시 에러 반환
                            result_key = f"semantic_chunk_result:{task_id}"
                            error_data = {
                                'success': False,
                                'error': str(semantic_error)
                            }
                            redis_client.setex(result_key, 300, json.dumps(error_data))
                            logger.error(f"[EMBEDDING_WORKER] 시맨틱 청킹 실패: {task_id}, 오류: {semantic_error}")
                        
                        continue  # Redis 작업 처리 후 다음 루프로
                        
            except redis.RedisError as redis_error:
                # Redis 연결 오류는 로그만 남기고 계속 진행
                if "timed out" not in str(redis_error).lower():
                    logger.warning(f"Redis 작업 확인 실패: {redis_error}")
            except Exception as task_error:
                logger.error(f"Redis 작업 처리 실패: {task_error}")
            
            # 연결 상태 확인 및 복구
            try:
                if conn.closed != 0:  # 연결이 끊어진 경우
                    logger.warning("임베딩 워커: PostgreSQL 연결이 끊어짐, 재연결 시도")
                    
                    # 기존 연결 반납 (가능한 경우)
                    try:
                        if document_processor:
                            document_processor.return_db_connection(conn)
                    except:
                        pass

                    # 새 연결 획득
                    conn = get_pool_connection()
                    
                    conn.autocommit = False
                    cursor = conn.cursor()
                    logger.info("임베딩 워커: PostgreSQL 재연결 완료")
            except Exception as conn_check_error:
                logger.error(f"연결 상태 확인/복구 실패: {conn_check_error}")
                try:
                    # 새 연결 획득 시도
                    conn = get_pool_connection()
                    conn.autocommit = False
                    cursor = conn.cursor()
                    logger.info("임베딩 워커: PostgreSQL 재연결 완료")
                except Exception as reconnect_error:
                    logger.error(f"재연결 실패: {reconnect_error}")
                    time.sleep(5)  # 5초 대기 후 다시 시도
                    continue
            
            if not embedding_queue.empty():
                request = embedding_queue.get()
                request_id = request['request_id']
                file_path = request['file_path']
                user_id = request.get('user_id', 'all')  # user_id 추출, 기본값 'all'
                temp_settings = request.get('temp_settings', None)  # temp_settings 추출
                
                logger.info(f"임베딩 워커: 임베딩 요청 처리 시작 - ID: {request_id}, 파일: {os.path.basename(file_path)}, 사용자 ID: {user_id}")
                if temp_settings:
                    logger.info(f"임베딩 워커: temp_settings 적용됨 - {temp_settings}")
                
                # 동적 설정 재로드 - 매 문서 처리 전에 최신 설정 반영 (temp_settings가 있을 때는 건너뛰기)
                if not temp_settings:
                    document_processor.reload_settings()
                    logger.info(f"임베딩 워커: 설정 재로드 완료 - process_text_only={document_processor.process_text_only}")
                else:
                    logger.info(f"임베딩 워커: temp_settings 사용으로 설정 재로드 건너뛰기 - temp_settings={temp_settings}")
                
                start_time = time.time()
                
                try:
                    # .extracts 폴더의 임시 이미지 파일 제외
                    if '/.extracts/' in file_path or os.sep + '.extracts' + os.sep in file_path:
                        logger.debug(f"임베딩 워커: .extracts 폴더의 임시 파일 제외 - {os.path.basename(file_path)}")
                        continue
                    
                    # 파일 확장자 확인
                    file_ext = os.path.splitext(file_path)[1].lower()
                    if file_ext not in SUPPORTED_EXTENSIONS:
                        logger.warning(f"임베딩 워커: 지원하지 않는 파일 형식 - {file_ext}")
                        continue
                    
                    logger.info(f"임베딩 워커: 문서 처리 시작 - {os.path.basename(file_path)} ({file_ext})")

                    # 파일명 추출
                    filename = os.path.basename(file_path)

                    # 임베딩 시작 시 데이터베이스 상태를 pending으로 업데이트
                    try:
                        status_update_connection = get_pool_connection()
                        status_update_cursor = status_update_connection.cursor()
                        status_update_cursor.execute("""
                            UPDATE chat_documents
                            SET embedding_status = 'pending',
                                updated_at = CURRENT_TIMESTAMP
                            WHERE user_id = %s AND filename = %s
                        """, (user_id, filename))
                        if status_update_cursor.rowcount > 0:
                            logger.info(f"임베딩 워커: DB 상태 초기화 완료 - {filename} -> pending")
                        status_update_connection.commit()
                        if document_processor:
                            document_processor.return_db_connection(status_update_connection)
                    except Exception as status_error:
                        logger.error(f"임베딩 워커: DB 상태 초기화 실패 - {filename}: {status_error}")

                    # 중복 처리 방지: 이미 처리된 파일인지 확인
                    doc_id = os.path.splitext(filename)[0]
                    
                    # PostgreSQL에서 이미 처리된 파일인지 확인 (DB 연결 재시도 로직 포함)
                    existing_count = 0
                    duplicate_check_success = False

                    for retry_attempt in range(3):  # 최대 3회 재시도
                        try:
                            # DB 연결 상태 확인 및 재연결
                            if conn.closed:
                                logger.warning(f"임베딩 워커: DB 연결이 끊어짐, 재연결 시도 {retry_attempt + 1}/3")
                                conn = get_pool_connection()
                                cursor = conn.cursor()
                                conn.autocommit = False

                            cursor.execute("SELECT COUNT(*) FROM document_embeddings WHERE doc_id = %s AND user_id = %s", (doc_id, user_id))
                            existing_count = cursor.fetchone()[0]
                            duplicate_check_success = True

                            if existing_count > 0:
                                logger.info(f"임베딩 워커: 파일이 이미 처리됨 - {filename} ({existing_count}개 청크 존재), 건너뛰기")
                                # Redis에서 완료 상태로 업데이트
                                try:
                                    redis_client.hset(f"embedding_status:{request_id}", mapping={
                                        "status": "already_processed",
                                        "filename": filename,
                                        "existing_chunks": existing_count,
                                        "skipped_at": time.time()
                                    })
                                    update_doc_status(filename, 'done', '이미 처리됨', {'existing_chunks': existing_count})
                                except Exception as redis_error:
                                    logger.warning(f"Redis 상태 업데이트 실패: {redis_error}")
                                break  # 중복 확인 성공, 이 파일은 건너뛰기
                            else:
                                break  # 중복 없음, 처리 계속

                        except Exception as db_error:
                            logger.warning(f"중복 확인 중 DB 오류 (시도 {retry_attempt + 1}/3): {db_error}")
                            if retry_attempt == 2:  # 마지막 시도에서도 실패
                                logger.error(f"중복 확인 최종 실패, 파일 처리 계속 진행: {filename}")
                                duplicate_check_success = False
                            else:
                                time.sleep(1)  # 1초 대기 후 재시도

                    # 중복 파일이 확인된 경우 다음 파일로 넘어감
                    if duplicate_check_success and existing_count > 0:
                        continue
                    
                    # 문서 처리 (텍스트 추출 + 이미지 설명 + 테이블 마크다운 변환 등)
                    if document_processor:
                        try:
                            # process_document를 호출하여 완전한 문서 처리 수행
                            process_result = document_processor.process_document(file_path, user_id, temp_settings=temp_settings)
                            
                            if process_result and process_result.get('status') == 'success':
                                text_content = process_result
                                logger.info(f"임베딩 워커: 문서 처리 완료 - {os.path.basename(file_path)}")
                                logger.info(f"임베딩 완료된 문서 완료 처리 상태 DB 업데이트 - {os.path.basename(file_path)}")
                                
                                # 실시간 진행 상황 업데이트 (Redis Pub/Sub)
                                try:
                                    import redis
                                    import json
                                    from datetime import datetime
                                    
                                    redis_host = os.getenv('REDIS_HOST', 'localhost')
                                    redis_port = int(os.getenv('REDIS_PORT', '6379'))
                                    redis_client = redis.Redis(host=redis_host, port=redis_port, decode_responses=True)
                                    filename = os.path.basename(file_path)
                                    logger.info("📡 Redis 연결 성공 (실시간 진행 상황 알림)")
                                    
                                    # job_id 찾기 (간단한 방법)
                                    pattern = "background_job:*"
                                    job_id = None
                                    for key in redis_client.scan_iter(match=pattern):
                                        job_data = redis_client.hgetall(key)
                                        if (job_data.get('filename') == filename and 
                                            job_data.get('userId') == user_id and
                                            job_data.get('type') == 'rag-upload'):
                                            job_id = key.replace('background_job:', '')
                                            break
                                    
                                    # 진행 상황 데이터 생성
                                    progress_data = {
                                        'jobId': job_id,
                                        'filename': filename,
                                        'status': 'completed',
                                        'progress': 100,
                                        'message': f'임베딩 완료: {filename}',
                                        'userId': user_id,
                                        'timestamp': datetime.now().isoformat()
                                    }
                                    
                                    # WebSocket으로 실시간 알림 전송
                                    redis_client.publish('websocket:job-status', json.dumps(progress_data))
                                    
                                    # Background job 상태도 업데이트
                                    if job_id:
                                        job_key = f"background_job:{job_id}"
                                        redis_client.hset(job_key, mapping={
                                            'status': 'completed',
                                            'progress': '100',
                                            'message': f'임베딩 완료: {filename}',
                                            'updatedAt': datetime.now().isoformat()
                                        })
                                    
                                    redis_client.close()
                                    logger.info(f"📡 임베딩 완료 알림 전송: {filename}")
                                    
                                except Exception as e:
                                    logger.error(f"⚠️ 진행 상황 알림 전송 실패: {str(e)}")
                               
                                db_update_success = False
                                for db_attempt in range(2):
                                    try:
                                        conn, cursor = ensure_active_connection(
                                            conn, cursor, logger, document_processor, force_new=(db_attempt > 0)
                                        )
                                        cursor.execute("""
                                            UPDATE document_embeddings
                                            SET is_embedding = TRUE
                                            WHERE filename = %s AND user_id = %s
                                        RETURNING id, chunk_index
                                        """, (filename, user_id))
                                        
                                        updated_rows = cursor.fetchall()
                                        updated_count = cursor.rowcount
                                        logger.info(f"DB 업데이트 is_embedding=TRUE - ({updated_count}건)")
                                        if updated_count:
                                            logger.info(f"완료된 문서 임베딩 (id, chunk_index): {updated_rows}")

                                            # chat_documents 테이블의 embedding_status도 completed로 업데이트
                                            filename = os.path.basename(file_path)
                                            try:
                                                cursor.execute("""
                                                    UPDATE chat_documents
                                                    SET embedding_status = 'completed',
                                                        processed_at = CURRENT_TIMESTAMP,
                                                        updated_at = CURRENT_TIMESTAMP
                                                    WHERE user_id = %s AND filename = %s
                                                """, (user_id, filename))
                                                chat_updated_count = cursor.rowcount
                                                if chat_updated_count > 0:
                                                    logger.info(f"임베딩 상태 업데이트 완료: {filename} -> completed ({chat_updated_count}건)")
                                                else:
                                                    logger.warning(f"임베딩 상태 업데이트 실패: {filename} - 해당 레코드 없음")
                                            except Exception as status_error:
                                                logger.error(f"임베딩 상태 업데이트 중 오류 ({filename}): {status_error}")

                                        conn.commit()
                                        db_update_success = True
                                        break
                                    except (psycopg2.InterfaceError, psycopg2.OperationalError) as db_error:
                                        logger.warning(f"임베딩 워커: 문서 상태 업데이트 중 DB 오류 (재시도 {db_attempt + 1}/2) - {db_error}")
                                        cursor = None
                                        time.sleep(1)
                                        if db_attempt == 1:
                                            logger.error(f"임베딩 워커: 문서 상태 업데이트 최종 실패 - {filename}: {db_error}")
                                    except Exception as status_update_error:
                                        logger.error(f"임베딩 워커: 문서 상태 업데이트 중 예기치 않은 오류 - {status_update_error}")
                                        break
                                
                                if not db_update_success:
                                    logger.warning(f"임베딩 워커: 문서 상태 업데이트가 완전히 적용되지 못했습니다 - {filename}")

                                    
                            else:
                                logger.warning(f"임베딩 워커: 문서 처리 실패 - {process_result.get('message', 'Unknown error')}")
                                text_content = None
                        except Exception as e:
                            logger.warning(f"임베딩 워커: 문서 처리 실패 - {e}")
                            text_content = None
                    else:
                        text_content = None
                    
                        # fallback: 기본 텍스트 파일 처리
                        if file_ext in ['.txt', '.md', '.text']:
                            try:
                                with open(file_path, 'r', encoding='utf-8') as f:
                                    text_content = f.read()
                            except Exception as read_error:
                                logger.error(f"임베딩 워커: 파일 읽기 실패 - {read_error}")
                                continue
                        else:
                            logger.warning(f"임베딩 워커: DocumentProcessor 없이 처리 불가한 파일 형식 - {file_ext}")
                            continue
                    
                    if not text_content:
                        logger.warning(f"임베딩 워커: 문서 처리 실패 - {os.path.basename(file_path)}")
                        # 에러 상태로 업데이트
                        try:
                            update_embedding_status_sync(user_id, filename, 'error', '문서 처리 실패')
                        except:
                            pass
                        continue
                    
                    # process_document()의 결과 처리
                    chunks = []
                    chunks_metadata = []
                    
                    if isinstance(text_content, dict) and text_content.get('status') == 'success':
                        # process_document()의 성공 결과인 경우
                        if 'chunks' in text_content:
                            # 이미 청킹된 데이터 사용
                            for chunk_data in text_content['chunks']:
                                if isinstance(chunk_data, dict):
                                    chunks.append(chunk_data.get('text', str(chunk_data)))
                                    chunks_metadata.append(chunk_data.get('metadata', {}))
                                else:
                                    chunks.append(str(chunk_data))
                                    chunks_metadata.append({})
                        else:
                            logger.info(f"임베딩 워커: process_document()가 통계 정보만 반환 (정상) - {os.path.basename(file_path)}")
                            continue
                    else:
                        # 기존 로직 fallback
                        rag_settings = get_rag_settings()
                        chunking_strategy = rag_settings.get('chunking_strategy', 'default')
                        
                        if isinstance(text_content, dict):
                            if 'page_documents' in text_content:
                                chunks = split_text("", file_ext=file_ext, 
                                                  page_texts=text_content['page_documents'],
                                                  chunking_strategy=chunking_strategy)
                            else:
                                plain_text = text_content.get('text', str(text_content))
                                chunks = split_text(plain_text, file_ext=file_ext, 
                                                  chunking_strategy=chunking_strategy)
                        else:
                            chunks = split_text(text_content, file_ext=file_ext,
                                              chunking_strategy=chunking_strategy)
                        
                        chunks_metadata = [{}] * len(chunks)
                    
                    if not chunks:
                        logger.warning(f"임베딩 워커: 청크 분할 실패 - {os.path.basename(file_path)}")
                        continue
                    
                    # 청크 텍스트 추출 (chunks_metadata는 이미 위에서 설정됨)
                    chunks_text = []
                    # chunks_metadata는 이미 설정되어 있음
                    
                    for chunk in chunks:
                        if isinstance(chunk, dict):
                            chunks_text.append(chunk.get('text', ''))
                            chunks_metadata.append({
                                'page_number': chunk.get('page', 1),
                                'total_pages': chunk.get('total_pages', 1)
                            })
                        else:
                            chunks_text.append(str(chunk))
                            chunks_metadata.append({'page_number': 1, 'total_pages': 1})
                    
                    if not chunks_text:
                        logger.warning(f"임베딩 워커: 유효한 청크 텍스트가 없음 - {os.path.basename(file_path)}")
                        continue
                    
                    logger.info(f"임베딩 워커: {len(chunks_text)}개 청크에 대한 임베딩 생성 시작")
                    
                    # RAG 설정 가져오기
                    rag_settings = get_rag_settings()
                    use_enhanced_embedding = str(rag_settings.get('use_enhanced_embedding', 'false')).lower() in ['true', 'yes', '1']
                    enhanced_embedding_strategy = rag_settings.get('enhanced_embedding_strategy', 'hybrid')
                    
                    # 임베딩 생성
                    if use_enhanced_embedding:
                        logger.info(f"임베딩 워커: 향상된 임베딩 사용 - 전략: {enhanced_embedding_strategy}")
                        embeddings = []
                        enhanced_metadata_list = []
                        
                        for i, chunk_text in enumerate(chunks_text):
                            try:
                                # 향상된 임베딩 생성
                                enhanced_result = question_generator.generate_questions_and_keywords(chunk_text)
                                
                                # 원본 텍스트 임베딩
                                original_embedding = embedding_model.encode([chunk_text], show_progress_bar=False)[0]
                                
                                # 예상 질문들 임베딩 (상위 3개만)
                                question_embeddings = []
                                if enhanced_result.get('questions'):
                                    questions = enhanced_result['questions'][:3]  # 상위 3개만
                                    if questions:
                                        question_embeddings = embedding_model.encode(questions, show_progress_bar=False)
                                
                                # 최종 임베딩 선택
                                final_embedding = original_embedding
                                
                                if enhanced_embedding_strategy == 'questions' and len(question_embeddings) > 0:
                                    final_embedding = question_embeddings[0]
                                elif enhanced_embedding_strategy == 'hybrid' and len(question_embeddings) > 0:
                                    embeddings_array = np.array([original_embedding] + list(question_embeddings))
                                    final_embedding = np.mean(embeddings_array, axis=0)
                                
                                embeddings.append(final_embedding)
                                enhanced_metadata_list.append({
                                    'questions': enhanced_result.get('questions', [])[:5],
                                    'keywords': enhanced_result.get('keywords', []),
                                    'topics': enhanced_result.get('topics', []),
                                    'enhanced': True,
                                    'strategy': enhanced_embedding_strategy
                                })
                                
                                if i % 10 == 0 and i > 0:
                                    logger.info(f"임베딩 워커: 향상된 임베딩 진행률 - {i}/{len(chunks_text)} 완료")
                                    
                            except Exception as e:
                                logger.warning(f"임베딩 워커: 청크 {i} 향상된 임베딩 실패, 기본 임베딩 사용 - {e}")
                                basic_embedding = embedding_model.encode([chunk_text], show_progress_bar=False)[0]
                                embeddings.append(basic_embedding)
                                enhanced_metadata_list.append({'enhanced': False, 'fallback_reason': 'generation_failed'})
                    else:
                        # 기본 임베딩 생성
                        logger.info("임베딩 워커: 기본 임베딩 생성")
                        embeddings = embedding_model.encode(chunks_text, show_progress_bar=False)
                        enhanced_metadata_list = [{'enhanced': False} for _ in chunks_text]
                    
                    logger.info(f"임베딩 워커: 임베딩 생성 완료")
                    
                    # PostgreSQL에 저장
                    filename = os.path.basename(file_path)
                    doc_id = os.path.splitext(filename)[0]
                    
                    # 트랜잭션 시작 및 기존 문서 삭제 (재처리의 경우)
                    try:
                        # 연결 상태 먼저 확인
                        if conn.closed != 0 or cursor.closed:
                            logger.warning("임베딩 워커: 연결/커서가 닫혀있음, 재연결 시도")
                            
                            # 기존 연결 반납
                            try:
                                if document_processor and conn.closed == 0:
                                    document_processor.return_db_connection(conn)
                            except:
                                pass

                            # 새 연결 획득
                            conn = get_pool_connection()
                            conn.autocommit = False
                            cursor = conn.cursor()
                            logger.info("임베딩 워커: 재연결 완료")
                        
                        cursor.execute("DELETE FROM document_embeddings WHERE doc_id = %s AND user_id = %s", (doc_id, user_id))
                    except (psycopg2.OperationalError, psycopg2.InterfaceError) as delete_error:
                        logger.error(f"DELETE 작업 중 오류: {delete_error}")
                        # 연결 복구 시도
                        try:
                            if document_processor:
                                document_processor.return_db_connection(conn)
                        except:
                            pass

                        # 새 연결 획득
                        conn = get_pool_connection()
                        conn.autocommit = False
                        cursor = conn.cursor()
                        cursor.execute("DELETE FROM document_embeddings WHERE doc_id = %s AND user_id = %s", (doc_id, user_id))
                    
                    # 새 임베딩 저장
                    insert_query = """
                        INSERT INTO document_embeddings 
                        (doc_id, filename, chunk_index, chunk_text, embedding, user_id, source, file_mtime, metadata)
                        VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)
                    """
                    
                    # 청크 즉시 저장 방식 (개선된 방식)
                    chunks_saved = 0
                    total_chunks = len(chunks_text)
                    
                    # 임베딩 상태를 'processing'으로 업데이트
                    try:
                        update_embedding_status_sync(user_id, filename, 'processing', f"처리 중... 0/{total_chunks} 청크")
                        logger.info(f"임베딩 워커: 상태를 'processing'으로 업데이트 - {filename}")
                    except Exception as status_error:
                        logger.warning(f"임베딩 워커: 상태 업데이트 실패 - {status_error}")
                    
                    # 배치 커밋을 위한 변수
                    batch_size = 10  # 10개씩 배치로 커밋
                    batch_counter = 0
                    
                    for i, (chunk_text, embedding, chunk_metadata) in enumerate(zip(chunks_text, embeddings, chunks_metadata)):
                        # 메타데이터 구성
                        from datetime import datetime
                        current_time = datetime.now().isoformat()
                        
                        metadata = {
                            'source': filename,
                            'chunk_index': i,
                            'page_number': chunk_metadata.get('page_number', chunk_metadata.get('page', 1)),
                            'total_chunks': len(chunks_text),
                            'timestamp': current_time,
                            'modification_date': current_time,
                            'processed_at': time.time(),
                            'user_id': user_id,
                            'document_processor_metadata': chunk_metadata,
                        }
                        
                        # 향상된 임베딩 정보 추가
                        if use_enhanced_embedding and i < len(enhanced_metadata_list):
                            metadata['enhanced_embedding'] = enhanced_metadata_list[i]
                        
                        # numpy array를 list로 변환하여 저장
                        embedding_list = embedding.tolist() if hasattr(embedding, 'tolist') else list(embedding)
                        
                        try:
                            cursor.execute(
                                insert_query,
                                (doc_id, filename, i, chunk_text, embedding_list, user_id, filename, None, Json(metadata))
                            )
                            batch_counter += 1
                            
                            # 배치 단위로 커밋 또는 마지막 청크일 때 커밋
                            if batch_counter >= batch_size or i == total_chunks - 1:
                                conn.commit()
                                chunks_saved += batch_counter
                                batch_counter = 0
                                
                                # 진행 상태 업데이트 (20% 단위로)
                                if chunks_saved % max(1, total_chunks // 5) == 0 or chunks_saved == total_chunks:
                                    progress_msg = f"처리 중... {chunks_saved}/{total_chunks} 청크"
                                    try:
                                        update_embedding_status_sync(user_id, filename, 'processing', progress_msg)
                                        logger.info(f"임베딩 워커: 진행률 업데이트 - {progress_msg}")
                                    except Exception as progress_error:
                                        logger.warning(f"임베딩 워커: 진행률 업데이트 실패 - {progress_error}")
                            
                        except (psycopg2.OperationalError, psycopg2.InterfaceError) as insert_error:
                            logger.warning(f"INSERT 중 연결 오류: {insert_error}, 재연결 후 재시도")
                            # 연결 복구
                            try:
                                if document_processor:
                                    document_processor.return_db_connection(conn)
                            except:
                                pass

                            # 새 연결 획득
                            conn = get_pool_connection()
                            conn.autocommit = False
                            cursor = conn.cursor()
                            # 재시도
                            cursor.execute(
                                insert_query,
                                (doc_id, filename, i, chunk_text, embedding_list, user_id, filename, None, Json(metadata))
                            )
                            conn.commit()
                            chunks_saved += 1
                    
                    logger.info(f"임베딩 워커: PostgreSQL 배치 저장 완료 - {chunks_saved}/{total_chunks}개 청크")
                    
                    # chat_documents 테이블에 파일 정보 저장 (support 폴더 제외)
                    if user_id != 'support' and user_id != 'shared':
                        try:
                            # 파일 정보 수집
                            file_size = os.path.getsize(file_path) if os.path.exists(file_path) else 0
                            file_extension = os.path.splitext(filename)[1].lower()
                            
                            # MIME 타입 결정
                            mime_type = 'application/pdf' if file_extension == '.pdf' else \
                                       'application/msword' if file_extension == '.doc' else \
                                       'application/vnd.openxmlformats-officedocument.wordprocessingml.document' if file_extension == '.docx' else \
                                       'text/plain' if file_extension == '.txt' else \
                                       'application/octet-stream'
                            
                            # chat_documents 테이블에 저장 및 상태 업데이트 (무결성 강화)
                            cursor.execute("""
                                INSERT INTO chat_documents (
                                    user_id, filename, filesize, mimetype, filepath, 
                                    upload_status, embedding_status, processed_at, created_at, updated_at
                                ) VALUES (%s, %s, %s, %s, %s, %s, %s, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
                                ON CONFLICT (user_id, filename) 
                                DO UPDATE SET 
                                    filesize = EXCLUDED.filesize,
                                    mimetype = EXCLUDED.mimetype,
                                    embedding_status = 'completed',
                                    processed_at = CURRENT_TIMESTAMP,
                                    error_message = NULL,
                                    updated_at = CURRENT_TIMESTAMP
                            """, (user_id, filename, file_size, mime_type, f"{user_id}/{filename}", 
                                  'uploaded', 'completed'))
                            
                            # chat_documents 테이블 상태는 ON CONFLICT에서 자동으로 completed로 업데이트됨
                            logger.info(f"임베딩 워커: chat_documents 테이블 저장 및 상태 업데이트 완료 - {filename} -> completed (사용자: {user_id})")
                        except Exception as chat_doc_error:
                            logger.error(f"임베딩 워커: chat_documents 테이블 저장 실패 - {filename}: {chat_doc_error}")
                            # chat_documents 저장 실패해도 임베딩은 이미 성공했으므로 계속 진행
                            # 상태 동기화를 위해 별도 함수로 재시도
                            try:
                                success = update_embedding_status_sync(user_id, filename, 'completed', None)
                                if success:
                                    logger.info(f"임베딩 워커: 상태 동기화 성공 - {filename} -> completed")
                                else:
                                    logger.warning(f"임베딩 워커: 상태 동기화 실패 - {filename} (수동 확인 필요)")
                            except Exception as sync_error:
                                logger.warning(f"임베딩 워커: 상태 동기화 재시도 실패 - {sync_error}")
                    
                    # Redis에 완료 상태 저장
                    redis_client.hset(f"embedding_status:{request_id}", mapping={
                        "status": "completed",
                        "filename": filename,
                        "doc_id": doc_id,
                        "num_chunks": len(chunks_text),
                        "processing_time": time.time() - start_time,
                        "completed_at": time.time()
                    })
                    
                    # Redis에서 처리 키 삭제
                    file_processing_key = f"file_processing:{file_path}"
                    redis_client.delete(file_processing_key)
                    
                    processing_time = time.time() - start_time
                    logger.info(f"임베딩 워커: 요청 처리 완료 - ID: {request_id}, 소요시간: {processing_time:.2f}초")
                    
                    # Redis에 문서 처리 완료 상태 업데이트
                    try:
                        update_doc_status(filename, 'done', '임베딩 처리 완료', {'completed_at': time.time()})
                        logger.info(f"임베딩 완료된 문서 완료 처리 상태 DB 업데이트 - {filename}")
                    except Exception as e:
                        logger.error(f"Redis 완료 상태 업데이트 실패: {str(e)}")
                    
                        
                        for i, chunk_text in enumerate(chunks_content):
                            try:
                                # 향상된 임베딩 생성
                                enhanced_result = question_generator.generate_questions_and_keywords(chunk_text)
                                
                                # 원본 텍스트 임베딩
                                original_embedding = embedding_model.encode([chunk_text], show_progress_bar=False)[0]
                                
                                # 예상 질문들 임베딩 (상위 3개만)
                                question_embeddings = []
                                if enhanced_result.get('questions'):
                                    questions = enhanced_result['questions'][:3]  # 상위 3개만
                                    if questions:
                                        question_embeddings = embedding_model.encode(questions, show_progress_bar=False)
                                
                                # 최종 임베딩 선택
                                final_embedding = original_embedding
                                
                                if enhanced_embedding_strategy == 'questions' and len(question_embeddings) > 0:
                                    # 가장 대표적인 질문의 임베딩 사용
                                    final_embedding = question_embeddings[0]
                                elif enhanced_embedding_strategy == 'hybrid' and len(question_embeddings) > 0:
                                    # 원본과 질문 임베딩의 가중 평균
                                    embeddings_array = np.array([original_embedding] + list(question_embeddings))
                                    final_embedding = np.mean(embeddings_array, axis=0)
                                
                                embeddings.append(final_embedding)
                                
                                # 향상된 메타데이터 저장
                                enhanced_metadata_list.append({
                                    'questions': enhanced_result.get('questions', [])[:5],  # 최대 5개
                                    'keywords': enhanced_result.get('keywords', []),
                                    'topics': enhanced_result.get('topics', []),
                                    'enhanced': True,
                                    'strategy': enhanced_embedding_strategy
                                })
                                
                                if i % 10 == 0 and i > 0:
                                    logger.info(f"임베딩 워커: 향상된 임베딩 진행률 - {i}/{len(chunks_content)} 완료")
                                    
                            except Exception as e:
                                logger.warning(f"임베딩 워커: 청크 {i} 향상된 임베딩 실패, 기본 임베딩 사용 - {e}")
                                # 실패 시 기본 임베딩 사용
                                basic_embedding = embedding_model.encode([chunk_text], show_progress_bar=False)[0]
                                embeddings.append(basic_embedding)
                                enhanced_metadata_list.append({
                                    'questions': [],
                                    'keywords': [],
                                    'topics': [],
                                    'enhanced': False,
                                    'fallback_reason': 'generation_failed'
                                })
                    else:
                        # 기본 임베딩 생성
                        logger.info("임베딩 워커: 기본 임베딩 생성")
                        embeddings = embedding_model.encode(chunks_text, show_progress_bar=False)
                        enhanced_metadata_list = [{'enhanced': False} for _ in chunks_text]
                    
                    logger.info(f"임베딩 워커: 임베딩 생성 완료")
                    
                    # PostgreSQL에 저장
                    filename = os.path.basename(file_path)
                    doc_id = os.path.splitext(filename)[0]
                    
                    # 트랜잭션 시작 및 기존 문서 삭제 (재처리의 경우)
                    try:
                        cursor.execute("DELETE FROM document_embeddings WHERE doc_id = %s", (doc_id,))
                    except psycopg2.OperationalError as delete_error:
                        logger.error(f"DELETE 작업 중 연결 오류: {delete_error}")
                        # 연결 복구 시도
                        return_pool_connection(conn)
                        conn = get_pool_connection()
                        conn.autocommit = False
                        cursor = conn.cursor()
                        # 다시 시도
                        cursor.execute("DELETE FROM document_embeddings WHERE doc_id = %s", (doc_id,))
                    
                    # 새 임베딩 저장
                    insert_query = """
                        INSERT INTO document_embeddings 
                        (doc_id, filename, chunk_index, chunk_text, embedding, user_id, source, file_mtime, metadata)
                        VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)
                    """
                    
                    for i, (chunk_text, embedding, chunk_metadata) in enumerate(zip(chunks_text, embeddings, chunks_metadata)):
                        chunk = chunk_text
                        page_number = chunk_metadata.get('page_number', 1)
                        # Calculate sub_scores for the chunk
                        import re
                        
                        # Structure score calculation
                        structure_score = 0.35  # base score
                        if re.search(r'^(?:제\s*)?[0-9０-９]+[.\s]*장\s+', chunk):
                            structure_score += 0.45
                        elif re.search(r'^(?:제\s*)?[0-9０-９]+[.\s]*절\s+', chunk):
                            structure_score += 0.35
                        elif re.search(r'^[0-9０-９]+[.][0-9０-９]+[.]*\s+[^\n]+', chunk):
                            structure_score += 0.25
                        if '\n\n' in chunk: 
                            structure_score += 0.35
                        if len(re.findall(r'[.!?]+', chunk)) > 1: 
                            structure_score += 0.25
                        structure_score = min(1.0, structure_score)
                        
                        # Content score calculation
                        content_score = 0.35  # base score
                        if not re.search(r'[\u0000-\u001F]', chunk):
                            content_score += 0.15
                        if not re.search(r'[ㄱ-ㅎㅏ-ㅣ]+', chunk):
                            content_score += 0.25
                        if len(chunk) > 0:
                            special_char_ratio = len(re.findall(r'[^\w\s가-힣]', chunk)) / len(chunk)
                            if special_char_ratio < 0.15:
                                content_score += 0.25
                        content_score = min(1.0, content_score)
                        
                        # Get RAG settings for chunk size
                        rag_settings = get_rag_settings()
                        rag_chunk_size = rag_settings.get('chunk_size', 150)
                        
                        # Length score calculation
                        optimal_min = rag_chunk_size * 0.15
                        optimal_max = rag_chunk_size * 3.5
                        chunk_length = len(chunk)
                        if optimal_min <= chunk_length <= optimal_max:
                            length_score = 1.0
                        elif chunk_length < optimal_min:
                            length_score = min(1.0, chunk_length / optimal_min) if optimal_min > 0 else 0.0
                        else:
                            length_score = min(1.0, optimal_max / chunk_length) if chunk_length > 0 else 0.0
                        
                        # Keyword quality (basic implementation)
                        keyword_quality = 0.5  # placeholder score
                        
                        # Get RAG settings for thresholds and weights
                        from datetime import datetime
                        current_time = datetime.now().isoformat()
                        
                        # Default RAG settings (matching rag_process.py defaults)
                        thresholds = {
                            'similarity': 0.7,
                            'exact_match': 0.3,
                            'keyword_match': 0.6,
                            'strong_keyword_match': 0.9,
                            'relevance': 0.5
                        }
                        
                        main_weights = {
                            'exact_match': 0.40,
                            'similarity': 0.30,
                            'relevance': 0.10,
                            'date': 0.10
                        }
                        
                        sub_weights = {
                            'structure': 0.10,
                            'content': 0.10,
                            'length': 0.10,
                            'keyword_quality': 0.20,
                            'partial_match': 0.50
                        }
                        
                        metadata = {
                            'source': filename,
                            'chunk_index': i,
                            'page_number': page_number,
                            'total_chunks': len(chunks_text),
                            'timestamp': current_time,
                            'modification_date': current_time,
                            'processed_at': time.time(),
                            'user_id': user_id,  # request에서 받은 user_id 사용
                            'main_scores': {
                                'similarity_score': 0.0,  # will be calculated during search
                                'exact_match_score': 0.0,  # will be calculated during search
                                'relevance_score': 0.0,  # will be calculated during search
                                'date_score': 1.0  # newly processed documents get full date score
                            },
                            'sub_scores': {
                                'structure_score': structure_score,
                                'content_score': content_score,
                                'length_score': length_score,
                                'keyword_quality': keyword_quality,
                                'partial_match_score': 0.0,  # will be calculated during search
                                'strong_keyword_match_score': 0.0  # will be calculated during search
                            },
                            'matches': {
                                'exact_matches': [],  # will be populated during search
                                'partial_matches': []  # will be populated during search
                            },
                            'thresholds': thresholds,
                            'main_weights': main_weights,
                            'sub_weights': sub_weights,
                            'filter_settings': {
                                'exact_match_threshold': thresholds['exact_match'],
                                'keyword_match_threshold': thresholds['keyword_match'],
                                'strong_keyword_match_threshold': thresholds['strong_keyword_match'],
                                'similarity_threshold': thresholds['similarity'],
                                'relevance_threshold': thresholds['relevance']
                            }
                        }
                        
                        # 향상된 임베딩 정보 추가
                        if use_enhanced_embedding and i < len(enhanced_metadata_list):
                            enhanced_info = enhanced_metadata_list[i]
                            metadata['enhanced_embedding'] = {
                                'enabled': enhanced_info.get('enhanced', False),
                                'strategy': enhanced_info.get('strategy', enhanced_embedding_strategy),
                                'questions': enhanced_info.get('questions', []),
                                'keywords': enhanced_info.get('keywords', []),
                                'topics': enhanced_info.get('topics', []),
                                'fallback_reason': enhanced_info.get('fallback_reason', None)
                            }
                        else:
                            metadata['enhanced_embedding'] = {
                                'enabled': False,
                                'strategy': None,
                                'questions': [],
                                'keywords': [],
                                'topics': [],
                                'fallback_reason': None
                            }
                        
                        # numpy array를 list로 변환하여 저장
                        embedding_list = embedding.tolist()
                        
                        cursor.execute(
                            insert_query,
                            (doc_id, filename, i, chunk, embedding_list, user_id, filename, None, Json(metadata))
                        )
                    
                    # 트랜잭션 커밋
                    conn.commit()
                    logger.info(f"임베딩 워커: PostgreSQL 저장 완료 - {len(chunks_text)}개 청크")
                    
                    # Redis에 완료 상태 저장
                    redis_client.hset(f"embedding_status:{request_id}", mapping={
                        "status": "completed",
                        "filename": filename,
                        "doc_id": doc_id,
                        "num_chunks": len(chunks_text),
                        "processing_time": time.time() - start_time,
                        "completed_at": time.time()
                    })
                    
                    # Redis에서 처리 키 삭제
                    file_processing_key = f"file_processing:{file_path}"
                    redis_client.delete(file_processing_key)
                    
                    processing_time = time.time() - start_time
                    logger.info(f"임베딩 워커: 요청 처리 완료 - ID: {request_id}, 소요시간: {processing_time:.2f}초")
                    
                    # Redis에 문서 처리 완료 상태 업데이트
                    try:
                        update_doc_status(filename, 'done', '임베딩 처리 완료', {'completed_at': time.time()})
                        logger.info(f"Redis에 문서 완료 상태 업데이트: {filename} -> done")
                    except Exception as e:
                        logger.error(f"Redis 완료 상태 업데이트 실패: {str(e)}")
                    
                except Exception as e:
                    logger.error(f"임베딩 워커: 임베딩 처리 실패 - ID: {request_id}, 오류: {e}")
                    logger.error(f"임베딩 워커: 상세 오류: {traceback.format_exc()}")
                    
                    # 트랜잭션 롤백 및 연결 복구
                    try:
                        if conn and conn.closed == 0:
                            conn.rollback()
                    except:
                        pass
                    
                    # 연결 재설정
                    try:
                        # 기존 연결 반납
                        try:
                            if document_processor:
                                document_processor.return_db_connection(conn)
                        except:
                            pass

                        # 새 연결 획득
                        conn = get_pool_connection()
                        conn.autocommit = False
                        cursor = conn.cursor()
                        logger.warning("임베딩 워커: PostgreSQL 재연결 완료")
                    except Exception as reconnect_error:
                        logger.error(f"재연결 실패: {reconnect_error}")
                        time.sleep(5)  # 5초 대기 후 다음 요청 처리
                        continue
                    
                    # Redis에 실패 상태 저장
                    redis_client.hset(f"embedding_status:{request_id}", mapping={
                        "status": "failed",
                        "error": str(e),
                        "failed_at": time.time()
                    })
                    
                    # Redis에 문서 처리 실패 상태 업데이트
                    try:
                        filename = os.path.basename(file_path)
                        update_doc_status(filename, 'error', '임베딩 처리 실패', {'error': str(e)})
                        logger.info(f"Redis에 문서 실패 상태 업데이트: {filename} -> error")
                    except Exception as status_error:
                        logger.error(f"Redis 실패 상태 업데이트 실패: {str(status_error)}")
                    
                    # Redis에서 처리 키 삭제 (실패 시에도)
                    try:
                        file_processing_key = f"file_processing:{file_path}"
                        redis_client.delete(file_processing_key)
                    except:
                        pass
                    continue
            
            time.sleep(0.1)  # CPU 사용률 조절
    
    except Exception as e:
        logger.error(f"임베딩 워커: 초기화 오류 발생: {e}")
        import traceback
        logger.error(f"임베딩 워커: 전체 스택 트레이스: {traceback.format_exc()}")
    finally:
        # 연결 정리
        try:
            if 'conn' in locals() and conn and 'document_processor' in locals() and document_processor:
                document_processor.return_db_connection(conn)
                logger.info("임베딩 워커: DocumentProcessor DB 풀에 연결 반납")
        except Exception as cleanup_error:
            logger.warning(f"임베딩 워커: 연결 정리 중 오류: {cleanup_error}")

# 전역 프로세스 관리
search_process = None
embedding_process = None
graphrag_process = None
search_queue = None
search_response_queue = None
embedding_queue = None
graphrag_queue = None
search_model_ready_event = None
embedding_model_ready_event = None
graphrag_ready_event = None
GLOBAL_DB_POOL = None

def get_vram_usage():
    """VRAM 사용량 확인"""
    try:
        import subprocess
        result = subprocess.run(['nvidia-smi', '--query-gpu=memory.used,memory.total', '--format=csv,noheader,nounits'], 
                              capture_output=True, text=True, timeout=5)
        
        if result.returncode == 0:
            lines = result.stdout.strip().split('\n')
            total_used = 0
            total_memory = 0
            
            for line in lines:
                if line.strip():
                    used, total = line.split(',')
                    total_used += int(used.strip())
                    total_memory += int(total.strip())
            
            used_gb = total_used / 1024
            total_gb = total_memory / 1024
            return f"{used_gb:.1f}GB / {total_gb:.1f}GB"
        else:
            return "GPU 정보 없음"
            
    except Exception:
        return "GPU 정보 없음"

def monitor_worker_readiness():
    """워커 준비 상태를 모니터링하고 완료되면 최종 상태 출력"""
    import time
    
    max_wait_time = 120  # 최대 2분 대기
    check_interval = 2   # 2초마다 체크
    
    for i in range(max_wait_time // check_interval):
        time.sleep(check_interval)
        
        # 두 워커 모두 준비되었는지 확인
        search_ready = search_model_ready_event.is_set() if search_model_ready_event else False
        embedding_ready = embedding_model_ready_event.is_set() if embedding_model_ready_event else False
        
        if search_ready and embedding_ready:
            # 추가 2초 대기 (마지막 로그 출력 완료를 위해)
            time.sleep(2)
            print_final_service_status()
            break
    else:
        # 타임아웃 시에도 상태 출력
        logger.info("⚠️ 워커 준비 상태 모니터링 타임아웃 - 부분적 준비 상태 출력")
        print_final_service_status()

def print_final_service_status():
    """최종 서비스 상태 정보 출력 (모든 워커 준비 완료 후)"""
    import os
    from datetime import datetime
    
    # 메인 프로세스 PID
    main_pid = os.getpid()
    
    # VRAM 사용량
    vram_usage = get_vram_usage()
    
    # 현재 시간
    current_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    
    # 워커 PID 정보 (전역 변수에서 가져오기)
    search_pid = search_process.pid if search_process else "N/A"
    embedding_pid = embedding_process.pid if embedding_process else "N/A"
    
    logger.info("")
    logger.info("🎉" + "=" * 78 + "🎉")
    logger.info("")
    logger.info("● ✅ RAG 서비스 상태")
    logger.info("")
    logger.info(f"  - RAG 서비스: 정상 실행 중 (systemd)")
    logger.info(f"  - 완료 시간: {current_time}")
    logger.info(f"  - Main PID: {main_pid}")
    logger.info(f"  - Worker PIDs: {search_pid} (Search), {embedding_pid} (Embedding)")
    logger.info(f"  - VRAM 사용량: {vram_usage}")
    logger.info("")
    # 현재 활성 모델명 가져오기
    current_embedding_model = os.environ.get('RAG_EMBEDDING_MODEL', 'nlpai-lab/KURE-v1')
    current_image_model = os.environ.get('RAG_IMAGE_EMBEDDING_MODEL', 'Bingsu/clip-vit-base-patch32-ko')
    current_semantic_model = os.environ.get('RAG_SENTENCE_TRANSFORMER_MODEL', 'snunlp/KR-SBERT-V40K-klueNLI-augSTS')

    # 파인튜닝된 모델인 경우 간단한 이름으로 표시
    display_embedding_model = current_embedding_model
    if 'finetuned' in current_embedding_model:
        # 경로에서 모델명만 추출 (예: nlpai-lab-KURE-v1-finetuned-850e6d54)
        model_name = os.path.basename(current_embedding_model)
        display_embedding_model = f"{model_name} (Fine-tuned)"

    logger.info("📊 워커별 모델 로딩 현황")
    logger.info("")
    logger.info(f"Search Worker (PID: {search_pid})")
    logger.info(f"  ✅ {display_embedding_model} - 텍스트 임베딩")
    logger.info(f"  ✅ {current_image_model} - 이미지 임베딩")
    logger.info("")
    logger.info(f"Embedding Worker (PID: {embedding_pid})")
    logger.info(f"  ✅ {display_embedding_model} - 텍스트 임베딩")
    logger.info(f"  ✅ {current_image_model} - 이미지 임베딩")
    logger.info(f"  📋 {current_semantic_model} - 시맨틱 청킹 (동적 로딩)")
    logger.info("")
    logger.info("🎉" + "=" * 78 + "🎉")
    logger.info("")

def print_service_status(search_pid, embedding_pid, num_search_threads):
    """서비스 상태 정보 출력"""
    import os
    from datetime import datetime
    
    # 메인 프로세스 PID
    main_pid = os.getpid()
    
    # VRAM 사용량
    vram_usage = get_vram_usage()
    
    # 현재 시간
    current_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    
    logger.info("")
    logger.info("● ✅ 서비스 상태")
    logger.info("")
    logger.info(f"  - RAG 서비스: 정상 실행 중 (systemd)")
    logger.info(f"  - 시작 시간: {current_time}")
    logger.info(f"  - Main PID: {main_pid}")
    logger.info(f"  - Worker PIDs: {search_pid} (Search), {embedding_pid} (Embedding)")
    logger.info(f"  - VRAM 사용량: {vram_usage}")
    logger.info("")
    # 현재 활성 모델명 가져오기 (로딩 중 상태)
    current_embedding_model = os.environ.get('RAG_EMBEDDING_MODEL', 'nlpai-lab/KURE-v1')
    current_image_model = os.environ.get('RAG_IMAGE_EMBEDDING_MODEL', 'Bingsu/clip-vit-base-patch32-ko')
    current_semantic_model = os.environ.get('RAG_SENTENCE_TRANSFORMER_MODEL', 'snunlp/KR-SBERT-V40K-klueNLI-augSTS')

    # 파인튜닝된 모델인 경우 간단한 이름으로 표시
    display_embedding_model = current_embedding_model
    if 'finetuned' in current_embedding_model:
        # 경로에서 모델명만 추출 (예: nlpai-lab-KURE-v1-finetuned-850e6d54)
        model_name = os.path.basename(current_embedding_model)
        display_embedding_model = f"{model_name} (Fine-tuned)"

    logger.info("📊 워커별 모델 로딩 현황")
    logger.info("")
    logger.info(f"Search Worker (PID: {search_pid}, {num_search_threads} threads)")
    logger.info(f"  ⏳ {display_embedding_model} - 텍스트 임베딩 (로딩 중)")
    logger.info(f"  ⏳ {current_image_model} - 이미지 임베딩 (로딩 중)")
    logger.info("  ⏳ DocumentProcessor - 싱글턴 인스턴스 획득 (로딩 중)")
    logger.info("")
    logger.info(f"Embedding Worker (PID: {embedding_pid})")
    logger.info(f"  ⏳ {display_embedding_model} - 텍스트 임베딩 (로딩 중)")
    logger.info(f"  ⏳ {current_image_model} - 이미지 임베딩 (로딩 중)")
    logger.info(f"  📋 {current_semantic_model} - 시맨틱 청킹 (필요시 동적 로딩)")
    logger.info("")
    logger.info("🔄 백그라운드 모델 로딩이 완료되면 ✅ 표시로 변경됩니다.")
    logger.info("")

def initialize_worker_processes():
    """워커 프로세스 초기화 - 단일 프로세스 내 다중 스레드 단위로 개선"""
    global search_process, embedding_process, graphrag_process, search_queue, search_response_queue, embedding_queue, graphrag_queue
    global search_model_ready_event, embedding_model_ready_event, graphrag_ready_event

    from multiprocessing import Process, Queue, Manager, Event
    import psutil

    logger.info("워커 프로세스 초기화 시작")

    # CPU 코어 수 기반으로 검색 스레드 수 결정
    cpu_count = psutil.cpu_count()
    # 검색 스레드 수: CPU 코어 수의 50-75% (최소 4개, 최대 12개)
    num_search_threads = min(max(4, int(cpu_count * 0.75)), 12)
    logger.info(f"CPU 코어: {cpu_count}개, 검색 스레드: {num_search_threads}개로 설정")

    # 큐 및 이벤트 생성
    search_queue = Queue()
    search_response_queue = Queue()
    embedding_queue = Queue()
    graphrag_queue = Queue()
    search_model_ready_event = Event()
    embedding_model_ready_event = Event()
    graphrag_ready_event = Event()

    # 단일 검색 워커 프로세스 시작 (내부에서 다중 스레드 사용)
    search_process = Process(
        target=search_worker,
        args=(search_queue, search_response_queue, search_model_ready_event, num_search_threads)
    )
    search_process.start()

    # 임베딩 워커 프로세스 시작
    embedding_process = Process(
        target=embedding_worker,
        args=(embedding_queue, embedding_model_ready_event)
    )
    embedding_process.start()
    
    # 워커 프로세스 시작 완료 로그 (한번에 표시)
    logger.info(f"워커 프로세스 시작: 검색(PID:{search_process.pid}, {num_search_threads}스레드), 임베딩(PID:{embedding_process.pid})")
    
    # 백그라운드에서 워커 준비 상태 모니터링 시작
    import threading
    monitor_thread = threading.Thread(target=monitor_worker_readiness, daemon=True)
    monitor_thread.start()

def cleanup_worker_processes():
    """워커 프로세스 정리"""
    global search_process, embedding_process, graphrag_process, GLOBAL_DB_POOL
    
    # 서비스 종료 시작 로그
    from datetime import datetime
    logger.info("")
    logger.info("🛑" + "=" * 78 + "🛑")
    logger.info("🛑 RAG 서비스 종료 시작")
    logger.info(f"⏰ 종료 시간: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    logger.info("🛑" + "=" * 78 + "🛑")

    # 검색 워커 프로세스 종료
    if search_process and search_process.is_alive():
        search_process.terminate()
        search_process.join(timeout=5)  # 5초 타임아웃
        logger.info("검색 워커 프로세스 종료됨")
    
    if embedding_process and embedding_process.is_alive():
        embedding_process.terminate()
        embedding_process.join()
        logger.info("임베딩 워커 프로세스 종료됨")
    
    if graphrag_process and graphrag_process.is_alive():
        graphrag_process.terminate()
        graphrag_process.join()
        logger.info("GraphRAG 워커 프로세스 종료됨")
    
    # Connection Pool 정리
    try:
        if 'GLOBAL_DB_POOL' in globals() and GLOBAL_DB_POOL:
            GLOBAL_DB_POOL.closeall()
            logger.info("PostgreSQL 연결 풀 정리 완료")
            GLOBAL_DB_POOL = None
    except NameError:
        logger.info("GLOBAL_DB_POOL이 정의되지 않음 (정상)")
    except Exception as pool_cleanup_error:
        logger.warning(f"연결 풀 정리 중 오류: {pool_cleanup_error}")
    
    # 서비스 종료 완료 로그
    from datetime import datetime
    logger.info("✅ RAG 서비스 종료 완료")
    logger.info(f"⏰ 종료 완료 시간: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    logger.info("🛑" + "=" * 78 + "🛑")
    logger.info("")

if __name__ == '__main__':
    # GPU 사용 시 CUDA forked subprocess 문제 해결을 위해 spawn 방식 사용
    multiprocessing.set_start_method('spawn', force=True)
    
    # 프로세스 종료 시 정리 핸들러 등록
    import signal
    import atexit
    
    def signal_handler(signum, frame):
        logger.info(f"메인 프로세스: 종료 신호 수신 ({signum})")
        cleanup_worker_processes()
        sys.exit(0)
    
    def exit_handler():
        logger.info("메인 프로세스: 정상 종료 시 정리")
        cleanup_worker_processes()
    
    signal.signal(signal.SIGTERM, signal_handler)
    signal.signal(signal.SIGINT, signal_handler)
    atexit.register(exit_handler)
    
    try:
        main()
    except Exception as e:
        if logger:
            logger.info(f"Fatal error: {str(e)}", error=True)
        else:
            print(f"Fatal error: {str(e)}")
        cleanup_worker_processes()
        sys.exit(1)
    finally:
        cleanup_worker_processes()


# GraphRAG 전용 워커 프로세스
def graphrag_worker(graphrag_queue, graphrag_ready_event):
    """GraphRAG 전용 워커 프로세스"""
    import sys
    import os
    import time
    import logging
    import json
    
    # 현재 스크립트의 디렉토리를 Python 경로에 추가
    current_dir = os.path.dirname(os.path.abspath(__file__))
    parent_dir = os.path.dirname(os.path.dirname(current_dir))
    if parent_dir not in sys.path:
        sys.path.insert(0, parent_dir)
    if current_dir not in sys.path:
        sys.path.insert(0, current_dir)
    
    # rag_process.py에서 필요한 기능들 import
    from rag_process import get_rag_settings, GraphRAGProcessor
    
    # 로깅 설정
    log_file = os.path.expanduser("~/.airun/logs/airun-rag.log")
    logging.basicConfig(
        level=logging.INFO,
        format='[%(asctime)s] [GRAPHRAG_WORKER] %(levelname)s: %(message)s',
        handlers=[
            logging.FileHandler(log_file, mode='a'),
            logging.StreamHandler()
        ]
    )
    
    logger = logging.getLogger("GraphRAGWorker")
    logger.info("=" * 60)
    logger.info("🌐 GraphRAG 워커 프로세스 시작")
    logger.info("🔗 엔터티 및 관계 추출 처리")
    logger.info("=" * 60)
    
    # GraphRAG 프로세서 초기화
    graphrag_processor = None
    
    try:
        # 설정 로드
        rag_settings = get_rag_settings()
        enable_graphrag = rag_settings.get('enable_graphrag', False)
        
        if isinstance(enable_graphrag, str):
            enable_graphrag = enable_graphrag.lower() == 'true'
        
        if enable_graphrag:
            logger.info("GraphRAG 설정이 활성화됨, 프로세서 초기화 중...")
            
            # rag_process.py에서 GraphRAGProcessor 사용
            graphrag_processor = get_or_create_graphrag_processor()
            logger.info("✅ GraphRAG 프로세서 초기화 성공!")
        else:
            logger.info("GraphRAG가 설정에서 비활성화됨")
        
        # 준비 완료 신호
        graphrag_ready_event.set()
        logger.info("GraphRAG 워커 준비 완료")
        
        # 큐에서 작업 처리
        while True:
            try:
                if not graphrag_queue.empty():
                    task = graphrag_queue.get(timeout=1)
                    
                    if task is None:  # 종료 신호
                        break
                    
                    if graphrag_processor:
                        if task['type'] == 'process_document':
                            result = graphrag_processor.process_document(task['data'])
                            logger.info(f"문서 처리 완료: {result}")
                        elif task['type'] == 'search':
                            result = graphrag_processor.search(task['query'], task.get('top_k', 5))
                            logger.info(f"검색 완료: {result}")
                    else:
                        logger.warning("GraphRAG 프로세서가 초기화되지 않음")
                
                time.sleep(0.1)
                
            except Exception as e:
                logger.error(f"GraphRAG 워커 작업 처리 중 오류: {str(e)}")
                time.sleep(1)
    
    except Exception as e:
        logger.error(f"GraphRAG 워커 초기화 실패: {str(e)}")
        graphrag_ready_event.set()  # 실패해도 신호는 보내서 무한 대기 방지
    
    logger.info("GraphRAG 워커 프로세스 종료")

# ===== 엔티티 추출 기반 처리 클래스 =====
class KnowledgeGraph:
    """
    지식 그래프 데이터 구조 및 멀티홉 탐색 기능
    """
    def __init__(self):
        self.entities = {}  # entity_id -> {name, type, attributes}
        self.relationships = {}  # rel_id -> {source, target, relation, weight}
        self.entity_index = {}  # name -> entity_id
        self.adjacency = {}  # entity_id -> [connected_entity_ids]
        
    def add_entity(self, name: str, entity_type: str, attributes: dict = None) -> str:
        """엔티티 추가"""
        entity_id = f"{entity_type}_{hashlib.md5(name.encode()).hexdigest()[:8]}"
        self.entities[entity_id] = {
            'name': name,
            'type': entity_type,
            'attributes': attributes or {}
        }
        self.entity_index[name.lower()] = entity_id
        if entity_id not in self.adjacency:
            self.adjacency[entity_id] = set()
        return entity_id
        
    def add_relationship(self, source_name: str, target_name: str, relation: str, weight: float = 1.0) -> str:
        """관계 추가"""
        source_id = self.entity_index.get(source_name.lower())
        target_id = self.entity_index.get(target_name.lower())
        
        if not source_id or not target_id:
            return None
            
        rel_id = f"rel_{hashlib.md5(f'{source_id}_{target_id}_{relation}'.encode()).hexdigest()[:8]}"
        self.relationships[rel_id] = {
            'source': source_id,
            'target': target_id,
            'relation': relation,
            'weight': weight
        }
        
        # 인접 리스트 업데이트
        self.adjacency[source_id].add(target_id)
        self.adjacency[target_id].add(source_id)  # 양방향 그래프
        
        return rel_id
        
    def find_path(self, start_entity: str, end_entity: str, max_hops: int = 3) -> List[List[str]]:
        """두 엔티티 간 경로 찾기 (멀티홉 탐색)"""
        start_id = self.entity_index.get(start_entity.lower())
        end_id = self.entity_index.get(end_entity.lower())
        
        if not start_id or not end_id:
            return []
            
        paths = []
        visited = set()
        current_path = [start_id]
        
        def dfs(current_id: str, target_id: str, path: List[str], depth: int):
            if depth > max_hops:
                return
                
            if current_id == target_id and len(path) > 1:
                paths.append(path.copy())
                return
                
            if current_id in visited:
                return
                
            visited.add(current_id)
            
            for neighbor_id in self.adjacency.get(current_id, []):
                if neighbor_id not in path:
                    path.append(neighbor_id)
                    dfs(neighbor_id, target_id, path, depth + 1)
                    path.pop()
                    
            visited.remove(current_id)
            
        dfs(start_id, end_id, current_path, 0)
        
        # 엔티티 ID를 이름으로 변환
        result_paths = []
        for path in paths:
            name_path = [self.entities[entity_id]['name'] for entity_id in path]
            result_paths.append(name_path)
            
        return result_paths
        
    def get_related_entities(self, entity_name: str, max_distance: int = 2) -> Dict[str, List[Dict]]:
        """특정 엔티티와 관련된 모든 엔티티 찾기"""
        entity_id = self.entity_index.get(entity_name.lower())
        if not entity_id:
            return {}
            
        related = {}
        visited = set()
        queue = [(entity_id, 0)]  # (entity_id, distance)
        
        while queue:
            current_id, distance = queue.pop(0)
            
            if distance > max_distance or current_id in visited:
                continue
                
            visited.add(current_id)
            
            if distance > 0:  # 시작 엔티티 제외
                if distance not in related:
                    related[distance] = []
                    
                entity_info = self.entities[current_id].copy()
                entity_info['distance'] = distance
                related[distance].append(entity_info)
                
            # 인접한 엔티티들을 큐에 추가
            for neighbor_id in self.adjacency.get(current_id, []):
                if neighbor_id not in visited:
                    queue.append((neighbor_id, distance + 1))
                    
        return related

    def graph_search(self, query: str, max_results: int = 5) -> List[Dict]:
        """그래프 기반 관계형 추론 검색"""
        # 쿼리에서 엔티티 추출 (간단한 키워드 매칭)
        query_lower = query.lower()
        matching_entities = []
        
        for entity_name, entity_id in self.entity_index.items():
            if entity_name in query_lower or any(word in entity_name for word in query_lower.split()):
                matching_entities.append((entity_id, entity_name))
        
        if not matching_entities:
            return []
        
        # 매칭된 엔티티들의 관련 정보 수집
        results = []
        for entity_id, entity_name in matching_entities[:max_results]:
            entity_info = self.entities[entity_id].copy()
            
            # 직접 연결된 관계들 찾기
            connected_relations = []
            for rel_id, rel in self.relationships.items():
                if rel['source'] == entity_id:
                    target_entity = self.entities[rel['target']]
                    connected_relations.append({
                        'relation': rel['relation'],
                        'target': target_entity['name'],
                        'target_type': target_entity['type'],
                        'weight': rel['weight']
                    })
                elif rel['target'] == entity_id:
                    source_entity = self.entities[rel['source']]
                    connected_relations.append({
                        'relation': f"[역방향] {rel['relation']}",
                        'target': source_entity['name'],
                        'target_type': source_entity['type'],
                        'weight': rel['weight']
                    })
            
            entity_info['connected_relations'] = connected_relations
            entity_info['relation_count'] = len(connected_relations)
            results.append(entity_info)
        
        # 관계 수가 많은 엔티티 우선 정렬
        results.sort(key=lambda x: x['relation_count'], reverse=True)
        return results[:max_results]

# get_graphrag_processor 함수는 상단으로 이동됨

# GraphRAG 전역 인스턴스
# 함수를 상단으로 이동함

def initialize_graphrag_processor(config: dict):
    """GraphRAG 프로세서 초기화 (호환성 유지)"""
    logger.info("GraphRAG processor using singleton pattern", error=False)

def get_graphrag_processor():
    """GraphRAG 프로세서 싱글톤 인스턴스 가져오기 (GraphRAGProcessor 클래스 정의 후 호출)"""
    try:
        rag_settings = get_rag_settings()
        enable_graphrag = rag_settings.get('enable_graphrag', False)
        
        # Boolean이든 string이든 처리
        if isinstance(enable_graphrag, str):
            enable_graphrag = enable_graphrag.lower() == 'true'
        
        logger.info(f"GraphRAG 설정 확인: enable_graphrag = {enable_graphrag}")
        
        if enable_graphrag:
            # 이제 GraphRAGProcessor 클래스가 이미 정의된 상태
            logger.info("GraphRAG 프로세서 생성 시도...")
            processor = get_or_create_graphrag_processor()
            logger.info("✅ GraphRAG 프로세서 생성 성공!")
            return processor
        else:
            logger.info("GraphRAG가 설정에서 비활성화됨")
            return None
    except Exception as e:
        logger.info(f"GraphRAG processor 초기화 실패: {str(e)}", error=True)
        return None

# ===== 임베딩 모델 평가 전용 함수들 =====

async def cleanup_eval_user_data():
    """eval_user의 기존 임베딩 데이터 정리"""
    try:
        logger.info("eval_user 기존 데이터 정리 중...")
        
        # PostgreSQL에서 eval_user의 문서와 임베딩 삭제
        pool = get_pool_connection()
        async with pool.acquire() as conn:
            # document_embeddings 테이블에서 eval_user 데이터 삭제
            result = await conn.execute(
                "DELETE FROM document_embeddings WHERE user_id = $1",
                "eval_user"
            )
            logger.info(f"eval_user 임베딩 데이터 {result} 개 삭제")
            
            # documents 테이블에서 eval_user 데이터 삭제 (있다면)
            try:
                result2 = await conn.execute(
                    "DELETE FROM documents WHERE user_id = $1",
                    "eval_user"
                )
                logger.info(f"eval_user 문서 메타데이터 {result2} 개 삭제")
            except Exception as e:
                logger.warning(f"문서 메타데이터 삭제 시 오류 (무시): {e}")
                
    except Exception as e:
        logger.error(f"eval_user 데이터 정리 실패: {e}")

async def embed_document_for_evaluation(document_text: str, user_id: str, doc_name: str, model_id: str) -> bool:
    """평가용 문서를 지정된 모델로 임베딩하여 저장"""
    try:
        # 문서를 청크로 분할
        chunks = split_text_into_chunks(document_text)
        logger.info(f"문서 '{doc_name}' 을 {len(chunks)}개 청크로 분할")
        
        # 각 청크를 지정된 모델로 임베딩
        for i, chunk in enumerate(chunks):
            try:
                # 모델 지정하여 임베딩 생성 (워커 큐 사용)
                embedding_result = await generate_embedding_with_model(chunk, model_id)
                
                if embedding_result and 'embedding' in embedding_result:
                    # PostgreSQL에 저장
                    await save_embedding_to_db(
                        user_id=user_id,
                        chunk_text=chunk,
                        embedding=embedding_result['embedding'],
                        doc_name=f"{doc_name}_chunk_{i+1}",
                        model_id=model_id
                    )
                    logger.debug(f"청크 {i+1}/{len(chunks)} 임베딩 저장 완료")
                else:
                    logger.warning(f"청크 {i+1} 임베딩 생성 실패")
                    return False
                    
            except Exception as e:
                logger.error(f"청크 {i+1} 처리 실패: {e}")
                return False
        
        logger.info(f"문서 '{doc_name}' 임베딩 완료 ({len(chunks)}개 청크)")
        return True
        
    except Exception as e:
        logger.error(f"문서 임베딩 실패 ({doc_name}): {e}")
        return False

async def generate_embedding_with_model(text: str, model_id: str) -> dict:
    """지정된 모델로 텍스트 임베딩 생성"""
    try:
        # 임베딩 워커 큐에 작업 요청 (모델 지정)
        embedding_request = {
            'text': text,
            'model_id': model_id,
            'user_id': 'eval_user'
        }
        
        # 워커 큐를 통해 임베딩 요청
        logger.debug(f"모델 {model_id}로 임베딩 요청: {text[:50]}...")
        
        # embedding_queue를 통해 작업 전송
        global embedding_queue
        if embedding_queue:
            # 동기화된 임베딩 생성을 위한 Future 객체 사용
            import asyncio
            future = asyncio.Future()
            
            # 요청에 callback 추가
            embedding_request['callback'] = future
            
            await embedding_queue.put(embedding_request)
            
            # 결과 대기 (최대 30초)
            try:
                result = await asyncio.wait_for(future, timeout=30.0)
                return result
            except asyncio.TimeoutError:
                logger.error(f"임베딩 생성 타임아웃 (모델: {model_id})")
                return None
        else:
            logger.error("임베딩 큐가 없음")
            return None
            
    except Exception as e:
        logger.error(f"모델 {model_id} 임베딩 생성 실패: {e}")
        return None

async def save_embedding_to_db(user_id: str, chunk_text: str, embedding: list, doc_name: str, model_id: str):
    """임베딩을 PostgreSQL 데이터베이스에 저장"""
    try:
        pool = get_pool_connection()
        async with pool.acquire() as conn:
            await conn.execute("""
                INSERT INTO document_embeddings 
                (user_id, embedding, chunk_text, document_title, model_id, created_at)
                VALUES ($1, $2, $3, $4, $5, NOW())
            """, user_id, embedding, chunk_text, doc_name, model_id)
            
    except Exception as e:
        logger.error(f"임베딩 저장 실패: {e}")
        raise

def split_text_into_chunks(text: str, chunk_size: int = 500, overlap: int = 50) -> list:
    """텍스트를 청크로 분할"""
    if len(text) <= chunk_size:
        return [text]
    
    chunks = []
    start = 0
    
    while start < len(text):
        end = start + chunk_size
        if end >= len(text):
            chunks.append(text[start:])
            break
        else:
            # 단어 경계에서 자르기
            chunk = text[start:end]
            last_space = chunk.rfind(' ')
            if last_space > 0:
                chunk = text[start:start + last_space]
                end = start + last_space
            
            chunks.append(chunk)
            start = end - overlap if overlap > 0 else end
    
    return chunks
