import { spawn } from 'child_process';

class ProcessManager {
    static async checkAndCleanupProcesses() {
        if (process.platform !== 'win32') return true;

        try {
            const checkProcess = spawn('tasklist', [], { windowsHide: true });
            let output = '';

            await new Promise((resolve, reject) => {
                checkProcess.stdout.on('data', (data) => {
                    output += data.toString();
                });

                checkProcess.on('close', async (code) => {
                    if (code === 0) {
                        const cmdProcesses = output.match(/cmd.exe/gi);
                        const pythonProcesses = output.match(/python.exe/gi);
                        
                        if ((cmdProcesses && cmdProcesses.length > 15) || 
                            (pythonProcesses && pythonProcesses.length > 10)) {
                            console.log('Cleaning up processes...');
                            
                            // cmd.exe 프로세스 정리
                            await new Promise(resolve => {
                                spawn('taskkill', ['/F', '/IM', 'cmd.exe'], { windowsHide: true })
                                    .on('close', () => setTimeout(resolve, 500));
                            });
                            
                            // python.exe 프로세스 정리
                            await new Promise(resolve => {
                                spawn('taskkill', ['/F', '/IM', 'python.exe'], { windowsHide: true })
                                    .on('close', () => setTimeout(resolve, 500));
                            });
                            
                            // 시스템이 안정화되도록 잠시 대기
                            await new Promise(resolve => setTimeout(resolve, 1000));
                        }
                    }
                    resolve();
                });

                checkProcess.on('error', () => resolve());
            });

            return true;
        } catch (error) {
            console.warn('Error checking process state:', error);
            return false;
        }
    }

    static getWindowsSpawnOptions(additionalEnv = {}) {
        if (process.platform !== 'win32') {
            return {
                shell: true,
                detached: true,
                stdio: 'ignore'
            };
        }

        return {
            windowsHide: true,
            detached: false,
            stdio: ['pipe', 'pipe', 'pipe'],
            shell: true,
            env: {
                ...process.env,
                PYTHONIOENCODING: 'utf-8',
                PYTHONLEGACYWINDOWSSTDIO: '1',
                PYTHONUNBUFFERED: '1',
                ...additionalEnv
            }
        };
    }

    static async handleWindowsProcessError(error, command, options, retryCount = 1) {
        if (process.platform !== 'win32' || retryCount > 3) {
            throw error;
        }

        if (error.code === 'EPERM') {
            console.warn(`Process creation permission error, checking system state... (attempt ${retryCount}/3)`);
            try {
                // 프로세스 정리
                await new Promise(resolve => {
                    spawn('taskkill', ['/F', '/IM', 'cmd.exe'], { windowsHide: true })
                        .on('close', () => setTimeout(resolve, 1000));
                });
                
                // 재시도
                return new Promise((resolve, reject) => {
                    const retryChild = spawn(command, [], options);
                    
                    retryChild.on('error', async (retryError) => {
                        try {
                            const result = await this.handleWindowsProcessError(
                                retryError, 
                                command, 
                                options, 
                                retryCount + 1
                            );
                            resolve(result);
                        } catch (e) {
                            reject(e);
                        }
                    });

                    retryChild.on('spawn', () => {
                        resolve(retryChild);
                    });
                });
            } catch (cleanupError) {
                throw new Error(`Process cleanup failed: ${cleanupError.message}`);
            }
        }
        
        throw error;
    }
}

export default ProcessManager; 