import fs from 'fs';
import path from 'path';
import os from 'os';
import { 
  enqueueApiRequest, 
  enqueueChatRequest, 
  enqueueReportGeneration, 
  chatRequestQueue,
  apiRequestQueue,
  reportGenerationQueue,
  queueApiUsage,
  redisClient
} from './index.js';
import { getVarVal, setVarVal } from '../configuration.js';
import chalk from 'chalk';
import { debugLog, logger } from '../utils/logger.js';

// 설정 파일 로드
const configPath = path.join(path.dirname(new URL(import.meta.url).pathname), 'config.json');
let config = { queuedEndpoints: {} };

try {
  if (fs.existsSync(configPath)) {
    const configData = fs.readFileSync(configPath, 'utf8');
    config = JSON.parse(configData);
  } else {
    logger.warn('큐 미들웨어: 설정 파일을 찾을 수 없습니다.');
  }
} catch (error) {
  logger.error('큐 미들웨어: 설정 파일 로드 중 오류:', error);
}

// API 요청을 큐로 처리하는 미들웨어
export function queueApiMiddleware(activityLogger) {
    return async (req, res, next) => {
        try {
            // Redis 연결 확인
            if (!redisClient) {
                throw new Error('Redis 연결이 초기화되지 않았습니다.');
            }

            // 전체 경로를 엔드포인트로 사용
            const endpoint = `/api/v1${req.path}`;

            // logger.debug(chalk.blue(`[Queue] 요청 처리 - 엔드포인트: ${endpoint}`));

            // 핸들러 찾기
            const handler = global.apiHandlers[endpoint];
            if (!handler) {
                logger.debug(chalk.yellow(`[Queue] 핸들러 없음 - 등록된 핸들러:`, Object.keys(global.apiHandlers).join(',')));
                return next();
            }

            // logger.debug(chalk.blue(`[Queue] 작업 추가 시작 - 엔드포인트: ${endpoint}`));

            let result;
            // 엔드포인트 타입에 따라 적절한 큐 사용
            if (['/api/v1/chat', '/api/v1/code', '/api/v1/agent'].includes(endpoint)) {
                // 세션 ID 처리 - 없으면 새로 생성
                let sessionId = req.body.sessionId || req.body.options?.sessionId;
                
                // logger.debug(chalk.blue(`[Queue] 세션 ID 처리`), {
                //     endpoint: endpoint,
                //     providedSessionId: sessionId,
                //     hasBodySessionId: !!req.body.sessionId,
                //     hasOptionsSessionId: !!req.body.options?.sessionId
                // });

                // 세션 ID가 없으면 새로 생성 (각 엔드포인트에 맞는 접두어 사용)
                // 단, /api/v1/agent의 경우 api_server.js에서 이미 세션을 생성하므로 건너뛰기
                if (!sessionId && endpoint !== '/api/v1/agent') {
                    const { sharedContext } = await import('../aiFeatures.js');
                    
                    // 사용자가 요청한 프로바이더와 모델 정보 확인
                    const requestedProvider = req.body.options?.provider || req.body.provider;
                    const requestedModel = req.body.options?.model || req.body.model;
                    const currentLLM = await getVarVal('USE_LLM');
                    const useProvider = requestedProvider || currentLLM;
                    
                    const userId = req.user?.id || 0;
                    
                    // 엔드포인트에 따라 적절한 세션 타입 결정
                    let sessionType = 'chat';
                    if (endpoint === '/api/v1/code') sessionType = 'code';
                    else if (endpoint === '/api/v1/agent') sessionType = 'agent';
                    
                    const session = await sharedContext.createSession(
                        null, useProvider, userId, sessionType
                    );
                    
                    // 사용자가 지정한 모델이 있으면 세션에 설정
                    if (requestedModel) {
                        session.setModel(requestedModel);
                        try {
                            await session.save();
                        } catch (saveError) {
                            logger.error(`[Queue] 세션 모델 정보 DB 저장 실패:`, saveError);
                        }
                    }
                    
                    sessionId = session.getId();
                    // logger.debug(chalk.green(`[Queue] 새 세션 생성: ${sessionId}, 타입: ${sessionType}, 모델: ${requestedModel || 'default'}`));
                    
                    // req.body에 세션 ID 추가
                    if (!req.body.options) {
                        req.body.options = {};
                    }
                    req.body.options.sessionId = sessionId;
                    req.body.sessionId = sessionId;
                }

                // 이전 형식과 새로운 형식 모두 지원
                const options = {
                    ...req.body.options,
                    sessionId: sessionId,  // 생성되거나 기존 세션 ID
                    userId: req.body.userId || req.body.options?.userId || req.user?.id,  // userId 기본 정보 포함
                    username: req.body.username || req.body.options?.username || req.user?.username,  // username 추가
                    temperature: req.body.temperature || req.body.parameters?.temperature || 0,
                    hideCode: endpoint === '/api/v1/agent' ? true : req.body.hideCode,
                    autoExecute: endpoint === '/api/v1/agent' ? true : req.body.autoExecute,
                    rag: req.body.rag === true || req.body.options?.rag === true || false,
                    web: req.body.web === true || req.body.options?.web === true || false,
                    enableSmartResearch: req.body.enableSmartResearch === true || req.body.options?.enableSmartResearch === true || false, // AI프로 버튼 상태 전달
                    useReview: req.body.USE_REVIEW === true || req.body.options?.USE_REVIEW === true || false, // 코드리뷰 옵션 추가
                    type: req.body.type || req.body.options?.type || 'chat', // type 필드 명시적으로 유지
                    ragSearchScope: req.body.ragSearchScope || req.body.options?.ragSearchScope || 'personal', // RAG 검색 범위 추가
                    // 멘션 기능 관련 필드 추가
                    mentionedDocumentContent: req.body.mentionedDocumentContent || req.body.options?.mentionedDocumentContent || '',
                    mentionedDocuments: req.body.mentionedDocuments || req.body.options?.mentionedDocuments || []
                };

                // 이미지 처리 - 원본 형식 유지
                // logger.debug(chalk.cyan('[Queue] 미들웨어 이미지 처리 시작:'), {
                //     hasOptionsImage: !!req.body.options?.image,
                //     hasBodyImage: !!req.body.image,
                //     optionsImageType: req.body.options?.image ? (typeof req.body.options.image) : 'none',
                //     bodyImageType: req.body.image ? (typeof req.body.image) : 'none'
                // });

                if (req.body.options?.image) {
                    options.image = req.body.options.image;
                    // logger.debug(chalk.cyan('[Queue] options.image 설정됨:'), {
                    //     type: typeof options.image,
                    //     isPath: !options.image.startsWith('data:'),
                    //     length: options.image.length
                    // });
                } else if (req.body.image) {
                    options.image = req.body.image;
                    // logger.debug(chalk.cyan('[Queue] body.image에서 설정됨:'), {
                    //     type: typeof options.image,
                    //     isPath: !options.image.startsWith('data:'),
                    //     length: options.image.length
                    // });
                }
                
                // 이미지 배열도 유지 (다중 이미지 지원을 위해)
                if (req.body.images) {
                    options.images = req.body.images;
                }

                // logger.debug(chalk.cyan('[Queue] 최종 이미지 데이터:'), {
                //     hasImage: !!options.image,
                //     hasImages: !!options.images,
                //     imageType: options.image ? (options.image.startsWith('data:') ? 'base64' : 'path') : 'none'
                // });

                // logger.debug('[Queue] req.body:', req.body);
                // logger.debug('[Queue] req.body.options:', req.body.options);

                result = await enqueueChatRequest(req.body.prompt, options, endpoint);
            } else if (endpoint === '/api/v1/report') {
                // 요청 처리 방식 개선
                let reportOptions = { ...req.body };
                
                // options 객체가 있다면 상위 레벨로 이동 (중첩 구조 해결)
                if (reportOptions.options) {
                    // 중첩된 options 객체가 있는 경우 해당 필드를 상위 레벨로 병합
                    reportOptions = {
                        ...reportOptions,
                        ...reportOptions.options,
                        options: undefined // 중첩된 객체 제거
                    };
                }
                
                // 사용자 정보 추가 (사용자명을 기본으로)
                reportOptions.userId = reportOptions.userId || req.user?.id;
                reportOptions.username = reportOptions.username || req.user?.username;
                
                // 사용자명이 없으면 사용자 ID에서 변환 시도
                if (!reportOptions.username && reportOptions.userId) {
                    try {
                        const { getUsernameFromUserId } = await import('../api_server.js');
                        const username = await getUsernameFromUserId(reportOptions.userId);
                        reportOptions.username = username || String(reportOptions.userId);
                    } catch (error) {
                        logger.warn('[Queue] 사용자명 변환 실패, 사용자 ID를 문자열로 사용:', error.message);
                        reportOptions.username = String(reportOptions.userId);
                    }
                }
                
                logger.debug(chalk.blue('[Queue] 처리된 보고서 옵션:'), {
                    ...reportOptions,
                    userId: reportOptions.userId,
                    username: reportOptions.username
                });
                result = await enqueueReportGeneration(reportOptions, endpoint);
            } else if (endpoint.startsWith('/api/v1/rag/')) {
                // RAG 관련 엔드포인트 처리
                if (endpoint === '/api/v1/rag/add') {
                    // 문서 추가 요청
                    result = await enqueueApiRequest(
                        endpoint,
                        req.method,
                        req.params,
                        req.body,
                        req.headers,
                        endpoint,
                        'high' // 높은 우선순위
                    );
                } else if (endpoint === '/api/v1/rag/init') {
                    // RAG 초기화 요청
                    result = await enqueueApiRequest(
                        endpoint,
                        req.method,
                        req.params,
                        req.body,
                        req.headers,
                        endpoint,
                        'high'
                    );
                } else {
                    // 기타 RAG 관련 요청
                    result = await enqueueApiRequest(
                        endpoint,
                        req.method,
                        req.params,
                        req.body,
                        req.headers,
                        endpoint
                    );
                }
            } else if (endpoint.startsWith('/api/v1/web/')) {
                result = await enqueueApiRequest(
                    endpoint,
                    req.method,
                    req.params,
                    req.body,
                    req.headers,
                    endpoint
                );
            }

            if (!result) {
                throw new Error(`큐 작업 추가 실패: ${endpoint}`);
            }

            // Bull Queue ID와 프로젝트 ID 모두 표시
            if (result.bullJobId) {
                logger.debug(chalk.green(`[Queue] 작업 추가 - Project ID: ${result.jobId}, Bull Queue ID: ${result.bullJobId}`));
            } else {
                logger.debug(chalk.green(`[Queue] 작업 추가 - JobID: ${result.jobId}`));
            }

            await activityLogger.logActivity(req.user?.id || 0, endpoint, {
              ip: req.ip,
              userAgent: `${req.headers["user-agent"]}, ${JSON.stringify(req.body)}, jobId:${result.jobId}`
            });
            
            return res.json({
                success: true,
                jobId: result.jobId,
                status: 'queued',
                message: '작업이 큐에 추가되었습니다.',
                downloadInfo: result.downloadInfo || []
            });
        } catch (error) {
            logger.error(chalk.red('[Queue] 작업 추가 실패:'), error);
            logger.error(chalk.red('[Queue] 에러 상세:'), {
                message: error.message,
                stack: error.stack,
                response: error.response?.data
            });
            next(error);
        }
    };
}

// API 핸들러 등록
export function registerApiHandlers(app, handlers) {
  if (!app.locals.apiHandlers) {
    app.locals.apiHandlers = {};
  }
  
  // 전역 핸들러 저장소 설정
  if (!global.apiHandlers) {
    global.apiHandlers = {};
  }
  
  // 핸들러 함수들을 app.locals와 전역 저장소에 등록
  Object.keys(handlers).forEach(endpoint => {
    app.locals.apiHandlers[endpoint] = handlers[endpoint];
    global.apiHandlers[endpoint] = handlers[endpoint];
    // logger.debug(`[Queue] 엔드포인트 핸들러 등록: ${endpoint}`);
  });
}

// 작업 상태 조회 API 미들웨어
export function queueStatusMiddleware() {
  return async (req, res, next) => {
    // 상태 확인 요청은 api_server.js의 엔드포인트에서 직접 처리하도록 함
    // 미들웨어에서는 상태 확인을 처리하지 않음
    return next();
  };
} 