#!/bin/bash

# NVIDIA Driver 580 Open 오프라인 패키지 다운로드 스크립트
# nvidia-driver-580-open 및 모든 의존성 패키지 다운로드

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
OFFLINE_DIR="$(cd "$SCRIPT_DIR/../offline-repo" && pwd)"
PACKAGES_DIR="$OFFLINE_DIR/packages"

# 색상 코드
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'

show_info() { echo -e "${BLUE}ℹ️  $1${NC}"; }
show_success() { echo -e "${GREEN}✅ $1${NC}"; }
show_warning() { echo -e "${YELLOW}⚠️  $1${NC}"; }
show_error() { echo -e "${RED}❌ $1${NC}"; }

echo ""
echo -e "${CYAN}╔═══════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║                                                           ║${NC}"
echo -e "${CYAN}║       NVIDIA Driver 580 오프라인 패키지 다운로드         ║${NC}"
echo -e "${CYAN}║                                                           ║${NC}"
echo -e "${CYAN}╚═══════════════════════════════════════════════════════════╝${NC}"
echo ""

# packages 디렉토리로 이동
cd "$PACKAGES_DIR"

# NVIDIA 드라이버 패키지 목록 (nvidia-driver-580-open의 의존성)
NVIDIA_PACKAGES=(
    "nvidia-driver-580-open"
    "nvidia-dkms-580-open"
    "nvidia-kernel-common-580"
    "nvidia-kernel-source-580-open"
    "libnvidia-compute-580"
    "libnvidia-extra-580"
    "libnvidia-decode-580"
    "libnvidia-encode-580"
    "libnvidia-fbc1-580"
    "libnvidia-gl-580"
    "xserver-xorg-video-nvidia-580"
    "nvidia-settings"
    "nvidia-utils-580"
    "nvidia-modprobe"
    "nvidia-prime"
    "screen-resolution-extra"
    "dkms"
    "libvdpau1"
    "mesa-vdpau-drivers"
    "vdpau-driver-all"
)

# 추가 필수 커널 헤더 및 빌드 도구
KERNEL_PACKAGES=(
    "linux-headers-generic"
    "linux-headers-$(uname -r)"
    "build-essential"
    "dkms"
)

show_info "NVIDIA 드라이버 패키지 다운로드 시작..."
echo ""

DOWNLOAD_COUNT=0
SKIP_COUNT=0
FAIL_COUNT=0

# NVIDIA 패키지 다운로드
for package in "${NVIDIA_PACKAGES[@]}"; do
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${BLUE}  패키지: $package${NC}"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

    # 이미 다운로드되었는지 확인
    if ls ${package}_*.deb 1> /dev/null 2>&1; then
        show_success "$package 이미 다운로드되어 있습니다"
        ((SKIP_COUNT++))
        continue
    fi

    # 패키지 다운로드
    show_info "$package 다운로드 중..."
    if apt-get download "$package" 2>/dev/null; then
        show_success "$package 다운로드 완료!"
        ((DOWNLOAD_COUNT++))
    else
        show_warning "$package 다운로드 실패 (패키지가 없을 수 있습니다)"
        ((FAIL_COUNT++))
    fi
    echo ""
done

# 커널 헤더 다운로드
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  커널 헤더 및 빌드 도구 다운로드                          ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""

for package in "${KERNEL_PACKAGES[@]}"; do
    if ls ${package%%\(*}_*.deb 1> /dev/null 2>&1; then
        show_success "$package 이미 다운로드되어 있습니다"
        ((SKIP_COUNT++))
    else
        show_info "$package 다운로드 중..."
        if apt-get download "$package" 2>/dev/null; then
            show_success "$package 다운로드 완료!"
            ((DOWNLOAD_COUNT++))
        else
            show_warning "$package 다운로드 실패"
            ((FAIL_COUNT++))
        fi
    fi
done

# 의존성 자동 다운로드
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  NVIDIA 드라이버 의존성 자동 다운로드                    ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""

show_info "nvidia-driver-580-open 의존성 분석 중..."

# apt-rdepends로 의존성 다운로드
if command -v apt-rdepends &> /dev/null; then
    apt-rdepends nvidia-driver-580-open | grep -v "^ " | while read -r dep; do
        if [ ! -z "$dep" ]; then
            if ! ls ${dep}_*.deb 1> /dev/null 2>&1; then
                apt-get download "$dep" 2>/dev/null || true
            fi
        fi
    done
else
    show_warning "apt-rdepends가 설치되어 있지 않습니다. 기본 의존성만 다운로드됩니다."
fi

# Repository 인덱스 재생성
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  저장소 인덱스 재생성                                    ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""

show_info "Packages.gz 생성 중..."
dpkg-scanpackages . /dev/null | gzip -9c > Packages.gz
show_success "Packages.gz 생성 완료!"

# 요약
echo ""
echo -e "${CYAN}╔═══════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║                                                           ║${NC}"
echo -e "${CYAN}║       ✅ NVIDIA 드라이버 패키지 다운로드 완료!           ║${NC}"
echo -e "${CYAN}║                                                           ║${NC}"
echo -e "${CYAN}╚═══════════════════════════════════════════════════════════╝${NC}"
echo ""

show_success "다운로드 완료: $DOWNLOAD_COUNT 개"
show_success "스킵: $SKIP_COUNT 개"
if [ $FAIL_COUNT -gt 0 ]; then
    show_warning "실패: $FAIL_COUNT 개"
fi

echo ""
NVIDIA_SIZE=$(du -sh . | cut -f1)
show_success "NVIDIA 패키지 디렉토리 총 크기: $NVIDIA_SIZE"

echo ""
echo -e "${YELLOW}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${YELLOW}  오프라인 환경 설치 방법                                ${NC}"
echo -e "${YELLOW}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""
echo "1. 전체 offline-repo/ 디렉토리를 오프라인 환경으로 전송"
echo ""
echo "2. 오프라인 환경에서 실행:"
echo "   cd offline-repo"
echo "   sudo ./scripts/install-offline.sh"
echo ""
echo "3. NVIDIA 드라이버 설치:"
echo "   sudo apt-get update"
echo "   sudo apt-get install nvidia-driver-580-open"
echo ""
echo "4. 재부팅 후 확인:"
echo "   nvidia-smi"
echo ""
