#!/bin/bash

#############################################
# Python Wheels Offline Downloader
# Downloads all Python packages for offline installation
#############################################

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

show_info() { echo -e "${BLUE}ℹ️  $1${NC}"; }
show_success() { echo -e "${GREEN}✅ $1${NC}"; }
show_warning() { echo -e "${YELLOW}⚠️  $1${NC}"; }
show_error() { echo -e "${RED}❌ $1${NC}"; }

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
WHEELS_DIR="$SCRIPT_DIR/../offline-repo/python-wheels"

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║       Python Wheels 오프라인 다운로더                ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════════════════════╝${NC}"
echo ""

# Create wheels directory
show_info "Wheels 디렉토리 생성 중..."
mkdir -p "$WHEELS_DIR"

# Check for requirements files
AIRUN_ROOT="$SCRIPT_DIR/.."
REQUIREMENTS_FILES=()

if [ -f "$AIRUN_ROOT/requirements-linux.txt" ]; then
    REQUIREMENTS_FILES+=("$AIRUN_ROOT/requirements-linux.txt")
fi

if [ -f "$AIRUN_ROOT/requirement-system.txt" ]; then
    REQUIREMENTS_FILES+=("$AIRUN_ROOT/requirement-system.txt")
fi

if [ ${#REQUIREMENTS_FILES[@]} -eq 0 ]; then
    show_error "requirements 파일을 찾을 수 없습니다"
    exit 1
fi

show_success "발견된 requirements 파일: ${#REQUIREMENTS_FILES[@]}개"

# Create a temporary virtual environment for downloading
TEMP_VENV="$SCRIPT_DIR/../.temp_venv"
show_info "임시 가상환경 생성 중..."
python3 -m venv "$TEMP_VENV"
source "$TEMP_VENV/bin/activate"

# Upgrade pip
show_info "pip 업그레이드 중..."
pip install --upgrade pip wheel

# Download wheels for each requirements file
for req_file in "${REQUIREMENTS_FILES[@]}"; do
    show_info "다운로드 중: $(basename $req_file)"

    # Download with PyTorch CUDA 12.9 support
    pip download \
        -r "$req_file" \
        --dest "$WHEELS_DIR" \
        --index-url https://download.pytorch.org/whl/cu129 \
        --extra-index-url https://pypi.org/simple \
        --platform manylinux2014_x86_64 \
        --python-version 312 \
        --only-binary=:all: \
        || show_warning "일부 패키지 다운로드 실패 (계속 진행)"

    # Also download source distributions for packages that don't have wheels
    pip download \
        -r "$req_file" \
        --dest "$WHEELS_DIR" \
        --index-url https://download.pytorch.org/whl/cu129 \
        --extra-index-url https://pypi.org/simple \
        --no-binary :all: \
        || show_warning "일부 소스 패키지 다운로드 실패"
done

# Download additional essential packages
show_info "추가 필수 패키지 다운로드 중..."
ADDITIONAL_PACKAGES=(
    "Cython"
    "numpy"
    "setuptools"
    "wheel"
    "pip"
    "huggingface-hub"
    "sentence-transformers"
    "transformers"
)

for package in "${ADDITIONAL_PACKAGES[@]}"; do
    show_info "다운로드: $package"
    pip download "$package" --dest "$WHEELS_DIR" || true
done

# Cleanup
deactivate
rm -rf "$TEMP_VENV"

# Create installation script
cat > "$WHEELS_DIR/install-wheels.sh" << 'INSTALL_SCRIPT'
#!/bin/bash

WHEELS_DIR="$(cd "$(dirname "$0")" && pwd)"

echo "🐍 Python 패키지 오프라인 설치 중..."

# Check if in virtual environment
if [ -z "$VIRTUAL_ENV" ]; then
    echo "⚠️  경고: 가상환경이 활성화되지 않았습니다"
    echo "권장: source ~/.airun_venv/bin/activate"
    read -p "계속하시겠습니까? (y/n) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        exit 1
    fi
fi

# Install all wheels
pip install --no-index --find-links="$WHEELS_DIR" "$WHEELS_DIR"/*.whl

echo "✅ 오프라인 설치 완료!"
INSTALL_SCRIPT

chmod +x "$WHEELS_DIR/install-wheels.sh"

# Create README
cat > "$WHEELS_DIR/README.md" << 'README'
# Python Wheels 오프라인 설치 가이드

## 사용 방법

### 1. 가상환경 활성화

```bash
source ~/.airun_venv/bin/activate
```

### 2. Wheels 설치

```bash
cd python-wheels
./install-wheels.sh
```

또는 수동 설치:

```bash
pip install --no-index --find-links=. *.whl
```

### 3. PyTorch CUDA 설정 확인

```bash
python -c "import torch; print(f'PyTorch {torch.__version__}, CUDA: {torch.cuda.is_available()}')"
```

## 파일 구조

- `*.whl`: Python wheel 패키지
- `*.tar.gz`: 소스 배포 패키지
- `install-wheels.sh`: 자동 설치 스크립트

## 주의사항

- CUDA 12.9 지원 PyTorch wheel이 포함되어 있습니다
- manylinux2014_x86_64 플랫폼용으로 다운로드되었습니다
- Python 3.12 버전을 대상으로 합니다
README

# Count and report
WHEEL_COUNT=$(ls -1 "$WHEELS_DIR"/*.whl 2>/dev/null | wc -l)
TOTAL_SIZE=$(du -sh "$WHEELS_DIR" | cut -f1)

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║           Wheels 다운로드 완료!                       ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${BLUE}저장 위치:${NC} $WHEELS_DIR"
echo -e "${BLUE}Wheel 개수:${NC} $WHEEL_COUNT"
echo -e "${BLUE}총 크기:${NC} $TOTAL_SIZE"
echo ""
echo -e "${YELLOW}다음 단계:${NC}"
echo "1. python-wheels 디렉토리를 오프라인 시스템으로 복사"
echo "2. 가상환경 활성화: source ~/.airun_venv/bin/activate"
echo "3. install-wheels.sh 실행"
echo ""
