// schema.js (refactored)

import bcrypt from 'bcrypt';
import crypto from 'crypto';
import fs from 'fs';
import path from 'path';

/**
 * =============================================================================
 * HAMONIZE (AIRUN) DATABASE SCHEMA
 * =============================================================================
 * - Safe, idempotent migrations with advisory lock
 * - Modular steps with helpers
 * - Standardized vector dims: embedding=1024, image_embedding=512
 * =============================================================================
 */

const SCHEMA_VERSION = '3.6.1'; // AI 폼 빌더 첨부파일 테이블 추가 및 CASCADE 제약조건 강화

/* ---------------------------------- Utils ---------------------------------- */

const compareVersions = (a, b) => {
  const pa = String(a).split('.').map(Number);
  const pb = String(b).split('.').map(Number);
  const len = Math.max(pa.length, pb.length);
  for (let i = 0; i < len; i++) {
    const va = pa[i] || 0;
    const vb = pb[i] || 0;
    if (va > vb) return 1;
    if (va < vb) return -1;
  }
  return 0;
};

const withTxn = async (db, fn) => {
  await db.query('BEGIN');
  try {
    const res = await fn();
    await db.query('COMMIT');
    return res;
  } catch (e) {
    await db.query('ROLLBACK');
    throw e;
  }
};

const log = (...args) => console.log('[SCHEMA]', ...args);
const warn = (...args) => console.warn('[SCHEMA][WARN]', ...args);
const err = (...args) => console.error('[SCHEMA][ERROR]', ...args);

/* ------------------------------ Version table ------------------------------ */

const createMigrationTable = async (db) => {
  await db.query(`
    CREATE TABLE IF NOT EXISTS schema_migrations (
      version VARCHAR(50) PRIMARY KEY,
      applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `);
};

const getCurrentVersion = async (db) => {
  try {
    await createMigrationTable(db);
    const r = await db.query(
      'SELECT version FROM schema_migrations ORDER BY applied_at DESC, version DESC LIMIT 1'
    );
    return r.rows[0]?.version || '0';
  } catch (e) {
    err('마이그레이션 버전 확인 실패:', e);
    return '0';
  }
};

const recordMigration = async (db, version) => {
  await db.query(
    'INSERT INTO schema_migrations (version) VALUES ($1) ON CONFLICT (version) DO NOTHING',
    [version]
  );
};

/* --------------------------------- Helpers --------------------------------- */

const advisoryKey = BigInt(
  // A stable hash; Node doesn't have built-in hashtext → use a fixed constant
  // Change only if you need a different lock per product.
  0x41_49_52_55_4e // 'AIRUN'
);

const acquireAdvisoryLock = async (db) => {
  const { rows } = await db.query('SELECT pg_try_advisory_lock($1)', [advisoryKey]);
  return rows[0]?.pg_try_advisory_lock === true;
};

const releaseAdvisoryLock = async (db) => {
  await db.query('SELECT pg_advisory_unlock($1)', [advisoryKey]);
};

const tableExists = async (db, table) => {
  const { rows } = await db.query(
    `
    SELECT EXISTS (
      SELECT FROM information_schema.tables
      WHERE table_schema='public' AND table_name=$1
    ) AS exists
  `,
    [table]
  );
  return !!rows[0]?.exists;
};

const columnExists = async (db, table, column) => {
  const { rows } = await db.query(
    `
    SELECT EXISTS (
      SELECT 1 FROM information_schema.columns
      WHERE table_name=$1 AND column_name=$2
    ) AS exists
  `,
    [table, column]
  );
  return !!rows[0]?.exists;
};

const indexExists = async (db, indexName) => {
  const { rows } = await db.query(
    `
    SELECT EXISTS (
      SELECT 1 FROM pg_indexes
      WHERE indexname = $1
    ) AS exists
  `,
    [indexName]
  );
  return !!rows[0]?.exists;
};

const constraintExists = async (db, name) => {
  const { rows } = await db.query(
    `
    SELECT EXISTS (
      SELECT 1 FROM pg_constraint WHERE conname = $1
    ) AS exists
  `,
    [name]
  );
  return !!rows[0]?.exists;
};

const ensureExtension = async (db, ext) => {
  await db.query(`CREATE EXTENSION IF NOT EXISTS ${ext};`);
};

const ensureTable = async (db, ddl) => {
  await db.query(ddl);
};

const ensureColumn = async (db, table, column, definition) => {
  if (!(await columnExists(db, table, column))) {
    await db.query(`ALTER TABLE ${table} ADD COLUMN ${definition};`);
    log(`컬럼 추가: ${table}.${column}`);
  }
};

const ensureIndex = async (db, name, ddl) => {
  if (!(await indexExists(db, name))) {
    await db.query(ddl);
    log(`인덱스 생성: ${name}`);
  }
};

const ensureUniqueIndex = async (db, name, ddl) => ensureIndex(db, name, ddl);

const ensureConstraint = async (db, name, ddl) => {
  if (!(await constraintExists(db, name))) {
    await db.query(ddl);
    log(`제약조건 생성: ${name}`);
  }
};

/* ------------------------- Section: Projects (base) ------------------------- */

const ensureProjectsTable = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS projects (
      id SERIAL PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      user_id VARCHAR(100) NOT NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      UNIQUE(user_id, name)
    );
  `
  );
};

/* --------------------- Section: Users/Auth & basic tables ------------------- */

const migrateUsersAndAuth = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS users (
      id SERIAL PRIMARY KEY,
      username VARCHAR(50) UNIQUE NOT NULL,
      email VARCHAR(255) UNIQUE,
      password_hash VARCHAR(255) NOT NULL,
      name VARCHAR(100),
      role VARCHAR(50) DEFAULT 'user',
      language VARCHAR(10) DEFAULT 'ko',
      status VARCHAR(20) DEFAULT 'active',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      last_login TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS permissions (
      id SERIAL PRIMARY KEY,
      name VARCHAR(100) UNIQUE NOT NULL,
      description TEXT
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS role_permissions (
      role VARCHAR(50),
      permission_id INTEGER REFERENCES permissions(id) ON DELETE CASCADE,
      PRIMARY KEY (role, permission_id)
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS api_keys (
      id SERIAL PRIMARY KEY,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      key_value VARCHAR(64) UNIQUE NOT NULL,
      name VARCHAR(100),
      status VARCHAR(20) DEFAULT 'active',
      permissions JSONB DEFAULT '{}',
      allowed_ips TEXT[],
      rate_limit INTEGER DEFAULT 1000,
      last_used TIMESTAMP,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      expires_at TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS activity_logs (
      id SERIAL PRIMARY KEY,
      user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
      action VARCHAR(255),
      details JSONB DEFAULT '{}',
      ip_address VARCHAR(45),
      user_agent TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS user_ip_logs (
      id SERIAL PRIMARY KEY,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      ip_address VARCHAR(45) NOT NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  // v3.1: system_prompt 컬럼
  await ensureColumn(db, 'users', 'system_prompt', 'system_prompt TEXT');
};

/* -------------------------- Section: Sessions ------------------------------- */

const migrateSessions = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS sessions (
      id VARCHAR(64) PRIMARY KEY,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      username VARCHAR(100),
      data JSONB NOT NULL DEFAULT '{}',
      provider VARCHAR(50),
      model VARCHAR(100),
      title VARCHAR(255),
      message_count INTEGER DEFAULT 0,
      last_message TEXT,
      status VARCHAR(20) DEFAULT 'active',
      type VARCHAR(20) DEFAULT 'chat',
      created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
      expires_at TIMESTAMP WITH TIME ZONE
    );
  `
  );
};

/* ---------------------- Section: RAG / Embeddings -------------------------- */

const VECTOR_DIM = 1024;
const IMAGE_VECTOR_DIM = 512; // 표준화

const migrateRAG = async (db) => {
  await ensureExtension(db, 'vector');

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS document_embeddings (
      id SERIAL PRIMARY KEY,
      is_embedding boolean NOT NULL DEFAULT FALSE,
      doc_id VARCHAR(500) NOT NULL,
      filename VARCHAR(500) NOT NULL,
      chunk_index INTEGER NOT NULL,
      chunk_text TEXT NOT NULL,
      embedding vector(${VECTOR_DIM}),
      image_embedding vector(${IMAGE_VECTOR_DIM}),
      user_id VARCHAR(255),
      source TEXT,
      file_mtime BIGINT,
      metadata JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      CONSTRAINT unique_doc_chunk UNIQUE (doc_id, chunk_index)
    );
  `
  );

  // 호환성: 기존 컬럼 추가(이미 존재하면 skip)
  // 주: 이전 스키마에서 image_embedding 768을 추가하는 로직이 있었음.
  if (await tableExists(db, 'document_embeddings')) {
    // is_embedding
    await ensureColumn(db, 'document_embeddings', 'is_embedding', 'is_embedding boolean NOT NULL DEFAULT FALSE');

    // image_embedding 존재 & 차원 확인
    const dimCheck = await db.query(`
      SELECT atttypmod
      FROM pg_attribute a
      JOIN pg_class c ON a.attrelid = c.oid
      JOIN pg_type t ON a.atttypid = t.oid
      WHERE c.relname='document_embeddings' AND a.attname='image_embedding' AND NOT a.attisdropped
      LIMIT 1;
    `);
    // pgvector stores dim in atttypmod-4; we avoid brittle calc and just warn
    if (dimCheck.rows.length) {
      // Unable to reliably decode dim here without function; just warn user to verify
      // Or you can try: SELECT vector_dims(image_embedding) FROM document_embeddings LIMIT 1; (pgvector>=0.7.0)
      try {
        const r = await db.query(`SELECT vector_dims(image_embedding) AS d FROM document_embeddings WHERE image_embedding IS NOT NULL LIMIT 1;`);
        const d = r.rows[0]?.d;
        if (d && Number(d) !== IMAGE_VECTOR_DIM) {
          warn(
            `document_embeddings.image_embedding 차원=${d} (권장=${IMAGE_VECTOR_DIM}). 자동 변경하지 않습니다. 백업 후 수동 마이그레이션을 권장합니다.`
          );
        }
      } catch {
        // older pgvector without vector_dims()
        warn(
          `document_embeddings.image_embedding 차원 확인 불가 (pgvector 구버전). 권장=${IMAGE_VECTOR_DIM}. 필요 시 수동 점검하세요.`
        );
      }
    } else {
      // 컬럼이 없다면 생성
      await ensureColumn(
        db,
        'document_embeddings',
        'image_embedding',
        `image_embedding vector(${IMAGE_VECTOR_DIM})`
      );
    }
  }

  // chat_documents: 먼저 create → 그 후 확장
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS chat_documents (
      id SERIAL PRIMARY KEY,
      filename VARCHAR(255) NOT NULL,
      filepath VARCHAR(500) NOT NULL,
      filesize BIGINT,
      mimetype VARCHAR(100),
      user_id VARCHAR(100) NOT NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  // 확장 컬럼
  const chatDocColumns = [
    ["upload_status", "upload_status VARCHAR(20) DEFAULT 'uploaded' CHECK (upload_status IN ('uploading','uploaded','failed'))"],
    ["embedding_status", "embedding_status VARCHAR(20) DEFAULT 'pending' CHECK (embedding_status IN ('pending','processing','completed','failed'))"],
    ['project_name', 'project_name VARCHAR(100)'],
    ['project_id', 'project_id INTEGER REFERENCES projects(id) ON DELETE SET NULL'],
    ['tags', 'tags TEXT[]'],
    ['metadata', "metadata JSONB DEFAULT '{}'"],
    ['error_message', 'error_message TEXT'],
    ['processed_at', 'processed_at TIMESTAMP'],
  ];
  for (const [name, def] of chatDocColumns) {
    await ensureColumn(db, 'chat_documents', name, def);
  }

  // UNIQUE(user_id, filename)
  await ensureUniqueIndex(
    db,
    'idx_chat_documents_user_filename',
    `CREATE UNIQUE INDEX idx_chat_documents_user_filename ON chat_documents(user_id, filename);`
  );

  // 채워주기(최초 상태 보정) - 존재한다면만
  await db.query(`
    UPDATE chat_documents cd
    SET embedding_status = CASE
      WHEN EXISTS (
        SELECT 1 FROM document_embeddings de
        WHERE de.filename LIKE '%' || cd.filename
          AND de.user_id = cd.user_id
          AND de.is_embedding = true
      ) THEN 'completed'
      ELSE 'pending'
    END,
    processed_at = CASE
      WHEN EXISTS (
        SELECT 1 FROM document_embeddings de
        WHERE de.filename LIKE '%' || cd.filename
          AND de.user_id = cd.user_id
          AND de.is_embedding = true
      ) THEN cd.updated_at
      ELSE NULL
    END
    WHERE cd.embedding_status IS NULL OR cd.embedding_status = 'pending';
  `);

  // RAG 전용(모델별) 테이블 인덱스 존재 시 인덱스 생성
  const ragTableName = 'nlpai_lab_kure_v1_document_embeddings';
  if (await tableExists(db, ragTableName)) {
    const ragIdx = (n, ddl) => ensureIndex(db, n, ddl.replaceAll('@@T', ragTableName));
    await ragIdx(
      `idx_${ragTableName}_doc_id`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_doc_id ON @@T(doc_id);`
    );
    await ragIdx(
      `idx_${ragTableName}_filename`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_filename ON @@T(filename);`
    );
    await ragIdx(
      `idx_${ragTableName}_chunk_index`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_chunk_index ON @@T(chunk_index);`
    );
    await ragIdx(
      `idx_${ragTableName}_user_id`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_user_id ON @@T(user_id);`
    );
    await ragIdx(
      `idx_${ragTableName}_source`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_source ON @@T(source);`
    );
    await ragIdx(
      `idx_${ragTableName}_created_at`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_created_at ON @@T(created_at);`
    );
    await ragIdx(
      `idx_${ragTableName}_has_image`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_has_image ON @@T((image_embedding IS NOT NULL));`
    );
    await ragIdx(
      `idx_${ragTableName}_is_embedding`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_is_embedding ON @@T(is_embedding);`
    );
    await ragIdx(
      `idx_${ragTableName}_vec_cos`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_vec_cos ON @@T USING ivfflat (embedding vector_cosine_ops) WITH (lists = 100);`
    );
    await ragIdx(
      `idx_${ragTableName}_img_vec_cos`,
      `CREATE INDEX IF NOT EXISTS idx_${ragTableName}_img_vec_cos ON @@T USING ivfflat (image_embedding vector_cosine_ops) WITH (lists = 100);`
    );
  } else {
    log(`${ragTableName} 미존재: RAG 전용 인덱스 건너뜀`);
  }
};

/* ----------------------------- Section: Reports ----------------------------- */

const migrateReports = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS report_jobs (
      id SERIAL PRIMARY KEY,
      job_id VARCHAR(255) UNIQUE NOT NULL,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      username VARCHAR(100),
      title VARCHAR(500),
      executive_summary TEXT,
      template VARCHAR(100),
      output_format VARCHAR(10) DEFAULT 'pdf',
      status VARCHAR(20) DEFAULT 'queued',
      progress INTEGER DEFAULT 0,
      output_path VARCHAR(1000),
      cache_key VARCHAR(255),
      error_message TEXT,
      metadata JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      completed_at TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS report_section_cache (
      id SERIAL PRIMARY KEY,
      job_id VARCHAR(255) NOT NULL,
      section_name VARCHAR(255) NOT NULL,
      subsection_name VARCHAR(255),
      content TEXT,
      chart_data JSONB,
      table_data JSONB,
      cache_key VARCHAR(255),
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      expires_at TIMESTAMP,
      FOREIGN KEY (job_id) REFERENCES report_jobs(job_id) ON DELETE CASCADE
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS user_report_settings (
      id SERIAL PRIMARY KEY,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      setting_key VARCHAR(255) NOT NULL,
      setting_value TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      UNIQUE(user_id, setting_key)
    );
  `
  );
};

/* --------------------------- Section: Scheduler ----------------------------- */

const migrateScheduler = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS saved_tasks (
      id SERIAL PRIMARY KEY,
      task_id VARCHAR(255) UNIQUE NOT NULL,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      prompt TEXT NOT NULL,
      python_code TEXT,
      agent_options JSONB DEFAULT '{}',
      execution_count INTEGER DEFAULT 0,
      success_count INTEGER DEFAULT 0,
      metadata JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      last_used TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS scheduled_tasks (
      id SERIAL PRIMARY KEY,
      schedule_id VARCHAR(255) UNIQUE NOT NULL,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      task_id VARCHAR(255) NOT NULL,
      name VARCHAR(255) NOT NULL,
      schedule_type VARCHAR(20) NOT NULL CHECK (schedule_type IN ('once','recurring')),
      scheduled_at TIMESTAMP NOT NULL,
      recurring_pattern JSONB,
      status VARCHAR(20) DEFAULT 'active' CHECK (status IN ('active','inactive','completed','failed','running')),
      next_execution TIMESTAMP,
      last_execution TIMESTAMP,
      execution_count INTEGER DEFAULT 0,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (task_id) REFERENCES saved_tasks(task_id) ON DELETE CASCADE
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS task_execution_logs (
      id SERIAL PRIMARY KEY,
      execution_id VARCHAR(255) UNIQUE NOT NULL,
      task_id VARCHAR(255) NOT NULL,
      schedule_id VARCHAR(255),
      user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
      execution_type VARCHAR(20) NOT NULL CHECK (execution_type IN ('manual','scheduled')),
      status VARCHAR(20) NOT NULL CHECK (status IN ('pending','running','completed','failed','cancelled')),
      python_code TEXT,
      stdout TEXT,
      stderr TEXT,
      exit_code INTEGER,
      error_message TEXT,
      agent_job_id VARCHAR(255),
      execution_time_ms INTEGER,
      metadata JSONB DEFAULT '{}',
      started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      completed_at TIMESTAMP,
      FOREIGN KEY (task_id) REFERENCES saved_tasks(task_id) ON DELETE CASCADE,
      FOREIGN KEY (schedule_id) REFERENCES scheduled_tasks(schedule_id) ON DELETE SET NULL
    );
  `
  );
};

/* --------------------------- Section: Web portal ---------------------------- */

const migrateWebPortal = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS support_tickets (
      id SERIAL PRIMARY KEY,
      title VARCHAR(255) NOT NULL,
      question TEXT NOT NULL,
      status VARCHAR(20) NOT NULL DEFAULT 'open',
      author_name VARCHAR(255) NOT NULL,
      author_role VARCHAR(20) NOT NULL,
      attachment_image BYTEA,
      attachment_image_mimetype VARCHAR(100),
      created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS support_ticket_replies (
      id SERIAL PRIMARY KEY,
      ticket_id INTEGER NOT NULL REFERENCES support_tickets(id) ON DELETE CASCADE,
      content TEXT NOT NULL,
      author_name VARCHAR(255) NOT NULL,
      author_role VARCHAR(20) NOT NULL,
      images BYTEA[] DEFAULT '{}',
      image_mimetypes VARCHAR(100)[] DEFAULT '{}',
      created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS support_documents (
      id SERIAL PRIMARY KEY,
      filename VARCHAR(255) NOT NULL,
      filepath TEXT NOT NULL,
      filesize BIGINT,
      mimetype VARCHAR(100),
      user_id VARCHAR(255),
      created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS user_memories (
      id SERIAL PRIMARY KEY,
      user_id VARCHAR(255) NOT NULL,
      title VARCHAR(255) NOT NULL,
      content TEXT NOT NULL,
      category VARCHAR(100) DEFAULT 'general',
      importance_level INTEGER DEFAULT 1 CHECK (importance_level BETWEEN 1 AND 5),
      tags TEXT[],
      is_active BOOLEAN DEFAULT true,
      access_count INTEGER DEFAULT 0,
      created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
      last_accessed TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};

/* ----------------------------- Section: FlowAI ------------------------------ */

const migrateFlowAI = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS flowai_workflows (
      id VARCHAR(255) PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      nodes TEXT NOT NULL DEFAULT '[]',
      connections TEXT NOT NULL DEFAULT '[]',
      variables TEXT DEFAULT '{}',
      user_id VARCHAR(255) NOT NULL,
      is_public BOOLEAN DEFAULT false,
      tags TEXT DEFAULT '[]',
      created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS flowai_executions (
      id SERIAL PRIMARY KEY,
      workflow_id VARCHAR(255) NOT NULL REFERENCES flowai_workflows(id) ON DELETE CASCADE,
      user_id VARCHAR(255) NOT NULL,
      status VARCHAR(20) NOT NULL DEFAULT 'pending',
      results TEXT DEFAULT '[]',
      total_duration INTEGER,
      error_message TEXT,
      final_output TEXT,
      start_time TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
      end_time TIMESTAMP WITH TIME ZONE
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS flowai_template_categories (
      id VARCHAR(255) PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      icon VARCHAR(50) DEFAULT '📁',
      color VARCHAR(7) DEFAULT '#666666',
      order_index INTEGER DEFAULT 999,
      is_active BOOLEAN DEFAULT true,
      created_at TIMESTAMP DEFAULT now(),
      updated_at TIMESTAMP DEFAULT now()
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS flowai_templates (
      id VARCHAR(255) PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      category_id VARCHAR(255) REFERENCES flowai_template_categories(id) ON DELETE SET NULL,
      tags TEXT DEFAULT '[]',
      icon VARCHAR(50) DEFAULT '⚡',
      difficulty VARCHAR(50) DEFAULT 'beginner',
      estimated_time INTEGER DEFAULT 5,
      nodes TEXT DEFAULT '[]',
      connections TEXT DEFAULT '[]',
      variables TEXT DEFAULT '[]',
      preview_image TEXT,
      preview_description TEXT,
      author_name VARCHAR(255),
      author_organization VARCHAR(255),
      version VARCHAR(50) DEFAULT '1.0.0',
      is_public BOOLEAN DEFAULT false,
      usage_count INTEGER DEFAULT 0,
      rating NUMERIC(3,2) DEFAULT 0.00,
      user_id INTEGER,
      created_at TIMESTAMP DEFAULT now(),
      updated_at TIMESTAMP DEFAULT now()
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS flowai_template_reviews (
      id SERIAL PRIMARY KEY,
      template_id VARCHAR(255) NOT NULL REFERENCES flowai_templates(id) ON DELETE CASCADE,
      user_id INTEGER NOT NULL,
      rating INTEGER CHECK (rating >= 1 AND rating <= 5),
      comment TEXT,
      created_at TIMESTAMP DEFAULT now(),
      updated_at TIMESTAMP DEFAULT now()
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS flowai_template_usage (
      id SERIAL PRIMARY KEY,
      template_id VARCHAR(255) NOT NULL REFERENCES flowai_templates(id) ON DELETE CASCADE,
      user_id INTEGER NOT NULL,
      used_at TIMESTAMP DEFAULT now()
    );
  `
  );
};

/* ----------------------- Section: AI QA Feedback --------------------------- */

const migrateQADataset = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS qa_dataset (
      id SERIAL PRIMARY KEY,
      user_question TEXT NOT NULL,
      ai_answer TEXT NOT NULL,
      user_id VARCHAR(100) NOT NULL,
      rating INTEGER NOT NULL CHECK (rating >= 1 AND rating <= 5),
      feedback_type VARCHAR(50) DEFAULT 'thumbs',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};

/* ------------------------- Section: Knowledge Graph ------------------------- */

const migrateGraph = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS graph_entities (
      id SERIAL PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      type VARCHAR(100) NOT NULL,
      description TEXT,
      properties JSONB DEFAULT '{}',
      source_documents TEXT[],
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      UNIQUE(name, type)
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS graph_relationships (
      id SERIAL PRIMARY KEY,
      source_entity_id INTEGER REFERENCES graph_entities(id) ON DELETE CASCADE,
      target_entity_id INTEGER REFERENCES graph_entities(id) ON DELETE CASCADE,
      relationship_type VARCHAR(100) NOT NULL,
      description TEXT,
      weight FLOAT DEFAULT 1.0,
      properties JSONB DEFAULT '{}',
      source_documents TEXT[],
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS graph_communities (
      id SERIAL PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      entities INTEGER[],
      level INTEGER DEFAULT 0,
      properties JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};

/* ------------------- Section: Business Announcements Suite ------------------ */

const migrateBusinessSuite = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS business_announcements (
      id VARCHAR(255) PRIMARY KEY,
      source VARCHAR(50) NOT NULL,
      external_id VARCHAR(255) NOT NULL,
      title TEXT NOT NULL,
      description TEXT,
      category VARCHAR(100),
      subcategory VARCHAR(100),
      budget BIGINT,
      support_amount BIGINT,
      support_ratio FLOAT,
      announcement_date TIMESTAMP,
      application_start TIMESTAMP,
      application_end TIMESTAMP,
      project_start TIMESTAMP,
      project_end TIMESTAMP,
      eligibility JSONB DEFAULT '{}',
      requirements JSONB DEFAULT '{}',
      evaluation JSONB DEFAULT '{}',
      contact JSONB DEFAULT '{}',
      documents JSONB DEFAULT '{}',
      status VARCHAR(20) DEFAULT 'active',
      view_count INTEGER DEFAULT 0,
      raw_data JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS companies (
      id SERIAL PRIMARY KEY,
      business_number VARCHAR(20) UNIQUE NOT NULL,
      company_name VARCHAR(200) NOT NULL,
      company_size VARCHAR(20),
      industry_code VARCHAR(10),
      industry_name VARCHAR(100),
      address TEXT,
      region VARCHAR(50),
      employee_count INTEGER,
      annual_revenue BIGINT,
      establishment_date TIMESTAMP,
      technologies JSONB DEFAULT '{}',
      certifications JSONB DEFAULT '{}',
      capabilities JSONB DEFAULT '{}',
      business_history JSONB DEFAULT '{}',
      financial_info JSONB DEFAULT '{}',
      contact_person VARCHAR(50),
      contact_phone VARCHAR(20),
      contact_email VARCHAR(100),
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS company_announcement_matches (
      id SERIAL PRIMARY KEY,
      company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
      announcement_id VARCHAR(255) REFERENCES business_announcements(id) ON DELETE CASCADE,
      total_score FLOAT NOT NULL,
      category_score FLOAT,
      size_score FLOAT,
      location_score FLOAT,
      technology_score FLOAT,
      experience_score FLOAT,
      match_details JSONB DEFAULT '{}',
      recommendation_status VARCHAR(20) DEFAULT 'pending',
      recommendation_reason TEXT,
      matched_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS business_proposals (
      id SERIAL PRIMARY KEY,
      company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
      announcement_id VARCHAR(255) REFERENCES business_announcements(id) ON DELETE CASCADE,
      proposal_title VARCHAR(500) NOT NULL,
      proposal_summary TEXT,
      sections JSONB DEFAULT '{}',
      generation_method VARCHAR(50),
      template_used VARCHAR(100),
      ai_model_used VARCHAR(100),
      status VARCHAR(20) DEFAULT 'draft',
      version INTEGER DEFAULT 1,
      file_path VARCHAR(500),
      file_format VARCHAR(10),
      file_size INTEGER,
      quality_score FLOAT,
      completeness_score FLOAT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS data_collection_logs (
      id SERIAL PRIMARY KEY,
      source VARCHAR(50) NOT NULL,
      collection_type VARCHAR(20) NOT NULL,
      total_items INTEGER DEFAULT 0,
      new_items INTEGER DEFAULT 0,
      updated_items INTEGER DEFAULT 0,
      failed_items INTEGER DEFAULT 0,
      start_time TIMESTAMP NOT NULL,
      end_time TIMESTAMP,
      duration_seconds INTEGER,
      status VARCHAR(20) NOT NULL,
      error_message TEXT,
      details JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS system_configurations (
      id SERIAL PRIMARY KEY,
      config_key VARCHAR(100) NOT NULL UNIQUE,
      config_value TEXT,
      config_type VARCHAR(20) DEFAULT 'string',
      description TEXT,
      category VARCHAR(50),
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};

/* ------------------------- Section: Context Memory -------------------------- */

const migrateContextMemory = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS context_memories (
      id SERIAL PRIMARY KEY,
      user_id VARCHAR(255) NOT NULL,
      session_id VARCHAR(255),
      memory_type VARCHAR(50) NOT NULL CHECK (memory_type IN ('entity','fact','preference','pattern','skill','project','relationship')),
      content TEXT NOT NULL,
      summary TEXT,
      entities JSONB DEFAULT '[]',
      keywords TEXT[],
      importance_score FLOAT DEFAULT 0.5 CHECK (importance_score >= 0 AND importance_score <= 1),
      confidence_score FLOAT DEFAULT 0.8 CHECK (confidence_score >= 0 AND confidence_score <= 1),
      valid_from TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      valid_until TIMESTAMP,
      source_context TEXT,
      reference_count INTEGER DEFAULT 0,
      last_referenced TIMESTAMP,
      embedding vector(${VECTOR_DIM}),
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureConstraint(
    db,
    'unique_user_memory_content',
    `
    ALTER TABLE context_memories
    ADD CONSTRAINT unique_user_memory_content UNIQUE (user_id, memory_type, content);
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS user_preferences (
      id SERIAL PRIMARY KEY,
      user_id VARCHAR(255) NOT NULL,
      category VARCHAR(100) NOT NULL,
      preference_key VARCHAR(255) NOT NULL,
      preference_value TEXT NOT NULL,
      strength FLOAT DEFAULT 0.5 CHECK (strength >= 0 AND strength <= 1),
      confidence FLOAT DEFAULT 0.8 CHECK (confidence >= 0 AND confidence <= 1),
      learned_from VARCHAR(50) DEFAULT 'conversation',
      evidence_count INTEGER DEFAULT 1,
      first_observed TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      last_confirmed TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      notes TEXT,
      tags TEXT[],
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      UNIQUE(user_id, category, preference_key)
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS context_relationships (
      id SERIAL PRIMARY KEY,
      user_id VARCHAR(255) NOT NULL,
      source_memory_id INTEGER REFERENCES context_memories(id) ON DELETE CASCADE,
      target_memory_id INTEGER REFERENCES context_memories(id) ON DELETE CASCADE,
      relationship_type VARCHAR(50) NOT NULL,
      strength FLOAT DEFAULT 0.5 CHECK (strength >= 0 AND strength <= 1),
      description TEXT,
      evidence TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS context_session_summaries (
      id SERIAL PRIMARY KEY,
      session_id VARCHAR(255) UNIQUE NOT NULL,
      user_id VARCHAR(255) NOT NULL,
      title VARCHAR(255),
      summary TEXT,
      key_topics TEXT[],
      mentioned_entities TEXT[],
      message_count INTEGER DEFAULT 0,
      session_type VARCHAR(50),
      provider VARCHAR(50),
      model VARCHAR(100),
      importance_score FLOAT DEFAULT 0.5,
      engagement_score FLOAT DEFAULT 0.5,
      session_start TIMESTAMP,
      session_end TIMESTAMP,
      duration_minutes INTEGER,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};

/* ---------------------------- Section: Webhooks ----------------------------- */

const migrateWebhooks = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS webhooks (
      id SERIAL PRIMARY KEY,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      url VARCHAR(500) NOT NULL,
      events TEXT[] NOT NULL,
      secret VARCHAR(255),
      is_active BOOLEAN DEFAULT true,
      retry_count INTEGER DEFAULT 3,
      timeout_seconds INTEGER DEFAULT 30,
      headers JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS webhook_logs (
      id SERIAL PRIMARY KEY,
      webhook_id INTEGER REFERENCES webhooks(id) ON DELETE CASCADE,
      event_type VARCHAR(100) NOT NULL,
      payload JSONB,
      response_status INTEGER,
      response_body TEXT,
      response_headers JSONB,
      attempts INTEGER DEFAULT 1,
      delivered_at TIMESTAMP,
      error_message TEXT,
      execution_time_ms INTEGER,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};

/* ----------------------- Section: Health Batch System ----------------------- */

const migrateHealthBatch = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS health_batch_jobs (
      id VARCHAR(255) PRIMARY KEY,
      user_id VARCHAR(255) NOT NULL,
      filename VARCHAR(255) NOT NULL,
      status VARCHAR(20) DEFAULT 'pending' CHECK (status IN ('pending','processing','completed','failed')),
      total_records INTEGER DEFAULT 0,
      processed_records INTEGER DEFAULT 0,
      successful_records INTEGER DEFAULT 0,
      failed_records INTEGER DEFAULT 0,
      progress FLOAT DEFAULT 0,
      current_patient VARCHAR(255),
      error_message TEXT,
      metadata JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      completed_at TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS health_batch_patients (
      id SERIAL PRIMARY KEY,
      job_id VARCHAR(255) NOT NULL REFERENCES health_batch_jobs(id) ON DELETE CASCADE,
      patient_name VARCHAR(255) NOT NULL,
      row_number INTEGER NOT NULL,
      raw_data JSONB DEFAULT '{}',
      blood_pressure_score INTEGER DEFAULT 3,
      cholesterol_score INTEGER DEFAULT 3,
      hba1c_received INTEGER DEFAULT 3,
      blood_sugar_score INTEGER DEFAULT 3,
      obesity_score INTEGER DEFAULT 3,
      urine_protein_score INTEGER DEFAULT 1,
      past_history_treatment TEXT,
      current_history_treatment TEXT,
      final_judgment INTEGER DEFAULT 3,
      analysis_status VARCHAR(20) DEFAULT 'pending' CHECK (analysis_status IN ('pending','processing','completed','failed')),
      analysis_error TEXT,
      analyzed_at TIMESTAMP,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};

/* ----------------------- Section: LLM Fine-tuning --------------------------- */

const migrateFineTuning = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS finetuning_datasets (
      id VARCHAR(255) PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      source VARCHAR(50) NOT NULL,
      file_path TEXT,
      total_samples INTEGER DEFAULT 0,
      status VARCHAR(20) DEFAULT 'pending' CHECK (status IN ('pending','processing','completed','failed')),
      metadata JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS finetuned_models (
      id VARCHAR(255) PRIMARY KEY,
      base_model VARCHAR(255) NOT NULL,
      dataset_id VARCHAR(255) REFERENCES finetuning_datasets(id) ON DELETE SET NULL,
      status VARCHAR(50) NOT NULL CHECK (status IN ('training','completed','failed','cancelled')),
      output_path TEXT,
      training_config JSONB DEFAULT '{}',
      metrics JSONB DEFAULT '{}',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS finetuning_jobs (
      id SERIAL PRIMARY KEY,
      model_id VARCHAR(255) REFERENCES finetuned_models(id) ON DELETE CASCADE,
      status VARCHAR(20) DEFAULT 'queued' CHECK (status IN ('queued','running','completed','failed','cancelled')),
      progress INTEGER DEFAULT 0,
      logs TEXT,
      error_message TEXT,
      started_at TIMESTAMP,
      completed_at TIMESTAMP,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );
};


/* ----------------------- Section: ESG Analysis ---------------------------- */

const migrateESGAnalysis = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS esg_batch_jobs (
      id VARCHAR(36) PRIMARY KEY,
      user_id VARCHAR(50) NOT NULL,
      filename VARCHAR(255) NOT NULL,
      status VARCHAR(20) DEFAULT 'pending' CHECK (status IN ('pending','processing','completed','failed')),
      total_records INTEGER DEFAULT 0,
      processed_records INTEGER DEFAULT 0,
      successful_records INTEGER DEFAULT 0,
      failed_records INTEGER DEFAULT 0,
      progress INTEGER DEFAULT 0,
      error_message TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      completed_at TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS esg_batch_documents (
      id SERIAL PRIMARY KEY,
      job_id VARCHAR(36) REFERENCES esg_batch_jobs(id) ON DELETE CASCADE,
      document_name VARCHAR(255),
      row_number INTEGER,
      raw_content TEXT,
      industry_major_code VARCHAR(10),
      industry_major_name VARCHAR(255),
      industry_sub_code VARCHAR(10),
      industry_sub_name VARCHAR(255),
      industry_confidence DECIMAL(5,2),
      estimated_revenue DECIMAL(15,2),
      revenue_currency VARCHAR(10) DEFAULT 'KRW',
      carbon_intensity DECIMAL(10,4),
      total_carbon_emission DECIMAL(15,4),
      emission_unit VARCHAR(20) DEFAULT 'tCO2e',
      analysis_status VARCHAR(20) DEFAULT 'pending' CHECK (analysis_status IN ('pending','processing','completed','failed')),
      analysis_error TEXT,
      analyzed_at TIMESTAMP,
      report_content TEXT,
      report_format VARCHAR(20) DEFAULT 'pdf',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  // Ensure report columns exist for HTML report storage
  if (await tableExists(db, 'esg_batch_documents')) {
    await ensureColumn(db, 'esg_batch_documents', 'report_content', 'report_content TEXT');
    await ensureColumn(db, 'esg_batch_documents', 'report_format', "report_format VARCHAR(20) DEFAULT 'html'");
    await ensureColumn(db, 'esg_batch_documents', 'total_carbon_emission', 'total_carbon_emission FLOAT');
  }

  // Industry classifications table for ESG carbon intensity reference
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS esg_industry_classifications (
      id SERIAL PRIMARY KEY,
      code VARCHAR(10) NOT NULL UNIQUE,
      major_category VARCHAR(200) NOT NULL,
      sub_categories JSONB DEFAULT '[]',
      carbon_intensity FLOAT NOT NULL,
      description TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  // Indexes for ESG tables
  await ensureIndex(db, 'idx_esg_jobs_user', `CREATE INDEX IF NOT EXISTS idx_esg_batch_jobs_user_id ON esg_batch_jobs(user_id);`);
  await ensureIndex(db, 'idx_esg_jobs_status', `CREATE INDEX IF NOT EXISTS idx_esg_batch_jobs_status ON esg_batch_jobs(status);`);
  await ensureIndex(db, 'idx_esg_jobs_created', `CREATE INDEX IF NOT EXISTS idx_esg_batch_jobs_created_at ON esg_batch_jobs(created_at);`);
  await ensureIndex(db, 'idx_esg_docs_job', `CREATE INDEX IF NOT EXISTS idx_esg_batch_documents_job_id ON esg_batch_documents(job_id);`);
  await ensureIndex(db, 'idx_esg_docs_status', `CREATE INDEX IF NOT EXISTS idx_esg_batch_documents_analysis_status ON esg_batch_documents(analysis_status);`);
  await ensureIndex(db, 'idx_esg_industry_code', `CREATE INDEX IF NOT EXISTS idx_esg_industry_classifications_code ON esg_industry_classifications(code);`);
};

/* ------------------------- Section: RPA Interface --------------------------- */

const migrateRPAJobs = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS rpa_jobs (
      id SERIAL PRIMARY KEY,
      request_id VARCHAR(255) UNIQUE NOT NULL,
      status VARCHAR(20) DEFAULT 'QUEUED' CHECK (status IN ('QUEUED','IN_PROGRESS','SUCCEEDED','FAILED')),
      application_type VARCHAR(100) NOT NULL,
      applicant_name VARCHAR(255) NOT NULL,
      applicant_info JSONB DEFAULT '{}',
      applicant_email VARCHAR(255),
      applicant_phone VARCHAR(50),
      applicant_address TEXT,
      application_data JSONB DEFAULT '{}',
      additional_notes TEXT,
      payload_hash VARCHAR(255),
      schema_version VARCHAR(20) DEFAULT '1.0',
      correlation_id VARCHAR(255),
      user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
      username VARCHAR(100),
      dici_key VARCHAR(255),
      error_message TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      completed_at TIMESTAMP
    );
  `
  );

  // v3.5.5: dici_key 컬럼 추가 (강진군 빈집 신청서용)
  await ensureColumn(db, 'rpa_jobs', 'dici_key', 'dici_key VARCHAR(255)');

  // Indexes for rpa_jobs
  await ensureIndex(db, 'idx_rpa_jobs_request_id', `CREATE INDEX IF NOT EXISTS idx_rpa_jobs_request_id ON rpa_jobs(request_id);`);
  await ensureIndex(db, 'idx_rpa_jobs_status', `CREATE INDEX IF NOT EXISTS idx_rpa_jobs_status ON rpa_jobs(status);`);
  await ensureIndex(db, 'idx_rpa_jobs_user_id', `CREATE INDEX IF NOT EXISTS idx_rpa_jobs_user_id ON rpa_jobs(user_id);`);
  await ensureIndex(db, 'idx_rpa_jobs_created_at', `CREATE INDEX IF NOT EXISTS idx_rpa_jobs_created_at ON rpa_jobs(created_at);`);
  await ensureIndex(db, 'idx_rpa_jobs_correlation_id', `CREATE INDEX IF NOT EXISTS idx_rpa_jobs_correlation_id ON rpa_jobs(correlation_id);`);
  await ensureIndex(db, 'idx_rpa_jobs_dici_key', `CREATE INDEX IF NOT EXISTS idx_rpa_jobs_dici_key ON rpa_jobs(dici_key);`);

  // v3.5.6: rpa_attachments 테이블 추가 (강진군 빈집 신청서 첨부서류)
  await db.query(
    `
    CREATE TABLE IF NOT EXISTS rpa_attachments (
      id SERIAL PRIMARY KEY,
      rpa_job_id INTEGER NOT NULL REFERENCES rpa_jobs(id) ON DELETE CASCADE,
      file_name VARCHAR(255) NOT NULL,
      original_name VARCHAR(255) NOT NULL,
      file_path TEXT NOT NULL,
      file_size BIGINT NOT NULL,
      file_type VARCHAR(100),
      file_hash VARCHAR(255),
      description TEXT,
      uploaded_by INTEGER REFERENCES users(id) ON DELETE SET NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  // Indexes for rpa_attachments
  await ensureIndex(db, 'idx_rpa_attachments_job_id', `CREATE INDEX IF NOT EXISTS idx_rpa_attachments_job_id ON rpa_attachments(rpa_job_id);`);
  await ensureIndex(db, 'idx_rpa_attachments_uploaded_by', `CREATE INDEX IF NOT EXISTS idx_rpa_attachments_uploaded_by ON rpa_attachments(uploaded_by);`);
  await ensureIndex(db, 'idx_rpa_attachments_created_at', `CREATE INDEX IF NOT EXISTS idx_rpa_attachments_created_at ON rpa_attachments(created_at);`);
};

/* --------------------------- Section: AI Form Builder ----------------------- */

const migrateAIFormBuilder = async (db) => {
  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS ai_forms (
      id VARCHAR(255) PRIMARY KEY,
      user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
      username VARCHAR(100),
      title VARCHAR(500) NOT NULL,
      description TEXT,
      document_id VARCHAR(255),
      document_filename VARCHAR(255),
      document_url TEXT,
      analyzed_content TEXT,
      form_schema JSONB NOT NULL DEFAULT '[]',
      share_link VARCHAR(255) UNIQUE,
      view_link VARCHAR(255) UNIQUE,
      status VARCHAR(20) DEFAULT 'active' CHECK (status IN ('draft','active','archived')),
      response_count INTEGER DEFAULT 0,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS ai_form_responses (
      id SERIAL PRIMARY KEY,
      form_id VARCHAR(255) REFERENCES ai_forms(id) ON DELETE CASCADE,
      response_data JSONB NOT NULL,
      submitted_by VARCHAR(255),
      submitted_ip VARCHAR(45),
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  await ensureTable(
    db,
    `
    CREATE TABLE IF NOT EXISTS ai_form_response_attachments (
      id SERIAL PRIMARY KEY,
      response_id INTEGER NOT NULL REFERENCES ai_form_responses(id) ON DELETE CASCADE,
      filename VARCHAR(255) NOT NULL,
      original_filename VARCHAR(255) NOT NULL,
      file_path VARCHAR(500) NOT NULL,
      file_size INTEGER NOT NULL,
      mime_type VARCHAR(100),
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `
  );

  // Indexes for ai_forms
  await ensureIndex(db, 'idx_ai_forms_user_id', `CREATE INDEX IF NOT EXISTS idx_ai_forms_user_id ON ai_forms(user_id);`);
  await ensureIndex(db, 'idx_ai_forms_username', `CREATE INDEX IF NOT EXISTS idx_ai_forms_username ON ai_forms(username);`);
  await ensureIndex(db, 'idx_ai_forms_status', `CREATE INDEX IF NOT EXISTS idx_ai_forms_status ON ai_forms(status);`);
  await ensureIndex(db, 'idx_ai_forms_share_link', `CREATE INDEX IF NOT EXISTS idx_ai_forms_share_link ON ai_forms(share_link);`);
  await ensureIndex(db, 'idx_ai_forms_view_link', `CREATE INDEX IF NOT EXISTS idx_ai_forms_view_link ON ai_forms(view_link);`);
  await ensureIndex(db, 'idx_ai_forms_created_at', `CREATE INDEX IF NOT EXISTS idx_ai_forms_created_at ON ai_forms(created_at);`);

  // Indexes for ai_form_responses
  await ensureIndex(db, 'idx_ai_form_responses_form_id', `CREATE INDEX IF NOT EXISTS idx_ai_form_responses_form_id ON ai_form_responses(form_id);`);
  await ensureIndex(db, 'idx_ai_form_responses_created_at', `CREATE INDEX IF NOT EXISTS idx_ai_form_responses_created_at ON ai_form_responses(created_at);`);

  // Indexes for ai_form_response_attachments
  await ensureIndex(db, 'idx_ai_form_attachments_response_id', `CREATE INDEX IF NOT EXISTS idx_ai_form_attachments_response_id ON ai_form_response_attachments(response_id);`);
  await ensureIndex(db, 'idx_ai_form_attachments_created_at', `CREATE INDEX IF NOT EXISTS idx_ai_form_attachments_created_at ON ai_form_response_attachments(created_at);`);
};

/* ------------------------------ Indexes (core) ------------------------------ */

const createCoreIndexes = async (db) => {
  // users & auth
  await ensureIndex(db, 'idx_users_username', `CREATE INDEX IF NOT EXISTS idx_users_username ON users(username);`);
  await ensureIndex(db, 'idx_users_email', `CREATE INDEX IF NOT EXISTS idx_users_email ON users(email);`);
  await ensureIndex(db, 'idx_users_role', `CREATE INDEX IF NOT EXISTS idx_users_role ON users(role);`);
  await ensureIndex(db, 'idx_users_status', `CREATE INDEX IF NOT EXISTS idx_users_status ON users(status);`);
  await ensureIndex(db, 'idx_api_keys_key_value', `CREATE INDEX IF NOT EXISTS idx_api_keys_key_value ON api_keys(key_value);`);
  await ensureIndex(db, 'idx_api_keys_user_id', `CREATE INDEX IF NOT EXISTS idx_api_keys_user_id ON api_keys(user_id);`);
  await ensureIndex(db, 'idx_activity_logs_user_id', `CREATE INDEX IF NOT EXISTS idx_activity_logs_user_id ON activity_logs(user_id);`);
  await ensureIndex(db, 'idx_activity_logs_created_at', `CREATE INDEX IF NOT EXISTS idx_activity_logs_created_at ON activity_logs(created_at);`);
  await ensureIndex(db, 'idx_user_ip_logs_user_id', `CREATE INDEX IF NOT EXISTS idx_user_ip_logs_user_id ON user_ip_logs(user_id);`);

  // sessions
  await ensureIndex(db, 'idx_sessions_user_id', `CREATE INDEX IF NOT EXISTS idx_sessions_user_id ON sessions(user_id);`);
  await ensureIndex(db, 'idx_sessions_username', `CREATE INDEX IF NOT EXISTS idx_sessions_username ON sessions(username);`);
  await ensureIndex(db, 'idx_sessions_created_at', `CREATE INDEX IF NOT EXISTS idx_sessions_created_at ON sessions(created_at);`);
  await ensureIndex(db, 'idx_sessions_updated_at', `CREATE INDEX IF NOT EXISTS idx_sessions_updated_at ON sessions(updated_at);`);
  await ensureIndex(db, 'idx_sessions_type', `CREATE INDEX IF NOT EXISTS idx_sessions_type ON sessions(type);`);
  await ensureIndex(db, 'idx_sessions_status', `CREATE INDEX IF NOT EXISTS idx_sessions_status ON sessions(status);`);

  // document_embeddings
  await ensureIndex(db, 'idx_de_doc_id', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_doc_id ON document_embeddings(doc_id);`);
  await ensureIndex(db, 'idx_de_filename', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_filename ON document_embeddings(filename);`);
  await ensureIndex(db, 'idx_de_chunk_index', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_chunk_index ON document_embeddings(chunk_index);`);
  await ensureIndex(db, 'idx_de_user_id', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_user_id ON document_embeddings(user_id);`);
  await ensureIndex(db, 'idx_de_source', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_source ON document_embeddings(source);`);
  await ensureIndex(db, 'idx_de_created_at', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_created_at ON document_embeddings(created_at);`);
  await ensureIndex(db, 'idx_de_has_img', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_has_image ON document_embeddings((image_embedding IS NOT NULL));`);
  await ensureIndex(db, 'idx_de_is_embedding', `CREATE INDEX IF NOT EXISTS idx_document_embeddings_is_embedding ON document_embeddings(is_embedding);`);

  // ANN indexes
  await ensureIndex(
    db,
    'idx_de_vec_cos',
    `CREATE INDEX IF NOT EXISTS idx_document_embeddings_vector_cosine ON document_embeddings USING ivfflat (embedding vector_cosine_ops) WITH (lists = 100);`
  );
  await ensureIndex(
    db,
    'idx_de_img_vec_cos',
    `CREATE INDEX IF NOT EXISTS idx_document_embeddings_image_vector_cosine ON document_embeddings USING ivfflat (image_embedding vector_cosine_ops) WITH (lists = 100);`
  );

  // chat_documents
  await ensureIndex(db, 'idx_chat_upload', `CREATE INDEX IF NOT EXISTS idx_chat_documents_upload_status ON chat_documents(upload_status);`);
  await ensureIndex(db, 'idx_chat_embed', `CREATE INDEX IF NOT EXISTS idx_chat_documents_embedding_status ON chat_documents(embedding_status);`);
  await ensureIndex(db, 'idx_chat_proj', `CREATE INDEX IF NOT EXISTS idx_chat_documents_project_name ON chat_documents(project_name);`);
  await ensureIndex(db, 'idx_chat_processed_at', `CREATE INDEX IF NOT EXISTS idx_chat_documents_processed_at ON chat_documents(processed_at);`);
  await ensureIndex(db, 'idx_chat_user', `CREATE INDEX IF NOT EXISTS idx_chat_documents_user_id ON chat_documents(user_id);`);
  await ensureIndex(db, 'idx_chat_created', `CREATE INDEX IF NOT EXISTS idx_chat_documents_created_at ON chat_documents(created_at);`);

  // reports
  await ensureIndex(db, 'idx_report_jobs_user', `CREATE INDEX IF NOT EXISTS idx_report_jobs_user_id ON report_jobs(user_id);`);
  await ensureIndex(db, 'idx_report_jobs_status', `CREATE INDEX IF NOT EXISTS idx_report_jobs_status ON report_jobs(status);`);
  await ensureIndex(db, 'idx_report_jobs_created', `CREATE INDEX IF NOT EXISTS idx_report_jobs_created_at ON report_jobs(created_at);`);
  await ensureIndex(db, 'idx_report_cache_job', `CREATE INDEX IF NOT EXISTS idx_report_section_cache_job_id ON report_section_cache(job_id);`);
  await ensureIndex(db, 'idx_report_cache_exp', `CREATE INDEX IF NOT EXISTS idx_report_section_cache_expires_at ON report_section_cache(expires_at);`);
  await ensureIndex(db, 'idx_user_report_settings_user', `CREATE INDEX IF NOT EXISTS idx_user_report_settings_user_id ON user_report_settings(user_id);`);

  // scheduler
  await ensureIndex(db, 'idx_saved_tasks_user', `CREATE INDEX IF NOT EXISTS idx_saved_tasks_user_id ON saved_tasks(user_id);`);
  await ensureIndex(db, 'idx_saved_tasks_task', `CREATE INDEX IF NOT EXISTS idx_saved_tasks_task_id ON saved_tasks(task_id);`);
  await ensureIndex(db, 'idx_saved_tasks_last', `CREATE INDEX IF NOT EXISTS idx_saved_tasks_last_used ON saved_tasks(last_used);`);
  await ensureIndex(db, 'idx_sched_tasks_user', `CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_user_id ON scheduled_tasks(user_id);`);
  await ensureIndex(db, 'idx_sched_tasks_task', `CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_task_id ON scheduled_tasks(task_id);`);
  await ensureIndex(db, 'idx_sched_tasks_status', `CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_status ON scheduled_tasks(status);`);
  await ensureIndex(db, 'idx_sched_tasks_next', `CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_next_execution ON scheduled_tasks(next_execution);`);
  await ensureIndex(db, 'idx_exec_logs_task', `CREATE INDEX IF NOT EXISTS idx_task_execution_logs_task_id ON task_execution_logs(task_id);`);
  await ensureIndex(db, 'idx_exec_logs_sched', `CREATE INDEX IF NOT EXISTS idx_task_execution_logs_schedule_id ON task_execution_logs(schedule_id);`);
  await ensureIndex(db, 'idx_exec_logs_user', `CREATE INDEX IF NOT EXISTS idx_task_execution_logs_user_id ON task_execution_logs(user_id);`);
  await ensureIndex(db, 'idx_exec_logs_status', `CREATE INDEX IF NOT EXISTS idx_task_execution_logs_status ON task_execution_logs(status);`);
  await ensureIndex(db, 'idx_exec_logs_started', `CREATE INDEX IF NOT EXISTS idx_task_execution_logs_started_at ON task_execution_logs(started_at);`);

  // web portal
  await ensureIndex(db, 'idx_support_tickets_status', `CREATE INDEX IF NOT EXISTS idx_support_tickets_status ON support_tickets(status);`);
  await ensureIndex(db, 'idx_support_tickets_created', `CREATE INDEX IF NOT EXISTS idx_support_tickets_created_at ON support_tickets(created_at);`);
  await ensureIndex(db, 'idx_support_replies_ticket', `CREATE INDEX IF NOT EXISTS idx_support_ticket_replies_ticket_id ON support_ticket_replies(ticket_id);`);
  await ensureIndex(db, 'idx_support_docs_user', `CREATE INDEX IF NOT EXISTS idx_support_documents_user_id ON support_documents(user_id);`);
  await ensureIndex(db, 'idx_support_docs_created', `CREATE INDEX IF NOT EXISTS idx_support_documents_created_at ON support_documents(created_at);`);
  await ensureIndex(db, 'idx_user_memories_user', `CREATE INDEX IF NOT EXISTS idx_user_memories_user_id ON user_memories(user_id);`);
  await ensureIndex(db, 'idx_user_memories_category', `CREATE INDEX IF NOT EXISTS idx_user_memories_category ON user_memories(category);`);
  await ensureIndex(db, 'idx_user_memories_importance', `CREATE INDEX IF NOT EXISTS idx_user_memories_importance ON user_memories(importance_level);`);
  await ensureIndex(db, 'idx_user_memories_active', `CREATE INDEX IF NOT EXISTS idx_user_memories_active ON user_memories(is_active);`);

  // flowAI
  await ensureIndex(db, 'idx_flowai_workflows_user', `CREATE INDEX IF NOT EXISTS idx_flowai_workflows_user_id ON flowai_workflows(user_id);`);
  await ensureIndex(db, 'idx_flowai_workflows_public', `CREATE INDEX IF NOT EXISTS idx_flowai_workflows_public ON flowai_workflows(is_public);`);
  await ensureIndex(db, 'idx_flowai_workflows_updated', `CREATE INDEX IF NOT EXISTS idx_flowai_workflows_updated ON flowai_workflows(updated_at);`);
  await ensureIndex(db, 'idx_flowai_exec_workflow', `CREATE INDEX IF NOT EXISTS idx_flowai_executions_workflow ON flowai_executions(workflow_id);`);
  await ensureIndex(db, 'idx_flowai_exec_user', `CREATE INDEX IF NOT EXISTS idx_flowai_executions_user ON flowai_executions(user_id);`);
  await ensureIndex(db, 'idx_flowai_exec_status', `CREATE INDEX IF NOT EXISTS idx_flowai_executions_status ON flowai_executions(status);`);
  await ensureIndex(db, 'idx_flowai_cat_active', `CREATE INDEX IF NOT EXISTS idx_flowai_template_categories_is_active ON flowai_template_categories(is_active);`);
  await ensureIndex(db, 'idx_flowai_cat_order', `CREATE INDEX IF NOT EXISTS idx_flowai_template_categories_order_index ON flowai_template_categories(order_index);`);
  await ensureIndex(db, 'idx_flowai_tpl_cat', `CREATE INDEX IF NOT EXISTS idx_flowai_templates_category_id ON flowai_templates(category_id);`);
  await ensureIndex(db, 'idx_flowai_tpl_public', `CREATE INDEX IF NOT EXISTS idx_flowai_templates_is_public ON flowai_templates(is_public);`);
  await ensureIndex(db, 'idx_flowai_tpl_user', `CREATE INDEX IF NOT EXISTS idx_flowai_templates_user_id ON flowai_templates(user_id);`);
  await ensureIndex(db, 'idx_flowai_tpl_rating', `CREATE INDEX IF NOT EXISTS idx_flowai_templates_rating ON flowai_templates(rating);`);
  await ensureIndex(db, 'idx_flowai_tpl_usage', `CREATE INDEX IF NOT EXISTS idx_flowai_templates_usage_count ON flowai_templates(usage_count);`);
  await ensureIndex(db, 'idx_flowai_tpl_created', `CREATE INDEX IF NOT EXISTS idx_flowai_templates_created_at ON flowai_templates(created_at);`);
  await ensureIndex(db, 'idx_flowai_review_tpl', `CREATE INDEX IF NOT EXISTS idx_flowai_template_reviews_template_id ON flowai_template_reviews(template_id);`);
  await ensureIndex(db, 'idx_flowai_review_user', `CREATE INDEX IF NOT EXISTS idx_flowai_template_reviews_user_id ON flowai_template_reviews(user_id);`);
  await ensureIndex(db, 'idx_flowai_review_rating', `CREATE INDEX IF NOT EXISTS idx_flowai_template_reviews_rating ON flowai_template_reviews(rating);`);
  await ensureIndex(db, 'idx_flowai_usage_tpl', `CREATE INDEX IF NOT EXISTS idx_flowai_template_usage_template_id ON flowai_template_usage(template_id);`);
  await ensureIndex(db, 'idx_flowai_usage_user', `CREATE INDEX IF NOT EXISTS idx_flowai_template_usage_user_id ON flowai_template_usage(user_id);`);
  await ensureIndex(db, 'idx_flowai_usage_used', `CREATE INDEX IF NOT EXISTS idx_flowai_template_usage_used_at ON flowai_template_usage(used_at);`);

  // QA dataset
  await ensureIndex(db, 'idx_qa_rating', `CREATE INDEX IF NOT EXISTS idx_qa_dataset_rating ON qa_dataset(rating);`);
  await ensureIndex(db, 'idx_qa_user', `CREATE INDEX IF NOT EXISTS idx_qa_dataset_user_id ON qa_dataset(user_id);`);
  await ensureIndex(db, 'idx_qa_created', `CREATE INDEX IF NOT EXISTS idx_qa_dataset_created_at ON qa_dataset(created_at);`);

  // graph
  await ensureIndex(db, 'idx_ge_name', `CREATE INDEX IF NOT EXISTS idx_graph_entities_name ON graph_entities(name);`);
  await ensureIndex(db, 'idx_ge_type', `CREATE INDEX IF NOT EXISTS idx_graph_entities_type ON graph_entities(type);`);
  await ensureIndex(db, 'idx_ge_created', `CREATE INDEX IF NOT EXISTS idx_graph_entities_created_at ON graph_entities(created_at);`);
  await ensureIndex(db, 'idx_gr_source', `CREATE INDEX IF NOT EXISTS idx_graph_relationships_source ON graph_relationships(source_entity_id);`);
  await ensureIndex(db, 'idx_gr_target', `CREATE INDEX IF NOT EXISTS idx_graph_relationships_target ON graph_relationships(target_entity_id);`);
  await ensureIndex(db, 'idx_gr_type', `CREATE INDEX IF NOT EXISTS idx_graph_relationships_type ON graph_relationships(relationship_type);`);
  await ensureIndex(db, 'idx_gr_weight', `CREATE INDEX IF NOT EXISTS idx_graph_relationships_weight ON graph_relationships(weight);`);
  await ensureIndex(db, 'idx_gc_name', `CREATE INDEX IF NOT EXISTS idx_graph_communities_name ON graph_communities(name);`);
  await ensureIndex(db, 'idx_gc_level', `CREATE INDEX IF NOT EXISTS idx_graph_communities_level ON graph_communities(level);`);

  // business suite
  await ensureIndex(db, 'idx_ba_source_ext', `CREATE INDEX IF NOT EXISTS idx_announcement_source_external ON business_announcements(source, external_id);`);
  await ensureIndex(db, 'idx_ba_category', `CREATE INDEX IF NOT EXISTS idx_announcement_category ON business_announcements(category);`);
  await ensureIndex(db, 'idx_ba_deadline', `CREATE INDEX IF NOT EXISTS idx_announcement_deadline ON business_announcements(application_end);`);
  await ensureIndex(db, 'idx_ba_status', `CREATE INDEX IF NOT EXISTS idx_announcement_status ON business_announcements(status);`);
  await ensureIndex(db, 'idx_ba_created', `CREATE INDEX IF NOT EXISTS idx_announcement_created_at ON business_announcements(created_at);`);
  await ensureIndex(db, 'idx_comp_bn', `CREATE INDEX IF NOT EXISTS idx_company_business_number ON companies(business_number);`);
  await ensureIndex(db, 'idx_comp_size_industry', `CREATE INDEX IF NOT EXISTS idx_company_size_industry ON companies(company_size, industry_code);`);
  await ensureIndex(db, 'idx_comp_region', `CREATE INDEX IF NOT EXISTS idx_company_region ON companies(region);`);
  await ensureIndex(db, 'idx_comp_created', `CREATE INDEX IF NOT EXISTS idx_company_created_at ON companies(created_at);`);
  await ensureIndex(db, 'idx_match_company_score', `CREATE INDEX IF NOT EXISTS idx_match_company_score ON company_announcement_matches(company_id, total_score);`);
  await ensureIndex(db, 'idx_match_announcement_score', `CREATE INDEX IF NOT EXISTS idx_match_announcement_score ON company_announcement_matches(announcement_id, total_score);`);
  await ensureIndex(db, 'idx_match_status', `CREATE INDEX IF NOT EXISTS idx_match_status ON company_announcement_matches(recommendation_status);`);
  await ensureIndex(db, 'idx_match_time', `CREATE INDEX IF NOT EXISTS idx_match_matched_at ON company_announcement_matches(matched_at);`);
  await ensureIndex(db, 'idx_bp_company_status', `CREATE INDEX IF NOT EXISTS idx_proposal_company_status ON business_proposals(company_id, status);`);
  await ensureIndex(db, 'idx_bp_announcement', `CREATE INDEX IF NOT EXISTS idx_proposal_announcement ON business_proposals(announcement_id);`);
  await ensureIndex(db, 'idx_bp_created', `CREATE INDEX IF NOT EXISTS idx_proposal_created_at ON business_proposals(created_at);`);
  await ensureIndex(db, 'idx_bp_quality', `CREATE INDEX IF NOT EXISTS idx_proposal_quality_score ON business_proposals(quality_score);`);
  await ensureIndex(db, 'idx_dcl_source_time', `CREATE INDEX IF NOT EXISTS idx_collection_log_source_time ON data_collection_logs(source, start_time);`);
  await ensureIndex(db, 'idx_dcl_status', `CREATE INDEX IF NOT EXISTS idx_collection_log_status ON data_collection_logs(status);`);
  await ensureIndex(db, 'idx_dcl_created', `CREATE INDEX IF NOT EXISTS idx_collection_log_created_at ON data_collection_logs(created_at);`);
  await ensureIndex(db, 'idx_sysconf_key', `CREATE INDEX IF NOT EXISTS idx_system_config_key ON system_configurations(config_key);`);
  await ensureIndex(db, 'idx_sysconf_category', `CREATE INDEX IF NOT EXISTS idx_system_config_category ON system_configurations(category);`);

  // context memory
  await ensureIndex(db, 'idx_cm_user', `CREATE INDEX IF NOT EXISTS idx_context_memories_user_id ON context_memories(user_id);`);
  await ensureIndex(db, 'idx_cm_type', `CREATE INDEX IF NOT EXISTS idx_context_memories_type ON context_memories(memory_type);`);
  await ensureIndex(db, 'idx_cm_importance', `CREATE INDEX IF NOT EXISTS idx_context_memories_importance ON context_memories(importance_score);`);
  await ensureIndex(db, 'idx_cm_valid', `CREATE INDEX IF NOT EXISTS idx_context_memories_valid_period ON context_memories(valid_from, valid_until);`);
  await ensureIndex(db, 'idx_cm_keywords', `CREATE INDEX IF NOT EXISTS idx_context_memories_keywords ON context_memories USING GIN(keywords);`);
  await ensureIndex(db, 'idx_cm_entities', `CREATE INDEX IF NOT EXISTS idx_context_memories_entities ON context_memories USING GIN(entities);`);
  await ensureIndex(db, 'idx_cm_created', `CREATE INDEX IF NOT EXISTS idx_context_memories_created_at ON context_memories(created_at);`);
  await ensureIndex(db, 'idx_cm_refcount', `CREATE INDEX IF NOT EXISTS idx_context_memories_reference_count ON context_memories(reference_count);`);
  await ensureIndex(db, 'idx_up_user', `CREATE INDEX IF NOT EXISTS idx_user_preferences_user_id ON user_preferences(user_id);`);
  await ensureIndex(db, 'idx_up_category', `CREATE INDEX IF NOT EXISTS idx_user_preferences_category ON user_preferences(category);`);
  await ensureIndex(db, 'idx_up_strength', `CREATE INDEX IF NOT EXISTS idx_user_preferences_strength ON user_preferences(strength);`);
  await ensureIndex(db, 'idx_up_last', `CREATE INDEX IF NOT EXISTS idx_user_preferences_last_confirmed ON user_preferences(last_confirmed);`);
  await ensureIndex(db, 'idx_cr_user', `CREATE INDEX IF NOT EXISTS idx_context_relationships_user_id ON context_relationships(user_id);`);
  await ensureIndex(db, 'idx_cr_src', `CREATE INDEX IF NOT EXISTS idx_context_relationships_source ON context_relationships(source_memory_id);`);
  await ensureIndex(db, 'idx_cr_tgt', `CREATE INDEX IF NOT EXISTS idx_context_relationships_target ON context_relationships(target_memory_id);`);
  await ensureIndex(db, 'idx_cr_type', `CREATE INDEX IF NOT EXISTS idx_context_relationships_type ON context_relationships(relationship_type);`);
  await ensureIndex(db, 'idx_css_user', `CREATE INDEX IF NOT EXISTS idx_context_session_summaries_user_id ON context_session_summaries(user_id);`);
  await ensureIndex(db, 'idx_css_session', `CREATE INDEX IF NOT EXISTS idx_context_session_summaries_session_id ON context_session_summaries(session_id);`);
  await ensureIndex(db, 'idx_css_importance', `CREATE INDEX IF NOT EXISTS idx_context_session_summaries_importance ON context_session_summaries(importance_score);`);
  await ensureIndex(db, 'idx_css_topics', `CREATE INDEX IF NOT EXISTS idx_context_session_summaries_topics ON context_session_summaries USING GIN(key_topics);`);
  await ensureIndex(db, 'idx_css_entities', `CREATE INDEX IF NOT EXISTS idx_context_session_summaries_entities ON context_session_summaries USING GIN(mentioned_entities);`);

  // webhooks
  await ensureIndex(db, 'idx_webhooks_user', `CREATE INDEX IF NOT EXISTS idx_webhooks_user_id ON webhooks(user_id);`);
  await ensureIndex(db, 'idx_webhooks_active', `CREATE INDEX IF NOT EXISTS idx_webhooks_active ON webhooks(is_active);`);
  await ensureIndex(db, 'idx_webhooks_events', `CREATE INDEX IF NOT EXISTS idx_webhooks_events ON webhooks USING GIN(events);`);
  await ensureIndex(db, 'idx_wh_logs_wh', `CREATE INDEX IF NOT EXISTS idx_webhook_logs_webhook_id ON webhook_logs(webhook_id);`);
  await ensureIndex(db, 'idx_wh_logs_type', `CREATE INDEX IF NOT EXISTS idx_webhook_logs_event_type ON webhook_logs(event_type);`);
  await ensureIndex(db, 'idx_wh_logs_created', `CREATE INDEX IF NOT EXISTS idx_webhook_logs_created_at ON webhook_logs(created_at);`);
  await ensureIndex(db, 'idx_wh_logs_status', `CREATE INDEX IF NOT EXISTS idx_webhook_logs_status ON webhook_logs(response_status);`);
};

/* --------------------------- Seed default data ------------------------------ */

const seedDefaults = async (db) => {
  const homeDir = process.env.HOME || process.env.USERPROFILE || '';
  const datasetDir = homeDir ? path.join(homeDir, '.airun', 'datasets') : null;

  await db.query(`
    INSERT INTO permissions (name, description) VALUES
      ('read:report','보고서 읽기 권한'),
      ('write:report','보고서 작성 권한'),
      ('delete:report','보고서 삭제 권한'),
      ('share:report','보고서 공유 권한'),
      ('manage:users','사용자 관리 권한'),
      ('manage:api_keys','API 키 관리 권한'),
      ('admin','관리자 권한')
    ON CONFLICT (name) DO NOTHING;
  `);

  await db.query(`
    INSERT INTO role_permissions (role, permission_id)
    SELECT 'admin', id FROM permissions
    ON CONFLICT DO NOTHING;
  `);

  await db.query(`
    INSERT INTO role_permissions (role, permission_id)
    SELECT 'manager', id FROM permissions
    WHERE name IN ('read:report','write:report','share:report','manage:api_keys')
    ON CONFLICT DO NOTHING;
  `);

  await db.query(`
    INSERT INTO role_permissions (role, permission_id)
    SELECT 'user', id FROM permissions
    WHERE name IN ('read:report','write:report')
    ON CONFLICT DO NOTHING;
  `);

  const { rows } = await db.query('SELECT COUNT(*) AS c FROM users');
  if (Number(rows[0].c) === 0) {
    const adminPassword = await bcrypt.hash('admin1234', 10);
    const demoPassword = await bcrypt.hash('demo', 10);

    const adminIns = await db.query(
      `INSERT INTO users (username, email, password_hash, name, role, status)
       VALUES ('admin','admin@airun.local',$1,'Admin User','admin','active') RETURNING id;`,
      [adminPassword]
    );
    await db.query(
      `INSERT INTO users (username, email, password_hash, name, role, status)
       VALUES ('demo','demo@airun.local',$1,'Demo User','user','active');`,
      [demoPassword]
    );

    if (adminIns.rows.length) {
      const adminId = adminIns.rows[0].id;
      const randomKey = crypto.randomBytes(16).toString('hex');
      const keyValue = `airun_${adminId}_${randomKey}`;
      await db.query(
        `INSERT INTO api_keys (user_id, name, key_value, permissions, rate_limit, status)
         VALUES ($1,$2,$3,$4,$5,$6)`,
        [
          adminId,
          '기본 API 키',
          keyValue,
          {
            reports: { read: true, write: true, delete: true, share: true },
            users: { read: true, write: true, manage: true },
          },
          1000,
          'active',
        ]
      );
    }
    log('기본 사용자 생성 완료 (admin/demo)');
  }

  await db.query(`
    INSERT INTO system_configurations (config_key, config_value, config_type, description, category) VALUES
      ('data_collection_schedule','{"interval_hours": 6, "enabled": true}','json','데이터 수집 스케줄 설정','collection'),
      ('matching_threshold','70.0','number','기업-공고 매칭 최소 임계점','matching'),
      ('proposal_template_path','templates/business_proposal.docx','string','제안서 템플릿 파일 경로','generation'),
      ('ai_model_config','{"provider":"openai","model":"gpt-4","temperature":0.7}','json','AI 모델 설정','generation'),
      ('notification_settings','{"email_enabled": false, "webhook_enabled": false}','json','알림 설정','notification')
    ON CONFLICT (config_key) DO NOTHING;
  `);

  // 기본 제공 파인튜닝 데이터셋 등록 (외부 FK 제약 충돌 방지)
  const builtinDatasets = [
    {
      id: 'domain_benchmark',
      name: '도메인 평가 데이터셋',
      description: '내장 도메인 평가 데이터셋',
      filename: 'domain_benchmark.json',
      metadata: { type: 'domain_eval', builtin: true }
    },
    {
      id: 'universal_benchmark',
      name: '범용 평가 데이터셋',
      description: '내장 범용 평가 데이터셋',
      filename: 'universal_benchmark.json',
      metadata: { type: 'universal_eval', builtin: true }
    }
  ];

  for (const dataset of builtinDatasets) {
    const filePath = datasetDir ? path.join(datasetDir, dataset.filename) : null;
    let totalSamples = null;

    if (filePath && fs.existsSync(filePath)) {
      try {
        const fileContent = fs.readFileSync(filePath, 'utf-8');
        const parsed = JSON.parse(fileContent);
        const examples = Array.isArray(parsed?.examples) ? parsed.examples : [];
        totalSamples = examples.length;
      } catch (error) {
        warn(`기본 데이터셋(${dataset.id}) 파일 분석 실패:`, error.message);
      }
    }

    await db.query(
      `
        INSERT INTO finetuning_datasets (id, name, description, source, file_path, total_samples, status, metadata)
        VALUES ($1, $2, $3, $4, $5, $6, 'completed', $7)
        ON CONFLICT (id) DO UPDATE SET
          name = EXCLUDED.name,
          description = EXCLUDED.description,
          source = EXCLUDED.source,
          file_path = EXCLUDED.file_path,
          total_samples = COALESCE(EXCLUDED.total_samples, finetuning_datasets.total_samples),
          status = CASE WHEN finetuning_datasets.status = 'pending' THEN EXCLUDED.status ELSE finetuning_datasets.status END,
          metadata = finetuning_datasets.metadata || EXCLUDED.metadata,
          updated_at = CURRENT_TIMESTAMP;
      `,
      [
        dataset.id,
        dataset.name,
        dataset.description,
        'system',
        filePath,
        totalSamples,
        dataset.metadata
      ]
    );
  }
};

/* --------------------------- PG timeouts migration -------------------------- */

const migratePgTimeouts = async (db, currentVersion) => {
  if (compareVersions(currentVersion, '2.8') < 0) {
    log('PostgreSQL 타임아웃 설정 마이그레이션 시작...');
    try {
      await db.query(`ALTER SYSTEM SET statement_timeout = '300s';`);
      await db.query(`ALTER SYSTEM SET idle_in_transaction_session_timeout = '600s';`);
      await db.query(`SELECT pg_reload_conf();`);
      log('PostgreSQL 타임아웃 설정 완료: statement_timeout=300s, idle_in_transaction_session_timeout=600s');
    } catch (e) {
      warn('PostgreSQL 타임아웃 설정 실패:', e.message);
      warn('필요 시 수동 적용 바랍니다.');
    }
  }
};

/* --------------------------------- Runner ---------------------------------- */

export const createTables = async (db) => {
  const locked = await acquireAdvisoryLock(db);
  if (!locked) {
    throw new Error('다른 마이그레이션이 실행 중입니다. 잠시 후 다시 시도하세요.');
  }

  try {
    const currentVersion = await getCurrentVersion(db);
    log(`현재 데이터베이스 버전: ${currentVersion}`);

    if (compareVersions(currentVersion, SCHEMA_VERSION) >= 0) {
      log(`데이터베이스가 이미 최신 버전(${SCHEMA_VERSION})입니다.`);
      return;
    }

    await withTxn(db, async () => {
      log(`데이터베이스 마이그레이션 시작 (→ v${SCHEMA_VERSION})`);

      await ensureProjectsTable(db);

      await migrateUsersAndAuth(db);
      await migrateSessions(db);
      await migrateRAG(db);
      await migrateReports(db);
      await migrateScheduler(db);
      await migrateWebPortal(db);
      await migrateFlowAI(db);
      await migrateQADataset(db);
      await migrateGraph(db);
      await migrateBusinessSuite(db);
      await migrateContextMemory(db);
      await migrateWebhooks(db);
      await migrateHealthBatch(db);
      await migrateFineTuning(db);
      await migrateESGAnalysis(db);
      await migrateRPAJobs(db);
      await migrateAIFormBuilder(db);

      await createCoreIndexes(db);
      await seedDefaults(db);

      await recordMigration(db, SCHEMA_VERSION);
      log(`데이터베이스 마이그레이션 완료 (v${SCHEMA_VERSION})`);
    });

    await migratePgTimeouts(db, currentVersion);
  } catch (e) {
    err('데이터베이스 마이그레이션 중 오류:', e);
    throw e;
  } finally {
    await releaseAdvisoryLock(db);
  }
};