import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';
import { getVarVal } from '../../configuration.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

/**
 * 가이드 문서를 읽어옵니다.
 */
export async function readGuide() {
    try {
        // 현재 설정된 언어 코드 가져오기
        const language = await getVarVal('LANGUAGE') || 'en';
        
        // pages 디렉토리에서 해당 언어의 도움말 파일 경로 설정
        const guidePath = path.join(__dirname, '../../pages', `help.${language.toLowerCase()}.md`);
        
        // 해당 언어의 도움말 파일이 없으면 영어 버전으로 폴백
        if (!fs.existsSync(guidePath)) {
            console.warn(`Guide file not found for language ${language}, falling back to English`);
            const fallbackPath = path.join(__dirname, '../../pages', 'help.en.md');
            if (!fs.existsSync(fallbackPath)) {
                throw new Error('Guide file not found');
            }
            return await fs.promises.readFile(fallbackPath, 'utf8');
        }
        
        const content = await fs.promises.readFile(guidePath, 'utf8');
        if (!content) {
            throw new Error('Guide file is empty');
        }
        return content;
    } catch (error) {
        console.error('Error reading guide file:', error);
        throw new Error(`Failed to read guide file: ${error.message}`);
    }
} 