import chalk from 'chalk';

class ProgressDisplay {
    constructor() {
        this.spinnerFrames = ['⠋', '⠙', '⠹', '⠸', '⠼', '⠴', '⠦', '⠧', '⠇', '⠏'];
        this.currentFrame = 0;
        this.intervalId = null;
        this.startTime = null;
    }

    startSpinner(message) {
        this.stopSpinner();
        this.currentFrame = 0;
        this.intervalId = setInterval(() => {
            process.stdout.write('\r' + this.spinnerFrames[this.currentFrame] + ' ' + message);
            this.currentFrame = (this.currentFrame + 1) % this.spinnerFrames.length;
        }, 80);
    }

    stopSpinner() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = null;
            process.stdout.write('\r');
        }
    }

    showTaskPlan(analysis, steps) {
        console.log(chalk.cyan('\n=== 작업 계획 ==='));
        
        // 작업 단계 표시
        console.log(chalk.yellow('\n📋 실행 단계'));
        steps.forEach((step, index) => {
            console.log(`${chalk.blue(index + 1)}. ${step.description}`);
        });

        // 의존성 표시 (간단한 형태로)
        const hasDependencies = steps.some(step => step.dependencies && step.dependencies.length > 0);
        if (hasDependencies) {
            console.log(chalk.yellow('\n🔄 실행 순서'));
            console.log(`   ${steps.map((_, i) => i + 1).join(' → ')}`);
        }

        // 경로 정보 표시 (마지막에 간단하게)
        if (analysis) {
            console.log(chalk.yellow('\n📂 작업 경로'));
            const inputMatch = analysis.match(/(?:입력 경로|Input Paths?):\s*([^\n]+)/i);
            const outputMatch = analysis.match(/(?:출력 경로|Output Path?):\s*([^\n]+)/i);
            
            if (inputMatch) {
                const inputs = inputMatch[1].trim();
                console.log(`   입력: ${inputs}`);
            }
            if (outputMatch) {
                const output = outputMatch[1].trim();
                console.log(`   출력: ${output}`);
            }
        }
        
        console.log(); // 빈 줄 추가
    }

    showExecutionStart() {
        this.startTime = process.hrtime();
        console.log(chalk.cyan('\n=== 작업 실행 시작 ===\n'));
    }

    showExecutionResult(results) {
        console.log(chalk.cyan('\n=== 작업 실행 결과 ===\n'));
        
        // Map에서 모든 결과를 가져와서 정렬
        const sortedResults = Array.from(results.values())
            .sort((a, b) => parseInt(a.id) - parseInt(b.id));

        for (const step of sortedResults) {
            // 상태 아이콘 개선
            let statusIcon, description;
            
            if (step.status === 'failed' || step.error) {
                statusIcon = chalk.red('❌');  // 실패
                description = chalk.red(step.description);
            } else if (step.status === 'completed') {
                statusIcon = chalk.green('✅');  // 성공
                description = chalk.white(step.description);
            } else if (step.status === 'running') {  // TaskStatus.IN_PROGRESS 값
                statusIcon = chalk.blue('🔄');  // 진행 중
                description = chalk.blue(step.description);
            } else if (step.status === 'blocked') {  // TaskStatus.BLOCKED 값
                statusIcon = chalk.yellow('🚫');  // 차단됨
                description = chalk.yellow(step.description);
            } else if (step.status === 'pending') {  // TaskStatus.NOT_STARTED 값
                statusIcon = chalk.gray('⏳');  // 대기 중
                description = chalk.gray(step.description);
            } else {
                statusIcon = chalk.yellow('❓');  // 알 수 없는 상태
                description = chalk.yellow(step.description);
            }

            // 단계 번호 포맷팅
            const stepNumber = String(step.number).padStart(2, '0');
            
            console.log(`${statusIcon} 단계 ${chalk.cyan(stepNumber)}: ${description}`);
            
            if (step.error) {
                console.log(chalk.red(`   ⚠️  오류: ${step.error}`));
            }
            
            if (step.result) {
                const resultStr = typeof step.result === 'string' ? 
                    step.result : JSON.stringify(step.result, null, 2);
                if (resultStr.trim()) {
                    console.log(chalk.gray(`   📝 결과: ${resultStr}`));
                }
            }
        }

        if (this.startTime) {
            const diff = process.hrtime(this.startTime);
            const timeInSeconds = (diff[0] + diff[1] / 1e9).toFixed(2);
            console.log(chalk.yellow(`\n⏱  총 실행 시간: ${timeInSeconds}초\n`));
        }
    }

    showStepHeader(step) {
        console.log('\n' + chalk.cyan('━'.repeat(50)));
        console.log(chalk.cyan(`단계 ${parseInt(step.id) + 1}: ${step.description}`));
        console.log(chalk.cyan('━'.repeat(50)) + '\n');
    }

    showSuccess(message) {
        this.stopSpinner();
        console.log(chalk.green('✓ ') + message);
    }

    showError(message) {
        this.stopSpinner();
        console.log(chalk.red('✗ ') + message);
    }

    showWarning(message) {
        this.stopSpinner();
        console.log(chalk.yellow('⚠ ') + message);
    }

    showInfo(message) {
        this.stopSpinner();
        console.log(chalk.blue('ℹ ') + message);
    }

    showPackageInstallation(packageName, status) {
        this.stopSpinner();
        if (status) {
            console.log(chalk.green(`✓ 패키지 설치 성공: ${packageName}`));
        } else {
            console.log(chalk.red(`✗ 패키지 설치 실패: ${packageName}`));
        }
    }

    showCodeBox(code) {
        if (!code) return;
        
        console.log(chalk.yellow('\n📝 생성된 코드'));
        console.log(chalk.cyan('┌' + '─'.repeat(78) + '┐'));
        
        const lines = code.split('\n');
        lines.forEach(line => {
            // 줄이 너무 길면 자르기
            if (line.length > 76) {
                line = line.substring(0, 73) + '...';
            }
            
            // 들여쓰기를 시각적으로 더 명확하게 표시
            const indentMatch = line.match(/^(\s+)/);
            const indentLevel = indentMatch ? indentMatch[1].length : 0;
            const visualIndent = '·'.repeat(indentLevel);
            const displayLine = line.replace(/^\s+/, visualIndent);
            
            // 왼쪽 여백을 포함하여 총 78자가 되도록 공백 추가
            const padding = ' '.repeat(Math.max(0, 76 - displayLine.length));
            console.log(chalk.cyan('│ ') + chalk.gray(displayLine) + padding + chalk.cyan(' │'));
        });
        
        console.log(chalk.cyan('└' + '─'.repeat(78) + '┘'));
    }

    showCodeGeneration(success = true, code = null) {
        this.stopSpinner();
        if (success) {
            console.log(chalk.green('✓ 코드 생성 완료'));
            if (code) {
                this.showCodeBox(code);
            }
        } else {
            console.log(chalk.red('✗ 코드 생성 실패'));
        }
    }

    showCodeExecution(attempt, maxRetries) {
        return this.startSpinner(
            chalk.yellow(`코드 실행 중... (시도 ${attempt}/${maxRetries})`)
        );
    }

    showRetrying(delay) {
        this.stopSpinner();
        console.log(chalk.yellow(`⟳ ${delay/1000}초 후 재시도...`));
    }
}

export default ProgressDisplay; 