/**
 * @swagger
 * tags:
 *   - name: RPA Tasks
 *     description: RPA 업무 처리 관련 API
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     RpaTask:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: 내부 작업 ID
 *         requestId:
 *           type: string
 *           description: 사용자가 전달받는 작업 요청 ID (REQ-YYYYMMDD-NNNNNN 형식)
 *         status:
 *           type: string
 *           description: 작업 상태
 *           enum: [QUEUED, IN_PROGRESS, SUCCEEDED, FAILED]
 *         applicationType:
 *           type: string
 *           description: 신청 유형
 *         applicantName:
 *           type: string
 *           description: 신청자 이름
 *         applicantInfo:
 *           type: object
 *           nullable: true
 *           description: 신청자 세부 정보(JSON)
 *         applicantEmail:
 *           type: string
 *           format: email
 *           nullable: true
 *           description: 신청자 이메일
 *         applicantPhone:
 *           type: string
 *           nullable: true
 *           description: 신청자 전화번호
 *         applicantAddress:
 *           type: string
 *           nullable: true
 *           description: 신청자 주소
 *         applicationData:
 *           type: object
 *           nullable: true
 *           description: 신청 데이터(payload)
 *         additionalNotes:
 *           type: string
 *           nullable: true
 *           description: 추가 메모
 *         payloadHash:
 *           type: string
 *           description: 요청 본문 해시(sha256)
 *         schemaVer:
 *           type: string
 *           nullable: true
 *           description: 스키마 버전
 *         correlationId:
 *           type: string
 *           nullable: true
 *           description: 외부 시스템 추적용 상관 ID
 *         userId:
 *           type: integer
 *           nullable: true
 *           description: 작업을 생성한 사용자 ID
 *         username:
 *           type: string
 *           nullable: true
 *           description: 작업을 생성한 사용자명
 *         errorMessage:
 *           type: string
 *           nullable: true
 *           description: 실패 사유(있는 경우)
 *         createdAt:
 *           type: string
 *           format: date-time
 *           description: 생성 시각
 *         updatedAt:
 *           type: string
 *           format: date-time
 *           nullable: true
 *           description: 마지막 갱신 시각
 *         completedAt:
 *           type: string
 *           format: date-time
 *           nullable: true
 *           description: 완료 시각
 *     RpaTaskCreateRequest:
 *       type: object
 *       required:
 *         - applicationType
 *         - applicantName
 *       properties:
 *         applicationType:
 *           type: string
 *           description: 신청 유형
 *           example: BUSINESS_LICENSE
 *         applicantName:
 *           type: string
 *           description: 신청자 이름
 *           example: "홍길동"
 *         applicantInfo:
 *           type: object
 *           nullable: true
 *           description: 신청자 세부 정보(JSON)
 *           example:
 *             department: "총무팀"
 *             position: "매니저"
 *         applicantEmail:
 *           type: string
 *           format: email
 *           nullable: true
 *           example: "hong@example.com"
 *         applicantPhone:
 *           type: string
 *           nullable: true
 *           example: "+82-10-1234-5678"
 *         applicantAddress:
 *           type: string
 *           nullable: true
 *           example: "서울특별시 중구 세종대로 110"
 *         applicationData:
 *           type: object
 *           nullable: true
 *           description: 신청 관련 상세 데이터
 *           example:
 *             businessNumber: "123-45-67890"
 *             companyName: "ACME Corp"
 *         additionalNotes:
 *           type: string
 *           nullable: true
 *           description: 담당자 전달 메모
 *           example: "긴급 처리 요청"
 *         correlationId:
 *           type: string
 *           nullable: true
 *           description: 외부 시스템과 연동 시 추적용 ID
 *     RpaTaskCreateResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             taskId:
 *               type: integer
 *               description: 내부 작업 ID
 *             requestId:
 *               type: string
 *               description: 요청 식별자
 *             status:
 *               type: string
 *               enum: [QUEUED, IN_PROGRESS, SUCCEEDED, FAILED]
 *             applicant:
 *               type: object
 *               properties:
 *                 fullName:
 *                   type: string
 *                 email:
 *                   type: string
 *                   format: email
 *                   nullable: true
 *                 phone:
 *                   type: string
 *                   nullable: true
 *                 address:
 *                   type: string
 *                   nullable: true
 *             applicationData:
 *               type: object
 *               nullable: true
 *             payloadHash:
 *               type: string
 *             schemaVer:
 *               type: string
 *               nullable: true
 *             correlationId:
 *               type: string
 *               nullable: true
 *             createdAt:
 *               type: string
 *               format: date-time
 *           required:
 *             - taskId
 *             - requestId
 *             - status
 *             - createdAt
 *         message:
 *           type: string
 *         timestamp:
 *           type: string
 *           format: date-time
 *     RpaTaskListResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             tasks:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/RpaTask'
 *             total:
 *               type: integer
 *             limit:
 *               type: integer
 *             offset:
 *               type: integer
 *         timestamp:
 *           type: string
 *           format: date-time
 *     RpaTaskDetailResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           $ref: '#/components/schemas/RpaTask'
 *         timestamp:
 *           type: string
 *           format: date-time
 *     RpaTaskStatusUpdateRequest:
 *       type: object
 *       properties:
 *         status:
 *           type: string
 *           enum: [QUEUED, IN_PROGRESS, SUCCEEDED, FAILED]
 *           description: 업데이트할 상태
 *         error_message:
 *           type: string
 *           nullable: true
 *           description: 실패 사유 또는 추가 메시지
 *       description: 상태 또는 에러 메시지 중 하나 이상을 전달해야 합니다.
 *     RpaTaskStatusUpdateResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             id:
 *               type: integer
 *             requestId:
 *               type: string
 *             status:
 *               type: string
 *               enum: [QUEUED, IN_PROGRESS, SUCCEEDED, FAILED]
 *             errorMessage:
 *               type: string
 *               nullable: true
 *             updatedAt:
 *               type: string
 *               format: date-time
 *             completedAt:
 *               type: string
 *               format: date-time
 *               nullable: true
 *         message:
 *           type: string
 *         timestamp:
 *           type: string
 *           format: date-time
 */

/**
 * @swagger
 * /api/v1/rpa_tasks:
 *   post:
 *     summary: 새 RPA 작업 생성
 *     description: RPA 처리 자동화를 위한 작업을 새로 등록합니다.
 *     tags: [RPA Tasks]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/RpaTaskCreateRequest'
 *           example:
 *             applicationType: BUSINESS_LICENSE
 *             applicantName: "홍길동"
 *             applicantEmail: "hong@example.com"
 *             applicantPhone: "+82-10-1234-5678"
 *             applicantInfo:
 *               department: "총무팀"
 *               priority: "HIGH"
 *             applicationData:
 *               businessNumber: "123-45-67890"
 *               companyName: "ACME Corp"
 *             additionalNotes: "긴급 처리 요청"
 *             correlationId: "EXT-REQ-20250101-0001"
 *     responses:
 *       201:
 *         description: RPA 작업이 성공적으로 생성됨
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RpaTaskCreateResponse'
 *             example:
 *               success: true
 *               data:
 *                 taskId: 42
 *                 requestId: "REQ-20250101-123456"
 *                 status: "QUEUED"
 *                 applicant:
 *                   fullName: "홍길동"
 *                   email: "hong@example.com"
 *                   phone: "+82-10-1234-5678"
 *                   address: "서울특별시 중구 세종대로 110"
 *                 applicationData:
 *                   businessNumber: "123-45-67890"
 *                 payloadHash: "sha256:1a2b3c4d5e6f7g8h"
 *                 schemaVer: "1.0"
 *                 correlationId: "EXT-REQ-20250101-0001"
 *                 createdAt: "2025-01-01T09:30:00.000Z"
 *               message: "RPA 작업이 성공적으로 등록되었습니다."
 *               timestamp: "2025-01-01T09:30:00.000Z"
 *       400:
 *         description: 잘못된 요청
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패(API Key 누락 또는 유효하지 않음)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *   get:
 *     summary: RPA 작업 목록 조회
 *     description: 등록된 RPA 작업들을 필터링 및 페이징하여 조회합니다.
 *     tags: [RPA Tasks]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: query
 *         name: status
 *         schema:
 *           type: string
 *           enum: [QUEUED, IN_PROGRESS, SUCCEEDED, FAILED]
 *         description: 특정 상태로 필터링
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 50
 *         description: 한 번에 조회할 항목 수
 *       - in: query
 *         name: offset
 *         schema:
 *           type: integer
 *           default: 0
 *         description: 조회 시작 위치
 *     responses:
 *       200:
 *         description: 작업 목록
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RpaTaskListResponse'
 *             example:
 *               success: true
 *               data:
 *                 tasks:
 *                   - id: 42
 *                     requestId: "REQ-20250101-123456"
 *                     status: "QUEUED"
 *                     applicationType: "BUSINESS_LICENSE"
 *                     applicantName: "홍길동"
 *                     applicantEmail: "hong@example.com"
 *                     applicantPhone: "+82-10-1234-5678"
 *                     createdAt: "2025-01-01T09:30:00.000Z"
 *                     updatedAt: null
 *                     completedAt: null
 *                 total: 1
 *                 limit: 50
 *                 offset: 0
 *               timestamp: "2025-01-01T09:30:00.000Z"
 *       401:
 *         description: 인증 실패(API Key 누락 또는 유효하지 않음)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *
 * /api/v1/rpa_tasks/{task_id}:
 *   get:
 *     summary: 특정 RPA 작업 상세 조회
 *     description: 숫자 ID 또는 requestId(REQ-...)로 단일 작업 상세 정보를 가져옵니다.
 *     tags: [RPA Tasks]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: task_id
 *         required: true
 *         schema:
 *           type: string
 *         description: 작업 ID 또는 requestId
 *     responses:
 *       200:
 *         description: 작업 상세 정보
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RpaTaskDetailResponse'
 *             example:
 *               success: true
 *               data:
 *                 id: 42
 *                 requestId: "REQ-20250101-123456"
 *                 status: "IN_PROGRESS"
 *                 applicationType: "BUSINESS_LICENSE"
 *                 applicantName: "홍길동"
 *                 applicantInfo:
 *                   department: "총무팀"
 *                 applicantEmail: "hong@example.com"
 *                 applicantPhone: "+82-10-1234-5678"
 *                 applicationData:
 *                   businessNumber: "123-45-67890"
 *                 additionalNotes: "긴급 처리 요청"
 *                 payloadHash: "sha256:1a2b3c4d5e6f7g8h"
 *                 schemaVer: "1.0"
 *                 correlationId: "EXT-REQ-20250101-0001"
 *                 userId: 1
 *                 username: "admin"
 *                 errorMessage: null
 *                 createdAt: "2025-01-01T09:30:00.000Z"
 *                 updatedAt: "2025-01-01T10:00:00.000Z"
 *                 completedAt: null
 *               timestamp: "2025-01-01T10:00:00.000Z"
 *       401:
 *         description: 인증 실패(API Key 누락 또는 유효하지 않음)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       404:
 *         description: 작업을 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *   post:
 *     summary: RPA 작업 상태 업데이트
 *     description: 작업 ID 또는 requestId를 기준으로 상태 및 에러 메시지를 갱신합니다.
 *     tags: [RPA Tasks]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: task_id
 *         required: true
 *         schema:
 *           type: string
 *         description: 작업 ID 또는 requestId
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/RpaTaskStatusUpdateRequest'
 *           example:
 *             status: "SUCCEEDED"
 *             error_message: null
 *     responses:
 *       200:
 *         description: 작업 상태가 갱신됨
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RpaTaskStatusUpdateResponse'
 *             example:
 *               success: true
 *               data:
 *                 id: 42
 *                 requestId: "REQ-20250101-123456"
 *                 status: "SUCCEEDED"
 *                 errorMessage: null
 *                 updatedAt: "2025-01-01T11:00:00.000Z"
 *                 completedAt: "2025-01-01T11:00:00.000Z"
 *               message: "RPA 작업 상태가 업데이트되었습니다."
 *               timestamp: "2025-01-01T11:00:00.000Z"
 *       400:
 *         description: 잘못된 상태 값 등 요청 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패(API Key 누락 또는 유효하지 않음)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       404:
 *         description: 작업을 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

export const rpaTaskSchemas = {
    RpaTask: {
        type: 'object',
        properties: {
            id: {
                type: 'integer',
                description: '내부 작업 ID'
            },
            requestId: {
                type: 'string',
                description: '사용자가 전달받는 작업 요청 ID (REQ-YYYYMMDD-NNNNNN 형식)'
            },
            status: {
                type: 'string',
                enum: ['QUEUED', 'IN_PROGRESS', 'SUCCEEDED', 'FAILED'],
                description: '작업 상태'
            },
            applicationType: {
                type: 'string',
                description: '신청 유형'
            },
            applicantName: {
                type: 'string',
                description: '신청자 이름'
            },
            applicantInfo: {
                type: 'object',
                nullable: true,
                description: '신청자 세부 정보(JSON)',
                additionalProperties: true
            },
            applicantEmail: {
                type: 'string',
                format: 'email',
                nullable: true,
                description: '신청자 이메일'
            },
            applicantPhone: {
                type: 'string',
                nullable: true,
                description: '신청자 전화번호'
            },
            applicantAddress: {
                type: 'string',
                nullable: true,
                description: '신청자 주소'
            },
            applicationData: {
                type: 'object',
                nullable: true,
                description: '신청 데이터(payload)',
                additionalProperties: true
            },
            additionalNotes: {
                type: 'string',
                nullable: true,
                description: '추가 메모'
            },
            payloadHash: {
                type: 'string',
                description: '요청 본문 해시(sha256)'
            },
            schemaVer: {
                type: 'string',
                nullable: true,
                description: '스키마 버전'
            },
            correlationId: {
                type: 'string',
                nullable: true,
                description: '외부 시스템 추적용 상관 ID'
            },
            userId: {
                type: 'integer',
                nullable: true,
                description: '작업을 생성한 사용자 ID'
            },
            username: {
                type: 'string',
                nullable: true,
                description: '작업을 생성한 사용자명'
            },
            errorMessage: {
                type: 'string',
                nullable: true,
                description: '실패 사유(있는 경우)'
            },
            createdAt: {
                type: 'string',
                format: 'date-time',
                description: '생성 시각'
            },
            updatedAt: {
                type: 'string',
                format: 'date-time',
                nullable: true,
                description: '마지막 갱신 시각'
            },
            completedAt: {
                type: 'string',
                format: 'date-time',
                nullable: true,
                description: '완료 시각'
            }
        }
    },
    RpaTaskCreateRequest: {
        type: 'object',
        required: ['applicationType', 'applicantName'],
        properties: {
            applicationType: {
                type: 'string',
                description: '신청 유형'
            },
            applicantName: {
                type: 'string',
                description: '신청자 이름'
            },
            applicantInfo: {
                type: 'object',
                nullable: true,
                description: '신청자 세부 정보(JSON)',
                additionalProperties: true
            },
            applicantEmail: {
                type: 'string',
                format: 'email',
                nullable: true
            },
            applicantPhone: {
                type: 'string',
                nullable: true
            },
            applicantAddress: {
                type: 'string',
                nullable: true
            },
            applicationData: {
                type: 'object',
                nullable: true,
                description: '신청 관련 상세 데이터',
                additionalProperties: true
            },
            additionalNotes: {
                type: 'string',
                nullable: true,
                description: '담당자 전달 메모'
            },
            correlationId: {
                type: 'string',
                nullable: true,
                description: '외부 시스템과 연동 시 추적용 ID'
            }
        }
    },
    RpaTaskCreateResponse: {
        type: 'object',
        properties: {
            success: {
                type: 'boolean'
            },
            data: {
                type: 'object',
                properties: {
                    taskId: {
                        type: 'integer',
                        description: '내부 작업 ID'
                    },
                    requestId: {
                        type: 'string',
                        description: '요청 식별자'
                    },
                    status: {
                        type: 'string',
                        enum: ['QUEUED', 'IN_PROGRESS', 'SUCCEEDED', 'FAILED'],
                        description: '현재 상태'
                    },
                    applicant: {
                        type: 'object',
                        properties: {
                            fullName: {
                                type: 'string'
                            },
                            email: {
                                type: 'string',
                                format: 'email',
                                nullable: true
                            },
                            phone: {
                                type: 'string',
                                nullable: true
                            },
                            address: {
                                type: 'string',
                                nullable: true
                            }
                        }
                    },
                    applicationData: {
                        type: 'object',
                        nullable: true,
                        additionalProperties: true
                    },
                    payloadHash: {
                        type: 'string'
                    },
                    schemaVer: {
                        type: 'string',
                        nullable: true
                    },
                    correlationId: {
                        type: 'string',
                        nullable: true
                    },
                    createdAt: {
                        type: 'string',
                        format: 'date-time'
                    }
                }
            },
            message: {
                type: 'string'
            },
            timestamp: {
                type: 'string',
                format: 'date-time'
            }
        }
    },
    RpaTaskListResponse: {
        type: 'object',
        properties: {
            success: {
                type: 'boolean'
            },
            data: {
                type: 'object',
                properties: {
                    tasks: {
                        type: 'array',
                        items: {
                            $ref: '#/components/schemas/RpaTask'
                        }
                    },
                    total: {
                        type: 'integer'
                    },
                    limit: {
                        type: 'integer'
                    },
                    offset: {
                        type: 'integer'
                    }
                }
            },
            timestamp: {
                type: 'string',
                format: 'date-time'
            }
        }
    },
    RpaTaskDetailResponse: {
        type: 'object',
        properties: {
            success: {
                type: 'boolean'
            },
            data: {
                $ref: '#/components/schemas/RpaTask'
            },
            timestamp: {
                type: 'string',
                format: 'date-time'
            }
        }
    },
    RpaTaskStatusUpdateRequest: {
        type: 'object',
        properties: {
            status: {
                type: 'string',
                enum: ['QUEUED', 'IN_PROGRESS', 'SUCCEEDED', 'FAILED'],
                description: '업데이트할 상태'
            },
            error_message: {
                type: 'string',
                nullable: true,
                description: '실패 사유 또는 추가 메시지'
            }
        },
        description: '상태 또는 에러 메시지 중 하나 이상을 전달해야 합니다.'
    },
    RpaTaskStatusUpdateResponse: {
        type: 'object',
        properties: {
            success: {
                type: 'boolean'
            },
            data: {
                type: 'object',
                properties: {
                    id: {
                        type: 'integer'
                    },
                    requestId: {
                        type: 'string'
                    },
                    status: {
                        type: 'string',
                        enum: ['QUEUED', 'IN_PROGRESS', 'SUCCEEDED', 'FAILED']
                    },
                    errorMessage: {
                        type: 'string',
                        nullable: true
                    },
                    updatedAt: {
                        type: 'string',
                        format: 'date-time'
                    },
                    completedAt: {
                        type: 'string',
                        format: 'date-time',
                        nullable: true
                    }
                }
            },
            message: {
                type: 'string'
            },
            timestamp: {
                type: 'string',
                format: 'date-time'
            }
        }
    }
};