import apiClient, { ApiResponse } from './apiClient';

// 채팅 메시지 인터페이스
export interface ChatMessage {
  role: 'user' | 'assistant' | 'system';
  content: string;
  timestamp?: string;
  image?: string;
}

// 채팅 요청 옵션 인터페이스
export interface ChatOptions {
  temperature?: number;
  max_tokens?: number;
  messages?: ChatMessage[];
  history?: ChatMessage[];
  summary?: string;
  first?: boolean;
  sessionId?: string;
  rag?: boolean;
  web?: boolean;
  image?: string;
  settings?: Record<string, any>;
}

// 채팅 응답 인터페이스
export interface ChatResponse {
  id: string;
  object: string;
  created: number;
  model: string;
  provider: string;
  sessionId: string;
  userId: string;
  choices: {
    index: number;
    message: {
      role: string;
      content: string;
      sessionId: string;
      userId: string;
    };
    finish_reason: string;
  }[];
  usage: {
    prompt_tokens: number;
    completion_tokens: number;
    total_tokens: number;
  };
  rag_enabled: boolean;
  web_search_enabled: boolean;
  rag_results?: any[];
}

// 채팅 작업 상태 인터페이스
export interface ChatJobStatus {
  jobId: string;
  state: string;
  message: string;
  data?: ChatResponse;
  timestamp: string;
}

// 채팅 API 서비스 클래스
class ChatService {
  // 채팅 요청 전송 (비동기)
  async sendChat(prompt: string, options: ChatOptions = {}): Promise<ApiResponse<{ jobId: string }>> {
    return apiClient.post('/chat', {
      prompt,
      options,
    });
  }

  // 채팅 요청 전송 (동기)
  async sendChatSync(prompt: string, options: ChatOptions = {}): Promise<ApiResponse<ChatResponse>> {
    return apiClient.post('/chat/sync', {
      prompt,
      options,
    });
  }

  // 채팅 작업 상태 확인
  async checkChatStatus(jobId: string): Promise<ApiResponse<ChatJobStatus>> {
    return apiClient.get(`/chat/status/${jobId}`);
  }

  // 세션 생성
  async createSession(provider?: string, model?: string): Promise<ApiResponse<{ id: string }>> {
    return apiClient.post('/sessions/create', {
      provider,
      model,
    });
  }

  // 세션 조회
  async getSession(sessionId: string): Promise<ApiResponse<{
    id: string;
    provider: string;
    model: string;
    history: ChatMessage[];
  }>> {
    return apiClient.get(`/sessions/${sessionId}`);
  }

  // 세션 목록 조회
  async getSessions(): Promise<ApiResponse<Record<string, any>>> {
    return apiClient.get('/sessions');
  }
}

// 채팅 서비스 인스턴스 생성
const chatService = new ChatService();

export default chatService; 