# Security Features Documentation

## Overview

The AIRUN web application implements comprehensive security measures to protect against common web attacks and malicious requests.

## Implemented Security Features

### 1. Malicious File Extension Blocking

The middleware automatically blocks requests for files with suspicious extensions commonly used in attacks:

**Blocked Extensions:**
- PHP files: `.php`, `.php3`, `.php4`, `.php5`, `.phtml`
- ASP files: `.asp`, `.aspx`, `.jsp`, `.jspx`
- Scripts: `.cgi`, `.pl`, `.py`, `.rb`, `.sh`
- Executables: `.exe`, `.dll`, `.bat`, `.cmd`
- Database files: `.sql`, `.db`, `.sqlite`
- Configuration files: `.env`, `.config`, `.ini`, `.conf`
- Backup files: `.bak`, `.backup`, `.old`, `.tmp`
- Version control: `.git`, `.svn`, `.htaccess`, `.htpasswd`

**Example:**
```bash
# These requests will be blocked with 403 Forbidden
GET /test.php
GET /admin.php?cmd=ls
GET /config.ini
```

### 2. Malicious Pattern Detection

The middleware detects and blocks common attack patterns:

**Attack Types Detected:**
- **Path Traversal**: `../`, `%2e%2e`
- **SQL Injection**: `union select`, `insert into`, `drop table`, etc.
- **Command Injection**: `system()`, `exec()`, `shell_exec`, etc.
- **XSS Attempts**: `<script>`, `javascript:`, `onerror=`, `onload=`
- **System File Access**: `/etc/passwd`, `/proc/`
- **Admin Path Scanning**: `/admin/`, `/phpmyadmin`, `/wp-admin`

**Example:**
```bash
# These requests will be blocked with 403 Forbidden
GET /../../etc/passwd
GET /user?id=1' UNION SELECT * FROM users--
GET /?cmd=system('ls')
GET /<script>alert('xss')</script>
```

### 3. Rate Limiting

Automatic rate limiting prevents abuse and DDoS attacks:

**Configuration:**
- **Window**: 1 minute (60 seconds)
- **Max Requests**: 100 requests per IP per minute
- **Response**: `429 Too Many Requests`

**Customization:**
```typescript
// In src/middleware.ts
const RATE_LIMIT_WINDOW = 60000; // milliseconds
const RATE_LIMIT_MAX_REQUESTS = 100; // requests per window
```

**Note**: Current implementation uses in-memory storage. For production with multiple servers, consider using Redis:

```typescript
// Production example with Redis
import { createClient } from 'redis';

const redis = createClient({
  url: process.env.REDIS_URL || 'redis://localhost:6379'
});

async function checkRateLimit(ip: string): Promise<boolean> {
  const key = `rate_limit:${ip}`;
  const count = await redis.incr(key);

  if (count === 1) {
    await redis.expire(key, 60); // 60 seconds
  }

  return count <= RATE_LIMIT_MAX_REQUESTS;
}
```

### 4. Security Logging

All security events are logged for monitoring and analysis:

**Logged Information:**
- Timestamp (ISO 8601 format)
- Event type (`MALICIOUS_REQUEST_BLOCKED`, `RATE_LIMIT_EXCEEDED`)
- Request path
- Client IP address
- User-Agent string
- Block reason

**Console Output Example:**
```
[SECURITY ALERT] 2025-10-13T10:30:45.123Z
  Type: MALICIOUS_REQUEST_BLOCKED
  Path: /admin.php
  IP: 192.168.1.100
  User-Agent: Mozilla/5.0...
  Details: Blocked file extension: .php
---
```

### 5. Security Monitoring API

Administrators can access security logs and statistics through the API:

#### Get Security Statistics

```bash
curl -X GET "http://localhost:5700/api/admin/security?action=stats" \
  -H "Cookie: auth_token=your_token"
```

**Response:**
```json
{
  "success": true,
  "data": {
    "total": 1542,
    "lastHour": 23,
    "last24Hours": 456,
    "byType": {
      "MALICIOUS_REQUEST_BLOCKED": 1420,
      "RATE_LIMIT_EXCEEDED": 122
    },
    "topIps": [
      { "ip": "192.168.1.100", "count": 45 },
      { "ip": "10.0.0.50", "count": 32 }
    ],
    "topPaths": [
      { "path": "/admin.php", "count": 234 },
      { "path": "/wp-login.php", "count": 189 }
    ]
  }
}
```

#### Get Security Logs

```bash
curl -X GET "http://localhost:5700/api/admin/security?action=logs&limit=50&offset=0" \
  -H "Cookie: auth_token=your_token"
```

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "timestamp": "2025-10-13T10:30:45.123Z",
      "type": "MALICIOUS_REQUEST_BLOCKED",
      "path": "/admin.php",
      "ip": "192.168.1.100",
      "userAgent": "Mozilla/5.0...",
      "reason": "Blocked file extension: .php"
    }
  ]
}
```

#### Clear Security Logs

```bash
curl -X DELETE "http://localhost:5700/api/admin/security" \
  -H "Cookie: auth_token=your_token"
```

## Configuration

### Customizing Blocked Extensions

Edit `src/middleware.ts`:

```typescript
const BLOCKED_EXTENSIONS = [
  '.php', '.php3', '.php4', '.php5', '.phtml',
  // Add your custom extensions here
  '.custom', '.dangerous',
];
```

### Customizing Malicious Patterns

Edit `src/middleware.ts`:

```typescript
const MALICIOUS_PATTERNS = [
  /\.\./,                    // Path traversal
  // Add your custom patterns here
  /\/custom-admin\//i,       // Custom admin path
  /dangerous-keyword/i,      // Custom keyword
];
```

### Adjusting Rate Limits

Edit `src/middleware.ts`:

```typescript
const RATE_LIMIT_WINDOW = 60000; // 1 minute
const RATE_LIMIT_MAX_REQUESTS = 100; // max requests

// For stricter limits:
const RATE_LIMIT_WINDOW = 30000; // 30 seconds
const RATE_LIMIT_MAX_REQUESTS = 50; // max requests
```

### Log Storage Configuration

The current implementation stores up to 1000 security logs in memory. For production:

1. **Database Storage** (Recommended):
```typescript
// In src/app/api/admin/security/route.ts
import { pool } from '@/app/api/db';

async addLog(log: SecurityLog) {
  await pool.query(
    'INSERT INTO security_logs (timestamp, type, path, ip, user_agent, reason) VALUES ($1, $2, $3, $4, $5, $6)',
    [log.timestamp, log.type, log.path, log.ip, log.userAgent, log.reason]
  );
}
```

2. **File-based Logging**:
```typescript
import { writeFile, appendFile } from 'fs/promises';

async addLog(log: SecurityLog) {
  const logLine = JSON.stringify(log) + '\n';
  await appendFile('/var/log/airun/security.log', logLine);
}
```

## Testing

### Test Malicious Request Blocking

```bash
# Test PHP file blocking
curl -I http://localhost:5700/test.php
# Expected: HTTP/1.1 403 Forbidden

# Test path traversal
curl -I "http://localhost:5700/../../../etc/passwd"
# Expected: HTTP/1.1 403 Forbidden

# Test SQL injection
curl -I "http://localhost:5700/user?id=1' UNION SELECT * FROM users--"
# Expected: HTTP/1.1 403 Forbidden
```

### Test Rate Limiting

```bash
# Send 150 requests rapidly
for i in {1..150}; do
  curl -s -o /dev/null -w "%{http_code}\n" http://localhost:5700/
done
# Expected: First 100 return 200, then 429
```

### Monitor Security Logs

```bash
# Watch console output
npm run dev

# Make malicious request
curl http://localhost:5700/admin.php

# Check console for security alert
```

## Best Practices

### 1. Regular Monitoring

- Check security logs daily
- Review statistics for unusual patterns
- Investigate repeated attacks from specific IPs

### 2. Update Patterns Regularly

- Monitor new attack vectors
- Add new patterns as threats emerge
- Review and update blocked extensions

### 3. Combine with Web Application Firewall (WAF)

For production, consider adding:
- Cloudflare WAF
- AWS WAF
- ModSecurity

### 4. IP Blacklisting

For persistent attackers:
```typescript
// Add to src/middleware.ts
const BLACKLISTED_IPS = [
  '1.2.3.4',
  '5.6.7.8',
];

export function middleware(request: NextRequest) {
  const ip = request.ip || request.headers.get('x-forwarded-for') || 'unknown';

  if (BLACKLISTED_IPS.includes(ip)) {
    return new NextResponse('Forbidden', { status: 403 });
  }

  // ... rest of middleware
}
```

### 5. HTTPS Only

Always use HTTPS in production:
```javascript
// In next.config.js
module.exports = {
  async headers() {
    return [
      {
        source: '/:path*',
        headers: [
          {
            key: 'Strict-Transport-Security',
            value: 'max-age=63072000; includeSubDomains; preload'
          }
        ]
      }
    ]
  }
}
```

## Troubleshooting

### False Positives

If legitimate requests are being blocked:

1. Check the security logs for the block reason
2. Adjust patterns or extensions in `src/middleware.ts`
3. Add exceptions for specific paths:

```typescript
const EXCEPTION_PATHS = ['/api/special-endpoint'];

export function middleware(request: NextRequest) {
  const { pathname } = request.nextUrl;

  if (EXCEPTION_PATHS.some(path => pathname.startsWith(path))) {
    return NextResponse.next();
  }

  // ... rest of middleware
}
```

### Rate Limiting Issues

If users are hitting rate limits:

1. Increase `RATE_LIMIT_MAX_REQUESTS`
2. Increase `RATE_LIMIT_WINDOW`
3. Implement authenticated user exceptions:

```typescript
function checkRateLimit(ip: string, isAuthenticated: boolean): boolean {
  const maxRequests = isAuthenticated
    ? RATE_LIMIT_MAX_REQUESTS * 2  // Double limit for authenticated users
    : RATE_LIMIT_MAX_REQUESTS;

  // ... rest of rate limit logic
}
```

## Performance Considerations

### Memory Usage

- In-memory rate limiting: ~100 bytes per IP
- In-memory log storage: ~500 bytes per log (max 1000 logs = ~500KB)

### Processing Overhead

- Pattern matching: ~0.1-0.5ms per request
- Rate limiting: ~0.01ms per request
- Logging: ~0.05ms per blocked request

### Optimization Tips

1. **Use Redis for Rate Limiting** (if scaling horizontally)
2. **Move logs to database** (for long-term storage)
3. **Implement pattern caching** (for frequently checked patterns)
4. **Use CDN** (to filter traffic before reaching the server)

## Security Checklist

- [x] Malicious file extension blocking
- [x] Path traversal protection
- [x] SQL injection detection
- [x] Command injection detection
- [x] XSS protection
- [x] Rate limiting
- [x] Security logging
- [x] Security monitoring API
- [ ] Database-backed log storage (recommended for production)
- [ ] Redis-backed rate limiting (recommended for production)
- [ ] IP blacklisting database
- [ ] Automated alert notifications
- [ ] Integration with SIEM systems

## References

- [OWASP Top 10](https://owasp.org/www-project-top-ten/)
- [NextJS Security Best Practices](https://nextjs.org/docs/advanced-features/security-headers)
- [Web Application Firewall Guide](https://cheatsheetseries.owasp.org/cheatsheets/Web_Application_Firewall_Cheat_Sheet.html)
