'use client';

import React, { useState, useRef } from 'react';
import { useAuth } from '@/hooks/useAuth';
import {
  Upload,
  FileText,
  CheckCircle,
  AlertCircle,
  Download,
  Activity,
  Leaf,
  BarChart3,
  Building2,
  DollarSign
} from 'lucide-react';

// 타입 정의
interface DocumentData {
  jobId: string;
  documentId: number;
  filename: string;
  contentPreview: string;
  estimatedRevenue: number | null;
}

interface IndustryClassification {
  majorCode: string;
  majorName: string;
  subCode: string;
  subName: string;
  confidence: number;
}

interface IndustryCategory {
  code: string;
  major_category: string;
  sub_categories: SubCategory[];
  carbon_intensity: number;
}

interface SubCategory {
  code: string;
  name: string;
}

interface AnalysisResult {
  totalCarbonEmission: number;
  report: string;
}

type ProcessingStep = 'upload' | 'verify' | 'results';

export default function ESGAnalysisPage() {
  const { user } = useAuth();
  const [currentStep, setCurrentStep] = useState<ProcessingStep>('upload');
  const [uploadedFile, setUploadedFile] = useState<File | null>(null);
  const [documentData, setDocumentData] = useState<DocumentData | null>(null);
  const [industryData, setIndustryData] = useState<IndustryClassification | null>(null);
  const [analysisResult, setAnalysisResult] = useState<AnalysisResult | null>(null);
  const [customRevenue, setCustomRevenue] = useState<string>('');
  const [isUploading, setIsUploading] = useState(false);
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [uploadError, setUploadError] = useState<string | null>(null);
  const [dragOver, setDragOver] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  // 산업분류 데이터
  const [allIndustries, setAllIndustries] = useState<IndustryCategory[]>([]);
  const [selectedMajorCode, setSelectedMajorCode] = useState<string>('');
  const [selectedSubCode, setSelectedSubCode] = useState<string>('');
  const [availableSubCategories, setAvailableSubCategories] = useState<SubCategory[]>([]);

  // 컴포넌트 마운트 시 산업분류 데이터 로드
  React.useEffect(() => {
    loadIndustryClassifications();
  }, []);

  // AI가 산업분류를 식별하면 선택 상태 업데이트
  React.useEffect(() => {
    if (industryData) {
      setSelectedMajorCode(industryData.majorCode);
      setSelectedSubCode(industryData.subCode);

      // 소분류 목록 업데이트
      const industry = allIndustries.find(i => i.code === industryData.majorCode);
      if (industry) {
        setAvailableSubCategories(industry.sub_categories);
      }
    }
  }, [industryData, allIndustries]);

  // 산업분류 데이터 로드
  const loadIndustryClassifications = async () => {
    try {
      const response = await fetch('/api/admin/esg-analysis/industries');
      const result = await response.json();

      if (result.success) {
        setAllIndustries(result.data.classifications);
      }
    } catch (error) {
      console.error('산업분류 데이터 로드 오류:', error);
    }
  };

  // 대분류 선택 변경 핸들러
  const handleMajorCodeChange = (code: string) => {
    setSelectedMajorCode(code);

    const industry = allIndustries.find(i => i.code === code);
    if (industry) {
      setAvailableSubCategories(industry.sub_categories);

      // 첫 번째 소분류 자동 선택
      if (industry.sub_categories.length > 0) {
        setSelectedSubCode(industry.sub_categories[0].code);
      } else {
        setSelectedSubCode('');
      }

      // industryData 업데이트
      if (industryData) {
        setIndustryData({
          ...industryData,
          majorCode: code,
          majorName: industry.major_category,
          subCode: industry.sub_categories[0]?.code || '',
          subName: industry.sub_categories[0]?.name || ''
        });
      }
    }
  };

  // 소분류 선택 변경 핸들러
  const handleSubCodeChange = (code: string) => {
    setSelectedSubCode(code);

    const industry = allIndustries.find(i => i.code === selectedMajorCode);
    const subCategory = industry?.sub_categories.find(s => s.code === code);

    if (subCategory && industryData) {
      setIndustryData({
        ...industryData,
        subCode: code,
        subName: subCategory.name
      });
    }
  };

  // 파일 드래그 앤 드롭 핸들러
  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    setDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    setDragOver(false);
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    setDragOver(false);

    const files = Array.from(e.dataTransfer.files);
    const docFile = files.find(file =>
      file.name.match(/\.(pdf|docx?|xlsx?|txt)$/i)
    );

    if (docFile) {
      handleFileSelect(docFile);
    } else {
      setUploadError('PDF, Word, Excel, 텍스트 파일만 업로드 가능합니다.');
    }
  };

  // 파일 선택 핸들러
  const handleFileSelect = async (file: File) => {
    setUploadError(null);
    setIsUploading(true);

    try {
      if (!file.name.match(/\.(pdf|docx?|xlsx?|txt)$/i)) {
        throw new Error('PDF, Word, Excel, 텍스트 파일만 업로드 가능합니다.');
      }

      if (file.size > 10 * 1024 * 1024) {
        throw new Error('파일 크기는 10MB 이하여야 합니다.');
      }

      setUploadedFile(file);

      const formData = new FormData();
      formData.append('file', file);

      const response = await fetch('/api/admin/esg-analysis/parse', {
        method: 'POST',
        body: formData,
        credentials: 'include'
      });

      if (!response.ok) {
        throw new Error('파일 파싱에 실패했습니다.');
      }

      const result = await response.json();

      if (!result.success) {
        throw new Error(result.error || '파일 파싱에 실패했습니다.');
      }

      setDocumentData({
        jobId: result.data.jobId,
        documentId: result.data.documentId,
        filename: result.data.filename,
        contentPreview: result.data.contentPreview,
        estimatedRevenue: result.data.estimatedRevenue
      });

      // 표준산업분류 자동 식별
      await identifyIndustry(result.data.jobId, result.data.documentId);

      setCurrentStep('verify');

    } catch (error) {
      console.error('파일 업로드 오류:', error);
      setUploadError(error instanceof Error ? error.message : '파일 업로드에 실패했습니다.');
      setUploadedFile(null);
    } finally {
      setIsUploading(false);
    }
  };

  // 표준산업분류 식별
  const identifyIndustry = async (jobId: string, documentId: number) => {
    try {
      const response = await fetch('/api/admin/esg-analysis/identify-industry', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ jobId, documentId }),
        credentials: 'include'
      });

      if (!response.ok) {
        throw new Error('표준산업분류 식별에 실패했습니다.');
      }

      const result = await response.json();

      if (result.success) {
        setIndustryData(result.data.industry);
      }
    } catch (error) {
      console.error('표준산업분류 식별 오류:', error);
    }
  };

  // ESG 분석 시작
  const startAnalysis = async () => {
    if (!documentData) return;

    setIsAnalyzing(true);
    setUploadError(null);

    try {
      // 사용자 지정 매출액이 있으면 사용, 없으면 원본 사용
      let finalRevenue = documentData.estimatedRevenue;
      if (customRevenue) {
        finalRevenue = parseFloat(customRevenue.replace(/,/g, ''));
      }

      if (!finalRevenue || finalRevenue <= 0) {
        throw new Error('매출액을 입력해주세요.');
      }

      // 1단계: 탄소배출량 분석 (수정된 매출액과 산업분류 전달)
      const analyzeResponse = await fetch('/api/admin/esg-analysis/analyze', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          jobId: documentData.jobId,
          documentId: documentData.documentId,
          // 수정된 데이터 전달
          revenue: finalRevenue,
          industryMajorCode: selectedMajorCode,
          industrySubCode: selectedSubCode
        }),
        credentials: 'include'
      });

      if (!analyzeResponse.ok) {
        throw new Error('ESG 분석에 실패했습니다.');
      }

      const analyzeResult = await analyzeResponse.json();

      if (!analyzeResult.success) {
        throw new Error(analyzeResult.error || 'ESG 분석에 실패했습니다.');
      }

      // 2단계: HTML 보고서 생성
      const reportResponse = await fetch('/api/admin/esg-analysis/generate-report', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          jobId: documentData.jobId,
          documentId: documentData.documentId
        }),
        credentials: 'include'
      });

      if (!reportResponse.ok) {
        throw new Error('보고서 생성에 실패했습니다.');
      }

      const reportResult = await reportResponse.json();

      if (!reportResult.success) {
        throw new Error(reportResult.error || '보고서 생성에 실패했습니다.');
      }

      setAnalysisResult({
        totalCarbonEmission: analyzeResult.data.totalCarbonEmission,
        report: analyzeResult.data.report
      });

      setCurrentStep('results');

    } catch (error) {
      console.error('ESG 분석 오류:', error);
      setUploadError(error instanceof Error ? error.message : 'ESG 분석에 실패했습니다.');
    } finally {
      setIsAnalyzing(false);
    }
  };

  // Canvas를 이미지로 변환하는 함수
  const convertCanvasToImages = async () => {
    console.log('[DEBUG] Starting canvas to image conversion...');

    const iframe = document.querySelector('iframe[title="ESG 분석 보고서"]') as HTMLIFrameElement;
    if (!iframe) {
      console.error('[DEBUG] iframe not found');
      return null;
    }

    if (!iframe.contentWindow) {
      console.error('[DEBUG] iframe.contentWindow is null');
      return null;
    }

    try {
      const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
      const canvases = iframeDoc.querySelectorAll('canvas');

      console.log(`[DEBUG] Found ${canvases.length} canvas elements in iframe`);

      const imageData: { [id: string]: string } = {};

      canvases.forEach((canvas, index) => {
        try {
          const id = canvas.id || `chart-${index}`;
          const originalWidth = canvas.width;
          const originalHeight = canvas.height;

          console.log(`[DEBUG] Converting canvas #${index}, id="${id}", original size=${originalWidth}x${originalHeight}`);

          // 고해상도 이미지 생성 (2배 스케일, 최대 가로 1200px)
          const scale = 2;
          const maxWidth = 1200;
          const targetWidth = Math.min(originalWidth, maxWidth);
          const targetHeight = (originalHeight / originalWidth) * targetWidth;

          // 임시 canvas 생성하여 고해상도로 렌더링
          const tempCanvas = iframeDoc.createElement('canvas');
          tempCanvas.width = targetWidth * scale;
          tempCanvas.height = targetHeight * scale;

          const tempCtx = tempCanvas.getContext('2d');
          if (tempCtx) {
            // 고품질 렌더링 설정
            tempCtx.imageSmoothingEnabled = true;
            tempCtx.imageSmoothingQuality = 'high';

            // 원본 canvas를 고해상도로 스케일링하여 그리기
            tempCtx.drawImage(
              canvas,
              0, 0, originalWidth, originalHeight,
              0, 0, targetWidth * scale, targetHeight * scale
            );

            const dataUrl = tempCanvas.toDataURL('image/png', 0.95);
            imageData[id] = dataUrl;

            console.log(`[DEBUG] Converted canvas "${id}" to high-res data URL (${targetWidth}x${targetHeight} @${scale}x, length: ${dataUrl.length})`);
          } else {
            // Fallback to original canvas
            const dataUrl = canvas.toDataURL('image/png');
            imageData[id] = dataUrl;
            console.log(`[DEBUG] Converted canvas "${id}" to data URL (original, length: ${dataUrl.length})`);
          }
        } catch (e) {
          console.error(`[DEBUG] Error converting canvas #${index}:`, e);
        }
      });

      console.log(`[DEBUG] Successfully converted ${Object.keys(imageData).length} canvases`);
      return imageData;
    } catch (e) {
      console.error('[DEBUG] iframe access error:', e);
      return null;
    }
  };

  // 보고서 다운로드
  const downloadReport = async (format: 'html' | 'pdf' | 'docx' | 'hwp' = 'html') => {
    if (!documentData) return;

    try {
      // HTML 형식은 차트 변환 없이 바로 다운로드
      if (format === 'html') {
        const response = await fetch(
          `/api/admin/esg-analysis/download/${documentData.jobId}?format=${format}`,
          { credentials: 'include' }
        );

        if (!response.ok) {
          throw new Error('보고서 다운로드에 실패했습니다.');
        }

        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `ESG_환경영향분석보고서_${new Date().toISOString().slice(0, 10)}.${format}`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
        return;
      }

      // PDF/DOCX/HWP 형식은 차트를 이미지로 변환 후 다운로드
      const chartImages = await convertCanvasToImages();

      const response = await fetch(
        `/api/admin/esg-analysis/download/${documentData.jobId}?format=${format}`,
        {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({ chartImages }),
          credentials: 'include'
        }
      );

      if (!response.ok) {
        throw new Error('보고서 다운로드에 실패했습니다.');
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `ESG_환경영향분석보고서_${new Date().toISOString().slice(0, 10)}.${format}`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      window.URL.revokeObjectURL(url);

    } catch (error) {
      console.error('보고서 다운로드 오류:', error);
      alert(error instanceof Error ? error.message : '보고서 다운로드에 실패했습니다.');
    }
  };

  // 단계 표시 컴포넌트
  const StepIndicator = () => {
    const steps = [
      { key: 'upload', label: '파일 업로드', icon: Upload },
      { key: 'verify', label: '데이터 확인', icon: FileText },
      { key: 'results', label: '분석 결과', icon: BarChart3 }
    ];

    const currentIndex = steps.findIndex(step => step.key === currentStep);

    return (
      <div className="flex items-center justify-center mb-8 px-4">
        {steps.map((step, index) => {
          const IconComponent = step.icon;
          const isActive = index === currentIndex;
          const isCompleted = index < currentIndex;
          const isDisabled = index > currentIndex;

          return (
            <React.Fragment key={step.key}>
              <div className={`flex flex-col items-center ${isDisabled ? 'opacity-50' : ''}`}>
                <div className={`w-12 h-12 rounded-full flex items-center justify-center transition-all duration-200 ${isCompleted
                  ? 'bg-green-500 text-white'
                  : isActive
                    ? 'bg-blue-500 text-white'
                    : 'bg-gray-200 dark:bg-gray-700 text-gray-500 dark:text-gray-400'
                  }`}>
                  {isCompleted ? (
                    <CheckCircle className="w-6 h-6" />
                  ) : (
                    <IconComponent className="w-6 h-6" />
                  )}
                </div>
                <span className={`mt-2 text-sm font-medium ${isActive ? 'text-blue-600 dark:text-blue-400' : 'text-gray-500 dark:text-gray-400'
                  }`}>
                  {step.label}
                </span>
              </div>
              {index < steps.length - 1 && (
                <div className={`flex-1 h-px mx-4 transition-all duration-200 ${isCompleted ? 'bg-green-500' : 'bg-gray-300 dark:bg-gray-600'
                  }`} style={{ minWidth: '60px' }} />
              )}
            </React.Fragment>
          );
        })}
      </div>
    );
  };

  // 단계별 컨텐츠 렌더링
  const renderStepContent = () => {
    switch (currentStep) {
      case 'upload':
        return (
          <div className="max-w-2xl mx-auto">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>
                문서 업로드
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                ESG 탄소배출량 분석을 위한 문서를 업로드해주세요.
              </p>
            </div>

            <div
              className={`drag-drop-zone border-2 border-dashed rounded-xl p-12 text-center transition-all duration-200 cursor-pointer ${dragOver
                ? 'border-blue-400 bg-blue-50 dark:bg-blue-900/20'
                : 'border-gray-300 dark:border-gray-600 hover:border-gray-400'
                }`}
              onDragOver={handleDragOver}
              onDragLeave={handleDragLeave}
              onDrop={handleDrop}
              onClick={() => fileInputRef.current?.click()}
              data-drag-over={dragOver}
            >
              <input
                ref={fileInputRef}
                type="file"
                accept=".pdf,.docx,.doc,.xlsx,.xls,.txt"
                onChange={(e) => {
                  const file = e.target.files?.[0];
                  if (file) handleFileSelect(file);
                }}
                className="hidden"
              />

              {isUploading ? (
                <div className="flex flex-col items-center">
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mb-4"></div>
                  <p className="text-gray-600 dark:text-gray-400">파일 처리 중...</p>
                </div>
              ) : (
                <div className="flex flex-col items-center">
                  <FileText className="w-16 h-16 text-gray-400 mb-4" />
                  <p className="text-lg font-medium mb-2" style={{ color: 'var(--text-primary)' }}>
                    파일을 드래그하거나 클릭하여 업로드
                  </p>
                  <p className="text-sm text-gray-500 dark:text-gray-400">
                    지원 형식: PDF, Word, Excel, TXT
                  </p>
                </div>
              )}
            </div>

            {uploadError && (
              <div className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                <div className="flex items-center">
                  <AlertCircle className="w-5 h-5 text-red-500 mr-2" />
                  <span className="text-red-700 dark:text-red-400">{uploadError}</span>
                </div>
              </div>
            )}
          </div>
        );

      case 'verify':
        return (
          <div className="max-w-4xl mx-auto">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>
                데이터 확인
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                분석된 표준산업분류와 예상 매출액을 확인하고 수정할 수 있습니다.
              </p>
            </div>

            <div className="space-y-6">
              {/* 표준산업분류 */}
              {industryData && (
                <div className="card p-6">
                  <div className="flex items-start">
                    <Building2 className="w-6 h-6 text-blue-500 mr-3 mt-1" />
                    <div className="flex-1">
                      <h3 className="text-lg font-semibold mb-4" style={{ color: 'var(--text-primary)' }}>
                        표준산업분류
                      </h3>

                      {/* AI 식별 결과 표시 */}
                      <div className="bg-blue-50 dark:bg-blue-900/20 p-3 rounded-lg mb-4">
                        <p className="text-sm text-blue-800 dark:text-blue-300 mb-1">
                          <strong>AI 식별 결과:</strong> [{industryData.majorCode}] {industryData.majorName}
                          {industryData.subName && ` > [${industryData.subCode}] ${industryData.subName}`}
                        </p>
                        <p className="text-sm text-blue-600 dark:text-blue-400">
                          신뢰도: {(industryData.confidence * 100).toFixed(1)}%
                        </p>
                      </div>

                      {/* 산업분류 선택 */}
                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <label className="text-sm text-gray-600 dark:text-gray-400 mb-2 block">
                            대분류
                          </label>
                          <select
                            value={selectedMajorCode}
                            onChange={(e) => handleMajorCodeChange(e.target.value)}
                            className="w-full px-4 py-2 border rounded-lg"
                            style={{
                              borderColor: 'var(--border-color)',
                              backgroundColor: 'var(--input-bg)',
                              color: 'var(--text-primary)'
                            }}
                          >
                            {allIndustries.map((industry) => (
                              <option key={industry.code} value={industry.code}>
                                [{industry.code}] {industry.major_category}
                              </option>
                            ))}
                          </select>
                        </div>

                        <div>
                          <label className="text-sm text-gray-600 dark:text-gray-400 mb-2 block">
                            소분류
                          </label>
                          <select
                            value={selectedSubCode}
                            onChange={(e) => handleSubCodeChange(e.target.value)}
                            className="w-full px-4 py-2 border rounded-lg"
                            style={{
                              borderColor: 'var(--border-color)',
                              backgroundColor: 'var(--input-bg)',
                              color: 'var(--text-primary)'
                            }}
                            disabled={availableSubCategories.length === 0}
                          >
                            {availableSubCategories.map((subCategory) => (
                              <option key={subCategory.code} value={subCategory.code}>
                                [{subCategory.code}] {subCategory.name}
                              </option>
                            ))}
                          </select>
                        </div>
                      </div>

                      {/* 탄소집약도 표시 */}
                      {selectedMajorCode && (
                        <div className="mt-4 p-3 bg-green-50 dark:bg-green-900/20 rounded-lg">
                          <p className="text-sm text-green-800 dark:text-green-300">
                            <strong>탄소집약도:</strong> {allIndustries.find(i => i.code === selectedMajorCode)?.carbon_intensity || 0} tCO2e/백만원
                          </p>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              )}

              {/* 예상 매출액 */}
              <div className="card p-6">
                <div className="flex items-start">
                  <DollarSign className="w-6 h-6 text-green-500 mr-3 mt-1" />
                  <div className="flex-1">
                    <h3 className="text-lg font-semibold mb-2" style={{ color: 'var(--text-primary)' }}>
                      예상 매출액
                    </h3>
                    {documentData?.estimatedRevenue ? (
                      <div>
                        <p className="text-base text-gray-600 dark:text-gray-400 mb-2">
                          문서에서 추출된 매출액: {new Intl.NumberFormat('ko-KR').format(documentData.estimatedRevenue)}원
                        </p>
                      </div>
                    ) : (
                      <p className="text-base text-yellow-600 dark:text-yellow-400 mb-2">
                        문서에서 매출액을 찾을 수 없습니다. 직접 입력해주세요.
                      </p>
                    )}
                    <div className="mt-4">
                      <label className="text-sm text-gray-600 dark:text-gray-400 mb-2 block">
                        매출액 수정 (원)
                      </label>
                      <input
                        type="text"
                        value={customRevenue}
                        onChange={(e) => {
                          const value = e.target.value.replace(/[^0-9]/g, '');
                          setCustomRevenue(value);
                        }}
                        placeholder={documentData?.estimatedRevenue?.toString() || '예: 1000000000'}
                        className="w-full px-4 py-2 border rounded-lg"
                        style={{
                          borderColor: 'var(--border-color)',
                          backgroundColor: 'var(--input-bg)',
                          color: 'var(--text-primary)'
                        }}
                      />
                      {customRevenue && (
                        <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                          {new Intl.NumberFormat('ko-KR').format(parseInt(customRevenue))}원
                        </p>
                      )}
                    </div>
                  </div>
                </div>
              </div>

              {/* 문서 미리보기 */}
              <div className="card p-6">
                <h3 className="text-lg font-semibold mb-4" style={{ color: 'var(--text-primary)' }}>
                  문서 미리보기
                </h3>
                <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg max-h-64 overflow-y-auto">
                  <p className="text-sm text-gray-700 dark:text-gray-300 whitespace-pre-wrap">
                    {documentData?.contentPreview}
                  </p>
                </div>
              </div>
            </div>

            <div className="flex justify-center gap-4 mt-8">
              <button
                onClick={() => setCurrentStep('upload')}
                className="px-6 py-2 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
              >
                다시 업로드
              </button>
              <button
                onClick={startAnalysis}
                disabled={isAnalyzing}
                className="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors flex items-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isAnalyzing ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                    분석 중...
                  </>
                ) : (
                  <>
                    <Activity className="w-4 h-4" />
                    분석 시작
                  </>
                )}
              </button>
            </div>

            {uploadError && (
              <div className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                <div className="flex items-center">
                  <AlertCircle className="w-5 h-5 text-red-500 mr-2" />
                  <span className="text-red-700 dark:text-red-400">{uploadError}</span>
                </div>
              </div>
            )}
          </div>
        );

      case 'results':
        return (
          <div className="max-w-7xl mx-auto">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>
                분석 완료
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                ESG 탄소배출량 분석이 완료되었습니다.
              </p>
            </div>

            {analysisResult && (
              <div className="space-y-6">
                {/* 탄소배출량 요약 */}
                <div className="card p-6 bg-gradient-to-br from-green-50 to-blue-50 dark:from-green-900/20 dark:to-blue-900/20">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center">
                      <Leaf className="w-12 h-12 text-green-600 mr-4" />
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">
                          예상 탄소배출량
                        </p>
                        <p className="text-3xl font-bold text-green-600">
                          {analysisResult.totalCarbonEmission.toFixed(2)} tCO2e
                        </p>
                      </div>
                    </div>
                    <div className="flex gap-2">
                      <button
                        onClick={() => downloadReport('html')}
                        className="px-4 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors flex items-center gap-2"
                      >
                        <Download className="w-5 h-5" />
                        HTML
                      </button>
                      <button
                        onClick={() => downloadReport('pdf')}
                        className="px-4 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors flex items-center gap-2"
                      >
                        <Download className="w-5 h-5" />
                        PDF
                      </button>
                      <button
                        onClick={() => downloadReport('docx')}
                        className="px-4 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors flex items-center gap-2"
                      >
                        <Download className="w-5 h-5" />
                        DOCX
                      </button>
                      <button
                        onClick={() => downloadReport('hwp')}
                        className="px-4 py-3 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors flex items-center gap-2"
                      >
                        <Download className="w-5 h-5" />
                        HWP
                      </button>
                    </div>
                  </div>
                </div>

                {/* HTML 보고서 미리보기 */}
                <div className="card p-6">
                  <h3 className="text-lg font-semibold mb-4" style={{ color: 'var(--text-primary)' }}>
                    ESG 환경영향 심층분석 보고서 미리보기
                  </h3>
                  <div className="bg-white dark:bg-gray-900 rounded-lg overflow-hidden border border-gray-200 dark:border-gray-700" style={{ height: '600px' }}>
                    <iframe
                      src={`/api/admin/esg-analysis/download/${documentData?.jobId}?preview=true`}
                      className="w-full h-full"
                      title="ESG 분석 보고서"
                      sandbox="allow-same-origin allow-scripts"
                    />
                  </div>
                  <p className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                    💡 보고서를 저장하려면 위의 "보고서 다운로드" 버튼을 클릭하세요.
                  </p>
                </div>
              </div>
            )}

            <div className="flex justify-center gap-4 mt-8">
              <button
                onClick={() => {
                  setCurrentStep('upload');
                  setDocumentData(null);
                  setIndustryData(null);
                  setAnalysisResult(null);
                  setUploadedFile(null);
                  setCustomRevenue('');
                }}
                className="px-6 py-2 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
              >
                새로 시작
              </button>
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  // 권한 확인
  if (!user || user.role !== 'admin') {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <AlertCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
          <h2 className="text-2xl font-bold mb-2" style={{ color: 'var(--text-primary)' }}>
            접근 권한 없음
          </h2>
          <p className="text-gray-600 dark:text-gray-400">
            이 페이지는 관리자만 접근할 수 있습니다.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen py-8 px-4" style={{ backgroundColor: 'var(--body-bg)' }}>
      <div className="max-w-7xl mx-auto">
        {/* 페이지 헤더 */}
        <div className="text-center mb-12">
          <div className="flex items-center justify-center mb-4">
            <Leaf className="w-12 h-12 text-green-600 mr-4" />
            <h3 className="text-2xl font-bold" style={{ color: 'var(--text-primary)' }}>
              ESG 탄소배출량 분석
            </h3>
          </div>
          <p className="text-lg text-gray-600 dark:text-gray-400">
            문서를 업로드하여 탄소배출량을 분석하고 ESG 보고서를 생성합니다
          </p>
        </div>

        {/* 단계 표시기 */}
        <StepIndicator />

        {/* 단계별 컨텐츠 */}
        {renderStepContent()}
      </div>
    </div>
  );
}
