'use client';

import React, { useEffect, useState, useRef } from 'react';
import { EvaluationItem, ModelInfo, LeaderboardRow } from '../../../types/evaluation';
import { Bar } from 'react-chartjs-2';
import { Chart, BarElement, CategoryScale, LinearScale, Tooltip, Legend } from 'chart.js';
import { getModelMetadata, calculateAverageResponseTime, type ModelMetadata } from '@/data/modelMetadata';
import { MarkdownRenderer } from '@/components/MarkdownRenderer';
import * as pdfjsLib from 'pdfjs-dist';

Chart.register(BarElement, CategoryScale, LinearScale, Tooltip, Legend);

// PDF.js worker 설정 (로컬 파일 사용)
if (typeof window !== 'undefined') {
  pdfjsLib.GlobalWorkerOptions.workerSrc = '/pdf-worker/pdf.worker.min.js';
}

const EVAL_CATEGORIES = [
  { key: 'math', label: '수학' },
  { key: 'coding', label: '코딩' },
  { key: 'reasoning', label: '추론' },
  { key: 'knowledge', label: '지식' },
  { key: 'language', label: '언어' },
];

interface EvalStatus {
  isRunning: boolean;
  isStopped?: boolean;
  progress: number;
  currentModel: string;
  currentCategory: string;
  totalModels: number;
  completedModels: number;
  error?: string;
  startTime?: string;
  endTime?: string;
  stoppedAt?: string;
}

interface ChatMessage {
  id: string;
  content: string;
  role: 'user' | 'assistant';
  timestamp: Date;
  responseTime?: number; // 응답 시간 (ms)
  startTime?: Date; // 요청 시작 시간
}

interface ChatState {
  isOpen: boolean;
  selectedModel: string;
  messages: ChatMessage[];
  isLoading: boolean;
  input: string;
  streaming: boolean;
  keepHistory: boolean;
  uploadedFiles: File[];
  enableTools: boolean;
  showSettings: boolean;
}

interface HuggingFaceModel {
  id: string;
  downloads: number;
  likes: number;
  tags: string[];
  library_name?: string;
  pipeline_tag?: string;
}

interface ModelDownloadStatus {
  model: string;
  status: 'downloading' | 'completed' | 'error';
  progress: number;
  error?: string;
}

interface ModelItem {
  name: string;
  size: number;
  modified_at?: string;
  digest?: string;
  type: 'ollama' | 'vllm';
  provider: string;
  object?: string;
  created?: number;
  owned_by?: string;

  context_length?: number;
  parameters?: string;
}

export default function EvaluationLeaderboard() {
  const [results, setResults] = useState<any>({});
  const [sortKey, setSortKey] = useState<string>('avg');
  const [sortOrder, setSortOrder] = useState<'asc' | 'desc'>('desc');
  const [isDark, setIsDark] = useState<null | boolean>(null);
  const [availableModels, setAvailableModels] = useState<ModelItem[]>([]);
  const [selectedModels, setSelectedModels] = useState<string[]>([]);
  const [judgeProvider, setJudgeProvider] = useState<'openai' | 'ollama' | 'vllm'>('ollama');
  const [evalStatus, setEvalStatus] = useState<EvalStatus>({
    isRunning: false,
    progress: 0,
    currentModel: '',
    currentCategory: '',
    totalModels: 0,
    completedModels: 0,
  });
  const [showModelSelector, setShowModelSelector] = useState(false);
  const [modelMetadataCache, setModelMetadataCache] = useState<Record<string, ModelMetadata>>({});
  const [modalMessage, setModalMessage] = useState<{
    type: 'success' | 'error' | 'warning' | 'info';
    title: string;
    message: string;
    show: boolean;
  }>({
    type: 'info',
    title: '',
    message: '',
    show: false
  });

  // 채팅 상태 관리
  const [chatState, setChatState] = useState<ChatState>({
    isOpen: false,
    selectedModel: '',
    messages: [],
    isLoading: false,
    input: '',
    streaming: true,
    keepHistory: true,
    uploadedFiles: [],
    enableTools: false, // 기본적으로 도구 기능 비활성화
    showSettings: false
  });

  // 폴링 인터벌 관리를 위한 ref
  const pollIntervalRef = useRef<NodeJS.Timeout | null>(null);
  
  // 메시지 스크롤을 위한 ref
  const messagesEndRef = useRef<HTMLDivElement>(null);
  
  // 입력창 포커스를 위한 ref
  const inputRef = useRef<HTMLTextAreaElement>(null);

  // 사이드바 상태 관리
  const [sidebarCollapsed, setSidebarCollapsed] = useState(false);
  const [activeTab, setActiveTab] = useState<'huggingface' | 'ollama' | 'upload' | 'manage'>('huggingface');
  const [huggingFaceModels, setHuggingFaceModels] = useState<HuggingFaceModel[]>([]);
  const [ollamaModels, setOllamaModels] = useState<string[]>([]);
  const [searchQuery, setSearchQuery] = useState('');
  const [downloadStatus, setDownloadStatus] = useState<ModelDownloadStatus[]>([]);
  const [uploadingFiles, setUploadingFiles] = useState<File[]>([]);
  const [searchLoading, setSearchLoading] = useState(false);

  // 평가 결과 불러오기
  const fetchResults = async () => {
    try {
      const res = await fetch('/api/eval-results');
      const data = await res.json();
      setResults(data);
    } catch {
      setResults({});
    }
  };

  // 사용 가능한 모델 목록 불러오기
  const fetchAvailableModels = async () => {
    try {
      const res = await fetch('/api/eval-models');
      const data = await res.json();
      setAvailableModels(data.models || []);
    } catch (error) {
      console.error('모델 목록 조회 실패:', error);
      setAvailableModels([]);
    }
  };

  // 평가 상태 조회
  const fetchEvalStatus = async () => {
    try {
      const res = await fetch('/api/eval-status');
      const data = await res.json();
      // console.log('상태 업데이트:', data);
      setEvalStatus(data);
    } catch (error) {
      console.error('평가 상태 조회 실패:', error);
    }
  };

  // 모델 메타데이터 캐시 로드
  const loadModelMetadata = async () => {
    const modelIds = Object.keys(results);
    const cache: Record<string, ModelMetadata> = {};
    
    for (const modelId of modelIds) {
      // 정적 메타데이터 먼저 확인
      const staticMetadata = getModelMetadata(modelId);
      if (staticMetadata) {
        cache[modelId] = staticMetadata;
      } else {
        // 동적 메타데이터 로드 (사용자 정의 모델의 경우)
        try {
          const response = await fetch(`/api/ollama-metadata?model=${encodeURIComponent(modelId)}`);
          if (response.ok) {
            const data = await response.json();
            
            // 베이스 모델이 있으면 해당 메타데이터 사용
            if (data.baseModel) {
              const baseMetadata = getModelMetadata(data.baseModel);
              if (baseMetadata) {
                cache[modelId] = {
                  ...baseMetadata,
                  description: `Custom model based on ${data.baseModel}`,
                  source: 'ollama-api'
                };
                continue;
              }
            }
            
            // 직접 추출한 정보로 메타데이터 생성
            cache[modelId] = {
              parameters: data.parameters || 'Unknown',
              contextLength: data.contextLength || 2048,
              supportsTools: false,
              supportsVision: false,
              family: data.family || 'Unknown',
              provider: 'Custom',
              description: 'Custom model from Ollama',
              source: 'ollama-api'
            };
          }
        } catch (error) {
          console.error(`Failed to load metadata for ${modelId}:`, error);
        }
      }
    }
    
    // 특별한 모델들에 대한 메타데이터 추가/수정
    if (results['airun-chat:latest']) {
      cache['airun-chat:latest'] = {
        parameters: '24B',
        contextLength: 131072,
        supportsTools: true,
        supportsVision: false,
        family: 'Mistral',
        provider: 'AI.RUN',
        description: 'AI.RUN에서 개발한 다목적 AI 모델',
        source: 'airun-custom'
      };
    }
    
    if (results['hamonize:latest']) {
      cache['hamonize:latest'] = {
        parameters: '20B',
        contextLength: 131072,
        supportsTools: true,
        supportsVision: false,
        family: 'GPT-OSS',
        provider: 'Hamonize',
        description: 'GPT-OSS 기반으로 커스텀한 도구 지원 AI 모델',
        source: 'hamonize-custom'
      };
    }
    
    setModelMetadataCache(cache);
  };

  // 모달 메시지 표시 함수
  const showModal = (type: 'success' | 'error' | 'warning' | 'info', title: string, message: string) => {
    setModalMessage({
      type,
      title,
      message,
      show: true
    });
  };

  const closeModal = () => {
    setModalMessage(prev => ({ ...prev, show: false }));
  };

  // 사이드바 관련 함수들
  const searchHuggingFaceModels = async (query: string) => {
    if (!query.trim()) {
      setHuggingFaceModels([]);
      return;
    }
    
    setSearchLoading(true);
    try {
      const response = await fetch(`https://huggingface.co/api/models?search=${encodeURIComponent(query)}&limit=20&filter=text-generation`);
      const data = await response.json();
      setHuggingFaceModels(data || []);
    } catch (error) {
      console.error('허깅페이스 모델 검색 오류:', error);
      setHuggingFaceModels([]);
    } finally {
      setSearchLoading(false);
    }
  };

  const searchOllamaModels = async (query: string) => {
    if (!query.trim()) {
      setOllamaModels([]);
      return;
    }
    
    setSearchLoading(true);
    try {
      // Ollama 공식 모델 목록 - 실제로는 API에서 가져와야 함
      const commonModels = [
        'llama3.2', 'llama3.1', 'llama3', 'llama2', 'mistral', 'mixtral', 'qwen2.5', 'qwen2',
        'phi3', 'gemma2', 'gemma', 'codellama', 'vicuna', 'openchat', 'neural-chat', 'starling-lm',
        'solar', 'nous-hermes2', 'yi', 'deepseek-coder', 'codeup', 'magicoder', 'phind-codellama',
        'wizardlm2', 'orca-mini', 'tinyllama', 'llava', 'bakllava', 'dolphin-mixtral', 'dolphin-mistral'
      ];
      
      const filtered = commonModels.filter(model => 
        model.toLowerCase().includes(query.toLowerCase())
      );
      setOllamaModels(filtered);
    } catch (error) {
      console.error('Ollama 모델 검색 오류:', error);
      setOllamaModels([]);
    } finally {
      setSearchLoading(false);
    }
  };

  const downloadHuggingFaceModel = async (modelId: string) => {
    try {
      const newDownload: ModelDownloadStatus = {
        model: modelId,
        status: 'downloading',
        progress: 0
      };
      setDownloadStatus(prev => [...prev, newDownload]);

      const response = await fetch('/api/download-huggingface-model', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ modelId }),
      });

      if (!response.ok) {
        throw new Error('모델 다운로드 실패');
      }

      // 실제로는 진행률을 추적해야 함
      setDownloadStatus(prev => prev.map(d => 
        d.model === modelId ? { ...d, status: 'completed', progress: 100 } : d
      ));
      
      // 모델 목록 새로고침
      await fetchAvailableModels();
      
    } catch (error) {
      console.error('모델 다운로드 오류:', error);
      setDownloadStatus(prev => prev.map(d => 
        d.model === modelId ? { ...d, status: 'error', error: '다운로드 실패' } : d
      ));
    }
  };

  const downloadOllamaModel = async (modelName: string) => {
    try {
      const newDownload: ModelDownloadStatus = {
        model: modelName,
        status: 'downloading',
        progress: 0
      };
      setDownloadStatus(prev => [...prev, newDownload]);

      const response = await fetch('/api/download-ollama-model', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ modelName }),
      });

      if (!response.ok) {
        throw new Error('모델 다운로드 실패');
      }

      setDownloadStatus(prev => prev.map(d => 
        d.model === modelName ? { ...d, status: 'completed', progress: 100 } : d
      ));
      
      await fetchAvailableModels();
      
    } catch (error) {
      console.error('모델 다운로드 오류:', error);
      setDownloadStatus(prev => prev.map(d => 
        d.model === modelName ? { ...d, status: 'error', error: '다운로드 실패' } : d
      ));
    }
  };

  const handleModelUpload = async (files: FileList) => {
    const fileArray = Array.from(files);
    setUploadingFiles(fileArray);
    
    try {
      const formData = new FormData();
      fileArray.forEach(file => {
        formData.append('models', file);
      });

      const response = await fetch('/api/upload-model', {
        method: 'POST',
        body: formData,
      });

      if (!response.ok) {
        throw new Error('모델 업로드 실패');
      }

      await fetchAvailableModels();
      setUploadingFiles([]);
      
    } catch (error) {
      console.error('모델 업로드 오류:', error);
      setUploadingFiles([]);
    }
  };

  const deleteModel = async (modelName: string) => {
    if (!confirm(`${modelName} 모델을 삭제하시겠습니까?`)) return;
    
    try {
      const response = await fetch('/api/delete-model', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ modelName }),
      });

      if (!response.ok) {
        throw new Error('모델 삭제 실패');
      }

      await fetchAvailableModels();
      
    } catch (error) {
      console.error('모델 삭제 오류:', error);
    }
  };

  // 채팅 관련 함수들
  const startChat = (modelName: string) => {
    setChatState({
      isOpen: true,
      selectedModel: modelName,
      messages: [],
      isLoading: false,
      input: '',
      streaming: true,
      keepHistory: true,
      uploadedFiles: [],
      enableTools: false, // 기본적으로 도구 기능 비활성화
      showSettings: false
    });
  };

  const closeChat = () => {
    setChatState(prev => ({ ...prev, isOpen: false }));
  };

  const sendMessage = async () => {
    if (!chatState.input.trim() || chatState.isLoading) return;

    const startTime = new Date();
    const userMessage: ChatMessage = {
      id: Date.now().toString(),
      content: chatState.input,
      role: 'user',
      timestamp: startTime,
      startTime: startTime
    };

    setChatState(prev => ({
      ...prev,
      messages: [...prev.messages, userMessage],
      input: '',
      isLoading: true,
      uploadedFiles: [] // 메시지 전송 후 파일 초기화
    }));

    try {
      // 파일을 base64로 변환 및 텍스트 추출 (크기 제한 적용)
      const fileData = await Promise.all(
        chatState.uploadedFiles.map(async (file) => {
          // 파일 크기 제한 확인 (10MB)
          if (file.size > 10 * 1024 * 1024) {
            throw new Error(`파일 크기가 너무 큽니다: ${file.name} (${(file.size / 1024 / 1024).toFixed(2)}MB). 최대 10MB까지 지원됩니다.`);
          }
          
          const base64 = await new Promise<string>((resolve) => {
            const reader = new FileReader();
            reader.onloadend = () => resolve(reader.result as string);
            reader.readAsDataURL(file);
          });
          
          // base64 데이터 크기 확인 (약 1.33배 커짐)
          const base64Size = base64.length * 0.75; // 실제 데이터 크기 추정
          if (base64Size > 8 * 1024 * 1024) { // 8MB 제한
            console.warn(`Base64 데이터 크기가 큽니다: ${file.name} (${(base64Size / 1024 / 1024).toFixed(2)}MB)`);
          }
          
            // PDF 파일인 경우 텍스트 추출
            let extractedText = '';
            if (file.type === 'application/pdf') {
              console.log(`📋 PDF 텍스트 추출 시작: ${file.name}`);
              try {
                extractedText = await extractPdfText(file);
                console.log(`✅ PDF 텍스트 추출 완료: ${file.name} (${extractedText.length}자 추출)`);
                if (extractedText.length > 5000) {
                  console.log(`⚠️ 큰 PDF 파일: ${file.name} - ${extractedText.length}자 추출됨`);
                }
              } catch (error) {
                console.error(`❌ PDF 텍스트 추출 실패: ${file.name}`, error);
                extractedText = `PDF 텍스트 추출 중 오류가 발생했습니다: ${error}`;
              }
            }
          
          return {
            name: file.name,
            type: file.type,
            size: file.size,
            data: base64,
            extractedText: extractedText || undefined
          };
        })
      );

      const requestBody = {
        model: chatState.selectedModel,
        message: userMessage.content,
        messages: chatState.keepHistory ? chatState.messages : [userMessage],
        files: fileData,
        streaming: chatState.streaming,
        enableTools: chatState.enableTools
      };

      if (chatState.streaming) {
        // 스트리밍 처리
        const response = await fetch('/api/chat-stream', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(requestBody),
        });

        if (!response.ok) {
          throw new Error('채팅 응답을 받을 수 없습니다.');
        }

        const reader = response.body?.getReader();
        const decoder = new TextDecoder();
        
        let assistantMessage: ChatMessage = {
          id: (Date.now() + 1).toString(),
          content: '',
          role: 'assistant',
          timestamp: new Date(),
          startTime: startTime
        };

        // 빈 메시지 추가
        setChatState(prev => ({
          ...prev,
          messages: [...prev.messages, assistantMessage],
          isLoading: false
        }));

        while (true) {
          const { done, value } = await reader!.read();
          if (done) {
            // 스트리밍 완료 시 응답 시간 계산
            const endTime = new Date();
            assistantMessage.responseTime = endTime.getTime() - startTime.getTime();
            setChatState(prev => ({
              ...prev,
              messages: prev.messages.map(msg => 
                msg.id === assistantMessage.id ? assistantMessage : msg
              )
            }));
            
            // 응답 완료 시 입력창으로 포커스 이동
            setTimeout(() => {
              inputRef.current?.focus();
            }, 100);
            break;
          }
          
          const chunk = decoder.decode(value);
          const lines = chunk.split('\n');
          
          for (const line of lines) {
            if (line.startsWith('data: ')) {
              const data = line.slice(6);
              if (data === '[DONE]') {
                // 스트리밍 완료 시 응답 시간 계산
                const endTime = new Date();
                assistantMessage.responseTime = endTime.getTime() - startTime.getTime();
                setChatState(prev => ({
                  ...prev,
                  messages: prev.messages.map(msg => 
                    msg.id === assistantMessage.id ? assistantMessage : msg
                  )
                }));
                
                // 응답 완료 시 입력창으로 포커스 이동
                setTimeout(() => {
                  inputRef.current?.focus();
                }, 100);
              } else {
                try {
                  const parsed = JSON.parse(data);
                  if (parsed.content) {
                    assistantMessage.content += parsed.content;
                    setChatState(prev => ({
                      ...prev,
                      messages: prev.messages.map(msg => 
                        msg.id === assistantMessage.id ? assistantMessage : msg
                      )
                    }));
                  }
                } catch (e) {
                  console.error('파싱 오류:', e);
                }
              }
            }
          }
        }
      } else {
        // 일반 처리
        const response = await fetch('/api/chat', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(requestBody),
        });

        if (!response.ok) {
          throw new Error('채팅 응답을 받을 수 없습니다.');
        }

        const data = await response.json();
        const endTime = new Date();
        
        const assistantMessage: ChatMessage = {
          id: (Date.now() + 1).toString(),
          content: data.response || '응답을 받을 수 없습니다.',
          role: 'assistant',
          timestamp: endTime,
          startTime: startTime,
          responseTime: endTime.getTime() - startTime.getTime()
        };

        setChatState(prev => ({
          ...prev,
          messages: [...prev.messages, assistantMessage],
          isLoading: false
        }));
        
        // 응답 완료 시 입력창으로 포커스 이동
        setTimeout(() => {
          inputRef.current?.focus();
        }, 100);
      }

    } catch (error) {
      console.error('채팅 오류:', error);
      const endTime = new Date();
      const errorMessage: ChatMessage = {
        id: (Date.now() + 1).toString(),
        content: '죄송합니다. 오류가 발생했습니다. 다시 시도해주세요.',
        role: 'assistant',
        timestamp: endTime,
        startTime: startTime,
        responseTime: endTime.getTime() - startTime.getTime()
      };

      setChatState(prev => ({
        ...prev,
        messages: [...prev.messages, errorMessage],
        isLoading: false
      }));
      
      // 오류 발생 시에도 입력창으로 포커스 이동
      setTimeout(() => {
        inputRef.current?.focus();
      }, 100);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  // PDF 텍스트 추출 함수
  const extractPdfText = async (file: File): Promise<string> => {
    try {
      console.log('🔧 PDF.js 설정 확인:', pdfjsLib.GlobalWorkerOptions.workerSrc);
      
      const arrayBuffer = await file.arrayBuffer();
      
      // PDF 문서 로드 (기본 옵션으로 설정)
      const loadingTask = pdfjsLib.getDocument({
        data: arrayBuffer
      });
      
      const pdf = await loadingTask.promise;
      console.log(`📖 PDF 로드 완료: ${pdf.numPages}페이지`);
      
      let fullText = '';
      
      // 모든 페이지에서 텍스트 추출
      for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
        try {
          const page = await pdf.getPage(pageNum);
          const textContent = await page.getTextContent();
          const pageText = textContent.items
            .map((item: any) => {
              if (item.str && typeof item.str === 'string') {
                return item.str;
              }
              return '';
            })
            .filter((text: string) => text.trim().length > 0)
            .join(' ')
            .replace(/\s+/g, ' ')
            .trim();
          if (pageText) {
            fullText += `\n[페이지 ${pageNum}]\n${pageText}\n`;
            console.log(`✅ 페이지 ${pageNum}: ${pageText.length}자 추출`);
          } else {
            console.log(`⚠️ 페이지 ${pageNum}: 텍스트 없음`);
          }
                 } catch (pageError) {
           console.error(`❌ 페이지 ${pageNum} 처리 오류:`, pageError);
           const pageErrorMessage = pageError instanceof Error ? pageError.message : String(pageError);
           fullText += `\n[페이지 ${pageNum}]\n(이 페이지에서 텍스트를 추출할 수 없습니다: ${pageErrorMessage})\n`;
         }
      }
      
      // PDF 문서 정리
      pdf.destroy();
      
      if (fullText.trim()) {
        return fullText.trim();
      } else {
        return `PDF 파일 "${file.name}"에서 읽을 수 있는 텍스트를 찾을 수 없습니다. 이미지 기반 PDF이거나 보호된 문서일 수 있습니다.`;
      }
      
    } catch (error) {
      console.error('❌ PDF 텍스트 추출 실패:', error);
      
      // 오류 타입에 따른 상세 메시지
      const errorMessage = error instanceof Error ? error.message : String(error);
      if (errorMessage.includes('Worker')) {
        return `PDF 처리 환경 오류: ${file.name} - PDF.js Worker 로드에 실패했습니다. 네트워크 환경을 확인해주세요.`;
      } else if (errorMessage.includes('Invalid PDF')) {
        return `PDF 형식 오류: ${file.name} - 손상되었거나 유효하지 않은 PDF 파일입니다.`;
      } else if (errorMessage.includes('Password')) {
        return `PDF 보안 오류: ${file.name} - 패스워드로 보호된 PDF 파일입니다.`;
      } else {
        return `PDF 텍스트 추출 실패: ${file.name} - ${errorMessage || '알 수 없는 오류가 발생했습니다.'}`;
      }
    }
  };

  // 이미지 압축 함수
  const compressImage = async (file: File, maxWidth: number = 800, maxHeight: number = 800, quality: number = 0.8): Promise<File> => {
    return new Promise((resolve) => {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const img = new Image();
      
      img.onload = () => {
        // 원본 크기 계산
        let { width, height } = img;
        
        // 최대 크기에 맞춰 비율 조정
        if (width > height) {
          if (width > maxWidth) {
            height = (height * maxWidth) / width;
            width = maxWidth;
          }
        } else {
          if (height > maxHeight) {
            width = (width * maxHeight) / height;
            height = maxHeight;
          }
        }
        
        // 캔버스 크기 설정
        canvas.width = width;
        canvas.height = height;
        
        // 이미지 그리기
        ctx?.drawImage(img, 0, 0, width, height);
        
        // 압축된 이미지를 Blob으로 변환
        canvas.toBlob((blob) => {
          if (blob) {
            const compressedFile = new File([blob], file.name, {
              type: file.type,
              lastModified: Date.now(),
            });
            resolve(compressedFile);
          } else {
            resolve(file); // 압축 실패 시 원본 반환
          }
        }, file.type, quality);
      };
      
      img.src = URL.createObjectURL(file);
    });
  };

  // 파일 업로드 핸들러
  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (files) {
      const validFiles = Array.from(files).filter(file => 
        file.type.startsWith('image/') || 
        file.type === 'application/pdf' ||
        file.type.startsWith('text/')
      );
      
      // PDF 파일이 있는지 확인
      const hasPdfFiles = validFiles.some(file => file.type === 'application/pdf');
      
      // PDF 파일이 있으면 처리 중 상태를 표시
      if (hasPdfFiles) {
        setChatState(prev => ({
          ...prev,
          isLoading: true
        }));
      }
      
      try {
        // 파일들을 처리
        const processedFiles = await Promise.all(
          validFiles.map(async (file) => {
            // 이미지 파일인 경우 압축 처리
            if (file.type.startsWith('image/')) {
              // 파일 크기가 5MB 이상인 경우에만 압축
              if (file.size > 5 * 1024 * 1024) {
                console.log(`이미지 압축 중: ${file.name} (${(file.size / 1024 / 1024).toFixed(2)}MB)`);
                const compressedFile = await compressImage(file);
                console.log(`이미지 압축 완료: ${compressedFile.name} (${(compressedFile.size / 1024 / 1024).toFixed(2)}MB)`);
                return compressedFile;
              }
              // 5MB 이하인 경우 원본 그대로 사용
              return file;
            }
            // PDF 파일인 경우 텍스트 미리 추출하지 않고 원본만 반환
            // (실제 텍스트 추출은 sendMessage에서 수행)
            return file;
          })
        );
        
        setChatState(prev => ({
          ...prev,
          uploadedFiles: [...prev.uploadedFiles, ...processedFiles],
          isLoading: false
        }));
      } catch (error) {
        console.error('파일 처리 오류:', error);
        setChatState(prev => ({
          ...prev,
          isLoading: false
        }));
      }
    }
  };

  // 파일 삭제 핸들러
  const removeFile = (index: number) => {
    setChatState(prev => ({
      ...prev,
      uploadedFiles: prev.uploadedFiles.filter((_, i) => i !== index)
    }));
  };

  // 설정 토글
  const toggleSettings = () => {
    setChatState(prev => ({ ...prev, showSettings: !prev.showSettings }));
  };

  // 대화 히스토리 초기화
  const clearHistory = () => {
    if (confirm('대화 히스토리를 초기화하시겠습니까?')) {
      setChatState(prev => ({ ...prev, messages: [] }));
    }
  };

  // 폴링 중단 함수
  const stopPolling = () => {
    if (pollIntervalRef.current) {
      clearInterval(pollIntervalRef.current);
      pollIntervalRef.current = null;
      console.log('폴링이 중단되었습니다.');
    }
  };

  // 특정 모델 재평가
  const reEvaluateModel = async (modelName: string) => {
    if (evalStatus.isRunning) {
      showModal('warning', '평가 진행 중', '현재 다른 평가가 진행 중입니다. 완료 후 다시 시도해주세요.');
      return;
    }

    const confirmed = confirm(`${modelName} 모델을 재평가하시겠습니까?\n기존 결과가 덮어쓰여집니다.`);
    if (!confirmed) return;

    // 기존 폴링이 있다면 중단
    stopPolling();

    try {
      const res = await fetch('/api/eval-start', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ 
          models: [modelName],
          reEvaluate: true  // 재평가 플래그
        }),
      });

      const data = await res.json();
      if (res.ok) {
        showModal('info', '재평가 시작', `${modelName} 모델의 재평가가 시작되었습니다.`);
        
        // 백엔드 상태 파일이 생성될 시간을 주기 위해 약간 지연
        setTimeout(async () => {
          await fetchEvalStatus();
          await fetchResults();
          
          // 상태 폴링 시작
          const startPolling = () => {
            pollIntervalRef.current = setInterval(async () => {
              try {
                const statusRes = await fetch('/api/eval-status');
                const statusData = await statusRes.json();
                console.log('폴링 상태 업데이트:', statusData);
                setEvalStatus(statusData);
                
                // 결과도 함께 업데이트
                const resultsRes = await fetch('/api/eval-results');
                const resultsData = await resultsRes.json();
                setResults(resultsData);
                
                // 평가가 완료되거나 중단된 경우 폴링 중단
                if (!statusData.isRunning) {
                  stopPolling();
                  console.log('평가 완료 또는 중단으로 폴링을 중단합니다.');
                }
              } catch (error) {
                console.error('폴링 중 오류:', error);
              }
            }, 3000); // 3초 간격
          };
          
          startPolling();
        }, 500); // 0.5초 지연
        
      } else {
        showModal('error', '재평가 시작 실패', data.error || '알 수 없는 오류가 발생했습니다.');
      }
    } catch (error) {
      showModal('error', '오류 발생', '재평가 시작 중 오류가 발생했습니다.');
      console.error('재평가 시작 오류:', error);
    }
  };

  // 평가 시작
  const startEvaluation = async () => {
    if (selectedModels.length === 0) {
      showModal('warning', '모델 선택 필요', '평가할 모델을 선택해주세요.');
      return;
    }

    // 기존 폴링이 있다면 중단
    stopPolling();

    try {
      const res = await fetch('/api/eval-start', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ models: selectedModels, judgeProvider }),
      });

      const data = await res.json();
      if (res.ok) {
        setShowModelSelector(false);
        
        // 백엔드 상태 파일이 생성될 시간을 주기 위해 약간 지연
        setTimeout(async () => {
          await fetchEvalStatus();
          await fetchResults();
          
          // 상태 폴링 시작
          const startPolling = () => {
            pollIntervalRef.current = setInterval(async () => {
              try {
                const statusRes = await fetch('/api/eval-status');
                const statusData = await statusRes.json();
                console.log('폴링 상태 업데이트:', statusData);
                setEvalStatus(statusData);
                
                // 결과도 함께 업데이트
                const resultsRes = await fetch('/api/eval-results');
                const resultsData = await resultsRes.json();
                setResults(resultsData);
                
                // 평가가 완료되거나 중단된 경우 폴링 중단
                if (!statusData.isRunning) {
                  stopPolling();
                  console.log('평가 완료 또는 중단으로 폴링을 중단합니다.');
                }
              } catch (error) {
                console.error('폴링 중 오류:', error);
              }
            }, 3000); // 3초 간격
          };
          
          startPolling();
        }, 500); // 0.5초 지연
        
      } else {
        showModal('error', '평가 시작 실패', data.error || '알 수 없는 오류가 발생했습니다.');
      }
    } catch (error) {
      showModal('error', '오류 발생', '평가 시작 중 오류가 발생했습니다.');
      console.error('평가 시작 오류:', error);
    }
  };

  // 평가 중단
  const stopEvaluation = async () => {
    try {
      const res = await fetch('/api/eval-stop', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      const data = await res.json();
      if (res.ok) {
        // 폴링 중단
        stopPolling();
        // 상태 업데이트
        await fetchEvalStatus();
        await fetchResults();
      } else {
        showModal('error', '평가 중단 실패', data.error || '알 수 없는 오류가 발생했습니다.');
      }
    } catch (error) {
      showModal('error', '오류 발생', '평가 중단 중 오류가 발생했습니다.');
      console.error('평가 중단 오류:', error);
    }
  };

  useEffect(() => {
    fetchResults();
    fetchAvailableModels();
    fetchEvalStatus();
    
    // 컴포넌트 언마운트 시 폴링 정리
    return () => {
      stopPolling();
    };
  }, []);

  // 결과가 로드된 후 메타데이터 캐시 로드
  useEffect(() => {
    if (Object.keys(results).length > 0) {
      loadModelMetadata();
    }
  }, [results]);

  // 평가 완료 감지 및 자동 닫힘
  useEffect(() => {
    if (!evalStatus.isRunning && !evalStatus.isStopped && evalStatus.progress === 100 && evalStatus.endTime) {
      // 평가 완료 시 5초 후 완료 메시지 자동 닫힘
      const timer = setTimeout(() => {
        setEvalStatus(prev => ({ ...prev, progress: 0, endTime: undefined }));
      }, 5000);
      
      return () => clearTimeout(timer);
    }
  }, [evalStatus.isRunning, evalStatus.isStopped, evalStatus.progress, evalStatus.endTime]);

  // 다크모드 감지 및 이벤트 리스너
  useEffect(() => {
    if (typeof window === 'undefined') return;
    const checkDark = () => setIsDark(document.documentElement.classList.contains('dark'));
    checkDark();
    window.addEventListener('themeChange', checkDark);
    return () => window.removeEventListener('themeChange', checkDark);
  }, []);

  // 메시지 변경 시 스크롤을 맨 아래로
  useEffect(() => {
    if (messagesEndRef.current) {
      messagesEndRef.current.scrollIntoView({ behavior: 'smooth' });
    }
  }, [chatState.messages]);

  // 채팅창이 열릴 때 입력창에 포커스 설정
  useEffect(() => {
    if (chatState.isOpen && inputRef.current) {
      setTimeout(() => {
        inputRef.current?.focus();
      }, 200); // 모달 애니메이션 완료 후 포커스
    }
  }, [chatState.isOpen]);

  // 모델 목록 추출
  const modelIds = Object.keys(results);
  // 카테고리별 점수 집계
  let leaderboard: LeaderboardRow[] = modelIds.map((modelId) => {
    const modelResult = results[modelId];
    const scores = EVAL_CATEGORIES.map((cat) => modelResult?.[cat.key]?.score ?? 0);
    const total = scores.reduce((a, b) => a + b, 0);
    const avg = scores.length ? Math.round((total / scores.length) * 100) / 100 : 0;
    return {
      modelId,
      modelName: modelId,
      accuracy: avg,
      ...Object.fromEntries(EVAL_CATEGORIES.map((cat, i) => [cat.key, scores[i]])),
      total,
      avg,
    };
  });

  // 정렬
  leaderboard = leaderboard.sort((a, b) => {
    // 메타데이터 기반 정렬
    if (sortKey === 'avgResponseTime') {
      const aTime = results[a.modelId] ? calculateAverageResponseTime(results[a.modelId]) : 0;
      const bTime = results[b.modelId] ? calculateAverageResponseTime(results[b.modelId]) : 0;
      return sortOrder === 'asc' ? aTime - bTime : bTime - aTime;
    }
    
    if (sortKey === 'parameters') {
      const aMetadata = modelMetadataCache[a.modelId] || getModelMetadata(a.modelName);
      const bMetadata = modelMetadataCache[b.modelId] || getModelMetadata(b.modelName);
      const aParams = aMetadata?.parameters || '';
      const bParams = bMetadata?.parameters || '';
      
      // 파라미터 문자열을 숫자로 변환 (예: "7B" -> 7, "1.5B" -> 1.5)
      const parseParams = (params: string): number => {
        if (!params) return 0;
        const match = params.match(/^(\d+(?:\.\d+)?)([BM])?/i);
        if (!match) return 0;
        const num = parseFloat(match[1]);
        const unit = match[2]?.toUpperCase();
        if (unit === 'B') return num * 1000; // B는 M 단위로 변환
        return num; // M 단위 또는 단위 없음
      };
      
      const aParamsNum = parseParams(aParams);
      const bParamsNum = parseParams(bParams);
      return sortOrder === 'asc' ? aParamsNum - bParamsNum : bParamsNum - aParamsNum;
    }
    
    if (sortKey === 'contextLength') {
      const aMetadata = modelMetadataCache[a.modelId] || getModelMetadata(a.modelName);
      const bMetadata = modelMetadataCache[b.modelId] || getModelMetadata(b.modelName);
      const aContext = aMetadata?.contextLength || 0;
      const bContext = bMetadata?.contextLength || 0;
      return sortOrder === 'asc' ? aContext - bContext : bContext - aContext;
    }
    
    // 기본 숫자 정렬
    const aVal = Number(a[sortKey]) || 0;
    const bVal = Number(b[sortKey]) || 0;
    if (sortOrder === 'asc') return aVal - bVal;
    return bVal - aVal;
  });

  // 정렬 핸들러
  const handleSort = (key: string) => {
    if (sortKey === key) {
      setSortOrder(sortOrder === 'asc' ? 'desc' : 'asc');
    } else {
      setSortKey(key);
      setSortOrder('desc');
    }
  };

  // 정렬 아이콘
  const sortIcon = (key: string) => {
    if (sortKey !== key) return null;
    return sortOrder === 'asc' ? <span>▲</span> : <span>▼</span>;
  };

  // Bar Chart 데이터 준비
  const chartLabels = leaderboard.map((row) => row.modelName);
  const datasets = [
    ...EVAL_CATEGORIES.map((cat, i) => ({
      label: cat.label,
      data: leaderboard.map((row) => row[cat.key]),
      backgroundColor: [
        'rgba(59,130,246,0.5)', // blue-400, 투명도 ↑
        'rgba(16,185,129,0.5)', // green
        'rgba(234,179,8,0.5)',  // yellow
        'rgba(239,68,68,0.5)',  // red
        'rgba(168,85,247,0.5)', // purple
      ][i],
      borderRadius: 6,
    })),
    {
      label: '평균',
      data: leaderboard.map((row) => row.avg),
      backgroundColor: 'rgba(37,99,235,1)', // blue-600, 불투명
      borderColor: '#1d4ed8', // blue-700
      borderWidth: 2,
      borderRadius: 6,
      // barThickness: 32, // 더 두껍게 (선택)
    },
  ];
  const barData = {
    labels: chartLabels,
    datasets,
  };

  function getCssVar(name: string, fallback: string) {
    if (typeof window === 'undefined') return fallback;
    const value = getComputedStyle(document.documentElement).getPropertyValue(name).trim();
    return value || fallback;
  }

  const axisColor = isDark
    ? getCssVar('--chart-text-dark', '#fff')
    : getCssVar('--chart-text-light', '#222');
  const gridColor = isDark
    ? getCssVar('--chart-grid-dark', 'rgba(255,255,255,0.15)')
    : getCssVar('--chart-grid-light', 'rgba(0,0,0,0.08)');

  const barOptions = {
    responsive: true,
    plugins: {
      legend: {
        display: true,
        position: 'top' as const,
        labels: {
          color: axisColor,
        },
      },
      tooltip: {
        enabled: true,
        bodyColor: axisColor,
        titleColor: axisColor,
        backgroundColor: isDark ? '#222' : '#fff',
        borderColor: isDark ? '#444' : '#eee',
        borderWidth: 1,
      },
    },
    scales: {
      x: {
        ticks: { color: axisColor },
        grid: { color: gridColor },
      },
      y: {
        beginAtZero: true,
        max: 10,
        ticks: { color: axisColor },
        grid: { color: gridColor },
      },
    },
  };

  if (isDark === null) return null;
  return (
    <div className="w-full min-h-screen flex">
      {/* 사이드바 */}
      <div className={`${sidebarCollapsed ? 'w-16' : 'w-80'} transition-all duration-300 bg-[var(--sidebar-bg)] border-r border-[var(--card-border)] flex flex-col`}>
        {/* 사이드바 헤더 */}
        <div className="p-4 border-b border-gray-200 dark:border-gray-700 flex items-center justify-between">
          {!sidebarCollapsed && (
            <h2 className="font-semibold text-gray-900 dark:text-gray-100">모델 관리</h2>
          )}
          <button
            onClick={() => setSidebarCollapsed(!sidebarCollapsed)}
            className="p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
          >
            <svg
              className={`w-5 h-5 text-gray-600 dark:text-gray-400 transition-transform duration-200 ${
                sidebarCollapsed ? 'rotate-180' : ''
              }`}
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
            </svg>
          </button>
        </div>

        {/* 사이드바 내용 */}
        {!sidebarCollapsed && (
          <div className="flex-1 overflow-y-auto">
            {/* 탭 메뉴 */}
            <div className="p-4 space-y-2">
              <button
                onClick={() => setActiveTab('huggingface')}
                className={`w-full text-left px-3 py-2 rounded-lg transition-colors ${
                  activeTab === 'huggingface'
                    ? 'bg-[var(--accent-blue)] text-white'
                    : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700'
                }`}
              >
                🤗 Hugging Face
              </button>
              <button
                onClick={() => setActiveTab('ollama')}
                className={`w-full text-left px-3 py-2 rounded-lg transition-colors ${
                  activeTab === 'ollama'
                    ? 'bg-[var(--accent-blue)] text-white'
                    : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700'
                }`}
              >
                🦙 Ollama
              </button>
              <button
                onClick={() => setActiveTab('upload')}
                className={`w-full text-left px-3 py-2 rounded-lg transition-colors ${
                  activeTab === 'upload'
                    ? 'bg-[var(--accent-blue)] text-white'
                    : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700'
                }`}
              >
                📤 업로드
              </button>
              <button
                onClick={() => setActiveTab('manage')}
                className={`w-full text-left px-3 py-2 rounded-lg transition-colors ${
                  activeTab === 'manage'
                    ? 'bg-[var(--accent-blue)] text-white'
                    : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700'
                }`}
              >
                ⚙️ 관리
              </button>
            </div>

            {/* 탭 내용 */}
            <div className="p-4">
              {activeTab === 'huggingface' && (
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                      모델 검색
                    </label>
                    <input
                      type="text"
                      value={searchQuery}
                      onChange={(e) => setSearchQuery(e.target.value)}
                      onKeyPress={(e) => e.key === 'Enter' && searchHuggingFaceModels(searchQuery)}
                      placeholder="모델 이름을 입력하세요"
                      className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-gray-100"
                    />
                    <button
                      onClick={() => searchHuggingFaceModels(searchQuery)}
                      disabled={searchLoading}
                      className="w-full mt-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed"
                    >
                      {searchLoading ? '검색 중...' : '검색'}
                    </button>
                  </div>

                  {huggingFaceModels.length > 0 && (
                    <div className="space-y-2 max-h-96 overflow-y-auto">
                      {huggingFaceModels.map((model) => (
                        <div key={model.id} className="p-3 border border-gray-200 dark:border-gray-600 rounded-lg">
                          <div className="flex items-center justify-between mb-2">
                            <h4 className="font-medium text-gray-900 dark:text-gray-100 text-sm">
                              {model.id}
                            </h4>
                            <button
                              onClick={() => downloadHuggingFaceModel(model.id)}
                              disabled={downloadStatus.some(d => d.model === model.id)}
                              className="px-2 py-1 text-xs bg-green-600 text-white rounded hover:bg-green-700 disabled:bg-gray-400 disabled:cursor-not-allowed"
                            >
                              다운로드
                            </button>
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            <p>다운로드: {model.downloads?.toLocaleString() || 0}</p>
                            <p>좋아요: {model.likes?.toLocaleString() || 0}</p>
                            {model.pipeline_tag && <p>유형: {model.pipeline_tag}</p>}
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}

              {activeTab === 'ollama' && (
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                      모델 검색
                    </label>
                    <input
                      type="text"
                      value={searchQuery}
                      onChange={(e) => setSearchQuery(e.target.value)}
                      onKeyPress={(e) => e.key === 'Enter' && searchOllamaModels(searchQuery)}
                      placeholder="모델 이름을 입력하세요"
                      className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-gray-100"
                    />
                    <button
                      onClick={() => searchOllamaModels(searchQuery)}
                      disabled={searchLoading}
                      className="w-full mt-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed"
                    >
                      {searchLoading ? '검색 중...' : '검색'}
                    </button>
                  </div>

                  {ollamaModels.length > 0 && (
                    <div className="space-y-2 max-h-96 overflow-y-auto">
                      {ollamaModels.map((model) => (
                        <div key={model} className="p-3 border border-gray-200 dark:border-gray-600 rounded-lg">
                          <div className="flex items-center justify-between">
                            <h4 className="font-medium text-gray-900 dark:text-gray-100 text-sm">
                              {model}
                            </h4>
                            <button
                              onClick={() => downloadOllamaModel(model)}
                              disabled={downloadStatus.some(d => d.model === model)}
                              className="px-2 py-1 text-xs bg-green-600 text-white rounded hover:bg-green-700 disabled:bg-gray-400 disabled:cursor-not-allowed"
                            >
                              다운로드
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}

              {activeTab === 'upload' && (
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                      모델 파일 업로드
                    </label>
                    <input
                      type="file"
                      multiple
                      accept=".bin,.safetensors,.gguf,.ggml"
                      onChange={(e) => e.target.files && handleModelUpload(e.target.files)}
                      className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-gray-100"
                    />
                    <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                      지원 형식: .bin, .safetensors, .gguf, .ggml
                    </p>
                  </div>

                  {uploadingFiles.length > 0 && (
                    <div className="space-y-2">
                      <h4 className="font-medium text-gray-900 dark:text-gray-100">업로드 중인 파일:</h4>
                      {uploadingFiles.map((file, index) => (
                        <div key={index} className="p-2 border border-gray-200 dark:border-gray-600 rounded">
                          <p className="text-sm text-gray-700 dark:text-gray-300">{file.name}</p>
                          <p className="text-xs text-gray-500">크기: {(file.size / 1024 / 1024).toFixed(2)}MB</p>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}

              {activeTab === 'manage' && (
                <div className="space-y-4">
                  <h4 className="font-medium text-gray-900 dark:text-gray-100">설치된 모델</h4>
                  <div className="space-y-2 max-h-128 overflow-y-auto">
                    {availableModels.map((model) => (
                      <div key={model.name} className="p-3 border border-gray-200 dark:border-gray-600 rounded-lg">
                        <div className="flex items-center justify-between mb-2">
                          <h5 className="font-medium text-gray-900 dark:text-gray-100 text-sm">
                            {model.name}
                          </h5>
                          <button
                            onClick={() => deleteModel(model.name)}
                            className="px-2 py-1 text-xs bg-red-600 text-white rounded hover:bg-red-700"
                          >
                            삭제
                          </button>
                        </div>
                        <div className="text-xs text-gray-500 dark:text-gray-400">
                          <p>유형: {model.type}</p>
                          <p>제공자: {model.provider}</p>
                          {model.size > 0 && <p>크기: {(model.size / 1024 / 1024 / 1024).toFixed(1)}GB</p>}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>

            {/* 다운로드 상태 */}
            {downloadStatus.length > 0 && (
              <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                <h4 className="font-medium text-gray-900 dark:text-gray-100 mb-2">다운로드 상태</h4>
                <div className="space-y-2">
                  {downloadStatus.map((status, index) => (
                    <div key={index} className="text-sm">
                      <div className="flex justify-between items-center">
                        <span className="text-gray-700 dark:text-gray-300 truncate">{status.model}</span>
                        <span className={`text-xs ${
                          status.status === 'completed' ? 'text-green-600' :
                          status.status === 'error' ? 'text-red-600' :
                          'text-blue-600'
                        }`}>
                          {status.status === 'completed' ? '완료' :
                           status.status === 'error' ? '오류' :
                           '다운로드 중'}
                        </span>
                      </div>
                      {status.status === 'downloading' && (
                        <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2 mt-1">
                          <div
                            className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                            style={{ width: `${status.progress}%` }}
                          ></div>
                        </div>
                      )}
                      {status.error && (
                        <p className="text-xs text-red-600 mt-1">{status.error}</p>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>
        )}
      </div>

      {/* 메인 컨텐츠 */}
      <div className="flex-1 py-6 px-4 overflow-x-auto">
        <div className="max-w-[1900px] mx-auto w-full">
      <style jsx>{`
        .leaderboard-row:hover {
          background-color: rgba(59, 130, 246, 0.05);
        }
        .leaderboard-row-first {
          background-color: rgba(255, 215, 0, 0.1) !important;
        }
        .leaderboard-row-first:hover {
          background-color: rgba(255, 215, 0, 0.15) !important;
        }
        @media (max-width: 768px) {
          .overflow-x-auto {
            -webkit-overflow-scrolling: touch;
          }
        }
      `}</style>
      <div className="flex justify-between items-center mb-4">
        <h2 className="font-bold">Leaderboard</h2>
        <div className="flex gap-2">
          <button
            onClick={() => setShowModelSelector(true)}
            disabled={evalStatus.isRunning}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed"
            style={{ color: 'white' }}
          >
            새 평가 시작
          </button>
          <button
            onClick={fetchResults}
            className="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700"
            style={{ color: 'white' }}
          >
            결과 새로고침
          </button>

        </div>
      </div>

      {/* 평가 방법 및 주요 기능 설명 */}
      <div className="mb-8 grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* 평가 방법 설명 */}
        <div className="bg-white dark:bg-gray-900 rounded-lg p-6 shadow-sm border border-gray-200 dark:border-gray-700">
          <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-gray-100 flex items-center gap-2">
            <svg className="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
            평가 방법론
          </h3>
          <div className="space-y-4 text-sm text-gray-700 dark:text-gray-300">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg border border-blue-200 dark:border-blue-800">
                <h4 className="font-semibold text-blue-800 dark:text-blue-200 mb-2">🤖 LLM as Judge</h4>
                <p className="text-xs text-blue-700 dark:text-blue-300">
                  AI 모델이 다른 모델의 응답을 평가하는 방식입니다. 확장 가능하고 일관된 평가가 가능하지만, 
                  편향성과 신뢰성 문제가 있어 다중 샘플 평가와 확률 분포 기반 평가를 권장합니다.
                </p>
              </div>
              
              <div className="bg-green-50 dark:bg-green-900/20 p-4 rounded-lg border border-green-200 dark:border-green-800">
                <h4 className="font-semibold text-green-800 dark:text-green-200 mb-2">📊 전통적 벤치마크</h4>
                <p className="text-xs text-green-700 dark:text-green-300">
                  MMLU(종합 지식), HellaSwag(상식 추론), TruthfulQA(사실성), HumanEval(코딩) 등 
                  검증된 벤치마크를 통해 객관적 성능을 측정합니다.
                </p>
              </div>
              
              <div className="bg-purple-50 dark:bg-purple-900/20 p-4 rounded-lg border border-purple-200 dark:border-purple-800">
                <h4 className="font-semibold text-purple-800 dark:text-purple-200 mb-2">👥 인간 평가</h4>
                <p className="text-xs text-purple-700 dark:text-purple-300">
                  Chatbot Arena의 Elo 레이팅 시스템처럼 실제 사용자의 선호도를 반영한 평가 방법으로, 
                  실용적 성능을 측정합니다.
                </p>
              </div>
              
              <div className="bg-orange-50 dark:bg-orange-900/20 p-4 rounded-lg border border-orange-200 dark:border-orange-800">
                <h4 className="font-semibold text-orange-800 dark:text-orange-200 mb-2">🔧 전문화된 평가</h4>
                <p className="text-xs text-orange-700 dark:text-orange-300">
                  RAG 시스템용 RAGAs, AI 에이전트용 AgentBench, 멀티모달 평가용 FID/CLIPScore 등 
                  특정 도메인에 특화된 평가 프레임워크를 활용합니다.
                </p>
              </div>
            </div>
            
            <div className="mt-4 space-y-2">
              <p className="text-sm text-gray-700 dark:text-gray-300">
                ℹ️ 본 시스템은 대규모 언어 모델(LLM)의 성능을 다각도로 평가하기 위해 여러 평가 방법론을 활용합니다.
              </p>
              <p className="text-sm text-gray-700 dark:text-gray-300">
                ⚠️ LLM의 개방형 특성으로 인해 맥락 의존적 평가, 다중 유효 답변 처리, 데이터 오염 방지 등의 문제로 평가 결과가 불안정할 수 있습니다.
              </p>
            </div>
          </div>
        </div>

        {/* 주요 기능 설명 */}
        <div className="bg-white dark:bg-gray-900 rounded-lg p-6 shadow-sm border border-gray-200 dark:border-gray-700">
          <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-gray-100 flex items-center gap-2">
            <svg className="w-5 h-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10" />
            </svg>
            주요 기능
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 text-sm">
            <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 dark:text-gray-100 mb-2 flex items-center gap-2">
                <span className="text-blue-600">📈</span>
                종합 평가 시스템
              </h4>
              <p className="text-gray-700 dark:text-gray-300 text-xs">
                수학, 글쓰기, 코딩, 추론, 문법 등 5개 핵심 영역에서 모델 성능을 평가하고 리더보드로 순위를 매깁니다.
              </p>
            </div>
            
            <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 dark:text-gray-100 mb-2 flex items-center gap-2">
                <span className="text-green-600">💬</span>
                실시간 채팅 테스트
              </h4>
              <p className="text-gray-700 dark:text-gray-300 text-xs">
                평가된 모델과 직접 대화하여 실제 성능을 체험할 수 있습니다. 파일 업로드와 도구 사용도 지원합니다.
              </p>
            </div>
            
            <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 dark:text-gray-100 mb-2 flex items-center gap-2">
                <span className="text-purple-600">⚙️</span>
                모델 메타데이터
              </h4>
              <p className="text-gray-700 dark:text-gray-300 text-xs">
                파라미터 수, 컨텍스트 길이, 도구 지원, 이미지 지원 등 모델의 상세 정보를 제공합니다.
              </p>
            </div>
            
            <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 dark:text-gray-100 mb-2 flex items-center gap-2">
                <span className="text-orange-600">⏱️</span>
                성능 분석
              </h4>
              <p className="text-gray-700 dark:text-gray-300 text-xs">
                평균 응답 시간, 처리 속도 등 실용적 성능 지표를 측정하여 실제 사용 환경에서의 효율성을 평가합니다.
              </p>
            </div>
            
            <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 dark:text-gray-100 mb-2 flex items-center gap-2">
                <span className="text-red-600">🔄</span>
                동적 평가
              </h4>
              <p className="text-gray-700 dark:text-gray-300 text-xs">
                새로운 모델 추가, 재평가, 실시간 진행 상황 모니터링 등을 통해 지속적인 성능 추적이 가능합니다.
              </p>
            </div>
            
            <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 dark:text-gray-100 mb-2 flex items-center gap-2">
                <span className="text-indigo-600">📊</span>
                시각화 및 비교
              </h4>
              <p className="text-gray-700 dark:text-gray-300 text-xs">
                차트와 표를 통해 모델 간 성능을 직관적으로 비교하고, 다양한 기준으로 정렬하여 분석할 수 있습니다.
              </p>
            </div>
          </div>
        </div>
      </div>

      <div className="mb-6 text-gray-700 dark:text-gray-300 text-base">
        ※ 사용 가능한 공개(Foundation) 모델 평가 결과를 리더보드로 표시합니다.
        <br />
      </div>

      {/* 평가 진행 상태 */}
      {evalStatus.isRunning && !evalStatus.isStopped && (
        <div className="mb-6 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
          <div className="flex items-center justify-between mb-2">
            <h3 className="text-lg font-semibold text-blue-800 dark:text-blue-200">
              {(evalStatus as any).reEvaluate ? '재평가 진행 중' : '평가 진행 중'}
            </h3>
            <div className="flex items-center gap-3">
                                      <span className="text-blue-600 dark:text-blue-300 font-bold">{evalStatus.progress.toFixed(1)}%</span>
              <button
                onClick={stopEvaluation}
                className="px-3 py-1 bg-red-600 text-white text-sm rounded hover:bg-red-700 transition-colors"
              >
                중단
              </button>
            </div>
          </div>
          <div className="w-full bg-blue-200 dark:bg-blue-800 rounded-full h-3 mb-3">
            <div
              className="bg-blue-600 h-3 rounded-full transition-all duration-500 flex items-center justify-end pr-2"
              style={{ width: `${evalStatus.progress}%` }}
            >
              {evalStatus.progress > 10 && (
                                      <span className="text-white text-xs font-medium">{evalStatus.progress.toFixed(1)}%</span>
              )}
            </div>
          </div>
          <div className="text-sm text-blue-700 dark:text-blue-300 space-y-1">
            {(evalStatus as any).reEvaluate && (evalStatus as any).targetModels && (
              <div className="flex justify-between">
                <span>재평가 대상:</span>
                <strong>{(evalStatus as any).targetModels.join(', ')}</strong>
              </div>
            )}
            {evalStatus.currentModel && (
              <div className="flex justify-between">
                <span>현재 모델:</span>
                <strong>{evalStatus.currentModel}</strong>
              </div>
            )}
            {evalStatus.currentCategory && (
              <div className="flex justify-between">
                <span>현재 카테고리:</span>
                <strong>{evalStatus.currentCategory}</strong>
              </div>
            )}
            <div className="flex justify-between">
              <span>완료된 모델:</span>
              <strong>{evalStatus.completedModels} / {evalStatus.totalModels}</strong>
            </div>
            {evalStatus.startTime && (
              <div className="flex justify-between">
                <span>시작 시간:</span>
                <span>{new Date(evalStatus.startTime).toLocaleString()}</span>
              </div>
            )}
          </div>
        </div>
      )}

      {/* 평가 완료 상태 */}
      {!evalStatus.isRunning && !evalStatus.isStopped && evalStatus.progress === 100 && evalStatus.endTime && (
        <div className="mb-6 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
          <div className="flex items-center justify-between mb-2">
            <h3 className="text-lg font-semibold text-green-800 dark:text-green-200">평가 완료</h3>
            <span className="text-green-600 dark:text-green-300">100%</span>
          </div>
          <div className="text-sm text-green-700 dark:text-green-300">
            <p>모든 모델의 평가가 성공적으로 완료되었습니다.</p>
            <p>완료 시간: {new Date(evalStatus.endTime).toLocaleString()}</p>
            <p>평가된 모델: {evalStatus.completedModels} / {evalStatus.totalModels}</p>
          </div>
        </div>
      )}

      {/* 평가 중단됨 상태 */}
      {evalStatus.isStopped && (
        <div className="mb-6 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
          <div className="flex items-center justify-between mb-2">
            <h3 className="text-lg font-semibold text-yellow-800 dark:text-yellow-200">평가 중단됨</h3>
                                  <span className="text-yellow-600 dark:text-yellow-300">{evalStatus.progress.toFixed(1)}%</span>
          </div>
          <div className="text-sm text-yellow-700 dark:text-yellow-300">
            <p>평가가 사용자에 의해 중단되었습니다.</p>
            {evalStatus.stoppedAt && (
              <p>중단 시간: {new Date(evalStatus.stoppedAt).toLocaleString()}</p>
            )}
            <p>완료된 모델: {evalStatus.completedModels} / {evalStatus.totalModels}</p>
          </div>
        </div>
      )}

      {/* 모델 선택 모달 */}
      {showModelSelector && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-2xl w-full mx-4 max-h-[80vh] overflow-y-auto">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-xl font-bold">평가할 모델 선택</h3>
              <button
                onClick={() => setShowModelSelector(false)}
                className="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
              >
                ✕
              </button>
            </div>
            
            <div className="mb-4">
              <div className="flex justify-between items-center mb-2">
                <p className="text-sm text-gray-600 dark:text-gray-400">
                  사용 가능한 모델: {availableModels.length}개
                  <span className="ml-2 text-xs">
                    (Ollama: {availableModels.filter(m => m.type === 'ollama').length}개, 
                     VLLM: {availableModels.filter(m => m.type === 'vllm').length}개)
                  </span>
                </p>
              </div>
              <div className="flex gap-2 mb-4">
                <button
                  onClick={() => setSelectedModels(availableModels.map(m => m.name))}
                  className="px-3 py-1 text-sm bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 rounded"
                >
                  전체 선택
                </button>
                <button
                  onClick={() => setSelectedModels([])}
                  className="px-3 py-1 text-sm bg-gray-100 dark:bg-gray-700 text-gray-800 dark:text-gray-200 rounded"
                >
                  전체 해제
                </button>
                <button
                  onClick={() => setSelectedModels(availableModels.filter(m => m.type === 'ollama').map(m => m.name))}
                  className="px-3 py-1 text-sm bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200 rounded"
                >
                  Ollama만
                </button>
                <button
                  onClick={() => setSelectedModels(availableModels.filter(m => m.type === 'vllm').map(m => m.name))}
                  className="px-3 py-1 text-sm bg-purple-100 dark:bg-purple-900 text-purple-800 dark:text-purple-200 rounded"
                >
                  VLLM만
                </button>
              </div>
            </div>

            <div className="space-y-2 mb-6 max-h-60 overflow-y-auto">
              {availableModels.map((model) => (
                <label key={model.name} className="flex items-center space-x-3 p-2 hover:bg-gray-50 dark:hover:bg-gray-700 rounded">
                  <input
                    type="checkbox"
                    checked={selectedModels.includes(model.name)}
                    onChange={(e) => {
                      if (e.target.checked) {
                        setSelectedModels([...selectedModels, model.name]);
                      } else {
                        setSelectedModels(selectedModels.filter(m => m !== model.name));
                      }
                    }}
                    className="rounded"
                  />
                  <div className="flex-1">
                    <div className="flex items-center gap-2">
                      <span className="font-medium">{model.name}</span>
                      <span className={`px-2 py-0.5 text-xs rounded-full ${
                        model.type === 'ollama' 
                          ? 'bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200'
                          : 'bg-purple-100 dark:bg-purple-900 text-purple-800 dark:text-purple-200'
                      }`}>
                        {model.provider}
                      </span>
                    </div>
                    <div className="text-sm text-gray-500 dark:text-gray-400">
                      <div className="flex flex-wrap gap-2 mt-1">
                        {model.type === 'ollama' && model.size > 0 && (
                          <span className="bg-gray-100 dark:bg-gray-700 px-2 py-0.5 rounded text-xs">
                            {(model.size / 1024 / 1024 / 1024).toFixed(1)}GB
                          </span>
                        )}
                        {model.parameters && (
                          <span className="bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 px-2 py-0.5 rounded text-xs">
                            {model.parameters}
                          </span>
                        )}
                        {model.context_length && (
                          <span className="bg-orange-100 dark:bg-orange-900 text-orange-800 dark:text-orange-200 px-2 py-0.5 rounded text-xs">
                            {model.context_length.toLocaleString()}ctx
                          </span>
                        )}
                      </div>
                    </div>
                  </div>
                </label>
              ))}
            </div>

            {/* Judge 모델 제공자 선택 */}
            <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-700 rounded-lg">
              <h4 className="text-sm font-semibold mb-3 text-gray-700 dark:text-gray-300">평가 Judge 모델 선택</h4>
              <div className="flex gap-3">
                <label className="flex items-center cursor-pointer">
                  <input
                    type="radio"
                    name="judgeProvider"
                    value="ollama"
                    checked={judgeProvider === 'ollama'}
                    onChange={(e) => setJudgeProvider(e.target.value as 'ollama' | 'openai' | 'vllm')}
                    className="mr-2"
                  />
                  <span className="flex items-center gap-2">
                    <span className="px-2 py-0.5 text-xs rounded-full bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200">
                      Ollama
                    </span>
                    <span className="text-sm text-gray-600 dark:text-gray-400">로컬 모델 (기본)</span>
                  </span>
                </label>
                <label className="flex items-center cursor-pointer">
                  <input
                    type="radio"
                    name="judgeProvider"
                    value="openai"
                    checked={judgeProvider === 'openai'}
                    onChange={(e) => setJudgeProvider(e.target.value as 'ollama' | 'openai' | 'vllm')}
                    className="mr-2"
                  />
                  <span className="flex items-center gap-2">
                    <span className="px-2 py-0.5 text-xs rounded-full bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200">
                      OpenAI
                    </span>
                    <span className="text-sm text-gray-600 dark:text-gray-400">GPT 모델</span>
                  </span>
                </label>
                <label className="flex items-center cursor-pointer">
                  <input
                    type="radio"
                    name="judgeProvider"
                    value="vllm"
                    checked={judgeProvider === 'vllm'}
                    onChange={(e) => setJudgeProvider(e.target.value as 'ollama' | 'openai' | 'vllm')}
                    className="mr-2"
                  />
                  <span className="flex items-center gap-2">
                    <span className="px-2 py-0.5 text-xs rounded-full bg-purple-100 dark:bg-purple-900 text-purple-800 dark:text-purple-200">
                      VLLM
                    </span>
                    <span className="text-sm text-gray-600 dark:text-gray-400">고성능 서버</span>
                  </span>
                </label>
              </div>
              <p className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                선택한 제공자의 Judge 모델이 평가에 사용됩니다. 실패 시 다른 제공자로 자동 전환됩니다.
              </p>
            </div>

            <div className="flex justify-end gap-2">
              <button
                onClick={() => setShowModelSelector(false)}
                className="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600"
              >
                취소
              </button>
              <button
                onClick={startEvaluation}
                disabled={selectedModels.length === 0}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed"
              >
                평가 시작 ({selectedModels.length}개 모델)
              </button>
            </div>
          </div>
        </div>
      )}
      {/* Bar Chart 조건부 렌더링 및 스크롤 대응 */}
      {leaderboard.length > 0 && (
        <div className="mb-8 bg-white dark:bg-gray-900 rounded-lg p-6 shadow-sm border border-gray-200 dark:border-gray-700">
          <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-gray-100">평가 점수 차트</h3>
          <div className="w-full" style={{ height: '400px' }}>
            <Bar
              key={isDark ? 'dark' : 'light'}
              data={barData}
              options={{
                ...barOptions,
                maintainAspectRatio: false,
                scales: {
                  ...barOptions.scales,
                  x: {
                    ...barOptions.scales.x,
                    ticks: {
                      ...barOptions.scales.x.ticks,
                      maxRotation: 45,
                      minRotation: 0
                    }
                  }
                }
              }}
            />
          </div>
        </div>
      )}
      <div className="overflow-x-auto bg-[var(--card-bg)]">
        <table className="w-full border text-center text-sm min-w-[1000px]">
            <thead>
            <tr>
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2 cursor-pointer hover:bg-gray-300 transition w-16" onClick={() => handleSort('rank')}>순위</th>
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-3 py-2 cursor-pointer hover:bg-gray-300 transition text-left min-w-[120px]" onClick={() => handleSort('modelName')}>모델 정보{sortIcon('modelName')}</th>
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2 cursor-pointer hover:bg-gray-300 transition" onClick={() => handleSort('avgResponseTime')}>
                <div className="flex items-center justify-center gap-1">
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                  </svg>
                  응답시간{sortIcon('avgResponseTime')}
                </div>
              </th>
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2">
                <div className="flex items-center justify-center gap-1">
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                  </svg>
                  도구
                </div>
              </th>
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2">
                <div className="flex items-center justify-center gap-1">
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                  </svg>
                  이미지
                </div>
              </th>
              {EVAL_CATEGORIES.map((cat) => (
                <th
                  key={cat.key}
                  className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2 cursor-pointer hover:bg-gray-300 transition"
                  onClick={() => handleSort(cat.key)}
                >
                  {cat.label}{sortIcon(cat.key)}
                </th>
              ))}
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2 cursor-pointer hover:bg-gray-300 transition" onClick={() => handleSort('total')}>총점{sortIcon('total')}</th>
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2 cursor-pointer hover:bg-gray-300 transition" onClick={() => handleSort('avg')}>평균{sortIcon('avg')}</th>
              <th className="text-gray-900 dark:text-gray-100 font-bold border px-2 py-2">작업</th>
            </tr>
          </thead>
          <tbody>
            {leaderboard.map((row, idx) => {
              const modelData = availableModels.find(m => m.name === row.modelId);
              const metadata = modelMetadataCache[row.modelId] || getModelMetadata(row.modelName);
              const avgResponseTime = results[row.modelId] ? calculateAverageResponseTime(results[row.modelId]) : 0;
              
              return (
                <tr
                  key={row.modelId}
                  className={
                    idx === 0
                      ? 'leaderboard-row leaderboard-row-first'
                      : 'leaderboard-row'
                  }
                >
                  <td className="border px-2 py-2">{idx + 1}</td>
                  <td className="border px-3 py-2 text-left">
                    <div className="flex flex-col gap-1">
                      <div className="flex items-center gap-2">
                        <span className="font-medium">{row.modelName}</span>
                        {results[row.modelId] && Object.values(results[row.modelId])[0] && (
                          <span className={`px-1.5 py-0.5 text-xs rounded-full ${
                            (Object.values(results[row.modelId])[0] as any)?.model_type === 'ollama'
                              ? 'bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200'
                              : 'bg-purple-100 dark:bg-purple-900 text-purple-800 dark:text-purple-200'
                          }`}>
                            {(Object.values(results[row.modelId])[0] as any)?.model_type === 'ollama' ? 'Ollama' : 'VLLM'}
                          </span>
                        )}
                      </div>
                      {/* 모델 패밀리, 제공자, 파라미터, 컨텍스트 정보를 한 줄에 표시 */}
                      {metadata && (
                        <div className="text-xs text-gray-500 dark:text-gray-400 flex items-center gap-2" title={`출처: ${metadata.source}`}>
                          <span>{metadata.family} by {metadata.provider}</span>
                          {metadata.source === 'ollama-api' && (
                            <span className="text-orange-500" title="Ollama API에서 동적으로 가져온 정보">*</span>
                          )}
                          {metadata.parameters && (
                            <span className="bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 px-1.5 py-0.5 rounded">
                              {metadata.parameters}
                            </span>
                          )}
                          {metadata.contextLength && (
                            <span className="bg-orange-100 dark:bg-orange-900 text-orange-800 dark:text-orange-200 px-1.5 py-0.5 rounded">
                              {(metadata.contextLength / 1024).toFixed(0)}K
                            </span>
                          )}
                        </div>
                      )}
                    </div>
                  </td>
                  
                  {/* 평균 응답 시간 */}
                  <td className="border px-2 py-2">
                    {avgResponseTime > 0 ? (
                      <div className="flex flex-col items-center">
                        <span className="font-medium">{(avgResponseTime / 1000).toFixed(1)}초</span>
                        <span className="text-xs text-gray-500 dark:text-gray-400">{avgResponseTime}ms</span>
                      </div>
                    ) : (
                      <span className="text-gray-400 dark:text-gray-500">-</span>
                    )}
                  </td>
                  
                  {/* 도구 지원 */}
                  <td className="border px-2 py-2">
                    {metadata ? (
                      metadata.supportsTools ? (
                        <div className="flex items-center justify-center">
                          <svg className="w-5 h-5 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                          </svg>
                        </div>
                      ) : (
                        <div className="flex items-center justify-center">
                          <svg className="w-5 h-5 text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                          </svg>
                        </div>
                      )
                    ) : (
                      <span className="text-gray-400 dark:text-gray-500">-</span>
                    )}
                  </td>
                  
                  {/* 이미지 지원 */}
                  <td className="border px-2 py-2">
                    {metadata ? (
                      metadata.supportsVision ? (
                        <div className="flex items-center justify-center">
                          <svg className="w-5 h-5 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                          </svg>
                        </div>
                      ) : (
                        <div className="flex items-center justify-center">
                          <svg className="w-5 h-5 text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                          </svg>
                        </div>
                      )
                    ) : (
                      <span className="text-gray-400 dark:text-gray-500">-</span>
                    )}
                  </td>

                  {EVAL_CATEGORIES.map((cat) => (
                    <td key={cat.key} className="border px-2 py-2 font-medium">{row[cat.key]}</td>
                  ))}
                  <td className="border px-2 py-2 font-bold">{row.total}</td>
                  <td className="border px-2 py-2 font-bold text-blue-600 dark:text-blue-400">{row.avg}</td>
                  <td className="border px-2 py-2">
                    <div className="flex gap-1 items-center justify-center">
                      <button
                        onClick={() => startChat(row.modelName)}
                        disabled={!results[row.modelId]}
                        style={{color: 'white'}}
                        className="px-2 py-1 bg-green-600 text-white rounded hover:bg-green-700 disabled:bg-gray-400 disabled:cursor-not-allowed text-xs"
                        title={
                          !results[row.modelId] 
                            ? "평가 결과가 없습니다" 
                            : "이 모델과 채팅하기"
                        }
                      >
                        채팅
                      </button>
                      <button
                        onClick={() => reEvaluateModel(row.modelName)}
                        disabled={evalStatus.isRunning || !results[row.modelId]}
                        style={{color: 'white'}}
                        className="px-2 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed text-xs"
                        title={
                          evalStatus.isRunning 
                            ? "평가 진행 중입니다" 
                            : !results[row.modelId] 
                              ? "평가 결과가 없습니다" 
                              : "이 모델을 재평가합니다"
                        }
                      >
                        재평가
                      </button>
                    </div>
                  </td>
                </tr>
              );
            })}
          </tbody>
        </table>
      </div>
      {/* 채팅 모달 */}
      {chatState.isOpen && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg max-w-4xl w-full mx-4 h-[600px] flex flex-col">
            {/* 헤더 */}
            <div className="flex justify-between items-center p-4 border-b border-gray-200 dark:border-gray-700">
              <div>
                <h3 className="text-xl font-bold text-gray-900 dark:text-gray-100">모델 채팅</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">{chatState.selectedModel}</p>
              </div>
              <div className="flex items-center gap-2">
                <button
                  onClick={clearHistory}
                  className="p-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
                  title="대화 히스토리 초기화"
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                  </svg>
                </button>
                <button
                  onClick={toggleSettings}
                  className="p-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
                  title="설정"
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                  </svg>
                </button>
                <button
                  onClick={closeChat}
                  className="p-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
                >
                  ✕
                </button>
              </div>
            </div>

            {/* 설정 패널 */}
            {chatState.showSettings && (
              <div className="p-4 bg-gray-50 dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">스트리밍 응답</label>
                    <button
                      onClick={() => setChatState(prev => ({ ...prev, streaming: !prev.streaming }))}
                      className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                        chatState.streaming ? 'bg-blue-600' : 'bg-gray-200 dark:bg-gray-600'
                      }`}
                    >
                      <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                        chatState.streaming ? 'translate-x-6' : 'translate-x-1'
                      }`} />
                    </button>
                  </div>
                  <div className="flex items-center justify-between">
                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">대화 히스토리 유지</label>
                    <button
                      onClick={() => setChatState(prev => ({ ...prev, keepHistory: !prev.keepHistory }))}
                      className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                        chatState.keepHistory ? 'bg-blue-600' : 'bg-gray-200 dark:bg-gray-600'
                      }`}
                    >
                      <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                        chatState.keepHistory ? 'translate-x-6' : 'translate-x-1'
                      }`} />
                    </button>
                  </div>
                  <div className="flex items-center justify-between">
                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">도구 사용</label>
                    <button
                      onClick={() => setChatState(prev => ({ ...prev, enableTools: !prev.enableTools }))}
                      className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                        chatState.enableTools ? 'bg-blue-600' : 'bg-gray-200 dark:bg-gray-600'
                      }`}
                    >
                      <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                        chatState.enableTools ? 'translate-x-6' : 'translate-x-1'
                      }`} />
                    </button>
                  </div>
                </div>
              </div>
            )}
            
            {/* 메시지 영역 */}
            <div className="flex-1 overflow-y-auto p-4 space-y-4">
                                {chatState.messages.length === 0 ? (
                <div className="text-center py-8">
                  {/* 환영 메시지 */}
                  <div className="mb-6">
                    <h3 className="text-xl font-semibold text-gray-900 dark:text-gray-100 mb-2">
                      안녕하세요! 🤖
                    </h3>
                    <p className="text-gray-600 dark:text-gray-400">
                      <span className="font-medium text-blue-600 dark:text-blue-400">{chatState.selectedModel}</span>과 대화를 시작해보세요.
                    </p>
                  </div>

                  {/* 현재 설정 요약 */}
                  <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-800 rounded-lg">
                    <div className="flex justify-center items-center gap-4 text-sm">
                      <div className="flex items-center gap-1">
                        <div className={`w-2 h-2 rounded-full ${chatState.streaming ? 'bg-green-500' : 'bg-gray-400'}`}></div>
                        <span>스트리밍 {chatState.streaming ? '활성' : '비활성'}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <div className={`w-2 h-2 rounded-full ${chatState.enableTools ? 'bg-blue-500' : 'bg-gray-400'}`}></div>
                        <span>도구 {chatState.enableTools ? '활성' : '비활성'}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <div className={`w-2 h-2 rounded-full ${chatState.keepHistory ? 'bg-purple-500' : 'bg-gray-400'}`}></div>
                        <span>히스토리 {chatState.keepHistory ? '유지' : '비유지'}</span>
                      </div>
                    </div>
                  </div>

                  {/* 주요 기능 카드 */}
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4 max-w-2xl mx-auto">
                    {/* 도구 기능 카드 */}
                    {chatState.enableTools ? (
                      <div className="p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg text-left">
                        <div className="flex items-center gap-2 mb-3">
                          <svg className="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                          </svg>
                          <span className="font-medium text-blue-800 dark:text-blue-200">사용 가능한 도구</span>
                        </div>
                        <div className="space-y-2 text-sm text-blue-700 dark:text-blue-300">
                          <div className="flex items-center gap-2">
                            <span>🕐</span>
                            <span>현재 시간 조회</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <span>🔢</span>
                            <span>계산기 및 수학 함수</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <span>🎲</span>
                            <span>랜덤 숫자 생성</span>
                          </div>
                        </div>
                      </div>
                    ) : (
                      <div className="p-4 bg-gray-50 dark:bg-gray-900/20 border border-gray-200 dark:border-gray-800 rounded-lg text-left">
                        <div className="flex items-center gap-2 mb-3">
                          <svg className="w-5 h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                          </svg>
                          <span className="font-medium text-gray-800 dark:text-gray-200">도구 기능</span>
                        </div>
                        <div className="space-y-2 text-sm text-gray-700 dark:text-gray-300">
                          <p>설정에서 도구 사용을 활성화하면 시간 조회, 계산기, 랜덤 숫자 생성 등의 기능을 사용할 수 있습니다.</p>
                          <p className="text-xs text-gray-500 dark:text-gray-400">※ 일부 모델은 도구 기능을 지원하지 않습니다.</p>
                        </div>
                      </div>
                    )}

                    {/* 파일 업로드 카드 */}
                    <div className="p-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg text-left">
                      <div className="flex items-center gap-2 mb-3">
                        <svg className="w-5 h-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                        </svg>
                        <span className="font-medium text-green-800 dark:text-green-200">파일 업로드 지원</span>
                      </div>
                      <div className="space-y-2 text-sm text-green-700 dark:text-green-300">
                        <div className="flex items-center gap-2">
                          <span>🖼️</span>
                          <span>이미지 분석 (.jpg, .png, .gif)</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <span>📄</span>
                          <span>텍스트 읽기 (.txt, .md, .csv)</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <span>📋</span>
                          <span>PDF 텍스트 추출 (.pdf)</span>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* 시작 팁 */}
                  <div className="mt-6 text-sm text-gray-500 dark:text-gray-400">
                    <div className="flex items-center justify-center gap-1">
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                      </svg>
                      <span>아래 입력창에 메시지를 입력하거나 파일을 업로드해보세요</span>
                    </div>
                    <div className="mt-2 text-xs text-center">
                      <span>💡 도구 기능이 필요한 경우 설정 버튼을 클릭하여 활성화하세요</span>
                    </div>
                  </div>
                </div>
              ) : (
                chatState.messages.map((message) => (
                  <div
                    key={message.id}
                    className={`flex ${message.role === 'user' ? 'justify-end' : 'justify-start'}`}
                  >
                    <div
                      className={`max-w-2xl lg:max-w-3xl px-4 py-2 rounded-lg ${
                        message.role === 'user'
                          ? 'bg-gray-200 dark:bg-gray-700 text-gray-900 dark:text-gray-100'
                          : 'bg-gray-200 dark:bg-gray-700 text-gray-900 dark:text-gray-100'
                      }`}
                    >
                      {/* 메시지 내용 */}
                      <div className="space-y-2">
                        {message.role === 'assistant' ? (
                          <MarkdownRenderer 
                            content={message.content.replace(/<think>[\s\S]*?<\/think>/g, '').trim()} 
                            className="text-sm" 
                          />
                        ) : (
                          <p className="text-sm whitespace-pre-wrap">{message.content}</p>
                        )}
                        
                                                 {/* 메시지 메타데이터 */}
                         <div className="flex items-center justify-between text-xs opacity-70">
                           <div className="flex items-center gap-2 flex-wrap">
                             <span>{message.timestamp.toLocaleTimeString()}</span>
                             
                             {/* 응답 시간 표시 (AI 응답에만) */}
                             {message.role === 'assistant' && message.responseTime && (
                               <span className="flex items-center gap-1 bg-gray-300 dark:bg-gray-600 px-1.5 py-0.5 rounded text-xs" title="응답 시간">
                                 <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                   <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                                 </svg>
                                 {message.responseTime < 1000 
                                   ? `${message.responseTime}ms` 
                                   : `${(message.responseTime / 1000).toFixed(1)}s`
                                 }
                               </span>
                             )}
                             
                             {message.role === 'user' && chatState.enableTools && (
                               <span className="flex items-center gap-1" title="도구 사용 가능">
                                 <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                   <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                                   <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                 </svg>
                               </span>
                             )}
                             {message.role === 'assistant' && chatState.streaming && (
                               <span className="flex items-center gap-1" title="스트리밍 응답">
                                 <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                   <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 4V2a1 1 0 011-1h8a1 1 0 011 1v2m-9 4v10a2 2 0 002 2h6a2 2 0 002-2V8M7 8h10M9 12h6m-6 4h6" />
                                 </svg>
                               </span>
                             )}
                           </div>
                         </div>
                      </div>
                    </div>
                  </div>
                ))
              )}
              
              {/* 로딩 메시지 */}
              {chatState.isLoading && (
                <div className="flex justify-start">
                  <div className="bg-gray-200 dark:bg-gray-700 text-gray-900 dark:text-gray-100 px-4 py-2 rounded-lg">
                    <div className="flex items-center space-x-2">
                      <div className="flex space-x-1">
                        <div className="w-2 h-2 bg-gray-500 rounded-full animate-bounce"></div>
                        <div className="w-2 h-2 bg-gray-500 rounded-full animate-bounce" style={{animationDelay: '0.1s'}}></div>
                        <div className="w-2 h-2 bg-gray-500 rounded-full animate-bounce" style={{animationDelay: '0.2s'}}></div>
                      </div>
                      <span className="text-sm">응답을 생성 중...</span>
                    </div>
                  </div>
                </div>
              )}
              
              {/* 스크롤 타겟 */}
              <div ref={messagesEndRef} />
            </div>
            
            {/* 입력 영역 */}
            <div className="p-4 border-t border-gray-200 dark:border-gray-700">
              {/* 업로드된 파일 표시 */}
              {chatState.uploadedFiles.length > 0 && (
                <div className="mb-3 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <svg className="w-4 h-4 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    <span className="text-sm font-medium text-gray-700 dark:text-gray-300">업로드된 파일</span>
                  </div>
                  <div className="space-y-2">
                    {chatState.uploadedFiles.map((file, index) => {
                      // 파일 타입에 따른 아이콘과 설명
                      const getFileInfo = (file: File) => {
                        if (file.type.startsWith('image/')) {
                          return {
                            icon: '🖼️',
                            color: 'bg-green-500',
                            description: '이미지 분석 가능'
                          };
                        } else if (file.type.startsWith('text/')) {
                          return {
                            icon: '📄',
                            color: 'bg-blue-500',
                            description: '텍스트 내용 읽기 가능'
                          };
                        } else if (file.type === 'application/pdf') {
                          return {
                            icon: '📋',
                            color: 'bg-orange-500',
                            description: '텍스트 추출 및 분석 가능'
                          };
                        } else {
                          return {
                            icon: '📎',
                            color: 'bg-gray-500',
                            description: '지원되지 않는 파일 타입'
                          };
                        }
                      };

                      const fileInfo = getFileInfo(file);

                      return (
                        <div key={index} className="flex items-center justify-between p-2 bg-white dark:bg-gray-700 rounded border">
                          <div className="flex items-center gap-2">
                            <span className="text-lg">{fileInfo.icon}</span>
                            <div className="flex flex-col">
                              <div className="flex items-center gap-2">
                                <span className="text-sm text-gray-700 dark:text-gray-300">{file.name}</span>
                                <span className="text-xs text-gray-500">({(file.size / 1024).toFixed(1)}KB)</span>
                              </div>
                              <span className="text-xs text-gray-500">{fileInfo.description}</span>
                            </div>
                          </div>
                          <button
                            onClick={() => removeFile(index)}
                            className="p-1 text-red-500 hover:text-red-700"
                          >
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                          </button>
                        </div>
                      );
                    })}
                  </div>
                </div>
              )}

              <div className="flex gap-2">
                <div className="flex flex-col flex-1">
                  <div className="flex gap-2">
                    <textarea
                      ref={inputRef}
                      value={chatState.input}
                      onChange={(e) => setChatState(prev => ({ ...prev, input: e.target.value }))}
                      onKeyPress={handleKeyPress}
                      placeholder="메시지를 입력하세요... (Enter로 전송, Shift+Enter로 줄바꿈)"
                      className="flex-1 px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-gray-100 resize-none"
                      rows={2}
                      disabled={chatState.isLoading}
                    />
                    <div className="flex flex-col gap-1">
                      <input
                        type="file"
                        id="file-upload"
                        multiple
                        accept="image/*,application/pdf,text/*"
                        onChange={handleFileUpload}
                        className="hidden"
                        disabled={chatState.isLoading}
                      />
                      <label
                        htmlFor="file-upload"
                        className="p-2 bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-300 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-600 cursor-pointer flex items-center justify-center disabled:cursor-not-allowed"
                        title="파일 업로드 (이미지: 분석 가능, 텍스트: 내용 읽기, PDF: 텍스트 추출)"
                      >
                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                        </svg>
                      </label>
                      <button
                        onClick={sendMessage}
                        disabled={!chatState.input.trim() || chatState.isLoading}
                        className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed flex items-center justify-center"
                      >
                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                        </svg>
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* 모달 */}
      {modalMessage.show && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-md w-full mx-4">
            <div className="flex justify-between items-center mb-4">
              <h3 className={`text-xl font-bold ${
                modalMessage.type === 'success' ? 'text-green-600 dark:text-green-400' :
                modalMessage.type === 'error' ? 'text-red-600 dark:text-red-400' :
                modalMessage.type === 'warning' ? 'text-yellow-600 dark:text-yellow-400' :
                'text-blue-600 dark:text-blue-400'
              }`}>
                {modalMessage.title}
              </h3>
              <button
                onClick={closeModal}
                className="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
              >
                ✕
              </button>
            </div>
            
            <p className="text-gray-700 dark:text-gray-300 mb-6">
              {modalMessage.message}
            </p>
            
            <div className="flex justify-end">
              <button
                onClick={closeModal}
                className="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600"
              >
                확인
              </button>
            </div>
          </div>
        </div>
      )}
        </div>
        </div>
      </div>
  );
} 