'use client';

import React, { useState, useEffect, useRef } from 'react';
import { useAuth } from '@/hooks/useAuth';
import { 
  Upload, 
  FileSpreadsheet, 
  Users, 
  Activity, 
  CheckCircle, 
  AlertCircle, 
  Download,
  Eye,
  Play,
  Pause,
  RotateCcw,
  RefreshCw
} from 'lucide-react';

// 타입 정의
interface PatientRecord {
  name: string;
  bloodPressure?: { systolic: number; diastolic: number };
  cholesterol?: { hdl: number; ldl: number; triglycerides: number };
  bloodSugar?: { glucose: number; hba1c: number };
  bmi?: number;
  waist?: number;
  urineProtein?: string;
  pastHistory?: string;
  currentHistory?: string;
  rawData: any;
}

interface AnalysisResult {
  patientName: string;
  bloodPressureScore: number;
  cholesterolScore: number;
  hba1cReceived: number;
  bloodSugarScore: number;
  obesityScore: number;
  urineProteinScore: number;
  pastHistoryTreatment: string;
  currentHistoryTreatment: string;
  finalJudgment: number;
  success: boolean;
  error?: string;
}

interface BatchJob {
  id: string;
  status: 'pending' | 'running' | 'completed' | 'failed';
  progress: number;
  currentPatient?: string;
  completed: number;
  total: number;
  startTime: Date;
  endTime?: Date;
  results?: AnalysisResult[];
}

type ProcessingStep = 'upload' | 'preview' | 'processing' | 'results';

export default function HealthBatchPage() {
  const { user } = useAuth();
  const [currentStep, setCurrentStep] = useState<ProcessingStep>('upload');
  const [uploadedFile, setUploadedFile] = useState<File | null>(null);
  const [patientData, setPatientData] = useState<PatientRecord[]>([]);
  const [batchJob, setBatchJob] = useState<BatchJob | null>(null);
  const [isUploading, setIsUploading] = useState(false);
  const [uploadError, setUploadError] = useState<string | null>(null);
  const [dragOver, setDragOver] = useState(false);
  const [isReanalyzing, setIsReanalyzing] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  // 파일 드래그 앤 드롭 핸들러
  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    setDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    setDragOver(false);
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    setDragOver(false);
    
    const files = Array.from(e.dataTransfer.files);
    const excelFile = files.find(file => 
      file.name.endsWith('.xlsx') || file.name.endsWith('.xls')
    );
    
    if (excelFile) {
      handleFileSelect(excelFile);
    } else {
      setUploadError('Excel 파일(.xlsx, .xls)만 업로드 가능합니다.');
    }
  };

  // 파일 선택 핸들러
  const handleFileSelect = async (file: File) => {
    setUploadError(null);
    setIsUploading(true);
    
    try {
      // 파일 유효성 검사
      if (!file.name.match(/\.(xlsx|xls)$/i)) {
        throw new Error('Excel 파일(.xlsx, .xls)만 업로드 가능합니다.');
      }
      
      if (file.size > 10 * 1024 * 1024) { // 10MB 제한
        throw new Error('파일 크기는 10MB 이하여야 합니다.');
      }

      setUploadedFile(file);
      
      // 파일 파싱하여 환자 데이터 추출
      const formData = new FormData();
      formData.append('file', file);
      
      const response = await fetch('/api/admin/health-batch/parse', {
        method: 'POST',
        body: formData,
        credentials: 'include'
      });
      
      if (!response.ok) {
        throw new Error('파일 파싱에 실패했습니다.');
      }
      
      const result = await response.json();
      console.log('파싱 결과:', result);

      if (!result.success) {
        throw new Error(result.error || '파일 파싱에 실패했습니다.');
      }

      // 데이터베이스에 저장된 작업 ID와 환자 데이터 설정
      const newBatchJob = {
        id: result.data.jobId,
        status: 'pending' as const,
        progress: 0,
        completed: 0,
        total: result.data.totalCount || result.data.patients.length,
        startTime: new Date()
      };

      console.log('배치 작업 설정:', newBatchJob);
      console.log('환자 데이터:', result.data.patients);

      setBatchJob(newBatchJob);
      setPatientData(result.data.patients);
      setCurrentStep('preview');

      console.log('상태 변경 완료: preview 단계로 이동');
      
    } catch (error) {
      console.error('파일 업로드 오류:', error);
      setUploadError(error instanceof Error ? error.message : '파일 업로드에 실패했습니다.');
      setUploadedFile(null);
    } finally {
      setIsUploading(false);
    }
  };

  // 배치 분석 시작
  const startBatchAnalysis = async () => {
    if (!batchJob?.id || patientData.length === 0) return;

    try {
      // 데이터베이스 기반 분석 API 호출
      const response = await fetch('/api/admin/health-batch/analyze-v2', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          jobId: batchJob.id
        }),
        credentials: 'include'
      });

      if (!response.ok) {
        throw new Error('배치 분석 시작에 실패했습니다.');
      }

      const result = await response.json();
      if (!result.success) {
        throw new Error(result.error || '배치 분석 시작에 실패했습니다.');
      }

      setBatchJob(prev => prev ? {
        ...prev,
        status: 'running'
      } : null);

      setCurrentStep('processing');

      // 진행률 폴링 시작
      startProgressPolling(batchJob.id);

    } catch (error) {
      console.error('배치 분석 시작 오류:', error);
      alert(error instanceof Error ? error.message : '배치 분석 시작에 실패했습니다.');
    }
  };

  // 진행률 폴링
  const startProgressPolling = (jobId: string) => {
    const interval = setInterval(async () => {
      try {
        const response = await fetch(`/api/admin/health-batch/status-v2/${jobId}`, {
          credentials: 'include'
        });
        
        if (!response.ok) return;
        
        const result = await response.json();
        if (!result.success) return;
        
        const jobData = result.data;
        setBatchJob(prev => prev ? {
          ...prev,
          status: jobData.status,
          progress: jobData.processed_records / jobData.total_records * 100,
          currentPatient: jobData.current_patient,
          completed: jobData.processed_records,
          total: jobData.total_records,
          endTime: jobData.status === 'completed' ? new Date() : undefined,
          results: jobData.results
        } : null);
        
        if (jobData.status === 'completed' || jobData.status === 'failed') {
          clearInterval(interval);
          if (jobData.status === 'completed') {
            setCurrentStep('results');
          }
        }
        
      } catch (error) {
        console.error('진행률 확인 오류:', error);
      }
    }, 2000); // 2초마다 체크
    
    // 10분 후 자동 중단
    setTimeout(() => clearInterval(interval), 600000);
  };

  // 스텝 표시 컴포넌트
  const StepIndicator = () => {
    const steps = [
      { key: 'upload', label: '파일 업로드', icon: Upload },
      { key: 'preview', label: '데이터 확인', icon: Eye },
      { key: 'processing', label: 'AI 분석', icon: Activity },
      { key: 'results', label: '결과 다운로드', icon: Download }
    ];
    
    const currentIndex = steps.findIndex(step => step.key === currentStep);
    
    return (
      <div className="flex items-center justify-center mb-8 px-4">
        {steps.map((step, index) => {
          const IconComponent = step.icon;
          const isActive = index === currentIndex;
          const isCompleted = index < currentIndex;
          const isDisabled = index > currentIndex;
          
          return (
            <React.Fragment key={step.key}>
              <div className={`flex flex-col items-center ${isDisabled ? 'opacity-50' : ''}`}>
                <div className={`w-12 h-12 rounded-full flex items-center justify-center transition-all duration-200 ${
                  isCompleted 
                    ? 'bg-green-500 text-white' 
                    : isActive 
                    ? 'bg-blue-500 text-white' 
                    : 'bg-gray-200 dark:bg-gray-700 text-gray-500 dark:text-gray-400'
                }`}>
                  {isCompleted ? (
                    <CheckCircle className="w-6 h-6" />
                  ) : (
                    <IconComponent className="w-6 h-6" />
                  )}
                </div>
                <span className={`mt-2 text-sm font-medium ${
                  isActive ? 'text-blue-600 dark:text-blue-400' : 'text-gray-500 dark:text-gray-400'
                }`}>
                  {step.label}
                </span>
              </div>
              {index < steps.length - 1 && (
                <div className={`flex-1 h-px mx-4 transition-all duration-200 ${
                  isCompleted ? 'bg-green-500' : 'bg-gray-300 dark:bg-gray-600'
                }`} style={{ minWidth: '60px' }} />
              )}
            </React.Fragment>
          );
        })}
      </div>
    );
  };

  // 단계별 컨텐츠 렌더링
  const renderStepContent = () => {
    switch (currentStep) {
      case 'upload':
        return (
          <div className="max-w-2xl mx-auto">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>
                건강검진 데이터 업로드
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                환자들의 원시 건강검진 데이터가 포함된 Excel 파일을 업로드해주세요.
              </p>
            </div>
            
            <div 
              className={`drag-drop-zone border-2 border-dashed rounded-xl p-12 text-center transition-all duration-200 cursor-pointer ${
                dragOver 
                  ? 'border-blue-400 bg-blue-50 dark:bg-blue-900/20' 
                  : 'border-gray-300 dark:border-gray-600 hover:border-gray-400'
              }`}
              onDragOver={handleDragOver}
              onDragLeave={handleDragLeave}
              onDrop={handleDrop}
              onClick={() => fileInputRef.current?.click()}
              data-drag-over={dragOver}
            >
              <input
                ref={fileInputRef}
                type="file"
                accept=".xlsx,.xls"
                onChange={(e) => {
                  const file = e.target.files?.[0];
                  if (file) handleFileSelect(file);
                }}
                className="hidden"
              />
              
              {isUploading ? (
                <div className="flex flex-col items-center">
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mb-4"></div>
                  <p className="text-gray-600 dark:text-gray-400">파일 처리 중...</p>
                </div>
              ) : (
                <div className="flex flex-col items-center">
                  <FileSpreadsheet className="w-16 h-16 text-gray-400 mb-4" />
                  <p className="text-lg font-medium mb-2" style={{ color: 'var(--text-primary)' }}>
                    파일을 드래그하거나 클릭하여 업로드
                  </p>
                  <p className="text-sm text-gray-500 dark:text-gray-400">
                    지원 형식: .xlsx, .xls, .csv
                  </p>
                </div>
              )}
            </div>
            
            {uploadError && (
              <div className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                <div className="flex items-center">
                  <AlertCircle className="w-5 h-5 text-red-500 mr-2" />
                  <span className="text-red-700 dark:text-red-400">{uploadError}</span>
                </div>
              </div>
            )}
          </div>
        );

      case 'preview':
        return (
          <div className="w-full px-4">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>
                환자 데이터 확인
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                총 {patientData.length}명의 환자 데이터가 발견되었습니다. 데이터를 확인한 후 분석을 시작하세요.
              </p>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
              <div className="card p-6">
                <div className="flex items-center">
                  <Users className="w-8 h-8 text-blue-500 mr-3" />
                  <div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">총 환자 수</p>
                    <p className="text-2xl font-bold" style={{ color: 'var(--text-primary)' }}>
                      {patientData.length}명
                    </p>
                  </div>
                </div>
              </div>
              
              <div className="card p-6">
                <div className="flex items-center">
                  <CheckCircle className="w-8 h-8 text-green-500 mr-3" />
                  <div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">완전한 데이터</p>
                    <p className="text-2xl font-bold text-green-600">
                      {patientData.filter(p => p.name && p.rawData && Object.keys(p.rawData).length > 0).length}명
                    </p>
                  </div>
                </div>
              </div>
              
              <div className="card p-6">
                <div className="flex items-center">
                  <AlertCircle className="w-8 h-8 text-yellow-500 mr-3" />
                  <div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">불완전한 데이터</p>
                    <p className="text-2xl font-bold text-yellow-600">
                      {patientData.filter(p => !p.name || !p.rawData || Object.keys(p.rawData).length === 0).length}명
                    </p>
                  </div>
                </div>
              </div>
            </div>
            
            <div className="card w-full">
              <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                  환자 목록 미리보기 (처음 10명) - 전체 {Object.keys(patientData[0]?.rawData || {}).length}개 컬럼
                </h3>
              </div>
              <div className="overflow-x-auto max-h-96">
                <table className="w-full min-w-max text-base">
                  <thead style={{ backgroundColor: 'var(--body-bg)' }}>
                    <tr>
                      <th className="px-4 py-3 text-left text-gray-600 dark:text-gray-400">환자명</th>
                      {/* 동적으로 모든 원시 데이터 컬럼 표시 */}
                      {patientData.length > 0 && Object.keys(patientData[0].rawData || {}).map(column => (
                        <th key={column} className="px-4 py-3 text-left text-gray-600 dark:text-gray-400">
                          {column}
                        </th>
                      ))}
                      <th className="px-4 py-3 text-left text-gray-600 dark:text-gray-400">상태</th>
                    </tr>
                  </thead>
                  <tbody>
                    {patientData.slice(0, 10).map((patient, index) => (
                      <tr key={index} className="border-b border-gray-200 dark:border-gray-700">
                        <td className="px-4 py-3" style={{ color: 'var(--text-primary)' }}>
                          {patient.name}
                        </td>
                        {/* 동적으로 모든 원시 데이터 값 표시 */}
                        {Object.keys(patient.rawData || {}).map(column => (
                          <td key={column} className="px-4 py-3 text-gray-600 dark:text-gray-400">
                            {patient.rawData[column] || '미입력'}
                          </td>
                        ))}
                        <td className="px-4 py-3">
                          {patient.name && patient.rawData && Object.keys(patient.rawData).length > 0 ? (
                            <span className="inline-flex items-center px-2 py-1 rounded-full bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400">
                              완료
                            </span>
                          ) : (
                            <span className="inline-flex items-center px-2 py-1 rounded-full bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400">
                              불완전
                            </span>
                          )}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
              {patientData.length > 10 && (
                <div className="p-4 text-center text-gray-500 dark:text-gray-400">
                  + {patientData.length - 10}명 더 있습니다
                </div>
              )}
            </div>
            
            <div className="flex justify-center gap-4 mt-8">
              <button
                onClick={() => setCurrentStep('upload')}
                className="px-6 py-2 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
              >
                다시 업로드
              </button>
              <button
                onClick={startBatchAnalysis}
                className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors flex items-center gap-2"
              >
                <Play className="w-4 h-4" />
                분석 시작
              </button>
            </div>
          </div>
        );

      case 'processing':
        return (
          <div className="max-w-4xl mx-auto">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>
                AI 건강검진 분석 진행 중
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                모든 환자의 건강검진 데이터를 분석하고 있습니다. 잠시만 기다려주세요.
              </p>
            </div>
            
            {batchJob && (
              <div className="space-y-6">
                {/* 진행률 표시 */}
                <div className="card p-6">
                  <div className="flex items-center justify-between mb-4">
                    <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                      분석 진행률
                    </h3>
                    <span className="text-sm text-gray-600 dark:text-gray-400">
                      {batchJob.completed} / {batchJob.total} 완료
                    </span>
                  </div>
                  
                  <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-3 mb-4">
                    <div 
                      className="bg-blue-600 h-3 rounded-full transition-all duration-500"
                      style={{ width: `${batchJob.progress}%` }}
                    ></div>
                  </div>
                  
                  <div className="flex justify-between text-sm text-gray-600 dark:text-gray-400">
                    <span>{batchJob.progress.toFixed(1)}% 완료</span>
                    <span>
                      {batchJob.currentPatient && `현재: ${batchJob.currentPatient}`}
                    </span>
                  </div>
                </div>
                
                {/* 실시간 상태 */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="card p-4">
                    <div className="flex items-center">
                      <div className="w-3 h-3 bg-green-500 rounded-full animate-pulse mr-3"></div>
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">처리 완료</p>
                        <p className="text-xl font-bold text-green-600">{batchJob.completed}명</p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="card p-4">
                    <div className="flex items-center">
                      <div className="w-3 h-3 bg-blue-500 rounded-full animate-pulse mr-3"></div>
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">처리 중</p>
                        <p className="text-xl font-bold text-blue-600">
                          {Math.max(0, batchJob.total - batchJob.completed)}명
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="card p-4">
                    <div className="flex items-center">
                      <Activity className="w-6 h-6 text-purple-500 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">전체</p>
                        <p className="text-xl font-bold" style={{ color: 'var(--text-primary)' }}>
                          {batchJob.total}명
                        </p>
                      </div>
                    </div>
                  </div>
                </div>
                
                {/* 예상 완료 시간 */}
                <div className="card p-4">
                  <div className="text-center">
                    <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                      시작 시간: {batchJob.startTime.toLocaleString()}
                    </p>
                    {batchJob.progress > 0 && (
                      <p className="text-sm text-gray-600 dark:text-gray-400">
                        예상 완료: 약 {Math.ceil((100 - batchJob.progress) / batchJob.progress * 
                          (Date.now() - batchJob.startTime.getTime()) / 1000 / 60)}분 후
                      </p>
                    )}
                  </div>
                </div>
              </div>
            )}
          </div>
        );

      case 'results':
        return (
          <div className="w-full px-4">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>
                분석 완료
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                모든 환자의 건강검진 분석이 완료되었습니다. 결과를 확인하고 다운로드하세요.
              </p>
            </div>
            
            {batchJob?.results && (
              <>
                {/* 결과 요약 */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-8">
                  <div className="card p-6">
                    <div className="flex items-center">
                      <CheckCircle className="w-8 h-8 text-green-500 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">성공</p>
                        <p className="text-2xl font-bold text-green-600">
                          {batchJob.results.filter(r => r.analysis_status === 'completed').length}명
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="card p-6">
                    <div className="flex items-center">
                      <AlertCircle className="w-8 h-8 text-red-500 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">실패</p>
                        <p className="text-2xl font-bold text-red-600">
                          {batchJob.results.filter(r => r.analysis_status === 'failed').length}명
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="card p-6">
                    <div className="flex items-center">
                      <Activity className="w-8 h-8 text-blue-500 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">처리 시간</p>
                        <p className="text-lg font-bold" style={{ color: 'var(--text-primary)' }}>
                          {batchJob.endTime && 
                            Math.ceil((batchJob.endTime.getTime() - batchJob.startTime.getTime()) / 1000 / 60)
                          }분
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="card p-6">
                    <div className="flex items-center">
                      <Download className="w-8 h-8 text-purple-500 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">다운로드</p>
                        <button 
                          onClick={() => downloadResults()}
                          className="text-lg font-bold text-purple-600 hover:text-purple-700 transition-colors"
                        >
                          Excel 파일
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
                
                {/* 결과 미리보기 */}
                <div className="card w-full">
                  <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                    <h3 className="text-lg font-semibold" style={{ color: 'var(--text-primary)' }}>
                      분석 결과 미리보기 (처음 10명)
                    </h3>
                  </div>
                  <div className="overflow-x-auto max-h-120">
                    <table className="w-full min-w-max text-base">
                      <thead style={{ backgroundColor: 'var(--body-bg)' }}>
                        <tr>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">환자명</th>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">혈압</th>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">지질</th>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">혈당</th>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">비만</th>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">요단백</th>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">최종판정</th>
                          <th className="px-3 py-2 text-left text-gray-600 dark:text-gray-400">상태</th>
                        </tr>
                      </thead>
                      <tbody>
                        {batchJob.results.slice(0, 10).map((result, index) => (
                          <tr key={index} className="border-b border-gray-200 dark:border-gray-700">
                            <td className="px-3 py-2" style={{ color: 'var(--text-primary)' }}>
                              {result.patient_name}
                            </td>
                            <td className="px-3 py-2">{result.blood_pressure_score ?? '-'}</td>
                            <td className="px-3 py-2">{result.cholesterol_score ?? '-'}</td>
                            <td className="px-3 py-2">{result.blood_sugar_score ?? '-'}</td>
                            <td className="px-3 py-2">{result.obesity_score ?? '-'}</td>
                            <td className="px-3 py-2">{result.urine_protein_score ?? '-'}</td>
                            <td className="px-3 py-2 font-bold">{result.final_judgment ?? '-'}</td>
                            <td className="px-3 py-2">
                              {result.analysis_status === 'completed' ? (
                                <CheckCircle className="w-4 h-4 text-green-500" />
                              ) : (
                                <AlertCircle className="w-4 h-4 text-red-500" />
                              )}
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
                
                <div className="flex justify-center gap-4 mt-8">
                  <button
                    onClick={() => {
                      setCurrentStep('upload');
                      setBatchJob(null);
                      setPatientData([]);
                      setUploadedFile(null);
                    }}
                    className="px-6 py-2 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors flex items-center gap-2"
                  >
                    <RotateCcw className="w-4 h-4" />
                    새로 시작
                  </button>

                  {/* 실패한 환자가 있으면 재분석 버튼 표시 */}
                  {batchJob?.results && batchJob.results.filter(r => r.analysis_status === 'failed').length > 0 && (
                    <button
                      onClick={reanalyzeFailedPatients}
                      disabled={isReanalyzing}
                      className="px-6 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700 transition-colors flex items-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      {isReanalyzing ? (
                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                      ) : (
                        <RefreshCw className="w-4 h-4" />
                      )}
                      {isReanalyzing ? '재분석 중...' : '실패한 환자 재분석'}
                    </button>
                  )}

                  <button
                    onClick={downloadResults}
                    className="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors flex items-center gap-2"
                  >
                    <Download className="w-4 h-4" />
                    Excel 다운로드
                  </button>
                </div>
              </>
            )}
          </div>
        );

      default:
        return null;
    }
  };

  // 실패한 환자 재분석
  const reanalyzeFailedPatients = async () => {
    if (!batchJob?.id) return;

    setIsReanalyzing(true);

    try {
      const response = await fetch('/api/admin/health-batch/reanalyze', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          jobId: batchJob.id
        }),
        credentials: 'include'
      });

      if (!response.ok) {
        throw new Error('재분석에 실패했습니다.');
      }

      const result = await response.json();
      if (!result.success) {
        throw new Error(result.error || '재분석에 실패했습니다.');
      }

      // 결과를 사용자에게 알리고 데이터 새로고침
      alert(result.data.message);

      // 진행률 다시 조회하여 UI 업데이트
      const statusResponse = await fetch(`/api/admin/health-batch/status-v2/${batchJob.id}`, {
        credentials: 'include'
      });

      if (statusResponse.ok) {
        const statusResult = await statusResponse.json();
        if (statusResult.success) {
          const jobData = statusResult.data;
          setBatchJob(prev => prev ? {
            ...prev,
            status: jobData.status,
            progress: jobData.processed_records / jobData.total_records * 100,
            completed: jobData.processed_records,
            total: jobData.total_records,
            results: jobData.patients
          } : null);
        }
      }

    } catch (error) {
      console.error('재분석 오류:', error);
      alert(error instanceof Error ? error.message : '재분석에 실패했습니다.');
    } finally {
      setIsReanalyzing(false);
    }
  };

  // 결과 다운로드
  const downloadResults = async () => {
    if (!batchJob?.id) return;

    try {
      const response = await fetch(`/api/admin/health-batch/download-v2/${batchJob.id}`, {
        credentials: 'include'
      });

      if (!response.ok) {
        throw new Error('다운로드에 실패했습니다.');
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `건강검진_분석결과_${new Date().toISOString().slice(0, 10)}.xlsx`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      window.URL.revokeObjectURL(url);

    } catch (error) {
      console.error('다운로드 오류:', error);
      alert(error instanceof Error ? error.message : '다운로드에 실패했습니다.');
    }
  };

  // 권한 확인
  if (!user || user.role !== 'admin') {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <AlertCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
          <h2 className="text-2xl font-bold mb-2" style={{ color: 'var(--text-primary)' }}>
            접근 권한 없음
          </h2>
          <p className="text-gray-600 dark:text-gray-400">
            이 페이지는 관리자만 접근할 수 있습니다.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen py-8 px-4" style={{ backgroundColor: 'var(--body-bg)' }}>
      <div className="max-w-7xl mx-auto">
        {/* 페이지 헤더 */}
        <div className="text-center mb-12">
          <div className="flex items-center justify-center mb-4">
            <Activity className="w-12 h-12 text-blue-600 mr-4" />
            <h3 className="text-2xl font-bold" style={{ color: 'var(--text-primary)' }}>
              건강검진 분석 Agent
            </h3>
          </div>
          <p className="text-lg text-gray-600 dark:text-gray-400">
            여러 환자의 건강검진 데이터를 일괄 분석하여 정규화된 판정 결과를 생성합니다
          </p>
        </div>
        
        {/* 단계 표시기 */}
        <StepIndicator />
        
        {/* 단계별 컨텐츠 */}
        {renderStepContent()}
      </div>
    </div>
  );
}