import { NextRequest, NextResponse } from 'next/server';
import pg from 'pg';
import { getDatabaseConfig } from '@/config/serverConfig';

const { Pool } = pg;

// 데이터베이스 연결
function getDbConnection() {
  const config = getDatabaseConfig();
  return new Pool({
    host: config.host,
    port: config.port,
    user: config.user,
    password: config.password,
    database: config.name,
    max: 5
  });
}

// 탄소배출량 계산
function calculateCarbonEmission(
  revenue: number,
  carbonIntensity: number
): number {
  // 탄소배출량 = 매출액(백만원) × 탄소집약도(tCO2e/백만원)
  const revenueInMillions = revenue / 1000000;
  return revenueInMillions * carbonIntensity;
}

// ESG 보고서 생성
async function generateESGReport(data: {
  documentName: string;
  industryMajorName: string;
  industrySubName: string;
  estimatedRevenue: number;
  carbonIntensity: number;
  totalCarbon: number;
}): Promise<string> {
  const revenueFormatted = new Intl.NumberFormat('ko-KR').format(data.estimatedRevenue);
  const carbonFormatted = data.totalCarbon.toFixed(2);

  const report = `
# ESG 탄소배출량 분석 보고서

## 1. 기업 정보
- **문서명**: ${data.documentName}
- **산업분류**: ${data.industryMajorName} > ${data.industrySubName}

## 2. 재무 정보
- **예상 매출액**: ${revenueFormatted}원

## 3. 탄소배출량 분석

### 3.1 탄소집약도
- **산업별 탄소집약도**: ${data.carbonIntensity} tCO2e/백만원

### 3.2 총 탄소배출량
- **계산식**: 매출액(백만원) × 탄소집약도
- **총 배출량**: **${carbonFormatted} tCO2e** (톤 CO2 환산량)

## 4. 탄소배출 등급

${getCarbonEmissionGrade(data.totalCarbon, data.estimatedRevenue)}

## 5. 개선 권고사항

### 5.1 단기 개선 방안
1. **에너지 효율 개선**: 사무실 및 생산시설의 에너지 효율 개선
2. **재생에너지 전환**: 태양광, 풍력 등 재생에너지 사용 확대
3. **전기차 도입**: 차량 운영 시 전기차 또는 하이브리드 차량 도입

### 5.2 중장기 개선 방안
1. **탄소중립 로드맵 수립**: 2050 탄소중립 목표 달성을 위한 단계별 계획 수립
2. **공급망 관리**: 협력업체의 탄소배출 관리 및 저탄소 제품 구매
3. **R&D 투자**: 저탄소 기술 개발 및 친환경 제품 개발

## 6. 결론

${data.documentName}의 예상 탄소배출량은 연간 **${carbonFormatted} tCO2e**로 추정됩니다.
${data.industryMajorName} 산업의 평균 탄소집약도를 기준으로 계산되었으며,
실제 배출량은 에너지 사용량, 생산 공정, 제품 특성에 따라 달라질 수 있습니다.

---
*본 보고서는 AI 기반 분석 도구를 사용하여 자동 생성되었습니다.*
*보고서 생성일: ${new Date().toLocaleDateString('ko-KR')}*
`;

  return report;
}

// 탄소배출 등급 계산
function getCarbonEmissionGrade(totalCarbon: number, revenue: number): string {
  const intensity = (totalCarbon * 1000000) / revenue; // tCO2e per 원 -> tCO2e per 백만원

  let grade = '';
  let description = '';

  if (intensity < 0.2) {
    grade = 'A (우수)';
    description = '탄소배출이 매우 낮은 친환경 기업입니다.';
  } else if (intensity < 0.5) {
    grade = 'B (양호)';
    description = '탄소배출이 평균보다 낮은 수준입니다.';
  } else if (intensity < 1.0) {
    grade = 'C (보통)';
    description = '산업 평균 수준의 탄소배출을 보이고 있습니다.';
  } else if (intensity < 2.0) {
    grade = 'D (개선 필요)';
    description = '탄소배출이 다소 높은 편이며, 개선이 필요합니다.';
  } else {
    grade = 'E (매우 높음)';
    description = '탄소배출이 매우 높은 수준이며, 적극적인 개선이 필요합니다.';
  }

  return `- **배출 등급**: ${grade}\n- **평가**: ${description}`;
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { jobId, documentId, revenue, industryMajorCode, industrySubCode } = body;

    if (!jobId || !documentId) {
      return NextResponse.json({
        success: false,
        error: '작업 ID 또는 문서 ID가 필요합니다.'
      }, { status: 400 });
    }

    // 데이터베이스 연결
    const dbManager = getDbConnection();

    // 문서 정보 조회
    const documentResult = await dbManager.query(`
      SELECT
        document_name,
        estimated_revenue,
        industry_major_code,
        industry_major_name,
        industry_sub_code,
        industry_sub_name,
        carbon_intensity
      FROM esg_batch_documents
      WHERE id = $1 AND job_id = $2
    `, [documentId, jobId]);

    if (documentResult.rows.length === 0) {
      await dbManager.end();
      return NextResponse.json({
        success: false,
        error: '문서를 찾을 수 없습니다.'
      }, { status: 404 });
    }

    let document = documentResult.rows[0];

    // 사용자가 수정한 매출액이 있으면 사용
    if (revenue && revenue > 0) {
      document.estimated_revenue = revenue;
    }

    // 사용자가 수정한 산업분류가 있으면 탄소집약도 재조회
    if (industryMajorCode && industrySubCode) {
      const industryResult = await dbManager.query(`
        SELECT
          code,
          major_category,
          carbon_intensity
        FROM esg_industry_classifications
        WHERE code = $1
      `, [industryMajorCode]);

      if (industryResult.rows.length > 0) {
        const industry = industryResult.rows[0];
        document.industry_major_code = industryMajorCode;
        document.industry_major_name = industry.major_category;
        document.carbon_intensity = industry.carbon_intensity;

        // 소분류 정보 조회
        const subResult = await dbManager.query(`
          SELECT sub_categories
          FROM esg_industry_classifications
          WHERE code = $1
        `, [industryMajorCode]);

        if (subResult.rows.length > 0) {
          const subCategories = subResult.rows[0].sub_categories || [];
          const selectedSub = subCategories.find((s: any) => s.code === industrySubCode);
          if (selectedSub) {
            document.industry_sub_code = industrySubCode;
            document.industry_sub_name = selectedSub.name;
          }
        }
      }
    }

    // 매출액 확인
    if (!document.estimated_revenue || document.estimated_revenue <= 0) {
      await dbManager.end();
      return NextResponse.json({
        success: false,
        error: '예상 매출액이 설정되지 않았습니다.'
      }, { status: 400 });
    }

    // 탄소집약도 확인
    if (!document.carbon_intensity || document.carbon_intensity <= 0) {
      await dbManager.end();
      return NextResponse.json({
        success: false,
        error: '탄소집약도가 설정되지 않았습니다. 먼저 표준산업분류를 식별해주세요.'
      }, { status: 400 });
    }

    // 탄소배출량 계산
    const totalCarbonEmission = calculateCarbonEmission(
      document.estimated_revenue,
      document.carbon_intensity
    );

    // ESG 보고서 생성
    const report = await generateESGReport({
      documentName: document.document_name,
      industryMajorName: document.industry_major_name,
      industrySubName: document.industry_sub_name,
      estimatedRevenue: document.estimated_revenue,
      carbonIntensity: document.carbon_intensity,
      totalCarbon: totalCarbonEmission
    });

    // 분석 결과 저장 (수정된 매출액과 산업분류도 함께 저장)
    await dbManager.query(`
      UPDATE esg_batch_documents
      SET
        estimated_revenue = $1,
        industry_major_code = $2,
        industry_major_name = $3,
        industry_sub_code = $4,
        industry_sub_name = $5,
        carbon_intensity = $6,
        total_carbon_emission = $7,
        emission_unit = 'tCO2e',
        report_content = $8,
        report_format = 'markdown',
        analysis_status = 'completed',
        analyzed_at = CURRENT_TIMESTAMP,
        updated_at = CURRENT_TIMESTAMP
      WHERE id = $9
    `, [
      document.estimated_revenue,
      document.industry_major_code,
      document.industry_major_name,
      document.industry_sub_code,
      document.industry_sub_name,
      document.carbon_intensity,
      totalCarbonEmission,
      report,
      documentId
    ]);

    // 작업 상태 업데이트
    await dbManager.query(`
      UPDATE esg_batch_jobs
      SET
        status = 'completed',
        processed_records = 1,
        successful_records = 1,
        progress = 100,
        completed_at = CURRENT_TIMESTAMP,
        updated_at = CURRENT_TIMESTAMP
      WHERE id = $1
    `, [jobId]);

    await dbManager.end();

    return NextResponse.json({
      success: true,
      data: {
        jobId,
        documentId,
        totalCarbonEmission,
        report,
        message: 'ESG 분석이 완료되었습니다.'
      }
    });

  } catch (error) {
    console.error('ESG 분석 오류:', error);

    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : 'ESG 분석에 실패했습니다.'
    }, { status: 500 });
  }
}
