import { NextRequest, NextResponse } from 'next/server';
import { v4 as uuidv4 } from 'uuid';
import pg from 'pg';
import { getDatabaseConfig } from '@/config/serverConfig';
import { exec } from 'child_process';
import { promisify } from 'util';
import fs from 'fs/promises';
import path from 'path';
import os from 'os';

const { Pool } = pg;
const execAsync = promisify(exec);
const pythonExecutable = process.env.AIRUN_PYTHON || path.join(
  os.homedir(),
  '.airun_venv',
  process.platform === 'win32' ? 'Scripts' : 'bin',
  process.platform === 'win32' ? 'python.exe' : 'python3'
);

// 데이터베이스 연결
function getDbConnection() {
  const config = getDatabaseConfig();
  return new Pool({
    host: config.host,
    port: config.port,
    user: config.user,
    password: config.password,
    database: config.name,
    max: 5
  });
}

// utils.py의 extract_text_from_document 함수를 사용한 텍스트 추출
async function extractTextFromDocument(filePath: string): Promise<string> {
  try {
    // 절대 경로 사용 (workspaces/web에서 실행되므로 상위 2단계로 이동)
    const scriptPath = path.resolve(process.cwd(), '../../scripts/extract_document_text.py');

    // Python 스크립트 실행 (StderrRedirector가 디버그 로그를 stderr로 리다이렉트)
    const { stdout, stderr } = await execAsync(`"${pythonExecutable}" "${scriptPath}" "${filePath}"`, {
      maxBuffer: 50 * 1024 * 1024 // 50MB 버퍼
    });

    // stderr에 경고가 있어도 정상 동작할 수 있으므로 로그만 출력 (error로 처리하지 않음)
    if (stderr && !stderr.includes('UserWarning')) {
      console.warn('Python 스크립트 stderr:', stderr);
    }

    // JSON 응답 파싱
    let result;
    try {
      result = JSON.parse(stdout);
    } catch (parseError) {
      console.error('JSON 파싱 오류:', parseError);
      console.error('stdout 내용:', stdout);
      throw new Error('문서 텍스트 추출 결과를 파싱할 수 없습니다.');
    }

    if (!result.success) {
      throw new Error(result.error || '문서 텍스트 추출 실패');
    }

    return result.text;
  } catch (error) {
    console.error('문서 텍스트 추출 오류:', error);
    throw new Error(`문서 처리에 실패했습니다: ${error instanceof Error ? error.message : String(error)}`);
  }
}

// 문서에서 매출액 추출 (간단한 패턴 매칭)
function extractRevenue(text: string): number | null {
  // 매출액 패턴: "매출액: 100억", "예상 매출액: 200억원", "약 50억원", "매출 100,000,000원", "revenue: 100M" 등
  const patterns = [
    /(?:예상|실적|목표)?\s*매출\s*[액:]?\s*(?:은|는|이|가)?\s*(?:약|대략|예상)?\s*(\d{1,3}(?:,\d{3})*)\s*억/i,
    /(?:예상|실적|목표)?\s*매출\s*[액:]?\s*(?:은|는|이|가)?\s*(?:약|대략|예상)?\s*(\d{1,3}(?:,\d{3})*)\s*만원/i,
    /(?:예상|실적|목표)?\s*매출\s*[액:]?\s*(?:은|는|이|가)?\s*(?:약|대략|예상)?\s*(\d{1,3}(?:,\d{3})*(?:\.\d+)?)\s*원/i,
    /revenue[:\s]*(?:approximately|about|estimated)?\s*\$?\s*(\d{1,3}(?:,\d{3})*(?:\.\d+)?)\s*[MmKkBb]?/i,
    /sales[:\s]*(?:approximately|about|estimated)?\s*\$?\s*(\d{1,3}(?:,\d{3})*(?:\.\d+)?)\s*[MmKkBb]?/i
  ];

  for (const pattern of patterns) {
    const match = text.match(pattern);
    if (match) {
      let value = parseFloat(match[1].replace(/,/g, ''));

      // 단위 변환 (억, 만, M, B 등)
      if (text.includes('억')) {
        value = value * 100000000;
      } else if (text.includes('만원')) {
        value = value * 10000;
      } else if (/[Bb]/.test(match[0])) {
        value = value * 1000000000;
      } else if (/[Mm]/.test(match[0])) {
        value = value * 1000000;
      } else if (/[Kk]/.test(match[0])) {
        value = value * 1000;
      }

      return value;
    }
  }

  return null;
}

export async function POST(request: NextRequest) {
  try {
    // 임시로 인증 건너뛰기
    const userId = 'admin';

    const formData = await request.formData();
    const file = formData.get('file') as File;

    if (!file) {
      return NextResponse.json({
        success: false,
        error: '파일이 업로드되지 않았습니다.'
      }, { status: 400 });
    }

    // 파일 확장자 검증 (utils.py가 지원하는 모든 형식)
    if (!file.name.match(/\.(pdf|docx?|xlsx?|pptx?|hwpx?|txt)$/i)) {
      return NextResponse.json({
        success: false,
        error: 'PDF, Word, Excel, PowerPoint, HWP, 텍스트 파일만 업로드 가능합니다.'
      }, { status: 400 });
    }

    // 파일 크기 검증 (50MB)
    if (file.size > 50 * 1024 * 1024) {
      return NextResponse.json({
        success: false,
        error: '파일 크기는 50MB 이하여야 합니다.'
      }, { status: 400 });
    }

    // 임시 파일로 저장 (Python 스크립트가 파일 경로를 필요로 함)
    const buffer = await file.arrayBuffer();
    const tempDir = os.tmpdir();
    const tempFilePath = path.join(tempDir, `esg_${uuidv4()}_${file.name}`);

    try {
      await fs.writeFile(tempFilePath, Buffer.from(buffer));

      // utils.py의 extract_text_from_document 함수를 사용하여 텍스트 추출
      const textContent = await extractTextFromDocument(tempFilePath);

      if (!textContent || textContent.trim().length === 0) {
        return NextResponse.json({
          success: false,
          error: '문서에서 텍스트를 추출할 수 없습니다.'
        }, { status: 400 });
      }

      // 매출액 추출
      const estimatedRevenue = extractRevenue(textContent);

      // 데이터베이스 연결
      const dbManager = getDbConnection();

      // 새로운 배치 작업 생성
      const jobId = uuidv4();

      await dbManager.query(`
        INSERT INTO esg_batch_jobs (id, user_id, filename, status, total_records)
        VALUES ($1, $2, $3, 'pending', 1)
      `, [jobId, userId, file.name]);

      // 문서 데이터 저장
      const documentResult = await dbManager.query(`
        INSERT INTO esg_batch_documents (
          job_id,
          document_name,
          row_number,
          raw_content,
          estimated_revenue,
          analysis_status
        )
        VALUES ($1, $2, $3, $4, $5, 'pending')
        RETURNING id
      `, [
        jobId,
        file.name,
        1,
        textContent.substring(0, 10000), // 텍스트 최대 10000자까지 저장
        estimatedRevenue
      ]);

      await dbManager.end();

      return NextResponse.json({
        success: true,
        data: {
          jobId,
          documentId: documentResult.rows[0].id,
          filename: file.name,
          contentLength: textContent.length,
          contentPreview: textContent.substring(0, 500),
          estimatedRevenue,
          message: '문서가 성공적으로 업로드되었습니다.'
        }
      });

    } finally {
      // 임시 파일 정리
      try {
        await fs.unlink(tempFilePath);
      } catch (unlinkError) {
        console.error('임시 파일 삭제 오류:', unlinkError);
      }
    }

  } catch (error) {
    console.error('파일 파싱 오류:', error);

    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : '파일 파싱에 실패했습니다.'
    }, { status: 500 });
  }
}
