import { NextRequest, NextResponse } from 'next/server';
import { pg } from '../../db';
import bcrypt from 'bcrypt';
import crypto from 'crypto';

// 사용자 목록 조회
export async function GET(req: NextRequest) {
  try {
    const { searchParams } = req.nextUrl;
    const search = searchParams.get("search")?.trim() || "";
    const page = parseInt(searchParams.get("page") || "1", 10);
    const pageSize = parseInt(searchParams.get("pageSize") || "20", 10);

    let where = "";
    let values: any[] = [];
    if (search) {
      where = `WHERE username ILIKE $1 OR name ILIKE $1 OR email ILIKE $1`;
      values.push(`%${search}%`);
    }

    // 전체 개수 쿼리
    const countResult = await pg.query(
      `SELECT COUNT(*) FROM users ${where}`,
      values
    );
    const total = parseInt(countResult.rows[0].count, 10);

    // 목록 쿼리
    const offset = (page - 1) * pageSize;
    const listResult = await pg.query(
      `
      SELECT id, username, email, name, role, created_at, last_login, status
      FROM users
      ${where}
      ORDER BY created_at DESC
      LIMIT $${values.length + 1} OFFSET $${values.length + 2}
      `,
      [...values, pageSize, offset]
    );

    const usersWithKeys = await Promise.all(listResult.rows.map(async (user) => {
      const keysRes = await pg.query(
        'SELECT key_value as key, status FROM api_keys WHERE user_id = $1 ORDER BY created_at DESC LIMIT 1',
        [user.id]
      );
      return { ...user, apiKeys: keysRes.rows };
    }));

    return NextResponse.json({
      success: true,
      data: {
        users: usersWithKeys,
        total,
        page,
        pageSize,
      },
    });
  } catch (error) {
    return NextResponse.json(
      { success: false, error: { message: "사용자 목록 조회 실패", details: error } },
      { status: 500 }
    );
  }
}

// 사용자 생성
export async function POST(req: NextRequest) {
  try {
    const body = await req.json();
    const { username, email, name, role, status, password } = body;
    if (!username || !role) {
      return NextResponse.json({ success: false, error: { message: '필수 항목 누락' } }, { status: 400 });
    }
    // 비밀번호는 기본값 또는 입력값 사용
    const passwordHash = await bcrypt.hash(password || 'changeme1234', 10);
    const result = await pg.query(
      `INSERT INTO users (username, email, name, role, status, password_hash)
       VALUES ($1, $2, $3, $4, $5, $6)
       RETURNING id, username, email, name, role, created_at, last_login, status`,
      [username, email, name || '', role, status ?? 'active', passwordHash]
    );
    const userId = result.rows[0].id;

    // API키 생성
    const randomKey = crypto.randomBytes(16).toString('hex');
    const apiKey = `airun_${userId}_${randomKey}`;
    await pg.query(
      `INSERT INTO api_keys (user_id, key_value) VALUES ($1, $2)`,
      [userId, apiKey]
    );

    // 사용자 정보 + API키 반환 (보안상 최초 1회만 노출 권장)
    return NextResponse.json({ success: true, data: { ...result.rows[0], apiKeys: [{ key: apiKey }] } });
  } catch (error) {
    return NextResponse.json({ success: false, error: { message: '사용자 생성 실패', details: error } }, { status: 500 });
  }
}

export async function PUT(req: NextRequest, context: { params: { userId: string } }) {
  try {
    // context.params를 await로 받아야 함 (Next.js 14+)
    const params = await context.params;
    const { userId } = params;
    // ... 이하 동일
  } catch (error) {
    return NextResponse.json(
      { success: false, error: { message: "사용자 정보 수정 실패", details: error } },
      { status: 500 }
    );
  }
}

export async function GET_USER(req: NextRequest, context: { params: { userId: string } }) {
  try {
    const params = await context.params;
    const { userId } = params;
    const userRes = await pg.query('SELECT ... FROM users WHERE id = $1', [userId]);
    const apiKeysRes = await pg.query('SELECT key_value as key, status FROM api_keys WHERE user_id = $1', [userId]);
    return NextResponse.json({ success: true, data: { ...userRes.rows[0], apiKeys: apiKeysRes.rows } });
  } catch (error) {
    return NextResponse.json(
      { success: false, error: { message: "사용자 상세 조회 실패", details: error } },
      { status: 500 }
    );
  }
} 