import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '@/app/api/middlewares';
import { pg } from '@/app/api/db';

const RAG_API_SERVER = process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';

export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user) => {
    try {
      const formData = await req.formData();
      const file = formData.get('file') as File;
      
      if (!file) {
        return NextResponse.json({ success: false, error: 'File is required.' }, { status: 400 });
      }
      
      const apiFormData = new FormData();
      apiFormData.append('file', file);
      if (user?.id) {
        apiFormData.append('user_id', user.id);
      }
      
      // RAG 서버에 파일 업로드
      const apiResponse = await fetch(`${RAG_API_SERVER}/upload-page-chunk`, {
        method: 'POST',
        body: apiFormData,
      });

      if (!apiResponse.ok) {
        const errorText = await apiResponse.text();
        return NextResponse.json({ success: false, error: `API Error: ${errorText}` }, { status: apiResponse.status });
      }

      const apiData = await apiResponse.json();

      // 채팅 페이지 전용 DB에 파일 정보 저장
      try {
        await pg.query(
          `INSERT INTO chat_documents (filename, filepath, filesize, mimetype, user_id)
           VALUES ($1, $2, $3, $4, $5)`,
          [
            apiData.file, // RAG 서버가 반환하는 파일명
            apiData.rag_docs_path, // RAG 서버가 반환하는 저장 경로
            file.size,
            file.type,
            user?.id
          ]
        );
        console.log('[CHAT_UPLOAD] DB 저장 완료:', apiData.file);
      } catch (dbError) {
        console.error('[CHAT_DB_INSERT_ERROR]', dbError);
        // DB 저장에 실패해도 RAG 업로드는 성공으로 간주
      }

      return NextResponse.json(apiData);

    } catch (error) {
      console.error('[CHAT_UPLOAD_ERROR]', error);
      return NextResponse.json({ success: false, error: 'Internal Server Error' }, { status: 500 });
    }
  });
} 