import { NextRequest, NextResponse } from 'next/server';
import { pg } from '@/app/api/db';
import fs from 'fs/promises';
import path from 'path';

// 개인 문서 목록 조회 (chat, code, document, agent 페이지 공통 사용)
export async function GET(req: NextRequest) {
  try {
    const url = new URL(req.url);
    const userId = url.searchParams.get('userId');
    const checkFiles = url.searchParams.get('checkFiles') === 'true';
    
    let query = 'SELECT * FROM chat_documents';
    let params: any[] = [];
    
    if (userId) {
      query += ' WHERE user_id = $1';
      params = [userId];
    }
    
    query += ' ORDER BY created_at DESC';
    
    const result = await pg.query(query, params);
    const documents = result.rows;

    // 파일 존재 여부 확인이 요청된 경우
    if (checkFiles) {
      const documentsWithFileCheck = await Promise.all(
        documents.map(async (doc) => {
          const filePath = path.join(doc.filepath, doc.filename);
          let fileExists = false;
          
          try {
            await fs.access(filePath);
            fileExists = true;
          } catch (error) {
            fileExists = false;
            console.log(`[CHAT_DOCS_FILE_CHECK] File not found: ${filePath}`);
          }
          
          return {
            ...doc,
            file_exists: fileExists
          };
        })
      );
      
      return NextResponse.json({ success: true, data: documentsWithFileCheck });
    }

    return NextResponse.json({ success: true, data: documents });
  } catch (error) {
    console.error('[CHAT_DOCS_GET_ERROR]', error);
    return NextResponse.json({ success: false, error: 'Internal Server Error' }, { status: 500 });
  }
}

// 채팅 페이지 문서 정보 저장
export async function POST(req: NextRequest) {
  try {
    const { filename, filepath, filesize, mimetype, user_id } = await req.json();

    if (!filename || !filepath) {
      return NextResponse.json({ success: false, error: 'Filename and filepath are required.' }, { status: 400 });
    }

    // 동일한 user_id와 filename 조합이 있는지 확인하고 삭제
    if (user_id && filename) {
      try {
        const existingDocs = await pg.query(
          'SELECT id FROM chat_documents WHERE user_id = $1 AND filename = $2',
          [user_id, filename]
        );

        if (existingDocs.rows.length > 0) {
          console.log(`[CHAT_DOCS_DUPLICATE_REMOVAL] Found ${existingDocs.rows.length} duplicate documents for user_id: ${user_id}, filename: ${filename}`);
          
          // 모든 중복 문서 삭제
          await pg.query(
            'DELETE FROM chat_documents WHERE user_id = $1 AND filename = $2',
            [user_id, filename]
          );
          
          console.log(`[CHAT_DOCS_DUPLICATE_REMOVAL] Deleted ${existingDocs.rows.length} duplicate documents`);
        }
      } catch (deleteError) {
        console.error('[CHAT_DOCS_DUPLICATE_REMOVAL_ERROR]', deleteError);
        // 중복 삭제 실패시에도 새 문서는 저장 시도
      }
    }

    // 새 문서 삽입
    const result = await pg.query(
      'INSERT INTO chat_documents (filename, filepath, filesize, mimetype, user_id) VALUES ($1, $2, $3, $4, $5) RETURNING *',
      [filename, filepath, filesize || null, mimetype || null, user_id]
    );

    console.log(`[CHAT_DOCS_INSERT] Successfully inserted document for user_id: ${user_id}, filename: ${filename}`);
    return NextResponse.json({ success: true, data: result.rows[0] });
  } catch (error) {
    console.error('[CHAT_DOCS_POST_ERROR]', error);
    return NextResponse.json({ success: false, error: 'Internal Server Error' }, { status: 500 });
  }
}

// 채팅 페이지 문서 삭제
export async function DELETE(req: NextRequest) {
  try {
    
    const token = req.cookies.get('auth_token')?.value;
    
    if (!token) {
      return NextResponse.json({ success: false, error: 'Authentication token is required.' }, { status: 401 });
    }

    const { id, filename } = await req.json();
    if (!filename || !id) {
      return NextResponse.json({ success: false, error: 'Filename and id are required.' }, { status: 400 });
    }

    // 와일드카드 지원: filename이 '*'인 경우 해당 사용자의 모든 문서 삭제
    let docResult;
    if (filename === '*') {
      // 모든 문서 조회
      docResult = await pg.query('SELECT id FROM chat_documents WHERE user_id = $1', [id]);
      console.log(`[CHAT_DOCS_DELETE] Deleting ALL documents for user_id: ${id}, found ${docResult.rows.length} documents`);
    } else {
      // 특정 파일 조회
      docResult = await pg.query('SELECT id FROM chat_documents WHERE user_id = $1 and filename = $2', [id, filename]);
      console.log(`[CHAT_DOCS_DELETE] Deleting specific document for user_id: ${id}, filename: ${filename}`);
    }

    if (docResult.rows.length === 0) {
      return NextResponse.json({ success: false, error: 'Document not found.' }, { status: 404 });
    }
    
    // 2. 파일 시스템에서 파일 삭제
    // const filePath = path.join(doc.filepath, doc.filename);
    // try {
    //   await fs.unlink(filePath);
    //   console.log(`[CHAT_FILE_DELETE_SUCCESS] ${filePath}`);
    // } catch (fileError: any) {
    //   if (fileError.code !== 'ENOENT') {
    //     console.error('[CHAT_FILE_DELETE_ERROR]', fileError);
    //   } else {
    //     console.log(`[CHAT_FILE_DELETE_SKIP] File not found: ${filePath}`);
    //   }
    // }

    // 3. DB에서 레코드 삭제
    if (filename === '*') {
      // 모든 문서 삭제
      await pg.query('DELETE FROM chat_documents WHERE user_id = $1', [id]);
      console.log(`[CHAT_DOCS_DELETE] Successfully deleted ${docResult.rows.length} documents for user_id: ${id}`);
    } else {
      // 특정 문서 삭제
      const doc = docResult.rows[0];
      await pg.query('DELETE FROM chat_documents WHERE id = $1', [doc.id]);
      console.log(`[CHAT_DOCS_DELETE] Successfully deleted document id: ${doc.id}`);
    }

    return NextResponse.json({ success: true, message: 'Document deleted successfully.' });
  } catch (error) {
    console.error('[CHAT_DOCS_DELETE_ERROR]', error);
    return NextResponse.json({ success: false, error: 'Internal Server Error' }, { status: 500 });
  }
} 