import { NextRequest, NextResponse } from 'next/server';

export async function POST(request: NextRequest) {
  try {
    const { modelName } = await request.json();
    
    if (!modelName) {
      return NextResponse.json({ error: '모델명이 필요합니다.' }, { status: 400 });
    }

    // HuggingFace API를 통한 모델 정보 조회
    const response = await fetch(`https://huggingface.co/api/models/${modelName}`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        'User-Agent': 'airun-eval/1.0'
      },
    });

    if (!response.ok) {
      throw new Error(`HuggingFace API 오류: ${response.status}`);
    }

    const data = await response.json();
    
    // config.json 파일 조회 (모델 설정 정보)
    let configData = null;
    try {
      const configResponse = await fetch(`https://huggingface.co/${modelName}/raw/main/config.json`, {
        headers: {
          'Accept': 'application/json',
          'User-Agent': 'airun-eval/1.0'
        },
      });
      if (configResponse.ok) {
        configData = await configResponse.json();
      }
    } catch (error) {
      console.warn('config.json 조회 실패:', error);
    }

    // 메타데이터 추출
    const tags = data.tags || [];
    const modelCard = data.cardData || {};
    
    // 파라미터 수 추출
    let parameterCount = 'Unknown';
    if (data.safetensors && data.safetensors.total) {
      const totalParams = data.safetensors.total;
      if (totalParams > 1e9) {
        parameterCount = `${Math.round(totalParams / 1e9)}B`;
      } else if (totalParams > 1e6) {
        parameterCount = `${Math.round(totalParams / 1e6)}M`;
      }
    } else if (modelName.includes('7b')) {
      parameterCount = '7B';
    } else if (modelName.includes('13b')) {
      parameterCount = '13B';
    } else if (modelName.includes('70b')) {
      parameterCount = '70B';
    }

    // 컨텍스트 길이 추출
    let contextLength = 4096; // 기본값
    if (configData) {
      contextLength = configData.max_position_embeddings || 
                     configData.n_positions || 
                     configData.max_sequence_length || 
                     configData.seq_length || 
                     4096;
    }

    // 도구 지원 여부
    const supportsTools = tags.includes('tool-use') || 
                         tags.includes('function-calling') ||
                         tags.includes('agents') ||
                         data.pipeline_tag === 'text-generation';

    // 비전 지원 여부
    const supportsVision = tags.includes('multimodal') ||
                          tags.includes('vision') ||
                          tags.includes('image-to-text') ||
                          data.pipeline_tag === 'image-to-text' ||
                          modelName.toLowerCase().includes('vision') ||
                          modelName.toLowerCase().includes('vl');

    // 라이센스 정보
    const license = data.cardData?.license || 'Unknown';

    const modelInfo = {
      name: modelName,
      parameters: parameterCount,
      context_length: contextLength,
      supports_tools: supportsTools,
      supports_vision: supportsVision,
      license: license,
      tags: tags,
      downloads: data.downloads || 0,
      likes: data.likes || 0,
      pipeline_tag: data.pipeline_tag,
      library_name: data.library_name,
      created_at: data.createdAt,
      updated_at: data.lastModified
    };

    return NextResponse.json(modelInfo);
  } catch (error) {
    console.error('HuggingFace 모델 정보 조회 오류:', error);
    return NextResponse.json({ 
      error: 'HuggingFace에서 모델 정보를 조회할 수 없습니다.',
      details: String(error)
    }, { status: 500 });
  }
} 