import { NextRequest, NextResponse } from 'next/server';
import { createReactAgent } from '@langchain/langgraph/prebuilt';
import { ChatOpenAI } from '@langchain/openai';
import { ChatAnthropic } from '@langchain/anthropic';
import { tool } from '@langchain/core/tools';
import fs from 'fs';
import path from 'path';
import os from 'os';

// 설정 파일 경로
const getAirunConfigPath = () => {
  const homeDir = os.homedir();
  return path.join(homeDir, '.airun', 'airun.conf');
};

// airun.conf에서 설정 읽기
const getAirunConfig = () => {
  const configPath = getAirunConfigPath();
  const config: any = {};
  
  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      for (const line of content.split('\n')) {
        const trimmed = line.trim();
        if (trimmed.startsWith('export ')) {
          const match = trimmed.match(/export\s+([A-Z_]+)=["']?([^"']+)["']?/);
          if (match) {
            config[match[1]] = match[2];
          }
        }
      }
    } catch (error) {
      console.error('airun.conf 읽기 실패:', error);
    }
  }
  
  return config;
};

// 간단한 도구 정의 (zod 없이)
const simpleTool = tool(
  async (input: { message: string }) => {
    console.log(`🔧 간단한 도구 호출됨: ${input.message}`);
    return `간단한 도구 응답: ${input.message}를 처리했습니다.`;
  },
  {
    name: "simple_tool",
    description: "간단한 테스트 도구입니다.",
    // zod 없이 간단한 스키마 정의
    schema: {
      type: "object",
      properties: {
        message: {
          type: "string",
          description: "처리할 메시지"
        }
      },
      required: ["message"]
    }
  }
);

// 모든 도구를 배열로 정의
const allTools = [simpleTool];

// LLM 모델 생성 함수
function createLLMModel(provider: string, model: string, config: any, temperature?: number) {
  switch (provider) {
    case 'openai':
      return new ChatOpenAI({
        apiKey: config.OPENAI_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    case 'anthropic':
      return new ChatAnthropic({
        apiKey: config.ANTHROPIC_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    default:
      throw new Error(`지원하지 않는 프로바이더: ${provider}`);
  }
}

// 스트리밍 응답 생성 함수
const createStreamingResponse = (generator: () => AsyncGenerator<string>) => {
  return new Response(
    new ReadableStream({
      async start(controller) {
        try {
          for await (const chunk of generator()) {
            controller.enqueue(new TextEncoder().encode(chunk));
          }
        } catch (error) {
          console.error('스트리밍 오류:', error);
          controller.enqueue(new TextEncoder().encode(`\n\n오류: ${error.message}`));
        } finally {
          controller.close();
        }
      },
    }),
    {
      headers: {
        'Content-Type': 'text/plain; charset=utf-8',
        'Cache-Control': 'no-cache',
        'Connection': 'keep-alive',
      },
    }
  );
};

// LangGraph 에이전트로 처리
async function* processWithLangGraph(
  message: string,
  provider: string,
  model: string,
  temperature?: number
) {
  console.log(`🚀 LangGraph 에이전트 처리 시작: ${provider}/${model}`);
  
  try {
    // 설정 로드
    const config = getAirunConfig();
    
    // LLM 모델 생성
    const llm = createLLMModel(provider, model, config, temperature);
    
    // LangGraph 에이전트 생성
    console.log('🔧 LangGraph 에이전트 생성 중...');
    const agent = createReactAgent({
      llm,
      tools: allTools,
    });
    
    // console.log('✅ LangGraph 에이전트 생성 완료');
    
    // 메시지 준비
    const messages = [
      {
        role: 'human',
        content: message,
      }
    ];
    
    console.log('📤 LangGraph 에이전트 실행 중...');
    
    // 에이전트 실행
    const result = await agent.invoke({
      messages: messages,
    });
    
    console.log('✅ LangGraph 에이전트 처리 완료');
    
    // 결과 처리
    if (result && result.messages && result.messages.length > 0) {
      const lastMessage = result.messages[result.messages.length - 1];
      
      // 응답 텍스트 추출
      let responseText = '';
      if (lastMessage.content) {
        responseText = lastMessage.content;
      } else {
        responseText = '응답을 생성했습니다.';
      }
      
      // 스트리밍 방식으로 응답 전송
      const words = responseText.split(' ');
      for (let i = 0; i < words.length; i++) {
        yield words[i] + (i < words.length - 1 ? ' ' : '');
        await new Promise(resolve => setTimeout(resolve, 50)); // 50ms 지연
      }
      
    } else {
      yield '응답을 생성할 수 없습니다.';
    }
    
  } catch (error) {
    console.error('❌ LangGraph 에이전트 처리 실패:', error);
    yield `처리 중 오류가 발생했습니다: ${error.message}`;
  }
}

// POST: 채팅 메시지 처리 (LangGraph 사용)
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { message, provider, model, temperature } = body;
    
    if (!message || message.trim() === '') {
      return NextResponse.json(
        { error: '메시지가 필요합니다.' },
        { status: 400 }
      );
    }
    
    // 지원하는 프로바이더 확인
    if (!['openai', 'anthropic'].includes(provider)) {
      return NextResponse.json(
        { error: `현재 LangGraph 모드에서는 OpenAI와 Anthropic만 지원됩니다. (요청: ${provider})` },
        { status: 400 }
      );
    }
    
    console.log(`🦜 LangGraph 간단 채팅 요청: ${message}`);

    // 스트리밍 응답 생성
    return createStreamingResponse(async function* () {
      yield* processWithLangGraph(message, provider, model, temperature);
    });
    
  } catch (error) {
    console.error('LangGraph 간단 채팅 처리 실패:', error);
    return NextResponse.json(
      { error: '채팅 처리 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}

// GET: 채팅 상태 확인
export async function GET() {
  try {
    const config = getAirunConfig();
    
    return NextResponse.json({
      available: true,
      mode: 'langgraph-simple',
      provider: config.USE_LLM || 'anthropic',
      model: config.ANTHROPIC_MODEL || config.OPENAI_MODEL || 'default',
      supportedProviders: ['openai', 'anthropic'],
      toolsCount: allTools.length,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error('LangGraph 간단 채팅 상태 확인 실패:', error);
    return NextResponse.json(
      { error: '채팅 상태 확인에 실패했습니다.' },
      { status: 500 }
    );
  }
} 