import { NextRequest, NextResponse } from 'next/server';
import { validateApiKey, getTokenFromRequest } from './auth/utils';
import { getApiServerUrl } from '@/config/serverConfig';

// API 서버의 기본 URL
const API_SERVER = getApiServerUrl();

/**
 * API 키 인증 미들웨어
 * (API 요청에 사용)
 */
export async function withApiKeyAuth(
  req: NextRequest,
  handler: (req: NextRequest, userData: any) => Promise<NextResponse>
): Promise<NextResponse> {
  try {
    // API 키 헤더에서 추출
    const apiKey = req.headers.get('x-api-key');
    
    if (!apiKey) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: 'API 키가 필요합니다.'
          }
        },
        { status: 401 }
      );
    }
    
    // API 키 유효성 검사
    const userData = await validateApiKey(apiKey);
    
    if (!userData) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '유효하지 않은 API 키입니다.'
          }
        },
        { status: 401 }
      );
    }
    
    // 핸들러 호출
    return handler(req, userData);
  } catch (error) {
    console.error('API 키 인증 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: '인증 처리 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * 토큰 인증 미들웨어
 * (웹 브라우저 인증에 사용)
 */
export async function withTokenAuth(
  req: NextRequest,
  handler: (req: NextRequest, userData: any) => Promise<NextResponse>
): Promise<NextResponse> {
  try {
    // 쿠키에서 토큰 추출
    const token = getTokenFromRequest(req);
    
    // 디버깅 로그 추가
    console.log('=== withTokenAuth 시작 ===');
    console.log('요청 URL:', req.url);
    console.log('토큰 존재 여부:', token ? '있음' : '없음');
    console.log('쿠키 정보:', req.cookies.getAll().map(c => `${c.name}=${c.value.substring(0, 10)}...`));
    
    if (!token) {
      console.log('토큰이 없어서 401 반환');
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '로그인이 필요합니다.'
          }
        },
        { status: 401 }
      );
    }
    
    // API 서버에 토큰 유효성 검사 요청
    console.log('API 서버에 토큰 검증 요청:', `${API_SERVER}/auth/me`);
    const response = await fetch(`${API_SERVER}/auth/me`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${token}`
      }
    });
    
    console.log('API 서버 응답 상태:', response.status, response.statusText);
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('토큰 검증 실패:', response.status, response.statusText, errorText);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '인증이 만료되었거나 유효하지 않습니다.'
          }
        },
        { status: 401 }
      );
    }
    
    const data = await response.json();
    console.log('API 서버 응답 데이터:', data);
    
    if (!data.success || !data.data) {
      console.error('API 서버 응답 실패:', data);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '사용자 정보를 가져올 수 없습니다.'
          }
        },
        { status: 401 }
      );
    }
    
    console.log('인증 성공, 핸들러 호출');
    // 핸들러에 사용자 정보 전달
    return handler(req, data.data);
  } catch (error) {
    console.error('인증 처리 중 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: '인증 처리 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * 둘 중 하나의 인증 방식 허용 (API 키 또는 토큰)
 */
export async function withAnyAuth(
  req: NextRequest,
  handler: (req: NextRequest, userData: any, authType: 'token' | 'apikey') => Promise<NextResponse>
): Promise<NextResponse> {
  try {
    // console.log('=== withAnyAuth 시작 ===');
    // console.log('요청 URL:', req.url);
    
    // API 키 확인
    const apiKey = req.headers.get('x-api-key');
    // console.log('API 키 존재:', apiKey ? '있음' : '없음');
    
    if (apiKey) {
      // console.log('API 키로 인증 시도:', apiKey.substring(0, 10) + '...');
      // 직접 API 서버에 API 키 검증 요청
      const response = await fetch(`${API_SERVER}/auth/validate-key`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ apiKey })
      });
      
      // console.log('API 키 검증 응답 상태:', response.status);
      
      if (response.ok) {
        const data = await response.json();
        // console.log('API 키 검증 성공:', data.success);
        if (data.success) {
          // console.log('API 키 인증 성공, 핸들러 호출');
          return handler(req, data.data.user, 'apikey');
        }
      } else {
        const errorText = await response.text();
        console.log('API 키 검증 실패:', errorText);
      }
    }
    
    // 토큰 확인
    const token = getTokenFromRequest(req);
    console.log('토큰 존재:', token ? '있음' : '없음');
    
    if (token) {
      console.log('토큰으로 인증 시도:', token.substring(0, 10) + '...');
      const response = await fetch(`${API_SERVER}/auth/me`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });
      
      console.log('토큰 검증 응답 상태:', response.status);
      
      if (response.ok) {
        const data = await response.json();
        console.log('토큰 검증 성공:', data.success);
        if (data.success) {
          console.log('토큰 인증 성공, 핸들러 호출');
          return handler(req, data.data, 'token');
        }
      } else {
        const errorText = await response.text();
        console.log('토큰 검증 실패:', errorText);
      }
    }
    
    // 둘 다 실패한 경우
    console.log('모든 인증 방법 실패');
    return NextResponse.json(
      {
        success: false,
        error: {
          message: 'API 키 또는 로그인이 필요합니다.'
        }
      },
      { status: 401 }
    );
  } catch (error) {
    console.error('인증 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: '인증 처리 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
} 