import { NextRequest, NextResponse } from 'next/server';
import { writeFile, unlink } from 'fs/promises';
import { join } from 'path';
import { tmpdir } from 'os';
import { v4 as uuidv4 } from 'uuid';
import { exec } from 'child_process';
import { promisify } from 'util';

const execAsync = promisify(exec);

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;
    const previewData = JSON.parse(formData.get('preview') as string);

    if (!file || !previewData) {
      return NextResponse.json(
        { success: false, error: 'File and preview data required' },
        { status: 400 }
      );
    }

    // 파일을 임시 디렉토리에 저장
    const buffer = Buffer.from(await file.arrayBuffer());
    const pdfPath = join(tmpdir(), `pdf-analyze-${uuidv4()}.pdf`);
    await writeFile(pdfPath, buffer);

    try {
      // AI 분석 실행
      const analysis = await analyzePdfWithAI(pdfPath, previewData);

      return NextResponse.json({
        success: true,
        data: analysis
      });

    } finally {
      // 임시 파일 정리
      await unlink(pdfPath).catch(() => {});
    }

  } catch (error) {
    console.error('PDF AI analysis error:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'AI analysis failed' 
      },
      { status: 500 }
    );
  }
}

async function analyzePdfWithAI(pdfPath: string, previewData: any) {
  const analysis = {
    autoRotate: false,
    detectTwoUp: false,
    deskew: false,
    suggestedCropMargin: 0,
    hasSkew: false,
    isTwoUp: false,
    rotation: 0,
    confidence: 0,
    recommendations: [] as string[]
  };

  try {
    // PDF 기본 정보 추출
    const { stdout: pdfInfo } = await execAsync(`pdfinfo "${pdfPath}"`);
    const pageCount = previewData.pageCount || 1;
    
    let totalScore = 0;
    let analysisCount = 0;

    // 1. 2-up 레이아웃 분석
    let twoUpPages = 0;
    let totalAspectRatio = 0;
    
    for (const page of previewData.pages || []) {
      if (page.width && page.height) {
        const aspectRatio = page.width / page.height;
        totalAspectRatio += aspectRatio;
        
        // 가로/세로 비율이 1.3보다 크면 2-up 가능성
        if (aspectRatio > 1.3) {
          twoUpPages++;
        }
      }
    }
    
    const avgAspectRatio = totalAspectRatio / Math.max(previewData.pages?.length || 1, 1);
    const twoUpRatio = twoUpPages / Math.max(previewData.pages?.length || 1, 1);
    
    if (twoUpRatio > 0.3) { // 30% 이상의 페이지가 2-up 형태
      analysis.detectTwoUp = true;
      analysis.isTwoUp = true;
      analysis.recommendations.push('2-up 레이아웃이 감지되었습니다. 페이지 분할을 권장합니다.');
      totalScore += 20;
    }
    
    analysisCount++;

    // 2. 회전 필요성 분석
    let rotationNeeded = false;
    let suggestedRotation = 0;
    
    for (const page of previewData.pages || []) {
      if (page.width && page.height) {
        const aspectRatio = page.width / page.height;
        
        // 세로가 매우 긴 경우 (세로 문서가 가로로 되어 있을 가능성)
        if (aspectRatio < 0.6) {
          rotationNeeded = true;
          suggestedRotation = 90;
        }
        // 가로가 매우 긴 경우 (가로 문서가 세로로 되어 있을 가능성)
        else if (aspectRatio > 1.8 && !analysis.isTwoUp) {
          rotationNeeded = true;
          suggestedRotation = 270;
        }
      }
    }
    
    if (rotationNeeded) {
      analysis.autoRotate = true;
      analysis.rotation = suggestedRotation;
      analysis.recommendations.push(`${suggestedRotation}도 회전이 필요한 페이지가 감지되었습니다.`);
      totalScore += 15;
    }
    
    analysisCount++;

    // 3. 텍스트 기반 분석 (간단한 휴리스틱)
    try {
      // pdftotext로 텍스트 추출하여 분석
      const { stdout: textContent } = await execAsync(`pdftotext -layout "${pdfPath}" -`);
      
      if (textContent) {
        const lines = textContent.split('\n');
        const nonEmptyLines = lines.filter(line => line.trim().length > 0);
        
        // 텍스트 밀도 분석
        const avgLineLength = nonEmptyLines.reduce((sum, line) => sum + line.length, 0) / Math.max(nonEmptyLines.length, 1);
        
        // 짧은 줄이 많으면 스캔 문서일 가능성 (OCR 오류)
        if (avgLineLength < 20 && nonEmptyLines.length > 10) {
          analysis.deskew = true;
          analysis.hasSkew = true;
          analysis.recommendations.push('스캔 문서로 보입니다. 기울기 보정을 권장합니다.');
          totalScore += 10;
        }
        
        // 텍스트가 페이지 가장자리에 몰려있으면 여백이 큼
        const edgeTextRatio = analyzeTextDistribution(textContent);
        if (edgeTextRatio < 0.7) {
          analysis.suggestedCropMargin = Math.min(10, Math.max(2, Math.floor((1 - edgeTextRatio) * 15)));
          analysis.recommendations.push(`여백이 큰 것으로 감지되었습니다. ${analysis.suggestedCropMargin}% 크롭을 권장합니다.`);
          totalScore += 5;
        }
      }
      
      analysisCount++;
      
    } catch (textError) {
      console.warn('Text analysis failed:', textError);
    }

    // 4. 파일 크기 기반 분석
    const stats = await require('fs').promises.stat(pdfPath);
    const fileSizeMB = stats.size / (1024 * 1024);
    
    // 큰 파일은 고해상도 스캔일 가능성
    if (fileSizeMB > 50) {
      analysis.recommendations.push('대용량 파일입니다. 압축 최적화가 도움이 될 수 있습니다.');
      totalScore += 5;
    }
    
    // 페이지당 크기가 큰 경우
    const avgSizePerPage = fileSizeMB / pageCount;
    if (avgSizePerPage > 5) {
      if (!analysis.suggestedCropMargin) {
        analysis.suggestedCropMargin = 3;
        analysis.recommendations.push('페이지당 용량이 큽니다. 여백 크롭으로 용량을 줄일 수 있습니다.');
      }
      totalScore += 3;
    }
    
    analysisCount++;

    // 5. 종합 신뢰도 계산
    analysis.confidence = Math.min(totalScore / Math.max(analysisCount * 10, 1), 1);
    
    // 권장사항이 없으면 기본 메시지 추가
    if (analysis.recommendations.length === 0) {
      analysis.recommendations.push('문서가 이미 잘 정리되어 있습니다. 추가 전처리가 필요하지 않을 수 있습니다.');
    }

    // 6. 한국어 문서 특별 처리
    try {
      const { stdout: textSample } = await execAsync(`pdftotext -f 1 -l 1 "${pdfPath}" -`);
      if (textSample && /[가-힣]/.test(textSample)) {
        analysis.recommendations.push('한국어 문서가 감지되었습니다. OCR 정확도 향상을 위해 전처리를 권장합니다.');
      }
    } catch (e) {
      // 무시
    }

  } catch (error) {
    console.error('PDF analysis failed:', error);
    analysis.recommendations.push('분석 중 오류가 발생했습니다. 수동 설정을 권장합니다.');
  }

  return analysis;
}

function analyzeTextDistribution(textContent: string): number {
  // 텍스트가 페이지 중앙에 집중되어 있는지 분석
  // 간단한 휴리스틱: 줄의 시작 공백과 길이를 분석
  const lines = textContent.split('\n');
  const nonEmptyLines = lines.filter(line => line.trim().length > 0);
  
  if (nonEmptyLines.length === 0) return 1;
  
  let centeredLines = 0;
  for (const line of nonEmptyLines) {
    const leadingSpaces = line.length - line.trimStart().length;
    const trimmedLength = line.trim().length;
    
    // 적절한 들여쓰기와 길이를 가진 줄
    if (leadingSpaces > 2 && leadingSpaces < 20 && trimmedLength > 10) {
      centeredLines++;
    }
  }
  
  return centeredLines / nonEmptyLines.length;
}