import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '@/app/api/middlewares';
import { getApiServerUrl } from '@/config/serverConfig';

const API_SERVER = getApiServerUrl();
console.log(API_SERVER);

export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user): Promise<any> => {
    try {
      const body = await req.json();

      const token = req.cookies.get('auth_token')?.value;
      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
      };
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      } else {
        // Return a non-streaming error response
        return NextResponse.json({ success: false, error: "Authentication token is required." }, { 
          status: 401, 
        });
      }

      const { messages } = body;
      if (!messages || !Array.isArray(messages) || messages.length === 0) {
        return NextResponse.json({ success: false, error: 'Invalid messages payload' }, { status: 400 });
      }

      const prompt = messages[messages.length - 1].content;
      const history = messages.slice(0, -1);
      const { sessionId } = body;

      const payload = {
        prompt,
        history,
        options: {
          rag: true,
          userId: user?.id,
          sessionId: sessionId,
        },
      };

      const apiResponse = await fetch(`${API_SERVER}/chat/sync`, {
        method: 'POST',
        headers,
        body: JSON.stringify(payload),
      });

      if (!apiResponse.ok) {
        const errorData = await apiResponse.json();
        const errorText = errorData.error || 'API Error';
        return NextResponse.json({ success: false, error: errorText }, { 
          status: apiResponse.status,
        });
      }

      // Handle synchronous JSON response instead of streaming
      const data = await apiResponse.json();
      return NextResponse.json(data);

    } catch (error) {
      console.error('[SUPPORT_CHAT_ERROR]', error);
      return NextResponse.json({ success: false, error: 'Internal Server Error' }, { 
        status: 500,
      });
    }
  });
} 