import { NextRequest, NextResponse } from 'next/server';
import { pg } from '@/app/api/db';

// 텍스트 전처리 함수
function preprocessText(text: string): string[] {
  return text.toLowerCase()
    .replace(/[^\w\s]/g, '') // 특수문자 제거
    .split(/\s+/)            // 공백으로 분리
    .filter(word => word.length > 1); // 1글자 단어 제거
}

// 키워드 매칭 점수 계산
function calculateKeywordMatchScore(query: string, text: string): number {
  const queryWords = preprocessText(query);
  const textWords = preprocessText(text);
  
  // 정확한 키워드 매칭
  const exactMatches = queryWords.filter(word => textWords.includes(word));
  const exactMatchScore = exactMatches.length / queryWords.length;
  
  // 부분 키워드 매칭 (포함 관계)
  const partialMatches = queryWords.filter(qWord => 
    textWords.some(tWord => tWord.includes(qWord) || qWord.includes(tWord))
  );
  const partialMatchScore = partialMatches.length / queryWords.length * 0.5; // 부분 매칭은 가중치 절반
  
  return Math.min(1, exactMatchScore + partialMatchScore);
}

// 코사인 유사도 계산
function calculateCosineSimilarity(text1: string, text2: string): number {
  const words1 = preprocessText(text1);
  const words2 = preprocessText(text2);
  
  const allWords = Array.from(new Set([...words1, ...words2]));
  
  const vector1 = allWords.map(word => words1.filter(w => w === word).length);
  const vector2 = allWords.map(word => words2.filter(w => w === word).length);
  
  const dotProduct = vector1.reduce((sum, val, i) => sum + val * vector2[i], 0);
  const magnitude1 = Math.sqrt(vector1.reduce((sum, val) => sum + val * val, 0));
  const magnitude2 = Math.sqrt(vector2.reduce((sum, val) => sum + val * val, 0));
  
  if (magnitude1 === 0 || magnitude2 === 0) return 0;
  return dotProduct / (magnitude1 * magnitude2);
}

// 최종 유사도 점수 계산
function calculateSimilarityScore(query: string, ticket: any): number {
  const titleWeight = 0.4;
  const questionWeight = 0.6;
  
  // 제목과 질문 각각에 대한 키워드 매칭 점수
  const titleKeywordScore = calculateKeywordMatchScore(query, ticket.title);
  const questionKeywordScore = calculateKeywordMatchScore(query, ticket.question);
  
  // 제목과 질문 각각에 대한 코사인 유사도
  const titleCosineSim = calculateCosineSimilarity(query, ticket.title);
  const questionCosineSim = calculateCosineSimilarity(query, ticket.question);
  
  // 최종 점수 계산 (키워드 매칭 60%, 코사인 유사도 40%)
  const titleScore = (titleKeywordScore * 0.6 + titleCosineSim * 0.4) * titleWeight;
  const questionScore = (questionKeywordScore * 0.6 + questionCosineSim * 0.4) * questionWeight;
  
  return titleScore + questionScore;
}

interface SearchResult {
  results: Array<{
    metadata?: {
      ticket_id?: string;
    };
    relevance_score: number;
  }>;
}

interface Ticket {
  id: string;
  title: string;
  question: string;
  status: string;
  created_at: string;
  author_name: string;
  author_role: string;
  replies: Array<{
    id: string;
    content: string;
    author_name: string;
    author_role: string;
    created_at: string;
  }>;
}

// 검색어 추출 함수
function extractSearchKeywords(query: string): string[] {
  // 1. 불필요한 조사/어미 제거
  const stopwords = ['은', '는', '이', '가', '을', '를', '에', '에서', '로', '으로', '와', '과', '이나', '나', '하고', '의', '도', '만', '에게', '께', '처럼', '같이'];
  
  // 2. 문장 부호 및 특수문자 제거하고 공백으로 분리
  let words = query.replace(/[,.?!~'"'""…()[\]{};:]/g, ' ')
                   .replace(/\s+/g, ' ')
                   .trim()
                   .split(' ');
  
  // 3. 불용어 제거 및 2글자 이상 단어만 선택
  words = words.filter(word => 
    word.length >= 2 && 
    !stopwords.some(stop => word.endsWith(stop))
  );

  // 4. 중복 제거
  return Array.from(new Set(words));
}

async function searchSimilarTickets(query: string, limit: number = 3): Promise<Ticket[]> {
  try {
    // 검색 키워드 추출
    const keywords = extractSearchKeywords(query);
    if (keywords.length === 0) {
      keywords.push(query); // 키워드 추출 실패시 원본 쿼리 사용
    }

    // 각 키워드에 대한 LIKE 조건 생성
    const conditions = keywords.map((_, i) => 
      `(LOWER(t.title) LIKE $${i + 1} OR LOWER(t.question) LIKE $${i + 1})`
    ).join(' OR ');

    // 검색 파라미터 준비 (%keyword%)
    const searchParams = keywords.map(k => `%${k.toLowerCase()}%`);
    searchParams.push(limit.toString());

    // 제목이나 내용에 검색어가 포함된 티켓 검색
    const { rows: tickets } = await pg.query<Ticket>(`
      SELECT 
        t.id,
        t.title,
        t.question,
        t.status,
        t.created_at,
        t.author_name,
        t.author_role,
        COALESCE(
          json_agg(
            json_build_object(
              'id', r.id,
              'content', r.content,
              'author_name', r.author_name,
              'author_role', r.author_role,
              'created_at', r.created_at
            )
          ) FILTER (WHERE r.id IS NOT NULL),
          '[]'
        ) as replies
      FROM support_tickets t
      LEFT JOIN support_ticket_replies r ON t.id = r.ticket_id
      WHERE ${conditions}
      GROUP BY t.id, t.title, t.question, t.status, t.created_at, t.author_name, t.author_role
      ORDER BY t.created_at DESC
      LIMIT $${keywords.length + 1}
    `, [...searchParams]);

    return tickets;

  } catch (error) {
    console.error('Error searching similar tickets:', error);
    return [];
  }
}

export async function POST(req: NextRequest) {
  try {
    const body = await req.json();
    const { query, limit = 3 } = body;

    if (!query) {
      return NextResponse.json({ success: false, error: 'Query is required' }, { status: 400 });
    }

    const similarTickets = await searchSimilarTickets(query, limit);

    return NextResponse.json({
      success: true,
      data: {
        similarTickets
      }
    });

  } catch (error) {
    console.error('Error in similar-tickets route:', error);
    return NextResponse.json(
      { success: false, error: 'Internal server error' },
      { status: 500 }
    );
  }
} 