import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';
import { fileURLToPath } from 'url';
import { dirname } from 'path';
import jwt from 'jsonwebtoken';
import { query } from '../../../lib/database';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// JWT 시크릿 키 (환경변수에서 읽어오거나 기본값 사용)
const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key';

/**
 * 요청에서 사용자 정보를 추출합니다.
 */
async function getUserFromRequest(request: NextRequest) {
    try {
        let token = null;
        console.log('[DEBUG] getUserFromRequest - Authorization header:', request.headers.get('authorization'));
        console.log('[DEBUG] getUserFromRequest - Cookie header:', request.headers.get('cookie'));

        // 1. Authorization 헤더에서 토큰 추출 시도
        const authorization = request.headers.get('authorization');
        if (authorization && authorization.startsWith('Bearer ')) {
            token = authorization.substring(7);
        }

        // 2. Authorization 헤더가 없으면 쿠키에서 토큰 추출 시도
        if (!token) {
            const cookieHeader = request.headers.get('cookie');
            if (cookieHeader) {
                const cookies = cookieHeader.split(';').reduce((acc, cookie) => {
                    const [key, value] = cookie.trim().split('=');
                    acc[key] = value;
                    return acc;
                }, {} as Record<string, string>);

                // 쿠키에서 JWT 토큰 찾기 (일반적인 쿠키 이름들 시도)
                token = cookies['token'] || cookies['auth-token'] || cookies['jwt'] || cookies['authToken'];
            }
        }

        if (!token) {
            return null;
        }

        console.log('[DEBUG] JWT_SECRET used:', JWT_SECRET);
        console.log('[DEBUG] Token to verify:', token);
        const decoded = jwt.verify(token, JWT_SECRET) as any;
        return {
            id: decoded.id,
            username: decoded.username
        };
    } catch (error) {
        console.error('토큰 검증 실패:', error);
        return null;
    }
}

/**
 * 앱의 설정 디렉토리 경로를 가져옵니다.
 */
async function getAppConfigPath() {
    const homedir = os.homedir();
    const configDir = path.join(homedir, '.airun');

    // 디렉토리가 없으면 생성
    if (!fs.existsSync(configDir)) {
        await fs.promises.mkdir(configDir, { recursive: true });
    }

    return configDir;
}

/**
 * 설정에서 현재 프로바이더를 읽어옵니다.
 */
async function getCurrentProvider() {
    try {
        const configDir = await getAppConfigPath();
        const configPath = path.join(configDir, 'airun.conf');

        if (fs.existsSync(configPath)) {
            const configContent = await fs.promises.readFile(configPath, 'utf8');
            const lines = configContent.split('\n');

            for (const line of lines) {
                if (line.startsWith('USE_LLM=')) {
                    return line.split('=')[1].trim().toLowerCase();
                }
            }
        }
    } catch (error) {
        console.error('프로바이더 설정 읽기 실패:', error);
    }

    return 'ollama'; // 기본값
}

/**
 * 프로바이더별 템플릿 파일에서 프롬프트를 읽어옵니다.
 */
async function readProviderPromptTemplate(provider: string) {
    // 프로젝트 루트 디렉토리 경로 계산 (웹 앱에서 메인 프로젝트로)
    const projectRoot = path.resolve(__dirname, '../../../../../..');
    const promptsDir = path.join(projectRoot, 'prompts');

    // 프로바이더별 chat 템플릿 파일 경로 시도
    const chatTemplatePath = path.join(promptsDir, `prompt.template.chat.${provider}`);
    if (fs.existsSync(chatTemplatePath)) {
        return await fs.promises.readFile(chatTemplatePath, 'utf8');
    }

    // 기본 chat 템플릿 시도
    const defaultChatPath = path.join(promptsDir, 'prompt.template.chat');
    if (fs.existsSync(defaultChatPath)) {
        return await fs.promises.readFile(defaultChatPath, 'utf8');
    }

    // 기본 템플릿 시도
    const defaultTemplatePath = path.join(promptsDir, 'prompt.template');
    if (fs.existsSync(defaultTemplatePath)) {
        return await fs.promises.readFile(defaultTemplatePath, 'utf8');
    }

    return null;
}

/**
 * 데이터베이스에서 사용자별 시스템프롬프트를 조회합니다.
 */
async function getUserSystemPrompt(userId: number) {
    try {
        const result = await query(
            'SELECT system_prompt FROM users WHERE id = $1',
            [userId]
        );

        if (result.rows.length > 0) {
            return result.rows[0].system_prompt;
        }

        return null;
    } catch (error) {
        console.error('사용자 시스템프롬프트 조회 실패:', error);
        return null;
    }
}

/**
 * 데이터베이스에 사용자별 시스템프롬프트를 저장합니다.
 */
async function saveUserSystemPrompt(userId: number, prompt: string) {
    try {
        await query(
            'UPDATE users SET system_prompt = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
            [prompt, userId]
        );

        return true;
    } catch (error) {
        console.error('사용자 시스템프롬프트 저장 실패:', error);
        return false;
    }
}

export async function GET(request: NextRequest) {
    try {
        // 사용자 인증 확인
        const user = await getUserFromRequest(request);
        if (!user) {
            return NextResponse.json({
                success: false,
                error: {
                    code: 'UNAUTHORIZED',
                    message: '인증이 필요합니다.'
                },
                timestamp: new Date().toISOString()
            }, { status: 401 });
        }

        // 쿼리 파라미터에서 mode 확인
        const { searchParams } = new URL(request.url);
        const mode = searchParams.get('mode') || 'default';

        let promptContent = '';
        let isCustom = false;

        // 먼저 사용자별 데이터베이스에서 시스템프롬프트 확인
        const userPrompt = await getUserSystemPrompt(user.id);
        if (userPrompt) {
            promptContent = userPrompt;
            isCustom = true;
        } else {
            // 사용자 설정이 없으면 현재 프로바이더의 템플릿 로드
            const currentProvider = await getCurrentProvider();

            // mode가 'chat'인 경우 chat 전용 템플릿만 사용
            if (mode === 'chat') {
                // chat 모드일 때는 chat 전용 템플릿만 사용하여 code 모드용 프롬프트 방지
                const projectRoot = path.resolve(__dirname, '../../../../../..');
                const promptsDir = path.join(projectRoot, 'prompts');
                const chatTemplatePath = path.join(promptsDir, `prompt.template.chat.${currentProvider}`);
                if (fs.existsSync(chatTemplatePath)) {
                    promptContent = await fs.promises.readFile(chatTemplatePath, 'utf8');
                } else {
                    // 프로바이더별 chat 템플릿이 없으면 기본 chat 템플릿 사용
                    const defaultChatPath = path.join(promptsDir, 'prompt.template.chat');
                    if (fs.existsSync(defaultChatPath)) {
                        promptContent = await fs.promises.readFile(defaultChatPath, 'utf8');
                    } else {
                        return NextResponse.json({
                            success: false,
                            error: {
                                code: 'NOT_FOUND',
                                message: 'chat 모드용 시스템 프롬프트 템플릿을 찾을 수 없습니다.'
                            },
                            timestamp: new Date().toISOString()
                        }, { status: 404 });
                    }
                }
            } else {
                // 기본 모드일 때는 기존 로직 사용
                const templateContent = await readProviderPromptTemplate(currentProvider);

                if (templateContent) {
                    promptContent = templateContent;
                } else {
                    return NextResponse.json({
                        success: false,
                        error: {
                            code: 'NOT_FOUND',
                            message: '시스템 프롬프트 템플릿을 찾을 수 없습니다.'
                        },
                        timestamp: new Date().toISOString()
                    }, { status: 404 });
                }
            }
        }

        // 현재 프로바이더 정보도 함께 반환
        const currentProvider = await getCurrentProvider();

        return NextResponse.json({
            success: true,
            data: {
                content: promptContent,
                provider: currentProvider,
                isCustom: isCustom,
                mode: mode // 사용된 모드 정보도 반환
            },
            timestamp: new Date().toISOString()
        });
    } catch (error) {
        console.error('시스템 프롬프트 읽기 실패:', error);

        return NextResponse.json({
            success: false,
            error: {
                code: 'INTERNAL_SERVER_ERROR',
                message: '시스템 프롬프트를 읽는 중 오류가 발생했습니다.'
            },
            timestamp: new Date().toISOString()
        }, { status: 500 });
    }
}

export async function PUT(request: NextRequest) {
    try {
        // 사용자 인증 확인
        const user = await getUserFromRequest(request);
        if (!user) {
            return NextResponse.json({
                success: false,
                error: {
                    code: 'UNAUTHORIZED',
                    message: '인증이 필요합니다.'
                },
                timestamp: new Date().toISOString()
            }, { status: 401 });
        }

        const body = await request.json();
        const { content } = body;

        // 내용 검증 (빈 문자열도 허용)
        if (typeof content !== 'string') {
            return NextResponse.json({
                success: false,
                error: {
                    code: 'INVALID_JSON',
                    message: '프롬프트 내용은 문자열이어야 합니다.'
                },
                timestamp: new Date().toISOString()
            }, { status: 400 });
        }

        // 데이터베이스에 사용자별 시스템프롬프트 저장
        const success = await saveUserSystemPrompt(user.id, content);

        if (!success) {
            return NextResponse.json({
                success: false,
                error: {
                    code: 'INTERNAL_SERVER_ERROR',
                    message: '시스템 프롬프트 저장에 실패했습니다.'
                },
                timestamp: new Date().toISOString()
            }, { status: 500 });
        }

        return NextResponse.json({
            success: true,
            data: {
                message: '시스템 프롬프트가 성공적으로 저장되었습니다.'
            },
            timestamp: new Date().toISOString()
        });
    } catch (error) {
        console.error('시스템 프롬프트 저장 실패:', error);

        return NextResponse.json({
            success: false,
            error: {
                code: 'INTERNAL_SERVER_ERROR',
                message: '시스템 프롬프트 저장 중 오류가 발생했습니다.'
            },
            timestamp: new Date().toISOString()
        }, { status: 500 });
    }
}