'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';

export default function NewCodeProject() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [prompt, setPrompt] = useState('');
  const [language, setLanguage] = useState('');
  const [framework, setFramework] = useState('');
  const [template, setTemplate] = useState('');

  // 언어별 프레임워크 목록
  const frameworks = {
    javascript: ['React', 'Vue', 'Angular', 'Next.js', 'Express'],
    python: ['Django', 'Flask', 'FastAPI', 'Streamlit'],
    java: ['Spring Boot', 'Jakarta EE', 'Quarkus'],
    csharp: ['.NET Core', 'ASP.NET', 'Blazor'],
    php: ['Laravel', 'Symfony', 'CodeIgniter'],
    go: ['Gin', 'Echo', 'Fiber'],
    rust: ['Actix', 'Rocket', 'Axum'],
  };

  // 선택한 언어와 프레임워크에 따른 템플릿 목록
  const templates = {
    'javascript-React': ['SPA', 'PWA', '컴포넌트 라이브러리'],
    'javascript-Next.js': ['블로그', '대시보드', 'E-커머스', 'API'],
    'python-Django': ['블로그', 'REST API', '관리 시스템'],
    'python-FastAPI': ['REST API', 'GraphQL API', '마이크로서비스'],
    'java-Spring Boot': ['REST API', '웹 애플리케이션', '마이크로서비스'],
  };

  // 프레임워크 옵션 목록 생성
  const frameworkOptions = language ? frameworks[language as keyof typeof frameworks] || [] : [];
  
  // 템플릿 옵션 목록 생성
  const templateKey = `${language}-${framework}`;
  const templateOptions = templates[templateKey as keyof typeof templates] || [];

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!prompt) {
      alert('프로젝트 설명을 입력하세요.');
      return;
    }

    try {
      setLoading(true);
      
      // API 호출 시뮬레이션
      await new Promise(resolve => setTimeout(resolve, 1500));
      
      // 성공 시 프로젝트 상세 페이지로 이동
      router.push('/code/list');
    } catch (error) {
      console.error('프로젝트 생성 오류:', error);
      alert('프로젝트 생성 중 오류가 발생했습니다.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="container mx-auto px-4 py-8">
      <h1 className="text-3xl font-bold mb-8 text-center">새 코드 프로젝트 생성</h1>
      
      <div className="max-w-3xl mx-auto bg-white dark:bg-gray-800 rounded-lg shadow p-6">
        <form onSubmit={handleSubmit}>
          <div className="mb-6">
            <label htmlFor="prompt" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              프로젝트 설명
            </label>
            <textarea
              id="prompt"
              rows={5}
              className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
              placeholder="만들고자 하는 프로젝트에 대해 자세히 설명해주세요. (예: 회원 관리 기능이 있는 Todo 앱, 주식 데이터를 분석하는 대시보드 등)"
              value={prompt}
              onChange={(e) => setPrompt(e.target.value)}
              required
            />
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
            <div>
              <label htmlFor="language" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                프로그래밍 언어
              </label>
              <select
                id="language"
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                value={language}
                onChange={(e) => {
                  setLanguage(e.target.value);
                  setFramework('');
                  setTemplate('');
                }}
              >
                <option value="">언어 선택</option>
                <option value="javascript">JavaScript</option>
                <option value="python">Python</option>
                <option value="java">Java</option>
                <option value="csharp">C#</option>
                <option value="php">PHP</option>
                <option value="go">Go</option>
                <option value="rust">Rust</option>
              </select>
            </div>
            
            <div>
              <label htmlFor="framework" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                프레임워크
              </label>
              <select
                id="framework"
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                value={framework}
                onChange={(e) => {
                  setFramework(e.target.value);
                  setTemplate('');
                }}
                disabled={!language}
              >
                <option value="">프레임워크 선택</option>
                {frameworkOptions.map((fw) => (
                  <option key={fw} value={fw}>
                    {fw}
                  </option>
                ))}
              </select>
            </div>
          </div>
          
          {framework && (
            <div className="mb-6">
              <label htmlFor="template" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                프로젝트 템플릿
              </label>
              <select
                id="template"
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                value={template}
                onChange={(e) => setTemplate(e.target.value)}
              >
                <option value="">템플릿 선택 (선택사항)</option>
                {templateOptions.map((tmpl) => (
                  <option key={tmpl} value={tmpl}>
                    {tmpl}
                  </option>
                ))}
              </select>
            </div>
          )}
          
          <div className="flex justify-end mt-8">
            <button
              type="button"
              onClick={() => router.back()}
              className="mr-4 px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
            >
              취소
            </button>
            <button
              type="submit"
              disabled={loading}
              className="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {loading ? (
                <span className="flex items-center">
                  <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  프로젝트 생성 중...
                </span>
              ) : (
                '프로젝트 생성'
              )}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
} 