'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { 
  Search, 
  Plus, 
  Building2, 
  MapPin, 
  Users, 
  Phone, 
  Mail,
  ChevronLeft,
  ChevronRight,
  Edit,
  Trash2
} from 'lucide-react';

interface Company {
  id: number;
  business_number: string;
  company_name: string;
  company_size: string;
  industry_name: string;
  region: string;
  employee_count: number;
  contact_person: string;
  contact_phone: string;
  contact_email: string;
  created_at: string;
}

interface CompanySearchResponse {
  success: boolean;
  data: {
    companies: Company[];
    pagination: {
      current_page: number;
      total_pages: number;
      total_count: number;
      page_size: number;
    };
  };
  message: string;
}

export default function CompaniesPage() {
  const [companies, setCompanies] = useState<Company[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchForm, setSearchForm] = useState({
    company_name: '',
    business_number: '',
    company_size: '',
    region: ''
  });
  const [pagination, setPagination] = useState({
    current_page: 1,
    total_pages: 1,
    total_count: 0,
    page_size: 20
  });

  // 기업 목록 조회
  const fetchCompanies = async (page = 1) => {
    try {
      setLoading(true);
      
      const params = new URLSearchParams({
        page: page.toString(),
        limit: '20'
      });
      
      // 검색 조건 추가
      Object.entries(searchForm).forEach(([key, value]) => {
        if (value.trim()) {
          params.append(key, value.trim());
        }
      });
      
      const response = await fetch(`/api/companies?${params}`);
      
      if (!response.ok) {
        throw new Error('기업 목록 조회 실패');
      }
      
      const data: CompanySearchResponse = await response.json();
      
      if (data.success) {
        setCompanies(data.data.companies);
        setPagination(data.data.pagination);
      } else {
        console.error('기업 목록 조회 실패:', data.message);
      }
    } catch (error) {
      console.error('기업 목록 조회 오류:', error);
    } finally {
      setLoading(false);
    }
  };

  // 검색 실행
  const handleSearch = () => {
    fetchCompanies(1);
  };

  // 검색 초기화
  const handleReset = () => {
    setSearchForm({
      company_name: '',
      business_number: '',
      company_size: '',
      region: ''
    });
    setTimeout(() => fetchCompanies(1), 100);
  };

  // 페이지 변경
  const handlePageChange = (page: number) => {
    fetchCompanies(page);
  };

  // 기업 삭제
  const handleDelete = async (companyId: number, companyName: string) => {
    if (!confirm(`'${companyName}' 기업을 삭제하시겠습니까?`)) {
      return;
    }

    try {
      const response = await fetch(`/api/companies/${companyId}`, {
        method: 'DELETE'
      });

      if (response.ok) {
        alert('기업이 삭제되었습니다.');
        fetchCompanies(pagination.current_page);
      } else {
        const errorData = await response.json();
        alert(`삭제 실패: ${errorData.message || '알 수 없는 오류'}`);
      }
    } catch (error) {
      console.error('기업 삭제 오류:', error);
      alert('기업 삭제 중 오류가 발생했습니다.');
    }
  };

  // 초기 로드
  useEffect(() => {
    fetchCompanies();
  }, []);

  // 기업 규모 배지 색상
  const getSizeBadgeColor = (size: string) => {
    switch (size) {
      case '대기업': return 'bg-blue-100 text-blue-800';
      case '중견기업': return 'bg-green-100 text-green-800';
      case '중소기업': return 'bg-yellow-100 text-yellow-800';
      case '스타트업': return 'bg-purple-100 text-purple-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  // 숫자 포맷팅
  const formatNumber = (num: number | null | undefined) => {
    if (!num) return '미정';
    return num.toLocaleString();
  };

  return (
    <div className="container mx-auto p-6">
      {/* 헤더 */}
      <div className="flex justify-between items-center mb-6">
        <div>
          <h1 className="text-3xl font-bold">기업 관리</h1>
          <p className="text-gray-600 mt-2">등록된 기업 정보를 관리합니다.</p>
        </div>
        <Link href="/companies/new">
          <Button className="flex items-center gap-2">
            <Plus className="w-4 h-4" />
            신규 기업 등록
          </Button>
        </Link>
      </div>

      {/* 검색 필터 */}
      <Card className="mb-6">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Search className="w-5 h-5" />
            기업 검색
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-4">
            <div>
              <label className="block text-sm font-medium mb-2">회사명</label>
              <Input
                placeholder="회사명 검색"
                value={searchForm.company_name}
                onChange={(e) => setSearchForm(prev => ({
                  ...prev,
                  company_name: e.target.value
                }))}
                onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
              />
            </div>
            
            <div>
              <label className="block text-sm font-medium mb-2">사업자등록번호</label>
              <Input
                placeholder="사업자등록번호 검색"
                value={searchForm.business_number}
                onChange={(e) => setSearchForm(prev => ({
                  ...prev,
                  business_number: e.target.value
                }))}
                onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
              />
            </div>
            
            <div>
              <label className="block text-sm font-medium mb-2">기업규모</label>
              <Select
                value={searchForm.company_size}
                onValueChange={(value) => setSearchForm(prev => ({
                  ...prev,
                  company_size: value
                }))}
              >
                <SelectTrigger>
                  <SelectValue placeholder="기업규모 선택" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">전체</SelectItem>
                  <SelectItem value="대기업">대기업</SelectItem>
                  <SelectItem value="중견기업">중견기업</SelectItem>
                  <SelectItem value="중소기업">중소기업</SelectItem>
                  <SelectItem value="스타트업">스타트업</SelectItem>
                </SelectContent>
              </Select>
            </div>
            
            <div>
              <label className="block text-sm font-medium mb-2">지역</label>
              <Input
                placeholder="지역 검색"
                value={searchForm.region}
                onChange={(e) => setSearchForm(prev => ({
                  ...prev,
                  region: e.target.value
                }))}
                onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
              />
            </div>
          </div>
          
          <div className="flex gap-2">
            <Button onClick={handleSearch} className="flex items-center gap-2">
              <Search className="w-4 h-4" />
              검색
            </Button>
            <Button variant="outline" onClick={handleReset}>
              초기화
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* 기업 목록 */}
      <Card>
        <CardHeader>
          <CardTitle className="flex justify-between items-center">
            <span>기업 목록</span>
            <span className="text-sm font-normal text-gray-600">
              총 {formatNumber(pagination.total_count)}개
            </span>
          </CardTitle>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-8">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900 mx-auto"></div>
              <p className="mt-2 text-gray-600">로딩 중...</p>
            </div>
          ) : companies.length === 0 ? (
            <div className="text-center py-8">
              <Building2 className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <p className="text-gray-600">검색 결과가 없습니다.</p>
            </div>
          ) : (
            <>
              {/* 기업 카드 목록 */}
              <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-4 mb-6">
                {companies.map((company) => (
                  <Card key={company.id} className="hover:shadow-md transition-shadow">
                    <CardContent className="p-4">
                      <div className="flex justify-between items-start mb-3">
                        <div className="flex-1">
                          <h3 className="font-semibold text-lg mb-1">{company.company_name}</h3>
                          <p className="text-sm text-gray-600">사업자번호: {company.business_number}</p>
                        </div>
                        <div className="flex gap-1">
                          <Link href={`/companies/${company.id}/edit`}>
                            <Button variant="ghost" size="sm">
                              <Edit className="w-4 h-4" />
                            </Button>
                          </Link>
                          <Button 
                            variant="ghost" 
                            size="sm"
                            onClick={() => handleDelete(company.id, company.company_name)}
                          >
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                      
                      <div className="space-y-2 mb-3">
                        {company.company_size && (
                          <Badge className={getSizeBadgeColor(company.company_size)}>
                            {company.company_size}
                          </Badge>
                        )}
                        
                        {company.industry_name && (
                          <div className="flex items-center gap-1 text-sm text-gray-600">
                            <Building2 className="w-4 h-4" />
                            {company.industry_name}
                          </div>
                        )}
                        
                        {company.region && (
                          <div className="flex items-center gap-1 text-sm text-gray-600">
                            <MapPin className="w-4 h-4" />
                            {company.region}
                          </div>
                        )}
                        
                        {company.employee_count && (
                          <div className="flex items-center gap-1 text-sm text-gray-600">
                            <Users className="w-4 h-4" />
                            {formatNumber(company.employee_count)}명
                          </div>
                        )}
                      </div>
                      
                      <div className="space-y-1 text-xs text-gray-500">
                        {company.contact_person && (
                          <div className="flex items-center gap-1">
                            <span>담당자: {company.contact_person}</span>
                          </div>
                        )}
                        
                        {company.contact_phone && (
                          <div className="flex items-center gap-1">
                            <Phone className="w-3 h-3" />
                            {company.contact_phone}
                          </div>
                        )}
                        
                        {company.contact_email && (
                          <div className="flex items-center gap-1">
                            <Mail className="w-3 h-3" />
                            {company.contact_email}
                          </div>
                        )}
                      </div>
                      
                      <div className="mt-3 pt-3 border-t">
                        <div className="flex justify-between items-center">
                          <span className="text-xs text-gray-500">
                            등록일: {new Date(company.created_at).toLocaleDateString()}
                          </span>
                          <Link href={`/companies/${company.id}`}>
                            <Button variant="outline" size="sm">
                              상세보기
                            </Button>
                          </Link>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>

              {/* 페이지네이션 */}
              {pagination.total_pages > 1 && (
                <div className="flex justify-center items-center gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handlePageChange(pagination.current_page - 1)}
                    disabled={pagination.current_page <= 1}
                  >
                    <ChevronLeft className="w-4 h-4" />
                    이전
                  </Button>
                  
                  <div className="flex gap-1">
                    {Array.from({ length: Math.min(5, pagination.total_pages) }, (_, i) => {
                      const pageNum = i + 1;
                      return (
                        <Button
                          key={pageNum}
                          variant={pageNum === pagination.current_page ? "default" : "outline"}
                          size="sm"
                          onClick={() => handlePageChange(pageNum)}
                        >
                          {pageNum}
                        </Button>
                      );
                    })}
                  </div>
                  
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handlePageChange(pagination.current_page + 1)}
                    disabled={pagination.current_page >= pagination.total_pages}
                  >
                    다음
                    <ChevronRight className="w-4 h-4" />
                  </Button>
                </div>
              )}
            </>
          )}
        </CardContent>
      </Card>
    </div>
  );
}