'use client';

import { useState, useEffect, useRef } from 'react';
import { Upload, FileText, Image, List, Eye, Plus, Trash2, Edit2, ExternalLink, Copy, Check } from 'lucide-react';
import { useRouter } from 'next/navigation';
import * as pdfjsLib from 'pdfjs-dist';
import FormSubmissionModal from '@/components/FormSubmissionModal';
import ResponsesViewModal from '@/components/ResponsesViewModal';

interface FormField {
  id: string;
  label: string;
  type: 'text' | 'email' | 'number' | 'date' | 'textarea' | 'select' | 'checkbox';
  required: boolean;
  options?: string[];
  group?: string; // 필드가 속한 그룹명 (예: "건물현황", "소유자")
  placeholder?: string; // 입력 힌트
}

interface AIForm {
  id: string;
  title: string;
  description: string;
  document_filename: string;
  share_link: string;
  view_link: string;
  response_count: number;
  status: string;
  created_at: string;
}

interface UploadedDocument {
  filename: string;
  preview?: string;
  type: 'pdf' | 'image';
}

type ViewMode = 'upload' | 'document-preview' | 'analyzing' | 'html-preview' | 'form-editor' | 'final';

export default function FormBuilderPage() {
  const router = useRouter();
  const [viewMode, setViewMode] = useState<ViewMode>('upload');
  const [uploadedDocument, setUploadedDocument] = useState<UploadedDocument | null>(null);
  const [uploadedFile, setUploadedFile] = useState<File | null>(null);
  const [pdfPreviewImages, setPdfPreviewImages] = useState<string[]>([]);
  const [analyzedContent, setAnalyzedContent] = useState<string>('');
  const [analyzedHtml, setAnalyzedHtml] = useState<string>('');
  const [analyzedMarkdown, setAnalyzedMarkdown] = useState<string>('');
  const [analyzedStructuredJson, setAnalyzedStructuredJson] = useState<any>(null);
  const [analyzedMetadata, setAnalyzedMetadata] = useState<any>(null);
  const [formFields, setFormFields] = useState<FormField[]>([]);
  const [formTitle, setFormTitle] = useState('');
  const [formDescription, setFormDescription] = useState('');
  const [savedForms, setSavedForms] = useState<AIForm[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [analysisProgress, setAnalysisProgress] = useState(0);
  const [user, setUser] = useState<{ id: string; username: string } | null>(null);
  const [isPdfLoading, setIsPdfLoading] = useState(false);

  // 페이지 선택 관련 state
  const [totalPages, setTotalPages] = useState(0);
  const [selectedPages, setSelectedPages] = useState<number[]>([]);
  const [pageSelectionMode, setPageSelectionMode] = useState<'all' | 'range'>('all');
  const [pageRangeInput, setPageRangeInput] = useState('');

  // 모달 상태
  const [isFormSubmissionModalOpen, setIsFormSubmissionModalOpen] = useState(false);
  const [isResponsesViewModalOpen, setIsResponsesViewModalOpen] = useState(false);
  const [selectedFormShareLink, setSelectedFormShareLink] = useState('');
  const [selectedFormViewLink, setSelectedFormViewLink] = useState('');
  const [copiedLinkId, setCopiedLinkId] = useState<string | null>(null);

  // 탭 상태
  const [activePreviewTab, setActivePreviewTab] = useState<1 | 2>(1);
  const [activeContentTab, setActiveContentTab] = useState<'html' | 'markdown' | 'json'>('html');

  // PDF.js 워커 설정
  useEffect(() => {
    // 로컬 워커 파일 사용
    pdfjsLib.GlobalWorkerOptions.workerSrc = '/pdf-worker/pdf.worker.min.js';
  }, []);

  useEffect(() => {
    // 사용자 정보 로드
    const userId = localStorage.getItem('userId');
    const username = localStorage.getItem('username');
    if (userId && username) {
      setUser({ id: userId, username });
      loadSavedForms(username);
    }
  }, []);

  const loadSavedForms = async (username: string) => {
    try {
      const response = await fetch(`/api/formbuilder/forms?username=${username}`);
      if (response.ok) {
        const data = await response.json();
        setSavedForms(data.data || []);
      }
    } catch (error) {
      console.error('Failed to load saved forms:', error);
    }
  };

  // 오피스 문서 확장자 확인 헬퍼 함수
  const isOfficeDocument = (filename: string): boolean => {
    const ext = filename.toLowerCase().split('.').pop();
    const officeExtensions = ['doc', 'docx', 'odt', 'hwp', 'hwpx', 'ppt', 'pptx', 'xls', 'xlsx'];
    return officeExtensions.includes(ext || '');
  };

  // 오피스 문서를 PDF로 변환하는 함수
  const convertOfficeToPdf = async (file: File): Promise<File> => {
    try {
      console.log('📄 오피스 문서 PDF 변환 시작:', file.name);

      const formData = new FormData();
      formData.append('file', file);

      const response = await fetch('/api/formbuilder/convert-to-pdf', {
        method: 'POST',
        body: formData,
      });

      if (!response.ok) {
        throw new Error('PDF 변환 실패');
      }

      const blob = await response.blob();
      const pdfFile = new File([blob], file.name.replace(/\.[^.]+$/, '.pdf'), { type: 'application/pdf' });
      console.log('✅ PDF 변환 완료:', pdfFile.name);

      return pdfFile;
    } catch (error) {
      console.error('❌ PDF 변환 오류:', error);
      throw error;
    }
  };

  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    let fileType: 'pdf' | 'image' | 'office';
    let processFile = file;

    // 파일 타입 판별
    if (file.type.includes('pdf')) {
      fileType = 'pdf';
    } else if (file.type.includes('image')) {
      fileType = 'image';
    } else if (isOfficeDocument(file.name)) {
      fileType = 'office';
    } else {
      // 기본값: 이미지로 처리
      fileType = 'image';
    }

    setUploadedDocument({
      filename: file.name,
      type: fileType,
      preview: fileType === 'image' ? URL.createObjectURL(file) : undefined
    });
    setUploadedFile(file);
    setViewMode('document-preview');

    if (fileType === 'office') {
      // 오피스 문서는 PDF로 변환 후 이미지로 변환
      setIsPdfLoading(true);
      try {
        processFile = await convertOfficeToPdf(file);
        // 변환된 PDF 파일을 uploadedFile로 설정 (분석 시 사용됨)
        setUploadedFile(processFile);
        // 파일 타입을 PDF로 업데이트
        setUploadedDocument({
          filename: file.name,
          type: 'pdf',
          preview: undefined
        });
        await convertPdfToImages(processFile);
      } catch (error) {
        console.error('오피스 문서 처리 오류:', error);
        alert('문서를 처리하는 중 오류가 발생했습니다. 지원하는 형식인지 확인해주세요.');
        setIsPdfLoading(false);
        return;
      }
      setIsPdfLoading(false);
    } else if (fileType === 'pdf') {
      // PDF를 이미지로 변환 (비동기)
      setIsPdfLoading(true);
      await convertPdfToImages(processFile);
      setIsPdfLoading(false);
    } else {
      // 이미지 파일은 1페이지로 설정
      setTotalPages(1);
      setSelectedPages([1]);
      setPageSelectionMode('all');
    }
  };

  const convertPdfToImages = async (file: File) => {
    try {
      console.log('📄 PDF 변환 시작:', file.name);
      const arrayBuffer = await file.arrayBuffer();
      console.log('📦 ArrayBuffer 생성 완료:', arrayBuffer.byteLength, 'bytes');

      const loadingTask = pdfjsLib.getDocument({ data: arrayBuffer });
      const pdf = await loadingTask.promise;
      console.log('📖 PDF 로드 완료, 총 페이지:', pdf.numPages);

      // 전체 페이지 수 저장
      setTotalPages(pdf.numPages);

      // 기본적으로 모든 페이지 선택
      setSelectedPages(Array.from({ length: pdf.numPages }, (_, i) => i + 1));

      const images: string[] = [];

      // 최대 5페이지까지만 미리보기
      const numPages = Math.min(pdf.numPages, 5);
      console.log(`🖼️ ${numPages}개 페이지를 이미지로 변환 시작 (전체: ${pdf.numPages}페이지)`);

      for (let pageNum = 1; pageNum <= numPages; pageNum++) {
        const page = await pdf.getPage(pageNum);
        const viewport = page.getViewport({ scale: 1.5 });

        const canvas = document.createElement('canvas');
        const context = canvas.getContext('2d');
        if (!context) {
          console.warn(`⚠️ 페이지 ${pageNum}: Canvas context를 가져올 수 없습니다`);
          continue;
        }

        canvas.height = viewport.height;
        canvas.width = viewport.width;

        await page.render({
          canvasContext: context,
          viewport: viewport,
        }).promise;

        const imageData = canvas.toDataURL('image/png');
        images.push(imageData);
        console.log(`✅ 페이지 ${pageNum}/${numPages} 변환 완료 (${Math.round(imageData.length / 1024)}KB)`);
      }

      console.log(`🎉 PDF 변환 완료: ${images.length}개 이미지 생성`);
      setPdfPreviewImages(images);
    } catch (error) {
      console.error('❌ PDF 미리보기 에러:', error);
      console.error('에러 상세:', error instanceof Error ? error.message : String(error));
      setPdfPreviewImages([]);
    }
  };

  // 페이지 범위 파싱 함수
  const parsePageRange = (input: string, maxPages: number): number[] => {
    if (!input.trim()) return [];

    const pages = new Set<number>();
    const parts = input.split(',');

    for (const part of parts) {
      const trimmed = part.trim();
      if (trimmed.includes('-')) {
        // 범위 (예: "1-5")
        const [start, end] = trimmed.split('-').map(s => parseInt(s.trim()));
        if (!isNaN(start) && !isNaN(end)) {
          for (let i = Math.max(1, start); i <= Math.min(maxPages, end); i++) {
            pages.add(i);
          }
        }
      } else {
        // 단일 페이지 (예: "3")
        const page = parseInt(trimmed);
        if (!isNaN(page) && page >= 1 && page <= maxPages) {
          pages.add(page);
        }
      }
    }

    return Array.from(pages).sort((a, b) => a - b);
  };

  const handleAnalyze = async () => {
    if (!uploadedFile) return;

    setIsLoading(true);
    setViewMode('analyzing');
    setAnalysisProgress(0);

    try {
      const formData = new FormData();
      formData.append('file', uploadedFile);

      // 페이지 범위 추가 (PDF이고 전체 페이지가 아닌 경우)
      if (uploadedDocument?.type === 'pdf' && selectedPages.length > 0 && selectedPages.length < totalPages) {
        const pageRange = selectedPages.join(',');
        formData.append('pageRange', pageRange);
        console.log('📄 Selected pages:', pageRange);
      }

      console.log('📤 Uploading file to Chandra API...');

      // 프로그레스 시뮬레이션
      const progressInterval = setInterval(() => {
        setAnalysisProgress(prev => {
          if (prev >= 90) return prev;
          return prev + Math.random() * 10;
        });
      }, 500);

      const response = await fetch('/api/formbuilder/chandra/process', {
        method: 'POST',
        body: formData,
      });

      clearInterval(progressInterval);
      setAnalysisProgress(100);

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Document processing failed');
      }

      const result = await response.json();
      console.log('✅ Chandra processing result:', result);

      if (result.success && result.data) {
        // Chandra 결과 저장
        const content = result.data.html || result.data.markdown || result.data.text || '';
        setAnalyzedContent(content);
        setAnalyzedHtml(result.data.html || '');
        setAnalyzedMarkdown(result.data.markdown || '');
        setAnalyzedStructuredJson(result.data.structuredJson || null);
        setAnalyzedMetadata(result.data.metadata || null);
        setIsLoading(false);

        // HTML 미리보기 단계로 이동
        setViewMode('html-preview');
      } else {
        throw new Error('Invalid response from Chandra');
      }
    } catch (error: any) {
      console.error('Document analysis error:', error);
      alert(`문서 분석 중 오류가 발생했습니다: ${error.message}`);
      setViewMode('document-preview');
      setIsLoading(false);
      setAnalysisProgress(0);
    }
  };

  const generateFormFields = async () => {
    setIsLoading(true);
    try {
      const apiKey = localStorage.getItem('apiKey') || '';

      const response = await fetch('/api/formbuilder/generate-fields', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-API-Key': apiKey,
        },
        body: JSON.stringify({
          html: analyzedHtml,
          markdown: analyzedMarkdown,
          text: analyzedContent,
          metadata: analyzedMetadata,
        }),
      });

      if (response.ok) {
        const data = await response.json();
        setFormFields(data.data.fields || []);
        setFormTitle(data.data.title || '자동 생성된 폼');
        setFormDescription(data.data.description || '');
        setViewMode('form-editor');
      } else {
        throw new Error('Failed to generate form fields');
      }
    } catch (error) {
      console.error('Form field generation error:', error);
      alert('폼 필드 생성 중 오류가 발생했습니다. 기본 필드로 시작합니다.');
      // 기본 필드 생성
      setFormFields([
        { id: '1', label: '이름', type: 'text', required: true },
        { id: '2', label: '이메일', type: 'email', required: true },
      ]);
      setViewMode('form-editor');
    } finally {
      setIsLoading(false);
    }
  };

  const addField = () => {
    const newField: FormField = {
      id: Date.now().toString(),
      label: '새 필드',
      type: 'text',
      required: false,
    };
    setFormFields([...formFields, newField]);
  };

  const updateField = (id: string, updates: Partial<FormField>) => {
    setFormFields(formFields.map(f => f.id === id ? { ...f, ...updates } : f));
  };

  const removeField = (id: string) => {
    setFormFields(formFields.filter(f => f.id !== id));
  };

  const saveForm = async () => {
    if (!user) {
      alert('로그인이 필요합니다.');
      return;
    }

    setIsLoading(true);
    try {
      const response = await fetch('/api/formbuilder/forms', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          title: formTitle,
          description: formDescription,
          document_filename: uploadedDocument?.filename,
          analyzed_content: analyzedContent,
          form_schema: formFields,
          user_id: parseInt(user.id),
          username: user.username,
        }),
      });

      if (response.ok) {
        const data = await response.json();
        alert('폼이 성공적으로 저장되었습니다!');
        setViewMode('final');
        loadSavedForms(user.username);
      } else {
        throw new Error('Form save failed');
      }
    } catch (error) {
      console.error('Form save error:', error);
      alert('폼 저장 중 오류가 발생했습니다.');
    } finally {
      setIsLoading(false);
    }
  };

  const resetBuilder = () => {
    setViewMode('upload');
    setUploadedDocument(null);
    setUploadedFile(null);
    setPdfPreviewImages([]);
    setAnalyzedContent('');
    setAnalyzedHtml('');
    setAnalyzedMarkdown('');
    setAnalyzedStructuredJson(null);
    setAnalyzedMetadata(null);
    setFormFields([]);
    setFormTitle('');
    setFormDescription('');
    setAnalysisProgress(0);
    setActivePreviewTab(1);

    // 페이지 선택 관련 state 초기화
    setTotalPages(0);
    setSelectedPages([]);
    setPageSelectionMode('all');
    setPageRangeInput('');
  };

  const handleDeleteForm = async (formId: string) => {
    if (!confirm('이 폼과 모든 응답 데이터를 삭제하시겠습니까?\n\n삭제된 데이터는 복구할 수 없습니다.')) {
      return;
    }

    try {
      const response = await fetch(`/api/formbuilder/form/${formId}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        alert('폼이 삭제되었습니다.');
        // 저장된 폼 목록 새로고침
        if (user) {
          loadSavedForms(user.username);
        }
      } else {
        throw new Error('Failed to delete form');
      }
    } catch (error) {
      console.error('Delete form error:', error);
      alert('폼 삭제 중 오류가 발생했습니다.');
    }
  };

  const handleCopyLink = async (shareLink: string, formId: string) => {
    const fullUrl = `${window.location.origin}/form/${shareLink}`;
    try {
      await navigator.clipboard.writeText(fullUrl);
      setCopiedLinkId(formId);
      setTimeout(() => setCopiedLinkId(null), 2000);
    } catch (err) {
      console.error('Failed to copy link:', err);
      alert('링크 복사에 실패했습니다.');
    }
  };

  return (
    <div className="flex h-full" style={{ backgroundColor: 'var(--body-bg)', color: 'var(--text-primary)' }}>
      {/* 초기 화면: 전체 화면 레이아웃 */}
      {viewMode === 'upload' ? (
        <div className="flex-1 overflow-y-auto">
          <div className="max-w-7xl mx-auto p-8">
            {/* 상단: 문서 업로드 영역 */}
            <div className="mb-12">

              <label className="formbuilder-upload-zone block w-full border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-xl cursor-pointer hover:border-blue-500 hover:bg-gray-50 dark:hover:bg-gray-800 transition-all">
                <div className="grid grid-cols-2 gap-8 p-12">
                  {/* 왼쪽: 업로드 영역 */}
                  <div className="flex flex-col items-center justify-center">
                    <Upload className="w-20 h-20 mb-6 text-blue-500" />
                    <h2 className="text-2xl font-bold mb-3">문서 업로드</h2>
                    <p className="text-gray-500 dark:text-gray-400 text-center">
                      파일을 드래그하거나<br />클릭하여 업로드하세요
                    </p>
                  </div>

                  {/* 오른쪽: 지원 파일 포맷 */}
                  <div className="flex flex-col justify-center">
                    <h3 className="text-lg font-semibold mb-4 text-gray-700 dark:text-gray-300">지원 파일 포맷</h3>
                    <div className="space-y-3 text-sm text-gray-600 dark:text-gray-400">
                      <div>
                        <p className="font-medium text-gray-700 dark:text-gray-300 mb-1">📄 문서 파일</p>
                        <p>PDF, DOC, DOCX, ODT, HWP, HWPX</p>
                      </div>
                      <div>
                        <p className="font-medium text-gray-700 dark:text-gray-300 mb-1">📊 프레젠테이션 / 스프레드시트</p>
                        <p>PPT, PPTX, XLS, XLSX</p>
                      </div>
                      <div>
                        <p className="font-medium text-gray-700 dark:text-gray-300 mb-1">🖼️ 이미지 파일</p>
                        <p>PNG, JPEG, JPG, WEBP, GIF, TIFF, BMP</p>
                      </div>
                    </div>
                  </div>
                </div>
                <input
                  type="file"
                  className="hidden"
                  accept=".pdf,.doc,.docx,.odt,.hwp,.hwpx,.ppt,.pptx,.xls,.xlsx,image/*"
                  onChange={handleFileUpload}
                  disabled={isLoading}
                />
              </label>
            </div>

            {/* 하단: 저장된 폼 목록 */}
            <div>
              <div className="flex items-center justify-between mb-6">
                <h2 className="text-2xl font-bold">저장된 폼</h2>
                <span className="text-sm text-gray-500 dark:text-gray-400">
                  {savedForms.length}개의 폼
                </span>
              </div>

              {savedForms.length === 0 ? (
                <div className="text-center py-16 bg-gray-50 dark:bg-gray-800/50 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-700">
                  <FileText className="w-16 h-16 mx-auto mb-4 text-gray-400" />
                  <p className="text-gray-500 dark:text-gray-400">아직 저장된 폼이 없습니다</p>
                  <p className="text-sm text-gray-400 dark:text-gray-500 mt-2">문서를 업로드하여 첫 폼을 생성해보세요</p>
                </div>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  {savedForms.map((form) => (
                    <div
                      key={form.id}
                      className="p-5 rounded-lg border border-gray-200 dark:border-gray-700 hover:shadow-lg hover:border-blue-500 transition-all bg-white dark:bg-gray-800"
                    >
                      <div className="flex items-start justify-between mb-3">
                        <h3 className="font-semibold text-lg flex-1 line-clamp-1">{form.title}</h3>
                        <button
                          onClick={() => handleDeleteForm(form.id)}
                          className="p-1.5 hover:bg-red-50 dark:hover:bg-red-900/20 rounded transition-colors flex-shrink-0"
                          title="폼 삭제"
                        >
                          <Trash2 className="w-4 h-4 text-red-600 dark:text-red-400" />
                        </button>
                      </div>

                      <p className="text-sm text-gray-600 dark:text-gray-400 mb-4 line-clamp-2 min-h-[2.5rem]">
                        {form.description || '설명 없음'}
                      </p>

                      <div className="flex flex-col gap-2">
                        <button
                          onClick={() => {
                            setSelectedFormShareLink(form.share_link);
                            setIsFormSubmissionModalOpen(true);
                          }}
                          className="w-full px-3 py-2 text-sm bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors flex items-center justify-center gap-2"
                        >
                          <ExternalLink className="w-4 h-4" />
                          폼 작성
                        </button>

                        <div className="flex gap-2">
                          <button
                            onClick={() => {
                              setSelectedFormViewLink(form.view_link);
                              setIsResponsesViewModalOpen(true);
                            }}
                            className="flex-1 px-3 py-2 text-sm bg-green-600 text-white rounded-md hover:bg-green-700 transition-colors flex items-center justify-center gap-1"
                          >
                            <Eye className="w-4 h-4" />
                            응답 ({form.response_count})
                          </button>

                          <button
                            onClick={() => handleCopyLink(form.share_link, form.id)}
                            className="flex-1 px-3 py-2 text-sm bg-gray-600 text-white rounded-md hover:bg-gray-700 transition-colors flex items-center justify-center gap-1"
                          >
                            {copiedLinkId === form.id ? (
                              <>
                                <Check className="w-4 h-4" />
                                복사됨
                              </>
                            ) : (
                              <>
                                <Copy className="w-4 h-4" />
                                링크
                              </>
                            )}
                          </button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>
        </div>
      ) : (
        <>
          {/* 작업 중일 때: 사이드바 + 메인 영역 */}
          <div className="w-80 border-r border-gray-200 dark:border-gray-700 p-6 overflow-y-auto" style={{ backgroundColor: 'var(--header-bg)' }}>
            <h2 className="text-xl font-bold mb-6">AI 폼 빌더</h2>

            {/* 문서 업로드 */}
            <div className="mb-8">
              <h3 className="text-sm font-semibold mb-3 text-gray-600 dark:text-gray-400">문서 업로드</h3>
              <label className="formbuilder-upload-zone flex flex-col items-center justify-center w-full h-32 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors">
                <div className="flex flex-col items-center">
                  <Upload className="w-8 h-8 mb-2 text-gray-400" />
                  <p className="text-sm text-gray-500 dark:text-gray-400 text-center">문서 또는 이미지<br/>업로드</p>
                </div>
                <input
                  type="file"
                  className="hidden"
                  accept=".pdf,.doc,.docx,.odt,.hwp,.hwpx,.ppt,.pptx,.xls,.xlsx,image/*"
                  onChange={handleFileUpload}
                  disabled={isLoading}
                />
              </label>
              {uploadedDocument && (
                <div className="mt-3 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                  <div className="flex items-center">
                    {uploadedDocument.type === 'pdf' ? (
                      <FileText className="w-5 h-5 mr-2 text-blue-600" />
                    ) : (
                      <Image className="w-5 h-5 mr-2 text-blue-600" />
                    )}
                    <span className="text-sm truncate">{uploadedDocument.filename}</span>
                  </div>
                </div>
              )}
            </div>

            {/* 저장된 폼 목록 */}
            <div>
              <h3 className="text-sm font-semibold mb-3 text-gray-600 dark:text-gray-400">저장된 폼</h3>
              <div className="space-y-2">
                {savedForms.length === 0 ? (
                  <p className="text-sm text-gray-400 dark:text-gray-500 italic">저장된 폼이 없습니다</p>
                ) : (
                  savedForms.map((form) => (
                    <div
                      key={form.id}
                      className="p-3 rounded-lg border border-gray-200 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors"
                    >
                      <div className="flex items-start justify-between mb-1">
                        <h4 className="font-medium text-sm flex-1">{form.title}</h4>
                        <button
                          onClick={() => handleDeleteForm(form.id)}
                          className="p-1 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors flex-shrink-0"
                          title="폼 삭제"
                        >
                          <Trash2 className="w-4 h-4 text-red-600 dark:text-red-400" />
                        </button>
                      </div>
                      <p className="text-xs text-gray-500 dark:text-gray-400 mb-2">{form.description}</p>
                      <div className="flex gap-2 flex-wrap">
                        <button
                          onClick={() => {
                            setSelectedFormShareLink(form.share_link);
                            setIsFormSubmissionModalOpen(true);
                          }}
                          className="text-xs text-blue-600 dark:text-blue-400 hover:underline flex items-center cursor-pointer"
                        >
                          <ExternalLink className="w-3 h-3 mr-1" />
                          폼 작성
                        </button>
                        <button
                          onClick={() => {
                            setSelectedFormViewLink(form.view_link);
                            setIsResponsesViewModalOpen(true);
                          }}
                          className="text-xs text-green-600 dark:text-green-400 hover:underline flex items-center cursor-pointer"
                        >
                          <Eye className="w-3 h-3 mr-1" />
                          응답 보기 ({form.response_count})
                        </button>
                        <button
                          onClick={() => handleCopyLink(form.share_link, form.id)}
                          className="text-xs text-purple-600 dark:text-purple-400 hover:underline flex items-center cursor-pointer"
                        >
                          {copiedLinkId === form.id ? (
                            <>
                              <Check className="w-3 h-3 mr-1" />
                              복사됨
                            </>
                          ) : (
                            <>
                              <Copy className="w-3 h-3 mr-1" />
                              링크 복사
                            </>
                          )}
                        </button>
                      </div>
                    </div>
                  ))
                )}
              </div>
            </div>
          </div>

          {/* 메인 영역 */}
          <div className="flex-1 overflow-y-auto p-8">

        {viewMode === 'document-preview' && uploadedDocument && (
          <div className="h-full flex flex-col">
            <h2 className="text-2xl font-bold mb-4">업로드된 문서</h2>

            <div className="flex-1 flex gap-4">
              {/* 왼쪽: 원본 문서 미리보기 */}
              <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                  <div className="flex items-center justify-between">
                    <h3 className="font-semibold">문서 미리보기</h3>
                    {totalPages > 0 && (
                      <span className="text-xs text-gray-600 dark:text-gray-400">
                        총 {totalPages}페이지
                      </span>
                    )}
                  </div>
                </div>
                <div className="h-[calc(100%-3rem)] overflow-y-auto p-4">
                  {uploadedDocument.type === 'image' && uploadedDocument.preview ? (
                    <img
                      src={uploadedDocument.preview}
                      alt={uploadedDocument.filename}
                      className="max-w-full h-auto rounded-lg shadow-lg"
                    />
                  ) : uploadedDocument.type === 'pdf' && isPdfLoading ? (
                    <div className="flex flex-col items-center justify-center h-full">
                      <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-blue-600 mb-4"></div>
                      <p className="text-lg font-semibold mb-2">PDF를 이미지로 변환 중...</p>
                      <p className="text-sm text-gray-500 dark:text-gray-400">잠시만 기다려주세요</p>
                    </div>
                  ) : uploadedDocument.type === 'pdf' && pdfPreviewImages.length > 0 ? (
                    <div className="space-y-4">
                      {pdfPreviewImages.map((img, index) => (
                        <div key={index}>
                          <p className="text-xs text-gray-500 dark:text-gray-400 mb-2">페이지 {index + 1}</p>
                          <img
                            src={img}
                            alt={`Page ${index + 1}`}
                            className="max-w-full h-auto rounded-lg shadow-lg"
                          />
                        </div>
                      ))}
                      {pdfPreviewImages.length >= 5 && totalPages > 5 && (
                        <div className="text-center py-4 px-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                          <p className="text-sm text-blue-700 dark:text-blue-300 font-medium">
                            ... 더 많은 페이지가 있습니다
                          </p>
                          <p className="text-xs text-blue-600 dark:text-blue-400 mt-1">
                            전체 {totalPages}페이지 중 {pdfPreviewImages.length}페이지 미리보기
                          </p>
                          <p className="text-xs text-gray-600 dark:text-gray-400 mt-2">
                            오른쪽에서 처리할 페이지를 선택할 수 있습니다
                          </p>
                        </div>
                      )}
                    </div>
                  ) : uploadedDocument.type === 'pdf' ? (
                    <div className="flex flex-col items-center justify-center h-full">
                      <FileText className="w-24 h-24 text-gray-400 dark:text-gray-500 mb-4" />
                      <p className="text-lg font-semibold mb-2">{uploadedDocument.filename}</p>
                      <p className="text-gray-500 dark:text-gray-400">PDF 미리보기를 생성할 수 없습니다</p>
                      <p className="text-sm text-gray-400 dark:text-gray-500 mt-2">브라우저 콘솔에서 에러를 확인하세요</p>
                    </div>
                  ) : (
                    <div className="flex flex-col items-center justify-center h-full">
                      <FileText className="w-24 h-24 text-gray-400 dark:text-gray-500 mb-4" />
                      <p className="text-lg font-semibold mb-2">{uploadedDocument.filename}</p>
                      <p className="text-gray-500 dark:text-gray-400">문서를 불러오는 중...</p>
                    </div>
                  )}
                </div>
              </div>

              {/* 오른쪽: 액션 영역 */}
              <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                  <h3 className="font-semibold">다음 단계</h3>
                </div>
                <div className="h-[calc(100%-3rem)] overflow-y-auto p-8">
                  <div className="text-center mb-8">
                    <div className="w-20 h-20 bg-blue-100 dark:bg-blue-900/30 rounded-full flex items-center justify-center mx-auto mb-4">
                      <FileText className="w-10 h-10 text-blue-600" />
                    </div>
                    <h3 className="text-xl font-bold mb-3">문서 업로드 완료</h3>
                    <p className="text-gray-600 dark:text-gray-400 mb-2">
                      AI가 문서를 분석하여 자동으로<br/>입력 폼을 생성할 준비가 되었습니다
                    </p>
                  </div>

                  {/* 페이지 선택 옵션 (PDF만) */}
                  {uploadedDocument.type === 'pdf' && totalPages > 0 && (
                    <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-900/30 rounded-lg border border-gray-200 dark:border-gray-700">
                      <h4 className="font-semibold mb-3 text-sm">페이지 선택</h4>
                      <div className="text-xs text-gray-600 dark:text-gray-400 mb-3">
                        총 {totalPages}페이지
                      </div>

                      <div className="space-y-3">
                        {/* 전체 페이지 */}
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="pageSelection"
                            checked={pageSelectionMode === 'all'}
                            onChange={() => {
                              setPageSelectionMode('all');
                              setSelectedPages(Array.from({ length: totalPages }, (_, i) => i + 1));
                            }}
                            className="w-4 h-4 text-blue-600"
                          />
                          <span className="text-sm">전체 페이지 ({totalPages}페이지)</span>
                        </label>

                        {/* 페이지 범위 지정 */}
                        <div>
                          <label className="flex items-center gap-2 cursor-pointer mb-2">
                            <input
                              type="radio"
                              name="pageSelection"
                              checked={pageSelectionMode === 'range'}
                              onChange={() => setPageSelectionMode('range')}
                              className="w-4 h-4 text-blue-600"
                            />
                            <span className="text-sm">페이지 범위 지정</span>
                          </label>

                          {pageSelectionMode === 'range' && (
                            <div className="ml-6 space-y-2">
                              <input
                                type="text"
                                value={pageRangeInput}
                                onChange={(e) => {
                                  setPageRangeInput(e.target.value);
                                  // 페이지 범위 파싱 (예: "1-3,5,7-9")
                                  const pages = parsePageRange(e.target.value, totalPages);
                                  setSelectedPages(pages);
                                }}
                                placeholder="예: 1-3,5,7"
                                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800"
                              />
                              <p className="text-xs text-gray-500 dark:text-gray-400">
                                선택된 페이지: {selectedPages.length}개
                              </p>
                            </div>
                          )}
                        </div>
                      </div>
                    </div>
                  )}

                  <div className="w-full max-w-md space-y-3 mx-auto">
                    <button
                      onClick={handleAnalyze}
                      disabled={isLoading || selectedPages.length === 0}
                      className="w-full px-6 py-4 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-semibold text-lg shadow-lg"
                    >
                      {isLoading ? '분석 중...' : `🔍 문서 분석 시작하기${selectedPages.length > 0 && selectedPages.length < totalPages ? ` (${selectedPages.length}페이지)` : ''}`}
                    </button>
                    <button
                      onClick={resetBuilder}
                      className="w-full px-6 py-3 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors font-medium"
                    >
                      다시 업로드
                    </button>
                  </div>

                  <div className="mt-8 text-sm text-gray-500 dark:text-gray-400 text-center">
                    <p>💡 분석 시간은 문서 크기에 따라 다를 수 있습니다</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}

        {viewMode === 'analyzing' && uploadedDocument && (
          <div className="h-full flex flex-col">
            <h2 className="text-2xl font-bold mb-4">문서 분석 중</h2>

            <div className="flex-1 flex gap-4">
              {/* 왼쪽: 원본 문서 */}
              <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                  <h3 className="font-semibold">원본 문서</h3>
                </div>
                <div className="h-[calc(100%-3rem)] overflow-y-auto p-4">
                  {uploadedDocument.type === 'image' && uploadedDocument.preview ? (
                    <img
                      src={uploadedDocument.preview}
                      alt={uploadedDocument.filename}
                      className="max-w-full h-auto rounded-lg shadow-lg"
                    />
                  ) : uploadedDocument.type === 'pdf' && pdfPreviewImages.length > 0 ? (
                    <div className="space-y-4">
                      {pdfPreviewImages.map((img, index) => (
                        <div key={index}>
                          <p className="text-xs text-gray-500 dark:text-gray-400 mb-2">페이지 {index + 1}</p>
                          <img src={img} alt={`Page ${index + 1}`} className="max-w-full h-auto rounded-lg shadow-lg" />
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="flex items-center justify-center h-full">
                      <FileText className="w-16 h-16 text-gray-400" />
                    </div>
                  )}
                </div>
              </div>

              {/* 오른쪽: 분석 진행 상태 */}
              <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                  <h3 className="font-semibold">AI 분석 진행</h3>
                </div>
                <div className="h-[calc(100%-3rem)] flex flex-col items-center justify-center p-8">
                  <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-blue-600 mb-6"></div>
                  <h3 className="text-xl font-bold mb-3">분석 중...</h3>
                  <p className="text-gray-500 dark:text-gray-400 mb-8 text-center">
                    문서 내용을 분석하고<br/>폼 필드를 자동 생성하고 있습니다
                  </p>

                  {/* 프로그레스 바 */}
                  <div className="w-full max-w-md">
                    <div className="bg-gray-200 dark:bg-gray-700 rounded-full h-4 overflow-hidden mb-3">
                      <div
                        className="bg-blue-600 h-full transition-all duration-500 ease-out rounded-full"
                        style={{ width: `${analysisProgress}%` }}
                      ></div>
                    </div>
                    <p className="text-center text-sm text-gray-600 dark:text-gray-400">
                      {Math.round(analysisProgress)}% 완료
                    </p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}

        {viewMode === 'html-preview' && uploadedDocument && (
          <div className="h-full flex flex-col">
            <div className="mb-4 flex justify-between items-center">
              <h2 className="text-2xl font-bold">문서 분석 결과</h2>
              <div className="flex gap-3">
                <button
                  onClick={generateFormFields}
                  disabled={isLoading}
                  className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-semibold"
                >
                  {isLoading ? '폼 생성 중...' : '🤖 AI로 폼 생성하기'}
                </button>
                <button
                  onClick={resetBuilder}
                  className="px-4 py-3 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
                >
                  다시 시작
                </button>
              </div>
            </div>

            {/* 탭 내비게이션 */}
            <div className="mb-4 flex gap-2 border-b border-gray-200 dark:border-gray-700">
              <button
                onClick={() => setActivePreviewTab(1)}
                className={`px-6 py-3 font-medium transition-colors ${
                  activePreviewTab === 1
                    ? 'text-blue-600 border-b-2 border-blue-600'
                    : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200'
                }`}
              >
                문서 미리보기
              </button>
              <button
                onClick={() => setActivePreviewTab(2)}
                className={`px-6 py-3 font-medium transition-colors ${
                  activePreviewTab === 2
                    ? 'text-blue-600 border-b-2 border-blue-600'
                    : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200'
                }`}
              >
                HTML 소스
              </button>
            </div>

            {/* 탭 내용 */}
            <div className="flex-1 overflow-hidden">
              {/* Tab 1: 문서 미리보기 + 렌더링 결과 */}
              {activePreviewTab === 1 && (
              <div className="h-full flex gap-4">
                {/* 왼쪽: 원본 문서 미리보기 */}
                <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                  <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                    <h3 className="font-semibold">원본 문서</h3>
                  </div>
                  <div className="h-[calc(100%-3rem)] overflow-y-auto p-4">
                    {uploadedDocument.type === 'image' && uploadedDocument.preview ? (
                      <img
                        src={uploadedDocument.preview}
                        alt={uploadedDocument.filename}
                        className="max-w-full h-auto rounded-lg shadow-lg"
                      />
                    ) : uploadedDocument.type === 'pdf' && pdfPreviewImages.length > 0 ? (
                      <div className="space-y-4">
                        {pdfPreviewImages.map((img, index) => (
                          <div key={index}>
                            <p className="text-xs text-gray-500 dark:text-gray-400 mb-2">페이지 {index + 1}</p>
                            <img src={img} alt={`Page ${index + 1}`} className="max-w-full h-auto rounded-lg shadow-lg" />
                          </div>
                        ))}
                      </div>
                    ) : (
                      <div className="flex items-center justify-center h-full">
                        <FileText className="w-16 h-16 text-gray-400" />
                      </div>
                    )}
                  </div>
                </div>

                {/* 오른쪽: 분석된 HTML/Markdown 렌더링 */}
                <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                  <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                    <div className="flex items-center justify-between">
                      <h3 className="font-semibold">분석 결과</h3>
                      <div className="flex gap-2">
                        <button
                          onClick={() => setActiveContentTab('html')}
                          className={`px-3 py-1 text-sm rounded ${
                            activeContentTab === 'html'
                              ? 'bg-blue-600 text-white'
                              : 'bg-gray-200 dark:bg-gray-600 text-gray-700 dark:text-gray-300'
                          }`}
                        >
                          HTML
                        </button>
                        <button
                          onClick={() => setActiveContentTab('markdown')}
                          className={`px-3 py-1 text-sm rounded ${
                            activeContentTab === 'markdown'
                              ? 'bg-blue-600 text-white'
                              : 'bg-gray-200 dark:bg-gray-600 text-gray-700 dark:text-gray-300'
                          }`}
                        >
                          Markdown
                        </button>
                        <button
                          onClick={() => setActiveContentTab('json')}
                          className={`px-3 py-1 text-sm rounded ${
                            activeContentTab === 'json'
                              ? 'bg-blue-600 text-white'
                              : 'bg-gray-200 dark:bg-gray-600 text-gray-700 dark:text-gray-300'
                          }`}
                        >
                          JSON
                        </button>
                      </div>
                    </div>
                  </div>
                  <div className="h-[calc(100%-3rem)] overflow-y-auto p-4">
                    {activeContentTab === 'html' && (
                      analyzedHtml ? (
                        <div
                          className="prose dark:prose-invert max-w-none html-preview-content"
                          dangerouslySetInnerHTML={{ __html: analyzedHtml }}
                        />
                      ) : (
                        <p className="text-gray-500 dark:text-gray-400">HTML 결과가 없습니다</p>
                      )
                    )}
                    {activeContentTab === 'markdown' && (
                      analyzedMarkdown ? (
                        <div className="prose dark:prose-invert max-w-none whitespace-pre-wrap">
                          {analyzedMarkdown}
                        </div>
                      ) : (
                        <p className="text-gray-500 dark:text-gray-400">Markdown 결과가 없습니다</p>
                      )
                    )}
                    {activeContentTab === 'json' && (
                      analyzedStructuredJson ? (
                        <pre className="text-xs bg-gray-50 dark:bg-gray-900 p-4 rounded overflow-auto">
                          {JSON.stringify(analyzedStructuredJson, null, 2)}
                        </pre>
                      ) : (
                        <p className="text-gray-500 dark:text-gray-400">JSON 구조화 데이터가 없습니다</p>
                      )
                    )}
                  </div>
                </div>
              </div>
              )}

              {/* Tab 2: HTML 소스 + 폼 미리보기 */}
              {activePreviewTab === 2 && (
              <div className="h-full flex gap-4">
                {/* 왼쪽: HTML 소스 코드 */}
                <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                  <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                    <h3 className="font-semibold">HTML 소스</h3>
                  </div>
                  <div className="h-[calc(100%-3rem)] overflow-y-auto p-4">
                    <pre className="text-xs bg-gray-50 dark:bg-gray-900 p-4 rounded-lg overflow-x-auto">
                      <code>{analyzedHtml || analyzedMarkdown || '소스 없음'}</code>
                    </pre>
                  </div>
                </div>

                {/* 오른쪽: 폼 생성 안내 */}
                <div className="w-1/2 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                  <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                    <h3 className="font-semibold">폼 생성</h3>
                  </div>
                  <div className="h-[calc(100%-3rem)] flex flex-col items-center justify-center p-8">
                    <div className="text-center mb-6">
                      <div className="w-16 h-16 bg-blue-100 dark:bg-blue-900/20 rounded-full flex items-center justify-center mx-auto mb-4">
                        <svg className="w-8 h-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                      </div>
                      <h3 className="text-lg font-bold mb-2">AI 폼 생성</h3>
                      <p className="text-gray-500 dark:text-gray-400 text-sm">
                        문서 구조를 분석하여<br/>자동으로 입력 폼을 생성합니다
                      </p>
                    </div>
                    <button
                      onClick={generateFormFields}
                      disabled={isLoading}
                      className="px-8 py-4 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-semibold text-lg"
                    >
                      {isLoading ? '폼 생성 중...' : '🤖 폼 생성하기'}
                    </button>
                  </div>
                </div>
              </div>
              )}
            </div>
          </div>
        )}

        {viewMode === 'form-editor' && uploadedDocument && (
          <div className="h-full flex flex-col">
            <h2 className="text-2xl font-bold mb-4">폼 편집</h2>

            <div className="flex-1 flex gap-4">
              {/* 왼쪽: 원본 문서 및 분석 결과 */}
              <div className="w-1/2 flex flex-col gap-4">
                {/* 원본 문서 */}
                <div className="flex-1 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                  <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                    <h3 className="font-semibold">원본 문서</h3>
                  </div>
                  <div className="h-[calc(100%-3rem)] overflow-y-auto p-4">
                    {uploadedDocument.type === 'image' && uploadedDocument.preview ? (
                      <img
                        src={uploadedDocument.preview}
                        alt={uploadedDocument.filename}
                        className="max-w-full h-auto rounded-lg shadow-lg"
                      />
                    ) : uploadedDocument.type === 'pdf' && pdfPreviewImages.length > 0 ? (
                      <div className="space-y-4">
                        {pdfPreviewImages.map((img, index) => (
                          <div key={index}>
                            <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">페이지 {index + 1}</p>
                            <img src={img} alt={`Page ${index + 1}`} className="max-w-full h-auto rounded-lg shadow-sm" />
                          </div>
                        ))}
                      </div>
                    ) : (
                      <div className="flex items-center justify-center h-full">
                        <FileText className="w-16 h-16 text-gray-400" />
                      </div>
                    )}
                  </div>
                </div>
              </div>

              {/* 오른쪽: 폼 편집 */}
              <div className="w-1/2 flex flex-col">
                <div className="flex-1 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                  <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 border-b border-gray-200 dark:border-gray-600">
                    <h3 className="font-semibold">폼 설정</h3>
                  </div>
                  <div className="h-[calc(100%-3rem)] overflow-y-auto p-4">
                    {/* 폼 제목 및 설명 */}
                    <div className="mb-6">
                      <input
                        type="text"
                        value={formTitle}
                        onChange={(e) => setFormTitle(e.target.value)}
                        placeholder="폼 제목"
                        className="w-full px-4 py-3 text-xl font-bold border border-gray-300 dark:border-gray-600 rounded-lg mb-3 bg-white dark:bg-gray-800"
                      />
                      <textarea
                        value={formDescription}
                        onChange={(e) => setFormDescription(e.target.value)}
                        placeholder="폼 설명"
                        rows={2}
                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800"
                      />
                    </div>

                    {/* 폼 필드 목록 (그룹별) */}
                    <div className="space-y-4 mb-6">
                      <h4 className="font-semibold text-sm text-gray-600 dark:text-gray-400">폼 필드</h4>
                      {(() => {
                        // 필드를 그룹별로 묶기
                        const fieldsByGroup: { [key: string]: FormField[] } = {};
                        formFields.forEach(field => {
                          const groupName = field.group || '';
                          if (!fieldsByGroup[groupName]) {
                            fieldsByGroup[groupName] = [];
                          }
                          fieldsByGroup[groupName].push(field);
                        });

                        // 그룹별로 렌더링
                        return Object.entries(fieldsByGroup).map(([groupName, groupFields]) => (
                          <div key={groupName || 'no-group'} className="space-y-2">
                            {groupName && (
                              <div className="font-semibold text-blue-600 dark:text-blue-400 bg-blue-50 dark:bg-blue-900/20 px-3 py-2 rounded-lg border-l-4 border-blue-600">
                                📋 {groupName}
                              </div>
                            )}
                            <div className="space-y-2 pl-2">
                              {groupFields.map((field) => (
                                <div key={field.id} className="p-3 border border-gray-200 dark:border-gray-700 rounded-lg bg-gray-50 dark:bg-gray-900">
                                  <div className="flex items-center gap-2 mb-2">
                                    <input
                                      type="text"
                                      value={field.label}
                                      onChange={(e) => updateField(field.id, { label: e.target.value })}
                                      placeholder="필드 라벨"
                                      className="flex-1 px-2 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800"
                                    />
                                    <button
                                      onClick={() => removeField(field.id)}
                                      className="p-1 text-red-600 hover:bg-red-50 dark:hover:bg-red-900/20 rounded"
                                    >
                                      <Trash2 className="w-4 h-4" />
                                    </button>
                                  </div>
                                  <div className="flex items-center gap-2 mb-1">
                                    <select
                                      value={field.type}
                                      onChange={(e) => updateField(field.id, { type: e.target.value as FormField['type'] })}
                                      className="flex-1 px-2 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800"
                                    >
                                      <option value="text">텍스트</option>
                                      <option value="email">이메일</option>
                                      <option value="number">숫자</option>
                                      <option value="date">날짜</option>
                                      <option value="textarea">긴 텍스트</option>
                                      <option value="select">선택</option>
                                      <option value="checkbox">체크박스</option>
                                    </select>
                                    <label className="flex items-center text-sm">
                                      <input
                                        type="checkbox"
                                        checked={field.required}
                                        onChange={(e) => updateField(field.id, { required: e.target.checked })}
                                        className="mr-1"
                                      />
                                      <span>필수</span>
                                    </label>
                                  </div>
                                  {field.group && (
                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                      그룹: {field.group}
                                    </div>
                                  )}
                                </div>
                              ))}
                            </div>
                          </div>
                        ));
                      })()}
                    </div>

                    <div className="flex flex-col gap-2">
                      <button
                        onClick={addField}
                        className="w-full px-4 py-2 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors flex items-center justify-center"
                      >
                        <Plus className="w-4 h-4 mr-2" />
                        필드 추가
                      </button>
                      <button
                        onClick={saveForm}
                        disabled={isLoading || !formTitle || formFields.length === 0}
                        className="w-full px-4 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-semibold"
                      >
                        {isLoading ? '저장 중...' : '💾 폼 저장'}
                      </button>
                      <button
                        onClick={resetBuilder}
                        className="w-full px-4 py-2 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
                      >
                        다시 시작
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}

        {viewMode === 'final' && (
          <div className="flex flex-col items-center justify-center h-full">
            <div className="bg-green-100 dark:bg-green-900/20 rounded-full p-6 mb-4">
              <svg className="w-16 h-16 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
              </svg>
            </div>
            <h2 className="text-2xl font-bold mb-2">폼이 생성되었습니다!</h2>
            <p className="text-gray-500 dark:text-gray-400 mb-6">왼쪽 사이드바에서 폼 링크를 확인하세요</p>
            <button
              onClick={resetBuilder}
              className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              새 폼 만들기
            </button>
          </div>
        )}
          </div>
        </>
      )}

      {/* 모달들 */}
      <FormSubmissionModal
        isOpen={isFormSubmissionModalOpen}
        onClose={() => setIsFormSubmissionModalOpen(false)}
        shareLink={selectedFormShareLink}
        showShareLink={false}
      />

      <ResponsesViewModal
        isOpen={isResponsesViewModalOpen}
        onClose={() => setIsResponsesViewModalOpen(false)}
        viewLink={selectedFormViewLink}
      />
    </div>
  );
}
