'use client';

import { useState, useEffect, useCallback, useRef, Suspense } from 'react';
import { useAuth } from '@/hooks/useAuth';
import { useSearchParams } from 'next/navigation';
import { Bar } from 'react-chartjs-2';
import { Line } from 'react-chartjs-2';
import { Chart, BarElement, CategoryScale, LinearScale, Tooltip, Legend, LineController, LineElement, PointElement, TimeScale, ArcElement } from 'chart.js';
import 'chartjs-adapter-date-fns';
import { Doughnut } from 'react-chartjs-2';

Chart.register(BarElement, CategoryScale, LinearScale, Tooltip, Legend, LineController, LineElement, PointElement, TimeScale, ArcElement);

interface SystemStats {
  cpu_usage: number;
  memory_usage: number;
  disk_usage: number;
  active_sessions: number;
  total_requests: number;
  error_rate: number;
  uptime: string;
  postgresql?: {
    current_connections: number;
    max_connections: number;
    connection_usage_percent: number;
  };
}

interface ServiceStatus {
  name: string;
  status: 'Online' | 'Offline' | 'Error';
  details: string;
}

interface LogEntry {
  timestamp: string;
  level?: string;
  message?: string;
  source?: string;
  details?: any;
  userId?: string;
  fileName?: string;
  lineNumber?: number;
}

interface UsageStats {
  daily_requests: number;
  weekly_requests: number;
  monthly_requests: number;
  top_users: Array<{
    user_id: string;
    username?: string;
    request_count: number;
  }>;
  feature_usage: Record<string, number>;
  total_cost?: number;
  total_tokens?: number;
  monthly_cost?: number;
  monthly_tokens?: number;
  provider_stats?: Record<string, any>;
  heatmap?: number[][];
  call_details?: any[];
  session_stats?: any[];
  today_session_bubble?: any[];
  model_stats?: any[];
  monthly_stats?: Record<string, { cost: number; tokens: number }>;
}

// CSV 다운로드 유틸(컴포넌트 하단에서 쓸 수 있게 일반 함수로 선언)
function downloadCSVFile(content: string, filename: string) {
  const blob = new Blob([content], { type: 'text/csv;charset=utf-8;' });
  const url = URL.createObjectURL(blob);
  const link = document.createElement('a');
  link.setAttribute('href', url);
  link.setAttribute('download', filename);
  link.style.visibility = 'hidden';
  document.body.appendChild(link);
  link.click();
  document.body.removeChild(link);
  URL.revokeObjectURL(url);
}

interface ApiStats {
  endpointStats: Record<string, { calls: number; avgResponseTime: number; successRate: number }>;
  callsHeatmap: number[][];
  timeLabels: string[];
  responseTimes: number[];
  errorRates: number[];
  logs: any[];
  totalCalls: number;
  limitedCalls: number;
  avgResponseTime: number;
  successRate: number;
}

function MonitoringPageContent() {
  const [systemStats, setSystemStats] = useState<SystemStats | null>(null);
  const [serviceStatuses, setServiceStatuses] = useState<ServiceStatus[]>([]);
  const [logs, setLogs] = useState<LogEntry[]>([]);
  const [usageStats, setUsageStats] = useState<UsageStats | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const searchParams = useSearchParams();
  const initialTab = searchParams.get('tab') as 'system' | 'logs' | 'usage' | 'api' | 'flop' || 'usage';
  const [activeTab, setActiveTab] = useState<'system' | 'logs' | 'usage' | 'api' | 'flop'>(initialTab);
  const [autoRefresh, setAutoRefresh] = useState(true);
  const { user, isLoading: authLoading } = useAuth();
  const [apiStats, setApiStats] = useState<ApiStats | null>(null);
  const [flopStats, setFlopStats] = useState<any>(null);
  const [logPage, setLogPage] = useState(1);
  const logsPerPage = 10;
  const totalPages = Math.ceil(logs.length / logsPerPage);
  const pagedLogs = logs.slice((logPage - 1) * logsPerPage, logPage * logsPerPage);


  // 다크모드 대응 차트 옵션 유틸리티
  const getChartOptions = (additionalOptions: any = {}) => {
    const isDark = typeof document !== 'undefined' && document.documentElement.classList.contains('dark');
    const textColor = isDark ? '#ffffff' : '#374151';
    const gridColor = isDark ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
    
    const baseOptions = {
      responsive: true,
      plugins: {
        legend: {
          labels: {
            color: textColor,
            ...additionalOptions.plugins?.legend?.labels
          },
          ...additionalOptions.plugins?.legend
        },
        tooltip: {
          backgroundColor: isDark ? 'rgba(0, 0, 0, 0.8)' : 'rgba(255, 255, 255, 0.9)',
          titleColor: textColor,
          bodyColor: textColor,
          borderColor: gridColor,
          borderWidth: 1,
          ...additionalOptions.plugins?.tooltip
        },
        ...additionalOptions.plugins
      },
      scales: {
        x: {
          ticks: {
            color: textColor
          },
          grid: {
            color: gridColor
          },
          title: {
            color: textColor,
            ...additionalOptions.scales?.x?.title
          },
          ...additionalOptions.scales?.x
        },
        y: {
          ticks: {
            color: textColor
          },
          grid: {
            color: gridColor
          },
          title: {
            color: textColor,
            ...additionalOptions.scales?.y?.title
          },
          ...additionalOptions.scales?.y
        },
        ...additionalOptions.scales
      },
      ...additionalOptions
    };
    
    return baseOptions;
  };

  // 권한 분기
  const isAdmin = user?.role === 'admin';
  const userId = user?.id;

  useEffect(() => {
    loadData();
    let interval: NodeJS.Timeout;
    if (autoRefresh) {
      interval = setInterval(loadData, 30000); // 30초마다 새로고침
    }
    return () => {
      if (interval) clearInterval(interval);
    };
  }, [autoRefresh]);

  // API 대시보드 데이터 로드 (관리자만)
  useEffect(() => {
    if (!isAdmin) return;
    fetch('/api/monitoring/api')
      .then((r) => r.json())
      .then((res) => {
        if (res.success && res.data) setApiStats(res.data);
      });
  }, [user, isAdmin]);

  // FLOP 데이터 로드
  useEffect(() => {
    if (!user || typeof window === 'undefined') return;
    
    const loadFlopData = async () => {
      try {
        // 토큰 가져오기 및 검증 (Next.js 앱에서는 userToken으로 저장)
        const token = localStorage.getItem('userToken') || localStorage.getItem('token') || localStorage.getItem('auth-token') || sessionStorage.getItem('token');
        if (!token) {
          console.warn('JWT 토큰이 없습니다.');
          setError('인증 토큰이 없습니다. 다시 로그인해주세요.');
          return;
        }

        // 토큰 형식 확인 (Bearer 접두사 제거)
        const cleanToken = token.replace(/^Bearer\s+/i, '').trim();
        
        // 토큰이 올바른 JWT 형식인지 확인 (3개 부분으로 구성)
        if (!cleanToken || cleanToken.split('.').length !== 3) {
          console.warn('JWT 토큰 형식이 올바르지 않습니다. 토큰 길이:', cleanToken.length);
          console.warn('토큰 부분 수:', cleanToken.split('.').length);
          setError(`JWT 토큰 형식이 올바르지 않습니다. (부분 수: ${cleanToken.split('.').length})`);
          return;
        }

        console.log('FLOP API 호출 시작 - 토큰 유효');

        const { getApiServerUrl } = await import('@/config/serverConfig');
        const apiServerUrl = getApiServerUrl();

        const [usageRes, licenseRes] = await Promise.all([
          
          fetch(`${apiServerUrl}/flop/usage`, {
            headers: {
              'Authorization': `Bearer ${cleanToken}`,
              'Content-Type': 'application/json'
            },
            credentials: 'include'
          }),
          fetch(`${apiServerUrl}/flop/license`, {
            headers: {
              'Authorization': `Bearer ${cleanToken}`,
              'Content-Type': 'application/json'
            },
            credentials: 'include'
          })
        ]);

        console.log('FLOP API 응답 상태:', {
          usage: usageRes.status,
          license: licenseRes.status
        });

        if (!usageRes.ok || !licenseRes.ok) {
          const errorInfo = {
            usage: { status: usageRes.status, statusText: usageRes.statusText },
            license: { status: licenseRes.status, statusText: licenseRes.statusText }
          };
          console.error('FLOP API 호출 실패:', errorInfo);
          setError(`FLOP API 호출 실패: ${!usageRes.ok ? `사용량 ${usageRes.status}` : ''} ${!licenseRes.ok ? `라이선스 ${licenseRes.status}` : ''}`);
          return;
        }

        const [usageData, licenseData] = await Promise.all([
          usageRes.json(),
          licenseRes.json()
        ]);

        console.log('FLOP 데이터 수신:', { usageData, licenseData });

        if (usageData.success && licenseData.success) {
          setFlopStats({
            usage: usageData.data,
            license: licenseData.data
          });
          console.log('FLOP 데이터 설정 완료');
        } else {
          console.error('FLOP 데이터 응답 오류:', { usageData, licenseData });
          setError('FLOP 데이터 응답 형식이 올바르지 않습니다.');
        }
      } catch (error: any) {
        console.error('FLOP 데이터 로드 실패:', error);
        setError(`FLOP 데이터 로드 실패: ${error.message || error}`);
      }
    };

    loadFlopData();
  }, [user]);

  // 서비스 상태 로드 (관리자만)
  useEffect(() => {
    if (!isAdmin) return;
    const loadServiceStatuses = async () => {
      try {
        const response = await fetch('/api/monitoring/services');
        const data = await response.json();
        if (data.success && data.data) {
          setServiceStatuses(data.data);
        }
      } catch (error) {
        console.error('서비스 상태 로드 실패:', error);
      }
    };
    loadServiceStatuses();
  }, [user, isAdmin]);

  const loadData = async () => {
    try {
      setError(null);
      const [systemRes, logsRes, usageRes] = await Promise.all([
        fetch('/api/monitoring/system').then(r => r.json()),
        fetch('/api/monitoring/logs').then(r => r.json()),
        fetch('/api/monitoring/usage').then(r => r.json())
      ]);
      if (systemRes.success && systemRes.data) setSystemStats(systemRes.data);
      if (logsRes.success && logsRes.data) setLogs(logsRes.data);
      if (usageRes.success && usageRes.data) setUsageStats(usageRes.data);
    } catch (error) {
      setError('모니터링 데이터를 불러오는 중 오류가 발생했습니다.');
    } finally {
      setIsLoading(false);
    }
  };

  // 사용자 권한에 따라 데이터 필터링
  const filteredLogs = isAdmin ? logs : logs.filter(l => l.userId === userId);
  const filteredUsageStats = isAdmin ? usageStats : filterUsageStatsByUser(usageStats, userId);

  // 탭 네비게이션 분기
  const tabs = isAdmin
    ? [
        { key: 'system', label: '시스템 상태' },
        { key: 'logs', label: '로그' },
        { key: 'usage', label: 'LLM 사용' },
        { key: 'api', label: 'API 사용' },
        { key: 'flop', label: 'FLOP 사용' },
      ]
    : [
        { key: 'usage', label: '내 사용 현황' },
        { key: 'logs', label: '내 로그' },
        { key: 'flop', label: 'FLOP 사용량' },
      ];

  const clearLogs = async () => {
    if (!confirm('모든 로그를 삭제하시겠습니까?')) return;

    try {
      const res = await fetch('/api/monitoring/logs', {
        method: 'DELETE'
      });
      const data = await res.json();
      if (data.success) {
        setLogs([]);
      } else {
        setError(data.error?.message || '로그 삭제 실패');
      }
    } catch (error) {
      console.error('로그 삭제 중 오류:', error);
      setError('로그 삭제 중 오류가 발생했습니다.');
    }
  };

  const resetAllData = async () => {
    if (!confirm('모든 모니터링 데이터를 초기화하시겠습니까?\n\n이 작업은 다음 데이터를 모두 삭제합니다:\n- 시스템 로그\n- 사용량 통계\n- API 호출 기록\n- 세션 데이터\n\n이 작업은 되돌릴 수 없습니다.')) {
      return;
    }

    try {
      console.log('🗑️ 모니터링 데이터 초기화 시작...');
      setIsLoading(true);
      
      const res = await fetch('/api/monitoring/reset', {
        method: 'DELETE'
      });
      
      console.log('📡 API 응답 상태:', res.status, res.statusText);
      
      const data = await res.json();
      console.log('📋 API 응답 데이터:', data);
      
      if (data.success) {
        // 모든 상태 초기화
        setSystemStats(null);
        setLogs([]);
        setUsageStats(null);
        setApiStats(null);
        setError(null);
        
        // 성공 메시지 표시
        const message = data.message || `모든 모니터링 데이터가 초기화되었습니다.`;
        const details = data.data?.deletedFiles ? `\n\n삭제된 파일: ${data.data.deletedFiles}개` : '';
        alert(message + details);
        console.log('✅ 데이터 초기화 성공:', data);
        
        // 데이터 다시 로드
        console.log('🔄 데이터 재로드 시작...');
        await loadData();
        console.log('🔄 데이터 재로드 완료');
      } else {
        const errorMsg = data.error?.message || '데이터 초기화 실패';
        console.error('❌ 데이터 초기화 실패:', data);
        
        // 부분적 성공인 경우 상세 정보 표시
        if (data.data?.deletedFiles > 0) {
          const partialMsg = `부분적으로 완료됨:\n- 삭제된 파일: ${data.data.deletedFiles}개\n- 오류: ${data.data.errors?.length || 0}개\n\n${errorMsg}`;
          setError(partialMsg);
          alert(partialMsg);
        } else {
          setError(errorMsg);
        }
      }
    } catch (error) {
      console.error('❌ 데이터 초기화 중 네트워크 오류:', error);
      setError('데이터 초기화 중 네트워크 오류가 발생했습니다. 관리자 권한이 있는지 확인해주세요.');
    } finally {
      setIsLoading(false);
    }
  };

  const exportLogs = () => {
    const logData = logs.map(log => ({
      timestamp: log.timestamp,
      level: log.level,
      message: log.message,
      source: log.source,
      details: log.details
    }));

    const blob = new Blob([JSON.stringify(logData, null, 2)], {
      type: 'application/json'
    });
    
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `logs-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const getLogLevelColor = (level: string) => {
    switch (level) {
      case 'error':
        return 'text-red-500';
      case 'warning':
        return 'text-yellow-500';
      case 'info':
        return 'text-blue-500';
      case 'debug':
        return 'text-gray-500';
      default:
        return 'text-gray-700';
    }
  };

  const getLogLevelBadge = (level: string) => {
    const baseClasses = 'px-2 py-1 rounded-full text-xs font-medium';
    switch (level) {
      case 'error':
        return `${baseClasses} bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200`;
      case 'warning':
        return `${baseClasses} bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200`;
      case 'info':
        return `${baseClasses} bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200`;
      case 'debug':
        return `${baseClasses} bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200`;
      default:
        return `${baseClasses} bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200`;
    }
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleString('ko-KR');
  };

  const formatUptime = (uptime: string) => {
    // uptime이 초 단위라고 가정
    const seconds = parseInt(uptime);
    const days = Math.floor(seconds / 86400);
    const hours = Math.floor((seconds % 86400) / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    
    if (days > 0) {
      return `${days}일 ${hours}시간 ${minutes}분`;
    } else if (hours > 0) {
      return `${hours}시간 ${minutes}분`;
    } else {
      return `${minutes}분`;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Online':
        return 'text-green-600 bg-green-100 dark:bg-green-900/30 dark:text-green-400';
      case 'Offline':
        return 'text-red-600 bg-red-100 dark:bg-red-900/30 dark:text-red-400';
      case 'Error':
        return 'text-yellow-600 bg-yellow-100 dark:bg-yellow-900/30 dark:text-yellow-400';
      default:
        return 'text-gray-600 bg-gray-100 dark:bg-gray-900/30 dark:text-gray-400';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'Online':
        return '🟢';
      case 'Offline':
        return '🔴';
      case 'Error':
        return '🟡';
      default:
        return '⚪';
    }
  };

  const refreshServiceStatuses = async () => {
    setIsLoading(true);
    try {
      const response = await fetch('/api/monitoring/services');
      const data = await response.json();
      if (data.success && data.data) {
        setServiceStatuses(data.data);
      }
    } catch (error) {
      console.error('서비스 상태 새로고침 실패:', error);
    } finally {
      setIsLoading(false); // 로딩 종료
    }
  };

  // 요약 카드 데이터 (API 대시보드)
  const apiSummary = isAdmin && apiStats && apiStats.totalCalls > 0 ? (() => {
    const totalCallsVal = apiStats.totalCalls;
    const avgResponseTime = apiStats.avgResponseTime;
    const successRate = apiStats.successRate;
    return { totalCalls: totalCallsVal, avgResponseTime, successRate };
  })() : null;

  // 엔드포인트별 Bar Chart 데이터
  const endpointBarData = isAdmin && apiStats ? {
    labels: Object.keys(apiStats.endpointStats),
    datasets: [
      {
        label: 'API Calls',
        data: Object.values(apiStats.endpointStats).map((s: any) => s.calls),
        backgroundColor: 'rgba(59,130,246,0.7)',
      },
    ],
  } : null;

  // 시간대별 히트맵 데이터 (24시간)
  const heatmapData = isAdmin && apiStats ? (() => {
    const hours = Array.from({ length: 24 }, (_, i) => i);
    const totalHourly = Array(24).fill(0);
    (apiStats.callsHeatmap as number[][]).forEach((row: number[]) => {
      row.forEach((cnt: number, hour: number) => {
        totalHourly[hour as number] += cnt as number;
      });
    });
    return {
      labels: hours.map((h) => `${h}시`),
      datasets: [
        {
          label: 'API Calls by Hour',
          data: totalHourly,
          backgroundColor: 'rgba(16,185,129,0.7)',
        } as const,
      ],
    };
  })() : null;

  // 응답시간/오류율 추이 (LineChart로 분리)
  const responseTimeTrend = isAdmin && apiStats ? (() => {
    const logs = apiStats.logs.sort((a: any, b: any) => new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime()).slice(-100);
    return {
      labels: logs.map((l: any) => l.timestamp),
      datasets: [
        {
          label: '응답시간(ms)',
          data: logs.map((l: any) => l.responseTime),
          borderColor: 'rgba(59,130,246,1)',
          backgroundColor: 'rgba(59,130,246,0.2)',
          yAxisID: 'y',
          tension: 0.2,
        } as const,
        {
          label: '오류(1=오류)',
          data: logs.map((l: any) => l.status >= 400 ? 1 : 0),
          borderColor: 'rgba(239,68,68,1)',
          backgroundColor: 'rgba(239,68,68,0.1)',
          yAxisID: 'y1',
          type: 'line',
          tension: 0.2,
        } as const,
      ],
    };
  })() : null;

  if (isLoading || authLoading) {
    return (
              <div className="fixed inset-0 backdrop-blur-sm z-50 flex items-center justify-center" style={{ backgroundColor: 'var(--background)' }}>
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
          <p className="text-lg text-gray-600 dark:text-gray-300 animate-pulse">모니터링 데이터를 불러오는 중...</p>
          <p className="text-sm text-gray-400 dark:text-gray-500 mt-2">잠시만 기다려주세요</p>
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8 max-w-6xl min-h-screen pb-40">
      <div className="flex justify-between items-center mb-8">
        <h1 className="text-3xl font-bold">모니터링</h1>
        <div className="flex items-center space-x-4">
          <label className="flex items-center space-x-2">
            <input
              type="checkbox"
              checked={autoRefresh}
              onChange={(e) => setAutoRefresh(e.target.checked)}
              className="rounded"
            />
            <span className="text-sm">자동 새로고침</span>
          </label>
          <button
            onClick={loadData}
            className="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600"
          >
            새로고침
          </button>
          {isAdmin && (
            <button
              onClick={resetAllData}
              className="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors"
              disabled={isLoading}
            >
              {isLoading ? '초기화 중...' : '데이터 초기화'}
            </button>
          )}
        </div>
      </div>

      {/* 에러 메시지 */}
      {error && (
        <div className="mb-6 p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-200 rounded-lg">
          {error}
          <button
            onClick={() => setError(null)}
            className="ml-4 text-sm underline"
          >
            닫기
          </button>
        </div>
      )}

      {/* 탭 네비게이션 */}
      <div className="mb-6">
        <nav className="flex space-x-8">
          {tabs.map((tab) => (
            <button
              key={tab.key}
              onClick={() => setActiveTab(tab.key as any)}
              className={`py-2 px-1 border-b-2 font-medium text-lg ${
                activeTab === tab.key
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              {tab.label}
            </button>
          ))}
        </nav>
      </div>

      {/* 시스템 상태 탭: 관리자만 - 서비스 상태 확인 */}
      {isAdmin && activeTab === 'system' && (
        <div className="space-y-8 overflow-hidden">
          {/* 서비스 상태 개요 */}
          <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
            <div className="flex justify-between items-center mb-6">
              <h2 className="text-xl font-semibold">서비스 상태</h2>
              <button
                onClick={refreshServiceStatuses}
                className="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
              >
                상태 새로고침
              </button>
            </div>
            
            {serviceStatuses.length === 0 ? (
              <div className="text-center py-8 text-gray-500">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto mb-4"></div>
                <p>서비스 상태를 확인하는 중...</p>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-1 lg:grid-cols-6 gap-4">
                {serviceStatuses.map((service, index) => (
                  <div
                    key={index}
                    className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 hover:shadow-md transition-shadow"
                  >
                    <div className="flex items-center justify-between mb-2">
                      <h3 className="font-semibold text-gray-800 dark:text-gray-200">
                        {service.name}
                      </h3>
                      <span className="text-lg">
                        {getStatusIcon(service.status)}
                      </span>
                    </div>
                    <div className="mb-2">
                      <span className={`inline-block px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(service.status)}`}>
                        {service.status}
                      </span>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>

          {/* 시스템 메트릭 */}
          {systemStats && (
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-6">시스템 메트릭</h2>
              <div className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-6 gap-4">
                <div className="text-center">
                  <div className="text-sm text-gray-500 mb-1">CPU 사용률</div>
                  <div className="text-2xl font-bold text-blue-600">
                    {systemStats.cpu_usage?.toFixed(1)}%
                  </div>
                </div>
                <div className="text-center">
                  <div className="text-sm text-gray-500 mb-1">메모리 사용률</div>
                  <div className="text-2xl font-bold text-green-600">
                    {systemStats.memory_usage?.toFixed(1)}%
                  </div>
                </div>
                <div className="text-center">
                  <div className="text-sm text-gray-500 mb-1">활성 세션</div>
                  <div className="text-2xl font-bold text-purple-600">
                    {systemStats.active_sessions}
                  </div>
                </div>
                <div className="text-center">
                  <div className="text-sm text-gray-500 mb-1">총 요청 수</div>
                  <div className="text-2xl font-bold text-indigo-600">
                    {systemStats.total_requests?.toLocaleString()}
                  </div>
                </div>
                
                {/* PostgreSQL 연결 정보 */}
                {systemStats.postgresql && (
                  <>
                    <div className="text-center">
                      <div className="text-sm text-gray-500 mb-1">PG 현재 연결</div>
                      <div className="text-2xl font-bold text-cyan-600">
                        {systemStats.postgresql.current_connections}
                      </div>
                    </div>
                    <div className="text-center">
                      <div className="text-sm text-gray-500 mb-1">PG 연결 사용률</div>
                      <div className={`text-2xl font-bold ${
                        systemStats.postgresql.connection_usage_percent > 80 
                          ? 'text-red-600' 
                          : systemStats.postgresql.connection_usage_percent > 60 
                            ? 'text-yellow-600' 
                            : 'text-green-600'
                      }`}>
                        {systemStats.postgresql.connection_usage_percent}%
                      </div>
                    </div>
                  </>
                )}
              </div>
            </div>
          )}

          {/* 서비스 상태 통계 */}
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">온라인 서비스</div>
              <div className="text-3xl font-bold text-green-600">
                {serviceStatuses.filter(s => s.status === 'Online').length}
              </div>
              <div className="text-sm text-gray-400 mt-1">
                / {serviceStatuses.length} 서비스
              </div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">오프라인 서비스</div>
              <div className="text-3xl font-bold text-red-600">
                {serviceStatuses.filter(s => s.status === 'Offline').length}
              </div>
              <div className="text-sm text-gray-400 mt-1">
                서비스 중단
              </div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">오류 서비스</div>
              <div className="text-3xl font-bold text-yellow-600">
                {serviceStatuses.filter(s => s.status === 'Error').length}
              </div>
              <div className="text-sm text-gray-400 mt-1">
                확인 필요
              </div>
            </div>
          </div>

          {/* 서비스 상태 분포 도넛 차트 */}
          {serviceStatuses.length > 0 && (
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-6 text-center">서비스 상태 분포</h2>
              <div className="w-full max-w-md mx-auto">
                <Doughnut
                  data={{
                    labels: ['온라인', '오프라인', '오류'],
                    datasets: [{
                      data: [
                        serviceStatuses.filter(s => s.status === 'Online').length,
                        serviceStatuses.filter(s => s.status === 'Offline').length,
                        serviceStatuses.filter(s => s.status === 'Error').length,
                      ],
                      backgroundColor: [
                        'rgb(34, 197, 94)',  // 초록 (온라인)
                        'rgb(239, 68, 68)',  // 빨강 (오프라인)
                        'rgb(245, 158, 11)', // 노랑 (오류)
                      ],
                      borderWidth: 2,
                      borderColor: '#ffffff'
                    }]
                  }}
                  options={{
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                      legend: {
                        position: 'bottom' as const,
                        labels: {
                          padding: 20,
                          usePointStyle: true,
                          color: document.documentElement.classList.contains('dark') ? '#ffffff' : '#374151',
                        }
                      },
                      tooltip: {
                        callbacks: {
                          label: (context) => {
                            const total = serviceStatuses.length;
                            const value = context.parsed;
                            const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0';
                            return `${context.label}: ${value}개 (${percentage}%)`;
                          }
                        }
                      }
                    }
                  }}
                  height={300}
                />
              </div>
            </div>
          )}
        </div>
      )}

      {/* 사용 현황 탭: 권한별 분기 */}
      {activeTab === 'usage' && filteredUsageStats && (
        <div className="flex flex-col gap-8 overflow-hidden">
          {/* 1. 상단 요약 카드 */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">총 비용</div>
              <div className="text-3xl font-bold text-blue-600">${filteredUsageStats.total_cost?.toFixed(2) ?? '0.00'}</div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">총 토큰</div>
              <div className="text-3xl font-bold text-green-600">{filteredUsageStats.total_tokens?.toLocaleString() ?? '0'}</div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">월별 비용</div>
              <div className="text-3xl font-bold text-purple-600">${filteredUsageStats.monthly_cost?.toFixed(2) ?? '0.00'}</div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">월별 토큰</div>
              <div className="text-3xl font-bold text-indigo-600">{filteredUsageStats.monthly_tokens?.toLocaleString() ?? '0'}</div>
            </div>
          </div>

          {/* 2. 제공자별 카드/도넛차트 */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-4">제공자별 분석</h2>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {filteredUsageStats.provider_stats && Object.entries(filteredUsageStats.provider_stats).map(([provider, stats]: any) => (
                  <div key={provider} className="border rounded p-4 text-left">
                    <div className="font-bold text-lg mb-1">{provider}</div>
                    <div className="text-sm text-gray-500 dark:text-gray-400">비용: ${stats.cost.toFixed(2)}</div>
                    <div className="text-sm text-gray-500 dark:text-gray-400">요청 토큰: {stats.promptTokens.toLocaleString()}</div>
                    <div className="text-sm text-gray-500 dark:text-gray-400">응답 토큰: {stats.completionTokens.toLocaleString()}</div>
                    <div className="text-sm text-gray-500 dark:text-gray-400">호출수: {stats.requests}</div>
                  </div>
                ))}
              </div>
            </div>
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-6 text-center">제공자별 비용 분포</h2>
              {(() => {
                // 디버깅을 위한 데이터 검증
                // console.log('🔍 Provider Stats Debug:', {
                //   filteredUsageStats: filteredUsageStats,
                //   provider_stats: filteredUsageStats?.provider_stats,
                //   hasProviderStats: !!filteredUsageStats?.provider_stats,
                //   providerKeys: filteredUsageStats?.provider_stats ? Object.keys(filteredUsageStats.provider_stats) : [],
                //   providerValues: filteredUsageStats?.provider_stats ? Object.values(filteredUsageStats.provider_stats) : []
                // });

                // provider_stats가 존재하고 데이터가 있는지 확인
                if (!filteredUsageStats?.provider_stats) {
                  return (
                    <div className="flex flex-col items-center justify-center h-64 text-gray-500">
                      <div className="text-lg mb-2">📊</div>
                      <div className="text-sm">제공자 통계 데이터가 없습니다</div>
                      <div className="text-xs mt-1">AI 호출 후 데이터가 표시됩니다</div>
                    </div>
                  );
                }

                const providerEntries = Object.entries(filteredUsageStats.provider_stats);
                if (providerEntries.length === 0) {
                  return (
                    <div className="flex flex-col items-center justify-center h-64 text-gray-500">
                      <div className="text-lg mb-2">📈</div>
                      <div className="text-sm">제공자 데이터가 비어있습니다</div>
                    </div>
                  );
                }

                // 비용 데이터 검증 및 필터링 - 모든 제공자를 비용 기준으로 통일
                const validProviders = providerEntries.filter(([provider, stats]: [string, any]) => {
                  // 통계 데이터가 있고, 요청이나 토큰이 있으면 유효한 것으로 판단
                  const hasStats = stats && (
                    (typeof stats.requests === 'number' && stats.requests > 0) ||
                    (typeof stats.promptTokens === 'number' && stats.promptTokens > 0) ||
                    (typeof stats.completionTokens === 'number' && stats.completionTokens > 0)
                  );
                  const hasCostData = stats && typeof stats.cost === 'number' && stats.cost >= 0;
                  const isValid = hasStats && hasCostData;
                  // console.log(`Provider ${provider} validation:`, { 
                  //   stats, 
                  //   isValid, 
                  //   cost: stats?.cost,
                  //   requests: stats?.requests,
                  //   tokens: {
                  //     prompt: stats?.promptTokens,
                  //     completion: stats?.completionTokens
                  //   }
                  // });
                  return isValid;
                });

                if (validProviders.length === 0) {
                  return (
                    <div className="flex flex-col items-center justify-center h-64 text-gray-500">
                      <div className="text-lg mb-2">💰</div>
                      <div className="text-sm">제공자 데이터가 없습니다</div>
                      <div className="text-xs mt-1">AI 호출 후 데이터가 표시됩니다</div>
                    </div>
                  );
                }

                // 차트 데이터 준비 - 모든 제공자를 비용 기준으로 통일
                const chartLabels = validProviders.map(([provider]) => provider);
                const chartData = validProviders.map(([, stats]: [string, any]) => stats.cost || 0);
                const totalValue = chartData.reduce((sum, value) => sum + value, 0);
                
                // 실제 비용을 지불한 제공자가 있는지 확인
                const hasPaidProviders = validProviders.some(([, stats]: [string, any]) => stats.cost > 0);

                // console.log('📊 Chart Data Prepared:', {
                //   labels: chartLabels,
                //   data: chartData,
                //   totalValue,
                //   hasPaidProviders,
                //   validProviderCount: validProviders.length,
                //   providerCosts: validProviders.map(([name, stats]) => ({ [name]: stats.cost }))
                // });

                return (
                  <div className="w-full flex flex-col items-center">
                    <div className="w-full max-w-md mx-auto">
                      <Doughnut
                        data={{
                          labels: chartLabels,
                          datasets: [{
                            data: chartData,
                            backgroundColor: [
                              'rgb(255, 99, 132)',   // 빨강
                              'rgb(54, 162, 235)',   // 파랑  
                              'rgb(255, 206, 86)',   // 노랑
                              'rgb(75, 192, 192)',   // 청록
                              'rgb(153, 102, 255)',  // 보라
                              'rgb(255, 159, 64)',   // 주황
                              'rgb(199, 199, 199)',  // 회색
                            ],
                            borderWidth: 2,
                            borderColor: '#ffffff'
                          }]
                        }}
                        options={{
                          responsive: true,
                          maintainAspectRatio: false,
                          plugins: {
                            legend: {
                              position: 'bottom' as const,
                              labels: {
                                padding: 15,
                                usePointStyle: true,
                                boxWidth: 12,
                                boxHeight: 12,
                                font: {
                                  size: 12
                                },
                                color: document.documentElement.classList.contains('dark') ? '#ffffff' : '#374151',
                                generateLabels: (chart) => {
                                  const data = chart.data;
                                  if (data.labels && data.datasets.length > 0) {
                                    const bgColors = data.datasets[0].backgroundColor as string[];
                                    const isDark = document.documentElement.classList.contains('dark');
                                    const textColor = isDark ? '#ffffff' : '#374151';
                                    return data.labels.map((label, i) => {
                                      const value = (data.datasets[0].data[i] as number).toFixed(2);
                                      return {
                                        text: `${label}: $${value}`,
                                        fillStyle: bgColors?.[i] || '#000',
                                        strokeStyle: bgColors?.[i] || '#000',
                                        lineWidth: 0,
                                        pointStyle: 'circle' as const,
                                        hidden: false,
                                        index: i,
                                        fontColor: textColor
                                      };
                                    });
                                  }
                                  return [];
                                }
                              }
                            },
                            tooltip: {
                              callbacks: {
                                label: (context) => {
                                  const provider = context.label || '';
                                  const value = context.parsed || 0;
                                  const percentage = totalValue > 0 ? ((value / totalValue) * 100).toFixed(1) : '0';
                                  return `${provider}: $${value.toFixed(2)} (${percentage}%)`;
                                }
                              }
                            }
                          }
                        }}
                        height={320}
                      />
                    </div>
                    <div className="mt-6 text-center space-y-2">
                      <div className="text-lg font-semibold text-gray-700 dark:text-gray-300">
                        총 비용: <span className="text-blue-600 dark:text-blue-400">${totalValue.toFixed(2)}</span>
                      </div>
                      <div className="flex flex-wrap justify-center gap-4 text-sm text-gray-500 dark:text-gray-400">
                        <span>📊 {validProviders.length}개 제공자</span>
                        {!hasPaidProviders && totalValue === 0 && (
                          <span className="text-green-600 dark:text-green-400">💰 모든 제공자 무료</span>
                        )}
                      </div>
                    </div>
                  </div>
                );
              })()}
            </div>
          </div>

          {/* 3. 월별 추이 Bar+Line Chart */}
          <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
            <h2 className="text-xl font-semibold mb-4">월별 추이</h2>
            {filteredUsageStats.monthly_stats && (
              <Bar
                data={{
                  labels: Object.keys(filteredUsageStats.monthly_stats),
                  datasets: [
                    {
                      label: '비용',
                      data: Object.values(filteredUsageStats.monthly_stats).map((s: any) => s.cost),
                      yAxisID: 'cost',
                      // @ts-ignore
                      type: 'line',
                      borderColor: 'rgb(255, 99, 132)',
                      backgroundColor: 'rgba(255, 99, 132, 0.5)'
                    },
                    {
                      label: '토큰',
                      data: Object.values(filteredUsageStats.monthly_stats).map((s: any) => s.tokens),
                      yAxisID: 'tokens',
                      backgroundColor: 'rgba(54, 162, 235, 0.5)'
                    }
                  ]
                }}
                options={(() => {
                  const isDark = typeof document !== 'undefined' && document.documentElement.classList.contains('dark');
                  const textColor = isDark ? '#ffffff' : '#374151';
                  const gridColor = isDark ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
                  return {
                    responsive: true,
                    plugins: { 
                      legend: { 
                        position: 'bottom',
                        labels: { color: textColor }
                      },
                      tooltip: {
                        backgroundColor: isDark ? 'rgba(0, 0, 0, 0.8)' : 'rgba(255, 255, 255, 0.9)',
                        titleColor: textColor,
                        bodyColor: textColor,
                        borderColor: gridColor,
                        borderWidth: 1
                      }
                    },
                    scales: {
                      x: {
                        ticks: { color: textColor },
                        grid: { color: gridColor }
                      },
                      cost: { 
                        type: 'linear', 
                        position: 'left', 
                        title: { display: true, text: '비용(USD)', color: textColor },
                        ticks: { color: textColor },
                        grid: { color: gridColor }
                      },
                      tokens: { 
                        type: 'linear', 
                        position: 'right', 
                        title: { display: true, text: '토큰', color: textColor },
                        ticks: { color: textColor },
                        grid: { display: false }
                      }
                    }
                  };
                })()}
                height={120}
              />
            )}
          </div>

          {/* 4. 시간대별 히트맵 (Bar로 대체) */}
          <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
            <h2 className="text-xl font-semibold mb-4">시간대별 사용량(요일/시간)</h2>
            {filteredUsageStats.heatmap && (
              <div className="overflow-x-auto">
                <table className="min-w-full text-xs text-center">
                  <thead>
                    <tr>
                      <th>요일/시간</th>
                      {[...Array(24)].map((_, h) => <th key={h}>{h}시</th>)}
                    </tr>
                  </thead>
                  <tbody>
                    {filteredUsageStats.heatmap.map((row: any, day: number) => (
                      <tr key={day}>
                        <td className="font-bold">{['일','월','화','수','목','금','토'][day]}</td>
                        {row.map((cnt: number, h: number) => <td key={h} className={cnt > 0 ? 'bg-blue-100 dark:bg-blue-900/40' : ''}>{cnt || ''}</td>)}
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>

          {/* 5. AI 호출 상세 테이블 (페이지네이션, CSV) */}
          <UsageCallTable callDetails={filteredUsageStats.call_details || []} />

          {/* 6. 세션별 사용량 리스트 (페이지네이션, CSV) */}
          <SessionUsageList sessionStats={filteredUsageStats.session_stats || []} />

          {/* 7. 세션별 버블차트 */}
          <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
            <h2 className="text-xl font-semibold mb-4">오늘 세션별 사용량(버블차트)</h2>
            <SessionBubbleChart data={filteredUsageStats.today_session_bubble || []} />
          </div>

          {/* 8. 모델별 토큰 비교/비용 효율성 */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-4">모델별 토큰 비교</h2>
              <ModelTokenChart data={filteredUsageStats.model_stats || []} />
            </div>
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-4">비용 효율성(모델별)</h2>
              <ModelCostEfficiencyChart data={filteredUsageStats.model_stats || []} />
            </div>
          </div>
        </div>
      )}

      {/* API 대시보드 탭: 관리자만 */}
      {isAdmin && activeTab === 'api' && apiStats && (
        <div className="flex flex-col gap-8 w-full pb-16 pr-2 overflow-hidden">
          {/* 1. 상단 요약 카드 */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">전체 호출 수</div>
              <div className="text-3xl font-bold text-blue-600">{apiStats.totalCalls?.toLocaleString() ?? '0'}</div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">제한 대상 호출 수</div>
              <div className="text-3xl font-bold text-green-600">{apiStats.limitedCalls?.toLocaleString() ?? '0'}</div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">평균 응답시간</div>
              <div className="text-3xl font-bold text-purple-600">{apiStats.avgResponseTime ?? 0} ms</div>
            </div>
            <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-sm text-gray-500 mb-1">성공률</div>
              <div className="text-3xl font-bold text-indigo-600">{apiStats.successRate !== undefined ? (apiStats.successRate * 100).toFixed(1) : 'N/A'}%</div>
            </div>
          </div>

          {/* 2. 엔드포인트별 호출수 Bar 차트 */}
          <div className="rounded-lg shadow-md p-6 overflow-x-auto min-w-[600px]" style={{ backgroundColor: 'var(--card-bg)' }}>
            <h2 className="text-xl font-semibold mb-4">엔드포인트별 호출 수</h2>
            {apiStats.endpointStats && (
              <Bar
                data={{
                  labels: Object.keys(apiStats.endpointStats),
                  datasets: [{
                    label: 'API Calls',
                    data: Object.values(apiStats.endpointStats).map((s: any) => s.calls),
                    backgroundColor: 'rgba(59,130,246,0.7)',
                  }]
                }}
                options={getChartOptions({ plugins: { legend: { display: false } } })}
                height={120}
              />
            )}
          </div>

          {/* 3. 시간대별 API 호출 히트맵(요일/시간) */}
          <div className="rounded-lg shadow-md p-6 overflow-x-auto min-w-[600px]" style={{ backgroundColor: 'var(--card-bg)' }}>
            <h2 className="text-xl font-semibold mb-4">시간대별 API 호출(히트맵)</h2>
            {apiStats.callsHeatmap && (
              <div className="overflow-x-auto">
                <table className="min-w-full text-xs text-center">
                  <thead>
                    <tr>
                      <th>요일/시간</th>
                      {[...Array(24)].map((_, h) => <th key={h}>{h}시</th>)}
                    </tr>
                  </thead>
                  <tbody>
                    {apiStats.callsHeatmap.map((row: any, day: number) => (
                      <tr key={day}>
                        <td className="font-bold">{['일','월','화','수','목','금','토'][day]}</td>
                        {row.map((cnt: number, h: number) => <td key={h} className={cnt > 0 ? 'bg-blue-100 dark:bg-blue-900/40' : ''}>{cnt || ''}</td>)}
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>

          {/* 4. 응답시간/오류율 추이(Line) */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-4">응답시간 추이</h2>
              {apiStats.timeLabels && apiStats.responseTimes && (
                <Line
                  data={{
                    labels: apiStats.timeLabels,
                    datasets: [{
                      label: '응답시간(ms)',
                      data: apiStats.responseTimes,
                      borderColor: 'rgba(59,130,246,1)',
                      backgroundColor: 'rgba(59,130,246,0.2)',
                      tension: 0.2,
                    }]
                  }}
                  options={getChartOptions({ 
                    plugins: { legend: { display: true } }, 
                    scales: { y: { beginAtZero: true, title: { display: true, text: '응답시간(ms)' } } } 
                  })}
                  height={120}
                />
              )}
            </div>
            <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-xl font-semibold mb-4">오류율 추이</h2>
              {apiStats.timeLabels && apiStats.errorRates && (
                <Line
                  data={{
                    labels: apiStats.timeLabels,
                    datasets: [{
                      label: '오류율(%)',
                      data: apiStats.errorRates,
                      borderColor: 'rgba(239,68,68,1)',
                      backgroundColor: 'rgba(239,68,68,0.1)',
                      tension: 0.2,
                    }]
                  }}
                  options={getChartOptions({ 
                    plugins: { legend: { display: true } }, 
                    scales: { y: { beginAtZero: true, title: { display: true, text: '오류율(%)' }, max: 100 } } 
                  })}
                  height={120}
                />
              )}
            </div>
          </div>

          {/* 5. API 로그 테이블 (페이지네이션, CSV) */}
          <ApiLogsTable logs={apiStats.logs || []} />
        </div>
      )}

      {/* 로그 탭: 권한별 분기 */}
      {activeTab === 'logs' && (
        <LogsViewer isAdmin={isAdmin} />
      )}

      {/* FLOP 사용량 탭 */}
      {activeTab === 'flop' && (
        <div className="space-y-6 overflow-hidden">
          {error && (
            <div className="bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-200 p-4 rounded-lg">
              <div className="flex justify-between items-center">
                <span>FLOP 데이터 로드 오류: {error}</span>
                <button
                  onClick={() => setError(null)}
                  className="text-red-500 hover:text-red-700"
                >
                  ✕
                </button>
              </div>
            </div>
          )}
          <FlopUsageTab flopStats={flopStats} isAdmin={isAdmin} />
        </div>
      )}
    </div>
  );
}

// 로그 뷰어 컴포넌트
function LogsViewer({ isAdmin }: { isAdmin: boolean }) {
  const [logs, setLogs] = useState<LogEntry[]>([]);
  const [logFiles, setLogFiles] = useState<any[]>([]);
  const [selectedFile, setSelectedFile] = useState<string>('');
  const [isLoading, setIsLoading] = useState(false);
  const [hasMore, setHasMore] = useState(true);
  const [fromEnd, setFromEnd] = useState(0);
  const [error, setError] = useState<string | null>(null);
  const [totalLines, setTotalLines] = useState(0);


  // 로그 파일 목록 로드
  const loadLogFiles = async () => {
    try {
      const response = await fetch('/api/monitoring/logs?action=list');
      const data = await response.json();
      if (data.success) {
        setLogFiles(data.data);
      }
    } catch (error) {
      console.error('로그 파일 목록 로드 실패:', error);
    }
  };

  // 로그 데이터 로드 (역방향)
  const loadLogs = useCallback(async (fromEndParam: number = 0, append: boolean = false) => {
    if (isLoading) return;
    
    setIsLoading(true);
    setError(null);
    
    try {
      const params = new URLSearchParams({
        fromEnd: fromEndParam.toString(),
        pageSize: '100'
      });
      
      if (selectedFile) {
        params.append('file', selectedFile);
      }
      
      const response = await fetch(`/api/monitoring/logs?${params}`);
      const data = await response.json();
      
      if (data.success) {
        const newLogs = data.data || [];
        
        if (append) {
          // 위로 스크롤할 때는 기존 로그 위에 추가
          setLogs(prev => [...newLogs, ...prev]);
        } else {
          // 처음 로드할 때는 교체
          setLogs(newLogs);
        }
        
        setHasMore(data.pagination?.hasMore || false);
        setFromEnd(fromEndParam);
        setTotalLines(data.pagination?.totalLines || 0);
        
        // 로그 파일 목록도 업데이트
        if (data.logFiles) {
          setLogFiles(data.logFiles);
        }
      } else {
        setError(data.error?.message || '로그 로드 실패');
      }
    } catch (error) {
      console.error('로그 로드 실패:', error);
      setError('로그를 불러오는 중 오류가 발생했습니다.');
    } finally {
      setIsLoading(false);
    }
  }, [isLoading, selectedFile]);

  // 무한 스크롤 핸들러 (역방향)
  const loadMoreLogs = useCallback(() => {
    if (hasMore && !isLoading) {
      loadLogs(fromEnd + 100, true);
    }
  }, [hasMore, isLoading, fromEnd, loadLogs]);

  // 로그 컨테이너 ref
  const logContainerRef = useRef<HTMLDivElement>(null);

  // 스크롤 상태를 ref로 관리
  const scrollStateRef = useRef({ hasMore, isLoading, fromEnd });
  scrollStateRef.current = { hasMore, isLoading, fromEnd };

  // 스크롤 이벤트 핸들러 (컨테이너 내부 스크롤 감지)
  useEffect(() => {
    const container = logContainerRef.current;
    if (!container) return;

    let ticking = false;
    
    const handleScroll = () => {
      if (!ticking) {
        requestAnimationFrame(() => {
          const scrollTop = container.scrollTop;
          const { hasMore, isLoading, fromEnd } = scrollStateRef.current;
          
          // 컨테이너 상단에서 100px 이내에 도달하면 이전 로그 로드
          if (scrollTop <= 100 && hasMore && !isLoading) {
            const prevScrollHeight = container.scrollHeight;
            
            // 직접 loadLogs 호출
            loadLogs(fromEnd + 100, true);
            
            // 로그 추가 후 스크롤 위치 조정 (새로 추가된 로그만큼 아래로 이동)
            setTimeout(() => {
              if (container.scrollHeight > prevScrollHeight) {
                const newContentHeight = container.scrollHeight - prevScrollHeight;
                container.scrollTop = container.scrollTop + newContentHeight;
              }
            }, 100);
          }
          ticking = false;
        });
        ticking = true;
      }
    };

    container.addEventListener('scroll', handleScroll, { passive: true });
    return () => container.removeEventListener('scroll', handleScroll);
  }, [loadLogs]); // loadLogs만 의존성으로 유지

  // 초기 로드
  useEffect(() => {
    loadLogFiles();
  }, []);

  useEffect(() => {
    loadLogs(0, false);
  }, []);

  // 초기 로드 후 스크롤을 맨 아래로 이동
  useEffect(() => {
    if (logs.length > 0 && logContainerRef.current) {
      logContainerRef.current.scrollTop = logContainerRef.current.scrollHeight;
    }
  }, [logs.length > 0 && fromEnd === 0]);

  // 필터 변경 시 로그 다시 로드
  useEffect(() => {
    if (selectedFile !== undefined) { // 초기 렌더링 방지
      setFromEnd(0);
      setHasMore(true);
      setLogs([]); // 기존 로그 초기화
      loadLogs(0, false);
    }
  }, [selectedFile]);

  // 로그 삭제
  const clearLogs = async () => {
    if (!isAdmin) return;
    
    const message = selectedFile 
      ? `${selectedFile} 파일을 삭제하시겠습니까?`
      : '모든 로그 파일을 삭제하시겠습니까?';
    
    if (!confirm(message)) return;

    try {
      const params = selectedFile ? `?file=${selectedFile}` : '';
      const response = await fetch(`/api/monitoring/logs${params}`, {
        method: 'DELETE'
      });
      
      const data = await response.json();
      if (data.success) {
        setLogs([]);
        setSelectedFile('');
        await loadLogFiles();
        await loadLogs();
      } else {
        setError(data.error?.message || '로그 삭제 실패');
      }
    } catch (error) {
      console.error('로그 삭제 중 오류:', error);
      setError('로그 삭제 중 오류가 발생했습니다.');
    }
  };

  // 로그 내보내기
  const exportLogs = () => {
    if (logs.length === 0) return;
    
    const logData = logs.map(log => ({
      timestamp: log.timestamp,
      level: log.level,
      message: log.message,
      source: log.source,
      fileName: log.fileName,
      lineNumber: log.lineNumber,
      details: log.details
    }));

    const blob = new Blob([JSON.stringify(logData, null, 2)], {
      type: 'application/json'
    });
    
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    const fileName = selectedFile || 'all-logs';
    a.download = `${fileName}-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const getLogLevelBadge = (level: string) => {
    const baseClasses = 'px-2 py-1 rounded-full text-xs font-medium';
    switch (level.toLowerCase()) {
      case 'error':
      case 'fatal':
      case 'critical':
        return `${baseClasses} bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200`;
      case 'warning':
      case 'warn':
        return `${baseClasses} bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200`;
      case 'info':
        return `${baseClasses} bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200`;
      case 'debug':
        return `${baseClasses} bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200`;
      default:
        return `${baseClasses} bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200`;
    }
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleString('ko-KR');
  };

  return (
    <div className="space-y-6">
      {/* 로그 파일 선택 및 필터 */}
      <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex flex-col">
              <label className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                로그 파일
              </label>
              <select
                value={selectedFile}
                onChange={(e) => setSelectedFile(e.target.value)}
                className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
              >
                <option value="">모든 파일</option>
                {logFiles.map((file) => (
                  <option key={file.name} value={file.name}>
                    {file.name} ({formatFileSize(file.size)})
                  </option>
                ))}
              </select>
            </div>
            

          </div>
          
          <div className="flex items-center gap-2">
            <span className="text-sm text-gray-500">
              {selectedFile ? `${logs.length}/${totalLines}개 로그` : `총 ${logs.length}개 로그`}
            </span>
            <button
              onClick={exportLogs}
              disabled={logs.length === 0}
              className="px-3 py-2 bg-green-500 text-white rounded hover:bg-green-600 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              내보내기
            </button>
            {isAdmin && (
              <button
                onClick={clearLogs}
                className="px-3 py-2 bg-red-500 text-white rounded hover:bg-red-600"
              >
                삭제
              </button>
            )}
          </div>
        </div>
      </div>



      {/* 에러 메시지 */}
      {error && (
        <div className="bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-200 p-4 rounded-lg">
          {error}
          <button
            onClick={() => setError(null)}
            className="ml-4 text-sm underline"
          >
            닫기
          </button>
        </div>
      )}

      {/* 로그 목록 */}
      <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
        <h3 className="text-lg font-semibold mb-4">로그 내용</h3>
        
        {/* 상단 로딩 인디케이터 */}
        {isLoading && hasMore && (
          <div className="text-center py-2 mb-4">
            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-500 mx-auto"></div>
            <p className="text-xs text-gray-500 mt-1">이전 로그를 불러오는 중...</p>
          </div>
        )}
        
        {/* 더 이상 로그가 없을 때 */}
        {!hasMore && logs.length > 0 && (
          <div className="text-center py-2 mb-4 text-gray-500 text-sm">
            --- 로그 시작 ---
          </div>
        )}
        
        {logs.length === 0 && !isLoading ? (
          <div className="text-center py-8 text-gray-500">
            표시할 로그가 없습니다.
          </div>
        ) : (
          <div 
            ref={logContainerRef}
            className="h-[600px] overflow-y-auto border border-gray-200 dark:border-gray-700"
          >
            <pre className="text-sm text-gray-800 dark:text-gray-200 whitespace-pre-wrap font-mono leading-relaxed p-4 bg-gray-50 dark:bg-gray-900 rounded">
              {logs.map((log, index) => (
                <div key={index}>
                  {log.message || ''}
                </div>
              ))}
            </pre>
          </div>
        )}
        
        {/* 하단 로딩 인디케이터 (초기 로드용) */}
        {isLoading && logs.length === 0 && (
          <div className="text-center py-4">
            <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-500 mx-auto"></div>
            <p className="text-sm text-gray-500 mt-2">로그를 불러오는 중...</p>
          </div>
        )}
      </div>
    </div>
  );
}

// AI 호출 상세 테이블 컴포넌트
function UsageCallTable({ callDetails }: { callDetails: any[] }) {
  const [page, setPage] = useState(1);
  const perPage = 20;
  const totalPages = Math.ceil(callDetails.length / perPage);
  const pageData = callDetails.slice((page-1)*perPage, page*perPage);

  // CSV 다운로드
  const handleDownload = () => {
    const headers = ['Timestamp','Provider','Model','PromptTokens','CompletionTokens','Cost','SessionId','Prompt','UserId','Username','Feature'];
    const rows = callDetails.map(r => [r.timestamp, r.provider, r.model, r.promptTokens, r.completionTokens, r.cost, r.sessionId, r.prompt, r.userId, r.username, r.feature]);
    const csv = [headers.join(','), ...rows.map(row => row.map(v => `"${String(v ?? '').replace(/"/g,'""')}"`).join(','))].join('\n');
    downloadCSVFile(csv, 'ai_call_details.csv');
  };

  return (
    <div className="rounded-lg shadow-md p-6 mt-8" style={{ backgroundColor: 'var(--card-bg)' }}>
      <div className="flex justify-between items-center mb-2">
        <h2 className="text-xl font-semibold">AI 호출 상세 내역</h2>
        <button className="px-3 py-1 bg-green-500 text-white rounded hover:bg-green-600" onClick={handleDownload}>CSV 다운로드</button>
      </div>
      <div className="overflow-x-auto">
        <table className="min-w-full text-xs text-left">
          <thead>
            <tr>
              <th>시간</th><th>제공자</th><th>모델</th><th>요청토큰</th><th>응답토큰</th><th>비용</th><th>세션</th><th>프롬프트</th><th>유저ID</th><th>이름</th><th>기능</th>
            </tr>
          </thead>
          <tbody>
            {pageData.map((r, i) => (
              <tr key={i}>
                <td>{new Date(r.timestamp).toLocaleString()}</td>
                <td>{r.provider}</td>
                <td>{r.model}</td>
                <td>{r.promptTokens}</td>
                <td>{r.completionTokens}</td>
                <td>${r.cost?.toFixed(4)}</td>
                <td>{r.sessionId}</td>
                <td className="max-w-[200px] truncate" title={r.prompt}>{r.prompt}</td>
                <td>{r.userId}</td>
                <td>{r.username}</td>
                <td>{r.feature}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <div className="flex justify-center items-center mt-4 space-x-2">
        <button onClick={()=>setPage(p=>Math.max(1,p-1))} disabled={page===1} className="px-2 py-1 rounded bg-gray-200 dark:bg-gray-700">이전</button>
        <span>{page} / {totalPages}</span>
        <button onClick={()=>setPage(p=>Math.min(totalPages,p+1))} disabled={page===totalPages} className="px-2 py-1 rounded bg-gray-200 dark:bg-gray-700">다음</button>
      </div>
          </div>
  );
}

// 세션별 사용량 리스트 컴포넌트
function SessionUsageList({ sessionStats }: { sessionStats: any[] }) {
  const [page, setPage] = useState(1);
  const perPage = 20;
  const totalPages = Math.ceil(sessionStats.length / perPage);
  const pageData = sessionStats.slice((page-1)*perPage, page*perPage);

  // CSV 다운로드
  const handleDownload = () => {
    const headers = ['SessionId','Timestamp','FirstPrompt','Calls','TotalTokens','Providers'];
    const rows = sessionStats.map(r => [r.sessionId, r.timestamp, r.firstPrompt, r.calls, r.totalTokens, (r.providers||[]).join(';')]);
    const csv = [headers.join(','), ...rows.map(row => row.map(v => `"${String(v ?? '').replace(/"/g,'""')}"`).join(','))].join('\n');
    downloadCSVFile(csv, 'session_usage.csv');
  };

  // 프롬프트 줄임 함수
  const truncate = (str: string, n: number) => str && str.length > n ? str.slice(0, n) + '...' : str;

  return (
    <div className="rounded-lg shadow-md p-6 mt-8" style={{ backgroundColor: 'var(--card-bg)' }}>
      <div className="flex justify-between items-center mb-2">
        <h2 className="text-xl font-semibold">세션별 사용량</h2>
        <button className="px-3 py-1 bg-green-500 text-white rounded hover:bg-green-600" onClick={handleDownload}>CSV 다운로드</button>
      </div>
      <div className="space-y-2">
        {pageData.map((r, i) => (
          <div key={i} className="border border-gray-200 dark:border-gray-700 rounded p-3">
              <div className="flex justify-between items-center">
              <div className="font-semibold max-w-[320px] truncate" title={r.firstPrompt}>{truncate(r.firstPrompt, 80)}</div>
              <div className="text-xs text-gray-500">{new Date(r.timestamp).toLocaleString()}</div>
              </div>
            <div className="flex flex-wrap gap-4 mt-2 text-xs">
              <span>호출수: <b>{r.calls}</b></span>
              <span>토큰: <b>{r.totalTokens.toLocaleString()}</b></span>
              <span>제공자: <b>{(r.providers||[]).join(', ')}</b></span>
              <span>세션ID: <b>{r.sessionId}</b></span>
            </div>
                </div>
              ))}
            </div>
    <div className="flex justify-center items-center mt-4 space-x-2">
      <button onClick={()=>setPage(p=>Math.max(1,p-1))} disabled={page===1} className="px-2 py-1 rounded bg-gray-200 dark:bg-gray-700">이전</button>
      <span>{page} / {totalPages}</span>
      <button onClick={()=>setPage(p=>Math.min(totalPages,p+1))} disabled={page===totalPages} className="px-2 py-1 rounded bg-gray-200 dark:bg-gray-700">다음</button>
      </div>
    </div>
  );
}

// 세션별 버블차트 컴포넌트
function SessionBubbleChart({ data }: { data: any[] }) {
  if (!data || data.length === 0) return <div className="text-center text-gray-400">데이터가 없습니다.</div>;
  const chartData = {
    datasets: [{
      label: 'Session',
      data: data.map((d: any) => ({
        x: new Date(d.timestamp).getTime(),
        y: d.calls,
        r: Math.sqrt(d.calls) * 3,
        calls: d.calls,
        tokens: d.totalTokens,
        providers: (d.providers||[]).join(', ')
      })),
      backgroundColor: 'rgba(75, 192, 192, 0.6)',
      borderColor: 'rgba(75, 192, 192, 1)',
      borderWidth: 1
    }]
  };
  
  const isDark = typeof document !== 'undefined' && document.documentElement.classList.contains('dark');
  const textColor = isDark ? '#ffffff' : '#374151';
  const gridColor = isDark ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
  
  return (
    <Line
      data={chartData}
      options={{
        responsive: true,
        plugins: { 
          legend: { 
            display: false,
            labels: { color: textColor }
          },
          tooltip: {
            backgroundColor: isDark ? 'rgba(0, 0, 0, 0.8)' : 'rgba(255, 255, 255, 0.9)',
            titleColor: textColor,
            bodyColor: textColor,
            borderColor: gridColor,
            borderWidth: 1
          }
        },
        scales: {
          x: {
            type: 'linear',
            min: Math.min(...data.map((d: any) => new Date(d.timestamp).getTime())),
            max: Math.max(...data.map((d: any) => new Date(d.timestamp).getTime())),
            ticks: {
              color: textColor,
              callback: (v: any) => {
                const date = new Date(Number(v));
                return `${date.getHours()}시`;
              }
            },
            grid: { color: gridColor },
            title: { display: true, text: '시간', color: textColor }
          },
          y: { 
            title: { display: true, text: '호출수', color: textColor }, 
            beginAtZero: true,
            ticks: { color: textColor },
            grid: { color: gridColor }
          }
        },
        elements: { point: { radius: (ctx: any) => ctx.raw?.r || 5 } }
      }}
      height={120}
    />
  );
}

// 모델별 토큰 비교 차트
function ModelTokenChart({ data }: { data: any[] }) {
  if (!data || data.length === 0) return <div className="text-center text-gray-400">데이터가 없습니다.</div>;
  
  const isDark = typeof document !== 'undefined' && document.documentElement.classList.contains('dark');
  const textColor = isDark ? '#ffffff' : '#374151';
  const gridColor = isDark ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
  
  return (
    <Bar
      data={{
        labels: data.map((d: any) => d.model),
        datasets: [
          {
            label: '요청 토큰',
            data: data.map((d: any) => d.promptTokens),
            backgroundColor: 'rgba(255, 99, 132, 0.8)'
          },
          {
            label: '응답 토큰',
            data: data.map((d: any) => d.completionTokens),
            backgroundColor: 'rgba(54, 162, 235, 0.8)'
          }
        ]
      }}
      options={{
        responsive: true,
        plugins: { 
          legend: { 
            position: 'bottom',
            labels: { color: textColor }
          },
          tooltip: {
            backgroundColor: isDark ? 'rgba(0, 0, 0, 0.8)' : 'rgba(255, 255, 255, 0.9)',
            titleColor: textColor,
            bodyColor: textColor,
            borderColor: gridColor,
            borderWidth: 1
          }
        },
        scales: { 
          x: { 
            stacked: true,
            ticks: { color: textColor },
            grid: { color: gridColor }
          }, 
          y: { 
            stacked: true, 
            beginAtZero: true,
            ticks: { color: textColor },
            grid: { color: gridColor }
          } 
        }
      }}
      height={120}
    />
  );
}

// 모델별 비용 효율성 차트
function ModelCostEfficiencyChart({ data }: { data: any[] }) {
  if (!data || data.length === 0) return <div className="text-center text-gray-400">데이터가 없습니다.</div>;
  const costPerToken = data.filter((d: any) => d.totalTokens > 0).map((d: any) => ({ model: d.model, efficiency: d.totalCost / d.totalTokens * 1000 }));
  
  const isDark = typeof document !== 'undefined' && document.documentElement.classList.contains('dark');
  const textColor = isDark ? '#ffffff' : '#374151';
  const gridColor = isDark ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
  
  return (
    <Bar
      data={{
        labels: costPerToken.map((d: any) => d.model),
        datasets: [{
          label: '1000토큰당 비용',
          data: costPerToken.map((d: any) => d.efficiency),
          backgroundColor: 'rgba(75, 192, 192, 0.8)'
        }]
      }}
      options={{
        responsive: true,
        plugins: { 
          legend: { 
            position: 'bottom',
            labels: { color: textColor }
          },
          tooltip: {
            backgroundColor: isDark ? 'rgba(0, 0, 0, 0.8)' : 'rgba(255, 255, 255, 0.9)',
            titleColor: textColor,
            bodyColor: textColor,
            borderColor: gridColor,
            borderWidth: 1
          }
        },
        scales: { 
          x: {
            ticks: { color: textColor },
            grid: { color: gridColor }
          },
          y: { 
            beginAtZero: true, 
            title: { display: true, text: '비용(USD/1000토큰)', color: textColor },
            ticks: { color: textColor },
            grid: { color: gridColor }
          } 
        }
      }}
      height={120}
    />
  );
}

// API 로그 테이블 컴포넌트
function ApiLogsTable({ logs }: { logs: any[] }) {
  const [page, setPage] = useState(1);
  const perPage = 20;
  const totalPages = Math.ceil(logs.length / perPage);
  const pageData = logs.slice((page-1)*perPage, page*perPage);

  // CSV 다운로드
  const handleDownload = () => {
    const headers = ['Timestamp','Endpoint','Method','Status','ResponseTime','UserId'];
    const rows = logs.map(r => [r.timestamp, r.endpoint, r.method, r.status, r.responseTime, r.userId]);
    const csv = [headers.join(','), ...rows.map(row => row.map(v => `"${String(v ?? '').replace(/"/g,'""')}"`).join(','))].join('\n');
    downloadCSVFile(csv, 'api_logs.csv');
  };

  return (
    <div className="rounded-lg shadow-md p-6 mt-8" style={{ backgroundColor: 'var(--card-bg)' }}>
      <div className="flex justify-between items-center mb-2">
        <h2 className="text-xl font-semibold">API 로그</h2>
        <button className="px-3 py-1 bg-green-500 text-white rounded hover:bg-green-600" onClick={handleDownload}>CSV 다운로드</button>
      </div>
      <div className="overflow-x-auto">
        <table className="min-w-full text-xs text-left">
          <thead>
            <tr>
              <th>시간</th><th>엔드포인트</th><th>메서드</th><th>상태</th><th>응답시간(ms)</th><th>유저</th>
            </tr>
          </thead>
          <tbody>
            {pageData.map((r, i) => (
              <tr key={i} className={r.status >= 400 ? 'bg-red-50 dark:bg-red-900/30' : ''}>
                <td>{new Date(r.timestamp).toLocaleString()}</td>
                <td>{r.endpoint}</td>
                <td>{r.method}</td>
                <td>{r.status}</td>
                <td>{r.responseTime}</td>
                <td>{r.userId || '-'}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <div className="flex justify-center items-center mt-4 space-x-2">
        <button onClick={()=>setPage(p=>Math.max(1,p-1))} disabled={page===1} className="px-2 py-1 rounded bg-gray-200 dark:bg-gray-700">이전</button>
        <span>{page} / {totalPages}</span>
        <button onClick={()=>setPage(p=>Math.min(totalPages,p+1))} disabled={page===totalPages} className="px-2 py-1 rounded bg-gray-200 dark:bg-gray-700">다음</button>
      </div>
    </div>
  );
}

// FLOP 사용량 탭 컴포넌트
function FlopUsageTab({ flopStats, isAdmin }: { flopStats: any; isAdmin: boolean }) {
  // 토큰 디버깅 함수
  const debugToken = () => {
    if (typeof window === 'undefined') return;
    
    const tokens = {
      userToken: localStorage.getItem('userToken'),
      token: localStorage.getItem('token'),
      authToken: localStorage.getItem('auth-token'),
      sessionToken: sessionStorage.getItem('token'),
    };
    
    console.log('🔍 토큰 디버깅:', tokens);
    
    Object.entries(tokens).forEach(([key, token]) => {
      if (token) {
        const cleanToken = token.replace(/^Bearer\s+/i, '').trim();
        const parts = cleanToken.split('.');
        console.log(`${key}:`, {
          length: cleanToken.length,
          parts: parts.length,
          preview: cleanToken.substring(0, 50) + '...',
          isValidJWT: parts.length === 3
        });
      }
    });
  };

  if (!flopStats) {
    return (
      <div className="flex flex-col items-center justify-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mb-4"></div>
        <p className="text-lg text-gray-600 dark:text-gray-300">FLOP 데이터를 불러오는 중...</p>
        <button 
          onClick={debugToken}
          className="mt-4 px-4 py-2 bg-gray-500 text-white rounded text-sm hover:bg-gray-600"
        >
          토큰 디버깅
        </button>
      </div>
    );
  }

  const { usage, license } = flopStats;

  // FLOP 단위 변환 함수
  const formatFlops = (flops: number | string) => {
    if (flops === "unlimited" || flops === Infinity) return "무제한";
    if (typeof flops === 'string') return flops;
    
    const numFlops = Number(flops);
    if (isNaN(numFlops)) return "0 FLOPS";
    
    if (numFlops >= 1e12) return `${(numFlops / 1e12).toFixed(2)} TFLOPS`;
    if (numFlops >= 1e9) return `${(numFlops / 1e9).toFixed(2)} GFLOPS`;
    if (numFlops >= 1e6) return `${(numFlops / 1e6).toFixed(2)} MFLOPS`;
    if (numFlops >= 1e3) return `${(numFlops / 1e3).toFixed(2)} KFLOPS`;
    return `${numFlops.toFixed(0)} FLOPS`;
  };

  // 사용률 계산
  const usagePercentage = (license.flop_limit === "unlimited" || license.flop_limit === Infinity)
    ? 0 
    : ((license.flop_used / Number(license.flop_limit)) * 100);

  return (
    <div className="space-y-8">
      {/* FLOP 사용량 설명 카드 */}
      <div className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 rounded-lg border border-blue-200 dark:border-blue-800 p-6">
        <div className="flex items-start space-x-4">
          <div className="flex-shrink-0">
            <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900 rounded-lg flex items-center justify-center">
              <span className="text-2xl">🧮</span>
            </div>
          </div>
          <div className="flex-1">
            <h3 className="text-lg font-semibold text-blue-900 dark:text-blue-100 mb-2">
              FLOP (Floating Point Operations) 사용량이란?
            </h3>
            <div className="text-sm text-blue-800 dark:text-blue-200 space-y-2">
              <p>
                <strong>FLOP</strong>는 AI 모델이 수행하는 부동소수점 연산의 총량을 측정하는 단위입니다. 
                모델의 크기와 처리하는 토큰 수에 따라 계산됩니다.
              </p>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-3">
                <div className="rounded-lg p-3" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="font-medium text-blue-900 dark:text-blue-100">📊 계산 방식</div>
                  <div className="text-xs mt-1">모델 파라미터 × 토큰 수 × 2</div>
                </div>
                <div className="rounded-lg p-3" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="font-medium text-blue-900 dark:text-blue-100">💰 과금 기준</div>
                  <div className="text-xs mt-1">사용한 FLOP 양에 따른 라이선스 비용</div>
                </div>
                <div className="rounded-lg p-3" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="font-medium text-blue-900 dark:text-blue-100">📈 단위</div>
                  <div className="text-xs mt-1">KFLOPS → MFLOPS → GFLOPS → TFLOPS</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* 라이선스 상태 카드 */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
          <div className="text-sm text-gray-500 mb-1">라이선스 플랜</div>
          <div className="text-xl font-bold text-blue-600 capitalize">{license.plan}</div>
          <div className="text-sm text-gray-400 mt-1">
            {license.active ? '활성' : '비활성'}
          </div>
        </div>
        <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
          <div className="text-sm text-gray-500 mb-1">FLOP 한도</div>
          <div className="text-xl font-bold text-green-600">
            {formatFlops(license.flop_limit)}
          </div>
        </div>
        <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
          <div className="text-sm text-gray-500 mb-1">사용량</div>
          <div className="text-xl font-bold text-purple-600">
            {formatFlops(license.flop_used)}
          </div>
        </div>
        <div className="rounded-lg shadow-md p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
          <div className="text-sm text-gray-500 mb-1">잔여량</div>
          <div className="text-xl font-bold text-indigo-600">
            {formatFlops(license.flop_remaining)}
          </div>
        </div>
      </div>

      {/* 사용률 진행바 */}
      <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
        <h2 className="text-xl font-semibold mb-4">FLOP 사용률</h2>
        <div className="mb-2 flex justify-between">
          <span className="text-sm text-gray-600 dark:text-gray-400">
            {formatFlops(license.flop_used)} / {formatFlops(license.flop_limit)}
          </span>
          <span className="text-sm text-gray-600 dark:text-gray-400">
            {(license.flop_limit === "unlimited" || license.flop_limit === Infinity) ? '0%' : `${usagePercentage.toFixed(1)}%`}
          </span>
        </div>
        <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-4">
          <div 
            className={`h-4 rounded-full transition-all duration-300 ${
              usagePercentage > 90 ? 'bg-red-500' :
              usagePercentage > 70 ? 'bg-yellow-500' :
              'bg-green-500'
            }`}
            style={{ width: (license.flop_limit === "unlimited" || license.flop_limit === Infinity) ? '0%' : `${Math.min(usagePercentage, 100)}%` }}
          ></div>
        </div>
                  {usagePercentage > 80 && license.flop_limit !== "unlimited" && license.flop_limit !== Infinity && (
          <div className="mt-2 text-sm text-orange-600 dark:text-orange-400">
            ⚠️ FLOP 사용량이 높습니다. 라이선스 업그레이드를 고려해보세요.
          </div>
        )}
      </div>

      {/* 사용량 상세 정보 */}
      <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
        <h2 className="text-xl font-semibold mb-4">사용량 상세</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="text-center">
            <div className="text-2xl font-bold text-gray-800 dark:text-gray-200">
              {usage.total_requests || 0}
            </div>
            <div className="text-sm text-gray-500">총 요청 수</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-gray-800 dark:text-gray-200">
              {(usage.total_tokens || 0).toLocaleString()}
            </div>
            <div className="text-sm text-gray-500">총 토큰 수</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-gray-800 dark:text-gray-200">
              {formatFlops(usage.total_flops || 0)}
            </div>
            <div className="text-sm text-gray-500">총 FLOP 수</div>
          </div>
        </div>
      </div>

      {/* 모델별 FLOP 사용량 */}
      {usage.model_stats && Object.keys(usage.model_stats).length > 0 && (
        <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
          <h2 className="text-xl font-semibold mb-4">모델별 FLOP 사용량</h2>
          <div className="space-y-3">
            {Object.entries(usage.model_stats).map(([model, stats]: [string, any]) => (
              <div key={model} className="flex justify-between items-center p-3 bg-gray-50 dark:bg-gray-700 rounded">
                <div>
                  <div className="font-medium">{model}</div>
                  <div className="text-sm text-gray-500">
                    {stats.requests}회 호출 · {stats.tokens.toLocaleString()} 토큰
                  </div>
                </div>
                <div className="text-right">
                  <div className="font-bold">{formatFlops(stats.flops)}</div>
                  <div className="text-sm text-gray-500">${stats.cost?.toFixed(4) || '0.0000'}</div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* 데이터가 없는 경우 안내 */}
      {(!usage.total_flops || usage.total_flops === 0) && (
        <div className="border border-blue-200 dark:border-blue-800 rounded-lg p-6 text-center" style={{ backgroundColor: 'var(--card-bg)' }}>
          <div className="text-blue-600 dark:text-blue-400 text-lg mb-2">📊</div>
          <h3 className="text-lg font-medium text-blue-900 dark:text-blue-100 mb-2">
            FLOP 사용 데이터가 없습니다
          </h3>
          <p className="text-blue-700 dark:text-blue-300 text-sm">
            AI 모델을 사용하면 FLOP 사용량이 자동으로 추적됩니다.
          </p>
        </div>
      )}
    </div>
  );
  }

// 사용자별 사용량 필터링 함수(간단 예시)
function filterUsageStatsByUser(usageStats: UsageStats | null, userId: string | undefined): UsageStats | null {
  if (!usageStats || !userId) return null;
  // call_details, session_stats 등에서 userId로 필터링
  return {
    ...usageStats,
    call_details: (usageStats.call_details || []).filter((c: any) => c.userId === userId),
    session_stats: (usageStats.session_stats || []).filter((s: any) => s.userId === userId),
    today_session_bubble: (usageStats.today_session_bubble || []).filter((s: any) => s.userId === userId),
    model_stats: (usageStats.model_stats || []).filter((m: any) => m.userId === userId),
    // provider_stats, monthly_stats 등은 필요시 추가 가공
  };
} 



// 메인 컴포넌트를 Suspense로 감싸기 - 개선된 버전
export default function MonitoringPage() {
  return (
    <Suspense fallback={
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-2 text-gray-600 dark:text-gray-400">채팅을 로딩하고 있습니다...</p>
        </div>
      </div>
    }>
      <MonitoringPageContent />
    </Suspense>
  );
}