import { NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';

function getAppDataPath() {
  switch (process.platform) {
    case 'win32':
      return path.join(process.env.APPDATA || '', 'airun');
    case 'darwin':
      return path.join(os.homedir(), 'Library', 'Application Support', 'airun');
    default:
      return path.join(os.homedir(), '.airun');
  }
}

async function readLastLinesFromBigFile(
  filePath: string,
  lines: number = 50,
  chunkSize = 128 * 1024
): Promise<string[]> {
  const fd = await fs.promises.open(filePath, 'r');
  try {
    const stat = await fd.stat();
    let pos = stat.size;
    let leftover = '';
    const collected: string[] = [];

    const buf = Buffer.allocUnsafe(chunkSize);

    while (pos > 0 && collected.length < lines) {
      const readLen = Math.min(chunkSize, pos);
      const start = pos - readLen;

      const { bytesRead } = await fd.read(buf, 0, readLen, start);
      const chunkStr = buf.toString('utf8', 0, bytesRead);
      const combined = chunkStr + leftover;

      let parts = combined.split(/\r?\n/);

      if (start > 0) {
        leftover = parts.shift() ?? '';
      } else {
        leftover = '';
      }

      if (parts.length) {
        collected.unshift(...parts);
        if (collected.length > lines) {
          collected.splice(0, collected.length - lines);
        }
      }

      pos = start;
    }

    if (leftover.trim().length && collected.length < lines) {
      collected.unshift(leftover);
      if (collected.length > lines) {
        collected.splice(0, collected.length - lines);
      }
    }

    return collected.slice(-lines);
  } finally {
    await fd.close();
  }
}

export async function readLogFile(fileName: string, lines = 50): Promise<string[]> {
  try {
    const appDataDir = getAppDataPath();
    const logFilePath = path.join(appDataDir, 'logs', fileName);

    if (!fs.existsSync(logFilePath)) {
      console.log(`로그 파일이 존재하지 않습니다: ${logFilePath}`);
      return [];
    }

    return await readLastLinesFromBigFile(logFilePath, lines);
  } catch (error) {
    console.error('로그 파일 읽기 오류:', error);
    return [];
  }
}

export async function GET(req: Request) {
  try {
    const { searchParams } = new URL(req.url);
    const fileName = searchParams.get('file') || 'airun-api.log';
    const lines = parseInt(searchParams.get('lines') || '50');
    
    const logs = await readLogFile(fileName, lines);
    
    return NextResponse.json({ 
      success: true, 
      logs,
      fileName,
      totalLines: logs.length
    });
  } catch (error) {
    console.error('로그 API 오류:', error);
    return NextResponse.json({ 
      success: false, 
      error: { message: error instanceof Error ? error.message : String(error) },
      logs: []
    }, { status: 500 });
  }
} 